package ch.rolfp.mond

import kotlin.math.*
//val PI:Double = (atan(1.0)*4.0)
const val PI:Double = kotlin.math.PI
const val ZWEIPI:Double = 2.0*PI
const val GRAD:Double = PI/180.0 // Umrechnungsfaktor von Grad in RAD
const val WINKELMINUTE:Double = GRAD/60.0
const val WINKELSTD:Double = 360.0/24.0
//const val erdradius:Double = 6.3675e6 //in Meter, polarer Radius=6357, Aequatorialer Radius=6378 km
const val sunMoonRadius:Double = 16.0/60.0  //in Grad
var mondAbstand:Double = 0.0 //in Meter
var moonPosx=0.0; var moonPosy=0.0; var moonPosz=0.0
var getPosFlag: Boolean = false
class jahrMonatTagStunde(val jahr:Int,val monat:Int,val tag:Int,val stunde:Int)

fun calcMJD(jahr:Int, monat:Int, tag:Int, stunde:Int, minute:Int, sekunde:Int):Double { //Zeit in UT
    var y:Int; var m:Int; var B:Int; val d:Int=tag
    val h:Double = stunde+minute/60.0+sekunde/3600.0
    if(monat<=2) {y=jahr-1; m=monat+13;}
    else         {y=jahr;   m=monat+1;}
    B = if(y<1582 || (y==1582 && (monat<10 || (monat==10 && tag<=4))))
              (y+4716)/4-1181 //bis 4.Okt.1582
        else  y/400-y/100+y/4 //ab 15.Okt.1582
    return 365.0*y-679004+B+floor(30.6*m)+d+h/24.0
}

// Einfache Formel fuer Mondposition, genauere Formel in moonequ.kt
// beides von Planetium kopiert.
/*-----------------------------------------------------------------------*/
/* MINI_MOON: low precision lunar coordinates (approx. 5'/1')            */
/*            T  : time in Julian centuries since J2000                  */
/*                 ( T=(JD-2451545)/36525 )                              */
/*            RA : right ascension (in h; equinox of date)               */
/*            DEC: declination (in deg; equinox of date)                 */
/*-----------------------------------------------------------------------*/
fun miniMoon(t:Double):Pair<Double,Double> { //Rueckgabewerte: ra, dec
    val arc:Double=206264.8062; val coseps:Double=0.91748; val sineps:Double=0.39778 // cos/sin(obliquity ecliptic)
    /* mean elements of lunar orbit */
    val l0 =     frac(0.606433+1336.855225*t) /* mean longitude Moon (in rev) */
    val l  = ZWEIPI*frac(0.374897+1325.552410*t) /* mean anomaly of the Moon  */
    val ls = ZWEIPI*frac(0.993133+  99.997361*t) /* mean anomaly of the Sun   */
    val d  = ZWEIPI*frac(0.827361+1236.853086*t) /* diff. longitude Moon-Sun  */
    val f  = ZWEIPI*frac(0.259086+1342.227825*t) /* mean argument of latitude */
    val dl = 22640*sin(l) - 4586*sin(l-2*d) + 2370*sin(2*d) + 769*sin(2*l) -
    	     668*sin(ls) - 412*sin(2*f) - 212*sin(2*l-2*d) - 206*sin(l+ls-2*d) +
	     192*sin(l+2*d) - 165*sin(ls-2*d) - 125*sin(d) - 110*sin(l+ls) +
	     148*sin(l-ls) - 55*sin(2*f-2*d)
    val s = f + (dl+412*sin(2*f)+541*sin(ls)) / arc
    val h = f-2*d
    val n = -526*sin(h) + 44*sin(l+h) - 31*sin(-l+h) - 23*sin(ls+h) +
            11*sin(-ls+h) - 25*sin(-2*l+f) + 21*sin(-l+f)
    val l_moon = ZWEIPI*frac(l0 + dl/1296e3) // in rad
    val b_moon = (18520.0*sin(s) + n ) / arc // in rad
    /* equatorial coordinates */
    val cb = cos(b_moon)
    val x = cb*cos(l_moon); val v = cb*sin(l_moon); val w = sin(b_moon)
    val y = coseps*v-sineps*w; val z = sineps*v+coseps*w; val rho = sqrt(1.0-z*z)
    val dec:Double = (360.0/ZWEIPI)*atan(z/rho) //Deklination in Grad
    var ra:Double  = ( 48.0/ZWEIPI)*atan(y/(x+rho)); if(ra<0) ra+=24.0
    return Pair(ra,dec)
}

// Berechnung von Hohe und Azimut aus Rektaszension und Deklination (ra dec):

fun lmst(mjd: Double, lambda: Double):Double {
    val mjd0:Double = floor(mjd)
    val ut = (mjd - mjd0) * 24;  val t = (mjd0 - 51544.5) / 36525.0
    val gmst = 6.697374558 + 1.0027379093*ut + (8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0
    return 24.0*frac((gmst-lambda/15.0) / 24.0)
}

fun equhor(dec:Double, tau:Double, phi:Double):Pair<Double,Double> {
    val cs_phi = cos(phi*GRAD); val sn_phi = sin(phi*GRAD)
    val cs_dec = cos(dec*GRAD); val sn_dec = sin(dec*GRAD); val cs_tau = cos(tau*GRAD)
    val x = cs_dec*sn_phi*cs_tau - sn_dec*cs_phi
    val y = cs_dec*sin(tau*GRAD)
    val z = cs_dec*cs_phi*cs_tau + sn_dec*sn_phi
    val rho:Double = sqrt(x*x+y*y)
    var az = atan2(y,x)/GRAD; if(az<0.0) az+=360.0
    val ho = atan2(z,rho)/GRAD
    return Pair(ho,az)
}

// Hoehe und Azimut berechnen:
// Eingaben: Zeit (mjd), Rektaszension, Deklination (ra,dec), Geographische Koordinaten (Lat,Lon)
// ra entspricht Geogr.Laenge am Himmelszelt, dec entspricht Breite
// Lon ist Geogr.Laenge, Lat ist Breite
// Ergebnisse: hoehe, azimut  (beides in Grad)

fun hoeheAzimutBerechnen(mjd:Double, ra:Double, dec:Double, rr:Double, lat:Double, lon:Double):Pair<Double,Double> {
    val abstand:Double = if(rr==0.0) 384.4e6 else rr //durchschnittlicher Abstand des Mondes in Meter
    val rr = abstand/erdradius //Abstand in Erdradien (ungenau)
    val kabst = 1.0/rr //Kehrwert des Abstands in Erdradien
    val paralax:Double = asin(kabst)/GRAD
    val g_phi=lat //in Grad
    val lambda= -lon //in Grad
    val sternzeit = 15.0*lmst(mjd,lambda) //lambda in Grad
    var(ho,az) = equhor(dec,sternzeit-ra,g_phi) //Liefert Werte nach ho,az  in Grad
    ho -= paralax*cos(ho*GRAD)
    return Pair(ho,az) //Hoehe und Azimut
}

// Die folgenden beiden Funktionen werden von  MainActivity.kt aufgerufen:

fun umrechnungZonenzeitUT(zeit:jahrMonatTagStunde, zeitzone:Int):jahrMonatTagStunde
{
    val monatstage = arrayOf(0,31,28,31,30,31,30,31,31,30,31,30,31)
    var jahr:Int=zeit.jahr; var monat:Int=zeit.monat; var tag:Int=zeit.tag; var stunde:Int=zeit.stunde
    var schaltjahr:Int = 0
    if((jahr%4)==0 && (jahr%100!=0 || jahr%400==0)) schaltjahr=1
    stunde -= zeitzone
    if(stunde>=24) {
    stunde -= 24
     val max:Int = if(schaltjahr==1 && monat==2) 29 else monatstage[monat]
     if(++tag>max) {
        tag=1
        if(++monat==13) {monat=1; ++jahr;}
    }
 } else if(stunde<0) {
    stunde += 24
     if(--tag==0) {
        if(--monat==0) {tag=31; monat=12; --jahr;}
        else {tag = if(schaltjahr==1 && monat==2) 29 else monatstage[monat];}
    }
  }
 return jahrMonatTagStunde(jahr,monat,tag,stunde)
}

fun mondpositionBerechnen(jahr1:Int, monat1:Int, tag1:Int, //Datum und
                 stunde1:Int, minute:Int, sekunde:Int, //Zeit in UT oder MEZ oder MESZ
                 lat:Double, lon:Double, //Geographische Koordinaten in Grad
                 zeitzone:Int, //0=UT, 1=MEZ, 2=MESZ
                 refraktflag:Int //1=Refraktion beruecksichtigen, 2=zweite Formel
			     ):Triple<Double,Double,Double> //Rueckgabewerte Hoehe Azimut in Rad
{
    var jahr:Int; var monat:Int; var tag:Int; var stunde:Int
    if(zeitzone!=0) { //Umrechnung der Zeit in UT
        val zeit1:jahrMonatTagStunde = jahrMonatTagStunde(jahr1, monat1, tag1, stunde1)
   	    val zeit = umrechnungZonenzeitUT(zeit1, zeitzone)
  	    jahr=zeit.jahr; monat=zeit.monat; tag=zeit.tag; stunde=zeit.stunde
    } else {
        jahr=jahr1; monat=monat1; tag=tag1; stunde=stunde1
    }
    val mjd:Double = calcMJD(jahr,monat,tag,stunde,minute,sekunde)
    val tt=(mjd-51544.5)/36525.0
    var ra:Double; var dec:Double; var rr:Double
    if(miniMoonFlag==1) {
      val (ra1,dec1) = miniMoon(tt)
      ra = ra1*WINKELSTD //ra in Grad umrechnen (Bereich -90 bis +90 Grad)
      dec = dec1
      mondAbstand = 384000.0*1000 //durchschnittlicher Mondabstand
      rr=mondAbstand
    } else {
      val (ra2,dec2,rr2) = moonequ(tt)
      ra = ra2; dec = dec2; rr=rr2*erdradius
      //mondAbstand = rr*erdradius //schon in moonequ() gemacht
    }
    var(hoehe,azimut) = hoeheAzimutBerechnen(mjd,ra,dec,rr,lat,lon)
    if(azimut>180.0) azimut -= 360.0 //Azimut in Bereich -180 bis +180 Grad bringen

    //Refraktion beruecksichtigen:
    var re:Double = 0.0
    if(refraktflag!=0 && hoehe<45 && hoehe > -45) {
        re = calcRefraction(hoehe,refraktflag)
        hoehe += re*WINKELMINUTE/GRAD
    }
    return Triple(hoehe*GRAD,azimut*GRAD,re)
}

//var testString:String = ""//test
fun barometerFormel(h:Double): Double {
    val p0:Double = 1010.0; val k:Double = 0.03417
    val p:Double = p0*exp(-k/283*h)
    return p //Druck auf h Meter ueber Meer
}
fun calcRefraction(hoehe:Double,refraktflag:Int): Double { //hoehe in Grad, Rueckgabe in Winkelminuten
    var h:Double = hoehe
    var re:Double = 0.0
    if(refraktflag==1) {
        if(h<0) {h = if(h < -2) -(h+2) else 0.0}
        re = 1.02/tan((h+10.3/(h+5.11))*GRAD)
    } else if(refraktflag==2) {
	    //TODO: alternative Refraktionsformel ausprobieren
        val a:Double = 3200.0 //Dicke der Atmosphaere in Meter
        val nLuft:Double = 1.00028 //Brechungsindex der Luft
        //val sinBeta = erdradius / (erdradius + a) * sin((h + 90)*GRAD)
        val sinBeta = erdradius / (erdradius + a) * cos(h*GRAD)
        val alfa = asin(nLuft * sinBeta)
        val beta = asin(sinBeta)
        re = (alfa - beta)/WINKELMINUTE
    }
    //val temp:Double = 20.0 //bei 20 Grad Celsius (statt 10Grad Standardformel)
    //re *= p/1010.0 * 283.0/(273.0+temp) //mit Korrektur von Druck und Temperatur
    re *= luftdruck/1010.0
    return re
}

fun eclequ(t:Double,x:Double,y:Double,z:Double): Triple<Double,Double,Double> {
    val eps = 23.43929111-(46.8150+(0.00059-0.001813*t)*t)*t/3600.0
    val c = cs(eps);  val s = sn(eps)
    val y1 = c*y-s*z;  val z1 = s*y+c*z
    return Triple(x,y1,z1)
}

var planetR:Double = 0.0//test
fun planetpositionBerechnen(jahr1:Int, monat1:Int, tag1:Int, //Datum und
                            stu1:Int, min:Int, sek:Int, //Zeit in UT oder MEZ oder MESZ
                            lat:Double, lon:Double, //Geographische Koordinaten in Grad
                            zeitzone:Int, //0=UT, 1=MEZ, 2=MESZ
                            refraktflag:Int, //1=Refraktion beruecksichtigen
                            nr:Int //Planetnummer (0=Merkur, 1=Venus, 2=Mars, 3=Jupiter, 4=Saturn)
                            ): Array<Double> { //Rueckgabewerte hoehe azimut in Rad, rr in Meter

    var jahr:Int; var monat:Int; var tag:Int; var stunde:Int
    val planetNr = if(nr<0 || nr>4) 1 else nr
    if(zeitzone!=0) {
        //Umrechnung der Zeit in UT
        val zeit1:jahrMonatTagStunde = jahrMonatTagStunde(jahr1, monat1, tag1, stu1)
        val zeit:jahrMonatTagStunde = umrechnungZonenzeitUT(zeit1, zeitzone)
        jahr=zeit.jahr; monat=zeit.monat; tag=zeit.tag; stunde=zeit.stunde
    }
    else {jahr=jahr1; monat=monat1; tag=tag1; stunde=stu1}
    val mjd:Double = calcMJD(jahr,monat,tag,stunde,min,sek)
    val tt=(mjd-51544.5)/36525.0

    val(ra,dec,rr,rs) = planetEqu(planetNr,tt)
    var(hoehe,azimut) = hoeheAzimutBerechnen(mjd,ra,dec,planetAbstand,lat,lon)
    if(azimut>180.0) azimut -= 360.0 //Azimut in Bereich -180 bis +180 Grad bringen

    //Refraktion beruecksichtigen:
    if(refraktflag!=0 && hoehe<45 && hoehe > -45) {
        val R:Double = calcRefraction(hoehe,refraktflag)
        hoehe += R*WINKELMINUTE/GRAD
        planetR = R//test
    } else planetR=0.0
    return arrayOf(hoehe,azimut,rr,rs) //rr=Abstand Erde-Planet, rs=Abstand Erde-Sonne
}
