package ch.rolfp.mond

import kotlin.math.*

//const val ZWEIPI:Double = 2.0*PI
//const val GRAD:Double = PI/180.0
// x*GRAD zum umrechnen von Grad in RAD
// x/GRAD zum umrechnen von RAD in Grad

const val arc:Double = 3600.0*180.0/PI // arcsec per radian
const val erdradius:Double = 6.3675e6 //Radius der Erde in Meter (Mittelwert Pol Equ)
/*
const val erdradiusMit:Double = 6.371e6; //mittlerer Radius gemaess Wikipedia
const val erdradiusPol:Double = 6.3567e6; //Radius an den Polen
const val erdradiusEqu:Double = 6.3781e6; //Radius am Aequator
const val erdradius45:Double =  6.3754e6; //Radius auf 45.Breitengrad (selbst berechnet)
*/

fun cs(x:Double):Double {return cos(x*GRAD)}
fun sn(x:Double):Double {return sin(x*GRAD)}
fun atn2(x:Double,y:Double):Double {return atan2(x,y)/GRAD}

/*** aus astro.cc und astro2.cc kopiertes: ***/
fun addthe(c1:Double, s1:Double, c2:Double, s2:Double): Pair<Double,Double> {
    val c=c1*c2-s1*s2
    val s=s1*c2+c1*s2
    return Pair(c,s)
}

fun addthey(c1:Double, s1:Double, c2:Double, s2:Double): Double {
    val s=s1*c2+c1*s2
    return s
}

/*-----------------------------------------------------------------------*/
/* CART: conversion of polar coordinates (r,theta,phi)                   */
/*       into cartesian coordinates (x,y,z)                              */
/*       (theta in [-90 deg,+90 deg]; phi in [-360 deg,+360 deg])        */
/*-----------------------------------------------------------------------*/
fun cart(r:Double, theta:Double, phi:Double): Array<Double> {
    val rcst:Double = r*cs(theta)
    val x = rcst*cs(phi); val y = rcst*sn(phi); val z = r*sn(theta)
    return arrayOf(x,y,z)
}

/*-----------------------------------------------------------------------*/
/* ECLEQU: Conversion of ecliptic into equatorial coordinates            */
/*         (T: equinox in Julian centuries since J2000)                  */
/*-----------------------------------------------------------------------*/
fun eclequ(t:Double, y:Double, z:Double): Array<Double> {
    val eps = 23.43929111-(46.8150+(0.00059-0.001813*t)*t)*t/3600.0
    val c = cs(eps);  val s = sn(eps)
    val y1 = c*y-s*z;  val z1 = s*y+c*z
    return arrayOf(y1,z1)
}


/*-----------------------------------------------------------------------*/
/* POLAR: conversion of cartesian coordinates (x,y,z)                    */
/*        into polar coordinates (r,theta,phi)                           */
/*        (theta in [-90 deg,+90 deg]; phi in [0 deg,+360 deg])          */
/*-----------------------------------------------------------------------*/
fun polar(x:Double, y:Double, z:Double): Array<Double> {
    var rho = x*x+y*y
    val r = sqrt(rho+z*z)
    var phi = atn2(y,x); if(phi<0.0) phi+=360.0
    rho = sqrt(rho); val theta = atn2(z,rho)
    return arrayOf(r,theta,phi)
}
/*** fertig: aus astro.cc und astro2.cc kopiertes ***/

/*-----------------------------------------------------------------------*/
/* MOON: analytical lunar theory by E.W.Brown (Improved Lunar Ephemeris) */
/*       with an accuracy of approx. 1"                                  */
/*                                                                       */
/*       T:      time in Julian centuries since J2000 (Ephemeris Time)   */
/*               (T=(JD-2451545.0)/36525.0)                              */
/*       LAMBDA: geocentric ecliptic longitude (equinox of date)         */
/*       BETA:   geocentric ecliptic latitude  (equinox of date)         */
/*       R:      geocentric distance (in Earth radii)                    */
/*-----------------------------------------------------------------------*/

//global variables: //TODO: in Klasse legen
    var dgam:Double=0.0; var fac:Double=0.0
    var dlam:Double=0.0; var gam1c:Double=0.0; var sinpi:Double=0.0
    var l0:Double=0.0; var ls:Double=0.0; var gf:Double=0.0
var dl0:Double=0.0; var dl:Double=0.0; var dls:Double=0.0; var df:Double=0.0
var dd:Double=0.0; var ds:Double=0.0

class ArrayReal(ug:Int,og:Int) {
    var offset:Int = -ug
    val size:Int = og-ug+1
    val feld = Array<Double>(size){_ -> 0.0}
    operator fun get(i:Int):Double {
        return feld[i+offset]
    }
    operator fun set(i:Int,x:Double) {
        feld[i+offset] = x
    }
}
var co1:ArrayReal = ArrayReal(-6,6)
var co2:ArrayReal = ArrayReal(-6,6)
var co3:ArrayReal = ArrayReal(-6,6)
var co4:ArrayReal = ArrayReal(-6,6)
var si1:ArrayReal = ArrayReal(-6,6)
var si2:ArrayReal = ArrayReal(-6,6)
var si3:ArrayReal = ArrayReal(-6,6)
var si4:ArrayReal = ArrayReal(-6,6)

fun frac(x:Double):Double {
    val result:Double = x - floor(x)
    return if(result<0.0) result+1.0 else result
}

/* calculate sin(phi); phi in units of 1 revolution = 360 degrees */
fun sine(phi:Double):Double {return sin(ZWEIPI*frac(phi));}

/*----------------------------------------------------------------*/
/* calculate long-periodic changes of the mean elements           */
/* l,l',F,D and L0 as well as dgamma                              */
/*----------------------------------------------------------------*/
fun longPeriodic(t:Double) {
 val s1 = sine(0.19833+0.05611*t); val s2 = sine(0.27869+0.04508*t)
    val s3 = sine(0.16827-0.36903*t); val s4 = sine(0.34734-5.37261*t)
    val s5 = sine(0.10498-5.37899*t); val s6 = sine(0.42681-0.41855*t)
    val s7 = sine(0.14943-5.37511*t)
    dl0 = 0.84*s1+0.31*s2+14.27*s3+7.26*s4+0.28*s5+0.24*s6
    dl  = 2.94*s1+0.31*s2+14.27*s3+9.34*s4+1.12*s5+0.83*s6
    dls = -6.40*s1 -1.89*s6
    df = 0.21*s1+0.31*s2+14.27*s3-88.70*s4-15.30*s5+0.24*s6-1.86*s7
    dd = dl0-dls
    dgam = -3332e-9 * sine(0.59734-5.37261*t) - 539e-9 * sine(0.35498-5.37899*t) - 64e-9 * sine(0.39943-5.37511*t)
 // Rueckgabe: globale Variablen gesetzt: (dl0,dl,dls,df,dd,dgam)
}

/*----------------------------------------------------------------*/
/* INIT: calculates the mean elements and their sine and cosine   */
/* l mean anomaly of the Moon     l' mean anomaly of the Sun      */
/* F mean distance from the node  D  mean elongation from the Sun */
/*----------------------------------------------------------------*/
fun init(t:Double)
{
 val t2:Double=t*t
 dlam=0.0; ds=0.0; gam1c=0.0; sinpi=3422.7000
    longPeriodic(t) //results in global variables: (dl0,dl,dls,df,dd,dgam)
 l0 = ZWEIPI*frac(0.60643382+1336.85522467*t-0.00000313*t2) + dl0/arc
    val l = ZWEIPI*frac(0.37489701+1325.55240982*t+0.00002565*t2) + dl /arc
    ls = ZWEIPI*frac(0.99312619+  99.99735956*t-0.00000044*t2) + dls/arc
    gf = ZWEIPI*frac(0.25909118+1342.22782980*t-0.00000892*t2) + df /arc
    val d  = ZWEIPI*frac(0.82736186+1236.85308708*t-0.00000397*t2) + dd /arc
    var arg:Double = l; var max = 4; var fac:Double = 1.000002208
    co1[0] = 1.0; co1[1] = cos(arg)*fac
    si1[0] = 0.0; si1[1] = sin(arg)*fac
    for(j in 2..max)
     {val(a,b) = addthe(co1[j-1],si1[j-1],co1[1],si1[1]); co1[j]=a; si1[j]=b}
 for(j in 1..max)
     {co1[-j] = co1[j]; si1[-j] = -si1[j];}

 arg = ls; max = 3; fac = 0.997504612-0.002495388*t
    co2[0] = 1.0; co2[1] = cos(arg)*fac
    si2[0] = 0.0; si2[1] = sin(arg)*fac
    for(j in 2..max)
   {val(a,b) = addthe(co2[j-1],si2[j-1],co2[1],si2[1]); co2[j]=a; si2[j]=b}
 for(j in 1..max)
   {co2[-j] = co2[j]; si2[-j] = -si2[j];}

 arg = gf;  max = 4; fac = 1.000002708+139.978*dgam
    co3[0] = 1.0; co3[1] = cos(arg)*fac
    si3[0] = 0.0; si3[1] = sin(arg)*fac
    for(j in 2..max)
   {val(a,b) = addthe(co3[j-1],si3[j-1],co3[1],si3[1]); co3[j]=a; si3[j]=b}
 for(j in 1..max)
   {co3[-j] = co3[j]; si3[-j] = -si3[j];}

 arg = d;  max = 6; fac = 1.0
    co4[0] = 1.0; co4[1] = cos(arg)*fac
    si4[0] = 0.0; si4[1] = sin(arg)*fac
    for(j in 2..max)
   {val(a,b) = addthe(co4[j-1],si4[j-1],co4[1],si4[1]); co4[j]=a; si4[j]=b} 
 for(j in 1..max)
   {co4[-j] = co4[j]; si4[-j] = -si4[j];}
}

/* TERM calculates X=cos(p*arg1+q*arg2+r*arg3+s*arg4) and   */
/*                 Y=sin(p*arg1+q*arg2+r*arg3+s*arg4)       */
fun term(p:Int, q:Int, r:Int, s:Int): Pair<Double,Double>
{
 //val(a1,b1) = addthe(co1[p],si1[p],co2[q],si2[q]); var x=a1; var y=b1
 var(x,y) = addthe(co1[p],si1[p],co2[q],si2[q])
 val(a2,b2) = addthe(x,y,co3[r],si3[r]); x=a2; y=b2 //noch weiter optimierbar
 val(a3,b3) = addthe(x,y,co4[s],si4[s]); x=a3; y=b3
 return Pair(x,y)
}

fun addsol(coeffl:Double, coeffs:Double, coeffg:Double, coeffp:Double,
	    p:Int, q:Int, r:Int, s:Int) {
 val(x,y) = term(p,q,r,s)
    dlam += coeffl*y; ds += coeffs*y
    gam1c += coeffg*x; sinpi += coeffp*x
}

fun solar1() {
 addsol(   13.902,   14.06,-0.001,   0.2607,0, 0, 0, 4)
    addsol(    0.403,   -4.01, 0.394,   0.0023,0, 0, 0, 3)
    addsol( 2369.912, 2373.36, 0.601,  28.2333,0, 0, 0, 2)
    addsol( -125.154, -112.79,-0.725,  -0.9781,0, 0, 0, 1)
    addsol(    1.979,    6.98,-0.445,   0.0433,1, 0, 0, 4)
    addsol(  191.953,  192.72, 0.029,   3.0861,1, 0, 0, 2)
    addsol(   -8.466,  -13.51, 0.455,  -0.1093,1, 0, 0, 1)
    addsol(22639.500,22609.07, 0.079, 186.5398,1, 0, 0, 0)
    addsol(   18.609,    3.59,-0.094,   0.0118,1, 0, 0,-1)
    addsol(-4586.465,-4578.13,-0.077,  34.3117,1, 0, 0,-2)
    addsol(    3.215,    5.44, 0.192,  -0.0386,1, 0, 0,-3)
    addsol(  -38.428,  -38.64, 0.001,   0.6008,1, 0, 0,-4)
    addsol(   -0.393,   -1.43,-0.092,   0.0086,1, 0, 0,-6)
    addsol(   -0.289,   -1.59, 0.123,  -0.0053,0, 1, 0, 4)
    addsol(  -24.420,  -25.10, 0.040,  -0.3000,0, 1, 0, 2)
    addsol(   18.023,   17.93, 0.007,   0.1494,0, 1, 0, 1)
    addsol( -668.146, -126.98,-1.302,  -0.3997,0, 1, 0, 0)
    addsol(    0.560,    0.32,-0.001,  -0.0037,0, 1, 0,-1)
    addsol( -165.145, -165.06, 0.054,   1.9178,0, 1, 0,-2)
    addsol(   -1.877,   -6.46,-0.416,   0.0339,0, 1, 0,-4)
    addsol(    0.213,    1.02,-0.074,   0.0054,2, 0, 0, 4)
    addsol(   14.387,   14.78,-0.017,   0.2833,2, 0, 0, 2)
    addsol(   -0.586,   -1.20, 0.054,  -0.0100,2, 0, 0, 1)
    addsol(  769.016,  767.96, 0.107,  10.1657,2, 0, 0, 0)
    addsol(    1.750,    2.01,-0.018,   0.0155,2, 0, 0,-1)
    addsol( -211.656, -152.53, 5.679,  -0.3039,2, 0, 0,-2)
    addsol(    1.225,    0.91,-0.030,  -0.0088,2, 0, 0,-3)
    addsol(  -30.773,  -34.07,-0.308,   0.3722,2, 0, 0,-4)
    addsol(   -0.570,   -1.40,-0.074,   0.0109,2, 0, 0,-6)
    addsol(   -2.921,  -11.75, 0.787,  -0.0484,1, 1, 0, 2)
    addsol(    1.267,    1.52,-0.022,   0.0164,1, 1, 0, 1)
    addsol( -109.673, -115.18, 0.461,  -0.9490,1, 1, 0, 0)
    addsol( -205.962, -182.36, 2.056,   1.4437,1, 1, 0,-2)
    addsol(    0.233,    0.36, 0.012,  -0.0025,1, 1, 0,-3)
    addsol(   -4.391,   -9.66,-0.471,   0.0673,1, 1, 0,-4)
}

fun solar2()
{
 addsol(    0.283,    1.53,-0.111,   0.0060,1,-1, 0, 4)
    addsol(   14.577,   31.70,-1.540,   0.2302,1,-1, 0, 2)
    addsol(  147.687,  138.76, 0.679,   1.1528,1,-1, 0, 0)
    addsol(   -1.089,    0.55, 0.021,   0.0   ,1,-1, 0,-1)
    addsol(   28.475,   23.59,-0.443,  -0.2257,1,-1, 0,-2)
    addsol(   -0.276,   -0.38,-0.006,  -0.0036,1,-1, 0,-3)
    addsol(    0.636,    2.27, 0.146,  -0.0102,1,-1, 0,-4)
    addsol(   -0.189,   -1.68, 0.131,  -0.0028,0, 2, 0, 2)
    addsol(   -7.486,   -0.66,-0.037,  -0.0086,0, 2, 0, 0)
    addsol(   -8.096,  -16.35,-0.740,   0.0918,0, 2, 0,-2)
    addsol(   -5.741,   -0.04, 0.0  ,  -0.0009,0, 0, 2, 2)
    addsol(    0.255,    0.0 , 0.0  ,   0.0   ,0, 0, 2, 1)
    addsol( -411.608,   -0.20, 0.0  ,  -0.0124,0, 0, 2, 0)
    addsol(    0.584,    0.84, 0.0  ,   0.0071,0, 0, 2,-1)
    addsol(  -55.173,  -52.14, 0.0  ,  -0.1052,0, 0, 2,-2)
    addsol(    0.254,    0.25, 0.0  ,  -0.0017,0, 0, 2,-3)
    addsol(    0.025,   -1.67, 0.0  ,   0.0031,0, 0, 2,-4)
    addsol(    1.060,    2.96,-0.166,   0.0243,3, 0, 0, 2)
    addsol(   36.124,   50.64,-1.300,   0.6215,3, 0, 0, 0)
    addsol(  -13.193,  -16.40, 0.258,  -0.1187,3, 0, 0,-2)
    addsol(   -1.187,   -0.74, 0.042,   0.0074,3, 0, 0,-4)
    addsol(   -0.293,   -0.31,-0.002,   0.0046,3, 0, 0,-6)
    addsol(   -0.290,   -1.45, 0.116,  -0.0051,2, 1, 0, 2)
    addsol(   -7.649,  -10.56, 0.259,  -0.1038,2, 1, 0, 0)
    addsol(   -8.627,   -7.59, 0.078,  -0.0192,2, 1, 0,-2)
    addsol(   -2.740,   -2.54, 0.022,   0.0324,2, 1, 0,-4)
    addsol(    1.181,    3.32,-0.212,   0.0213,2,-1, 0, 2)
    addsol(    9.703,   11.67,-0.151,   0.1268,2,-1, 0, 0)
    addsol(   -0.352,   -0.37, 0.001,  -0.0028,2,-1, 0,-1)
    addsol(   -2.494,   -1.17,-0.003,  -0.0017,2,-1, 0,-2)
    addsol(    0.360,    0.20,-0.012,  -0.0043,2,-1, 0,-4)
    addsol(   -1.167,   -1.25, 0.008,  -0.0106,1, 2, 0, 0)
    addsol(   -7.412,   -6.12, 0.117,   0.0484,1, 2, 0,-2)
    addsol(   -0.311,   -0.65,-0.032,   0.0044,1, 2, 0,-4)
    addsol(    0.757,    1.82,-0.105,   0.0112,1,-2, 0, 2)
    addsol(    2.580,    2.32, 0.027,   0.0196,1,-2, 0, 0)
    addsol(    2.533,    2.40,-0.014,  -0.0212,1,-2, 0,-2)
    addsol(   -0.344,   -0.57,-0.025,   0.0036,0, 3, 0,-2)
    addsol(   -0.992,   -0.02, 0.0  ,   0.0   ,1, 0, 2, 2)
    addsol(  -45.099,   -0.02, 0.0  ,  -0.0010,1, 0, 2, 0)
    addsol(   -0.179,   -9.52, 0.0  ,  -0.0833,1, 0, 2,-2)
    addsol(   -0.301,   -0.33, 0.0  ,   0.0014,1, 0, 2,-4)
    addsol(   -6.382,   -3.37, 0.0  ,  -0.0481,1, 0,-2, 2)
    addsol(   39.528,   85.13, 0.0  ,  -0.7136,1, 0,-2, 0)
    addsol(    9.366,    0.71, 0.0  ,  -0.0112,1, 0,-2,-2)
    addsol(    0.202,    0.02, 0.0  ,   0.0   ,1, 0,-2,-4)
}

fun solar3()
{
 addsol(    0.415,    0.10, 0.0  ,  0.0013,0, 1, 2, 0)
    addsol(   -2.152,   -2.26, 0.0  , -0.0066,0, 1, 2,-2)
    addsol(   -1.440,   -1.30, 0.0  ,  0.0014,0, 1,-2, 2)
    addsol(    0.384,   -0.04, 0.0  ,  0.0   ,0, 1,-2,-2)
    addsol(    1.938,    3.60,-0.145,  0.0401,4, 0, 0, 0)
    addsol(   -0.952,   -1.58, 0.052, -0.0130,4, 0, 0,-2)
    addsol(   -0.551,   -0.94, 0.032, -0.0097,3, 1, 0, 0)
    addsol(   -0.482,   -0.57, 0.005, -0.0045,3, 1, 0,-2)
    addsol(    0.681,    0.96,-0.026,  0.0115,3,-1, 0, 0)
    addsol(   -0.297,   -0.27, 0.002, -0.0009,2, 2, 0,-2)
    addsol(    0.254,    0.21,-0.003,  0.0   ,2,-2, 0,-2)
    addsol(   -0.250,   -0.22, 0.004,  0.0014,1, 3, 0,-2)
    addsol(   -3.996,    0.0 , 0.0  ,  0.0004,2, 0, 2, 0)
    addsol(    0.557,   -0.75, 0.0  , -0.0090,2, 0, 2,-2)
    addsol(   -0.459,   -0.38, 0.0  , -0.0053,2, 0,-2, 2)
    addsol(   -1.298,    0.74, 0.0  ,  0.0004,2, 0,-2, 0)
    addsol(    0.538,    1.14, 0.0  , -0.0141,2, 0,-2,-2)
    addsol(    0.263,    0.02, 0.0  ,  0.0   ,1, 1, 2, 0)
    addsol(    0.426,    0.07, 0.0  , -0.0006,1, 1,-2,-2)
    addsol(   -0.304,    0.03, 0.0  ,  0.0003,1,-1, 2, 0)
    addsol(   -0.372,   -0.19, 0.0  , -0.0027,1,-1,-2, 2)
    addsol(    0.418,    0.0 , 0.0  ,  0.0   ,0, 0, 4, 0)
    addsol(   -0.330,   -0.04, 0.0  ,  0.0   ,3, 0, 2, 0)
}

/* part N of the perturbations of ecliptic latitude */
//#define addn(coeffn,p,q,r,s) term(p,q,r,s,x,y); n += coeffn*y

fun addn(coeffn:Double, p:Int, q:Int, r:Int, s:Int): Double {
    var(x,y) = addthe(co1[p],si1[p],co2[q],si2[q])
    val(a2,b2) = addthe(x,y,co3[r],si3[r]); x=a2; y=b2 //noch weiter optimierbar
    y = addthey(x,y,co4[s],si4[s])
    return coeffn*y
}

fun solarn(): Double
{
 var n:Double = 0.0
 n += addn(-526.069, 0, 0,1,-2)
 n += addn( -3.352,  0, 0,1,-4)
 n += addn(  44.297, 1, 0,1,-2); n += addn( -6.000, 1, 0,1,-4)
 n += addn(  20.599,-1, 0,1, 0); n += addn(-30.598,-1, 0,1,-2)
 n += addn( -24.649,-2, 0,1, 0); n += addn( -2.000,-2, 0,1,-2)
 n += addn( -22.571, 0, 1,1,-2); n += addn( 10.985, 0,-1,1,-2)
 return n
}

/* perturbations of ecliptic latitude by Venus and Jupiter          */
fun planetary(dlam0:Double,t:Double) {
    dlam = dlam0 + 0.82*sine(0.7736  -62.5512*t)+0.31*sine(0.0466 -125.1025*t)
         + 0.35*sine(0.5785  -25.1042*t)+0.66*sine(0.4591+1335.8075*t)
         + 0.64*sine(0.3130  -91.5680*t)+1.14*sine(0.1480+1331.2898*t)
         + 0.21*sine(0.5918+1056.5859*t)+0.44*sine(0.5784+1322.8595*t)
         + 0.24*sine(0.2275   -5.7374*t)+0.28*sine(0.2965   +2.6929*t)
         + 0.33*sine(0.3132   +6.3368*t)
}

fun moon(t:Double): Array<Double>
{
    init(t); solar1(); solar2(); solar3(); val n:Double = solarn(); planetary(dlam,t)
    val lambda:Double = 360.0*frac((l0+dlam/arc)/ZWEIPI)
    val s = gf + ds/arc
    fac  = 1.000002708+139.978*dgam
    val beta = (fac*(18518.511+1.189+gam1c)*sin(s)-6.24*sin(3*s)+n) / 3600.0
    sinpi *= 0.999953253
    val r = arc/sinpi
    return arrayOf(lambda,beta,r)
}

/*-----------------------------------------------------------------------*/
/* MOONEQU: geocentric equatorial coordinates of the Moon                */
/*          referred to the true equinox of date                         */
/*   T   time in Julian centuries ephemeris time since J2000             */
/*       ( T = (JD-2451545.0)/36525 )                                    */
/*   RA  right ascension (deg)                                           */
/*   DEC declination (deg)                                               */
/*   R   distance (in earth radii)                                       */
/*-----------------------------------------------------------------------*/
fun moonequ(t:Double): Array<Double>
{
    val(l,b,r) = moon(t)         // ecliptic coordinates (mean equinox of date)
    val(x1,y1,z1) = cart(r,b,l) // transform into xyz coordinates
    val(y2,z2) = eclequ(t,y1,z1)  // transform into equatorial coordinates
    val(x,y,z) = nutequ(t,x1,y2,z2) // Nutation der Erde
    val(rr,dec,ra) = polar(x,y,z) //rr in Erdradien; dec in [-90 deg,+90 deg]; ra in [0 deg,+360 deg])
    if(getPosFlag) {
        moonPosx=x*erdradius; moonPosy=y*erdradius; moonPosz=z*erdradius
        mondAbstand=rr*erdradius
        moonRa=ra
    }
    return arrayOf(ra,dec,rr)
}
var moonRa=0.0; var sunRa=0.0
fun calcMondPhase():Pair<Double,String> {
    val es = sonneAbstand //Abstand Erde-Sonne
    val em = mondAbstand  //Abstand Erde-Mond
    val ms = sqrt(quadrat(sunPosx-moonPosx)+quadrat(sunPosy-moonPosy)+quadrat(sunPosz-moonPosz))
    val cosArc = (ms*ms+em*em-es*es)/(2.0*em*ms)
    //val phaseArc = acos(cosArc)
    val phase = (cosArc+1.0)*50.0
    /* ungenau:
    var ph:Double = sunRa-moonRa;
    if(ph<0.0) ph+=360.0 else if(ph>360.0) ph-=360.0
    val str:String = if(ph>180.0) "zunehmend" else "abnehmend"
    */
    val alfa = -atan2(sunPosy,sunPosx)
    val py = sin(alfa)*moonPosx + cos(alfa)*moonPosy
    var str = if(py>0) "zunehmend" else if(py<0) "abnehmend" else ""
    if(phase>=99.5) str="Vollmond"
    else if(phase<=0.5) str="Neumond"
    return Pair(phase, str)
}
