/* bahn.cc       letzte Aenderungen: 29.12.2010, 23.9.2023
Steuerung fuer Modelleisenbahn
*/
#define VERSION "0.11"
/*
 Prozessor: Atmega32, 16MHz Quarz
 Schaltung: schaltung/bahnsteuerung2023.sch
	    4 Ausgaenge PB0...PB3 bisher nicht benutzt
	    4 LEDs auf PB4...PB7
	    3 Taster auf PC2...PC4
	    2 Ausgaenge fuer DCC-Steuerung PC0-PC1,PC6-PC7
	    1 Leitung: PC5 als Synchronisation beim Debuggen
   Grafik-LCD: PA0...PA7, PD2-PD7
   Tastatur: PD0 (RxD)

 Autor: Rolf Pfister
 Copyright: Freeware
 History:
 15.5.2010	Erstellung, Uhr vom Lichtwecker uebernommen
 13.6.2010	Testen eines Grafik-LCD-Displays
 7.8.2010	Testen Textausgabe auf Grafik-LCD-Display
 8.8.2010	Tastatur mit Tiny2313
 9.8.2010	Datenuebertragung von Tastatur mit UART
 15.8.2010	DCC-Steuerung, Schema-Aenderungen: Taster statt PC0...PC2
		neu auf PC2...PC4
 19.8.10  0.06	Grafik-LCD neu auf PORTD (statt vorher PORTC)
 23.12.10 0.07  Geschwindigkeiten der Zuege senden
 26.12.10       Synchranisations-Signal zum Debuggen
 28.12.10 0.08  Pruefsumme, Richtungswechsel, Zuglicht
 28.8.23  0.09  Grafikprobleme geloest.
                Grafische Anzeige von Anlage.
 11.9.23  0.10  Weichensteuerung, zweites Zuglicht
                milliwait() verbessert
 18.9.23        Notstop wenn Rote Taste lang gedrueckt
 23.9.23  0.11  Bessere Transistoren eingesetzt: IRF4905 statt IRF9620
                Problem: zu viel Strom beim Umschalten (das war wohl auch das
		eigentliche Problem mit IRF9620)
		Behebung: Transistor-Treiber und Verzoegerung zwischen ausschalten 
		der einen und einschalten der andern Transistoren (in mosfets_schalten()).
		Dazu zusaetzlich PC6 und PC7 verwendet.

*/
#define F_CPU 16000000UL
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <util/delay.h>
#include "ulong.h"
#include "stdio.h"

//#define GRAFIKTEST //Test von Grafikroutinen beim Einschalten
//#define GLEICHSTROMTEST //geht noch nicht
//#define AUSTEST  //Test des Stromverbrauchs wenn Schienenstrom aus

#define UARTVARIANTE
//#define DEBUG
#define DEBUG2

#ifdef GLEICHSTROMTEST
static int8_t gleichspannung=0;
#endif

/**** Vordeklarationen ****/
//void variable_feiertage_berechnen();
void tastaturcheck();
char tastaturabfrage();
void zugeinstellen(int8 zugnr);
void zuganzeigen();
void zugschneller();
void zuglangsamer(int8 vneu= -1);
void zuglichtschalten(int8 nr);
void zugnotstop();
void glcd_printzeile(int zeile,const char *cstr,
		     int p1=0,int p2=0,int p3=0,int p4=0);
void zugdatensenden();
void gleisplan_zeichnen();
void weichestellen(int nr);
void ftaste(int8 loknr,int8 fx);
void ftaste_aktuellerzug(int8 fx);

//Globale Variablen:
static int8_t zeitzumsenden=0;

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=8;
static volatile uchar tage=4,monat=9; //aktuelles Datum
static volatile uint jahr=2023;
static char monatstage[13]={0,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=0; //0=Montag ... 6=Sonntag
static volatile char sommerzeit=1; //0=Winterzeit 1=Sommerzeit

void schaltjahrtest()
{
 if(jahr%4==0 && (jahr%100!=0 || jahr%400==0)) monatstage[2]=29;
 else monatstage[2]=28;
}
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
void schaltjahr_und_wochentagberechnen()
{
 schaltjahrtest();
 wochentagberechnen();
 // variable_feiertage_berechnen();
}

/*
// Feiertage:
const char NFEST=6;//Anzahl feste Feiertage
const char festfeiertag[NFEST]={1, 2, 1, 1, 25, 26};
const char festfeiermon[NFEST]={1, 1, 5, 8, 12, 12};
const char NVARI=4;//Anzahl von Ostern abhaengige Feiertage
static char varifeiertag[NVARI];
static char varifeiermon[NVARI];
void variable_feiertage_berechnen()
{
 int a,b,c,k,p,q,M,N;
 char d,e,tag,i;
 const char mon=3;
 //Osterformel nach Gauss:
 a=jahr%19;
 b=jahr%4;
 c=jahr%7;
 k=jahr/100;
 p=(8*k+13)/25;
 q=k/4;
 M=(15+k-p-q)%30;
 N=(4+k-q)%7;
 d=(19*a+M)%30;
 e=(2*b+4*c+6*d+N)%7;
 tag=22+d+e;
 varifeiertag[0]=tag+1; varifeiermon[0]=mon; //Ostermontag
 varifeiertag[1]=tag-2; varifeiermon[1]=mon; //Karfreitag
 varifeiertag[2]=tag+39; varifeiermon[2]=mon; //Auffahrt
 varifeiertag[3]=tag+50; varifeiermon[3]=mon; //Pfingstmontag
 for(i=0;i<NVARI;i++)
   while(varifeiertag[i]>monatstage[varifeiermon[i]])
    {varifeiertag[i] -= monatstage[varifeiermon[i]]; varifeiermon[i]++;}
}

bool istfeiertag()
{
 char i;
 for(i=0;i<NFEST;i++)
   if(tage==festfeiertag[i] && monat==festfeiermon[i]) return true;
 for(i=0;i<NVARI;i++)
   if(tage==varifeiertag[i] && monat==varifeiermon[i]) return true;
 return false;
}
*/

// Fuer genaue Uhr: Quarz moeglichst genau ausmessen und hier
// eintragen in Hertz:

//#define Hz    16000000UL //exakt 16MHz
//Beispiel: 16MHz ist nach n Tagen z sec vorgegangen:
//nach n Tagen sind also 16000000*(3600*24*n+z) Takte vergangen
//Frequenz ist also:
// x = (3600*24*n+z)*16000000 / (3600*24*n)
//   = 16000000 + z*16000000/(3600*24*n)
#define Hz 16001422UL //ausgemessener Wert fuer genaue Uhr

#define Rest1 (Hz%1000) //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (Hz/1000-1)

//fuer Tastenentprellung:
static volatile char entprelltetasten=0xFF;
static char status1=0xFF,status2=0xFF;
static volatile char tastegedrueckt=0;
static volatile char langgedrueckt=0;
static char status3=0,status4=0,status5=0;
static volatile char tasten_aktiv=1;

//fuer LEDs:
static volatile uint rothell=0,gruenhell=0,gelbhell=0,orangehell=0;//Helligkeit der LEDs
const uchar LEDROT=0x10,LEDGRUEN=0x20,LEDGELB=0x40,LEDORANGE=0x80;
static volatile uchar LEDBITS=0;//Eingeschaltete LEDs
static volatile uchar aktportb=0x00;//aktueller Zustand von PORTB

ISR(TIMER1_COMPA_vect)
{
 uchar mehr;
 uint ms=millisec;
 uint h;//fuer LEDs
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0;
   //if(++sec>=60-Rest2) mehr=1; else mehr=0;
   mehr=0;//einfacher wenn Rest2 und Rest3 sowieso ==0
   if(++sec==60)
    {sec=0;
     //if(++min>=60-Rest3) mehr=1; else mehr=0;
     if(++min==60)
      {min=0; mehr=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
      }
    }
  }
 else if((ms&0x0F)==1) //alle 16ms
  {if(tasten_aktiv)
   {//Tasten-Entprellung: fuer 8 Tasten unabhaengig voneinander
    char pin,ung;
    pin = PINC;
    ung = (pin^status1) |   //Ist PINC verschieden zu status1 oder
          (status2^status1);//ist status2 verschieden zu status1 ?
    entprelltetasten = //entweder Alt oder Neu uebernehmen:
     (entprelltetasten&ung) //ja: Alt=Alter Zustand, nein: Alt=0
     | (pin & (ung^0xFF)); //nein: Neu=PINC, ja: Neu=0
    status2=status1;
    status1=pin;
    tastegedrueckt |= (entprelltetasten^0xFF);
   }
  }
 else if((ms&0xFF)==0xE7) //alle ca 250ms
  {if(tasten_aktiv)
   {//Erkennung von langen Tastendrucken
    langgedrueckt = (entprelltetasten^0xFF) & status3 & status4 & status5;
    status5=status4; status4=status3; status3=(entprelltetasten^0xFF);
   }
  }
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
 h=ms%10;//LEDs in 10 Helligkeitsstufen
 aktportb &= 0x0F;
 if((LEDBITS&LEDROT) && h<rothell) aktportb |= LEDROT;
 if((LEDBITS&LEDGRUEN) && h<gruenhell) aktportb |= LEDGRUEN;
 if((LEDBITS&LEDGELB) && h<gelbhell) aktportb |= LEDGELB;
 if((LEDBITS&LEDORANGE) && h<orangehell) aktportb |= LEDORANGE;
 PORTB = aktportb;
 if(zeitzumsenden<30) zeitzumsenden++;
}

static volatile unsigned char send_byte=0,send_bit=0x80;
//static volatile unsigned char send_byte=0xFF,send_bit=0x80;//test
//wenn send_bit==0x00 dann ist ganzes send_byte gesendet.

void bytesenden(unsigned char c)
{
 while(send_bit) ;//warten bis letztes Byte gesendet
 send_byte=c; send_bit=0x80;
}

void bitsenden(unsigned char c)
{
 while(send_bit) ;//warten bis letztes Byte gesendet
 send_byte=c; send_bit=0x01;
}

#ifdef AUSTEST
ISR(TIMER2_COMP_vect)
{
 static char n=0,pc=1,aktbit=0;
 if(pc==1 && n==0) aktbit=(send_byte&send_bit);
 if(aktbit!=0)
  {//1-Bit senden
   PORTC=(PORTC&0xFC); //+pc;
   pc^=3; //Polaritaet wechseln fuer naechstes Halbbit
   if(pc==1)  send_bit >>= 1; //naechstes zu sendendes Bit einstellen
  }
 else
  {//0-Bit senden
   if(++n==2)
    {PORTC=(PORTC&0xFC); //+pc;
     pc^=3; //Polaritaet wechseln
     n=0;
     if(pc==1)  send_bit >>= 1; //naechstes zu sendendes Bit einstellen
    }
  }
}
#else

inline void mosfets_schalten(char pc)
{
 if(pc==1)
  {
   PORTC &= ~(1<<PC1); PORTC &= ~(1<<PC6); //MOSFETs ausschalten
   _delay_us(1); //TODO: je nach Transistoren und Treiber anpassen
   PORTC |= (1<<PC0); PORTC |= (1<<PC7);  //andere MOSFETs einschalten
  }
 else //if(pc==2)
  {
   PORTC &= ~(1<<PC0); PORTC &= ~(1<<PC7); //MOSFETs ausschalten
   _delay_us(1); //TODO: je nach Transistoren und Treiber anpassen
   PORTC |= (1<<PC1); PORTC |= (1<<PC6);  //andere MOSFETs einschalten
  }
}

ISR(TIMER2_COMP_vect)
{
 //dieser Timer2-Interrupt sollte alle 58usec kommen
 //1-Bits werden mit 58usec = 8620.7Hz moduliert
 //0-Bits mit 116usec = 4310.3Hz
 //(nach Spezifikation nominal 100usec, aber 95 bis 9800 erlaubt)
 static char n=0,pc=1,aktbit=0;
 if(pc==1 && n==0) aktbit=(send_byte&send_bit);
 if(aktbit!=0)
  {//1-Bit senden
   //PORTC=(PORTC&0xFC)+pc; pc^=3; //Polaritaet wechseln fuer naechstes Halbbit
   mosfets_schalten(pc); pc^=3; //Polaritaet wechseln fuer naechstes Halbbit
   if(pc==1)  send_bit >>= 1; //naechstes zu sendendes Bit einstellen
  }
 else
  {//0-Bit senden
#ifdef GLEICHSTROMTEST
   static char k=0;
   if(gleichspannung>0 && n==0)
    {
     if(++k==2) {n++; k=0;} //erste Bithaelfte verlaengern
    }
   else if(gleichspannung<0 && n==1)
    {
     if(++k==2) {n++; k=0;} //zweite Bithaelfte verlaengern
    }
   else {n++; k=0;}//beide Bithaelften gleich lang
   if(n==2)
    {PORTC=(PORTC&0xFC)+pc; pc^=3; //Polaritaet wechseln
     n=0;
     if(pc==1)  send_bit >>= 1; //naechstes zu sendendes Bit einstellen
    }
#else
   if(++n==2)
    {//PORTC=(PORTC&0xFC)+pc; pc^=3; //Polaritaet wechseln
     mosfets_schalten(pc); pc^=3; //Polaritaet wechseln
     n=0;
     if(pc==1)  send_bit >>= 1; //naechstes zu sendendes Bit einstellen
    }
#endif
  }
}

#endif //AUSTEST else ende

//Bei einem unerwarteten Interrupt nichts machen:
//SIGNAL (__vector_default) {}
//Oder besser wirklich einfach nur ein reti machen:
void __attribute__ ((naked)) __vector_default(void) {__asm__ ("reti");}
//Ohne diese Funktion wird nach 0 gesprungen, was einem Reset entspricht.

void milliwait(int n) //n Millisekunden warten  1000 >= n >= 1
{
 uint16_t ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) {sleep_mode();}//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
  {sleep_mode();}//auf n. Millisekunde warten
  //{__asm__("sleep");} //wenn sicher dass sleep nicht versehentlich aufgerufen
}

void secwait(int n)
{
 while(n>0) {milliwait(1000); n--;}
}

/**** Routinen fuer LCD-Ansteuerung ****/
/*
dies hat neuer Compiler wegoptimiert, neue Version in glcd.cc
char microwait(int n) //etwa n Microsekunden warten
{
 char vx=0;
 n*=2;//test
 while(n>0)
  {vx+=n; --n;}
 return vx;
}
*/

#include "glcd.cc"

//Globale Variablen:
const char ROTETASTE=0x08,GRUENETASTE=0x10,SCHWARZETASTE=0x04,ALLETASTEN=0x1C;
const char MINSEK=0,UHRZEIT=1,MOTG=2,WOTAG=3,JJJ=4,WECK1=5,WECK2=6;
static char zeitstellen=0;

void zeitauflcdanzeigen()
{
 char txt[40];
 int zeile=6;
 glcd_goto(zeile,1);
 sprintf(txt,"%02d:%02d:%02d ",stunden,min,sec);
 glcd_write(txt);
 sprintf(txt,"%02d.%02d.%d",tage,monat,jahr);
 glcd_write(txt);
 if(zeitstellen>0)
  {int pos=zeitstellen*3-1;
   if(pos>16) pos=16;//test
   glcd_goto(zeile,pos);
  }
}

void schwarzetaste()
{
 //if(++zeitstellen==7) zeitstellen=0;
 if(zeitstellen>0)
  {
   const char *s="                ";
   if(zeitstellen==8) zeitstellen=1; else zeitstellen++;
   switch(zeitstellen)
    {case 1: s="Std stellen";
     CASE 2: s="Min stellen";
     CASE 3: s="Sec stellen";
     CASE 4: s="Tag stellen";
     CASE 5: s="Monat stellen";
     CASE 6: s="Jahr stellen";
     CASE 7: zeitstellen=0;
    }
   glcd_printzeile(7,s);
  }
 else
  {//Weiche 1 umschalten
   weichestellen(1);
  }
}

void rotetaste()
{
 if(zeitstellen>0)
  switch(zeitstellen)
   {case 1: cli(); if(++stunden==24) stunden=0; sei();
    CASE 2: cli(); if(++min==60) min=0; sei();
    CASE 3: cli(); if(sec>=30) ++min; sec=0; sei();
    CASE 4: if(++tage>monatstage[monat]) tage=1; wochentagberechnen();
    CASE 5: if(++monat==13) monat=1; wochentagberechnen();
    CASE 6: ++jahr; schaltjahr_und_wochentagberechnen();
   }
 else
  {//Weiche 3 umschalten
   milliwait(500); //wenn Rote Taste laenger als halbe Sekunde gedrueckt
   if((entprelltetasten&ROTETASTE)==0) zugnotstop();//alle Zuege stoppen
   else //sonst normale Funktion
    weichestellen(3);
  }
}

void gruenetaste()
{
 if(zeitstellen>0)
  switch(zeitstellen)
   {case 1: cli(); if(stunden==0) stunden=23; else --stunden; sei();
    CASE 2: cli(); if(min==0) min=59; else --min; sei();
    CASE 3: cli(); if(sec>=30) ++min; sec=0; sei();
    CASE 4: if(--tage==0) tage=monatstage[monat]; wochentagberechnen();
    CASE 5: if(--monat==0) monat=12; wochentagberechnen();
    CASE 6: --jahr; schaltjahr_und_wochentagberechnen();
   }
 else
  {//Weiche 2 umschalten
   weichestellen(2);
  }
#ifdef GRAFIKTEST
 glcd_test(2);//test
#endif
}

#ifdef UARTVARIANTE
void uart_init();
#endif

int main(void)
{
 const char WARTE_AUF_TASTENDRUCK=0,WARTE_AUF_TASTELOSLASSEN=1;
 char warten=WARTE_AUF_TASTENDRUCK;
 DDRA  = 0x00; //PortA alles Eingaenge
 DDRB  = 0xFF; //PortB alles Ausgaenge
 DDRC  = 0xC3; //PortC untere 2 Bits und oberste 2 Bits Ausgaenge, Rest Eingaenge
#ifdef DEBUG
 DDRC |= (1<<5); //PortC PC5 auch als Ausgang
#endif
#ifdef DEBUG2
 DDRC |= (1<<5); //PortC PC5 auch als Ausgang
#endif
 DDRD  = 0xFC; //PortD obere 6 Bits Ausgaenge, unter 2 Bits offen fuer UART
 PORTA = 0;//keine Pullups
 PORTB = 0;//alle Ausgaenge auf 0
 PORTC = 0x1C; //C2-C4 Pullup-Widerstaende fuer Taster
 PORTD = 0xE0; //C7-C5 RST,CS1,CS2 auf H (=inaktiv fuer LCD)

 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert

#define DccCompwert (29-1)
 TCCR2 = (1<<WGM21)|(3<<CS20); //CTC-Modus, Prescaler=32, 16MHz/32 --> 2usec
 OCR2 = DccCompwert;
 TCNT2 = 0; //Startwert Timer2
 TIMSK |= (1<<OCIE2); //Timer2 Interrupts bei Vergleichswert
 
 set_sleep_mode(SLEEP_MODE_IDLE);
 sleep_enable(); //Schlaf-Modus waehlen, aber noch nicht schlafen

 sei(); //Interrupt einschalten
 schaltjahr_und_wochentagberechnen();
 do {milliwait(200);} while((entprelltetasten&ALLETASTEN)!=ALLETASTEN);
 langgedrueckt=tastegedrueckt=0;//erste Entprellungen abwarten 
 PORTB=0x20; //test gruene LED an

 milliwait(200);
 glcd_init();
 glcd_goto(5,1);
 glcd_write("Eisenbahn-Steuerung");
 glcd_goto(6,1);
 glcd_write("Version "); glcd_write(VERSION);
 glcd_goto(7,1);
 glcd_write("Zug1 Zug2 Zug3 Zug4 ");//test
 
#ifdef UARTVARIANTE
 uart_init();
#endif

 //Testschlaufen:
 rothell=gruenhell=gelbhell=orangehell=1;//Helligkeit der LEDs
#ifdef GRAFIKTEST 
 while((tastegedrueckt&ALLETASTEN)==0) //Startschlaufe, weiter mit Tastendruck
  {LEDBITS=(sec<<4)&0xF0; //laufende Sekunden auf den LEDs anzeigen
  }
#else
 LEDBITS = 0xF0; //alle LEDs ein
 secwait(3);
#endif
 /** test **
 tastegedrueckt=0;
 while((langgedrueckt&7)==0) //Testschlaufe, weiter mit langem Tastendruck
  {//PORTB=(entprelltetasten<<4); //Status der Tasten auf den LEDs anzeigen
   PORTB=((tastegedrueckt&7)<<4); //Status der Tasten auf den LEDs anzeigen
   if((tastegedrueckt&7)==7) {milliwait(200); tastegedrueckt=0;}//test
  }
 **/
 langgedrueckt=tastegedrueckt=0;

 LEDBITS=0;//alle LEDs aus
#ifdef GRAFIKTEST
 glcd_init();
 //glcd_test(0);//Punkte loeschen
 glcd_test(1);//Punkte setzen
 //glcd_test(2);//Punkte umkehren
 secwait(3);
#else
 glcd_init();
 milliwait(500);//test
#endif
 
 int8_t pos=0,modus=0,altsec=0;
#define ZUGSTEUERN  1//Werte fuer modus
#define FTASTEN     2
#define ZEITSTELLEN 3
#define MAXMODUS    4
 zugeinstellen(0);//test
 glcd_clear(0);
 gleisplan_zeichnen();
 weichestellen(99); //alle Weichen auf gerade
 while(1) //Hauptschlaufe
  {char i,c;
   if(altsec!=sec) {zeitauflcdanzeigen(); altsec=sec;}
   for(i=0;i<4;i++) {milliwait(1); tastaturcheck();}
   if((c=tastaturabfrage())!=0)
    {
     if(modus==0)
      {glcd_goto(7,pos); glcd_write(c);//Tastatureingabe auf 7.Zeile darstellen
       if(++pos==21) pos=0;
      }
     else if(modus==ZEITSTELLEN)
      {if(zeitstellen==0) modus=ZUGSTEUERN;//TODO: Zeitstellen beenden
      }
     switch(c) //Reaktionen auf Tastatureingabe
      {case 'A': case 'B': case 'C': case 'D':
	 glcd_goto(7,pos=0); for(i=0;i<21;i++) glcd_write(' ');//Testzeile loeschen
	 zugeinstellen(c);
	 modus=ZUGSTEUERN; zeitstellen=0;
	 break;
       case '.':
	if(++modus==MAXMODUS) modus=1;
	if(modus==ZEITSTELLEN) zeitstellen=8; else zeitstellen=0;
	int8 z=7;
	switch(modus)
	 {case ZUGSTEUERN:  glcd_printzeile(z,"Zug steuern "); break;
	  case ZEITSTELLEN: glcd_printzeile(z,"Zeit stellen"); break;
	  case FTASTEN:     glcd_printzeile(z,"F-Tasten (F1 bis F4)"); break;
	 }
      }
     if(modus==ZUGSTEUERN)
      {static int8 nr2[4]={2,2,4,4}, i2=0;
       if(c>='0' && c<='9') zuglangsamer(c-'0');
       else switch(c)
	    {case '+': zugschneller();
	     CASE '-': zuglangsamer();
	     CASE '*': zuglichtschalten(1);
	     CASE ':': zuglichtschalten(nr2[i2]);
	               if(++i2==4) i2=0; //zwischen 2 und 4 abwechseln
	     CASE '=': zugeinstellen(0); pos=0;
	    }
      }
     else if(modus==FTASTEN)
      {
       if(c>='0' && c<='4') ftaste_aktuellerzug(c-'0');//TODO
      }
    }
   if(warten==WARTE_AUF_TASTENDRUCK)
    {if(tastegedrueckt&ALLETASTEN)
      {if(tastegedrueckt&SCHWARZETASTE) schwarzetaste();
       else if(tastegedrueckt&ROTETASTE) rotetaste(); 
       else if(tastegedrueckt&GRUENETASTE) gruenetaste();
       warten=WARTE_AUF_TASTELOSLASSEN;
      }
    }
   else //warten==WARTE_AUF_TASTELOSLASSEN
    {if((entprelltetasten&ALLETASTEN)==ALLETASTEN)
      {warten=WARTE_AUF_TASTENDRUCK; langgedrueckt=tastegedrueckt=0;}
    }
   if(zeitzumsenden>=8) //etwa alle 30ms aktuelle Daten senden
    {
     zugdatensenden(); //Daten fuer alle Zuege senden
    }
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

/*** Tastatur mit Tiny2313 ***/
#ifdef UARTVARIANTE

#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 16000000"
#define F_CPU 16000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 8192UL      // Baudrate
 
// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart_init()
{
 UBRRH = UBRR_VAL >> 8;
 UBRRL = UBRR_VAL & 0xFF;
 //UCSRB |= (1<<TXEN); // UART TX einschalten
 UCSRB |= (1<<RXEN); // UART RX einschalten
 UCSRC = (1<<URSEL)|(3<<UCSZ0); // Asynchron 8N1 Atmega32
}

/*
void uart_putc(unsigned char c)
{
 while (!(UCSRA & (1<<UDRE))) ;// warten bis Senden moeglich
 UDR = c;                      // sende Zeichen
}
char uart_getc(void)
{
 while (!(UCSRA & (1<<RXC))) ;// warten bis Zeichen verfuegbar
 return UDR; // Zeichen aus UDR an Aufrufer zurueckgeben
}
*/

static int8_t uart_puffer[8],uart_i1=0,uart_i2=0;

void tastaturcheck()
{
 if(!(UCSRA & (1<<RXC))) return;// warten bis Zeichen verfuegbar
 if(((uart_i2+1)&0x07)==uart_i1) return;//Puffer voll, noch nichts tun
 uart_puffer[uart_i2++]=UDR;
 uart_i2 &= 0x07;
}
char tastaturabfrage()
{
 char c=0;
 if(uart_i1!=uart_i2)
  {c=uart_puffer[uart_i1++]; uart_i1&=0x07;}
 return c;
}

#else
/*
Datenuebertragung auf 1 Leitung:
-------------------------------

                | 1.Bit |  2. Bit  |   ...  9. Bit |    Rueckmeldung

-----+ Startbit +---+   +---+      +---            +----+   +--
     |  3 ms    |1ms|1ms|1ms| 2ms  |   ...         |    |1ms|
     |          |   |   |   |      |               |    |   |
     +----------+   +---+   +------+             --+    +---+

Das Startbit definiert die Laenge eines zu uebertragenden Bits und ist
3 Einheiten lang (z.B. 3ms). Eine 0 wird durch einen 2 Einheiten langen
Puls uebertragen, eine 1 durch einen 1 Einheit langen Puls.
(Eine 1 also ein kurzer negativer Puls, eine 0 ein langer Puls)
Wenn das erste Datenbit eine 1 ist wird ein Kommando fuer das LCD, sonst
ein darzustellendes Zeichen uebertragen. Die folgenden 8 Bit ergeben
ein Byte beginnend mit dem hoechstwertigen Bit.
Fuer eine schnelle Uebertragung muss am Ende die Datenrichtung umgekehrt
werden, und der LCD-Kontroller gibt einen 1 Einheiten langen Puls zurueck.
(Fuer langsame Uebertragung kann man darauf verzichten und einfach eine
genuegend lange Warteschlaufe einbauen.)
*/
static volatile char taste_ready=0,taste_c=0;
void tastaturcheck()
{
 char d0;
 d0=PIND&1;
 if(d0==1) return;
 char i,j,ok=0;
 int startbit,kurzesbit,langesbit,sehrlangesbit,tmitte,t1,daten;
 for(j=0;ok==0 && j<3;j++)
  {for(startbit=0;(PIND&1)==0;startbit++)
     microwait(10);//Laenge des Startbits messen
   langesbit=startbit*2/3;
   kurzesbit=startbit/3;
   sehrlangesbit=langesbit+(kurzesbit+1)/2;
   tmitte=(langesbit+kurzesbit+1)/2;
   for(i=0,daten=0,ok=1;i<9;i++)
    {for(t1=0;(PIND&1)!=0;t1++)
      {microwait(10);//Low abwarten
       if(t1>sehrlangesbit) {ok=0; daten=0; break;} //Abbruch bei Fehler
      }
     if(ok==0) break;
     for(t1=0;(PIND&1)==0;t1++) microwait(10);//Bitlaenge messen
     daten <<= 1;
     if(t1<tmitte) daten++;
     else if(t1>sehrlangesbit) {ok=0; daten=0; break;} //Abbruch bei Fehler
    }
  }
 if((d0=daten&0xFF)!=0) {taste_ready=1; taste_c=d0;}
 return;
}
char tastaturabfrage()
{
 char c=0;
 if(taste_ready)
  {c=taste_c; taste_ready=0;}
 return c;
}
#endif

/************ Einstellungen fuer die Zuege ************/
const int8 VMAX=28;
static int8 aktuellerzug=0, vzug[4]={0,0,0,0}, zuglicht[4]={0,0,0,0};

#define C_IST_VBIT 1 //C wird fuer feinere Geschwindigkeitsabstufung benutzt (normal fuer Licht)
                     //28 Fahrstufen wenn gesetzt, sonst 14 Fahrstufen (ev. in CV29 setzbar)
#define DINVERS    2 //Richtungsflag D ist gesetzt wenn "Vorwaerts" (normal fuer "Rueckwarts")
static int8 zugmodus[4]={0,0,DINVERS,0};

void glcd_printzeile(int zeile,const char *cstr,int p1,int p2,int p3,int p4)
{
 const int8_t N=22; //Breite der Anzeige
 int8_t i;
 char text[N];
 glcd_goto(zeile,0); //Anfang 6.Zeile
 sprintf(text,cstr,p1,p2,p3,p4);
 for(i=0;i<N-1 && text[i]!=0;i++) ;
 for(;i<N-1;i++) text[i]=' ';
 text[i]=0;
 glcd_write(text);
}

void zuganzeigen()
{
 int zeile=7;
 if(aktuellerzug==0)
   glcd_printzeile(zeile,"v[4] = %d %d %d %d",vzug[0],vzug[1],vzug[2],vzug[3]);
 else glcd_printzeile(zeile, "Zug %d  v=%d Licht=%d",
		      aktuellerzug, vzug[aktuellerzug-1], zuglicht[aktuellerzug-1]);
}

void ftaste_aktuellerzug(int8 fx)
{
 int zeile=7;
 if(aktuellerzug!=0)
  {
   ftaste(aktuellerzug,fx);
   glcd_printzeile(zeile, "Zug %d  Taste F%d",aktuellerzug,fx);
  }
 else glcd_printzeile(zeile, "zuerst Zug einstellen");

}

/*
void zugfehler(int fehlernummer)
{
 char text[22];
 glcd_goto(7,0); //Anfang 7.Zeile
 sprintf(text,"Fehler %d ",fehlernummer);
}
*/
void zugeinstellen(int8 zugnr)
{
 if(zugnr>='A') zugnr=zugnr-'A'+1;
 aktuellerzug=zugnr;
 zuganzeigen();
}
void zugschneller()
{
 int n=aktuellerzug-1;
 if(n<0) return;
 if(vzug[n]<VMAX) vzug[n]++;
 zuganzeigen();
#ifdef GLEICHSTROMTEST
 if(n==3) gleichspannung=vzug[n];
#endif
}
void zuglangsamer(int8 vneu)
{
 int n=aktuellerzug-1;
 if(n<0) return;
 if(vneu== -1)
   {if(vzug[n] > -VMAX) --vzug[n];
   }
 else if(vneu>=0 && vneu<=VMAX)
   {if(vzug[n]>=0) vzug[n]=vneu;
    else vzug[n] = -vneu;
   }
 zuganzeigen();
#ifdef GLEICHSTROMTEST
 if(n==3) gleichspannung=vzug[n];
#endif
}

void zuglichtschalten(int8 nr)
{
 if(aktuellerzug<1) return;
 int n=aktuellerzug-1;
 zuglicht[n] ^= nr;
 if(nr==2)
  {
   //Licht2 ein/aus-schalten
   if(zuglicht[n]&nr)
    {
     ftaste(aktuellerzug,1); //mit F1-Taste AUX1 einschalten
     //ftaste(aktuellerzug,2); //test: mit F2-Taste AUX2 einschalten
     //beide AUX ein: geht nicht
    }
   else ftaste(aktuellerzug,3); //mit F3-Taste AUX1 und AUX2 aus
  }
 else if(nr==4)
  {
   //drittes Licht ein/aus-schalten
   if(zuglicht[n]&nr) ftaste(aktuellerzug,2); //mit F2-Taste AUX2 einschalten
   else ftaste(aktuellerzug,3); //mit F3-Taste AUX1 und AUX2 aus
  }
 zuganzeigen();
}

void zugnotstop() //alle Zuege anhalten
{
 uint8_t i;
 for(i=0;i<4;i++) vzug[i]=0;
 zuganzeigen();
}

void zugdatensenden() //Daten fuer alle Zuege senden
{
 static int8_t i=0;
 int8_t v,absv,kom=0x40,D,C=0,SSSS,b,pruefbyte;
 // Befehlsaufbau: erstes Byte Loknr, zweites Byte KKDCSSSS, letztes Byte Pruefsumme
 // Erste 2 Bits (KK) ist das Kommando. Alle 3 Bytes exclusiv oder muessen 0 ergeben.
 //for(i=0;i<4;i++)
   {//14 1-Bits gefolgt von einem 0-Bit als Paketerkennung:
#ifdef DEBUG
    PORTC |= (1<<5); //Synchronisationspuls starten
#endif
    bytesenden(0x7F); //7 1-Bits senden
    bytesenden(0xFE); //nochmals 7 1-Bits, dann 0-Bit
#ifdef DEBUG
    PORTC &= ~(1<<5); //Synchronisationspuls fertig
#endif
    bytesenden(b=i+1); //Lok-Nummer
    pruefbyte=b;
    bitsenden(0);
    if((absv=vzug[i])<0) {absv= -absv; D=0x20;} //D=Richtung rueckwaerts
    else D=0; //D=Richtung vorwaerts
    if(zugmodus[i]&DINVERS) D ^= 0x20;
    v=absv+3;
    if(v<4) SSSS=0;
    else
     {SSSS = (v>>1);
      if(zugmodus[i]&C_IST_VBIT) C=((v&1)<<4);
      else C = (zuglicht[i]&1)!=0 ? 0x10 : 0;
     }
    bytesenden(b=kom+D+C+SSSS); //Kommando 0b01DCSSSS
    pruefbyte ^= b;
    bitsenden(0);
    bytesenden(pruefbyte); //Pruefsumme
    bitsenden(1);
    if(++i==4) i=0; //naechster Zug bei naechstem Durchgang
   }
 zeitzumsenden=0;
}

void weiche_dcc(int8_t weichenr,int8_t richtung) //Weichen ueber DCC stellen
{
 int8_t loknr=61;  //muss mit LOKNR im weichendecoder uebereinstimmen
 int8_t kom=0x40,C=0x10,SSSS=0x0F,b,pruefbyte; //,D=0x20;
 // Befehlsaufbau: erstes Byte Loknr, zweites Byte KKDCSSSS, letztes Byte Pruefsumme
 // Erste 2 Bits (KK) ist das Kommando. Alle 3 Bytes exclusiv oder muessen 0 ergeben.
 // Eigentlich wird mit den Bits DCSSSS Richtung, Licht und Geschwindigkeit gesetzt,
 // aber hier fuer Weichensteuerung wird diese Bedeutung verwendet:
 // SSSS entspricht den 4 Weichen (unterste 2 Bits, ev. in Erweiterung bis 16 Weichen)
 // C entspricht der Stellung auf die geschaltet werden soll
 // D auf 0 gesetzt fuer Weichenpulse
 // Mit D auf 1 gesetzt koennten noch andere Funktionen realisiert werden.

#ifdef DEBUG2
 PORTC |= (1<<5); //Synchronisationspuls starten
#endif
 //14 1-Bits gefolgt von einem 0-Bit als Paketerkennung:
 bytesenden(0x7F); //7 1-Bits senden
 bytesenden(0xFE); //nochmals 7 1-Bits, dann 0-Bit
#ifdef DEBUG2
 PORTC &= ~(1<<5); //Synchronisationspuls fertig
#endif
 bytesenden(b=loknr); //Lok-Nummer
 pruefbyte=b;
 bitsenden(0);
 if(richtung==0) C=0;
 SSSS=weichenr-1;
 bytesenden(b=kom+C+SSSS); //Kommando 0b010CSSSS
 pruefbyte ^= b;
 bitsenden(0);
 bytesenden(pruefbyte); //Pruefsumme
 bitsenden(1);
}

/*
Aufbau von CV29:
Bit 0  Fahrtrichtung (1=invertiert, 0=normal)
Bit 1  Fahrstufen    (2=28 Fahrstufen, 0=14 Fahrstufen)
Bit 2  Analogerkennung (4=ein)
Bit 3  RailCom  (8=ein)
Bit 4  Geschwindigkeitskennlinie (16=ein)
Bit 5  Erweiterte Adressen (32=ein)
Defaultwert in Funktionsdecoder: 14, also 8+4+2
*/

void ftaste(int8 loknr,int8 fx)
{
 //Kommando fuer F0-F4-Tasten
 int8_t kom=0x80,b,pruefbyte;
 if(fx<0 || fx>4) return;//bei Fehler nichts tun
 //wenn in CV29 Bit 1 geloescht hat fx==0 keine Funktion
 bytesenden(0x7F); //7 1-Bits senden
 bytesenden(0xFE); //nochmals 7 1-Bits, dann 0-Bit
 bytesenden(pruefbyte=loknr); //Lok-Nummer
 bitsenden(0);
 if(fx==0) b = kom + 0x10;
 else      b = kom + (1<<(fx-1));
 bytesenden(b); //Kommando fuer F-Taste
 pruefbyte ^= b;
 bitsenden(0);
 bytesenden(pruefbyte); //Pruefsumme
 bitsenden(1);
}

void cvdatensetzen(int8 loknr,int cvnr,uint8_t cvinhalt)
{
 //TODO
}

// Konstanten fuer CV setzen:
#define F0f  1
#define F0r  2
#define AUX1 4
#define AUX2 8

void aux2einschalten(int8 loknr)
{
 // cvnr aus Tabelle in "Anleitung Funktionsdecoder FD-R Basic.3"
 // gleiche Tabelle in "Anleitung Lokdecoder LD-G-42 und LD-W-42"
 /* Default-Werte:
 cvdatensetzen(loknr,257,F0f); //fuer F0-Taste vorwaerts
 cvdatensetzen(loknr,261,F0r); //fuer F0-Taste rueckwaerts
 cvdatensetzen(loknr,265,AUX1); //fuer F1-Taste vorwaerts
 cvdatensetzen(loknr,269,AUX1); //fuer F1-Taste rueckwaerts
 cvdatensetzen(loknr,273,AUX2); //fuer F2-Taste vorwaerts
 cvdatensetzen(loknr,277,AUX2); //fuer F2-Taste rueckwaerts
  */
 /* test um AUX2 immer zu schalten: * /
 cvdatensetzen(loknr,257,F0f+AUX2); //fuer F0-Taste vorwaerts
 cvdatensetzen(loknr,261,F0r+AUX2); //fuer F0-Taste rueckwaerts
 cvdatensetzen(loknr,265,AUX1+AUX2); //fuer F1-Taste vorwaerts
 cvdatensetzen(loknr,269,AUX1+AUX2); //fuer F1-Taste rueckwaerts
 cvdatensetzen(loknr,273,AUX2); //fuer F2-Taste vorwaerts
 cvdatensetzen(loknr,277,AUX2); //fuer F2-Taste rueckwaerts
 / * */
 ftaste(loknr,2); //F2-Taste ein/aus
}

/******************** Grafische Darstellung der Anlage ********************/
static int8_t weiche1=0,weiche2=0,weiche3=0; //Zustand der Weichen

void schienenkreis_zeichnen(int zeile,int spalte,int8_t nx,int8_t ny)
{
 int8_t i,z=zeile,sp=spalte+2;
 glcd_goto(z,sp); for(i=0;i<nx;i++) glcd_write(0x05); //gerade waagrechte Schiene oben
 glcd_goto(z+ny+3,sp); for(i=0;i<nx;i++) glcd_write(0x15); //gerade waagrechte Schiene unten
 glcd_goto(z,sp+nx); glcd_write(0x07); glcd_write(0x08); //Kurve rechts oben
 glcd_goto(z+1,sp+nx); glcd_write(0x09); glcd_write(0x0A); //Kurve rechts oben
 for(i=0;i<ny;i++)
  {
   glcd_goto(z+2+i,sp+nx+1); glcd_write(0x06); //gerade senkrechte Schiene rechts
   glcd_goto(z+2+i,spalte); glcd_write(0x16); //gerade senkrechte Schiene links
  }
 glcd_goto(z+ny+2,sp+nx); glcd_write(0x19); glcd_write(0x1A); //Kurve rechts unten
 glcd_goto(z+ny+3,sp+nx); glcd_write(0x17); glcd_write(0x18); //Kurve rechts unten

 sp=spalte;
 glcd_goto(z,sp); glcd_write(0x03); glcd_write(0x04); //Kurve links oben
 glcd_goto(z+1,sp); glcd_write(0x01); glcd_write(0x02); //Kurve links oben
 glcd_goto(z+ny+2,sp); glcd_write(0x11); glcd_write(0x12); //Kurve links unten
 glcd_goto(z+ny+3,sp); glcd_write(0x13); glcd_write(0x14); //Kurve links unten
}

void gleisplan_zeichnen()
{
 //aeusserer Kreis:
 schienenkreis_zeichnen(0,1,15,2);
 
 //innerer Kreis:
 schienenkreis_zeichnen(1,3,11,0);

 //Abstellgleis:
 int8_t ze=3,sp=6;
 glcd_goto(ze,sp); for(int i=0;i<6;i++) glcd_write(0x15); //gerade waagrechte Schiene unten
}

void weichestellen(int nr)
{
 const int8_t w1posx=30, w1posy=32;    //Werte von Links Oben nach Rechts Unten
 const int8_t w2posx=110-30, w2posy=32;// x durch 6 teilbar, y durch 8 teilbar (wegen Font 6x8)
 const int8_t w3posx=66, w3posy=24;// x durch 6 teilbar, y durch 8 teilbar (wegen Font 6x8)
 if(nr==1)      weiche1 ^= 1;
 else if(nr==2) weiche2 ^= 1;
 else if(nr==3) weiche3 ^= 1;
 else weiche1=weiche2=weiche3=0;
 
 glcd_clear(0);
 gleisplan_zeichnen();
 
 if(nr==1) weiche_dcc(1,weiche1);
 if(weiche1==0)
  {
   for(uint8_t i=8,j=8*2; i<=13; i++,j++) //Linie von oberer nach unterer Schiene
     glcd_punkt(w1posx+i, 64-(w1posy+j/2), 1);
  }
 else
  {
   glcd_goto(w1posy/8, w1posx/6); glcd_write(0x0B); glcd_write(0x0E);
   glcd_goto(w1posy/8+1,w1posx/6+1); glcd_write(0x1E); glcd_write(0x1D);
  }
 
 if(nr==2) weiche_dcc(2,weiche2);
 if(weiche2==0)
  {
   for(uint8_t i=1,j=10*2+1; i<=6; i++,j--) //Linie von unterer nach oberer Schiene
     glcd_punkt(w2posx+i, 64-(w2posy+j/2), 1);
  }
 else
  {
   glcd_goto(w2posy/8, w2posx/6+1); glcd_write(0x0C); glcd_write(0x0D);
   glcd_goto(w2posy/8+1, w2posx/6); glcd_write(0x1B); glcd_write(0x1C);
  }

 //Weiche zum Abstellgleis:
 if(nr==3) weiche_dcc(3,weiche3);
 if(weiche3==0)
  {
   for(uint8_t i=8,j=8*2; i<=13; i++,j++) //Linie von oberer nach unterer Schiene
     glcd_punkt(w3posx+i, 64-(w3posy+j/2), 1);
  }
 else
  {
   glcd_goto(w3posy/8, w3posx/6); glcd_write(0x0B); glcd_write(0x0E);
   glcd_goto(w3posy/8+1,w3posx/6+1); glcd_write(0x1E); glcd_write(0x1D);
  }
}
