/* decoder.cc  DCC-Signal decodieren   letzte Aenderung: 27.12.2010
*/
#define VERSION "0.0"
/*
 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 24.12.2010      Erstellung

*/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdio.h>
#define F_CPU 3686400
#include "ulong.h"

#define DCCPORT PORTC //Port fuer zu decodierendes Signal
#define DCCINP  PINC  //Eingangsport fuer zu decodierendes Signal
#define DCCPIN  5     //PC5 als Eingang fuer zu decodierendes Signal

/*********************** Zeitbehandlung ******************************/
static volatile uint millisec=0,tage=0;
static volatile uchar sec=0,min=0,stunden=0;

//#define F_CPU 8000000
const ulong Hz=F_CPU,MilliHz=0; //exakt 8MHz

const uint Rest1 = Hz%1000;//jede Sekunde zu korrigierender Fehler
const uint NormalerCompwert=Hz/1000-1;

//die Namen *_vect sind in /usr/avr/include/avr/iom8.h zu finden
ISR(TIMER1_COMPA_vect)
{
 uint ms=millisec;
 if(++ms>=1000-Rest1 && ms<1000) OCR1A=NormalerCompwert+1;
 else OCR1A=NormalerCompwert;
 if(ms==1000)
  {ms=0;
   if(sec==60)
    {sec=0;
     if(min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0; ++tage;}
      }
    }
  }
 millisec=ms;
}

//Bei einem unerwarteten Interrupt nichts machen:
void __attribute__ ((naked)) __vector_default(void) {__asm__ ("reti");}

void milliwait(int n) //n Millisekunden warten  1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) ;//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
   ;//auf n. Millisekunde warten
}

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
}

/*********************** LCD-Ansteuerung *****************************/
#include "lcd-routines.h"
void lcd_printf(const char* cstr,int8 n)
{
 char text[40];
 sprintf(text,cstr,n);
 lcd_string(text);
}
void lcd_goprintf(int8 spalte,int8 zeile,const char* cstr,int8 n)
{
 lcd_setcursor(spalte,zeile);  //spalte = {0,... 15}, zeile = {1, 2}
 lcd_printf(cstr,n);
}

/************************ Hauptprogramm ******************************/

//Konstanten mit F_CPU==3686400 mit bitsausmessen() ermittelt:
#define LZEIT 64
#define HZEIT 22
#define MZEIT ((LZEIT+HZEIT)/2)

static int8 paketdata[8], paketlaenge=0,startflanke=0;
static int8 zeit1=0,zeit2=0;
static int8 testdata[8];//test

int8 bitlesen()
{
 int8 zaehler1=0,zaehler2=0;
 if(startflanke==0)
  {while((DCCINP & (1<<DCCPIN))!=0) zaehler1++;//Zeit im H-Zustand messen
   while((DCCINP & (1<<DCCPIN))==0) zaehler2++;//Zeit im L-Zustand messen
  }
 else
  {while((DCCINP & (1<<DCCPIN))==0) zaehler1++;//Zeit im L-Zustand messen
   while((DCCINP & (1<<DCCPIN))!=0) zaehler2++;//Zeit im H-Zustand messen
  }
 zeit1=zaehler1; zeit2=zaehler2;//test
 if(zaehler1<=MZEIT && zaehler2<=MZEIT) return 1; //kurzer Puls entspricht 1-Bit
 return 0; //langer Puls entspricht 0-Bit
}

/*
static int8 mini1,maxi1,mini2,maxi2;//test
void bitsausmessen() //test
{
 int i;
 mini1=mini2=127; maxi1=maxi2=0;
 while((DCCINP & (1<<DCCPIN))!=0) ;//warten auf L-Zustand
 while((DCCINP & (1<<DCCPIN))==0) ;//warten auf L/H-Flanke
 for(i=0;i<1000;i++)
  {bitlesen();
   if(zeit1<mini1) mini1=zeit1;
   if(zeit2<mini2) mini2=zeit2;
   if(zeit1>maxi1) maxi1=zeit1;
   if(zeit2>maxi2) maxi2=zeit2;
  }
}
*/

int8 paket_lesen1() //mit ausgeschaltetem Interrupt aufzurufen,
{                   //Rueckgabe: Paketlaenge oder negative Werte als Fehlernummer
 int8 i,j,n,neuesbyte;
 for(i=0;i<8;i++) paketdata[i]=12;//test
 for(i=0;i<8;i++) testdata[i]=i;//test

 startflanke=0;
 while((DCCINP & (1<<DCCPIN))!=0) ;//warten auf L-Zustand
 while((DCCINP & (1<<DCCPIN))==0) ;//warten auf L/H-Flanke

 do {n=bitlesen();}
 while(n==0); //auf erstes 1-Bit warten

 testdata[0]=zeit1; testdata[1]=zeit2; //test

 for(i=1;i<12;i++) //mindestens 12 1-Bits fuer Paketstart erkennen
  {n=bitlesen();
   if(n==0)
    {testdata[2]=zeit1; testdata[3]=zeit2; //test
     return -i; //keine 12 1-Bits erkannt
    }
  }

 do {n=bitlesen();} //auf 0-Bit warten
 while(n==1);
 if(zeit2<MZEIT) return -16; //Fehler im Abschlussbit
 if(zeit1>MZEIT)
    startflanke=0;//Bit startet mit L/H-Flanke
 else
   {startflanke=1;//Bit startet mit H/L-Flanke
    zeit1=zeit2;
    zeit2=0;
    while((DCCINP & (1<<DCCPIN))!=0) zeit2++;//warten auf H/L-Flanke
    if(zeit2<MZEIT) return -17; //Fehler im Abschlussbit
   }
 //0-Bit erfolgreich erkannt
 testdata[2]=zeit1; testdata[3]=zeit2;//test

 PORTC |= (1<<2); //gruene LED an

 for(j=0;j<8;) //Bytes einlesen bis mit einem 1-Bit abgeschlossen wird, oder maximal 8
  {neuesbyte=0;
   for(i=0;i<8;i++) //8 Bits einlesen
    {neuesbyte = (neuesbyte<<1) + bitlesen();
    }
   paketdata[j++] = neuesbyte;
   //Abschluss-Bit einlesen:
   n=bitlesen();
   PORTC &= ~(1<<2); //gruene LED aus
   if(n==1) break; //Bei 1-Bit ist das Paket fertig empfangen
  }
 // testdata[2]=zaehler1; testdata[3]=zaehler2;
 paketlaenge=j;
 return j;
}

int8 paket_lesen()
{
 int8 antwort;
 cli(); //Interrupt aus
 antwort=paket_lesen1();
 sei(); //Interrupt ein
 return antwort;
}

int main(void)
{
 int8 j;
 DDRC = 0x07; //3 LEDs zum testen
 DCCPORT |= (1<<DCCPIN); //Pullup-Widerstand setzen

 lcd_init();
 timer1_init();

 sei(); //Interrupt einschalten
 
 lcd_goprintf(0,1,"Test %d",17);//test
 lcd_setcursor(0,2);  // Die Ausgabemarke in die 2te Zeile setzen
 lcd_string("Decoder ");
 lcd_string(VERSION);

 /*
 while(1) //test
  {
   bitsausmessen();
   lcd_goprintf(0,1,"%d ",mini1);
   lcd_printf("%d ",maxi1);
   lcd_printf("%d ",mini2);
   lcd_printf("%d  ",maxi2);
   milliwait(1000);
  }
 */
 int8 zugnr=1;//test

 while(1)
  {
   j=paket_lesen();
   if(j>=0)
    {
     PORTC = (PORTC&0xFC) + (j&3); //test auf 2 LEDs: Paketlaenge
     lcd_goprintf(0,1,"%d ",j);
     lcd_printf("pak=%d",paketdata[0]);
     for(j=1;j<paketlaenge;j++) lcd_printf(" %d",paketdata[j]);
     if(paketlaenge==2) lcd_printf(" 0x%02X",paketdata[1]);
     lcd_data('.');
     lcd_goprintf(0,2,"T=%d",testdata[0]);
     for(j=1;j<4;j++) lcd_printf(" %d",testdata[j]);
     lcd_data('.');
     if(paketdata[0]==zugnr)
      {milliwait(1000);
       if(++zugnr==5) zugnr=1;
      }
    }
   else
    {PORTC = (PORTC&0xFC); //test auf LEDs: geloescht = kein Paket erkannt
     lcd_goprintf(0,1,"%d Test=",j);
     for(j=0;j<4;j++) lcd_printf(" %d",testdata[j]);//test
     lcd_data('.');//test
    }
  }
 
 return 0;
}
