/* wdecoder.cc       letzte Aenderung: 20.9.2023
DCC-Weichendecoder

 Version: 0.1
 Prozessor: ATmega8A
 Schaltung: schaltung/weichemega8smd (Quarz weggelassen)
            Port PB5 mit LED verbunden
	    Port PC0-PC5 fuer Weichen 1 bis 3
	    Port PD3-PD4 fuer Weiche 4
 Autor: Rolf Pfister
 Copyright: Freeware
 History:
 12.9.2023   0.0  Erstellung
 20.9.2023   0.1  auch noch Leitungen PB0-PB2, PD0,PD1,PD5-PD6 setzbar
*/

//#define TEST1  //erster Test: blinkende LED

#define LOKNR 61  //Loknummer fuer dieses Geraet

#define F_CPU 1000000  //interner Oszillator 1MHz
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
#include <ctype.h>
#include <util/delay.h>

#define LED_PORT PORTB
#define LED_DDR  DDRB
#define LED1     PB5
#define WEICHE1_PORT PORTC
#define WEICHE2_PORT PORTC
#define WEICHE3_PORT PORTC
#define WEICHE1BIS3_DDR  DDRC
#define WEICHE4_PORT PORTD
#define WEICHE4_DDR  DDRD
#define A1 PC0
#define B1 PC1
#define A2 PC2
#define B2 PC3
#define A3 PC4
#define B3 PC5
#define A4 PD3
#define B4 PD4

#define DIO0 PB0
#define DIO1 PB1
#define DIO2 PB2
#define DIO3 PD0
#define DIO4 PD1
#define DIO5 PD5
#define DIO6 PD6
#define DIO7 PD7

/*
void dio_richtung_all(uint8_t k)
{
 if(k&(1<<0)) DDRB |= (1<<0); else DDRB &= ~(1<<0);
 if(k&(1<<1)) DDRB |= (1<<1); else DDRB &= ~(1<<1);
 if(k&(1<<2)) DDRB |= (1<<2); else DDRB &= ~(1<<2);
 if(k&(1<<3)) DDRD |= (1<<0); else DDRD &= ~(1<<0);
 if(k&(1<<4)) DDRD |= (1<<1); else DDRD &= ~(1<<1);
 if(k&(1<<5)) DDRD |= (1<<5); else DDRD &= ~(1<<5);
 if(k&(1<<6)) DDRD |= (1<<6); else DDRD &= ~(1<<6);
 if(k&(1<<7)) DDRD |= (1<<7); else DDRD &= ~(1<<7);
}

void dio_set_all(uint8_t k)
{
 if(k&(1<<0)) PORTB |= (1<<0); else PORTB &= ~(1<<0);
 if(k&(1<<1)) PORTB |= (1<<1); else PORTB &= ~(1<<1);
 if(k&(1<<2)) PORTB |= (1<<2); else PORTB &= ~(1<<2);
 if(k&(1<<3)) PORTD |= (1<<0); else PORTD &= ~(1<<0);
 if(k&(1<<4)) PORTD |= (1<<1); else PORTD &= ~(1<<1);
 if(k&(1<<5)) PORTD |= (1<<5); else PORTD &= ~(1<<5);
 if(k&(1<<6)) PORTD |= (1<<6); else PORTD &= ~(1<<6);
 if(k&(1<<7)) PORTD |= (1<<7); else PORTD &= ~(1<<7);
}
*/

void dio_richtung(uint8_t nr,uint8_t x)
{
 if(nr==0)      {if(x!=0) DDRB |= (1<<0); else DDRB &= ~(1<<0);}
 else if(nr==1) {if(x!=0) DDRB |= (1<<1); else DDRB &= ~(1<<1);}
 else if(nr==2) {if(x!=0) DDRB |= (1<<2); else DDRB &= ~(1<<2);}
 else if(nr==3) {if(x!=0) DDRD |= (1<<0); else DDRD &= ~(1<<0);}
 else if(nr==4) {if(x!=0) DDRD |= (1<<1); else DDRD &= ~(1<<1);}
 else if(nr==5) {if(x!=0) DDRD |= (1<<5); else DDRD &= ~(1<<5);}
 else if(nr==6) {if(x!=0) DDRD |= (1<<6); else DDRD &= ~(1<<6);}
 else if(nr==7) {if(x!=0) DDRD |= (1<<7); else DDRD &= ~(1<<7);}
}

#define INPUT    1
#define OUTPUT   2
#define TRISTATE 3

void dio_set(int8_t richtung,int8_t nr,int8_t x=0)
{
 if(richtung==OUTPUT) dio_richtung(nr,1);
 else if(richtung==INPUT)    {dio_richtung(nr,0); x=1;}
 else if(richtung==TRISTATE) {dio_richtung(nr,0); x=0;}
 if(nr==0)      {if(x!=0) PORTB |= (1<<0); else PORTB &= ~(1<<0);}
 else if(nr==1) {if(x!=0) PORTB |= (1<<1); else PORTB &= ~(1<<1);}
 else if(nr==2) {if(x!=0) PORTB |= (1<<2); else PORTB &= ~(1<<2);}
 else if(nr==3) {if(x!=0) PORTD |= (1<<0); else PORTD &= ~(1<<0);}
 else if(nr==4) {if(x!=0) PORTD |= (1<<1); else PORTD &= ~(1<<1);}
 else if(nr==5) {if(x!=0) PORTD |= (1<<5); else PORTD &= ~(1<<5);}
 else if(nr==6) {if(x!=0) PORTD |= (1<<6); else PORTD &= ~(1<<6);}
 else if(nr==7) {if(x!=0) PORTD |= (1<<7); else PORTD &= ~(1<<7);}
}

/************************ Zeitbehandlung ******************************/
static volatile uint16_t millisec=0,tage=0;
static volatile uint8_t sec=0,min=0,stunden=0;

#define Rest1 (F_CPU%1000)  //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (F_CPU/1000-1)

ISR(TIMER1_COMPA_vect)
{
 uint16_t ms=millisec+1;
#if(Rest1!=0)
 if(ms>1000-Rest1) OCR1A=(NormalerCompwert+1); else OCR1A=NormalerCompwert;
#endif
 if(ms==1000)
  {ms=0;
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0; ++tage;}
      }
    }
  }
 millisec=ms;
}

void milliwait(int n) //n Millisekunden warten  1000 >= n >= 1
{
 uint16_t ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) {}//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
  {sleep_mode();}//auf n. Millisekunde warten
  //{__asm__("sleep");} //wenn sicher dass sleep nicht versehentlich aufgerufen
}

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
#if defined (__AVR_ATmega8__)
 TIMSK = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
#else
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
#endif
 set_sleep_mode(SLEEP_MODE_IDLE);
 sleep_enable(); //Schlaf-Modus waehlen, aber noch nicht schlafen
}

/************************ DCC-Auswertung ******************************/
#define DCCPORT PORTD //Port fuer zu decodierendes Signal
#define DCCINP  PIND  //Eingangsport fuer zu decodierendes Signal
#define DCCBIT  2     //PD2 als Eingang fuer zu decodierendes Signal

//#define BITSAUSMESSEN
/*Konstanten mit F_CPU==16000000 mit bitsausmessen() ermittelt:
#define LZEIT 297
#define HZEIT 126
*/

//Konstanten mit F_CPU==1000000 berechnet:
#define LZEIT 19
#define HZEIT 9

#define MZEIT ((LZEIT+HZEIT)/2)

#define UINT uint8_t  //sollte so bis etwa 4MHz ok sein
//#define UINT uint16_t  //so bei 16MHz, da LZEIT sonst ueberlaeuft
static uint8_t paketdata[8], paketlaenge=0,startflanke=0;
static volatile UINT zeit1=0,zeit2=0;
//static UINT testdata[8];//test

uint8_t bitlesen()
{
 UINT zaehler1=0,zaehler2=0;
 if(startflanke==0)
  {while((DCCINP & (1<<DCCBIT))!=0) zaehler1++;//Zeit im H-Zustand messen
   _delay_us(2); //5 bei 16MHz
   while((DCCINP & (1<<DCCBIT))==0) zaehler2++;//Zeit im L-Zustand messen
  }
 else
  {while((DCCINP & (1<<DCCBIT))==0) zaehler1++;//Zeit im L-Zustand messen
   _delay_us(2); //5 bei 16MHz
   while((DCCINP & (1<<DCCBIT))!=0) zaehler2++;//Zeit im H-Zustand messen
  }
 //_delay_us(4); //nur bei 16MHz noetig
 zeit1=zaehler1; zeit2=zaehler2;
 if(zaehler1<=MZEIT && zaehler2<=MZEIT) return 1; //kurzer Puls entspricht 1-Bit
 return 0; //langer Puls entspricht 0-Bit
}

#ifdef BITSAUSMESSEN
static UINT mini1=(-1),maxi1=0,mini2=(-1),maxi2=0;
void bitsausmessen()
{
 int i;
 mini1=mini2=(-1); maxi1=maxi2=0;
 while((DCCINP & (1<<DCCBIT))!=0) {}//warten auf L-Zustand
 _delay_us(5);
 while((DCCINP & (1<<DCCBIT))==0) {}//warten auf L/H-Flanke
 _delay_us(5);
 //for(i=0;i<1000;i++)
 for(i=0;i<8;i++)
  {bitlesen();
   if(zeit1<mini1) mini1=zeit1;
   if(zeit2<mini2) mini2=zeit2;
   if(zeit1>maxi1) maxi1=zeit1;
   if(zeit2>maxi2) maxi2=zeit2;
  }
}
#endif

int8_t paket_lesen1() //mit ausgeschaltetem Interrupt aufzurufen,
{                   //Rueckgabe: Paketlaenge oder negative Werte als Fehlernummer
 int8_t j;
 uint8_t i,n,neuesbyte;
 //for(i=0;i<8;i++) paketdata[i]=12;//test
 //for(i=0;i<8;i++) testdata[i]=i;//test

 if(startflanke==0)
  {
   while((DCCINP & (1<<DCCBIT))!=0) {}//warten auf L-Zustand
   _delay_us(2);
   while((DCCINP & (1<<DCCBIT))==0) {}//warten auf L/H-Flanke
   _delay_us(2);
  }
 else
  {
   while((DCCINP & (1<<DCCBIT))==0) {}//warten auf H-Zustand
   _delay_us(2);
   while((DCCINP & (1<<DCCBIT))!=0) {}//warten auf H/L-Flanke
   _delay_us(2);
  }

 do {n=bitlesen();}
 while(n==0); //auf erstes 1-Bit warten

 //testdata[0]=zeit1; testdata[1]=zeit2; //test

 for(i=1;i<12;i++) //mindestens 12 1-Bits fuer Paketstart erkennen
  {n=bitlesen();
   if(n==0)
    {
     //testdata[2]=zeit1; testdata[3]=zeit2; //test
     return -i; //keine 12 1-Bits erkannt
    }
  }

 do {n=bitlesen();} //auf 0-Bit warten
 while(n==1);
 //wenn 0-Bit korrekt gelesen dann muss sowohl zeit1 als auch zeit2 groesser MZEIT sein
 if(zeit1>MZEIT && zeit2>MZEIT)
    {}//eingestgellte startflanke ok
 else if(zeit1<=MZEIT && zeit2<=MZEIT)
    return -18;//da stimmt was gar nicht
 else if(zeit2<=MZEIT)
    return -16; //Fehler im Abschlussbit
 else //(zeit1<=MZEIT)
   {
    startflanke ^= 1;//andere startflanke setzen
    //testdata[2]=zeit1; testdata[3]=zeit2;//test
    if(zeit2<=MZEIT) return -17; //Fehler im Abschlussbit
   }
 //0-Bit erfolgreich erkannt
 //testdata[2]=zeit1; testdata[3]=zeit2;//test

 for(j=0;j<8;) //Bytes einlesen bis mit einem 1-Bit abgeschlossen wird, oder maximal 8
  {neuesbyte=0;
   for(i=0;i<8;i++) //8 Bits einlesen
    {neuesbyte = (neuesbyte<<1) + bitlesen();
    }
   if(j==0 && neuesbyte==LOKNR) LED_PORT |= (1<<LED1); //gruene LED ein
   paketdata[j++] = neuesbyte;
   //Abschluss-Bit einlesen:
   n=bitlesen();
   LED_PORT &= ~(1<<LED1); //gruene LED aus
   if(n==1) break; //Bei 1-Bit ist das Paket fertig empfangen
  }
 paketlaenge=j;
 return j;
}

int8_t paket_lesen()
{
 int8_t antwort;
 cli(); //Interrupt aus
 antwort=paket_lesen1();
 sei(); //Interrupt ein
 return antwort;
}

void setzen(int8_t stellung,int8_t weichenr)
{
 if(weichenr==1)
  {
   if(stellung==0) WEICHE1_PORT |= (1<<A1);
   else            WEICHE1_PORT |= (1<<B1);
  }
 else if(weichenr==2)
  {
   if(stellung==0) WEICHE2_PORT |= (1<<A2);
   else            WEICHE2_PORT |= (1<<B2);
  }
 else if(weichenr==3)
  {
   if(stellung==0) WEICHE3_PORT |= (1<<A3);
   else            WEICHE3_PORT |= (1<<B3);
  }
 else //if(weichenr==4)
  {
   if(stellung==0) WEICHE4_PORT |= (1<<A4);
   else            WEICHE4_PORT |= (1<<B4);
  }
}

void ruecksetzen(int8_t stellung,int8_t weichenr)
{
 if(weichenr==1)
  {
   if(stellung==0) WEICHE1_PORT &= ~(1<<A1);
   else            WEICHE1_PORT &= ~(1<<B1);
  }
 else if(weichenr==2)
  {
   if(stellung==0) WEICHE2_PORT &= ~(1<<A2);
   else            WEICHE2_PORT &= ~(1<<B2);
  }
 else if(weichenr==3)
  {
   if(stellung==0) WEICHE3_PORT &= ~(1<<A3);
   else            WEICHE3_PORT &= ~(1<<B3);
  }
 else //if(weichenr==4)
  {
   if(stellung==0) WEICHE4_PORT &= ~(1<<A4);
   else            WEICHE4_PORT &= ~(1<<B4);
  }
}

void paket_auswerten()
{
 int8_t j,kom=0x40,D=0x20,C=0x10,SSSS=0x0F,b,pruefbyte;
 // Befehlsaufbau: erstes Byte Loknr, zweites Byte KKDCSSSS, letztes Byte Pruefsumme
 // Erste 2 Bits (KK) ist das Kommando. Alle 3 Bytes exclusiv oder muessen 0 ergeben.
 // Eigentlich wird mit den Bits DCSSSS Richtung, Licht und Geschwindigkeit gesetzt,
 // aber hier fuer Weichensteuerung wird diese Bedeutung verwendet:
 // SSSS entspricht den 4 Weichen (nur unterste 2 Bits)
 // C entspricht der Stellung auf die geschaltet werden soll
 // D auf 0 gesetzt fuer Weichenpulse
 // Mit D auf 1 gesetzt koennten noch andere Funktionen realisiert werden.
 pruefbyte = paketdata[0];
 for(j=1;j<paketlaenge;j++) pruefbyte ^= paketdata[j];
 if(pruefbyte!=0) return; //Pruefsummen-Fehler --> nichts tun
 b = paketdata[1];
 if((b&0xC0)!=kom) return; //anderes Kommando --> nichts tun
 if((b&D)==0)
  {
   int8_t weichenr = (b&SSSS)+1;
   int8_t stellung = (b&C)==0 ? 0 : 1;
   if(weichenr<=4)
    {
     LED_PORT |= (1<<LED1); //gruene LED ein
     setzen(stellung,weichenr);
     if(weichenr==2 && stellung!=0) milliwait(1000); else//test: Ausnahme fuer Weiche2
     milliwait(100); // 100msec langer Puls (TODO: ev andere Pulslaenge?)
     ruecksetzen(stellung,weichenr);
     LED_PORT &= ~(1<<LED1); //gruene LED aus
    }
   else //if(weichenr>4)
    {
     // andere Funktion: ein/aus schalten des entsprechenden Pins
     // KKDCSSSS: D=0, C=richtung, ein: SSSS=12+weichenr-1, aus: SSSS=8+weichenr-1
     if(weichenr>12)
      {weichenr -= 12;
       setzen(stellung,weichenr);
      }
     else if(weichenr>8)
      {weichenr -= 8;
       ruecksetzen(stellung,weichenr);
      }
    }
  }
 else
  {
   // andere Funktion: ein/aus schalten eines DIO-Pins
   // KKDCTSSS: D=1, C==1: ein, C==0: aus, T=Tristate, SSS=Nummer (0 bis 7)
   // T=0x08, SSS=0x07
   int8_t nr=b&0x07;
   if(b&0x08)
    {//Tristate oder Input
     if(b&C) dio_set(INPUT,nr,1);//input mit Pullup-Widerstand
     else    dio_set(TRISTATE,nr);
    }
   else if(b&C)
    {//einschalten
     dio_set(OUTPUT,nr,1);
    }
   else
    {//ausschalten
     dio_set(OUTPUT,nr,0);
    }
  }
}

/************************ Hauptprogramm *******************************/
int main(void)
{
 WEICHE1BIS3_DDR = (1<<A1)|(1<<B1) | (1<<A2)|(1<<B2) | (1<<A3)|(1<<B3);
 WEICHE4_DDR = (1<<A4)|(1<<B4);
 LED_DDR |= (1<<LED1);
 LED_PORT |= (1<<LED1); //gruene LED ein als Einschalttest
 timer1_init();
 sei(); //Interrupt einschalten
 milliwait(1000);
 LED_PORT &= ~(1<<LED1); //gruene LED aus
 
#ifdef TEST1
 while(1)
  {
   milliwait(500);
   LED_PORT |= (1<<LED1); //gruene LED ein
   milliwait(500);
   LED_PORT &= ~(1<<LED1); //gruene LED aus
  }
#endif
 
 while(1) //Hauptschlaufe
  {
   int8_t j=paket_lesen();
   if(j>0 && paketdata[0]==LOKNR) //nur Pakete fuer diese Loknummer auswerten
    {
     paket_auswerten();
    }
   else if(j<=0)
    {
     //TODO: Fehlerbehandlung
     for(j=0;j<3;j++) //LED 3 mal schnell blinken
      {
       LED_PORT |= (1<<LED1); //LED ein
       milliwait(200);
       LED_PORT &= ~(1<<LED1); //LED aus
       milliwait(200);
      }
    }
   else //Pakte fuer andere Geraete ignorieren
    {
     milliwait(2); //Pause zwischen Paketen?
    } 
  }

 return 0; //ende main(), wird nie erreicht
}
