/* edelgas.cc  Edelgas simulieren */
#define VERSION "Version 0.2"
/*
Erste Variante:
 N Atome in einer rechteckigen Box
 Umkehrung der Geschwindigkeit bei Kontakt mit Wand

30.6.2019   Version 0.1  mit xtekplot1
21.10.2019  Version 0.2  Anpassung an OpenGL

*/

#include <stdio.h>
#include <thread>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <vector_types.h>
//#include <xtekplot1.h>
#include "edelgas.h"
#include "myopencl.h"
#include "kernel.h" //BOXX...BOXZ, NATOMS, OPTIMIERUNG
#include "vektor3dklasse.cc"
#include "myfonts.h"
#include "menu.h"

//#define OPTIMIERUNG 0 // 0=ohne Optimierung, 3=staerkste Optimierung (in kernel.h)
//#define DEBUG
#define MAXBER 42000 //maximale Zeit fuer Berechnung in Microsekunden

static char kernel_name[200]="kernel.cc";

class Kernelname
{
public:
 int opt;
 char *name;
 Kernelname() {opt=OPTIMIERUNG; name=kernel_name; set(opt);}
 void set(int opt) {if(opt>0) sprintf(name,"kernel-opt%d.cc",opt);}
};
static Kernelname kname;

// Konstanten fuer konkrete Simulation:
#define NATOME NATOMS //in kernel.h definiert
//#define NATOME 512 //statt hier: in kernel.h  NATOMS setzen
//#define NATOME 4 //Test mit definierten Anfangsbedingungen
//#define NATOME 8 //3 davon mit definierten Anfangsbedingungen

// Tabelle fuer Optimierte Parallelisierung:
#define MAXPAAR ((NATOME-1)*NATOME/2)  //jedes Atom mit jedem vergleichen
static short2 paartabelle[MAXPAAR];  //Geordnete Tabelle, so dass immer NATOME/2
                                     //voneinander unabhaengig.

// Grafik-Konstanten und Variablen:
#define XMAX 1024 //Grafik-Breite
#define YMAX 820  //Grafik-Hoehe
#define TIEFE 24  //Farbtiefe der Grafik
#define SCHWARZAUFWEISS //auskommentieren fuer weiss auf schwarz
//static double xmax=3.122, ymax=2.5;
static double xmax=3.122*BOXX/1.55, ymax=2.5*BOXX/1.55;
static double xmin= -xmax, ymin= -ymax;
static double zmax=xmax;//TODO

int pixel(double x) //Umrechnung von Radius-User-Werten in Pixel-Durchmesser
{
 return int(2*XMAX/(xmax-xmin)*x+0.5);
}

// Variablen fuer konkrete Simulation:
static double atom_posx[NATOME];
static double atom_posy[NATOME];
static double atom_posz[NATOME];
static double atom_vx[NATOME]; //Geschwindigkeiten
static double atom_vy[NATOME];
static double atom_vz[NATOME];
static float atom_masse[NATOME];
static float atom_radius[NATOME];

// Serielle Variante des Kernels:
void kernel_neupos(double *posx,double *posy,double *posz,
		   double *vx,double *vy,double *vz,
		   float *mass,float *radi,int natoms,double dt)
{
 for(int i=0;i<natoms;i++)
  {
   posx[i] += vx[i]*dt;
   posy[i] += vy[i]*dt;
   posz[i] += vz[i]*dt;
   float radius=radi[i];
   if(posx[i] < radius-BOXX)      {if(vx[i]<0) vx[i] = -vx[i];}
   else if(posx[i] > BOXX-radius) {if(vx[i]>0) vx[i] = -vx[i];}
   if(posy[i] < radius-BOXY)      {if(vy[i]<0) vy[i] = -vy[i];}
   else if(posy[i] > BOXY-radius) {if(vy[i]>0) vy[i] = -vy[i];}
   if(posz[i] < radius-BOXZ)      {if(vz[i]<0) vz[i] = -vz[i];}
   else if(posz[i] > BOXZ-radius) {if(vz[i]>0) vz[i] = -vz[i];}
  }
}

double sq(double x) {return x*x;}

//schon in kernel.h definiert:
//const double KF=1.0; //Kraftkonstante, wie realistischer Wert ermitteln?
//const double KA=1e-9; //Beschleunigungskonstante: Umrechnung zwischen m/s^2 und nm/s^2

void kernel_kraefte(double *posx,double *posy,double *posz,
		    double *vx,double *vy,double *vz,
		    float *mass,float *radi,int natoms,double dt)
{
 for(int i=0;i<natoms;i++)
  {
   double fx=0,fy=0,fz=0;
   for(int j=0;j<natoms;j++)
    if(j!=i)
     {
      double ax,ay,az;
      double rx2=sq(ax=posx[j]-posx[i]); //Abstand x im Quadrat
      double ry2=sq(ay=posy[j]-posy[i]);
      double rz2=sq(az=posz[j]-posz[i]);
      double abstand = sqrt(rx2+ry2+rz2);
      if(abstand < (radi[i]+radi[j])*1.5)
       {
	double f=KF*KA/pow(abstand,6); //?wie gross muss KF sein?
	fx += f*ax/abstand; //Vorzeichen ist Glueckssache
	fy += f*ay/abstand;
	fz += f*az/abstand;
       }
     }
   double m=mass[i];
   //double a=fx/m; //Beschleunigung, KA noetig wegen Einheiten
   vx[i] -= fx/m*dt; //Vorzeichen ist Glueckssache: -= scheint richtig zu sein (ausprobiert)
   vy[i] -= fy/m*dt;
   vz[i] -= fz/m*dt;
  }
}

void kernel_cpu(double dt)
{
 kernel_neupos(atom_posx,atom_posy,atom_posz,
	       atom_vx,atom_vy,atom_vz,
	       atom_masse,atom_radius,NATOME,dt);
 kernel_kraefte(atom_posx,atom_posy,atom_posz,
		atom_vx,atom_vy,atom_vz,
		atom_masse,atom_radius,NATOME,dt);
}

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
static int neues_TPB=0;
static int neues_TPBx=0;
static int neues_TPBy=0;

void setargflags(char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
   if(c=='P') sscanf(s,"%d",&neues_TPB);
   if(c=='Q') sscanf(s,"%d:%d",&neues_TPBx,&neues_TPBy);
  }
}

/************************* Menu Behandlung ****************************/
static int exitflag=0;
static bool stopflag=false;
bool fastflag=false; //wenn gesetzt: fuer Neuberechnung nicht warten bis fertig gezeichnet

//void menu_exit() {exitflag=1;}
void menu_exit(long id)
{
 stopflag=true; //Berechnung stoppen
 menu.stop_loop();
 exitflag=1;
}

/*********************** Grafik mit OpenGL ****************************/
//#define BELEUCHTUNG  //mit Lichtquelle
#ifdef BELEUCHTUNG
static int ambientflag=0;
#endif
int bildbreite=XMAX,bildhoehe=YMAX; //Groesse des Fensters beim Start
int bildposx=30, bildposy=10;       //Position des Fensters beim Start
void display(void);
void reshape(int w, int h);
void zeichne_alles();

void reshape(int w, int h)
{
 glViewport(0, 0, w, h);
 bildbreite=w; bildhoehe=h;
 printf("reshape(w=%d,h=%d)\n",w,h);//test
 //DX=(xmax-xmin)/bildbreite; DY=(ymax-ymin)/bildhoehe;
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 //gluPerspective(GLdouble fovy, aspect, zNear, zFar);
 // zNear muss groesser 0 sein, Verhaeltnis zFar/zNear sollte nicht zu gross sein
 // roughly log2(zFar/zNear) bits of depth buffer precision are lost
 //gluPerspective(45.0, 1.0, 0.1, 200);
 gluPerspective(15.0, 1.0, 0.1, 200);
 gluLookAt(0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
 glMatrixMode(GL_MODELVIEW);
};

const char *hilfetext=
"    Hilfe\n\
    ----\n\n\
Simulation eines Edelgases\n\
OpenGL-Version von edelgas.cc";

void key(unsigned char c, int x, int y)
{
 if(requester1.aktiv)
   requester1.key(c,x,y);
 else if(menu.aktiv)
   menu.key(c,x,y);
 else
  {
   if(c=='?' || c=='H' || c=='h')
    {
     printf("\n%s\n",hilfetext);
    }
   else
    {
     if(c>=' ' && c<127) printf("key(c='%c',x=%d,y=%d)\n",c,x,y);//test
     else                printf("key(c=%d=0x%02X,x=%d,y=%d)\n",c,c,x,y);//test
    }
  }
 glutPostRedisplay();
};

void spezial(int key,int x,int y) //Tastatur spezialzeichen
{
 if(requester1.aktiv)
   requester1.spezial(key,x,y);
 else if(menu.aktiv)
   menu.spezial(key,x,y);
 else
  {
   printf("spezial(key=%d,x=%d,y=%d)\n",key,x,y);//test
  }
 glutPostRedisplay();
}

void mausknopf(int button,int state,int x,int y)
{
 static int x0=0,y0=0;
 if(requester1.aktiv)
  {requester1.mausknopf(button,state,x,y);}
 else if(menu.aktiv ||                //wenn aktives Menu
    (y<MBH && button==0 && state==0)) //oder linker Mausknopf im Menubalken gedrueckt
  {menu.mausknopf(button,state,x,y);}
 else
  {
   if(button==0 && state==GLUT_DOWN) {x0=x; y0=y;}
   if(button==0 && state==GLUT_UP && x==x0 && y==y0)
    {
     //wenn linker Mausknopf losgelassen an gleicher Stelle wie gedrueckt
     printf("Mausklick x=%d y=%d\n",x,y);//test
     //TODO: Mausklicks auswerten
    }
  }
 glutPostRedisplay();
}

void mausmotion(int x,int y)
{
 if(requester1.aktiv)
   requester1.mausmotion(x,y);
 else if(menu.aktiv)
   menu.mausmotion(x,y);
 else
  {
   //TODO: Mausbewegungen mit gedrueckter Maustaste auswerten
  }
 glutPostRedisplay();
}

void mausmotion2(int x,int y) //Mausbewegungen ohne gedrueckte Taste
{
 if(requester1.aktiv)
  {requester1.mausmotionpassiv(x,y); glutPostRedisplay();}
 else if(menu.aktiv)
  {menu.mausmotionpassiv(x,y); glutPostRedisplay();}
 else
  {
   //TODO: Mausbewegungen ohne gedrueckte Taste auswerten
  }
 //glutPostRedisplay();
}

void grafik_init(const char *titel)
{
 int screenWidth  = glutGet(GLUT_SCREEN_WIDTH);
 int screenHeight = glutGet(GLUT_SCREEN_HEIGHT);
 //printf("screenWidht/Height = %d / %d\n",screenWidth,screenHeight);//test
 if(screenWidth>2*bildbreite && screenHeight>2*bildhoehe)
  {bildbreite *= 2; bildhoehe *= 2; bildposx *= 2; bildposy *= 2;}
 if(screenWidth>2500) {menu.grosserfont(2); requester1.grosserfont(2);}
 glutInitDisplayMode(GLUT_DOUBLE|GLUT_RGB|GLUT_DEPTH);
 glutInitWindowSize(bildbreite,bildhoehe);
 glutInitWindowPosition(bildposx,bildposy);
 glutCreateWindow(titel);

 //ev. Menu hier setzen statt in main() ?
 setmenu(1,"File");
 setmenu(1,"Exit",&menu_exit);
 
 glShadeModel(GL_SMOOTH);
 glEnable(GL_DEPTH_TEST);

 /* Register GLUT callbacks. */
 glutDisplayFunc(display);
 glutReshapeFunc(reshape);
 glutKeyboardFunc(key);
 glutMouseFunc(mausknopf);
 glutMotionFunc(mausmotion);
 glutPassiveMotionFunc(mausmotion2);
 // TODO: vermutlich noch gebraucht:
 //glutSpecialFunc(spezial);

 // wahrscheinlich nicht gebraucht:
 //glutIdleFunc(idle);
 // zum Loslassen von Tasten:
 //glutKeyboardUpFunc(void (*func)(unsigned char key,int x,int y));
 //glutSpecialUpFunc(void (*func)(int key,int x, int y));

 /* Init the GL state */
 glLineWidth(1.0);
}

void menu_text_zeichnen() //Menu zeichnen und evt. auch noch andere Texte
{
 glMatrixMode(GL_PROJECTION);
#ifdef BELEUCHTUNG
 glDisable(GL_LIGHTING);
#endif
 glPushMatrix();
 glLoadIdentity();
 gluOrtho2D(0, bildbreite, 0, bildhoehe);
 /*
 Mit dem DEPTH_BUFFER wird im Gegensatz zum normalen Verhalten das zuerst
 gezeichnete behalten.
 Schon gezeichnete Pixel werden also nicht ueberzeichnet.
 Also z.B. schwarze Schrift auf hellgrauem Hintergrund zuerst die Schrift,
 dann erst den Hintergrund zeichnen!
 */
 if(requester1.aktiv) requester1.zeichnen();
 if(menu.aktiv) menu.zeichnen();
 else if(menu.anzahl>0)
  {
   double x=REQ_RAND, y=bildhoehe-MBH+3;
   menu.mtextsize();
   glColor3ub(0,0,0); // Schwarze Schrift
   schrift(x,y,"Menuleiste"); //auskommentieren fuer Menu unsichtbar wenn nicht aktiv
  }
#ifdef BELEUCHTUNG
 glEnable(GL_LIGHTING);
#endif
 glPopMatrix();
 glMatrixMode(GL_MODELVIEW);
}

static Vektor3d eye(0.0, 0.0, 10.0); //Kameraposition
static Vektor3d cent(0.0, 0.0, 0.0); //Punkt auf den die Kamera schaut
static Vektor3d up(0.0 , 1.0, 0.0); //Oben-Richtung der Kamera

void display(void)
{
 //static int testnr=0;
 //printf("display() %d.Aufruf\n",++testnr);//test
 if(exitflag) exit(0);
#ifdef SCHWARZAUFWEISS
 glClearColor(1.0, 1.0, 1.0, 1); //Hintergrundfarbe weiss
 //glClearColor(0.5, 0.5, 0.5, 1); //Hintergrundfarbe grau
#else
 glClearColor(0.0, 0.0, 0.0, 1); //Hintergrundfarbe schwarz
#endif
 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
 
#ifdef BELEUCHTUNG
 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
 GLfloat light_pos[]={(float)xmax,(float)ymax,(float)zmax,1};
  GLfloat ambient_hell[]={0.25,0.25,0.25,1};
  GLfloat ambient_dunkel[]={0,0,0,1};
  glLightfv(GL_LIGHT0,GL_POSITION,light_pos);
  if(ambientflag==0) glLightfv(GL_LIGHT0,GL_AMBIENT,ambient_dunkel);
  else               glLightfv(GL_LIGHT0,GL_AMBIENT,ambient_hell);
  glEnable(GL_COLOR_MATERIAL);
#endif

 menu_text_zeichnen(); //Menu und event. andere Texte (z.B. Requester) zeichnen

 glMatrixMode(GL_MODELVIEW);
 glLoadIdentity();
 glPushMatrix(); // GL_MODELVIEW is default
 glScalef(1.0, bildbreite/(float)bildhoehe, 1.0);
 gluLookAt(eye.x, eye.y, eye.z, cent.x, cent.y, cent.z, up.x,up.y,up.z);
 
 glEnable(GL_BLEND); //antialias
 glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); //antialias
 glEnable(GL_LINE_SMOOTH); //antialias
 zeichne_alles(); //Box und alle Atome zeichnen
 glPopMatrix();
 glutSwapBuffers();
};

/*********************** Zeichnen-Funktionen **************************/
// Einfache Projektion:
void projektion(double x1,double y1,double z1,double *x,double *y)
{
 *x = x1+0.35*z1;
 *y = y1+0.35*z1;
}

void linie(double x1,double y1,double z1,double x2,double y2,double z2)
{
 double x,y,z=0; //provisorisch
 projektion(x1,y1,z1,&x,&y);
 glVertex3d(x, y, z);
 projektion(x2,y2,z2,&x,&y);
 glVertex3d(x, y, z);
}

void box_hinten_zeichnen()
{
 glBegin(GL_LINES);
 linie(-BOXX,-BOXY,-BOXZ, -BOXX,-BOXY,BOXZ);
 linie(-BOXX,-BOXY, BOXZ,  -BOXX, BOXY, BOXZ);
 linie(-BOXX, BOXY, BOXZ,   BOXX, BOXY, BOXZ);
 linie( BOXX, BOXY, BOXZ,   BOXX,-BOXY, BOXZ);
 linie( BOXX,-BOXY, BOXZ,  -BOXX,-BOXY, BOXZ);
 glEnd();
}

void box_vorne_zeichnen()
{
 glBegin(GL_LINES);
 linie(-BOXX,-BOXY,-BOXZ,  -BOXX, BOXY,-BOXZ);
 linie(-BOXX, BOXY,-BOXZ,   BOXX, BOXY,-BOXZ);
 linie( BOXX, BOXY,-BOXZ,   BOXX,-BOXY,-BOXZ);
 linie( BOXX,-BOXY,-BOXZ,  -BOXX,-BOXY,-BOXZ);
 double x=BOXX, y=BOXY;
 for(int i=0;i<4;i++)
  {
   if(i!=2) linie(x,y,-BOXZ, x,y,BOXZ);
   if(i%2==1) x= -x; else y= -y;
  }
 glEnd();
}

static int sortierliste[NATOME];

void sortieren(int n,double *z,int *s)
{
 for(int i=0;i<n;i++) s[i]=i;
 bool aenderung=true;
 while(aenderung)
  {
   aenderung=false;
   for(int j=0;j<n-2;j++)
    for(int i=j;i<n-1;i++)
     {
      if(z[s[i]] < z[s[i+1]]) //vertauschen, so dass groesste z-Werte zuerst
       {int h=s[i]; s[i]=s[i+1]; s[i+1]=h;
	aenderung=true;
       }
     }
  }
}

void atome_zeichnen()
{
 int *s=sortierliste;
 static int test=0;//test
 if(test<=5) stoppuhr_reset();
 //if(true) //test: immer mit CPU sortieren
 if(argflag['T'])
  sortieren(NATOME,atom_posz,s);
 else
  {
   //bool ok=
   kernel_aufruf_sort(s);
   //if(!ok) {fprintf(stderr,"Error: kernel_aufruf_sort()\n"); exit(1);}//test
  }
 int usec=0;
 if(test<=5) usec=stoppuhr_read();
 if(test<5)//test
  {
   /*
   printf("Sortierte Liste:");
   for(int i=0;i<8 && i<NATOME;i++) printf(" %d",s[i]);
   printf("\nsort. atom_posz:");
   for(int i=0;i<8 && i<NATOME;i++) printf(" %f",(s[i]<NATOME)?atom_posz[s[i]]:s[i]);
   printf("\n");
   */
   printf("Sortierzeit: "); zeit_print(usec);
   test++;
   stoppuhr_reset();
  }
 //double r0=0.002;
 //glPointSize(pixel(r0));
 glEnable(GL_POINT_SMOOTH); //fuer runde Punkte
 //glBegin(GL_POINTS); //provisorisch: Atome als grosse Punkte zeichnen
 //for(int i=0;i<NATOME;i++)
 for(int i=NATOME;--i>=0;) //umgekehrte Reihenfolge mit GLUT_DEPTH 
  {
   double x,y;
   double z=atom_posz[s[i]];
   double r=atom_radius[s[i]];
   r -= (z+BOXZ)/(2*BOXZ)*0.5*r; //kleinerer Radius wenn weiter weg
   if(r<0.002) r=0.002;//test
   /*
   if(r!=r0)
    {glEnd();
     glPointSize(pixel(r0=r));
     glBegin(GL_POINTS);
    }
   */
   glPointSize(pixel(r));
   projektion(atom_posx[s[i]],atom_posy[s[i]],z,&x,&y);
   if(s[i]==0) glColor3ub(0xFF,0,0); //1.Atom rot
   else if(s[i]==(NATOME-1)) glColor3ub(0x3F,0x57,0xFF);//letztes Atom blau
   else glColor3ub(0xBF,0xBF,0xBF); //grau
   //fillcircle(x,y,r,r);
   glBegin(GL_POINTS); glVertex2d(x,y); glEnd();
#ifdef SCHWARZAUFWEISS
   glColor3ub(0,0,0);//schwarzer Kreis
#else
   glColor3ub(255,255,255); //weiss fuer weiss auf schwarz
#endif
   glPointSize(pixel(r)+4); //2 Pixel groesserer Durchmesser mehr fuer Kreis
   glBegin(GL_POINTS); glVertex2d(x,y); glEnd();
   //drawcircle(x,y,r,r);
  }
 //glEnd();
 glDisable(GL_POINT_SMOOTH); //wieder normale eckige Punkte
 glPointSize(1);             //und kleinste Groesse (wichtig fuer schrift)
 if(test<=5)
  {
   int usec2=stoppuhr_read();
   printf("Zeit zum Atome zeichnen: "); zeit_print(usec2);
   test++;
  }
}

/************************* Hauptprogramm ******************************/
void hauptschlaufe(int nr); //fuer Schlaufe mit Threads

//const
int TAUSEND=1000;

bool zunahe(int i,double dx)
{
 double dx2=dx*dx;
 for(int j=0;j<i;j++)
  {
   if(atom_posx[j]*atom_posx[j]+atom_posy[j]*atom_posy[j]+atom_posz[j]*atom_posz[j]
      < dx2) return true; //zu nahe
  }
 return false;//nicht zu nahe
}

int main(int argc,char **argv)
{
 char quellname[80],zielname[80];
 quellname[0]=zielname[0]=0;
 int i,j,c;
 bool ok=true;
 for(j=0,i=1;i<argc;i++)
  {if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
   else	{if(++j==1) strcpy(quellname,argv[i]);
         else if(j==2) strcpy(zielname,argv[i]);
  }	}
 if(argflag['?'] || j>MAXARG)
  {printf("edelgas  %s\n",VERSION);
   printf("Anwendung: edelgas [-Flags]\n");
   printf("  Flags: t=test, rechnen auf CPU\n");
#if(OPTIMIERUNG==1)
   printf("         p=TPB setzen, z.B. -p32 (maximal -p1024 oder NATOME/2)\n");
#else
   printf("         p=TPB setzen, z.B. -p32 (maximal -p1024 oder NATOME)\n");
#endif
#if(OPTIMIERUNG==3)
   printf("         q=TPBx:y setzen, z.B. -q16:16 (Voreingestellt 32:32)\n");
   printf("         a=Automatisch optimale TPBs suchen\n");
#endif
   exit(0);
  }

#if(OPTIMIERUNG==1)
 init_paartabelle(NATOME);
 set_TPB(NATOME/2);
#endif
 if(argflag['P'] && neues_TPB!=0)
  {
   int tst=neues_TPB;
   while((tst&1)==0) {tst>>=1;}
   if(tst!=1) printf("TPB sollte die Form 2^n haben. TPB=%d\n",neues_TPB);//test
   set_TPB(neues_TPB);
  }
 if(argflag['Q'] && neues_TPBy!=0)
  {
   set_TPBxy(neues_TPBx,neues_TPBy);
  }
 for(i=0;i<NATOME;i++)
  {
   const double r=0.031; //Radius in Nanometer
   atom_masse[i]=4.0026; //alles Helium
   atom_radius[i]=r;
   //Zufaellige Positionen:
   const double platz=2*(BOXX-r); //provi. fuer BOXX==BOXY==BOXZ
   int mz=0;//maximalzaehler
   do {
   atom_posx[i] = (random()&0xFFFFFF)/float(0xFFFFFF)*platz-(BOXX-r);
   atom_posy[i] = (random()&0xFFFFFF)/float(0xFFFFFF)*platz-(BOXY-r);
   atom_posz[i] = (random()&0xFFFFFF)/float(0xFFFFFF)*platz-(BOXZ-r);
   } while(zunahe(i,r*2*1.5) && ++mz<100);
   //Zufaellige Geschwindigkeiten in nm/sec:
   //etwa in 10 Sekunden die Box durchqueren:
   const float vmax=(sqrt(3.0)*2*BOXX)/10; //nm/sec
   atom_vx[i] = (random()&0xFFFFFF)/float(0xFFFFFF)*(2*vmax)-vmax;
   atom_vy[i] = (random()&0xFFFFFF)/float(0xFFFFFF)*(2*vmax)-vmax;
   atom_vz[i] = (random()&0xFFFFFF)/float(0xFFFFFF)*(2*vmax)-vmax;
  }
 if(NATOME>=4 && NATOME<=8) //test mit bekannten Zahlen
  {
   atom_posz[0]=atom_posz[1]=atom_posz[2]=0;
   atom_posx[0]= -BOXX*3/4; atom_posx[1]=BOXX*3/4; atom_posx[2]= -BOXX*3/4;
   atom_posy[0]=atom_posy[1]= -BOXY*3/4; atom_posy[2]=BOXY*3/4;
   atom_vz[0]=atom_vz[1]=0; atom_vz[2]=BOXX/10; //0;
   atom_vy[0]=atom_vy[1]=0; atom_vy[2]=0;
   atom_vx[0]=BOXX/20; atom_vx[1]= -BOXX/20; atom_vx[2]=0; //BOXX/10;
   atom_posx[3]= BOXX*3/4; atom_posy[3]=BOXY*3/4; atom_posz[4]=0;
   atom_vx[3]=0; atom_vy[3]=0; atom_vz[4]=BOXZ/10;
  }
 printf("\nStart-Positionen: "); //einige Werte anzeigen
 for(int i=0;i<3;i++) printf("%f ",atom_posx[i]);
 printf(".... ");
 for(int i=NATOME-3;i<NATOME;i++) printf("%f ",atom_posx[i]);
 printf("\n");
#if(OPTIMIERUNG>=1)
 copy_to_device_opt(atom_posx,atom_posy,atom_posz, atom_vx,atom_vy,atom_vz,
		atom_masse,atom_radius,paartabelle);
#else
 copy_to_device(atom_posx,atom_posy,atom_posz, atom_vx,atom_vy,atom_vz,
		atom_masse,atom_radius,NATOME);
#endif
 ok=kernel_compilieren(kernel_name);
 if(!ok) {printf("kernel_compilieren() misslungen.\n"); exit(1);}
 double dt=0.05*1e-3; //1000 Schritte fuer 0.05 Sekunden
 int usec=0;
 printf("Anzahl Atome: %d\n",NATOME);//test
 printf("Kernel_name: \"%s\"\n",kernel_name);//test
 if(neues_TPB!=0) printf("TPB=%d\n",neues_TPB);//test
 if(neues_TPBx!=0 || neues_TPBy!=0) printf("TPBxy=%d:%d\n",
					   neues_TPBx,neues_TPBy);//test
 if(argflag['T'])
  {
   printf("auf CPU gerechnet:\n");
   stoppuhr_reset();
   for(int i=0;ok && i<1000;i++) kernel_cpu(dt); //test mit CPU
   usec=stoppuhr_read();
   printf("Rechenzeit auf CPU: "); zeit_print(usec);
  }
 else
  {
   printf("auf Grafikkarte mit OpenCL gerechnet:\n");
   if(argflag['A'])
    { //erste 0.05 sec simulieren und dabei TPBs optimieren:
     int i0=100;
     for(int i=0;ok && i<i0;i++) //zuerst einige Zyklen rechnen,
      ok=kernel_aufrufen(dt);      
     i0 += tpbs_optimieren(dt); //dann Optimierung machen.
     for(int i=i0;ok && i<1000;i++) //total erste 0.05 sec simulieren
      ok=kernel_aufrufen(dt);
    }
   else
    {
     for(int i=0;ok && i<1000;i++) //erste 0.05 sec simulieren
      ok=kernel_aufrufen(dt);
    }
   stoppuhr_reset();
   for(int i=0;ok && i<1000;i++) //zweite 0.05 sec simulieren mit Zeitmessung
     ok=kernel_aufrufen(dt);
   if(ok)
    {
     int usec1=stoppuhr_read();
     copy_from_device(atom_posx,atom_posy,atom_posz,NATOME);
     int usec2=stoppuhr_read();
     printf("Rechenzeit mit OpenCL: "); zeit_print(usec1);
     printf("  mit zurueckkopieren: "); zeit_print(usec2);
     usec=usec2;
    }
  }
 if(usec>MAXBER) //maximal soviele usec fuer Berechnung brauchen
  {
   TAUSEND=1000*MAXBER/usec; //kleinere Anzahl als 1000 Schritte pro 0.05 Sekunden
   if(TAUSEND<10) TAUSEND=10; //minimale Anzahl trotzdem
   printf("TAUSEND=%d\n",TAUSEND);
  }
#ifdef DEBUG
 if(ok)
  {
   printf("\nNeue Positionen: "); //einige Ergebnisse anzeigen
   for(int i=0;i<3;i++) printf("%f ",atom_posx[i]);
   printf(".... ");
   for(int i=NATOME-3;i<NATOME;i++) printf("%f ",atom_posx[i]);
   printf("\n");
  }
#endif

/** Grafikteil xtekplot1: **
#ifdef DEBUG
 //tek_setdebug(1);//test
#endif
 int breite,hoehe,tiefe,visklasse;
 getmaxsize(&breite,&hoehe,&tiefe,&visklasse);
 if(tiefe>TIEFE) tiefe=TIEFE;
 if(breite>XMAX) breite=XMAX;
 if(hoehe>YMAX) hoehe=YMAX;
 setsize(breite,hoehe,tiefe);
 setmenu(1,"File");
 setmenu(1,"Exit",&menu_exit);
 inital(xmin,ymin,xmax,ymax); // Grafikfenster oeffnen
#ifdef SCHWARZAUFWEISS
 //fuer schwarz auf weiss (default ist weiss auf schwarz):
 if(TIEFE==24)
  {screenclear(0xFFFFFF); rgbcolor(0x00,0x00,0x00);}
 else
  {screenclear(1); color(0);}
#endif
 term_refresh();
 std::thread second(hauptschlaufe,1);
 hauptschlaufe(0);
 second.join();
 device_speicher_freigeben();
 term_exit);
**/
/** Grafikteil OpenGL: **/
 // weiter oben noch grafik_init() reshape() display() und weitere Funktionen
 glutInit(&argc, argv);
 grafik_init("Molz - Molekuele zeichnen mit OpenGL");
 std::thread second(hauptschlaufe,1);
 //printf("starte menu.startMainLoop()\n");//test
 menu.startMainLoop(); //startet glutMainLoop() und menu.loop() in separaten Threads
 printf("second.join();\n");//test, wird nie erreicht!
 second.join();
 printf("device_speicher_freigeben();\n");//test
 device_speicher_freigeben();
/**/
 return 0;
}

static int zwischenzeiten[4];
static int zyklen=0;

void zeiten_anzeigen()
{
 int *z=zwischenzeiten;
 printf("Kernelzeiten:\n");
 printf("kernel_hello: %d, _fsum: %d, _vpos: %d us\n",
	             z[0], z[1]-z[0], z[2]-z[1]);
 printf(" total: "); zeit_print(z[2]);
 int t,x,y; get_tpb(&t,&x,&y);
 printf("TPB=%d TPBxy=%d:%d\n",t,x,y);
}

bool neue_positionen_berechnen()
{
 bool ok=true;
 //const double dt=0.05*1e-3; //1000 Schritte fuer 0.05 Sekunden
 const double dt=0.05/TAUSEND; //1000 Schritte fuer 0.05 Sekunden
 if(argflag['T'])
  {
   for(int i=0;ok && i<TAUSEND;i++) kernel_cpu(dt); //test mit CPU
  }
 else
  {
   for(int i=0;ok && i<TAUSEND;i++) //naechste 0.05 sec simulieren
     ok=kernel_aufrufen(dt);
   if(ok) copy_from_device(atom_posx,atom_posy,atom_posz,NATOME);
  }
 return ok;
}

#ifdef SCHWARZAUFWEISS
#define HINTERGRUNDFARBE 0xFFFFFF
#define LINIENFARBE 0
#else
#define HINTERGRUNDFARBE 0
#define LINIENFARBE 0xFFFFFF
#endif

/*
void myscreenclear()
{
 screenclear(HINTERGRUNDFARBE);
}
*/

static bool neuzeichnen=true;

void zeichne_alles()
{
 if(zyklen>=10 && zyklen<=13) stoppuhr_reset();//test
 glColor3ub(0,0,0); //schwarz
 box_vorne_zeichnen();
 atome_zeichnen();
 glColor3ub(0,0,0); //schwarz
 box_hinten_zeichnen();
 if(zyklen>=10 && zyklen<=13)//test
  {
   int usec=stoppuhr_read();
   printf("Zeit zum alles zeichnen: "); zeit_print(usec);
  }
 neuzeichnen=false;
}

void hauptschlaufe(int nr)
{
 printf("hauptschlaufe(nr=%d)\n",nr);//test
 if(nr==0)
  { // CPU-Thread fuer Grafische Darstellung: fuer OpenGL schon in main() gemacht
  }
 else
  { // CPU-Thread fuer Berechnungen
   bool ok;
   glutPostRedisplay();
   while(exitflag==0)
    {
     if(!fastflag)
      while(neuzeichnen) {usleep(10);}//warten bis aktuelle Generation fertig gezeichnet
     //eventuell besser zu loesen mit semaphore statt usleep()
     if(!stopflag)
      {
       ok=neue_positionen_berechnen();
       zyklen++;
       neuzeichnen=true;
       glutPostRedisplay();
       if(!ok)
	{printf("Fehler in neue_positionen_berechnen() nach %d Zyklen.\n",zyklen); break;}
       //if(zyklen==maxzyklen)
       //  {printf("maxzyklen erreicht: %d\n",zyklen); exitflag=1; break;}
      }
    }
  }
 return;
}

/****************************** Kleinkram ******************************/
bool fastgleich(float a,float b)
{
 const float fasteins=0.9999, guteins=1.0001;
 if(a==b) return true;
 if(a==0 || b==0) return false;
 double eins=a/b;
 return (eins>=fasteins && eins<=guteins);
}

/****************************** Stoppuhr: ******************************/
#include <time.h>
#include <sys/time.h>
static struct timeval stoppuhr_tv0, stoppuhr_tv;
static struct timezone stoppuhr_tz0, stoppuhr_tz;

void stoppuhr_reset()
{
 gettimeofday(&stoppuhr_tv0,&stoppuhr_tz0);
}

int stoppuhr_read() //Zeit seit letztem stoppuhr_reset() in usec
{
 int usec,sec;
 gettimeofday(&stoppuhr_tv,&stoppuhr_tz);
 usec = stoppuhr_tv.tv_usec - stoppuhr_tv0.tv_usec;
 if((sec=stoppuhr_tv.tv_sec-stoppuhr_tv0.tv_sec)!=0) usec += sec*1000000;
 return usec;
}

void zeit_print(int usec,long flop)
{
 if(usec>=500000)
      printf("%d.%03d sec",usec/1000000,(usec/1000)%1000);
 else printf("%d.%03d ms",usec/1000,usec%1000);
 if(flop>0)
  {
   double gflops=flop/(usec*1e-6)/1e9;
   if(gflops<2)        printf("  (%.3f GFLOPS)\n",gflops);
   else if(gflops<200) printf("  (%.1f GFLOPS)\n",gflops);
   else                printf("  (%.0f GFLOPS)\n",gflops);
  }
 else printf("\n");
}
/**************************** Ende Stoppuhr ****************************/

/******************* Tabellen fuer Parallel-Rechnung *******************/
//#define MAXPAAR ((NATOME-1)*NATOME/2) //schon weiter oben
//static short2 paartabelle[MAXPAAR]; //schon weiter oben
static short3 paartabsort[MAXPAAR];

#define MAXKNOTEN NATOME
static int knotentabelle[MAXKNOTEN];

bool schonvorhanden(short knr,short2 paar,short3 *paarsort,int mins,int maxs)
{
 //Suche ob unter der gleichen Knotennummer (knr) schon ein i- oder j-Wert
 //vorhanden ist.
 for(int s=mins;s<maxs;s++)
  {
   if(paarsort[s].x==knr &&
      (paarsort[s].y==paar.x || paarsort[s].y==paar.y ||
       paarsort[s].z==paar.x || paarsort[s].z==paar.y
      )
     ) return true;
  }
 return false;
}

void paartab_save(const int natome)
{
 const int imax=(natome-1)*natome/2;
 char name[200];
 sprintf(name,"paartabelle%d.txt",natome);
 FILE *fp=fopen(name,"w");
 if(fp==NULL) {printf("Fehler: kann \"%s\" nicht erstellen\n",name); return;}
 for(int i=0;i<imax;i++)
  {
   fprintf(fp,"%d %d\n",paartabelle[i].x,paartabelle[i].y);
  }
 fclose(fp);
}

void init_paartabelle(int natome)
{
 if(natome>256)
  {//versuchen aus Datei zu lesen, da Berechnung ziemlich lang dauert:
   char name[200];
   sprintf(name,"paartabelle%d.txt",natome);
   FILE *fp=fopen(name,"r");
   if(fp!=NULL)
    {
     bool ok=true;
     for(int i=0;i<MAXPAAR;i++)
      {
       int n,x,y;
       n=fscanf(fp,"%d %d",&x,&y);
       if(n!=2)
	{printf("Fehler: fscanf() kann nicht genuegend Werte lesen\n"); ok=false; break;}
       paartabelle[i].x=x;
       paartabelle[i].y=y;
      }
     fclose(fp);
     if(ok) return;
    }
  }
 int p=0;
 for(int i=0;i<natome-1;i++)
  for(int j=i+1;j<natome;j++)
   {
    paartabelle[p++]={(short)i,(short)j};
   }
 printf("Anzahl Paare: %d\n",p);//test
 if(p!=MAXPAAR) printf("Fehler: p=%d MAXPAAR=%d sollten gleich sein\n",p,MAXPAAR);

 //sortierte Paartabelle aufbauen:
 short knr=0; //Knotennummer
 int s0=0,s1=0;
 for(int s=0;s<p;) //s=Index fuer sortierte Tabelle
  {
   for(int i=0;i<p;i++) //gesamte unsortierte Tabelle durchsuchen
    {
     if(paartabelle[i].x>=0 &&
	!schonvorhanden(knr,paartabelle[i],paartabsort,s0,s))
      {
       paartabsort[s].x=knr; //Knotennummer
       paartabsort[s].y=paartabelle[i].x; //i-Wert
       paartabsort[s].z=paartabelle[i].y; //j-Wert
       s++;
       paartabelle[i].x = -1; //Eintrag aus unsortierter Tabelle entfernen
      }
    }
   s0=s1; s1=s-1;
   knr++;
  }
 printf("Anzahl Knoten: %d\n",knr);//test
 if(NATOME<=32) //test
  {
   printf("knr i j\n");//test
   for(int i=0;i<p;i++) //test
    {
     short3 p=paartabsort[i];
     printf("%d %d %d\n",p.x,p.y,p.z);
    }
  }

 if(knr!=MAXKNOTEN-1)
  printf("Fehler: knr=%d MAXKNOTEN-1=%d sollten gleich sein\n",knr,MAXKNOTEN-1);//test
 int k=0,i;
 for(i=0;i<knr;i++)
  {
   knotentabelle[i]=k;
   while(paartabsort[k].x==i) {k++;}
  }
 knotentabelle[i]=k;
  for(int i=0;i<=knr;i++) //test: Resultate pruefen
  {
   if(knotentabelle[i]!=i*NATOME/2)
    printf("Fehler: knotentabelle[%d]=%d i*NATOME/2=%d\n",i,knotentabelle[i],i*NATOME/2);
  }
 
 if(NATOME<=32)//test
  {
   printf("Knotentabelle:\n");//test
   for(i=0;i<=knr;i++)//test
    printf(" %d\n",knotentabelle[i]);//test
  }

 for(int i=0;i<MAXPAAR;i++) //sortierte Tabelle umkopieren
  {
   paartabelle[i].x = paartabsort[i].y;
   paartabelle[i].y = paartabsort[i].z;
  }
 paartab_save(natome);
}

int tpbs_optimieren(double dt)
{
 int i0=0;
 //Suche nach besten TPBs
 bool ok;
 int zwi[4];
 int tpb,tpbx,tpby;
 get_tpb(&tpb,&tpbx,&tpby);
 int best=1000000; //beste also kleinste Zeit in usec
 int best_tpb=tpb,best_tpbx=tpbx,best_tpby=tpby;
 best=1000000;
 for(tpb=1;tpb<=1024;tpb*=2) //nach bestem TPB fuer 1D-Kernels suchen
  for(tpby=1;tpby<=1024;tpby*=2) //nach bestem TPBy fuer 2D-Kernels suchen
   for(tpbx=1;tpbx*tpby<=1024;tpbx*=2) //nach bestem TPBx fuer 2D-Kernels suchen
    {
     set_TPB(tpb);
     set_TPBxy(tpbx,tpby);
     stoppuhr_reset();
     ok=kernel_aufrufen(dt,zwi); i0++;
     zwi[2]=stoppuhr_read();
     if(zwi[1]<zwi[0] || zwi[2]<zwi[1])//test
      printf("Fehler: zwi[0]=%d zwi[1]=%d zwi[2]=%d\n",zwi[0],zwi[1],zwi[2]);//test
     if(ok && zwi[2]<best)
      {best=zwi[2]; best_tpb=tpb; best_tpbx=tpbx; best_tpby=tpby;}
    }
 set_TPB(best_tpb);
 set_TPBxy(best_tpbx,best_tpby);
 stoppuhr_reset();
 ok=kernel_aufrufen(dt,zwischenzeiten); i0++;
 if(!ok) printf("Fehler: Kernelaufruf mit optimierten TPBs misslungen\n");//test
 zwischenzeiten[2]=stoppuhr_read();
 zeiten_anzeigen();
#ifdef DEBUG
 printf("i0=%d\n",i0);//test
#endif
 return i0;
}
