/* biber.cc			letzte Aenderung: 27.8.2017 */
#define VERSION "Version 0.1"
/*
Uebersetzen auf Unix (Linux):
> make  ;siehe makefile

Fleissiger Biber:
 Suche nach der Turingmaschine mit Z Zustaenden, die am laengste laeuft,
 aber nicht unendlich lang.
 Gestartet wird mit einem Band mit lauter Nullen. Es sind  nur die
 Zahlen 0 und 1 auf dem Band erlaubt.

History:
Datum      Vers
20.8.2017       Erstellung
27.8.2017  0.1  groesseres ZMAX

*/

#define BAND_DARSTELLEN  //bei Z>=5 auskommentieren
#define ESC_MARKIERUNGEN //markiere aktuelle Position farbig, zuletzt gschriebenes fett
#define LOOPTEST //Testen ob in einer unendlichen Schlaufe, und dann abbrechen

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dateiklasse.cc"

/************************* Vordeklarationen ***************************/

/************************* Globale Variablen **************************/
bool errflag=false;
int errnum=0;
#define BANDVOLL 1
#define LOOPUNENDLICH 2

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(const char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
  }
}

/*************************** Kleinkram ********************************/
/*
bool istja(const char *s)
{
 int c= *s;
 return (c=='j' || c=='J' || c=='y' || c=='Y');
}
*/

/************** Klassen und Hauptteil des Programms *******************/
#define ZMAX 20 //maximale Anzahl Zustaende, ein fleissiger Biber mit mehr als 6 Zustaenden ist zwar zu kompliziert,
                //aber fuer andere Turingmaschinen sinnvoll (z.B. Addierer)
//#define MAXBAND 1000 //Laenge des Bandes
//#define MAXBAND 100000 //Laenge des Bandes
#define MAXBAND 1000000 //Laenge des Bandes (sollte fast unendlich sein)
#define BANDMITTE (MAXBAND/2)

char zustand_char[8]={'H','A','B','C','D','E','F','G'}; //H=HALT

const char *zustand_name(int i)
{
 if(i==0) return "Halt";
 static char str[8];
 if(i<=8) {str[0]=zustand_char[i]; str[1]=0; return str;}
 sprintf(str,"Z%d",i);
 return str;
}

int zustand_einlesen(const char *str) //Zustaende mussen mit 'A' 'B' ... 'G' oder Z0 ... Z19 bezeichnet sein, 'H' fuer Halt
{
 int zustand=0;
 char c = *str++;
 if(c=='H') return 0; //Haltezustand (Nummer 0 in aktueller Programmversion)
 if(c>='A' && c<='G') zustand = c+1-'A';
 else if(c=='Z')
  {sscanf(str,"%d",&zustand); zustand++;} // Z0 entspricht Zustand 'A' und bekommt Nummer 1
 return zustand; //0 fuer Halt oder Zahl 1 ... ZMAX
}

char band[MAXBAND];
int minband,maxband;
int bandpos,lastbandpos;

void band_init()
{
 lastbandpos=bandpos=BANDMITTE; //in der Mitte des unendlichen Bandes starten
 maxband=bandpos+1;
 minband=bandpos-1;
 for(int i=0;i<MAXBAND;i++) band[i]=0;//mit leerem Band starten
}

void band_einsen_setzen(uint u) //test
{
 int i=bandpos;
 while(u!=0)
  {
   band[i++] = (u&1);
   u >>= 1;
   maxband++;
  }
}

#define RECHTS 1
#define LINKS  0

class Biber
{
public:
 int Z; //Anzahl moegliche Zustaende
 int programm[(ZMAX+1)*2]; //Index ist Zustand*2 + gelesene Zahl
 int zustand; //aktueller Zustand als Zahl, 0=Halt, 1='A', 2='B'...
 Biber()
   {zustand=1; Z=1;
    setprog('A',0, 1,RECHTS,'H'); //einzige 2 Moeglichkeiten dass er anhaelt (LINKS geht auch)
    setprog('A',1, 0,RECHTS,'A'); //irrelevant was hier steht.
   }
 void init(int z);
 void prog_init(int z);
 void setprog(int zust,int zahl,int neuezahl,int richtung,int neuerzustand);
 void step();
};

void Biber::init(int z)
{
 Z=z;
 zustand=1; //es wird mit Zustand A gestartet
 //band_init();
}

void Biber::prog_init(int z)
{
 if(z==2)
  {
   setprog('A',0, 1,RECHTS,'B');
   setprog('A',1, 1,LINKS, 'B');
   setprog('B',0, 1,LINKS, 'A');
   setprog('B',1, 1,RECHTS,'H');
  }
 else if(z==3)
  {
   setprog('A',0, 1,RECHTS,'B'); //ein optimaler Biber fuer Z=3
   setprog('A',1, 1,RECHTS,'H');
   setprog('B',0, 1,LINKS, 'B');
   setprog('B',1, 0,RECHTS,'C');
   setprog('C',0, 1,LINKS, 'C');
   setprog('C',1, 1,LINKS, 'A');
  }
 else if(z==4)
  {
   setprog('A',0, 1,RECHTS,'B'); //bis z==4 wurden schon optimale Biber gefunden
   setprog('A',1, 1,LINKS, 'B');
   setprog('B',0, 1,LINKS, 'A');
   setprog('B',1, 0,LINKS, 'C');
   setprog('C',0, 1,RECHTS,'H');
   setprog('C',1, 1,LINKS, 'D');
   setprog('D',0, 1,RECHTS,'D');
   setprog('D',1, 0,RECHTS,'A');
  }
 else if(z==5)
  {
   setprog('A',0, 1,LINKS, 'B'); //vermutlich optimaler Biber
   setprog('A',1, 1,RECHTS,'C');
   setprog('B',0, 1,LINKS, 'C');
   setprog('B',1, 1,LINKS, 'B');
   setprog('C',0, 1,LINKS, 'D');
   setprog('C',1, 0,RECHTS,'E');
   setprog('D',0, 1,RECHTS,'A');
   setprog('D',1, 1,RECHTS,'D');
   setprog('E',0, 1,LINKS, 'H');
   setprog('E',1, 0,RECHTS,'A');
  }
 else if(z==6)
  {
   setprog('A',0, 1,LINKS, 'B');
   setprog('A',1, 1,RECHTS,'C');
   setprog('B',0, 1,LINKS, 'C');
   setprog('B',1, 1,LINKS, 'B');
   setprog('C',0, 1,LINKS, 'D');
   setprog('C',1, 0,RECHTS,'E');
   setprog('D',0, 1,RECHTS,'A');
   setprog('D',1, 1,RECHTS,'D');
   setprog('E',0, 1,LINKS, 'F'); //statt Halt wie beim z==5 hier zum F
   setprog('E',1, 0,RECHTS,'A');
   setprog('F',0, 1,RECHTS,'F'); //1 Schritt mehr ist immer moeglich
   setprog('F',1, 1,LINKS, 'H'); //hier auch 2.Schritt bei gleich lassen der andern
  }
}

void Biber::setprog(int zust,int zahl,int neuezahl,int richtung,int neuerzustand)
{
 if(zahl>1) {zahl=1; fprintf(stderr,"zahl darf nur 0 oder 1 sein.\n");}
 int i;
 if(zust=='H') {fprintf(stderr,"Fehler: Halt-Zustand darf nicht definiert werden\n"); return;}
 if(zust>='A') i = (zust-'A'+1)*2+zahl;
 else          i = zust*2+zahl;
 if(i>ZMAX*2+1) {fprintf(stderr,"zust*2+zahl = i=%d zu gross\n",i);}
 if(richtung>1) {richtung=1; fprintf(stderr,"richtung darf nur 0 (=links) oder 1 (=rechts) sein.\n");}
 if(neuerzustand=='H') neuerzustand=0;
 else if(neuerzustand>='A') neuerzustand = neuerzustand+1-'A';
 programm[i] = (neuezahl<<5)+(richtung<<4)+neuerzustand; //je 1 Bit fuer zu schreibende Zahl und Richtung, 4 Bits fuer Zustand
}

void Biber::step()
{
 if(zustand==0) return; //bei HALT nichts machen
 int zahl=band[bandpos];
 int i=zustand*2+zahl;
 int neuezahl = (programm[i]>>5);
 int richtung = (programm[i]>>4)&0x1;
 //printf("neuezahl=%d, richtung=%d\n",neuezahl,richtung);//test
 band[bandpos]=neuezahl;
 lastbandpos=bandpos;
 if(richtung==LINKS)
  {
   if(bandpos==0) {fprintf(stderr,"Band zu kurz, linker Rand erreicht.\n"); errnum=BANDVOLL; errflag=true;}
   else if(--bandpos < minband) minband=bandpos;
  }
 else
  {
   if(bandpos==MAXBAND-1) {fprintf(stderr,"Band zu kurz, rechter Rand erreicht.\n"); errnum=BANDVOLL; errflag=true;}
   else if(++bandpos > maxband) maxband=bandpos;
  }
 zustand = programm[i]&0xF;
#ifdef LOOPTEST
 if(zustand==1 && band[bandpos]==0) //wieder im Startzustand
  {
   for(i=minband;i<=maxband;i++)
    if(band[i]!=0) break;
   if(i>maxband) {errnum=LOOPUNENDLICH; errflag=true;}
  }
#endif
}

void turingmaschine_einlesen(Datei& datei,Biber& biber)
{
 int z=0,j;
 datei.getzeile_ohne_kommentare(';');
 if(datei.zeile[1]=='=')
  {if(datei.zeile[0]=='Z' || datei.zeile[0]=='z') sscanf(&datei.zeile[2],"%d",&z);
  }
 if(z==0) {fprintf(stderr,"Syntaxfehler: %s\n",datei.zeile); return;}
 if(z>ZMAX) {fprintf(stderr,"z=%d zu gross. Z bisher auf %d beschraenkt\n",z,ZMAX); return;}

 biber.init(z);
 printf("Testausdrucke: Turingmaschine von Datei einlesen:\n");//test
 for(j=0;j<z*2;j++)
  {
   char zustand,neuerzustand,richtung;
   int rzahl,wzahl;
   char str_zust[20], str_neuzust[20];
   if(!datei.getzeile_ohne_kommentare(';')) {fprintf(stderr,"Zu wenige Daten in Datei\n"); return;}
   sscanf(&datei.zeile[0],"%s %d %d %c %s",str_zust,&rzahl,&wzahl,&richtung,str_neuzust);
   zustand=zustand_einlesen(str_zust);         //Zustaende mussen mit 'A' 'B' ... 'G' oder Z0 ... Z19 bezeichnet sein
   neuerzustand=zustand_einlesen(str_neuzust); //'H' oder Halt oder HALT fuer Haltezustand
   if(zustand>ZMAX || neuerzustand>ZMAX) printf("Syntaxfehler: zustand oder neuerzustand zu gross\n\"%s\"\n",datei.zeile);
   richtung = (richtung=='R') ? RECHTS : LINKS;
   printf("%s\n",datei.zeile);//test
   printf("zustand=%d rzahl=%d  wzahl=%d, richtung=%d,neuerzustand=%d\n",zustand,rzahl, wzahl,richtung,neuerzustand);//test
   biber.setprog(zustand,rzahl, wzahl,richtung,neuerzustand);
  }
 if(datei.getzeile_ohne_kommentare(';'))
  {
   if(strncmp("Band=",datei.zeile,5)==0)
    {
     int c,i,j=0;
     uint u=0,u2=0;
     for(i=5;(c=datei.zeile[i])=='1' || c=='0';i++)
      {u=(u<<1)+(c-'0'); j++;}
     for(i=0;i<j;i++) {c=u&1; u2=(u2<<1)+c; u>>=1;}
     band_einsen_setzen(u2);
     printf("band_einsen_setzen(%d)\n",u2);//test
    }
  }
 printf("Ende Testausdrucke: Turingmaschine von Datei einlesen.\n");//test
}

void print_pos_markiert(int c)
{
#ifdef ESC_MARKIERUNGEN
 printf("\033[33m%d\033[0m",c); //Position gelb markieren ([31=rot, [35=magenta, [33=gelb)
#else
 if(c==0) printf("o"); else printf("i"); //Position mit o fuer 0 und i fuer 1 markieren
#endif
}

void print_letztes_markiert(int c)
{
#ifdef ESC_MARKIERUNGEN
 printf("\033[1m%d\033[0m",c); //Letztes fett markieren
#else
 if(c==0) printf("O"); else printf("I");} //statt fett mit O fuer 0 und I fuer 1 markieren
#endif
}

/************************* Hauptprogramm ******************************/
int main(int argc,char *argv[])
{
 char quellname[80];
 int64_t i;
 int j,c;
 int z=3;
 quellname[0]=0;
 
 for(j=0,i=1;i<argc;i++)
  {if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
   else if(++j==1)
      {if(isdigit(*argv[i])) sscanf(argv[i],"%d",&z);
       else strcpy(quellname,argv[i]);
      }
   else if(j==2) strcpy(quellname,argv[i]);
  }
 
 if(argflag['?'] || j>MAXARG)
	{printf("biber  %s\n",VERSION);
	 printf("Anwendung: biber [Z] [datei.tm]\n");
	 printf(" fuer Z sind bisher nur Zahlen von 2 bis %d moeglich.\n",ZMAX);//test
	 printf("sizeof(long)=%d sizeof(int)=%d (int64_t)=%d\n",sizeof(long),sizeof(int),sizeof(int64_t));
	 exit(0);
	}

 Datei datei(quellname);
 bool dateiflag=false;
 if(quellname[0]!=0)
  {if(datei.open_read()) dateiflag=true;
   else {fprintf(stderr,"Datei '%s' nicht gefunden.\n",quellname); exit(0);}
  }

 Biber biber;
 band_init();
 printf("band_init() gemacht\n");//test
 if(dateiflag)
  {
   turingmaschine_einlesen(datei,biber);
   printf("von Datei eingelesen. Z=%d\n",biber.Z);
  }
 else
  {
   if(z<1 || z>6) {printf("keine Voreinstellung mit z=%d eingebaut.\n",z); z=2;}
   biber.init(z);
   biber.prog_init(z);
   printf("Initialisert mit Z=%d\n",biber.Z);
  }

#ifdef BAND_DARSTELLEN
 printf("  i Zust Band (aktuelle Position farbig oder mit o oder i markiert)\n");
 printf("              (zuletzt geschriebenes fett oder mit O oder I markiert)\n");
#endif
 long ioverflow=0;
 for(i=0;;i++)
  {
   if(i<0) {i=0; ioverflow++;}
   if(ioverflow==0 && (i<10000 || (i%1000000)==0))
    {
     printf("i=%2Ld  %s  ",i,zustand_name(biber.zustand));
#ifdef BAND_DARSTELLEN
     if(i<10000)
      {
       //printf("bandpos:%2d  Band: ",bandpos-BANDMITTE);
       for(int j=minband;j<=maxband;j++)
	{
	 if(j==bandpos) print_pos_markiert(band[j]); //aktuelle Position markieren
	 else if(j==lastbandpos) print_letztes_markiert(band[j]); //letztes fett markieren
	 else printf("%d",band[j]);//Band darstellen
	}
      }
     printf("\n");
#else //BAND_DARSTELLEN
     printf("bandpos = %2d\n",bandpos-BANDMITTE); //ohne Band-Darstellung: nur Position zeigen
#endif
    }
   if(biber.zustand==0 || errflag) break;
   biber.step();
  }
#ifdef BAND_DARSTELLEN
 //printf("bandpos=%2d  Band: ",bandpos-BANDMITTE);
 for(int j=minband;j<=maxband;j++)
  {
   if(j==bandpos) print_pos_markiert(band[j]); //aktuelle Position markieren
   else if(j==lastbandpos) print_letztes_markiert(band[j]); //letztes fett markieren
   else printf("%d",band[j]);//Band darstellen
  }
 printf("\n");
#endif
 if(ioverflow!=0) printf("Maschine ist %Ld+%ld*2^63 Schritte gelaufen.\n",i,ioverflow);
 else printf("Maschine ist %Ld Schritte gelaufen.\n",i);
 for(i=minband,c=0;i<=maxband;i++)
  if(band[i]==1) c++;
 printf("Und hat %d Einsen geschrieben.\n",c);
 printf("Benutzte Bandlaenge: %d\n",maxband-minband+1);
 if(errflag)
  {
   if(errnum==LOOPUNENDLICH) printf("Gestoppt wegen Unendlicher Schlaufe.\n");
   else printf("Fehler: Band zu kurz?\n");
  }
 else printf("HALT regulaer erreicht.\n");
 
 return 0;
}/* ende von main */
