/* biber.cc			letzte Aenderung: 6.9.2017 */
#define VERSION "Version 0.3"
/*
Uebersetzen auf Unix (Linux):
> make  ;siehe makefile

Fleissiger Biber:
 Suche nach der Turingmaschine mit Z Zustaenden, die am laengste laeuft,
 aber nicht unendlich lang.
 Gestartet wird mit einem Band mit lauter Nullen. Es sind  nur die
 Zahlen 0 und 1 auf dem Band erlaubt.

History:
Datum      Vers
20.8.2017       Erstellung
27.8.2017  0.1  groesseres ZMAX
29.8.2017  0.2  bessere Klasse Biber und Band jetzt auch als Klasse, 'A' ist jetzt 0, Buchstaben von A biz Y ohne H erlaubt
 6.9.2017  0.3  Erweiterung fuer mehr als 2 Symbole (SMAX und Biber.smax)

*/

#define BAND_DARSTELLEN  //bei Z>=5 auskommentieren
#define ESC_MARKIERUNGEN //markiere aktuelle Position farbig, zuletzt gschriebenes fett
//#define LOOPTEST //Testen ob in einer unendlichen Schlaufe, und dann abbrechen
//#define DEBUG  //Testausdrucke fuer Fehlersuche (bisher die meisten mit //test auskommentiert)

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dateiklasse.cc"

/************************* Vordeklarationen ***************************/
typedef unsigned char uchar;
typedef unsigned char uint8_t;

/************************* Globale Variablen **************************/
bool errflag=false;
int errnum=0;
#define BANDVOLL 1
#define LOOPUNENDLICH 2

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(const char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
  }
}

/*************************** Kleinkram ********************************/
/*
bool istja(const char *s)
{
 int c= *s;
 return (c=='j' || c=='J' || c=='y' || c=='Y');
}
*/

/************** Klassen und Hauptteil des Programms *******************/
#define SMAX 4  //Maximale Anzahl Symbole (bisher auf 4 beschraenkt)
#define ZMAX 25 //maximale Anzahl Zustaende, ein fleissiger Biber mit mehr als 6 Zustaenden ist zwar zu kompliziert,
                //aber fuer andere Turingmaschinen sinnvoll (z.B. Addierer)
//#define MAXBAND 100000 //Laenge des Bandes
#define MAXBAND 5000000 //Laenge des Bandes (sollte fast unendlich sein)
#define BANDMITTE (MAXBAND/2)

const char *zustand_name(int i)
{
 static char str[8];
 if(i==ZMAX) return "Halt";
 if(i<=24) {str[0] = (i>=7) ? i-7+'I' : 'A'+i; str[1]=0;}
#if(ZMAX>25)
 else sprintf(str,"Z%d",i);
#endif
 return str;
}

#define HALT ZMAX

int zustand_einlesen(const char *str)
{  //Zustaende mussen mit 'A' 'B' ... 'Y' ohne 'H', oder Z0 ... Z99 bezeichnet sein, 'H' fuer Halt
 int zustand=0;
 char c = *str++;
 if(c=='H') return HALT; //Haltezustand (Nummer ZMAX in aktueller Programmversion)
 if(c>='A' && c<='G') zustand = c-'A';
 else if(c>='I' && c<='Y') zustand = c-1-'A';
 else if(c=='Z')
  {sscanf(str,"%d",&zustand);} // Z0 entspricht Zustand 'A' und bekommt Nummer 0
 return zustand; // ZMAX fuer Halt oder Zahl 0 ... ZMAX-1
}

class Band
{
 uchar feld[MAXBAND];
public:
 uint pos,lastpos;
 uint posmin,posmax; //neu statt minband,maxband
 uint symbol_setz_j;
 Band() {init(); symbol_setz_j=pos;}
 void init();
 uchar& operator[](uint i) {return feld[i];}
 void einsen_setzen(uint u); //provi. fuer Voreinstellungen auf dem Band
 void symbol_setzen(uchar u,bool alleflag=false); //fuer Voreinstellungen auf dem Band
 uchar read() {return feld[pos];}
 void write(uchar neuezahl,uchar richtung,uchar neuerzustand);
};

static Band band;

void Band::init()
{
 lastpos=pos=BANDMITTE; //in der Mitte des unendlichen Bandes starten
 posmax=pos+1;
 posmin=pos-1;
 for(int i=0;i<MAXBAND;i++) feld[i]=0;//mit leerem Band starten
 //printf("Band::init() gemacht\n");//test
}

void Band::symbol_setzen(uchar u,bool alleflag)
{
 feld[symbol_setz_j++] = u;
 if(symbol_setz_j==MAXBAND) symbol_setz_j=0;
 if(alleflag)
  while(symbol_setz_j!=pos)
   {
    feld[symbol_setz_j++] = u;
    if(symbol_setz_j==MAXBAND) symbol_setz_j=0;
   }
}

void Band::einsen_setzen(uint u) //provi.
{
 int i=pos;
 while(u!=0)
  {
   feld[i++] = (u&1);
   u >>= 1;
   posmax++;
  }
}

#define RECHTS 1
#define LINKS  0

class Eintrag
{
public:
 uchar schreibwert,richtung,neuerzustand; //schreibwert muss im Bereich 0 bis SMAX-1 liegen
 void set(uchar s,uchar r,uchar z) {schreibwert=s; richtung=r; neuerzustand=z;}
};

class Biber
{
public:
 int Z; //Anzahl moegliche Zustaende
 uint8_t smax; //Anzahl moegliche Symbole
 Eintrag tabelle[ZMAX*SMAX]; //Index ist Zustand*AnzahlSymbole + gelesene Zahl
 int zustand; //aktueller Zustand als Zahl, 0='A', 1='B'... , ZMAX=Halt
 Biber()
   {zustand=0; Z=1;
    setprog('A',0, 1,RECHTS,'H'); //einzige 2 Moeglichkeiten dass er anhaelt (LINKS geht auch)
    setprog('A',1, 0,RECHTS,'A'); //irrelevant was hier steht.
   }
 void init(int z,uint8_t s);
 void prog_init(int z);
 void setprog(int zust,int zahl,int neuezahl,int richtung,int neuerzustand);
 void step();
};

void Biber::init(int z,uint8_t s=2)
{
 Z=z;
 smax=s;
 zustand=0; //es wird mit Zustand A gestartet
}

void Biber::prog_init(int z)
{
 if(z==2)
  {
   setprog('A',0, 1,RECHTS,'B');
   setprog('A',1, 1,LINKS, 'B');
   setprog('B',0, 1,LINKS, 'A');
   setprog('B',1, 1,RECHTS,'H');
  }
 else if(z==3)
  {
   setprog('A',0, 1,RECHTS,'B'); //ein optimaler Biber fuer Z=3
   setprog('A',1, 1,RECHTS,'H');
   setprog('B',0, 1,LINKS, 'B');
   setprog('B',1, 0,RECHTS,'C');
   setprog('C',0, 1,LINKS, 'C');
   setprog('C',1, 1,LINKS, 'A');
  }
 else if(z==4)
  {
   setprog('A',0, 1,RECHTS,'B'); //bis z==4 wurden schon optimale Biber gefunden
   setprog('A',1, 1,LINKS, 'B');
   setprog('B',0, 1,LINKS, 'A');
   setprog('B',1, 0,LINKS, 'C');
   setprog('C',0, 1,RECHTS,'H');
   setprog('C',1, 1,LINKS, 'D');
   setprog('D',0, 1,RECHTS,'D');
   setprog('D',1, 0,RECHTS,'A');
  }
 else if(z==5)
  {
   setprog('A',0, 1,LINKS, 'B'); //vermutlich optimaler Biber
   setprog('A',1, 1,RECHTS,'C');
   setprog('B',0, 1,LINKS, 'C');
   setprog('B',1, 1,LINKS, 'B');
   setprog('C',0, 1,LINKS, 'D');
   setprog('C',1, 0,RECHTS,'E');
   setprog('D',0, 1,RECHTS,'A');
   setprog('D',1, 1,RECHTS,'D');
   setprog('E',0, 1,LINKS, 'H');
   setprog('E',1, 0,RECHTS,'A');
  }
 else if(z==6)
  {
   setprog('A',0, 1,LINKS, 'B');
   setprog('A',1, 1,RECHTS,'C');
   setprog('B',0, 1,LINKS, 'C');
   setprog('B',1, 1,LINKS, 'B');
   setprog('C',0, 1,LINKS, 'D');
   setprog('C',1, 0,RECHTS,'E');
   setprog('D',0, 1,RECHTS,'A');
   setprog('D',1, 1,RECHTS,'D');
   setprog('E',0, 1,LINKS, 'F'); //statt Halt wie beim z==5 hier zum F
   setprog('E',1, 0,RECHTS,'A');
   setprog('F',0, 1,RECHTS,'F'); //1 Schritt mehr ist immer moeglich
   setprog('F',1, 1,RECHTS, 'H'); //hier auch 2.Schritt bei gleich lassen der andern
  }
}

void Biber::setprog(int zust,int zahl,int neuezahl,int richtung,int neuerzustand)
{
 if(zahl>=smax) {zahl=smax-1; fprintf(stderr,"SymbolZahl auf %d beschraenkt (smax-1).\n",smax-1);}
 int i;
 if(zust=='H' || zust==ZMAX) {fprintf(stderr,"Fehler: Halt-Zustand darf nicht definiert werden\n"); return;}
 if(zust>='I') i = (zust-'A'-1)*smax+zahl;
 else if(zust>='A') i = (zust-'A')*smax+zahl;
 else          i = zust*smax+zahl;
 if(i>=ZMAX*smax) {fprintf(stderr,"zust*smax+zahl = i = %d  zu gross\n",i);}
 if(richtung>1) {richtung=1; fprintf(stderr,"richtung darf nur 0 (=links) oder 1 (=rechts) sein.\n");}
 if(neuerzustand=='H') neuerzustand=ZMAX;
 else if(neuerzustand>='I') neuerzustand = neuerzustand-'A'-1;
 else if(neuerzustand>='A') neuerzustand = neuerzustand-'A';
 tabelle[i].set(neuezahl,richtung,neuerzustand);
 //printf(" tabelle[%d].set(%d,%d,%d)\n",i,neuezahl,richtung,neuerzustand);//test
}

void Biber::step()
{
 if(zustand==HALT) return; //bei HALT nichts machen
 int zahl=band.read();
 int i=zustand*smax+zahl;
 int neuezahl = tabelle[i].schreibwert;
 int richtung = tabelle[i].richtung;
 //printf("neuezahl=%d, richtung=%d\n",neuezahl,richtung);//test
 zustand = tabelle[i].neuerzustand;
 band.write(neuezahl,richtung,zustand);
}

void Band::write(uchar neuezahl,uchar richtung,uchar zustand)
{
 feld[pos]=neuezahl;
 lastpos=pos;
 if(richtung==LINKS)
   {
    if(pos==0) {fprintf(stderr,"Band zu kurz, linker Rand erreicht.\n"); errnum=BANDVOLL; errflag=true;}
    else if(--pos < posmin) posmin=pos;
   }
 else
   {
    if(pos==MAXBAND-1) {fprintf(stderr,"Band zu kurz, rechter Rand erreicht.\n"); errnum=BANDVOLL; errflag=true;}
    else if(++pos > posmax) posmax=pos;
   }
#ifdef LOOPTEST
 if(zustand==0 && feld[pos]==0) //wieder im Startzustand
   {
    uint i;
    for(i=posmin;i<=posmax;i++)
      {if(feld[i]!=0) break;}
    if(i>posmax) {errnum=LOOPUNENDLICH; errflag=true;}
   }
#endif
}

uchar symbol_name2nr(char c)
{
 if(c=='0' || c=='1') return c-'0';
 if(c=='_') return 2;
 if(c=='+') return 3;
 return 3;//provi. (nur Symbole 0 1 _ + erlaubt)
}

char symbol_nr2name(uchar u)
{
 if(u<=1) return u+'0';
 if(u==2) return '_';
 if(u==3) return '+';
 return u-2+'a';//provi. (nur Symbole 0 1 _ + erlaubt)
}

void turingmaschine_einlesen(Datei& datei,Biber& biber)
{
 int z=0,smax=2,j;
 char c;
 while(datei.getzeile_ohne_kommentare(';'))
  {
   if(datei.zeile[1]!='=') break;
   if((c=datei.zeile[0])=='Z' || c=='z')
    {
     sscanf(&datei.zeile[2],"%d",&z);
     if(z==0) {fprintf(stderr,"Syntaxfehler: %s\n",datei.zeile); return;}
     if(z>ZMAX) {fprintf(stderr,"z=%d zu gross. Z bisher auf %d beschraenkt\n",z,ZMAX); return;}
    }
   else if(c=='S' || c=='s')
    {
     sscanf(&datei.zeile[2],"%d",&smax);
     if(smax<2 || smax >4) {printf("S bisher auf 2 bis 4 beschraenkt.\n"); if(smax<2) smax=2; else smax=4;}
    }
  }
 if(z==0) {fprintf(stderr,"Fehlendes \"Z=\" in %s\n",datei.name); return;}

 biber.init(z,smax);
#ifdef DEBUG
 printf("Testausdrucke: Turingmaschine von Datei einlesen:\n");//test
 printf(" Z=%d  smax=%d\n",z,smax);//test
#endif
 for(j=0;j<z*smax;j++)
  {
   char zustand,rz,wz,neuerzustand,richtung;
   int rzahl,wzahl;
   char str_zust[20], str_neuzust[20];
   if(j!=0)
    {
     if(!datei.getzeile_ohne_kommentare(';')) {fprintf(stderr,"Zu wenige Daten in Datei\n"); return;}
    }
   //printf("zeile:'%s'\n",datei.zeile);//test
   sscanf(&datei.zeile[0],"%s %c %c %c %s",str_zust,&rz,&wz,&richtung,str_neuzust);
   rzahl = symbol_name2nr(rz);
   wzahl = symbol_name2nr(wz);
   zustand=zustand_einlesen(str_zust);         //Zustaende mussen mit 'A' 'B' ... 'G' oder Z0 ... Z19 bezeichnet sein
   neuerzustand=zustand_einlesen(str_neuzust); //'H' oder Halt oder HALT fuer Haltezustand
   if(zustand>ZMAX || neuerzustand>ZMAX) printf("Syntaxfehler: zustand oder neuerzustand zu gross\n\"%s\"\n",datei.zeile);
   richtung = (richtung=='R') ? RECHTS : LINKS;
   //printf("%s\n",datei.zeile);//test
   //printf("zustand=%d rzahl=%d  wzahl=%d, richtung=%d,neuerzustand=%d\n",zustand,rzahl, wzahl,richtung,neuerzustand);//test
   biber.setprog(zustand,rzahl, wzahl,richtung,neuerzustand);
  }
 if(datei.getzeile_ohne_kommentare(';'))
  {
   if(strncmp("Band=",datei.zeile,5)==0)
    {
     if(datei.zeile[5]=='"')
      { //Voreingestellte Zeichen auf dem Band:
       //printf("Band aus String '%s' setzen\n",&datei.zeile[5]);//test
       uchar u=0;
       for(int i=6;(c=datei.zeile[i])!=0 && c!='"';i++)
	{
	 u = symbol_name2nr(c);
	 band.symbol_setzen(u);
	}
       if(c=='"') band.symbol_setzen(u,true); //restliches Band fuellen
      }
     else
      { //Voreingestellte 0 und 1 auf dem Band:
       //printf("Band aus binaereingabe (veraltet) '%s' setzen\n",&datei.zeile[5]);//test
       int c,i,j=0;
       uint u=0,u2=0;
       for(i=5;(c=datei.zeile[i])=='1' || c=='0';i++)
	{u=(u<<1)+(c-'0'); j++;}
       for(i=0;i<j;i++) {c=u&1; u2=(u2<<1)+c; u>>=1;}
       band.einsen_setzen(u2);
      }
    }//Ende "Band="
  }//Ende getzeile
#ifdef DEBUG
 printf("Ende Testausdrucke: Turingmaschine von Datei einlesen.\n");//test
#endif
}

void print_pos_markiert(int c)
{
 c=symbol_nr2name(c);
#ifdef ESC_MARKIERUNGEN
 printf("\033[33m%c\033[0m",c); //Position gelb markieren ([31=rot, [35=magenta, [33=gelb)
#else
 if(c==0) printf("o");
 else if(c==1) printf("i"); //Position mit o fuer 0 und i fuer 1 markieren
 else printf("<%c>",c);     //Position mit < > markieren
#endif
}

void print_letztes_markiert(int c)
{
 c=symbol_nr2name(c);
#ifdef ESC_MARKIERUNGEN
 printf("\033[1m%c\033[0m",c); //Letztes fett markieren
#else
 if(c==0) printf("O");
 else if(c==1) printf("I");} //statt fett mit O fuer 0 und I fuer 1 markieren
 else printf("(%c)",c);      //statt fett mit ( ) markieren
#endif
}

/************************* Hauptprogramm ******************************/
int main(int argc,char *argv[])
{
 char quellname[80];
 int64_t i;
 int j,c;
 int z=3;
 quellname[0]=0;
 
 for(j=0,i=1;i<argc;i++)
  {if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
   else if(++j==1)
      {if(isdigit(*argv[i])) sscanf(argv[i],"%d",&z);
       else strcpy(quellname,argv[i]);
      }
   else if(j==2) strcpy(quellname,argv[i]);
  }
 
 if(argflag['?'] || j>MAXARG)
	{printf("biber  %s\n",VERSION);
	 printf("Anwendung: biber [Z] [datei.tm]\n");
	 printf("  fuer Z sind bisher nur Zahlen von 2 bis %d moeglich.\n",ZMAX);
	 //printf("sizeof(long)=%d sizeof(int)=%d (int64_t)=%d\n",sizeof(long),sizeof(int),sizeof(int64_t));//test
	 exit(0);
	}

 Datei datei(quellname);
 bool dateiflag=false;
 if(quellname[0]!=0)
  {if(datei.open_read()) dateiflag=true;
   else {fprintf(stderr,"Datei '%s' nicht gefunden.\n",quellname); exit(0);}
  }

 Biber biber;
 if(dateiflag)
  {
   turingmaschine_einlesen(datei,biber);
   printf("Turingmaschine von Datei eingelesen: Z=%d  smax=%d\n",biber.Z, biber.smax);
  }
 else
  {
   if(z<1 || z>6) {printf("keine Voreinstellung mit z=%d eingebaut.\n",z); z=2;}
   biber.init(z);
   biber.prog_init(z);
   printf("Initialisert mit Z=%d\n",biber.Z);
  }
 printf("Zustandstabelle:\n"); //gesamte Zustandstabelle anzeigen
 for(int i=0;i<biber.Z;i++)
   for(j=0;j<biber.smax;j++)
     {
      Eintrag *e= &biber.tabelle[i*biber.smax+j];
      char neuz[8];
      if(e->neuerzustand==HALT) strcpy(neuz,"Halt");
      else {neuz[0] = 'A'+e->neuerzustand; neuz[1]=0;}
      printf("%c %c %c %c %s\n",
	     'A'+i, symbol_nr2name(j), symbol_nr2name(e->schreibwert),
	     (e->richtung==0)?'L':'R', neuz);
     }

#ifdef BAND_DARSTELLEN
 printf("Band Darstellung: (zuletzt geschriebenes fett oder mit O I oder (x) markiert)\n");
 printf("  i Zust Band  (aktuelle Position farbig oder mit o oder i oder <x> markiert)\n");
#endif
 long ioverflow=0;
 for(i=0;;i++)
  {
   if(i<0) {i=0; ioverflow++;}
   if(ioverflow==0 && (i<10000 || (i%100000000)==0))
    {
     if(i>=100000000) printf("i=%Ld Millionen  Zustand=%s  ",i/1000000,zustand_name(biber.zustand));
     else printf("i=%2Ld  %s  ",i,zustand_name(biber.zustand));
#ifdef BAND_DARSTELLEN
     if(i<10000)
      {
       //printf("bandpos:%2d  Band: ",bandpos-BANDMITTE);
       for(uint j=band.posmin;j<=band.posmax;j++)
	{
	 if(j==band.pos) print_pos_markiert(band[j]); //aktuelle Position markieren
	 else if(j==band.lastpos) print_letztes_markiert(band[j]); //letztes fett markieren
	 else printf("%c",symbol_nr2name(band[j]));//Band darstellen
	}
      }
     printf("\n");
#else //BAND_DARSTELLEN
     printf("bandpos = %2d\n",bandpos-BANDMITTE); //ohne Band-Darstellung: nur Position zeigen
#endif
    }
   if(biber.zustand==HALT || errflag) break;
   biber.step();
  }
#ifdef BAND_DARSTELLEN
 //printf("bandpos=%2d  Band: ",bandpos-BANDMITTE);
 for(uint j=band.posmin;j<=band.posmax;j++)
  {
   if(j==band.pos) print_pos_markiert(band[j]); //aktuelle Position markieren
   else if(j==band.lastpos) print_letztes_markiert(band[j]); //letztes fett markieren
   else printf("%c",symbol_nr2name(band[j]));//Band darstellen
  }
 printf("\n");
#endif
 if(ioverflow!=0) printf("Maschine ist %Ld+%ld*2^63 Schritte gelaufen.\n",i,ioverflow);
 else printf("Maschine ist %Ld Schritte gelaufen.\n",i);
 for(i=band.posmin,c=0;i<=band.posmax;i++)
  if(band[i]==1) c++;
 printf("Und hat %d Einsen geschrieben.\n",c);
 printf("Benutzte Bandlaenge: %d\n",band.posmax-band.posmin+1);
 if(errflag)
  {
   if(errnum==LOOPUNENDLICH) printf("Gestoppt wegen Unendlicher Schlaufe.\n");
   else printf("Fehler: Band zu kurz?\n");
  }
 else printf("HALT regulaer erreicht.\n");
 
 return 0;
}/* ende von main */
