/* fonteditor.cc			letzte Aenderung: 13.6.2014 */
#define VERSION "Version 0.02"
/*
Uebersetzen auf Unix (Linux):
> make  ;siehe makefile

 Kurzbeschreibung: Editor um Fonts anzupassen

History:
2.6.2014        Erstellung (RP)
10.6.           Version 0.02


Einteilung in Fonttypen:
Typ  Grund-Aufbau           x-Pixel                      y-Pixel                    Leerbits
 1   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von oben nach unten  rechts
 2   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von oben nach unten  links
 3   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von oben nach unten  rechts
 4   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von oben nach unten  links
 5   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von unten nach oben  rechts
 6   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von unten nach oben  links
 7   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von unten nach oben  rechts
 8   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von unten nach oben  links
 9   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von oben nach unten   unten
10   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von oben nach unten   oben
11   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von unten nach oben   unten
12   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von unten nach oben   oben
13   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von oben nach unten   unten
14   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von oben nach unten   oben
15   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von unten nach oben   unten
16   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von unten nach oben   oben

// Beispiele:
--------------------------------------------------------------------------------
;// font6x8:              ;aus 6x8_horizontal_LSB_1.asm (font6x8.png)
;// Fonttyp 3
;// Daten (je eine Zeile fuer 1 Zeichen)        ; Code Kommentar
;//-------------------------------------------------------------
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x20 ' '
.db 0x70,0x88,0x88,0x88,0xF8,0x88,0x88,0x00,	; 0x41 A
.db 0x08,0x08,0x08,0x08,0x08,0x08,0xF8,0x00,	; 0x4C L
--------------------------------------------------------------------------------
;// Chemiefont 12x16
;// Fonttyp 1
;// Daten (je 4 Zeilen fuer 1 Zeichen)          ; Code Kommentar
;//-------------------------------------------------------------
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	;
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	;
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	;
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x20 ' '
.db 0x3F,0xC0,0x3F,0xC0,0x60,0x60,0x60,0x60,	;
.db 0xC0,0x30,0xC0,0x30,0xC0,0x30,0xC0,0x30,	;
.db 0xFC,0xFC,0xFC,0xFC,0xC0,0x30,0xC0,0x30,	;
.db 0xC0,0x30,0xC0,0x30,0x00,0x00,0x00,0x00,	; 0x41 A
.db 0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,	;
.db 0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,	;
.db 0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,	;
.db 0xFF,0xF0,0xFF,0xF0,0x00,0x00,0x00,0x00,	; 0x4C L
--------------------------------------------------------------------------------

*/

#include <stdio.h>
//#include <iostream>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <xtekplot1.h>

typedef unsigned char uchar;

#define XMAX 1280
#define YMAX 1000
#define TIEFE 24
#define OBERE_LEISTENHOEHE 24  //Hoehe der nicht ausblendbaren Unity-Leiste

//#define SCHWARZAUFWEISS //auskommentieren fuer weiss auf schwarz

/************************* Vordeklarationen ***************************/
void neuzeichnen();
void font_einlesen(FILE *fp);
void zeichne_raster(double boxx,double boxy,double kante);
void zeichne_zeichen(double x0,double y0,double kante,uint zeichen);
void bitumkehren(int i,int j);
void speicher_aufraeumen();
void zeichen_copy(int zeichen);
void zeichen_paste(int zeichen);
void zeichen_vertauschen(int c1,int c2);
void font_convert(int bx,int hy,int typ,int skalierung);
void fonts_vertauschen();
void probe_schreiben(int c);
void probefeld_neuzeichnen();
void probefeld_loeschen();
void hintergrundfarbe();
void vordergrundfarbe();
void berechne_position_aus_zeichen(int zeichen,double *x,double *y);
void zeichen_zurechtschieben(int richtung);

/********************* Globale Variablen ******************************/
static int exitflag=0;
static int savedflag=1;
static char quellname[200+8],zielname[200+8];

// Position und Groesse der Uebersichtstabelle:
const double tabellex=0.05, tabelley=0.31, tabelle_kante=0.64;

// Position und Groesse des Probierbereichs:
const double probex=0.02, probey=0.01, probe_kantex=0.67, probe_kantey=0.26;
const double probe_statusx=probex, probe_statusy=probey+probe_kantey+0.001, probe_status_kantey=0.025;
static int probemodus=0,probe_offset=0;

// Positionen und Groessen der Darstellungs-Boxen fuer das aktuelle Zeichen: 
const double kboxx=0.8, kboxy=0.85, kkante=0.025;
const double mboxx=0.78, mboxy=0.65, mkante=0.1;
const double gboxx=0.74, gboxy=0.05, gkante=0.5;

// Position und Groesse des Copy/Paste-Zeichens:
const double cboxx=1.1, cboxy=0.65, ckante=0.1;

static uchar *aktueller_font=NULL;
static int maxfontbytes=0;
static int font_bx=0, font_hy=0, font_typ=0;
static int default_bx=0, default_hy=0, default_typ=0;
static uchar aktuelles_zeichen=0x30;
static int bytesprozeichen=0;

static uchar *neuer_font=NULL;
static int neu_maxfontbytes=0;
static int neu_bx=0, neu_hy=0, neu_font_typ=0;
static int neu_bytesprozeichen=0;

static uchar *copypaste_puffer=NULL;
const uint copypaste_zeichen=0x100;

// Globale Variablen fuer Mausbewegungen:
static double x0=0,y0=0,x1=0,y1=0,x2=0,y2=0;
static int mb_flag=0;
const double tka=tabelle_kante/16; //totale Kantenlaenge pro Zeichen
const double dk=0.001; //Haelfte des Abstands zwischen den Zeichen
const double mb_dx=tka-0.002;
const double mb_dy=mb_dx;//provi.
static uchar startzeichen,zielzeichen;

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 1
static char argflag[128];
void setargflags(char *s)
{
 int c;
 while((c= *s++)!=0)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
   if(c=='X') default_hy=atoi(s);
   else if(c=='G') default_bx=atoi(s);
   else if(c=='T') default_typ=atoi(s);
  }
}

/**************************** Kleinkram *******************************/
int kleinerer_wert(int a,int b)
{
 if(a<b) return a;
 return b;
}

double groesserer_wert(double a,double b)
{
 if(a>b) return a;
 return b;
}

int mygetline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
	*s++ = c;
 *s=0;
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

int farbnummer(int r,int g,int b)
{
#if(TIEFE>=24)
   return (r<<16)+(g<<8)+b;
#elif(TIEFE==16)
 {while(r<8 && g<8 && b<8 && (r>1 || g>1 || b>1)) {r<<=1; g<<=1; b<<=1;}
  return ((r&0xF8)<<8)+((g&0xF8)<<3)+((b&0xF8)>>2);
 }
#elif(TIEFE==12)
   return ((r&0xF0)<<4)+(g&0xF0)+((b&0xF0)>>4);
#else //elif(TIEFE==8)
 return (r&0xE0)+((g>>3)&0x1C)+((b>>6)&0x03);
#endif
}

void mycolor(int r,int g,int b)
{
 if(TIEFE==24) rgbcolor(r,g,b); else color(farbnummer(r,g,b));
}

#ifdef SCHWARZAUFWEISS
void hintergrundfarbe()
{
 if(TIEFE==24) rgbcolor(0xFF,0xFF,0xFF);
 else  color(1);
}
void vordergrundfarbe()
{
 if(TIEFE==24) rgbcolor(0x00,0x00,0x00);
 else  color(0);
}
#else
void hintergrundfarbe()
{
 if(TIEFE==24) rgbcolor(0x00,0x00,0x00);
 else  color(0);
}
void vordergrundfarbe()
{
 if(TIEFE==24) rgbcolor(0xFF,0xFF,0xFF);
 else  color(1);
}
#endif

void wprintf(double x,double y,const char *cstr,int p1,int p2=0)
{
 char str[200];
 sprintf(str,cstr,p1,p2);
 schrift(x,y,str);
}

uchar *mycalloc(int n)
{
 uchar *p=(uchar*)calloc(n,1);
 if(p==NULL) fprintf(stderr,"Error: Unable to allocate memory!\n");
 return p;
}

/***************************** Klassen ********************************/
const int RE=1,LI=2,OB=3,UN=4; //Pfeilrichtungen

class Pfeil
{
 double x0,y0,x1,y1;
 int richtung;
public:
 Pfeil(double x,double y,double x2,double y2,int ri)
  {x0=x; y0=y; x1=x2; y1=y2; richtung=ri;}
 bool istinnerhalb(double x,double y)
  {return (x>x0 && x<x1 && y>y0 && y<y1);}
 void shift(double dx,double dy=0) {x0+=dx; x1+=dx; y0+=dy; y1+=dy;}
 void draw();
};

void Pfeil::draw()
{
 double dx=x1-x0, dy=y1-y0;
 double mx=x0+dx/2, my=y0+dy/2;
 if(((richtung==RE || richtung==LI) && dy>dx) || // Breite des Pfeils groesser
    ((richtung==OB || richtung==UN) && dx>dy))   // als die Laenge?
  switch(richtung)
  {
  case RE:
    plot(x0,y0,PENUP); plot(mx,y0,PENDOWN); plot(x1,my,PENDOWN);
    plot(mx,y1,PENDOWN); plot(x0,y1,PENDOWN); plot(x0,y0,PENDOWN);
    break;
  case LI:
    plot(x1,y0,PENUP); plot(mx,y0,PENDOWN); plot(x0,my,PENDOWN);
    plot(mx,y1,PENDOWN); plot(x1,y1,PENDOWN); plot(x1,y0,PENDOWN);
    break;
  case OB:
    plot(x0,y0,PENUP); plot(x0,my,PENDOWN); plot(mx,y1,PENDOWN);
    plot(x1,my,PENDOWN); plot(x1,y0,PENDOWN); plot(x0,y0,PENDOWN);
    break;
  case UN:
    plot(x0,y1,PENUP); plot(x0,my,PENDOWN); plot(mx,y0,PENDOWN);
    plot(x1,my,PENDOWN); plot(x1,y1,PENDOWN); plot(x0,y1,PENDOWN);
    break;
  }
 else
  switch(richtung)
  {
  case RE:
    plot(x0,y0,PENUP); plot(x1-dy/2,y0,PENDOWN); plot(x1,my,PENDOWN);
    plot(x1-dy/2,y1,PENDOWN); plot(x0,y1,PENDOWN); plot(x0,y0,PENDOWN);
    break;
  case LI:
    plot(x1,y0,PENUP); plot(x0+dy/2,y0,PENDOWN); plot(x0,my,PENDOWN);
    plot(x0+dy/2,y1,PENDOWN); plot(x1,y1,PENDOWN); plot(x1,y0,PENDOWN);
    break;
  case OB:
    plot(x0,y0,PENUP); plot(x0,y1-dx/2,PENDOWN); plot(mx,y1,PENDOWN);
    plot(x1,y1-dx/2,PENDOWN); plot(x1,y0,PENDOWN); plot(x0,y0,PENDOWN);
    break;
  case UN:
    plot(x0,y1,PENUP); plot(x0,y0+dx/2,PENDOWN); plot(mx,y0,PENDOWN);
    plot(x1,y0+dx/2,PENDOWN); plot(x1,y1,PENDOWN); plot(x0,y1,PENDOWN);
    break;
  }
}

class Box
{
public:
 double x0,y0,x1,y1;
 Box(double x,double y,double x2,double y2)
  {x0=x; y0=y; x1=x2; y1=y2;}
 bool istinnerhalb(double x,double y)
  {return (x>x0 && x<x1 && y>y0 && y<y1);}
 void draw() {drawbox(x0,y0,x1,y1);}
};

double berechne_xfo()
{
 if(font_bx==font_hy) return 0.0;
 return (gkante-gkante*font_bx/font_hy)/2;
}

/***************** Globale Klassen-Variablen **************************/
//const double mboxx=0.78, mboxy=0.65, mkante=0.1;
//const double cboxx=1.1, cboxy=0.65, ckante=0.1;
static Pfeil
  copy_pfeil (mboxx+mkante+0.04, mboxy+mkante/2+0.01, cboxx-0.04, mboxy+mkante-0.01,RE),
  paste_pfeil(mboxx+mkante+0.04, mboxy+0.01,          cboxx-0.04, mboxy+mkante/2-0.01,LI),
#define BR (gkante*4/16) //Breite
#define LA (gkante/16)   //Laenge
#define AB 0.005         //Abstand von Box
//Distanz abhaengig von Verhaeltnis Fonthoehe zu Fontbreite:
#define XFO 0 //berechne_xfo() //geht nicht weil font_bx und font_hy erst spaeter gesetzt.
  shiftr_pfeil(gboxx+gkante+AB-XFO, gboxy+gkante/2-BR/2, gboxx+gkante+AB+LA-XFO, gboxy+gkante/2+BR/2, RE),
  shiftl_pfeil(gboxx-AB-LA+XFO,     gboxy+gkante/2-BR/2, gboxx-AB+XFO,           gboxy+gkante/2+BR/2, LI),
  shifto_pfeil(gboxx+(gkante-BR)/2, gboxy+gkante+AB,     gboxx+(gkante+BR)/2,  gboxy+gkante+AB+LA,  OB),
  shiftu_pfeil(gboxx+(gkante-BR)/2, gboxy-AB-LA,         gboxx+(gkante+BR)/2,  gboxy-AB,            UN);

static Box probe_loeschbox(probex+probe_kantex-0.05, probey+probe_kantey,
			   probex+probe_kantex,      probey+probe_kantey+0.02);

/************************* Menu Behandlung ****************************/
void menu_save()
{
 FILE *fp;
 int err;
 uchar *pfont=aktueller_font;
 if(pfont==NULL)
   {fprintf(stderr,"Error: kein Font geladen.\n"); return;}
 if(zielname[0]==0)
  {
   sprintf(zielname,"%.200s.bak",quellname);
   err=rename(quellname,zielname);
   if(err)
    {fprintf(stderr,"kann nicht auf \"%s\" umbenennen. (errno=%d)\n",zielname,errno); zielname[0]=0; return;}
   strcpy(zielname,quellname);
  }
 if((fp=fopen(zielname,"w"))==NULL)
   {fprintf(stderr,"kann Datei \"%s\" nicht speichern.\n",zielname); return;}
 int i,j,zeichen;
 //fprintf(fp,"Test Fonteditor %s\n",VERSION);
 fprintf(fp,";// font %dx%d\n",font_bx,font_hy);
 fprintf(fp,";// Fonttyp %d\n",font_typ);
 fprintf(fp,";//-------------------------------------------------------------\n");
 if(bytesprozeichen<=8)
  fprintf(fp,";// Daten (je 1 Zeile fuer jedes Zeichen)       ; Code Kommentar\n");
 else
  fprintf(fp,";// Daten (je %d Zeilen fuer 1 Zeichen)         ; Code Kommentar\n",
	 (bytesprozeichen+7)/8);
 fprintf(fp,";//-------------------------------------------------------------\n");
 for(zeichen=0;zeichen<256;zeichen++)
  {
   for(j=bytesprozeichen;j>8;j-=8)
    {
     fprintf(fp,".db ");
     for(i=0;i<8;i++) fprintf(fp,"0x%02X,",*pfont++);
     fprintf(fp,"    ;\n");
    }
   fprintf(fp,".db ");
   for(i=0;i<8;i++)
    {if(i<j) fprintf(fp,"0x%02X,",*pfont++);
     else    fprintf(fp,"       ");
    }
   fprintf(fp,"    ; 0x%02X",zeichen);
   if(isprint(zeichen)) fprintf(fp," '%c'\n",zeichen);
   else fprintf(fp,"\n");
  }
 fclose(fp);
 savedflag=1;
}

void menu_save_as()
{
 if(zielname[0]==0) strcpy(zielname,"unnamed.i");
 int ok=nachfilenamefragen("Save Font",zielname,200);
 if(ok)
  {
   if(zielname[0]==0)
    {fprintf(stderr,"keine Datei angegeben - nichts gespeichert.\n"); return;}
   menu_save();
  }
}

void drawline(double x1,double y1,double x2,double y2)
{
 plot(x1,y1,PENUP); plot(x2,y2,PENDOWN); // eine Linie zeichnen
}

void menu_refresh()
{
 double x,y,h;
 inital_new();
#ifdef SCHWARZAUFWEISS
 //fuer schwarz auf weiss (default ist weiss auf schwarz):
 if(TIEFE==24)
  {screenclear(0xFFFFFF); rgbcolor(0x00,0x00,0x00);}
 else
  {screenclear(1); color(0);}
#else
 screenclear(0);
#endif
 //drawbox(0.0, 0.0, 1.28, 1.0); //test Rahmen um alles
 // Rahmen um die Uebersichtstabelle:
 drawbox(tabellex,tabelley,tabellex+tabelle_kante,tabelley+tabelle_kante);
 // Rahmen um den Probierbereich:
 drawbox(probex,probey,probex+probe_kantex,probey+probe_kantey);
 drawbox(probe_statusx,probe_statusy,probe_statusx+probe_kantex,probe_statusy+probe_status_kantey);

 drawbox(kboxx,kboxy,kboxx+kkante,kboxy+kkante); // kleine Box fuer das aktuelle Zeichen
 drawbox(mboxx,mboxy,mboxx+mkante,mboxy+mkante); // mittlere Box fuer das aktuelle Zeichen
 drawbox(gboxx,gboxy,gboxx+gkante,gboxy+gkante); // grosse Box fuer das aktuelle Zeichen
 zeichne_raster(gboxx,gboxy,gkante);
 drawbox(cboxx,cboxy,cboxx+ckante,cboxy+ckante); // mittlere Box fuer das Copy/Paste-Zeichen
 copy_pfeil.draw();
 paste_pfeil.draw();
 shiftr_pfeil.draw();
 shiftl_pfeil.draw();
 shifto_pfeil.draw();
 shiftu_pfeil.draw();

 h=textsize(kkante*0.75,kkante,NULL,0); //breite, hoehe, font, style
 //printf("Texthoehe des Codes vom aktuellen Zeichen: h=%lf\n",h);//test

 wprintf(kboxx-kkante/2,kboxy+2*kkante,"0x%02X",aktuelles_zeichen);
 //Aktuelles Zeichen in verschiedenen Groessen darstellen:
 zeichne_zeichen(kboxx,kboxy,kkante,aktuelles_zeichen);
 zeichne_zeichen(mboxx,mboxy,mkante,aktuelles_zeichen);
 zeichne_zeichen(gboxx,gboxy,gkante,aktuelles_zeichen);
 //Copy/Paste Zeichen darstellen:
 if(copypaste_puffer!=NULL)
   zeichne_zeichen(cboxx,cboxy,ckante,copypaste_zeichen);

 // Uebersichtstabelle zeichnen:
 double ka=tka-0.002; //Kantenlaenge der Zeichen in der Uebersicht
 int zeichen,i,j;
 for(zeichen=0,j=0,y=tabelley+tabelle_kante-tka+0.001; j<16; j++,y-=tka)
  {
   for(i=0,x=tabellex+0.001; i<16; i++,x+=tka)
    {
     zeichne_zeichen(x,y,ka,zeichen);
     zeichen++;
    }
  }
 // Beschriftung der Uebersichtstabelle:
 h=textsize(0.015,0.02,NULL,0); //breite, hoehe, font, style
 //printf("Schrifthoehe Beschriftung Uebersichtstabelle: h=%f\n",h);//test
 for(i=0,x=tabellex+(tka-h*3/4)/2,y=tabelley+tabelle_kante+0.005; i<16; i++,x+=tka)
  {
   wprintf(x,y,"%X",i);
  }
 for(i=0,x=tabellex-2*h,y=tabelley+tabelle_kante-tka+(tka-h)/2; i<16; i++,y-=tka)
  {
   wprintf(x,y,"%X0",i);
  }
 // Einrahmung des aktuellen Zeichens in der Uebersichtstabelle:
 berechne_position_aus_zeichen(aktuelles_zeichen,&x,&y);
 mycolor(0,255,0); //gruener Rahmen um aktuelles Zeichen zeichnen
 drawbox(x,y,x+mb_dx,y+mb_dy);
 vordergrundfarbe();

 //Probierbereich zeichnen:
 probefeld_neuzeichnen();

 term_refresh();
}

void neuzeichnen()
{
 menu_refresh(); //provi. es muesste nicht alles neu gezeichnet werden
}

void menu_exit()
{
 if(savedflag==0)
  {
   int ok=janeinrequester("Aktueller Font noch nicht gespeichert. Speichern?",
			  " Ja ","Cancel");
   if(ok)   menu_save();
   else
    {
     ok=janeinrequester("Ungespeicherte Daten verwerfen?"," Ja ","Nein");
     if(!ok) return;
    }
  }
 exitflag=1;
}

inline bool ist_innerhalb(double x,double y,double x1,double y1,double x2,double y2)
{
 return (x>x1 && x<x2 && y>y1 && y<y2);
}

void mb_abrunden(double &x,double &y)
{
 x = int((x-tabellex)/tka)*tka+tabellex+dk;
 y = int((y-tabelley)/tka)*tka+tabelley+dk;
}

int berechne_zeichen_aus_position(double x,double y)
{
 int n1 = int((x-tabellex)/tka);
 int n2 = int((tabelley+tabelle_kante-mb_dy-y)/tka);
 if(n1>15 || n1<0 || n2>15 || n2<0) return (-1);
 return (n2<<4)+n1;
}

void berechne_position_aus_zeichen(int zeichen,double *x,double *y)
{
 int n1=zeichen&0x0F, n2=(zeichen>>4)&0x0F;
 *x = tabellex+n1*tka+dk;
 *y = tabelley+(15-n2)*tka+dk;
}

void maus_gedrueckt()
{
 double x,y,dy,rand;
 int taste=mausposition(&x,&y);
 //printf("maus_gedrueckt() taste=%d x=%lf y=%lf\n",taste,x,y);//test
 dy=gkante/font_hy;
 rand=(font_hy-font_bx)*dy/2;
 if(probemodus==0 && ist_innerhalb(x,y, gboxx+rand, gboxy, gboxx+gkante-rand, gboxy+gkante))
  {
   if(taste&LIMAUS)
    {
     int i,j;
     //printf("linke Maustaste in grosser Box gedrueckt\n");//test
     i = int((x-gboxx-rand)/dy);
     j = int((gboxy+gkante-y)/dy);
     bitumkehren(i,j);
     menu_refresh();
    }
   else
    {
     printf("Maus innerhalb grosser Box gedrueckt. taste=%d\n",taste);//test
     if(taste&MIMAUS) printf("mittlere Maustaste gedrueckt\n");//test
     if(taste&REMAUS) printf("rechte Maustaste gedrueckt\n");//test
    }
  }
 else if(copy_pfeil.istinnerhalb(x,y))
  {
   zeichen_copy(aktuelles_zeichen);
   neuzeichnen();
   //printf("Zeichen kopiert.\n");//test
   probemodus=0;
  }
 else if(paste_pfeil.istinnerhalb(x,y))
  {
   zeichen_paste(aktuelles_zeichen);
   neuzeichnen();
   //printf("Zeichen zurueck kopiert.\n");//test
   probemodus=0;
  }
 else if(ist_innerhalb(x,y, tabellex, tabelley, tabellex+tabelle_kante, tabelley+tabelle_kante))
  {
   if(mb_flag==2) // Flag fuer Vertauschung von 2 Zeichen gesetzt?
    {
     mb_abrunden(x,y);
     int c=berechne_zeichen_aus_position(x,y);
     if(c==zielzeichen || c==startzeichen)
      {
       //printf("Testpunkt: Zeichen 0x%02X mit 0x%02X vertauschen\n",startzeichen,zielzeichen);//test
       zeichen_vertauschen(startzeichen,zielzeichen);
       menu_refresh();
      }
     else
      {
       hintergrundfarbe(); //Rahmen wieder loeschen
       drawbox(x0,y0,x0+mb_dx,y0+mb_dy);
       drawbox(x2,y2,x2+mb_dx,y2+mb_dy);
       vordergrundfarbe();
      }
     x0=x2=0; mb_flag=0;
     return;
    }
   else if(probemodus==0)
    {
     int zeichen;
     zeichen = int((x-tabellex)/tka) + 16*(15-int((y-tabelley)/tka));
     //printf("Angeklicktes Zeichen: 0x%0X\n",zeichen);//test
     if(zeichen>=0 && zeichen<256)
      {
       aktuelles_zeichen=zeichen;
       neuzeichnen();
      }
     else fprintf(stderr,"Error in maus_gedrueckt()\n");//test
    }
   else //if(probemodus==1)
    {
     int zeichen = int((x-tabellex)/tka) + 16*(15-int((y-tabelley)/tka));
     probe_schreiben(zeichen);
    }
  }
 else if(ist_innerhalb(x,y, probex, probey, probex+probe_kantex, probey+probe_kantey))
  {
   probemodus ^= 1; //Probefeild ein/aus-schalten
   //if(probemodus==0) probefeld_loeschen();//provi.
   probefeld_neuzeichnen();
  }
 else if(shiftr_pfeil.istinnerhalb(x,y))
  {
   zeichen_zurechtschieben(RE);
   neuzeichnen();
   probemodus=0;
  }
 else if(shiftl_pfeil.istinnerhalb(x,y))
  {
   zeichen_zurechtschieben(LI);
   neuzeichnen();
   probemodus=0;
  }
 else if(shifto_pfeil.istinnerhalb(x,y))
  {
   zeichen_zurechtschieben(OB);
   neuzeichnen();
   probemodus=0;
  }
 else if(shiftu_pfeil.istinnerhalb(x,y))
  {
   zeichen_zurechtschieben(UN);
   neuzeichnen();
   probemodus=0;
  }
 else if(probe_loeschbox.istinnerhalb(x,y) && probemodus!=0)
  {
   probefeld_loeschen();
   neuzeichnen();
  }
}

void maus_bewegung()
{
 int taste; double x,y;
 taste=mausposition(&x,&y);
 //printf("maus_bewegung() tasten=%d x=%f y=%f\n",taste,x,y); //test
 if(ist_innerhalb(x,y, tabellex, tabelley, tabellex+tabelle_kante, tabelley+tabelle_kante))
  {
   mb_abrunden(x,y);
   if(taste&LIMAUS)
    {
     if(mb_flag==0)
      {
       mycolor(255,0,0); //roter Rahmen um Start-Zeichen zeichnen
       drawbox(x,y,x+mb_dx,y+mb_dy);
       vordergrundfarbe();
       startzeichen=berechne_zeichen_aus_position(x,y);
       x0=x; y0=y; mb_flag=1;
       return;
      }
     else if(mb_flag==1)
      {
       drawmode(COMPLEMENT);
       if(x1!=0) drawbox(x1,y1,x1+mb_dx,y1+mb_dy);
       drawbox(x,y,x+mb_dx,y+mb_dy);
       drawmode(JAM1);
       x1=x; y1=y;
      }
     else printf("Fehler: mb_flag=%d\n",mb_flag);//test
    }
   else if(taste&REMAUS) //test
    {
    }
   else if(taste&MIMAUS) //test
    {
    }
  }
}

void maus_losgelassen()
{
 double x,y;
 int c;
 //int taste=
 mausposition(&x,&y);
 if(mb_flag!=0)
  {
   if(mb_flag==2) {printf("Fehler in maus_losgelassen() mb_flag=%d\n",mb_flag); return;}//test
   mb_abrunden(x,y);
   if(x1!=0)
    {
     drawmode(COMPLEMENT);
     drawbox(x1,y1,x1+mb_dx,y1+mb_dy);
     drawmode(JAM1);
     x1=0;
    }
   zielzeichen=c=berechne_zeichen_aus_position(x,y);
   if(c==(-1) || zielzeichen==startzeichen)
    {
     hintergrundfarbe(); //Rahmen wieder loeschen
     drawbox(x0,y0,x0+mb_dx,y0+mb_dy);
     x0=x2=0; mb_flag=0;
    }
   else
    {
     mycolor(0,255,255); //blaugruener Rahmen um Ziel-Zeichen zeichnen
     drawbox(x,y,x+mb_dx,y+mb_dy);
     x2=x; y2=y; mb_flag=2;
    }
   vordergrundfarbe();
   x1=0;
  }
}

void menu_convert()
{
 int neu_bx=font_bx, neu_hy=font_hy, neu_typ=font_typ;
 char skal[80]="auto"; int skalierung=1;
 int ok=requester_input(4,
			"Font-Breite","%d","%d",&neu_bx,
			"Font-Hoehe", "%d","%d",&neu_hy,
			"Font-Typ","%d","%d\n",&neu_typ,
			"Skalierung (ja, nein, auto)","%s","%s\n",skal);
 if(ok)
  {
   //printf("nach ok: neu_bx=%d neu_hy=%d neu_typ=%d\n",neu_bx,neu_hy,neu_typ);//test
   if(*skal=='j' || *skal=='J' || *skal=='y' || *skal=='Y') skalierung=2;
   else if(*skal=='n' || *skal=='N') skalierung=0;
   font_convert(neu_bx,neu_hy,neu_typ,skalierung);
   fonts_vertauschen();
   menu_refresh();
  }
}

void menu_undo()
{
 if(neuer_font!=NULL) fonts_vertauschen();
 menu_refresh();
}

/************************* Hauptprogramm ******************************/
void hilfetext()
{
 printf("Einteilung in Fonttypen:\n\
Typ Grund-Aufbau  x-Pixel                      y-Pixel                    Leerbits\n\
 1  Zeilenweise   Bits von links nach rechts   Bytes von oben nach unten  rechts\n\
 2  Zeilenweise   Bits von links nach rechts   Bytes von oben nach unten  links\n\
 3  Zeilenweise   Bits von rechts nach links   Bytes von oben nach unten  rechts\n\
 4  Zeilenweise   Bits von rechts nach links   Bytes von oben nach unten  links\n\
 5  Zeilenweise   Bits von links nach rechts   Bytes von unten nach oben  rechts\n\
 6  Zeilenweise   Bits von links nach rechts   Bytes von unten nach oben  links\n\
 7  Zeilenweise   Bits von rechts nach links   Bytes von unten nach oben  rechts\n\
 8  Zeilenweise   Bits von rechts nach links   Bytes von unten nach oben  links\n\
 9  Spaltenweise  Bytes von links nach rechts  Bits von oben nach unten   unten\n\
10  Spaltenweise  Bytes von links nach rechts  Bits von oben nach unten   oben\n\
11  Spaltenweise  Bytes von links nach rechts  Bits von unten nach oben   unten\n\
12  Spaltenweise  Bytes von links nach rechts  Bits von unten nach oben   oben\n\
13  Spaltenweise  Bytes von rechts nach links  Bits von oben nach unten   unten\n\
14  Spaltenweise  Bytes von rechts nach links  Bits von oben nach unten   oben\n\
15  Spaltenweise  Bytes von rechts nach links  Bits von unten nach oben   unten\n\
16  Spaltenweise  Bytes von rechts nach links  Bits von unten nach oben   oben\n\
");
}

int main(int argc,char *argv[])
{
 quellname[0]=zielname[0]=0;
 FILE *fp1;
 int i,j,c;
 int breite,hoehe,tiefe,visklasse;
 //double xmin= -0.125,ymin=0.,xmax=1.125,ymax=1.;
 double xmin=0, ymin=0, xmax=XMAX/(double)YMAX, ymax=1.0;
 char titel[80];
 if(argc<=0)
   ;/* es wurde von WorkBench gestartet */
 else
   /* es wurde von der Shell gestartet */
   for(j=0,i=1;i<argc;i++)
	{if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
	 else	{if(++j==1) strcpy(quellname,argv[i]);
	         //else if(j==2) strcpy(zielname,argv[i]);
	}	}
 if(argflag['?'] || argflag['H'] || j>MAXARG || quellname[0]==0
    || (argflag['X'] && !argflag['G']) || (argflag['G'] && !argflag['X']))
	{printf("fonteditor  %s\n",VERSION);
	 printf("Anwendung: fonteditor [-Flags] Fontdatei\n");
	 printf(" Flags: t = Fonttyp (zB. -t12 fuer Fonttyp 12)\n");
	 printf("        g = Fontgrosse (zB. -g12x16)\n");
	 printf("        mit Flags t, g werden Eintraege in der Datei ignoriert.\n");
	 printf("        h = Hilfe\n");
	 printf("Beispielaufruf:\n");
	 printf(" fonteditor -g6x8 -t3 testfont.i\n");
	 if(argflag['H']) {printf("\n"); hilfetext();}
	 exit(0);
	}
 if((fp1=fopen(quellname,"r"))==NULL)
   {fprintf(stderr,"kann \"%s\" nicht oeffnen.\n",quellname); return 1;}
 //tek_setdebug(1);//test
 getmaxsize(&breite,&hoehe,&tiefe,&visklasse);
#ifdef OBERE_LEISTENHOEHE
 hoehe -= OBERE_LEISTENHOEHE;
#endif
 if(tiefe>TIEFE) tiefe=TIEFE;
 if(breite>XMAX) breite=XMAX;
 if(hoehe>YMAX) hoehe=YMAX;
 //maxcol=(1<<TIEFE);
 setsize(breite,hoehe,tiefe);
 setmenu(3,"File",       "Edit",            "Draw");
 setmenu(3,"Save as ...","Undo ",           "Refresh",&menu_save_as,&menu_undo,&menu_refresh);
 setmenu(2,"Save",       "Convert Font ...",&menu_save,&menu_convert);
 setmenu(1,"Exit (q)",&menu_exit);
 //set_funktions(maus_gedrueckt,maus_losgelassen,fenster_verdeckung,maus_bewegung);
 set_funktions(maus_gedrueckt,maus_losgelassen,NULL,maus_bewegung);
 inital(xmin,ymin,xmax,ymax); /* Grafikfenster oeffnen */
 sprintf(titel,"Fonteditor %s",VERSION);
 set_tektitel(titel);

#ifdef SCHWARZAUFWEISS
 //fuer schwarz auf weiss (default ist weiss auf schwarz):
 if(TIEFE==24)
  {screenclear(0xFFFFFF); rgbcolor(0x00,0x00,0x00);}
 else
  {screenclear(1); color(0);}
#endif

 font_einlesen(fp1);
 fclose(fp1);
 menu_refresh();

 while(exitflag==0 && waitmenu(0)==0)
  {
   int in,asci; ULONG rawcode;
   waitTOF();	//auf Beginn des Bildaufbaus warten
   //oder waitmenu(0);	//nicht warten
   if(keyget(&in,&asci,&rawcode) != 0)
    {
     if(probemodus!=0)
      {
       if(in!=0)
	{
	 probe_schreiben((asci+probe_offset)&0xFF);
	}
       else switch(rawcode)
	     {
	     case 0xFF52: //UP-Taste
	       if((probe_offset-=0x40)<0) probe_offset=0;
	       probefeld_neuzeichnen();
	       break;
	     case 0xFF54: //DOWN-Taste
	       if((probe_offset+=0x40)>0xC0) probe_offset=0xC0;
	       probefeld_neuzeichnen();
	       break;
	     default:
	       printf("rawcode=0x%04lX\n",(long)rawcode);
	     }
      }
     else if(in!=0 && (asci=='q' || asci=='Q'))
      {
       if(savedflag==1) menu_exit();
      }
     else switch(rawcode)
	     {
	     case 0xFF52: //UP-Taste
	       if(aktuelles_zeichen>0) --aktuelles_zeichen;
	       menu_refresh();
	       break;
	     case 0xFF54: //DOWN-Taste
	       if(aktuelles_zeichen<255) ++aktuelles_zeichen;
	       menu_refresh();
	       break;
 	     default:
	       if(in) printf("asci = %02X = '%c' ",asci,asci);
	       printf("rawcode=0x%04lX\n",(long)rawcode);
	     }
    }
  }

 term_exit();
 speicher_aufraeumen();
 return 0;
}// ende von main

void speicher_aufraeumen() //mit calloc reservierter Speicher wieder freigeben
{
 if(aktueller_font!=NULL) free(aktueller_font);
 if(copypaste_puffer!=NULL) free(copypaste_puffer);
 if(neuer_font!=NULL) free(neuer_font);
}

bool istkommentar(const char *zeile)
{
 if(*zeile==0 || *zeile==';') return true;
 if(zeile[0]=='/' && (zeile[1]=='/' || zeile[1]=='*')) return true;
 return false;
}

bool ermittle_fontgroesse(const char *zeile,int* breite,int* hoehe)
{
 int br,ho;
 while(*zeile==';') zeile++;
 while(*zeile=='/') zeile++;
 do
  {
   if(*zeile!=0 && isdigit(zeile[1]) && zeile[2]=='x' && isdigit(zeile[3]))
    {
     ho = atoi(&zeile[3]);
     if(!isdigit(zeile[0])) zeile++;
     br = atoi(zeile);
     if(*hoehe==0) *hoehe=ho;
     if(*breite==0) *breite=br;
     return true;
    }
   else
    {
     if(*zeile!=0) zeile++;
    }
  } while(!istkommentar(zeile));
 return false;
}

bool ermittle_fonttyp(const char *zeile,int* typ)
{
 //z.B.: ";; Fonttyp 5" oder "fonttyp=5"
 while(*zeile!=0)
  {
   if((zeile[0]=='F' || zeile[0]=='f') && strncmp(&zeile[1],"onttyp",6)==0)
    {
     zeile += 7;
     while(isblank(*zeile) || *zeile=='=') zeile++;
     *typ=atoi(zeile);
     if(*typ>=1 && *typ<=16) return true;
     fprintf(stderr,"unbekannter Fonttyp %d  '%s'\n",*typ,zeile); *typ=0;
     return false;
    }
   else
    {
     zeile++;
    }
  }
 return false;
}

int hexzahl(int c)
{
 if(c<='9') return c-'0';
 if(c<='F') return c-'A'+10;
 return c-'a'+10;
}

uchar* daten_einlesen(const char *zeile,uchar *fontzeiger,int& limit)
{
 uchar c1,c2;
 if(fontzeiger==NULL)
  {
   if(font_bx==0 || font_hy==0) {limit=(-1); return NULL;}
   maxfontbytes = 256 * ((font_typ<=8) ? (font_bx+7)/8*font_hy : (font_hy+7)/8*font_bx);
   fontzeiger = (uchar*)calloc(maxfontbytes, 1);
   if(fontzeiger==NULL)
    {fprintf(stderr,"Error: Unable to allocate memory!\n"); limit=(-1); return NULL;}
   aktueller_font=fontzeiger;
   limit=maxfontbytes;
  }
 if(istkommentar(zeile))
  {
   //Kommentare ueberlesen
  }
 else
  {
   while(isblank(*zeile)) zeile++;
   while(zeile[0]!='0' || zeile[1]!='x') //suche erstes 0x
    {
     if(istkommentar(zeile)) return fontzeiger;
     zeile++;
    }
   do
    {
     zeile++; zeile++;
     if(!isxdigit(zeile[0]) || !isxdigit(zeile[1]))
      {fprintf(stderr,"Syntaxfehler: %s\n",zeile); break;}
     if(--limit<0)
      {fprintf(stderr,"Fonttabelle zu lang: %s\n",zeile); break;}
     c1= *zeile++;
     c2= *zeile++;
     *fontzeiger++ = (hexzahl(c1)<<4)+hexzahl(c2);
     while(isblank(*zeile) || *zeile==',') zeile++;
    }
   while(!istkommentar(zeile));
  }
 return fontzeiger;
}

/* Beispiele:
--------------------------------------------------------------------------------
;// font6x8:              ;aus 6x8_horizontal_LSB_1.asm (font6x8.png)
;// Fonttyp 5
;// Daten (je eine Zeile fuer 1 Zeichen)        ; Code Kommentar
;//-------------------------------------------------------------
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x20 ' '
.db 0x70,0x88,0x88,0x88,0xF8,0x88,0x88,0x00,	; 0x41 A
.db 0x08,0x08,0x08,0x08,0x08,0x08,0xF8,0x00,	; 0x4C L
--------------------------------------------------------------------------------
*/

void font_einlesen(FILE *fp)
{
 char zeile[200];
 int status=0,limit=0;
 //uchar *fontzeiger=aktueller_font;
 //for(int i=0;i<MAXFONTBYTES;i++) aktueller_font[i]=0;
 uchar *fontzeiger=NULL;
 //printf("font_einlesen(FILE *fp)\n");//test
 if(default_typ!=0) {font_typ=default_typ; status=1;}
 if(default_bx!=0) font_bx=default_bx;
 if(default_hy!=0) font_hy=default_hy;
 if(font_bx!=0 && font_hy!=0) status++;
 while(mygetline(fp,zeile,200))
  {
   if(status<=1 && strncmp(zeile,".db",3)!=0)
    {
     if((font_bx==0 || font_hy==0)
	&& ermittle_fontgroesse(zeile,&font_bx,&font_hy)) status++;
     if(font_typ==0 && ermittle_fonttyp(zeile,&font_typ)) status++;
    }
   else
    {
     if(status<=1) {fprintf(stderr,"Warning: missing fonttyp or fontsize\n"); status=2;}
     fontzeiger=daten_einlesen(zeile,fontzeiger,limit); //bei 1.Aufruf wird fontzeiger u. limit gesetzt
     if(limit<0) break;
    }
  }
 if(font_bx==0 || font_hy==0) fprintf(stderr,"konnte Fontgroesse nicht ermitteln\n");
 else
  {
   int anzahl=(int)(fontzeiger-aktueller_font);
   if(font_typ==0)
    {
     fprintf(stderr,"konnte den Fonttyp nicht ermitteln"); font_typ=3;
     fprintf(stderr," - auf %d gesetzt\n",font_typ=3);
    }
   fprintf(stderr,"Font: Typ=%d, Groesse=%dx%d,  %d Bytes eingelesen",font_typ,font_bx,font_hy,anzahl);
   bytesprozeichen = (font_typ<=8) ? (font_bx+7)/8*font_hy : (font_hy+7)/8*font_bx;
   anzahl /= bytesprozeichen;
   fprintf(stderr," = %d Zeichen\n",anzahl);
   //printf("Testpunkt: font_bx=%d font_hy=%d\n",font_bx,font_hy);//test
   shiftr_pfeil.shift(-berechne_xfo()); //Rechts-Pfeil auf schoenere Position setzen
   shiftl_pfeil.shift(berechne_xfo()); //Links-Pfeil auf schoenere Position setzen
  }
}

void zeichne_raster(double boxx,double boxy,double kante)
{
 int i;
 double x,y;
 double dy=kante/font_hy;
 double x1=boxx+(font_hy-font_bx)*dy/2;
 double x2=boxx+kante-(font_hy-font_bx)*dy/2;
 for(x=x1,i=0; i<=font_bx; i++,x+=dy)
   drawline(x,boxy,x,boxy+kante);
 for(y=boxy+dy,i=1;i<font_hy;i++,y+=dy)
   drawline(x1,y,x2,y);
}

uchar *bitlesen(uchar *pfont,int i,int j,int *maske,int bx,int hy,int typ)
{
 // i kann Werte von 0 bis bx-1 haben, 0 ist Spalte ganz links
 // j kann Werte von 0 bis hy-1 haben, 0 ist oberste Zeile
 // bx ist Fontbreite, hy ist Fonthoehe, typ ist Fonttyp
 uchar leerbits;
 uchar *p1;
 int bytesprozeile,bytesprospalte,nbits;
 int bx1=bx-1, hy1=hy-1;
 //printf("bitlesen(pfont,i=%d,j=%d,bx=%d,hy=%d,typ=%d)\n",i,j,bx,hy,typ);//test
 if(typ<=8) {bytesprozeile = (bx+7)/8;  leerbits = 8-bx%8;}
 else       {bytesprospalte = (hy+7)/8; leerbits = 8-hy%8;}
 if(leerbits==8) leerbits=0;
 if(typ<1 || typ>16)
  {
   fprintf(stderr,"Error: in bitlesen() unknown Fonttyp %d - defaulting to Typ 3\n",typ);
   typ=3;
  }
 switch(typ)
  {
   //       Typ  Grund-Aufbau           x-Pixel                      y-Pixel                    Leerbits
   case 1:// 1   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von oben nach unten  rechts
     p1 = &pfont[bytesprozeile*j+i/8];
     nbits = 7-i%8;
     break;
   case 2:// 2   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von oben nach unten  links
     p1 = &pfont[bytesprozeile*j+(i+leerbits)/8]; //noch testen
     nbits = 7-(i+leerbits)%8; //noch testen
     break;
   case 3:// 3   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von oben nach unten  rechts
     p1 = &pfont[bytesprozeile*j+(bx1-i+leerbits)/8];
     nbits = (i+leerbits)%8;
     break;
   case 4:// 4   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von oben nach unten  links
     p1 = &pfont[bytesprozeile*j+(bx1-i)/8];
     nbits = i%8;
     break;
   case 5:// 5   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von unten nach oben  rechts
     p1 = &pfont[bytesprozeile*(hy1-j)+i/8]; //noch testen
     nbits = 7-i%8; //noch testen
     break;
   case 6:// 6   Zeilenweiser Aufbau    Bits von links nach rechts   Bytes von unten nach oben  links
     p1 = &pfont[bytesprozeile*(hy1-j)+(i+leerbits)/8]; //noch testen
     nbits = 7-(i+leerbits)%8; //noch testen
     break;
   case 7:// 7   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von unten nach oben  rechts
     p1 = &pfont[bytesprozeile*(hy1-j)+(bx1-i+leerbits)/8]; //noch testen
     nbits = (i+leerbits)%8; //noch testen
     break;
   case 8:// 8   Zeilenweiser Aufbau    Bits von rechts nach links   Bytes von unten nach oben  links
     p1 = &pfont[bytesprozeile*(hy1-j)+(bx1-i)/8]; //noch testen
     nbits = i%8; //noch testen
     break;
   //       Typ  Grund-Aufbau           x-Pixel                      y-Pixel                    Leerbits
  case  9:// 9   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von oben nach unten   unten
     p1 = &pfont[bytesprospalte*i+j/8]; //noch testen
     nbits = j%8+leerbits; //noch testen
     break;
  case 10://10   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von oben nach unten   oben
     p1 = &pfont[bytesprospalte*i+(j+leerbits)/8]; //noch testen
     nbits = 7-(j+leerbits)%8; //noch testen
     break;
  case 11://11   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von unten nach oben   unten
     p1 = &pfont[bytesprospalte*i+(hy1-j+leerbits)/8]; //noch testen
     nbits = (j+leerbits)%8; //noch testen
     break;
  case 12://12   Spaltenweiser Aufbau   Bytes von links nach rechts  Bits von unten nach oben   oben
     p1 = &pfont[bytesprospalte*i+(hy1-j)/8]; //noch testen
     nbits = j%8+leerbits; //noch testen
     break;
  case 13://13   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von oben nach unten   unten
     p1 = &pfont[bytesprospalte*(bx1-i)+j/8]; //noch testen
     nbits = 7-j%8; //noch testen
     break;
  case 14://14   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von oben nach unten   oben
     p1 = &pfont[bytesprospalte*(bx1-i)+(j+leerbits)/8]; //noch testen
     nbits = 7-(j+leerbits)%8; //noch testen
     break;
  case 15://15   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von unten nach oben   unten
     p1 = &pfont[bytesprospalte*(bx1-i)+(hy1-j+leerbits)/8]; //noch testen
     nbits = (j+leerbits)%8; //noch testen
     break;
  case 16://16   Spaltenweiser Aufbau   Bytes von rechts nach links  Bits von unten nach oben   oben
     p1 = &pfont[bytesprospalte*(bx1-i)+(hy1-j)/8]; //noch testen
     nbits = j%8; //noch testen
     break;
  }
 *maske = (1<<nbits);
 return p1;
}

bool bitgesetzt(uchar *pfont,int i,int j)  //i=Zaehler in x-Richtung, j=y-Richtung
{
 int maske,c;
 uchar *p;
 p = bitlesen(pfont,i,j,&maske,font_bx,font_hy,font_typ);
 c = *p;
 //printf("Testpunkt bitgesetzt(): c=0x%02X maske=0x%02X\n",c,maske);//test
 return (c&maske)!=0;
}

void zeichne_zeichen(double x0,double y0,double kante,uint zeichen)
{
 int i,j;
 double x,y,x1;
 uchar *pfont;
 double dy=kante/font_hy;
 if(zeichen==copypaste_zeichen)
      pfont = copypaste_puffer;
 else pfont = &aktueller_font[zeichen*bytesprozeichen];
 x1=x0+(font_hy-font_bx)*dy/2;
 for(x=x1,i=0; i<font_bx; i++,x+=dy)
  for(y=y0+(font_hy-1)*dy,j=0; j<font_hy; j++,y-=dy)
   {
    if(bitgesetzt(pfont,i,j))
      fillbox(x,y,x+dy,y+dy);
   }
}

void bitumkehren(int i,int j)
{
 int maske;
 uchar *pfont, *p;
 pfont = &aktueller_font[aktuelles_zeichen*bytesprozeichen];
 p = bitlesen(pfont,i,j,&maske,font_bx,font_hy,font_typ);
 if(p==NULL)
  {
   fprintf(stderr,"Error in bitumkehren()\n");//test
  }
 else
  {
   savedflag=0;
   *p ^= maske;
  }
}

void zeichen_copy(int zeichen)
{
 uchar* pfont = &aktueller_font[zeichen*bytesprozeichen];
 if(copypaste_puffer==NULL)
  {
   copypaste_puffer = (uchar*)calloc(bytesprozeichen, 1);
   if(copypaste_puffer==NULL)
    {fprintf(stderr,"Error: Unable to allocate memory!\n"); return;}
  }
 for(int i=0;i<bytesprozeichen;i++)
   copypaste_puffer[i] = pfont[i];
}

void zeichen_paste(int zeichen)
{
 if(copypaste_puffer!=NULL)
  {
   uchar* pfont = &aktueller_font[zeichen*bytesprozeichen];
   for(int i=0;i<bytesprozeichen;i++)
     pfont[i] = copypaste_puffer[i];
   savedflag=0;
  }
}

void zeichen_vertauschen(int c1,int c2) //zwei Zeichen innerhalb des aktuellen Fonts vertauschen
{
 uchar h;
 uchar* p1 = &aktueller_font[c1*bytesprozeichen];
 uchar* p2 = &aktueller_font[c2*bytesprozeichen];
 for(int i=0;i<bytesprozeichen;i++)
  {
   h=p1[i]; p1[i]=p2[i]; p2[i]=h;
  }
 savedflag=0;
}

void font_convert(int bx,int hy,int typ,int skalierung) //aktueller_font nach neuer_font konvertieren
{
 //skalierung: 0=ohne, 1=auto, 2=ja
 int z,i,j,maske,i1,j1,i2,j2,i3,j3;
 uchar *palt,*pneu,*p;
 int skalierfaktor=1;
 double dskal=0;
 if((bx<font_bx && hy>font_hy) || (bx>font_bx && hy<font_hy))
  {fprintf(stderr,"Error: cant convert to different font aspect.\n"); return;}//provi.
 if(neuer_font!=NULL) free(neuer_font);
 neu_bytesprozeichen = (typ<=8) ? (bx+7)/8*hy : (hy+7)/8*bx;
 neu_maxfontbytes = 256*neu_bytesprozeichen;
 neuer_font = mycalloc(neu_maxfontbytes);
 if(neuer_font==NULL) return;
 neu_bx=bx, neu_hy=hy, neu_font_typ=typ;
 pneu=neuer_font;
 palt=aktueller_font;
 if(bx>=font_bx && hy>=font_hy)
  {
   if(skalierung>=1)
    {
     if(skalierung==2) dskal=groesserer_wert(bx/(double)font_bx,hy/(double)font_hy);
     else skalierfaktor=kleinerer_wert(bx/font_bx,hy/font_hy);
    }
   for(z=0; z<256; z++,palt+=bytesprozeichen,pneu+=neu_bytesprozeichen) //alle Zeichen konvertieren
     for(i=i1=0;i<font_bx;i++,i1=i2)
       {
	if(dskal==0) i2=(i+1)*skalierfaktor;
	else         i2=int((i+1)*dskal+0.5);
	for(j=j1=0;j<font_hy;j++,j1=j2)
	 {
	  if(dskal==0) j2=(j+1)*skalierfaktor;
	  else         j2=int((j+1)*dskal+0.5);
	  if(bitgesetzt(palt,i,j))
	   {
	    for(i3=i1;i3<i2;i3++)
	     for(j3=j1;j3<j2;j3++)
	      {
	       p = bitlesen(pneu,i3,j3,&maske,bx,hy,typ);
	       *p |= maske;
	      }
	   }
	 }
       }
  }
 else //if(bx<=font_bx && hy<=font_hy)
  {
   if(skalierung>=1)
    {
     if(skalierung==2) dskal=groesserer_wert(font_bx/(double)bx,font_hy/(double)hy);
     else  skalierfaktor = kleinerer_wert(font_bx/bx,font_hy/hy);
    }
   for(z=0; z<256; z++,palt+=bytesprozeichen,pneu+=neu_bytesprozeichen) //alle Zeichen konvertieren
     for(i2=0;i2<bx;i2++)
       {
	if(dskal==0) i=i2*skalierfaktor;
	else         i=int(i2*dskal+0.5);
	for(j2=0;j2<hy;j2++)
	 {
	  if(dskal==0) j=j2*skalierfaktor;
	  else         j=int(j2*dskal+0.5);
	  if(bitgesetzt(palt,i,j))
	   {
	    p = bitlesen(pneu,i2,j2,&maske,bx,hy,typ);
	    *p |= maske;
	   }
	 }
       }
  }
 savedflag=0;
}

void fonts_vertauschen() //aktueller_font mit neuer_font vertauschen
{
 if(neuer_font!=NULL && neu_bx!=0 && neu_hy!=0 && neu_font_typ!=0)
  {
   uchar *p;
   int h;
   p=aktueller_font; aktueller_font=neuer_font; neuer_font=p;
   h=font_bx; font_bx=neu_bx; neu_bx=h;
   h=font_hy; font_hy=neu_hy; neu_hy=h;
   h=font_typ; font_typ=neu_font_typ; neu_font_typ=h;
   h=maxfontbytes; maxfontbytes=neu_maxfontbytes; neu_maxfontbytes=h;
   h=bytesprozeichen; bytesprozeichen=neu_bytesprozeichen; neu_bytesprozeichen=h;
  }
 else
  fprintf(stderr,"Error in fonts_vertauschen()\n");
}

/************************* Probefeld **********************************/
// Globale Variablen:
//static int probemodus...  schon weiter oben
static uchar probe_puffer[200]; //Puffer fuer Probefeld
static int  probe_jp=0; //Index fuer puffer
static double prob_hy=0.04, prob_bx=0.03; //Hoehe und Breite der Zeichen im Probe-Bereich
static double prob_ay=0.004, prob_ax=0.003; //Abstand zwischen den Zeichen im Probe-Bereich
static double cursor_maxy=probey+probe_kantey-prob_hy-prob_ay, cursor_miny=probey+prob_ay;
static double cursor_maxx=probex+probe_kantex-prob_bx-prob_ax, cursor_minx=probex+prob_ax-(prob_hy-prob_bx)/2;
static double cursor_x=cursor_minx, cursor_y=cursor_maxy;
//static int zeichen_pro_zeile = int((probe_kantex-prob_ax)/(prob_bx+prob_ax));
const double dd=1e-8; //zum Rundungsfehler ausgleichen

void probefeld_loeschen()
{
 probe_jp=0;
}

void probe_writec(int c)
{
 if(c==(-1))
  {cursor_x=cursor_minx; cursor_y=cursor_maxy; return;}
 zeichne_zeichen(cursor_x,cursor_y,prob_hy,c);
 cursor_x += prob_bx+prob_ax;
 if(cursor_x>cursor_maxx+dd)
    {
     cursor_x = cursor_minx;
     cursor_y -= prob_hy+prob_ay;
     if(cursor_y < cursor_miny-dd)
       cursor_y=cursor_maxy;
    }
}

void probe_schreiben(int c) //ein Zeichen im Probe-Bereich schreiben
{
 if(prob_bx==0) prob_bx=prob_hy/font_hy*font_bx;
 if(probe_offset==0 && c==0x08) //Loeschtaste?
  {
   if(probe_jp>0)
    {
     double x;
     cursor_x -= prob_bx+prob_ax;
     if(cursor_x < cursor_minx-dd) cursor_x=cursor_minx;//provi.
     else --probe_jp; //provi. nur bis Zeilananfang loeschbar
     hintergrundfarbe();
     x=cursor_x+(prob_hy-prob_bx)/2;
     fillbox(x,cursor_y,x+prob_bx+prob_ax+dd,cursor_y+prob_hy+dd); //Zeichen loeschen
     vordergrundfarbe();
    }
  }
 else if(probe_jp<200) //sonst normale Taste
  {
   probe_puffer[probe_jp++]=c;
   probe_writec(c);
  }
}

void probefeld_neuzeichnen()
{
 // Beschriftung in der Probe-Statuszeile:
 double h=textsize(0.015,0.02,NULL,0); //breite, hoehe, font, style
 hintergrundfarbe();
 fillbox(probe_statusx,probe_statusy,probe_statusx+probe_kantex,probe_statusy+probe_status_kantey);
 vordergrundfarbe();
 wprintf(probe_statusx+h/8,probe_statusy+h/8,"Probemodus: %d   Offset: 0x%02X",probemodus,probe_offset);
 if(probemodus!=0)
  {
   wprintf(probe_loeschbox.x0+h/8,probe_loeschbox.y0+h/8,"Clear",0,0);
   probe_loeschbox.draw();
  }
 probe_writec(-1); //cursor in Anfangsposition
 for(int j=0;j<probe_jp;j++)
   probe_writec(probe_puffer[j]); //Zeichen vom Puffer schreiben
}

void zeichen_zurechtschieben(int richtung)
{
 //aktuelles_zeichen um 1 Bit verschieben, sofern dabei keine Bits verloren gehen
 int i,j,n,maske;
 uchar *pfont, *p, bits[font_bx][font_hy];
 pfont = &aktueller_font[aktuelles_zeichen*bytesprozeichen];

 //Das Feld bits[i][j] einlesen:
 for(i=0; i<font_bx; i++)
  for(j=0; j<font_hy; j++)
   {
    bits[i][j] = bitgesetzt(pfont,i,j) ? 1 : 0;
   }

 switch(richtung)
  {
  case RE:
    for(n=0,j=0;j<font_hy;j++)  n += bits[font_bx-1][j];
    if(n!=0) return;
    for(j=0;j<font_hy;j++)
     {
      for(i=font_bx;--i>=1;)
        bits[i][j]=bits[i-1][j];
      bits[0][j]=0;
     }
    break;
  case LI:
    for(n=0,j=0;j<font_hy;j++)  n += bits[0][j];
    if(n!=0) return;
    for(j=0;j<font_hy;j++)
     {
      for(i=0;++i<font_bx;)
        bits[i-1][j]=bits[i][j];
      bits[font_bx-1][j]=0;
     }
   break;
  case OB:
    for(n=0,i=0;i<font_bx;i++)  n += bits[i][0];
    if(n!=0) return;
    for(i=0;i<font_bx;i++)
     {
      for(j=0;++j<font_hy;)
        bits[i][j-1]=bits[i][j];
      bits[i][font_hy-1]=0;
     }
   break;
  case UN:
    for(n=0,i=0;i<font_bx;i++)  n += bits[i][font_hy-1];
    if(n!=0) return;
    for(i=0;i<font_bx;i++)
     {
      for(j=font_hy;--j>=1;)
        bits[i][j]=bits[i][j-1];
      bits[i][0]=0;
     }
   break;
  }

 //bits[i][j] wieder speichern:
 for(i=0; i<font_bx; i++)
  for(j=0; j<font_hy; j++)
   {
    p = bitlesen(pfont,i,j,&maske,font_bx,font_hy,font_typ);
    if(bits[i][j]==0) *p &= ~maske; //Bit loeschen
    else              *p |= maske;  //oder Bit setzen
   }
 savedflag=0;
}
