/* biber.cc       letzte Aenderung: 1.9.2017
Simulation einer Touringmaschine fuer LED-Matrix mit 4 Modulen
*/
#define VERSION "0.1"
/*
 Prozessor: Atmega1284P, 20MHz Quarz
 Schaltung: ledmatrix.png
            4 Stueck so verschaltet:
	      jeweils mit allen verbunden: GND, +5V, MISO, TAKT
	      IN1 ist jeweils OUT von Rechts, IN2 ist OUT von Oben

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 26.8.2017	Erstellung aus nimm.cc

*/
#define F_CPU 20000000UL  //20 MHz Quarz
//#define DEBUG           //zur Fehlersuche
//#define UART_BASIRGB  //Darstellung auch auf Fernseher (noch nicht getestet)
//#define BAND_BEWEGEN //auskommentieren fuer fixes Band und dafuer Kopf bewegen

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
//#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>

#include "led_treiber.c"

/**** Vordeklarationen ****/
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t

void status_anzeigen(ulong anzahlsteps);

#ifdef UART_BASIRGB
void basi_viereck(uchar j,uchar i,uchar farbe);
#endif

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=0;
static volatile uchar tage=28,monat=8; //aktuelles Datum
static volatile uint jahr=2017;
static char monatstage[13]={31,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=0; //0=Montag ... 6=Sonntag

void schaltjahrtest()
{
 if(jahr%4==0 && (jahr%100!=0 || jahr%400==0)) monatstage[2]=29;
 else monatstage[2]=28;
}
/*
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
*/

// Fuer sehr genaue Uhr Quarz moeglichst genau ausmessen und hier eintragen:
#define Hz 20000000L //exakt 20MHz

#define Rest1 (Hz%1000) //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (Hz/1000-1)

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
}

ISR(TIMER1_COMPA_vect)
{
 uchar mehr;
 uint ms=millisec;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0; mehr=0;
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
      }
    }
  }
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
 //tastaturcheck();
}

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) ;//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
   ;//auf n. Millisekunde warten
}

void secwait(uchar n) //n Sekunden warten
{
 while(n!=0) {milliwait(1000); --n;}
}

void microwait(uchar n) //mindestens n Microsekunden warten
{
 while(--n!=0) _delay_us(1);
}

/***************************** Timer0 fuer LED-Anzeige ********************************/
static volatile uchar ledfeld[5][10]; // 5 Reihen, 10 Spalten
void ledfeld_init(uchar n)
{
 uchar i,j;
 for(i=0;i<5;i++)
  for(j=0;j<10;j++)
    ledfeld[i][j]=n; //alle gleich setzen
}

void timer0_init()
{
 TCCR0A = (0<<WGM01)|(0<<WGM00); //Normal-Modus, Interrupt bei Ueberlauf (8 Bit, also Teiler durch 256)
 //TCCR0A = (1<<WGM01)|(0<<WGM00); //CTC-Modus, Interrupt bei Erreichen von OCRA
 //TCCR0B = (1<<CS00); //Takt nicht teilen
 TCCR0B = (1<<CS01); //Takt durch 8teilen
 TCNT0 = 0;
 TIMSK0 = (1<<TOIE0); //Interrupt bei Ueberlauf
 //TIMSK0 = (1<<OCIE0A); //Interrupt bei Erreichen von OCRA
}

ISR(TIMER0_OVF_vect)
{
 static uchar spalte=0;
 uchar b,c,d,tmp2,tmp3;
 bool flag;
 tmp2 = (~ledfeld[2][spalte])&7;
 tmp3 = (~ledfeld[3][spalte])<<3;
 //PORTB |= 0x0F; PORTC |= 0x7F; PORTD = 0xFF;//test
 b = ((~ledfeld[4][spalte])&7) | (tmp3&0x08);
 c = (((~ledfeld[0][spalte])&7)<<4) | (((~ledfeld[1][spalte])&7)<<1) | (tmp2>>2);
 d = ((tmp2&3)<<6) | (tmp3&0x30);
 flag = (b!=0x0F) || (c!=0x7F) || (d!=0xF0);
 PORTD |= 3; PORTA = 0xFF; //alle Transistoren aus
 PORTB = (PORTB&0xF0) | b;
 PORTC = (PORTC&0x80) | c;
 PORTD = 0x0F | d;
 if(flag) //nur wenn auch mindestens eine LED an:
  switch(spalte) //entsprechender Transistor einschalten
   {
    case 0: PORTD &= ~(1<<1); break;
    case 1: PORTD &= ~(1<<0); break;
    case 2: PORTA &= ~(1<<7); break;
    case 3: PORTA &= ~(1<<6); break;
    case 4: PORTA &= ~(1<<5); break;
    case 5: PORTA &= ~(1<<4); break;
    case 6: PORTA &= ~(1<<3); break;
    case 7: PORTA &= ~(1<<2); break;
    case 8: PORTA &= ~(1<<1); break;
    case 9: PORTA &= ~(1<<0); break;
   }
 if(++spalte==10) spalte=0;
 //if(spalte==0) spalte=9; else --spalte;//test
}

/************************ Datenuebertragung zwischen den Modulen **********************/
#define MISO_DDR  DDRB
#define MISO_PORT PORTB
#define MISO_PIN  PINB
#define MISOBIT 6
#define TAKT_DDR  DDRB
#define TAKT_PORT PORTB
#define TAKT_PIN  PINB
#define TAKTBIT 7

/* Der Master sendet einen TAKT und auf MISO jeweils 8 Bits, steigende Flanke von
   TAKT soll das Bit in MISO uebernehmen.
   Erstes gesendetes Byte ist Modulnummer, dann 1 Byte j, 1 Byte i, und 1 Byte Datenwert.
 */
bool masterflag=false; //darf nur in erstem Modul gesetzt werden
uchar modulnummer=0; //gueltige Nummern: 1...4, 0=noch nicht gesetzt

void byte_senden(uchar c)
{
 uchar i;
 for(i=0;i<8;c<<=1,i++)
  {
   if(c&0x80) MISO_PORT |= (1<<MISOBIT);
   else       MISO_PORT &= ~(1<<MISOBIT);
   TAKT_PORT &= ~(1<<TAKTBIT);
   microwait(20);
   TAKT_PORT |= (1<<TAKTBIT);
   microwait(20);
  }
}

void daten_senden(uchar nummer,uchar j,uchar i,uchar wert)
{
 byte_senden(nummer);
 byte_senden(j);
 byte_senden(i);
 byte_senden(wert);
}

bool byte_empfangen(uchar& c)
{
 uchar i,bit;
 int timeout=0;
#define TIMEOUT 1000
 c=0;
 for(i=0;i<8;i++)
  {
   while(TAKT_PIN&(1<<TAKTBIT)) //warten bis Low
     {_delay_us(1); if(++timeout>=TIMEOUT) return false;} //Fehler bei Timeout
   while((TAKT_PIN&(1<<TAKTBIT))==0) ;//warten auf High
   bit = (MISO_PIN&(1<<MISOBIT))==0 ? 0 : 1;
   c = (c<<1)+bit;
  }
 return true;
}

void daten_empfangen()
{
 uchar n,i,j,wert;
 if(!byte_empfangen(n)) return; //Rueckkehr bei Fehler
 if(!byte_empfangen(j)) return;
 if(!byte_empfangen(i)) return;
 if(!byte_empfangen(wert)) return;
 if(n==modulnummer)
   ledfeld[j][i]=wert;
}

/******************* alle LED-Felder vom Master aus behandeln *************************/
// Master ist das Modul oben links
//tatic volatile uchar ledfeld[5][10]; //Modul oben links //schon weiter oben definiert
static volatile uchar ledfeld2[5][10]; //Modul oben rechts
static volatile uchar ledfeld3[5][10]; //Modul unten links
static volatile uchar ledfeld4[5][10]; //Modul unten rechts
#ifdef UART_BASIRGB
static volatile uchar gesamtfeld[10][20]; //gesamtes LED-Feld (nur von Master benoetigt)
#endif

void gesamtes_ledfeld_setzen(uchar j,uchar i,uchar wert) //nur von Master aufrufen
{
 if(j>=5)
  {
   if(i>=10)
    {
     ledfeld4[j-5][i-10]=wert;
     daten_senden(4,j-5,i-10,wert);
    }
   else
    {
     ledfeld3[j-5][i]=wert;
     daten_senden(3,j-5,i,wert);
    }
  }
 else if(i>=10)
  {
   ledfeld2[j][i-10]=wert;
   daten_senden(2,j,i-10,wert);
  }
 else
  ledfeld[j][i]=wert;
#ifdef UART_BASIRGB
 if(wert!=gesamtfeld[j][i])
  {
   basi_viereck(j,i,wert);
   gesamtfeld[j][i]=wert;
  }
#endif
}

uchar gesamtes_ledfeld_lesen(uchar j,uchar i) //nur von Master aufrufen
{
 if(j>=5)
  {
   if(i>=10) return ledfeld4[j-5][i-10];
   else      return ledfeld3[j-5][i];
  }
 else if(i>=10) return ledfeld2[j][i-10];
 return ledfeld[j][i];
}

/*************************** Serielle Schnittstelle ***********************************/
#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 20000000"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif

#define BAUD 9600UL      // Baudrate
//#define BAUD 1200UL      //test
 
// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart1_init()
{
 UBRR1 = UBRR_VAL;
#ifdef UART_BASIRGB
 UCSR1B = (1<<RXEN1)|(1<<TXEN1); // UART TX und RX einschalten
#else
 UCSR1B = (1<<RXEN1); // nur UART RX einschalten
#endif
 //UCSR1C = (3<<UCSZ10); // Asynchron 8N1
 UCSR1C = (3<<UCSZ10)|(1<<USBS1); // Asynchron 8N2
}

#ifdef UART_BASIRGB
void uart_putc(unsigned char c)
{
 while (!(UCSR1A & (1<<UDRE1))) ;// warten bis Senden moeglich
 UDR1 = c;                      // sende Zeichen
 milliwait(3);//test
}
#endif

unsigned char uart_getc(void)
{
 while (!(UCSR1A & (1<<RXC1))) ;// warten bis Zeichen verfuegbar
 return UDR1; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart_check()
{
 return (UCSR1A & (1<<RXC1)) ;//true wenn Zeichen verfuegbar
}

/************************** Sachen noch von Nimm **************************************/
#ifdef UART_BASIRGB
void basi_screenclear()
{
 uchar j,i;
 uart_putc(0x10); uart_putc(0x00); //Hintergrundfarbe Schwarz
 uart_putc(0x04); //Screenclear auf Fernseher
 for(j=0;j<10;j++)
  for(i=0;i<20;i++)
   gesamtfeld[j][i]=0;
}

void basi_viereck(uchar j,uchar i,uchar farbe)
{
 j *= 8;
 i *= 8;
 uart_putc(0x11); uart_putc(farbe); //Vordergrundfarbe
 uart_putc(0x12);
 uart_putc(i&0xFF); uart_putc(0); //(eigentlich uart_putc(j>>8)) Position vom Viereck links oben
 uart_putc(j&0xFF); uart_putc(0); //Position vom Viereck links oben
 uart_putc(0x16); //Viereck auf Fernseher
 uart_putc((i+6)&0xFF); uart_putc(0); //Position vom Viereck rechts unten
 uart_putc((j+6)&0xFF); uart_putc(0); //Position vom Viereck rechts unten
 //milliwait(10); //test
}
#endif

#define STARTPOSY 0
#define STARTPOSX 1
#define CURSOR_STARTPOSX 0
static uchar cursorx=CURSOR_STARTPOSX, cursory=STARTPOSY;

void cursor_zeichnen_loeschen(uchar flag)
{
 static uchar x0=0,y0=0,vorher=0,vorherflag=0;
 if(vorherflag!=0) gesamtes_ledfeld_setzen(y0,x0,vorher);
 vorher=gesamtes_ledfeld_lesen(y0=cursory,x0=cursorx);
 if(flag==2)
   vorherflag=0;
 else
  {
   gesamtes_ledfeld_setzen(cursory,cursorx,1);
   vorherflag=1;
  }
}
void cursor_zeichnen()
{
 static uchar flag=0;
 cursor_zeichnen_loeschen(flag);
 flag=1;
}
void cursor_loeschen()
{
 cursor_zeichnen_loeschen(2);
}

uchar rand8() //8-Bit-Zufallszahl
{
 uchar c=(TCNT1&0xFF);
 return c;
}

/****************************** Testroutinen ******************************************/
void einige_led_leuchten_lassen(uchar startspalte=0,uchar startreihe=0,uchar startfarbe=1,uchar n=8)
{
 uchar i,j=startreihe,farbe=0;
 for(i=startspalte;i<startspalte+n;i++) //n Felder auf dunkel
  gesamtes_ledfeld_setzen(j,i,farbe);
 secwait(1); //1 Sec warten
 
 farbe=startfarbe;
 for(i=startspalte;i<startspalte+n;i++) //n Felder leuchten lassen
    {
     gesamtes_ledfeld_setzen(j,i,farbe);
     if(++farbe>=8) farbe=0;
    }
 gesamtes_ledfeld_setzen(j,i+startspalte,0); //Feld nach letztem Stein der Reihe auf Dunkel setzen
}

/********************************* Kleinkram ******************************************/
#define SCHWARZ 0
#define ROT   1
#define GRUEN 2
#define GELB  3
#define BLAU  4
#define VIOLET 5
#define CYAN  6
#define WEISS 7

void alle_LED_aus()
{
 uchar i,j,farbe=0; //i=spalte j=reihe
 for(j=0;j<10;j++)
  for(i=0;i<20;i++)
   gesamtes_ledfeld_setzen(j,i,farbe); //alle LEDs aus
}

void binaerzahl_auf_LED(uint64_t zahl,uchar laenge,uchar farbe0,uchar farbe1,uchar j,uchar i)//j=reihe, i=spalte
{
 uint64_t bit, mask=1;
 mask <<= (laenge-1);
 for(;mask!=0;mask>>=1)
  {
   bit = zahl&mask;
   gesamtes_ledfeld_setzen(j,i,(bit==0)?farbe0:farbe1);
   if(++i==20) {i=0; if(++j==10) j=0;}
  }
}

void zahl_anzeigen(ulong zahl,uchar farbe,uchar basis=10)
{
 char txt[8];
 ulong basishoch5 = (basis==16)?100000:0x100000;
 led_farbe=farbe;//provi.
 led_clear();
 //led_cursor(0,0); //Position fuer led_print (schon in led_clear auf 0,0 gesetzt)
 if(zahl>basishoch5-1) //Zahl laenger als 5 Stellen?
  {
   ulong zoben=zahl/basishoch5;
   zahl = zahl%basishoch5;
   if(zoben>basishoch5-1) zoben=basishoch5-1;
   sprintf(txt,(basis==16)?"%5lX":"%5ld",zoben);
   led_print(txt);
   milliwait(750); //obere 5 Stellen waehrend sovielen Millisec anzeigen
   led_clear();
  }
 sprintf(txt,(basis==16)?"%4lX":"%4ld",zahl);
 led_print(txt);
}

/********************** Simulation einer Touringmaschine ******************************/
#define MAX_ZUSTAENDE 8 //Maximale Anzahl Zustaende
//#define MAX_BAND 200  //Groesse des Bandes, 200=minimale Groesse
#define MAX_BAND 65528  //Groesse des Bandes
                        //(Atmega hat 16KB RAM --> maximal also etwa 15k*8=120000 verfuegber, aber wegen uint nur 16 Bit --> 65528 Maximalwert)
#define ANZ_MOEGL_WERTE 2 //vorlaeufig nur 0 und 1 erlaubt (so definiert fuer "Fleissige Biber")
#define MAX ((MAX_BAND+7)/8)  //Anzahl Bytes um die entsprechenden Bits zu speichern
#define RECHTS 1
#define LINKS  0
#define R RECHTS
#define L LINKS
#define A 0
#define B 1
#define C 2
#define D 3
#define E 4
#define F 5
#define G 6
#define I 7  //kein H wegen Verwechslung mit Halt
//#define J 8  //wenn MAX_ZUSTAENDE>=9
//#define K 9  //wenn MAX_ZUSTAENDE>=10
#define HALT 0xFF

class Band
{
public:
 uchar feld[MAX];
 uint pos,altpos;
#ifndef BAND_BEWEGEN
 uint posmin,posmax;
#endif
 uint32_t overflow;
 Band() {clear();}
 void clear();
 uchar read(uint k) {uchar c=feld[k/8]; return (c>>(k%8))&1;}
 uchar read() {return read(pos);}
 void write(uchar bit,uchar richtung);
 void led_setzen(uchar j,uchar i,uchar wert,bool bflag);
#ifdef BAND_BEWEGEN
 void darstellen();
#else
 void darstellen(uint k1,uint k2);
 void neuzeichnen();
#endif
 uint zaehlen(); //Anzahl Einsen auf dem Band zaehlen
 uint benutzte_bandlaenge(); //Laenge des benutzten Teil des Bandes
};
static Band band;

uint Band::zaehlen() //Anzahl Einsen auf dem Band zaehlen
{
 uint anzahl=0;
 for(uint k=0;k<MAX_BAND;k++)
  anzahl += read(k);
 return anzahl;
}
uint Band::benutzte_bandlaenge() //Laenge des benutzten Teil des Bandes
{
 if(overflow!=0) return MAX_BAND;
 if(posmax>=posmin) return posmax-posmin+1;
 return ulong(posmax)+MAX_BAND+1-posmin;
}

void Band::clear()
{
#ifdef BAND_BEWEGEN
 pos=altpos=MAX_BAND/2;
#else
 pos=altpos=10;
#endif
 posmin=posmax=pos;
 overflow=0;
 for(int i=0;i<MAX;i++) feld[i]=0;
 //feld[MAX/2]=0xE3;//test mit nicht leerem Band
}

void Band::write(uchar bit,uchar richtung)
{
 uint i=pos/8;
 uchar c=feld[i];
 if(bit==1) c |=  (1<<(pos%8));
 else       c &= ~(1<<(pos%8));
 feld[i]=c;
 altpos=pos;
 if(richtung==RECHTS)
  {
   if(posmax==pos)
    {if(++pos==MAX_BAND) pos=0;
     posmax=pos;
     if(posmax==posmin) overflow++;//TODO noch austesten ob es korrekt funktioniert
    }
   else
    {if(++pos==MAX_BAND) pos=0;}
  }
 else //richtung==LINKS
  {
   if(posmin==pos)
    {if(pos==0) pos=MAX_BAND-1; else --pos;
     posmin=pos;
     if(posmax==posmin) overflow++;//TODO noch austesten ob es korrekt funktioniert
    }
   else
    {if(pos==0) pos=MAX_BAND-1; else --pos;}
  }
}

void Band::led_setzen(uchar j,uchar i,uchar wert,bool bflag)
{
 uchar farbe0=BLAU, farbe0b=CYAN;
 uchar farbe1=ROT, farbe1b=VIOLET;
 if(wert==1) gesamtes_ledfeld_setzen(j,i,(bflag)?farbe1b:farbe1); //darstellung einer 1 mit farbe1 (rot)
 else        gesamtes_ledfeld_setzen(j,i,(bflag)?farbe0b:farbe0); //darstellung einer 0 mit farbe0 (blau)
}

#ifdef BAND_BEWEGEN
void Band::darstellen()
{
 uchar i,j; //i=spalte j=reihe
 uint k;
 for(k=pos,i=9,j=4;j<10;)
  {
   led_setzen(j,i,read(k),k==altpos);
   if(++k==MAX_BAND) k=0;
   if(++i==20) {i=0; ++j;}
  }
 for(k=pos,i=9,j=4;j!=9;)
  {
   if(k==0) k=MAX_BAND-1; else --k;
   if(i==0) {i=19; if(j==0) j=9; else --j;}
   else --i;
   led_setzen(j,i,read(k),k==altpos);
  }
}
#else
/*
uchar divmod20(uchar x,uchar *rest)
{
 *rest = x%20; //Compiler optimiert das schon
 return x/20;
}
*/

void Band::darstellen(uint k1,uint k2)
{
 uchar i,j; //i=spalte j=reihe
 if(k2<200)
  {
   //j=divmod20(k2,&i);
   j=k2/20; i=k2%20;
   led_setzen(j,i,read(k2),true); //neu gesetztes markieren
   if(k1<200)
    {
     //divmod20(k1,&i);
     j=k1/20; i=k1%20;
     led_setzen(j,i,read(k1),false); //vorher markiertes wieder unmarkiert
    }
  }
}
void Band::neuzeichnen()
{
 uchar i,j; //i=spalte j=reihe
 uint k;
 for(i=j=0,k=0;k<200 && k<MAX_BAND;k++)
  {
   led_setzen(j,i,read(k),false); //zeichnen ohne zu markieren
   if(++i==20) {i=0; ++j;}
  }
}
#endif

class Eintrag
{
public:
 uchar schreibwert,richtung,neuerzustand;
 void set(uchar s,uchar r,uchar z) {schreibwert=s; richtung=r; neuerzustand=z;}
};

class Tm
{
public:
 //Eintrag *tabelle;
 Eintrag tabelle[MAX_ZUSTAENDE*ANZ_MOEGL_WERTE]; //auf diesen Wert begrenzt, aber new und delete vermeidbar
 uchar zustand; //aktueller Zustand
 uchar zmax; //Anzahl moegliche Zustaende (von 0 bis zmax-1 nummeriert)
 /* new und delete[] auf Microkontroller geht vielleicht nicht
 Tm(uint z) {zmax=z; zustand=A; tabelle=new Eintrag[z*ANZ_MOEGL_WERTE];}
 ~Tm() {delete[] tabelle;}
 */
 Tm() {zmax=1; zustand=A; setprog(A,0,1,R,HALT); setprog(A,1,1,L,HALT);}//Minimalstes falls vergessen zu initialisieren
 Tm(uint z) {zmax=z; zustand=A;}
 void setprog(uchar z,uchar lesewert,uchar schreibwert,uchar richtung,uchar neuz);
 bool step();
 void reset() {zustand=A;}
};

void Tm::setprog(uchar z,uchar lesewert,uchar schreibwert,uchar richtung,uchar neuz)
{
 tabelle[z*ANZ_MOEGL_WERTE+lesewert].set(schreibwert,richtung,neuz);
}

bool Tm::step()
{
 if(zustand>=zmax) return false; //HALT erreicht
 uchar lesewert=band.read();
 Eintrag *e = &tabelle[zustand*ANZ_MOEGL_WERTE+lesewert];
 band.write(e->schreibwert,e->richtung);
 zustand = e->neuerzustand;
 return true;
}

/********************************** Hauptprogramm *************************************/
void startschirm_zeichnen()
{
 alle_LED_aus();
 // Binaerzahlen fuer Auswahl von z:
 binaerzahl_auf_LED(2,8,BLAU,ROT,0,2);//j=reihe, i=spalte
 binaerzahl_auf_LED(3,8,BLAU,ROT,1,2);//j=reihe, i=spalte
 binaerzahl_auf_LED(4,8,BLAU,ROT,2,2);//j=reihe, i=spalte
 binaerzahl_auf_LED(5,8,BLAU,ROT,3,2);//j=reihe, i=spalte
 binaerzahl_auf_LED(8,8,BLAU,ROT,4,2); //8 Zustande, wenn MAX_ZUSTAENDE erhoeht entsprechend anpassen
 cursorx=0; cursory=0;
 cursor_zeichnen();
}

int main(void)
{
 uchar i,j,c;

 PORTA = 0xFF; //PortA alle auf High fuer inaktiv (P-Chann-Transistoren)
 PORTD = 0x03; //PortD: unterste 2 Bits High fuer inaktiv (P-Chann-Transistoren),
 //PORTD = 0x0F;  //      INT0-INT1 Pullup-Widerstaende fuer Tasten setzen
 DDRA = 0xFF; //PortA alles Ausgaenge
 DDRD = 0xF3; //PortD alle ausser INT0-INT1 auf Ausgang
 DDRB = 0x0F; //PortB untere 4 Bits auf Ausgang
 DDRC = 0xFF; //PortC alles Ausgaenge
 PORTB = 0xFF; //alle Ausgaenge inaktiv, Eingaenge Pullups setzen
 PORTC = 0x7F; //Ausgaenge fuer LEDs inaktiv, OUT auf aktiv low

 timer0_init();
 timer1_init();
 ledfeld_init(0);
 sei(); //Interrupt einschalten

 milliwait(10); //warten bis andere Module auch bereit
 c=(PINB&0x30); //Status der Eingaenge lesen
 switch(c)
  {
   case 0x00: modulnummer=3; break;
   case 0x10: modulnummer=4; break;
   case 0x20: masterflag=true; modulnummer=1; break;
   case 0x30: modulnummer=2; break;
  }

 // Einschalt-Test:
 if(masterflag) c=7; //weiss fuer Modul oben links
 else {c=modulnummer-1; if(c==3) c=4;} //rot, gruen, blau fuer die andern Module
 ledfeld[0][0]=c; //LED oben links setzen
 milliwait(500);
 ledfeld[0][0]=c; //wieder loeschen

 if(!masterflag) //andere Module nur Daten vom Master empfangen
  {
   for(;;) //Hauptschlaufe fuer Sklaven-Module
    {
     if((TAKT_PIN&(1<<TAKTBIT))==0) //warten bis Start einer Datenuebertragung
       daten_empfangen();
    }
  }

 //Aktionen fuer den Master:
 uart1_init(); //Serielle Schnittstelle (UART) fuer Tastatur und fuer Video-Ausgabe auf Fernseher
 milliwait(100); //warten bis andere Module bereit
#ifdef UART_BASIRGB
 milliwait(900); //warten bis Fernseher bereit
 basi_screenclear(); //Screenclear auf Fernseher
 /*
 uart_putc(0x12); //test: Position fuer Test-Kreis
 uart_putc(0x40); uart_putc(0x00); uart_putc(0x40); uart_putc(0x00); //test: x- und y-Wert fuer Position
 uart_putc(0x17); //test: Kreis auf Fernseher zeichnen
 uart_putc(0x78); uart_putc(0x00); uart_putc(0x80); uart_putc(0x00); //test: Kreisdurchmesser in x- und y-Richtung
 */
#endif

 TAKT_DDR |= (1<<TAKTBIT);
 MISO_DDR |= (1<<MISOBIT);
 c=BLAU;
 for(j=0;j<10;j++)
 for(i=0;i<20;i++)
  {
   gesamtes_ledfeld_setzen(j,i,c); //alle LEDs nacheinander setzen
   //milliwait(50);//test
  }
 for(j=0;j<10;j++)
 for(i=0;i<20;i++)
  {
   gesamtes_ledfeld_setzen(j,i,0); //alle LEDs wieder aus
   //milliwait(50);//test
  }

 led_cursor(0,0); //Position fuer led_print
#ifdef DEBUG
 ulong zahl = sizeof(uchar); //1
 /*
 zahl_anzeigen(zahl,WEISS); secwait(1);
 zahl = sizeof(short); //2
 zahl_anzeigen(zahl,ROT); secwait(1);
 zahl = sizeof(int); //2
 zahl_anzeigen(zahl,GELB); secwait(1);
 zahl = sizeof(ulong); //4
 zahl_anzeigen(zahl,GRUEN); secwait(1);
 zahl = sizeof(ulong *); //2
 zahl_anzeigen(zahl,BLAU);
 */
 zahl = sizeof(char PROGMEM *); //2
 zahl_anzeigen(zahl,WEISS);
 secwait(3);
 led_clear();
 led_print("Fleis");//test
 milliwait(100); led_clear();//test
 led_print("siger");//test
 milliwait(500); led_clear();//test
#endif
 led_print("Biber");
 secwait(2);
 led_clear();

 uint z;
 Tm biber2(z=2);
 biber2.setprog(A,0, 1,R,B);
 biber2.setprog(A,1, 1,L,B);
 biber2.setprog(B,0, 1,L,A);
 biber2.setprog(B,1, 1,R,HALT);
 
 Tm biber3(z=3);
 biber3.setprog(A,0, 1,R,B);
 biber3.setprog(A,1, 1,R,HALT);
 biber3.setprog(B,0, 1,L,B);
 biber3.setprog(B,1, 0,R,C);
 biber3.setprog(C,0, 1,L,C);
 biber3.setprog(C,1, 1,L,A);
 
 Tm biber4(z=4);
 biber4.setprog(A,0, 1,R,B);
 biber4.setprog(A,1, 1,L,B);
 biber4.setprog(B,0, 1,L,A);
 biber4.setprog(B,1, 0,L,C);
 biber4.setprog(C,0, 1,R,HALT);
 biber4.setprog(C,1, 1,L,D);
 biber4.setprog(D,0, 1,R,D);
 biber4.setprog(D,1, 0,R,A);

 Tm biber5(z=5);
 biber5.setprog(A,0, 1,L,B);
 biber5.setprog(A,1, 1,R,C);
 biber5.setprog(B,0, 1,L,C);
 biber5.setprog(B,1, 1,L,B);
 biber5.setprog(C,0, 1,L,D);
 biber5.setprog(C,1, 0,R,E);
 biber5.setprog(D,0, 1,R,A);
 biber5.setprog(D,1, 1,R,D);
 biber5.setprog(E,0, 1,L,HALT);
 biber5.setprog(E,1, 0,R,A);

 Tm biber8(z=8);
 biber8.setprog(A,0, 1,R,B);
 biber8.setprog(A,1, 1,R,B);
 biber8.setprog(B,0, 1,R,C);
 biber8.setprog(B,1, 1,R,C);
 biber8.setprog(C,0, 1,L,E);
 biber8.setprog(C,1, 1,R,F);
 biber8.setprog(D,0, 1,L,A);
 biber8.setprog(D,1, 1,L,E);
 biber8.setprog(E,0, 1,R,A);
 biber8.setprog(E,1, 0,L,D);
 biber8.setprog(F,0, 1,R,C);
 biber8.setprog(F,1, 1,R,HALT);
 biber8.setprog(G,0, 1,R,F);
 biber8.setprog(G,1, 1,R,F);
 biber8.setprog(I,0, 1,L,D);
 biber8.setprog(I,1, 1,L,I);
 
 Tm biber;
 uchar startflag=0,pauseflag=0,anzeigeflag=0;
 int wartezeit=1000;
 ulong anzahlsteps=0;
 startschirm_zeichnen();
 //anzahlsteps=0xA7; band.overflow=0xE5;//test
 //binaerzahl_auf_LED(anzahlsteps,40,BLAU,GRUEN,8,0);//test Steps auf 2 letzten Reihen j=reihe, i=spalte
 //binaerzahl_auf_LED(band.overflow,20,BLAU,ROT,7,0);//test Ueberlauf auf 3.letzter Reihe anzeigen

 while(1) //Hauptschlaufe
  {
   if(anzeigeflag>=1)
    {//Anzeige der Ergebnisse nach einem Durchlauf
     if(anzeigeflag==2) {anzeigeflag=0; anzahlsteps=0; startschirm_zeichnen();}
     //else {}//warten auf Tastendrucke
    }
   else if(startflag>=1) //Biber simulieren
    {
     if(startflag==1)
      {
       z=cursory+2;
       switch(z)
	{case 2: biber=biber2; break;
	 case 3: biber=biber3; break;
	 case 4: biber=biber4; break;
	 case 5: biber=biber5; break;
	 case 6: default: biber=biber8; break;
	}
       band.clear();
       alle_LED_aus();
#ifdef BAND_BEWEGEN
       band.darstellen();
#else
       band.neuzeichnen();
       band.darstellen(0,band.pos);
#endif
       secwait(3);
       wartezeit=3000; //langsamste Einstellung, dann mit blauer Tastenreihe einstellbar
       biber.reset();
       startflag++;
      }
     else if(startflag==2 && pauseflag==0)
      {
       bool ok=biber.step();
#ifdef BAND_BEWEGEN
       band.darstellen();
#else
       band.darstellen(band.altpos,band.pos);
#endif
       if(ok)
	{
	 anzahlsteps++;
	 if(band.overflow!=0)
	  {
	   binaerzahl_auf_LED(band.overflow,4,SCHWARZ,ROT,9,16);//Ueberlauf auf letzten 4 Feldern anzeigen
	  }//TODO
	}
       else
	{
	 status_anzeigen(anzahlsteps);
	 startflag=0; anzeigeflag=1;
	}
       if(wartezeit>1000) secwait(wartezeit/1000);
       else if(wartezeit>0) milliwait(wartezeit);
      }
    }
   if(uart_check())   // Zeichen von Tastatur bereit?
    {
     c=uart_getc();
     uchar loslassflag=c&0x80;
     c &= 0x7F;

     //test:
     //j = (c-'A')/20; i = (c-'A')%20; //oben links = 'A', naechste LED = 'B', ...
     //gesamtes_ledfeld_setzen(j,i, loslassflag ? 0 : 1); //test-LED rot wenn gedrueckt, aus wenn losgelassen

     //Reaktionen bei Tastendruck:
     if(loslassflag==0)
      {
       if(c=='R') //Mittlere weisse Taste
	{
	 if(anzeigeflag>0)
	  {
	   if(cursory==6) zahl_anzeigen(band.benutzte_bandlaenge(),GELB);
	   else if(cursory==7) zahl_anzeigen(band.zaehlen(),CYAN);
	   else if(cursory==8) zahl_anzeigen(anzahlsteps,GRUEN);
	   else anzeigeflag=2;
	   if(anzeigeflag==1) secwait(3);
	   status_anzeigen(anzahlsteps);
	  }
	 else
	  {if(startflag==0)
	    {
	     if(anzahlsteps!=0)
	      {anzahlsteps=0; startschirm_zeichnen();}//vorheriger Lauf loeschen
	     else startflag=1; //Biber starten
	    }
	   else pauseflag ^= 1;
	  }
	}
       else if(c=='U') //Taste unten links
	{
	 if(anzeigeflag) anzeigeflag=2; //Resultat-Anzeigemodus beenden
	}
       else if(c=='Y') //Taste unten rechts
	{
	 if(anzeigeflag) anzeigeflag=2;
	}
       else if(c=='C') //oberste weisse Taste
	{
	 if(anzeigeflag) anzeigeflag=2;
	 //else einige_led_leuchten_lassen(0,9,1,8); //test auf letzter Reiche: 0=Spalte, 9=Reihe (10.)
	}
       else if(c=='F') {wartezeit=3000;}//obere blaue Reihe, erste Taste
       else if(c=='G') {wartezeit=1000;}//obere blaue Reihe, 2. Taste
       else if(c=='H') {wartezeit=100;}//obere blaue Reihe, 3. Taste
       else if(c=='I') {wartezeit=10;}//obere blaue Reihe, 4. Taste
       else if(c=='J') {wartezeit=0;}//obere blaue Reihe, letzte Taste
       else
	{
	 if(c=='M') //nach oben
	  {if(cursory>0) --cursory;}
	 else if(c=='W') //nach unten
	  {if(cursory<9) cursory++;}
	 else if(c=='Q') //nach links
	  {if(cursorx>0) --cursorx;}
	 else if(c=='S') //nach rechts
	  {if(cursorx<19) cursorx++;}
	 cursor_loeschen();
	 milliwait(500);
	 cursor_zeichnen();
	}
      }//Ende loslassflag
    }//Ende uart_check
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

void status_anzeigen(ulong anzahlsteps)
{
 binaerzahl_auf_LED(anzahlsteps,40,SCHWARZ,GRUEN,8,0);//8=Reihe, 0=Spalte, grosse Zahl letzte 2 Reihen
 binaerzahl_auf_LED(band.zaehlen(),20,SCHWARZ,CYAN,7,0);//Anzahl gezeichnete Einsen, 3.letzte Reihe
 if(band.overflow!=0)
      binaerzahl_auf_LED(band.overflow,20,SCHWARZ,ROT,6,0);//Ueberlauf auf 4.letzter Reihe anzeigen
 else binaerzahl_auf_LED(band.benutzte_bandlaenge(),20,SCHWARZ,GELB,6,0);//Benutzte Bandlaenge anzeigen
 cursorx=0; cursory=6;
 cursor_zeichnen();
}
