/* led_treiber.c
Dies ist ein Treiber fuer Text-Ausgabe auf dem ledmatrix-Modul mit 200 LEDs, also 10*20 Pixeln

Version 0.1   letzte Aenderungen: 29.8.2017

Autor:     Rolf Pfister
Homepage:  www.rolfp.ch
Copyright: Opensource Freeware

Installation:
-------------
- led_treiber.h   Editieren und Anpassungen ans Zielsystem machen (vorlaeufig nicht noetig, da nur 1 System)
- led_treiber.c und led_treiber.h  an gewuenschte Stelle kopieren

Oder statt umkopieren vielleicht sowas machen:
> ln -s /gesamter_Pfad/led_treiber.c led_treiber.c
> ln -s /gesamter_Pfad/led_treiber.h led_treiber.h

Benutzung im Hauptprogramm:
---------------------------
#include "led_treiber.h"    //Wenn led_treiber seperat compiliert werden soll
#include "led_treiber.c"    //Wenn led_treiber gleich mit dem Hauptprogramm compiliert werden soll

Beispiele wie es jeweils compiliert wird sind im makefile vom bas_treiber zu finden.

History:
--------
29.8.2017  0.1 Erstellung aus bas_treiber.c (Rolf Pfister)

*/
#ifndef F_CPU
#define F_CPU 20000000UL
#endif

#include "led_treiber.h"

#ifndef LED_TREIBER_C
#define LED_TREIBER_C

//------------------------------------------------------------------------------
//------------------ Globale Variablen im Treiber ------------------------------
static uint8_t cursor_x=0; //aktuelle Cursor-Position
static uint8_t cursor_y=0; //aktuelle Cursor-Position

//#ifdef LINIENZEICHNEN
//static int8_t drawmodus=JAM1;
//#endif

#ifdef FONTTABELLE
#ifdef FONT_8x5
static uint8_t text_zeilenabstand=10; //Zeilenabstand fuer Text
#else
static uint8_t text_zeilenabstand=5; //Zeilenabstand fuer kleinen Font
#endif
void setze_zeilenabstand(uint8_t n) {text_zeilenabstand=n;}
#endif //FONTTABELLE

#if BILDPUFFER>=1
//wahrscheinlich nicht benoetigt, sonst siehe bas_treiber.c
/*
Aufbau des Bildspeichers bei BAS-Grafik:
 Das erste Bit des ersten Bytes (Bit 0, LSB) ist die linke obere Ecke.
 Das ganze erste Byte entspricht den erste 8 Pixel in der ersten Reihe des Bildes.
 Das naechste Byte entspricht dann den naechsten 8 Pixeln der Zeile.
 Dabei ist jeweils Bit 0 das erste Pixel und Bit 7 das letzte Pixel.
 Pro Reihe gibt es 40 Bytes (BAS_MAXX/8).
 Im Ganzen gibt es 256 Reihen (BAS_MAXY). Bei Zaehlung von 0 bis 255 geht das
 gerade noch mit dem Typ uint8_t.

Der Bildpuffer bei der BAS-Grafik ist in bas_routinen.c definiert:
static uint8_t grafikspeicher[320/8*256];
*/
//#endif
//#if BILDPUFFER>=2
//static uchar bildpuffer2[LED_MAXJ]; //(provisorisch: noch nicht benutzt)
//static uint16_t j2=0;
#endif

static uint8_t grafikspeicher[(LED_MAXX+7)/8*LED_MAXY];

//-----------------------------------------------------------------------------
//------------------------- Grundfunktionen -----------------------------------

void ledSendData(uchar c) // Datenbyte senden, will heissen Daten an aktueller Cursorposition speichern
{
 uint16_t j=cursor_y;
 j *= (LED_MAXX+7)/8;
 j += cursor_x/8;
 grafikspeicher[j]=c;
}

inline void ledSendDataCur(uchar c) // Datenbyte senden und Cursor erhoehen
{
 ledSendData(c);
 cursor_x += 8;
 if(cursor_x>=LED_MAXX) cursor_x=0;
}

uchar ledReadData()        // Liest ein Byte an Cursor-Position aus
{
 uint16_t j=cursor_y;
 j *= (LED_MAXX+7)/8;
 j += cursor_x/8;
 return grafikspeicher[j];
}

//void ledSendCmd(uchar c)  // Commando senden
//{
//}

//-----------------------------------------------------------------------------
//--------------- Funktionen fuer alle Grafik-Anzeigen ------------------------

void led_cursor(uint8_t x,uint8_t y)  // Setzen des Cursors
{
  cursor_x=x;
  cursor_y=y;  //aktuelle Cursor-Position speichern
}

void led_clear()      // Loeschen der Grafik
{
 uint16_t j;
 for(j=0;j<((LED_MAXX+7)/8*LED_MAXY);j++)
  {
   grafikspeicher[j]=0;
  }
 led_cursor(0,0);
}

//-----------------------------------------------------------------------------
//--------------------- Hoehere Funktionen ------------------------------------
#ifdef LINIENZEICHNEN
//Wahrscheinlich nicht benoetigt. Sonst von bas_treiber.c kopieren
#endif

#ifdef FIGURENFUELLEN
//Wahrscheinlich nicht benoetigt. Sonst von bas_treiber.c kopieren
#endif

#ifdef FONTTABELLE

#include <avr/pgmspace.h>  //Definitionen damit die Fonts im Flash bleiben
//#undef PROGMEM
//#define PROGMEM __attribute__((section(".progmem.data")))

#ifdef FONT_8x5
#define FONTBREITE 5
#define FONTHOEHE  8

/************************************************************************/
/* Filename: font90_8x5                                                 */
/* Version : 1.00                                                       */
/* Date    : 08.02.2010                                                 */
/* Owner   : UB                                                         */
/* leicht abgeaendert: RP                                               */
/*----------------------------------------------------------------------*/
/* Remark  : Font-File zum BAS-Modul                                    */
/*           Schriftgroesse : 8x5 bit                                   */
/*                            wobei ein Zeichen aus 5x7 Pixel besteht   */
/*----------------------------------------------------------------------*/
/*                                                                      */
/*  z.B. Ascii 82 = R -> 82,0x7F,0x09,0x19,0x29,0x46,                   */
/*                                                                      */
/*                  (linke obere Ecke = LSB                             */
/*                                                                      */
/*                       12345 (Spalte)                                 */
/*                                                                      */
/*                  01h  ####:         = 0x7F (Spalte-1)                */
/*                  02h  #:::#         = 0x09 (Spalte-2)                */
/*                  04h  #:::#         = 0x19 (Spalte-3)                */
/*                  08h  ####:         = 0x29 (Spalte-4)                */
/*                  10h  #:#::         = 0x46 (Spalte-5)                */
/*                  20h  #::#:                                          */
/*                  40h  #:::#                                          */
/*                  80h  :::::                                          */
/*                                                                      */
/************************************************************************/

typedef struct {              // def von Font 8x5
//char ascii;                 // ascii-nr des buchstabens
  uchar b0;
  uchar b1;
  uchar b2;
  uchar b3;
  uchar b4;
} Font8x5_t;

const Font8x5_t font8x5[] PROGMEM = {
  {/*32,*/0x00,0x00,0x00,0x00,0x00}, // Space
  {/*33,*/0x00,0x00,0x4F,0x00,0x00}, // !
  {/*34,*/0x00,0x07,0x00,0x07,0x00}, // "
  {/*35,*/0x14,0x7F,0x14,0x7F,0x14}, // #
  {/*36,*/0x24,0x2A,0x7F,0x2A,0x12}, // $
  {/*37,*/0x63,0x13,0x08,0x64,0x63}, // %
  {/*38,*/0x30,0x4E,0x5D,0x26,0x50}, // &
  {/*39,*/0x04,0x02,0x01,0x00,0x00}, // '
  {/*40,*/0x00,0x1C,0x22,0x41,0x00}, // (
  {/*41,*/0x00,0x41,0x22,0x1C,0x00}, // )
  {/*42,*/0x14,0x08,0x3E,0x08,0x14}, // *
  {/*43,*/0x08,0x08,0x3E,0x08,0x08}, // +
  {/*44,*/0xC0,0x30,0x00,0x00,0x00}, // ,
  {/*45,*/0x08,0x08,0x08,0x08,0x08}, // -
  {/*46,*/0x00,0x60,0x60,0x00,0x00}, // .
  {/*47,*/0x60,0x10,0x08,0x04,0x03}, // /
  {/*48,*/0x3E,0x41,0x41,0x3E,0x00}, // 0
  {/*49,*/0x00,0x42,0x7F,0x40,0x00}, // 1
  {/*50,*/0x72,0x49,0x49,0x49,0x46}, // 2
  {/*51,*/0x21,0x41,0x49,0x4D,0x33}, // 3
  {/*52,*/0x18,0x14,0x12,0x7F,0x10}, // 4
  {/*53,*/0x47,0x45,0x45,0x45,0x39}, // 5
  {/*54,*/0x38,0x4C,0x4A,0x49,0x30}, // 6
  {/*55,*/0x01,0x71,0x09,0x05,0x03}, // 7
  {/*56,*/0x36,0x49,0x49,0x49,0x36}, // 8
  {/*57,*/0x06,0x49,0x29,0x19,0x0E}, // 9
  {/*58,*/0x00,0x00,0x44,0x00,0x00}, // :
  {/*59,*/0x40,0x34,0x00,0x00,0x00}, // ;
  {/*60,*/0x08,0x14,0x22,0x41,0x41}, // <
  {/*61,*/0x14,0x14,0x14,0x14,0x14}, // =
  {/*62,*/0x41,0x41,0x22,0x14,0x08}, // >
  {/*63,*/0x02,0x01,0x51,0x09,0x06}, // ?
  {/*64,*/0x3E,0x41,0x5D,0x55,0x5E}, // @
  {/*65,*/0x7E,0x09,0x09,0x09,0x7E}, // A
  {/*66,*/0x7F,0x49,0x49,0x49,0x36}, // B
  {/*67,*/0x3E,0x41,0x41,0x41,0x22}, // C
  {/*68,*/0x7F,0x41,0x41,0x22,0x1C}, // D
  {/*69,*/0x7F,0x49,0x49,0x49,0x41}, // E
  {/*70,*/0x7F,0x09,0x09,0x09,0x01}, // F
  {/*71,*/0x3E,0x41,0x41,0x49,0x7A}, // G
  {/*72,*/0x7F,0x08,0x08,0x08,0x7F}, // H
  {/*73,*/0x00,0x41,0x7F,0x41,0x00}, // I
  {/*74,*/0x20,0x40,0x41,0x3F,0x01}, // J
  {/*75,*/0x7F,0x08,0x14,0x22,0x41}, // K
  {/*76,*/0x7F,0x40,0x40,0x40,0x40}, // L
  {/*77,*/0x7F,0x02,0x0C,0x02,0x7F}, // M
  {/*78,*/0x7F,0x06,0x08,0x30,0x7F}, // N
  {/*79,*/0x3E,0x41,0x41,0x41,0x3E}, // O
  {/*80,*/0x7F,0x09,0x09,0x09,0x06}, // P
  {/*81,*/0x3E,0x41,0x51,0x21,0x5E}, // Q
  {/*82,*/0x7F,0x09,0x19,0x29,0x46}, // R
  {/*83,*/0x46,0x49,0x49,0x49,0x31}, // S
  {/*84,*/0x01,0x01,0x7F,0x01,0x01}, // T
  {/*85,*/0x3F,0x40,0x40,0x40,0x3F}, // U
  {/*86,*/0x1F,0x20,0x40,0x20,0x1F}, // V
  {/*87,*/0x7F,0x20,0x18,0x20,0x7F}, // W
  {/*88,*/0x63,0x14,0x08,0x14,0x63}, // X
  {/*89,*/0x07,0x08,0x78,0x08,0x07}, // Y
  {/*90,*/0x61,0x51,0x49,0x45,0x43}, // Z
  {/*91,*/0x7F,0x41,0x41,0x00,0x00}, // [
  {/*92,*/0x02,0x04,0x08,0x10,0x20}, // \ (Backslash)
  {/*93,*/0x00,0x00,0x41,0x41,0x7F}, // ]
  {/*94,*/0x04,0x02,0x01,0x02,0x04}, // ^
  {/*95,*/0x80,0x80,0x80,0x80,0x80}, // _
  {/*96,*/0x00,0x01,0x02,0x04,0x00}, // `
  {/*97,*/0x38,0x44,0x44,0x28,0x7C}, // a
  {/*98,*/0x7F,0x28,0x44,0x44,0x38}, // b
  {/*99,*/0x38,0x44,0x44,0x44,0x00}, // c
 {/*100,*/0x38,0x44,0x44,0x28,0x7F}, // d
 {/*101,*/0x38,0x54,0x54,0x54,0x18}, // e
 {/*102,*/0x00,0x08,0x7E,0x09,0x00}, // f
 {/*103,*/0x18,0xA4,0xA4,0x98,0x7C}, // g
 {/*104,*/0x7F,0x08,0x04,0x04,0x78}, // h
 {/*105,*/0x00,0x44,0x7D,0x40,0x00}, // i
 {/*106,*/0x40,0x80,0x80,0x7D,0x00}, // j
 {/*107,*/0x7F,0x10,0x18,0x24,0x42}, // k
 {/*108,*/0x00,0x41,0x7F,0x40,0x00}, // l
 {/*109,*/0x7C,0x04,0x78,0x04,0x78}, // m
 {/*110,*/0x7C,0x04,0x04,0x78,0x00}, // n
 {/*111,*/0x38,0x44,0x44,0x44,0x38}, // o
 {/*112,*/0xFC,0x18,0x24,0x24,0x18}, // p
 {/*113,*/0x18,0x24,0x24,0x18,0xFC}, // q
 {/*114,*/0x7C,0x08,0x04,0x04,0x04}, // r
 {/*115,*/0x48,0x54,0x54,0x54,0x24}, // s
 {/*116,*/0x00,0x04,0x3F,0x44,0x00}, // t
 {/*117,*/0x3C,0x40,0x40,0x40,0x7C}, // u
 {/*118,*/0x0C,0x30,0x40,0x30,0x0C}, // v
 {/*119,*/0x3C,0x40,0x20,0x40,0x3C}, // w
 {/*120,*/0x44,0x28,0x10,0x28,0x44}, // x
 {/*121,*/0x1C,0xA0,0xA0,0xA0,0x7C}, // y
 {/*122,*/0x44,0x64,0x54,0x4C,0x44}, // z
 {/*123,*/0x08,0x36,0x41,0x00,0x00}, // {
 {/*124,*/0x00,0x00,0x77,0x00,0x00}, // |
 {/*125,*/0x00,0x00,0x41,0x36,0x08}, // }
 {/*126,*/0x10,0x08,0x08,0x08,0x04}, // ~
#define VIERECKZEICHEN 127
 {/*127,*/0xFF,0x81,0x81,0x81,0xFF}, // Viereck fuer nicht darstellbare Zeichen

#ifdef UTF8ZEICHEN
#define GRADZEICHEN 128
 {0x0E,0x11,0x11,0x11,0x0E}, // ^o
#define AUMLAUT 129
 {0x78,0x15,0x14,0x15,0x78}, // "A
#define OUMLAUT 130
 {0x7C,0x45,0x44,0x45,0x7C}, // "O
#define UUMLAUT 131
 {0x3C,0x41,0x40,0x41,0x3C}, // "U
#define aUMLAUT 132
 {0x38,0x45,0x44,0x29,0x7C}, // "a
#define oUMLAUT 133
 {0x38,0x45,0x44,0x45,0x38}, // "o
#define uUMLAUT 134
 {0x3C,0x41,0x40,0x41,0x7C}, // "u
#define SHARPSS 135
 {0x7E,0x01,0x49,0x4C,0x32}, // Deusches Scharfes S
#define EUROZEICHEN 136
 {0x3E,0x55,0x55,0x41,0x22}, // Eurozeichen
//weitere Zeichen definierbar bis maximal 225
#endif

};
const uint8_t ANZAHLZEICHEN = sizeof(font8x5)/sizeof(Font8x5_t);  // Anzahl der Zeichen
#endif //Ende FONT_8x5

#ifdef FONT_5x3
#define FONTBREITE 3
#define FONTHOEHE  5

/*************************************************************/
/* Remark  : Font-File fuer LED-Matrix                       */
/*           Schriftgroesse : 5x3 bit                        */
/*-----------------------------------------------------------*/
/*  Fonttyp 11                                               */
/*  z.B. Ascii 54 = 6 -> 54,0x1F,0x15,0x1D,                  */
/*                                                           */
/*                  (linke obere Ecke = LSB                  */
/*                                                           */
/*                       123 (Spalte)                        */
/*                                                           */
/*                  08h  ###         = 0xF8 (Spalte-1)       */
/*                  10h  #::         = 0xA8 (Spalte-2)       */
/*                  20h  ###         = 0xE8 (Spalte-3)       */
/*                  40h  #:#                                 */
/*                  80h  ###                                 */
/*                                                           */
/*************************************************************/

typedef struct {  // def von Font 5x3
//char ascii;     // ascii-nr des buchstabens
  uchar b0;
  uchar b1;
  uchar b2;
} Font5x3_t;

const Font5x3_t font5x3[] PROGMEM = {
  {/*32,*/0x00,0x00,0x00}, // Space
  {0x00,0xB8,0x00},                                   // 0x21 '!'
  {0x18,0x00,0x18},                                   // 0x22 '"'
  {0xF8,0x50,0xF8},                                   // 0x23 '#'
  {0xC8,0xF8,0x98},                                   // 0x24 '$'
  {0xC8,0x20,0x98},                                   // 0x25 '%'
  {0xC0,0x80,0xC0},                                   // 0x26 '&'
  {0x00,0x18,0x00},                                   // 0x27 '''
  {0x70,0x88,0x00},                                   // 0x28 '('
  {0x00,0x88,0x70},                                   // 0x29 ')'
  {0x50,0x20,0x50},                                   // 0x2A '*'
  {0x20,0x70,0x20},                                   // 0x2B '+'
  {0x00,0xC0,0x00},                                   // 0x2C ','
  {0x20,0x20,0x20},                                   // 0x2D '-'
  {0x00,0x80,0x00},                                   // 0x2E '.'
  {0x40,0x20,0x10},                                   // 0x2F '/'
  
  {0xF8,0x88,0xF8},                                   // 0x30 '0'
  {0x10,0xF8,0x00},                                   // 0x31 '1'
  {0xE8,0xA8,0xB8},                                   // 0x32 '2'
  {0xA8,0xA8,0xF8},                                   // 0x33 '3'
  {0x38,0x20,0xF8},                                   // 0x34 '4'
  {0xB8,0xA8,0xE8},                                   // 0x35 '5'
  {0xF8,0xA8,0xE8},                                   // 0x36 '6'
  {0x08,0x08,0xF8},                                   // 0x37 '7'
  {0xF8,0xA8,0xF8},                                   // 0x38 '8'
  {0xB8,0xA8,0xF8},                                   // 0x39 '9'

  {0x00,0x50,0x00},                                   // 0x3A ':'
  {0x00,0xD0,0x00},                                   // 0x3B ';'
  {0x20,0x50,0x88},                                   // 0x3C '<'
  {0x50,0x50,0x50},                                   // 0x3D '='
  {0x88,0x50,0x20},                                   // 0x3E '>'
  {0x08,0xA8,0x18},                                   // 0x3F '?'
  {0x70,0x50,0x70},                                   // 0x40 '@'
  {0xF0,0x28,0xF0},                                   // 0x41 'A'
  {0xF8,0xA8,0x70},                                   // 0x42 'B'
  {0xF8,0x88,0x88},                                   // 0x43 'C'
  {0xF8,0x88,0x70},                                   // 0x44 'D'
  {0xF8,0xA8,0xA8},                                   // 0x45 'E'
  {0xF8,0x28,0x28},                                   // 0x46 'F'
  {0xF8,0x88,0xC8},                                   // 0x47 'G'
  {0xF8,0x20,0xF8},                                   // 0x48 'H'
  {0x88,0xF8,0x88},                                   // 0x49 'I'
  {0x40,0x80,0xF8},                                   // 0x4A 'J'
  {0xF8,0x30,0xC8},                                   // 0x4B 'K'
  {0xF8,0x80,0x80},                                   // 0x4C 'L'
  {0xF8,0x10,0xF8},                                   // 0x4D 'M'
  {0xF0,0x20,0xE0},                                   // 0x4E 'N'
  {0xF8,0x88,0xF8},                                   // 0x4F 'O'
  {0xF8,0x28,0x38},                                   // 0x50 'P'
  {0x38,0x28,0xF8},                                   // 0x51 'Q'
  {0xF8,0x68,0xB8},                                   // 0x52 'R'
  {0xB8,0xA8,0xE8},                                   // 0x53 'S'
  {0x08,0xF8,0x08},                                   // 0x54 'T'
  {0xF8,0x80,0xF8},                                   // 0x55 'U'
  {0x78,0x80,0x78},                                   // 0x56 'V'
  {0xF8,0x40,0xF8},                                   // 0x57 'W'
  {0xD8,0x20,0xD8},                                   // 0x58 'X'
  {0x38,0xE0,0x38},                                   // 0x59 'Y'
  {0xE8,0xA8,0xB8},                                   // 0x5A 'Z'
  {0xF8,0x88,0x00},                                   // 0x5B '['
  {0x10,0x20,0x40},                                   // 0x5C '\'
  {0x00,0x88,0xF8},                                   // 0x5D ']'
  {0x10,0x08,0x10},                                   // 0x5E '^'
  {0x80,0x80,0x80},                                   // 0x5F '_'
  {0x00,0x08,0x10},                                   // 0x60 '`'
  {0xE0,0x50,0xE0},                                   // 0x61 'a'
  {0xF8,0xA0,0xE0},                                   // 0x62 'b'
  {0xE0,0xA0,0xA0},                                   // 0x63 'c'
  {0xE0,0xA0,0xF8},                                   // 0x64 'd'
  {0xF0,0xB0,0xB0},                                   // 0x65 'e'
  {0xF0,0x48,0x10},                                   // 0x66 'f'
  {0xB8,0xA8,0xF8},                                   // 0x67 'g'
  {0xF8,0x20,0xE0},                                   // 0x68 'h'
  {0x00,0xE8,0x00},                                   // 0x69 'i'
  {0x80,0xE8,0x00},                                   // 0x6A 'j'
  {0xF8,0x40,0xA0},                                   // 0x6B 'k'
  {0x00,0xF8,0x00},                                   // 0x6C 'l'
  {0xE0,0x20,0xE0},                                   // 0x6D 'm'
  {0xF0,0x20,0xE0},                                   // 0x6E 'n'
  {0xE0,0xA0,0xE0},                                   // 0x6F 'o'
  {0xF0,0x50,0x70},                                   // 0x70 'p'
  {0x70,0x50,0xF0},                                   // 0x71 'q'
  {0xE0,0x20,0x20},                                   // 0x72 'r'
  {0xB0,0xD0,0xD0},                                   // 0x73 's'
  {0x10,0xF8,0x10},                                   // 0x74 't'
  {0xE0,0x80,0xE0},                                   // 0x75 'u'
  {0x60,0x80,0x60},                                   // 0x76 'v'
  {0xF0,0x40,0xF0},                                   // 0x77 'w'
  {0xA0,0x40,0xA0},                                   // 0x78 'x'
  {0xB0,0xC0,0x30},                                   // 0x79 'y'
  {0xD0,0xD0,0xB0},                                   // 0x7A 'z'
  {0x20,0x70,0x88},                                   // 0x7B '{'
  {0x00,0xD8,0x00},                                   // 0x7C '|'
  {0x88,0x70,0x20},                                   // 0x7D '}'
  {0x08,0x10,0x08},                                   // 0x7E '~'

#define VIERECKZEICHEN 127
 {/*127,*/0x1F,0x1F,0x1F}, // Viereck fuer nicht darstellbare Zeichen

};
const uint8_t ANZAHLZEICHEN = sizeof(font5x3)/sizeof(Font5x3_t);  // Anzahl der Zeichen
#endif //Ende FONT_5x3

//------------------------------------------------------------------
void led_printc(uchar c) // schreibe ein Zeichen erhoehe Cursor
{                        // (muss uchar sein weil auch Nicht-Ascii-Zeichen unterstuetzt werden sollen)
 uint8_t n;
#ifdef UTF8ZEICHEN
 static uchar u1=0,u2=0; //zum Zwischenspeichern der ersten 2 Bytes eines UTF8-Zeichens
#endif

 if(c<' ') return; //Steuerzeichen ignorieren
 n = c-' ';

#ifdef UTF8ZEICHEN
 if(u1!=0 || n>=ANZAHLZEICHEN)
   {
    if(u1==0)        //ist es erstes Byte eines UTF8-Zeichens?
     {u1=c; return;} //ja: speichern und return
    if(u1==0xC2)
     {
      if(c==0xB0) n=GRADZEICHEN;
      //else if... //hier weitere 2-Byte UTF8-Zeichen definierbar
      else n=VIERECKZEICHEN; //Viereck fuer unbekanntes Zeichen
     }
    else if(u1==0xC3)
     {
      if(c==0x84)      n=AUMLAUT;
      else if(c==0x96) n=OUMLAUT;
      else if(c==0x9C) n=UUMLAUT;
      else if(c==0xA4) n=aUMLAUT;
      else if(c==0xB6) n=oUMLAUT;
      else if(c==0xBC) n=uUMLAUT;
      else if(c==0x9F) n=SHARPSS;
      //else if... //hier weitere 2-Byte UTF8-Zeichen definierbar
      else n=VIERECKZEICHEN;
     }
    else if(u1==0xE2)
     {
      if(u2==0)        //ist es zweites Byte eines 3-Byte UTF8-Zeichens?
       {u2=c; return;} //ja: speichern und return
      if(u2==0x82)
       {
	if(c==0xAC) n=EUROZEICHEN;
	//else if... //hier weitere 3-Byte UTF8-Zeichen definierbar
	else n=VIERECKZEICHEN; //unbekanntes Zeichen
       }
      //else if(u2==0x... //hier waeren weitere 3-Byte UTF8-Zeichen mit anderem 2.Byte definierbar
      else n=VIERECKZEICHEN;
     }
    else  //andernfalls unbekanntes Zeichen
     {
      n=VIERECKZEICHEN;
     }
    u1=u2=0; //beide ruecksetzen fuers naechste Zeichen
    if(n==VIERECKZEICHEN         //Wenn ein ungueltiges UTF8-Zeichen erkannt wurde,
       && c-' '<ANZAHLZEICHEN)   //aber das aktuelle Zeichen ein gueltiges Zeichen ist,
                     {
		      n = c;     //dann dieses Zeichen uebernehmen. (und das ungueltige UTF8 ignorieren)
		     }
    n -= ' ';
   }

#else

 if(n >= ANZAHLZEICHEN)
   {
    n=VIERECKZEICHEN-' '; //Viereck fuer nicht darstellbare Zeichen
   }

#endif

 /* ohne PROGMEM:
 led_printb(font8x5[n].b0);
 led_printb(font8x5[n].b1);
 led_printb(font8x5[n].b2);
 led_printb(font8x5[n].b3);
 led_printb(font8x5[n].b4);
 */
#ifdef FONT_8x5
 led_printb_progmem(&font8x5[n].b0); //mit PROGMEM
#else
 led_printb_progmem(&font5x3[n].b0); //mit PROGMEM
#endif
 led_printb(0x00);

 //led_refresh(); //erst in led_print() machen
}

void led_printb_progmem(const uchar *b0)
{
 uint8_t i;
 unsigned long addr;

 // der Zeiger ist nur 16 Bit gross! (sizeof(const uchar *) ergibt 2 Bytes)
 // int ist 16Bit, long ist 32Bit
 addr = (unsigned long)(unsigned int)b0; //wenn Adresse unterhalb 64KB
 //addr = (unsigned long)(unsigned int)b0 | 0x10000L; //wenn Adresse oberhalb 64KB (auf Atmega1284P)
 //addr = (long)b0 & 0xFFFF; if(addr&0x8000) addr |= 0x10000L; //sollte so gehen wenn Adresse in der Naehe der 64KB-Grenze ist
 
 //fuer Atmega2560 muesste es so gehen:
 //addr = (unsigned long)(unsigned int)b0 | 0x10000L; //wenn Adresse zwischen 64KB und 128KB
 //addr = (unsigned long)(unsigned int)b0 | 0x20000L; //wenn Adresse zwischen 128KB und 192KB
 //addr = (unsigned long)(unsigned int)b0 | 0x30000L; //wenn Adresse oberhalb 192KB
 
 for(i=0;i<FONTBREITE;i++) //5 Bytes fuer Font8x5
   led_printb(__ELPM_classic__(addr++));
}

#if(GROSSERFONT<2)
//sollte hier immer 1 sein (also kleinstmoeglicher Font)

void led_printb(uchar c) // Schreibe ein Byte eines Buchstabens
{
 // beispiel font8x5[L].b0...b4={0x7F,0x40,0x40,0x40,0x40}, // L
 // beispiel font5x3[L].b0...b2={0xF8,0x80,0x80}, // 'L'
 uint8_t j, c2, altes_y=cursor_y, maske = 1<<(cursor_x&7);
 if(FONTHOEHE<8)
   c >>= (8-FONTHOEHE);
 for(j=0;j<FONTHOEHE;j++)
  {
   c2=ledReadData();
   if((c&0x01)==0) c2 &= ~maske; //Punkt loeschen
   else            c2 |= maske; //Punkt setzen
   ledSendData(c2);
   c >>= 1;
   cursor_y++;
  }
 cursor_x++;
 cursor_y=altes_y;
}

#else //GROSSERFONT
//macht hier keinen Sinn. Sonst von bas_treiber.c kopieren
#endif //GROSSERFONT

void led_print(const char *ptr) // zum schreiben von Text ab Cursor
{
 char c;
 uchar jstart=cursor_y;
 
 while((c= *ptr++)!=0)
  {
   // Sonderzeichen behandeln:
   if(c=='\t') c=' '; //Tabulator durch Leerzeichen ersetzen
   else if(c=='\r')
    {
     if(*ptr=='\n')  c = *ptr++;  //CR LF durch LF ersezen (LF = Zeilentrenner)
     else c = '\n';  //einzelnes CR auch als Zeilentrenner behandeln
    }

   // Zeilentrenner behandeln:
   if(c=='\n' || cursor_x >= LED_MAXX-FONTBREITE)  //Zeilentrennzeichen, oder Zeile voll?
      {
       led_cursor(0,cursor_y+text_zeilenabstand); //ja: neue Zeile
      }

   led_printc(c);
  }

 led_refresh(jstart);
}

//Ende #ifdef FONTTABELLE
#endif

void gesamtes_ledfeld_setzen(uchar j,uchar i,uchar wert); //muss im Hauptprogramm programmiert werden
                                                          //oder direkt led_refresh() im Hauptprogramm
static uchar led_farbe=7; //WEISS
void led_setfontcolor(uchar c) {led_farbe=c;} // Farbe zum Text drucken setzen (default ist 7=WEISS)
uchar led_getfontcolor() {return led_farbe;}  // eingestellte Farbe lesen 

void led_refresh(uchar jstart)
{
 uchar i,j,c,farbe; //i=spalte j=reihe
 uint8_t *pj;
 for(j=jstart;j<10;j++)
  {
   pj = &grafikspeicher[(LED_MAXX+7)/8*j];
   for(i=0;i<20;i++)
    {
     c = pj[i/8];
     if((c & (1<<(i%8)))==0) farbe=0; else farbe=led_farbe;
     gesamtes_ledfeld_setzen(j,i,farbe);
    }
  }
}

//Ende #ifndef LED_TREIBER_C
#endif
