/* nimm.cc       letzte Aenderung: 2.8.2014
Erstes Spielprogramm fuer LED-Matrix mit 4 Modulen
*/
#define VERSION "0.0"
/*
 Prozessor: Atmega1284P, 20MHz Quarz
 Schaltung: ledmatrix.png
            4 Stueck so verschaltet:
	      jeweils mit allen verbunden: GND, +5V, MISO, TAKT
	      IN1 ist jeweils OUT von Rechts, IN2 ist OUT von Oben

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 2.8.2014	Erstellung aus test2.cc
 3.8.           erweitert auf mehr als 4 Reihen
26.8.2017       Kommentare fuer Tasten eingefuegt

*/
#define F_CPU 20000000UL
#define ANZAHL_REIHEN 10  //4=klassische Variante, maximal 10 erlaubt
#define UART_BASIRGB  //Darstellung auch auf Fernseher

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
//#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>

//#define DEBUG  //zur Fehlersuche

/**** Vordeklarationen ****/
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t
void basi_viereck(uchar j,uchar i,uchar farbe);

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=0;
static volatile uchar tage=20,monat=7; //aktuelles Datum
static volatile uint jahr=2014;
static char monatstage[13]={31,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=6; //0=Montag ... 6=Sonntag
static volatile char sommerzeit=1; //0=Winterzeit 1=Sommerzeit

void schaltjahrtest()
{
 if(jahr%4==0 && (jahr%100!=0 || jahr%400==0)) monatstage[2]=29;
 else monatstage[2]=28;
}
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
void schaltjahr_und_wochentagberechnen()
{
 schaltjahrtest();
 wochentagberechnen();
 //variable_feiertage_berechnen();
}

/** Feiertage: ** /
const char NFEST=6;//Anzahl feste Feiertage
const char festfeiertag[NFEST]={1, 2, 1, 1, 25, 26};
const char festfeiermon[NFEST]={1, 1, 5, 8, 12, 12};
const char NVARI=4;//Anzahl von Ostern abhaengige Feiertage
static char varifeiertag[NVARI];
static char varifeiermon[NVARI];
void variable_feiertage_berechnen()
{
 int a,b,c,k,p,q,M,N;
 char d,e,tag,i;
 const char mon=3;
 //Osterformel nach Gauss:
 a=jahr%19;
 b=jahr%4;
 c=jahr%7;
 k=jahr/100;
 p=(8*k+13)/25;
 q=k/4;
 M=(15+k-p-q)%30;
 N=(4+k-q)%7;
 d=(19*a+M)%30;
 e=(2*b+4*c+6*d+N)%7;
 tag=22+d+e;
 varifeiertag[0]=tag+1; varifeiermon[0]=mon; //Ostermontag
 varifeiertag[1]=tag-2; varifeiermon[1]=mon; //Karfreitag
 varifeiertag[2]=tag+39; varifeiermon[2]=mon; //Auffahrt
 varifeiertag[3]=tag+50; varifeiermon[3]=mon; //Pfingstmontag
 for(i=0;i<NVARI;i++)
   while(varifeiertag[i]>monatstage[varifeiermon[i]])
    {varifeiertag[i] -= monatstage[varifeiermon[i]]; varifeiermon[i]++;}
}

bool istfeiertag()
{
 char i;
 for(i=0;i<NFEST;i++)
   if(tage==festfeiertag[i] && monat==festfeiermon[i]) return true;
 for(i=0;i<NVARI;i++)
   if(tage==varifeiertag[i] && monat==varifeiermon[i]) return true;
 return false;
}
/ **/

// Quarz moeglichst genau ausmessen und hier eintragen:
const ulong Hz=20000000; //exakt 20MHz

#define Rest1 (Hz%1000) //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (Hz/1000-1)

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
}

ISR(TIMER1_COMPA_vect)
{
 uchar mehr;
 uint ms=millisec;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0; mehr=0;
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
      }
    }
  }
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
 //tastaturcheck();
}

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) ;//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
   ;//auf n. Millisekunde warten
}

void secwait(uchar n) //n Sekunden warten
{
 while(n!=0) {milliwait(1000); --n;}
}

void microwait(uchar n) //mindestens n Microsekunden warten
{
 while(--n!=0) _delay_us(1);
}

/***************************** Timer0 fuer LED-Anzeige ********************************/
static volatile uchar ledfeld[5][10]; // 5 Reihen, 10 Spalten
void ledfeld_init(uchar n)
{
 uchar i,j;
 for(i=0;i<5;i++)
  for(j=0;j<10;j++)
    ledfeld[i][j]=n; //alle gleich setzen
}

void timer0_init()
{
 TCCR0A = (0<<WGM01)|(0<<WGM00); //Normal-Modus, Interrupt bei Ueberlauf (8 Bit, also Teiler durch 256)
 //TCCR0A = (1<<WGM01)|(0<<WGM00); //CTC-Modus, Interrupt bei Erreichen von OCRA
 //TCCR0B = (1<<CS00); //Takt nicht teilen
 TCCR0B = (1<<CS01); //Takt durch 8teilen
 TCNT0 = 0;
 TIMSK0 = (1<<TOIE0); //Interrupt bei Ueberlauf
 //TIMSK0 = (1<<OCIE0A); //Interrupt bei Erreichen von OCRA
}

ISR(TIMER0_OVF_vect)
{
 static uchar spalte=0;
 uchar b,c,d,tmp2,tmp3;
 bool flag;
 tmp2 = (~ledfeld[2][spalte])&7;
 tmp3 = (~ledfeld[3][spalte])<<3;
 //PORTB |= 0x0F; PORTC |= 0x7F; PORTD = 0xFF;//test
 b = ((~ledfeld[4][spalte])&7) | (tmp3&0x08);
 c = (((~ledfeld[0][spalte])&7)<<4) | (((~ledfeld[1][spalte])&7)<<1) | (tmp2>>2);
 d = ((tmp2&3)<<6) | (tmp3&0x30);
 flag = (b!=0x0F) || (c!=0x7F) || (d!=0xF0);
 PORTD |= 3; PORTA = 0xFF; //alle Transistoren aus
 PORTB = (PORTB&0xF0) | b;
 PORTC = (PORTC&0x80) | c;
 PORTD = 0x0F | d;
 if(flag) //nur wenn auch mindestens eine LED an:
  switch(spalte) //entsprechender Transistor einschalten
   {
    case 0: PORTD &= ~(1<<1); break;
    case 1: PORTD &= ~(1<<0); break;
    case 2: PORTA &= ~(1<<7); break;
    case 3: PORTA &= ~(1<<6); break;
    case 4: PORTA &= ~(1<<5); break;
    case 5: PORTA &= ~(1<<4); break;
    case 6: PORTA &= ~(1<<3); break;
    case 7: PORTA &= ~(1<<2); break;
    case 8: PORTA &= ~(1<<1); break;
    case 9: PORTA &= ~(1<<0); break;
   }
 if(++spalte==10) spalte=0;
 //if(spalte==0) spalte=9; else --spalte;//test
}

/************************ Datenuebertragung zwischen den Modulen **********************/
#define MISO_DDR  DDRB
#define MISO_PORT PORTB
#define MISO_PIN  PINB
#define MISOBIT 6
#define TAKT_DDR  DDRB
#define TAKT_PORT PORTB
#define TAKT_PIN  PINB
#define TAKTBIT 7

/* Der Master sendet einen TAKT und auf MISO jeweils 8 Bits, steigende Flanke von
   TAKT soll das Bit in MISO uebernehmen.
   Erstes gesendetes Byte ist Modulnummer, dann 1 Byte j, 1 Byte i, und 1 Byte Datenwert.
 */
bool masterflag=false; //darf nur in erstem Modul gesetzt werden
uchar modulnummer=0; //gueltige Nummern: 1...4, 0=noch nicht gesetzt

void byte_senden(uchar c)
{
 uchar i;
 for(i=0;i<8;c<<=1,i++)
  {
   if(c&0x80) MISO_PORT |= (1<<MISOBIT);
   else       MISO_PORT &= ~(1<<MISOBIT);
   TAKT_PORT &= ~(1<<TAKTBIT);
   microwait(20);
   TAKT_PORT |= (1<<TAKTBIT);
   microwait(20);
  }
}

void daten_senden(uchar nummer,uchar j,uchar i,uchar wert)
{
 byte_senden(nummer);
 byte_senden(j);
 byte_senden(i);
 byte_senden(wert);
}

bool byte_empfangen(uchar& c)
{
 uchar i,bit;
 int timeout=0;
#define TIMEOUT 1000
 c=0;
 for(i=0;i<8;i++)
  {
   while(TAKT_PIN&(1<<TAKTBIT)) //warten bis Low
     {_delay_us(1); if(++timeout>=TIMEOUT) return false;} //Fehler bei Timeout
   while((TAKT_PIN&(1<<TAKTBIT))==0) ;//warten auf High
   bit = (MISO_PIN&(1<<MISOBIT))==0 ? 0 : 1;
   c = (c<<1)+bit;
  }
 return true;
}

void daten_empfangen()
{
 uchar n,i,j,wert;
 if(!byte_empfangen(n)) return; //Rueckkehr bei Fehler
 if(!byte_empfangen(j)) return;
 if(!byte_empfangen(i)) return;
 if(!byte_empfangen(wert)) return;
 if(n==modulnummer)
   ledfeld[j][i]=wert;
}

/******************* alle LED-Felder vom Master aus behandeln *************************/
// Master ist das Modul oben links
static volatile uchar ledfeld2[5][10]; //Modul oben rechts
static volatile uchar ledfeld3[5][10]; //Modul unten links
static volatile uchar ledfeld4[5][10]; //Modul unten rechts
static volatile uchar gesamtfeld[10][20]; //gesamtes LED-Feld (nur von Master benoetigt)

void gesamtes_ledfeld_setzen(uchar j,uchar i,uchar wert) //nur von Master aufrufen
{
 if(j>=5)
  {
   if(i>=10)
    {
     ledfeld4[j-5][i-10]=wert;
     daten_senden(4,j-5,i-10,wert);
    }
   else
    {
     ledfeld3[j-5][i]=wert;
     daten_senden(3,j-5,i,wert);
    }
  }
 else if(i>=10)
  {
   ledfeld2[j][i-10]=wert;
   daten_senden(2,j,i-10,wert);
  }
 else
  ledfeld[j][i]=wert;
#ifdef UART_BASIRGB
 if(wert!=gesamtfeld[j][i])
  {
   basi_viereck(j,i,wert);
   gesamtfeld[j][i]=wert;
  }
#endif
}

uchar gesamtes_ledfeld_lesen(uchar j,uchar i) //nur von Master aufrufen
{
 if(j>=5)
  {
   if(i>=10) return ledfeld4[j-5][i-10];
   else      return ledfeld3[j-5][i];
  }
 else if(i>=10) return ledfeld2[j][i-10];
 return ledfeld[j][i];
}

/*************************** Serielle Schnittstelle ***********************************/
#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 16000000"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 9600UL      // Baudrate
//#define BAUD 1200UL      //test
 
// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart1_init()
{
 UBRR1 = UBRR_VAL;
#ifdef UART_BASIRGB
 UCSR1B = (1<<RXEN1)|(1<<TXEN1); // UART TX und RX einschalten
#else
 UCSR1B = (1<<RXEN1); // nur UART RX einschalten
#endif
 //UCSR1C = (3<<UCSZ10); // Asynchron 8N1
 UCSR1C = (3<<UCSZ10)|(1<<USBS1); // Asynchron 8N2
}

#ifdef UART_BASIRGB
void uart_putc(unsigned char c)
{
 while (!(UCSR1A & (1<<UDRE1))) ;// warten bis Senden moeglich
 UDR1 = c;                      // sende Zeichen
 milliwait(3);//test
}
#endif

unsigned char uart_getc(void)
{
 while (!(UCSR1A & (1<<RXC1))) ;// warten bis Zeichen verfuegbar
 return UDR1; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart_check()
{
 return (UCSR1A & (1<<RXC1)) ;//true wenn Zeichen verfuegbar
}

/********************************** Nimm-Spiel ****************************************/
#define MAXSTEINE (1+2*(ANZAHL_REIHEN-1))
static int steine[ANZAHL_REIHEN];
static int steinfarben[ANZAHL_REIHEN][MAXSTEINE];
#if ANZAHL_REIHEN<=4
 #define STARTPOSY 3
 #define STARTPOSX 5
 #define CURSOR_STARTPOSX 3
#elif ANZAHL_REIHEN<=8
 #define STARTPOSY 1
 #define STARTPOSX 2
 #define CURSOR_STARTPOSX 0
#else
 #define STARTPOSY 0
 #define STARTPOSX 1
 #define CURSOR_STARTPOSX 0
#endif
static uchar cursorx=CURSOR_STARTPOSX, cursory=STARTPOSY;

#ifdef UART_BASIRGB
void basi_screenclear()
{
 uchar j,i;
 uart_putc(0x10); uart_putc(0x00); //Hintergrundfarbe Schwarz
 uart_putc(0x04); //Screenclear auf Fernseher
 for(j=0;j<10;j++)
  for(i=0;i<20;i++)
   gesamtfeld[j][i]=0;
}

void basi_viereck(uchar j,uchar i,uchar farbe)
{
 j *= 8;
 i *= 8;
 uart_putc(0x11); uart_putc(farbe); //Vordergrundfarbe
 uart_putc(0x12);
 uart_putc(i&0xFF); uart_putc(0); //(eigentlich uart_putc(j>>8)) Position vom Viereck links oben
 uart_putc(j&0xFF); uart_putc(0); //Position vom Viereck links oben
 uart_putc(0x16); //Viereck auf Fernseher
 uart_putc((i+6)&0xFF); uart_putc(0); //Position vom Viereck rechts unten
 uart_putc((j+6)&0xFF); uart_putc(0); //Position vom Viereck rechts unten
 //milliwait(10); //test
}
#endif

void steine_init(uchar nreihen)
{
 uchar j,i,n,farbe=7;
 for(j=0,n=1; j<ANZAHL_REIHEN; j++,n+=2)
  {
   steine[j] = (j<nreihen) ? n : 0;
   for(i=0;i<n;i++)
     steinfarben[j][i]=farbe;
   if(++farbe==8) farbe=2; //naechste Farbe (weiss, gruen, gelb, blau, ...)
  }
}

void steine_zeichnen()
{
 uchar i,j,farbe;
 for(j=0;j<ANZAHL_REIHEN;j++)
  {
   for(i=0;i<steine[j];i++)
    {
     farbe=steinfarben[j][i];
     gesamtes_ledfeld_setzen(j+STARTPOSY,i+STARTPOSX+ANZAHL_REIHEN-1-j,farbe);
    }
   gesamtes_ledfeld_setzen(j+STARTPOSY,i+STARTPOSX+ANZAHL_REIHEN-1-j,0); //Feld nach letztem Sein der Reihe auf Dunkel setzen
  }
}

void cursor_zeichnen_loeschen(uchar flag)
{
 static uchar x0=0,y0=0,vorher=0,vorherflag=0;
 if(vorherflag!=0) gesamtes_ledfeld_setzen(y0,x0,vorher);
 vorher=gesamtes_ledfeld_lesen(y0=cursory,x0=cursorx);
 if(flag==2)
   vorherflag=0;
 else
  {
   gesamtes_ledfeld_setzen(cursory,cursorx,1);
   vorherflag=1;
  }
}
void cursor_zeichnen()
{
 static uchar flag=0;
 cursor_zeichnen_loeschen(flag);
 flag=1;
}
void cursor_loeschen()
{
 cursor_zeichnen_loeschen(2);
}

static uchar wegnehm_reihe=0;
static uchar wegnehm_anzahl=0;

void steine_farben_anpassen()
{
 uchar i,j,farbe=7;
 wegnehm_anzahl=0;
 for(j=0;j<ANZAHL_REIHEN;j++)
  {
   for(i=0;i<steine[j];i++)
    {
     if(j+STARTPOSY==cursory && i+STARTPOSX+ANZAHL_REIHEN-1-j <= cursorx)
         {
	  steinfarben[j][i]=1; //rot
	  wegnehm_reihe=j;
	  wegnehm_anzahl++;
	 }
     else steinfarben[j][i]=farbe;
    }
   if(++farbe==8) farbe=2; //naechste Farbe (weiss, gruen, gelb, blau, ...)
  }
 steine_zeichnen();
}

void steine_wegnehmen()
{
 if(wegnehm_anzahl!=0)
  {
   do
    {
     --cursorx;
     --steine[wegnehm_reihe];
     steine_farben_anpassen();
     cursor_loeschen();
     steine_zeichnen();
     cursor_zeichnen();
     milliwait(200);
    }
   while(wegnehm_anzahl!=0);
   while(cursorx>CURSOR_STARTPOSX)
    {
     --cursorx;
     cursor_zeichnen();
     milliwait(200);
    }
  }
}

static uchar wer_ist_am_zug=0;
#define MENSCH 0
#define COMPUTER 1
#define MENSCH_GEWINNT 10
#define COMPUTER_GEWINNT 11
static uchar strategie=2; //Strategie des Computerzugs (0=primitiv, 2=perfekt)

void wer_ist_dran_wechseln()
{
 wer_ist_am_zug = MENSCH+COMPUTER - wer_ist_am_zug;
}

uchar rand8() //8-Bit-Zufallszahl
{
 uchar c=(TCNT1&0xFF);
 return c;
}

int computerzug()
{
 uchar reihe,anzahl,anzreihen,nr=0,n2=0,n2nr=0,j,i,retwert=0;

 for(reihe=0,anzahl=0,anzreihen=0; reihe<ANZAHL_REIHEN; reihe++) //totale Anzahl Steine zaehlen
  {
   anzahl += steine[reihe]; //Anzahl Steine
   if(steine[reihe]!=0)
    {
     anzreihen++; //Anzahl noch vorhandene Reihen
     nr=reihe; //letzte Reihennummer
     if(steine[reihe]>=2) {n2nr=reihe; n2++;} //Anzahl Reihen mit mehr als 1 Stein
    }
  }

 if(anzahl==0) return COMPUTER_GEWINNT;
 if(anzahl==1)
  {
   retwert=MENSCH_GEWINNT;
   reihe=nr; anzahl=1; //Letzten Stein nehmen
  }
 else if(anzreihen==1) //Gewinnzug?
  {
   reihe=nr; anzahl=steine[reihe]-1; //Alle bis auf einen nehmen
  }
 else if(n2==1) //nur 1 Reihe mit 2 Steinen oder mehr?
  {
   reihe=n2nr;
   anzahl = anzreihen%2==0 ? steine[reihe] : steine[reihe]-1; //alle oder alle bis auf einen nehmen
  }
 else if(strategie==0)
  {
   //primitive Strategie: zufaellige Reihe und zufaellige Anzahl wegnehmen:
   do {reihe = (rand8()%ANZAHL_REIHEN);}
   while(steine[reihe]==0);
   anzahl = (rand8()%steine[reihe])+1;
  }
 else //if(strategie==2)
  {
   //Gute Strategie: Stellung versuchen auf gradzahlige Bit-Anzahlen zu setzen
   uchar bitmaske=0,n;
   for(reihe=0;reihe<ANZAHL_REIHEN;reihe++)
    {
     bitmaske ^= steine[reihe]; //Bits auf Gradzahligkeit testen
    }
   if(bitmaske==0) //verlorene Stellung?
    {
     //primitive Strategie bei verlorener Stellung
     do {reihe = (rand8()%ANZAHL_REIHEN);}
     while(steine[reihe]==0);
     anzahl = (rand8()%steine[reihe])+1;
    }
   else
    {
     int k;
     for(k=0;k<1000;k++) //eine Reihe suchen von der wir soviele Steine nehmen koennen
      {                      //dass gradzahlige Bitzahlen entstehen
       reihe = (rand8()%ANZAHL_REIHEN); //zufaellige Reihe
       n=steine[reihe];
       nr=(n^bitmaske);
       if(nr<n)
	{
	 anzahl=n-nr;
	 break;
	}
      }
     if(k==1000) //war es erfolglos?
      {
       do {reihe = (rand8()%ANZAHL_REIHEN);} //zufaellige Reihe suchen
       while(steine[reihe]==0); //die noch Steine hat
       anzahl=1;
      }
    }
  }

 j=reihe+STARTPOSY;
 i=STARTPOSX-1+anzahl+(ANZAHL_REIHEN-1-reihe);
 while(cursory!=j)
    {
     if(cursory<j) ++cursory; else --cursory;
     cursor_zeichnen();
     milliwait(500);
    }
 milliwait(500);
 while(cursorx!=i)
    {
     if(cursorx<i) ++cursorx; else --cursorx;
     cursor_loeschen();
     steine_farben_anpassen();
     cursor_zeichnen();
     milliwait(200);
    }
 steine_wegnehmen();
 return retwert;
}

void neues_spiel(uchar nreihen)
{
 //screenclear();
#ifdef UART_BASIRGB
 basi_screenclear();
#endif
 uchar i,j;
 for(j=0;j<10;j++)
 for(i=0;i<20;i++)
  {
   gesamtes_ledfeld_setzen(j,i,0); //alle LEDs aus
  }

 steine_init(nreihen);
 cursor_loeschen();
 steine_farben_anpassen();
 steine_zeichnen();
 cursor_zeichnen();
}

void gewinner_zeigen(int gewonnen,uchar nreihen)
{
 uchar i,j,farbe=0;
 if(gewonnen==MENSCH_GEWINNT)
  {
   farbe=4; //Blau
  }
 if(gewonnen==COMPUTER_GEWINNT)
  {
   farbe=1; //Rot
  }
 for(j=0;j<10;j++)
 for(i=0;i<20;i++)
  {
   gesamtes_ledfeld_setzen(j,i,farbe); //alle LEDs setzen
  }

 for(i=0;i<10;i++) milliwait(1000);

 neues_spiel(nreihen);
 wer_ist_am_zug = (rand8()%2)==1 ? MENSCH : COMPUTER;
}

/********************************** Hauptprogramm *************************************/

int main(void)
{
 uchar i,j,c,startflag=1,anzahl_reihen=ANZAHL_REIHEN;

 PORTA = 0xFF; //PortA alle auf High fuer inaktiv (P-Chann-Transistoren)
 PORTD = 0x03; //PortD: unterste 2 Bits High fuer inaktiv (P-Chann-Transistoren),
 //PORTD = 0x0F;  //      INT0-INT1 Pullup-Widerstaende fuer Tasten setzen
 DDRA = 0xFF; //PortA alles Ausgaenge
 DDRD = 0xF3; //PortD alle ausser INT0-INT1 auf Ausgang
 DDRB = 0x0F; //PortB untere 4 Bits auf Ausgang
 DDRC = 0xFF; //PortC alles Ausgaenge
 PORTB = 0xFF; //alle Ausgaenge inaktiv, Eingaenge Pullups setzen
 PORTC = 0x7F; //Ausgaenge fuer LEDs inaktiv, OUT auf aktiv low

 timer0_init();
 timer1_init();
 ledfeld_init(0);
 sei(); //Interrupt einschalten

 milliwait(10); //warten bis andere Module auch bereit
 c=(PINB&0x30); //Status der Eingaenge lesen
 switch(c)
  {
   case 0x00: modulnummer=3; break;
   case 0x10: modulnummer=4; break;
   case 0x20: masterflag=true; modulnummer=1; break;
   case 0x30: modulnummer=2; break;
  }

 // Einschalt-Test:
 if(masterflag) c=7; //weiss fuer Modul oben links
 else {c=modulnummer-1; if(c==3) c=4;} //rot, gruen, blau fuer die andern Module
 ledfeld[0][0]=c; //LED oben links setzen
 milliwait(500);
 ledfeld[0][0]=c; //wieder loeschen

 if(!masterflag) //andere Module nur Daten vom Master empfangen
  {
   for(;;) //Hauptschlaufe fuer Sklaven-Module
    {
     if((TAKT_PIN&(1<<TAKTBIT))==0) //warten bis Start einer Datenuebertragung
       daten_empfangen();
    }
  }

 //Aktionen fuer den Master:
 uart1_init(); //Serielle Schnittstelle (UART) fuer Tastatur und fuer Video-Ausgabe auf Fernseher
 milliwait(100); //warten bis andere Module bereit
#ifdef UART_BASIRGB
 milliwait(900); //warten bis Fernseher bereit
 basi_screenclear(); //Screenclear auf Fernseher
 /*
 uart_putc(0x12); //test: Position fuer Test-Kreis
 uart_putc(0x40); uart_putc(0x00); uart_putc(0x40); uart_putc(0x00); //test: x- und y-Wert fuer Position
 uart_putc(0x17); //test: Kreis auf Fernseher zeichnen
 uart_putc(0x78); uart_putc(0x00); uart_putc(0x80); uart_putc(0x00); //test: Kreisdurchmesser in x- und y-Richtung
 */
#endif

 TAKT_DDR |= (1<<TAKTBIT);
 MISO_DDR |= (1<<MISOBIT);
 c=4; //blau
 for(j=0;j<10;j++)
 for(i=0;i<20;i++)
  {
   gesamtes_ledfeld_setzen(j,i,c); //alle LEDs nacheinander setzen
   //milliwait(50);
  }
 for(j=0;j<10;j++)
 for(i=0;i<20;i++)
  {
   gesamtes_ledfeld_setzen(j,i,0); //alle LEDs wieder aus
   //milliwait(50);
  }

 uchar loslassflag,gewonnen;
 steine_init(anzahl_reihen);
 steine_zeichnen();
 cursor_zeichnen();

 while(1) //Hauptschlaufe
  {
   if(wer_ist_am_zug==COMPUTER)
    {
     gewonnen=computerzug();
     if(gewonnen!=0) {gewinner_zeigen(gewonnen,anzahl_reihen); startflag=1;}
     wer_ist_dran_wechseln();
    }
   else if(uart_check())   // Zeichen von Tastatur bereit?
    {
     c=uart_getc();
     loslassflag=c&0x80;
     c &= 0x7F;

     //test:
     //j = (c-'A')/20; i = (c-'A')%20;
     //gesamtes_ledfeld_setzen(j,i, loslassflag ? 0 : 1); //test LED auf rot wenn gedruckt, aus wenn losgelassen

     //Reaktionen bei Tastendruck:
     if(loslassflag==0)
      {
       if(c=='R') //Mittlere Taste
	{
	 if(wegnehm_anzahl!=0)
	  {
	   startflag=0;
	   steine_wegnehmen();
	   wer_ist_dran_wechseln();
	  }
	}
       else if(c=='U') //Taste unten links
	{
	 if(startflag!=0 && anzahl_reihen>4)
	   neues_spiel(--anzahl_reihen);
	}
       else if(c=='Y') //Taste unten rechts
	{
	 if(startflag!=0 && anzahl_reihen<ANZAHL_REIHEN)
	   neues_spiel(++anzahl_reihen);
	}
       else if(c=='C') //oberste wseisse Taste
	{
	 if(startflag!=0)
	  {neues_spiel(anzahl_reihen); if((startflag^=2)==3) wer_ist_dran_wechseln();}
	}
       //weitere Tasten: F G H I J ist obere blaue Reihe
       else
	{//Tasten zum Cursor bewegen:
	 if(c=='M') //nach oben
	  {if(cursory>0) --cursory;}
	 else if(c=='W') //nach unten
	  {if(cursory<9) cursory++;}
	 else if(c=='Q') //nach links
	  {if(cursorx>0) --cursorx;}
	 else if(c=='S') //nach rechts
	  {if(cursorx<19) cursorx++;}
	 cursor_loeschen();
	 steine_farben_anpassen();
	 cursor_zeichnen();
	}
      }
    }
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}
