/* molz.cc  letzte Aenderung: 1.10.2019
Version 0.3

Ein Molekuel mit OpenGL zeichnen

*/
#define BELEUCHTUNG  //mit Lichtquelle

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <GL/freeglut.h>
#include "vektor3dklasse.cc"
#include "quaternionklasse.cc"

typedef unsigned char uchar;

static int bildbreite=1200, bildhoehe=900; //Fenstergroesse in Pixeln
static int bildposx=30, bildposy=10;       //Position des Fensters beim Start

const char *filename="Alanin.xyz"; //das darzustellende Molekuel, TODO: mehrere Molekuele

static int menu_status=0; //0=inaktiv (ev. versteckt), 

//const double PI=4*atan(1.0); //schon in vektor3dklasse definiert
const int anz_kreispunkte=360; //also 1 Grad Aufloesung
const double xmin= -6, xmax=6, ymin= -4.5, ymax=4.5;
const double zmin= -10, zmax=10; //z-Achse vom Bildschirm gegen Betrachter zeigend
static double DX=(xmax-xmin)/bildbreite, DY=(ymax-ymin)/bildhoehe;
#include "myfonts.cc"

static int solidflag=1;
static int ambientflag=1;
static int cylinderflag=1; //1: Bindungen als Zylinder zeichnen, 0:nur dicke Linien

static double kugelskal=1.0; //Groesse der Kugeln (Atome) skalieren

class Atomlist
{
public:
 char name[4];
 Vektor3d p;
 double ra; //radius
 uchar r,g,b; //Farbe
 void set(const char *nam,Vektor3d& v);
};

void Atomlist::set(const char *nam,Vektor3d& v)
{
 p=v;
 strncpy(name,nam,3); name[3]=0;
 ra=1.0; //Radius in Angstrom
 r=g=b=127; //mittleres Grau wenn keine andere Farbe definiert
 if(strlen(name)==1)
  {switch(name[0])
    {// Radius in Angstrom (0.1nm)
     case 'C': ra=0.772; r=g=b=50; break; //Kohlenstoff, fast schwarz
     case 'H': ra=0.373; r=g=b=200; break; //Wasserstoff, weiss
     case 'B': ra=0.795; r=g=b=140; break; //Bor, grau
     case 'N': ra=0.549; r=50;g=50;b=255; break; //Stickstoff
     case 'O': ra=0.604; r=255;g=b=0; break; //Sauerstoff
     case 'F': ra=0.709; r=204;g=255;b=127; break; //Fluor
     case 'I': case 'J': ra=1.331; r=127;g=b=255; break; //Jod
     case 'S': ra=1.035; r=255;g=255;b=0; break; //Schwefel
     case 'P': ra=1.105; r=255;g=0;b=255; break; //Phosphor
     default: printf("unbekanntes Atom '%c'\n",name[0]);
    }
  }
 else if(strcmp(name,"Na")==0) {ra=1.858; r=g=b=154;} //Natrium
 else if(strcmp(name,"Cl")==0) {ra=0.994; r=0;g=255;b=0;} //Chlor
 else if(strcmp(name,"Br")==0) {ra=1.145; r=g=179;b=0;} //Brom
}

class Bondlist
{
public:
 int n1,n2; //Nummern der Atome die miteinander verbunden
 int strong; //1=einfach- 2=zweifach- 3=dreifach-Bindung
 void set(int a,int b,int c) {n1=a; n2=b; strong=(c==0)?1:c;}
};

#define MAXATOMS 1000
Atomlist atomliste[MAXATOMS];
int anz_atome=0;
#define MAXBONDS 2000
Bondlist bondliste[MAXBONDS];
int anz_bonds=0;

/** Requester **/
const int REQ_X=12, REQ_Y=16; //Fontgroesse fuer alle Requester
static char REQ_FONTSTRING[32]="12x16";
const int REQ_RAND=4; //provi. Rand innerhalb von Requestern

class Knopf
{
public:
 int x1,y1,x2,y2; //Viereck unten links und oben rechts in Pixelkoordinaten
 //(x von links nach rechts, y von unten nach oben)
 void set(int xa,int ya,int b,int h) {x1=xa; y1=ya; x2=x1+b; y2=y1+h;}
 void zeichnen(const char *text,bool preselect);
 bool klick(int x,int y) {return (x>x1 && x<x2 && y>y1 && y<y2);}
};

void Knopf::zeichnen(const char *text,bool preselect)
{
 int x=x1+REQ_X, y=y1+REQ_Y/2;
 glColor3f(0, 0, 0); // Schwarzer Rahmen um Knoepfe und auch schwarzer Text
 schrift(x,y,text);
 glBegin(GL_LINE_LOOP);
 glVertex2i(x1, y2);
 glVertex2i(x1, y1);
 glVertex2i(x2, y1);
 glVertex2i(x2, y2);
 glEnd();
 if(preselect)
  {
   int d=4; //4 Pixel Abstand fuer doppelter Rahmen
   glBegin(GL_LINE_LOOP);
   glVertex2i(x1+d, y2-d);
   glVertex2i(x1+d, y1+d);
   glVertex2i(x2-d, y1+d);
   glVertex2i(x2-d, y2-d);
   glEnd();
  }
 glColor3f(1.9, 1.9, 1.9); // Hellgrauer Hintergrund
 glBegin(GL_QUADS);
 glVertex2f(x1, y2);
 glVertex2f(x1, y1);
 glVertex2f(x2, y1);
 glVertex2f(x2, y2);
 glEnd();
}

class Myrequester
{
public:
 bool aktiv;
 int typ;
 char text[4000];
 char jatext[80],neintext[80],canceltext[80];
 //Subfenster unten links und oben rechts in Pixelkoordinaten:
 //(x von links nach rechts, y von unten nach oben)
 int x1,y1,x2,y2;
 Knopf knopf1,knopf2,knopf3; //Ja-, Nein-, und Cancel-Box
 int preselect; //1, 2, oder 3: entsprechender Knopf mit Enter auswaehlbar
 Myrequester() {aktiv=false; typ=0; preselect=0;}
 void set(int t,int br,int ho,const char *txt,
	  const char *ja=NULL,const char *nein=NULL,const char *canc=NULL,int pre= -1);
 void reset() {aktiv=false;}
 void zeichnen();
};

//Requester-Typen:
#define REQ_JANEIN 1
//TODO: weitere wie in xtekplot1

static Myrequester requester1; //mehr als einer macht vermutlich keinen Sinn
static int requester_antwort= -2; //-2 = ungueltig, 0=nein, 1=ja, -1=cancel

void Myrequester::set(int t,int br,int ho,const char *txt,
		      const char *ja,const char *nein,const char *canc,int pre)
{
 typ=t;
 strcpy(text,txt);
 preselect=1; //automatisch falls pre nicht gesetzt
 if(ja==NULL) strcpy(jatext,"ok"); else strcpy(jatext,ja);
 if(nein==NULL) strcpy(neintext,""); else {strcpy(neintext,nein); preselect=2;}
 if(canc==NULL) strcpy(canceltext,""); else {strcpy(canceltext,canc); preselect=3;}
 if(pre>=0) preselect=pre;
 x1=50; y2=bildhoehe-50; //linke obere Eche des Requesters //provisorische Werte
 if(y2<ho) y2=bildhoehe;
 if(x1+br>bildbreite) x1=0;
 x2=x1+br; y1=y2-ho; //rechte untere Ecke des Requesters
 if(x2>bildbreite) x2=bildbreite;
 if(y1<0) y1=0;
 aktiv=true;
}

/** eigene Routine zum Kreise und Kugel zeichnen: ** /
static Vektor3d einheitskreis[anz_kreispunkte];

void init_einheitskreis()
{
 Vektor3d p;
 for(int i=0;i<anz_kreispunkte;i++)
  {
   double alfa=2*PI/anz_kreispunkte*i;
   p.x=cos(alfa); p.y=sin(alfa); p.z=0;
   einheitskreis[i]=p;
  }
}

void zeichne_kugel(double radius,double x0,double y0,double z0)
{
 radius *= kugelskal;
 Vektor3d p;
 const int anz_rot=18;
 double rotwinkelschritt=PI/anz_rot;
 for(int j=0;j<anz_rot;j++) //Laengenlinien
  {
   double phi=rotwinkelschritt*j;
   double cosphi=cos(phi), sinphi=sin(phi);
   glBegin(GL_LINE_LOOP);
   for(int i=0;i<anz_kreispunkte;i++)
    {
     p=einheitskreis[i];
     double x1=p.x*radius; 
     p.x = x1*cosphi + x0;
     p.y = p.y*radius + y0;
     p.z = x1*sinphi + z0;
     glVertex3d(p.x, p.y, p.z);
    }
   glEnd();
  }
 for(int j=1;j<anz_rot;j++) //Breitenlinien
  {
   double r=sin(j*rotwinkelschritt)*radius;
   p.y = cos(j*rotwinkelschritt)*radius + y0;
   glBegin(GL_LINE_LOOP);
   for(int i=0;i<anz_kreispunkte;i++)
    {
     p.x = einheitskreis[i].x*r + x0;
     p.z = einheitskreis[i].y*r + z0;
     glVertex3d(p.x, p.y, p.z);
    }
   glEnd();
  }
}
/ ** :eigene Routine zum Kreise und Kugel zeichnen **/

/** einfacher mit Glut: **/
void zeichne_kugel(double radius,double x,double y,double z)
{
 glPushMatrix();
 glTranslated(x,y,z);
 glutWireSphere(radius*kugelskal, 36, 18);
 glPopMatrix();
}

void zeichne_bindung(int a,int b,int c)
{
 //double radius=0.08+0.02*c; //willkuerlicher Wert
 Vektor3d p1,p2,pa;
 //double r1,r2;
 p1=atomliste[a].p; //r1=atomliste[a].ra;
 p2=atomliste[b].p; //r2=atomliste[b].ra;
 glColor3ub(50,50,50); // Farbe setzen, dunkelgrau
 if(cylinderflag==0)
  {
   //float dicke=4.0; if(c==2) dicke=8; else if(c>=3) dicke=20;
   float dicke=bildbreite/200; if(c==2) dicke*=2; else if(c>=3) dicke*=5;
   glLineWidth(dicke); //dicke Linien
   glBegin(GL_LINES);
   glVertex3d(p1.x, p1.y, p1.z);
   glVertex3d(p2.x, p2.y, p2.z);
   glEnd();
   glLineWidth(1.0); //duenne Linien
  }
 else
  {
   //Bindung als Zylinder um Verbindungslinie p1--p2 zeichnen:
   GLUquadric *quad=gluNewQuadric();
   double dicke = 0.373/4; //Zylinder-Radius = 1/4 von H-Atom-Radius
   if(c==2) dicke*=1.75; else if(c>=3) dicke*=2.5;
   pa=p2-p1;
   double zylhoehe=abs(pa);
   glPushMatrix();
   glTranslated(p1.x,p1.y,p1.z);
   float alfa= -atan2(pa.x,pa.y)/GRAD; //Vorzeichen ist Glueckssache
   glRotatef(alfa,0,0,1);
   float beta= -acos(pa.z/zylhoehe)/GRAD; //Vorzeichen ist Glueckssache
   glRotatef(beta,1,0,0);
   gluCylinder(quad,dicke,dicke,zylhoehe,36,100);
   glPopMatrix();
  }
}

void zeichne_solidekugel(double radius,double x,double y,double z)
{
 glPushMatrix();
 glTranslated(x,y,z);
 glutSolidSphere(radius*kugelskal, 36, 18);
 glPopMatrix();
}

void zeichne_molekuel()
{
 Atomlist *p=atomliste;
 for(int i=0;i<anz_atome;i++)
  {
   glColor3ub(p[i].r, p[i].g, p[i].b); // Farbe setzen
   if(solidflag==0)
    zeichne_kugel(p[i].ra, p[i].p.x, p[i].p.y, p[i].p.z);
   else
    zeichne_solidekugel(p[i].ra, p[i].p.x, p[i].p.y, p[i].p.z);
  }
 Bondlist *b=bondliste;
 for(int i=0;i<anz_bonds;i++)
  {
   zeichne_bindung(b[i].n1, b[i].n2, b[i].strong);
  }
}

void zeichne_koordinatenkreuz()
{
 glColor3ub(0,0,0); // Farbe setzen, schwarz
 glBegin(GL_LINES); // GL einschalten Modus: LINIE
 glVertex3f(xmin,0,0); // erst der x-wert (links), y=0
 glVertex3f(xmax,0,0); // dann Linie nach rechts ziehen
 glVertex3f(0,ymin,0); // y-Wert unten Mitte
 glVertex3f(0,ymax,0); // y nach oben hochziehen
 glVertex3f(0,0,zmin); // z-Wert vorne
 glVertex3f(0,0,zmax); // z nach hinten ziehen
 glEnd(); // fertig, GL ausschalten
}

//static Vektor3d eye(2.0, 2.5, 10.0); //Kameraposition
static Vektor3d eye(0.0, 0.0, 10.0); //Kameraposition
static Vektor3d cent(0.0, 0.0, 0.0); //Punkt auf den die Kamera schaut
static Vektor3d up(0.0 , 1.0, 0.0); //Oben-Richtung der Kamera
static int fadenkreuz=0; //1=Koordinatenkreuz zeichnen

void kamera_reset()
{
 //eye=Vektor3d(2.0, 2.5, 10.0);
 eye=Vektor3d(0.0, 0.0, 10.0);
 cent.x=cent.y=cent.z=0;
 up.x=up.z=0; up.y=1;
}

/**** Menu und eventuell Requester zeichnen: ****/
#define MENU_BALKEN_HOEHE 23
#define MBH MENU_BALKEN_HOEHE

void schrift_umbruch(int x0,int y0,char *text,int br,int ho)
{
 char *s,*p;
 int x=x0,y=y0;
 for(s=text;;)
  {
   if((p=index(s,'\n'))==NULL && (p-s)<=br) {schrift(x,y,s); break;}
   if(p==NULL)
    {
     p=index(s,0);
     if((p-s)<=br) {schrift(x,y,s); break;}
    }
   else if((p-s)<=br)
    {
     char c= *p; *p = 0; schrift(x,y,s); *p++ = c;
     s=p;
    }
   else
    {
     char c=s[br]; s[br]=0; schrift(x,y,s); s[br]=c;
     s= &s[br];
    }
   x=x0;
   y -= ho;
  }
}

void Myrequester::zeichnen()
{
 //Myrequester *req= &requester1;
 //if(REQ_FONTSTRING[0]==0) sprintf(REQ_FONTSTRING,"%dx%d",REQ_X,REQ_Y);
 textsize(REQ_X,REQ_Y,REQ_FONTSTRING);
 glColor3ub(0,0,0); // Schwarze Schrift
 int br = (x2-x1-2*REQ_RAND)/REQ_X; //Anzahl Zeichen, die pro Zeile platz haben
 int ho=REQ_Y*4/3; //hoehe der Buchstaben inklusiv Zeilenabstand
 schrift_umbruch(x1+REQ_RAND, y2-REQ_Y-REQ_RAND, text, br,ho);
 if(*jatext!=0) knopf1.zeichnen(jatext,preselect==1);
 if(*neintext!=0) knopf2.zeichnen(neintext,preselect==2);
 if(*canceltext!=0) knopf3.zeichnen(canceltext,preselect==3);
 glColor3f(1.7, 1.7, 1.7); // Hellgrauer Hintergrund
 glBegin(GL_QUADS);
 glVertex2f(x1, y2);
 glVertex2f(x1, y1);
 glVertex2f(x2, y1);
 glVertex2f(x2, y2);
 glEnd();
}

void menu_text_zeichnen() //Menu zeichnen und evt. auch noch andere Texte
{
 glMatrixMode(GL_PROJECTION);
 glPushMatrix();
 glLoadIdentity();
 gluOrtho2D(0, bildbreite, 0, bildhoehe);

 /*
 Mit dem DEPTH_BUFFER wird im Gegensatz zum normalen Verhalten das zuerst
 gezeichnete behalten.
 Schon gezeichnete Pixel werden also nicht ueberzeichnet.
 Also z.B. schwarze Schrift auf hellgrauem Hintergrund zuerst die Schrift,
 dann erst den Hintergrund zeichnen!
 */

 if(requester1.aktiv)
  {
   requester1.zeichnen();
  }
 else if(requester_antwort != -2)
  {
   printf("requester_antwort war %d\n",requester_antwort);//test
   //TODO: Antwort evtl. auswerten
   requester_antwort= -2;
  }
 //textsize(6,8,"6x8");
 //textsize(9,12,"9x12");
 textsize(12,16,"12x16");
 double x=REQ_RAND, y=bildhoehe-MBH+3;
 glColor3ub(0,0,0); // Schwarze Schrift
 schrift(x,y,"Menuleiste");
 if(menu_status!=0)
  {
   glColor3f(1.7, 1.7, 1.7); // Hellgraue Menuleiste
   glBegin(GL_QUADS);
   glVertex2d(2, bildhoehe-2);
   glVertex2d(2, bildhoehe-MBH+2);
   glVertex2d(bildbreite-2, bildhoehe-MBH+2);
   glVertex2d(bildbreite-2, bildhoehe-2);
   glColor3f(2.5, 2.5, 2.5); // Weiss fuer oberer Rahmen der Menuleiste
   glVertex2d(0, bildhoehe);
   glVertex2d(0, bildhoehe-2);
   glVertex2d(bildbreite, bildhoehe-2);
   glVertex2d(bildbreite, bildhoehe);
   glVertex2d(0, bildhoehe);
   glVertex2d(0, bildhoehe-MBH);
   glVertex2d(2, bildhoehe-MBH);
   glVertex2d(2, bildhoehe);
   glColor3f(0.75, 0.75, 0.75); // dunkelgrau fuer unteren Rahmen der Menuleiste
   glVertex2d(0, bildhoehe-MBH+2);
   glVertex2d(0, bildhoehe-MBH);
   glVertex2d(bildbreite, bildhoehe-MBH);
   glVertex2d(bildbreite, bildhoehe-MBH+2);
   glVertex2d(bildbreite-2, bildhoehe-MBH+2);
   glVertex2d(bildbreite-2, bildhoehe-MBH);
   glVertex2d(bildbreite, bildhoehe-MBH);
   glVertex2d(bildbreite, bildhoehe-MBH+2);
   glEnd();
  }
 
 glPopMatrix();
 glMatrixMode(GL_MODELVIEW);
}

void display(void)
{
 //static int ntest=0;//test
 //printf("display() %d.Aufruf\n",++ntest);//test
 glClearColor(0.5, 0.5, 0.5, 1); //Hintergrundfarbe grau
 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
 
#ifdef BELEUCHTUNG
 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
#endif
#ifdef BELEUCHTUNG
  GLfloat light_pos[]={xmax,ymax,zmax,1};
  GLfloat ambient_hell[]={0.25,0.25,0.25,1};
  GLfloat ambient_dunkel[]={0,0,0,1};
  glLightfv(GL_LIGHT0,GL_POSITION,light_pos);
  if(ambientflag==0) glLightfv(GL_LIGHT0,GL_AMBIENT,ambient_dunkel);
  else               glLightfv(GL_LIGHT0,GL_AMBIENT,ambient_hell);
  glEnable(GL_COLOR_MATERIAL);
#endif

 menu_text_zeichnen(); //provisorisches Menu, TODO: Routinen fuer richtiges Menu
 
 glMatrixMode(GL_MODELVIEW);
 glLoadIdentity();
 glPushMatrix(); // GL_MODELVIEW is default
 glScalef(1.0, bildbreite/(float)bildhoehe, 1.0);
 //glTranslatef(-xmin/10, -ymin/10, -zmin/10);
 
 //gluLookAt(0, 0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
 //gluLookAt(eye[0], eye[1], eye[2], cent[0], cent[1], cent[2], up[0],up[1],up[2]);
 gluLookAt(eye.x, eye.y, eye.z, cent.x, cent.y, cent.z, up.x,up.y,up.z);
 
 glEnable(GL_BLEND); //antialias
 glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); //antialias
 glEnable(GL_LINE_SMOOTH); //antialias
 
 zeichne_molekuel();
 if(fadenkreuz!=0) zeichne_koordinatenkreuz();
 glPopMatrix();
 glutSwapBuffers();
};

/* Idle proc. Redisplays, if called. *  /
void idle(void)
{
 static int ntest=0;//test
 printf("idle() %d.Aufruf\n",++ntest);//test
 glutPostRedisplay();
};
*/

/*
void mpause(int n) //n millisekunden warten
{
 usleep(n*1000); //soviele Mikrosekunden warten
}
*/

const char *hilfetext=
"Pfeiltasten: Kamera verschieben:\n\
   Pfeil links/rechts: um y-Achse drehen\n\
   Pfeil oben/unten:   um x-Achse drehen\n\
   Tasten PageUp/Down: um z-Achse drehen\n\
   Taste Home: Kamera auf Anfangsposition ruecksetzen\n\
Maus mit linker Taste: Objekt drehen um x- und y-Achse\n\
Maus mit mittlerer Taste: Objekt drehen um z-Achse\n\
Maus mit rechter Taste: Objekt verschieben in x-y-Richtung\n\
Mausrad: Objekt von Kamera weg/hin bewegen (zoomen)\n\
Tastatur:\n\
  F: Koordinaten-Kreuz zeichnen/ausschalten\n\
  S: umschalten zw. Drahtkugeln oder Solide Kugeln\n\
  A: Ambientlicht ein/aus\n\
  K: Kugelgroessen skalieren\n\
  B: Bindungen als Zylinder oder als Dicke Linien zeichnen\n\
  Q: Programm beenden\n\
  H: diese Hilfe";

void key(unsigned char c, int x, int y)
{
 static int escflag=0;
 if(requester1.aktiv)
  {
   escflag=0;
   if(c==27) {requester_antwort= -1; requester1.reset();}
   else if(c=='\r') {requester_antwort=1; requester1.reset();}
   else if(c=='n' || c=='N') {requester_antwort=0; requester1.reset();}
   glutPostRedisplay();
   return;
  }
 if(c==27) escflag=1;
 else if(escflag!=0 && (c=='q' || c=='Q')) exit(0);//mit ESC Q beenden
 else escflag=0;
 
 if(c>='a' && c<='z') c -= 'a'-'A';// c=toupper(c);
 if(c=='?' || c=='H')
  {
   printf("\n%s\n",hilfetext);
  }
 else if(c=='F') fadenkreuz ^= 1;
 else if(c=='S') solidflag ^= 1;
 else if(c=='A') ambientflag ^= 1;
 else if(c=='K') {if(kugelskal<=0.25) kugelskal=1.0; else kugelskal-=0.25;}
 else if(c=='B') cylinderflag ^= 1;
 else printf("key('%c',%d,%d)\n",c,x,y);//test
 glutPostRedisplay();
};

/*  Drehungen um xyz-Achsen des globalen Koordinatensystems
void kamera_drehen(int achse,int richtung)
{
 Vektor3d p; p=eye-cent;
 const double sina=sin(ZWEIPI/360), cosa=cos(ZWEIPI/360);
 if(achse==0) //um x-Achse drehen
  {
   if(richtung==1) {p.drehenyz(sina,cosa); up.drehenyz(sina,cosa);}
   else            {p.drehenzy(sina,cosa); up.drehenzy(sina,cosa);}
  }
 else if(achse==1) //um y-Achse drehen
  {
   if(richtung==1) {p.drehenzx(sina,cosa); up.drehenzx(sina,cosa);}
   else            {p.drehenxz(sina,cosa); up.drehenxz(sina,cosa);}
  }
 else if(achse==2) //um z-Achse drehen
  {
   if(richtung==1) {p.drehenyx(sina,cosa); up.drehenyx(sina,cosa);}
   else            {p.drehenxy(sina,cosa); up.drehenxy(sina,cosa);}
  }
 eye=p+cent;
}
*/

/* Drehungen mit Koordinatensystem von Kamera aus gesehen */
void kamera_drehen(int achse,int richtung)
{
 Vektor3d p; p=eye-cent;
 Quaternion q;
 const double dw=ZWEIPI/360;
 if(achse==0) //um x-Achse drehen
  {
   Vektor3d k;
   k=vektorkreuzprodukt(p,up);
   p=q.vektordrehen(p,k,-richtung*dw);
   up=q.vektordrehen(up);
   eye=p+cent;
  }
 else if(achse==1) //um y-Achse drehen
  {
   p=q.vektordrehen(p,up,richtung*dw);
   eye=p+cent;
  }
 else if(achse==2) //um z-Achse drehen
  {
   up=q.vektordrehen(up,p,richtung*dw);
  }
}

#define PFEIL_LINKS 100
#define PFEIL_OBEN 101
#define PFEIL_RECHTS 102
#define PFEIL_UNTEN 103
#define PAGE_UP 104
#define PAGE_DOWN 105
#define KEY_HOME 106
#define KEY_END 107
#define KEY_INSERT 108

#define KEY_DEL 111
#define KEY_SHIFTL 112
#define KEY_SHIFTR 113

void spezial(int key,int x,int y) //Tastatur spezialzeichen
{
 if(requester1.aktiv)
  {
   //printf("spezial(key=%d,x=%d,y=%d)\n",key,x,y);//test
   if(key=='\n') {requester_antwort=1; requester1.reset();} //TODO: preselect auswerten
   glutPostRedisplay();
   return;
  }
 int flag=0;
 if(key==PFEIL_LINKS)       kamera_drehen(1,flag= 1); //{eye.x -= 0.1; flag=1;}
 else if(key==PFEIL_RECHTS) kamera_drehen(1,flag= -1); //{eye.x += 0.1; flag=1;}
 else if(key==PFEIL_OBEN)   kamera_drehen(0,flag= 1); //{eye.y += 0.1; flag=1;}
 else if(key==PFEIL_UNTEN)  kamera_drehen(0,flag= -1); //{eye.y -= 0.1; flag=1;}
 else if(key==PAGE_UP)      kamera_drehen(2,flag= 1); //{up.x += 0.1; flag=1;}//provi.
 else if(key==PAGE_DOWN)    kamera_drehen(2,flag= -1); //{up.x -= 0.1; flag=1;}//provi.
 else if(key==KEY_HOME)     kamera_reset(); //{eye.z += 0.1; flag=1;}
 //else if(key==KEY_END)      {eye.z -= 0.1; flag=1;}
 else printf("spezial(key=%d,x=%d,y=%d)\n",key,x,y);//test
 /*
 if(flag!=0) //test
  {
   printf("\n eyeX Y Z: %f %f %f\n",eye.x,eye.y,eye.z);
   printf("centX Y Z: %f %f %f\n",cent.x,cent.y,cent.z);
   printf("  upX Y Z: %f %f %f\n",up.x,up.y,up.z);
  }
 */
 glutPostRedisplay();
}

void reshape(int w, int h)
{
 glViewport(0, 0, w, h);
 bildbreite=w; bildhoehe=h;
 printf("reshape() bildbreite=%d bildhoehe=%d\n",w,h);//test
 DX=(xmax-xmin)/bildbreite; DY=(ymax-ymin)/bildhoehe;
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 gluPerspective(45.0, 1.0, 0.001, 1000);
 gluLookAt(0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
 glMatrixMode(GL_MODELVIEW);
};

class Intvektor
{
public:
 int x,y;
 Intvektor() {x=y=0;}
 Intvektor(int a,int b) {x=a; y=b;}
 Intvektor& operator-=(Intvektor p) {x-=p.x; y-=p.y; return *this;}
 Intvektor& operator/=(int n) {x/=n; y/=n; return *this;}
};

static Intvektor maus_pos; //Position der Maus beim druecken einer Taste
static int maus_knopf= -1; //aktuell gedrueckter Mausknopf (-1 wenn keiner gedrueckt)

#define MAXDISTANZ 100 //test
#define DISTSTEP 0.5 //test

void menustatus(int state,int y)
{
 //printf("menu_status=%d  y=%d\n",menu_status,y);//test
 if(menu_status==0) //wenn Menu inaktiv
  {
   if(y<MBH) //wenn Maus an oberen Rand gefahren
    {menu_status=1;
     glutAttachMenu(GLUT_LEFT_BUTTON);
    }
  }
 else //bei aktivem Menu
  {
   if(y>MBH) //wenn Maus aus Menu-Balkenbereich hinausgefahren
    {menu_status=0;
     glutDetachMenu(GLUT_LEFT_BUTTON);
    }
  }
}

void mausknopf(int button,int state,int x,int y)
{
 if(requester1.aktiv)
  {
   if(button==0)
    {
     y=bildhoehe-y; //y-Achse konvertieren zu unten nach oben
     if(requester1.knopf1.klick(x,y)) {requester_antwort=1; requester1.reset();}
     if(requester1.knopf2.klick(x,y)) {requester_antwort=0; requester1.reset();}
     if(requester1.knopf3.klick(x,y)) {requester_antwort= -1; requester1.reset();}
    }
  }
 if(button==0 || button==1 || button==2) //linke, mittlere oder rechte Maustaste
  {
   if(y<MBH || menu_status!=0) menustatus(state,y);
   Intvektor p(x,y);
   if(state==1) {maus_knopf= -1;} //beim loslassen
   else {maus_pos=p; maus_knopf=button;}
  }
 else if((button==3 || button==4) && state==0) //Mausrad rauf oder runter
  {
   Vektor3d p;
   p=eye-cent;
   double distanz=abs(p),neudist=distanz;
   if(button==3 && distanz>=2*DISTSTEP) //naeher ans Objekt ran
    {
     neudist -= DISTSTEP;
     p *= neudist/distanz;
     eye=cent+p;
    }
   else if(button==4 && distanz<MAXDISTANZ)
    {
     neudist += DISTSTEP;
     p *= neudist/distanz;
     eye=cent+p;
    }
   //printf("Distanz der Kamera zum Blickpunkt: %f\n",neudist);//test
  }
 glutPostRedisplay();
}

void mausmotion(int x,int y)
{
 if(maus_knopf==0) //linke Maustaste: Drehen um y-Achse oder um x-Achse
  {
   Intvektor p0(x,y),p1(x,y),p;
   p1 -= maus_pos;
   p=p1; p /= 2; //mindestens 2 Pixel bewegen
   if(abs(p.x)>0) //um y-Achse drehen
    {
     kamera_drehen(1, (p1.x<0) ? 1 : -1);
     maus_pos=p0;
    }
   if(abs(p.y)>0) //um x-Achse drehen
    {
     kamera_drehen(0, (p1.y<0) ? 1 : -1);
     maus_pos=p0;
    }
  }
 else if(maus_knopf==1) //mittlere Maustaste: Drehen um z-Achse
  {
   Intvektor p0(x,y),p1(x,y),p;
   p1 -= maus_pos;
   p=p1; p /= 2; //mindestens 2 Pixel bewegen
   if(abs(p.x)>0) //um z-Achse drehen
    {
     kamera_drehen(2, (p1.x<0) ? 1 : -1);
     maus_pos=p0;
    }
   if(abs(p.y)>0) //um z-Achse drehen
    {
     kamera_drehen(2, (p1.y<0) ? 1 : -1);
     maus_pos=p0;
    }
  }
 else if(maus_knopf==2) //rechte Maustaste: in x-y-Ebene bewegen
  {
   Intvektor p0(x,y),p1(x,y),p;
   p1 -= maus_pos;
   p=p1; p /= 5; //mindestens 5 Pixel bewegen
   if(abs(p.x)>0) //links-rechts Bewegung
    {
     //eye.x -= p1.x*DX;  cent.x -= p1.x*DX; //auf globalem Koordinatensystem schieben
     double anzpixel=p1.x*DX;
     Vektor3d kp; kp = vektorkreuzprodukt(eye-cent,up); kp.einheitsvektor();
     kp *= anzpixel;
     eye += kp;  cent += kp; //senkrecht zu Sichtlinie nach rechts/links schieben
     maus_pos=p0;
    }
   if(abs(p.y)>0) //oben-unten Bewegung
    {
     //eye.y += p1.y*DY;  cent.y += p1.y*DY; //auf globalem Koordinatensystem schieben
     double anzpixel=p1.y*DY; Vektor3d kp; kp=up*anzpixel;
     eye += kp;  cent += kp; //senkrecht zu Sichtlinie nach oben/unten schieben
     maus_pos=p0;
    }
  }
 else printf("mausmotion(x=%d,y=%d) maus_knopf=%d\n",x,y,maus_knopf);//test
 glutPostRedisplay();
}

int win0,win1;
//static char scratch[4000];

int strlen_zeile(const char *s,int *nz) //ermittelt Laenge der laengsten Zeile
{				        //und Anzahl Zeilen
 int c,i,n;
 for(i=n=0,*nz=1;(c= *s++);i++)
   if(c=='\n')
     {(*nz)++; if(n<i) n=i;
      i= -1;
     }
 if(n<i) n=i;
 return n;
}

void janeinrequester(const char *text,const char *jatext="ok",const char *neintext=NULL)
{
 int breite,hoehe; //Groesse des ganzen Requesters
 int br,ho; //Groesse der Antwortboxen
 int nzeilen, n1=strlen_zeile(text,&nzeilen); //n1=Laenge der laengsten Zeile
 int n2=strlen(jatext), n3=(neintext==NULL)?0:strlen(neintext);
 br = (((n2>n3) ? n2 : n3) + 2) * REQ_X;
 ho = 2*REQ_Y;
 breite=n1*REQ_X;
 if(breite < 3*br) breite=3*br;
 breite += 2*REQ_RAND;
 if(nzeilen<2) nzeilen=2;
 hoehe = 3*ho + nzeilen*REQ_Y*4/3 + 2*REQ_RAND;
 requester1.set(REQ_JANEIN,breite,hoehe,text,jatext,neintext);
 int x1,y1;
 if(neintext==NULL) {x1=breite/2;} else {x1=breite/3;}
 x1 -= br/2;
 x1 += requester1.x1;
 y1 = requester1.y1+ho;
 requester1.knopf1.set(x1,y1,br,ho);
 if(neintext!=NULL)
  {
   x1 += breite/3;
   requester1.knopf2.set(x1,y1,br,ho);
  }
}

void menu(int id)
{
 double neuskal;
 switch(id)
  {
  case 1: key('?',0,0); //Hilfe auf dem Terminal
          janeinrequester(hilfetext); //Hilfe als Requester
          //janeinrequester(hilfetext," ok","cancel"); //test
	  break;
  case 2: exit(0); break;
  case 3: printf("Skalierfaktor fuer Kugeln = %f\n",kugelskal);
          printf(" neuer Wert: "); scanf("%lf",&neuskal);
	  if(neuskal>0 && neuskal<=2.0) kugelskal=neuskal;
	  else printf("ungueltiger Wert - nichts veraendert\nnur Werte zwischen 0 und 1.0 erlaubt\n");
	  break;
  default: printf("menu(id=%d) unbekannte id\n",id);//test
  }
 glutPostRedisplay();
}
 
void grafik_init(const char *titel)
{
 int screenWidth  = glutGet(GLUT_SCREEN_WIDTH);
 int screenHeight = glutGet(GLUT_SCREEN_HEIGHT);
 //printf("screenWidht/Height = %d / %d\n",screenWidth,screenHeight);//test
 if(screenWidth>2*bildbreite && screenHeight>2*bildhoehe)
  {bildbreite *= 2; bildhoehe *= 2; bildposx *= 2; bildposy *= 2;}
 //glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE);
 glutInitDisplayMode(GLUT_DOUBLE|GLUT_RGB|GLUT_DEPTH);
 glutInitWindowSize(bildbreite,bildhoehe);
 glutInitWindowPosition(bildposx,bildposy);
 win0=glutCreateWindow(titel);
 
 glutCreateMenu(menu);
 glutAddMenuEntry("Hilfe  H",1);  
 glutAddMenuEntry("Kugelgroesse...",3);  
 glutAddMenuEntry("Exit   Q",2);  
 //glutAttachMenu(GLUT_RIGHT_BUTTON);  

 glShadeModel(GL_SMOOTH);
 glEnable(GL_DEPTH_TEST);

 /* Register GLUT callbacks. */
 glutDisplayFunc(display);
 glutKeyboardFunc(key);
 glutMouseFunc(mausknopf);
 glutMotionFunc(mausmotion);
 glutPassiveMotionFunc(mausmotion);
 glutSpecialFunc(spezial);
 glutReshapeFunc(reshape);
 //glutIdleFunc(idle);

 /* Init the GL state */
 glLineWidth(1.0);
}

/**************************** kleinkram *******************************/
inline bool istja(char *s) {return (*s!='n' && *s!='N');}
inline bool istnein(char *s) {return (*s=='n' || *s=='N');}

//Die Standard abs() Funktion ist unbrauchbar weil bei jeder Compilerversion
//wieder anders.
inline double myabs(double x) {return (x<0.) ? -x : x;}	/* Betrag */

bool getline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
	*s++ = c;
 *s='\0';
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

/************************** Hauptprogramm *****************************/
void xyzeinlesen(FILE *fp)
{
 char zeile[200],name[40];
 int natoms,nbonds;
 bool ok;
 do {ok=getline(fp,zeile,200);}
 while(ok && (*zeile==0 || *zeile==';')); //Leerzeilen und Kommentare ignorieren
 sscanf(zeile,"%d %d",&natoms,&nbonds);
 if(natoms<1) {printf("Fehler in xyzeinlesen(): keine Atome\n"); return;}
 if(natoms>MAXATOMS) {printf("zu viele Atome, bisher auf %d begrenzt\n",MAXATOMS); return;}
 Vektor3d v;
 for(int i=0;i<natoms && getline(fp,zeile,200);)
  {
   if(*zeile==0 || *zeile==';') continue; //Leerzeilen und Kommentare ignorieren
   sscanf(zeile,"%s %lf %lf %lf",name,&v.x,&v.y,&v.z);
   v.z = -v.z; //weil Koordinatensystem im OpenGL anders
   if(strlen(name)>2) printf("Fehlerhaftes Atom: \"%s\"\n",name);
   atomliste[i++].set(name,v);
   anz_atome=i;
  }
 int n1,n2,c;
 for(int i=0;i<nbonds && getline(fp,zeile,200);)
  {
   if(*zeile==0 || *zeile==';') continue; //Leerzeilen und Kommentare ignorieren
   c=0;
   sscanf(zeile,"%d %d %d",&n1,&n2,&c);
   //printf("Bond %d: %d %d %d\n",i,n1,n2,c);//test
   bondliste[i++].set(n1,n2,c);
   anz_bonds=i;
  }
}

int main(int argc, char **argv)
{
 int flags=0;
 if(argc>1)
  {if(*argv[1]=='-' || *argv[1]=='?') flags=1; //setargfalgs(argv[1]);
   else filename=argv[1]; //TODO: mehrer Molekuele schon beim Aufruf laden
   if(argc>2) printf("Mehrere Molekuele geht noch nicht. Nur erstes geladen.\n");
  }
 FILE *fp=fopen(filename,"r");
 if(fp==NULL) {printf("kann \"%s\" nicht oeffnen.\n",filename); exit(0);}
 xyzeinlesen(fp);
 fclose(fp);
 //init_einheitskreis();
 glutInit(&argc, argv);
 grafik_init(filename);
 if(flags>0) key('?',0,0);//test
 glutMainLoop();
 return 0;
}
