/* molz.cc  letzte Aenderung: 14.10.2019 */
#define VERSION "Version 0.5"
/*

Molekuele mit OpenGL zeichnen

History:
6.9.2019	Erstellung (RP)
4.10.2019  0.4  einfaches Menu
7.10.2019  0.5  mehrere Molekuele, bessere Requester

*/
#define BELEUCHTUNG  //mit Lichtquelle

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <GL/freeglut.h>
#include "vektor3dklasse.cc"
#include "quaternionklasse.cc"
typedef unsigned char uchar;
#include "myfonts.h"
#include "menu.h"

int bildbreite=1200, bildhoehe=900; //Fenstergroesse in Pixeln
int bildposx=30, bildposy=10;       //Position des Fensters beim Start

char filename0[MAXP]="Alanin.xyz"; //das erste darzustellende Molekuel

//const double PI=4*atan(1.0); //schon in vektor3dklasse definiert
const int anz_kreispunkte=360; //also 1 Grad Aufloesung
const double xmin= -6, xmax=6, ymin= -4.5, ymax=4.5;
const double zmin= -10, zmax=10; //z-Achse vom Bildschirm gegen Betrachter zeigend
static double DX=(xmax-xmin)/bildbreite, DY=(ymax-ymin)/bildhoehe;
//#include "myfonts.cc"

static int solidflag=1;
static int ambientflag=1;
static int cylinderflag=1; //1: Bindungen als Zylinder zeichnen, 0:nur dicke Linien

static double kugelskal=1.0; //Groesse der Kugeln (Atome) skalieren

#define EDIT_PAUSE 0x100 //Pause-Bit zum Edit kurzfristig abschalten
#define MOLEK_ROT  1  //Werte fuer edit_modus
#define EDIT_ATOMS 2  //Werte fuer edit_modus
#define EDIT_BONDS 4  //Werte fuer edit_modus
static int edit_modus=0;
static int aktives_molekuel=0;

class Atomlist
{
public:
 char name[4];
 Vektor3d p;
 double ra; //radius
 uchar r,g,b; //Farbe
 int molekuelnr;
 void set(const char *nam,Vektor3d& v,int m);
};

void Atomlist::set(const char *nam,Vektor3d& v,int m)
{
 molekuelnr=m;
 p=v;
 strncpy(name,nam,3); name[3]=0;
 ra=1.0; //Radius in Angstrom (Kovalent-Radius)
 //vwra=1.5; //Vanderwaals-Radius? //TODO
 r=g=b=127; //mittleres Grau wenn keine andere Farbe definiert
 if(strlen(name)==1)
  {switch(name[0])
    {// Radius in Angstrom (0.1nm) (Angstrom = Picometer/100)
     case 'C': ra=0.772; r=g=b=50; break; //Kohlenstoff, fast schwarz
     case 'H': ra=0.373; r=g=b=200; break; //Wasserstoff, weiss (vwra=1,2)
     case 'D': ra=0.4;//TODO
      r=g=200; b=150; break; //Deuterium, fast weiss
     case 'B': ra=0.795; r=g=b=140; break; //Bor, grau
     case 'N': ra=0.549; r=50;g=50;b=255; break; //Stickstoff (vwra=1.55)
     case 'O': ra=0.604; r=255;g=b=0; break; //Sauerstoff (vwra=1.52)
     case 'F': ra=0.709; r=204;g=255;b=127; break; //Fluor
     case 'I': case 'J': ra=1.331; r=127;g=b=255; break; //Jod
     case 'S': ra=1.035; r=255;g=255;b=0; break; //Schwefel
     case 'P': ra=1.105; r=255;g=0;b=255; break; //Phosphor
     case 'K': ra=2.03; r=g=b=154; break; //Kalium (vwra=2.75)
     default: printf("unbekanntes Atom '%c'\n",name[0]);
    }
  }
 else if(strcmp(name,"Na")==0) {ra=1.858; r=g=b=154;} //Natrium
 else if(strcmp(name,"Cl")==0) {ra=0.994; r=0;g=255;b=0;} //Chlor
 else if(strcmp(name,"Br")==0) {ra=1.145; r=g=179;b=0;} //Brom
 else if(strcmp(name,"Ni")==0) {ra=1.24; r=g=b=154;} //Nickel
 else if(strcmp(name,"Li")==0) {ra=1.52; r=g=b=154;}
 else if(strcmp(name,"Mg")==0) {ra=1.60; r=g=b=154;}
 else if(strcmp(name,"Fe")==0) {ra=1.37; r=g=b=154;}
 else if(strcmp(name,"Cu")==0) {ra=1.32; r=0x8e;g=0x40;b=0x2a;}
 //TODO: weitere Elemente anfuegen
 else printf("Werte von Atom '%s' auf Default gesetzt\n",name);//test
}

class Bondlist
{
public:
 int molekuelnr;
 int n1,n2; //Nummern der Atome die miteinander verbunden sind
            //(eindeutige Numern ueber alle Molekuele = Index in atomliste[])
 int strong; //Staerke der Bindung: 1=einfach- 2=zweifach- 3=dreifach-Bindung
 void set(int a,int b,int c,int m) {n1=a; n2=b; strong=(c==0)?1:c; molekuelnr=m;}
};

#define MAXATOMS 1000 //maximale Anzahl Atome in allen Molekuelen
Atomlist atomliste[MAXATOMS]; //Liste aller Atome
int anz_atome=0; //Anzahl Atome in allen Molekuelen
#define MAXBONDS 2000
Bondlist bondliste[MAXBONDS]; //Liste aller Bindungen
int anz_bonds=0; //Anzahl Bindungen in allen Molekuelen

class Molekuel
{
public:
 Atomlist *atom;
 Bondlist *bond;
 int natoms,nbonds;
 int nr; //Molekuel-Nummer
 int ia; //erster Index in atomliste
 Molekuel() {natoms=nbonds=nr=0; atom=NULL; bond=NULL;}
 void init(int m,int maxa,int maxb)
   {atom = &atomliste[ia=anz_atome]; bond = &bondliste[anz_bonds];
    nr=m; anz_atome += maxa; anz_bonds += maxb; natoms=nbonds=0;}
 void addatom(const char *nam,Vektor3d& v);
 void addbond(int a,int b,int c);
 void drehen(int achse,int richtung);
 void schieben(Vektor3d v);
};

#define MAXM 10 //maximale Anzahl Molekuele
Molekuel molekuel[MAXM];
int molek_nr=0; //naechste freie Molekuelnummer

void Molekuel::addatom(const char *nam,Vektor3d& v)
{
 if(natoms==anz_atome) {printf("Fehler: zu viele Atome in Melekuel %d\n",nr); return;}
 atom[natoms++].set(nam,v,nr);
}

void Molekuel::addbond(int a,int b,int c)
{
 if(nbonds==anz_bonds) {printf("Fehler: zu viele Bindungen in Melekuel %d\n",nr); return;}
 bond[nbonds++].set(a,b,c,nr);
}

void Molekuel::schieben(Vektor3d v)
{
 for(int i=0;i<natoms;i++)  atom[i].p -= v;
}

/****** Vordeklarationen ******/
void xyzeinlesen(FILE *fp);
void molekuel_xyzspeichern(int nr,FILE *fp,const char *name);
void alles_xyzspeichern(FILE *fp,const char *name,int flag);
void pdb_einlesen(FILE *fp);

/**************************** kleinkram *******************************/
inline bool istja(char *s) {return (*s!='n' && *s!='N');}
inline bool istnein(char *s) {return (*s=='n' || *s=='N');}

bool getline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
	*s++ = c;
 *s='\0';
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

void strclean(char *str)
{
 char c, *t1=str;
 while((c= *t1)==' ' || c=='\r' || c=='\n') t1++; //fuerende Leerstellen ueberlesen
 int i=strlen(str);
 while(i>0 && str[--i]==' ') {str[i]=0;} //Leerstellen am Ende des Strings loeschen
 if(t1!=str) //wenn fuerende Leerstellen
  {
   while((c= *t1++)!=0) {*str++ = c;} //dann String entsprechend nach vorne schieben
   *str=0;
  }
}

bool fastgleich(int a,int b,int diff)
{
 int ab=a-b; if(ab<0) ab= -ab;
 return (ab<=diff);
}

void myprints(const char *cstr,const char *str)
{
 char txt[strlen(cstr)+strlen(str)+16];
 sprintf(txt,cstr,str);
 janeinrequester(txt);
}

void myprintd(const char *cstr,int n)
{
 char txt[strlen(cstr)+32];
 sprintf(txt,cstr,n);
 janeinrequester(txt);
}

/****** Requester ******/

//TODO: Funktionen der Pfeiltasten aendern

const char *hilfetext=
"    Hilfe\n\
    ----\n\n\
Pfeiltasten: Kamera verschieben:\n\
   Pfeil links/rechts: um y-Achse drehen\n\
   Pfeil oben/unten:   um x-Achse drehen\n\
   Tasten PageUp/Down: um z-Achse drehen\n\
   Taste Home: Kamera auf Anfangsposition ruecksetzen\n\
Maus mit linker Taste: Objekt drehen um x- und y-Achse\n\
Maus mit mittlerer Taste: Objekt drehen um z-Achse\n\
Maus mit rechter Taste: Objekt verschieben in x-y-Richtung\n\
Mausrad: Objekt von Kamera weg/hin bewegen (zoomen)\n\
Tastatur:\n\
  F: Koordinaten-Kreuz zeichnen/ausschalten\n\
  S: umschalten zw. Drahtkugeln oder Solide Kugeln\n\
  A: Ambientlicht ein/aus\n\
  K: Kugelgroessen skalieren\n\
  B: Bindungen als Zylinder oder als Dicke Linien zeichnen\n\
  H: diese Hilfe";

/****** richtiges Menu ******/
int menu_debug=1; //test
int exitflag=0; //zum Programm verlassen

void m_quit(long id)
{
 menu.stop_loop();
 exitflag=1; //erst in display() dann exit(0) machen!
}

void m_hilfe(int id)
{
 janeinrequester(hilfetext);
}

void m_open_file(long id)
{
 static char filename[200]="test.xyz";
 int ok=nachfilenamefragen("Lade Molekuel von xyz-Datei",filename,200);
 if(ok)
  {
   FILE *fp=fopen(filename,"r");
   if(fp==NULL) {myprints("kann \"%s\" nicht oeffnen.\n",filename); return;}
   xyzeinlesen(fp);
   fclose(fp);
   printf("\"%s\" eingelesen\n",filename);//test
  }
}

void m_save_file(long id)
{
 int ok, mnr= -1;
 static char filename[200]="default.xyz";
 static int defaultnr=0;
 if(defaultnr>0 && strncmp(filename,"default",7)==0) sprintf(filename,"default%d.xyz",defaultnr);
 if(molek_nr>1)
  {
   ok=janeinrequester3("Was soll gespeichert werden?","alles","nur 1 Molekuel","cancel");
   if(ok<0) {printf("cancel - nichts gespeichert\n"); return;}
   if(ok==0)
    {
     char txt[80]; sprintf(txt,"Molekuel-Nr: (0 - %d):",molek_nr-1);
     char antw[80]="0";
     mnr=0;
     //ok=requester_input(1,txt,"%d","%d",&mnr);
     ok=requester_input(1,txt,"%s","%s",antw);
     if(ok)
      {
       if(antw[0]=='*')
	{
	 mnr= -2;//alle Molekuele speichern
	}
       else
	{sscanf(antw,"%d",&mnr);
	 if(mnr>=molek_nr || mnr<0) {myprintd("ungueltige Molekuel-Nr: %d\n",mnr); ok=0;}
	}
      }
     else printf("Eingabe von Melekuel-Nr mit Cancel abgebrochen\n");//test
     if(!ok) return;
    }
   else
    {
     int ok2=janeinrequester("Alles als einzelnes Molekuel speichern\noder mehrere Molekuele in selber Datei?","einzelnes","mehrere");
     if(ok2==1) mnr= -1;//alles speichern als einziges Molekuel
     else       mnr= -2;//alle Molekuele speichern
    }
  }
 if(mnr== -2) ok=nachfilenamefragen("Speichere alle Molekuele in xyz-Datei",filename,200);
 else ok=nachfilenamefragen("Speichere Molekuel in xyz-Datei",filename,200);
 if(ok)
  {
   if(strncmp(filename,"default",7)==0) defaultnr++;
   FILE *fp=fopen(filename,"r");
   if(fp!=NULL)
    {
     fclose(fp);
     if(strncmp(filename,"default",7)==0)
      {
       do {sprintf(filename,"default%d.xyz",defaultnr++); fp=fopen(filename,"r"); if(fp!=NULL) fclose(fp);}
       while(fp!=NULL);
      }
     else {myprints("Datei \"%s\" schon vorhanden.\nNichts gespeichert.\n",filename); return;}
    }
   fp=fopen(filename,"w");
   if(fp==NULL)
    {myprints("kann \"%s\" nicht erstellen.\nNichts gespeichert.\n",filename);
     return;
    }
   if(mnr<0) alles_xyzspeichern(fp,filename,mnr);
   else molekuel_xyzspeichern(mnr,fp,filename);
   fclose(fp);
  }
}

static int undo_typ=0;
static int undo_nr,undo_mnr;
static char undo_atomname[4];
static Vektor3d undo_vektor;

void m_undo(long id)
{
 if(undo_typ==1) //m_atom_aendern zuruecknehmen?
  {
   atomliste[undo_nr].set(undo_atomname,undo_vektor,undo_mnr);
   undo_typ=0;
  }
 //else if(undo_typ==2) //TODO
 else janeinrequester("Undo Buffer empty\nnichts zu tun");
}

void m_atom_aendern(long id)
{
 static int nr=0;
 char txt[80]; sprintf(txt,"Atom-Nummer (0 - %d)",anz_atome-1);
 int ok=requester_input(1,txt,"%d","%d",&nr);
 if(ok)
  {
   if(nr>=anz_atome) {janeinrequester("falsche Atom-Nummer"); return;}
   char atomname[80]; Vektor3d p; int mnr;
   strcpy(atomname,atomliste[nr].name);
   p=atomliste[nr].p;
   mnr=atomliste[nr].molekuelnr;
   sprintf(txt,"Atom %d (in Molekuel %d)", nr, mnr);
   //sprintf(txt,"Atom %d", nr);
   ok=requester_input(4,
		      txt,"%s","%s\n",atomname,
		      "Position x","%12.6f","%lf",&p.x,
		      "Position y","%12.6f","%lf",&p.y,
		      "Position z","%12.6f","%lf",&p.z
		      );
   if(ok)
    {
     strclean(atomname);
     if(strlen(atomname)>2 || atomname[0]==0 || islower(atomname[0]) || isupper(atomname[1]))
      {
       myprints(" ungueltiges Atom '%s' \n nichts veraendert",atomname);
      }
     else
      {
       undo_typ=1; undo_nr=nr; undo_mnr=atomliste[nr].molekuelnr;
       undo_vektor=atomliste[nr].p; strncpy(undo_atomname,atomliste[nr].name,4);
       atomliste[nr].set(atomname,p,mnr);
      }
    }
  }
}

void m_bindung(long id)
{
 if(menu_debug) printf("Menu \"m_bindung\" aufgerufen (id=%ld)\n",id);//test
 static int bnr=0,anr1=0,anr2=0;
 Bondlist bond;
 char txt[80]; sprintf(txt,"Bindungs-Nummer (0 - %d)",anz_bonds-1);
 int ok=janeinrequester(" Bindungs-Nummer oder Atom-Nummern ? ","Bindungs","Atom-Nr");
 int i,flag=0;
 if(ok==1) {ok=requester_input(1,txt,"%d","%d",&bnr); flag=1;}
 else      {ok=requester_input(2,"Atom-Nr 1","%d","%d",&anr1,"Atom-Nr 2","%d","%d",&anr2);}
 if(ok)
  {
   if(flag==1)
    {
     if(bnr>=anz_bonds) {janeinrequester("falsche Bindungs-Nummer"); return;}
     bond=bondliste[bnr];
     anr1=bond.n1; anr2=bond.n2;
    }
   else
    {
     for(i=0;i<anz_bonds;i++)
      if(bondliste[i].n1==anr1 && bondliste[i].n2==anr2) break;
     if(i==anz_bonds) {janeinrequester("keine entsprechende Bindung vorhanden"); return;}
     bnr=i;
     bond=bondliste[bnr];
    }
   Vektor3d p; p=atomliste[anr1].p-atomliste[anr2].p;
   double abstand=abs(p);
   ok=requester_input(2,"Abstand anr1 - anr2","%f","%lf\n",&abstand,
		      "Bindung 1-, 2- oder 3-fach","%d","%d",&bond.strong);
   if(ok)
    {
     if(bond.strong>=1 && bond.strong<=3)
       bondliste[bnr].strong=bond.strong;
     else {}//TODO
    }
  }
}

void m_ueber(int id)
{
 char text[80];
 sprintf(text,"molz.cc\n%s",VERSION);
 int ok=janeinrequester(text);
 printf("m_ueber(%d) ok=%d\n",id,ok);//test
}

void m_edit_off(long id)
{
 if(menu_debug) printf("Menu \"m_edit_off\" aufgerufen (id=%ld)\n",id);//test
 edit_modus=0;
}

void m_molek_rot(long id)
{
 int nr=0;
 static char txt[80];
 sprintf(txt,"Molekuel-Nr: (0 - %d): ",molek_nr-1);
 //printf("%s",txt); scanf("%d",&nr);//provi.
 int ok=requester_input(1,txt,"%d","%d",&nr);
 if(ok!=1) {janeinrequester("Abbruch - Editmodus off\n"); edit_modus=0; return;}
 if(nr>=0 && nr<molek_nr)
  {
   aktives_molekuel=nr;
   edit_modus=MOLEK_ROT;
  }
 else
  {myprintd(" falsche Molekuel-Nr (%d)\n Editmodus off\n",nr); edit_modus=0;}
}

/** eigene Routine zum Kreise und Kugel zeichnen: ** /
static Vektor3d einheitskreis[anz_kreispunkte];

void init_einheitskreis()
{
 Vektor3d p;
 for(int i=0;i<anz_kreispunkte;i++)
  {
   double alfa=2*PI/anz_kreispunkte*i;
   p.x=cos(alfa); p.y=sin(alfa); p.z=0;
   einheitskreis[i]=p;
  }
}

void zeichne_kugel(double radius,double x0,double y0,double z0)
{
 radius *= kugelskal;
 Vektor3d p;
 const int anz_rot=18;
 double rotwinkelschritt=PI/anz_rot;
 for(int j=0;j<anz_rot;j++) //Laengenlinien
  {
   double phi=rotwinkelschritt*j;
   double cosphi=cos(phi), sinphi=sin(phi);
   glBegin(GL_LINE_LOOP);
   for(int i=0;i<anz_kreispunkte;i++)
    {
     p=einheitskreis[i];
     double x1=p.x*radius; 
     p.x = x1*cosphi + x0;
     p.y = p.y*radius + y0;
     p.z = x1*sinphi + z0;
     glVertex3d(p.x, p.y, p.z);
    }
   glEnd();
  }
 for(int j=1;j<anz_rot;j++) //Breitenlinien
  {
   double r=sin(j*rotwinkelschritt)*radius;
   p.y = cos(j*rotwinkelschritt)*radius + y0;
   glBegin(GL_LINE_LOOP);
   for(int i=0;i<anz_kreispunkte;i++)
    {
     p.x = einheitskreis[i].x*r + x0;
     p.z = einheitskreis[i].y*r + z0;
     glVertex3d(p.x, p.y, p.z);
    }
   glEnd();
  }
}
/ ** :eigene Routine zum Kreise und Kugel zeichnen **/

/** einfacher mit Glut: **/
void zeichne_kugel(double radius,double x,double y,double z)
{
 glPushMatrix();
 glTranslated(x,y,z);
 glutWireSphere(radius*kugelskal, 36, 18);
 glPopMatrix();
}

void zeichne_bindung(int a,int b,int c)
{
 //double radius=0.08+0.02*c; //willkuerlicher Wert
 Vektor3d p1,p2,pa;
 //double r1,r2;
 p1=atomliste[a].p; //r1=atomliste[a].ra;
 p2=atomliste[b].p; //r2=atomliste[b].ra;
 glColor3ub(50,50,50); // Farbe setzen, dunkelgrau
 if(cylinderflag==0)
  {
   //float dicke=4.0; if(c==2) dicke=8; else if(c>=3) dicke=20;
   float dicke=bildbreite/200; if(c==2) dicke*=2; else if(c>=3) dicke*=5;
   glLineWidth(dicke); //dicke Linien
   glBegin(GL_LINES);
   glVertex3d(p1.x, p1.y, p1.z);
   glVertex3d(p2.x, p2.y, p2.z);
   glEnd();
   glLineWidth(1.0); //duenne Linien
  }
 else
  {
   //Bindung als Zylinder um Verbindungslinie p1--p2 zeichnen:
   GLUquadric *quad=gluNewQuadric();
   double dicke = 0.373/4; //Zylinder-Radius = 1/4 von H-Atom-Radius
   if(c==2) dicke*=1.75; else if(c>=3) dicke*=2.5;
   pa=p2-p1;
   double zylhoehe=abs(pa);
   glPushMatrix();
   glTranslated(p1.x,p1.y,p1.z);
   float alfa= -atan2(pa.x,pa.y)/GRAD; //Vorzeichen ist Glueckssache
   glRotatef(alfa,0,0,1);
   float beta= -acos(pa.z/zylhoehe)/GRAD; //Vorzeichen ist Glueckssache
   glRotatef(beta,1,0,0);
   gluCylinder(quad,dicke,dicke,zylhoehe,36,100);
   glPopMatrix();
  }
}

void zeichne_solidekugel(double radius,double x,double y,double z)
{
 glPushMatrix();
 glTranslated(x,y,z);
 glutSolidSphere(radius*kugelskal, 36, 18);
 glPopMatrix();
}

void zeichne_molekuel()
{
 Atomlist *p=atomliste;
 for(int i=0;i<anz_atome;i++)
  {
   glColor3ub(p[i].r, p[i].g, p[i].b); // Farbe setzen
   if(solidflag==0)
    zeichne_kugel(p[i].ra, p[i].p.x, p[i].p.y, p[i].p.z);
   else
    zeichne_solidekugel(p[i].ra, p[i].p.x, p[i].p.y, p[i].p.z);
  }
 Bondlist *b=bondliste;
 for(int i=0;i<anz_bonds;i++)
  {
   zeichne_bindung(b[i].n1, b[i].n2, b[i].strong);
  }
}

void zeichne_koordinatenkreuz()
{
 glColor3ub(0,0,0); // Farbe setzen, schwarz
 glBegin(GL_LINES); // GL einschalten Modus: LINIE
 glVertex3f(xmin,0,0); // erst der x-wert (links), y=0
 glVertex3f(xmax,0,0); // dann Linie nach rechts ziehen
 glVertex3f(0,ymin,0); // y-Wert unten Mitte
 glVertex3f(0,ymax,0); // y nach oben hochziehen
 glVertex3f(0,0,zmin); // z-Wert vorne
 glVertex3f(0,0,zmax); // z nach hinten ziehen
 glEnd(); // fertig, GL ausschalten
}

//static Vektor3d eye(2.0, 2.5, 10.0); //Kameraposition
static Vektor3d eye(0.0, 0.0, 10.0); //Kameraposition
static Vektor3d cent(0.0, 0.0, 0.0); //Punkt auf den die Kamera schaut
static Vektor3d up(0.0 , 1.0, 0.0); //Oben-Richtung der Kamera
static int fadenkreuz=0; //1=Koordinatenkreuz zeichnen

void kamera_reset()
{
 //eye=Vektor3d(2.0, 2.5, 10.0);
 eye=Vektor3d(0.0, 0.0, 10.0);
 cent.x=cent.y=cent.z=0;
 up.x=up.z=0; up.y=1;
}

/**** Menu und eventuell Requester zeichnen: ****/

void menu_text_zeichnen() //Menu zeichnen und evt. auch noch andere Texte
{
 glMatrixMode(GL_PROJECTION);
#ifdef BELEUCHTUNG
 glDisable(GL_LIGHTING);
#endif
 glPushMatrix();
 glLoadIdentity();
 gluOrtho2D(0, bildbreite, 0, bildhoehe);

 /*
 Mit dem DEPTH_BUFFER wird im Gegensatz zum normalen Verhalten das zuerst
 gezeichnete behalten.
 Schon gezeichnete Pixel werden also nicht ueberzeichnet.
 Also z.B. schwarze Schrift auf hellgrauem Hintergrund zuerst die Schrift,
 dann erst den Hintergrund zeichnen!
 */

 if(requester1.aktiv)
   requester1.zeichnen();
 if(menu.aktiv)
   menu.zeichnen();
 else if(menu.anzahl>0)
  {
   double x=REQ_RAND, y=bildhoehe-MBH+3;
   menu.mtextsize();
   glColor3ub(0,0,0); // Schwarze Schrift
   schrift(x,y,"Menuleiste"); //auskommentieren fuer Menu unsichtbar wenn nicht aktiv
  }

#ifdef BELEUCHTUNG
 glEnable(GL_LIGHTING);
#endif
 glPopMatrix();
 glMatrixMode(GL_MODELVIEW);
}

void display(void)
{
 if(exitflag) exit(0);
 glClearColor(0.5, 0.5, 0.5, 1); //Hintergrundfarbe grau
 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
 
#ifdef BELEUCHTUNG
 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
#endif
#ifdef BELEUCHTUNG
  GLfloat light_pos[]={xmax,ymax,zmax,1};
  GLfloat ambient_hell[]={0.25,0.25,0.25,1};
  GLfloat ambient_dunkel[]={0,0,0,1};
  glLightfv(GL_LIGHT0,GL_POSITION,light_pos);
  if(ambientflag==0) glLightfv(GL_LIGHT0,GL_AMBIENT,ambient_dunkel);
  else               glLightfv(GL_LIGHT0,GL_AMBIENT,ambient_hell);
  glEnable(GL_COLOR_MATERIAL);
#endif

 menu_text_zeichnen(); //Menu und event. andere Texte (z.B. Requester) zeichnen
 
 glMatrixMode(GL_MODELVIEW);
 glLoadIdentity();
 glPushMatrix(); // GL_MODELVIEW is default
 glScalef(1.0, bildbreite/(float)bildhoehe, 1.0);
 //glTranslatef(-xmin/10, -ymin/10, -zmin/10);
 
 //gluLookAt(0, 0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
 //gluLookAt(eye[0], eye[1], eye[2], cent[0], cent[1], cent[2], up[0],up[1],up[2]);
 gluLookAt(eye.x, eye.y, eye.z, cent.x, cent.y, cent.z, up.x,up.y,up.z);
 
 glEnable(GL_BLEND); //antialias
 glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); //antialias
 glEnable(GL_LINE_SMOOTH); //antialias
 
 zeichne_molekuel();
 if(fadenkreuz!=0) zeichne_koordinatenkreuz();
 glPopMatrix();
 glutSwapBuffers();
};

/* Idle proc. Redisplays, if called. *  /
void idle(void)
{
 static int ntest=0;//test
 printf("idle() %d.Aufruf\n",++ntest);//test
 glutPostRedisplay();
};
*/

/*
void mpause(int n) //n millisekunden warten
{
 usleep(n*1000); //soviele Mikrosekunden warten
}
*/

void key(unsigned char c, int x, int y)
{
 if(requester1.aktiv)
   requester1.key(c,x,y);
 else if(menu.aktiv)
   menu.key(c,x,y);
 else
  {
   c=toupper(c);
   if(c=='?' || c=='H')
    {
     printf("\n%s\n",hilfetext);
    }
   else if(c=='F') fadenkreuz ^= 1;
   else if(c=='S') solidflag ^= 1;
   else if(c=='A') ambientflag ^= 1;
   else if(c=='K') {if(kugelskal<=0.25) kugelskal=1.0; else kugelskal-=0.25;}
   else if(c=='B') cylinderflag ^= 1;
   else printf("key(c='%c',x=%d,y=%d)\n",c,x,y);//test
  }
 glutPostRedisplay();
};

/*  Drehungen um xyz-Achsen des globalen Koordinatensystems
void kamera_drehen(int achse,int richtung)
{
 Vektor3d p; p=eye-cent;
 const double sina=sin(ZWEIPI/360), cosa=cos(ZWEIPI/360);
 if(achse==0) //um x-Achse drehen
  {
   if(richtung==1) {p.drehenyz(sina,cosa); up.drehenyz(sina,cosa);}
   else            {p.drehenzy(sina,cosa); up.drehenzy(sina,cosa);}
  }
 else if(achse==1) //um y-Achse drehen
  {
   if(richtung==1) {p.drehenzx(sina,cosa); up.drehenzx(sina,cosa);}
   else            {p.drehenxz(sina,cosa); up.drehenxz(sina,cosa);}
  }
 else if(achse==2) //um z-Achse drehen
  {
   if(richtung==1) {p.drehenyx(sina,cosa); up.drehenyx(sina,cosa);}
   else            {p.drehenxy(sina,cosa); up.drehenxy(sina,cosa);}
  }
 eye=p+cent;
}
*/

/* Drehungen mit Koordinatensystem von Kamera aus gesehen */
void kamera_drehen(int achse,int richtung)
{
 if(edit_modus==MOLEK_ROT)
  {
   molekuel[aktives_molekuel].drehen(achse,richtung);
   return;
  }
 Vektor3d p; p=eye-cent;
 Quaternion q;
 const double dw=ZWEIPI/360;
 if(achse==0) //um x-Achse drehen
  {
   Vektor3d k;
   k=vektorkreuzprodukt(p,up);
   p=q.vektordrehen(p,k,-richtung*dw);
   up=q.vektordrehen(up);
   eye=p+cent;
  }
 else if(achse==1) //um y-Achse drehen
  {
   p=q.vektordrehen(p,up,richtung*dw);
   eye=p+cent;
  }
 else if(achse==2) //um z-Achse drehen
  {
   up=q.vektordrehen(up,p,richtung*dw);
  }
}

void Molekuel::drehen(int achse,int richtung)
{
 Vektor3d p;
 Quaternion q;
 const double dw=ZWEIPI/360;
 if(achse==0) //um x-Achse drehen
  {
   Vektor3d k;
   k=vektorkreuzprodukt(eye-cent,up);
   q.drehung_definieren(k,richtung*dw);
  }
 else if(achse==1) //um y-Achse drehen
  {
   q.drehung_definieren(up,-richtung*dw);
  }
 else if(achse==2) //um z-Achse drehen
  {
   q.drehung_definieren(eye-cent,-richtung*dw);
  }
 for(int i=0;i<natoms;i++)
  {
   p = atom[i].p-cent;
   p = q.vektordrehen(p);
   atom[i].p = p+cent;
  }
}

#define PFEIL_LINKS 100
#define PFEIL_OBEN 101
#define PFEIL_RECHTS 102
#define PFEIL_UNTEN 103
#define PAGE_UP 104
#define PAGE_DOWN 105
#define KEY_HOME 106
#define KEY_END 107
#define KEY_INSERT 108

#define KEY_DEL 111
#define KEY_SHIFTL 112
#define KEY_SHIFTR 113

void spezial(int key,int x,int y) //Tastatur spezialzeichen
{
 if(requester1.aktiv)
   requester1.spezial(key,x,y);
 else if(menu.aktiv)
   menu.spezial(key,x,y);
 else
  {
   int flag=0;
   //TODO: Funktionen der Pfeiltasten aendern
   if(key==PFEIL_LINKS)       kamera_drehen(1,flag= 1);
   else if(key==PFEIL_RECHTS) kamera_drehen(1,flag= -1);
   else if(key==PFEIL_OBEN)   kamera_drehen(0,flag= 1);
   else if(key==PFEIL_UNTEN)  kamera_drehen(0,flag= -1);
   else if(key==PAGE_UP)      kamera_drehen(2,flag= 1);
   else if(key==PAGE_DOWN)    kamera_drehen(2,flag= -1);
   else if(key==KEY_HOME && edit_modus!=MOLEK_ROT)  kamera_reset();
   //else if(key==KEY_END)      {eye.z -= 0.1; flag=1;}
   else if(key==KEY_SHIFTL || key==KEY_SHIFTR)  {edit_modus ^= EDIT_PAUSE;}
   else printf("spezial(key=%d,x=%d,y=%d)\n",key,x,y);//test
   /*
   if(flag!=0) //test
    {
     printf("\n eyeX Y Z: %f %f %f\n",eye.x,eye.y,eye.z);
     printf("centX Y Z: %f %f %f\n",cent.x,cent.y,cent.z);
     printf("  upX Y Z: %f %f %f\n",up.x,up.y,up.z);
    }
   */
  }
 glutPostRedisplay();
}

void reshape(int w, int h)
{
 glViewport(0, 0, w, h);
 bildbreite=w; bildhoehe=h;
 DX=(xmax-xmin)/bildbreite; DY=(ymax-ymin)/bildhoehe;
 /*
 printf("reshape() bildbreite=%d bildhoehe=%d\n",w,h);//test
 if(bildbreite>1240) //test
  {menu.grosserfont(2); requester1.grosserfont(2);}//test
 else //test
  {menu.grosserfont(1); requester1.grosserfont(1);}//test
 */
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 gluPerspective(45.0, 1.0, 0.001, 1000);
 gluLookAt(0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
 glMatrixMode(GL_MODELVIEW);
};

class Intvektor
{
public:
 int x,y;
 Intvektor() {x=y=0;}
 Intvektor(int a,int b) {x=a; y=b;}
 Intvektor& operator-=(Intvektor p) {x-=p.x; y-=p.y; return *this;}
 Intvektor& operator/=(int n) {x/=n; y/=n; return *this;}
};

static Intvektor maus_pos; //Position der Maus beim druecken einer Taste
static int maus_knopf= -1; //aktuell gedrueckter Mausknopf (-1 wenn keiner gedrueckt)

//provisorisch:
#define MAXDISTANZ 100
#define DISTSTEP 0.5

void mausknopf(int button,int state,int x,int y)
{
 static int x0=0,y0=0;
 if(requester1.aktiv)
   requester1.mausknopf(button,state,x,y);
 if(menu.aktiv ||                     //wenn aktives Menu
    (y<MBH && button==0 && state==0)) //oder linker Mausknopf im Menubalken gedrueckt
   menu.mausknopf(button,state,x,y);
 else
  {
   if(button==0 && state==GLUT_DOWN) {x0=x; y0=y;}
   if(button==0 && state==GLUT_UP && fastgleich(x,x0,1) && fastgleich(y,y0,1))
    { //wenn linker Mausknopf losgelassen an gleicher Stelle wie gedrueckt
     printf("atomklick: x=%d y=%d\n",x,y);//test
     //TODO: wenn auf ein Atom geklickt, Atom oder Bindung editieren
     maus_knopf= -1;
    }
   else if(button==0 || button==1 || button==2) //linke, mittlere oder rechte Maustaste
    {
     Intvektor p(x,y);
     if(state==1) {maus_knopf= -1;} //beim loslassen
     else {maus_pos=p; maus_knopf=button;}
    }
   else if((button==3 || button==4) && state==0) //Mausrad rauf oder runter
    {
     Vektor3d p;
     p=eye-cent;
     double distanz=abs(p),neudist=distanz;
     if(button==3 && distanz>=2*DISTSTEP) //naeher ans Objekt ran
      {
       neudist -= DISTSTEP;
       p *= neudist/distanz;
       eye=cent+p;
      }
     else if(button==4 && distanz<MAXDISTANZ)
      {
       neudist += DISTSTEP;
       p *= neudist/distanz;
       eye=cent+p;
      }
     //printf("Distanz der Kamera zum Blickpunkt: %f\n",neudist);//test
    }
  }
 glutPostRedisplay();
}

void mausmotion(int x,int y)
{
 if(requester1.aktiv)
   requester1.mausmotion(x,y);
 else if(menu.aktiv)
   menu.mausmotion(x,y);
 else
  {
   if(maus_knopf==0) //linke Maustaste: Drehen um y-Achse oder um x-Achse
    {
     Intvektor p0(x,y),p1(x,y),p;
     p1 -= maus_pos;
     p=p1; p /= 2; //mindestens 2 Pixel bewegen
     if(abs(p.x)>0) //um y-Achse drehen
      {
       kamera_drehen(1, (p1.x<0) ? 1 : -1);
       maus_pos=p0;
      }
     if(abs(p.y)>0) //um x-Achse drehen
      {
       kamera_drehen(0, (p1.y<0) ? 1 : -1);
       maus_pos=p0;
      }
    }
   else if(maus_knopf==1) //mittlere Maustaste: Drehen um z-Achse
    {
     Intvektor p0(x,y),p1(x,y),p;
     p1 -= maus_pos;
     p=p1; p /= 2; //mindestens 2 Pixel bewegen
     if(abs(p.x)>0) //um z-Achse drehen
      {
       //kamera_drehen(2, (p1.x<0) ? 1 : -1);
       int h = (y<bildhoehe/2) ? -1 : 1;
       kamera_drehen(2, (p1.x<0) ? h : -h);
       maus_pos=p0;
      }
     if(abs(p.y)>0) //um z-Achse drehen
      {
       //kamera_drehen(2, (p1.y<0) ? 1 : -1);
       int h = (x<bildbreite/2) ? 1 : -1;
       kamera_drehen(2, (p1.y<0) ? h : -h);
       maus_pos=p0;
      }
    }
   else if(maus_knopf==2) //rechte Maustaste: in x-y-Ebene bewegen
    {
     Intvektor p0(x,y),p1(x,y),p;
     p1 -= maus_pos;
     p=p1; p /= 5; //mindestens 5 Pixel bewegen
     if(abs(p.x)>0) //links-rechts Bewegung
      {
       //eye.x -= p1.x*DX;  cent.x -= p1.x*DX; //auf globalem Koordinatensystem schieben
       double anzpixel=p1.x*DX;
       Vektor3d kp; kp = vektorkreuzprodukt(eye-cent,up); kp.einheitsvektor();
       kp *= anzpixel;
       if(edit_modus==MOLEK_ROT)
	{molekuel[aktives_molekuel].schieben(kp);} //Molekuel nach rechts/links schieben
       else
	{eye += kp;  cent += kp;} //Kamera senkrecht zu Sichtlinie nach rechts/links schieben
       maus_pos=p0;
      }
     if(abs(p.y)>0) //oben-unten Bewegung
      {
       //eye.y += p1.y*DY;  cent.y += p1.y*DY; //auf globalem Koordinatensystem schieben
       double anzpixel=p1.y*DY; Vektor3d kp; kp=up*anzpixel;
       if(edit_modus==MOLEK_ROT)
	{molekuel[aktives_molekuel].schieben(kp);} //Molekuel nach oben/unten schieben
       else
	{eye += kp;  cent += kp;} //senkrecht zu Sichtlinie nach oben/unten schieben
       maus_pos=p0;
      }
    }
   else
    {
     //printf("mausmotion(x=%d,y=%d) maus_knopf=%d\n",x,y,maus_knopf);//test
    }
  }
 glutPostRedisplay();
}

void mausmotion2(int x,int y) //Mausbewegungen ohne gedrueckte Taste
{
 if(requester1.aktiv)
  {requester1.mausmotionpassiv(x,y); glutPostRedisplay();}

 else if(menu.aktiv)
  {menu.mausmotionpassiv(x,y); glutPostRedisplay();}
 else
  {
  }
 //glutPostRedisplay();
}

int win0;

void grafik_init(const char *titel)
{
 int screenWidth  = glutGet(GLUT_SCREEN_WIDTH);
 int screenHeight = glutGet(GLUT_SCREEN_HEIGHT);
 //printf("screenWidht/Height = %d / %d\n",screenWidth,screenHeight);//test
 if(screenWidth>2*bildbreite && screenHeight>2*bildhoehe)
  {bildbreite *= 2; bildhoehe *= 2; bildposx *= 2; bildposy *= 2;}
 if(screenWidth>2500) {menu.grosserfont(2); requester1.grosserfont(2);}
 //glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE);
 glutInitDisplayMode(GLUT_DOUBLE|GLUT_RGB|GLUT_DEPTH);
 glutInitWindowSize(bildbreite,bildhoehe);
 glutInitWindowPosition(bildposx,bildposy);
 win0=glutCreateWindow(titel);

 setmenu(3,"File",         "Edit",            "Hilfe"); // Menuleiste
 setmenu(3,"Open File...", "Undo",            "Ueber",  // Menupunkte
	   m_open_file,    m_undo,            m_ueber); // Zeiger auf Funktionen
 setmenu(3,"Save File...", "Atom aendern...", "Hilfe  H",
	   m_save_file,    m_atom_aendern,    m_hilfe);
 setmenu(2,"Quit  Q",   "Bindung aendern...", m_quit, m_bindung);
 setmenu(2, NULL,       "Molekuel rotieren...", NULL, m_molek_rot);
 setmenu(2, NULL,       "Editmodus off", NULL, m_edit_off);

 glShadeModel(GL_SMOOTH);
 glEnable(GL_DEPTH_TEST);

 /* Register GLUT callbacks. */
 glutDisplayFunc(display);
 glutKeyboardFunc(key);
 glutMouseFunc(mausknopf);
 glutMotionFunc(mausmotion);
 glutPassiveMotionFunc(mausmotion2);
 glutSpecialFunc(spezial);
 glutReshapeFunc(reshape);
 //glutIdleFunc(idle);
 // zum Loslassen von Tasten:
 //glutKeyboardUpFunc(void (*func)(unsigned char key,int x,int y));
 //glutSpecialUpFunc(void (*func)(int key,int x, int y));

 /* Init the GL state */
 glLineWidth(1.0);
}

/************************** Hauptprogramm *****************************/
int myscanf(const char *str,int *zahl)
{
 while(*str!=0 && !isdigit(*str)) {str++;}
 if(isdigit(*str)) sscanf(str,"%d",zahl);
 else {*zahl=0; return 0;}
 return 1;
}

void xyzeinlesen(FILE *fp)
{
 char zeile[200],name[40];
 int i,natoms,nbonds;
 int anzmol=1;
 bool ok;
 do { //eventuell Parameter von erweitertem xyz-Format einlesen
 do {ok=getline(fp,zeile,200);}
 while(ok && (*zeile==0 || *zeile==';')); //Leerzeilen und Kommentare ignorieren
 for(i=0;zeile[i]==' ';i++) {}
 if(!isdigit(zeile[i]))
  {
   if(strncmp(&zeile[i],"mol",3)==0)
    {myscanf(&zeile[i+3],&anzmol);
     if(anzmol<1) {printf("Fehler in xyz-Datei: \"%s\"\n",zeile); anzmol=1;}
    }
   else if(strncmp(&zeile[i],"koo",3)==0)
    {int koordinatensystem;
     myscanf(&zeile[i+3],&koordinatensystem);
     //TODO
    }
   else if(strncmp(&zeile[i],"ATOM",4)==0) //vermutlich pdb-Format
    {
     pdb_einlesen(fp);
     return;
    }
  }
 } while(!isdigit(zeile[i])); //fertig Parameter einlesen
 while(anzmol>=1) { //eventulle mehrere Molekuele von selber Datei einlesen
 sscanf(zeile,"%d %d",&natoms,&nbonds);
 if(natoms<1) {printf("Fehler in xyzeinlesen(): keine Atome\n"); return;}
 if(anz_atome+natoms>MAXATOMS) {printf("zu viele Atome, auf %d begrenzt\n",MAXATOMS); return;}
 if(anz_bonds+nbonds>MAXBONDS) {printf("zu viele Bindungen, auf %d begrenzt\n",MAXBONDS); return;}
 Vektor3d v;
 int nrstart=anz_atome;
 if(molek_nr==MAXM) {printf("zu viele Molekuele, auf %d begrenzt\n",MAXM); return;}
 int molekuelnr=molek_nr++;
 molekuel[molekuelnr].init(molekuelnr,natoms,nbonds);
 for(int i=0;i<natoms && getline(fp,zeile,200);)
  {
   if(*zeile==0 || *zeile==';') continue; //Leerzeilen und Kommentare ignorieren
   int n=sscanf(zeile,"%s %lf %lf %lf",name,&v.x,&v.y,&v.z);
   if(n!=4) printf("Fehler: zeile=\"%s\"\n",zeile);//test
   v.z = -v.z; //weil Koordinatensystem im OpenGL anders als in molz3d
   if(strlen(name)>2) printf("Fehlerhaftes Atom: \"%s\"\n",name);
   molekuel[molekuelnr].addatom(name,v);
   i++;
  }
 int n1,n2,c;
 for(int i=0;i<nbonds && getline(fp,zeile,200);)
  {
   if(*zeile==0 || *zeile==';') continue; //Leerzeilen und Kommentare ignorieren
   c=0;
   sscanf(zeile,"%d %d %d",&n1,&n2,&c);
   molekuel[molekuelnr].addbond(n1+nrstart,n2+nrstart,c);
   i++;
  }
 if(molekuelnr>0)
  { //bei mehreren Molekuelen: neues so verschieben dass nicht mit andern ueberlagert
   Vektor3d p0,q0; //groesste und kleinste Position der bisherigen Atome
   int i=0;
   q0=p0=atomliste[i++].p;
   for(;i<nrstart;i++) //alle Atome der bisherigen Molekuele durchsuchen
    {
     Vektor3d v=atomliste[i].p;
     if(v.x>p0.x) p0.x=v.x;
     if(v.y>p0.y) p0.y=v.y;
     if(v.z>p0.z) p0.z=v.z;
     if(v.x<q0.x) q0.x=v.x;
     if(v.y<q0.y) q0.y=v.y;
     if(v.z<q0.z) q0.z=v.z;
    }
   Vektor3d p1,q1; //groesste und kleinste Position der Atome des neuen Molekuels
   q1=p1=atomliste[i++].p;
   for(;i<anz_atome;i++) //alle Atome des neuen Molekuels
    {
     Vektor3d v=atomliste[i].p;
     if(v.x>p1.x) p1.x=v.x;
     if(v.y>p1.y) p1.y=v.y;
     if(v.z>p1.z) p1.z=v.z;
     if(v.x<q1.x) q1.x=v.x;
     if(v.y<q1.y) q1.y=v.y;
     if(v.z<q1.z) q1.z=v.z;
    }
   Vektor3d p3(2.5, 2.5, 2.5);
   p0 -= q1;
   p1 -= q0;
   p0 += p3; //mindestens 2.5 Angstroem Abstand zu bisherigen Molekuelen
   q0 -= p3; //oder auch 2.5 wenn in negative Richtung verschoben werden soll
   //Um in die Richtung zu schieben, wo es moeglchst wenig weit ist, den kleinsten
   //Wert von den 6 moeglichen (p0.x p0.y p0.z -q0.x -q0.y -q0.z) ermitteln:
   if(p0.x<p0.y && p0.x<p0.z && p0.x< -q0.x && p0.x< -q0.y && p0.x< -q0.z)
    p0.y=p0.z=0;
   else if(p0.y<p0.x && p0.y<p0.z && p0.y< -q0.x && p0.y< -q0.y && p0.y< -q0.z)
    p0.x=p0.z=0;
   else if(p0.z<p0.x && p0.z<p0.y && p0.z< -q0.x && p0.z< -q0.y && p0.z< -q0.z)
    p0.x=p0.y=0;
   else
    {
     p3=p0; p0=q0; q0=p3; //p0 und q0 vertauschen
     if(p0.x<p0.y && p0.x<p0.z && p0.x< -q0.x && p0.x< -q0.y && p0.x< -q0.z)
      p0.y=p0.z=0;
     else if(p0.y<p0.x && p0.y<p0.z && p0.y< -q0.x && p0.y< -q0.y && p0.y< -q0.z)
      p0.x=p0.z=0;
     else //if(p0.z<p0.x && p0.z<p0.y && p0.z< -q0.x && p0.z< -q0.y && p0.z< -q0.z)
      p0.x=p0.y=0;
    }
   for(int i=nrstart;i<anz_atome;i++) //neues Molekuel entsprechend verschieben
      atomliste[i].p += p0;
  }
 if(--anzmol>=1)
  {
   do {ok=getline(fp,zeile,200);}
   while(ok && (*zeile==0 || *zeile==';')); //Leerzeilen und Kommentare ignorieren
  }
 } //ende while(anzmol>=1)
}

void molekuel_xyzspeichern(int nr,FILE *fp,const char *name)
{
 Molekuel *p= &molekuel[nr];
 int i;
 fprintf(fp,"%d  %d\n",p->natoms,p->nbonds);
 for(i=0;i<p->natoms;i++)
  {
   fprintf(fp," %-2.2s %12.6f %12.6f %12.6f\n",
	   p->atom[i].name, p->atom[i].p.x, p->atom[i].p.y, -p->atom[i].p.z);
   //z-Wert negativ damit rueckwaertskompatibel zu molz3d
  }
 int ia=p->ia;
 for(i=0;i<p->nbonds;i++)
  {
   if(p->bond[i].strong==1)
    fprintf(fp,"%d %d\n",p->bond[i].n1-ia, p->bond[i].n2-ia);
   else
    fprintf(fp,"%d %d %d\n",p->bond[i].n1-ia, p->bond[i].n2-ia, p->bond[i].strong);
  }
 if(name!=NULL) printf("Molekuel %d als \"%s\" gespeichert.\n",nr,name);
}

void alles_xyzspeichern(FILE *fp,const char *name,int flag)
{
 int i;
 if(flag== -1) //alles als einziges Molekuel speichern?
  {
   fprintf(fp,"%d  %d\n",anz_atome,anz_bonds);
   for(i=0;i<anz_atome;i++)
    {
     fprintf(fp," %-2.2s %12.6f %12.6f %12.6f\n",
	     atomliste[i].name, atomliste[i].p.x, atomliste[i].p.y, -atomliste[i].p.z);
     //z-Wert negativ damit rueckwaertskompatibel zu molz3d
    }
   for(i=0;i<anz_bonds;i++)
    {
     if(bondliste[i].strong==1)
      fprintf(fp,"%d %d\n",bondliste[i].n1, bondliste[i].n2);
     else
      fprintf(fp,"%d %d %d\n",bondliste[i].n1, bondliste[i].n2, bondliste[i].strong);
    }
   printf("Alles als einziges Molekuel in \"%s\" gespeichert.\n",name);
  }
 else
  { //alle Molekuele in selber Datei speichern
   if(molek_nr>1) fprintf(fp,"molekuele=%d  ;Anzahl Molekuele in dieser Datei\n",molek_nr);
   for(i=0;i<molek_nr;i++)
     molekuel_xyzspeichern(i,fp,NULL);
   printf("Alle Molekuele in \"%s\" gespeichert.\n",name);
  }
}

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 3
static char argflag[128];
void setargflags(const char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
  }
}
/**************** Routinen zur Parameterauswertung ********************/

void hauptschlaufe(int nr);

int main(int argc, char **argv)
{
 int i,j,c;
 char quellname1[80],quellname2[80],quellname3[80];
 quellname1[0]=quellname2[0]=quellname3[0]=0;
 FILE *fp;
 for(j=0,i=1;i<argc;i++)
  {if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
   else	{if(++j==1) strcpy(quellname1,argv[i]);
         else if(j==2) strcpy(quellname2,argv[i]);
         else if(j==3) strcpy(quellname3,argv[i]);
  }	}
 if(argflag['?'] || j>MAXARG)
	{
	 printf("molz %s\n",VERSION);
	 printf("Anwendung: molz [Quellname1] [Quellname2] [Quellname3]\n");
	 exit(0);
	}
 if(j==0 || quellname1[0]==0)
  {
   fp=fopen(filename0,"r");
   if(fp==NULL) {printf("kann \"%s\" nicht oeffnen.\n",filename0); exit(0);}
   else {xyzeinlesen(fp); fclose(fp);}
  }
 else
  {
   fp=fopen(quellname1,"r");
   if(fp!=NULL) {xyzeinlesen(fp); fclose(fp); printf("%s eingelesen\n",quellname1);}
   else printf("konnte \"%s\" nicht oeffnen\n",quellname1);
   if(j>=2)
    {
     fp=fopen(quellname2,"r");
     if(fp!=NULL) {xyzeinlesen(fp); fclose(fp); printf("%s eingelesen\n",quellname2);}
     else printf("konnte \"%s\" nicht oeffnen\n",quellname2);
    }
   if(j>=3)
    {
     fp=fopen(quellname3,"r");
     if(fp!=NULL) {xyzeinlesen(fp); fclose(fp); printf("%s eingelesen\n",quellname3);}
     else printf("konnte \"%s\" nicht oeffnen\n",quellname3);
    }
  }
 //init_einheitskreis();
 glutInit(&argc, argv);
 grafik_init("Molz - Molekuele zeichnen mit OpenGL");
 menu.startMainLoop();
 //glutMainLoop(); //wird schon in menu.startMainLoop() gemacht
 return 0; //wird nie erreicht
}

void pdb_einlesen(FILE *fp)
{
 Vektor3d v;
 char zeile[200],name[40];
 int nextflag=0;
 do
  {
   nextflag=0;
   int endflag=0;
   int nrstart=anz_atome;
   if(molek_nr==MAXM) {printf("zu viele Molekuele, auf %d begrenzt\n",MAXM); return;}
   int molekuelnr=molek_nr++;
   int i,natoms=1000,nbonds=0; //TODO: zuerst richtige Werte herausfinden
   molekuel[molekuelnr].init(molekuelnr,natoms,nbonds);
   for(i=0;i<natoms && getline(fp,zeile,200);)
    {
     if(strncmp(zeile,"ATOM",4)==0)
      {
       int gruppe;
       name[0]=zeile[13]; name[1]=0; //nur Atome mit 1-Buchstaben? (C,O,N,H)
       if(zeile[14]=='X') {} //TODO: ungueltiges Atom?
       else
	{
	 sscanf(&zeile[22],"%d %lf %lf %lf",&gruppe,&v.x,&v.y,&v.z);
	 molekuel[molekuelnr].addatom(name,v);
	 i++;
	}
      }
     else if(strncmp(zeile,"TER",3)==0)
      {
       nextflag=1; break;
      }
     else if(strncmp(zeile,"END",3)==0)
      {
       endflag=1; break;
      }
     else printf("unbekanntes Schluesselwort in pdb-Datei: \"%s\"\n",zeile);
    }
   if(i<natoms)
    {
     molekuel[molekuelnr].natoms=i;
     anz_atome -= natoms-i;
    }
   else if(nextflag!=1 && endflag!=1)
    {
     printf("Fehler in pdb-lesen: fehlendes END oder zu viele Atome pro Molekuel\n");
     printf("natoms=%d nrstart=%d\n",natoms,nrstart);//test
    }
   //TODO: Bindungen herausfinden
  }
 while(nextflag!=0);
}
