//astro.cc
#include "astro.h"
const double p2=ZWEIPI;

double frac(double x) {if((x-=long(x))<0.) return x+1.0; else return x;}
double cubr(double x) {if(x==0.) return 0.;  return exp(log(x)/3.0);}

char *hms(double t)
{
 static char str[16];
 int sig; if(t<0) {t= -t; sig= -1;} else sig=1;
 long ut=long(3600*t+0.5);
 int s=ut%60;
 ut/=60;
 sprintf(str,"%2d:%02d:%02d",(int)(sig*ut/60),(int)(ut%60),s);
 return str;
}
char *gms(double t)
{
 static char str[16];
 int sig; if(t<0) {t= -t; sig= -1;} else sig=1;
 long ut=long(3600*t+0.5);
 int s=ut%60;
 ut/=60;
 sprintf(str,"%3d%02d'%02d\"",(int)(sig*ut/60),(int)(ut%60),s);
 return str;
}

char *planet_name(int plan)
{
 switch(plan)
  {case 0: return "Sun";	case 1: return "Mercury";
   case 2: return "Venus";	case 3: return "Earth";
   case 4: return "Mars";	case 5: return "Jupiter";
   case 6: return "Saturn";	case 7: return "Uranus";
   case 8: return "Neptune";	case 9: return "Pluto";
  }
 return "unknownPlanet";
}

FILE *fopen_num(char *name,char *rw)
{
 if(*rw!='r' || rw[1]!=0) return fopen(name,rw);
 static char *TMPFILE="ram:fopen_num.tmp";
 int c;
 FILE *fp=fopen(name,rw); if(fp==NULL) return NULL;
 FILE *fp2=fopen(TMPFILE,"w");
 if(fp2==NULL)
   {printf("can't open '%s'\n",TMPFILE); fclose(fp); return NULL;}
 for(;(c=getc(fp))!=EOF;)
   {if(c=='!') //Pascal-Kommentar
      {while((c=getc(fp))!=EOF && c!='\n') ;}
    if(c!=EOF) putc(c,fp2);
   }
 fclose(fp2); fclose(fp);
 return fopen(TMPFILE,rw);
}

/*-----------------------------------------------------------------------*/
/* CART: conversion of polar coordinates (r,theta,phi)                   */
/*       into cartesian coordinates (x,y,z)                              */
/*       (theta in [-90 deg,+90 deg]; phi in [-360 deg,+360 deg])        */
/*-----------------------------------------------------------------------*/
void cart(double r,double theta,double phi, double& x,double& y,double& z)
{
 double rcst=r*cs(theta);
 x = rcst*cs(phi); y = rcst*sn(phi); z = r*sn(theta);
}
/*-----------------------------------------------------------------------*/
/* POLAR: conversion of cartesian coordinates (x,y,z)                    */
/*        into polar coordinates (r,theta,phi)                           */
/*        (theta in [-90 deg,+90 deg]; phi in [0 deg,+360 deg])          */
/*-----------------------------------------------------------------------*/
void polar(double x,double y,double z, double& r,double& theta,double& phi)
{
 double rho = x*x+y*y;
 r = sqrt(rho+z*z);  
 phi = atn2(y,x); if(phi<0.) phi+=360.0;
 rho = sqrt(rho); theta = atn2(z,rho);
}
/*-----------------------------------------------------------------------*/
/* DDD: conversion of degrees, minutes and seconds into                  */
/*      degrees and fractions of a degree                                */
/*-----------------------------------------------------------------------*/
void ddd(long d,long m,double s,double& dd)
{
 double sign;
 if(d<0 || m<0 || s<0) sign = -1.0; else sign = 1.0;
 dd = sign*(abs(d)+abs(m)/60.0+abs(s)/3600.0);
}
/*-----------------------------------------------------------------------*/
/* DMS: conversion of degrees and fractions of a degree                  */
/*      into degrees, minutes and seconds                                */
/*-----------------------------------------------------------------------*/
void dms(double dd,long& d,long& m,double& s)
{
 double d1;
 d1=abs(dd);  d=long(d1);  
 d1=(d1-d)*60.0; m=long(d1); s=(d1-m)*60.0;
 if(dd<0)
   {if(d!=0) d = -d;
    else if(m!=0) m = -m;
    else s = -s;
   }
}
/*-----------------------------------------------------------------------*/
/* MJD: Modified Julian Date                                             */
/*      The routine is valid for any date since 4713 BC.                 */
/*      Julian calendar is used up to 1582 October 4,                    */
/*      Gregorian calendar is used from 1582 October 15 onwards.         */
/*-----------------------------------------------------------------------*/
double mjd(long day,long month,long year,double hour)
{
 double a; long b;
 if(month<=2) {month += 12; year--;}
 if(year<0) ++year;//Korrektur wegen fehlendem Jahr 0
 if(year<1582 || (year==1582 && (100*month+day)<=1004))
	b = (year+4716)/4 - 1181;
 else	b = year/400 - year/100 + year/4;
 a = 365.0*year-679004.0;
 return a+b+long(30.6001*(month+1))+day+hour/24.0;
}
/*----------------------------------------------------------------------*/
/* CALDAT: Finds the civil calendar date for a given value              */
/*         of the Modified Julian Date (MJD).                           */
/*         Julian calendar is used up to 1582 October 4,                */
/*         Gregorian calendar is used from 1582 October 15 onwards.     */
/*----------------------------------------------------------------------*/
void caldat(double mjd, long& day,long& month,long& year,double& hour)
{
 long b,d,f;
 double jd,jd0,c,e;
 jd = mjd + 2400000.5;
 jd0 = long(jd+0.5);
 if(jd0<2299161.0)              /* calendar:    */
   c=jd0+1524.0;                /* -> Julian    */
   else {                       /* -> Gregorian */
         b = long((jd0-1867216.25)/36524.25);
	 c = jd0+(b-long(b/4))+1525.0;
        }
 d    = long((c-122.1)/365.25);		e     = 365.0*d+long(d/4);
 f    = long((c-e)/30.6001);
 day  = long(c-e+0.5)-long(30.6001*f);	month = f-1-12*long(f/14);
 year = d-4715-(7+month)/10;		hour  = 24.0*(jd+0.5-jd0);
 if(year<=0) --year;//Korrektur wegen fehlendem Jahr 0
}
void caldat(double mjd, int& day,int& month,int& year,double& hour)
{
 long d,m,y;
 caldat(mjd,d,m,y,hour);
 day=d; month=m; year=y;
}

/*-----------------------------------------------------------------------*/
/* ECLEQU: Conversion of ecliptic into equatorial coordinates            */
/*         (T: equinox in Julian centuries since J2000)                  */
/*-----------------------------------------------------------------------*/
void eclequ(double t,double& x,double& y,double& z)
{
 double eps,c,s,v;
 eps = 23.43929111-(46.8150+(0.00059-0.001813*t)*t)*t/3600.0;
 c = cs(eps);  s = sn(eps);
 v = c*y-s*z;  z = s*y+c*z;  y=v;
}
/*-----------------------------------------------------------------------*/
/* ECLEQU: Conversion of ecliptic into equatorial coordinates            */
/*         (T: equinox in Julian centuries since J2000)                  */
/*-----------------------------------------------------------------------*/
void equecl(double t,double& x,double& y,double& z)
{
 double  eps,c,s,v  ;
 eps = 23.43929111-(46.8150+(0.00059-0.001813*t)*t)*t/3600.0;
 c = cs(eps);  s = sn(eps);
 v = c*y+s*z;  z = -s*y+c*z;  y=v;
}
/*-----------------------------------------------------------------------*/
/* PMATECL: calculates the precession matrix A[i,j] for                  */
/*          transforming ecliptic coordinates from equinox T1 to T2      */
/*          ( T=(JD-2451545.0)/36525 )                                   */
/*-----------------------------------------------------------------------*/
void pmatecl(double t1,double t2,REAL33& a)
{
 const double sec=3600.0;
 double dt,ppi,pi,pa;
 double c1,s1,c2,s2,c3,s3;
 dt = t2-t1;
 ppi = 174.876383889 +( ((3289.4789+0.60622*t1)*t1) +
		       ((-869.8089-0.50491*t1) + 0.03536*dt)*dt )/sec;
 pi  = ( (47.0029-(0.06603-0.000598*t1)*t1)+
	((-0.03302+0.000598*t1)+0.000060*dt)*dt )*dt/sec;
 pa  = ( (5029.0966+(2.22226-0.000042*t1)*t1)+
	((1.11113-0.000042*t1)-0.000006*dt)*dt )*dt/sec;
 c1 = cs(ppi+pa);  c2 = cs(pi);  c3 = cs(ppi);
 s1 = sn(ppi+pa);  s2 = sn(pi);  s3 = sn(ppi);
 a[1][1] = c1*c3+s1*c2*s3; a[1][2] = c1*s3-s1*c2*c3; a[1][3] = -s1*s2;
 a[2][1] = s1*c3-c1*c2*s3; a[2][2] = s1*s3+c1*c2*c3; a[2][3] = c1*s2;
 a[3][1] = s2*s3;          a[3][2] = -s2*c3;         a[3][3] = c2;
}
/*-----------------------------------------------------------------------*/
/* PMATEQU: calculates the precession matrix A[i,j] for                  */
/*          transforming equatorial coordinates from equinox T1 to T2    */
/*          (T=(JD-2451545.0)/36525 )                                    */
/*-----------------------------------------------------------------------*/
void pmatequ(double t1,double t2,REAL33& a)
{
 const double sec=3600.0;
 double dt,zeta,z,theta;
 double c1,s1,c2,s2,c3,s3;
 dt = t2-t1;
 zeta  = ( (2306.2181+(1.39656-0.000139*t1)*t1)+
	  ((0.30188-0.000345*t1)+0.017998*dt)*dt )*dt/sec;
 z     = zeta + ( (0.79280+0.000411*t1)+0.000205*dt)*dt*dt/sec;
 theta = ( (2004.3109-(0.85330+0.000217*t1)*t1)-
	  ((0.42665+0.000217*t1)+0.041833*dt)*dt )*dt/sec;
 c1 = cs(z);  c2 = cs(theta);  c3 = cs(zeta);
 s1 = sn(z);  s2 = sn(theta);  s3 = sn(zeta);
 a[1][1] = -s1*s3+c1*c2*c3; a[1][2] = -s1*c3-c1*c2*s3; a[1][3] = -c1*s2;
 a[2][1] = c1*s3+s1*c2*c3;  a[2][2] = c1*c3-s1*c2*s3;  a[2][3] = -s1*s2;
 a[3][1] = s2*c3;           a[3][2] = -s2*s3;          a[3][3] = c2;
}
/*-----------------------------------------------------------------------*/
/* PRECART: calculate change of coordinates due to precession            */
/*          for given transformation matrix A[i,j]                       */
/*          (to be used with PMATECL und PMATEQU)                        */
/*-----------------------------------------------------------------------*/
void precart(REAL33& a,double& x,double& y,double& z)
{
 double u,v,w;
 u = a[1][1]*x+a[1][2]*y+a[1][3]*z;
 v = a[2][1]*x+a[2][2]*y+a[2][3]*z;
 w = a[3][1]*x+a[3][2]*y+a[3][3]*z;
 x = u; y = v; z = w;
}
/*-----------------------------------------------------------------------*/
/* LMST: local mean sidereal time                                        */
/*-----------------------------------------------------------------------*/
double lmst(double mjd,double lambda)
{
 double mjd0,t,ut,gmst;
 mjd0 = long(mjd);
 ut = (mjd-mjd0)*24; t = (mjd0-51544.5)/36525.0;
 gmst = 6.697374558 + 1.0027379093*ut
           +(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;
 return 24.0*frac( (gmst-lambda/15.0) / 24.0 );
}
/*-----------------------------------------------------------------------*/
/* QUAD: finds a parabola through 3 points                               */
/*       (-1,Y_MINUS), (0,Y_0) und (1,Y_PLUS),                           */
/*       that do not lie on a straight line.                             */
/*                                                                       */
/*      Y_MINUS,Y_0,Y_PLUS: three y-values                               */
/*      XE,YE   : x and y of the extreme value of the parabola           */
/*      ZERO1   : first root within [-1,+1] (for NZ=1,2)                 */
/*      ZERO2   : second root within [-1,+1] (only for NZ=2)             */
/*      NZ      : number of roots within the interval [-1,+1]            */
/*-----------------------------------------------------------------------*/
void quad(double y_minus,double y_0,double y_plus,
	  double& xe,double& ye,double& zero1,double& zero2,int& nz)
{
 double a,b,c,dis,dx;
 nz=0;
 a=0.5*(y_minus+y_plus)-y_0; b=0.5*(y_plus-y_minus); c=y_0;
 xe = -b/(2.0*a); ye = (a*xe + b) * xe + c;
 dis= b*b - 4.0*a*c; /* discriminant of y = axx+bx+c */
 if(dis>=0)          /* parabola intersects x-axis   */
  {dx = 0.5*sqrt(dis)/abs(a); zero1 = xe-dx; zero2 = xe+dx;
   if(abs(zero1)<=1.0) nz++;
   if(abs(zero2)<=1.0) nz++;
   if(zero1 < -1.0) zero1=zero2;
  }
}
/*-----------------------------------------------------------------------*/
/* SUN200: ecliptic coordinates L,B,R (in deg and AU) of the             */
/*         Sun referred to the mean equinox of date                      */
/*         (T: time in Julian centuries since J2000)                     */
/*         (   = (JED-2451545.0)/36525             )                     */
/*-----------------------------------------------------------------------*/
//const double p2=ZWEIPI;
static	ARRAYREAL c3(-1,7),s3(-1,7),
		  c(-8,0),s(-8,0);
static	double	m2,m3,m4,m5,m6,
		d,a,uu,
		u,v,dl,dr,db,
		sun200_t;

void addthe(double c1,double s1,double c2,double s2, double& c,double& s)
	{c=c1*c2-s1*s2; s=s1*c2+c1*s2;}

static void term(long i1,long i,long it,double dlc,double dls,
		double drc,double drs,double dbc,double dbs)
{
 if(it==0) addthe(c3[i1],s3[i1],c[i],s[i],u,v);
 else {u*=sun200_t; v*=sun200_t;}
 dl += dlc*u+dls*v; dr += drc*u+drs*v; db += dbc*u+dbs*v;
}

void pertven()  /* Keplerian terms and perturbations by Venus */
{
 long i;
 c[0] = 1.0; s[0] = 0.0; c[-1] = cos(m2); s[-1] = -sin(m2);
 for(i = -1;i >= -5; i--) addthe(c[i],s[i],c[-1],s[-1],c[i-1],s[i-1]);
 term(1, 0,0,-0.22,6892.76,-16707.37, -0.54, 0.00, 0.00);
 term(1, 0,1,-0.06, -17.35,    42.04, -0.15, 0.00, 0.00);
 term(1, 0,2,-0.01,  -0.05,     0.13, -0.02, 0.00, 0.00);
 term(2, 0,0, 0.00,  71.98,  -139.57,  0.00, 0.00, 0.00);
 term(2, 0,1, 0.00,  -0.36,     0.70,  0.00, 0.00, 0.00);
 term(3, 0,0, 0.00,   1.04,    -1.75,  0.00, 0.00, 0.00);
 term(0,-1,0, 0.03,  -0.07,    -0.16, -0.07, 0.02,-0.02);
 term(1,-1,0, 2.35,  -4.23,    -4.75, -2.64, 0.00, 0.00);
 term(1,-2,0,-0.10,   0.06,     0.12,  0.20, 0.02, 0.00);
 term(2,-1,0,-0.06,  -0.03,     0.20, -0.01, 0.01,-0.09);
 term(2,-2,0,-4.70,   2.90,     8.28, 13.42, 0.01,-0.01);
 term(3,-2,0, 1.80,  -1.74,    -1.44, -1.57, 0.04,-0.06);
 term(3,-3,0,-0.67,   0.03,     0.11,  2.43, 0.01, 0.00);
 term(4,-2,0, 0.03,  -0.03,     0.10,  0.09, 0.01,-0.01);
 term(4,-3,0, 1.51,  -0.40,    -0.88, -3.36, 0.18,-0.10);
 term(4,-4,0,-0.19,  -0.09,    -0.38,  0.77, 0.00, 0.00);
 term(5,-3,0, 0.76,  -0.68,     0.30,  0.37, 0.01, 0.00);
 term(5,-4,0,-0.14,  -0.04,    -0.11,  0.43,-0.03, 0.00);
 term(5,-5,0,-0.05,  -0.07,    -0.31,  0.21, 0.00, 0.00);
 term(6,-4,0, 0.15,  -0.04,    -0.06, -0.21, 0.01, 0.00);
 term(6,-5,0,-0.03,  -0.03,    -0.09,  0.09,-0.01, 0.00);
 term(6,-6,0, 0.00,  -0.04,    -0.18,  0.02, 0.00, 0.00);
 term(7,-5,0,-0.12,  -0.03,    -0.08,  0.31,-0.02,-0.01);
}

void pertmar()  /* perturbations by Mars */
{
 long i;
 c[-1] = cos(m4); s[-1] = -sin(m4);
 for(i = -1;i >= -7;i--) addthe(c[i],s[i],c[-1],s[-1],c[i-1],s[i-1]);
 term(1,-1,0,-0.22,   0.17,    -0.21, -0.27, 0.00, 0.00);
 term(1,-2,0,-1.66,   0.62,     0.16,  0.28, 0.00, 0.00);
 term(2,-2,0, 1.96,   0.57,    -1.32,  4.55, 0.00, 0.01);
 term(2,-3,0, 0.40,   0.15,    -0.17,  0.46, 0.00, 0.00);
 term(2,-4,0, 0.53,   0.26,     0.09, -0.22, 0.00, 0.00);
 term(3,-3,0, 0.05,   0.12,    -0.35,  0.15, 0.00, 0.00);
 term(3,-4,0,-0.13,  -0.48,     1.06, -0.29, 0.01, 0.00);
 term(3,-5,0,-0.04,  -0.20,     0.20, -0.04, 0.00, 0.00);
 term(4,-4,0, 0.00,  -0.03,     0.10,  0.04, 0.00, 0.00);
 term(4,-5,0, 0.05,  -0.07,     0.20,  0.14, 0.00, 0.00);
 term(4,-6,0,-0.10,   0.11,    -0.23, -0.22, 0.00, 0.00);
 term(5,-7,0,-0.05,   0.00,     0.01, -0.14, 0.00, 0.00);
 term(5,-8,0, 0.05,   0.01,    -0.02,  0.10, 0.00, 0.00);
}

void pertjup()  /* perturbations by Jupiter */
{
 long i;
 c[-1] = cos(m5); s[-1] = -sin(m5);
 for(i = -1;i >= -3;i--) addthe(c[i],s[i],c[-1],s[-1],c[i-1],s[i-1]);
 term(-1,-1,0,0.01,   0.07,     0.18, -0.02, 0.00,-0.02);
 term(0,-1,0,-0.31,   2.58,     0.52,  0.34, 0.02, 0.00);
 term(1,-1,0,-7.21,  -0.06,     0.13,-16.27, 0.00,-0.02);
 term(1,-2,0,-0.54,  -1.52,     3.09, -1.12, 0.01,-0.17);
 term(1,-3,0,-0.03,  -0.21,     0.38, -0.06, 0.00,-0.02);
 term(2,-1,0,-0.16,   0.05,    -0.18, -0.31, 0.01, 0.00);
 term(2,-2,0, 0.14,  -2.73,     9.23,  0.48, 0.00, 0.00);
 term(2,-3,0, 0.07,  -0.55,     1.83,  0.25, 0.01, 0.00);
 term(2,-4,0, 0.02,  -0.08,     0.25,  0.06, 0.00, 0.00);
 term(3,-2,0, 0.01,  -0.07,     0.16,  0.04, 0.00, 0.00);
 term(3,-3,0,-0.16,  -0.03,     0.08, -0.64, 0.00, 0.00);
 term(3,-4,0,-0.04,  -0.01,     0.03, -0.17, 0.00, 0.00);
}

void pertsat()  /* perturbations by Saturn */
{
 c[-1] = cos(m6); s[-1] = -sin(m6);
 addthe(c[-1],s[-1],c[-1],s[-1],c[-2],s[-2]);
 term(0,-1,0, 0.00,   0.32,     0.01,  0.00, 0.00, 0.00);
 term(1,-1,0,-0.08,  -0.41,     0.97, -0.18, 0.00,-0.01);
 term(1,-2,0, 0.04,   0.10,    -0.23,  0.10, 0.00, 0.00);
 term(2,-2,0, 0.04,   0.10,    -0.35,  0.13, 0.00, 0.00);
}

void pertmoo()  /* difference between the Earth-Moon      */
{               /* barycenter and the center of the Earth */
 dl += 6.45*sin(d) - 0.42*sin(d-a) + 0.18*sin(d+a)
	+ 0.17*sin(d-m3) - 0.06*sin(d+m3);
 dr += 30.76*cos(d) - 3.06*cos(d-a)+ 0.85*cos(d+a)
	- 0.58*cos(d+m3) + 0.57*cos(d-m3);
 db += 0.576*sin(uu);
}

//fuer Mond-Phasen-Berechnung:
#include "astro2.h"
const double AE=1.4959787e11; //Astronomische Einheit in Meter
double sunPosx=0, sunPosy=0, sunPosz=0, sunRa=0, sunAbst=0;

void sun200(double t, double& l,double& b,double& r)
{
 int i;
 sun200_t=t;
 dl=dr=db=0.0;
 m2 = p2*frac(0.1387306+162.5485917*t);
 m3 = p2*frac(0.9931266+99.9973604*t);
 m4 = p2*frac(0.0543250+ 53.1666028*t);
 m5 = p2*frac(0.0551750+ 8.4293972*t);
 m6 = p2*frac(0.8816500+  3.3938722*t);  d = p2*frac(0.8274+1236.8531*t);
 a  = p2*frac(0.3749+1325.5524*t);      uu = p2*frac(0.2591+1342.2278*t);
 c3[0] = 1.0;     s3[0] = 0.0;
 c3[1] = cos(m3); s3[1] = sin(m3);  c3[-1] = c3[1]; s3[-1] = -s3[1];
 for(i=2;i<=7;i++) addthe(c3[i-1],s3[i-1],c3[1],s3[1],c3[i],s3[i]);
 pertven(); pertmar(); pertjup(); pertsat(); pertmoo();
 dl += 6.40*sin(p2*(0.6983+0.0561*t))+1.87*sin(p2*(0.5764+0.4174*t))
	+ 0.27*sin(p2*(0.4189+0.3306*t))+0.20*sin(p2*(0.3581+2.4814*t));
 l = 360.0*frac(0.7859453 + m3/p2 + ((6191.2+1.1*t)*t+dl)/1296.0e3 );
 r = 1.0001398 - 0.0000007*t  +  dr*1e-6;
 b = db/3600.0;
 //Vorbereitung fuer Mond-Phasen-Berechnung:
 double xp,yp,zp, xs,ys,zs, xx,yy,zz, abstand;
 geocen(t,l,b,r, l,b,r, 3,2, xp,yp,zp, xs,ys,zs, xx,yy,zz,abstand);
 sunPosx=xs*AE; sunPosy=ys*AE; sunPosz=zs*AE;
 sunAbst=r*AE;
}

/*-----------------------------------------------------------------------*/
/* ECCANOM: calculation of the eccentric anomaly E=ECCANOM(MAN,ECC)      */
/*          from the mean anomaly MAN and the eccentricity ECC.          */
/*          (solution of Kepler's equation by Newton's method)           */
/*          (E, MAN in degrees)                                          */
/*-----------------------------------------------------------------------*/
double eccanom(double man,double ecc)
{
 const double eps=1e-11;
 const int maxit=15;
 double m,e,f;
 int i;
 m=man/360.0;  m=ZWEIPI*(m-long(m)); if(m<0) m += ZWEIPI;
 if(ecc<0.8) e=m; else e=PI;
 f=e - ecc*sin(e) - m; i=0;
 while(abs(f)>eps && i<maxit)
	{e -= f/(1.0-ecc*cos(e)); f=e-ecc*sin(e)-m; i++;}
 if(i==maxit) writeln(" convergence problems in ECCANOM");
 return e/RAD;
}
/*-----------------------------------------------------------------------*/
/* ELLIP: calculation of position and velocity vector                    */
/*        for elliptic orbits                                            */
/*                                                                       */
/*        M    mean anomaly in degrees       X,Y    position   (in AU)   */
/*        A    semimajor axis                VX,VY  velocity (in AU/day) */
/*        ECC  eccentricity                                              */
/*-----------------------------------------------------------------------*/
void ellip(double m,double a,double ecc,
	   double& x,double& y,double& vx,double& vy)
{
 double k,e,c,s,fac,rho;
 k = kgauss / sqrt(a);
 e = eccanom(m,ecc);  c=cs(e); s=sn(e);
 fac=sqrt((1.0-ecc)*(1+ecc));  rho=1.0-ecc*c;
 x = a*(c-ecc);  y = a*fac*s;  vx = -k*s/rho;  vy = k*fac*c/rho;
}
/*-----------------------------------------------------------------------*/
/* HYPANOM: calculation of the eccentric anomaly H=HYPANOM(MH,ECC) from  */
/*          mean anomaly MH and eccentricity ECC for                     */
/*          hyperbolic orbits                                            */
/*-----------------------------------------------------------------------*/
double hypanom(double mh,double ecc)
{
 const double eps=1e-10, maxit=15;
 double h,f,exh,sinhh,coshh;
 int i;
 h=ln(2.0*abs(mh)/ecc+1.8); if(mh<0.0) h = -h;
 exh=exp(h); sinhh=0.5*(exh-1.0/exh); coshh=0.5*(exh+1.0/exh);
 f=ecc*sinhh-h-mh; i=0;
 while(abs(f)>eps*(1.0+abs(h+mh)) && i<maxit)
	{h -= f/(ecc*coshh-1.0);
	 exh=exp(h); sinhh=0.5*(exh-1.0/exh); coshh=0.5*(exh+1.0/exh);
	 f = ecc*sinhh-h-mh; i++;
	}
 if(i==maxit) writeln(" convergence problems in HYPANOM");
 return h;
}
/*-----------------------------------------------------------------------*/
/* HYPERB: calculation of the position and velocity vector               */
/*         for hyperbolic orbits                                         */
/*                                                                       */
/*   T0   time of perihelion passage             X,Y   position          */
/*   T    time                                   VX,VY velocity          */
/*   A    semimajor axis (arbitrary sign)                                */
/*   ECC  eccentricity                                                   */
/*   (T0,T in julian centuries since J2000)                              */
/*-----------------------------------------------------------------------*/
void hyperb(double t0,double t,double a,double ecc,
	    double& x,double& y,double& vx,double& vy)
{
 double k,mh,h,exh,coshh,sinhh,rho,fac;
 a   =  abs(a);
 k   =  kgauss / sqrt(a);
 mh  =  k*36525.0*(t-t0)/a;
 h   =  hypanom(mh,ecc);
 exh =  exp(h);  coshh = 0.5*(exh+1.0/exh);  sinhh = 0.5*(exh-1.0/exh);
 fac =  sqrt((ecc+1.0)*(ecc-1.0));   rho = ecc*coshh-1.0;
 x   =  a*(ecc-coshh);  y  = a*fac*sinhh;
 vx  = -k*sinhh/rho;    vy = k*fac*coshh/rho;
}
/*-----------------------------------------------------------------------*/
/*  STUMPFF: calculation of Stumpff's functions C1 = sin(E)/E ,          */
/*           C2 = (1-cos(E))/(E**2) and C3 = (E-sin(E))/(E**3)           */
/*           for argument E2=E**2                                        */
/*           (E: eccentric anomaly in radian)                            */
/*-----------------------------------------------------------------------*/
void stumpff(double e2,double& c1,double& c2,double& c3)
{
 const double eps=1e-12;
 double add=1.0;
 int n=1;
 c1=c2=c3=0.0;
 do {c1 += add; add /= 2*n;
     c2 += add; add /= 2*n+1;
     c3 += add; add *= -e2; n++;
    }
 while(abs(add)>=eps);
}
/*-----------------------------------------------------------------------*/
/* PARAB: calculation of position and velocity for                       */
/*        parabolic and near parabolic orbits according to Stumpff       */
/*                                                                       */
/*        T0   time of perihelion passage         X,Y    position        */
/*        T    time                              VX,VY  velocity         */
/*        Q    perihelion distance                                       */
/*        ECC  eccentricity                                              */
/*        (T0,T in julian centuries since J2000)                         */
/*-----------------------------------------------------------------------*/
void parab(double t0,double t,double q,double ecc,
	   double& x,double& y,double& vx,double& vy)
{
 const double eps=1e-9;
 double e2,e20,fac,c1,c2,c3,k,tau,a,u,u2,r;
 int i=0,maxit=15;
 e2=0.0; fac=0.5*ecc;
 k   = kgauss / sqrt(q*(1.0+ecc));
 tau = kgauss * 36525.0*(t-t0);
 do	{i++;
	 e20=e2;
	 a=1.5*sqrt(fac/(q*q*q))*tau; a=cubr(sqrt(a*a+1.0)+a);
	 u=a-1.0/a; u2=u*u; e2=u2*(1.0-ecc)/fac;
	 stumpff(e2,c1,c2,c3); fac=3.0*ecc*c3;
	}
 while(abs(e2-e20)>=eps && i<maxit);
 if(i==maxit) writeln(" convergence problems in PARAB");
 r = q*(1.0+u2*c2*ecc/fac);
 x = q*(1.0-u2*c2/fac);          vy =  k*(x/r+ecc);
 y = q*sqrt((1.0+ecc)/fac)*u*c1; vx = -k*y/r;
}
/*-----------------------------------------------------------------------*/
/* GAUSVEC: calculation of the Gaussian vectors (P,Q,R) from             */
/*          ecliptic orbital elements:                                   */
/*          LAN = longitude of the ascending node                        */
/*          INC = inclination                                            */
/*          AOP = argument of perihelion                                 */
/*-----------------------------------------------------------------------*/
void gausvec(double lan,double inc,double aop,REAL33& pqr)
{
 double c1,s1,c2,s2,c3,s3;
 c1=cs(aop);  c2=cs(inc);  c3=cs(lan);
 s1=sn(aop);  s2=sn(inc);  s3=sn(lan);
 pqr[1][1]= c1*c3-s1*c2*s3; pqr[1][2]= -s1*c3-c1*c2*s3; pqr[1][3] = s2*s3;
 pqr[2][1]= c1*s3+s1*c2*c3; pqr[2][2]= -s1*s3+c1*c2*c3; pqr[2][3] = -s2*c3;
 pqr[3][1]= s1*s2;          pqr[3][2]=  c1*s2;          pqr[3][3] =  c2;
}
/*-----------------------------------------------------------------------*/
/* ORBECL: transformation of coordinates XX,YY referred to the           */
/*         orbital plane into ecliptic coordinates X,Y,Z using           */
/*         Gaussian vectors PQR                                          */
/*-----------------------------------------------------------------------*/
void orbecl(double xx,double yy,REAL33& pqr,double& x,double& y,double& z)
{
 x = pqr[1][1]*xx+pqr[1][2]*yy;
 y = pqr[2][1]*xx+pqr[2][2]*yy;
 z = pqr[3][1]*xx+pqr[3][2]*yy;
}
/*-----------------------------------------------------------------------*/
/* KEPLER: calculation of position and velocity for unperturbed          */
/*         elliptic, parabolic and hyperbolic orbits                     */
/*                                                                       */
/*        T0   time of perihelion passage         X,Y,Z     position     */
/*        T    time                               VX,VY,VZ  velocity     */
/*        Q    perihelion distance                                       */
/*        ECC  eccentricity                                              */
/*        PQR  matrix of Gaussian vectors                                */
/*        (T0,T in Julian centuries since J2000)                         */
/*-----------------------------------------------------------------------*/
void kepler(double t0,double t,double q,double ecc,REAL33& pqr,
	    double& x,double& y,double& z,double& vx,double& vy,double& vz)
{
 const double m0=5.0, eps=0.1, deg=180/PI;
 double m,delta,tau,invax,xx,yy,vvx,vvy;
 delta= abs(1.0-ecc);
 invax= delta/q;
 tau  = kgauss*36525.0*(t-t0);
 m    = deg*tau*sqrt(invax*invax*invax);
 if(m<m0 && delta<eps) parab(t0,t,q,ecc,xx,yy,vvx,vvy);
 else if(ecc<1.0) ellip(m,1.0/invax,ecc,xx,yy,vvx,vvy);
 else hyperb(t0,t,1.0/invax,ecc,xx,yy,vvx,vvy);
 orbecl(xx,yy,pqr,x,y,z); orbecl(vvx,vvy,pqr,vx,vy,vz);
}

/*-----------------------------------------------------------------------*/
/*                                                                       */
/* XYZKEP: conversion of the state vector into Keplerian elements        */
/*         for elliptical orbits                                         */
/*                                                                       */
/*  X,Y,Z    : Position [AU]                                             */
/*  VX,VY,VZ : Velocity [AU/d]                                           */
/*  AX       : Semi-major axis [AU]                                      */
/*  ECC      : Eccentricity                                              */
/*  INC      : Inclination [deg]                                         */
/*  LAN      : Longitude of the ascending node [deg]                     */
/*  AOP      : Argument of perihelion [deg]                              */
/*  MA       : Mean anomaly [deg]                                        */
/*                                                                       */
/*-----------------------------------------------------------------------*/
void xyzkep(double x,double y,double z,double vx,double vy,double vz,
	    double& ax,double& ecc,double& inc,
	    double& lan,double& aop,double& ma)
{
 double hx,hy,hz,h, r,v2,  gm, c,s,e2, ea,u,nu;
 hx = y*vz-z*vy;                               /* Areal velocity     */
 hy = z*vx-x*vz;
 hz = x*vy-y*vx;
 h  = sqrt(hx*hx + hy*hy + hz*hz);
 lan = atn2(hx, -hy);                       /* Long. ascend. node */
 inc = atn2(sqrt(hx*hx+hy*hy), hz);         /* Inclination        */
 u   = atn2(z*h, -x*hy+y*hx);               /* Arg. of latitude   */
 gm = kgauss*kgauss;
 r  = sqrt ( x*x + y*y + z*z );             /* Distance           */
 v2 = vx*vx + vy*vy + vz*vz;                /* Velocity squared   */
 ax = 1.0 / (2.0/r-v2/gm);                  /* Semi-major axis    */
 c  = 1.0-r/ax;                             /* e*cos(E)           */
 s  = (x*vx+y*vy+z*vz)/(sqrt(ax)*kgauss);   /* e*sin(E)           */
 e2  = c*c + s*s;
 ecc = sqrt(e2);                            /* Eccentricity       */
 ea  = atn2(s, c);                          /* Eccentric anomaly  */
 ma  = ea - s*DEG;                          /* Mean anomaly       */
 nu  = atn2 ( sqrt(1.0-e2)*s, c-e2 );       /* True anomaly       */
 aop = u - nu;                              /* Arg. of perihelion */
 if(lan<0.0) lan += 360.0;
 if(aop<0.0) aop += 360.0;
 if(ma <0.0)  ma += 360.0;
}
