//astro2.cc
#include "astro2.h"
const double p2=ZWEIPI;

/*-----------------------------------------------------------------------*/
/*                                                                       */
/* GEOCEN: geocentric coordinates (geometric and light-time corrected)   */
/*                                                                       */
/*   T:        time in Julian centuries since J2000                      */
/*             T=(JD-2451545.0)/36525.0                                  */
/*   LP,BP,RP: ecliptic heliocentric coordinates of the planet           */
/*   LS,BS,RS: ecliptic geocentric   coordinates of the sun              */
/*                                                                       */
/*   IPLAN:    planet (0=Sun,1=Mercury,2=Venus,3=Earth,...,9=Pluto       */
/*   IMODE:    desired type of coordinates (see description of X,Y,Z)    */
/*             (0=geometric,1=astrometric,2=apparent)                    */
/*   XP,YP,ZP: ecliptic heliocentric coordinates of the planet           */
/*   XS,YS,ZS: ecliptic geocentric coordinates of the Sun                */
/*   X, Y, Z : ecliptic geocentric cordinates of the planet (geometric   */
/*             if IMODE=0, astrometric if IMODE=1, apparent if IMODE=2)  */
/*   DELTA0:   geocentric distance (geometric)                           */
/*                                                                       */
/*   (all angles in degrees, distances in AU)                            */
/*                                                                       */
/*-----------------------------------------------------------------------*/
static void posvel(double l,double b,double r,double dl,double db,double dr,
	double& x,double& y,double& z,double& vx,double& vy,double& vz)
{
 double cl,sl,cb,sb;
 cl=cs(l); sl=sn(l); cb=cs(b); sb=sn(b);
 x = r*cl*cb;  vx = dr*cl*cb-dl*r*sl*cb-db*r*cl*sb;
 y = r*sl*cb;  vy = dr*sl*cb+dl*r*cl*cb-db*r*sl*sb;
 z = r*sb;     vz = dr*sb              +db*r*cb;
}
void geocen(double t,double lp,double bp,double rp,
	    double ls,double bs,double rs,int iplan,int imode,
	    double& xp,double& yp,double& zp, double& xs,double& ys,
	    double& zs, double& x,double& y,double& z, double& delta0)
{
 double fac, vx,vy,vz, vxs,vys,vzs, m;
 double dl=0.0,db=0.0,dr=0.0,dls=0.0,dbs=0.0,drs=0.0;
 double sinm,sin2m,sin3m,sin4m,cosm,cos2m,cos3m,cos4m; //zur Optimierung
 if(imode>0)
   {m = p2*frac(0.9931266+ 99.9973604*t);              // Sun     //
    dls = 172.00+5.75*sin(m);  drs = 2.87*cos(m); //dbs = 0.0;
    // dl,db in 1e-4 rad/d, dr in 1e-4 AU/d //
    switch(iplan)
     {case 0: //dl=db=dr=0.0;        // Sun     //
      CASE 1: m = p2*frac(0.4855407+415.2014314*t);    // Mercury //
	      sinm=sin(m); sin2m=sin(2*m); sin3m=sin(3*m); sin4m=sin(4*m);
	      cosm=cos(m); cos2m=cos(2*m); cos3m=cos(3*m); cos4m=cos(4*m);
              dl = 714.00+292.66*cosm+71.96*cos2m+18.16*cos3m
                     +4.61*cos4m+3.81*sin2m+2.43*sin3m
                     +1.08*sin4m;
              dr = 55.94*sinm+11.36*sin2m+2.60*sin3m;
              db = 73.40*cosm+29.82*cos2m+10.22*cos3m
                     +3.28*cos4m-40.44*sinm-16.55*sin2m
                     -5.56*sin3m-1.72*sin4m;
      CASE 2: m = p2*frac(0.1400197+162.5494552*t);    // Venus   //
	      sinm=sin(m); cosm=cos(m);
              dl = 280.00+3.79*cosm;   dr = 1.37*sinm;
              db = 9.54*cosm-13.57*sinm;
      CASE 3: dl = dls; dr = drs; db = -dbs;           // Earth   //
      CASE 4: m = p2*frac(0.0538553+53.1662736*t);     // Mars    //
	      sinm=sin(m); cosm=cos(m); cos2m=cos(2*m);
              dl = 91.50+17.07*cosm+2.03*cos2m;
              dr = 12.98*sinm+1.21*cos2m;
              db =  0.83*cosm+2.80*sinm;
      CASE 5: m = p2*frac(0.0565314+8.4302963*t);      // Jupiter //
	      sinm=sin(m);
              dl = 14.50+1.41*cos(m); dr = 3.66*sinm; db = 0.33*sinm;
      CASE 6: m = p2*frac(0.8829867+3.3947688*t);      // Saturn  //
	      cosm=cos(m);
              dl = 5.84+0.65*cosm; dr = 3.09*sin(m); db = 0.24*cosm;
      CASE 7: m = p2*frac(0.3967117+1.1902849*t);      // Uranus  //
	      sinm=sin(m);
              dl = 2.05+0.19*cos(m); dr = 1.86*sinm; db = -0.03*sinm;
      CASE 8: m = p2*frac(0.7214906+0.6068526*t);      // Neptune //
	      sinm=sin(m);
              dl = 1.04+0.02*cos(m); dr = 0.27*sinm; db = 0.03*sinm;
      CASE 9: m = p2*frac(0.0385795+0.4026667*t);      // Pluto   //
	      sinm=sin(m); cosm=cos(m); sin2m=sin(2*m);
              dl = 0.69+0.34*cosm+0.12*cos(2*m)+0.05*cos(3*m);
              dr = 6.66*sinm+1.64*sin2m;
              db = -0.08*cosm-0.17*sinm-0.09*sin2m;
     }
   }
 posvel(ls,bs,rs,dls,dbs,drs,xs,ys,zs,vxs,vys,vzs);
 posvel(lp,bp,rp,dl ,db ,dr, xp,yp,zp,vx ,vy ,vz );
 if(iplan==3)  {x=y=z=delta0=0.0;}
 else {x=xp+xs; y=yp+ys; z=zp+zs; delta0=sqrt(x*x+y*y+z*z);}
 fac = 0.00578 * delta0 * 1e-4;
 switch(imode)
  {case 1: x -= fac*vx;  y -= fac*vy;  z -= fac*vz;
   CASE 2: x -= fac*(vx+vxs); y -= fac*(vy+vys); z -= fac*(vz+vzs);
  }
}

/*-----------------------------------------------------------------------*/
/* NUTEQU: transformation of mean to true coordinates                    */
/*         (including terms >0.1" according to IAU 1980)                 */
/*         T = (JD-2451545.0)/36525.0                                    */
/*-----------------------------------------------------------------------*/
void nutequ(double t,double& x,double& y,double& z)
{
 const double arc=3600*180/PI; // arcseconds per radian
 double ls,d,f,n,eps, dpsi,deps,c,s, dx,dy,dz;
 ls  = p2*frac(0.993133+  99.997306*t); // mean anomaly Sun
 d   = p2*frac(0.827362+1236.853087*t); // diff. longitude Moon-Sun
 f   = p2*frac(0.259089+1342.227826*t); // mean argument of latitude
 n   = p2*frac(0.347346-   5.372447*t); // longit. ascending node
 eps = 0.4090928-2.2696e-4*t;           // obliquity of the ecliptic
 double fdn2=2*(f-d+n), fn2=2*(f+n), n2=n+n;
 dpsi = (-17.200*sin(n)  - 1.319*sin(fdn2) - 0.227*sin(fn2)
	 + 0.206*sin(n2) + 0.143*sin(ls) ) / arc;
 deps = (  9.203*cos(n)  + 0.574*cos(fdn2) + 0.098*cos(fn2)
	 - 0.090*cos(n2) ) / arc;
 c = dpsi*cos(eps);  s = dpsi*sin(eps);
 dx = -(c*y+s*z); dy = c*x-deps*z; dz = s*x+deps*y;
 x += dx;  y += dy;  z += dz;
}

/*-----------------------------------------------------------------------*/
/* HOREQU: conversion of horizontal to equatorial coordinates            */
/*   H,AZ : azimuth and altitude (in deg)                                */
/*   PHI  : geographical latitude (in deg)                               */
/*   DEC  : declination (-90 deg .. +90 deg)                             */
/*   TAU  : hour angle (0 deg .. 360 deg)                                */
/*-----------------------------------------------------------------------*/
void horequ(double h,double az,double phi,double& dec,double& tau)
{
 double cs_phi,sn_phi, cs_h,sn_h, cs_az, x,y,z, dummy;
 cs_phi = cs(phi); sn_phi = sn(phi);
 cs_h  = cs(h);   sn_h  = sn(h); cs_az = cs(az);
 x = cs_h*sn_phi*cs_az+sn_h*cs_phi;
 y = cs_h*sn(az);
 z = sn_h*sn_phi-cs_h*cs_phi*cs_az;
 polar(x,y,z, dummy,dec,tau);
}

/*-----------------------------------------------------------------------*/
/* EQUHOR: conversion of equatorial into horizontal coordinates          */
/*   DEC  : declination (-90 deg .. +90 deg)                             */
/*   TAU  : hour angle (0 deg .. 360 deg)                                */
/*   PHI  : geographical latitude (in deg)                               */
/*   H    : altitude (in deg)                                            */
/*   AZ   : azimuth (0 deg .. 360 deg, counted S->W->N->E->S)            */
/*-----------------------------------------------------------------------*/
void equhor(double dec,double tau,double phi,double& h,double& az)
{
 double cs_phi,sn_phi, cs_dec,sn_dec, cs_tau, x,y,z, dummy;
 cs_phi = cs(phi); sn_phi = sn(phi);
 cs_dec = cs(dec); sn_dec = sn(dec); cs_tau = cs(tau);
 x = cs_dec*sn_phi*cs_tau - sn_dec*cs_phi;
 y = cs_dec*sn(tau);
 z = cs_dec*cs_phi*cs_tau + sn_dec*sn_phi;
 polar(x,y,z, dummy,h,az);
}
