;; bas.asm
;* Testprogramm fuer RS232-BAS-Wandler
;* Controller: ATmega8
;* Copyright: Freeware
;-----------------------------------------------------------------------------
#define MITINTERRUPT

.include "m8def.inc"		;ATmega8
.equ F_CPU=16000000		;16 MHz Quarz
.equ SRAM_START = 0x60		;Start vom SRAM beim Atmega8

	rjmp main
#ifdef MITINTERRUPT
.org OC1Aaddr
	rjmp timer1_match_A
#endif

;-----------------------------------------------------------------------------
; Registerbelegungen:
; r4=Sicherung von SREG im Interrupt
; r5=0  Global auf 0 gesetzt
.def fontzeile=r22
.def zeilenzaehlerL=r24
.def zeilenzaehlerH=r25

;-----------------------------------------------------------------------------
; Konstanten:
.equ zeilentotal=312
.equ obereleerzeilen=38
.equ inhaltzeilen=240
.equ untereleerzeilen=zeilentotal-obereleerzeilen-inhaltzeilen
.equ pixelprotextzeile=12
.equ textzeilen=inhaltzeilen/pixelprotextzeile ;=20
.equ zeichenprozeile=27
.equ ANZAHLTEXTBYTES=textzeilen*zeichenprozeile
	
;-----------------------------------------------------------------------------
; Variablen im SRAM:
                .DSEG
textspeicher:	.BYTE ANZAHLTEXTBYTES

		.CSEG

;-----------------------------------------------------------------------------
; Zeichentabelle im FLASH:
font6x8:				     ;aus 6x8_horizontal_LSB_1.asm
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x00
.db 0x70,0x88,0xD8,0x88,0xA8,0x88,0x70,0x00,	; 0x01
.db 0x70,0xF8,0xA8,0xF8,0x88,0xF8,0x70,0x00,	; 0x02
.db 0x00,0x50,0xF8,0xF8,0xF8,0x70,0x20,0x00,	; 0x03
.db 0x00,0x20,0x70,0xF8,0xF8,0x70,0x20,0x00,	; 0x04
.db 0x20,0x70,0x70,0x20,0xF8,0xF8,0x20,0x00,	; 0x05
.db 0x00,0x20,0x70,0xF8,0xF8,0x20,0x70,0x00,	; 0x06
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x07
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x08
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x09
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x0A
.db 0x00,0xE0,0xC0,0xB0,0x48,0x48,0x30,0x00,	; 0x0B
.db 0x70,0x88,0x88,0x70,0x20,0x70,0x20,0x00,	; 0x0C
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x0D
.db 0xC0,0xB0,0xD0,0xB0,0xD0,0xD8,0x18,0x00,	; 0x0E
.db 0x00,0xA8,0x70,0xD8,0x70,0xA8,0x00,0x00,	; 0x0F
.db 0x10,0x30,0x70,0xF0,0x70,0x30,0x10,0x00,	; 0x10
.db 0x40,0x60,0x70,0x78,0x70,0x60,0x40,0x00,	; 0x11
.db 0x20,0x70,0xF8,0x20,0xF8,0x70,0x20,0x00,	; 0x12
.db 0x50,0x50,0x50,0x50,0x50,0x00,0x50,0x00,	; 0x13
.db 0xF0,0xA8,0xA8,0xB0,0xA0,0xA0,0xA0,0x00,	; 0x14
.db 0x70,0x88,0x30,0x50,0x60,0x88,0x70,0x00,	; 0x15
.db 0x00,0x00,0x00,0x00,0x00,0x78,0x78,0x00,	; 0x16
.db 0x20,0x70,0xF8,0x20,0xF8,0x70,0x20,0x70,	; 0x17
.db 0x20,0x70,0xF8,0x20,0x20,0x20,0x20,0x00,	; 0x18
.db 0x20,0x20,0x20,0x20,0xF8,0x70,0x20,0x00,	; 0x19
.db 0x00,0x20,0x60,0xF8,0x60,0x20,0x00,0x00,	; 0x1A
.db 0x00,0x20,0x30,0xF8,0x30,0x20,0x00,0x00,	; 0x1B
.db 0x00,0x00,0x00,0x08,0x08,0x08,0xF8,0x00,	; 0x1C
.db 0x00,0x50,0x50,0xF8,0x50,0x50,0x00,0x00,	; 0x1D
.db 0x20,0x20,0x70,0x70,0xF8,0xF8,0x00,0x00,	; 0x1E
.db 0xF8,0xF8,0x70,0x70,0x20,0x20,0x00,0x00,	; 0x1F
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x20
.db 0x20,0x70,0x70,0x20,0x20,0x00,0x20,0x00,	; 0x21
.db 0xD8,0xD8,0x48,0x00,0x00,0x00,0x00,0x00,	; 0x22
.db 0x00,0x50,0xF8,0x50,0x50,0xF8,0x50,0x00,	; 0x23
.db 0x10,0x70,0x08,0x30,0x40,0x38,0x20,0x00,	; 0x24
.db 0x98,0x98,0x40,0x20,0x10,0xC8,0xC8,0x00,	; 0x25
.db 0x10,0x28,0x28,0x10,0xA8,0x48,0xB0,0x00,	; 0x26
.db 0x30,0x30,0x10,0x00,0x00,0x00,0x00,0x00,	; 0x27
.db 0x20,0x10,0x10,0x10,0x10,0x10,0x20,0x00,	; 0x28
.db 0x10,0x20,0x20,0x20,0x20,0x20,0x10,0x00,	; 0x29
.db 0x00,0x50,0x70,0xF8,0x70,0x50,0x00,0x00,	; 0x2A
.db 0x00,0x20,0x20,0xF8,0x20,0x20,0x00,0x00,	; 0x2B
.db 0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x10,	; 0x2C
.db 0x00,0x00,0x00,0xF8,0x00,0x00,0x00,0x00,	; 0x2D
.db 0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,	; 0x2E
.db 0x00,0x80,0x40,0x20,0x10,0x08,0x00,0x00,	; 0x2F
.db 0x70,0x88,0xC8,0xA8,0x98,0x88,0x70,0x00,	; 0x30
.db 0x20,0x30,0x20,0x20,0x20,0x20,0x70,0x00,	; 0x31
.db 0x70,0x88,0x80,0x60,0x10,0x08,0xF8,0x00,	; 0x32
.db 0x70,0x88,0x80,0x70,0x80,0x88,0x70,0x00,	; 0x33
.db 0x40,0x60,0x50,0x48,0xF8,0x40,0x40,0x00,	; 0x34
.db 0xF8,0x08,0x08,0x78,0x80,0x88,0x70,0x00,	; 0x35
.db 0x60,0x10,0x08,0x78,0x88,0x88,0x70,0x00,	; 0x36
.db 0xF8,0x80,0x40,0x20,0x10,0x10,0x10,0x00,	; 0x37
.db 0x70,0x88,0x88,0x70,0x88,0x88,0x70,0x00,	; 0x38
.db 0x70,0x88,0x88,0xF0,0x80,0x40,0x30,0x00,	; 0x39
.db 0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x00,	; 0x3A
.db 0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x10,	; 0x3B
.db 0x40,0x20,0x10,0x08,0x10,0x20,0x40,0x00,	; 0x3C
.db 0x00,0x00,0xF8,0x00,0x00,0xF8,0x00,0x00,	; 0x3D
.db 0x10,0x20,0x40,0x80,0x40,0x20,0x10,0x00,	; 0x3E
.db 0x70,0x88,0x80,0x60,0x20,0x00,0x20,0x00,	; 0x3F
.db 0x70,0x88,0xE8,0xA8,0xE8,0x08,0x70,0x00,	; 0x40
.db 0x70,0x88,0x88,0x88,0xF8,0x88,0x88,0x00,	; 0x41
.db 0x78,0x88,0x88,0x78,0x88,0x88,0x78,0x00,	; 0x42
.db 0x70,0x88,0x08,0x08,0x08,0x88,0x70,0x00,	; 0x43
.db 0x78,0x88,0x88,0x88,0x88,0x88,0x78,0x00,	; 0x44
.db 0xF8,0x08,0x08,0x78,0x08,0x08,0xF8,0x00,	; 0x45
.db 0xF8,0x08,0x08,0x78,0x08,0x08,0x08,0x00,	; 0x46
.db 0x70,0x88,0x08,0xE8,0x88,0x88,0xF0,0x00,	; 0x47
.db 0x88,0x88,0x88,0xF8,0x88,0x88,0x88,0x00,	; 0x48
.db 0x70,0x20,0x20,0x20,0x20,0x20,0x70,0x00,	; 0x49
.db 0x80,0x80,0x80,0x80,0x88,0x88,0x70,0x00,	; 0x4A
.db 0x88,0x48,0x28,0x18,0x28,0x48,0x88,0x00,	; 0x4B
.db 0x08,0x08,0x08,0x08,0x08,0x08,0xF8,0x00,	; 0x4C
.db 0x88,0xD8,0xA8,0x88,0x88,0x88,0x88,0x00,	; 0x4D
.db 0x88,0x98,0xA8,0xC8,0x88,0x88,0x88,0x00,	; 0x4E
.db 0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00,	; 0x4F
.db 0x78,0x88,0x88,0x78,0x08,0x08,0x08,0x00,	; 0x50
.db 0x70,0x88,0x88,0x88,0xA8,0x48,0xB0,0x00,	; 0x51
.db 0x78,0x88,0x88,0x78,0x48,0x88,0x88,0x00,	; 0x52
.db 0x70,0x88,0x08,0x70,0x80,0x88,0x70,0x00,	; 0x53
.db 0xF8,0x20,0x20,0x20,0x20,0x20,0x20,0x00,	; 0x54
.db 0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,	; 0x55
.db 0x88,0x88,0x88,0x88,0x88,0x50,0x20,0x00,	; 0x56
.db 0x88,0x88,0xA8,0xA8,0xA8,0xA8,0x50,0x00,	; 0x57
.db 0x88,0x88,0x50,0x20,0x50,0x88,0x88,0x00,	; 0x58
.db 0x88,0x88,0x88,0x50,0x20,0x20,0x20,0x00,	; 0x59
.db 0x78,0x40,0x20,0x10,0x08,0x08,0x78,0x00,	; 0x5A
.db 0x70,0x10,0x10,0x10,0x10,0x10,0x70,0x00,	; 0x5B
.db 0x00,0x08,0x10,0x20,0x40,0x80,0x00,0x00,	; 0x5C
.db 0x70,0x40,0x40,0x40,0x40,0x40,0x70,0x00,	; 0x5D
.db 0x20,0x50,0x88,0x00,0x00,0x00,0x00,0x00,	; 0x5E
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFC,	; 0x5F
.db 0x30,0x30,0x20,0x00,0x00,0x00,0x00,0x00,	; 0x60
.db 0x00,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x61
.db 0x08,0x08,0x78,0x88,0x88,0x88,0x78,0x00,	; 0x62
.db 0x00,0x00,0x70,0x88,0x08,0x88,0x70,0x00,	; 0x63
.db 0x80,0x80,0xF0,0x88,0x88,0x88,0xF0,0x00,	; 0x64
.db 0x00,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x65
.db 0x60,0x10,0x10,0x78,0x10,0x10,0x10,0x00,	; 0x66
.db 0x00,0x00,0xF0,0x88,0x88,0xF0,0x80,0x70,	; 0x67
.db 0x08,0x08,0x38,0x48,0x48,0x48,0x48,0x00,	; 0x68
.db 0x20,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x69
.db 0x40,0x00,0x60,0x40,0x40,0x40,0x48,0x30,	; 0x6A
.db 0x08,0x08,0x48,0x28,0x18,0x28,0x48,0x00,	; 0x6B
.db 0x20,0x20,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x6C
.db 0x00,0x00,0x58,0xA8,0xA8,0x88,0x88,0x00,	; 0x6D
.db 0x00,0x00,0x38,0x48,0x48,0x48,0x48,0x00,	; 0x6E
.db 0x00,0x00,0x70,0x88,0x88,0x88,0x70,0x00,	; 0x6F
.db 0x00,0x00,0x78,0x88,0x88,0x88,0x78,0x08,	; 0x70
.db 0x00,0x00,0xF0,0x88,0x88,0x88,0xF0,0x80,	; 0x71
.db 0x00,0x00,0x68,0x90,0x10,0x10,0x38,0x00,	; 0x72
.db 0x00,0x00,0x70,0x08,0x70,0x80,0x70,0x00,	; 0x73
.db 0x00,0x10,0x78,0x10,0x10,0x50,0x20,0x00,	; 0x74
.db 0x00,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x75
.db 0x00,0x00,0x88,0x88,0x88,0x50,0x20,0x00,	; 0x76
.db 0x00,0x00,0x88,0x88,0xA8,0xF8,0x50,0x00,	; 0x77
.db 0x00,0x00,0x48,0x48,0x30,0x48,0x48,0x00,	; 0x78
.db 0x00,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0x79
.db 0x00,0x00,0x78,0x40,0x30,0x08,0x78,0x00,	; 0x7A
.db 0x60,0x10,0x10,0x18,0x10,0x10,0x60,0x00,	; 0x7B
.db 0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,	; 0x7C
.db 0x30,0x40,0x40,0xC0,0x40,0x40,0x30,0x00,	; 0x7D
.db 0x50,0x28,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x7E
.db 0x20,0x70,0xD8,0x88,0x88,0xF8,0x00,0x00,	; 0x7F
.db 0x70,0x88,0x08,0x08,0x88,0x70,0x20,0x30,	; 0x80
.db 0x48,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x81
.db 0xC0,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x82
.db 0x70,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x83
.db 0x50,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x84
.db 0x30,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x85
.db 0x70,0x50,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x86
.db 0x00,0x70,0x88,0x08,0x88,0x70,0x20,0x30,	; 0x87
.db 0x70,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x88
.db 0x50,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x89
.db 0x30,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x8A
.db 0x50,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8B
.db 0x70,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8C
.db 0x10,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8D
.db 0x50,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0x8E
.db 0x70,0x50,0x70,0xD8,0x88,0xF8,0x88,0x00,	; 0x8F
.db 0xC0,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0x90
.db 0x00,0x00,0x78,0xA0,0xF8,0x28,0xF0,0x00,	; 0x91
.db 0xF0,0x28,0x28,0xF8,0x28,0x28,0xE8,0x00,	; 0x92
.db 0x70,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x93
.db 0x50,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x94
.db 0x18,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x95
.db 0x70,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x96
.db 0x18,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x97
.db 0x50,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0x98
.db 0x48,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0x99
.db 0x50,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0x9A
.db 0x00,0x00,0x80,0x70,0x68,0x58,0x38,0x04,	; 0x9B
.db 0x60,0x90,0x10,0x78,0x10,0x90,0xE8,0x00,	; 0x9C
.db 0xF0,0xC8,0xA8,0xA8,0xA8,0x98,0x78,0x00,	; 0x9D
.db 0x00,0x88,0x50,0x20,0x50,0x88,0x00,0x00,	; 0x9E
.db 0x40,0xA0,0x20,0x70,0x20,0x20,0x28,0x10,	; 0x9F
.db 0x60,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0xA0
.db 0x60,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0xA1
.db 0x60,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0xA2
.db 0x60,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0xA3
.db 0x50,0x28,0x00,0x38,0x48,0x48,0x48,0x00,	; 0xA4
.db 0x50,0x28,0x00,0x48,0x58,0x68,0x48,0x00,	; 0xA5
.db 0x70,0x80,0xF0,0x88,0xF0,0x00,0xF0,0x00,	; 0xA6
.db 0x30,0x48,0x48,0x48,0x30,0x00,0x78,0x00,	; 0xA7
.db 0x20,0x00,0x20,0x30,0x08,0x88,0x70,0x00,	; 0xA8
.db 0x78,0xA4,0xD4,0xB4,0xD4,0x84,0x78,0x00,	; 0xA9
.db 0x00,0x00,0xFC,0x80,0x80,0x00,0x00,0x00,	; 0xAA
.db 0x08,0x48,0x28,0x70,0x88,0x40,0xE0,0x00,	; 0xAB
.db 0x08,0x48,0x28,0xD0,0xA8,0xE0,0x80,0x00,	; 0xAC
.db 0x20,0x00,0x20,0x20,0x70,0x70,0x20,0x00,	; 0xAD
.db 0x00,0x00,0x90,0x48,0x90,0x00,0x00,0x00,	; 0xAE
.db 0x00,0x00,0x48,0x90,0x48,0x00,0x00,0x00,	; 0xAF
.db 0xA8,0x00,0x54,0x00,0xA8,0x00,0x54,0x00,	; 0xB0
.db 0xA8,0x54,0xA8,0x54,0xA8,0x54,0xA8,0x54,	; 0xB1
.db 0x54,0xFC,0xA8,0xFC,0x54,0xFC,0xA8,0xFC,	; 0xB2
.db 0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,	; 0xB3
.db 0x20,0x20,0x20,0x3C,0x20,0x20,0x20,0x20,	; 0xB4
.db 0x60,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB5
.db 0x70,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB6
.db 0x30,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB7
.db 0x78,0x84,0xB4,0x94,0xB4,0x84,0x78,0x00,	; 0xB8
.db 0x28,0x2C,0x20,0x2C,0x28,0x28,0x28,0x28,	; 0xB9
.db 0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,	; 0xBA
.db 0x00,0x3C,0x20,0x2C,0x28,0x28,0x28,0x28,	; 0xBB
.db 0x28,0x2C,0x20,0x3C,0x00,0x00,0x00,0x00,	; 0xBC
.db 0x00,0x20,0x70,0x08,0x08,0x70,0x20,0x00,	; 0xBD
.db 0x88,0x50,0x20,0xF8,0x20,0xF8,0x20,0x00,	; 0xBE
.db 0x00,0x00,0x00,0x3C,0x20,0x20,0x20,0x20,	; 0xBF
.db 0x20,0x20,0x20,0xE0,0x00,0x00,0x00,0x00,	; 0xC0
.db 0x20,0x20,0x20,0xFC,0x00,0x00,0x00,0x00,	; 0xC1
.db 0x00,0x00,0x00,0xFC,0x20,0x20,0x20,0x20,	; 0xC2
.db 0x20,0x20,0x20,0xE0,0x20,0x20,0x20,0x20,	; 0xC3
.db 0x00,0x00,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xC4
.db 0x20,0x20,0x20,0xFC,0x20,0x20,0x20,0x20,	; 0xC5
.db 0xA0,0x50,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0xC6
.db 0xA0,0x50,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xC7
.db 0x28,0xE8,0x08,0xF8,0x00,0x00,0x00,0x00,	; 0xC8
.db 0x00,0xF8,0x08,0xE8,0x28,0x28,0x28,0x28,	; 0xC9
.db 0x28,0xEC,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xCA
.db 0x00,0xFC,0x00,0xEC,0x28,0x28,0x28,0x28,	; 0xCB
.db 0x28,0xE8,0x08,0xE8,0x28,0x28,0x28,0x28,	; 0xCC
.db 0x00,0xFC,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xCD
.db 0x28,0xEC,0x00,0xEC,0x28,0x28,0x28,0x28,	; 0xCE
.db 0x88,0x70,0x88,0x88,0x88,0x70,0x88,0x00,	; 0xCF
.db 0x30,0x08,0x10,0x20,0x70,0x48,0x30,0x00,	; 0xD0
.db 0x70,0x90,0x90,0xB8,0x90,0x90,0x70,0x00,	; 0xD1
.db 0x70,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD2
.db 0x50,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD3
.db 0x30,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD4
.db 0x20,0x20,0x20,0x00,0x00,0x00,0x00,0x00,	; 0xD5
.db 0x60,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD6
.db 0x70,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD7
.db 0x50,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD8
.db 0x20,0x20,0x20,0x3C,0x00,0x00,0x00,0x00,	; 0xD9
.db 0x00,0x00,0x00,0xE0,0x20,0x20,0x20,0x20,	; 0xDA
.db 0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,	; 0xDB
.db 0x00,0x00,0x00,0x00,0xFC,0xFC,0xFC,0xFC,	; 0xDC
.db 0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,	; 0xDD
.db 0x30,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xDE
.db 0xFC,0xFC,0xFC,0xFC,0x00,0x00,0x00,0x00,	; 0xDF
.db 0x60,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE0
.db 0x00,0x38,0x48,0x38,0x48,0x48,0x38,0x08,	; 0xE1
.db 0x70,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE2
.db 0x18,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE3
.db 0x50,0x28,0x00,0x30,0x48,0x48,0x30,0x00,	; 0xE4
.db 0x50,0x28,0x30,0x48,0x48,0x48,0x30,0x00,	; 0xE5
.db 0x00,0x00,0x48,0x48,0x48,0x38,0x08,0x08,	; 0xE6
.db 0x00,0x18,0x08,0x38,0x48,0x38,0x08,0x18,	; 0xE7
.db 0x18,0x08,0x38,0x48,0x48,0x38,0x08,0x18,	; 0xE8
.db 0x60,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE9
.db 0x70,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xEA
.db 0x18,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xEB
.db 0x60,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0xEC
.db 0x60,0x00,0x88,0x50,0x20,0x20,0x20,0x00,	; 0xED
.db 0x00,0x70,0x00,0x00,0x00,0x00,0x00,0x00,	; 0xEE
.db 0x30,0x30,0x10,0x00,0x00,0x00,0x00,0x00,	; 0xEF
.db 0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,	; 0xF0
.db 0x00,0x20,0x70,0x20,0x00,0x70,0x00,0x00,	; 0xF1
.db 0x00,0x00,0xF8,0x00,0x00,0xF8,0x00,0x00,	; 0xF2
.db 0x0C,0x58,0x2C,0xD0,0xA8,0xE0,0x80,0x00,	; 0xF3
.db 0xF0,0xA8,0xA8,0xB0,0xA0,0xA0,0xA0,0x00,	; 0xF4
.db 0x70,0x88,0x30,0x50,0x60,0x88,0x70,0x00,	; 0xF5
.db 0x00,0x20,0x00,0xF8,0x00,0x20,0x00,0x00,	; 0xF6
.db 0x00,0x00,0x00,0x70,0x60,0x00,0x00,0x00,	; 0xF7
.db 0x30,0x48,0x48,0x30,0x00,0x00,0x00,0x00,	; 0xF8
.db 0x00,0x00,0x00,0x50,0x00,0x00,0x00,0x00,	; 0xF9
.db 0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,	; 0xFA
.db 0x10,0x18,0x10,0x10,0x00,0x00,0x00,0x00,	; 0xFB
.db 0x38,0x10,0x30,0x18,0x00,0x00,0x00,0x00,	; 0xFC
.db 0x18,0x20,0x10,0x38,0x00,0x00,0x00,0x00,	; 0xFD
.db 0x00,0x00,0x78,0x78,0x78,0x78,0x00,0x00,	; 0xFE
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 	; 0xFF

;; Eine gesamte Zeile sieht so aus:
;; 5us Sync-Puls, 9us Schwarzschulter, 45.6us Text, 4.4us leer --> total 64us
;; 80 Takte       144 Takte            729 Takte    71 Takte             1024 Takte
;;                             729 = 27Zeichen * 27Takte/Zeichen 

;-----------------------------------------------------------------------------
; Fuer Variante mit Interrupts
#ifdef MITINTERRUPT
;.equ NormalerCompwert = (F_CPU*64/1000000)-1 ;64 usec fuer Zeilenlaenge
.equ NormalerCompwert = (16*64)-1 ;64 usec fuer Zeilenlaenge

timer1_init:
	push r16
	push r17
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	out TCCR1B, r16
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	out OCR1AH, r17
	out OCR1AL, r16
	ldi r16, 0		;Startwert des Timers = 0
	out TCNT1H, r16
	out TCNT1L, r16
	ldi r16, 1<<OCIE1A	;Timer1 Interrupts bei Vergleichswert
	out TIMSK, r16
	ldi zeilenzaehlerL, low(312-1)
	ldi zeilenzaehlerH, high(312-1)
	pop r17
	pop r16
	ret

timer1_match_A:			;Interrupt-Routine
	cbi PORTC, 0		;Synchronisationspuls starten
	in r4, SREG
	push r16
	push r17
	push r18

	adiw zeilenzaehlerL, 1
	ldi r16, low(312)
	ldi r17, high(312)
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerL, r17
	brne t1_L1
	ldi zeilenzaehlerL, 0
	ldi zeilenzaehlerH, 0
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi fontzeile, 0
	rjmp t1_L2
t1_L1:	adiw r24, 0		;statt 2 mal nop
	adiw r24, 0
	adiw r24, 0
t1_L2:	rcall zeile_darstellen
	
t1_ret:	pop r18
	pop r17
	pop r16
	out SREG, r4
	reti
#endif

;-----------------------------------------------------------------------------
; Hauptprogramm:
main:	ldi r16, low(RAMEND)
        out SPL, r16		; Init Stackpointer L
        ldi r16, high(RAMEND)
        out SPH, r16		; Init Stackpointer H
	clr r5			; r5 global mit 0 belegen
	ldi r16, (1<<3)		; PB3
	out DDRB, r16		; Ausgang fuer Bildinhalt
	ldi r16, (1<<0)		; PC0
	out DDRC, r16		; Ausgang fuer Synchronisations-Pulse
	out PORTC, r16		; Ausgang gesetzt fuer inaktiv
#ifdef MITINTERRUPT
	rcall timer1_init
	ldi r16, (1<<SE)
	out MCUCR, r16
	sei			; Interrupts einschalten
#endif

;; tests:
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi r16, 'L'
	st Y+, r16
	ldi r16, 'A'
	st Y+, r16
	
mainloop:
#ifdef MITINTERRUPT
	sleep
#else
	rcall gesamte_seite_darstellen
#endif
	rjmp mainloop

;-----------------------------------------------------------------------------
#ifdef ALTEVERSION
zeichendarstellen:		;r16=ASCII-Code
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	ldi r17, 0
	lsl r16
	rol r17
	lsl r16
	rol r17
	lsl r16
	rol r17			;r17:r16 = 8*r16
	add ZL, r16
	adc ZH, r17		;Zeiger auf entsprechendes Zeichen
	add ZL, fontzeile
	adc ZH, r5		;Zeiger auf entsprechende Zeile
	lpm r16, Z
	rol r16
	out PORTB, r16		;1.Bit auf PB3 ausgeben
	nop			;Zeichenbreite 12 Takte ohne nop, 18 Takte mit nop
	ror r16
	out PORTB, r16		;2.Bit auf PB3 ausgeben
	nop
	lsr r16
	out PORTB, r16		;3.Bit auf PB3 ausgeben
	nop
	lsr r16
	out PORTB, r16		;4.Bit auf PB3 ausgeben
	nop
	lsr r16
	out PORTB, r16		;5.Bit auf PB3 ausgeben
	nop
	lsr r16
	out PORTB, r16		;6.Bit auf PB3 ausgeben
	ret

textdarstellen_variante1:	;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	ld r16, Y+
	rcall zeichendarstellen	;braucht ausser den 18 Takte fuers Zeichen nochmals 25 Takte
	dec r18
	brne textdarstellen_variante1
	ret
#endif

#ifdef ALTEVERSION
textdarstellen_variante2:	;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	ld r16, Y+
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	ldi r17, 0
	lsl r16
t_L1:	rol r17
	lsl r16
	rol r17
	lsl r16
	rol r17			;r17:r16 = 8*r16
	add ZL, r16
	adc ZH, r17		;Zeiger auf entsprechendes Zeichen
	add ZL, fontzeile
	adc ZH, r1		;Zeiger auf entsprechende Zeile
	lpm r19, Z
	rol r19
	out PORTB, r19		;1.Bit auf PB3 ausgeben
	ld r16, Y+              ;mit vorgezogenen Befehlen statt den nops
	ror r19
	out PORTB, r19		;2.Bit auf PB3 ausgeben
	ldi ZL, low(font6x8<<1)
	lsr r19
	out PORTB, r19		;3.Bit auf PB3 ausgeben
	ldi ZH, high(font6x8<<1)
	lsr r19
	out PORTB, r19		;4.Bit auf PB3 ausgeben
	ldi r17, 0
	lsr r19
	out PORTB, r19		;5.Bit auf PB3 ausgeben
	lsr r19
	lsl r16
	out PORTB, r19		;6.Bit auf PB3 ausgeben
	dec r18
	brne t_L1		;braucht nur noch 12 Takte zwischen den Zeichen, immer noch zu viel!
	ret
#endif

#ifdef ALTEVERSION
textdarstellen_variante3:	;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	ld r16, Y+
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	mov r17, r16
	andi r17, 0x0F
	swap r16
	andi r16, 0xF0
	lsr r17
	ror r16			;r17:r16 = 16*r16/2 = 8*r16
	add ZL, r16
td_L1:	adc ZH, r17		;Zeiger auf entsprechendes Zeichen
	add ZL, fontzeile
	adc ZH, r5		;Zeiger auf entsprechende Zeile
	lpm r19, Z
	rol r19
	out PORTB, r19		;1.Bit auf PB3 ausgeben
	ld r16, Y+		;mit 2 Befehlen vorgezogen
	ldi ZL, low(font6x8<<1)	;werden 4 Takte pro Bit, also 24 Takte pro Zeichen gebraucht
	ror r19
	out PORTB, r19		;2.Bit auf PB3 ausgeben
	ldi ZH, high(font6x8<<1)
	mov r17, r16
	lsr r19
	out PORTB, r19		;3.Bit auf PB3 ausgeben
	andi r17, 0x0F
	swap r16
	lsr r19
	out PORTB, r19		;4.Bit auf PB3 ausgeben
	andi r16, 0xF0
	lsr r19
	lsr r17
	out PORTB, r19		;5.Bit auf PB3 ausgeben
	ror r16
	lsr r19
	add ZL, r16
	out PORTB, r19		;6.Bit auf PB3 ausgeben
	dec r18
	brne td_L1		;braucht nur noch 7 Takte zwischen den Zeichen
	ret
#endif

textdarstellen:			;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	add ZL, fontzeile	 ;Zeiger auf entsprechende Zeile
	adc ZH, r5		 ;r5 = Globales Null
	movw r3:r2, ZH:ZL	;Z in r3:r2 zwischenspeichern
	ld r16, Y+
	ldi r17, 8
	mul r16, r17		;r1:r0 = r16*8
	add ZL, r0
	adc ZH, r1		;Zeiger auf entsprechendes Zeichen
L1:	lpm r19, Z
	rol r19
	out PORTB, r19		;1.Bit auf PB3 ausgeben
	movw ZH:ZL, r3:r2	;mit 2 Befehlen vorgezogen
	ld r16, Y+		;werden 4 Takte pro Bit, also 24 Takte pro Zeichen gebraucht
	ror r19
	out PORTB, r19		;2.Bit auf PB3 ausgeben
	nop
	ldi r17, 8
	lsr r19
	out PORTB, r19		;3.Bit auf PB3 ausgeben
	mul r16, r17
	lsr r19
	out PORTB, r19		;4.Bit auf PB3 ausgeben
	add ZL, r0
	adc ZH, r1
	lsr r19
	out PORTB, r19		;5.Bit auf PB3 ausgeben
	nop
	lsr r19
	dec r18
	out PORTB, r19		;6.Bit auf PB3 ausgeben
	brne L1			;braucht nur noch 3 Takte zwischen den Zeichen
	sbiw YH:YL, 1		;zu viel gelesenes Zeichen rueckgaengig machen
	ret

#ifndef MITINTERRUPT
seite_darstellen:		;Textseite darstellen: 20 Zeilen * 27 Zeichen
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi r20, 20		;r20 = Anzahl Textzeilen
	rjmp sd_L1a
sd_L1:	nop
	nop
	nop
	nop
sd_L1a:	ldi fontzeile, 0
	rcall zeile_darstellen
	dec r20
	brne sd_L1
	ret
#endif

;; Eine gesamte Zeile sieht so aus:
;; 5us Sync-Puls, 9us Schwarzschulter, 45.6us Text, 4.4us leer --> total 64us
;; 80 Takte       144 Takte            729 Takte    71 Takte             1024 Takte
;;                             729 = 27Zeichen * 27Takte/Zeichen 

#ifdef MITINTERRUPT
zeile_darstellen:
;	cbi PORTC, 0		;Synchronisationspuls starten (schon weiter oben gemacht)
	ldi r16, 80-26		;80 - Takte seit Synchr. start
	rcall tickswait
	sbi PORTC, 0		;Synchronisationspuls ende
	ldi r16, 38
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, r5
	brcs zd_ret		;erste 38 Zeilen sind Leerzeilen
	ldi r16, low(38+240)
	ldi r17, high(38+240)
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, r17
	brcc zd_ret		;letzte Zeilen sind Leerzeilen
	cpi fontzeile, 8
	brcc zd_L1
	ldi r16, 144-34		;144 - Anzahl Takte bis zum ersten Bit
	rcall tickswait		;Schwarzschulter
	ldi r18, 27		;Anzahl darzustellende Zeichen
	movw r7:r6, YH:YL
	rcall textdarstellen
	cpi fontzeile, 7	;war das die letzte Fontzeile?
	breq zd_L1		;ja-> erhoehtes Y lassen
	movw YH:YL, r7:r6	;nein: Y wieder auf Anfang der Fontzeile
zd_L1:	inc fontzeile
	cpi fontzeile, 12
	brne zd_ret
	ldi fontzeile, 0
zd_ret:	ret

#else

zd_L1:	nop
	nop
	nop
zeile_darstellen:
	cbi PORTC, 0		;Synchronisationspuls starten
	ldi r16, 80-2
	rcall tickswait
	sbi PORTC, 0		;Synchronisationspuls ende
	ldi r16, 144-21		;144 - Anzahl Takte bis zum ersten Bit
	rcall tickswait		;Schwarzschulter
	ldi r18, 27		;Anzahl darzustellende Zeichen
	rcall textdarstellen
	ldi r16, 71-8		;r16 = 71 - Anzahl Takte bis zum naechsten Synchronpuls
	rcall tickswait		;Hintere Schwarzschulter
	inc fontzeile
	cpi fontzeile, 8
	brcs zd_L1
	rcall leerzeile
	ldi r16, 71-5
	rcall tickswait		;Hintere Schwarzschulter
	rcall leerzeile
	ldi r16, 71-5
	rcall tickswait		;Hintere Schwarzschulter
	rcall leerzeile
	ldi r16, 71-5
	rcall tickswait		;Hintere Schwarzschulter
	rcall leerzeile
	ldi r16, 71-16
	rcall tickswait		;Hintere Schwarzschulter
	ret
#endif

leerzeile:
	cbi PORTC, 0		;Synchronisationspuls starten
	ldi r16, 72-2
	rcall tickswait
	sbi PORTC, 0		;Synchronisationspuls ende
	ldi r16, 144
	rcall tickswait		;Schwarzschulter
	ldi r16, 200
	rcall tickswait
	ldi r16, 200
	rcall tickswait
	ldi r16, 200
	rcall tickswait
	ldi r16, 129
	rcall tickswait		;total 729 Ticks warten
	ret

;; Eine gesamte Zeile sieht so aus:
;; 5us Sync-Puls, 9us Schwarzschulter, 45.6us Text, 4.4us leer --> total 64us
;; 80 Takte       144 Takte            729 Takte    71 Takte             1024 Takte
;;                             729 = 27Zeichen * 27Takte/Zeichen 

gesamte_seite_darstellen:
	cbi PORTC, 0		;Seiten-Synchronisationspuls starten
	ldi r16, 160		;160 usec langer Puls fuer Seitensync, naechster Zeilenpuls erst 3*64=192 usec spaeter
	rcall usecwait
	sbi PORTC, 0		;Seiten-Synchronisationspuls ende
	ldi r16, 30		;noch 192-160 = 32 usec bis zum naechsten Zeilensyncpuls
	rcall usecwait
	ldi r16, 32-8		;r16 = 32 - Anzahl Ticks bis zum naechsten Synchronpuls
	rcall tickswait
	ldi r21, obereleerzeilen ;es braucht mindestens 37 Zeilen bis zum eigentlichen Bildinhalt
	rjmp ge_L1a
ge_L1:	ldi r16, 71-8
	rcall tickswait		;Hintere Schwarzschulter
ge_L1a:	rcall leerzeile
	dec r21
	brne ge_L1
	ldi r16, 71-15
	rcall tickswait		;Hintere Schwarzschulter
	rcall seite_darstellen		;Textseite darstellen: 20 Textzeilen = 240 Video-Zeilen
	ldi r21, 312-obereleerzeilen-240
	rjmp ge_L2a
ge_L2:	ldi r16, 71-8
	rcall tickswait		;Hintere Schwarzschulter
ge_L2a:	rcall leerzeile
	dec r21
	brne ge_L2
	ldi r16, 71-10
	rcall tickswait		;Hintere Schwarzschulter
	ret

tickswait:			;r16 = Anzahl zu wartende Ticks (inklusive ldi und rcall und ret)
	cpi r16, 18		;Minimale Anzahl Ticks: bei weniger stimmts nicht
	brcs ti_ret
	subi r16, 14
	lsr r16
	brcs ti_L1
ti_L1:	lsr r16
	brcc ti_L3
	nop
	nop
ti_L2:	nop
ti_L3:	dec r16
	brne ti_L2
ti_ret:	ret

tickswait17: nop
tickswait16: nop
tickswait15: nop
tickswait14: nop
tickswait13: nop
tickswait12: nop
tickswait11: nop
tickswait10: nop
tickswait9:  nop
tickswait8:  nop
tickswait7:  ret


us_L1:	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
usecwait:			;r16 = Anzahl Microsekunden warten (bei 16MHz) (inklusive ldi und rcall und ret)
	nop
	nop
	nop
	nop
	nop
	dec r16
	brne us_L1
	nop
	ret
