;; bas.asm                                       letzte Aenderung: 25.11.2012
;* Programm fuer RS232-BAS-Wandler (Pollin-Bausatz oder Nachbau)
;* Controller: ATmega8A oder ATmega1284P mit 16 oder 20 MHz Quarz
;*
;* Schaltschema: PB3 (MOSI, Pin17) ueber Diode und 330 Ohm auf Chinch-Buchse
;*		 beim Atmega1284P PB0 verwendet (oder fuer MOSI PB5)
;*               PC0 (Pin23) ueber Diode und 1k auf Chinch-Buchse
;*               16MHz Quarz an Pin9 und Pin10, je 22pF auf GND
;*		 beim Atmega1284P 20MHz Quarz an Pin12 und Pin13
;*               Reset (Pin1 oder Pin9): 47k auf VCC, Reset-Taster gegen GND
;*               RXD (PD0, Pin 2) = Serieller Eingang
;*               (nicht benutzt: TXD (PD1, Pin 3) = Serieller Ausgang)
;*
;* Copyright: Opensource, Freeware
;*
.define VERSION="1.0"		;weiter unten auch anpassen!
;*
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
;- Useroptionen:

#define ATMEGA8
;#define ATMEGA1284P

;#define DREIZEHNZEILEN		;13 Zeilen a 27 Zeichen mit grosser Schrift
;#define VIELEZEILEN1		;22 Zeilen a 27 Zeichen mit breiter Schrift
#define VIELEZEILEN2		;22 Zeilen a 42 Zeichen mit kleiner Schrift

;#define INTERLACE		;fuer Interlace-Modus, ohne sollte es weniger flimmern

.equ BAUDRATE=19200		;moegliche Werte: 9600, 19200, 38400, 57600, (76800)

;- Ende Useroptionen
;-----------------------------------------------------------------------------

;#define TICKSTESTS	;um das Unterprogramm tickswait zu ueberpruefen
;#define DEBUG1		;zur Fehlersuche
;#define DEBUG2

#define VSYNC1		;VSYNC auf separater Leitung PC1
;#define VGATEST	;VGA braucht offenbar andere Synchronisation

#ifdef VIELEZEILEN1
#define ZWANZIGZEILEN
#endif

#ifdef VIELEZEILEN2
#define ZWANZIGZEILEN
#define MITSPI		;wird fuer mehr als 27 Zeichen pro Zeile benoetigt
#define SPIDOPPEL	;fuer doppelte SPI-Geschwindigkeit -> schmalere Zeichen
;#define SPITEST		;test
;#define SPITEST1		;test
#endif

#ifdef ATMEGA8
.include "m8def.inc"		;ATmega8
.equ F_CPU=16000000		;16 MHz Quarz
.equ SRAM_START = 0x60		;Start vom SRAM beim Atmega8
#else
.include "m1284Pdef.inc"        ;ATmega1284P
.equ F_CPU=20000000		;20 MHz Quarz
#endif

	rjmp main
.org OC1Aaddr
	rjmp timer1_match_A

#ifdef INTERLACE
.org OC1Baddr
	rjmp timer1_match_B
#endif

;-----------------------------------------------------------------------------
; Globale Registerbelegungen:
; r4=Sicherung von SREG im Interrupt
; r5=0  Global auf 0 gesetzt
; r7:r6 = Zwischenspeicher fuer Y
; r3:r2 = Zwischenspeicher fuer Z
; Y (r29:r28) = Zeiger ins RAM auf den darzustellenden Text
; Z (r31:r30) = Zeiger ins Flash auf die Fonttabelle
.def zero=r5
.def fontzeile=r22
.def fontzeileflag=r23
.def zeilenzaehlerL=r24
.def zeilenzaehlerH=r25
.def syncstatus=r8
#ifdef INTERLACE
.def halbbildflag=r9
.def mitteflag=r10
.def vsyncflag=r11
#endif
; Nur noch r0-r1, r12-r21 und XH:XL (r27:r26) sind im Hauptprogramm verwendbar

;-----------------------------------------------------------------------------
; Registerbelegung im Hauptprogramm:
; XH:XL (r27:r26) = aktuelle Position im Textpuffer
.def status=r19

;-----------------------------------------------------------------------------
; Konstanten:
#ifdef ATMEGA8
.equ SYNCBIT=PC0
.equ BILDBIT=PB3
#else
.equ SYNCBIT=PC0
.equ BILDBIT=PB0
#endif

#ifdef MITSPI
.equ DDR_SPI=DDRB
.equ PORT_SPI=PORTB
#ifdef ATMEGA8
.equ MOSI_BIT=PB3
.equ SCK_BIT=PB5
.equ SS_BIT=PB2
#endif
#ifdef ATMEGA1284P
.equ MOSI_BIT=PB5
.equ SCK_BIT=PB7
.equ SS_BIT=PB4
#endif
#endif

#ifdef VGATEST
.equ zeilentotal=525
.equ obereleerzeilen=45
#else
.equ zeilentotal=312
.equ obereleerzeilen=36
#endif
.equ Z312=zeilentotal

#ifdef ZWANZIGZEILEN
.equ PIXELPROTEXTZEILE=12
#ifdef VGATEST
.equ INHALTZEILEN=200
#else
.equ INHALTZEILEN=264		;so gibts sogar 22 Zeilen
#endif
.equ MAXFONTZEILE=PIXELPROTEXTZEILE
#else
.equ PIXELPROTEXTZEILE=20
#ifdef VGATEST
.equ INHALTZEILEN=200
#else
.equ INHALTZEILEN=260
#endif
.equ MAXFONTZEILE=PIXELPROTEXTZEILE/2
#endif

#ifdef SPITEST
.equ ZEICHENPROZEILE=8
#else

#ifdef MITSPI
#ifdef SPIDOPPEL
.equ ZEICHENPROZEILE=42
#else
.equ ZEICHENPROZEILE=22
#endif
#else
.equ ZEICHENPROZEILE=27
#endif

#endif

.equ ANZAHLTEXTZEILEN=INHALTZEILEN/PIXELPROTEXTZEILE ;22 oder 13
.equ ANZAHLTEXTBYTES=ANZAHLTEXTZEILEN*ZEICHENPROZEILE
;.equ untereleerzeilen=zeilentotal-obereleerzeilen-INHALTZEILEN

.equ SERBUFFERSIZE=8		;Fuer rs232-Empfang (nur Werte 2^n verwendbar)

#ifdef DEBUG2
.equ LED_PORT=PORTD
.equ LED_DDR=DDRD
.equ LED_ROT=PD4
.equ LED_GRUEN=PD3
.equ LED_GELB=PD2
#endif

;-----------------------------------------------------------------------------
; Variablen im SRAM:
                .DSEG
serbuffer:	.BYTE SERBUFFERSIZE
jread:		.BYTE 1			; wenn jread==jwrite dann ist der Buffer leer
jwrite:		.BYTE 1			; wenn jread==jwrite+1 dann ist der Buffer voll
textspeicher:	.BYTE ANZAHLTEXTBYTES
		.CSEG

;-----------------------------------------------------------------------------
; RS232-Teil:
;.equ UBRR_VAL = (F_CPU+BAUDRATE*8)/(BAUDRATE*16)-1  ;clever runden
.equ UBRR_VAL = (F_CPU+BAUDRATE*4)/(BAUDRATE*8)-1  ;clever runden, Wert fur mit Doublespeed
.equ BAUD_REAL  = (F_CPU/(8*(UBRR_VAL+1)))     	   ; Reale Baudrate
.equ BAUD_ERROR = ((BAUD_REAL*1000)/BAUDRATE-1000) ; Fehler in Promille

.if ((BAUD_ERROR>10) || (BAUD_ERROR<-10))       ; max. +/-10 Promille Fehler
  .warning Systematischer Fehler der Baudrate groesser 1%
.endif 

.if ((BAUD_ERROR>20) || (BAUD_ERROR<-20))       ; max. +/-20 Promille Fehler
  .error Systematischer Fehler der Baudrate groesser 2% und damit zu hoch! 
.endif 

#ifdef ATMEGA1284P
.equ UCSRA=UCSR0A
.equ UCSRB=UCSR0B
.equ UCSRC=UCSR0C
.equ UDR=UDR0
.equ U2X=U2X0
.equ RXEN=RXEN0
.equ TXEN=TXEN0
.equ UCSZ1=UCSZ01
.equ UCSZ0=UCSZ00
.equ RXC=RXC0
.equ UBRRH=UBRR0H
.equ UBRRL=UBRR0L
#endif

uart_init:			; Initialisierung fuer Asynchron 8N1
	clr r16
	sts jread, r16
	sts jwrite, r16
	ldi r16, LOW(UBRR_VAL)
	ldi r17, HIGH(UBRR_VAL)
	ldi r18, (1<<U2X)		;fuer Doublespeed
	ldi r19, (1<<RXEN) | (0<<TXEN)	;nur Empfangen, ohne Interrupts
#ifdef ATMEGA8
	out UBRRH, r17
	out UBRRL, r16
	out UCSRA, r18
	out UCSRB, r19
	ldi r20, (1<<URSEL) | (1<<UCSZ1)|(1<<UCSZ0)
	out UCSRC, r20
#else
	sts UBRRH, r17
	sts UBRRL, r16
	sts UCSRA, r18
	sts UCSRB, r19
	ldi r20, (1<<UCSZ1)|(1<<UCSZ0)
	sts UCSRC, r20
#endif
	ret

uart_check:		    ; Pruefen ob ein Zeichen empfangen wurde, und wenn ja im Puffer speichern
			    ; braucht 27 Takte, r16-r18,XH,XL zerstoert
	ldi XL, low(serbuffer)
	ldi XH, high(serbuffer)
	lds r17, jwrite
	lds r18, jread
	add XL, r17
	adc XH, zero
#ifdef ATMEGA8
	in r16, UCSRA
#else
	lds r16, UCSRA
#endif
	andi r16, (1<<RXC)	;neues Zeichen empfangen?
	breq ua_L1		;nein->
#ifdef ATMEGA8
	in r16, UDR		;ja: einlesen
#else
	lds r16, UDR		;ja: einlesen
#endif
	inc r17
	andi r17, SERBUFFERSIZE-1
	cp r17, r18		;ist Puffer voll?
	breq ua_L2		;ja-> Zeichen ignorieren
	st X, r16		;nein: Zeichen im Puffer speichern
	sts jwrite, r17
	ret
ua_L1:
#ifdef ATMEGA8
	nop
#else
	adiw r24, 0		;statt 2 mal nop
#endif
	adiw r24, 0		;statt 2 mal nop
	adiw r24, 0		;statt 2 mal nop
ua_L2:	nop
	adiw r24, 0		;statt 2 mal nop
ua_ret:
	ret

;-----------------------------------------------------------------------------
; Zeichentabelle im FLASH:
font6x8:				     ;aus 6x8_horizontal_LSB_1.asm
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x00
.db 0x70,0x88,0xD8,0x88,0xA8,0x88,0x70,0x00,	; 0x01
.db 0x70,0xF8,0xA8,0xF8,0x88,0xF8,0x70,0x00,	; 0x02
.db 0x00,0x50,0xF8,0xF8,0xF8,0x70,0x20,0x00,	; 0x03
.db 0x00,0x20,0x70,0xF8,0xF8,0x70,0x20,0x00,	; 0x04
.db 0x20,0x70,0x70,0x20,0xF8,0xF8,0x20,0x00,	; 0x05
.db 0x00,0x20,0x70,0xF8,0xF8,0x20,0x70,0x00,	; 0x06
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x07
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x08
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x09
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x0A
.db 0x00,0xE0,0xC0,0xB0,0x48,0x48,0x30,0x00,	; 0x0B
.db 0x70,0x88,0x88,0x70,0x20,0x70,0x20,0x00,	; 0x0C
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x0D
.db 0xC0,0xB0,0xD0,0xB0,0xD0,0xD8,0x18,0x00,	; 0x0E
.db 0x00,0xA8,0x70,0xD8,0x70,0xA8,0x00,0x00,	; 0x0F
.db 0x10,0x30,0x70,0xF0,0x70,0x30,0x10,0x00,	; 0x10
.db 0x40,0x60,0x70,0x78,0x70,0x60,0x40,0x00,	; 0x11
.db 0x20,0x70,0xF8,0x20,0xF8,0x70,0x20,0x00,	; 0x12
.db 0x50,0x50,0x50,0x50,0x50,0x00,0x50,0x00,	; 0x13
.db 0xF0,0xA8,0xA8,0xB0,0xA0,0xA0,0xA0,0x00,	; 0x14
.db 0x70,0x88,0x30,0x50,0x60,0x88,0x70,0x00,	; 0x15
.db 0x00,0x00,0x00,0x00,0x00,0x78,0x78,0x00,	; 0x16
.db 0x20,0x70,0xF8,0x20,0xF8,0x70,0x20,0x70,	; 0x17
.db 0x20,0x70,0xF8,0x20,0x20,0x20,0x20,0x00,	; 0x18
.db 0x20,0x20,0x20,0x20,0xF8,0x70,0x20,0x00,	; 0x19
.db 0x00,0x20,0x60,0xF8,0x60,0x20,0x00,0x00,	; 0x1A
.db 0x00,0x20,0x30,0xF8,0x30,0x20,0x00,0x00,	; 0x1B
.db 0x00,0x00,0x00,0x08,0x08,0x08,0xF8,0x00,	; 0x1C
.db 0x00,0x50,0x50,0xF8,0x50,0x50,0x00,0x00,	; 0x1D
.db 0x20,0x20,0x70,0x70,0xF8,0xF8,0x00,0x00,	; 0x1E
.db 0xF8,0xF8,0x70,0x70,0x20,0x20,0x00,0x00,	; 0x1F
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x20
.db 0x20,0x70,0x70,0x20,0x20,0x00,0x20,0x00,	; 0x21
.db 0xD8,0xD8,0x48,0x00,0x00,0x00,0x00,0x00,	; 0x22
.db 0x00,0x50,0xF8,0x50,0x50,0xF8,0x50,0x00,	; 0x23
.db 0x10,0x70,0x08,0x30,0x40,0x38,0x20,0x00,	; 0x24
.db 0x98,0x98,0x40,0x20,0x10,0xC8,0xC8,0x00,	; 0x25
.db 0x10,0x28,0x28,0x10,0xA8,0x48,0xB0,0x00,	; 0x26
.db 0x30,0x30,0x10,0x00,0x00,0x00,0x00,0x00,	; 0x27
.db 0x20,0x10,0x10,0x10,0x10,0x10,0x20,0x00,	; 0x28
.db 0x10,0x20,0x20,0x20,0x20,0x20,0x10,0x00,	; 0x29
.db 0x00,0x50,0x70,0xF8,0x70,0x50,0x00,0x00,	; 0x2A
.db 0x00,0x20,0x20,0xF8,0x20,0x20,0x00,0x00,	; 0x2B
.db 0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x10,	; 0x2C
.db 0x00,0x00,0x00,0xF8,0x00,0x00,0x00,0x00,	; 0x2D
.db 0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,	; 0x2E
.db 0x00,0x80,0x40,0x20,0x10,0x08,0x00,0x00,	; 0x2F
.db 0x70,0x88,0xC8,0xA8,0x98,0x88,0x70,0x00,	; 0x30
.db 0x20,0x30,0x20,0x20,0x20,0x20,0x70,0x00,	; 0x31
.db 0x70,0x88,0x80,0x60,0x10,0x08,0xF8,0x00,	; 0x32
.db 0x70,0x88,0x80,0x70,0x80,0x88,0x70,0x00,	; 0x33
.db 0x40,0x60,0x50,0x48,0xF8,0x40,0x40,0x00,	; 0x34
.db 0xF8,0x08,0x08,0x78,0x80,0x88,0x70,0x00,	; 0x35
.db 0x60,0x10,0x08,0x78,0x88,0x88,0x70,0x00,	; 0x36
.db 0xF8,0x80,0x40,0x20,0x10,0x10,0x10,0x00,	; 0x37
.db 0x70,0x88,0x88,0x70,0x88,0x88,0x70,0x00,	; 0x38
.db 0x70,0x88,0x88,0xF0,0x80,0x40,0x30,0x00,	; 0x39
.db 0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x00,	; 0x3A
.db 0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x10,	; 0x3B
.db 0x40,0x20,0x10,0x08,0x10,0x20,0x40,0x00,	; 0x3C
.db 0x00,0x00,0xF8,0x00,0x00,0xF8,0x00,0x00,	; 0x3D
.db 0x10,0x20,0x40,0x80,0x40,0x20,0x10,0x00,	; 0x3E
.db 0x70,0x88,0x80,0x60,0x20,0x00,0x20,0x00,	; 0x3F
.db 0x70,0x88,0xE8,0xA8,0xE8,0x08,0x70,0x00,	; 0x40
.db 0x70,0x88,0x88,0x88,0xF8,0x88,0x88,0x00,	; 0x41
.db 0x78,0x88,0x88,0x78,0x88,0x88,0x78,0x00,	; 0x42
.db 0x70,0x88,0x08,0x08,0x08,0x88,0x70,0x00,	; 0x43
.db 0x78,0x88,0x88,0x88,0x88,0x88,0x78,0x00,	; 0x44
.db 0xF8,0x08,0x08,0x78,0x08,0x08,0xF8,0x00,	; 0x45
.db 0xF8,0x08,0x08,0x78,0x08,0x08,0x08,0x00,	; 0x46
.db 0x70,0x88,0x08,0xE8,0x88,0x88,0xF0,0x00,	; 0x47
.db 0x88,0x88,0x88,0xF8,0x88,0x88,0x88,0x00,	; 0x48
.db 0x70,0x20,0x20,0x20,0x20,0x20,0x70,0x00,	; 0x49
.db 0x80,0x80,0x80,0x80,0x88,0x88,0x70,0x00,	; 0x4A
.db 0x88,0x48,0x28,0x18,0x28,0x48,0x88,0x00,	; 0x4B
.db 0x08,0x08,0x08,0x08,0x08,0x08,0xF8,0x00,	; 0x4C
.db 0x88,0xD8,0xA8,0x88,0x88,0x88,0x88,0x00,	; 0x4D
.db 0x88,0x98,0xA8,0xC8,0x88,0x88,0x88,0x00,	; 0x4E
.db 0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00,	; 0x4F
.db 0x78,0x88,0x88,0x78,0x08,0x08,0x08,0x00,	; 0x50
.db 0x70,0x88,0x88,0x88,0xA8,0x48,0xB0,0x00,	; 0x51
.db 0x78,0x88,0x88,0x78,0x48,0x88,0x88,0x00,	; 0x52
.db 0x70,0x88,0x08,0x70,0x80,0x88,0x70,0x00,	; 0x53
.db 0xF8,0x20,0x20,0x20,0x20,0x20,0x20,0x00,	; 0x54
.db 0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,	; 0x55
.db 0x88,0x88,0x88,0x88,0x88,0x50,0x20,0x00,	; 0x56
.db 0x88,0x88,0xA8,0xA8,0xA8,0xA8,0x50,0x00,	; 0x57
.db 0x88,0x88,0x50,0x20,0x50,0x88,0x88,0x00,	; 0x58
.db 0x88,0x88,0x88,0x50,0x20,0x20,0x20,0x00,	; 0x59
.db 0x78,0x40,0x20,0x10,0x08,0x08,0x78,0x00,	; 0x5A
.db 0x70,0x10,0x10,0x10,0x10,0x10,0x70,0x00,	; 0x5B
.db 0x00,0x08,0x10,0x20,0x40,0x80,0x00,0x00,	; 0x5C
.db 0x70,0x40,0x40,0x40,0x40,0x40,0x70,0x00,	; 0x5D
.db 0x20,0x50,0x88,0x00,0x00,0x00,0x00,0x00,	; 0x5E
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFC,	; 0x5F
.db 0x30,0x30,0x20,0x00,0x00,0x00,0x00,0x00,	; 0x60
.db 0x00,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x61
.db 0x08,0x08,0x78,0x88,0x88,0x88,0x78,0x00,	; 0x62
.db 0x00,0x00,0x70,0x88,0x08,0x88,0x70,0x00,	; 0x63
.db 0x80,0x80,0xF0,0x88,0x88,0x88,0xF0,0x00,	; 0x64
.db 0x00,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x65
.db 0x60,0x10,0x10,0x78,0x10,0x10,0x10,0x00,	; 0x66
.db 0x00,0x00,0xF0,0x88,0x88,0xF0,0x80,0x70,	; 0x67
.db 0x08,0x08,0x38,0x48,0x48,0x48,0x48,0x00,	; 0x68
.db 0x20,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x69
.db 0x40,0x00,0x60,0x40,0x40,0x40,0x48,0x30,	; 0x6A
.db 0x08,0x08,0x48,0x28,0x18,0x28,0x48,0x00,	; 0x6B
.db 0x20,0x20,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x6C
.db 0x00,0x00,0x58,0xA8,0xA8,0x88,0x88,0x00,	; 0x6D
.db 0x00,0x00,0x38,0x48,0x48,0x48,0x48,0x00,	; 0x6E
.db 0x00,0x00,0x70,0x88,0x88,0x88,0x70,0x00,	; 0x6F
.db 0x00,0x00,0x78,0x88,0x88,0x88,0x78,0x08,	; 0x70
.db 0x00,0x00,0xF0,0x88,0x88,0x88,0xF0,0x80,	; 0x71
.db 0x00,0x00,0x68,0x90,0x10,0x10,0x38,0x00,	; 0x72
.db 0x00,0x00,0x70,0x08,0x70,0x80,0x70,0x00,	; 0x73
.db 0x00,0x10,0x78,0x10,0x10,0x50,0x20,0x00,	; 0x74
.db 0x00,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x75
.db 0x00,0x00,0x88,0x88,0x88,0x50,0x20,0x00,	; 0x76
.db 0x00,0x00,0x88,0x88,0xA8,0xF8,0x50,0x00,	; 0x77
.db 0x00,0x00,0x48,0x48,0x30,0x48,0x48,0x00,	; 0x78
.db 0x00,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0x79
.db 0x00,0x00,0x78,0x40,0x30,0x08,0x78,0x00,	; 0x7A
.db 0x60,0x10,0x10,0x18,0x10,0x10,0x60,0x00,	; 0x7B
.db 0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,	; 0x7C
.db 0x30,0x40,0x40,0xC0,0x40,0x40,0x30,0x00,	; 0x7D
.db 0x50,0x28,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x7E
.db 0x20,0x70,0xD8,0x88,0x88,0xF8,0x00,0x00,	; 0x7F
.db 0x70,0x88,0x08,0x08,0x88,0x70,0x20,0x30,	; 0x80
.db 0x48,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x81
.db 0xC0,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x82
.db 0x70,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x83
.db 0x50,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x84
.db 0x30,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x85
.db 0x70,0x50,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x86
.db 0x00,0x70,0x88,0x08,0x88,0x70,0x20,0x30,	; 0x87
.db 0x70,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x88
.db 0x50,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x89
.db 0x30,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x8A
.db 0x50,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8B
.db 0x70,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8C
.db 0x10,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8D
.db 0x50,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0x8E
.db 0x70,0x50,0x70,0xD8,0x88,0xF8,0x88,0x00,	; 0x8F
.db 0xC0,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0x90
.db 0x00,0x00,0x78,0xA0,0xF8,0x28,0xF0,0x00,	; 0x91
.db 0xF0,0x28,0x28,0xF8,0x28,0x28,0xE8,0x00,	; 0x92
.db 0x70,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x93
.db 0x50,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x94
.db 0x18,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x95
.db 0x70,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x96
.db 0x18,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x97
.db 0x50,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0x98
.db 0x48,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0x99
.db 0x50,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0x9A
.db 0x00,0x00,0x80,0x70,0x68,0x58,0x38,0x04,	; 0x9B
.db 0x60,0x90,0x10,0x78,0x10,0x90,0xE8,0x00,	; 0x9C
.db 0xF0,0xC8,0xA8,0xA8,0xA8,0x98,0x78,0x00,	; 0x9D
.db 0x00,0x88,0x50,0x20,0x50,0x88,0x00,0x00,	; 0x9E
.db 0x40,0xA0,0x20,0x70,0x20,0x20,0x28,0x10,	; 0x9F
.db 0x60,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0xA0
.db 0x60,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0xA1
.db 0x60,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0xA2
.db 0x60,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0xA3
.db 0x50,0x28,0x00,0x38,0x48,0x48,0x48,0x00,	; 0xA4
.db 0x50,0x28,0x00,0x48,0x58,0x68,0x48,0x00,	; 0xA5
.db 0x70,0x80,0xF0,0x88,0xF0,0x00,0xF0,0x00,	; 0xA6
.db 0x30,0x48,0x48,0x48,0x30,0x00,0x78,0x00,	; 0xA7
.db 0x20,0x00,0x20,0x30,0x08,0x88,0x70,0x00,	; 0xA8
.db 0x78,0xA4,0xD4,0xB4,0xD4,0x84,0x78,0x00,	; 0xA9
.db 0x00,0x00,0xFC,0x80,0x80,0x00,0x00,0x00,	; 0xAA
.db 0x08,0x48,0x28,0x70,0x88,0x40,0xE0,0x00,	; 0xAB
.db 0x08,0x48,0x28,0xD0,0xA8,0xE0,0x80,0x00,	; 0xAC
.db 0x20,0x00,0x20,0x20,0x70,0x70,0x20,0x00,	; 0xAD
.db 0x00,0x00,0x90,0x48,0x90,0x00,0x00,0x00,	; 0xAE
.db 0x00,0x00,0x48,0x90,0x48,0x00,0x00,0x00,	; 0xAF
.db 0xA8,0x00,0x54,0x00,0xA8,0x00,0x54,0x00,	; 0xB0
.db 0xA8,0x54,0xA8,0x54,0xA8,0x54,0xA8,0x54,	; 0xB1
.db 0x54,0xFC,0xA8,0xFC,0x54,0xFC,0xA8,0xFC,	; 0xB2
.db 0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,	; 0xB3
.db 0x20,0x20,0x20,0x3C,0x20,0x20,0x20,0x20,	; 0xB4
.db 0x60,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB5
.db 0x70,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB6
.db 0x30,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB7
.db 0x78,0x84,0xB4,0x94,0xB4,0x84,0x78,0x00,	; 0xB8
.db 0x28,0x2C,0x20,0x2C,0x28,0x28,0x28,0x28,	; 0xB9
.db 0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,	; 0xBA
.db 0x00,0x3C,0x20,0x2C,0x28,0x28,0x28,0x28,	; 0xBB
.db 0x28,0x2C,0x20,0x3C,0x00,0x00,0x00,0x00,	; 0xBC
.db 0x00,0x20,0x70,0x08,0x08,0x70,0x20,0x00,	; 0xBD
.db 0x88,0x50,0x20,0xF8,0x20,0xF8,0x20,0x00,	; 0xBE
.db 0x00,0x00,0x00,0x3C,0x20,0x20,0x20,0x20,	; 0xBF
.db 0x20,0x20,0x20,0xE0,0x00,0x00,0x00,0x00,	; 0xC0
.db 0x20,0x20,0x20,0xFC,0x00,0x00,0x00,0x00,	; 0xC1
.db 0x00,0x00,0x00,0xFC,0x20,0x20,0x20,0x20,	; 0xC2
.db 0x20,0x20,0x20,0xE0,0x20,0x20,0x20,0x20,	; 0xC3
.db 0x00,0x00,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xC4
.db 0x20,0x20,0x20,0xFC,0x20,0x20,0x20,0x20,	; 0xC5
.db 0xA0,0x50,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0xC6
.db 0xA0,0x50,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xC7
.db 0x28,0xE8,0x08,0xF8,0x00,0x00,0x00,0x00,	; 0xC8
.db 0x00,0xF8,0x08,0xE8,0x28,0x28,0x28,0x28,	; 0xC9
.db 0x28,0xEC,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xCA
.db 0x00,0xFC,0x00,0xEC,0x28,0x28,0x28,0x28,	; 0xCB
.db 0x28,0xE8,0x08,0xE8,0x28,0x28,0x28,0x28,	; 0xCC
.db 0x00,0xFC,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xCD
.db 0x28,0xEC,0x00,0xEC,0x28,0x28,0x28,0x28,	; 0xCE
.db 0x88,0x70,0x88,0x88,0x88,0x70,0x88,0x00,	; 0xCF
.db 0x30,0x08,0x10,0x20,0x70,0x48,0x30,0x00,	; 0xD0
.db 0x70,0x90,0x90,0xB8,0x90,0x90,0x70,0x00,	; 0xD1
.db 0x70,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD2
.db 0x50,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD3
.db 0x30,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD4
.db 0x20,0x20,0x20,0x00,0x00,0x00,0x00,0x00,	; 0xD5
.db 0x60,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD6
.db 0x70,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD7
.db 0x50,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD8
.db 0x20,0x20,0x20,0x3C,0x00,0x00,0x00,0x00,	; 0xD9
.db 0x00,0x00,0x00,0xE0,0x20,0x20,0x20,0x20,	; 0xDA
.db 0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,	; 0xDB
.db 0x00,0x00,0x00,0x00,0xFC,0xFC,0xFC,0xFC,	; 0xDC
.db 0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,	; 0xDD
.db 0x30,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xDE
.db 0xFC,0xFC,0xFC,0xFC,0x00,0x00,0x00,0x00,	; 0xDF
.db 0x60,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE0
.db 0x00,0x38,0x48,0x38,0x48,0x48,0x38,0x08,	; 0xE1
.db 0x70,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE2
.db 0x18,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE3
.db 0x50,0x28,0x00,0x30,0x48,0x48,0x30,0x00,	; 0xE4
.db 0x50,0x28,0x30,0x48,0x48,0x48,0x30,0x00,	; 0xE5
.db 0x00,0x00,0x48,0x48,0x48,0x38,0x08,0x08,	; 0xE6
.db 0x00,0x18,0x08,0x38,0x48,0x38,0x08,0x18,	; 0xE7
.db 0x18,0x08,0x38,0x48,0x48,0x38,0x08,0x18,	; 0xE8
.db 0x60,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE9
.db 0x70,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xEA
.db 0x18,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xEB
.db 0x60,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0xEC
.db 0x60,0x00,0x88,0x50,0x20,0x20,0x20,0x00,	; 0xED
.db 0x00,0x70,0x00,0x00,0x00,0x00,0x00,0x00,	; 0xEE
.db 0x30,0x30,0x10,0x00,0x00,0x00,0x00,0x00,	; 0xEF
.db 0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,	; 0xF0
.db 0x00,0x20,0x70,0x20,0x00,0x70,0x00,0x00,	; 0xF1
.db 0x00,0x00,0xF8,0x00,0x00,0xF8,0x00,0x00,	; 0xF2
.db 0x0C,0x58,0x2C,0xD0,0xA8,0xE0,0x80,0x00,	; 0xF3
.db 0xF0,0xA8,0xA8,0xB0,0xA0,0xA0,0xA0,0x00,	; 0xF4
.db 0x70,0x88,0x30,0x50,0x60,0x88,0x70,0x00,	; 0xF5
.db 0x00,0x20,0x00,0xF8,0x00,0x20,0x00,0x00,	; 0xF6
.db 0x00,0x00,0x00,0x70,0x60,0x00,0x00,0x00,	; 0xF7
.db 0x30,0x48,0x48,0x30,0x00,0x00,0x00,0x00,	; 0xF8
.db 0x00,0x00,0x00,0x50,0x00,0x00,0x00,0x00,	; 0xF9
.db 0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,	; 0xFA
.db 0x10,0x18,0x10,0x10,0x00,0x00,0x00,0x00,	; 0xFB
.db 0x38,0x10,0x30,0x18,0x00,0x00,0x00,0x00,	; 0xFC
.db 0x18,0x20,0x10,0x38,0x00,0x00,0x00,0x00,	; 0xFD
.db 0x00,0x00,0x78,0x78,0x78,0x78,0x00,0x00,	; 0xFE
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 	; 0xFF

.equ LF=0x0A
starttext:
;	.db "basi.asm Version ",VERSION,LF,0	;geht in avra-1.3.0 nicht
	.db "basi.asm Version ","1.0",LF,0
beispieltext:
	.db "Autor: Rolf Pfister",LF
	.db "Copyr: Freeware OpenSource",LF,0
beispieltext2:
	.db "www.rolfp.ch/elektronik/#rs232bas",LF
#ifdef ZWANZIGZEILEN
	.db "Steuerzeichen: ",LF
	.db " 0x01: Sonderzeichen ",LF
	.db " 0x02: Zeilenauswahl ",LF
	.db " 0x03: Spaltenauswahl",LF
	.db " 0x04: Seite loeschen",LF
	.db " 0x05: Hexmodus ein/aus",LF
#endif
	.db 0,0

;-----------------------------------------------------------------------------
#ifdef VGATEST
.equ NormalerCompwert = 509-1	 ;31.78 usec fuer Zeilenlaenge
#else
.equ NormalerCompwert = (F_CPU*64/1000000)-1	 ;64 usec fuer Zeilenlaenge
#endif

#ifdef INTERLACE

.equ ZwischenpulsCompwert = (F_CPU*28/1000000)-1 ;28 usec fuer Inverse Zwischenpulse
.equ InversepulseCompwert = (F_CPU*60/1000000)-1 ;60 usec fuer Inverse Hauptpulse
.equ InverseoffCompwert = (F_CPU*64/1000000)+1	 ;unerreichbarer Wert zum Zwischenpulse ausschalten

timer1_init:
#ifdef ATMEGA8
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	out TCCR1B, r16
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	out OCR1AH, r17
	out OCR1AL, r16
	ldi r16, low(ZwischenpulsCompwert)
	ldi r17, high(ZwischenpulsCompwert)
	out OCR1BH, r17
	out OCR1BL, r16
	ldi r16, 0		;Startwert des Timers = 0
	out TCNT1H, r16
	out TCNT1L, r16
	ldi r16, (1<<OCIE1A)|(1<<OCIE1B)  ;Timer1 Interrupts bei Vergleichswert
	out TIMSK, r16
#else
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	sts TCCR1B, r16			;beim Atmega1284P muss fuer einige io-Register sts statt out verwendet werden
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	sts OCR1AH, r17
	sts OCR1AL, r16
	ldi r16, low(ZwischenpulsCompwert)
	ldi r17, high(ZwischenpulsCompwert)
	sts OCR1BH, r17
	sts OCR1BL, r16
	ldi r16, 0		;Startwert des Timers = 0
	sts TCNT1H, r16
	sts TCNT1L, r16
	ldi r16, (1<<OCIE1A)|(1<<OCIE1B)  ;Timer1 Interrupts bei Vergleichswert
	sts TIMSK1, r16
#endif
	ldi zeilenzaehlerL, low(Z312-1)
	ldi zeilenzaehlerH, high(Z312-1) ;Starten mit letzter Zeile
	clr syncstatus
	clr halbbildflag
	inc halbbildflag		;Starten mit letzter Zeile vom 2. Halbbild
	clr mitteflag
	inc mitteflag			;naechster Zwischenpuls kommt in Zeilenmitte
	clr vsyncflag
	ret

#else

timer1_init:			;fuer Variante ohne Interlace
#ifdef ATMEGA8
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	out TCCR1B, r16
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	out OCR1AH, r17
	out OCR1AL, r16
	ldi r16, 0		;Startwert des Timers = 0
	out TCNT1H, r16
	out TCNT1L, r16
	ldi r16, (1<<OCIE1A)	;Timer1 Interrupts bei Vergleichswert
	out TIMSK, r16
#else
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	sts TCCR1B, r16			;beim Atmega1284P muss fuer einige io-Register sts statt out verwendet werden
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	sts OCR1AH, r17
	sts OCR1AL, r16
	ldi r16, 0		;Startwert des Timers = 0
	sts TCNT1H, r16
	sts TCNT1L, r16
	ldi r16, (1<<OCIE1A)	;Timer1 Interrupts bei Vergleichswert
	sts TIMSK1, r16
#endif
	ldi zeilenzaehlerL, low(Z312-1)
	ldi zeilenzaehlerH, high(Z312-1) ;Starten mit letzter Zeile
	clr syncstatus
	ret

#endif

;; Eine normale Video-Zeile sieht so aus:
;; 4.5us Sync-Puls, 7.5us Schwarzschulter, 0.75us Rand, 48.0us Text, 3.25us leer --> total 64us
;; 72 Takte         120 Takte              12           768 Takte    52 Takte           1024 Takte bei 16MHz
;; 90 Takte         150 Takte              15           960 Takte    65 Takte           1280 Takte bei 20MHz
;;                             27Zeichen * 27Takte/Zeichen = 729 (bei 16MHz)
;;                             42Zeichen * 18Takte/Zeichen = 756 (bei 16MHz)
;;                             30Zeichen * 32Takte/Zeichen = 960 (bei 20MHz)

;; VGA-Zeile:
;; 3.81us Sync-Puls, 0.64us Schwarzschulter, 25.42us Rand+Text, 1.91us leer --> total 31.78us
;; 61 Takte          11 Takte                406 Takte          31 Takte              509 Takte bei 16MHz
;;                             22Zeichen * 18Takte/Zeichen = 396 (bei 16MHz)

#ifdef VGATEST
	.equ SYNC1=61
	.equ SCHWARZSCHULTER=11+2*18 ;=47
	.equ RANDLINKS=6
#else
.if F_CPU==16000000
	.equ SYNC1=72		;4.5us
	.equ SCHWARZSCHULTER=120 ;Schwarzschulter sollte zusammen mit Syncpuls genau 12us sein
	.equ RANDLINKS=12
.endif
.if F_CPU==20000000
	.equ SYNC1=90		;4.5us
	.equ SCHWARZSCHULTER=150
	.equ RANDLINKS=15
.endif
#endif
	.equ LEER1=SCHWARZSCHULTER+RANDLINKS

#ifdef INTERLACE

timer1_match_A:			;Interrupt-Routine, Variante mit INERLACE-Modus
	cbi PORTC, 0		;Synchronisationspuls starten
	in r4, SREG
	push r16
	push r17
	push r18
	push r19
	push r21
	push XL
	push XH			;16 Takte bis hier
	rcall uart_check	;nochmals 27 Takte

	adiw zeilenzaehlerL, 1
	tst syncstatus
	brne t1_L2
	cpi zeilenzaehlerL, 5	;Test auf 6. Zeile (5 weil Zaehlung bei 0 beginnend)
	cpc zeilenzaehlerH, zero
	breq t1_L4
	ldi r16, low(Z312-2)
	ldi r17, high(Z312-2)	;Test auf 311. Zeile (310 weil Zaehlung bei 0 beginnend)
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, r17
	breq t1_L3
	ldi r16, low(Z312)
	ldi r17, high(Z312)	;Test auf 313. Zeile
	add r16, halbbildflag
	adc r17, zero		;Oder 626. Zeile beim zweiten Halbbild (626-312=314)
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, r17
	breq t1_neues_halbbild
	rcall zeile_darstellen
	rjmp t1_ret
t1_neues_halbbild:
	ldi zeilenzaehlerL, 0
	ldi zeilenzaehlerH, 0
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi fontzeile, 0
	ldi r16, 1
	eor halbbildflag, r16	;welches Halbbild?
	breq t1_L1		;zweites-->
	inc vsyncflag		;erstes: VSYNC erst 32us spaeter starten
#if F_CPU==20000000
	ldi r16, SYNC1-71	;90 - Takte seit Synchronisationspuls gestartet
	rcall tickswait
#else
	nop			;72 - Takte seit Synchronisationspuls gestartet = 72 - 71 = 1
#endif
	sbi PORTC, 0		;Synchronisationspuls ende
t1_L1:
.if PIXELPROTEXTZEILE==20
	ldi fontzeileflag, 2
.endif
#ifdef DEBUG2
	sbi LED_PORT, LED_ROT	;Rote LED ein zur Markierung vom Start des VSYNC
#endif
	ldi r16, 2		;Statt hier 140 usec zu vertroedeln, 2 Interrupts abwarten und nur noch
	mov syncstatus, r16	;beim 3. mal die restlichen 140-2*64=12 usec abwarten
	rjmp t1_ret
t1_L2:	dec syncstatus
	brne t1_ret
	tst vsyncflag		;wurde VSYNC verzoegert gestartet?
	brne t1_L2a		;ja-->
	ldi r16, 12-1		;12 - ca. Anzahl schon verbrauchte Microsekunden (oder 17 fuer 145us gesamt)
	rcall usecwait		;2*64+12=140 Microsekunden fuer Seitensynchronisation
	sbi PORTC, 0		;Synchronisationspuls ende
#ifdef DEBUG2
	cbi LED_PORT, LED_ROT	;Rote LED aus zur Markierung vom Ende des VSYNC
#endif
	rjmp t1_ret
t1_L2a:	inc vsyncflag		;vsyncflag auf 2 setzen zum dann im match_B das VSYNC zu beenden
	rjmp t1_ret
t1_L3:	ldi r18, low(ZwischenpulsCompwert)  ;ab 311.Zeile Zusatzpulse in Zeilenmitte
	ldi r17, high(ZwischenpulsCompwert)
	inc mitteflag
#ifdef DEBUG2
	sbi LED_PORT, LED_GRUEN	;Gruene LED ein zur Markierung der Zusatzpulse
#endif
#if F_CPU==20000000
	ldi r16, SYNC1-60	;90 - Takte seit Synchronisationspuls-Start bis -Ende
	rcall tickswait
#else
	rcall tickswait12	;72 - Takte seit Synchronisationspuls-Start bis -Ende = 72 - 60 = 12
#endif
	rjmp t1_L5
t1_L4:	ldi r18, low(InverseoffCompwert)  ;ab 6.Zeile keine Zusatzpulse mehr
	ldi r17, high(InverseoffCompwert)
	clr mitteflag
#ifdef DEBUG2
	cbi LED_PORT, LED_GRUEN	;Gruene LED aus zur Markierung der Zusatzpulse ende
#endif
	ldi r16, SYNC1-53	;72 - Takte seit Synchronisationspuls-Start bis -Ende
	rcall tickswait
t1_L5:
	sbi PORTC, 0		;Synchronisationspuls ende
#ifdef ATMEGA8
	out OCR1BH, r17
	out OCR1BL, r18
#else
	sts OCR1BH, r17
	sts OCR1BL, r18
#endif
t1_ret:
	pop XH
	pop XL
	pop r21
	pop r19
	pop r18
	pop r17
	pop r16
	out SREG, r4
	reti

;; Im Interlace-Modus:
;; Erstes Halbbild: ab 311. Zeile jeweils in der Mitte ein HSYNC-Puls, bei 313. Zeile in der Mitte Beginn des VSYNC,
;; innerhalb des VSYNC inverse HSYNC-Pulse in 32us-Abstand (fallende Flanke jeweils in der Mitte). Laenge des VSYNC: 140us
;; Ab Zeile 318 wieder ohne Pulse in der Mitte. Zeilen innerhalb des VSYNC werden mitgezaehlt.
;; Zweites Halbbild: ab 623. Zeile jeweils in der Mitte ein HSYNC-Puls, am Anfang von 1. Zeile Beginn des VSYNC,
;; innerhalb des VSYNC wieder inverse HSYNC-Pulse. Ab Zeile 6 wieder normal.
;; Diese Informationen von hier: http://www.ni.com/white-paper/4750/de

;; Eine VSYNC-Impulsfolge sieht also so aus:
;; 28us LOW, 4us HIGH, 28us LOW, 4us HIGH --> 64us
;; 28us LOW, 4us HIGH, 28us LOW, 4us HIGH --> 64us
;; 12us LOW, 16us HIGH, 4us LOW, 32us HIGH --> 64us

timer1_match_B:			;Interrupt-Routine fuer Zwischen-Synchronisations-Pulse
	sbic PORTC, 0		;Ist PC0 auf 0 gesetzt? (skip bit ioreg clear)
	rjmp t2_L2		;nein->
t2_L1:	sbi PORTC, 0		;ja: Inverser Synchronisationspuls starten
	in r4, SREG
	push r16
	push r17

	tst mitteflag		;ist es ein Synchpuls in der Zeilenmitte
	brne t2_L1a		;ja-->
	ldi r16, low(ZwischenpulsCompwert)  ;nein: Ende des Synchpulses wird dann im timer1_match_A gemacht
	ldi r17, high(ZwischenpulsCompwert) ;und naechster Zusatzpuls ist wieder in Zeilenmitte
	inc mitteflag
	rjmp t2_L3
t2_L1a:	ldi r16, SYNC1-11	;80 - Takte seit Interrupt-Start
	rcall tickswait
	cbi PORTC, 0		;Inverser Synchronisationspuls ende
	ldi r16, 2
	cp vsyncflag, r16	;ist vsyncflag==2 ?
	brne t2_L1b		;nein-->
	clr vsyncflag
	ldi r16, 12		;ja: VSYNC nach 12us beenden (140us Gesamtzeit des VSYNC, oder 17 fuer 145us)
	rcall usecwait
	sbi PORTC, 0		;Synchronisationspuls ende
#ifdef DEBUG2
	cbi LED_PORT, LED_ROT	;Rote LED aus zur Markierung vom Ende des VSYNC
#endif
t2_L1b:	ldi r16, low(InversepulseCompwert) ;naechster Puls ist dann am Zeilenende
	ldi r17, high(InversepulseCompwert)
	clr mitteflag
	rjmp t2_L3
t2_L2:				;ja: Synchronisationspuls verzoegert starten
	in r4, SREG
	push r16
	push r17

	tst mitteflag		;ist es ein Syncpuls in der Zeilenmitte
	brne t2_L2a		;ja-->
	ldi r16, low(ZwischenpulsCompwert)  ;nein: Synchpuls wird dann im timer1_match_A gemacht
	ldi r17, high(ZwischenpulsCompwert) ;und naechster Zusatzpuls ist wieder in Zeilenmitte
	inc mitteflag
	rjmp t2_L3
t2_L2a:	ldi r16, SYNC1-11	;80 - Takte seit Interrupt-Start
	rcall tickswait
	cbi PORTC, 0
	tst vsyncflag		;soll VSYNC gestartet werden?
	breq t2_L2b		;nein->
				;ja: dann PC0 auf 0 belassen
#ifdef DEBUG2
	sbi LED_PORT, LED_ROT	;Rote LED ein zur Markierung vom Start des VSYNC
#endif
	ldi r16, low(InversepulseCompwert) ;naechster Puls ist dann am Zeilenende
	ldi r17, high(InversepulseCompwert)
	clr mitteflag
	rjmp t2_L3
t2_L2b:	ldi r16, SYNC1-4	;72 bei 16MHz, 90 bei 20MHz --> 4.5us
	rcall tickswait
	sbi PORTC, 0		;Synchronisationspuls ende
	rjmp t2_ret
t2_L3:
#ifdef ATMEGA8
	out OCR1BH, r17
	out OCR1BL, r16
#else
	sts OCR1BH, r17
	sts OCR1BL, r16
#endif
t2_ret:
	pop r17
	pop r16
	out SREG, r4
	reti

#else

timer1_match_A:			;Interrupt-Routine, Variante ohne INTERLACE
	cbi PORTC, 0		;Synchronisationspuls starten
	in r4, SREG
	push r16
	push r17
	push r18
	push r19
	push r21
	push XL
	push XH			;16 Takte bis hier
	rcall uart_check	;nochmals 27 Takte
	
	tst syncstatus			;44 Takte bis hier
	brne t1_L2
	adiw zeilenzaehlerL, 1
	ldi r16, low(Z312)
	ldi r17, high(Z312)
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, r17		;51 Takte bis hier
	brne t1_L1
#ifdef VSYNC1
	cbi PORTC, 1		;Starte Synchpuls auf separater VSYNC-Leitung auch
#endif
	ldi zeilenzaehlerL, 0
	ldi zeilenzaehlerH, 0
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi fontzeile, 0
.if PIXELPROTEXTZEILE==20
	ldi fontzeileflag, 2
.endif
	ldi r16, 2		;Statt hier 160 usec zu vertroedeln, 2 Interrupts abwarten und nur noch
	mov syncstatus, r16	;beim 3. mal die restlichen 160-2*64=32 usec abwarten
	rjmp t1_ret
t1_L2:	dec syncstatus
	brne t1_ret
	ldi r16, 32-3		;32 - ca. Anzahl schon verbrauchte Microsekunden
	rcall usecwait		;160 Microsekunden fuer Seitensynchronisation
	sbi PORTC, 0		;Synchronisationspuls ende
#ifdef VSYNC1
	sbi PORTC, 1		;separate VSYNC-Leitung auch
#endif
	rjmp t1_ret
t1_L1:
	rcall tickswait8	;Pause damit Verzoegerung gleich lang wie in Interlace-Variante
	rcall zeile_darstellen
t1_ret:
	pop XH
	pop XL
	pop r21
	pop r19
	pop r18
	pop r17
	pop r16
	out SREG, r4
	reti

#endif

;-----------------------------------------------------------------------------
; Hauptprogramm:
main:	ldi r16, low(RAMEND)
        out SPL, r16		; Init Stackpointer L
        ldi r16, high(RAMEND)
        out SPH, r16		; Init Stackpointer H
	clr zero		; r5 global mit 0 belegen
	ldi r16, (1<<BILDBIT)	; PB3 beim Atmega8, oder PB0 beim Atmega1284P
	out DDRB, r16		; Ausgang fuer Bildinhalt
#ifdef VSYNC1
	ldi r16, (3<<SYNCBIT)	; PC0|PC1
#else
	ldi r16, (1<<SYNCBIT)	; PC0
#endif
	out DDRC, r16		; Ausgang fuer Synchronisations-Pulse
	out PORTC, r16		; Ausgang gesetzt fuer inaktiv
#ifdef DEBUG2
	ldi r16, (1<<LED_ROT)|(1<<LED_GRUEN)|(1<<LED_GELB)
	out LED_DDR, r16	; Ausgaenge fuer Test-LEDs
	out LED_PORT, r16	; Test: alle LEDs ein
#endif

#ifdef TICKSTESTS
test1:	cbi PORTC, 0
	ldi r16, 18
	rcall tickswait
	sbi PORTC, 0
	ldi r16, 19		;390.244 kHz
;	ldi r16, 20		;380.952
;	ldi r16, 21		;372.093
;	ldi r16, 22		;363.636
	rcall tickswait
	rjmp test1		;gesamte Schlaufe sollte 18+19+4=41 Takte brauchen, sollten also 16MHz/41=390.244kHz ergeben
#endif

#ifdef SPITEST1
	ldi r16, (1<<MOSI_BIT) | (1<<SCK_BIT) | (1<<SS_BIT)
	out DDR_SPI, r16	;MOSI, SCK, und SS auf Ausgang setzen
	ldi r16, (1<<SS_BIT)
	out PORT_SPI, r16	;SS auf 1 setzen, MOSI auf 0 setzen
	ldi r16, (1<<SPI2X)	;Doppelte Geschwindigkeit
	out SPSR, r16
	ldi r16, (1<<SPE) | (1<<MSTR) | (0<<CPOL)  |  (0<<CPHA) | (1<<DORD)   |    (0<<SPR1) | (0<<SPR0)
;; 	         SPI ein,   SPI-Master, SCK=Lowactiv, SCK-Phase,  Schieberichtung, Geschwindigkeit
;; Einstellbare Geschwindigkeiten:
; SPR1-SPR0: 00=f/4, 01=f/16, 10=f/64, 11=f/128 (f=F_CPU oder wenn SPI2X gesetzt f=2*F_CPU)
	mov r20, r16
	mov r21, r20
	andi r21, ~(1<<SPE)
	clr r5
test1:
	ldi r19, 0x3C		;0x10 = einzelnes Bit gesetzt, 0x55 = jedes 2. Bit gesetzt
	out SPCR, r20		; SPI einschalten
	ldi r16, 1
	rcall usecwait	
	out SPDR, r19		; SPI mit neuem Wert starten
test2:
	in r16, SPSR
	andi r16, (1<<SPIF)	;auf Ende der Uebertragung warten
	breq test2
	out SPDR, r5
	nop
	out SPCR, r21		; SPI ausschalten
	cbi PORT_SPI, MOSI_BIT
	ldi r16, 5
	rcall usecwait
	rjmp test1
#endif

;; Beispieltexte kopieren:
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi ZL, low(starttext<<1)
	ldi ZH, high(starttext<<1)
	ldi r24, low(ANZAHLTEXTBYTES)
	ldi r25, high(ANZAHLTEXTBYTES)
	ldi r18, ZEICHENPROZEILE
	rcall textkopieren
	rjmp st2
st1:	cpi r16, LF
	brne st1a
	rcall auffuellen
	rjmp textkopieren
st1a:	st Y+, r16
	sbiw r25:r24, 1
	dec r18
	brne textkopieren
	ldi r18, ZEICHENPROZEILE
textkopieren:
	lpm r16, Z+
	tst r16
	brne st1
	ret
auffuellen:	
	ldi r16,' '
	st Y+, r16		;aktuelle Zeile mit Leerzeichen auffuellen
	sbiw r25:r24, 1
	dec r18
	brne auffuellen
	ldi r18, ZEICHENPROZEILE
	ret
st2:	ldi ZL, low(beispieltext<<1)
	ldi ZH, high(beispieltext<<1)
	rcall textkopieren
	ldi ZL, low(beispieltext2<<1)
	ldi ZH, high(beispieltext2<<1)
	rcall textkopieren
	ldi r16, ' '		;Restlicher Textspeicher mit Zeichensatz auffuellen
st3:	st Y+, r16
	inc r16
	sbiw r25:r24, 1
	brne st3
;; Ende Beispieltexte

#ifdef MITSPI
	rcall spi_init
#endif
	rcall uart_init
	rcall timer1_init
	sei			; Interrupts einschalten

;; Initialisierungen fuers Hauptprogramm:
	ldi XL, low(textspeicher)
	ldi XH, high(textspeicher)
	ldi status, 0		;r19=status
	
mainloop:
	lds r17, jwrite
	lds r18, jread
	cp r18, r17		;neues Zeichen von UART-Empfang vorhanden?
	breq ma_sleep		;nein-->
	push XL
	push XH
	ldi XL, low(serbuffer)
	ldi XH, high(serbuffer)
	add XL, r18
	adc XH, zero
	ld r16, X		;Empfangenes Zeichen vom Puffer einlesen
	pop XH
	pop XL
	inc r18
	andi r18, SERBUFFERSIZE-1
	sts jread, r18		;aktualisierter Lesezeiger speichern
	tst status
	brne ma_status_auswerten
	cpi r16, 0x20
	brcs ma_steuerzeichen
ma_L1:	st X+, r16		;empfangenes Zeichen an aktueller Position speichern
	ldi r17, high(textspeicher+ANZAHLTEXTBYTES)
	cpi XL, low(textspeicher+ANZAHLTEXTBYTES)
	cpc XH, r17		;Ende des Textspeichers erreicht?
	brne ma_sleep
ma_newpage:	
	ldi XL, low(textspeicher) ;ja: wieder auf Anfang des Textspeichers
	ldi XH, high(textspeicher)
ma_sleep:
	ldi r16, (1<<SE)
#ifdef ATMEGA8
	out MCUCR, r16
#else
	out SMCR, r16
#endif
	sleep
	rjmp mainloop

ma_screenclear:
	ldi XL, low(textspeicher) ;auf Anfang des Textspeichers
	ldi XH, high(textspeicher)
	ldi r16, ' '
	ldi r20, low(ANZAHLTEXTBYTES)
	ldi r21, high(ANZAHLTEXTBYTES)
ma_L3:	st X+, r16
	subi r20, 1
	sbci r21, 0
	brne ma_L3
	rjmp ma_newpage
ma_status_setzen:
	mov status, r16
	rjmp ma_sleep
ma_steuerzeichen:
	cpi r16, 0x0A		;Sonderzeichen LF?
	breq ma_einezeilerunter
	cpi r16, 0x0D		;Sonderzeichen CR?
	brne ma_s1
	rcall aufzeilenanfang
	rjmp ma_sleep
ma_s1:	cpi r16, 0x0C		;Sonderzeichen Newpage?
	breq ma_newpage
	cpi r16, 5		;Steuerzeichen fuer Hexmodus?
	breq ma_status_setzen
	cpi r16, 4		;Steuerzeichen fuer Seite loeschen?
	breq ma_screenclear
 	cpi r16, 3		;Steuerzeichen Spaltenwahl?
 	breq ma_status_setzen
 	cpi r16, 2		;Steuerzeichen Zeilenauswahl?
 	breq ma_status_setzen
 	cpi r16, 1		;Naechstes Zeichen Code im Steuerzeichen-Bereich?
	brne ma_L1		;nein: alle andern Zeichen als normale Zeichen behandeln
ma_status_auswerten:
	dec status		;war status==1 ?
	breq ma_L1		;ja: status geloescht und Sprung zum Zeichen als normales Zeichen behandeln
	inc status
	cpi status, 2
	breq ma_neuezeile
	cpi status, 3
	breq ma_neuespalte
; 	cpi status, 5		; Hexmodus?  einzig verbleibende Moglichkeit, darum Test nicht wirklich noetig
; 	brne error
ma_hexmodus:
	cpi r16, 5		;Steuerzeichen fuer Hexmodus?
	breq ma_L2		;ja--> Hexmodus wieder ausschalten
	mov r17, r16
	swap r16		;obere 4 Bits nach unten,
	rcall hexdigit		;in eine Zahl ('0'...'F') wandeln
	rcall putzeichen	;und im Textspeicher speichern
	mov r16, r17
	rcall hexdigit		;gleiches fuer untere 4 Bits
	rcall putzeichen
	ldi r16, ' '		;zwischen den 2-stelligen Hexzahlen jeweils eine Leerstelle
	rcall putzeichen
	rjmp ma_sleep
ma_neuespalte:
	rcall aufzeilenanfang
	cpi r16, ZEICHENPROZEILE   ;r16 zu gross?
	brcs ma_n1
	ldi r16, ZEICHENPROZEILE-1 ;ja: Wert auf Anzahl vorhandene Spalten beschraenken
ma_n1:	add XL, r16		;gewuenschte Spalte addieren
	adc XH, zero
ma_L2:	clr status
	rjmp ma_sleep
ma_neuezeile:
	mov r12, XL
	mov r13, XH
	ldi r17, low(textspeicher)
	ldi r18, high(textspeicher)
	sub r12, r17
	sbc r13, r18
	ldi r18, ZEICHENPROZEILE ;aktuelle Spalte berechnen:
	rcall divmodu16u8	 ;r13:r12 / r18 --> r13:r12=aktuelle Zeile  r18=Rest=aktuelle Spalte
	ldi r17, ZEICHENPROZEILE
	cpi r16, ANZAHLTEXTZEILEN   ;r16 zu gross?
	brcs ma_n2
	ldi r16, ANZAHLTEXTZEILEN-1 ;ja: Wert auf Anzahl vorhandene Zeilen beschraenken
ma_n2:	mul r16, r17		;r16*ZEICHENPROZEILE --> r1:r0
	ldi XL, low(textspeicher)
	ldi XH, high(textspeicher)
	add XL, r0
	adc XH, r1
	add XL, r18		;aktuelle Spalte wieder addieren
	adc XH, zero
	rjmp ma_L2
ma_einezeilerunter:
	ldi r18, ZEICHENPROZEILE
	add XL, r18
	adc XH, zero
	ldi r16, low(textspeicher+ANZAHLTEXTBYTES)
	ldi r17, high(textspeicher+ANZAHLTEXTBYTES)
	cp  XL, r16
	cpc XH, r17		;Grenze erreicht?
	brcs ma_e1		;nein-->
	subi XL, low(ANZAHLTEXTBYTES)	;ja: subtrahieren um wieder auf 1. Zeile zu kommen
	sbci XH, high(ANZAHLTEXTBYTES)
ma_e1:	rjmp ma_sleep

aufzeilenanfang:
	mov r12, XL
	mov r13, XH
	ldi r17, low(textspeicher)
	ldi r18, high(textspeicher)
	sub r12, r17
	sbc r13, r18
	ldi r18, ZEICHENPROZEILE ;aktuelle Zeile berechnen:
	rcall divmodu16u8	 ;r13:r12 / r18 --> r13:r12=aktuelle Zeile,  r18=Rest
	sub XL, r18		 ;Rest subtrahieren, somit kommen wir auf Spalte 0
	sbc XH, zero
	ret

hexdigit:			;untere 4 Bits von r16 in eine Hex-Zahl (0...A) wandeln
	push r18
	andi r16, 0x0F
	cpi r16, 10
	brcs he_L1
	ldi r18, 7
	add r16, r18
he_L1:	ldi r18, '0'
	add r16, r18
	pop r18
	ret

putzeichen:
	st X+, r16		;Zeichen an aktueller Position speichern
	ldi r16, high(textspeicher+ANZAHLTEXTBYTES)
	cpi XL, low(textspeicher+ANZAHLTEXTBYTES)
	cpc XH, r16		;Ende des Textspeichers erreicht?
	brne ma_h1
	ldi XL, low(textspeicher) ;ja: wieder auf Anfang des Textspeichers
	ldi XH, high(textspeicher)
ma_h1:	ret
	
;-----------------------------------------------------------------------------
; komplizierte Unterprogramme

#ifdef MITSPI
spi_init:
	ldi r16, (1<<MOSI_BIT) | (1<<SCK_BIT) | (1<<SS_BIT)
	out DDR_SPI, r16	;MOSI, SCK, und SS auf Ausgang setzen
	ldi r16, (1<<SS_BIT)
	out PORT_SPI, r16	;SS auf 1 setzen, MOSI auf 0 setzen
#ifdef SPIDOPPEL
	ldi r16, (1<<SPI2X)	;Doppelte Geschwindigkeit
#else
	ldi r16, (0<<SPI2X)	;einfache Geschwindigkeit
#endif
	out SPSR, r16
	ldi r16, (1<<SPE) | (1<<MSTR) | (0<<CPOL)  |  (1<<CPHA) | (1<<DORD)   |    (0<<SPR1) | (0<<SPR0)
;; 	         SPI ein,   SPI-Master, SCK=Lowactiv, SCK-Phase,  Schieberichtung, Geschwindigkeit
	out SPCR, r16
;; mit diesen Einstellungen gibt es 4 Takte pro Bit oder mit SPI2X gesetzt 2 Takte pro Bit
;; Einstellbare Geschwindigkeiten:
;; SPR1-SPR0: 00=f/4, 01=f/16, 10=f/64, 11=f/128 (f=F_CPU oder wenn SPI2X gesetzt f=2*F_CPU)
	ldi r16, 0
	out SPDR, r16		;Null-Byte abschicken damit der Ausgang (MOSI) auf 0 gesetzt wird
	ret

;; Variante mit Verwendung von SPI:
textdarstellen:			;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	push r0
	push r1
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	add ZL, fontzeile	 ;Zeiger auf entsprechende Zeile
	adc ZH, zero
	movw r3:r2, ZH:ZL	;Z in r3:r2 zwischenspeichern
	ldi r17, 8
L1:	ld r16, Y+		;darzustellendes Zeichen vom RAM lesen
	mul r16, r17		;r1:r0 = r16*8
	add ZL, r0
	adc ZH, r1		;Zeiger auf entsprechendes Zeichen in der Font-Tabelle
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen
	lsr r19			;zurechtschieben damit letztes Bit ein 0 sein wird (abhaengig von der Fonttabelle)
	out SPDR, r19		;SPI Uebertragung starten, sollte 2*8=16 Takte dauern (oder 32 Takte)
	movw ZH:ZL, r3:r2	;Z zurueckholen
	nop
#ifdef SPIDOPPEL	;(ifndef funktioniert nicht)
#else
	adiw r24, 0		; fuer Variante mit 32 Takten
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0		;statt 2 mal nop
#endif
	adiw r24, 0		;statt 2 mal nop
	dec r18
	brne L1			;Schlaufe braucht 18 Takte (nach 17 Takten ist SPI noch nicht fuer neues Zeichen bereit)
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
	ldi r19, 0
	out SPDR, r19		;Null-Byte abschicken damit der Ausgang (MOSI) sicher auf 0 gesetzt wird
	pop r1
	pop r0
	ret

#else

;; Variante ohne SPI:
textdarstellen:			;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	push r0
	push r1
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	add ZL, fontzeile	 ;Zeiger auf entsprechende Zeile
	adc ZH, zero
	ld r16, Y+		;darzustellendes Zeichen vom RAM lesen
	ldi r17, 8
	mul r16, r17		;r1:r0 = r16*8
	movw r3:r2, ZH:ZL	;Z in r3:r2 zwischenspeichern
	add ZL, r0
	adc ZH, r1		;Zeiger auf entsprechendes Zeichen in der Font-Tabelle
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen
L1:
#ifdef ATMEGA8
	rol r19			;zurechtschieben damit 1.Bit auf PB3 zu liegen kommt
#else
	lsr r19			;zurechtschieben damit 1.Bit auf PB0 zu liegen kommt
	lsr r19			;zurechtschieben damit 1.Bit auf PB0 zu liegen kommt
#endif
	out PORTB, r19		;1.Bit auf PB3 ausgeben (oder beim Atmega1284P auf PB0)
	ld r16, Y+		;mit 2 Befehlen vorgezogen (ld braucht 2 Takte)
				;werden 4 Takte pro Bit, also 24 Takte pro Zeichen gebraucht
#if F_CPU==20000000
	nop			;oder bei 20 MHz 5 Takte pro Bit, ergibt auch 0.25 usec pro Bit
#endif
	ror r19
	out PORTB, r19		;2.Bit auf PB3 ausgeben
	mul r16, r17		;mul braucht 2 Takte (r17 ist immer noch gleich 8)
	lsr r19
#if F_CPU==20000000
	nop			;bei 20 MHz 5 Takte pro Bit
#endif
	out PORTB, r19		;3.Bit auf PB3 ausgeben
	movw ZH:ZL, r3:r2	;movw braucht nur 1 Takt
	lsr r19
	add ZL, r0
#if F_CPU==20000000
	nop			;bei 20 MHz 5 Takte pro Bit
#endif
	out PORTB, r19		;4.Bit auf PB3 ausgeben
	adc ZH, r1
	lsr r19
	nop
#if F_CPU==20000000
	nop			;bei 20 MHz 5 Takte pro Bit
#endif
	out PORTB, r19		;5.Bit auf PB3 ausgeben
	lsr r19
	nop
	dec r18
#if F_CPU==20000000
	nop			;bei 20 MHz 5 Takte pro Bit
#endif
	out PORTB, r19		;6.Bit auf PB3 ausgeben (oder beim Atmega1284P auf PB0)
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen, braucht 3 Takte
#if F_CPU==20000000
	nop			;bei 20 MHz 5 Takte pro Bit
#endif
	out PORTB, zero		;PB3 nach letztem Bit wieder auf 0
	brne L1			;springen wenn r18 nicht 0 ist, es werden 4 Takte zwischen den Zeichen gebraucht
	sbiw YH:YL, 1		;zu viel gelesenes Zeichen rueckgaengig machen
	pop r1
	pop r0
	ret

#endif	//Ende ifdef MITSPI else


;; Eine normale Video-Zeile sieht so aus:
;; 4.5us Sync-Puls, 7.5us Schwarzschulter, 0.75us Rand, 48.0us Text, 3.25us leer --> total 64us
;; 72 Takte         120 Takte              12           768 Takte    52 Takte           1024 Takte bei 16MHz
;; 90 Takte         150 Takte              15           960 Takte    65 Takte           1280 Takte bei 20MHz
;;                             27Zeichen * 27Takte/Zeichen = 729 (bei 16MHz)
;;                             42Zeichen * 18Takte/Zeichen = 756 (bei 16MHz)
;;                             30Zeichen * 32Takte/Zeichen = 960 (bei 20MHz)

zeile_darstellen:
.if F_CPU==20000000
	ldi r16, SYNC1-65	;90 - Takte seit Synchronisationspuls gestartet = 90 - 65 = 25
	rcall tickswait
.else
	rcall tickswait7	;72 - Takte seit Synchronisationspuls gestartet = 72 - 65 = 7
.endif
	sbi PORTC, 0		;Synchronisationspuls ende
	ldi r16, obereleerzeilen ;38
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, zero
	brcs zd_ret		;erste 38 Zeilen sind Leerzeilen
	ldi r16, low(obereleerzeilen+INHALTZEILEN)
	ldi r17, high(obereleerzeilen+INHALTZEILEN)
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, r17
	brcc zd_ret		;letzte Zeilen sind Leerzeilen
	cpi fontzeile, 8
	brcc zd_L1
	ldi r18, ZEICHENPROZEILE ;Anzahl darzustellende Zeichen
	movw r7:r6, YH:YL	;14 Takte bis hier
#ifdef VGATEST
	rcall tickswait14	;LEER1-39 = 14 (bei VGA-Synchronisation)
#else
	ldi r16, LEER1-39	;120 - Anzahl Takte von Synchpuls bis zum ersten Bit
	rcall tickswait		;7.5 usec Schwarzschulter
#endif
	rcall textdarstellen	;+25 Takte bis erstes Bit
	cpi fontzeile, 7	;war das die letzte Fontzeile?
.if PIXELPROTEXTZEILE==20
	brne zd_L0		;nein->
	cpi fontzeileflag, 1
.endif
	breq zd_L1		;ja-> erhoehtes Y lassen
zd_L0:	movw YH:YL, r7:r6	;nein: Y wieder auf Anfang der Textzeile
zd_L1:
.if PIXELPROTEXTZEILE==20
	dec fontzeileflag
	brne zd_ret
	ldi fontzeileflag, 2
.endif
	inc fontzeile
	cpi fontzeile, MAXFONTZEILE
	brne zd_ret
	ldi fontzeile, 0
zd_ret:	ret

;-----------------------------------------------------------------------------
; Genaue Wartefunktionen:
; tickswait wartet genau soviele Takte wie in r16 steht, mindestens 18
; usecwait wartet genau soviele Microsekunden wie in r16 steht. 16MHz-Variante
;-----------------------------------------------------------------------------
tickswait:			;r16 = Anzahl zu wartende Takte (inklusive ldi und rcall und ret)
	cpi r16, 18		;Minimale Anzahl Ticks: bei weniger stimmts nicht
	brcs ti_ret
	subi r16, 14
	lsr r16
	brcs ti_L1
ti_L1:	lsr r16
	brcc ti_L3
	adiw r24, 0		;statt 2 mal nop
ti_L2:	nop
ti_L3:	dec r16
	brne ti_L2
ti_ret:	ret

tickswait17: nop
tickswait16: nop
tickswait15: nop
tickswait14: nop
tickswait13: nop
tickswait12: nop
tickswait11: nop
tickswait10: nop
tickswait9:  nop
tickswait8:  nop
tickswait7:  ret		;rcall und ret brauchen zusammen 7 Takte


us_L1:	adiw r24, 0		;statt 2 mal nop
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
usecwait:			;r16 = Anzahl Microsekunden warten (bei 16MHz) (inklusive ldi und rcall und ret)
	adiw r24, 0
	adiw r24, 0
	nop
	dec r16
	brne us_L1
	nop
	ret			;ldi und rcall und ret brauchen zusammen 8 Takte

;-----------------------------------------------------------------------------
; divmodu16u8
;  16Bit vorzeichenlose Zahl dividieren durch 8Bit vorzeichenlosen Teiler
; Eingabe: r13:r12 = Zahl (16Bit), r18 = Teiler (8Bit)
; Ausgabe: r13:r12 = Resultat, r18 = Rest
; Speicherverbrauch: 34 Byte
; Anzahl Takte: 158 - 174
;  Wenn Teiler 0 ist wird das Resultat 0xFFFF sein
;-----------------------------------------------------------------------------
divmodu16u8:
	push r16
	push r17
	mov r16, r18
	clr r18
	ldi r17, 16
di_L1:	lsl r12
	rol r13
	rol r18
	cp r18, r16
	brcs di_L2
	sub r18, r16
	inc r12
di_L2:	dec r17
	brne di_L1
	pop r17
	pop r16
	ret
