;; basi.asm                        letzte Aenderungen: 20.6.2014, 22.1.2017
;* Programm fuer RS232-BAS-Wandler (Pollin-Bausatz oder Nachbau)
;* Controller: ATmega8A oder ATmega1284P mit 16 oder 20 MHz Quarz
;*
;* Schaltschema: basischema.png
;*		 PB3 (MOSI, Pin17) ueber Diode und 330 Ohm auf Chinch-Buchse
;*		 beim Atmega1284P PB0 verwendet (oder fuer MOSI PB5)
;*               PC0 (Pin23) ueber Diode und 1k auf Chinch-Buchse
;*               16MHz Quarz an Pin9 und Pin10, je 22pF auf GND (Atmega1284P 20MHz)
;*               Reset (Pin1 oder Pin9): 47k auf VCC, Reset-Taster gegen GND
;*               RXD (PD0, Pin 2) = Serieller Eingang
;*               (nicht benutzt: TXD (PD1, Pin 3) = Serieller Ausgang)
;*
;* Letzte Aenderungen:
;* 25.11.2012  Version 1.0
;* 24.4.2014   Versuch einer Variante mit doppelter Zeichengroesse
;*  5.5.2014   einige ESC-Sequenzen eingefuegt (Version 1.2)
;* 10.5.2014   Komentare eingefuegt, Option fuer anderen Zeichensatz
;* 11.5.2014   Neues "m8def.inc" ausprobiert: SRAM_START ist jetzt auch drin
;* 26.5.-8.6.  Version 1.3, Invers-Modus, User-Optionen vereinfacht, test fuer BASIRGB
;*  9.6.       Gesamter Zeichensatz auswaehlbar, direkt Fonts von Fonteditor verwendbar
;* 10.6.       Version 1.4, Vereinfachungen: BASIRGB-Test, VGATEST, SPITEST entfernt
;*             (fuer BASIRGB jetzt gesamtes Programm in basirgb.asm)
;*             Macros optimiert, Stoereffekte bei Datenuebertragung minimiert (T-Flag)
;* 16.6.       Unterstuetzung von Piepston mit ASCII-Zeichen 7 eingebaut.
;* 18.6.       Test von Jitterausgleich statt T-Flag zum Stoereffekte verhindern
;* 12.1.2017   Fehlender Sprung bei Steuerzeichen 01 korrigiert: ma_status_setzen verschoben um Sprung zu sparen.
;*
;* Copyright: Opensource, Freeware
;*
;* VERSION.REVISION = "1.5"
#define VERSION 1
#define REVISION 5
;*
;-----------------------------------------------------------------------------
;- Useroptionen:

.equ BAUDRATE=9600	  ;moegliche Werte: 9600, 19200, 38400, 57600, (76800)

.set ANZAHL_ZEILEN  = 13  ;moegliche Werte:      6, 13, 22
.set ZEICHENPROZEILE=27   ;empfohlene Werte: 14-22, 27, 42 (im Inversmodus 26 statt 27)
.set RAND_LINKS=1800	  ;Linker Rand in Nanosekunden (Default 1800)
.set RAND_OBEN=6	  ;Oberer Rand in Zeilen (Default 6)

;#define INVERSMODUS	  ;Schwarze Zeichen auf weissem Hintergrund
#define INTERLACE	  ;fuer Interlace-Modus (ohne ergibt weniger flimmern, mit entspricht dem Standard)

#define MITESCSEQUENZEN	  ;Steuerzeichen mit ESC-Sequenzen (siehe Liesmich.txt)
#define PIEPSOFF	  ;Zeichen 0x07 (Piepston) ignorieren, auskommentieren zum Piepston aktivieren
.equ ZEICHENSATZ=1	  ;0=default, 1=Latin1, 2=Zeichensatz aehnlich LCD von Beckmann, 3=myzeichensatz.i
			  ;Fuer eigene Fonts mit Fonteditor: Fonttyp 3 und Groesse 6x8 verwenden.

#define ATMEGA8 	;mit 16MHz Quarz, (es kann auch ein Atmega8A sein)
;#define ATMEGA328	;mit diesem koennte auch ein 20MHz Quarz verwendet werden (noch nicht getestet)
;#define ATMEGA1284P	;mit 20MHz Quarz
.equ F_CPU=16000000	;16 MHz oder 20 MHz
;.equ F_CPU=20000000	;16 MHz oder 20 MHz

;- Ende Useroptionen
;-----------------------------------------------------------------------------
;- Programmanmerkungen:
; SPI wird bisher nur bei 22 Zeilen a 42 Zeichen gebraucht.
; Mit INVERSMODUS koennte auch bei DREIZEHNZEILEN und VIELEZEILEN1 das SPI gebraucht werden,
; es sind dann aber nur 22 Zeichen pro Zeile moeglich (statt 27).

;; Sachen zur Fehlersuche:
;#define TICKSTESTS	;um das Unterprogramm tickswait zu ueberpruefen
;#define DEBUG1		;zur Fehlersuche
;#define DEBUG2		;Fehlersuche mit zusaetzlich angeschlossenenen LEDs
;#define VSYNC1 1	;VSYNC auf separater Leitung PC1
;#define STRESSTEST	;Test mit Hauptprogramm stark beschaeftigt
;#define STRESS_VIELE_ZEICHEN	;noch staerker beschaeftigt
;#define STRESS_SEITE_LOESCHEN	;und noch jedesmal Screenclear
;#define JITTER_TEST		;einschalten fuer MINMAXTESTs
;#define JITTER_MINMAXTEST	;in der match_A Routine
;#define JITTER_MINMAXTEST2	;in der match_B Routine
;#define VEREINFACHUNG	;Vereinfachungen ausprobieren

#define JITTER_AUSGLEICH	;Jitterausgleich statt T-Flag-Methode verwenden

#ifdef JITTER_AUSGLEICH
#else
  #define TFLAG_METHODE
#endif
	
.include "ifndeftest.inc"	;//test damit man vor veraltetem avra gewarnt wird

.equ MU=F_CPU/1000000		;MU = Anzahl Takte fuer 1 Microsekunde

#ifdef RAND_LINKS
#else
  .set RAND_LINKS=1800	;Default-Wert
#endif

#ifdef RAND_OBEN
#else
  .set RAND_OBEN=6	;Default-Wert
#endif
	
#if ANZAHL_ZEILEN<=6
  #define SECHSZEILEN	  ;6 Zeilen a 14 bis 22 Zeichen mit sehr grosser Schrift
  #if ZEICHENPROZEILE>22
    .warning "bei 6 Zeilen sind maximal 22 Zeichen/Zeile moeglich - auf 22 gesetzt"
    .set ZEICHENPROZEILE=22
  #endif
  .equ PIXELPROTEXTZEILE=40
  .equ INHALTZEILEN=240
  .equ MAXFONTZEILE=PIXELPROTEXTZEILE/4
#elif ANZAHL_ZEILEN<=13
  #ifdef INVERSMODUS
	#if ZEICHENPROZEILE!=26
	.warning "bei 13 Zeilen im INVERSMODUS gehen nur 26 Zeiche/Zeile - so gesetzt"
	.set ZEICHENPROZEILE=26
	#endif
	#if RAND_LINKS>1800
	.warning "bei 13 Zeilen im INVERSMODUS kann RAND_LINKS maximal 1800 sein - so gesetzt"
	.set RAND_LINKS=1800
	#endif
  #endif
  #if ZEICHENPROZEILE>27
	.warning "bei 13 Zeilen sind maximal 27 Zeichen/Zeile moeglich - auf 27 gesetzt"
	.set ZEICHENPROZEILE=27
  #endif
  .equ PIXELPROTEXTZEILE=20
  .equ INHALTZEILEN=260
  .equ MAXFONTZEILE=PIXELPROTEXTZEILE/2
#else ;//elif ANZAHL_ZEILEN>13
  #if ZEICHENPROZEILE<=27
    #define VIELEZEILEN1	  ;22 Zeilen a 27 Zeichen mit breiter Schrift
    #ifdef INVERSMODUS
      #if ZEICHENPROZEILE==27
	.warning "bei 22 Zeilen sind im INVERSMODUS nur 26 oder 42 Zeichen/Zeile moeglich - auf 26 gesetzt"
	.set ZEICHENPROZEILE=26
      #endif
      #if RAND_LINKS>1800
	.warning "bei 22 Zeilen im INVERSMODUS kann RAND_LINKS maximal 1800 sein - so gesetzt"
	.set RAND_LINKS=1800
      #endif
    #else
      .set ZEICHENPROZEILE=27
    #endif
  #else
    #define VIELEZEILEN2	  ;22 Zeilen a 42 Zeichen mit kleiner Schrift
    #define MITSPI
    #define SPIDOPPEL
    #if ZEICHENPROZEILE>42
      .warning "es sind maximal 42 Zeichen/Zeile moeglich - auf 42 gesetzt"
    #endif
    .set ZEICHENPROZEILE=42
  #endif
  .equ PIXELPROTEXTZEILE=12
  .equ INHALTZEILEN=264		;so gibts 22 Zeilen (12*22=264)
  .equ MAXFONTZEILE=PIXELPROTEXTZEILE
#endif

.set RANDLINKS=RAND_LINKS*MU/1000 ;Umrechnung von ns in Ticks
#ifdef INVERSMODUS
  #if RANDLINKS<25
	.warning "Linker Rand muss im Inversmodus mindestens 25 Takte sein - so gesetzt."
	.set RANDLINKS=25
  #endif
#endif
	
;-----------------------------------------------------------------------------
;; Beginn des eigentlichen Codes

#ifdef ATMEGA8
.include "m8def.inc"		;ATmega8
#endif
#ifdef ATMEGA328
.include "m328def.inc"		;ATmega328
#endif
#ifdef ATMEGA1284P
.include "m1284Pdef.inc"	;ATmega1284P
#endif

	rjmp main
.org OC1Aaddr
	rjmp timer1_match_A

#ifdef INTERLACE
.org OC1Baddr
	rjmp timer1_match_B
#endif

;-----------------------------------------------------------------------------
; Globale Registerbelegungen:
; r4=Sicherung von SREG im Interrupt
; r5=0  Global auf 0 gesetzt
; r7:r6 = Zwischenspeicher fuer Y
; r3:r2 = Zwischenspeicher fuer Z
; Y (r29:r28) = Zeiger ins RAM auf den darzustellenden Text
; Z (r31:r30) = Zeiger ins Flash auf die Fonttabelle
.def zero=r5
.def fontzeile=r22
.def fontzeileflag=r23
.def zeilenzaehlerL=r24
.def zeilenzaehlerH=r25
.def syncstatus=r8
#ifdef INTERLACE
.def halbbildflag=r9
.def mitteflag=r10
.def vsyncflag=r11
#endif
#ifdef INVERSMODUS
.def inversmask=r15
#endif
; Nur noch r0-r1, r12-r14, r16-r21 und XH:XL (r27:r26) sind im Hauptprogramm verwendbar

;-----------------------------------------------------------------------------
; Registerbelegung im Hauptprogramm:
; XH:XL (r27:r26) = aktuelle Position im Textpuffer
.def status=r19
#ifdef STRESSTEST
  .def altsec=r14
  #ifdef JITTER_TEST
    .def minwert=r20
    .def maxwert=r21
  #endif
#endif

;-----------------------------------------------------------------------------
; Konstanten:
  .equ SYNC_PORT=PORTC
  .equ SYNC_DDR =DDRC
  .equ SYNCBIT=PC0
  .equ BILD_PORT=PORTB
  .equ BILD_DDR =DDRB
#ifdef ATMEGA8
  .equ BILDBIT=PB3
#else
  .equ BILDBIT=PB0
#endif

#ifdef MITSPI
.equ DDR_SPI=DDRB
.equ PORT_SPI=PORTB
#ifdef ATMEGA8
.equ MOSI_BIT=PB3
.equ SCK_BIT=PB5
.equ SS_BIT=PB2
#endif
#ifdef ATMEGA1284P
.equ MOSI_BIT=PB5
.equ SCK_BIT=PB7
.equ SS_BIT=PB4
#endif
#endif

#ifdef PIEPSOFF
#else
  #define MITPIEPSTON
  .equ SOUND_PORT=PORTC
  .equ SOUND_DDR =DDRC
  .equ SOUND_BIT =PC5
#endif

.equ zeilentotal=312
#ifdef SECHSZEILEN
.equ oberedunkelzeilen=38
#else
.equ oberedunkelzeilen=30
#endif
.equ obereleerzeilen=oberedunkelzeilen+RAND_OBEN
.equ Z312=zeilentotal
.equ RAND_UNTEN=2	;muss kleiner als Z312-(obereleerzeilen+INHALTZEILEN) sein

#ifdef MITSPI
 #ifdef SPIDOPPEL
 .set ZEICHENPROZEILE=42
 #else
 .set ZEICHENPROZEILE=22
 #endif
#endif

#endif

.equ ANZAHLTEXTZEILEN=INHALTZEILEN/PIXELPROTEXTZEILE ;22 oder 13 oder 6
.equ ANZAHLTEXTBYTES=ANZAHLTEXTZEILEN*ZEICHENPROZEILE
.equ untereleerzeilen=zeilentotal-obereleerzeilen-INHALTZEILEN

.equ SERBUFFERSIZE=64		;Fuer rs232-Empfang (nur Werte 2^n verwendbar, max.256)

#ifdef DEBUG2
.equ LED_PORT=PORTD
.equ LED_DDR=DDRD
.equ LED_ROT=PD4
.equ LED_GRUEN=PD3
.equ LED_GELB=PD2
#endif

;-----------------------------------------------------------------------------
; Macros:

.if ZEICHENPROZEILE<=14
  .macro WAIT_IF_XX
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
    #if F_CPU==20000000
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
    #endif
  .endmacro
  .macro WAIT_IF_XX1		;gleiche Zusatzverzoegerung mit 1 Takt weniger
	nop			;und ohne Cy zu veraendern.
	nop
	nop
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=16
  .macro WAIT_IF_XX
	nop			
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
    #if F_CPU==20000000
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
    #endif
  .endmacro
  .macro WAIT_IF_XX1		;gleiche Zusatzverzoegerung mit 1 Takt weniger
	nop
	nop
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=18
  .macro WAIT_IF_XX
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
    #if F_CPU==20000000
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
    #endif
  .endmacro
  .macro WAIT_IF_XX1		;gleiche Zusatzverzoegerung mit 1 Takt weniger
	nop
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=20
  .macro WAIT_IF_XX
	nop
    #if F_CPU==20000000
	adiw r24, 0		;statt 2 mal nop (Problem: Flags werden veraendert)
    #endif
  .endmacro
  .macro WAIT_IF_XX1		;gleiche Zusatzverzoegerung mit 1 Takt weniger
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=22
  .macro WAIT_IF_XX
	nop
    #if F_CPU==20000000
	nop
    #endif
  .endmacro
  .macro WAIT_IF_XX1		;gleiche Zusatzverzoegerung mit 1 Takt weniger
    #if F_CPU==20000000
	nop
    #endif
  .endmacro
.else  ;.elif ZEICHENPROZEILE>=22
  .macro WAIT_IF_XX
    #if F_CPU==20000000
	nop
    #endif
  .endmacro
  .macro WAIT_IF_XX1		;gleiche Zusatzverzoegerung mit 1 Takt weniger
  .endmacro
.endif

.if ZEICHENPROZEILE<=14
  .macro WAIT_IF_XX2		;gleich wie WAIT_IF_XX aber ohne Flags zu veraendern.
	nop
	nop
	nop
	nop
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=16
  .macro WAIT_IF_XX2
	nop			
	nop
	nop
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=18
  .macro WAIT_IF_XX2
	nop
	nop
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=20
  .macro WAIT_IF_XX2
	nop
    #if F_CPU==20000000
	nop
	nop
    #endif
  .endmacro
.elif ZEICHENPROZEILE<=22
  .macro WAIT_IF_XX2
	nop
    #if F_CPU==20000000
	nop
    #endif
  .endmacro
.else  ;.elif ZEICHENPROZEILE>=22
  .macro WAIT_IF_XX2
    #if F_CPU==20000000
	nop
    #endif
  .endmacro
.endif

.macro	ticksw
.endm
.macro	ticksw_i
  #if @0<0
	.warning "negativer Wert in macro ticksw"
  #endif
  #if @0==1
	nop
  #endif
  #if @0==2
	adiw r24,0
  #endif
  #if @0==3
	nop
	adiw r24,0
  #endif
  #if @0==4
	adiw r24,0
	adiw r24,0
  #endif
  #if @0==5
	nop
	adiw r24,0
	adiw r24,0
  #endif
  #if @0==6
	adiw r24,0
	adiw r24,0
	adiw r24,0
  #endif
  #if @0==7
	rcall tickswait7
  #endif
  #if @0==8
	rcall tickswait8
  #endif
  #if @0==9
	rcall tickswait9
  #endif
  #if @0==10
	rcall tickswait10
  #endif
  #if @0==11
	rcall tickswait11
  #endif
  #if @0==12
	rcall tickswait12
  #endif
  #if @0==13
	rcall tickswait13
  #endif
  #if @0==14
	rcall tickswait14
  #endif
  #if @0==15
	rcall tickswait15
  #endif
  #if @0==16
	rcall tickswait16
  #endif
  #if @0==17
	rcall tickswait17
  #endif
  #if @0>=18
	ldi r16, @0
	rcall tickswait
  #endif
.endm


.macro input
.endm
.macro input_8_i
#if @1>=64
	lds @0, @1
#else
	in  @0, @1
#endif
.endm

.macro output
.endm
.macro output_i_8
#if @0>=64
	sts @0, @1
#else
	out @0, @1
#endif
.endm

#ifdef INVERSMODUS
  #ifdef ATMEGA8
  .equ MINA=10
  .equ MAXA=16
  #else
  .equ MINA=11	;dies entspricht der minimalen Anzahl Takte bis reagiert wird
  .equ MAXA=17	;Maximaler Wert
  #endif
#else
  #ifdef ATMEGA8
  .equ MINA=9
  .equ MAXA=15
  #else
  .equ MINA=10	;dies entspricht der minimalen Anzahl Takte bis reagiert wird
  .equ MAXA=16	;Maximaler Wert
  #endif
#endif

#ifdef TFLAG_METHODE
.macro jitterausgleich
.endm
.set JITAKTE=0
#else
.macro jitterausgleich
	push ZL			;2 Takte zum retten von ZL
	input [ZL,TCNT1L]	;1 Takt beim Atmega8, 2 Takte beim Atmega1284P
#ifdef JITTER_MINMAXTEST
	cp ZL, minwert
	brcc ji1
	mov minwert, ZL		;Minimalwert speichern
  ji1:	cp ZL, maxwert
	brcs ji2
	mov maxwert, ZL		;Maximalwert speichern
  ji2:
#endif
	push ZH
	clr  ZH
	cpi ZL, MAXA+1		;TCNT1 doch groesser als Maximaler Zaehlerstand
	brcc L1			;ja--> Absturz verhindern
	subi ZL, low(-(L1-MINA))  ;16-Bit-Addition von (L1-Minimum)
	sbci ZH, high(-(L1-MINA)) ;also Z = L1 + Zaehlerstand-Minimum
	ijmp
	nop
	nop
	nop
	nop	;zusaetzliche nop falls TCNT1 doch kleiner als MINA ist
L1:	nop	;bis zu 6 Takte fuer Jitterausgleich
	nop
	nop
	nop
	nop
	nop
	pop ZH
	pop ZL
	; 17 bis 23 Takte gebraucht (1 weniger mit Atmega8, 6 mehr mit MINMAXTEST)
	;; hier sollte es jitterfrei sein.
.endm
#endif ;//TFLAG_METHODE else

#ifdef JITTER_MINMAXTEST
.set JITAKTE=23
#else
.set JITAKTE=17
#endif
#ifdef ATMEGA8
.set JITAKTE=JITAKTE-1
#endif

#ifdef INVERSMODUS
  #ifdef ATMEGA8
  .equ MINB=11
  .equ MAXB=17
  #else
  .equ MINB=12	;dies entspricht der minimalen Anzahl Takte bis reagiert wird
  .equ MAXB=18	;Maximaler Wert
  #endif
#else
  #ifdef ATMEGA8
  .equ MINB=10
  .equ MAXB=16
  #else
  .equ MINB=11	;dies entspricht der minimalen Anzahl Takte bis reagiert wird
  .equ MAXB=17	;Maximaler Wert
  #endif
#endif

#ifdef TFLAG_METHODE
.macro jitterausgleichB
.endm
.set JITAKTEB=0
#else
.macro jitterausgleichB
	push ZL			;2 Takte zum retten von ZL
	input [ZL,TCNT1L]	;1 Takt beim Atmega8, 2 Takte beim Atmega1284P
	push ZH
	input [ZH,OCR1BL]	;1 Takt beim Atmega8, 2 Takte beim Atmega1284P
	sub ZL, ZH
	clr ZH
#ifdef JITTER_MINMAXTEST2
	cp ZL, minwert
	brcc ji1
	mov minwert, ZL		;Minimalwert speichern
  ji1:	cp ZL, maxwert
	brcs ji2
	mov maxwert, ZL		;Maximalwert speichern
  ji2:
#endif
	cpi ZL, MAXB+1	 	;TCNT1 doch groesser als Maximum?
	brcc L1			;ja--> Absturz verhindern
	subi ZL, low(-(L1-MINB))  ;16-Bit-Addition von (L1-Minimum)
	sbci ZH, high(-(L1-MINB)) ;also Z = L1 + Zaehlerstand-Minimum
	ijmp
	nop
	nop
	nop
	nop	;zusaetzliche nop falls TCNT1 doch kleiner als MINB ist
L1:	nop	;bis zu 6 Takte fuer Jitterausgleich
	nop
	nop
	nop
	nop
	nop
	pop ZH
	pop ZL
	; 20 bis 26 Takte gebraucht (2 weniger mit Atmega8, 6 mehr mit MINMAXTEST2)
	;; hier sollte es jitterfrei sein.
.endm

#ifdef JITTER_MINMAXTEST2
.set JITAKTEB=26
#else
.set JITAKTEB=20
#endif
#ifdef ATMEGA8
.set JITAKTEB=JITAKTEB-2
#endif
#endif ;//TFLAG_METHODE else

;-----------------------------------------------------------------------------
; Variablen im SRAM:
                .DSEG
serbuffer:	.BYTE SERBUFFERSIZE
jread:		.BYTE 1			; wenn jread==jwrite dann ist der Buffer leer
jwrite:		.BYTE 1			; wenn jread==jwrite+1 dann ist der Buffer voll
textspeicher:	.BYTE ANZAHLTEXTBYTES
uhr_ticks:	.BYTE 1		; 50stels Sekunden (Wertebereich 0...49)
uhr_sec:	.BYTE 1		; Zeit seit dem Einschalten
uhr_min:	.BYTE 1
uhr_std:	.BYTE 1
uhr_tage:	.BYTE 1
pieps_endzeit:	.BYTE 1		; Endzeit+1 fuer Piepser in Ticks (+1 fuer Wertebereich 1...50)
		.CSEG

;-----------------------------------------------------------------------------
; RS232-Teil:
;.equ UBRR_VAL = (F_CPU+BAUDRATE*8)/(BAUDRATE*16)-1  ;clever runden
.equ UBRR_VAL = (F_CPU+BAUDRATE*4)/(BAUDRATE*8)-1  ;clever runden, Wert fur mit Doublespeed
.equ BAUD_REAL  = (F_CPU/(8*(UBRR_VAL+1)))     	   ; Reale Baudrate
.equ BAUD_ERROR = ((BAUD_REAL*1000)/BAUDRATE-1000) ; Fehler in Promille

.if ((BAUD_ERROR>10) || (BAUD_ERROR<-10))       ; max. +/-10 Promille Fehler
  .warning "Systematischer Fehler der Baudrate groesser 1%"
.endif 

.if ((BAUD_ERROR>20) || (BAUD_ERROR<-20))       ; max. +/-20 Promille Fehler
  .error "Systematischer Fehler der Baudrate groesser 2% und damit zu hoch!"
.endif 

#ifdef ATMEGA1284P
.equ UCSRA=UCSR0A
.equ UCSRB=UCSR0B
.equ UCSRC=UCSR0C
.equ UDR=UDR0
.equ U2X=U2X0
.equ RXEN=RXEN0
.equ TXEN=TXEN0
.equ UCSZ1=UCSZ01
.equ UCSZ0=UCSZ00
.equ RXC=RXC0
.equ UBRRH=UBRR0H
.equ UBRRL=UBRR0L
#endif

uart_init:			; Initialisierung fuer Asynchron 8N1
	clr r16
	sts jread, r16
	sts jwrite, r16
	ldi r16, LOW(UBRR_VAL)
	ldi r17, HIGH(UBRR_VAL)
	ldi r18, (1<<U2X)		;fuer Doublespeed
	ldi r19, (1<<RXEN) | (0<<TXEN)	;nur Empfangen, ohne Interrupts
#ifdef ATMEGA8
	out UBRRH, r17
	out UBRRL, r16
	out UCSRA, r18
	out UCSRB, r19
	ldi r16, (1<<URSEL) | (1<<UCSZ1)|(1<<UCSZ0)
	out UCSRC, r16
#else
	sts UBRRH, r17
	sts UBRRL, r16
	sts UCSRA, r18
	sts UCSRB, r19
	ldi r16, (1<<UCSZ1)|(1<<UCSZ0)
	sts UCSRC, r16
#endif
	ret

uart_check:		    ; Pruefen ob ein Zeichen empfangen wurde, und wenn ja im Puffer speichern
			    ; braucht 27 Takte, r16-r18,XH,XL zerstoert
	ldi XL, low(serbuffer)
	ldi XH, high(serbuffer)
	lds r17, jwrite
	lds r18, jread
	add XL, r17
	adc XH, zero
#ifdef ATMEGA8
	in r16, UCSRA
#else
	lds r16, UCSRA
#endif
	andi r16, (1<<RXC)	;neues Zeichen empfangen?
	breq ua_L1		;nein->
#ifdef ATMEGA8
	in r16, UDR		;ja: einlesen
#else
	lds r16, UDR		;ja: einlesen
#endif
	inc r17
	andi r17, SERBUFFERSIZE-1
	cp r17, r18		;ist Puffer voll?
	breq ua_L2		;ja-> Zeichen ignorieren
	st X, r16		;nein: Zeichen im Puffer speichern
	sts jwrite, r17
	ret
ua_L1:
#ifdef ATMEGA8
	nop
#else
	adiw r24, 0		;statt 2 mal nop
#endif
	adiw r24, 0		;statt 2 mal nop
	adiw r24, 0		;statt 2 mal nop
ua_L2:	nop
	adiw r24, 0		;statt 2 mal nop
ua_ret:
	ret

;-----------------------------------------------------------------------------
; Zeichentabelle im FLASH:
font6x8:			;aus 6x8_horizontal_LSB_1.asm (font6x8.png)
#if ZEICHENSATZ==1
.include "isolatin1.i"
#elif ZEICHENSATZ==2
.include "lcdzeichensatz.i"
#elif ZEICHENSATZ==3
.include "myzeichensatz.i"
#else
; Voreingestellter Zeichensatz:
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x00
.db 0x70,0x88,0xD8,0x88,0xA8,0x88,0x70,0x00,	; 0x01
.db 0x70,0xF8,0xA8,0xF8,0x88,0xF8,0x70,0x00,	; 0x02
.db 0x00,0x50,0xF8,0xF8,0xF8,0x70,0x20,0x00,	; 0x03
.db 0x00,0x20,0x70,0xF8,0xF8,0x70,0x20,0x00,	; 0x04
.db 0x20,0x70,0x70,0x20,0xF8,0xF8,0x20,0x00,	; 0x05
.db 0x00,0x20,0x70,0xF8,0xF8,0x20,0x70,0x00,	; 0x06
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x07
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x08
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x09
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x0A
.db 0x00,0xE0,0xC0,0xB0,0x48,0x48,0x30,0x00,	; 0x0B
.db 0x70,0x88,0x88,0x70,0x20,0x70,0x20,0x00,	; 0x0C
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x0D
.db 0xC0,0xB0,0xD0,0xB0,0xD0,0xD8,0x18,0x00,	; 0x0E
.db 0x00,0xA8,0x70,0xD8,0x70,0xA8,0x00,0x00,	; 0x0F
.db 0x10,0x30,0x70,0xF0,0x70,0x30,0x10,0x00,	; 0x10
.db 0x40,0x60,0x70,0x78,0x70,0x60,0x40,0x00,	; 0x11
.db 0x20,0x70,0xF8,0x20,0xF8,0x70,0x20,0x00,	; 0x12
.db 0x50,0x50,0x50,0x50,0x50,0x00,0x50,0x00,	; 0x13
.db 0xF0,0xA8,0xA8,0xB0,0xA0,0xA0,0xA0,0x00,	; 0x14
.db 0x70,0x88,0x30,0x50,0x60,0x88,0x70,0x00,	; 0x15
.db 0x00,0x00,0x00,0x00,0x00,0x78,0x78,0x00,	; 0x16
.db 0x20,0x70,0xF8,0x20,0xF8,0x70,0x20,0x70,	; 0x17
.db 0x20,0x70,0xF8,0x20,0x20,0x20,0x20,0x00,	; 0x18
.db 0x20,0x20,0x20,0x20,0xF8,0x70,0x20,0x00,	; 0x19
.db 0x00,0x20,0x60,0xF8,0x60,0x20,0x00,0x00,	; 0x1A
.db 0x00,0x20,0x30,0xF8,0x30,0x20,0x00,0x00,	; 0x1B
.db 0x00,0x00,0x00,0x08,0x08,0x08,0xF8,0x00,	; 0x1C
.db 0x00,0x50,0x50,0xF8,0x50,0x50,0x00,0x00,	; 0x1D
.db 0x20,0x20,0x70,0x70,0xF8,0xF8,0x00,0x00,	; 0x1E
.db 0xF8,0xF8,0x70,0x70,0x20,0x20,0x00,0x00,	; 0x1F
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x20 ' '
.db 0x20,0x70,0x70,0x20,0x20,0x00,0x20,0x00,	; 0x21 !
.db 0xD8,0xD8,0x48,0x00,0x00,0x00,0x00,0x00,	; 0x22 "
.db 0x00,0x50,0xF8,0x50,0x50,0xF8,0x50,0x00,	; 0x23 #
.db 0x10,0x70,0x08,0x30,0x40,0x38,0x20,0x00,	; 0x24 $
.db 0x98,0x98,0x40,0x20,0x10,0xC8,0xC8,0x00,	; 0x25
.db 0x10,0x28,0x28,0x10,0xA8,0x48,0xB0,0x00,	; 0x26
.db 0x30,0x30,0x10,0x00,0x00,0x00,0x00,0x00,	; 0x27
.db 0x20,0x10,0x10,0x10,0x10,0x10,0x20,0x00,	; 0x28
.db 0x10,0x20,0x20,0x20,0x20,0x20,0x10,0x00,	; 0x29
.db 0x00,0x50,0x70,0xF8,0x70,0x50,0x00,0x00,	; 0x2A
.db 0x00,0x20,0x20,0xF8,0x20,0x20,0x00,0x00,	; 0x2B
.db 0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x10,	; 0x2C
.db 0x00,0x00,0x00,0xF8,0x00,0x00,0x00,0x00,	; 0x2D
.db 0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,	; 0x2E
.db 0x00,0x80,0x40,0x20,0x10,0x08,0x00,0x00,	; 0x2F
.db 0x70,0x88,0xC8,0xA8,0x98,0x88,0x70,0x00,	; 0x30 0
.db 0x20,0x30,0x20,0x20,0x20,0x20,0x70,0x00,	; 0x31 1
.db 0x70,0x88,0x80,0x60,0x10,0x08,0xF8,0x00,	; 0x32 2
.db 0x70,0x88,0x80,0x70,0x80,0x88,0x70,0x00,	; 0x33 3
.db 0x40,0x60,0x50,0x48,0xF8,0x40,0x40,0x00,	; 0x34
.db 0xF8,0x08,0x08,0x78,0x80,0x88,0x70,0x00,	; 0x35
.db 0x60,0x10,0x08,0x78,0x88,0x88,0x70,0x00,	; 0x36
.db 0xF8,0x80,0x40,0x20,0x10,0x10,0x10,0x00,	; 0x37
.db 0x70,0x88,0x88,0x70,0x88,0x88,0x70,0x00,	; 0x38
.db 0x70,0x88,0x88,0xF0,0x80,0x40,0x30,0x00,	; 0x39 9
.db 0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x00,	; 0x3A :
.db 0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x10,	; 0x3B ;
.db 0x40,0x20,0x10,0x08,0x10,0x20,0x40,0x00,	; 0x3C <
.db 0x00,0x00,0xF8,0x00,0x00,0xF8,0x00,0x00,	; 0x3D =
.db 0x10,0x20,0x40,0x80,0x40,0x20,0x10,0x00,	; 0x3E >
.db 0x70,0x88,0x80,0x60,0x20,0x00,0x20,0x00,	; 0x3F ?
.db 0x70,0x88,0xE8,0xA8,0xE8,0x08,0x70,0x00,	; 0x40 @
.db 0x70,0x88,0x88,0x88,0xF8,0x88,0x88,0x00,	; 0x41 A
.db 0x78,0x88,0x88,0x78,0x88,0x88,0x78,0x00,	; 0x42 B
.db 0x70,0x88,0x08,0x08,0x08,0x88,0x70,0x00,	; 0x43 C
.db 0x78,0x88,0x88,0x88,0x88,0x88,0x78,0x00,	; 0x44
.db 0xF8,0x08,0x08,0x78,0x08,0x08,0xF8,0x00,	; 0x45
.db 0xF8,0x08,0x08,0x78,0x08,0x08,0x08,0x00,	; 0x46
.db 0x70,0x88,0x08,0xE8,0x88,0x88,0xF0,0x00,	; 0x47
.db 0x88,0x88,0x88,0xF8,0x88,0x88,0x88,0x00,	; 0x48
.db 0x70,0x20,0x20,0x20,0x20,0x20,0x70,0x00,	; 0x49
.db 0x80,0x80,0x80,0x80,0x88,0x88,0x70,0x00,	; 0x4A
.db 0x88,0x48,0x28,0x18,0x28,0x48,0x88,0x00,	; 0x4B
.db 0x08,0x08,0x08,0x08,0x08,0x08,0xF8,0x00,	; 0x4C
.db 0x88,0xD8,0xA8,0x88,0x88,0x88,0x88,0x00,	; 0x4D
.db 0x88,0x98,0xA8,0xC8,0x88,0x88,0x88,0x00,	; 0x4E
.db 0x70,0x88,0x88,0x88,0x88,0x88,0x70,0x00,	; 0x4F
.db 0x78,0x88,0x88,0x78,0x08,0x08,0x08,0x00,	; 0x50
.db 0x70,0x88,0x88,0x88,0xA8,0x48,0xB0,0x00,	; 0x51
.db 0x78,0x88,0x88,0x78,0x48,0x88,0x88,0x00,	; 0x52
.db 0x70,0x88,0x08,0x70,0x80,0x88,0x70,0x00,	; 0x53
.db 0xF8,0x20,0x20,0x20,0x20,0x20,0x20,0x00,	; 0x54
.db 0x88,0x88,0x88,0x88,0x88,0x88,0x70,0x00,	; 0x55
.db 0x88,0x88,0x88,0x88,0x88,0x50,0x20,0x00,	; 0x56
.db 0x88,0x88,0xA8,0xA8,0xA8,0xA8,0x50,0x00,	; 0x57
.db 0x88,0x88,0x50,0x20,0x50,0x88,0x88,0x00,	; 0x58 X
.db 0x88,0x88,0x88,0x50,0x20,0x20,0x20,0x00,	; 0x59 Y
.db 0x78,0x40,0x20,0x10,0x08,0x08,0x78,0x00,	; 0x5A Z
.db 0x70,0x10,0x10,0x10,0x10,0x10,0x70,0x00,	; 0x5B
.db 0x00,0x08,0x10,0x20,0x40,0x80,0x00,0x00,	; 0x5C
.db 0x70,0x40,0x40,0x40,0x40,0x40,0x70,0x00,	; 0x5D
.db 0x20,0x50,0x88,0x00,0x00,0x00,0x00,0x00,	; 0x5E
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFC,	; 0x5F _
.db 0x30,0x30,0x20,0x00,0x00,0x00,0x00,0x00,	; 0x60
.db 0x00,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x61 a
.db 0x08,0x08,0x78,0x88,0x88,0x88,0x78,0x00,	; 0x62 b
.db 0x00,0x00,0x70,0x88,0x08,0x88,0x70,0x00,	; 0x63 c
.db 0x80,0x80,0xF0,0x88,0x88,0x88,0xF0,0x00,	; 0x64
.db 0x00,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x65
.db 0x60,0x10,0x10,0x78,0x10,0x10,0x10,0x00,	; 0x66
.db 0x00,0x00,0xF0,0x88,0x88,0xF0,0x80,0x70,	; 0x67
.db 0x08,0x08,0x38,0x48,0x48,0x48,0x48,0x00,	; 0x68
.db 0x20,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x69
.db 0x40,0x00,0x60,0x40,0x40,0x40,0x48,0x30,	; 0x6A
.db 0x08,0x08,0x48,0x28,0x18,0x28,0x48,0x00,	; 0x6B
.db 0x20,0x20,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x6C
.db 0x00,0x00,0x58,0xA8,0xA8,0x88,0x88,0x00,	; 0x6D
.db 0x00,0x00,0x38,0x48,0x48,0x48,0x48,0x00,	; 0x6E
.db 0x00,0x00,0x70,0x88,0x88,0x88,0x70,0x00,	; 0x6F
.db 0x00,0x00,0x78,0x88,0x88,0x88,0x78,0x08,	; 0x70
.db 0x00,0x00,0xF0,0x88,0x88,0x88,0xF0,0x80,	; 0x71
.db 0x00,0x00,0x68,0x90,0x10,0x10,0x38,0x00,	; 0x72
.db 0x00,0x00,0x70,0x08,0x70,0x80,0x70,0x00,	; 0x73
.db 0x00,0x10,0x78,0x10,0x10,0x50,0x20,0x00,	; 0x74
.db 0x00,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x75
.db 0x00,0x00,0x88,0x88,0x88,0x50,0x20,0x00,	; 0x76
.db 0x00,0x00,0x88,0x88,0xA8,0xF8,0x50,0x00,	; 0x77
.db 0x00,0x00,0x48,0x48,0x30,0x48,0x48,0x00,	; 0x78
.db 0x00,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0x79
.db 0x00,0x00,0x78,0x40,0x30,0x08,0x78,0x00,	; 0x7A z
.db 0x60,0x10,0x10,0x18,0x10,0x10,0x60,0x00,	; 0x7B {
.db 0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,	; 0x7C |
.db 0x30,0x40,0x40,0xC0,0x40,0x40,0x30,0x00,	; 0x7D }
.db 0x50,0x28,0x00,0x00,0x00,0x00,0x00,0x00,	; 0x7E ~ (Tilde)
.db 0x20,0x70,0xD8,0x88,0x88,0xF8,0x00,0x00,	; 0x7F
.db 0x70,0x88,0x08,0x08,0x88,0x70,0x20,0x30,	; 0x80
.db 0x48,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x81 &uuml;
.db 0xC0,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x82
.db 0x70,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x83
.db 0x50,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x84 &auml;
.db 0x30,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x85
.db 0x70,0x50,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0x86 &aring;
.db 0x00,0x70,0x88,0x08,0x88,0x70,0x20,0x30,	; 0x87
.db 0x70,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x88
.db 0x50,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x89 &euml;
.db 0x30,0x00,0x70,0x88,0x78,0x08,0x70,0x00,	; 0x8A
.db 0x50,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8B &iuml;
.db 0x70,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8C
.db 0x10,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0x8D
.db 0x50,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0x8E &Auml;
.db 0x70,0x50,0x70,0xD8,0x88,0xF8,0x88,0x00,	; 0x8F &Aring; (Angstroem)
.db 0xC0,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0x90
.db 0x00,0x00,0x78,0xA0,0xF8,0x28,0xF0,0x00,	; 0x91
.db 0xF0,0x28,0x28,0xF8,0x28,0x28,0xE8,0x00,	; 0x92
.db 0x70,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x93
.db 0x50,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x94 &ouml;
.db 0x18,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0x95
.db 0x70,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x96
.db 0x18,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0x97
.db 0x50,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0x98 &yuml;
.db 0x48,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0x99 &Ouml;
.db 0x50,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0x9A &Uuml;
.db 0x00,0x00,0x80,0x70,0x68,0x58,0x38,0x04,	; 0x9B
.db 0x60,0x90,0x10,0x78,0x10,0x90,0xE8,0x00,	; 0x9C
.db 0xF0,0xC8,0xA8,0xA8,0xA8,0x98,0x78,0x00,	; 0x9D
.db 0x00,0x88,0x50,0x20,0x50,0x88,0x00,0x00,	; 0x9E Kreuz
.db 0x40,0xA0,0x20,0x70,0x20,0x20,0x28,0x10,	; 0x9F
.db 0x60,0x00,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0xA0
.db 0x60,0x00,0x20,0x20,0x20,0x20,0x60,0x00,	; 0xA1
.db 0x60,0x00,0x30,0x48,0x48,0x48,0x30,0x00,	; 0xA2
.db 0x60,0x00,0x48,0x48,0x48,0x68,0x50,0x00,	; 0xA3
.db 0x50,0x28,0x00,0x38,0x48,0x48,0x48,0x00,	; 0xA4 ntilde
.db 0x50,0x28,0x00,0x48,0x58,0x68,0x48,0x00,	; 0xA5 Ntilde
.db 0x70,0x80,0xF0,0x88,0xF0,0x00,0xF0,0x00,	; 0xA6
.db 0x30,0x48,0x48,0x48,0x30,0x00,0x78,0x00,	; 0xA7
.db 0x20,0x00,0x20,0x30,0x08,0x88,0x70,0x00,	; 0xA8
.db 0x78,0xA4,0xD4,0xB4,0xD4,0x84,0x78,0x00,	; 0xA9 Kopfstehendes ?
.db 0x00,0x00,0xFC,0x80,0x80,0x00,0x00,0x00,	; 0xAA
.db 0x08,0x48,0x28,0x70,0x88,0x40,0xE0,0x00,	; 0xAB 1/2
.db 0x08,0x48,0x28,0xD0,0xA8,0xE0,0x80,0x00,	; 0xAC 1/4
.db 0x20,0x00,0x20,0x20,0x70,0x70,0x20,0x00,	; 0xAD Kopfstehendes !
.db 0x00,0x00,0x90,0x48,0x90,0x00,0x00,0x00,	; 0xAE <<
.db 0x00,0x00,0x48,0x90,0x48,0x00,0x00,0x00,	; 0xAF >>
.db 0xA8,0x00,0x54,0x00,0xA8,0x00,0x54,0x00,	; 0xB0
.db 0xA8,0x54,0xA8,0x54,0xA8,0x54,0xA8,0x54,	; 0xB1 Schachbrettmuster
.db 0x54,0xFC,0xA8,0xFC,0x54,0xFC,0xA8,0xFC,	; 0xB2
.db 0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,	; 0xB3
.db 0x20,0x20,0x20,0x3C,0x20,0x20,0x20,0x20,	; 0xB4
.db 0x60,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB5
.db 0x70,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB6
.db 0x30,0x00,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xB7
.db 0x78,0x84,0xB4,0x94,0xB4,0x84,0x78,0x00,	; 0xB8
.db 0x28,0x2C,0x20,0x2C,0x28,0x28,0x28,0x28,	; 0xB9
.db 0x28,0x28,0x28,0x28,0x28,0x28,0x28,0x28,	; 0xBA
.db 0x00,0x3C,0x20,0x2C,0x28,0x28,0x28,0x28,	; 0xBB
.db 0x28,0x2C,0x20,0x3C,0x00,0x00,0x00,0x00,	; 0xBC
.db 0x00,0x20,0x70,0x08,0x08,0x70,0x20,0x00,	; 0xBD
.db 0x88,0x50,0x20,0xF8,0x20,0xF8,0x20,0x00,	; 0xBE
.db 0x00,0x00,0x00,0x3C,0x20,0x20,0x20,0x20,	; 0xBF
.db 0x20,0x20,0x20,0xE0,0x00,0x00,0x00,0x00,	; 0xC0
.db 0x20,0x20,0x20,0xFC,0x00,0x00,0x00,0x00,	; 0xC1
.db 0x00,0x00,0x00,0xFC,0x20,0x20,0x20,0x20,	; 0xC2
.db 0x20,0x20,0x20,0xE0,0x20,0x20,0x20,0x20,	; 0xC3
.db 0x00,0x00,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xC4
.db 0x20,0x20,0x20,0xFC,0x20,0x20,0x20,0x20,	; 0xC5
.db 0xA0,0x50,0x70,0x80,0xF0,0x88,0xF0,0x00,	; 0xC6
.db 0xA0,0x50,0x20,0x50,0x88,0xF8,0x88,0x00,	; 0xC7
.db 0x28,0xE8,0x08,0xF8,0x00,0x00,0x00,0x00,	; 0xC8
.db 0x00,0xF8,0x08,0xE8,0x28,0x28,0x28,0x28,	; 0xC9
.db 0x28,0xEC,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xCA
.db 0x00,0xFC,0x00,0xEC,0x28,0x28,0x28,0x28,	; 0xCB
.db 0x28,0xE8,0x08,0xE8,0x28,0x28,0x28,0x28,	; 0xCC
.db 0x00,0xFC,0x00,0xFC,0x00,0x00,0x00,0x00,	; 0xCD
.db 0x28,0xEC,0x00,0xEC,0x28,0x28,0x28,0x28,	; 0xCE
.db 0x88,0x70,0x88,0x88,0x88,0x70,0x88,0x00,	; 0xCF
.db 0x30,0x08,0x10,0x20,0x70,0x48,0x30,0x00,	; 0xD0
.db 0x70,0x90,0x90,0xB8,0x90,0x90,0x70,0x00,	; 0xD1
.db 0x70,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD2
.db 0x50,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD3 &Euml;
.db 0x30,0x00,0xF8,0x08,0x78,0x08,0xF8,0x00,	; 0xD4
.db 0x20,0x20,0x20,0x00,0x00,0x00,0x00,0x00,	; 0xD5
.db 0x60,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD6
.db 0x70,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD7
.db 0x50,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xD8
.db 0x20,0x20,0x20,0x3C,0x00,0x00,0x00,0x00,	; 0xD9
.db 0x00,0x00,0x00,0xE0,0x20,0x20,0x20,0x20,	; 0xDA
.db 0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,	; 0xDB
.db 0x00,0x00,0x00,0x00,0xFC,0xFC,0xFC,0xFC,	; 0xDC
.db 0x20,0x20,0x20,0x00,0x20,0x20,0x20,0x00,	; 0xDD
.db 0x30,0x00,0x70,0x20,0x20,0x20,0x70,0x00,	; 0xDE
.db 0xFC,0xFC,0xFC,0xFC,0x00,0x00,0x00,0x00,	; 0xDF
.db 0x60,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE0
.db 0x00,0x38,0x48,0x38,0x48,0x48,0x38,0x08,	; 0xE1 Beta (Deutsches sz)
.db 0x70,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE2
.db 0x18,0x30,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE3
.db 0x50,0x28,0x00,0x30,0x48,0x48,0x30,0x00,	; 0xE4
.db 0x50,0x28,0x30,0x48,0x48,0x48,0x30,0x00,	; 0xE5
.db 0x00,0x00,0x48,0x48,0x48,0x38,0x08,0x08,	; 0xE6
.db 0x00,0x18,0x08,0x38,0x48,0x38,0x08,0x18,	; 0xE7
.db 0x18,0x08,0x38,0x48,0x48,0x38,0x08,0x18,	; 0xE8
.db 0x60,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xE9
.db 0x70,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xEA
.db 0x18,0x00,0x48,0x48,0x48,0x48,0x30,0x00,	; 0xEB
.db 0x60,0x00,0x48,0x48,0x48,0x70,0x20,0x18,	; 0xEC
.db 0x60,0x00,0x88,0x50,0x20,0x20,0x20,0x00,	; 0xED
.db 0x00,0x70,0x00,0x00,0x00,0x00,0x00,0x00,	; 0xEE
.db 0x30,0x30,0x10,0x00,0x00,0x00,0x00,0x00,	; 0xEF
.db 0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,	; 0xF0 -
.db 0x00,0x20,0x70,0x20,0x00,0x70,0x00,0x00,	; 0xF1 +-
.db 0x00,0x00,0xF8,0x00,0x00,0xF8,0x00,0x00,	; 0xF2
.db 0x0C,0x58,0x2C,0xD0,0xA8,0xE0,0x80,0x00,	; 0xF3
.db 0xF0,0xA8,0xA8,0xB0,0xA0,0xA0,0xA0,0x00,	; 0xF4
.db 0x70,0x88,0x30,0x50,0x60,0x88,0x70,0x00,	; 0xF5
.db 0x00,0x20,0x00,0xF8,0x00,0x20,0x00,0x00,	; 0xF6
.db 0x00,0x00,0x00,0x70,0x60,0x00,0x00,0x00,	; 0xF7
.db 0x30,0x48,0x48,0x30,0x00,0x00,0x00,0x00,	; 0xF8 ^o (Hochgestelltes 0, Gradzeichen)
.db 0x00,0x00,0x00,0x50,0x00,0x00,0x00,0x00,	; 0xF9
.db 0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,	; 0xFA
.db 0x10,0x18,0x10,0x10,0x00,0x00,0x00,0x00,	; 0xFB ^1 (Hochgestelltes 1)
.db 0x38,0x10,0x30,0x18,0x00,0x00,0x00,0x00,	; 0xFC ^3
.db 0x18,0x20,0x10,0x38,0x00,0x00,0x00,0x00,	; 0xFD ^2
.db 0x00,0x00,0x78,0x78,0x78,0x78,0x00,0x00,	; 0xFE
.db 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 	; 0xFF
#endif ;//ZEICHENSATZ

.equ LF=0x0A
#ifdef SECHSZEILEN
starttext:
;	.db "basi.asm 1.3",LF,0
	.db "basi.asm ",'0'+VERSION,'.','0'+REVISION,LF,0
beispieltext:
	.db "Rolf Pfister ",LF
	.db "OpenSource",LF,0
beispieltext2:
	.db "www.rolfp.ch/",LF
#else
starttext:
;	.db "basi.asm Version ","1.3",LF,0
	.db "basi.asm Version ",'0'+VERSION,'.','0'+REVISION,LF,0
beispieltext:
	.db "Autor: Rolf Pfister",LF
	.db "Copyr: Freeware OpenSource",LF,0
beispieltext2:
	.db "www.rolfp.ch/elektronik/#rs232bas",LF
#endif

#if ZEICHENPROZEILE>=15
	.db "Baudrate: "
#else
	.db "Baud: "
#endif
.if BAUDRATE==9600
	.db "9600 ",LF
.endif
.if BAUDRATE==19200
	.db "19200",LF
.endif
.if BAUDRATE==38400
	.db "38400",LF
.endif
.if BAUDRATE==57600
	.db "57600",LF
.endif
.if BAUDRATE==76800
	.db "76800",LF
.endif

#if ANZAHL_ZEILEN>13
	.db "Steuerzeichen: ",LF
	.db " 0x01: Sonderzeichen ",LF
	.db " 0x02: Zeilenauswahl ",LF
	.db " 0x03: Spaltenauswahl",LF
	.db " 0x04: Seite loeschen",LF
	.db " 0x05: Hexmodus ein/aus",LF
  #ifdef PIEPSOFF
	.db " 0x07: ignorieren",LF
  #else
	.db " 0x07: Pieps-Ton ",LF
  #endif
#elif ANZAHL_ZEILEN>=11
	.db " 0x01: Sonderzeichen ",LF
	.db " 0x02/0x03: Zeile/Spalte ",LF
	.db " 0x04: Seite loeschen",LF
	.db " 0x05: Hexmodus ein/aus",LF
  #ifdef PIEPSOFF
	.db " 0x07: ignorieren",LF
  #else
	.db " 0x07: Pieps-Ton ",LF
  #endif
#endif
	.db 0,0

;-----------------------------------------------------------------------------
.equ NormalerCompwert = (F_CPU*64/1000000)-1	 ;64 usec fuer Zeilenlaenge

.equ SYNC1=(MU*47+9)/10	;4.7us in Ticks (ev. aufgerundet)
.equ SYNC2=140		;Anzahl Microsekunden fuer VSYNC-Puls mit Interlace (140-150, mind 137, max 150)
.equ SYNC3=160		;Anzahl Microsekunden fuer VSYNC-Puls ohne Interlace

#ifdef INTERLACE

;.equ ZwischenpulsCompwert = (F_CPU*27/1000000)-1 ;28 usec fuer Inverse Zwischenpulse (32-SYNC1--> 27.3us)
;.equ InversepulseCompwert = (F_CPU*59/1000000)-1 ;60 usec fuer Inverse Hauptpulse    (64-SYNC1--> 59.3us)
;.equ InverseoffCompwert = (F_CPU*64/1000000)+1	 ;unerreichbarer Wert zum Zwischenpulse ausschalten
.equ ZwischenpulsCompwert = MU*32-SYNC1-1 ;32us - SYNC1 --> 27.3us
.equ InversepulseCompwert = MU*64-SYNC1-1 ;64us - SYNC1 --> 59.3us
.equ InverseoffCompwert   = MU*64+1	 ;unerreichbarer Wert zum Zwischenpulse ausschalten

timer1_init:
#ifdef ATMEGA8
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	out TCCR1B, r16
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	out OCR1AH, r17
	out OCR1AL, r16
	ldi r16, low(ZwischenpulsCompwert)
	ldi r17, high(ZwischenpulsCompwert)
	out OCR1BH, r17
	out OCR1BL, r16
	ldi r16, 0		;Startwert des Timers = 0
	out TCNT1H, r16
	out TCNT1L, r16
	ldi r16, (1<<OCIE1A)|(1<<OCIE1B)  ;Timer1 Interrupts bei Vergleichswert
	out TIMSK, r16
#else
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	sts TCCR1B, r16			;beim Atmega1284P muss fuer einige io-Register sts statt out verwendet werden
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	sts OCR1AH, r17
	sts OCR1AL, r16
	ldi r16, low(ZwischenpulsCompwert)
	ldi r17, high(ZwischenpulsCompwert)
	sts OCR1BH, r17
	sts OCR1BL, r16
	ldi r16, 0		;Startwert des Timers = 0
	sts TCNT1H, r16
	sts TCNT1L, r16
	ldi r16, (1<<OCIE1A)|(1<<OCIE1B)  ;Timer1 Interrupts bei Vergleichswert
	sts TIMSK1, r16
#endif
	ldi zeilenzaehlerL, low(Z312-1)
	ldi zeilenzaehlerH, high(Z312-1) ;Starten mit letzter Zeile
	clr syncstatus
	clr halbbildflag
	inc halbbildflag		;Starten mit letzter Zeile vom 2. Halbbild
	clr mitteflag
	inc mitteflag			;naechster Zwischenpuls kommt in Zeilenmitte
	clr vsyncflag
	ret

#else

timer1_init:			;fuer Variante ohne Interlace
#ifdef ATMEGA8
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	out TCCR1B, r16
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	out OCR1AH, r17
	out OCR1AL, r16
	ldi r16, 0		;Startwert des Timers = 0
	out TCNT1H, r16
	out TCNT1L, r16
	ldi r16, (1<<OCIE1A)	;Timer1 Interrupts bei Vergleichswert
	out TIMSK, r16
#else
	ldi r16, (1<<WGM12)|(1<<CS10)	;CTC-OCR1A-Modus, Prescaler=1
	sts TCCR1B, r16			;beim Atmega1284P muss fuer einige io-Register sts statt out verwendet werden
	ldi r16, low(NormalerCompwert)
	ldi r17, high(NormalerCompwert)
	sts OCR1AH, r17
	sts OCR1AL, r16
	ldi r16, 0		;Startwert des Timers = 0
	sts TCNT1H, r16
	sts TCNT1L, r16
	ldi r16, (1<<OCIE1A)	;Timer1 Interrupts bei Vergleichswert
	sts TIMSK1, r16
#endif
	ldi zeilenzaehlerL, low(Z312-1)
	ldi zeilenzaehlerH, high(Z312-1) ;Starten mit letzter Zeile
	clr syncstatus
	ret

#endif

;; Eine normale Video-Zeile sieht so aus:
;; Schwarzschulter   Sync-Puls     Schwarzschulter, 
;; 1.5us (vordere)   4.7us HSync,  5.8us (hintere)   2us Rand links, 48us Text, 2us Rand rechts --> total 64us
;; 24 Takte          75 Takte      93 Takte          32 Takte        768 Takte  32    --> 1024 Takte bei 16MHz
;; 30 Takte          94 Takte      116 Takte         40 Takte        960 Takte  40    --> 1280 Takte bei 20MHz

.if F_CPU==16000000
	.equ VORDERE=24		;Vordere Schwarzschulter
	.equ SCHWARZSCHULTER=192-VORDERE-SYNC1 ;Hintere Schwarzschulter = 12us - Vordere - Sync1
.endif
.if F_CPU==20000000
	.equ VORDERE=30		;Vordere Schwarzschulter
	.equ SCHWARZSCHULTER=240-VORDERE-SYNC1 ;Hintere Schwarzschulter = 12us - Vordere - Sync1
.endif
	.equ LEER1=SCHWARZSCHULTER+RANDLINKS

#ifdef INTERLACE

timer1_match_A:			;Interrupt-Routine, Variante mit INERLACE-Modus
#ifdef INVERSMODUS
	out BILD_PORT, zero	;vordere Schwarzschulter 1.5us
	in r4, SREG
	jitterausgleich
	push r16
	ticksw [VORDERE-3-JITAKTE]
#else
	in r4, SREG
	jitterausgleich
	push r16
#endif
	cbi SYNC_PORT, SYNCBIT	;Synchronisationspuls starten
#ifdef TFLAG_METHODE
	clt-r4			;T-Flag loeschen
#else
	nop
#endif
	nop
	nop
	nop
	push r17
	push r18
	push r19
	push XL
	push XH			;14 Takte bis hier
	rcall uart_check	;nochmals 27 Takte
	pop XH
	pop XL			;45 Takte bis hier

	adiw zeilenzaehlerL, 1
	tst syncstatus		;48 Takte bis hier
	brne t1_L2
	cpi zeilenzaehlerL, 5	;Test auf 6. Zeile (5 weil Zaehlung bei 0 beginnend)
	cpc zeilenzaehlerH, zero
	breq t1_L4
	ldi r16, high(Z312-2)	;Test auf 311. Zeile (310 weil Zaehlung bei 0 beginnend)
	cpi zeilenzaehlerL, low(Z312-2)
	cpc zeilenzaehlerH, r16
	breq t1_L3
	ldi r16, low(Z312)
	ldi r17, high(Z312)	;Test auf 313. Zeile
	add r16, halbbildflag
	adc r17, zero		;Oder 626. Zeile beim zweiten Halbbild (626-312=314)
	cp  zeilenzaehlerL, r16
	cpc zeilenzaehlerH, r17
	breq t1_neues_halbbild
; 	rcall zeile_darstellen
;	rjmp t1_ret
 	rjmp zeile_darstellen	;Optimierung: dann Ruecksprung direkt nach t1_ret
t1_neues_halbbild:
	ldi zeilenzaehlerL, 0
	ldi zeilenzaehlerH, 0
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi fontzeile, 0
	ldi r16, 1
	eor halbbildflag, r16	;welches Halbbild?
	breq t1_L1		;zweites-->
	inc vsyncflag		;erstes: VSYNC erst 32us spaeter starten
	ticksw [SYNC1-71]	;SYNC1 - Takte seit Synchronisationspuls gestartet
	sbi SYNC_PORT, SYNCBIT	;Synchronisationspuls ende
t1_L1:
.if PIXELPROTEXTZEILE>=20
	ldi fontzeileflag, (PIXELPROTEXTZEILE/10)
;	ldi fontzeileflag, 2	;test
.endif
#ifdef DEBUG2
	sbi LED_PORT, LED_ROT	;Rote LED ein zur Markierung vom Start des VSYNC
#endif
	ldi r16, 2		;Statt hier 140 usec zu vertroedeln, 2 Interrupts abwarten und nur noch
	mov syncstatus, r16	;beim 3. mal die restlichen 140-2*64=12 usec abwarten
	rjmp t1_ret_tflag
t1_L2:				;49 Takte bis hier
	dec syncstatus
	brne t1_ret_tflag
	tst vsyncflag		;wurde VSYNC verzoegert gestartet?
	brne t1_L2a		;ja-->
;				;54 Takte bis hier
	rcall uhr_update	;106 Takte bis hier
	ticksw [8*MU-106]	;etwas warten damit genau 8 Microsekunden verbraucht
	ldi r16, SYNC2-2*64-8	;12 - Anzahl schon verbrauchte Microsekunden
	rcall usecwait		;2*64+12=140 Microsekunden fuer Seitensynchronisation
	sbi SYNC_PORT, SYNCBIT	;Synchronisationspuls ende
#ifdef DEBUG2
	cbi LED_PORT, LED_ROT	;Rote LED aus zur Markierung vom Ende des VSYNC
#endif
	rjmp t1_ret_tflag
t1_L2a:	inc vsyncflag		;vsyncflag auf 2 setzen zum dann im match_B das VSYNC zu beenden
	rjmp t1_ret_tflag
t1_L3:	ldi r18, low(ZwischenpulsCompwert)  ;ab 311.Zeile Zusatzpulse in Zeilenmitte
	ldi r17, high(ZwischenpulsCompwert)
	inc mitteflag
#ifdef DEBUG2
	sbi LED_PORT, LED_GRUEN	;Gruene LED ein zur Markierung der Zusatzpulse
#endif
	ticksw [SYNC1-60]
	rjmp t1_L5
t1_L4:	ldi r18, low(InverseoffCompwert)  ;ab 6.Zeile keine Zusatzpulse mehr
	ldi r17, high(InverseoffCompwert)
	clr mitteflag
#ifdef DEBUG2
	cbi LED_PORT, LED_GRUEN	;Gruene LED aus zur Markierung der Zusatzpulse ende
#endif
	ticksw [SYNC1-53]	;72 - Takte seit Synchronisationspuls-Start bis -Ende
t1_L5:
	sbi SYNC_PORT, SYNCBIT	;Synchronisationspuls ende
	output [OCR1BH, r17]
	output [OCR1BL, r18]
t1_ret_tflag:
	pop r19
	pop r18
	pop r17
	pop r16
	out SREG, r4
#ifdef TFLAG_METHODE
	set			;T-Flag setzen
#endif
	reti

;; Im Interlace-Modus:
;; Erstes Halbbild: ab 311. Zeile jeweils in der Mitte ein HSYNC-Puls, bei 313. Zeile in der Mitte Beginn des VSYNC,
;; innerhalb des VSYNC inverse HSYNC-Pulse in 32us-Abstand (fallende Flanke jeweils in der Mitte). Laenge des VSYNC: 140us
;; Ab Zeile 318 wieder ohne Pulse in der Mitte. Zeilen innerhalb des VSYNC werden mitgezaehlt.
;; Zweites Halbbild: ab 623. Zeile jeweils in der Mitte ein HSYNC-Puls, am Anfang von 1. Zeile Beginn des VSYNC,
;; innerhalb des VSYNC wieder inverse HSYNC-Pulse. Ab Zeile 6 wieder normal.
;; Diese Informationen von hier: http://www.ni.com/white-paper/4750/de

;; Eine VSYNC-Impulsfolge sieht also so aus:
;; 28us LOW, 4us HIGH, 28us LOW, 4us HIGH --> 64us
;; 28us LOW, 4us HIGH, 28us LOW, 4us HIGH --> 64us
;; 12us LOW, 16us HIGH, 4us LOW, 32us HIGH --> 64us

timer1_match_B:			;Interrupt-Routine fuer Zwischen-Synchronisations-Pulse
#ifdef INVERSMODUS
	out BILD_PORT, zero	;vordere Schwarzschulter 1.5us
	in r4, SREG		     ;Eigentlich wird hier keine Schwarzschulter gebraucht,
	jitterausgleichB
	push r16		     ;aber wir brauchen das damit die Verzoegerung bis zum
	ticksw [VORDERE-9-JITAKTEB]  ;Synchronpuls gleich ist wie bei timer1_match_A.
	pop r16
#else
	in r4, SREG
	jitterausgleichB
#endif
	sbic SYNC_PORT, SYNCBIT	;Ist PC0 auf 0 gesetzt? (skip bit ioreg clear)
	rjmp t2_L2		;nein->
t2_L1:	sbi SYNC_PORT, SYNCBIT	;ja: Inverser Synchronisationspuls starten
	push r16
	push r17

	tst mitteflag		;ist es ein Synchpuls in der Zeilenmitte
	brne t2_L1a		;ja-->
	ldi r16, low(ZwischenpulsCompwert)  ;nein: Ende des Synchpulses wird dann im timer1_match_A gemacht
	ldi r17, high(ZwischenpulsCompwert) ;und naechster Zusatzpuls ist wieder in Zeilenmitte
	inc mitteflag
	rjmp t2_L3
t2_L1a:	ldi r16, SYNC1-11	;80 - Takte seit Interrupt-Start
	rcall tickswait
	cbi SYNC_PORT, SYNCBIT	;Inverser Synchronisationspuls ende
	ldi r16, 2
	cp vsyncflag, r16	;ist vsyncflag==2 ?
	brne t2_L1b		;nein-->
	clr vsyncflag
;	ldi r16, 12		;ja: VSYNC nach 12us beenden (2*64+12=140us Gesamtzeit des VSYNC)
	rcall uhr_update	;+52 Takte --> 56 Takte seit Inverser Syncpuls ende
	ticksw [8*MU-56]	;Verzoegerung damit genau 8 Microsekunden verbraucht
	ldi r16, SYNC2-2*64-8	;140 - 2*64 - schon gebrauchte Microsekunden
	rcall usecwait
	sbi SYNC_PORT, SYNCBIT	;Synchronisationspuls ende
#ifdef DEBUG2
	cbi LED_PORT, LED_ROT	;Rote LED aus zur Markierung vom Ende des VSYNC
#endif
t2_L1b:	ldi r16, low(InversepulseCompwert) ;naechster Puls ist dann am Zeilenende
	ldi r17, high(InversepulseCompwert)
	clr mitteflag
	rjmp t2_L3
t2_L2:				;ja: Synchronisationspuls verzoegert starten
	push r16
	push r17
	tst mitteflag		;ist es ein Syncpuls in der Zeilenmitte
	brne t2_L2a		;ja-->
	ldi r16, low(ZwischenpulsCompwert)  ;nein: Synchpuls wird dann im timer1_match_A gemacht
	ldi r17, high(ZwischenpulsCompwert) ;und naechster Zusatzpuls ist wieder in Zeilenmitte
	inc mitteflag
	rjmp t2_L3
t2_L2a:	ldi r16, SYNC1-11	;80 - Takte seit Interrupt-Start
	rcall tickswait
	cbi SYNC_PORT, SYNCBIT
	tst vsyncflag		;soll VSYNC gestartet werden?
	breq t2_L2b		;nein->
				;ja: dann PC0 auf 0 belassen
#ifdef DEBUG2
	sbi LED_PORT, LED_ROT	;Rote LED ein zur Markierung vom Start des VSYNC
#endif
	ldi r16, low(InversepulseCompwert) ;naechster Puls ist dann am Zeilenende
	ldi r17, high(InversepulseCompwert)
	clr mitteflag
	rjmp t2_L3
t2_L2b:	ldi r16, SYNC1-4	;75 bei 16MHz, 94 bei 20MHz --> 4.7us
	rcall tickswait
	sbi SYNC_PORT, SYNCBIT	;Synchronisationspuls ende
	rjmp t2_ret
t2_L3:
	output [OCR1BH, r17]
	output [OCR1BL, r16]
t2_ret:
	pop r17
	pop r16
	out SREG, r4
#ifdef TFLAG_METHODE
	set
#endif
	reti
;; Ende Variante mit Interlace

#else ;//INTERLACE

timer1_match_A:			;Interrupt-Routine, Variante ohne INTERLACE
#ifdef INVERSMODUS
	out BILD_PORT, zero	;vordere Schwarzschulter 1.5us
	in r4, SREG
	jitterausgleich
	push r16
	ticksw [VORDERE-8-JITAKTE]
	pop r16
	out SREG, r4
#else
  #ifdef JITTER_AUSGLEICH
	in r4, SREG
	jitterausgleich
	out SREG, r4
  #endif
#endif
	cbi SYNC_PORT, SYNCBIT	;Synchronisationspuls starten
#ifdef TFLAG_METHODE
	clt			;T-Flag loeschen
#else
	nop
#endif
	in r4, SREG
	push r16
	push r17
	push r18
	push r19
	push XL
	push XH			;14 Takte bis hier
	rcall uart_check	;nochmals 27 Takte
	pop XH
	pop XL
	
	tst syncstatus		;46 Takte bis hier
	brne t1_L2
	adiw zeilenzaehlerL, 1
	ldi r16, high(Z312)
	cpi  zeilenzaehlerL, low(Z312)
	cpc zeilenzaehlerH, r16		;52 Takte bis hier
	brne t1_L1
#ifdef VSYNC1
	cbi SYNC_PORT, VSYNC1	;Starte Synchpuls auf separater VSYNC-Leitung auch
#endif
	ldi zeilenzaehlerL, 0
	ldi zeilenzaehlerH, 0
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi fontzeile, 0
.if PIXELPROTEXTZEILE>=20
	ldi fontzeileflag, (PIXELPROTEXTZEILE/10)
;	ldi fontzeileflag, 2	;test
.endif
	ldi r16, 2		;Statt hier 160 usec zu vertroedeln, 2 Interrupts abwarten und nur noch
	mov syncstatus, r16	;beim 3. mal die restlichen 160-2*64=32 usec abwarten
;	rjmp t1_ret_tflag
t1_ret_tflag:
	pop r19
	pop r18
	pop r17
	pop r16
	out SREG, r4
#ifdef TFLAG_METHODE
	set			;T-Flag setzen
#endif
	reti
t1_L2:				;48 Takte bis hier
	dec syncstatus
	brne t1_ret_tflag
	rcall uhr_update	;102 Takte bis hier
	ticksw [8*MU-102]	;Verzoegerung damit genau 8 Microsekunden verbraucht
	ldi r16, SYNC3-2*64-8	;160 - 2*64 - Anzahl schon verbrauchte Microsekunden
	rcall usecwait		;160 Microsekunden fuer Seitensynchronisation (2*64+32=160)
	sbi SYNC_PORT, SYNCBIT	;Synchronisationspuls ende
#ifdef VSYNC1
	sbi SYNC_PORT, VSYNC1	;separate VSYNC-Leitung auch
#endif
	rjmp t1_ret_tflag
t1_L1:			 	;54 Takte bis hier
	rcall tickswait9	;Pause damit Verzoegerung gleich lang wie in Interlace-Variante
;	rcall zeile_darstellen
 	rjmp zeile_darstellen	;Optimierung: dann Ruecksprung direkt nach t1_ret
;; Ende Variante ohne Interlace
	
#endif ;//INTERLACE
	
;-----------------------------------------------------------------------------
; komplizierte Unterprogramme: zeile_darstellen und textdarstellen

;-----------------------------------------------------------------------------
;; zeile_darstellen: dieses Unterprogramm sorgt fuer die Ausgabe einer Zeile
;; und ruft fuer die Zeichenausgabe textdarstellen auf.

;; Eine normale Video-Zeile sieht so aus:
;; Schwarzschulter   Sync-Puls     Schwarzschulter, 
;; 1.5us (vordere)   4.7us HSync,  5.8us (hintere)   2us Rand links, 48us Text, 2us Rand rechts --> total 64us
;; 24 Takte          75 Takte      93 Takte          32 Takte        768 Takte  32    --> 1024 Takte bei 16MHz
;; 30 Takte          94 Takte      116 Takte         40 Takte        960 Takte  40    --> 1280 Takte bei 20MHz

zeile_darstellen:
	ticksw [SYNC1-65]	 ;SYNC1 - Takte seit Synchronisationspuls gestartet:  94-65=29, 76-65=11
	sbi SYNC_PORT, SYNCBIT	 ;Synchronisationspuls ende
	cpi zeilenzaehlerL, obereleerzeilen
	cpc zeilenzaehlerH, zero
	brcs zd_Leerzeile1	;erste Zeilen sind Leerzeilen
	ldi r17, high(obereleerzeilen+INHALTZEILEN)
	cpi zeilenzaehlerL, low(obereleerzeilen+INHALTZEILEN)
	cpc zeilenzaehlerH, r17
	brcc zd_Leerzeile2	;letzte Zeilen sind Leerzeilen
	cpi fontzeile, 8
#ifdef INVERSMODUS
	brcc zd_Leerzeile3	;Zeilen zwischen den Textzeilen sind Leerzeilen
#else
	brcc zd_L1
#endif
	ldi r18, ZEICHENPROZEILE ;Anzahl darzustellende Zeichen
	movw r7:r6, YH:YL	;12 Takte bis hier
#ifdef MITSPI
  #ifdef SPIDOPPEL
  #define SPIZEIT 14
  #else
  #define SPIZEIT 28
  #endif
	ldi r16, SCHWARZSCHULTER-SPIZEIT-12	;(12=bisherige Takte seit Sync)
	rcall tickswait
    #ifdef INVERSMODUS
	ldi r16, 0x80		;nach Schwarzschulter auf weissen Hintergrund schalten
    #else
	ldi r16, 0x00		;nach Schwarzschulter auf schwarzem Hintergrund lassen
    #endif
	sbi SPCR, SPE		;SPI einschalten
	out SPDR, r16		;SPI-Daten senden  ;15 Takte seit Sync bis hier
	ticksw [LEER1-(SCHWARZSCHULTER-SPIZEIT-12)-(15+25)]
	      ;LEER1 -  schon gewartete Takte     - Anzahl Takte von Sync bis zum ersten Bit
#else ;//MITSPI
	ticksw [LEER1-(12+25)]	;LEER1 - Anzahl Takte von Sync bis zum ersten Bit
  #ifdef INVERSMODUS
	out BILD_PORT, inversmask	;PB3 auf 1 setzen fuer weissen Hintergrund
	ticksw [RANDLINKS-25]
  #endif
#endif ;//MITSPI
;	rcall textdarstellen	;+27 Takte bis erstes Bit
	rjmp textdarstellen	;Optimierung2: +25 Takte bis erstes Bit
zd_textret:			;Optimierung2
	cpi fontzeile, 7	;war das die letzte Fontzeile?
.if PIXELPROTEXTZEILE>=20
	brne zd_L0		;nein->
	cpi fontzeileflag, 1
.endif
	breq zd_L1		;ja-> erhoehtes Y lassen
zd_L0:	movw YH:YL, r7:r6	;nein: Y wieder auf Anfang der Textzeile
zd_L1:
.if PIXELPROTEXTZEILE>=20
	dec fontzeileflag
	brne zd_ret
	ldi fontzeileflag, (PIXELPROTEXTZEILE/10)
;	ldi fontzeileflag, 2	;test
.endif
	inc fontzeile
	cpi fontzeile, MAXFONTZEILE
	brne zd_ret
	ldi fontzeile, 0
zd_ret:
;	rjmp t1_ret		;Optimierung
;	ret
t1_ret:
	pop r19
	pop r18
	pop r17
	pop r16
	out SREG, r4
	reti
	
zd_Leerzeile1:
#ifdef INVERSMODUS
	cpi zeilenzaehlerL, oberedunkelzeilen
	cpc zeilenzaehlerH, zero
	brcs zd_ret_tflag		;erste Zeilen sind dunkle Leerzeilen
	ticksw [SCHWARZSCHULTER-7]
	out BILD_PORT, inversmask	;weisser Hintergrund
	rjmp zd_ret_tflag
#endif
zd_Leerzeile2:
#ifdef INVERSMODUS
	ldi r17, high(obereleerzeilen+INHALTZEILEN+RAND_UNTEN)
	cpi zeilenzaehlerL, low(obereleerzeilen+INHALTZEILEN+RAND_UNTEN)
	cpc zeilenzaehlerH, r17
	brcc zd_ret_tflag		;allerletzte Zeilen sind dunkle Leerzeilen
	ticksw [SCHWARZSCHULTER-12]
	out BILD_PORT, inversmask	;weisser Hintergrund
#endif
zd_ret_tflag:			;gleich wie t1_ret_tflag
	pop r19
	pop r18
	pop r17
	pop r16
	out SREG, r4
#ifdef TFLAG_METHODE
	set			;T-Flag setzen
#endif
	reti

#ifdef INVERSMODUS
zd_Leerzeile3:
	ticksw [SCHWARZSCHULTER-10]
	out BILD_PORT, inversmask	;weisser Hintergrund
#ifdef TFLAG_METHODE
	ldi r16, (1<<SREG_T)
	or  r4, r16			;T-Flag setzen
#endif
	rjmp zd_L1
#endif

;-----------------------------------------------------------------------------
;; textdarstellen: dieses Unterprogramm sorgt fuer die Ausgabe der Zeichen
;; es werden 3 Versionen gebraucht: mit SPI, ohne SPI mit INVERSMODUS, und ohne SPI ohne INVERSMODUS

#ifdef MITSPI
spi_init:
	ldi r16, (1<<MOSI_BIT) | (1<<SCK_BIT) | (1<<SS_BIT)
	out DDR_SPI, r16	;MOSI, SCK, und SS auf Ausgang setzen
	ldi r16, (1<<SS_BIT)
	out PORT_SPI, r16	;SS auf 1 setzen, MOSI auf 0 setzen
#ifdef SPIDOPPEL
	ldi r16, (1<<SPI2X)	;Doppelte Geschwindigkeit
#else
	ldi r16, (0<<SPI2X)	;einfache Geschwindigkeit
#endif
	out SPSR, r16
	ldi r16, (1<<SPE) | (1<<MSTR) | (0<<CPOL)  |  (1<<CPHA) | (1<<DORD)   |    (0<<SPR1) | (0<<SPR0)
;; 	         SPI ein,   SPI-Master, SCK=Lowactiv, SCK-Phase,  Schieberichtung, Geschwindigkeit
	out SPCR, r16
;; mit diesen Einstellungen gibt es 4 Takte pro Bit oder mit SPI2X gesetzt 2 Takte pro Bit
;; Einstellbare Geschwindigkeiten:
;; SPR1-SPR0: 00=f/4, 01=f/16, 10=f/64, 11=f/128 (f=F_CPU oder wenn SPI2X gesetzt f=2*F_CPU)
	ldi r16, 0
	out SPDR, r16		;Null-Byte abschicken damit der Ausgang (MOSI) auf 0 gesetzt wird
	ret

;; Variante mit Verwendung von SPI:
textdarstellen:			;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	push r0
	push r1
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	add ZL, fontzeile	 ;Zeiger auf entsprechende Zeile
	adc ZH, zero
	movw r3:r2, ZH:ZL	;Z in r3:r2 zwischenspeichern
	ldi r17, 8
#ifdef INVERSMODUS
	nop		;Justierung damit gleich viele Takte gebraucht wie bei nicht-SPI
#else
	adiw r24, 0	;(statt 2 nop) Justierung damit gleich viele Takte gebraucht wie bei nicht-SPI
#endif
L1:	ld r16, Y+		;darzustellendes Zeichen vom RAM lesen
	mul r16, r17		;r1:r0 = r16*8
	add ZL, r0
	adc ZH, r1		;Zeiger auf entsprechendes Zeichen in der Font-Tabelle
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen
	lsr r19			;zurechtschieben damit letztes Bit ein 0 sein wird (abhaengig von der Fonttabelle)
#ifdef INVERSMODUS
	eor r19, inversmask	;Bits umkehren fuer INVERSMODUS (1 Takt korrigieren)
#else
	nop
#endif
	out SPDR, r19		;SPI Uebertragung starten, sollte 2*8=16 Takte dauern (oder 32 Takte)
	movw ZH:ZL, r3:r2	;Z zurueckholen
#ifdef SPIDOPPEL
	adiw r24, 0	;statt 2 mal nop
#else
	rcall tickswait18	; fuer langsames SPI (32 Takte) hier also 16 Takte mehr
#endif
	dec r18
	brne L1			;Schlaufe braucht 18 Takte damit SPI jeweils fertig ist
	rcall tickswait7
#ifdef INVERSMODUS
	ldi r19, 0xFF		;bei INVERSMODUS nach letztem Zeichen 1-Bits ausgeben
#else
	ldi r19, 0		;sonst 0-Bits ausgeben
#endif
	pop r1
	pop r0
	out BILD_PORT, r19	;Hintergrund-Bit ausgeben
	cbi SPCR, SPE		;SPI ausschalten
;	ret
	rjmp zd_textret		;Optimierung2

#else   ;//MITSPI

;; Variante ohne SPI:
#ifdef INVERSMODUS
textdarstellen:			;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	push r0
	push r1
				;Variante fuer Inversmodus:
	dec r18			;in Schlaufe mit subi statt dec, und auf Cy testen

	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	add ZL, fontzeile	 ;Zeiger auf entsprechende Zeile
	adc ZH, zero
	ld r16, Y+		;darzustellendes Zeichen vom RAM lesen
	ldi r17, 8
	mul r16, r17		;r1:r0 = r16*8
	movw r3:r2, ZH:ZL	;Z in r3:r2 zwischenspeichern
	add ZL, r0
	adc ZH, r1		;Zeiger auf entsprechendes Zeichen in der Font-Tabelle
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen
	eor r19, inversmask	;Bits umkehren (total 2 Takte mehr als ohne INVERSMODUS)
L1:
#ifdef ATMEGA8
	rol r19			;zurechtschieben damit 1.Bit auf PB3 zu liegen kommt
#else
	lsr r19			;zurechtschieben damit 1.Bit auf PB0 zu liegen kommt
	lsr r19			;zurechtschieben damit 1.Bit auf PB0 zu liegen kommt
#endif
	out BILD_PORT, r19	;1.Bit auf PB3 ausgeben (oder beim Atmega1284P auf PB0)
	ld r16, Y+		;mit 2 Befehlen vorgezogen (ld braucht 2 Takte)
				;werden 4 Takte pro Bit, also 24 Takte pro Zeichen gebraucht
	ror r19
	WAIT_IF_XX	;ev. zusaetzliche Wartetakte abhaengig von Zeichenbreite und CPU-Frequenz
	out BILD_PORT, r19	;2.Bit auf PB3 ausgeben
	mul r16, r17		;mul braucht 2 Takte (r17 ist immer noch gleich 8)
	lsr r19
	WAIT_IF_XX
	out BILD_PORT, r19	;3.Bit auf PB3 ausgeben
	movw ZH:ZL, r3:r2	;movw braucht nur 1 Takt
	lsr r19
	WAIT_IF_XX
	add ZL, r0
	out BILD_PORT, r19	;4.Bit auf PB3 ausgeben
	adc ZH, r1
	lsr r19
	nop
	WAIT_IF_XX
	out BILD_PORT, r19	;5.Bit auf PB3 ausgeben
	lsr r19
	nop
	WAIT_IF_XX
	subi r18, 1
	out BILD_PORT, r19	;6.Bit auf PB3 ausgeben (oder beim Atmega1284P auf PB0)
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen, braucht 3 Takte
	eor r19, inversmask	;wie COM aber ohne Cy zu veraendern
				;(bei nicht SECHSZEILEN und 16MHz haben wir 1 Takt zu viel)
	WAIT_IF_XX1	;ev. zusaetzliche Wartetakte, aber 1 Takt weniger
	out BILD_PORT, inversmask	;im Inversmodus PB3 nach letztem Bit auf 1
	brcc L1			;springen wenn r18 >= 0 ist, es werden 4 Takte zwischen den Zeichen gebraucht
	sbiw YH:YL, 1		;zu viel gelesenes Zeichen rueckgaengig machen
	pop r1
	pop r0
;	ret
	rjmp zd_textret		;Optimierung2

#else  ;//INVERSMODUS

;//ohne SPI und ohne INVERSMODUS:
textdarstellen:			;Y=Zeiger auf den Text im RAM, r18=Anzahl Zeichen
	push r0
	push r1
	ldi ZL, low(font6x8<<1)
	ldi ZH, high(font6x8<<1) ;Zeiger auf Font-Tabelle
	add ZL, fontzeile	 ;Zeiger auf entsprechende Zeile
	adc ZH, zero
	ld r16, Y+		;darzustellendes Zeichen vom RAM lesen
	ldi r17, 8
	mul r16, r17		;r1:r0 = r16*8
	movw r3:r2, ZH:ZL	;Z in r3:r2 zwischenspeichern
	add ZL, r0
	adc ZH, r1		;Zeiger auf entsprechendes Zeichen in der Font-Tabelle
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen
	nop		;Justierung damit gleich viele Takte gebraucht wie bei INVERSMODUS
	nop		;Justierung damit gleich viele Takte gebraucht wie bei INVERSMODUS
L1:
#ifdef ATMEGA8
	rol r19			;zurechtschieben damit 1.Bit auf PB3 zu liegen kommt
#else
	lsr r19			;zurechtschieben damit 1.Bit auf PB0 zu liegen kommt
	lsr r19			;zurechtschieben damit 1.Bit auf PB0 zu liegen kommt
#endif
	out BILD_PORT, r19	;1.Bit auf PB3 ausgeben (oder beim Atmega1284P auf PB0)
	ld r16, Y+		;mit 2 Befehlen vorgezogen (ld braucht 2 Takte)
				;werden 4 Takte pro Bit, also 24 Takte pro Zeichen gebraucht
	ror r19
	WAIT_IF_XX	;ev. zusaetzliche Wartetakte abhaengig von Zeichenbreite und CPU-Frequenz
	out BILD_PORT, r19	;2.Bit auf PB3 ausgeben
	mul r16, r17		;mul braucht 2 Takte (r17 ist immer noch gleich 8)
	lsr r19
	WAIT_IF_XX
	out BILD_PORT, r19	;3.Bit auf PB3 ausgeben
	movw ZH:ZL, r3:r2	;movw braucht nur 1 Takt
	lsr r19
	WAIT_IF_XX
	add ZL, r0
	out BILD_PORT, r19	;4.Bit auf PB3 ausgeben
	adc ZH, r1
	lsr r19
	nop
	WAIT_IF_XX
	out BILD_PORT, r19	;5.Bit auf PB3 ausgeben
	lsr r19
	nop
	WAIT_IF_XX
	dec r18
	out BILD_PORT, r19	;6.Bit auf PB3 ausgeben (oder beim Atmega1284P auf PB0)
	lpm r19, Z		;Byte von der Font-Tabelle im Flash einlesen, braucht 3 Takte
	WAIT_IF_XX2
	out BILD_PORT, zero	;PB3 nach letztem Bit wieder auf 0
	brne L1			;springen wenn r18 nicht 0 ist, es werden 4 Takte zwischen den Zeichen gebraucht
	sbiw YH:YL, 1		;zu viel gelesenes Zeichen rueckgaengig machen
	pop r1
	pop r0
;	ret
	rjmp zd_textret		;Optimierung2
#endif  ;//Ende ifdef INVERSMODUS else
#endif	;//Ende ifdef MITSPI else

;-----------------------------------------------------------------------------
; Genaue Wartefunktionen:
; tickswait wartet genau soviele Takte wie in r16 steht, mindestens 18
; usecwait wartet genau soviele Microsekunden wie in r16 steht. 16MHz-Variante
;-----------------------------------------------------------------------------
tickswait:			;r16 = Anzahl zu wartende Takte (inklusive ldi und rcall und ret)
	cpi r16, 18		;Minimale Anzahl Ticks: bei weniger stimmts nicht
	brcs ti_ret
	subi r16, 14
	lsr r16
	brcs ti_L1
ti_L1:	lsr r16
	brcc ti_L3
	adiw r24, 0		;statt 2 mal nop
ti_L2:	nop
ti_L3:	dec r16
	brne ti_L2
ti_ret:	ret

tickswait18: nop
tickswait17: nop
tickswait16: nop
tickswait15: nop
tickswait14: nop
tickswait13: nop
tickswait12: nop
tickswait11: nop
tickswait10: nop
tickswait9:  nop
tickswait8:  nop
tickswait7:  ret		;rcall und ret brauchen zusammen 7 Takte


us_L1:	adiw r24, 0		;statt 2 mal nop
	adiw r24, 0
	adiw r24, 0
	adiw r24, 0
usecwait:			;r16 = Anzahl Microsekunden warten (bei 16MHz) (inklusive ldi und rcall und ret)
	adiw r24, 0
	adiw r24, 0
	nop
	dec r16
	brne us_L1
	nop
	ret			;ldi und rcall und ret brauchen zusammen 8 Takte

;-----------------------------------------------------------------------------
; divmodu16u8
;  16Bit vorzeichenlose Zahl dividieren durch 8Bit vorzeichenlosen Teiler
; Eingabe: r13:r12 = Zahl (16Bit), r18 = Teiler (8Bit)
; Ausgabe: r13:r12 = Resultat, r18 = Rest
; Speicherverbrauch: 34 Byte
; Anzahl Takte: 158 - 174
;  Wenn Teiler 0 ist wird das Resultat 0xFFFF sein
;-----------------------------------------------------------------------------
divmodu16u8:
	push r16
	push r17
	mov r16, r18
	clr r18
	ldi r17, 16
di_L1:	lsl r12
	rol r13
	rol r18
	cp r18, r16
	brcs di_L2
	sub r18, r16
	inc r12
di_L2:	dec r17
	brne di_L1
	pop r17
	pop r16
	ret

;-----------------------------------------------------------------------------
; Hauptprogramm:
main:	ldi r16, low(RAMEND)
        out SPL, r16		; Init Stackpointer L
        ldi r16, high(RAMEND)
        out SPH, r16		; Init Stackpointer H
	clr zero		; r5 global mit 0 belegen
	ldi r16, (1<<BILDBIT)	; PB3 beim Atmega8, oder PB0 beim Atmega1284P
	out BILD_DDR, r16	; Ausgang fuer Bildinhalt
#ifdef INVERSMODUS
	clr inversmask
	dec inversmask		; inversmask = 0xFF
#endif
#ifdef VSYNC1
	ldi r16, (1<<SYNCBIT)|(1<<VSYNC1)	;z.B. PC0|PC1
#else
	ldi r16, (1<<SYNCBIT)	; z.B. PC0
#endif
	out SYNC_DDR, r16	; Ausgang fuer Synchronisations-Pulse
	out SYNC_PORT, r16	; Ausgang gesetzt fuer inaktiv
	rcall uhr_reset
#ifdef MITPIEPSTON
	sbi SOUND_DDR, SOUND_BIT
	rcall piepston_stop	;Piepser aus
#endif

#ifdef DEBUG2
	ldi r16, (1<<LED_ROT)|(1<<LED_GRUEN)|(1<<LED_GELB)
	out LED_DDR, r16	; Ausgaenge fuer Test-LEDs
	out LED_PORT, r16	; Test: alle LEDs ein
#endif

#ifdef TICKSTESTS
	clr r17
	ldi r18, (1<<SYNCBIT)
	out SYNC_DDR, r18	;nur dieses eine Bit auf Ausgang
test1:	out SYNC_PORT, r17
	ldi r16, 18
	rcall tickswait
	out SYNC_PORT, r18
	ldi r16, 19		;390.244 kHz
;	ldi r16, 20		;380.952
;	ldi r16, 21		;372.093
;	ldi r16, 22		;363.636
	rcall tickswait
	rjmp test1		;gesamte Schlaufe sollte 18+19+4=41 Takte brauchen, sollten also 16MHz/41=390.244kHz ergeben
#endif

;; Beispieltexte kopieren:
	ldi YL, low(textspeicher)
	ldi YH, high(textspeicher)
	ldi ZL, low(starttext<<1)
	ldi ZH, high(starttext<<1)
	ldi r24, low(ANZAHLTEXTBYTES)
	ldi r25, high(ANZAHLTEXTBYTES)
	ldi r18, ZEICHENPROZEILE
	rcall textkopieren
	rjmp st2
st1:	cpi r16, LF
	brne st1a
	rcall auffuellen
	rjmp textkopieren
st1a:	st Y+, r16
	sbiw r25:r24, 1
	dec r18
	brne textkopieren
	ldi r18, ZEICHENPROZEILE
textkopieren:
	lpm r16, Z+
	tst r16
	brne st1
	ret
auffuellen:	
	ldi r16,' '
	st Y+, r16		;aktuelle Zeile mit Leerzeichen auffuellen
	sbiw r25:r24, 1
	dec r18
	brne auffuellen
	ldi r18, ZEICHENPROZEILE
	ret
st2:	ldi ZL, low(beispieltext<<1)
	ldi ZH, high(beispieltext<<1)
	rcall textkopieren
	ldi ZL, low(beispieltext2<<1)
	ldi ZH, high(beispieltext2<<1)
	rcall textkopieren
	ldi r16, '!'		;Restlicher Textspeicher mit Zeichensatz auffuellen
st3:	st Y+, r16
	inc r16
	sbiw r25:r24, 1
	brne st3
;; Ende Beispieltexte

#ifdef MITSPI
	rcall spi_init
#endif
	rcall uart_init
	rcall timer1_init
	sei			; Interrupts einschalten

;; Initialisierungen fuers Hauptprogramm:
	ldi XL, low(textspeicher)
	ldi XH, high(textspeicher)
	ldi status, 0		;r19=status

	ldi r16, (1<<SE)	;Schlafmodus IDLE setzen
#ifdef ATMEGA8
	out MCUCR, r16
#else
	out SMCR, r16
#endif

#ifdef STRESSTEST	;Test mit Hauptprogramm stark beschaeftigt
	clr altsec
  #ifdef JITTER_TEST
	clr maxwert
	ldi minwert, 99
	rcall minmax_anzeige	;Startwerte anzeigen
	rjmp mainloop
testanzeige:
	mov altsec, r16
	rcall minmax_anzeige	;alle Sekunden aktualisierte Werte anzeigen
	lds r16, uhr_sec
	andi r16, 7
	brne mainloop
	ldi maxwert, 1
	ldi minwert, 98		;ca. alle 8 Sekunden Werte ruecksetzen
  #endif
mainloop:			;Hauptschlaufe mit Stresstest
  #ifdef TFLAG_METHODE
	sleep
  #endif
  #ifdef JITTER_TEST
;	sleep
;	brtc mainloop
	lds r16, uhr_sec
	cp  r16, altsec
	brne testanzeige
  #else
;	brtc mainloop		
	lds r16, uhr_sec
	cp  r16, altsec
	breq s_L1
	mov altsec, r16
	rcall zeitanzeige
s_L1:
  #endif
#else
mainloop:			;Hauptschlaufe ohne Testoptionen
  #ifdef TFLAG_METHODE
	sleep
	brtc mainloop		;nur was machen wenn T-Flag gesetzt ist
  #else
;	sleep	;mit JITTER_AUSGLEICH ist sleep nicht mehr unbedingt noetig
  #endif
#endif
#ifdef MITPIEPSTON
	rcall piepston_pruefen
#endif
	lds r17, jwrite
	lds r18, jread
	cp r18, r17		;neues Zeichen von UART-Empfang vorhanden?
	breq mainloop		;nein-->
	push XL
	push XH
	ldi XL, low(serbuffer)
	ldi XH, high(serbuffer)
	add XL, r18
	adc XH, zero
	ld r16, X		;Empfangenes Zeichen vom Puffer einlesen
	pop XH
	pop XL
	inc r18
	andi r18, SERBUFFERSIZE-1
	sts jread, r18		;aktualisierter Lesezeiger speichern
	tst status
	brne ma_status_auswerten
	cpi r16, 0x20
	brcs ma_steuerzeichen
ma_L1:	st X+, r16		;empfangenes Zeichen an aktueller Position speichern
	ldi r17, high(textspeicher+ANZAHLTEXTBYTES)
	cpi XL, low(textspeicher+ANZAHLTEXTBYTES)
	cpc XH, r17		;Ende des Textspeichers erreicht?
	brne mainloop
ma_newpage:
	ldi XL, low(textspeicher) ;ja: wieder auf Anfang des Textspeichers
	ldi XH, high(textspeicher)
	rjmp mainloop

ma_screenclear:
	ldi XL, low(textspeicher) ;auf Anfang des Textspeichers
	ldi XH, high(textspeicher)
	ldi r16, ' '
	ldi r17, low(ANZAHLTEXTBYTES)
	ldi r18, high(ANZAHLTEXTBYTES)
ma_L3:	st X+, r16
	subi r17, 1
	sbci r18, 0
	brne ma_L3
	rjmp ma_newpage
ma_steuerzeichen:
	cpi r16, 0x0A		;Sonderzeichen LF?
	breq ma_einezeilerunter
	cpi r16, 0x0D		;Sonderzeichen CR?
	brne ma_s1
	rcall aufzeilenanfang
	rjmp mainloop
ma_s1:
	cpi r16, 0x07		;Sonderzeichen fuer Piepston?
#ifdef PIEPSOFF
	breq mainloop		;ja --> ignorieren
#else
	breq ma_pi1		;ja --> Piepston starten
#endif
	cpi r16, 0x0C		;Sonderzeichen Newpage?
	breq ma_newpage
#ifdef MITESCSEQUENZEN
	cpi r16, 0x1B		;Sonderzeichen ESC
	breq ma_status_setzen
#endif
	cpi r16, 5		;Steuerzeichen fuer Hexmodus?
	breq ma_status_setzen
	cpi r16, 4		;Steuerzeichen fuer Seite loeschen?
	breq ma_screenclear
 	cpi r16, 3		;Steuerzeichen Spaltenwahl?
 	breq ma_status_setzen
 	cpi r16, 2		;Steuerzeichen Zeilenauswahl?
 	breq ma_status_setzen
 	cpi r16, 1		;Steuerzeichen "Naechstes Zeichen Code im Steuerzeichen-Bereich"?
	brne ma_L1		;nein: alle andern Zeichen als normale Zeichen behandeln
				;ja: Sprung auf ma_status_setzen
ma_status_setzen:
	mov status, r16
	rjmp mainloop
ma_status_auswerten:
	dec status		;war status==1 ?
	breq ma_L1		;ja: status geloescht und Sprung zum Zeichen als normales Zeichen behandeln
	inc status
	cpi status, 2
	breq ma_neuezeile
	cpi status, 3
	breq ma_neuespalte
#ifdef MITESCSEQUENZEN
	cpi status, 0x1B
	breq ma_escape
	cpi status, 0x1C
	breq ma_escape1
	cpi status, 0x1D
	breq ma_escape2
#endif
; 	cpi status, 5		; Hexmodus?  einzig verbleibende Moglichkeit, darum Test nicht wirklich noetig
; 	brne error
ma_hexmodus:
	cpi r16, 5		;Steuerzeichen fuer Hexmodus?
	breq ma_L2		;ja--> Hexmodus wieder ausschalten
	mov r17, r16
	swap r16		;obere 4 Bits nach unten,
	rcall hexdigit		;in eine Zahl ('0'...'F') wandeln
	rcall putzeichen	;und im Textspeicher speichern
	mov r16, r17
	rcall hexdigit		;gleiches fuer untere 4 Bits
	rcall putzeichen
	ldi r16, ' '		;zwischen den 2-stelligen Hexzahlen jeweils eine Leerstelle
	rcall putzeichen
	rjmp mainloop
#ifdef MITPIEPSTON
ma_pi1:	rcall piepston_starten
	rjmp mainloop
#endif
ma_neuespalte:
	rcall neuespalte
ma_L2:	clr status
	rjmp mainloop
ma_neuezeile:
	rcall neuezeile
	rjmp ma_L2
ma_einezeilerunter:
	ldi r18, ZEICHENPROZEILE
	add XL, r18
	adc XH, zero
	ldi r16, low(textspeicher+ANZAHLTEXTBYTES)
	ldi r17, high(textspeicher+ANZAHLTEXTBYTES)
	cp  XL, r16
	cpc XH, r17		;Grenze erreicht?
	brcs ma_e1		;nein-->
	subi XL, low(ANZAHLTEXTBYTES)	;ja: subtrahieren um wieder auf 1. Zeile zu kommen
	sbci XH, high(ANZAHLTEXTBYTES)
ma_e1:	rjmp mainloop
#ifdef MITESCSEQUENZEN
.def escapezahl=r12
ma_escape:			;<ESC> erkannt
	cpi r16, '&'
	breq ma_esc1_setzen
	cpi r16, '['
	brne ma_L2		;andere Zeichen ignorieren
ma_esc1_setzen:
	clr escapezahl
	ldi r16, 0x1C
	rjmp ma_status_setzen
ma_esc_setzen:
	ldi r16, 0x1B
	rjmp ma_status_setzen
ma_esc_zeile:
	mov r16, escapezahl
	rcall neuezeile
ma_esc2_setzen:
	clr escapezahl
	ldi r16, 0x1D
	rjmp ma_status_setzen
ma_screenclear1:
	clr status
	rjmp ma_screenclear
ma_escape1:			;<ESC>& oder <ESC>[ erkannt
	cpi r16, '#'
	breq ma_screenclear1	;<ESC>&# --> Seite loeschen
	cpi r16, 'D'
	breq ma_L2		;<ESC>&D --> Cursor aus ignorieren
	cpi r16, 'S'
	breq ma_esc_setzen	;<ESC>&S --> <ESC>&S5 ignorieren
	cpi r16, ';'		;Endzeichen fuer (Zeile) ?
	breq ma_esc_zeile	;auf Zeile escapezahl gehen und nach esc2 wechseln
	cpi r16, '0'		;Ziffer >= 0 fuer (Zeile) ?
	brcc ma_e1a		;ja -->
	mov escapezahl, r16	;nein: direkt binaere Zahl verwenden
	rjmp mainloop
ma_e1a:	subi r16, '0'
	cpi r16, 10		;Ziffer 0 bis 9 fuer (Zeile) ?
	brcc ma_L2		;nein --> ESC-Sequenz ignorieren
	rcall put_escapezahl
	rjmp mainloop
ma_escape2:			;<ESC>[(Zeile); erkannt
	cpi r16, 'H'		;Endzeichen fuer (Spalte) ?
	breq ma_esc_spalte	;auf Spalte escapezahl gehen und ESC-Modus beenden
	cpi r16, '0'		;Ziffer >= 0 fuer (Spalte) ?
	brcc ma_e2a		;ja -->
	mov escapezahl, r16	;nein: direkt binaere Zahl verwenden
	rjmp mainloop
ma_e2a:	subi r16, '0'
	cpi r16, 10		;Ziffer 0 bis 9 fuer (Spalte) ?
	brcc ma_L2		;nein --> restliche ESC-Sequenz ignorieren
	rcall put_escapezahl
	rjmp mainloop
ma_esc_spalte:
	mov r16, escapezahl
	rcall neuespalte
	rjmp ma_L2
put_escapezahl:
	mov r17, escapezahl
	lsl escapezahl
	lsl escapezahl		;4*x
	add escapezahl, r17	;5*x
	lsl escapezahl		;10*x
	add escapezahl, r16
	ret
#endif

neuespalte:
	rcall aufzeilenanfang
	cpi r16, ZEICHENPROZEILE   ;r16 zu gross?
	brcs ma_n1
	ldi r16, ZEICHENPROZEILE-1 ;ja: Wert auf Anzahl vorhandene Spalten beschraenken
ma_n1:	add XL, r16		;gewuenschte Spalte addieren
	adc XH, zero
	ret

neuezeile:
	push r12
	push r13
	mov r12, XL
	mov r13, XH
	ldi r17, low(textspeicher)
	ldi r18, high(textspeicher)
	sub r12, r17
	sbc r13, r18
	ldi r18, ZEICHENPROZEILE ;aktuelle Spalte berechnen:
	rcall divmodu16u8	 ;r13:r12 / r18 --> r13:r12=aktuelle Zeile  r18=Rest=aktuelle Spalte
	ldi r17, ZEICHENPROZEILE
	cpi r16, ANZAHLTEXTZEILEN   ;r16 zu gross?
	brcs ma_n2
	ldi r16, ANZAHLTEXTZEILEN-1 ;ja: Wert auf Anzahl vorhandene Zeilen beschraenken
ma_n2:	mul r16, r17		;r16*ZEICHENPROZEILE --> r1:r0
	ldi XL, low(textspeicher)
	ldi XH, high(textspeicher)
	add XL, r0
	adc XH, r1
	add XL, r18		;aktuelle Spalte wieder addieren
	adc XH, zero
	pop r13
	pop r12
	ret

aufzeilenanfang:
	push r12
	push r13
	mov r12, XL
	mov r13, XH
	ldi r17, low(textspeicher)
	ldi r18, high(textspeicher)
	sub r12, r17
	sbc r13, r18
	ldi r18, ZEICHENPROZEILE ;aktuelle Zeile berechnen:
	rcall divmodu16u8	 ;r13:r12 / r18 --> r13:r12=aktuelle Zeile,  r18=Rest
	sub XL, r18		 ;Rest subtrahieren, somit kommen wir auf Spalte 0
	sbc XH, zero
	pop r13
	pop r12
	ret

hexdigit:			;untere 4 Bits von r16 in eine Hex-Zahl (0...A) wandeln
	push r18
	andi r16, 0x0F
	cpi r16, 10
	brcs he_L1
	ldi r18, 7
	add r16, r18
he_L1:	ldi r18, '0'
	add r16, r18
	pop r18
	ret

putzeichen:
	st X+, r16		;Zeichen an aktueller Position speichern
	ldi r16, high(textspeicher+ANZAHLTEXTBYTES)
	cpi XL, low(textspeicher+ANZAHLTEXTBYTES)
	cpc XH, r16		;Ende des Textspeichers erreicht?
	brne ma_h1
	ldi XL, low(textspeicher) ;ja: wieder auf Anfang des Textspeichers
	ldi XH, high(textspeicher)
ma_h1:	ret

#ifdef STRESSTEST	;Test mit Hauptprogramm stark beschaeftigt
zeitanzeige:
	push r12
	push r13
	push r18
	push r19
	push XL
	push XH
#ifdef STRESS_SEITE_LOESCHEN
	ldi r16, 0x04
	rcall putchar		;Screenclear
	rcall viele_zeichen_darstellen
#else
  #ifdef STRESS_VIELE_ZEICHEN
	ldi r16, 0x02
	rcall putchar
	ldi r16, 0x00		;auf 1.Zeile gehen
	rcall putchar
	ldi r16, 0x03
	rcall putchar
	ldi r16, 0x00		;auf Spalte 0 gehen
	rcall putchar
	rcall viele_zeichen_darstellen
  #endif
#endif
	ldi r16, 0x02
	rcall putchar
	ldi r16, 0x05		;auf 6.Zeile gehen
	rcall putchar
	ldi r16, 0x03
	rcall putchar
	ldi r16, 0x00		;auf Spalte 0 gehen
	rcall putchar

	;; Zeit anzeigen:
	lds r12, uhr_std
	rcall zahl_anzeigen_2stellig
	ldi r16, ':'
	rcall putchar
	lds r12, uhr_min
	rcall zahl_anzeigen_2stellig
	ldi r16, ':'
	rcall putchar
	lds r12, uhr_sec
	rcall zahl_anzeigen_2stellig
	ldi r16, ' '
	rcall putchar
	pop XH
	pop XL
	pop r19
	pop r18
	pop r13
	pop r12
	ret

#ifdef JITTER_TEST
minmax_anzeige:			;r16 r17 zerstoert
	push r12
	push r13
	push r18
	push r19
	push XL
	push XH
	mov r19, maxwert
#ifdef STRESS_SEITE_LOESCHEN
	ldi r16, 0x04
	rcall putchar		;Screenclear
	rcall viele_zeichen_darstellen
#else
  #ifdef STRESS_VIELE_ZEICHEN
	ldi r16, 0x02
	rcall putchar
	ldi r16, 0x00		;auf 1.Zeile gehen
	rcall putchar
	ldi r16, 0x03
	rcall putchar
	ldi r16, 0x00		;auf Spalte 0 gehen
	rcall putchar
	rcall viele_zeichen_darstellen
  #endif
#endif
	ldi r16, 0x02
	rcall putchar
	ldi r16, 0x05		;auf 6.Zeile gehen
	rcall putchar
	ldi r16, 0x03
	rcall putchar
	ldi r16, 0x00		;auf Spalte 0 gehen
	rcall putchar

	ldi r16, 'T'
	rcall putchar
	ldi r16, 'C'
	rcall putchar
	ldi r16, 'N'
	rcall putchar
	ldi r16, 'T'
	rcall putchar
	ldi r16, '1'
	rcall putchar
	ldi r16, '='
	rcall putchar

	;; Minimalwert anzeigen:
	mov r12, minwert
	rcall zahl_anzeigen_2stellig
	ldi r16, '.'
	rcall putchar
	ldi r16, '.'
	rcall putchar
	
	;; Maximalwert anzeigen:
;	mov r12, maxwert
	mov r12, r19
	rcall zahl_anzeigen_2stellig
	ldi r16, ' '
	rcall putchar
	;; Zeit anzeigen:
;	lds r12, uhr_std
;	rcall zahl_anzeigen_2stellig
;	ldi r16, ':'
;	rcall putchar
	lds r12, uhr_min
	rcall zahl_anzeigen_2stellig
	ldi r16, ':'
	rcall putchar
	lds r12, uhr_sec
	rcall zahl_anzeigen_2stellig
	ldi r16, ' '
	rcall putchar

	pop XH
	pop XL
	pop r19
	pop r18
	pop r13
	pop r12
	ret
#endif

zahl_anzeigen_2stellig:		;Zahl r12 anzeigen, 2 stellig mit fuehrender 0, r13:r12 r16-r18 XH:XL zerstoert
	push r19
	clr r13
	ldi r18, 10		;zahl/10
	rcall divmodu16u8	;r13:r12=Resultat, r18=Rest
	ldi r19, '0'
	mov r16, r12		;r16=Resultat=erste Ziffer
	cpi r16, 9+1		;vordere Stelle groesser 9?
	brcs L9
	ldi r16, 9		;auf 9 beschraenken
L9:	add r16, r19		;erste Ziffer als ASCII-Zeichen
	add r19, r18		;r19=Rest als ASCII-Zeichen
	rcall putchar
	mov r16, r19
	pop r19
;	rcall putchar
;	ret

putchar:			;Zeichen r16 im serbuffer speichern (sofern nicht schon voll)
	ldi XL, low(serbuffer)	;zerstoerte Register: r17, r18, X
	ldi XH, high(serbuffer)
	lds r17, jwrite
	lds r18, jread
	add XL, r17
	adc XH, zero
	inc r17
	andi r17, SERBUFFERSIZE-1
	cp r17, r18		;ist Puffer voll?
	breq _ret		;ja-> Zeichen ignorieren
	st X, r16		;nein: Zeichen im Puffer speichern
	sts jwrite, r17
_ret:	ret

viele_zeichen_darstellen:
	ldi r16, 'A'
_L2:	rcall putchar
	inc r16
	cpi r16, 'A'+12
	brne _L2
	ldi r16, 0x02
	rcall putchar
	ldi r16, 0x01		;auf 2.Zeile gehen
	rcall putchar
	ldi r16, 0x03
	rcall putchar
	ldi r16, 0x00		;auf Spalte 0 gehen
	rcall putchar
	ldi r16, 'a'
_L3:	rcall putchar
	inc r16
	cpi r16, 'a'+12
	brne _L3
  #if SERBUFFERSIZE>=128
	ldi r16, 0x02
	rcall putchar
	ldi r16, 0x02		;auf 3.Zeile gehen
	rcall putchar
	ldi r16, 0x03
	rcall putchar
	ldi r16, 0x00		;auf Spalte 0 gehen
	rcall putchar
	ldi r16, '0'
_L4:	rcall putchar
	inc r16
	cpi r16, 'A'
	brne _L4
	ldi r16, 0x02
	rcall putchar
	ldi r16, 0x03		;auf 4.Zeile gehen
	rcall putchar
	ldi r16, 0x03
	rcall putchar
	ldi r16, 0x00		;auf Spalte 0 gehen
	rcall putchar
	ldi r16, ' '
_L5:	rcall putchar
	inc r16
	cpi r16, '0'
	brne _L5
  #endif ;//SERBUFFERSIZE
	ret
#endif

; Ende Hauptprogramm
;-----------------------------------------------------------------------------

;-----------------------------------------------------------------------------
;; Uhr: Zeit seit dem Einschalten
uhr_reset:
	sts uhr_ticks, zero
	sts uhr_sec, zero
	sts uhr_min, zero
	sts uhr_std, zero
	sts uhr_tage, zero
	ret

uhr_update:			;Dieses Unterprogramm braucht immer genau die gleiche Anzahl Takte
	lds r16, uhr_ticks	;unabhaengig davon welche Daten wirklich aktualisiert werden.
	inc r16			;52 Takte inklusive rcall und ret,  r16-r17 zerstoert
	cpi r16, 50
	breq u_L1
	clr r17
	rjmp u_L1a
u_L1:	clr r16
	ldi r17, 1
u_L1a:	sts uhr_ticks, r16
	lds r16, uhr_sec
	add r16, r17
	cpi r16, 60
	breq u_L2
	clr r17
	rjmp u_L2a
u_L2:	clr r16
	ldi r17, 1
u_L2a:	sts uhr_sec, r16
	lds r16, uhr_min
	add r16, r17
	cpi r16, 60
	breq u_L3
	clr r17
	rjmp u_L3a
u_L3:	clr r16
	ldi r17, 1
u_L3a:	sts uhr_min, r16
	lds r16, uhr_std
	add r16, r17
	cpi r16, 24
	breq u_L4
	clr r17
	rjmp u_L4a
u_L4:	clr r16
	ldi r17, 1
u_L4a:	sts uhr_std, r16
	lds r16, uhr_tage
	add r16, r17
	sts uhr_tage, r16
	ret

#ifdef MITPIEPSTON
piepston_starten:
	lds r16, uhr_ticks
	subi r16, -25		;0.5 Sec addieren
	cpi r16, 50
	brcs pi1
	subi r16, 50
pi1:	inc r16			;Wertebereich 1...50
	sts pieps_endzeit, r16
	sbi SOUND_PORT, SOUND_BIT
	ret
piepston_pruefen:
	lds r16, pieps_endzeit
	tst r16
	breq pi_ret
	lds r17, uhr_ticks
	dec r16
	cp r16,r17
	breq piepston_stop
pi_ret:	ret
piepston_stop:
	sts pieps_endzeit, zero
	cbi SOUND_PORT, SOUND_BIT
	ret
#endif

;-----------------------------------------------------------------------------

