/* bahn.cc       letzte Aenderung: 19.8.2010
Steuerung fuer Modelleisenbahn
*/
#define VERSION "0.06"
/*
 Prozessor: Atmega32, 16MHz Quarz
 Schaltung: bahnsteuerung.png
	    4 Ausgaenge fuer Weichen PB0...PB3
	    4 LEDs auf PB4...PB7
	    3 Taster auf PC2...PC4
	    2 Ausgaenge fuer DCC-Steuerung PC0-PC1
	    4 Leitungen Reservere PC5-PC7
   Grafik-LCD: PA0...PA7, PD2-PD7
   Tastatur: PD0 (RxD)

 Autor: Rolf Pfister
 Copyright: Freeware
 History:
 15.5.2010	Erstellung, Uhr vom Lichtwecker uebernommen
 13.6.2010	Testen eines Grafik-LCD-Displays
 7.8.2010	Testen Textausgabe auf Grafik-LCD-Display
 8.8.2010	Tastatur mit Tiny2313
 9.8.2010	Datenuebertragung von Tastatur mit UART
 15.8.2010	DCC-Steuerung, Schema-Aenderungen: Taster statt PC0...PC2
		neu auf PC2...PC4
 19.8.10  0.06	Grafik-LCD neu auf PORTD (statt vorher PORTC)

*/
#include <avr/io.h>
#include <avr/interrupt.h>
#include "ulong.h"
#include "stdio.h"

#define UARTVARIANTE

/**** Vordeklarationen ****/
void tastaturcheck();
char tastaturabfrage();
void zugeinstellen(int zugnr);
void zugschneller();
void zuglangsamer(int vneu= -1);
void zugnotstop();
void glcd_printzeile(int zeile,const char *cstr,
		     int p1=0,int p2=0,int p3=0,int p4=0);

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=12,stunden=9;
static volatile uchar tage=14,monat=8;
static volatile uint jahr=2010;
static char monatstage[13]={0,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=5; //0=Montag ... 6=Sonntag
static volatile char sommerzeit=1; //0=Winterzeit 1=Sommerzeit

void schaltjahrtest()
{
 if(jahr%4==0 && jahr%100!=0 || jahr%400==0) monatstage[2]=29;
 else monatstage[2]=28;
}
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
void variable_feiertage_berechnen();
void schaltjahr_und_wochentagberechnen()
{
 schaltjahrtest();
 wochentagberechnen();
 variable_feiertage_berechnen();
}

// Feiertage:
const char NFEST=6;//Anzahl feste Feiertage
const char festfeiertag[NFEST]={1, 2, 1, 1, 25, 26};
const char festfeiermon[NFEST]={1, 1, 5, 8, 12, 12};
const char NVARI=4;//Anzahl von Ostern abhaengige Feiertage
static char varifeiertag[NVARI];
static char varifeiermon[NVARI];
void variable_feiertage_berechnen()
{
 int a,b,c,k,p,q,M,N;
 char d,e,tag,i;
 const char mon=3;
 //Osterformel nach Gauss:
 a=jahr%19;
 b=jahr%4;
 c=jahr%7;
 k=jahr/100;
 p=(8*k+13)/25;
 q=k/4;
 M=(15+k-p-q)%30;
 N=(4+k-q)%7;
 d=(19*a+M)%30;
 e=(2*b+4*c+6*d+N)%7;
 tag=22+d+e;
 varifeiertag[0]=tag+1; varifeiermon[0]=mon; //Ostermontag
 varifeiertag[1]=tag-2; varifeiermon[1]=mon; //Karfreitag
 varifeiertag[2]=tag+39; varifeiermon[2]=mon; //Auffahrt
 varifeiertag[3]=tag+50; varifeiermon[3]=mon; //Pfingstmontag
 for(i=0;i<NVARI;i++)
   while(varifeiertag[i]>monatstage[varifeiermon[i]])
    {varifeiertag[i] -= monatstage[varifeiermon[i]]; varifeiermon[i]++;}
}

bool istfeiertag()
{
 char i;
 for(i=0;i<NFEST;i++)
   if(tage==festfeiertag[i] && monat==festfeiermon[i]) return true;
 for(i=0;i<NVARI;i++)
   if(tage==varifeiertag[i] && monat==varifeiermon[i]) return true;
 return false;
}

// Quarz moeglichst genau ausmessen und hier eintragen
// in Hertz und Tausendstel Hertz:

//const ulong Hz=16000000,MilliHz=0; //exakt 16MHz
//Beispiel: 16MHz ist nach n Tagen z sec vorgegangen:
//nach n Tagen sind also 16000000*(3600*24*n+z) Takte vergangen
//Frequenz ist also:
// x = (3600*24*n+z)*16000000 / (3600*24*n)
//   = 16000000 + z*16000000/(3600*24*n)
const ulong Hz=16001422,MilliHz=0;
#define F_CPU 16001422UL

const uint Rest1 = Hz%1000;//jede Sekunde zu korrigierender Fehler
//const uint Rest2 = MilliHz*6/100;//jede Minute
//const uint Rest3 = ((MilliHz*6-Rest2*100)*60+50)/100;//jede Stunde
const uint NormalerCompwert=Hz/1000-1;
//der Compiler sollte bis hier noch keinen Code erzeugt haben.
//(der Quarz ist wahrscheinlich zu wenig stabil als dass Rest2 und Rest3
//eine Rolle spielen wuerden. Also MilliHz vorlaeufig auf 0 lassen.)

//fuer Tastenentprellung:
static volatile char entprelltetasten=0xFF;
static char status1=0xFF,status2=0xFF;
static volatile char tastegedrueckt=0;
static volatile char langgedrueckt=0;
static char status3=0,status4=0,status5=0;
static volatile char tasten_aktiv=1;

//fuer LEDs:
static volatile int rothell=0,gruenhell=0,gelbhell=0,orangehell=0;//Helligkeit der LEDs
const uchar LEDROT=0x10,LEDGRUEN=0x20,LEDGELB=0x40,LEDORANGE=0x80;
static volatile uchar LEDBITS=0;//Eingeschaltete LEDs
static volatile uchar aktportb=0x00;//aktueller Zustand von PORTB

//die Namen SIG_* sind in /usr/avr/include/avr/iom32.h zu finden?
SIGNAL (SIG_OUTPUT_COMPARE1A)
{
 uchar mehr;
 uint ms=millisec;
 uint h;//fuer LEDs
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0;
   //if(++sec>=60-Rest2) mehr=1; else mehr=0;
   mehr=0;//einfacher wenn Rest2 und Rest3 sowieso ==0
   if(++sec==60)
    {sec=0;
     //if(++min>=60-Rest3) mehr=1; else mehr=0;
     if(++min==60)
      {min=0; mehr=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      variable_feiertage_berechnen();
	     }
	  }
	}
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
      }
    }
  }
 else if((ms&0x0F)==1) //alle 16ms
  {if(tasten_aktiv)
   {//Tasten-Entprellung: fuer 8 Tasten unabhaengig voneinander
    char pin,ung;
    pin = PINC;
    ung = (pin^status1) |   //Ist PINC verschieden zu status1 oder
          (status2^status1);//ist status2 verschieden zu status1 ?
    entprelltetasten = //entweder Alt oder Neu uebernehmen:
     (entprelltetasten&ung) //ja: Alt=Alter Zustand, nein: Alt=0
     | (pin & (ung^0xFF)); //nein: Neu=PINC, ja: Neu=0
    status2=status1;
    status1=pin;
    tastegedrueckt |= (entprelltetasten^0xFF);
   }
  }
 else if((ms&0xFF)==0xE7) //alle ca 250ms
  {if(tasten_aktiv)
   {//Erkennung von langen Tastendrucken
    langgedrueckt = (entprelltetasten^0xFF) & status3 & status4 & status5;
    status5=status4; status4=status3; status3=(entprelltetasten^0xFF);
   }
  }
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
 h=ms%10;//LEDs in 10 Helligkeitsstufen
 aktportb &= 0x0F;
 if((LEDBITS&LEDROT) && h<rothell) aktportb |= LEDROT;
 if((LEDBITS&LEDGRUEN) && h<gruenhell) aktportb |= LEDGRUEN;
 if((LEDBITS&LEDGELB) && h<gelbhell) aktportb |= LEDGELB;
 if((LEDBITS&LEDORANGE) && h<orangehell) aktportb |= LEDORANGE;
 PORTB = aktportb;
}

//static volatile unsigned char send_byte=0,send_bit=0x80;
static volatile unsigned char send_byte=0xFF,send_bit=0x80;//test
//wenn sind_bit==0x00 dann ist ganzes send_byte gesendet.
void bytesenden(unsigned char c)
{
 while(send_bit) ;//warten bis letztes Byte gesendet
 send_byte=c; send_bit=0x80;
}

SIGNAL (SIG_OUTPUT_COMPARE2)
{
 //Timer2 sollte alle 58usec kommen
 //1-Bits werden mit 58usec = 8620.7Hz moduliert
 //0-Bits mit 116usec = 4310.3Hz
 //(nach Spezifikation nominal 100usec, aber 95 bis 9800 erlaubt)
 static char n=0,pc=1;
 //static unsigned char aktbyte=0,aktbit=0; //ev. noch 1 Byte puffern
 //if(aktbit==0) {aktbyte=send_byte; aktbit=send_bit; send_bit=0;} //ev. puff.
 if(++n==2 || (send_byte&send_bit)!=0)
  {PORTC=(PORTC&0xFC)+pc; pc^=3; //Polaritaet wechseln
   n=0;
   if(pc==2)
    {send_bit>>=1;
     if(send_bit==0) send_bit=0x80;//test zum Frequenz von 1-Bits messen
    }
  }
}

//Bei einem unerwarteten Interrupt nichts machen:
//SIGNAL (__vector_default) {}
//Oder besser wirklich einfach nur ein reti machen:
void __attribute__ ((naked)) __vector_default(void) {__asm__ ("reti");}
//Ohne diese Funktion wird nach 0 gesprungen, was einem Reset entspricht.

void milliwait(int n) //n Millisekunden warten  999 >= n >= 1
{
 uint ms;
 //ms=millisec+n;  das macht der Compiler nicht ganz korrekt!
 cli(); ms=millisec; sei(); ms+=n;
 if(ms>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) ;
}

/**** Routinen fuer LCD-Ansteuerung ****/
void milliwait(int n);
char microwait(int n) //etwa n Microsekunden warten
{
 char vx=0;
 while(n>0)
  {vx+=n; --n;}
 return vx;
}

#include "glcd.cc"

//Globale Variablen:
const char ROTETASTE=0x08,GRUENETASTE=0x10,SCHWARZETASTE=0x04,ALLETASTEN=0x1C;
const char MINSEK=0,UHRZEIT=1,MOTG=2,WOTAG=3,JJJ=4,WECK1=5,WECK2=6;
static char uhrmodus=UHRZEIT,stellen=0;

void zeitauflcdanzeigen()
{
 char txt[40];
 glcd_goto(4,1);
 sprintf(txt,"%02d:%02d:%02d ",stunden,min,sec);
 glcd_write(txt);
 sprintf(txt,"%02d.%02d.%d",tage,monat,jahr);
 glcd_write(txt);
 if(stellen>0)
  {int pos=stellen*3-1;
   if(pos>16) pos=16;//test
   glcd_goto(4,pos);
  }
}
void schwarzetaste()
{
 if(++stellen==7) stellen=0;
 const char *s="                ";
 switch(stellen)
  {case 1: s="Std stellen";
   CASE 2: s="Min stellen";
   CASE 3: s="Sec stellen";
   CASE 4: s="Tag stellen";
   CASE 5: s="Monat stellen";
   CASE 6: s="Jahr stellen";
  }
 glcd_printzeile(7,s);
}
void rotetaste()
{
 if(stellen>0)
  switch(stellen)
   {case 1: cli(); if(++stunden==24) stunden=0; sei();
    CASE 2: cli(); if(++min==60) min=0; sei();
    CASE 3: cli(); if(sec>=30) ++min; sec=0; sei();
    CASE 4: if(++tage>monatstage[monat]) tage=1; wochentagberechnen();
    CASE 5: if(++monat==13) monat=1; wochentagberechnen();
    CASE 6: ++jahr; schaltjahr_und_wochentagberechnen();
   }
}
void gruenetaste()
{
 if(stellen>0)
  switch(stellen)
   {case 1: cli(); if(stunden==0) stunden=23; else --stunden; sei();
    CASE 2: cli(); if(min==0) min=59; else --min; sei();
    CASE 3: cli(); if(sec>=30) ++min; sec=0; sei();
    CASE 4: if(--tage==0) tage=monatstage[monat]; wochentagberechnen();
    CASE 5: if(--monat==0) monat=12; wochentagberechnen();
    CASE 6: --jahr; schaltjahr_und_wochentagberechnen();
   }
}

#ifdef UARTVARIANTE
void uart_init();
#endif

int main(void)
{
 const char WARTE_AUF_TASTENDRUCK=0,WARTE_AUF_TASTELOSLASSEN=1;
 char warten=WARTE_AUF_TASTENDRUCK;
 int h;
 char DccCompwert=29-1;
 DDRA  = 0x00; //PortA alles Eingaenge
 DDRB  = 0xFF; //PortB alles Ausgaenge
 DDRC  = 0x03; //PortC untere 2 Bits Ausgaenge, Rest Eingaenge
 DDRD  = 0xFC; //PortD obere 6 Bits Ausgaenge, unter 2 Bits offen fuer UART
 PORTA = 0;//keine Pullups
 PORTB = 0;//alle Ausgaenge auf 0
 PORTC = 0x1C; //C2-C4 Pullup-Widerstaende fuer Taster
 PORTD = 0xE0; //C7-C5 RST,CS1,CS2 auf H (=inaktiv fuer LCD)

 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert

 TCCR2 = (1<<WGM21)|(3<<CS20); //CTC-Modus, Prescaler=32, 16MHz/32 --> 2usec
 OCR2 = DccCompwert;
 TCNT2 = 0; //Startwert Timer2
 TIMSK |= (1<<OCIE2); //Timer2 Interrupts bei Vergleichswert

 sei(); //Interrupt einschalten
 schaltjahr_und_wochentagberechnen();
 do {milliwait(200);} while((entprelltetasten&ALLETASTEN)!=ALLETASTEN);
 langgedrueckt=tastegedrueckt=0;//erste Entprellungen abwarten 
 PORTB=0x20; //test gruene LED an

 milliwait(200);
 glcd_init();
 glcd_goto(5,1);
 glcd_write("Eisenbahn-Steuerung");
 glcd_goto(6,1);
 glcd_write("Version "); glcd_write(VERSION);
 glcd_goto(7,1);
 glcd_write("Zug1  Zug2  Zug3");//test

#ifdef UARTVARIANTE
 uart_init();
#endif

 //Testschlaufen:
 rothell=gruenhell=gelbhell=orangehell=1;//Helligkeit der LEDs
 while((tastegedrueckt&ALLETASTEN)==0) //Startschlaufe, weiter mit Tastendruck
  {LEDBITS=(sec<<4)&0xF0; //laufende Sekunden auf den LEDs anzeigen
  }
 /** test **
 tastegedrueckt=0;
 while((langgedrueckt&7)==0) //Testschlaufe, weiter mit langem Tastendruck
  {//PORTB=(entprelltetasten<<4); //Status der Tasten auf den LEDs anzeigen
   PORTB=((tastegedrueckt&7)<<4); //Status der Tasten auf den LEDs anzeigen
   if((tastegedrueckt&7)==7) {milliwait(200); tastegedrueckt=0;}//test
  }
 /**/
 langgedrueckt=tastegedrueckt=0;

 LEDBITS=0;//alle LEDs aus
 glcd_init();
 glcd_test(2);//test
 //glcd_test2();//test

 char pos=0,modus=0,altsec=0;
 const int ZUGSTEUERN=1;//Gueltige Werte fuer modus
 while(1) //Hauptschlaufe
  {char i,c;
   if(altsec!=sec) {zeitauflcdanzeigen(); altsec=sec;}
   //zeitauflcdanzeigen();//test  ergibt unerfindliche Stoerungen
   //milliwait(25);
   for(i=0;i<25;i++) {milliwait(1); tastaturcheck();}
   if((c=tastaturabfrage())!=0) //test
        {if(modus==0)
	  {glcd_goto(7,pos); //Tastatureingabe auf 7.Zeile darstellen
	   glcd_write(c);
	   if(++pos==21) pos=0;
	  }
	 switch(c) //Reaktionen auf Tastatureingabe
	  {case 'R': case 'M': case 'N': case 'C':
	    zugeinstellen(c);
	    modus=ZUGSTEUERN;
	  }
	 if(modus==ZUGSTEUERN)
	  {if(c>='0' && c<='9') zuglangsamer(c-'0');
	   else switch(c)
	    {case '+': zugschneller();
	     CASE '-': zuglangsamer();
	     CASE '=': zugeinstellen(0); modus=0; pos=0;
	    }
	  }
	}
   if(warten==WARTE_AUF_TASTENDRUCK)
    {if(tastegedrueckt&ALLETASTEN)
      {if(tastegedrueckt&SCHWARZETASTE) schwarzetaste();
       else if(tastegedrueckt&ROTETASTE)
	          {if(stellen) rotetaste(); else zugnotstop();}
       else if(tastegedrueckt&GRUENETASTE)
	          {gruenetaste(); glcd_test(3);}//test
       warten=WARTE_AUF_TASTELOSLASSEN;
      }
    }
   else //warten==WARTE_AUF_TASTELOSLASSEN
    {if((entprelltetasten&ALLETASTEN)==ALLETASTEN)
      {warten=WARTE_AUF_TASTENDRUCK; langgedrueckt=tastegedrueckt=0;}
    }
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

/*** Tastatur mit Tiny2313 ***/
#ifdef UARTVARIANTE

#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 16000000"
#define F_CPU 16000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 8192UL      // Baudrate
 
// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart_init()
{
 UBRRH = UBRR_VAL >> 8;
 UBRRL = UBRR_VAL & 0xFF;
 //UCSRB |= (1<<TXEN); // UART TX einschalten
 UCSRB |= (1<<RXEN); // UART RX einschalten
 UCSRC = (1<<URSEL)|(3<<UCSZ0); // Asynchron 8N1 Atmega32
}

/*
void uart_putc(unsigned char c)
{
 while (!(UCSRA & (1<<UDRE))) ;// warten bis Senden moeglich
 UDR = c;                      // sende Zeichen
}
char uart_getc(void)
{
 while (!(UCSRA & (1<<RXC))) ;// warten bis Zeichen verfuegbar
 return UDR; // Zeichen aus UDR an Aufrufer zurueckgeben
}
*/

static char uart_puffer[8],uart_i1=0,uart_i2=0;
void tastaturcheck()
{
 if(!(UCSRA & (1<<RXC))) return;// warten bis Zeichen verfuegbar
 if(((uart_i2+1)&0x07)==uart_i1) return;//Puffer voll, noch nichts tun
 uart_puffer[uart_i2++]=UDR;
 uart_i2 &= 0x07;
}
char tastaturabfrage()
{
 char c=0;
 if(uart_i1!=uart_i2)
  {c=uart_puffer[uart_i1++]; uart_i1&=0x07;}
 return c;
}

#else
/*
Datenuebertragung auf 1 Leitung:
-------------------------------

                | 1.Bit |  2. Bit  |   ...  9. Bit |    Rueckmeldung

-----+ Startbit +---+   +---+      +---            +----+   +--
     |  3 ms    |1ms|1ms|1ms| 2ms  |   ...         |    |1ms|
     |          |   |   |   |      |               |    |   |
     +----------+   +---+   +------+             --+    +---+

Das Startbit definiert die Laenge eines zu uebertragenden Bits und ist
3 Einheiten lang (z.B. 3ms). Eine 0 wird durch einen 2 Einheiten langen
Puls uebertragen, eine 1 durch einen 1 Einheit langen Puls.
(Eine 1 also ein kurzer negativer Puls, eine 0 ein langer Puls)
Wenn das erste Datenbit eine 1 ist wird ein Kommando fuer das LCD, sonst
ein darzustellendes Zeichen uebertragen. Die folgenden 8 Bit ergeben
ein Byte beginnend mit dem hoechstwertigen Bit.
Fuer eine schnelle Uebertragung muss am Ende die Datenrichtung umgekehrt
werden, und der LCD-Kontroller gibt einen 1 Einheiten langen Puls zurueck.
(Fuer langsame Uebertragung kann man darauf verzichten und einfach eine
genuegend lange Warteschlaufe einbauen.)
*/
static volatile char taste_ready=0,taste_c=0;
void tastaturcheck()
{
 char d0;
 d0=PIND&1;
 if(d0==1) return;
 char i,j,ok=0;
 int startbit,kurzesbit,langesbit,sehrlangesbit,tmitte,t1,daten;
 for(j=0;ok==0 && j<3;j++)
  {for(startbit=0;(PIND&1)==0;startbit++)
     microwait(10);//Laenge des Startbits messen
   langesbit=startbit*2/3;
   kurzesbit=startbit/3;
   sehrlangesbit=langesbit+(kurzesbit+1)/2;
   tmitte=(langesbit+kurzesbit+1)/2;
   for(i=0,daten=0,ok=1;i<9;i++)
    {for(t1=0;(PIND&1)!=0;t1++)
      {microwait(10);//Low abwarten
       if(t1>sehrlangesbit) {ok=0; daten=0; break;} //Abbruch bei Fehler
      }
     if(ok==0) break;
     for(t1=0;(PIND&1)==0;t1++) microwait(10);//Bitlaenge messen
     daten <<= 1;
     if(t1<tmitte) daten++;
     else if(t1>sehrlangesbit) {ok=0; daten=0; break;} //Abbruch bei Fehler
    }
  }
 if((d0=daten&0xFF)!=0) {taste_ready=1; taste_c=d0;}
 return;
}
char tastaturabfrage()
{
 char c=0;
 if(taste_ready)
  {c=taste_c; taste_ready=0;}
 return c;
}
#endif

/************ Einstellungen fuer die Zuege ************/
const int VMAX=32;
static int aktuellerzug=0,vzug[4]={0,0,0,0};
void glcd_printzeile(int zeile,const char *cstr,int p1,int p2,int p3,int p4)
{
 const int N=22; //Breite der Anzeige
 char i,text[N];
 glcd_goto(zeile,0); //Anfang 6.Zeile
 sprintf(text,cstr,p1,p2,p3,p4);
 for(i=0;i<N-1 && text[i]!=0;i++) ;
 for(;i<N-1;i++) text[i]=' ';
 text[i]=0;
 glcd_write(text);
}
void zuganzeigen()
{
 if(aktuellerzug==0)
   glcd_printzeile(6,"v[4] = %d %d %d %d",vzug[0],vzug[1],vzug[2],vzug[3]);
 else glcd_printzeile(6,"Zug %d  v=%d",aktuellerzug,vzug[aktuellerzug-1]);
}
void zugfehler(int fehlernummer)
{
 char text[22];
 glcd_goto(7,0); //Anfang 7.Zeile
 sprintf(text,"Fehler %d",fehlernummer);
}
void zugeinstellen(int zugnr)
{
 switch(zugnr)
  {case 'R': zugnr=1;CASE 'M': zugnr=2;CASE 'N': zugnr=3;CASE 'C': zugnr=4;}
 if(zugnr<0 || zugnr>4) {zugnr=0; zugfehler(1);}
 aktuellerzug=zugnr;
 zuganzeigen();
}
void zugschneller()
{
 int n=aktuellerzug-1;
 if(n<0) return;
 if(vzug[n]<VMAX) vzug[n]++;
 zuganzeigen();
}
void zuglangsamer(int vneu)
{
 int n=aktuellerzug-1;
 if(n<0) return;
 if(vneu>=0 && vneu<=VMAX) vzug[n]=vneu;
 else if(vzug[n]>0) --vzug[n];
 zuganzeigen();
}
void zugnotstop() //alle Zuege anhalten
{
 char i;
 for(i=0;i<4;i++) vzug[i]=0;
 zuganzeigen();
}
