/* schach.cc			letzte Aenderung: 28.12.2016
 dies ist das Hauptprogramm unter Linux
 ausfuehrlichere Beschreibungen siehe:
   ../avr/schachavr/Liesmich.txt und kartonschach.cc
*/

#define VERSION "Version 1.02"

/*
Uebersetzen unter Unix (Linux):
> make  ;siehe makefile

History:
30.8.2016	 Erstellung (RP)
16.10.2016	 AVR-Variante
27.10.2016       Fehlerkorrekturen
31.10.           auslagerung des Hauptteils in schach-engine.cc
9.11.            spielerzug() ins Hauptprogramm verschoben
13.11.2016  1.0  aktualisierte schach-engine von schachavr uebernommen
10.12.2017  1.02 eingefuegte Grafik (mit xtekplot1)

*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <ctype.h>
#include <string.h>

#include <xtekplot1.h>
static int exitflag=0;
bool grafik_scanfzug(char *str,int max);

/******************** Hauptteil des Programms *************************/
#include "schach-engine.h"

static int8_t spielsuchtiefe=0;

void hilfe()
{
 printf("help oder hilfe eingegeben\n");
#ifdef BAUERNSPIEL
 printf("Provisorische Regeln fuer Bauernspiel:\n");
 printf("Wer zuerst mit einem Bauern die gegenrische Grundlinie erreicht gewinnt.\n");
 printf("Sonst, wenn kein gueltiger Zug mehr moeglich ist, ist unentschieden.\n");
#endif
 printf("Das Eingabeformat ist XN-YM wobei XN die Startposition und YM die Zielposition\n");
 printf("der gezogenen Figur ist. (Das - Zeichen kann auch weggelassen werden)\n");
 printf("Mit Eingabe von '?' kann die momentane Stellung angezeigt werden.\n");
 printf("Bei Rochade wird nur der Koenigszug angegeben.\n");
 printf("Eingabe bei Bauerumwandlung: XNYMZ wobei Z die neue Figur ist.\n");
 printf("Abkuerz. fuer Figuren: T=Turm, L=Laeufer, S=Springer, D=Dame, K=Koenig, B=Bauer.\n");
 printf("Englische Figuren-Abk: R=Turm, B=Laeufer, N=Springer, Q=Dame, K=Koenig, P=Bauer.\n");
}

Zug spielerzug(Brett& brett,uchar farbe)
{
 Zugliste zugliste;
 Zug zug;
 char mzug[80];
 static int8_t startflag=1;
 alle_moeglichen_zuege(brett,farbe,&zugliste,0);
 if(zugliste.istleer()) {zug.von=zug.nach=0; return zug;}//kein Zug mehr moeglich
 for(;;)
  {
   uchar von,nach,i;
   uchar errflag=0;
   //printf("Dein Zug: "); scanf("%s",mzug);
   if(!grafik_scanfzug(mzug,80)) {zug.von=zug.nach=0; return zug;} //Beendet mit exit
   printf("Dein Zug: %s\n",mzug); //test
   if(mzug[0]=='?') {brett.print(); continue;}
   if(mzug[0]=='h' && (mzug[1]=='e' || mzug[1]=='i')) {hilfe(); continue;}
   for(i=0;mzug[i]!=0;i++) if(islower(mzug[i])) mzug[i] -= 'a'-'A';
   if(mzug[0]<'A' || mzug[0]>'H' || mzug[1]<'1' || mzug[1]>'9') errflag=1;
   else
    {
     for(i=2;i<10 && !isalnum(mzug[i]);i++) {}
     if(mzug[i]<'A' || mzug[i]>'H' || mzug[i+1]<'1' || mzug[i+1]>'9') errflag=1;
     else
      {
       von = mzug[0]-'A' + (mzug[1]-'1')*8;
       nach = mzug[i]-'A' + (mzug[i+1]-'1')*8;
       zug.set(von,nach);
       if(isalpha(mzug[i+2])) //Bauerumwandlung?
	{
	 int c=toupper(mzug[i+2]);
	 switch(c)
	  {case 'D': case 'Q': zug.setpromo(DAME); break;
	   case 'S': case 'N': zug.setpromo(SPRINGER); break;
	   case 'L': case 'B': zug.setpromo(LAEUFER); break;
	   case 'T': case 'R': zug.setpromo(TURM); break;
	  }
	}
       if(zugliste.zugsuchen(zug)==NULL) errflag=2; //ungueltig wenn nicht in Liste vorhanden
       else if(zug.von&PROMOFLAG)
	{
	 char fig[80]="Dame";
	 int ok=requester_input(1,"Bauernumwandlung, Figur auswaehlen:","%s","%s",fig);
	 if(ok)
	  {int c=toupper(fig[0]);
	   switch(c)
	    {case 'D': case 'Q': zug.setpromo(DAME); break;
	     case 'S': case 'N': zug.setpromo(SPRINGER); break;
	     case 'L': case 'B': zug.setpromo(LAEUFER); break;
	     case 'T': case 'R': zug.setpromo(TURM); break;
	    }
	  }	  
	}
      }
    }
   if(errflag==0) break;
   if(errflag!=0 && startflag!=0)
    {
     printf("zum Stellung anzeigen: ? eingeben\n");
     printf("sonst Zug eingeben: Starposition-Zielposition\n");
     --startflag;
    }
   if(errflag==1) printf("Eingabefehler - Eingabe-Beispiel: E2-E4 oder ? zum Stellung zeigen\n");
   if(errflag==2) printf("Ungueltiger Zug\n");
  }
 return zug;
}


bool getline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
	*s++ = c;
 *s='\0';
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

void startstellung_setzen(const char *name)
{
 FILE *fp=fopen(name,"r");
 if(fp==NULL) return;
 int i,j,n,c;
 if((c=getc(fp))=='8')
  {
   int c2,k;
   if((c2=getc(fp))==':')
    {//einfaches Format: Zeilen mit 8: bis 1: markiert, Brettdarstellung wie bei Brett.print()
     char zeile[80], *s;
     for(k=8,j=64-8; k>=1 && getline(fp,zeile,80); k--,j-=16)
      {
       if(k==8) s=zeile; else  s = &zeile[2];
       for(i=8;i>0;i--)
	{
	 while((c= *s++)==' ') {}
	 c2= *s++;
	 if(c=='.' || c==':')
	  {s++; n=LEER;}//Leeres Feld
	 else
	  {
	   if(c=='s') n=SCHWARZ; else n=WEISS;
	   if(c2=='B') n += BAUER;
	   else if(c2=='T') n += TURM;
	   else if(c2=='S') n += SPRINGER;
	   else if(c2=='L') n += LAEUFER;
	   else if(c2=='D') n += DAME;
	   else if(c2=='K') n += KOENIG;
	  }
	 spielbrett[j++]=n;
	}
      }
     fclose(fp);
     return;
    }
   else
    {ungetc(c2,fp);}
  }
 ungetc(c,fp);
 for(j=64-8,i=8;j>=0;)
  {
   fscanf(fp,"%d",&n); //provisorisches Format: alle Felder als Zahlen in der Datei
   spielbrett[j++]=n;
   if(--i==0) {j-=16; i=8;}
  }
 fclose(fp);
}

/************************ Grafik-Routinen *****************************/
static double xmin= -90,ymin=0,xmax=990,ymax=900;
const double x0=50,y0=50,dx=100;
static int gbreite=1200,ghoehe=1000,gtiefe=24;
const int hintergrundgrau= 0x808080;
const int weissesfeld    = 0xC0C0C0;
const int schwarzesfeld  = 0x404040;
void maus_gedrueckt();
void maus_losgelassen();
void maus_bewegung();
void figur_zeichnen(int i,int j,int figur);

int scanfzug_von= -1; //wird von maus_gedrueckt() gesetzt
int scanfzug_nach= -1; //wird von maus_losgelassen() gesetzt

bool grafik_scanfzug(char *str,int max) //TODO
{
 scanfzug_von = scanfzug_nach = -1;
 while(exitflag==0 && waitmenu(0)==0 && (scanfzug_von<0 || scanfzug_nach<0))
  {
   if(exitflag) return false;
   waitTOF();
  }
 int i=scanfzug_von%8, j=scanfzug_von/8;
 char vonb = i+'A';
 char vonz = j+'1';
 char nachb = scanfzug_nach%8+'A';
 char nachz = scanfzug_nach/8+'1';
 char promo = ' ';
 if(nachz=='8' && (spielbrett[scanfzug_von]&FIGUR)==BAUER)
  {
   promo='Q'; //TODO
  }
 sprintf(str,"%c%c-%c%c%c",vonb,vonz,nachb,nachz,promo);
 return true;
}

void linie(double x1,double y1,double x2,double y2)
{
 plot(x1,y1,PENUP); plot(x2,y2,PENDOWN);
}

void m_load() {janeinrequester("neue Stellung laden geht noch nicht"); }
void m_save() {janeinrequester("Stellung speichern geht noch nicht"); }
void m_exit() {exitflag=1;}
void m_level() {janeinrequester("Level einstellen geht noch nicht"); }
void m_hilfe()
{
 janeinrequester("Hilfe\n\
Zug machen durch Ziehen einer Figur mit Maus oder Stift auf Touchscreen\n\
vorlaeufig: Zug machen durch Eingabe in Terminal\n\
");
}

void figuren_zeichnen()
{
 int i,j,n;
 for(j=64-8,i=8;j>=0;)
  {
   n=spielbrett[j];
   figur_zeichnen(8-i,j/8,n);
   j++;
   if(--i==0) {j-=16; i=8;}
  }
}

static double buchstabenhoehe=100,buchstabenbreite=50;

void m_refresh()
{
 inital_new();
 screenclear(hintergrundgrau);
 double x2=x0+8*dx,y2=y0+8*dx;
 color(schwarzesfeld);
 int i,j;
 for(i=0;i<8;i+=2) //Schachbrett zeichnen
  {
   for(j=0;j<8;j++)
    {
     if((j&1)==0) fillbox(x0+i*dx,y0+j*dx,x0+(i+1)*dx,y0+(j+1)*dx);
     else if(i<7) fillbox(x0+(i+1)*dx,y0+j*dx,x0+(i+2)*dx,y0+(j+1)*dx);
    }
  }
 color(0);//schwarz
 char c,text[2]; text[1]=0;
 double b=dx/2-buchstabenbreite/2;
 for(c='A',i=0;c<='H';c++,i++) //Beschriftungen
  {
   text[0]=c;
   schrift(x0+i*dx+b,y0-1.5*buchstabenhoehe,text);
  }
 b=dx/2-buchstabenhoehe/2;
 for(c='1',i=0;c<='8';c++,i++) //Beschriftungen
  {
   text[0]=c;
   schrift(x0-buchstabenbreite,y0+i*dx+b,text);
  }
 for(i=0;i<9;i++)
  {
   for(j=0;j<6;j++)
    {
     double x1=x0+i*dx+j*0.5;
     linie(x1,y0,x1,y2);
     double y1=y0+i*dx+j*0.5;
     linie(x0,y1,x2,y1);
    }
  }
 figuren_zeichnen();
 term_refresh();
}

void neuzeichnen_nach_verdeckung() {m_refresh();}

static bool grosse_figuren=false;

void grafikfenster_oeffnen()
{
 int vi;
 getmaxsize(&gbreite,&ghoehe,&gtiefe,&vi);
 if(gtiefe>24) gtiefe=24;
 if(gbreite>2000) grosse_figuren=true;
 if(gbreite>ghoehe*12/10) gbreite=ghoehe*12/10;
 if(gtiefe!=24) printf("TODO: gtiefe=%d\n",gtiefe);//provi.
 setsize(gbreite,ghoehe,gtiefe);
 setmenu(3,"File",  "Play",   "Help");
 setmenu(3,"Load..","Level..","Hilfe",&m_load,&m_level,&m_hilfe);
 setmenu(2,"Save..","Refresh  Y",&m_save,&m_refresh);
 setmenu(1,"Exit",&m_exit);
 set_funktions(maus_gedrueckt,maus_losgelassen,neuzeichnen_nach_verdeckung,maus_bewegung);
 inital(xmin,ymin,xmax,ymax);
 term_refresh();
 buchstabenhoehe=textsize(ghoehe/15,buchstabenbreite=gbreite/40);
 m_refresh();
}

void grafikfenster_schliessen()
{
 term_exit();
}

void maus_gedrueckt()
{
 if(scanfzug_von<0)
  {
   double x,y;
   mausposition(&x,&y);
   if(x>x0 && x<x0+8*dx && y>y0 && y<y0+8*dx)
    {
     int i=(x-x0)/dx;
     int j=(y-y0)/dx;
     scanfzug_von=i+8*j;
    }
  }
}

void maus_losgelassen()
{
 if(scanfzug_nach<0 && scanfzug_von>=0)
  {
   double x,y;
   mausposition(&x,&y);
   if(x>x0 && x<x0+8*dx && y>y0 && y<y0+8*dx)
    {
     int i=(x-x0)/dx;
     int j=(y-y0)/dx;
     scanfzug_nach=i+8*j;
    }
  }
}

void maus_bewegung()
{
}

uchar *bild_einlesen_ppm(const char *name,int *breite,int *hoehe)
{
 FILE *fp=fopen(name,"rb");
 if(fp==NULL) {printf("Fehler: Datei \"%s\" nicht gefunden\n",name); return NULL;}
 int c,c1,c2,ti;
 if((c1=getc(fp))!='P' || (c2=getc(fp))!='6')
    {printf("Fehler: Datei %s kein PPM\n",name); fclose(fp); return NULL;}
 while((c=getc(fp))!='\n' && c!=' ') ;
 while(c=='#') {while(getc(fp)!='\n') ;  c=getc(fp);}
 ungetc(c,fp);
 c=fscanf(fp,"%d %d %d",breite,hoehe,&ti);
 if(c!=3) {printf("Fehler in PPM-Datei\n"); fclose(fp); return NULL;}
 if((c=getc(fp))!='\n' && c!=' ') printf("fehlendes Leerzeichen in PPM\n");
 if(ti!=255 && ti!=65535) printf("falsche Farbtiefe %d in PPM\n",ti);
 int bytes_pro_pixel = 4;
 uchar *pmem = (uchar*)malloc((*breite)*(*hoehe)*bytes_pro_pixel);
 uchar r,g,b;
 if(pmem==NULL)
   {printf("kann keinen Speicher reservieren\n"); fclose(fp); return NULL;}
 uchar *p=pmem;
 //printf("Farbtiefe=%d\n",ti);//test
 if(ti<=255)
  {
   for(int j=0;j<(*hoehe);j++)
    for(int i=0;i<(*breite);i++)
     {
      r=getc(fp); g=getc(fp); b=getc(fp);
      *p++ = r; *p++ = g; *p++ = b; *p++ = 0;
     }
  }
 else //if(ti==65535)
  {
   for(int j=0;j<(*hoehe);j++)
    for(int i=0;i<(*breite);i++)
     {
      r=getc(fp); getc(fp); g=getc(fp); getc(fp); b=getc(fp); getc(fp); 
      *p++ = r; *p++ = g; *p++ = b; *p++ = 0;
     }
  }
 fclose(fp);
 return pmem;
}

void figur_zeichnen(int i,int j,int figur)
{
 double x=x0+i*dx, y=y0+j*dx;
 const char *name=NULL;
 switch(figur&(FIGUR+FARBE))
  {
   case BAUER: name="bauerw"; break;
   case TURM: name="turmw"; break;
   case LAEUFER: name="laeuferw"; break;
   case SPRINGER: name="springerw"; break;
   case DAME: name="damew"; break;
   case KOENIG: name="koenigw"; break;
   case BAUER+SCHWARZ: name="bauers"; break;
   case TURM+SCHWARZ: name="turms"; break;
   case LAEUFER+SCHWARZ: name="laeufers"; break;
   case SPRINGER+SCHWARZ: name="springers"; break;
   case DAME+SCHWARZ: name="dames"; break;
   case KOENIG+SCHWARZ: name="koenigs"; break;
  }
 if(name!=NULL)
  {
   char pfadname[256];
   if(grosse_figuren) sprintf(pfadname,"%s/figuren/%s2.ppm",getenv("PWD"),name);
   else sprintf(pfadname,"%s/figuren/%s.ppm",getenv("PWD"),name);
   int bildbreite,bildhoehe;
   uchar *p, *pmem=bild_einlesen_ppm(pfadname,&bildbreite,&bildhoehe);
   if(pmem==NULL) return;
   //printf("%s eingelesen\n",name);//test
   XImage *image=tek_xcreateimage(bildbreite,bildhoehe);
   uchar r,g,b; int rgb;
   int iy,ix;
   int grau = ((i+j)&1) ? hintergrundgrau : schwarzesfeld;
   for(p=pmem,iy=0; iy<bildhoehe; iy++)
    for(ix=0; ix<bildbreite; ix++)
     {
      r = *p++; g = *p++; b = *p++; p++;
      rgb = (r<<16)+(g<<8)+b;
      //if(ix<5 && iy<5) printf("rgb=0x%X\n",rgb);//test
      if(r>=0x38 && r<=0x78 && g>=0x38 && g<=0x78 && b>=0x38 && b<=0x78)//test
        rgb=grau;//test
      XPutPixel(image, ix, iy, rgb);
     }
   int offsetx,offsety;
   koorduser2pix(x+dx/2,y+dx/2,&offsetx,&offsety); //Pixelposition fuer Feldmitte
   offsetx -= bildbreite/2;
   offsety -= bildhoehe/2;
   tek_xputimage(image,0,0, offsetx, offsety, bildbreite,bildhoehe);
   XDestroyImage(image); //speicher vom image wieder freigeben
   free(pmem); //mit malloc reservierter Speicher freigeben
  }
}

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(const char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
   if(c=='T')
    {
     if((c= *s)!=0 && isdigit(c)) {spielsuchtiefe=c-'0'; s++;}
    }
  }
}

/************************* Hauptprogramm ******************************/
static bool geschlagenflag=false;
//Vordeklarationen:
void fertig(const char *str);
void stockfish_init();
Zug besterzug_stockfish(Brett& brett,Zug& letzterzug,uchar farbe,int suchtiefe);
void fenbrett_aktualisieren(Brett& brett,Zug& letzterzug);
void stockfish_beenden();

int main(int argc,char *argv[])
{
 char quellname[80],zielname[80];
 //FILE *fp1,*fp2;
 int i,j,c;
 quellname[0]=zielname[0]=0;
 if(argc<=0)
  {// es wurde von WorkBench (GUI, Desktop, ...) gestartet
   j=0;
  }
 else
  {// es wurde von Shell (Terminal, Konsole, ...) gestartet
   for(j=0,i=1;i<argc;i++)
	{if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
	 else	{if(++j==1) strcpy(quellname,argv[i]);
		 else if(j==2) strcpy(zielname,argv[i]);
	}	}
  }
 if(argflag['?'] || j>MAXARG)
	{printf("schach  %s\n",VERSION);
	 printf("Anwendung: schach [-Flags] [Quelle] [Ziel]\n");
	 printf(" Flags: t = Suchtiefe setzen (0 bis 9)\n");
	 printf("        w = Computer hat weiss\n");
	 printf("        b = Computer hat schwarz (black)\n");
	 printf("        s = Stockfish verwenden\n");
	 exit(0);
	}
 if(!argflag['T']) spielsuchtiefe=3;//voreingestellte Suchtiefe

 int computerfarbe,spielerfarbe;
 Zug zug; zug.von=zug.nach=0;
 bool stockfish_flag = (argflag['S'] || spielsuchtiefe>=10);
 spielbrett.grundstellung();
 grafikfenster_oeffnen();
 if(quellname[0]!=0) startstellung_setzen(quellname);
 if(!argflag['W'] && !argflag['B'])
  {
   char cf[80]="Black", an[80]="nein";
   char enginversion[80]; if(stockfish_flag) strcpy(enginversion,"Stockfish"); else strcpy(enginversion,"Eigene");
   int stiefe=spielsuchtiefe;
   int ok=requester_input(4,"Computerfarbe:","%s","%s",cf,
			  "Suchtiefe:","%d","%d\n",&stiefe,
			  "Engine-Version: (Eigene oder Stockfish)","%s","%s\n",enginversion,
			  "Analysemodus:","%s","%s",an);
   if(ok)
    {
     if(toupper(an[0])!='N') spielsuchtiefe=stiefe;
     else if(toupper(cf[0])=='W') setargflags("W");
     else setargflags("B");
     if(enginversion[0]=='E' || enginversion[0]=='e') stockfish_flag=false;
     else stockfish_flag=true;
    }
   else
    {
     printf("Computer hat %s, Suchtiefe=%d, engine=%s\n",
	    argflag['B']?"Schwarz":"Weiss", spielsuchtiefe, stockfish_flag?"Stockfish":"Eigene");
    }
  }
 else printf("Suchtiefe=%d  engine=%s\n",spielsuchtiefe,stockfish_flag?"stockfish":"eigene");//test
 if(stockfish_flag) stockfish_init();
 if(argflag['W'])
  {
   computerfarbe=WEISS;
   spielerfarbe=SCHWARZ;
  }
 else if(argflag['B'])
  {
   computerfarbe=SCHWARZ;
   spielerfarbe=WEISS;
  }
 else
  {
   if(stockfish_flag)
    zug=besterzug_stockfish(spielbrett,zug,WEISS,spielsuchtiefe);
   else
    zug=besterzug(spielbrett,WEISS,spielsuchtiefe);
   printf("Bester erster Zug: %s\n",zug.print());
   computerfarbe= -1;//test
  }

 if(computerfarbe>=0)
  {
   if(spielerfarbe==WEISS)
    {
     zug=spielerzug(spielbrett,spielerfarbe);
     zug_wirklich_machen(spielbrett,zug);
     m_refresh();
    }
   for(;;)
    {
     if(spielbrett.gewonnen(spielerfarbe)) {fertig("Computer ist Schachmatt - Du hast gewonnen"); break;}
     if(stockfish_flag)
      zug=besterzug_stockfish(spielbrett,zug,computerfarbe,spielsuchtiefe);
     else
      zug=besterzug(spielbrett,computerfarbe,spielsuchtiefe);
     if(zug.von==zug.nach) {fertig("Computer kann keinen Zug mehr machen - Unentschieden"); break;}
     geschlagenflag=(spielbrett[zug.nach]!=LEER);
     zug_wirklich_machen(spielbrett,zug);
     m_refresh();
     printf("Computer: %s\n",zug.print());
     if(spielbrett.gewonnen(computerfarbe)) {fertig("Schachmatt - Computer hat gewonnen"); break;}
     if(stockfish_flag) fenbrett_aktualisieren(spielbrett,zug);
     zug=spielerzug(spielbrett,spielerfarbe);
     if(exitflag) break;
     if(zug.von==zug.nach) {fertig("Du kannst keinen Zug mehr machen - Unentschieden"); break;}
     geschlagenflag=(spielbrett[zug.nach]!=LEER);
     zug_wirklich_machen(spielbrett,zug);
     m_refresh();
    }
  }

 if(stockfish_flag) stockfish_beenden();
 while(exitflag==0 && waitmenu(1)==0)
  {
  }
 grafikfenster_schliessen();
 return 0;
}// ende von main

void fertig(const char *str)
{
 spielbrett.print();
 printf("%s\n",str);
}

/************************* Kommunikation mit Stockfish ******************************/
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

const int MAXFEN=200;
const int MAXZ=240;
static int fd1=0,fd2=0;
static char zeile[MAXZ];
const char *myfifo1="myfifo1", *myfifo2="myfifo2";

int read1(int fd) //lies 1 Zeichen von einer Pipe
{
 char c;
 read(fd,&c,1);
 return c&0xFF;
}

bool readp(int fd,char *s,int lim)
{		// lies von einer Pipe eine Zeile
		// und ersetzt den Zeilentrenner durch 0
 int c=0;
 while(--lim && (c=read1(fd))!='\n')
	*s++ = c;
 *s='\0';
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

void printp(int fd,const char *s)
{
 write(fd,s,strlen(s));
}

void printp1(int fd,const char *s,int p1)
{
 char str[MAXZ];
 sprintf(str,s,p1);
 write(fd,str,strlen(str));
}

void printp1s(int fd,const char *s,const char *p1)
{
 char str[MAXZ];
 sprintf(str,s,p1);
 write(fd,str,strlen(str));
}

void fifos_init()
{
 mkfifo(myfifo1, 0666);
 mkfifo(myfifo2, 0666);
 int pid;
 if((pid=fork())!=0)
  {
   system("stockfish <myfifo1 >myfifo2");
   exit(0); //Fork beenden
  }
 fd1=open(myfifo1, O_WRONLY);
 fd2=open(myfifo2, O_RDONLY);
 printf("fifos geoeffnet: fd1=%d fd2=%d\n",fd1,fd2);//test
 printp(fd1,"uci\n");
 printf("uci an fd1 gesendet\n");//test
 bool ok=false;
 while(readp(fd2,zeile,200))
  {
   if(strncmp(zeile,"uciok",5)==0) {ok=true; break;}
   //fprintf(stderr,"Stockfish: %s\n",zeile);//test
  }
 if(!ok)
  {
   fprintf(stderr,"Error: kein uciok empfangen.\n");
   stockfish_beenden();
  }
}

class Zugstock
{
public:
 uchar von,nach;
 uchar promotfig;
 void set(uchar from,uchar to) {von=from; nach=to; promotfig=0;}
 const char *print();
};

const char *Zugstock::print()
{
 static char zugtext[6];
 zugtext[0] = 'A'+(von%8);  zugtext[1] = '1'+(von/8);
 //zugtext[2] = '-'; 
 zugtext[2] = 'A'+(nach%8); zugtext[3] = '1'+(nach/8);
 char c;
 switch(promotfig) //fuer Umwandlung von Bauer
  {case DAME:     c='q'; break; //Queen
   case SPRINGER: c='n'; break; //kNight
   case TURM:     c='r'; break; //Rook
   case LAEUFER:  c='b'; break; //Bishop
   default: c=0;
  }
 zugtext[4] = c;
 zugtext[5] = 0;
 return zugtext;
}

class Fen
{
public:
 char fen[MAXFEN];
 int halbzug,zugnummer;
 void grundstellung();
 //void fen2brett();
 void brett2fen(Brett& brett,Zug& letzterzug);
 //void print() {fen2brett(); brett.print();}
 Zugstock besterzug(int farbe,int suchtiefe);
 Zugstock spielerzug(int farbe);
};
static Fen fenbrett;

void Fen::grundstellung()
{
 strcpy(fen,"rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1");
 halbzug=0,zugnummer=1;
 printp(fd1,"position startpos\n");
}

char fig_eng_abk(uchar u)
{
 char c = u&FIGUR;
 switch(c)
  {case DAME:     c='q'; break; //Queen
   case SPRINGER: c='n'; break; //kNight
   case TURM:     c='r'; break; //Rook
   case LAEUFER:  c='b'; break; //Bishop
   case KOENIG:   c='k'; break; //Bishop
   case BAUER:    c='p'; break; //Bishop
   default: c='1';
  }
 if((u&FARBE)==WEISS) c -= 'a'-'A';
 return c;
}

void Fen::brett2fen(Brett& brett,Zug& letzterzug)
{
 int i,j,k,zahl;
 uchar c;
 char *f=fen;
 uchar farbeamzug;
 if(letzterzug.nach==letzterzug.von) farbeamzug=WEISS;
 else
  {
   uchar fig=(brett[letzterzug.nach]&FIGUR);
   farbeamzug = WEISS+SCHWARZ-(brett[letzterzug.nach]&FARBE);
   if(fig==BAUER || geschlagenflag) halbzug=0; //provi.
   else halbzug++;
   if(farbeamzug==WEISS) zugnummer++;
  }
 // Beispiel Grundstellung:
// "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1"
 for(j=64-8;j>=0;)
  {
   for(zahl=0,k=0;k<8;k++)
    {
     c=brett[j++];
     if(c==LEER) zahl++;
     else
      {if(zahl!=0) {*f++ = zahl+'0'; zahl=0;}
       *f++ = fig_eng_abk(c);
      }
    }
   if(zahl!=0) *f++ = zahl+'0';
   if((j-=16)>=0) *f++ = '/';
  }
 *f++ = ' ';
 *f++ = (farbeamzug==WEISS)?'w':'b';
 *f++ = ' ';
 bool flag=false;
 if(brett[4]==KOENIG+WEISS+MOVFLAG)
  {
   if(brett[7]==TURM+WEISS+MOVFLAG) {*f++ = 'K'; flag=true;}
   if(brett[0]==TURM+WEISS+MOVFLAG) {*f++ = 'Q'; flag=true;}
  }
 if(brett[60]==KOENIG+SCHWARZ+MOVFLAG)
  {
   if(brett[63]==TURM+SCHWARZ+MOVFLAG) {*f++ = 'k'; flag=true;}
   if(brett[56]==TURM+SCHWARZ+MOVFLAG) {*f++ = 'q'; flag=true;}
  }
 if(flag==false) *f++ = '-';
 *f++ = ' ';
 flag=false;
 c='-';
 for(j=24,i=0;i<8;j++,i++)
  if((brett[j]&(FIGUR|ENPASSFLAG))==(BAUER|ENPASSFLAG)) {*f++ = 'a'+i%8; *f++ = '3'; flag=true;}
 if(flag==false)
  for(j=32,i=0;i<8;j++,i++)
   if((brett[j]&(FIGUR|ENPASSFLAG))==(BAUER|ENPASSFLAG)) {*f++ = 'a'+i%8; *f++ = '6'; flag=true;}
 if(flag==false) *f++ = '-';
 *f++ = ' ';
 sprintf(f,"%d %d",halbzug,zugnummer);
 printf("fen='%s'\n",fen);//test
}

Zugstock Fen::besterzug(int farbe,int suchtiefe)
{
 Zugstock zug;
 printp1s(fd1,"position fen %s\n",fen);
 printp1(fd1,"go depth %d\n",suchtiefe);
 while(readp(fd2,zeile,200))
    {
     //fprintf(stderr,"Stockfish: %s\n",zeile);//test
     if(strncmp(zeile,"bestmove",8)==0)
      {
       printf("Bester Zug: %s\n",&zeile[9]);//test
       break;
      }
    }
 const char *p = &zeile[9];
 int c1= *p++;
 int c2= *p++;
 if(c1>='a' && c1<='h') c1 -= 'a'-'A';
 if(c1<'A' || c1>'H') printf("Fehler c1='%c'\n",c1);//test
 if(c2<'1' || c2>'8') printf("Fehler c2='%c'\n",c2);//test
 zug.von = c1-'A'+(c2-'1')*8;
 c1= *p++;
 c2= *p++;
 if(c1>='a' && c1<='h') c1 -= 'a'-'A';
 if(c1<'A' || c1>'H') printf("Fehler c1='%c'\n",c1);//test
 if(c2<'1' || c2>'8') printf("Fehler c2='%c'\n",c2);//test
 zug.nach = c1-'A'+(c2-'1')*8;
 if((c1= *p++)>='A')
  {uchar c=0;
   if(c1>='A' && c1<='Z') c1 += 'a'-'A';
   switch(c1)
    {case 'q': c=DAME; break; //Queen
     case 'n': c=SPRINGER; break; //kNight
     case 'r': c=TURM; break; //Rook
     case 'b': c=LAEUFER; break; //Bishop
    }
   zug.promotfig=c;
  }
 else zug.promotfig=0;
 return zug;
}

void stockfish_init()
{
 fenbrett.grundstellung();
 fifos_init();
}

Zug besterzug_stockfish(Brett& brett,Zug& letzterzug,uchar farbe,int suchtiefe)
{
 Zugliste zugliste;
 Zug zug;
 alle_moeglichen_zuege(brett,farbe,&zugliste,suchtiefe);
 if(zugliste.istleer()) {zug.set(0,0); return zug;}
 fenbrett.brett2fen(brett,letzterzug);
 Zugstock zugst=fenbrett.besterzug(farbe,suchtiefe);
 zug.set(zugst.von,zugst.nach);
 if(zugst.promotfig!=0) zug.setpromo(zugst.promotfig);
 return zug;
}

void fenbrett_aktualisieren(Brett& brett,Zug& letzterzug)
{
 fenbrett.brett2fen(brett,letzterzug);
}

void stockfish_beenden()
{
 printp(fd1,"quit\n");
 close(fd1);
 close(fd2);
}
