/* robot.cc       letzte Aenderung: 27.2.2023
Programm um Schrittmotoren des Schachroboters anzusteuern
*/
#define VERSION "0.2"
/*
 Prozessor: Atmega1284P, 20MHz Quarz
 Schaltung: schaltung/robot.cc

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 1.1.2023       Erstellung
23.2.2023  0.2  einstellbare Brettlage

*/
#define F_CPU 20000000UL
#define XSENSORPOS 0  //auf 0 lassen und A1XPOS anpassen
#define YSENSORPOS 0  //auf 0 lassen und A1YPOS anpassen
#define ZSENSORPOS 20
//#define MOT3_NIE_AUS  //Motor fuer z-Richtung nie ganz ausschalten
//#define AUTOCALIB 10  //nach jedem sovielten Zug neu calibrieren
//#define GREIFER_LANGSAMER  2  //test mit 20 fuer sehr langsam

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
//#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>

#define KIPPFAKTOR 50L //Korrektur fuer Neigung wenn Arm weit ausgefahren
#define ZEILE1KORR  2  //Korrektur von y in mm wegen leichtem Kippen nach hinten auf erster Zeile
#define DIAGONAL_MOVES //auskommentieren wenn Mechanik Diagonal-Fahrt nicht kann
//#define DIAGONALTEST
//#define KIPPTEST  //D1-D8 und zurueck zum Kippfaktor testen
//#define CALIBTEST

#define DEBUG  //zur Fehlersuche
//#define TEST1  //erster Test: nur x-Richtung
//#define TEST2  //Teste nur y-Richtung
//#define TEST3  //auf Eck-Positionen fahren
//#define TEST4  //Teste nur Motor4 (Greifer)
//#define TEST5  //nacheinander alle Bewegungen testen
//#define TEST6  //UART testen

/**** Vordeklarationen ****/
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t
void schrittmotoren_steps();

/**** Zeitbehandlung ****/
// von nimm.cc uebernommen
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=0;
static volatile uchar tage=25,monat=2; //aktuelles Datum
static volatile uint jahr=2023;
static char monatstage[13]={31,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=5; //0=Montag ... 6=Sonntag
//static volatile char sommerzeit=1; //0=Winterzeit 1=Sommerzeit

void schaltjahrtest()
{
 if(jahr%4==0 && (jahr%100!=0 || jahr%400==0)) monatstage[2]=29;
 else monatstage[2]=28;
}
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
void schaltjahr_und_wochentagberechnen()
{
 schaltjahrtest();
 wochentagberechnen();
}

// fuer genaue Uhr: Quarz moeglichst genau ausmessen und hier eintragen:
const ulong Hz=20000000; //exakt 20MHz

#define Rest1 (Hz%1000) //jede Sekunde zu korrigierender Fehler
#define SCHNELLER 2  //zum Schrittmotoren schneller laufen lassen
#ifdef SCHNELLER
#define NormalerCompwert (Hz/(1000*SCHNELLER)-1)
#else
#define NormalerCompwert (Hz/1000-1)
#endif

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
 set_sleep_mode(SLEEP_MODE_IDLE);
 sleep_enable(); //Schlaf-Modus waehlen, aber noch nicht schlafen
}

ISR(TIMER1_COMPA_vect)
{
 schrittmotoren_steps();
#ifdef SCHNELLER
 static uchar j=0;
 if(++j==SCHNELLER)  j=0;
 else return;
#endif
 uchar mehr;
 uint ms=millisec;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0; mehr=0;
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
       /*
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
       */
      }
    }
  }
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
 //tastaturcheck();
}

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) {sleep_mode();}//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
  {sleep_mode();}//auf n. Millisekunde warten
}

void secwait(uchar n) //n Sekunden warten
{
 while(n!=0) {milliwait(1000); --n;}
}

void microwait(uchar n) //mindestens n Microsekunden warten
{
 while(--n!=0) _delay_us(1);
}

/************************** Serielle Schnittstelle **************************/
#define TXUNDRX  //senden und empfangen, auskommentieren wenn nur empfangen
#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 20000000"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 9600UL      // Baudrate

// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart1_init()
{
 UBRR1 = UBRR_VAL;
#ifdef TXUNDRX
 UCSR1B = (1<<RXEN1)|(1<<TXEN1); // UART TX und RX einschalten
#else
 UCSR1B = (1<<RXEN1); // nur UART RX einschalten
#endif
 UCSR1C = (3<<UCSZ10); // Asynchron 8N1
 // UCSR1C = (3<<UCSZ10)|(1<<USBS1); // Asynchron 8N2
}

#ifdef TXUNDRX
void uart_putc(unsigned char c)
{
 while (!(UCSR1A & (1<<UDRE1))) {milliwait(1);}// warten bis Senden moeglich
 UDR1 = c;                      // sende Zeichen
 milliwait(3);//test
}

void uart_print(const char *text)
{
 for(int i=0; text[i] != 0; i++)
  {
   uart_putc(text[i]);
  }
}
#endif

unsigned char uart_getc(void)
{
 while(!(UCSR1A & (1<<RXC1))) {milliwait(1);}// warten bis Zeichen verfuegbar
 return UDR1; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart_check()
{
 return (UCSR1A & (1<<RXC1)) ;//true wenn Zeichen verfuegbar
}

char* uart_zeile_empfangen()
{
 static char zeile[32];
 uint8_t i;
 char c=' ';
 for(i=0;i<31 && c!=0;)
  {
   c=uart_getc();
   if(c=='\r') continue; //CR ignorieren
   if(c=='\n') c=0; //mit LF beenden
   zeile[i++]=c;
  }
 zeile[i]=0;
 return zeile;
}

/***************************** Port-Belegungen *******************************/
#define LED_PORT PORTB
#define LED_DDR  DDRB
#define LED0     PB5
#define LED1     PB4
#define LED0_ON  LED_PORT |= (1<<LED0)
#define LED0_OFF LED_PORT &= ~(1<<LED0)
#define LED1_ON  LED_PORT |= (1<<LED1)
#define LED1_OFF LED_PORT &= ~(1<<LED1)

#define MOT1_PORT PORTA
#define MOT1_DDR  DDRA
#define MOT1_PIN  PINA
#define MOT1_SEL  4
#define MOT1_DIR  5
#define MOT1_STEP 6
#define MOT1_SENS 7
#define MOT1_SENS2 0 //bisher unbenutzt
#define MOT1_SENS3 1 //bisher unbenutzt
// 0-3 fuer ADC0-3 falls noch A/D-Wandlung gebraucht

#define MOT2_PORT PORTB
#define MOT2_DDR  DDRB
#define MOT2_PIN  PINB
#define MOT2_SEL  0
#define MOT2_DIR  1
#define MOT2_STEP 2
#define MOT2_SENS 3
// B5-B6 sind fuer Programmer belegt

#define MOT3_PORT PORTC
#define MOT3_DDR  DDRC
#define MOT3_PIN  PINC
#define MOT3_SEL  2
#define MOT3_DIR  3
#define MOT3_STEP 4
#define MOT3_SENS 5
// C0-C1  SCL und SDA falls noch I2C gebraucht wir

#define MOT4_PORT PORTD
#define MOT4_DDR  DDRD
#define MOT4_PIN  PIND
#define MOT4_SEL  4
#define MOT4_DIR  5
#define MOT4_STEP 6
#define MOT4_SENS 7
// D0-D1 reserviert fuer UART0
// D2-D3 verwendet fuer UART1  //koennten ev. auch fuer INT0 INT1 Interrupts verwendet werden

#define TAST_PORT PORTC
#define TASTE1    6 //PCINT22
#define TASTE2    7 //PCINT23

void motor_onoff(uchar nr,bool on)
{
 uchar flags = (nr==0 || nr>4) ? 0x0F : (1<<(nr-1));
 if(flags&0x08)
  {if(on) MOT4_PORT &= ~(1<<MOT4_SEL);
   else   MOT4_PORT |= (1<<MOT4_SEL);
  }
 if(flags&0x04)
  {if(on) MOT3_PORT &= ~(1<<MOT3_SEL);
#ifndef MOT3_NIE_AUS
   else   MOT3_PORT |= (1<<MOT3_SEL);
#endif
  }
 if(flags&0x02)
  {if(on) MOT2_PORT &= ~(1<<MOT2_SEL);
   else   MOT2_PORT |= (1<<MOT2_SEL);
  }
 if(flags&1)
  {if(on) MOT1_PORT &= ~(1<<MOT1_SEL);
   else   MOT1_PORT |= (1<<MOT1_SEL);
  }
}

/********************************** PCINT ************************************/
volatile uint8_t got_pcint=0;
volatile bool notstopp=false;

ISR(PCINT2_vect)
{
 if((PINC&(1<<6))==0)        {got_pcint = 22; notstopp=true;} //Taste1 gedrueckt
 else if((PINC&(1<<7))==0) {got_pcint = 23; notstopp=true;} //Taste2 gedrueckt
 //TODO: vorlaeufig beide Tasten fuer Notstopp verwendet
}

void pcint_init()
{
 TAST_PORT |= (1<<TASTE1); //Pullup setzen
 TAST_PORT |= (1<<TASTE2); //Pullup setzen
 PCMSK2 = (1<<PCINT22)|(1<<PCINT23); //Auswahl der aktiven PC-Interrupts
 PCICR = (1<<PCIE2); //PCINT23 bis PCINT16 ermoeglichen
}

/******************************* Hauptprogramm *******************************/
static uint xposition=0, yposition=0, zposition=0; //aktuelle Position in Steps
static volatile int steps1=0, steps2=0, steps3=0, steps4=0; //zu fahrende Schritte
static volatile uchar mot3langsam=0; //fuer in Diagonalen zu fahren

#define STEPS_PRO_METER 40000L
#define FELDGROESSE_IN_MM 38
#define STEPS_PRO_FELD (FELDGROESSE_IN_MM*STEPS_PRO_METER/1000)
//#define A1XPOS_MM 75 //x-Position von Mitte des A1 Feldes
#define A1XPOS_MM 70 //x-Position von Mitte des A1 Feldes
#define A1YPOS_MM  6 //y-Position von Mitte des A1 Feldes
#define A1XPOS_STEPS (STEPS_PRO_METER*A1XPOS_MM/1000)
#define A1YPOS_STEPS (STEPS_PRO_METER*A1YPOS_MM/1000)
#define CM  (STEPS_PRO_METER/100)
#define MM  (STEPS_PRO_METER/1000)
#define CM9 (9*CM) //Greifer ueber hoechsten Figuren

#define sensorcheck(pin,bit) ((pin&bit)==0) //wenn Sensoren Lowaktiv
//#define sensorcheck(pin,bit) ((pin&bit)!=0) //wenn Sensoren Highaktiv

void fehlermeldung(int n,int s) //n mal pro Sekunde blinken, s Sekunden lang
{
 int wartezeit=500/n;
 motor_onoff(0,false); //alle Motoren aus
 s *= n;
 for(int i=0;i<s;i++)
  {
   LED0_ON;
   milliwait(wartezeit);
   LED0_OFF; //schnell Blinkende LED zur Fehleranzeige
   milliwait(wartezeit);
  }
}

void schrittmotoren_steps() //von Interrupt-Routine aufzurufen
{
 if(notstopp)
  {
   MOT1_PORT |= (1<<MOT1_SEL); MOT2_PORT |= (1<<MOT2_SEL);
   MOT3_PORT |= (1<<MOT3_SEL); MOT4_PORT |= (1<<MOT4_SEL); //alle Motoren aus
   steps1=steps2=steps3=steps4=0;
   return;
  }
 if(steps1!=0)
  {
   if(steps1<0)
    {MOT1_PORT |= (1<<MOT1_DIR);
     //auf Sensor am x-Anschlag pruefen:
     if(steps2<=0 && sensorcheck(MOT1_PIN,(1<<MOT1_SENS))) {steps1=0; steps2=0;}
     else  {steps1++; MOT1_PORT |= (1<<MOT1_STEP);}
    }
   else
    {MOT1_PORT &= ~(1<<MOT1_DIR);
     steps1--; MOT1_PORT |= (1<<MOT1_STEP);
    }
   microwait(10);
   MOT1_PORT &= ~(1<<MOT1_STEP);
  }
 if(steps2!=0)
  {
   if(steps2>0)
    {MOT2_PORT &= ~(1<<MOT2_DIR);
     //auf Sensor am y-Anschlag pruefen:
     if(steps1<=0 && sensorcheck(MOT2_PIN,(1<<MOT2_SENS))) {steps2=0; steps1=steps3=0;}
     else  {steps2--; MOT2_PORT |= (1<<MOT2_STEP);}
    }
   else
    {MOT2_PORT |= (1<<MOT2_DIR);
     steps2++; MOT2_PORT |= (1<<MOT2_STEP);
    }
   microwait(10);
   MOT2_PORT &= ~(1<<MOT2_STEP);
  }
 static uchar mot3takt=0;
 if(steps3!=0 && (mot3langsam==0 || ++mot3takt==2))
  {
   mot3takt=0;
   if(steps3<0)
    {MOT3_PORT |= (1<<MOT3_DIR);
     //auf Sensor am z-Anschlag pruefen:
     if(sensorcheck(MOT3_PIN,(1<<MOT3_SENS))) {steps3=0; steps1=steps2=0;}
     else  {steps3++; MOT3_PORT |= (1<<MOT3_STEP);}
    }
   else
    {MOT3_PORT &= ~(1<<MOT3_DIR);
     steps3--; MOT3_PORT |= (1<<MOT3_STEP);
    }
   microwait(10);
   MOT3_PORT &= ~(1<<MOT3_STEP);
  }
#ifdef GREIFER_LANGSAMER
 static uchar mot4takt=0; //Greifer-Motor etwas langsamer
 if(steps4!=0 && ++mot4takt==GREIFER_LANGSAMER)
  {
   mot4takt=0;
#else
 if(steps4!=0)
  {
#endif
   if(steps4<0)
    {MOT4_PORT |= (1<<MOT4_DIR);
     //auf Sensor fuer "Greifer zu" pruefen:
     //if(sensorcheck(MOT4_PIN, (1<<MOT4_SENS))) {steps4=0;}
     static uint8_t sens4flag=0; //erst beim zweiten Mal den Sensor erkennen
     if(sensorcheck(MOT4_PIN, (1<<MOT4_SENS)))
      {
       //steps4=0;
       if(++sens4flag==2) {sens4flag=0; steps4=0;}
      }
     else {steps4++; MOT4_PORT |= (1<<MOT4_STEP); sens4flag=0;}
    }
   else
    {MOT4_PORT &= ~(1<<MOT4_DIR);
     steps4--; MOT4_PORT |= (1<<MOT4_STEP);
    }
   microwait(10);
   MOT4_PORT &= ~(1<<MOT4_STEP);
  }
}

void motorwait()
{
 while(steps1!=0 || steps2!=0 || steps3!=0 || steps4!=0)
  {milliwait(10);}
}

void movemot1(int steps)
{
 cli(); steps1 += steps; sei();
}
void movemot2(int steps)
{
 cli(); steps2 += steps; sei();
}
void movemot3(int steps)
{
 MOT3_PORT &= ~(1<<MOT3_SEL); //Motor3 ein
 cli(); steps3 += steps; sei();
}
void movemot4(int steps)
{
 MOT4_PORT &= ~(1<<MOT4_SEL); //Motor4 ein
 cli(); steps4 += steps; sei();
}

void movex(int steps)
{
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 cli(); steps1 += steps; steps2 += steps; sei();
 xposition += steps;
}

void movey(int steps)
{
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 MOT3_PORT &= ~(1<<MOT3_SEL); //Motor3 ein
 cli(); steps1 += steps; //movemot1(steps);
 steps2 -= steps;        //movemot2(-steps);
 steps3 -= steps; sei(); //movemot3(-steps);
 yposition += steps;
}

void movez(int steps)
{
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 movemot3(steps);
 zposition += steps;
}

void movexy(int xsteps,int ysteps)
{
 xsteps -= xposition;
 ysteps -= yposition;
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 MOT3_PORT &= ~(1<<MOT3_SEL); //Motor3 ein
#ifdef DIAGONAL_MOVES
 mot3langsam=1;
 if(xsteps>0 && ysteps>0)
  {
   int h = (xsteps<ysteps) ? xsteps : ysteps; //h=kleinerer Wert
   cli(); steps1 = h*2;
           steps3 = -h; sei();
   motorwait();
   xsteps -= h; ysteps -= h;
   xposition += h; yposition += h;
  }
 else if(xsteps<0 && ysteps<0)
  {
   int h = (-xsteps < -ysteps) ? -xsteps : -ysteps; //h=kleinerer absoluter Wert
   cli(); steps1 = -h*2;
           steps3 = h; sei();
   motorwait();
   xsteps += h; ysteps += h;
   xposition -= h; yposition -= h;
  }
 else if(xsteps>0 && ysteps<0)
  {
   int h = (xsteps < -ysteps) ? xsteps : -ysteps; //h=kleinerer absoluter Wert
   cli(); steps2 = h*2;
           steps3 = h; sei();
   motorwait();
   xsteps -= h; ysteps += h;
   xposition += h; yposition -= h;
  }
 else if(xsteps<0 && ysteps>0)
  {
   int h = (-xsteps < ysteps) ? -xsteps : ysteps; //h=kleinerer absoluter Wert
   cli(); steps2 = -h*2;
           steps3 = -h; sei();
   motorwait();
   xsteps += h; ysteps -= h;
   xposition -= h; yposition += h;
  }
 mot3langsam=0;
#endif  //DIAGONAL_MOVES
 if(xsteps!=0) {movex(xsteps); motorwait();}
 if(ysteps!=0) {movey(ysteps); motorwait();}
}

void movezabs(int steps)
{
 steps += KIPPFAKTOR*yposition/1000;
 steps -= zposition;
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 movemot3(steps);
 zposition += steps;
 motorwait(); 
}

uint8_t greiferstatus=0; //1=zu, 2=auf, 0=ausgeschaltet

void greifer_zu()
{
 movemot4(-800); //maximal 1/2 Umdrehung bis Sensor anspricht
 motorwait();
 greiferstatus=1;
 milliwait(200);//test
}

void greifer_auf(int ns=110) //ns = Anzahl Schritte, Defaultwert  (200) ev. anpassen
{
 if(greiferstatus==0) greifer_zu();
 //milliwait(500);//testgreifer
 movemot4(ns);
 motorwait();
 greiferstatus=2;
 //milliwait(100);//test
}

void greifer_off()
{
 motor_onoff(4,false);
 greiferstatus=0;
}

static uint8_t brettlage=1; //SCHWARZ; //schwarze Seite zum Spieler zeigend
//static char figurabk[7]={' ','P','N','B','R','Q','K'}; //englische Figurenabkuerzungen
static char aktfigur=0;

uint letter_xwert(char c)
{
 c = toupper(c);
 if(c<'A' || c>'I') c='A'; //bei Fehler auf gueltigen Wert setzen
 uint x = c-'A';
 if(brettlage==0 && c != 'I')  x = 7-x; //wenn weisse Seite beim Spieler
 //uint x = (toupper(c)-'A')*STEPS_PRO_FELD + A1XPOS_STEPS;
 x = x*STEPS_PRO_FELD + A1XPOS_STEPS;
 //if(c=='I') x += 4*MM;//TODO: Position von geschlagenen Figuren anpassen
 return x;
}

uint ziffer_ywert(char c)
{
 if(c<'1' || c>'9') c='1'; //bei Fehler auf gueltigen Wert setzen
 c -= '1';
 if(brettlage==0)  c = 7-c; //wenn weisse Seite beim Spieler
 uint y = c*STEPS_PRO_FELD + A1YPOS_STEPS;
 if(c==0) y -= ZEILE1KORR*MM;
 return y;
}

void figur_vom_brett_nehmen(uint x,uint y)
{
 movexy(x, y);
 //Figur erfassen:
 #define WEITERAUF 30
 greifer_auf(); greifer_auf(WEITERAUF);//erst mal weit oeffnen
 uint z = 30*MM; //bis auf Hoehe Bauernkopf fahren
 movezabs(z);
 greifer_auf(-80); //Greifer etwas wieter zu zum Figur positionieren
  z = 20*MM; //bis knapp ueber Sockel fahren
 movezabs(z);
 greifer_auf(WEITERAUF); //dann wieder ganz oeffnen
 z = 2*MM; //knapp ueber dem Brett
 movezabs(z);
 milliwait(500);//test: warten bis fertig gewackelt
 greifer_zu();
 z = CM9; //9cm ueber dem Brett
 movezabs(z); //entsprechend nach oben fahren
}

void figur_abstellen(uint x,uint y)
{
 movexy(x,y);
 uint z = 2*MM; //2mm ueber dem Brett
 movezabs(z); //entsprechend nach unten fahren
 greifer_auf();
 z = CM9;
 movezabs(z); //entsprechend nach oben fahren
 greifer_off();
}

void figur_abstellen2(uint x,uint y) //beim Abstellen geschlagener Figur: etwas hoeher loslassn
{
 movexy(x,y);
 uint z = 7*MM;
 movezabs(z);
 movexy(x+12*MM,y); //ev. vorher geschlagene Figur wegschieben, TODO: groesserer Wert wenn Mechanisch moeglich
 greifer_auf();
 z = CM9;
 movezabs(z);
 greifer_off();
}

void figur_ziehen(const char* zeile)
{
 uint8_t i=0;
 uint x = letter_xwert(zeile[i++]);
 uint y = ziffer_ywert(zeile[i++]);
 figur_vom_brett_nehmen(x,y);
 if(zeile[i]=='-' || zeile[i]=='x' || zeile[i]=='X') i++;
 x = letter_xwert(zeile[i++]);
 y = ziffer_ywert(zeile[i++]);
 figur_abstellen(x,y);
}

void calibrieren(int ywert=2*CM)
{
 if(ywert!=0) {movey(ywert); motorwait();}
 movex(-32767); motorwait();
 movey(-32767); motorwait();
 greifer_zu();
 movez(-32767); motorwait();
 xposition=XSENSORPOS*MM; yposition=YSENSORPOS*MM; zposition=ZSENSORPOS*MM;
 //static int test=1000; if(test) {milliwait(test); test-=500;}//test
 milliwait(500);//test
 movez(CM9-zposition); //auf 9cm ueber dem Brett nach oben fahren
 motorwait();
 greifer_auf();
 greifer_off();
 motor_onoff(0,false); //alle Motoren aus
}

#define WEISS 0
#define SCHWARZ 1

int main(void)
{
 uint8_t computerfarbe=WEISS;
 uint x,y;
 LED_DDR = (1<<LED0) | (1<<LED1);
 MOT1_DDR |= (1<<MOT1_SEL)|(1<<MOT1_DIR)|(1<<MOT1_STEP); //Ausgaenge setzen
 MOT2_DDR |= (1<<MOT2_SEL)|(1<<MOT2_DIR)|(1<<MOT2_STEP); //Ausgaenge setzen
 MOT3_DDR |= (1<<MOT3_SEL)|(1<<MOT3_DIR)|(1<<MOT3_STEP); //Ausgaenge setzen
 MOT4_DDR |= (1<<MOT4_SEL)|(1<<MOT4_DIR)|(1<<MOT4_STEP); //Ausgaenge setzen
 MOT1_PORT |= (1<<MOT1_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT2_PORT |= (1<<MOT2_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT3_PORT |= (1<<MOT3_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT4_PORT |= (1<<MOT4_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT1_PORT |= (1<<MOT1_SENS); //Pullup fuer Sensor
 MOT2_PORT |= (1<<MOT2_SENS); //Pullup fuer Sensor
 MOT3_PORT |= (1<<MOT3_SENS); //Pullup fuer Sensor
 MOT4_PORT |= (1<<MOT4_SENS); //Pullup fuer Sensor
 MOT1_PORT |= (1<<MOT1_SENS2); //Pullup fuer zusaetzlichen Sensor //TODO
 MOT1_PORT |= (1<<MOT1_SENS3); //Pullup fuer zusaetzlichen Sensor //TODO

 timer1_init(); //Uhr und Schrittmotorentakt
 sei(); //Interrupts einschalten
 uart1_init(); //Serielle Schnittstelle
 secwait(1);
 pcint_init(); //fuer Tasten-Interrupts

#ifdef TEST1
 #define TEST1BIS4
 LED0_ON;
 movex(1600); //eine Umdrehung
 motorwait();
 LED0_OFF;
 secwait(1);
 LED0_ON;
 movex(-32767); //andere Richtung bis Sensor anspricht
 motorwait(); secwait(1);
 movex(STEPS_PRO_METER*25/100); //25cm nach links fahren
 motorwait(); LED0_OFF; 
#endif

#ifdef TEST2
 #define TEST1BIS4
 LED1_ON;
 movey(1600); //eine Umdrehung
 motorwait();
 LED1_OFF;
 secwait(1);
 LED1_ON;
 movey(-32767); //andere Richtung bis Sensor anspricht
 motorwait(); secwait(1);
 movey(STEPS_PRO_METER*25/100); //25cm nach vorn fahren
 motorwait(); LED1_OFF;
#endif

#ifdef TEST4
 #define TEST1BIS4
 greifer_auf(); motorwait();
 for(int i=0;i<10;i++)
  {
   LED1_ON;
   greifer_zu(); motorwait();
   secwait(5);
   LED1_OFF;
   greifer_auf(); motorwait();
   secwait(5);
  }
#endif

#ifdef TEST5
 #define TEST1BIS4
 LED1_ON;
 movex(1600*4); motorwait();
 LED1_OFF;
 secwait(5);
 LED1_ON;
 movey(1600*4); motorwait();
 LED1_OFF;
 secwait(5);
 LED1_ON;
 movez(1600*4); motorwait();
 LED1_OFF;
 secwait(5);

 greifer_auf();
 LED1_OFF;
 secwait(1);
 LED1_ON;
 greifer_zu(); //max 1 Umdrehung bis Sensor anspricht
 motorwait(); secwait(5);
 greifer_auf();
 motorwait(); LED1_OFF;
#endif

 //auf Startposition fahren:
 movez(7*CM); motorwait();
 movey(2*CM); motorwait();
 movex(2*CM); motorwait();
 calibrieren();

#ifdef TEST6
 movez(4*CM-zposition); //auf 4cm ueber dem Brett nach oben fahren
 motorwait();
 motor_onoff(0,false); //alle Motoren aus
 for(;;)
  {
   uart_putc('a');
   //uart_getc();
    secwait(1);//test
  }
#endif
 
#ifdef TEST3
 #define TEST1BIS4
 LED1_ON;
 movez(8*CM-zposition); //auf 8cm ueber dem Brett nach oben fahren
 LED1_OFF;
 secwait(1);
 x = letter_xwert('A'); //nach A1 fahren
 y = ziffer_ywert('1');
 LED1_ON;
 movex(x-xposition); motorwait();
 movey(y-yposition); motorwait();
 LED1_OFF;
 secwait(10);
 x = letter_xwert('H'); //nach H1 fahren
 y = ziffer_ywert('1');
 LED1_ON;
 movex(x-xposition); motorwait();
 movey(y-yposition); motorwait();
 LED1_OFF;
 secwait(10);
 x = letter_xwert('H'); //nach H8 fahren
 y = ziffer_ywert('8');
 LED1_ON;
 movex(x-xposition); motorwait();
 movey(y-yposition); motorwait();
 LED1_OFF;
 secwait(10);
 x = letter_xwert('A'); //nach A8 fahren
 y = ziffer_ywert('8');
 LED1_ON;
 movex(x-xposition); motorwait();
 movey(y-yposition); motorwait();
 LED1_OFF;
 secwait(10);
 x = letter_xwert('A'); //nach A1 fahren
 y = ziffer_ywert('1');
 LED1_ON;
 movex(x-xposition); motorwait();
 movey(y-yposition); motorwait();
 LED1_OFF;
#endif

#ifdef TEST1BIS4
 //motor_onoff(0,true); secwait(5); //test: alle Motoren ein
 motor_onoff(0,false); //alle Motoren aus
 for(;;)
  {
   if(sec&1) LED0_ON;
   else      LED0_OFF; //Blinkende LED wenn fertig
  }
#endif

#ifdef DIAGONALTEST
 //movex(A1XPOS_STEPS+2*STEPS_PRO_FELD - xposition); motorwait();
 //movey(A1YPOS_STEPS+2*STEPS_PRO_FELD - yposition); motorwait();
 movexy(A1XPOS_STEPS+2*STEPS_PRO_FELD, A1YPOS_STEPS+STEPS_PRO_FELD); motorwait();
 secwait(3);
 movexy(xposition+2*STEPS_PRO_FELD, yposition); motorwait(); secwait(3);
 movexy(xposition, yposition+2*STEPS_PRO_FELD); motorwait(); secwait(3);
 while(1)
  {
   movexy(xposition + 2*STEPS_PRO_FELD, yposition + 3*STEPS_PRO_FELD); secwait(10);
   movexy(xposition - 2*STEPS_PRO_FELD, yposition - 3*STEPS_PRO_FELD); secwait(10);
   movexy(xposition - 2*STEPS_PRO_FELD, yposition + 2*STEPS_PRO_FELD); secwait(10);
   movexy(xposition + 2*STEPS_PRO_FELD, yposition - 2*STEPS_PRO_FELD); secwait(10);
  }
#endif
 
 const char *zeile="";
 while(1) //Hauptschlaufe
  {
   while(notstopp) {milliwait(1000);}//TODO
#ifdef KIPPTEST
   static int testflag=1;
   if(testflag)
    {if(testflag==1) zeile="D1-D8";
     if(testflag==2) zeile="D8-D7";
     if(testflag==3) zeile="D7-D6";
     if(testflag==4) zeile="D6-D5";
     if(testflag==5) zeile="D5-D4";
     if(testflag==6) zeile="D4-D3";
     if(testflag==7) zeile="D3-D2";
     if(testflag==8) zeile="D2-D1";
     if(++testflag == 9) testflag=1;
     uart_print(zeile); uart_putc('\n');
    } else
#endif
   //UART abfragen und entsprechend reagieren:
   zeile=uart_zeile_empfangen();
   //Form der Antwort: "E2-E4" oder z.B. "e2xe4"
   //x = Schlagen von Figuren
   //Spezialzuege: en passant Beispiel: "d4xe3e.p.",
   //              Rochaden: "o-o" "o-o-o"
   //              Bauerumwandlung: "b7-b8Q" (Q=Queen, R=Turm, B=Laeufer, N=Springer)
   if(strlen(zeile) < 3)
    {
#ifdef DEBUG
     if(zeile[0]!='A')
      {
       if(zeile[0]==0) fehlermeldung(3,1); //bei leerer Zeile mit 3 Hz 1 Sekunde lang blinken
       else if(zeile[0]==':') fehlermeldung(3,2); //bei fehlendem ersten Zeichen von Parameter: mit 3 Hz 2 Sekunde lang blinken
       else fehlermeldung(5,6);  //bei unbekanntem Fehler 5 mal pro Sekunde blinken, 6 Sekunden lang
       secwait(1);
    }
#endif
     uart_putc('B'); uart_putc('\n'); //Antwort auf Verbindungsaufbau
     continue;
    }
   if(zeile[1]==':')
    {//Parameter setzen:
     if(zeile[0]=='F') computerfarbe=zeile[2]-'0';
     else if(zeile[0]=='G') brettlage=zeile[2]-'0';
     else if(zeile[0]=='P') aktfigur=zeile[2]; //Figur die gezogen wird
     //TODO: eventuell weitere Parameter
     uart_putc('\n'); //bereit fuer naechsten Befehl
     continue;
    }
   LED1_ON;//test
   if(zeile[2]=='x' || zeile[2]=='X') //Figur schlagen?
    {
     char c1,c2;
     if(strncmp(&zeile[5],"e.p.",4)==0) //en passant?
          {c1=zeile[3]; c2=zeile[1];}
     else {c1=zeile[3]; c2=zeile[4];}
     x = letter_xwert(c1);
     y = ziffer_ywert(c2);
     figur_vom_brett_nehmen(x,y);
     x = letter_xwert('i'); //geschlagene Figur neben das Brett stellen
     figur_abstellen2(x,y);
    }
   if(zeile[0]=='o') //Rochade?
    {
     if(strncmp(zeile,"o-o-o",5)==0)
      {//grosse Rochade:
       if(computerfarbe==WEISS)
	    {figur_ziehen("e1-c1"); figur_ziehen("a1-d1");}
       else {figur_ziehen("e8-c8"); figur_ziehen("a8-d8");}
      }
     else
      {//kleine Rochade:
       if(computerfarbe==WEISS)
	    {figur_ziehen("e1-g1"); figur_ziehen("h1-f1");}
       else {figur_ziehen("e8-g8"); figur_ziehen("h8-f8");}
      }
    }
   else if(strlen(zeile)==6) //Bauerumwandlung
    {
     figur_ziehen(zeile);
#ifndef CALIBTEST
     if(brettlage==1)  {x = letter_xwert('A');  y = ziffer_ywert('1');}
     else                     {x = letter_xwert('H');  y = ziffer_ywert('8');}
     movexy(x,y);
     movey(2*CM); motorwait();
     calibrieren(0);
#endif
    }
   else
    {//gewoehnlicher Zug:
     figur_ziehen(zeile);
    }
#ifdef CALIBTEST
   //test: jedes mal kalibrieren:
   if(brettlage==1)  {x = letter_xwert('A');  y = ziffer_ywert('2');}
   else                     {x = letter_xwert('H');  y = ziffer_ywert('7');}
   movexy(x,y);
   calibrieren(0);
#else
   movey(ziffer_ywert(brettlage==1 ? '1' : '8') - yposition); //Arm zurueckfahren nach abgeschlossenem Zug
   motorwait();
   milliwait(200);//test
  #ifdef AUTOCALIB
   static int recalib=0;//nach jedem 10. Zug neu calibrieren
   if(++recalib >= AUTOCALIB) {calibrieren(); recalib=0;}
  #endif
#endif
   motor_onoff(0,false); //alle Motoren aus
   LED1_OFF;//test
#ifdef KIPPTEST
   if(testflag)   secwait(10);//test
#endif
   uart_putc('\n'); //bereit fuer naechsten Zug
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}
