/* kartonschach.cc       letzte Aenderung: 31.10.2016
Erkennung von Schachzuegen auf einem Kartonbrett
*/
#define VERSION "0.1" //Version des Programms
#define EEVERSION 100 //Version der EEPROM-Belegung
/*
 Prozessor: Atmega1284P, 20MHz Quarz
 Schaltung: schaltung/hall8x8.sch

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 16.10.2016     Erstellung

 Todo:
 - Umwandlung von Bauern (promotion)
 - 3 mal gleiche Stellung --> unentschieden
 - Eingabe einer Stellung
 - Rochade und enpassant ausfuehrlicher testen
 - Ueber UART Verbindung zu Computer (ev. RaspberryPi) mit Stockfish
 - Ueber die zweite UART Verbindung zu Roboterarm

*/
//#define TEST3 1
//#define SCROLLTEST 1
//#define STACKTEST 1
//#define GESCHWINDIGKEITSTEST
#define ENTPRELLPAUSE 200 //zusaetzliche Pause in ms fuer entprelltes Sensorfeld

#define F_CPU 20000000UL

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>
#include <avr/eeprom.h>

//#define DEBUG  //zur Fehlersuche

/**** Vordeklarationen ****/
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t

#define TASTE_OBEN (1<<0)
#define TASTE_UNTEN (1<<1)
#define TASTE_RECHTS (1<<2)
#define TASTE_LINKS (1<<3)
#define TASTE_MITTE (1<<4)

void startmenu(int8_t modus,int8_t auswahlzeile);
uchar auswahlsetzen(uchar taste,int8_t auswahlzeile,int8_t d);
int8_t modus_wechseln(int8_t modus,int8_t* pwahl);
void zug_aus_sensoren_ermitteln(uchar *von,uchar *nach,uchar farbe);

/**** Tastenentprellung ****/
#define PIN_TASTER PINB
void entprell(); //Vordeklaration fuer Tasten-Entprellung - declaration for debouncing function
static volatile char entprelltetasten=0xFF;
static char status1=0xFF,status2=0xFF;
static volatile char tastegedrueckt=0;

void entprell() //Tasten-Entprellung fuer mehrere Tasten unabhaengig voneinander
{               //Debouncing several buttons independent at the same time
 char pin,ung;
 pin = PIN_TASTER;
 //In "ung" diejenigen Bits setzen die ungleich zu status1 oder status2 sind:
 //Set bits to 1 if different to status1 or different to status2:
 ung = (pin^status1) | (status2^status1);
 //Bits so lassen wenn sie in "ung" auf 1 sind, oder gleich wie in "pin" setzen wenn sie in "ung" auf 0 sind:
 //Set bits to previous state when it is 1 in "ung", or set to same as "pin" when it is 0 in "ung":
 entprelltetasten = (entprelltetasten & ung) | (pin & (ung^0xFF));
 status2=status1;
 status1=pin;
 tastegedrueckt |= (entprelltetasten^0xFF);
}

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=0,tage=0;

#define Rest1 (F_CPU%1000) //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (F_CPU/1000-1)

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
}

ISR(TIMER1_COMPA_vect)
{
 uchar mehr;
 uint ms=millisec;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0; mehr=0;
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0;
	 ++tage; //neuertag=1;
	}
      }
    }
  }
 else if((ms&0x03)==1) entprell(); //Tastenentprellung alle 4 Millisekunden - Debouncing every 4 milliseconds
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
}

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) ;//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
   ;//auf n. Millisekunde warten
}

void secwait(uchar n) //n Sekunden warten
{
 while(n!=0) {milliwait(1000); --n;}
}

/*
void microwait(uchar n) //mindestens n Microsekunden warten
{
 while(--n!=0) _delay_us(1);
}
*/

/******************************** AD-Wandler ******************************************/
static volatile uchar adamlaufen=0;
static volatile int messwert=0;
static volatile uint adwandler_zufallszahl=0;

ISR(ADC_vect) //AD-Wandler Interrupt-Routine
{
 uchar c1,c2;
 c1=ADCL;
 c2=ADCH;
 messwert = (c2<<8)+c1;
 adwandler_zufallszahl += c1&3; //unterste 2 Bits zum Zufallszahlen generieren
 adamlaufen=0;
}

void ad_init(uchar adkanal)
{
//Prescaler Beispiel: 2^7=128, 20MHz/128=156.25kHz (muss im Bereich 50-200 liegen)
 //uchar PRESCALER; int pre;
 //for(PRESCALER=1,pre=2*200;  PRESCALER<7 && (F_CPU/1000)>pre;  PRESCALER++, pre *= 2)
 //          {}          //PRESCALER ausrechnen lassen
#define PRESCALER 7  //oder direkt definieren

 adamlaufen=1;

//Fuer Referenzspannung eine der folgenden 4 Varianten benutzen:
 //ADMUX = (0<<REFS0)+adkanal; //Externe Referenz an AREF
 ADMUX = (1<<REFS0)+adkanal; //AVCC als Referenz verwenden
 //ADMUX = (2<<REFS0)+adkanal; //Interne Referenz = 1.1V
 //ADMUX = (3<<REFS0)+adkanal; //Interne Referenz = 2.56V

 ADCSRA = (1<<ADEN)+(1<<ADSC)+(1<<ADIE)+PRESCALER;//Enable+Interruptenable+MessungStarten

 //while(adamlaufen)  ; //erste Wandlung abwarten und verwerfen
 //_delay_ms(1); 
}

void ad_start()
{
 adamlaufen=1;
 ADCSRA |= (1<<ADSC); //Starte AD-Wandlung
}

int ad_lesen()
{
 if(adamlaufen) return 0;
 return messwert;
}

/*
void ad_off()
{
 ADCSRA &= ~(1<<ADEN);
}
*/

//Umrechnungen von Bitwerten auf Millivolt:
//1024 = referenz, somit Bitrauschen = referenz/1024
#define REFERENZ 5000L //Versorgungsspannung (5.0 V) als Referenz
//#define REFERENZ 1100L //Referenzspannung 1.1 V
//#define REFERENZ 2560L //Referenzspannung 2.56 V
#define DU ((REFERENZ+1023)/1024) //REFERENZ/1024 aufgerundet = Ungenauigkeit von AD-Wandler

int mvolt(uint ad) //Messwert umrechnen in Millivolt
{
 return (REFERENZ*ad+512)/1024;
}

/***************************** Timer0 fuer Sensoren auslesen **************************/
static volatile uint sensorfeld[64];
static uint sensorCalibLeer[64]; //Calibrierdaten bei leerem Brett
static uint minDiff; //Minimale Differenz wenn Figur gesetzt zu leerem Brett
static volatile uchar sensorlesezyklus=0; //wird jeweils erhoeht wenn alle 64 Sensoren gelesen

void timer0_init()
{
 TCCR0A = (0<<WGM01)|(0<<WGM00); //Normal-Modus, Interrupt bei Ueberlauf (8 Bit, also Teiler durch 256)
 //TCCR0A = (1<<WGM01)|(0<<WGM00); //CTC-Modus, Interrupt bei Erreichen von OCRA
 //TCCR0B = (1<<CS00); //Takt nicht teilen
 TCCR0B = (1<<CS01); //Takt durch 8teilen
 //TCCR0B = (1<<CS01)|(1<<CS00); //Takt durch 64teilen
 TCNT0 = 0;
 TIMSK0 = (1<<TOIE0); //Interrupt bei Ueberlauf
 //TIMSK0 = (1<<OCIE0A); //Interrupt bei Erreichen von OCRA
}

ISR(TIMER0_OVF_vect)
{
 // wird alle etwa 0.1 ms aufgerufen (102.4 usec)
 // 9765.625 Aufrufe = 1 Sekunde
#ifdef LANGSAMESENSOREN
 // neuer Cyclus alle 8 Aufrufe, 4 Cyclen pro Sensor, 64 Sensoren --> 8*4*64 = 2048 Durchlaeufe benoetigt
 // 2048*102.4usec = ca. 210 msec
#define MAX_ZAEHLER (4*8)
 static uchar zaehler=0;
 if(++zaehler==MAX_ZAEHLER) zaehler=0;
 if((zaehler&0x07)==0) //etwa alle 0.8 ms
  {
   static uchar j=0,maske=0x80;
   static uchar kanal=0;
   uchar cyclus = zaehler>>3;
   switch(cyclus)
    {
     case 1: ad_init(kanal); break;
     case 2: ad_start(); break;
     case 3: sensorfeld[j+kanal]=ad_lesen();
             if(kanal==7) PORTC=0; //wenn letzte Reihe gelesen, Spannung fuer Spalte ausschalten
             break;
     default:
         if(++kanal==8)
	  {
	   kanal=0;
	   j += 8;
	   if(j==64) {j=0; maske=0x80; sensorlesezyklus++;}
	   else  maske >>= 1;
	   PORTC = maske; //Spannung fuer neue Spalte einschalten
	  }
	 break;
    }
  }
#else
 // 4 Cyclen pro Sensor, 64 Sensoren --> 4*64 = 256 Durchlaeufe benoetigt
 // 256*102.4usec = 26.2144 msec
 static uchar cyclus=0;
 static uchar j=0,maske=0x80;
 static uchar kanal=0;
 if(++cyclus==4) cyclus=0;
 switch(cyclus)
    {
     case 1: ad_init(kanal); break;
     case 2: ad_start(); break;
     case 3: sensorfeld[j+kanal]=ad_lesen();
             if(kanal==7) PORTC=0; //wenn letzte Reihe gelesen, Spannung fuer Spalte ausschalten
             break;
     default:
         if(++kanal==8)
	  {
	   kanal=0;
	   j += 8;
	   if(j==64) {j=0; maske=0x80; sensorlesezyklus++;}
	   else  maske >>= 1;
	   PORTC = maske; //Spannung fuer neue Spalte einschalten
	  }
	 break;
    }
#endif
}

/*************************** Serielle Schnittstelle ***********************************/
#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 20000000"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 9600UL      // Baudrate
//#define BAUD 1200UL      //test
 
// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart1_init()
{
 UBRR1 = UBRR_VAL;
#ifdef UART_BASIRGB
 UCSR1B = (1<<RXEN1)|(1<<TXEN1); // UART TX und RX einschalten
#else
 UCSR1B = (1<<RXEN1); // nur UART RX einschalten
#endif
 //UCSR1C = (3<<UCSZ10); // Asynchron 8N1
 UCSR1C = (3<<UCSZ10)|(1<<USBS1); // Asynchron 8N2
}

void uart_putc(unsigned char c)
{
 while (!(UCSR1A & (1<<UDRE1))) ;// warten bis Senden moeglich
 UDR1 = c;                      // sende Zeichen
 milliwait(3);//test
}

unsigned char uart_getc(void)
{
 while (!(UCSR1A & (1<<RXC1))) ;// warten bis Zeichen verfuegbar
 return UDR1; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart_check()
{
 return (UCSR1A & (1<<RXC1)) ;//true wenn Zeichen verfuegbar
}

/********************************** LCD-Modul *****************************************/
#include "lcdmodul2.h"

void itoa(char *s,int n)
{
 char c,*t;
 int k;
 if(n<0) {*s++ = '-'; n = -n;}
 for(t=s;;)
  {
   if(n<10) {*t++ = n+'0'; break;}
   k = n%10;
   *t++ = k+'0';
   n /= 10;
  }
 *t=0;
 for(--t;t!=s;)
  {
   c = *t;
   *t-- = *s;
   *s = c;
   if(t!=s) s++;
  }
}

void lcd_print1(const char *txt,int n)
{
 char text[22];
 uchar i;
 for(i=0;*txt!=0 && i<15;i++) text[i] = *txt++;
 //sprintf(&text[i],"%d",n);
 itoa(&text[i],n);
 lcd_write(text);
}

void longtoa(char *s,long n)
{
 char c,*t;
 if(n<0) {*s++ = '-'; n = -n;}
 for(t=s;n>=10;n/=10)
  {
   c = n%10;
   *t++ = c+'0';
  }
 *t++ = n+'0';
 *t=0;
 for(--t;t!=s;)
  {
   c = *t;
   *t-- = *s;
   *s = c;
   if(t!=s) s++;
  }
}

void lcd_printz1L(int8_t zeile,const char *txt,long n,const char *txt2=NULL)
{
 lcd_goto(zeile,1);
 char text[32];
 uchar i;
 for(i=0;*txt!=0 && i<15;i++) text[i] = *txt++;
 longtoa(&text[i],n);
 lcd_write(text);
 if(txt2!=NULL) lcd_write(txt2);
}

#ifdef TEST3
void lcd_print_sensor(uchar j)
{
 lcd_print1("ad[",j); lcd_print1("] ",sensorfeld[j]);
 lcd_print1(" ",mvolt(sensorfeld[j])); lcd_write("mV ");
}
#endif

void lcd_write_fill(const char *txt) //gesamte Zeile schreiben (mit Blanks auffuellen)
{
 char text[17];
 uchar i;
 for(i=0;*txt!=0 && i<16;i++) text[i] = *txt++;
 for(;i<16;i++) text[i] = ' ';
 text[16]=0;
 lcd_write(text);
}

void lcd_write_zeile(int8_t zeile,const char *text)
{
 lcd_goto(zeile,1); lcd_write_fill(text);
}

void lcd_write4(const char *txt)
{
 lcd_write_zeile(4,txt);
}

void lcd_printf(const char *langer_text)
{
 char txt[17],c;
 int8_t i,zeile;
 for(zeile=1; zeile<=4 && *langer_text!=0; zeile++)
  {
   for(i=0;(c= *langer_text)!=0 && i<16;langer_text++) {if(c!='\n') txt[i++] = c;}
   while(i<16) txt[i++]=' ';
   txt[i]=0; lcd_goto(zeile,1); lcd_write(txt);
  }
}

typedef char string17[17];

int8_t lcd_auswahlfrage(const char *titel,int8_t n,const string17 *str,int8_t j0=1)
{
 int8_t i,i0=0,j,wahl=0;
 lcd_clear();
 lcd_write_zeile(1,titel);
 for(j=2,i=0;j<=4 && i<n;j++,i++)
  {
   lcd_goto(j,3); lcd_write(str[i]);
  }
 if(j0>n) j0=n;
 j0++;
 while(j0>4) //eventuell runterscrollen 
  {
   for(j=1,i= ++i0; j<4; i++)
	{lcd_goto(++j,3); lcd_write(str[i]);}
   j0--;
  }
 lcd_goto(j=j0,1); lcd_writec('>');
 for(;;)
  {
   milliwait(50);
   if((tastegedrueckt&TASTE_OBEN)!=0)
    {
     lcd_goto(j,1); lcd_writec(' ');
     if(j==2 && i0>0) //raufscrollen?
      {
       for(j=5,i= --i0 + 2; j>2; --i)
	{lcd_goto(--j,3); if(i<n) lcd_write(str[i]);}
      }
     else if(--j<2) j=2;
     lcd_goto(j,1); lcd_writec('>');
     while((entprelltetasten&TASTE_OBEN)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_OBEN; //reset pressed state for the button
    }
   if((tastegedrueckt&TASTE_UNTEN)!=0)
    {
     lcd_goto(j,1); lcd_writec(' ');
     if(j==4 && i0<n-3) //runterscrollen?
      {
       for(j=1,i= ++i0; j<4; i++)
	{lcd_goto(++j,3); lcd_write(str[i]);}
      }
     else if(++j>4) j=4;
     lcd_goto(j,1); lcd_writec('>');
     while((entprelltetasten&TASTE_UNTEN)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_UNTEN; //reset pressed state for the button
    }
   if((tastegedrueckt&TASTE_MITTE)!=0) //Ok-Taste
    {
     wahl=j-1+i0;
     while((entprelltetasten&TASTE_MITTE)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_MITTE; //reset pressed state for the button
     break;
    }
   if((tastegedrueckt&TASTE_LINKS)!=0) //Cancel-Taste
    {
     wahl=0;
     while((entprelltetasten&TASTE_LINKS)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_LINKS; //reset pressed state for the button
     break;
    }
  }
 return wahl;
}

/************************************ EEPROM ******************************************/
uint eeversion EEMEM;
uint eeCalibLeer[64] EEMEM;  //Calibrierdaten bei Leerem Brett
uint16_t eeMinDiff EEMEM;    //Minimale Differenz von leerem Feld zu Figur gesetzt
uint8_t eeCalibStatus EEMEM; //Status: 255 = noch nicht calibriert, 1=Leerfeld calibriert, 2=auch MinDiff gesetzt
uint8_t eeSpielstaerke EEMEM; //gleich spielsuchtiefe: 0=sehr schwach, 3=mittel, ca. 6 = staerkste Spielstufe
#define MAXSPIELSTAERKE 6  //noch testen was wirklich das Maximum ist

void eeversion_setzen()
{
 if(eeprom_read_word(&eeversion)!=EEVERSION)
   eeprom_write_word(&eeversion, EEVERSION);
}

void calib_von_eeprom_einlesen()
{
 int8_t j;
 uint version=eeprom_read_word(&eeversion);
 uint8_t status=eeprom_read_byte(&eeCalibStatus);

 if(version==EEVERSION && status != 255) //Calibrierdaten fuer Leerfeld vorhanden?
  {
   //for(j=0;j<64;j++) sensorCalibLeer[j]=eeprom_read_word(&eeCalibLeer[j]); //Calibrierdaten vom EEPROM einlesen
   eeprom_read_block(sensorCalibLeer,eeCalibLeer,64<<1); //einfacher mit Block-Befehl
  }
 else for(j=0;j<64;j++) sensorCalibLeer[j]=488; //provisorische Daten fuer Leerfeld setzen

 if(version==EEVERSION && status == 2) //eeMinDiff gesetzt?
   minDiff=eeprom_read_word(&eeMinDiff); //vom EEPROM einlesen
 else minDiff=10; //sonst provisorischer Wert setzen
}

void calibLeer_im_eeprom_speichern()
{
 eeversion_setzen();
 for(int8_t j=0;j<64;j++)
  {
   if(eeprom_read_word(&eeCalibLeer[j]) != sensorCalibLeer[j])
     eeprom_write_word(&eeCalibLeer[j], sensorCalibLeer[j]);
  }
 if(eeprom_read_byte(&eeCalibStatus)==255)
    eeprom_write_byte(&eeCalibStatus,1); //eeCalibStatus auf 1 setzen: Leerfeld calibriert
}

void minDiff_im_eeprom_speichern()
{
 uint8_t status=eeprom_read_byte(&eeCalibStatus);
 if(status!=1 && status!=2) return;//noch nicht mit Leerfeld calibriert? --> Fehler
 if(eeprom_read_word(&eeMinDiff)!=minDiff)
   eeprom_write_word(&eeMinDiff, minDiff);
 if(status!=2) eeprom_write_byte(&eeCalibStatus,2); //eeCalibStatus auf 2 setzen: auch eeMinDiff calibriert
}

uint8_t spielstaerke_von_eeprom_einlesen()
{
 uint version=eeprom_read_word(&eeversion);
 uint8_t spielstaerke=2;
 if(version==EEVERSION)
  {
   spielstaerke=eeprom_read_byte(&eeSpielstaerke);
   if(spielstaerke>MAXSPIELSTAERKE) //noch nicht im EEPROM gesetzt, oder ungueltiger Wert im EEPROM?
     spielstaerke=2;
  }
 return spielstaerke;
}

void spielstaerke_in_eeprom_speichern(uint8_t spielstaerke)
{
 if(eeprom_read_byte(&eeSpielstaerke)!=spielstaerke)
   eeprom_write_byte(&eeSpielstaerke,spielstaerke);
}

/************************* Sensorfeld calibrieren und auswerten ***********************/
const char *feldname(int8_t j) //zuordnung von Index zu Feld auf dem Schachbrett
{
 static char str[3];
 str[0]=j%8+'A';
 str[1]=j/8+'1';
 str[2]=0;
 return str;
}

void warte_bis_alle_sensoren_gelesen()
{
 static uchar zaehlervorher=0;
 while(sensorlesezyklus==zaehlervorher) milliwait(1);
 zaehlervorher=sensorlesezyklus;
 return;
}

bool fastgleich(uint a,uint b,uint delta)
{
 if(a==b) return true;
 return (a <= b+delta && a+delta >= b);
}

bool feldfastgleich(uint *f1,uint *f2)
{
 for(int8_t j=0;j<64;j++) if(!fastgleich(f1[j],f2[j],4)) return false;
 return true;
}

void sensorfeld_einlesen(uint *feld)
{
 warte_bis_alle_sensoren_gelesen();
 for(int8_t j=0;j<64;j++) feld[j]=sensorfeld[j];
}

void feld_kopieren(uint *f1,volatile uint *f2)
{
 for(int8_t j=0;j<64;j++) *f1++ = *f2++;
}

void sensorfeld_einlesen_entprellt(uint *feld)
{
 uint feld1[64],feld2[64];
 uint *p1=feld1,*p2=feld2,*h;
#ifdef ENTPRELLPAUSE
 milliwait(ENTPRELLPAUSE);
#endif
 warte_bis_alle_sensoren_gelesen();
 feld_kopieren(p1,sensorfeld); //erstes Feld einlesen
 for(int8_t n=0;n<2;) //die Sensoren einlesen bis 2 mal mit erstem Feld uebereinstimmend
  {
   warte_bis_alle_sensoren_gelesen();
   feld_kopieren(p2,sensorfeld); //naechstes Feld einlesen
   h=p1; p1=p2; p2=h; //Zeiger vertauschen
   if(!feldfastgleich(p1,p2)) {n=0; continue;} //wenn ungleich aelteres Feld ueberschreiben
   else n++;
  }
 feld_kopieren(feld,p2); //Resultat zurueckgeben
}

/*
void lcd_print_felddiff(uint *feld1,uint *feld2,int8_t max) //Test
{
 int8_t i,j,z=1;
 uchar c;
 lcd_goto(z,1);
 for(i=j=0;j<max;j++)
  {
   if(feld1[j]>feld2[j]+10) c=((feld1[j]-feld2[j])/10)%10 + '0';
   else if(feld2[j]>feld1[j]+10) c=((feld2[j]-feld1[j])/10)%10 + 'A';
   else c='0';
   lcd_writec(c);
   if(++i==16 && j<max-1) {lcd_goto(++z,1); i=0;}
  }
}
*/

void calib_anzeige(int8_t wahl)
{
 lcd_clear();
 if(wahl==1)      lcd_printf("Leerbrett calibrieren...");
 else if(wahl==2) lcd_printf("Grundstellung calibrieren...");
 else             lcd_printf("Sensorentests...");
}

void calibrieren(int8_t wahl)
{
 uint feld1[64],feld2[64];
 int8_t j;
 calib_anzeige(wahl);
 milliwait(500);
 sensorfeld_einlesen(feld1);
 sensorfeld_einlesen(feld2);
 /*
 for(int i=0;i<50;i++) //test 50*0.2 = 10 Sec
  {
   sensorfeld_einlesen(feld2);
   lcd_print_felddiff(feld1,feld2,64); //test
  }
 calib_anzeige(wahl);
 */
 if(wahl==1) //Leeres Brett
  {
   for(j=0;j<64;j++) sensorCalibLeer[j]=feld2[j];
   calibLeer_im_eeprom_speichern();
   lcd_write4("fertig.");
  }
 else if(wahl==2) //mit Grundaufstellung
  {
   int8_t errcnt=0;
   uint diff=32000; //mindestens Differenz zwischen leerem Feld und Figur auf Feld
   for(j=0;j<64;j++)
    {
     if(!fastgleich(feld1[j],feld2[j],4)) errcnt++;
     if(j>=16 && j<64-16)
      {
       if(!fastgleich(sensorCalibLeer[j],feld2[j],4)) errcnt++;
      }
     else
      {
       uint d=sensorCalibLeer[j]-feld2[j];
       if(d<diff) diff=d; //minimale Differenz ermitteln
      }
    }
   minDiff = diff*3/4; //minimale Differenz mit etwas Toleranz speichern
   lcd_goto(4,1);
   lcd_print1("minDiff=",minDiff);
   if(errcnt==0)
    {
     minDiff_im_eeprom_speichern();
     lcd_write(" ok.");
    }
   else  lcd_print1("er=",errcnt);
   lcd_write(" ");
  }
 else //sonst Sensorentests
  {
   for(;;)
    {
     sensorfeld_einlesen(feld2);
     for(j=0;j<64;j++)
       if(!fastgleich(feld1[j],feld2[j],minDiff))
	{
	 lcd_write_zeile(2,feld2[j]>feld1[j]?"Figur weg    ":"Figur gesetzt");
	 lcd_goto(3,1); lcd_write(feldname(j)); lcd_print1(" ADwert:",feld2[j]); lcd_write("  ");
	 lcd_goto(4,1); //AD-Werte von drumherum auch anzeigen:
	 if(j%8>0) {lcd_print1(" ",feld2[j-1]);}
	 if(j%8<7) {lcd_print1(" ",feld2[j+1]);}
	 if(j>=8)  {lcd_print1(" ",feld2[j-8]);}
	 if(j<64-8) {lcd_print1(" ",feld2[j+8]);}
	 lcd_write("   ");
	 //milliwait(500);
	 sensorfeld_einlesen(feld1);
	 break;
	}
     if((tastegedrueckt&TASTE_MITTE)!=0)
      {
       lcd_clear();
       while((entprelltetasten&TASTE_MITTE)==0) {} //wait until button released
       tastegedrueckt &= ~TASTE_MITTE; //reset pressed state for the button
       return;
      }
    }
  }
 secwait(10);
 lcd_clear();
}

/****************************** Fehler-Behandlungen ***********************************/
static uint8_t fehlerflags=0;
#define FEHLER_RAM             1
#define FEHLER_FIGUR_GESETZT   2 
#define FEHLER_SENSOREN        4
#define FEHLER_ROCHADESENSOR   16

void fehler_zu_wenig_ram()
{
 fehlerflags |= FEHLER_RAM;
}

void fehler_auf_lcd_anzeigen()
{
 if(fehlerflags&FEHLER_RAM)                lcd_write4("zu wenig RAM");
 else if(fehlerflags&FEHLER_FIGUR_GESETZT) lcd_write4("neue Fig gsetzt?");
 else if(fehlerflags&FEHLER_SENSOREN)      lcd_write4("Sensorfehler");
 else if(fehlerflags&FEHLER_ROCHADESENSOR) lcd_write4("Rochade-Fehler?");
 else lcd_write4("unbek. Fehler");
}

/********************************** Schach-Engine *************************************/

#ifdef STACKTEST
static int freezaehler=0;

void* mymalloc(int n){
 void* p=malloc(n);
 if(p!=NULL) freezaehler++;
 return p;
}

void myfree(void* p)
{
 if(p!=NULL) freezaehler--;
 free(p);
}

#define malloc mymalloc
#define free myfree
#endif

#define AVRSCHACH
#include "schach-engine.cc"

class LcdZugliste //Anzeige der bisher gemachten Zuege auf ersten 3 Zeilen des LCD-Moduls
{
public:
 int8_t zeile;
 char txt1[17],txt2[17],txt3[17];
 char *ptxt;
 LcdZugliste() {zeile=0; txt1[0]=txt2[0]=txt3[0]=0; ptxt=txt1;}
 void reset();
 void nextline();
 void scrollen();
 void write(int nr,uchar farbe,Zug& zug);
 void refresh();
};
static LcdZugliste lcdzugliste;

void LcdZugliste::reset()
{
 zeile=0; txt1[0]=txt2[0]=txt3[0]=0; ptxt=txt1;
 for(int8_t j=1;j<=3;j++) lcd_write_zeile(j,txt1);
}

void LcdZugliste::nextline()
{
 if(zeile==3) scrollen();
 else
  {
   zeile++;
   if(zeile==1)      ptxt=txt1;
   else if(zeile==2) ptxt=txt2;
   else              ptxt=txt3;
  }
}

void LcdZugliste::scrollen()
{
 strcpy(txt1,txt2); lcd_write_zeile(1,txt1);
 strcpy(txt2,txt3); lcd_write_zeile(2,txt2);
 txt3[0]=0;         lcd_write_zeile(3,txt3);
}

void LcdZugliste::refresh() //neu zeichnen
{
 lcd_write_zeile(1,txt1);
 lcd_write_zeile(2,txt2);
 lcd_write_zeile(3,txt3);
}

void LcdZugliste::write(int nr,uchar farbe,Zug& zug) //nr: Zugnummern (1, 2, 3, ...), farbe: WEISS oder SCHWARZ, zug: Halbzug
{
 int8_t i;
 if(nr<=1) {nr=1; if(farbe==WEISS) reset();}
 if(farbe==WEISS)
  {
   nextline();
   itoa(ptxt,nr);
   i=strlen(ptxt);
   ptxt[i++]=':';
  }
 else i=strlen(ptxt);
 ptxt[i++]=' ';
 strcpy(&ptxt[i],zug.print()); i+=5;
 lcd_goto(zeile,1); lcd_write(ptxt);
}

/********************************** Hauptprogramm *************************************/
static int8_t computerfarbe=WEISS,spielerfarbe=SCHWARZ;

void auf_taste_warten(uchar taste)
{
 tastegedrueckt=0; //eventuell schon zuvor gedrueckte Tasten ignorieren
 for(;;)
  {
   milliwait(100);
   if((tastegedrueckt&taste)!=0)
      {
       while((entprelltetasten&taste)==0) {} //wait until button released
       tastegedrueckt &= ~taste; //reset pressed state for the button
       return;
      }
  }
}

bool rochade(uchar von,uchar nach) //ist es Koenigszug einer Rochade?
{
 if((spielbrett[von]&FIGUR)!=KOENIG) return false;
 return (nach==von+2 || nach==von-2);
}

void rochade_fertig_machen(uchar nach,uchar farbe)
{
 uchar turmvon,turmnach;
 lcd_write4("Rochade");
 zug_aus_sensoren_ermitteln(&turmvon,&turmnach,farbe);
 if(nach==2 || nach==58) {if(turmnach!=nach+1) fehlerflags|=FEHLER_ROCHADESENSOR;}
 else {if(turmnach!=nach-1) fehlerflags|=FEHLER_ROCHADESENSOR;}
}

bool enpassant(uchar von,uchar nach) //ist es ein en-passant-Zug?
{
 return ((spielbrett[von]&FIGUR)==BAUER && spielbrett[nach]==LEER
	 && (nach==von+7 || nach==von+9 || nach==von-7 || nach==von-9));
}

void enpassant_fertig_machen(uchar nach,uchar geschlagen,uchar farbe)
{
 lcd_write4("En Passant");
 secwait(5);//provi.
}

void zug_aus_sensoren_ermitteln(uchar *von,uchar *nach,uchar farbe)
{
 //*von=12; *nach=12+16; return; //test E2-E4
 uint feld1[64],feld2[64];
 int8_t j, geschlagen=64;
 sensorfeld_einlesen_entprellt(feld1);
 do {
  sensorfeld_einlesen(feld2);
  for(j=0;j<64;j++)
   if(!fastgleich(feld2[j],feld1[j],minDiff))
    {
     if(feld2[j]>feld1[j] && spielbrett[j]!=LEER)
          {
	   if((spielbrett[j]&FARBE)==farbe) {*von=j;}
	   else //der Roboterarm wird zuerst geschlagene Figur nehmen, ausser bei enpassant
	        {geschlagen=j; sensorfeld_einlesen_entprellt(feld1); j=64;}
	   break;
	  }
     else //ignoriere: Figur gesetzt aber keine abgehoben - oder Figur von leerem Feld abgehoben
          {sensorfeld_einlesen_entprellt(feld1); j=64; break;}
    }
 } while(j==64);
 sensorfeld_einlesen_entprellt(feld1);
 do {
  sensorfeld_einlesen(feld2);
  for(j=0;j<64;j++)
   if(!fastgleich(feld2[j],feld1[j],minDiff))
    {
     if(feld2[j]>feld1[j])
      {if(geschlagen==64) geschlagen=j; else fehlerflags|=FEHLER_SENSOREN;
       sensorfeld_einlesen_entprellt(feld1); j=64; break;
      } //schlagen einer Figur
     else {*nach=j; break;}
    }
 } while(j==64);
 if(geschlagen!=64 && geschlagen!= *nach) fehlerflags|=FEHLER_SENSOREN;
 if(*nach == *von)  fehlerflags|=FEHLER_SENSOREN;
 if(fehlerflags!=0) return;
 if(rochade(*von,*nach)) rochade_fertig_machen(*nach,farbe);
 else if(enpassant(*von,*nach)) enpassant_fertig_machen(*nach,geschlagen,farbe);
 return;
}

void zug_aus_sensoren_ermitteln_mitfehlertest(uchar *von,uchar *nach,uchar farbe)
{
 for(;;)
  {
   int8_t i;
   for(i=0;i<2;i++)
    {
     fehlerflags=0;
     zug_aus_sensoren_ermitteln(von,nach,farbe);
     if(fehlerflags==0) return;
     fehler_auf_lcd_anzeigen(); secwait(3);
     if(fehlerflags==FEHLER_ROCHADESENSOR)
      {
       fehler_auf_lcd_anzeigen(); secwait(3);
       lcd_printf("Bitte Turm von  Rochade korrekt ziehen.");
       lcd_write4("dann ok-Taste");
       auf_taste_warten(TASTE_MITTE);
       lcdzugliste.refresh();
       fehlerflags=0;
       return;
      }
     if(i+1<2)
      {
       if(*von != *nach) {lcd_write4("zurueck,ok-Taste"); auf_taste_warten(TASTE_MITTE);}
       lcd_write4("nochmals ziehen");
      }
    }
   lcd_clear();
   lcd_printf("Sensoren-Fehler Bitte justiere  Figuren.");
   lcd_write4("dann ok-Taste");
   auf_taste_warten(TASTE_MITTE);
   lcdzugliste.refresh();
   lcd_write4("neu ziehen");
  }
}

void lcd_falscher_zug(Zug& zug)
{
 lcd_goto(4,1); lcd_write("Zug falsch "); lcd_write(zug.print());
 if(zug.von>=64 || zug.nach>=64) //test
  {
   lcd_goto(4,1); lcd_print1("falsch von=",zug.von);
   secwait(3);
   lcd_goto(4,1); lcd_print1("falsch nach=",zug.nach);
   secwait(3);
  }
}

void zug_von_hand_machen(Zug& zug)
{
 uchar von,nach,fehlerflag=0;
 for(;;)
  {
   zug_aus_sensoren_ermitteln_mitfehlertest(&von, &nach, spielbrett[zug.von]&FARBE);
   if(von==zug.von && nach==zug.nach) break;
   fehlerflag=1;
   Zug falscher; falscher.set(von,nach);
   lcd_falscher_zug(falscher); //provi.
   secwait(3);
   lcd_write4("zurueck dann ok");
   auf_taste_warten(TASTE_MITTE);
   lcd_goto(4,1); lcd_write(" ziehe "); lcd_write(zug.print()); lcd_write("    ");
  }
 if(fehlerflag)
  {
   lcd_write4("Ok richtiger Zug"); //provi.
   milliwait(1000);
   lcd_write4(zug.print());
   milliwait(1000);
  }
}

void fertig(const char *str)
{
 lcd_write4(" ");
 lcd_printf(str);
 auf_taste_warten(TASTE_MITTE);
}

int main(void)
{

 // PortA alles Eingaenge fuer A/D-Wandler
 PORTB = 0x1F; //PortB untere 5 Bits Pullups fuer Tasten
 DDRB  = 0x20; //PortB alles Eingaenge ausser 6.Bit fuer LCD-Beleuchtung, oder hier TestLED
 PORTC = 0x00; //PortC alle auf Low fuer inaktiv (N-Chann-Transistoren)
 DDRC  = 0xFF; //PortC alles Ausgaenge
 // PortD wird fuer LCD benutzt

 timer0_init();
 timer1_init();
 sei(); //Interrupt einschalten

 lcd_init();
 PORTB |= (1<<5); //Hintergrundbeleuchung ein
 //lcd_goto(1,1); lcd_write("Schach 0.0      "); //1.Zeile (16 Zeichen)
 lcd_goto(1,1); lcd_write("Schach "); lcd_write(VERSION); lcd_write("      "); //1.Zeile

 calib_von_eeprom_einlesen(); //Calibrationswerte fuer Sensoren
 spielsuchtiefe=spielstaerke_von_eeprom_einlesen();
 lcd_goto(2,1);
 uint version=eeprom_read_word(&eeversion);
 if(version==100)
  {
   uint8_t status=eeprom_read_byte(&eeCalibStatus);
   lcd_print1("CalibStatus=",status);
   lcd_goto(3,1); lcd_print1("Spielstaerke=",spielsuchtiefe);
   secwait(1);
  }
 else
  {
   if(version==0xFFFF) lcd_write("EEPROM leer");
   else lcd_print1("eeVersion=",version);
   secwait(3);
  }

#ifdef STACKTEST
 lcd_goto(4,1); lcd_print1("SP=",SP);
 secwait(3);
 lcd_goto(4,1); lcd_print1("freezaehler=",freezaehler);
 secwait(3);
#endif

#ifdef TEST3
 uchar j,taste=0;
 while(1) //Hauptschlaufe
  {
   static int pause=500;
   static uchar spalte=0,reihe=0;
   milliwait(pause);
   for(j=0;j<5;j++) //5 Tasten pruefen
    {
     uchar m=(1<<j);
     if((tastegedrueckt&m)!=0)
      {
       switch(j)
	{case 0: taste=1; if(reihe<5) reihe++; break; //obere Taste -> naechste Reihe
	 case 1: taste=2; if(reihe>0) reihe--; break; //untere Taste -> vorherige Reihe
	 case 2: taste=3; if(spalte>0) spalte--; break; //rechte Taste -> vorherige Spalte
	 case 3: taste=4; if(spalte<7) spalte++; break; //linke Taste -> naechste Spalte
	 case 4: taste=5; spalte=reihe=0; break; //mittlere Taste -> erste Spalte und erste Reihe
	}
       while((entprelltetasten&m)==0) {} //wait until button released
       tastegedrueckt &= ~m; //reset pressed state for the button
       lcd_goto(1,1); lcd_print1("Taste: ",taste); lcd_write(" ");
      }
    }
   uchar j0=spalte*8+reihe;
   lcd_goto(2,1); lcd_print_sensor(j0);
   lcd_goto(3,1); lcd_print_sensor(j0+1);
   lcd_goto(4,1); lcd_print_sensor(j0+2);
  }//Ende Hauptschlaufe von TEST3
#endif

#ifdef SCROLLTEST
 {
  string17 texte[4]={"Auswahl 1 ","2.Wahl  ","dritte  ","vierte "};
  int8_t wahl=lcd_auswahlfrage("Scrolltest:",4,texte);
  lcd_goto(1,1); lcd_print1("wahl=",wahl);
  secwait(3);
  Zug zug;
  int nr=1;
  zug.set(8,16); lcdzugliste.write(nr,WEISS,zug);  secwait(1);
  zug.set(6*8,5*8); lcdzugliste.write(nr++,SCHWARZ,zug);  secwait(1);
  zug.set(9,17); lcdzugliste.write(nr,WEISS,zug);  secwait(1);
  zug.set(6*8+1,5*8+1); lcdzugliste.write(nr++,SCHWARZ,zug);  secwait(1);
  zug.set(10,18); lcdzugliste.write(nr,WEISS,zug);  secwait(1);
  zug.set(6*8+2,5*8+2); lcdzugliste.write(nr++,SCHWARZ,zug);  secwait(1);
  zug.set(11,19); lcdzugliste.write(nr,WEISS,zug);  secwait(1);
  zug.set(6*8+3,5*8+3); lcdzugliste.write(nr++,SCHWARZ,zug);  secwait(1);
  secwait(5);
 }
#endif

 int8_t modus=0;
#define MODUS_HAUPTMENU       0
#define MODUS_FARBWAHL        1
#define MODUS_SPIELSTAERKE    2
#define MODUS_STELLUNG_SETZEN 3
#define MODUS_CALIB           4
#define MODUS_RESERVE         5 //Reserve zum Hauptmenu erweitern
#define MODUS_SPIELEN        10

 int8_t stellungsflag=0;

 while(1) //Hauptschlaufe
  {
   milliwait(100);
   if(modus==MODUS_HAUPTMENU)
    {
     string17 texte[4]={"Spiel starten ", //Reihenfoge muss mit MODUS-Nummerierung uebereinstimmen
			"Spielstaerke  ",
			"neue Stellung ",
			"Calib Sensoren"};
     int8_t wahl = lcd_auswahlfrage("Hauptmenu:",4,texte);
     //if(wahl==1) modus=MODUS_FARBWAHL; //zum Spiel starten zuerst Farbe auswaehlen lassen
     //else if(wahl==2) ...
     modus=wahl; //Ist einfacher so, da MODUS-Nummerierung schon uebereinstimmt.
    }
   else if(modus==MODUS_SPIELSTAERKE)
    {
     string17 texte[7]={"0  ","1  ","2  ","3  ","4  ","5  ","6  "};
     int8_t wahl=lcd_auswahlfrage("Spielstaerke:",7,texte,spielsuchtiefe+1);
     if(wahl!=0) spielsuchtiefe=wahl-1;
     lcd_goto(4,1); lcd_print1("Suchtiefe=",spielsuchtiefe); lcd_write("     ");//test
     secwait(2);//test
     modus=0;
    }
   else if(modus==MODUS_FARBWAHL)
    {
     string17 texte[2]={"Weiss  ","Schwarz"};
     int8_t wahl=lcd_auswahlfrage("Deine Farbwahl:",2,texte);
     if(wahl==0) modus=MODUS_HAUPTMENU;
     else
      {
       spielerfarbe = (wahl==1)?WEISS:SCHWARZ;
       computerfarbe = WEISS+SCHWARZ-spielerfarbe;
       modus=MODUS_SPIELEN;
      }
    }
   else if(modus==MODUS_SPIELEN)
    {
     spielstaerke_in_eeprom_speichern(spielsuchtiefe);
     if(stellungsflag==0) spielbrett.grundstellung();
     Zug zug;
     int nr=1;
#ifdef STACKTEST
     lcd_goto(4,1); lcd_print1("SP=",SP);
     secwait(1);
#endif
     lcd_clear();
     lcd_write4("Weiss am Zug ...");
     if(spielerfarbe==WEISS)
      {
       zug=spielerzug(spielbrett,spielerfarbe);
       zug_wirklich_machen(spielbrett,zug);
       lcdzugliste.write(nr,spielerfarbe,zug);
      }
     for(;;)
      {
#ifdef STACKTEST
     lcd_goto(4,1); lcd_print1("SP=",SP); lcd_print1("fr=",freezaehler);
     secwait(3);
#endif
       lcd_write4("Computer rechnet"); //Computer ist am Zug
       if(spielsuchtiefe==0) milliwait(500);//test
#ifdef GESCHWINDIGKEITSTEST
       long startzeit = ((tage*24L+stunden)*60+min)*60+sec;
#endif
       zug=besterzug(spielbrett,computerfarbe,spielsuchtiefe);
#ifdef GESCHWINDIGKEITSTEST
       long endzeit = ((tage*24L+stunden)*60+min)*60+sec;
       lcd_printz1L(4,"Zeit:",endzeit-startzeit," sec"); secwait(1);
       auf_taste_warten(TASTE_MITTE);
#endif
       if(fehlerflags!=0) {fehler_auf_lcd_anzeigen(); secwait(3); fehlerflags=0;}
       if(zug.von==zug.nach) {fertig("Computer kann keinen Zug mehr machen - Unentschieden"); break;}
       lcdzugliste.write(nr,computerfarbe,zug);
       if(computerfarbe==SCHWARZ) ++nr;
       lcd_write4((computerfarbe==WEISS)?"Weiss zieht ...":"Schwarz zieht...");
       zug_von_hand_machen(zug); //fuer Schachroboter: hier Zug vom Roboter machen lassen
       zug_wirklich_machen(spielbrett,zug);
       if(spielbrett.gewonnen(computerfarbe)) {fertig("Schachmatt - Computer hat gewonnen"); break;}
       //Spieler ist am Zug:
       lcd_write4((spielerfarbe==WEISS)?"Weiss am Zug ...":"Schwarz am Zug..");
       zug=spielerzug(spielbrett,spielerfarbe);
       if(zug.von==zug.nach) {fertig("Du kannst keinen Zug mehr machen - Unentschieden"); break;}
       zug_wirklich_machen(spielbrett,zug);
       lcdzugliste.write(nr,spielerfarbe,zug);
       if(spielerfarbe==SCHWARZ) ++nr;
       if(spielbrett.gewonnen(spielerfarbe)) {fertig("Computer ist Schachmatt - Du hast gewonnen"); break;}
      }
     modus=0;
    }
   else if(modus==MODUS_CALIB)
    {
     string17 texte[3]={"leeres Brett ","Startstellung","Sensortests  "};
     int8_t wahl=lcd_auswahlfrage("Calibrieren:",3,texte);
     if(wahl!=0) calibrieren(wahl);
     modus=0;
    }
   else if(modus==MODUS_STELLUNG_SETZEN) //eine Stellung setzen
    {
     lcd_goto(1,1); lcd_write_fill("Stellung setzen ");//test
     secwait(3); modus=0; //provi.
    }
   else
    {
     lcd_write_zeile(1,"Err: unbek.Modus");
     lcd_goto(2,1); lcd_print1("modus=",modus);
     secwait(10); modus=0; //test
    }
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}
