/* kartonschach.cc       letzte Aenderung: 2.1.2017
Erkennung von Schachzuegen auf einem Kartonbrett
Die Hallsensoren sind unter dem Karton montiert, also Abstand
zwischen Figuren-Magnete und Sensoren nur etwa 2mm
*/
#define VERSION "0.2" //Version des Programms
#define EEVERSION 100 //Version der EEPROM-Belegung
/*
 Prozessor: Atmega1284P, 20MHz Quarz
 Schaltung: schaltung/hall8x8.sch

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 16.10.2016     Erstellung
 5.11.2016      neues zug_aus_sensoren_ermitteln()
 11.11.         Bauerumwandlung
 30.12.         UART testen
 1.1.2017       Verbindung ueber UART mit Linux um Stockfisch als Engine zu verwenden

 Todo:
 - Rochade und enpassant ausfuehrlicher testen
 - Umwandlung von Bauern (promotion) ausfuehrlicher testen
 - Eingabe einer Stellung (entsprechende Stellen mit //TODO markiert)
 - aktuelles Spiel/Stellung im EEPROM speichern
 - aktuelle Stellung auf LCD anzeigen, PCINT fuer Taste rechts
 - Ueber zweite UART Verbindung zu Roboterarm, Hardwarekonflikt mit LCD: ev. Soft-UART
 - 3 mal gleiche Stellung, oder 50 Zuege ohne schlagen oder Bauer ziehen --> unentschieden

*/
//#define STACKTEST 1
//#define GESCHWINDIGKEITSTEST
#define ENTPRELLPAUSE 200 //zusaetzliche Pause in ms fuer entprelltes Sensorfeld
//#define UMLAUTTEST
#define MITUART  //muss aktiviert sein fuer Serielle Schnittstelle
//#define UARTTEST

#define F_CPU 20000000UL

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>
#include <avr/eeprom.h>

//#define DEBUG  //zur Fehlersuche

/**** Vordeklarationen ****/
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t

#define TASTE_OBEN (1<<0)
#define TASTE_UNTEN (1<<1)
#define TASTE_RECHTS (1<<2)
#define TASTE_LINKS (1<<3)
#define TASTE_MITTE (1<<4)

void startmenu(int8_t modus,int8_t auswahlzeile);
uchar auswahlsetzen(uchar taste,int8_t auswahlzeile,int8_t d);
int8_t modus_wechseln(int8_t modus,int8_t* pwahl);
void zug_aus_sensoren_ermitteln(uchar *von,uchar *nach,uchar farbe);
void stockfish_fertig();

/**** Tastenentprellung ****/
#define PIN_TASTER PINB
void entprell(); //Vordeklaration fuer Tasten-Entprellung - declaration for debouncing function
static volatile char entprelltetasten=0xFF;
static char status1=0xFF,status2=0xFF;
static volatile char tastegedrueckt=0;

void entprell() //Tasten-Entprellung fuer mehrere Tasten unabhaengig voneinander
{               //Debouncing several buttons independent at the same time
 char pin,ung;
 pin = PIN_TASTER;
 //In "ung" diejenigen Bits setzen die ungleich zu status1 oder status2 sind:
 //Set bits to 1 if different to status1 or different to status2:
 ung = (pin^status1) | (status2^status1);
 //Bits so lassen wenn sie in "ung" auf 1 sind, oder gleich wie in "pin" setzen wenn sie in "ung" auf 0 sind:
 //Set bits to previous state when it is 1 in "ung", or set to same as "pin" when it is 0 in "ung":
 entprelltetasten = (entprelltetasten & ung) | (pin & (ung^0xFF));
 status2=status1;
 status1=pin;
 tastegedrueckt |= (entprelltetasten^0xFF);
}

void warte_losgelassen_mitte()
{
 while((entprelltetasten&TASTE_MITTE)==0) {} //wait until button released
 tastegedrueckt &= ~TASTE_MITTE; //reset pressed state for the button
}

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=0,tage=0;

#define Rest1 (F_CPU%1000) //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (F_CPU/1000-1)

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
}

ISR(TIMER1_COMPA_vect)
{
 uchar mehr;
 uint ms=millisec;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0; mehr=0;
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0;
	 ++tage; //neuertag=1;
	}
      }
    }
  }
 else if((ms&0x03)==1) entprell(); //Tastenentprellung alle 4 Millisekunden - Debouncing every 4 milliseconds
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
}

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) ;//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
   ;//auf n. Millisekunde warten
}

void secwait(uchar n) //n Sekunden warten
{
 while(n!=0) {milliwait(1000); --n;}
}

/*
void microwait(uchar n) //mindestens n Microsekunden warten
{
 while(--n!=0) _delay_us(1);
}
*/

/******************************** AD-Wandler ******************************************/
static volatile uchar adamlaufen=0;
static volatile int messwert=0;
volatile uint adwandler_zufallszahl=0;

ISR(ADC_vect) //AD-Wandler Interrupt-Routine
{
 uchar c1,c2;
 c1=ADCL;
 c2=ADCH;
 messwert = (c2<<8)+c1;
 adwandler_zufallszahl += c1&3; //unterste 2 Bits zum Zufallszahlen generieren
 adamlaufen=0;
}

void ad_init(uchar adkanal)
{
//Prescaler Beispiel: 2^7=128, 20MHz/128=156.25kHz (muss im Bereich 50-200 liegen)
 //uchar PRESCALER; int pre;
 //for(PRESCALER=1,pre=2*200;  PRESCALER<7 && (F_CPU/1000)>pre;  PRESCALER++, pre *= 2)
 //          {}          //PRESCALER ausrechnen lassen
#define PRESCALER 7  //oder direkt definieren

 adamlaufen=1;

//Fuer Referenzspannung eine der folgenden 4 Varianten benutzen:
 //ADMUX = (0<<REFS0)+adkanal; //Externe Referenz an AREF
 ADMUX = (1<<REFS0)+adkanal; //AVCC als Referenz verwenden
 //ADMUX = (2<<REFS0)+adkanal; //Interne Referenz = 1.1V
 //ADMUX = (3<<REFS0)+adkanal; //Interne Referenz = 2.56V

 ADCSRA = (1<<ADEN)+(1<<ADSC)+(1<<ADIE)+PRESCALER;//Enable+Interruptenable+MessungStarten

 //while(adamlaufen)  ; //erste Wandlung abwarten und verwerfen
 //_delay_ms(1); 
}

void ad_start()
{
 adamlaufen=1;
 ADCSRA |= (1<<ADSC); //Starte AD-Wandlung
}

int ad_lesen()
{
 if(adamlaufen) return 0;
 return messwert;
}

/*
void ad_off()
{
 ADCSRA &= ~(1<<ADEN);
}
*/

//Umrechnungen von Bitwerten auf Millivolt:
//1024 = referenz, somit Bitrauschen = referenz/1024
#define REFERENZ 5000L //Versorgungsspannung (5.0 V) als Referenz
//#define REFERENZ 1100L //Referenzspannung 1.1 V
//#define REFERENZ 2560L //Referenzspannung 2.56 V
#define DU ((REFERENZ+1023)/1024) //REFERENZ/1024 aufgerundet = Ungenauigkeit von AD-Wandler

int mvolt(uint ad) //Messwert umrechnen in Millivolt
{
 return (REFERENZ*ad+512)/1024;
}

/***************************** Timer0 fuer Sensoren auslesen **************************/
static volatile uint sensorfeld[64];
static uint sensorCalibLeer[64]; //Calibrierdaten bei leerem Brett
static uint minDiff; //Minimale Differenz wenn Figur gesetzt zu leerem Brett
static volatile uchar sensorlesezyklus=0; //wird jeweils erhoeht wenn alle 64 Sensoren gelesen
static uchar timer0status=0; //0=gestoppt, 1=laufend

void timer0_init()
{
 TCCR0A = (0<<WGM01)|(0<<WGM00); //Normal-Modus, Interrupt bei Ueberlauf (8 Bit, also Teiler durch 256)
 //TCCR0A = (1<<WGM01)|(0<<WGM00); //CTC-Modus, Interrupt bei Erreichen von OCRA
 //TCCR0B = (1<<CS00); //Takt nicht teilen
 TCCR0B = (1<<CS01); //Takt durch 8teilen (sollte mit Hallsensoren SS495A funktionieren)
 //TCCR0B = (1<<CS01)|(1<<CS00); //Takt durch 64teilen (fuer langsamere Sensoren)
 TCNT0 = 0;
 TIMSK0 = (1<<TOIE0); //Interrupt bei Ueberlauf
 //TIMSK0 = (1<<OCIE0A); //Interrupt bei Erreichen von OCRA
 timer0status=1;
}

void timer0_stop()
{
 TCCR0B = 0; //Takt fuer Timer0 anhalten
 timer0status=0;
}

ISR(TIMER0_OVF_vect)
{
 // wird alle etwa 0.1 ms aufgerufen (102.4 usec)
 // 9765.625 Aufrufe = 1 Sekunde
 // 4 Cyclen pro Sensor, 64 Sensoren --> 4*64 = 256 Durchlaeufe benoetigt
 // 256*102.4usec = 26.2144 msec
 static uchar cyclus=0;
 static uchar j=0,maske=0x80;
 static uchar kanal=0;
 if(++cyclus==4) cyclus=0;
 switch(cyclus)
    {
     case 1: ad_init(kanal); break;
     case 2: ad_start(); break;
     case 3: sensorfeld[j+kanal]=ad_lesen();
             if(kanal==7) PORTC=0; //wenn letzte Reihe gelesen, Spannung fuer Spalte ausschalten
             break;
     default:
         if(++kanal==8)
	  {
	   kanal=0;
	   j += 8;
	   if(j==64) {j=0; maske=0x80; sensorlesezyklus++;}
	   else  maske >>= 1;
	   PORTC = maske; //Spannung fuer neue Spalte einschalten
	  }
	 break;
    }
}

/*************************** Serielle Schnittstelle ***********************************/
#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 20000000"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 19200UL      // Baudrate
//#define BAUD 9600UL      //test

// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

#ifdef MITUART
static bool uart_ok=false;
static bool stockflag=false; //gesetzt wenn Stockfisch als Engine verwendet werden soll

void uart0_init()
{
 //PRR |= (1<<PRUSART1); //Power Reduction Register: nicht gebrauchter UART
 //PRR &= ~(1<<PRUSART0); //UART0 Power Reduktion aus
 UBRR0 = UBRR_VAL;
 //UCSR0B = 0; //noch nichts senden und empfangen
 UCSR0B = (1<<RXEN0)|(1<<TXEN0); // UART TX und RX einschalten
 //UCSR0B = (1<<RXEN0); // nur UART RX einschalten
 //UCSR0C = (3<<UCSZ00); // Asynchron 8N1
 UCSR0C = (3<<UCSZ00)|(1<<USBS0); // Asynchron 8N2
}

void uart_start()
{
 UCSR0B |= (1<<RXEN0); // UART RX einschalten
 UCSR0B |= (1<<TXEN0); // UART TX einschalten
}

void uart_stopp()
{
 milliwait(1);//test
 while (!(UCSR0A & (1<<UDRE0))) ;// warten bis letztes Zeichen fertig gesendet
 milliwait(1);//test
 UCSR0B &= ~(1<<RXEN0); // UART RX aus
 UCSR0B &= ~(1<<TXEN0); // UART TX aus
 uart_ok=false; 
}

void uart_putc(unsigned char c)
{
 while (!(UCSR0A & (1<<UDRE0))) ;// warten bis Senden moeglich
 UDR0 = c;                      // sende Zeichen
}

unsigned char uart_getc(void)
{
 while (!(UCSR0A & (1<<RXC0))) ;// warten bis Zeichen verfuegbar
 return UDR0; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart_check()
{
 return (UCSR0A & (1<<RXC0)) ;//true wenn Zeichen verfuegbar
}

void uart_write(const char *txt)
{
 while(*txt!=0) uart_putc(*txt++);
}

void uart_print(uchar n)
{
 if(n>=100) {uart_putc(n/100+'0'); n=n%100;}
 if(n>=10)  {uart_putc(n/10+'0'); n=n%10;}
 uart_putc(n+'0');
}

/*
void uart1_init()
{
 UBRR1 = UBRR_VAL;
 UCSR1B = (1<<RXEN1)|(1<<TXEN1); // UART TX und RX einschalten
 //UCSR1B = (1<<RXEN1); // nur UART RX einschalten
 //UCSR1C = (3<<UCSZ10); // Asynchron 8N1
 UCSR1C = (3<<UCSZ10)|(1<<USBS1); // Asynchron 8N2
}

void uart1_putc(unsigned char c)
{
 while (!(UCSR1A & (1<<UDRE1))) ;// warten bis Senden moeglich
 UDR1 = c;                      // sende Zeichen
 milliwait(3);//test
}

unsigned char uart1_getc(void)
{
 while (!(UCSR1A & (1<<RXC1))) ;// warten bis Zeichen verfuegbar
 return UDR1; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart1_check()
{
 return (UCSR1A & (1<<RXC1)) ;//true wenn Zeichen verfuegbar
}
*/

bool uart_verbindung_aufbauen()
{
 //Startprotokoll: A an Linux-Computer senden, Antwort sollte B sein
 int i;
 uchar c0=0;
 uart_putc('A'); uart_putc('\n'); //A senden
 for(i=0;i<200;i++) //maximal 200 Millisekunden warten bis B empfangen wird
  {
   if(uart_check())
    {
     uchar c=uart_getc();
     if(c=='\n') break;
     if(c>=' ') c0=c;
    }
   milliwait(1);
  }
 return (c0=='B');
}
#endif //MITUART

/********************************** LCD-Modul *****************************************/
#include "lcdmodul2.h"

void itoa(char *s,int n)
{
 char c,*t;
 int k;
 if(n<0) {*s++ = '-'; n = -n;}
 for(t=s;;)
  {
   if(n<10) {*t++ = n+'0'; break;}
   k = n%10;
   *t++ = k+'0';
   n /= 10;
  }
 *t=0;
 for(--t;t!=s;)
  {
   c = *t;
   *t-- = *s;
   *s = c;
   if(t!=s) s++;
  }
}

int atoi(const char *s)
{
 int n=0;
 while(isdigit(*s)) n=10*n+(*s++)-'0';
 return n;
}

void lcd_print1(const char *txt,int n)
{
 char text[22];
 uchar i;
 for(i=0;*txt!=0 && i<15;i++) text[i] = *txt++;
 //sprintf(&text[i],"%d",n);
 itoa(&text[i],n);
 lcd_write(text);
}

void longtoa(char *s,long n)
{
 char c,*t;
 if(n<0) {*s++ = '-'; n = -n;}
 for(t=s;n>=10;n/=10)
  {
   c = n%10;
   *t++ = c+'0';
  }
 *t++ = n+'0';
 *t=0;
 for(--t;t!=s;)
  {
   c = *t;
   *t-- = *s;
   *s = c;
   if(t!=s) s++;
  }
}

void lcd_printz1L(int8_t zeile,const char *txt,long n,const char *txt2=NULL)
{
 lcd_goto(zeile,1);
 char text[32];
 uchar i;
 for(i=0;*txt!=0 && i<15;i++) text[i] = *txt++;
 longtoa(&text[i],n);
 lcd_write(text);
 if(txt2!=NULL) lcd_write(txt2);
}

void lcd_write_fill(const char *txt) //gesamte Zeile schreiben (mit Blanks auffuellen)
{
 char text[17];
 uchar i;
 for(i=0;*txt!=0 && i<16;i++) text[i] = *txt++;
 for(;i<16;i++) text[i] = ' ';
 text[16]=0;
 lcd_write(text);
}

void lcd_write_zeile(int8_t zeile,const char *text)
{
 lcd_goto(zeile,1); lcd_write_fill(text);
}

#ifdef MITUART
void lcd_keine_UART_Verbindung()
{
 lcd_write("keine UART-Verb.");
}
#endif

void lcd_write4(const char *txt)
{
 lcd_write_zeile(4,txt);
}

class Zeile4meldung
{
 const char *text;
public:
 Zeile4meldung() {text=NULL;}
 void write(const char *txt) {lcd_write4(text=txt);}
 void refresh() {if(text!=NULL) lcd_write4(text);}
};
static Zeile4meldung zeile4;

void lcd_printf(const char *langer_text)
{
 char txt[17],c;
 int8_t i,zeile;
 for(zeile=1; zeile<=4 && *langer_text!=0; zeile++)
  {
   for(i=0;(c= *langer_text)!=0 && i<16;langer_text++) {if(c!='\n') txt[i++] = c;}
   while(i<16) txt[i++]=' ';
   txt[i]=0; lcd_goto(zeile,1); lcd_write(txt);
  }
}

typedef char string17[17];

int8_t lcd_auswahlfrage(const char *titel,int8_t n,const string17 *str,int8_t j0=1)
{
 int8_t i,i0=0,j,wahl=0;
 lcd_clear();
 lcd_write_zeile(1,titel);
 for(j=2,i=0;j<=4 && i<n;j++,i++)
  {
   lcd_goto(j,3); lcd_write(str[i]);
  }
 if(j0>n) j0=n;
 j0++;
 while(j0>4) //eventuell runterscrollen 
  {
   for(j=1,i= ++i0; j<4; i++)
	{lcd_goto(++j,3); lcd_write(str[i]);}
   j0--;
  }
 lcd_goto(j=j0,1); lcd_writec('>');
 for(;;)
  {
   milliwait(50);
   if((tastegedrueckt&TASTE_OBEN)!=0)
    {
     lcd_goto(j,1); lcd_writec(' ');
     if(j==2 && i0>0) //raufscrollen?
      {
       for(j=5,i= --i0 + 2; j>2; --i)
	{lcd_goto(--j,3); if(i<n) lcd_write(str[i]);}
      }
     else if(--j<2) j=2;
     lcd_goto(j,1); lcd_writec('>');
     while((entprelltetasten&TASTE_OBEN)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_OBEN; //reset pressed state for the button
    }
   if((tastegedrueckt&TASTE_UNTEN)!=0)
    {
     lcd_goto(j,1); lcd_writec(' ');
     if(j==4 && i0<n-3) //runterscrollen?
      {
       for(j=1,i= ++i0; j<4; i++)
	{lcd_goto(++j,3); lcd_write(str[i]);}
      }
     else if(++j>4) j=4;
     lcd_goto(j,1); lcd_writec('>');
     while((entprelltetasten&TASTE_UNTEN)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_UNTEN; //reset pressed state for the button
    }
   if((tastegedrueckt&TASTE_MITTE)!=0) //Ok-Taste
    {
     wahl=j-1+i0;
     while((entprelltetasten&TASTE_MITTE)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_MITTE; //reset pressed state for the button
     break;
    }
   if((tastegedrueckt&TASTE_LINKS)!=0) //Cancel-Taste
    {
     wahl=0;
     while((entprelltetasten&TASTE_LINKS)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_LINKS; //reset pressed state for the button
     break;
    }
  }
 return wahl;
}

/************************************ EEPROM ******************************************/
//je nach Compilerversion stehen die Daten in umgekehrter Reihenfolge im EEPROM!
#ifdef ALTERCOMPILER
// avr-gcc ?.?.?  (weiss nicht mehr welche Version vor dem Update auf 16.04.1 LTS da war)
uint eeversion EEMEM;
uint eeCalibLeer[64] EEMEM;  //Calibrierdaten bei Leerem Brett
uint16_t eeMinDiff EEMEM;    //Minimale Differenz von leerem Feld zu Figur gesetzt
uint8_t eeCalibStatus EEMEM; //Status: 255 = noch nicht calibriert, 1=Leerfeld calibriert, 2=auch MinDiff gesetzt
uint8_t eeSpielstaerke EEMEM; //gleich spielsuchtiefe: 0=sehr schwach, 3=mittel, 4=staerkste Spielstufe
                              //fuer Stockfish 5,6,10...30?, 10=sehr schnell antwortend
#else
// avr-gcc 4.9.2  (nach Update auf xubuntu 16.04.1 LTS)
uint16_t eeLetzteStellung[64] EEMEM; //gespeicherte Stellung
uint8_t eeSpielstaerke EEMEM; //gleich spielsuchtiefe: 0=sehr schwach, 3=mittel, 4=staerkste Spielstufe
                              //fuer Stockfish 5,6,10...30?, 10=sehr schnell antwortend
uint8_t eeCalibStatus EEMEM; //Status: 255 = noch nicht calibriert, 1=Leerfeld calibriert, 2=auch MinDiff gesetzt
uint16_t eeMinDiff EEMEM;    //Minimale Differenz von leerem Feld zu Figur gesetzt
uint eeCalibLeer[64] EEMEM;  //Calibrierdaten bei Leerem Brett
uint eeversion EEMEM;
#endif

#define MAXSPIELSTAERKE 50  //noch testen was wirklich das Maximum ist (fuer Stockfish-Engine)
#define MAXEIGENEENGINE 4   //maximale Suchtiefe fuer eigene Engine

void eeversion_setzen()
{
 if(eeprom_read_word(&eeversion)!=EEVERSION)
   eeprom_write_word(&eeversion, EEVERSION);
}

void calib_von_eeprom_einlesen()
{
 int8_t j;
 uint version=eeprom_read_word(&eeversion);
 uint8_t status=eeprom_read_byte(&eeCalibStatus);

 if(version==EEVERSION && status != 255) //Calibrierdaten fuer Leerfeld vorhanden?
  {
   //for(j=0;j<64;j++) sensorCalibLeer[j]=eeprom_read_word(&eeCalibLeer[j]); //Calibrierdaten vom EEPROM einlesen
   eeprom_read_block(sensorCalibLeer,eeCalibLeer,64<<1); //einfacher mit Block-Befehl
  }
 else for(j=0;j<64;j++) sensorCalibLeer[j]=488; //provisorische Daten fuer Leerfeld setzen

 if(version==EEVERSION && status == 2) //eeMinDiff gesetzt?
   minDiff=eeprom_read_word(&eeMinDiff); //vom EEPROM einlesen
 else minDiff=10; //sonst provisorischer Wert setzen
}

void calibLeer_im_eeprom_speichern()
{
 eeversion_setzen();
 for(int8_t j=0;j<64;j++)
  {
   if(eeprom_read_word(&eeCalibLeer[j]) != sensorCalibLeer[j])
     eeprom_write_word(&eeCalibLeer[j], sensorCalibLeer[j]);
  }
 if(eeprom_read_byte(&eeCalibStatus)==255)
    eeprom_write_byte(&eeCalibStatus,1); //eeCalibStatus auf 1 setzen: Leerfeld calibriert
}

void minDiff_im_eeprom_speichern()
{
 uint8_t status=eeprom_read_byte(&eeCalibStatus);
 if(status!=1 && status!=2) return;//noch nicht mit Leerfeld calibriert? --> Fehler
 if(eeprom_read_word(&eeMinDiff)!=minDiff)
   eeprom_write_word(&eeMinDiff, minDiff);
 if(status!=2) eeprom_write_byte(&eeCalibStatus,2); //eeCalibStatus auf 2 setzen: auch eeMinDiff calibriert
}

uint8_t spielstaerke_von_eeprom_einlesen()
{
 uint version=eeprom_read_word(&eeversion);
 uint8_t spielstaerke=2;
 if(version==EEVERSION)
  {
   spielstaerke=eeprom_read_byte(&eeSpielstaerke);
   if(spielstaerke>MAXSPIELSTAERKE) //noch nicht im EEPROM gesetzt, oder ungueltiger Wert im EEPROM?
     spielstaerke=2;
  }
 return spielstaerke;
}

void spielstaerke_in_eeprom_speichern(uint8_t spielstaerke)
{
 if(eeprom_read_byte(&eeSpielstaerke)!=spielstaerke)
   eeprom_write_byte(&eeSpielstaerke,spielstaerke);
}

/************************* Sensorfeld calibrieren und auswerten ***********************/
const char *feldname(int8_t j) //zuordnung von Index zu Feld auf dem Schachbrett
{
 static char str[3];
 str[0]=j%8+'A';
 str[1]=j/8+'1';
 str[2]=0;
 return str;
}

void warte_bis_alle_sensoren_gelesen()
{
 static uchar zaehlervorher=0;
 while(sensorlesezyklus==zaehlervorher) milliwait(1);
 zaehlervorher=sensorlesezyklus;
 return;
}

bool fastgleich(uint a,uint b,uint delta)
{
 if(a==b) return true;
 return (a <= b+delta && a+delta >= b);
}

bool feldfastgleich(uint *f1,uint *f2)
{
 for(int8_t j=0;j<64;j++) if(!fastgleich(f1[j],f2[j],4)) return false;
 return true;
}

void sensorfeld_einlesen(uint *feld)
{
 warte_bis_alle_sensoren_gelesen();
 for(int8_t j=0;j<64;j++) feld[j]=sensorfeld[j];
}

void feld_kopieren(uint *f1,volatile uint *f2)
{
 for(int8_t j=0;j<64;j++) *f1++ = *f2++;
}

void sensorfeld_einlesen_entprellt(uint *feld)
{
 uint feld1[64],feld2[64];
 uint *p1=feld1,*p2=feld2,*h;
#ifdef ENTPRELLPAUSE
 milliwait(ENTPRELLPAUSE);
#endif
 warte_bis_alle_sensoren_gelesen();
 feld_kopieren(p1,sensorfeld); //erstes Feld einlesen
 for(int8_t n=0;n<2;) //die Sensoren einlesen bis 2 mal mit erstem Feld uebereinstimmend
  {
   warte_bis_alle_sensoren_gelesen();
   feld_kopieren(p2,sensorfeld); //naechstes Feld einlesen
   h=p1; p1=p2; p2=h; //Zeiger vertauschen
   if(!feldfastgleich(p1,p2)) {n=0; continue;} //wenn ungleich aelteres Feld ueberschreiben
   else n++;
  }
 feld_kopieren(feld,p2); //Resultat zurueckgeben
}

void sensorfeld_einlesen_digital(int8_t *feld)
{
 uint feld1[64],diff;
 sensorfeld_einlesen_entprellt(feld1);
 for(int8_t j=0;j<64;j++)
  {
   diff = (feld1[j]>=sensorCalibLeer[j]) ? 0 : sensorCalibLeer[j] - feld1[j];
   feld[j] = (diff>minDiff) ? 1 : 0;
  }
}

/*
void lcd_print_felddiff(uint *feld1,uint *feld2,int8_t max) //Test
{
 int8_t i,j,z=1;
 uchar c;
 lcd_goto(z,1);
 for(i=j=0;j<max;j++)
  {
   if(feld1[j]>feld2[j]+10) c=((feld1[j]-feld2[j])/10)%10 + '0';
   else if(feld2[j]>feld1[j]+10) c=((feld2[j]-feld1[j])/10)%10 + 'A';
   else c='0';
   lcd_writec(c);
   if(++i==16 && j<max-1) {lcd_goto(++z,1); i=0;}
  }
}
*/

bool ist_feld_leer(const int8_t *feld)
{
 for(int8_t i=0;i<64;i++) if(feld[i]!=0) return false;
 return true;
}

int8_t feld_vergleich(const int8_t *feld1,const int8_t *feld2)
{
 for(int8_t i=0;i<64;i++) if(feld1[i]!=feld2[i]) return (feld1[i]>feld2[i]) ? 1 : -1;
 return 0;
}

void calib_anzeige(int8_t wahl)
{
 lcd_clear();
 if(wahl==1)      lcd_printf("Leerbrett calibrieren...");
 else if(wahl==2) lcd_printf("Grundstellung calibrieren...");
 else             lcd_printf("Sensorentests...");
}

void calibrieren(int8_t wahl)
{
 uint feld1[64],feld2[64];
 int8_t j;
 calib_anzeige(wahl);
 milliwait(500);
 sensorfeld_einlesen(feld1);
 sensorfeld_einlesen(feld2);
 /*
 for(int i=0;i<50;i++) //test 50*0.2 = 10 Sec
  {
   sensorfeld_einlesen(feld2);
   lcd_print_felddiff(feld1,feld2,64); //test
  }
 calib_anzeige(wahl);
 */
 if(wahl==1) //Leeres Brett
  {
   for(j=0;j<64;j++) sensorCalibLeer[j]=feld2[j];
   calibLeer_im_eeprom_speichern();
   zeile4.write("fertig.");
  }
 else if(wahl==2) //mit Grundaufstellung
  {
   int8_t errcnt=0;
   uint diff=32000; //mindestens Differenz zwischen leerem Feld und Figur auf Feld
   for(j=0;j<64;j++)
    {
     if(!fastgleich(feld1[j],feld2[j],4)) errcnt++;
     if(j>=16 && j<64-16)
      {
       if(!fastgleich(sensorCalibLeer[j],feld2[j],4)) errcnt++;
      }
     else
      {
       uint d=sensorCalibLeer[j]-feld2[j];
       if(d<diff) diff=d; //minimale Differenz ermitteln
      }
    }
   minDiff = diff*3/4; //minimale Differenz mit etwas Toleranz speichern
   lcd_goto(4,1);
   lcd_print1("minDiff=",minDiff);
   if(errcnt==0)
    {
     minDiff_im_eeprom_speichern();
     lcd_write(" ok.");
    }
   else  lcd_print1("er=",errcnt);
   lcd_write(" ");
  }
 else //sonst Sensorentests
  {
   for(;;)
    {
     sensorfeld_einlesen(feld2);
     for(j=0;j<64;j++)
       if(!fastgleich(feld1[j],feld2[j],minDiff))
	{
	 lcd_write_zeile(2,feld2[j]>feld1[j]?"Figur weg    ":"Figur gesetzt");
	 lcd_goto(3,1); lcd_write(feldname(j)); lcd_print1(" ADwert:",feld2[j]); lcd_write("  ");
	 lcd_goto(4,1); //AD-Werte von drumherum auch anzeigen:
	 if(j%8>0) {lcd_print1(" ",feld2[j-1]);}
	 if(j%8<7) {lcd_print1(" ",feld2[j+1]);}
	 if(j>=8)  {lcd_print1(" ",feld2[j-8]);}
	 if(j<64-8) {lcd_print1(" ",feld2[j+8]);}
	 lcd_write("   ");
	 //milliwait(500);
	 sensorfeld_einlesen(feld1);
	 break;
	}
     if((tastegedrueckt&TASTE_MITTE)!=0)
      {
       lcd_clear();
       warte_losgelassen_mitte();
       return;
      }
    }
  }
 secwait(10);
 lcd_clear();
}

/****************************** Fehler-Behandlungen ***********************************/
static uint8_t fehlerflags=0;
#define FEHLER_RAM             1
#define FEHLER_FIGUR_GESETZT   2 
#define FEHLER_SENSOREN        4
#define FEHLER_ROCHADESENSOR   16

void fehler_zu_wenig_ram()
{
 fehlerflags |= FEHLER_RAM;
}

void fehler_auf_lcd_anzeigen()
{
 if(fehlerflags&FEHLER_RAM)                lcd_write4("zu wenig RAM");
 else if(fehlerflags&FEHLER_FIGUR_GESETZT) lcd_write4("neue Fig gsetzt?");
 else if(fehlerflags&FEHLER_SENSOREN)      lcd_write4("Sensorfehler");
 else if(fehlerflags&FEHLER_ROCHADESENSOR) lcd_write4("Rochade-Fehler?");
 else lcd_write4("unbek. Fehler");
}

/********************************** Schach-Engine *************************************/
static int8_t spielsuchtiefe=0;

#ifdef STACKTEST
static int freezaehler=0;

void* mymalloc(int n){
 void* p=malloc(n);
 if(p!=NULL) freezaehler++;
 return p;
}

void myfree(void* p)
{
 if(p!=NULL) freezaehler--;
 free(p);
}

#define malloc mymalloc
#define free myfree
#endif

#define AVRSCHACH
#include "schach-engine.h"

#ifdef MITUART
Zug besterzug_stockfish(Brett& brett,Zug& letzterzug,uchar farbe,int8_t suchtiefe);
#endif

void letzte_stellung_von_eeprom_einlesen(Brett& brett)
{
 for(int8_t i=0;i<64;i++) brett[i]=eeprom_read_word(&eeLetzteStellung[i]);
}

void stellung_im_eeprom_speichern(Brett& brett)
{
 for(int8_t i=0;i<64;i++)
   if(eeprom_read_word(&eeLetzteStellung[i]) != brett[i])
     eeprom_write_word(&eeLetzteStellung[i], brett[i]);
}

class LcdZugliste //Anzeige der bisher gemachten Zuege auf ersten 3 Zeilen des LCD-Moduls
{
public:
 int8_t zeile;
 char txt1[17],txt2[17],txt3[17];
 char *ptxt;
 LcdZugliste() {zeile=0; txt1[0]=txt2[0]=txt3[0]=0; ptxt=txt1;}
 void reset();
 void nextline();
 void scrollen();
 void write(int nr,uchar farbe,Zug& zug);
 void refresh();
};
static LcdZugliste lcdzugliste;

void LcdZugliste::reset()
{
 zeile=0; txt1[0]=txt2[0]=txt3[0]=0; ptxt=txt1;
 for(int8_t j=1;j<=3;j++) lcd_write_zeile(j,txt1);
}

void LcdZugliste::nextline()
{
 if(zeile==3) scrollen();
 else
  {
   zeile++;
   if(zeile==1)      ptxt=txt1;
   else if(zeile==2) ptxt=txt2;
   else              ptxt=txt3;
  }
}

void LcdZugliste::scrollen()
{
 strcpy(txt1,txt2); lcd_write_zeile(1,txt1);
 strcpy(txt2,txt3); lcd_write_zeile(2,txt2);
 txt3[0]=0;         lcd_write_zeile(3,txt3);
}

void LcdZugliste::refresh() //neu zeichnen
{
 lcd_write_zeile(1,txt1);
 lcd_write_zeile(2,txt2);
 lcd_write_zeile(3,txt3);
 zeile4.refresh();
}

void LcdZugliste::write(int nr,uchar farbe,Zug& zug) //nr: Zugnummern (1, 2, 3, ...), farbe: WEISS oder SCHWARZ, zug: Halbzug
{
 int8_t i;
 if(nr<=1) {nr=1; if(farbe==WEISS) reset();}
 if(farbe==WEISS)
  {
   nextline();
   itoa(ptxt,nr);
   i=strlen(ptxt);
   ptxt[i++]=':';
  }
 else i=strlen(ptxt);
 ptxt[i++]=' ';
 strcpy(&ptxt[i],zug.print()); i+=5;
 lcd_goto(zeile,1); lcd_write(ptxt);
}

/********************************** Hauptprogramm *************************************/
static int8_t computerfarbe=WEISS,spielerfarbe=SCHWARZ;

void auf_taste_warten(uchar taste)
{
 tastegedrueckt=0; //eventuell schon zuvor gedrueckte Tasten ignorieren
 for(;;)
  {
   milliwait(100);
   if((tastegedrueckt&taste)!=0)
      {
       while((entprelltetasten&taste)==0) {} //wait until button released
       tastegedrueckt &= ~taste; //reset pressed state for the button
       return;
      }
  }
}

bool stellung_pruefen(int8_t *feld,Brett& brett)
{
 for(int8_t j=0;j<64;j++)
  {
   if(feld[j]==0) {if(brett[j]!=LEER) return false;}
   else           {if(brett[j]==LEER) return false;}
  }
 return true;
}

void stellung_fehler()
{
 lcd_write_zeile(1,"Stellung falsch");
 lcd_write_zeile(2,"odr Sensorfehler");
 lcd_write_zeile(3,"justiere Figuren");
 lcd_write4("dann ok-Taste");
 auf_taste_warten(TASTE_MITTE);
}

void stellung_pruefen_und_korrigieren()
{
 int8_t feld1[64];
 for(;;)
  {
   sensorfeld_einlesen_digital(feld1);
   if(stellung_pruefen(feld1,spielbrett)) break;
   stellung_fehler();
   lcdzugliste.refresh();
   secwait(1);
  }
}

bool rochade(uchar von,uchar nach) //ist es Koenigszug einer Rochade?
{
 if((spielbrett[von]&FIGUR)!=KOENIG) return false;
 return (nach==von+2 || nach==von-2);
}

void rochade_fertig_machen(uchar nach,uchar farbe)
{
 uchar turmvon,turmnach;
 lcd_write4("Rochade");
 zug_aus_sensoren_ermitteln(&turmvon,&turmnach,farbe);
 if(nach==2 || nach==58) {if(turmnach!=nach+1) fehlerflags|=FEHLER_ROCHADESENSOR;}
 else {if(turmnach!=nach-1) fehlerflags|=FEHLER_ROCHADESENSOR;}
}

bool enpassant(uchar von,uchar nach) //ist es ein en-passant-Zug?
{
 if((spielbrett[von]&FIGUR)!=BAUER || spielbrett[nach]!=LEER) return false;
 if(nach==von+8 || nach==von-8) return false;
 uchar gegnerfarbe=WEISS+SCHWARZ-(spielbrett[von]&FARBE);
 if(nach==von+7 || nach==von-9) return (spielbrett[von-1]==BAUER+ENPASSFLAG+gegnerfarbe);
 if(nach==von+9 || nach==von-7) return (spielbrett[von+1]==BAUER+ENPASSFLAG+gegnerfarbe);
 return false;
}

void enpassant_fertig_machen(uchar von,uchar nach,uchar farbe,int8_t *feld1,int8_t *feld2)
{
 lcd_write4("En Passant");
 uchar schlagfeld = (nach==von+7 || nach==von-9) ? von-1 : von+1;
 secwait(1);//test
 for(int i=0;i<10;i++) //maximal etwa 5 Sec warten bis geschlagene Figur genommen
  {
   sensorfeld_einlesen_digital(feld2);
   if(feld2[schlagfeld]==0) break;
   milliwait(500);
  }
}

void zug_aus_sensoren_ermitteln(uchar *von,uchar *nach,uchar farbe)
{
 //*von=12; *nach=12+16; return; //test E2-E4
 int8_t feld1[64],feld2[64];
 int8_t j, geschlagen=64, fehler=0;
 sensorfeld_einlesen_digital(feld1);
 do {
  sensorfeld_einlesen_digital(feld2);
  for(j=0;j<64;j++)
   {
    if(feld2[j]!=feld1[j])
     {
      if(feld2[j]==0) //Figur weggenommen
          {
	   feld1[j]=feld2[j];
	   if((spielbrett[j]&FARBE)==farbe) {*von=j;}
	   else //der Roboterarm wird zuerst geschlagene Figur nehmen, ausser bei enpassant
	     {geschlagen=j; j=64;}
	  }
      else //Figur gesetzt? aber keine abgehoben - oder Figur von leerem Feld abgehoben?
          {
	   if(++fehler>=3) {fehlerflags|=FEHLER_SENSOREN; return;}
	   j=64;
	  }
      milliwait(200);
      break;
     }
    }//ende for
 } while(j==64);
 //Jetzt steht in *von die Startposition der gezogenen Figur,
 //und in feld1 ist die Stellung bevor Figur wieder abgestellt.
 do {
  sensorfeld_einlesen_digital(feld2);
  for(j=0;j<64;j++)
   {
    if(feld2[j]!=feld1[j])
     {
      feld1[j]=feld2[j];
      if(feld2[j]==1) //Figur abgesetzt
       {*nach=j;
       }
      else //Figur weggenommen, also Figur geschlagen?
       {if(geschlagen==64) geschlagen=j; else fehlerflags|=FEHLER_SENSOREN;
	//if((spielbrett[j]&FARBE)==farbe) //eigene Figur geschlagen?
	//   {} //dieser Fehler wird schon mit Ueberpruefung auf gueltigen Zug abgefangen
	j=64;
       }
      break;
     }
   }//ende for
 } while(j==64);
 if(geschlagen!=64 && geschlagen!= *nach) fehlerflags|=FEHLER_SENSOREN;
 if(*nach == *von)  fehlerflags|=FEHLER_SENSOREN;
 if(fehlerflags!=0) return;
 if(rochade(*von,*nach)) rochade_fertig_machen(*nach,farbe);
 else if(enpassant(*von,*nach)) enpassant_fertig_machen(*von,*nach,farbe,feld1,feld2);
 return;
}

void zug_aus_sensoren_ermitteln_mitfehlertest(uchar *von,uchar *nach,uchar farbe)
{
 stellung_pruefen_und_korrigieren();
 for(;;)
  {
   for(int8_t i=0;i<2;)
    {
     fehlerflags=0;
     zug_aus_sensoren_ermitteln(von,nach,farbe);
     if(fehlerflags==0) return;
     fehler_auf_lcd_anzeigen(); secwait(3);
     if(fehlerflags==FEHLER_ROCHADESENSOR)
      {
       fehler_auf_lcd_anzeigen(); secwait(3);
       lcd_printf("Bitte Turm von  Rochade korrekt ziehen.");
       lcd_write4("dann ok-Taste");
       auf_taste_warten(TASTE_MITTE);
       lcdzugliste.refresh();
       fehlerflags=0;
       return;
      }
     if(++i==2) break; //nach 2-maligem Fehler Schlaufe abbrechen
     if(*von != *nach) {lcd_write4("zurueck,ok-Taste"); auf_taste_warten(TASTE_MITTE);}
     lcd_write4("nochmals ziehen");
    }
   lcd_clear();
   lcd_printf("Sensoren-Fehler Bitte justiere  Figuren.");
   lcd_write4("dann ok-Taste");
   auf_taste_warten(TASTE_MITTE);
   lcdzugliste.refresh();
   secwait(1); lcd_write4("neu ziehen");//test
  }
}

void zug_von_hand_machen(Zug& zug) //wird aufgerufen wenn der Computerzug gemacht werden soll
{
 uchar von,nach,fehlerflag=0;
 timer0_init();
 for(;;)
  {
   zug_aus_sensoren_ermitteln_mitfehlertest(&von, &nach, spielbrett[zug.von&IMASK]&FARBE);
   if(von==(zug.von&IMASK) && nach==(zug.nach&IMASK)) break;
   fehlerflag=1;
   Zug falscher; falscher.set(von,nach);
   lcd_goto(4,1); lcd_write("Zug falsch "); lcd_write(zug.print());
   secwait(3);
   lcd_write4("zurueck dann ok");
   auf_taste_warten(TASTE_MITTE);
   lcd_goto(4,1); lcd_write(" ziehe "); lcd_write(zug.print()); lcd_write("    ");
  }
 if(fehlerflag)
  {
   lcd_write4("Ok richtiger Zug"); //provi.
   milliwait(1000);
   lcd_write4(zug.print());
   milliwait(1000);
  }
 timer0_stop();
}

bool bauer_umwandlung(Brett& brett,uchar from,uchar to,uchar farbe)
{
 if((brett[from]&FIGUR)!=BAUER) return false;
 return ((farbe==WEISS && to>=64-8) || (farbe==SCHWARZ && to<8)); //gegnerische Grundlinie erreicht?
}

Zug spielerzug(Brett& brett,uchar farbe)
{
 Zugliste zugliste;
 Zug zug;
 alle_moeglichen_zuege(brett,farbe,&zugliste,0);
 if(zugliste.istleer()) {zug.von=zug.nach=0; return zug;}//kein Zug mehr moeglich
 timer0_init();
 for(;;)
  {
   uchar errflag=0;
   uchar von,nach;
   zug_aus_sensoren_ermitteln_mitfehlertest(&von,&nach,farbe);
   zug.set(von,nach);
   if(bauer_umwandlung(brett,von,nach,farbe)) zug.setpromo(DAME); //zum Gueltigkeit des Zuges testen mal DAME annehmen
   if(zugliste.zugsuchen(zug)==NULL) errflag=2; //ungueltig wenn nicht in Liste vorhanden
   if(errflag==0) break;
   //if(errflag==1) printf("Eingabefehler - Eingabe-Beispiel: E2-E4 oder ? zum Stellung zeigen\n");
   if(errflag==2)
    {
     lcd_goto(3,1); lcd_write(zug.print()); lcd_write(" Ung\365ltig");
     lcd_write4("zur\365ck, ok-Taste"); //sollte \xF5 auch gehen?
     auf_taste_warten(TASTE_MITTE);
     lcdzugliste.refresh();
    }
  }
 if(zug.getpromo()!=0) //Bauerumwandlung?
  {
   string17 texte[4]={"Dame    ","Turm    ","Springer","Laeufer "};
   int8_t wahl=lcd_auswahlfrage("BauerUmwandlung:",4,texte);
   if(wahl!=0)
    {
     uchar fig;
     switch(wahl)
      {case 2: fig=TURM; break;
       case 3: fig=SPRINGER; break;
       case 4: fig=LAEUFER; break;
       default: fig=DAME; break;
      }
     zug.setpromo(fig);
    }
   lcdzugliste.refresh();
  }
 timer0_stop();
 return zug;
}

void fertig(const char *str)
{
 zeile4.write(" ");
 lcd_printf(str);
#ifdef MITUART
 if(uart_ok) stockfish_fertig();
#endif
 auf_taste_warten(TASTE_MITTE);
}

int main(void)
{
 // PortA alles Eingaenge fuer A/D-Wandler
 PORTB = 0x1F; //PortB untere 5 Bits Pullups fuer Tasten
 DDRB  = 0x20; //PortB alles Eingaenge ausser 6.Bit fuer LCD-Beleuchtung, oder hier TestLED
 PORTC = 0x00; //PortC alle auf Low fuer inaktiv (N-Chann-Transistoren)
 DDRC  = 0xFF; //PortC alles Ausgaenge
 // PortD wird fuer LCD benutzt

 //timer0_init(); //nur bei Bedarf ein- und aus-schalten
 timer1_init();
 sei(); //Interrupt einschalten
 
 lcd_init();
 PORTB |= (1<<5); //Hintergrundbeleuchung ein
 //lcd_goto(1,1); lcd_write("Schach 0.0      "); //1.Zeile (16 Zeichen)
 lcd_goto(1,1); lcd_write("Schach "); lcd_write(VERSION); lcd_write("      "); //1.Zeile

#ifdef UARTTEST
 uart0_init();
 secwait(10);
 for(int i=0;i<30000;i++)
  {
   if(i==0 || uart_check())
    {
     char txt[40];
     uchar c=0,j=0;
     if(uart_check())
      {while(c!='\n') {c=uart_getc(); if(c>=' ' && j<39) txt[j++]=c;} //Zeichen empfangen
      }
     else {txt[0]='A'; txt[1]='B'; txt[2]='C'; j=3;}
     txt[j]=0;
     lcd_goto(2,1); lcd_write("uart: "); lcd_write(txt); lcd_writec(' ');
     for(j=0;(c=txt[j])>=' ';j++) uart_putc(c); //Zeichen zuruecksenden
     uart_putc('\n');
     i=0;
    }
   milliwait(1);
  }
#endif

#ifdef MITUART
 uart0_init();
 lcd_goto(2,1); 
 uart_ok=uart_verbindung_aufbauen();
 if(uart_ok) {lcd_write_fill("uart ok"); stockflag=true;}
 else        {lcd_keine_UART_Verbindung(); uart_stopp();}
 secwait(2);
#endif

 calib_von_eeprom_einlesen(); //Calibrationswerte fuer Sensoren
 spielsuchtiefe=spielstaerke_von_eeprom_einlesen();
 lcd_goto(2,1);
 uint version=eeprom_read_word(&eeversion);
 if(version==100)
  {
   uint8_t status=eeprom_read_byte(&eeCalibStatus);
   lcd_print1("CalibStatus=",status); lcd_write("   ");
   lcd_goto(3,1); lcd_print1("Spielstaerke=",spielsuchtiefe);
  }
 else
  {
   if(version==0xFFFF) lcd_write("EEPROM leer");
   else lcd_print1("eeVersion=",version);
  }
#ifdef UMLAUTTEST
 for(uchar c0=0xD0;;c0+=0x10)
  {
   lcd_goto(4,1);
   for(uchar c=c0,i=0;i<16;c++,i++) lcd_writec(c);
   auf_taste_warten(TASTE_MITTE);
   if(c0==0xF0) break;
  }
 lcd_goto(4,1); lcd_write("Ae..Ue: \xE1\xEF\xF5");
 auf_taste_warten(TASTE_MITTE);
 // Umrechnung Hex-Octal: 0xE1 = 1110 0001 = 11 100 001=oktal 341 
 lcd_goto(4,1); lcd_write("ae..ue: \341\357\365");
 auf_taste_warten(TASTE_MITTE);
 lcd_write4("zur\365ck, ok-Taste");
#endif
 secwait(5); //Startmeldung 5 Sekunden stehen lassen

#ifdef STACKTEST
 lcd_goto(4,1); lcd_print1("SP=",SP);
 secwait(3);
 lcd_goto(4,1); lcd_print1("freezaehler=",freezaehler);
 secwait(3);
#endif

 int8_t modus=0;
#define MODUS_HAUPTMENU       0
#define MODUS_FARBWAHL        1
#define MODUS_SPIELSTAERKE    2
#define MODUS_STELLUNG_SETZEN 3
#define MODUS_CALIB           4
#define MODUS_RESERVE         5 //Reserve
#define MODUS_SPIELEN        10

 int8_t stellungsflag=0;

 while(1) //Hauptschlaufe
  {
   milliwait(100);
   if(modus==MODUS_HAUPTMENU)
    {
     string17 texte[4]={"Spiel starten ", //Reihenfoge muss mit MODUS-Nummerierung uebereinstimmen
			"Spielst\341rke   ",
			"neue Stellung ",
			"Calib Sensoren"};
     int8_t wahl = lcd_auswahlfrage("Hauptmenu:",4,texte);
     //if(wahl==1) modus=MODUS_FARBWAHL; //zum Spiel starten zuerst Farbe auswaehlen lassen
     //else if(wahl==2) ...
     modus=wahl; //Ist einfacher so, da MODUS-Nummerierung schon uebereinstimmt.
    }
   else if(modus==MODUS_SPIELSTAERKE)
    {
#ifdef MITUART
     string17 texte[10]={"0  ","1  ","2  ","3  ","4  ","5  ","6  ","10 ","20 ","30 "};
     int8_t n=0; while(n<10 && atoi(texte[n++])!=spielsuchtiefe) {}
     int8_t wahl=lcd_auswahlfrage("Spielst\341rke:",10,texte,n);
     if(wahl!=0) spielsuchtiefe=atoi(texte[wahl-1]);
#else
     string17 texte[8]={"0  ","1  ","2  ","3  ","4  ","5 ","6 ","7 "};
     int8_t wahl=lcd_auswahlfrage("Spielst\341rke:",8,texte,spielsuchtiefe+1);
     if(wahl!=0) spielsuchtiefe=wahl-1;
#endif
     lcd_goto(4,1); lcd_print1("Suchtiefe=",spielsuchtiefe); lcd_write("     ");
     milliwait(500);
#ifdef MITUART
     if(spielsuchtiefe>MAXEIGENEENGINE) stockflag=true;
     else
      {string17 text[2]={"Eigene   ","Stockfish"};
       wahl=lcd_auswahlfrage("Schach-Engine:",2,text,stockflag?2:1);
       if(wahl>=1) stockflag=(wahl==2);
      }
#endif
     modus=0;
    }
   else if(modus==MODUS_FARBWAHL)
    {
     string17 texte[2]={"Weiss  ","Schwarz"};
     int8_t wahl=lcd_auswahlfrage("Deine Farbwahl:",2,texte);
     if(wahl==0) modus=MODUS_HAUPTMENU;
     else
      {
       spielerfarbe = (wahl==1)?WEISS:SCHWARZ;
       computerfarbe = WEISS+SCHWARZ-spielerfarbe;
       modus=MODUS_SPIELEN;
      }
    }
   else if(modus==MODUS_SPIELEN)
    {
     spielstaerke_in_eeprom_speichern(spielsuchtiefe);
     if(stellungsflag==0) spielbrett.grundstellung();
#ifdef MITUART
     uchar alte_spielsuchtiefe=spielsuchtiefe;
     if(stockflag && !uart_ok)
      {uart_start();
       uart_ok=uart_verbindung_aufbauen();
       lcd_goto(3,1); 
       if(uart_ok)
	lcd_write_fill("uart ok");
       else
	{lcd_keine_UART_Verbindung();
	 uart_stopp();
	 if(spielsuchtiefe>MAXEIGENEENGINE) spielsuchtiefe=MAXEIGENEENGINE;
	}
       secwait(1);
      }
     if(stockflag && uart_ok)
      {
       uart_write("suchtiefe:"); uart_print(spielsuchtiefe); uart_putc('\n');
       milliwait(10); //provi. keine Antwort abwarten bei Suchtiefe setzen
      }
#endif
     Zug zug; zug.set(0,0);
     int nr=1;
#ifdef STACKTEST
     lcd_goto(4,1); lcd_print1("SP=",SP);
     secwait(1);
#endif
     //lcd_clear();
     lcdzugliste.reset();
     zeile4.write("Weiss am Zug ...");
     if(spielerfarbe==WEISS)
      {
       timer0_init();
       zug=spielerzug(spielbrett,spielerfarbe);
       timer0_stop();
       zug_wirklich_machen(spielbrett,zug);
       lcdzugliste.write(nr,spielerfarbe,zug);
      }
     for(;;)
      {
#ifdef STACKTEST
       lcd_goto(4,1); lcd_print1("SP=",SP); lcd_print1("fr=",freezaehler);
       secwait(3);
#endif
       timer0_init();
       stellung_pruefen_und_korrigieren();
       timer0_stop();
       zeile4.write("Computer rechnet"); //Computer ist am Zug
       //if(spielsuchtiefe==0) milliwait(500);//test
#ifdef GESCHWINDIGKEITSTEST
       long startzeit = ((tage*24L+stunden)*60+min)*60+sec;
#endif
       
#ifdef MITUART
       //lcd_write_zeile(3,"besterzug()");//test
       if(stockflag && uart_ok) zug=besterzug_stockfish(spielbrett,zug,computerfarbe,spielsuchtiefe);
       else
#endif
        zug=besterzug(spielbrett,computerfarbe,spielsuchtiefe);

#ifdef GESCHWINDIGKEITSTEST
       long endzeit = ((tage*24L+stunden)*60+min)*60+sec;
       lcd_printz1L(4,"Zeit:",endzeit-startzeit," sec"); secwait(1);
       auf_taste_warten(TASTE_MITTE);
#endif
       if(fehlerflags!=0) {fehler_auf_lcd_anzeigen(); secwait(3); fehlerflags=0;}
       if(zug.von==zug.nach) {fertig("Computer kann keinen Zug mehr machen - Unentschieden"); break;}
       lcdzugliste.write(nr,computerfarbe,zug);
       if(computerfarbe==SCHWARZ) ++nr;
       zeile4.write((computerfarbe==WEISS)?"Weiss zieht ...":"Schwarz zieht...");
       zug_von_hand_machen(zug); //fuer Schachroboter: hier Zug vom Roboter machen lassen
       zug_wirklich_machen(spielbrett,zug);
       if(spielbrett.gewonnen(computerfarbe)) {fertig("Schachmatt - Computer hat gewonnen"); break;}
       //Spieler ist am Zug:
       zeile4.write((spielerfarbe==WEISS)?"Weiss am Zug ...":"Schwarz am Zug..");
       timer0_init();
       zug=spielerzug(spielbrett,spielerfarbe);
       timer0_stop();
       if(zug.von==zug.nach) {fertig("Du kannst keinen Zug mehr machen - Unentschieden"); break;}
       zug_wirklich_machen(spielbrett,zug);
       lcdzugliste.write(nr,spielerfarbe,zug);
       if(spielerfarbe==SCHWARZ) ++nr;
       if(spielbrett.gewonnen(spielerfarbe)) {fertig("Computer ist Schachmatt - Du hast gewonnen"); break;}
       stellungsflag=0;
      }
#ifdef MITUART
     if(stockflag && !uart_ok) spielsuchtiefe=alte_spielsuchtiefe;
#endif
     modus=0;
    }
   else if(modus==MODUS_CALIB)
    {
     timer0_init();
     string17 texte[3]={"leeres Brett ","Startstellung","Sensortests  "};
     int8_t wahl=lcd_auswahlfrage("Calibrieren:",3,texte);
     if(wahl!=0) calibrieren(wahl);
     modus=0;
     timer0_stop();
    }
   else if(modus==MODUS_STELLUNG_SETZEN) //eine Stellung setzen
    {
     string17 texte[3]={"ab Leerbrett  ","ab Startstlg.","gesp. Stellg."};
     int8_t wahl=lcd_auswahlfrage("Stellung setzen:",3,texte);
     if(wahl!=0)
      {
       int8_t i,feld1[64],feld2[64];
       timer0_init();
       lcd_clear();
       if(wahl==1)
	{lcd_printf("alle Figuren entfernen");
	 spielbrett.clear();
	}
       else if(wahl==2)
	{lcd_printf("Startstellung setzen");
	 for(i=0;i<64;i++) feld2[i] = (i<16 || i>=64-16) ? 1 : 0;
	 spielbrett.grundstellung();
	}
       else //wahl==3
	{lcd_printf("Gesp. Stellung wird vom EEPROM geladen...");
	 letzte_stellung_von_eeprom_einlesen(spielbrett);
	 for(i=0;i<64;i++) feld2[i] = (spielbrett[i]==LEER) ? 0 : 1;
	 milliwait(500);//test
	 lcd_printf("Stellung setzen (anzeigen mit TASTE_RECHTS)");
	 //TODO
	}
       bool ok=false;
       do {
	sensorfeld_einlesen_digital(feld1);
	if(wahl==1) ok = ist_feld_leer(feld1);
	else        ok = (feld_vergleich(feld1,feld2)==0);
       } while(!ok);
       milliwait(500);//test
       if(wahl!=3)
	{
	 int8_t i,wahl1,wahl2;
	 lcd_printf("jetzt Figuren setzen, beenden mit ok-Taste");
	 do {
	  sensorfeld_einlesen_digital(feld2);
	  for(i=0;i<64;i++) if(feld2[i]!=feld1[i]) break;
	  if(i<64)
	   {if(feld2[i]==1)
	     {string17 figuren[6]={"Bauer   ","Turm    ","Springer","L\341ufer  ","Dame    ","K\357nig   "};
	      string17 farben[2]={"weiss  ","schwarz"};
	      wahl1=lcd_auswahlfrage("Figur gesetzt:",6,figuren);
	      wahl2=lcd_auswahlfrage("Farbe:",2,farben);
	      uchar figur=0,farbe=0,flag=0;
	      switch(wahl1) {case 1: figur=BAUER; break;  case 2: figur=TURM;  break; case 3: figur=SPRINGER; break;
	                     case 4: figur=LAEUFER; break;  case 5: figur=DAME;  break; case 6: figur=KOENIG; break;
	                    }
	      if(wahl1!=0)
	       {farbe=(wahl2==1)?WEISS:SCHWARZ;
		if(figur==KOENIG) {if((farbe==WEISS && i==4) || (farbe==SCHWARZ && i==60)) flag=MOVFLAG;}
		if(figur==TURM) {if((farbe==WEISS && (i==0 || i==7)) || (farbe==SCHWARZ && (i==63 || i==56))) flag=MOVFLAG;}
		spielbrett[i]=figur+farbe+flag;
	       }
	      lcd_printf("n\341chste Figur setzen oder ok-Taste");//provi.
	     }
	    else
	     {spielbrett[i]=LEER;
	      lcd_write4("Figur entfernt");//provi.
	      //TODO
	     }
	    feld1[i]=feld2[i];
	   } 
	  ok = (tastegedrueckt&TASTE_MITTE)!=0;
	  if(ok)
	   {
	    warte_losgelassen_mitte(); //warte auf Taste losgelassen
	    i=koenigposition(spielbrett,WEISS);
	    if((spielbrett[i]&(FIGUR+FARBE))!=KOENIG+WEISS)
	     {lcd_printf("Weisser Koenig fehlt noch!"); ok=false;}
	    i=koenigposition(spielbrett,SCHWARZ);
	    if((spielbrett[i]&(FIGUR+FARBE))!=KOENIG+SCHWARZ)
	     {lcd_printf("Schwarzer Koenig fehlt noch!"); ok=false;}
	   }
	 } while(!ok);
	}
       stellungsflag=1; //gewuenschte Stellung fertig gesetzt
       timer0_stop();
       //stellung_im_eeprom_speichern(spielbrett);//test
       warte_losgelassen_mitte(); //warte auf Taste losgelassen
      }
     modus=0;
    }
   else
    {
     lcd_write_zeile(1,"Err: unbek.Modus");
     lcd_goto(2,1); lcd_print1("modus=",modus);
     secwait(10); modus=0; //test
    }
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

/** Stockfisch benutzen fuer starke Spielstaerke **/
#ifdef MITUART
void uart_readline(char *text)
{
 uchar c,i;
 for(i=1; i<80 && (c=uart_getc())!='\n';)
  {
   if(c>=' ') {*text++ = c; i++;}
  }
 *text = 0;
}

Zug besterzug_stockfish(Brett& brett,Zug& letzterzug,uchar farbe,int8_t suchtiefe)
{
 Zugliste zugliste;
 Zug zug;
 //lcd_write_zeile(1,"best_stockfish");//test
 alle_moeglichen_zuege(brett,farbe,&zugliste,0);
 if(zugliste.istleer()) {zug.set(0,0); return zug;}
 uart_write("letzterzug: ");
 uart_print(letzterzug.von);
 uart_putc(' ');
 uart_print(letzterzug.nach);
 uart_putc(' ');
 uart_write("Brett:");
 for(int8_t i=0;i<64;i++)
  {
   uart_putc(' ');
   uart_print(brett[i]);
  }
 uart_putc('\n'); //Letzter Zug und Aufstellung an Stockfisch senden
 //lcd_write_zeile(2,"uart gesendet");//test
 char text[80];
 uart_readline(text); //Antwort von Stockfisch lesen
 //lcd_write_zeile(3,"empfang uart");//test
 int von,nach;
 //lcd_write_zeile(3,text);//test
 sscanf(text,"%d %d",&von,&nach);
 zug.set(von,nach);
 return zug;
}

void stockfish_fertig()
{
 uart_putc('Q'); uart_putc('\n');
 uart_stopp();
}
#endif //MITUART
