/* schach-engine.cc			letzte Aenderung: 5.11.2016

einfaches Schachprogramms in c++

Einbinden ins Hauptprogramm:
#define AVRSCHACH  //auskommentieren fuer Linux-Version
#include "schach-avr.cc"  //so im Hauptprogramm einfuegen (nach den andern includes)

History:
30.8.2016	Erstellung (RP)
16.10.2016	AVR-Variante
27.10.2016      Fehlerkorrekturen
31.10.          Hauptprogramm ausgelagert, so dass fast alles gleich fuer Linux- und AVR-Variante
31.10.          Optimierung in stellung_bewerten: figuren_punkte() rechnet jetzt gleichzeitig fuer beide Farben
5.11.2016       spielerzug() fuer AVR-Variante ins Hauptprogramm ausgelagert

*/

//#define AVRSCHACH   //fuer AVR-Variante, sollte im Hauptprogramm definiert werden
//#define BAUERNSPIEL //auskommentieren fuer normales Schach

#ifndef AVRSCHACH
typedef unsigned char uchar;
#define lcd_printf printf
#endif

/************************* Globale Variablen **************************/
static int8_t spielsuchtiefe=0;

/************** Klassen und Hauptteil des Programms *******************/
#define LEER 0

#define RESERVEBITS 0x38
#define FIGUR  0x07
#define BAUER     1
#define SPRINGER  2
#define LAEUFER   3
#define TURM      4
#define DAME      5
#define KOENIG    6

#define FARBE   0x40
#define WEISS   0
#define SCHWARZ 0x40

#define MOVFLAG 0x80 //bei Bauern fuer en passant, bei Koenig u. Turm fuer Rochade
#define ENPASSFLAG 0x80 //bei Bauern fuer en passant, bei Koenig u. Turm fuer Rochade

#define MAXPUNKTE 32000

#ifdef BAUERNSPIEL
#define MAXBAUERPUNKTE 10000 //fuer Bauernspiel
#else
#define MAXBAUERPUNKTE 900 //fuer Umwandlung von Bauer in Dame
#endif

const int16_t material[6]={100,300,300,500,900,0}; //Werte in Centibauer

class Brett
{
 uchar feld[64]; //Spielbrett von A1 B1 C1 ... G8 H8 als 0 1 2 ... 62 63 nummeriert
public:
 Brett() {clear();}
 void clear() {for(int8_t i=0;i<64;i++) feld[i]=0;}
 void grundstellung();
 uchar& operator[](int8_t i) {return feld[i];}
 //int get(int i) {return feld[i];}
 //void set(int i,uchar u) {feld[i]=u;}
#ifndef AVRSCHACH
 void print();
#endif
 bool gewonnen(uchar farbe);
 int8_t gewonnen_oder_unentschieden(uchar farbe);
 bool istzugmoeglich(uchar farbe);
};

// weitere Vordeklarationen:
bool isincheck(Brett& brett,uchar farbe);

bool Brett::gewonnen(uchar farbe)
{
#ifdef BAUERNSPIEL
 //provisorisch: Gewonnen wenn ein Bauer gegnerische Grundlinie erreicht
 //unentschieden wenn jemand nicht mehr fahren kann
 int8_t i;
 if(farbe==WEISS)
  {
   for(i=64-8;i<64;i++)
    if((feld[i]&FIGUR)==BAUER) return true;
  }
 else
  {
   for(i=0;i<8;i++)
    if((feld[i]&FIGUR)==BAUER) return true;
  }

#else

 uchar gegnerfarbe=WEISS+SCHWARZ-farbe;
 if(isincheck((*this),gegnerfarbe) && !istzugmoeglich(gegnerfarbe)) return true;
 //if(!isincheck((*this),gegnerfarbe) && !istzugmoeglich(gegnerfarbe)) return unentschieden;

#endif
 return false;
}

#ifndef AVRSCHACH
void Brett::print()
{
 int i,j,k;
 for(j=64-8;j>=0;j-=8)
  {
   k = (j>>3)&1;
   for(i=0;i<8;i++,k^=1)
    {switch(feld[j+i]&(FIGUR+FARBE))
      {case 0: printf((k==0)?" :: ":" .. "); break;
       case BAUER+WEISS: printf(" wB "); break;
       case BAUER+SCHWARZ: printf(" sB "); break;
       case KOENIG+WEISS: printf(" wK "); break;
       case KOENIG+SCHWARZ: printf(" sK "); break;
       case TURM+WEISS: printf(" wT "); break;
       case TURM+SCHWARZ: printf(" sT "); break;
       case SPRINGER+WEISS: printf(" wS "); break;
       case SPRINGER+SCHWARZ: printf(" sS "); break;
       case LAEUFER+WEISS: printf(" wL "); break;
       case LAEUFER+SCHWARZ: printf(" sL "); break;
       case DAME+WEISS: printf(" wD "); break;
       case DAME+SCHWARZ: printf(" sD "); break;
      }
    }
   printf("\n");
  }
}
#endif

void Brett::grundstellung()
{
 int8_t i,j;
 uchar farbe;
 clear();
 //Bauern setzen:
 for(j=1,farbe=WEISS;j<=6;j+=5,farbe=SCHWARZ)
  for(i=0;i<8;i++)
    feld[j*8+i] = BAUER+farbe;
 //Koenige:
 feld[4] = KOENIG+WEISS+MOVFLAG; //MOVEFLAG gesetzt wenn Koenig noch nicht bewegt wurde
 feld[4+7*8] = KOENIG+SCHWARZ+MOVFLAG;
#ifndef BAUERNSPIEL
 //restliche Figuren:
 feld[0] = feld[7] = TURM+WEISS+MOVFLAG; //MOVEFLAG gesetzt wenn Turm noch nicht bewegt wurde
 feld[1] = feld[6] = SPRINGER+WEISS;
 feld[2] = feld[5] = LAEUFER+WEISS;
 feld[3] = DAME+WEISS;
 feld[56] = feld[63] = TURM+SCHWARZ+MOVFLAG; //MOVEFLAG gesetzt wenn Turm noch nicht bewegt wurde
 feld[57] = feld[62] = SPRINGER+SCHWARZ;
 feld[58] = feld[61] = LAEUFER+SCHWARZ;
 feld[59] = DAME+SCHWARZ;
#endif
}

Brett spielbrett;

class Zug
{
public:
 uchar von,nach;
 void set(uchar from,uchar to) {von=from; nach=to;}
 const char *print();
};
const char *Zug::print()
{
 static char zugtext[6];
 zugtext[0] = 'A'+(von%8);  zugtext[1] = '1'+(von/8);
 zugtext[2] = '-'; 
 zugtext[3] = 'A'+(nach%8); zugtext[4] = '1'+(nach/8);
 zugtext[5] = 0;
 return zugtext;
}

class Zugliste
{
 uchar von,nach;
 int16_t bewertung;
 Zugliste *next;
public:
 Zugliste() {next=NULL; von=nach=0; bewertung=0;}
 ~Zugliste() {clear();}
#ifdef AVRSCHACH
 void clear() {if(next!=NULL) {next->clear(); free(next); next=NULL;} von=nach=0;}
#else
 void clear() {if(next!=NULL) {next->clear(); delete[] next; next=NULL;} von=nach=0;}
#endif
 bool istleer() {return von==nach;}
 void vertauschen(Zugliste *a,Zugliste *b);
 void addsorted(uchar from,uchar to,int16_t wert);
 void add(uchar from,uchar to,Brett& brett,uchar farbe,int8_t suchtiefe);
 Zug getbest(Brett& brett,uchar farbe);
 Zugliste* zugsuchen(Zug zug);
};

Zugliste* Zugliste::zugsuchen(Zug zug)
{
 Zugliste *p;
 for(p=this; p!=NULL && (p->von!=zug.von || p->nach!=zug.nach); p=p->next) {}
 return p;
}

void Zugliste::vertauschen(Zugliste *a,Zugliste *b)
{
 uchar h=a->von; a->von=b->von; b->von=h;
 h=a->nach; a->nach=b->nach; b->nach=h;
 int16_t h2=a->bewertung; a->bewertung=b->bewertung; b->bewertung=h2;
}

void Zugliste::addsorted(uchar from,uchar to,int16_t wert)
{
 if(istleer()) {von=from; nach=to; bewertung=wert; return;}
#ifdef AVRSCHACH
 Zugliste *neu=(Zugliste*)malloc(sizeof(Zugliste));
 if(neu==NULL) {fehler_zu_wenig_ram(); return;}
#else
 Zugliste *neu=new Zugliste[1];
#endif
 neu->von=from; neu->nach=to; neu->bewertung=wert;
 if(wert>bewertung)
  {
   neu->next=next;
   next=neu;
   vertauschen(this,neu);
   return;
  }
 Zugliste *p;
 for(p=this; p->next!=NULL && p->next->bewertung >= wert; p=p->next) {}
 neu->next=p->next;
 p->next=neu;
}

// weitere Vordeklarationen:
int16_t stellung_bewerten(Brett& brett,uchar farbe,int8_t suchtiefe);
int16_t zug_bewerten(Brett& brett,uchar farbe,Zug& zug,int8_t suchtiefe);
Zug besterzug(Brett& brett,uchar farbe,int8_t suchtiefe);
void zug_machen(Brett& brett,uchar von,uchar nach,Brett& hilfsbrett);
void alle_moeglichen_zuege(Brett& brett,uchar farbe,Zugliste *zugliste,int8_t suchtiefe);

int8_t koenigposition(Brett& brett,uchar farbe)
{
 int8_t i,j;
 if(farbe==WEISS)
  {
   for(i=4,j=3;i<8;j--,i++)
    {
     if((brett[i]&(FIGUR+FARBE))==KOENIG+WEISS) return i;
     if((brett[j]&(FIGUR+FARBE))==KOENIG+WEISS) return j;
    }
   for(;i<64;i++)
     if((brett[i]&(FIGUR+FARBE))==KOENIG+WEISS) return i;
  }
 else
  {
   for(i=60,j=61;i>57;j++,i--)
    {
     if((brett[i]&(FIGUR+FARBE))==KOENIG+SCHWARZ) return i;
     if((brett[j]&(FIGUR+FARBE))==KOENIG+SCHWARZ) return j;
    }
   for(;i>=0;--i)
     if((brett[i]&(FIGUR+FARBE))==KOENIG+SCHWARZ) return i;
  }
#ifndef AVRSCHACH
 //printf("Error: Koenig nicht gefunden\n");//test
#endif
 return 0;//sollte nie erreicht werden
}

bool isincheck(Brett& brett,uchar farbe) //steht Koenig im Schach?
{
 int8_t i,j,k,m;
 uchar gegnerfarbe=WEISS+SCHWARZ-farbe;
 int8_t n=koenigposition(brett,farbe);
 i=n%8;
 j=n/8;
 //Test auf Schach durch Springer:
 for(k=i-2; k <= i+2; k+=4)
  if(k>=0 && k<=7)
   {
    if(j>0 && (brett[(j-1)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
    if(j<7 && (brett[(j+1)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
   }
 for(k=i-1; k <= i+1; k+=2)
  if(k>=0 && k<=7)
   {
    if(j>1 && (brett[(j-2)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
    if(j<6 && (brett[(j+2)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
   }
 //Test auf gegnerischen Koenig:
 for(k=i-1; k <= i+1; k++)
 for(m=j-1; m <= j+1; m++)
  {
   if(k==i && m==j) continue;
   if(k>=0 && k<=7 && m>=0 && m<=7 && (brett[m*8+k]&(FIGUR+FARBE))==KOENIG+gegnerfarbe) return true;
  }
 //Test auf Schach durch Turm oder Dame:
 uchar fe;
 for(k=i,m=n;--k>=0;)
  {
   fe=brett[--m];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 for(k=i,m=n;++k<=7;)
  {
   fe=brett[++m];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 for(k=j,m=n;--k>=0;)
  {
   fe=brett[m-=8];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 for(k=j,m=n;++k<=7;)
  {
   fe=brett[m+=8];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 //Test auf Schach durch Laeufer oder Dame oder Bauer:
 for(k=i-1,m=j-1; k>=0 && m>=0; --k,--m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==SCHWARZ && m==j-1) return true;
     break;
    }
  }
 for(k=i+1,m=j-1; k<=7 && m>=0; ++k,--m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==SCHWARZ && m==j-1) return true;
     break;
    }
  }
 for(k=i-1,m=j+1; k>=0 && m<=7; --k,++m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==WEISS && m==j+1) return true;
     break;
    }
  }
 for(k=i+1,m=j+1; k<=7 && m<=7; ++k,++m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==WEISS && m==j+1) return true;
     break;
    }
  }
 return false;
}

bool Brett::istzugmoeglich(uchar farbe)
{
 Zugliste zugliste;
 alle_moeglichen_zuege((*this),farbe,&zugliste,0);
 if(zugliste.istleer()) return false;
 return true;
}

void Zugliste::add(uchar from,uchar to,Brett& brett,uchar farbe,int8_t suchtiefe)
{
 Brett hilfsbrett;
 zug_machen(brett,from,to,hilfsbrett);
 if(isincheck(hilfsbrett,farbe)) return; //wenn Koenig im Schach steht --> ungueltiger Zug
 int16_t wert=stellung_bewerten(hilfsbrett,farbe,suchtiefe);
 addsorted(from,to,wert);
}

bool ist_schraeger_zug(uchar von,uchar nach)
{
 return (nach==von+9 || nach==von+7 || nach==von-9 || nach==von-7);
}

void en_passant_schlagen(Brett& brett,uchar von,uchar nach)
{
 if(nach==von+9 || nach==von+7) brett[nach-8]=LEER;
 else if(nach==von-9 || nach==von-7) brett[nach+8]=LEER;
}

void zug_machen(Brett& brett,uchar von,uchar nach,Brett& hilfsbrett)
{
 int8_t i;
 for(i=0;i<64;i++)
  {
   if((brett[i]&FIGUR)==BAUER)
     hilfsbrett[i] = brett[i] & ~MOVFLAG; //eventuelle en passant Flags loeschen
   else hilfsbrett[i] = brett[i];
  }
 hilfsbrett[nach]=brett[von];
 hilfsbrett[von]=LEER;
 hilfsbrett[nach] &= ~MOVFLAG;//MOVFLAG loeschen (auch bei Koenig u. Turm wenn gezogen)
 if((brett[von]&FIGUR)==BAUER)
  {
   if(ist_schraeger_zug(von,nach) && brett[nach]==LEER) en_passant_schlagen(hilfsbrett,von,nach);
   else if((nach==von+2*8 || nach==von-2*8)) //wenn Bauer 2 Schritte gezogen
      hilfsbrett[nach] |= MOVFLAG; //Flag fuer moegliches en passant setzen
  }
 else if((brett[von]&FIGUR)==KOENIG)
  {
   if(nach==von+2 || nach==von-2) //Rochade?
      {
       uchar turmvon,turmnach;
       if(nach>von) {turmvon=nach+1; turmnach=nach-1;}
       else         {turmvon=nach-2; turmnach=nach+1;}
       hilfsbrett[turmnach] = brett[turmvon] & ~MOVFLAG; //Turm ziehen
       hilfsbrett[turmvon] = LEER;
      }
   hilfsbrett[nach] &= ~MOVFLAG; //Flag zum Rochade erlauben loeschen
  }
}

void zug_wirklich_machen(Brett& brett,Zug zug)
{
 Brett hilfsbrett;
 zug_machen(brett,zug.von,zug.nach,hilfsbrett);
 brett=hilfsbrett;
}

int figuren_punkte(Brett& brett,uchar farbe)
{
 int8_t i;
 int punkte=0;
 uchar fig,figfarbe;
 for(i=0;i<64;i++)
  {
   if((fig=brett[i]&FIGUR)==0) continue;
   figfarbe = brett[i]&FARBE;
   if(figfarbe==farbe)  punkte += material[fig-1];
   else                 punkte -= material[fig-1];
   if(fig==BAUER) //Bei Bauern pro vorgeruecktes Feld 1 weiterer Punkt
    {
     if(figfarbe==farbe)
      {
       if(figfarbe==WEISS)
	{if(i>=16)    punkte += (i>64-8) ? MAXBAUERPUNKTE : (i-8)/8;}
       else //SCHWARZ
	{if(i<=63-16) punkte += (i<8) ? MAXBAUERPUNKTE : (63-8-i)/8;}
      }
     else //figfarbe!=farbe
      {
       if(figfarbe==WEISS)
	{if(i>=16)    punkte -= (i>64-8) ? MAXBAUERPUNKTE : (i-8)/8;}
       else //SCHWARZ
	{if(i<=63-16) punkte -= (i<8) ? MAXBAUERPUNKTE : (63-8-i)/8;}
      }
    }
   else if(fig==KOENIG)
    {
     if(brett[i]&MOVFLAG) {if(figfarbe==farbe) punkte++; else punkte--;} //zusaetzlicher Punkt wenn Koenig noch Rochadefaehig
    }
  }
 //printf("figuren_punkte() farbe=%d punkte=%d\n",farbe,punkte);//test
 return punkte;
}

int16_t stellung_bewerten(Brett& brett,uchar farbe,int8_t suchtiefe) //Wert fuer farbe wenn gegnerfarbe am Zug ist
{
 int16_t bewertung;
 int8_t gegnerfarbe=WEISS+SCHWARZ-farbe;
 if(suchtiefe==0)
  {
   bewertung = figuren_punkte(brett,farbe);
  }
 else
  {
   Zug zug=besterzug(brett,gegnerfarbe,suchtiefe-1);
   if(zug.von==zug.nach) //kein Zug mehr moeglich
    {
     if(isincheck(brett,gegnerfarbe)) bewertung = MAXPUNKTE; //farbe hat gewonnen
     else bewertung = 0; //Unentschieden
    }
   else bewertung = -zug_bewerten(brett,gegnerfarbe,zug,suchtiefe-1);
  }
 //printf("Farbe=%d bewertung=%d\n",farbe,bewertung);//test
 return bewertung;
}

int16_t zug_bewerten(Brett& brett,uchar farbe,Zug& zug,int8_t suchtiefe)
{
 Brett hilfsbrett;
 zug_machen(brett, zug.von, zug.nach, hilfsbrett);
 int16_t wert=stellung_bewerten(hilfsbrett,farbe,suchtiefe);
 return wert;
}

#ifdef AVRSCHACH

//Zufallszahlen auf AVR-Microkontroller:
#define MYRAND_MAX (1<<14)
int myrand()
{
 return adwandler_zufallszahl&(MYRAND_MAX-1);
}
int zufall(int max) //Zufallszahl zwischen 0 und max (inklusive 0, exclusive max)
{
 int z = myrand()/(MYRAND_MAX/max);
 //printf("Test zufall(%d) --> z=%d (RAND_MAX=%d)\n",max,z,RAND_MAX);
 if(z==max) --z;
 return z;
}

#else

//Zufallszahlen unter Linux:
#include <sys/times.h>
int zufall(int max) //Zufallszahl zwischen 0 und max (inklusive 0, exclusive max)
{
 static int seed=0;
 if(seed==0)
  {
   struct tms buf;
   clock_t t=times(&buf);
   srand(seed=t);
  }
 int z = rand()/(RAND_MAX/max);
 //printf("Test zufall(%d) --> z=%d (RAND_MAX=%d)\n",max,z,RAND_MAX);
 if(z==max) --z;
 return z;
}

#endif

Zug Zugliste::getbest(Brett& brett,uchar farbe)
{
 Zugliste *p;
 int n,i;
 //Da schon mit addsort() sortiert wurde, ist erster Eintrag mit bester Bewertung.
 int16_t fastbestebewertung=bewertung-1;
 for(p=next,n=1; p!=NULL && p->bewertung >= fastbestebewertung ;p=p->next)
  {
   n++; //Anzahl fast gleich gute Bewertungen zaehlen
  }
 p=this;
 if(n>1) //bei mehreren fast gleich guten Zuegen, zufaelligen auswaehlen
  {
   i=zufall(n);
   for(;i>0;p=p->next,i--) {}
  }
 Zug zug;
 zug.set(p->von,p->nach);
 return zug;
}

void alle_moeglichen_zuege(Brett& brett,uchar farbe,Zugliste *zugliste,int8_t suchtiefe)
{
 int8_t i;
 uchar fig,col;
 for(i=0;i<64;i++)
  {
   if(brett[i]==LEER || (col=brett[i]&FARBE)!=farbe) continue;
   fig = brett[i]&FIGUR;
   if(fig==BAUER)
    {
     if(col==WEISS)
      {
       if(brett[i+8]==LEER) zugliste->add(i,i+8,brett,farbe,suchtiefe);
       if(i<16 && brett[i+8]==LEER && brett[i+16]==LEER) zugliste->add(i,i+16,brett,farbe,suchtiefe);
       if(i>=4*8 && i<5*8) //Bauer auf 5.Linie: auf en passant testen
	{
	 if(i > 4*8 && brett[i-1]==BAUER+SCHWARZ+MOVFLAG) zugliste->add(i,i+8-1,brett,farbe,suchtiefe);
	 if(i+1<5*8 && brett[i+1]==BAUER+SCHWARZ+MOVFLAG) zugliste->add(i,i+8+1,brett,farbe,suchtiefe);
	}
       if(i%8!=0 && brett[i+7]!=LEER && (brett[i+7]&FARBE)==SCHWARZ) zugliste->add(i,i+7,brett,farbe,suchtiefe);
       if(i%8!=7 && brett[i+9]!=LEER && (brett[i+9]&FARBE)==SCHWARZ) zugliste->add(i,i+9,brett,farbe,suchtiefe);
      }
     else //if(col==SCHWARZ)
      {
       if(brett[i-8]==LEER) zugliste->add(i,i-8,brett,farbe,suchtiefe);
       if(i>=64-16 && brett[i-8]==LEER && brett[i-16]==LEER) zugliste->add(i,i-16,brett,farbe,suchtiefe);
       if(i>=3*8 && i<4*8) //Bauer auf 4.Linie: auf en passant testen
	{
	 if(i > 3*8 && brett[i-1]==BAUER+WEISS+MOVFLAG) zugliste->add(i,i-8-1,brett,farbe,suchtiefe);
	 if(i+1<4*8 && brett[i+1]==BAUER+WEISS+MOVFLAG) zugliste->add(i,i-8+1,brett,farbe,suchtiefe);
	}
       if(i%8!=0 && brett[i-9]!=LEER && (brett[i-9]&FARBE)==WEISS) zugliste->add(i,i-9,brett,farbe,suchtiefe);
       if(i%8!=7 && brett[i-7]!=LEER && (brett[i-7]&FARBE)==WEISS) zugliste->add(i,i-7,brett,farbe,suchtiefe);
      }
    }
   else if(fig==KOENIG)
    {
     if((i%8)<7)
	{if(brett[i+1]==LEER || (brett[i+1]&FARBE)!=farbe) zugliste->add(i,i+1,brett,farbe,suchtiefe);
	 if(i<64-8 && (brett[i+9]==LEER || (brett[i+9]&FARBE)!=farbe)) zugliste->add(i,i+9,brett,farbe,suchtiefe);
	 if(i>=8   && (brett[i-7]==LEER || (brett[i-7]&FARBE)!=farbe)) zugliste->add(i,i-7,brett,farbe,suchtiefe);
	}
     if((i%8)>0)
	{if(brett[i-1]==LEER || (brett[i-1]&FARBE)!=farbe) zugliste->add(i,i-1,brett,farbe,suchtiefe);
	 if(i<64-8 && (brett[i+7]==LEER || (brett[i+7]&FARBE)!=farbe)) zugliste->add(i,i+7,brett,farbe,suchtiefe);
	 if(i>8    && (brett[i-9]==LEER || (brett[i-9]&FARBE)!=farbe)) zugliste->add(i,i-9,brett,farbe,suchtiefe);
	}
     if(i<64-8 && (brett[i+8]==LEER || (brett[i+8]&FARBE)!=farbe)) zugliste->add(i,i+8,brett,farbe,suchtiefe);
     if(i >= 8 && (brett[i-8]==LEER || (brett[i-8]&FARBE)!=farbe)) zugliste->add(i,i-8,brett,farbe,suchtiefe);
     //hier noch Rochade:
     // Rochade ist nur erlaubt wenn: Koenig und beteiligter Turm noch nicht bewegt wurden,
     //                               alle Felder zwischen Koenig und Turm leer sind,
     //                               Koenig nicht im Schach steht und ueber kein bedrohtes Feld fahren muss.
     if(col==WEISS)
	{
	 if(brett[i]&MOVFLAG && !isincheck(brett,farbe)) //Koenig noch nicht bewegt, und nicht im Schach
	  {
	   int8_t k;
	   if((brett[0]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer grosse Rochade noch nicht bewegt
	      && brett[1]==LEER && brett[2]==LEER && brett[3]==LEER) //und alle Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && --k>=i-2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER; 
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i-2,brett,farbe,suchtiefe); //grosse Rochade
	    }
	   if((brett[7]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer kleine Rochade noch nicht bewegt
	      && brett[6]==LEER && brett[5]==LEER)   //und beide Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && ++k<=i+2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER;
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i+2,brett,farbe,suchtiefe); //kleine Rochade
	    }
	  }
	}
     else //if(col==SCHWARZ) //Rochade fuer Schwarzen Koenig
	{
	 if(brett[i]&MOVFLAG && !isincheck(brett,farbe)) //Koenig noch nicht bewegt, und nicht im Schach
	  {
	   int8_t k;
	   if((brett[56]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer grosse Rochade noch nicht bewegt
	      && brett[57]==LEER && brett[58]==LEER && brett[59]==LEER) //und alle Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && --k>=i-2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER; 
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i-2,brett,farbe,suchtiefe); //grosse Rochade
	    }
	   if((brett[63]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer kleine Rochade noch nicht bewegt
	      && brett[62]==LEER && brett[61]==LEER)   //und beide Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && ++k<=i+2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER;
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i+2,brett,farbe,suchtiefe); //kleine Rochade
	    }
	  }
	}
    }
#ifdef BAUERNSPIEL
   else lcd_printf("Error: unbekannte Figur fig=%d\n",fig);//test
#else
   else if(fig==SPRINGER)
    {
     int8_t j,k, ix=i%8, jy=i/8;
     for(k=ix-2;k<=ix+2;k+=4)
      for(j=jy-1;j<=jy+1;j+=2)
      {
       if(k>=0 && k<8 && j>=0 && j<8)
	{
	 int8_t n=j*8+k;
	 if(brett[n]==LEER || (brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	}
      }
     for(k=ix-1;k<=ix+1;k+=2)
      for(j=jy-2;j<=jy+2;j+=4)
      {
       if(k>=0 && k<8 && j>=0 && j<8)
	{
	 int8_t n=j*8+k;
	 if(brett[n]==LEER || (brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	}
      }
    }
   else
    {
     int8_t ix=i%8, jy=i/8;
     if(fig==TURM || fig==DAME)
      {
       int8_t n,k;
       for(k=ix-1,n=i-1;k>=0;k--,n--)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix+1,n=i+1;k<8;k++,n++)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=jy-1,n=i-8;k>=0;k--,n-=8)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=jy+1,n=i+8;k<8;k++,n+=8)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
      }
     if(fig==LAEUFER || fig==DAME)
      {
       int8_t n,k,j;
       for(k=ix-1,j=jy-1,n=i-9; k>=0 && j>=0; k--,j--,n-=9)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix+1,j=jy+1,n=i+9; k<8 && j<8; k++,j++,n+=9)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix-1,j=jy+1,n=i+7; k>=0 && j<8; k--,j++,n+=7)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix+1,j=jy-1,n=i-7; k<8 && j>=0; k++,j--,n-=7)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
      }
    }
#endif //BAUERNSPIEL
  }
}

Zug besterzug(Brett& brett,uchar farbe,int8_t suchtiefe)
{
 Zugliste zugliste;
 alle_moeglichen_zuege(brett,farbe,&zugliste,suchtiefe);
 return zugliste.getbest(brett,farbe);
}

#ifndef AVRSCHACH
void hilfe()
{
 printf("help oder hilfe eingegeben\n");
#ifdef BAUERNSPIEL
 printf("Provisorische Regeln fuer Bauernspiel:\n");
 printf("Wer zuerst mit einem Bauern die gegenrische Grundlinie erreicht gewinnt.\n");
 printf("Sonst, wenn kein gueltiger Zug mehr moeglich ist, ist unentschieden.\n");
#endif
 printf("Das Eingabeformat ist XN-YM wobei XN die Startposition und YM die Zielposition einer Figur ist.\n");
 printf("Mit Eingabe von '?' kann die momentane Stellung angezeigt werden.\n");
}
#endif

#ifdef AVRSCHACH

Zug spielerzug(Brett& brett,uchar farbe);

#else

Zug spielerzug(Brett& brett,uchar farbe)
{
 Zugliste zugliste;
 Zug zug;
 char mzug[80];
 static int8_t startflag=1;
 alle_moeglichen_zuege(brett,farbe,&zugliste,0);
 if(zugliste.istleer()) {zug.von=zug.nach=0; return zug;}//kein Zug mehr moeglich
 for(;;)
  {
   uchar von,nach,i;
   uchar errflag=0;
   printf("Dein Zug: "); scanf("%s",mzug);
   if(mzug[0]=='?') {brett.print(); continue;}
   if(mzug[0]=='h' && (mzug[1]=='e' || mzug[1]=='i')) {hilfe(); continue;}
   for(i=0;mzug[i]!=0;i++) if(islower(mzug[i])) mzug[i] -= 'a'-'A';
   if(mzug[0]<'A' || mzug[0]>'H' || mzug[1]<'1' || mzug[1]>'9') errflag=1;
   else
    {
     for(i=2;i<10 && !isalnum(mzug[i]);i++) {}
     if(mzug[i]<'A' || mzug[i]>'H' || mzug[i+1]<'1' || mzug[i+1]>'9') errflag=1;
     else
      {
       von = mzug[0]-'A' + (mzug[1]-'1')*8;
       nach = mzug[i]-'A' + (mzug[i+1]-'1')*8;
       zug.set(von,nach);
       if(zugliste.zugsuchen(zug)==NULL) errflag=2; //ungueltig wenn nicht in Liste vorhanden
      }
    }
   if(errflag==0) break;
   if(errflag!=0 && startflag!=0)
    {
     printf("zum Stellung anzeigen: ? eingeben\n");
     printf("sonst Zug eingeben: Starposition-Zielposition\n");
     --startflag;
    }
   if(errflag==1)
     printf("Eingabefehler - Eingabe-Beispiel: E2-E4 oder ? zum Stellung zeigen\n");
   if(errflag==2) printf("Ungueltiger Zug\n");
  }
 return zug;
}
#endif
