/* schach-engine.cc			letzte Aenderung: 11.11.2016 */
#define ENGINE_VERSION 1.0
/*
einfaches Schachprogramms in c++

Einbinden ins Hauptprogramm:
#define AVRSCHACH  //auskommentieren fuer Linux-Version
#include "schach-avr.cc"  //so im Hauptprogramm einfuegen (nach den andern includes)

History:
30.8.2016	 Erstellung (RP)
16.10.2016	 AVR-Variante
31.10.           Hauptprogramm ausgelagert, so dass fast alles gleich fuer Linux- und AVR-Variante
31.10.           Optimierung in stellung_bewerten: figuren_punkte() rechnet jetzt gleichzeitig fuer beide Farben
5.11.            spielerzug() ins Hauptprogramm ausgelagert
11.11.           Bauernumwandlung (Promotion) programmiert: dazu bei Zug und Zugliste unbenutzte Bits benutzt
12.11.2016  1.0  Aufspaltung in schach-engine.h und schach-engine.cc zum separaten optimierten Uebersetzen

*/

//#define AVRSCHACH   //fuer AVR-Variante, sollte im Hauptprogramm definiert werden
//#define BAUERNSPIEL //auskommentieren fuer normales Schach

#ifdef AVRSCHACH

#define F_CPU 20000000UL
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>
#include <avr/eeprom.h>
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t

#else //AVRSCHACH

typedef unsigned char uchar;
#define lcd_printf printf

#endif

#include "schach-engine.h"

/************** Klassen und Hauptteil des Programms *******************/
#define LEER 0

#define RESERVEBITS 0x38
#define FIGUR  0x07
#define BAUER     1
#define SPRINGER  2
#define LAEUFER   3
#define TURM      4
#define DAME      5
#define KOENIG    6

#define FARBE   0x40
#define WEISS   0
#define SCHWARZ 0x40

#define MOVFLAG 0x80 //bei Bauern fuer en passant, bei Koenig u. Turm fuer Rochade
#define ENPASSFLAG 0x80 //bei Bauern fuer en passant, bei Koenig u. Turm fuer Rochade

#define MAXPUNKTE 32000

#ifdef BAUERNSPIEL
#define MAXBAUERPUNKTE 10000 //fuer Bauernspiel
#else
#define MAXBAUERPUNKTE 900 //fuer Umwandlung von Bauer in Dame
#endif

bool Brett::gewonnen(uchar farbe)
{
#ifdef BAUERNSPIEL
 //provisorisch: Gewonnen wenn ein Bauer gegnerische Grundlinie erreicht
 //unentschieden wenn jemand nicht mehr fahren kann
 int8_t i;
 if(farbe==WEISS)
  {
   for(i=64-8;i<64;i++)
    if((feld[i]&FIGUR)==BAUER) return true;
  }
 else
  {
   for(i=0;i<8;i++)
    if((feld[i]&FIGUR)==BAUER) return true;
  }

#else

 uchar gegnerfarbe=WEISS+SCHWARZ-farbe;
 if(isincheck((*this),gegnerfarbe) && !istzugmoeglich(gegnerfarbe)) return true;
 //if(!isincheck((*this),gegnerfarbe) && !istzugmoeglich(gegnerfarbe)) return unentschieden;

#endif
 return false;
}

#ifndef AVRSCHACH
void Brett::print()
{
 int i,j,k;
 for(j=64-8;j>=0;j-=8)
  {
   k = (j>>3)&1;
   for(i=0;i<8;i++,k^=1)
    {switch(feld[j+i]&(FIGUR+FARBE))
      {case 0: printf((k==0)?" :: ":" .. "); break;
       case BAUER+WEISS: printf(" wB "); break;
       case BAUER+SCHWARZ: printf(" sB "); break;
       case KOENIG+WEISS: printf(" wK "); break;
       case KOENIG+SCHWARZ: printf(" sK "); break;
       case TURM+WEISS: printf(" wT "); break;
       case TURM+SCHWARZ: printf(" sT "); break;
       case SPRINGER+WEISS: printf(" wS "); break;
       case SPRINGER+SCHWARZ: printf(" sS "); break;
       case LAEUFER+WEISS: printf(" wL "); break;
       case LAEUFER+SCHWARZ: printf(" sL "); break;
       case DAME+WEISS: printf(" wD "); break;
       case DAME+SCHWARZ: printf(" sD "); break;
      }
    }
   printf("\n");
  }
}
#endif

void Brett::grundstellung()
{
 int8_t i,j;
 uchar farbe;
 clear();
 //Bauern setzen:
 for(j=1,farbe=WEISS;j<=6;j+=5,farbe=SCHWARZ)
  for(i=0;i<8;i++)
    feld[j*8+i] = BAUER+farbe;
 //Koenige:
 feld[4] = KOENIG+WEISS+MOVFLAG; //MOVEFLAG gesetzt wenn Koenig noch nicht bewegt wurde
 feld[4+7*8] = KOENIG+SCHWARZ+MOVFLAG;
#ifndef BAUERNSPIEL
 //restliche Figuren:
 feld[0] = feld[7] = TURM+WEISS+MOVFLAG; //MOVEFLAG gesetzt wenn Turm noch nicht bewegt wurde
 feld[1] = feld[6] = SPRINGER+WEISS;
 feld[2] = feld[5] = LAEUFER+WEISS;
 feld[3] = DAME+WEISS;
 feld[56] = feld[63] = TURM+SCHWARZ+MOVFLAG; //MOVEFLAG gesetzt wenn Turm noch nicht bewegt wurde
 feld[57] = feld[62] = SPRINGER+SCHWARZ;
 feld[58] = feld[61] = LAEUFER+SCHWARZ;
 feld[59] = DAME+SCHWARZ;
#endif
}

Brett spielbrett;

const char *Zug::print()
{
 static char zugtext[6];
 //char figurabk[7]={' ','P','N','B','R','Q','K'}; //englische Figurenabkuerzungen
 char promofigurabk[4]={'N','B','R','Q'}; //englische Figurenabkuerzungen
 int8_t i=0;
 uchar v=(von&IMASK), n=(nach&IMASK);
 zugtext[i++] = 'A'+(v%8);  zugtext[i++] = '1'+(v/8);
 if((von&PROMOFLAG)==0) zugtext[i++] = '-';
 zugtext[i++] = 'A'+(n%8); zugtext[i++] = '1'+(n/8);
 if(von&PROMOFLAG) zugtext[i++] = promofigurabk[nach>>6];
 zugtext[i] = 0;
 return zugtext;
}

Zugliste* Zugliste::zugsuchen(Zug zug)
{
 Zugliste *p;
 for(p=this; p!=NULL && (p->von!=zug.von || p->nach!=zug.nach); p=p->next) {}
 return p;
}

void Zugliste::vertauschen(Zugliste *a,Zugliste *b)
{
 uchar h=a->von; a->von=b->von; b->von=h;
 h=a->nach; a->nach=b->nach; b->nach=h;
 int16_t h2=a->bewertung; a->bewertung=b->bewertung; b->bewertung=h2;
}

void Zugliste::addsorted(uchar from,uchar to,int16_t wert)
{
 if(istleer()) {von=from; nach=to; bewertung=wert; return;}
#ifdef AVRSCHACH
 Zugliste *neu=(Zugliste*)malloc(sizeof(Zugliste));
 if(neu==NULL) {fehler_zu_wenig_ram(); return;}
#else
 Zugliste *neu=new Zugliste[1];
#endif
 neu->von=from; neu->nach=to; neu->bewertung=wert;
 if(wert>bewertung)
  {
   neu->next=next;
   next=neu;
   vertauschen(this,neu);
   return;
  }
 Zugliste *p;
 for(p=this; p->next!=NULL && p->next->bewertung >= wert; p=p->next) {}
 neu->next=p->next;
 p->next=neu;
}

int8_t koenigposition(Brett& brett,uchar farbe)
{
 int8_t i,j;
 if(farbe==WEISS)
  {
   for(i=4,j=3;i<8;j--,i++)
    {
     if((brett[i]&(FIGUR+FARBE))==KOENIG+WEISS) return i;
     if((brett[j]&(FIGUR+FARBE))==KOENIG+WEISS) return j;
    }
   for(;i<64;i++)
     if((brett[i]&(FIGUR+FARBE))==KOENIG+WEISS) return i;
  }
 else
  {
   for(j=60,i=59;j<64;i--,j++)
    {
     if((brett[j]&(FIGUR+FARBE))==KOENIG+SCHWARZ) return j;
     if((brett[i]&(FIGUR+FARBE))==KOENIG+SCHWARZ) return i;
    }
   for(;i>=0;--i)
     if((brett[i]&(FIGUR+FARBE))==KOENIG+SCHWARZ) return i;
  }
#ifndef AVRSCHACH
 //printf("Error: Koenig nicht gefunden\n");//test
#endif
 return 0;//sollte nie erreicht werden
}

bool isincheck(Brett& brett,uchar farbe) //steht Koenig im Schach?
{
 int8_t i,j,k,m;
 uchar gegnerfarbe=WEISS+SCHWARZ-farbe;
 int8_t n=koenigposition(brett,farbe);
 i=n%8;
 j=n/8;
 //Test auf Schach durch Springer:
 for(k=i-2; k <= i+2; k+=4)
  if(k>=0 && k<=7)
   {
    if(j>0 && (brett[(j-1)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
    if(j<7 && (brett[(j+1)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
   }
 for(k=i-1; k <= i+1; k+=2)
  if(k>=0 && k<=7)
   {
    if(j>1 && (brett[(j-2)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
    if(j<6 && (brett[(j+2)*8+k]&(FIGUR+FARBE))==SPRINGER+gegnerfarbe) return true;
   }
 //Test auf gegnerischen Koenig:
 for(k=i-1; k <= i+1; k++)
 for(m=j-1; m <= j+1; m++)
  {
   if(k==i && m==j) continue;
   if(k>=0 && k<=7 && m>=0 && m<=7 && (brett[m*8+k]&(FIGUR+FARBE))==KOENIG+gegnerfarbe) return true;
  }
 //Test auf Schach durch Turm oder Dame:
 uchar fe;
 for(k=i,m=n;--k>=0;)
  {
   fe=brett[--m];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 for(k=i,m=n;++k<=7;)
  {
   fe=brett[++m];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 for(k=j,m=n;--k>=0;)
  {
   fe=brett[m-=8];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 for(k=j,m=n;++k<=7;)
  {
   fe=brett[m+=8];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==TURM || fe==DAME) return true;
     break;
    }
  }
 //Test auf Schach durch Laeufer oder Dame oder Bauer:
 for(k=i-1,m=j-1; k>=0 && m>=0; --k,--m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==SCHWARZ && m==j-1) return true;
     break;
    }
  }
 for(k=i+1,m=j-1; k<=7 && m>=0; ++k,--m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==SCHWARZ && m==j-1) return true;
     break;
    }
  }
 for(k=i-1,m=j+1; k>=0 && m<=7; --k,++m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==WEISS && m==j+1) return true;
     break;
    }
  }
 for(k=i+1,m=j+1; k<=7 && m<=7; ++k,++m)
  {
   fe=brett[m*8+k];
   if(fe!=LEER)
    {
     if((fe&FARBE)!=gegnerfarbe) break;
     fe &= FIGUR;
     if(fe==LAEUFER || fe==DAME) return true;
     if(fe==BAUER && farbe==WEISS && m==j+1) return true;
     break;
    }
  }
 return false;
}

bool Brett::istzugmoeglich(uchar farbe)
{
 Zugliste zugliste;
 alle_moeglichen_zuege((*this),farbe,&zugliste,0);
 if(zugliste.istleer()) return false;
 return true;
}

void Zugliste::add(uchar from,uchar to,Brett& brett,uchar farbe,int8_t suchtiefe)
{
 Brett hilfsbrett;
 zug_machen(brett,from,to,hilfsbrett);
 if(isincheck(hilfsbrett,farbe)) return; //wenn Koenig im Schach steht --> ungueltiger Zug
 int16_t wert=stellung_bewerten(hilfsbrett,farbe,suchtiefe);
 addsorted(from,to,wert);
}

void Zugliste::addb(uchar from,uchar to,Brett& brett,uchar farbe,int8_t suchtiefe)
{
 Brett hilfsbrett;
 zug_machen(brett,from,to,hilfsbrett);
 if(isincheck(hilfsbrett,farbe)) return; //wenn Koenig im Schach steht --> ungueltiger Zug
 if((farbe==WEISS && to>=64-8) || (farbe==SCHWARZ && to<8)) //gegnerische Grundlinie erreicht?
  {//alle Varianten von Bauernumwandlungen:
   from |= PROMOFLAG;
   for(uchar promot=SPRINGER; promot<=DAME; promot++)
    {
     hilfsbrett[to]++; //aus Bauer einen Springer machen, aus Springer einen Laeufer ... bis Dame
     int16_t wert=stellung_bewerten(hilfsbrett,farbe,suchtiefe);
     addsorted(from, to+((promot-2)<<6), wert);
    }
  }
 else //sonst ohne Umwandlung
  {
   int16_t wert=stellung_bewerten(hilfsbrett,farbe,suchtiefe);
   addsorted(from,to,wert);
  }
}

bool ist_schraeger_zug(uchar von,uchar nach)
{
 return (nach==von+9 || nach==von+7 || nach==von-9 || nach==von-7);
}

void en_passant_schlagen(Brett& brett,uchar von,uchar nach)
{
 if(nach==von+9 || nach==von+7) brett[nach-8]=LEER;
 else if(nach==von-9 || nach==von-7) brett[nach+8]=LEER;
}

void zug_machen(Brett& brett,uchar von,uchar nach,Brett& hilfsbrett)
{
 int8_t i;
 //uchar promo = (von&PROMOFLAG) ? (nach>>6)+2 : 0; //Bauerwandlung noch nicht in diesem Unterprogramm machen!
 von &= IMASK;
 nach &= IMASK;
 for(i=0;i<64;i++)
  {
   if((brett[i]&FIGUR)==BAUER)
     hilfsbrett[i] = brett[i] & ~MOVFLAG; //eventuelle en passant Flags loeschen
   else hilfsbrett[i] = brett[i];
  }
 hilfsbrett[nach]=brett[von];
 hilfsbrett[von]=LEER;
 hilfsbrett[nach] &= ~MOVFLAG;//MOVFLAG loeschen (auch bei Koenig u. Turm wenn gezogen)
 if((brett[von]&FIGUR)==BAUER)
  {
   if(ist_schraeger_zug(von,nach) && brett[nach]==LEER) en_passant_schlagen(hilfsbrett,von,nach);
   else if((nach==von+2*8 || nach==von-2*8)) //wenn Bauer 2 Schritte gezogen
      hilfsbrett[nach] |= MOVFLAG; //Flag fuer moegliches en passant setzen
  }
 else if((brett[von]&FIGUR)==KOENIG)
  {
   if(nach==von+2 || nach==von-2) //Rochade?
      {
       uchar turmvon,turmnach;
       if(nach>von) {turmvon=nach+1; turmnach=nach-1;}
       else         {turmvon=nach-2; turmnach=nach+1;}
       hilfsbrett[turmnach] = brett[turmvon] & ~MOVFLAG; //Turm ziehen
       hilfsbrett[turmvon] = LEER;
      }
   hilfsbrett[nach] &= ~MOVFLAG; //Flag zum Rochade erlauben loeschen
  }
}

void zug_wirklich_machen(Brett& brett,Zug zug)
{
 Brett hilfsbrett;
 zug_machen(brett,zug.von,zug.nach,hilfsbrett);
 if(zug.getpromo()!=0) hilfsbrett[zug.nach&IMASK] += zug.getpromo()-BAUER;
 brett=hilfsbrett;
}

int figuren_punkte(Brett& brett,uchar farbe)
{
 int8_t i;
 int punkte=0;
 uchar fig,figfarbe;
 for(i=0;i<64;i++)
  {
   if((fig=brett[i]&FIGUR)==0) continue;
   figfarbe = brett[i]&FARBE;
   if(figfarbe==farbe)  punkte += material[fig-1];
   else                 punkte -= material[fig-1];
   if(fig==BAUER) //Bei Bauern pro vorgeruecktes Feld 1 weiterer Punkt
    {
     if(figfarbe==farbe)
      {
       if(figfarbe==WEISS)
	{if(i>=16)    punkte += (i>64-8) ? MAXBAUERPUNKTE : (i-8)/8;}
       else //SCHWARZ
	{if(i<=63-16) punkte += (i<8) ? MAXBAUERPUNKTE : (63-8-i)/8;}
      }
     else //figfarbe!=farbe
      {
       if(figfarbe==WEISS)
	{if(i>=16)    punkte -= (i>64-8) ? MAXBAUERPUNKTE : (i-8)/8;}
       else //SCHWARZ
	{if(i<=63-16) punkte -= (i<8) ? MAXBAUERPUNKTE : (63-8-i)/8;}
      }
    }
   else if(fig==KOENIG)
    {
     if(brett[i]&MOVFLAG) {if(figfarbe==farbe) punkte++; else punkte--;} //zusaetzlicher Punkt wenn Koenig noch Rochadefaehig
    }
  }
 //printf("figuren_punkte() farbe=%d punkte=%d\n",farbe,punkte);//test
 return punkte;
}

int16_t stellung_bewerten(Brett& brett,uchar farbe,int8_t suchtiefe) //Wert fuer farbe wenn gegnerfarbe am Zug ist
{
 int16_t bewertung;
 int8_t gegnerfarbe=WEISS+SCHWARZ-farbe;
 if(suchtiefe==0)
  {
   bewertung = figuren_punkte(brett,farbe);
  }
 else
  {
   Zug zug=besterzug(brett,gegnerfarbe,suchtiefe-1);
   if(zug.von==zug.nach) //kein Zug mehr moeglich
    {
     if(isincheck(brett,gegnerfarbe)) bewertung = MAXPUNKTE; //farbe hat gewonnen
     else bewertung = 0; //Unentschieden
    }
   else bewertung = -zug_bewerten(brett,gegnerfarbe,zug,suchtiefe-1);
  }
 //printf("Farbe=%d bewertung=%d\n",farbe,bewertung);//test
 return bewertung;
}

int16_t zug_bewerten(Brett& brett,uchar farbe,Zug& zug,int8_t suchtiefe)
{
 Brett hilfsbrett;
 zug_machen(brett, zug.von, zug.nach, hilfsbrett);
 if(zug.getpromo()!=0) hilfsbrett[zug.nach&IMASK] += zug.getpromo()-BAUER;
 int16_t wert=stellung_bewerten(hilfsbrett,farbe,suchtiefe);
 return wert;
}

#ifdef AVRSCHACH

//Zufallszahlen auf AVR-Microkontroller:
#define MYRAND_MAX (1<<14)
int myrand()
{
 return adwandler_zufallszahl&(MYRAND_MAX-1);
}
int zufall(int max) //Zufallszahl zwischen 0 und max (inklusive 0, exclusive max)
{
 int z = myrand()/(MYRAND_MAX/max);
 //printf("Test zufall(%d) --> z=%d (RAND_MAX=%d)\n",max,z,RAND_MAX);
 if(z==max) --z;
 return z;
}

#else

//Zufallszahlen unter Linux:
#include <sys/times.h>
int zufall(int max) //Zufallszahl zwischen 0 und max (inklusive 0, exclusive max)
{
 static int seed=0;
 if(seed==0)
  {
   struct tms buf;
   clock_t t=times(&buf);
   srand(seed=t);
  }
 int z = rand()/(RAND_MAX/max);
 //printf("Test zufall(%d) --> z=%d (RAND_MAX=%d)\n",max,z,RAND_MAX);
 if(z==max) --z;
 return z;
}

#endif

Zug Zugliste::getbest(Brett& brett,uchar farbe)
{
 Zugliste *p;
 int n,i;
 //Da schon mit addsort() sortiert wurde, ist erster Eintrag mit bester Bewertung.
 int16_t fastbestebewertung=bewertung-1;
 for(p=next,n=1; p!=NULL && p->bewertung >= fastbestebewertung ;p=p->next)
  {
   n++; //Anzahl fast gleich gute Bewertungen zaehlen
  }
 p=this;
 if(n>1) //bei mehreren fast gleich guten Zuegen, zufaelligen auswaehlen
  {
   i=zufall(n);
   for(;i>0;p=p->next,i--) {}
  }
 Zug zug;
 zug.set(p->von,p->nach);
 return zug;
}

void alle_moeglichen_zuege(Brett& brett,uchar farbe,Zugliste *zugliste,int8_t suchtiefe)
{
 int8_t i;
 uchar fig,col;
 for(i=0;i<64;i++)
  {
   if(brett[i]==LEER || (col=brett[i]&FARBE)!=farbe) continue;
   fig = brett[i]&FIGUR;
   if(fig==BAUER)
    {
     if(col==WEISS)
      {
       if(brett[i+8]==LEER) zugliste->addb(i,i+8,brett,farbe,suchtiefe);
       if(i<16 && brett[i+8]==LEER && brett[i+16]==LEER) zugliste->add(i,i+16,brett,farbe,suchtiefe);
       if(i>=4*8 && i<5*8) //Bauer auf 5.Linie: auf en passant testen
	{
	 if(i > 4*8 && brett[i-1]==BAUER+SCHWARZ+MOVFLAG) zugliste->add(i,i+8-1,brett,farbe,suchtiefe);
	 if(i+1<5*8 && brett[i+1]==BAUER+SCHWARZ+MOVFLAG) zugliste->add(i,i+8+1,brett,farbe,suchtiefe);
	}
       if(i%8!=0 && brett[i+7]!=LEER && (brett[i+7]&FARBE)==SCHWARZ) zugliste->addb(i,i+7,brett,farbe,suchtiefe);
       if(i%8!=7 && brett[i+9]!=LEER && (brett[i+9]&FARBE)==SCHWARZ) zugliste->addb(i,i+9,brett,farbe,suchtiefe);
      }
     else //if(col==SCHWARZ)
      {
       if(brett[i-8]==LEER) zugliste->addb(i,i-8,brett,farbe,suchtiefe);
       if(i>=64-16 && brett[i-8]==LEER && brett[i-16]==LEER) zugliste->add(i,i-16,brett,farbe,suchtiefe);
       if(i>=3*8 && i<4*8) //Bauer auf 4.Linie: auf en passant testen
	{
	 if(i > 3*8 && brett[i-1]==BAUER+WEISS+MOVFLAG) zugliste->add(i,i-8-1,brett,farbe,suchtiefe);
	 if(i+1<4*8 && brett[i+1]==BAUER+WEISS+MOVFLAG) zugliste->add(i,i-8+1,brett,farbe,suchtiefe);
	}
       if(i%8!=0 && brett[i-9]!=LEER && (brett[i-9]&FARBE)==WEISS) zugliste->addb(i,i-9,brett,farbe,suchtiefe);
       if(i%8!=7 && brett[i-7]!=LEER && (brett[i-7]&FARBE)==WEISS) zugliste->addb(i,i-7,brett,farbe,suchtiefe);
      }
    }
   else if(fig==KOENIG)
    {
     if((i%8)<7)
	{if(brett[i+1]==LEER || (brett[i+1]&FARBE)!=farbe) zugliste->add(i,i+1,brett,farbe,suchtiefe);
	 if(i<64-8 && (brett[i+9]==LEER || (brett[i+9]&FARBE)!=farbe)) zugliste->add(i,i+9,brett,farbe,suchtiefe);
	 if(i>=8   && (brett[i-7]==LEER || (brett[i-7]&FARBE)!=farbe)) zugliste->add(i,i-7,brett,farbe,suchtiefe);
	}
     if((i%8)>0)
	{if(brett[i-1]==LEER || (brett[i-1]&FARBE)!=farbe) zugliste->add(i,i-1,brett,farbe,suchtiefe);
	 if(i<64-8 && (brett[i+7]==LEER || (brett[i+7]&FARBE)!=farbe)) zugliste->add(i,i+7,brett,farbe,suchtiefe);
	 if(i>8    && (brett[i-9]==LEER || (brett[i-9]&FARBE)!=farbe)) zugliste->add(i,i-9,brett,farbe,suchtiefe);
	}
     if(i<64-8 && (brett[i+8]==LEER || (brett[i+8]&FARBE)!=farbe)) zugliste->add(i,i+8,brett,farbe,suchtiefe);
     if(i >= 8 && (brett[i-8]==LEER || (brett[i-8]&FARBE)!=farbe)) zugliste->add(i,i-8,brett,farbe,suchtiefe);
     //hier noch Rochade:
     // Rochade ist nur erlaubt wenn: Koenig und beteiligter Turm noch nicht bewegt wurden,
     //                               alle Felder zwischen Koenig und Turm leer sind,
     //                               Koenig nicht im Schach steht und ueber kein bedrohtes Feld fahren muss.
     if(col==WEISS)
	{
	 if(brett[i]&MOVFLAG && !isincheck(brett,farbe)) //Koenig noch nicht bewegt, und nicht im Schach
	  {
	   int8_t k;
	   if((brett[0]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer grosse Rochade noch nicht bewegt
	      && brett[1]==LEER && brett[2]==LEER && brett[3]==LEER) //und alle Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && --k>=i-2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER; 
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i-2,brett,farbe,suchtiefe); //grosse Rochade
	    }
	   if((brett[7]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer kleine Rochade noch nicht bewegt
	      && brett[6]==LEER && brett[5]==LEER)   //und beide Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && ++k<=i+2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER;
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i+2,brett,farbe,suchtiefe); //kleine Rochade
	    }
	  }
	}
     else //if(col==SCHWARZ) //Rochade fuer Schwarzen Koenig
	{
	 if(brett[i]&MOVFLAG && !isincheck(brett,farbe)) //Koenig noch nicht bewegt, und nicht im Schach
	  {
	   int8_t k;
	   if((brett[56]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer grosse Rochade noch nicht bewegt
	      && brett[57]==LEER && brett[58]==LEER && brett[59]==LEER) //und alle Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && --k>=i-2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER; 
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i-2,brett,farbe,suchtiefe); //grosse Rochade
	    }
	   if((brett[63]&(FIGUR+MOVFLAG))==TURM+MOVFLAG //Turm fuer kleine Rochade noch nicht bewegt
	      && brett[62]==LEER && brett[61]==LEER)   //und beide Felder zwischen Turm und Koenig leer
	    {
	     bool ok=true;
	     uchar king=brett[i];
	     brett[i]=LEER; 
	     for(k=i; ok && ++k<=i+2;)
		{
		 brett[k]=king; //Probehalber Koenig 2 mal 1 Feld weit ziehen
		 if(isincheck(brett,farbe)) ok=false;
		 brett[k]=LEER;
		}
	     brett[i]=king; //Koenig wieder zurueckstellen
	     if(ok) zugliste->add(i,i+2,brett,farbe,suchtiefe); //kleine Rochade
	    }
	  }
	}
    }
#ifdef BAUERNSPIEL
   else lcd_printf("Error: unbekannte Figur fig=%d\n",fig);//test
#else
   else if(fig==SPRINGER)
    {
     int8_t j,k, ix=i%8, jy=i/8;
     for(k=ix-2;k<=ix+2;k+=4)
      for(j=jy-1;j<=jy+1;j+=2)
      {
       if(k>=0 && k<8 && j>=0 && j<8)
	{
	 int8_t n=j*8+k;
	 if(brett[n]==LEER || (brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	}
      }
     for(k=ix-1;k<=ix+1;k+=2)
      for(j=jy-2;j<=jy+2;j+=4)
      {
       if(k>=0 && k<8 && j>=0 && j<8)
	{
	 int8_t n=j*8+k;
	 if(brett[n]==LEER || (brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	}
      }
    }
   else
    {
     int8_t ix=i%8, jy=i/8;
     if(fig==TURM || fig==DAME)
      {
       int8_t n,k;
       for(k=ix-1,n=i-1;k>=0;k--,n--)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix+1,n=i+1;k<8;k++,n++)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=jy-1,n=i-8;k>=0;k--,n-=8)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=jy+1,n=i+8;k<8;k++,n+=8)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
      }
     if(fig==LAEUFER || fig==DAME)
      {
       int8_t n,k,j;
       for(k=ix-1,j=jy-1,n=i-9; k>=0 && j>=0; k--,j--,n-=9)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix+1,j=jy+1,n=i+9; k<8 && j<8; k++,j++,n+=9)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix-1,j=jy+1,n=i+7; k>=0 && j<8; k--,j++,n+=7)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
       for(k=ix+1,j=jy-1,n=i-7; k<8 && j>=0; k++,j--,n-=7)
	{
	 if(brett[n]==LEER)
	   zugliste->add(i,n,brett,farbe,suchtiefe);
	 else
	  {
	   if((brett[n]&FARBE)!=farbe) zugliste->add(i,n,brett,farbe,suchtiefe);
	   break;
	  }
	}
      }
    }
#endif //BAUERNSPIEL
  }
}

Zug besterzug(Brett& brett,uchar farbe,int8_t suchtiefe)
{
 Zugliste zugliste;
 alle_moeglichen_zuege(brett,farbe,&zugliste,suchtiefe);
 return zugliste.getbest(brett,farbe);
}
