/* kartonschach.cc       letzte Aenderungen: 27.4.2023, 7.4.2024, 24.9.2024
Erkennung von Schachzuegen auf einem Kartonbrett
Die Hallsensoren sind unter dem Karton montiert, also Abstand
zwischen Figuren-Magnete und Sensoren nur etwa 2mm
*/
#define VERSION "0.8" //Version des Programms
#define EEVERSION 100 //Version der EEPROM-Belegung
#define SPULENVERSION //Spulen als Sensoren statt Hallsensoren
//#define SPULENFREQUENZTEST //erster Test um richtige Spulenfrequenz einzustellen
//#define SPULENTEST //alle Spulen testen
/*
 Prozessor: Atmega1284P, 20MHz Quarz
 Schaltung: schaltung/hall8x8.sch
 Details fuer Spulenversion: schaltung/spulenmatrix.sch

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 16.10.2016     Erstellung
 5.11.2016      neues zug_aus_sensoren_ermitteln()
 11.11.         Bauerumwandlung
 30.12.         UART testen
 1.1.2017       Verbindung ueber UART mit Linux um Stockfisch als Engine zu verwenden
 16.2.2023  0.3 Soft-UART auf Pins B6 B7 fuer Verbindung zu Roboterarm
 21.2.2023      Bei Rochade sicherstellen dass Computerfarbe gesetzt ist
 23.2.2023      SLEEP_MODE_IDLE angefuegt, neu Seitenwahl um Brettlage zu aendern
 27.2.2023      Fehler in suart_parameter_senden() korrigiert. Entprellte Tasten von 4ms auf 8ms erhoeht
                Definition von SCHWARZ korrigiert, gleich wie in schach-engine.h (anders in robot.cc)
  1.3.2023  0.4 Verbesserungsversuch bei Sensorproblemen
  5.3.2023      Bei Spielende letzter Zug auch noch an Linux-Computer uebermitteln
 10.3.2023      Kommandos fuer Werte calibrieren an robot.cc schicken (MODUS_ROBOTCALIB)
 19.3.2023  0.5 Spulenversion: statt Hallsensoren Spulen verwenden (Prinzip Metalldetektor)
 27.4.2023  0.6 Test auf 3 mal gleiche Stellung und 50-Zug-Regel
                (50 Zuege ohne schlagen oder Bauer ziehen --> untentschieden)
 28.3.2024  0.7 Erweiterung fuer Schach960
  7.4.2024      Stellung in EEPROM speichern, und koenig_startpos, turm1_startpos, turm2_starpos
                setzen vereinfacht mit neuer Funktion check_stellung()
 19.9.2024      Verbesserungen zur Erkennung von Rochade von Benutzer gemacht. (neu 19.9.24)
                Bessere Fehlermeldungen bei Rochade-Fehler
		3 Teststellungen fuer Schach 960 auswaehlbar
 21.9.2024  0.8 letzter Zug zuruecknehmen (wenn z.B. Sensoren falscher Zug erkannt)

 Todo:
 - aktuelles Spiel/Stellung im EEPROM speichern
 - aktuelle Stellung auf LCD anzeigen, PCINT fuer Taste rechts

*/
//#define STACKTEST 1
//#define GESCHWINDIGKEITSTEST
#define SCHACH960

//#define DEBUG  //zur Fehlersuche
//#define DEBUG2 //Fehlersuche Schach960
//#define DEBUG3 //Fehlersuche Rochade

#ifdef SPULENVERSION
#define ENTPRELLPAUSE 20
#define PROZDIFF 20  //soviele Prozent Unterschied um Figur setzen zu erkennen
#define PROZGRENZ (100-PROZDIFF) //soviele Prozent bleiben uebrig bei Figur setzen
#else
#define ENTPRELLPAUSE 200 //zusaetzliche Pause in ms fuer entprelltes Sensorfeld
#endif
//#define UMLAUTTEST
#define MITUART  //muss aktiviert sein fuer Serielle Schnittstelle
//#define UARTTEST

#define F_CPU 20000000UL

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>
#include <avr/eeprom.h>

/**** Vordeklarationen ****/
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t

#define TASTE_OBEN (1<<0)
#define TASTE_UNTEN (1<<1)
#define TASTE_RECHTS (1<<2)
#define TASTE_LINKS (1<<3)
#define TASTE_MITTE (1<<4)
#define TASTE_ALLE 0x1F

void startmenu(int8_t modus,int8_t auswahlzeile);
uchar auswahlsetzen(uchar taste,int8_t auswahlzeile,int8_t d);
int8_t modus_wechseln(int8_t modus,int8_t* pwahl);
void zug_aus_sensoren_ermitteln(uchar *von,uchar *nach,uchar farbe);
bool zug_zuruecknehmen();

/**** Tastenentprellung ****/
#define PIN_TASTER PINB
void entprell(); //Vordeklaration fuer Tasten-Entprellung - declaration for debouncing function
static volatile char entprelltetasten=0xFF; //Bit auf 0 wenn Taste gedrueckt, wird von entprell() rueckges.
static char status1=0xFF,status2=0xFF;
static volatile char tastegedrueckt=0; //Bit gesetzt wenn Taste gedrueckt, muss aktiv rueckgesetzt werden

void entprell() //Tasten-Entprellung fuer mehrere Tasten unabhaengig voneinander
{               //Debouncing several buttons independent at the same time
 char pin,ung;
 pin = PIN_TASTER;
 //In "ung" diejenigen Bits setzen die ungleich zu status1 oder status2 sind:
 //Set bits to 1 if different to status1 or different to status2:
 ung = (pin^status1) | (status2^status1);
 //Bits so lassen wenn sie in "ung" auf 1 sind, oder gleich wie in "pin" setzen wenn sie in "ung" auf 0 sind:
 //Set bits to previous state when it is 1 in "ung", or set to same as "pin" when it is 0 in "ung":
 entprelltetasten = (entprelltetasten & ung) | (pin & (ung^0xFF));
 status2=status1;
 status1=pin;
 tastegedrueckt |= (entprelltetasten^0xFF);
}

void warte_losgelassen_mitte()
{
 while((entprelltetasten&TASTE_MITTE)==0) {} //warte bis taste losgelassen
 tastegedrueckt &= ~TASTE_MITTE; //entsprechendes Bit ruecksetzen
}

void warte_losgelassen_rechts()
{
 while((entprelltetasten&TASTE_RECHTS)==0) {}
 tastegedrueckt &= ~TASTE_RECHTS;
}

void warte_losgelassen_links()
{
 while((entprelltetasten&TASTE_LINKS)==0) {}
 tastegedrueckt &= ~TASTE_LINKS;
}

void warte_losgelassen_oben()
{
 while((entprelltetasten&TASTE_OBEN)==0) {}
 tastegedrueckt &= ~TASTE_OBEN;
}

void warte_losgelassen_unten()
{
 while((entprelltetasten&TASTE_UNTEN)==0) {}
 tastegedrueckt &= ~TASTE_UNTEN;
}

void warte_taste_losgelassen()
{
 while((entprelltetasten&TASTE_ALLE)!=TASTE_ALLE) {} //warte bis alle Tasten losgelassen
 tastegedrueckt=0; //alle Bits ruecksetzen
}

bool taste_gedrueckt(uchar taste)
{
 return ((entprelltetasten&taste)!=taste);
}

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=0,tage=0;

//#define Rest1 (F_CPU%1000) //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (F_CPU/1000-1)

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
 set_sleep_mode(SLEEP_MODE_IDLE);
 sleep_enable(); //Schlaf-Modus waehlen, aber noch nicht schlafen
}

void zeit_aktualisieren()
{
 uint ms=millisec;
#ifdef Rest1
 uchar mehr;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
#else
 ++ms;
#endif
 if(ms==1000)
  {ms=0;
#ifdef Rest1
   mehr=0;
#endif
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0;
	 ++tage; //neuertag=1;
	}
      }
    }
  }
 else if((ms&0x07)==1) entprell(); //Tastenentprellung alle 8 Millisekunden - Debouncing every 8 milliseconds
 millisec=ms;
#ifdef Rest1
 OCR1A=NormalerCompwert+mehr;
#endif
}

ISR(TIMER1_COMPA_vect, ISR_NAKED)
{
 __asm__("push	r1");
 __asm__("push	r0");
 __asm__("in	r0, 0x3f");
 __asm__("push	r0");
 __asm__("eor	r1, r1");
 __asm__("in	r0, 0x3b");
 __asm__("push	r0");
 sei(); //Interrupts moeglichst schnell wieder erlauben fuer Timer0
 __asm__("push	r18");
 __asm__("push	r19");
 __asm__("push	r20");
 __asm__("push	r21");
 __asm__("push	r22");
 __asm__("push	r23");
 __asm__("push	r24");
 __asm__("push	r25");
 __asm__("push	r26");
 __asm__("push	r27");
 __asm__("push	r30");
 __asm__("push	r31");
 zeit_aktualisieren();
 __asm__("pop  r31");
 __asm__("pop  r30");
 __asm__("pop	r27");
 __asm__("pop	r26");
 __asm__("pop	r25");
 __asm__("pop	r24");
 __asm__("pop	r23");
 __asm__("pop	r22");
 __asm__("pop	r21");
 __asm__("pop	r20");
 __asm__("pop	r19");
 __asm__("pop	r18");
 __asm__("pop	r0");
 __asm__("out	0x3b, r0");
 __asm__("pop	r0");
 __asm__("out	0x3f, r0");
 __asm__("pop	r0");
 __asm__("pop	r1");
 __asm__("reti"); //bei ISR_NAKED benoetigt!
}

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) {sleep_mode();}//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
  {sleep_mode();}//auf n. Millisekunde warten
}

void secwait(uchar n) //n Sekunden warten
{
 while(n!=0) {milliwait(1000); --n;}
}

void microwait(uchar n) //mindestens n Microsekunden warten
{
 while(--n!=0) _delay_us(1);
}

/******************************** AD-Wandler ******************************************/
static volatile uchar adamlaufen=0;
static volatile int messwert=0;
volatile uint adwandler_zufallszahl=0;

ISR(ADC_vect) //AD-Wandler Interrupt-Routine
{
 uchar c1,c2;
 c1=ADCL;
 c2=ADCH;
 messwert = (c2<<8)+c1;
 adwandler_zufallszahl += c1&3; //unterste 2 Bits zum Zufallszahlen generieren
 adamlaufen=0;
}

void ad_init(uchar adkanal)
{
//Prescaler Beispiel: 2^7=128, 20MHz/128=156.25kHz (muss im Bereich 50-200 liegen)
 //uchar PRESCALER; int pre;
 //for(PRESCALER=1,pre=2*200;  PRESCALER<7 && (F_CPU/1000)>pre;  PRESCALER++, pre *= 2)
 //          {}          //PRESCALER ausrechnen lassen
#define PRESCALER 7  //oder direkt definieren

 adamlaufen=1;

//Fuer Referenzspannung eine der folgenden 4 Varianten benutzen:
 //ADMUX = (0<<REFS0)+adkanal; //Externe Referenz an AREF
 ADMUX = (1<<REFS0)+adkanal; //AVCC als Referenz verwenden
 //ADMUX = (2<<REFS0)+adkanal; //Interne Referenz = 1.1V
 //ADMUX = (3<<REFS0)+adkanal; //Interne Referenz = 2.56V

 ADCSRA = (1<<ADEN)+(1<<ADSC)+(1<<ADIE)+PRESCALER;//Enable+Interruptenable+MessungStarten

 //while(adamlaufen)  ; //erste Wandlung abwarten und verwerfen
 //_delay_ms(1); 
}

void ad_start()
{
 adamlaufen=1;
 ADCSRA |= (1<<ADSC); //Starte AD-Wandlung
}

int ad_lesen()
{
 if(adamlaufen) return 0;
 return messwert;
}

/*
void ad_off()
{
 ADCSRA &= ~(1<<ADEN);
}
*/

//Umrechnungen von Bitwerten auf Millivolt:
//1024 = referenz, somit Bitrauschen = referenz/1024
#define REFERENZ 5000L //Versorgungsspannung (5.0 V) als Referenz
//#define REFERENZ 1100L //Referenzspannung 1.1 V
//#define REFERENZ 2560L //Referenzspannung 2.56 V
#define DU ((REFERENZ+1023)/1024) //REFERENZ/1024 aufgerundet = Ungenauigkeit von AD-Wandler

int mvolt(uint ad) //Messwert umrechnen in Millivolt
{
 return (REFERENZ*ad+512)/1024;
}

/***************************** Timer0 fuer Sensoren auslesen **************************/
static volatile uint sensorfeld[64];
static uint sensorCalibLeer[64]; //Calibrierdaten bei leerem Brett
#ifdef SPULENVERSION
static uint minDiff; //TODO: in Spulenversion nicht wirklich verwendet
#else
static uint minDiff; //Minimale Differenz wenn Figur gesetzt zu leerem Brett
static volatile uchar sensorlesezyklus=0; //wird jeweils erhoeht wenn alle 64 Sensoren gelesen
#endif
static uchar timer0status=0; //0=gestoppt, 1=laufend

#define F_SPULEN 217000L //Resonanzfrequenz der verwendeten Spulen-Kondensator-Schwingkreise

void timer0_init()
{
#ifdef SPULENVERSION
 TCCR0A = (1<<WGM01)|(0<<WGM00); //CTC-Modus, Interrupt bei Erreichen von OCRA
 TCCR0B = (1<<CS00); //Takt gleich F_CPU
 //OCR0A = 92-1; // 20MHz / 92 = 217 kHz = Frequenz der Oszillatoren die mit Spule und 3.3nF gebildet werden
 OCR0A = (F_CPU+F_SPULEN/2)/F_SPULEN - 1; //oder direkt berechnet
 TCNT0 = 0;
 TIMSK0 = (1<<OCIE0A); //Interrupt bei Erreichen von OCRA
#else
 TCCR0A = (0<<WGM01)|(0<<WGM00); //Normal-Modus, Interrupt bei Ueberlauf (8 Bit, also Teiler durch 256)
 TCCR0B = (1<<CS01); //Takt durch 8teilen (sollte mit Hallsensoren SS495A funktionieren)
 //TCCR0B = (1<<CS01)|(1<<CS00); //Takt durch 64teilen (fuer langsamere Sensoren)
 TCNT0 = 0;
 TIMSK0 = (1<<TOIE0); //Interrupt bei Ueberlauf
#endif
 timer0status=1;
}

void timer0_stop()
{
 TCCR0B = 0; //Takt fuer Timer0 anhalten
 timer0status=0;
}

#ifdef SPULENVERSION
static volatile uchar anregpin=0; //(1<<PC0) bis (1<<PC7)

ISR(TIMER0_COMPA_vect)
{
 PORTC = anregpin; //Frequenz auf entsprechenden Pin (PC0...PC7) ausgeben
 _delay_us(0.9);
 PORTC = 0;
}

#else

ISR(TIMER0_OVF_vect)
{
 // wird alle etwa 0.1 ms aufgerufen (102.4 usec)
 // 9765.625 Aufrufe = 1 Sekunde
 // 4 Cyclen pro Sensor, 64 Sensoren --> 4*64 = 256 Durchlaeufe benoetigt
 // 256*102.4usec = 26.2144 msec
 static uchar cyclus=0;
 static uchar j=0,maske=0x80;
 static uchar kanal=0;
 if(++cyclus==4) cyclus=0;
 switch(cyclus)
    {
     case 1: ad_init(kanal); break;
     case 2: ad_start(); break;
     case 3: sensorfeld[j+kanal]=ad_lesen();
             if(kanal==7) PORTC=0; //wenn letzte Reihe gelesen, Spannung fuer Spalte ausschalten
             break;
     default:
         if(++kanal==8)
	  {
	   kanal=0;
	   j += 8;
	   if(j==64) {j=0; maske=0x80; sensorlesezyklus++;}
	   else  maske >>= 1;
	   PORTC = maske; //Spannung fuer neue Spalte einschalten
	  }
	 break;
    }
}
#endif

/*************************** Serielle Schnittstelle ***********************************/
#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 20000000"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 19200UL      // Baudrate
//#define BAUD 9600UL      //test

// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

#ifdef MITUART
static bool uart_ok=false;
static bool stockflag=false; //gesetzt wenn Stockfisch als Engine verwendet werden soll

void uart0_init()
{
 //PRR |= (1<<PRUSART1); //Power Reduction Register: nicht gebrauchter UART
 //PRR &= ~(1<<PRUSART0); //UART0 Power Reduktion aus
 UBRR0 = UBRR_VAL;
 //UCSR0B = 0; //noch nichts senden und empfangen
 UCSR0B = (1<<RXEN0)|(1<<TXEN0); // UART TX und RX einschalten
 //UCSR0B = (1<<RXEN0); // nur UART RX einschalten
 //UCSR0C = (3<<UCSZ00); // Asynchron 8N1
 UCSR0C = (3<<UCSZ00)|(1<<USBS0); // Asynchron 8N2
}

void uart_start()
{
 UCSR0B |= (1<<RXEN0); // UART RX einschalten
 UCSR0B |= (1<<TXEN0); // UART TX einschalten
}

void uart_stopp()
{
 milliwait(1);//test
 while (!(UCSR0A & (1<<UDRE0))) ;// warten bis letztes Zeichen fertig gesendet
 milliwait(1);//test
 UCSR0B &= ~(1<<RXEN0); // UART RX aus
 UCSR0B &= ~(1<<TXEN0); // UART TX aus
 uart_ok=false; 
}

void uart_putc(unsigned char c)
{
 while(!(UCSR0A & (1<<UDRE0))) {}// warten bis Senden moeglich
 UDR0 = c;                      // sende Zeichen
}

unsigned char uart_getc(void)
{
 while(!(UCSR0A & (1<<RXC0))) {}// warten bis Zeichen verfuegbar
 return UDR0; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart_check()
{
 return (UCSR0A & (1<<RXC0));//true wenn Zeichen verfuegbar
}

void uart_write(const char *txt)
{
 while(*txt!=0) uart_putc(*txt++);
}

void uart_print(uchar n)
{
 if(n>=100) {uart_putc(n/100+'0'); n=n%100;}
 if(n>=10)  {uart_putc(n/10+'0'); n=n%10;}
 uart_putc(n+'0');
}

/*
void uart1_init()
{
 UBRR1 = UBRR_VAL;
 UCSR1B = (1<<RXEN1)|(1<<TXEN1); // UART TX und RX einschalten
 //UCSR1B = (1<<RXEN1); // nur UART RX einschalten
 //UCSR1C = (3<<UCSZ10); // Asynchron 8N1
 UCSR1C = (3<<UCSZ10)|(1<<USBS1); // Asynchron 8N2
}

void uart1_putc(unsigned char c)
{
 while (!(UCSR1A & (1<<UDRE1))) ;// warten bis Senden moeglich
 UDR1 = c;                      // sende Zeichen
 milliwait(3);//test
}

unsigned char uart1_getc(void)
{
 while (!(UCSR1A & (1<<RXC1))) ;// warten bis Zeichen verfuegbar
 return UDR1; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart1_check()
{
 return (UCSR1A & (1<<RXC1)) ;//true wenn Zeichen verfuegbar
}
*/

bool uart_verbindung_aufbauen()
{
 //Startprotokoll: A an Linux-Computer senden, Antwort sollte B sein
 int i;
 uchar c0=0;
 uart_putc('A'); uart_putc('\n'); //A senden
 for(i=0;i<200;i++) //maximal 200 Millisekunden warten bis B empfangen wird
  {
   if(uart_check())
    {
     uchar c=uart_getc();
     if(c=='\n') break;
     if(c>=' ') c0=c;
    }
   milliwait(1);
  }
 return (c0=='B');
}
#endif //MITUART

/*********************************** Software-UART ************************************/
/* Soft-UART auf Pins B6 B7 fuer Verbindung zu Roboterarm
 Schaltung:
   Senden:    PB7 (TxD)
   Empfangen: PB6 (RxD)
*/

//#include <inttypes.h>
static bool suart_ok=false;

#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 20MHz"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
#endif

#define BAUDRATE 9600UL      // Baudrate der Soft-UART
#define BAUDMICROSEC (1e6/BAUDRATE)

#define SUART_TXD  //senden
#define SUART_RXD  //empfangen

#ifdef SUART_TXD
 #define SUART_TXD_PORT PORTB
 #define SUART_TXD_DDR  DDRB
 #define SUART_TXD_BIT  PB7
#endif //SUART_TXD 

#ifdef SUART_RXD
 #define SUART_RXD_PORT PORTB
 #define SUART_RXD_PIN  PINB
 #define SUART_RXD_DDR  DDRB
 #define SUART_RXD_BIT  PB6
#endif //SUART_RXD 

void suart_init()
{
 uint8_t sreg = SREG;
 cli();

#ifdef SUART_RXD
 SUART_RXD_DDR  &= ~(1 << SUART_RXD_BIT);
 SUART_RXD_PORT |=  (1 << SUART_RXD_BIT);
#endif

#ifdef SUART_TXD
 SUART_TXD_PORT |= (1 << SUART_TXD_BIT);
 SUART_TXD_DDR  |= (1 << SUART_TXD_BIT);
#endif

 SREG = sreg;
}

#ifdef SUART_TXD
void suart_putc(uint8_t c)
{
 uint8_t sreg = SREG;
 cli();

 // frame = *.P.7.6.5.4.3.2.1.0.S   S=Start(0), P=Stop(1), *=Endemarke(1) 
 uint16_t data = (3 << 9) | (c << 1);
 for(; data!=1; data >>= 1)
  {
   if(data & 1) SUART_TXD_PORT |=  (1 << SUART_TXD_BIT);
   else         SUART_TXD_PORT &= ~(1 << SUART_TXD_BIT);
   //Verzoegerung mit 10 Takte fuer for-Schlaufe beruecksichtigt:
   _delay_us(BAUDMICROSEC - 10*1e6/F_CPU);
  }

 SREG = sreg;
}

void suart_write(const char *s)
{
 char c;
 while((c= *s++)!=0) suart_putc(c);
}
#endif //SUART_TXD

#ifdef SUART_RXD
inline bool suart_check()
{
 return ((SUART_RXD_PIN & (1<<SUART_RXD_BIT))==0); // warte auf Start-Bit
}

uint8_t suart_getc(void)
{
 uint8_t sreg = SREG;
 cli();
 
 // frame = *.P.7.6.5.4.3.2.1.0.S   S=Start(0), P=Stop(1), *=Endemarke(1) 
 uint8_t data=0;
 while((SUART_RXD_PIN & (1<<SUART_RXD_BIT))!=0) {}// warte auf Start-Bit
 _delay_us(BAUDMICROSEC/2);
 for(uint8_t i=0;i<9;i++)
  {
   //Verzoegerung mit 12 Takte fuer for-Schlaufe beruecksichtigt:
   _delay_us(BAUDMICROSEC - 12*1e6/F_CPU);
   uint8_t bit = ((SUART_RXD_PIN & (1<<SUART_RXD_BIT)) != 0) ? 0x80 : 0;
   if(i<8) data = (data>>1)+bit;
  }
 
 SREG = sreg;
 return data; //Zeichen an Aufrufer zurueckgeben
}

void suart_write_wait(const char *s,char *antwort)
{
 suart_write(s);
 char c;
 while((c=suart_getc()) != '\n') //Antwort abwarten
  {
   *antwort++ = c;
  }
 *antwort=0;
}

char* suart_zeile_empfangen()
{
 static char zeile[32];
 uint8_t i;
 char c=' ';
 for(i=0; i<31 && c!=0;)
  {
   c=suart_getc();
   if(c=='\r') continue; //CR ignorieren
   if(c=='\n') c=0; //mit LF beenden
   zeile[i++]=c;
  }
 zeile[i]=0;
 return zeile;
}

bool suart_verbindung_aufbauen()
{
 //Startprotokoll: A an robot.cc senden, Antwort sollte B sein
 uchar c,c0=0;
#ifdef SPULENVERSION
 for(int j=0; j < 10/5 && c0==0; j++) //test maximal ca. 10 Sekunden probieren
#else
 for(int j=0; j < 60/5 && c0==0; j++) //maximal ca. 60 Sekunden probieren
#endif
  {
   milliwait(100);
   suart_putc('A'); suart_putc('\n'); //A senden
   int32_t imax = 5*100000; //maximal ca. 5 Sekunden warten bis B empfangen wird
   for(int32_t i=0;i<imax;i++) //imax * 10us = maximale Wartezeit
    {
     if(suart_check())
      {
       c=suart_getc();
       if(c=='\n') break;
       if(c>' ') c0=c;
      }
     _delay_us(9); //Schlaufe braucht schon ca. 1us
    }
  }
 return (c0=='B');
}
#endif //SUART_RXD

/********************************** LCD-Modul *****************************************/
#include "lcdmodul2.h"

void itoa(char *s,int n)
{
 char c,*t;
 int k;
 if(n<0) {*s++ = '-'; n = -n;}
 for(t=s;;)
  {
   if(n<10) {*t++ = n+'0'; break;}
   k = n%10;
   *t++ = k+'0';
   n /= 10;
  }
 *t=0;
 for(--t;t!=s;)
  {
   c = *t;
   *t-- = *s;
   *s = c;
   if(t!=s) s++;
  }
}

int atoi(const char *s)
{
 int n=0;
 while(isdigit(*s)) n=10*n+(*s++)-'0';
 return n;
}

void lcd_print1(const char *txt,int n)
{
 char text[22];
 uchar i;
 for(i=0;*txt!=0 && i<15;i++) text[i] = *txt++;
 //sprintf(&text[i],"%d",n);
 itoa(&text[i],n);
 lcd_write(text);
}

void lcd_print(uchar zeile,const char *cstr,int p1=0,int p2=0)
{
 char txt[32];
 lcd_goto(zeile,1);
 sprintf(txt,cstr,p1,p2);
 lcd_write(txt);
}

void lcd_print(uchar zeile,const char *cstr,int p1,const char *p2)
{
 char txt[32];
 lcd_goto(zeile,1);
 sprintf(txt,cstr,p1,p2);
 lcd_write(txt);
}

void longtoa(char *s,long n)
{
 char c,*t;
 if(n<0) {*s++ = '-'; n = -n;}
 for(t=s;n>=10;n/=10)
  {
   c = n%10;
   *t++ = c+'0';
  }
 *t++ = n+'0';
 *t=0;
 for(--t;t!=s;)
  {
   c = *t;
   *t-- = *s;
   *s = c;
   if(t!=s) s++;
  }
}

void lcd_printz1L(int8_t zeile,const char *txt,long n,const char *txt2=NULL)
{
 lcd_goto(zeile,1);
 char text[32];
 uchar i;
 for(i=0;*txt!=0 && i<15;i++) text[i] = *txt++;
 longtoa(&text[i],n);
 lcd_write(text);
 if(txt2!=NULL) lcd_write(txt2);
}

void lcd_write_fill(const char *txt) //gesamte Zeile schreiben (mit Blanks auffuellen)
{
 char text[17];
 uchar i;
 for(i=0;*txt!=0 && i<16;i++)
  {if(*txt=='\n') {text[i]=' '; txt++;} else text[i] = *txt++;}
 for(;i<16;i++) text[i] = ' ';
 text[16]=0;
 lcd_write(text);
}

void lcd_write_zeile(int8_t zeile,const char *text)
{
 lcd_goto(zeile,1); lcd_write_fill(text);
}

void lcd_write_zeiles(int8_t zeile,const char *cstr,const char *p)
{
 char text[18];
 sprintf(text,cstr,p);
 lcd_goto(zeile,1); lcd_write_fill(text);
}

#ifdef MITUART
void lcd_keine_UART_Verbindung()
{
 lcd_write("keine UART-Verb.");
}
#endif

void lcd_write3(const char *txt)
{
 lcd_write_zeile(3,txt);
}
void lcd_write4(const char *txt)
{
 lcd_write_zeile(4,txt);
}
void lcd_write4i(const char *cstr,int8_t n)
{
 char txt[32];
 sprintf(txt,cstr,n);
 lcd_write_zeile(4,txt);
}
void lcd_write4s(const char *cstr,const char *p)
{
 char txt[32];
 sprintf(txt,cstr,p);
 lcd_write_zeile(4,txt);
}

class Zeile4meldung
{
 const char *text;
public:
 Zeile4meldung() {text=NULL;}
 void write(const char *txt) {lcd_write4(text=txt);}
 void refresh() {if(text!=NULL) lcd_write4(text);}
};
static Zeile4meldung zeile4;

void lcd_printf(const char *langer_text)
{
 char txt[17],c;
 int8_t i,zeile;
 for(zeile=1; zeile<=4 && *langer_text!=0; zeile++)
  {
   for(i=0;(c= *langer_text)!=0 && i<16;langer_text++) {if(c!='\n') txt[i++] = c;}
   while(i<16) txt[i++]=' ';
   txt[i]=0; lcd_goto(zeile,1); lcd_write(txt);
  }
}

typedef char string17[17];

int8_t lcd_auswahlfrage(const char *titel,int8_t n,const string17 *str,int8_t j0=1)
{
 int8_t i,i0=0,j,wahl=0;
 lcd_clear();
 lcd_write_zeile(1,titel);
 for(j=2,i=0;j<=4 && i<n;j++,i++)
  {
   lcd_goto(j,3); lcd_write(str[i]);
  }
 if(j0>n) j0=n;
 j0++;
 while(j0>4) //eventuell runterscrollen 
  {
   for(j=1,i= ++i0; j<4; i++)
	{lcd_goto(++j,3); lcd_write(str[i]);}
   j0--;
  }
 lcd_goto(j=j0,1); lcd_writec('>');
 for(;;)
  {
   milliwait(50);
   if((tastegedrueckt&TASTE_OBEN)!=0)
    {
     lcd_goto(j,1); lcd_writec(' ');
     if(j==2 && i0>0) //raufscrollen?
      {
       for(j=5,i= --i0 + 2; j>2; --i)
	{lcd_goto(--j,3); if(i<n) lcd_write(str[i]);}
      }
     else if(--j<2) j=2;
     lcd_goto(j,1); lcd_writec('>');
     while((entprelltetasten&TASTE_OBEN)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_OBEN; //reset pressed state for the button
    }
   if((tastegedrueckt&TASTE_UNTEN)!=0)
    {
     lcd_goto(j,1); lcd_writec(' ');
     if(j==4 && i0<n-3) //runterscrollen?
      {
       for(j=1,i= ++i0; j<4; i++)
	{lcd_goto(++j,3); lcd_write(str[i]);}
      }
     else if(++j>4) j=4;
     lcd_goto(j,1); lcd_writec('>');
     while((entprelltetasten&TASTE_UNTEN)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_UNTEN; //reset pressed state for the button
    }
   if((tastegedrueckt&TASTE_MITTE)!=0) //Ok-Taste
    {
     wahl=j-1+i0;
     while((entprelltetasten&TASTE_MITTE)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_MITTE; //reset pressed state for the button
     break;
    }
   if((tastegedrueckt&TASTE_LINKS)!=0) //Cancel-Taste
    {
     wahl=0;
     while((entprelltetasten&TASTE_LINKS)==0) {} //wait until button released
     tastegedrueckt &= ~TASTE_LINKS; //reset pressed state for the button
     break;
    }
  }
 return wahl;
}

/************************************ EEPROM ******************************************/
//je nach Compilerversion stehen die Daten in umgekehrter Reihenfolge im EEPROM!
#ifdef ALTERCOMPILER
// avr-gcc ?.?.?  (weiss nicht mehr welche Version vor dem Update auf 16.04.1 LTS da war)
uint eeversion EEMEM;
uint eeCalibLeer[64] EEMEM;  //Calibrierdaten bei Leerem Brett
uint16_t eeMinDiff EEMEM;    //Minimale Differenz von leerem Feld zu Figur gesetzt
uint8_t eeCalibStatus EEMEM; //Status: 255 = noch nicht calibriert, 1=Leerfeld calibriert, 2=auch MinDiff gesetzt
uint8_t eeSpielstaerke EEMEM; //gleich spielsuchtiefe: 0=sehr schwach, 3=mittel, 4=staerkste Spielstufe
                              //fuer Stockfish 5,6,10...30?, 10=sehr schnell antwortend
uint8_t eeLetzteStellung[64] EEMEM; //gespeicherte Stellung
uint8_t eeBrettlage EEMEM; //TODO: aktuelle Brettlage
uint8_t eeReserve EEMEM; //TODO
#else
// avr-gcc 4.9.2  (nach Update auf xubuntu 16.04.1 LTS)
uint8_t eeReserve EEMEM; //TODO
uint8_t eeBrettlage EEMEM; //TODO: aktuelle Brettlage
uint8_t eeLetzteStellung[64] EEMEM; //gespeicherte Stellung
uint8_t eeSpielstaerke EEMEM; //gleich spielsuchtiefe: 0=sehr schwach, 3=mittel, 4=staerkste Spielstufe
                              //fuer Stockfish 5,6,10...30?, 10=sehr schnell antwortend
uint8_t eeCalibStatus EEMEM; //Status: 255 = noch nicht calibriert, 1=Leerfeld calibriert, 2=auch MinDiff gesetzt
uint16_t eeMinDiff EEMEM;    //Minimale Differenz von leerem Feld zu Figur gesetzt
uint eeCalibLeer[64] EEMEM;  //Calibrierdaten bei Leerem Brett
uint eeversion EEMEM;
#endif

#define MAXSPIELSTAERKE 50  //noch testen was wirklich das Maximum ist (fuer Stockfish-Engine)
#define MAXEIGENEENGINE 4   //maximale Suchtiefe fuer eigene Engine

void eeversion_setzen()
{
 if(eeprom_read_word(&eeversion)!=EEVERSION)
   eeprom_write_word(&eeversion, EEVERSION);
}

void calib_von_eeprom_einlesen()
{
 int8_t j;
 uint version=eeprom_read_word(&eeversion);
 uint8_t status=eeprom_read_byte(&eeCalibStatus);

 if(version==EEVERSION && status != 255) //Calibrierdaten fuer Leerfeld vorhanden?
  {
   //for(j=0;j<64;j++) sensorCalibLeer[j]=eeprom_read_word(&eeCalibLeer[j]); //Calibrierdaten vom EEPROM einlesen
   eeprom_read_block(sensorCalibLeer,eeCalibLeer,64<<1); //einfacher mit Block-Befehl
  }
 else for(j=0;j<64;j++) sensorCalibLeer[j]=488; //provisorische Daten fuer Leerfeld setzen

 if(version==EEVERSION && status == 2) //eeMinDiff gesetzt?
   minDiff=eeprom_read_word(&eeMinDiff); //vom EEPROM einlesen
 else minDiff=10; //sonst provisorischer Wert setzen
}

void calibLeer_im_eeprom_speichern()
{
 eeversion_setzen();
 for(int8_t j=0;j<64;j++)
  {
   if(eeprom_read_word(&eeCalibLeer[j]) != sensorCalibLeer[j])
     eeprom_write_word(&eeCalibLeer[j], sensorCalibLeer[j]);
  }
 if(eeprom_read_byte(&eeCalibStatus)==255)
    eeprom_write_byte(&eeCalibStatus,1); //eeCalibStatus auf 1 setzen: Leerfeld calibriert
}

void minDiff_im_eeprom_speichern()
{
 uint8_t status=eeprom_read_byte(&eeCalibStatus);
 if(status!=1 && status!=2) return;//noch nicht mit Leerfeld calibriert? --> Fehler
 if(eeprom_read_word(&eeMinDiff)!=minDiff)
   eeprom_write_word(&eeMinDiff, minDiff);
 if(status!=2) eeprom_write_byte(&eeCalibStatus,2); //eeCalibStatus auf 2 setzen: auch eeMinDiff calibriert
}

uint8_t spielstaerke_von_eeprom_einlesen()
{
 uint version=eeprom_read_word(&eeversion);
 uint8_t spielstaerke=2;
 if(version==EEVERSION)
  {
   spielstaerke=eeprom_read_byte(&eeSpielstaerke);
   if(spielstaerke>MAXSPIELSTAERKE) //noch nicht im EEPROM gesetzt, oder ungueltiger Wert im EEPROM?
     spielstaerke=2;
  }
 return spielstaerke;
}

void spielstaerke_in_eeprom_speichern(uint8_t spielstaerke)
{
 if(eeprom_read_byte(&eeSpielstaerke)!=spielstaerke)
   eeprom_write_byte(&eeSpielstaerke,spielstaerke);
}

/************************* Sensorfeld calibrieren und auswerten ***********************/
#define WEISS   0
#define SCHWARZ 0x40  //so in schach-engine.h definiert!
static int8_t brettlage=SCHWARZ; //Seite die zum Spieler zeigt

const char *feldname(int8_t j) //zuordnung von Index zu Feld auf dem Schachbrett
{
 static char str[3];
 if(j<0 || j>=64) {str[0]=str[1]='?';}
 else if(brettlage==SCHWARZ)
  {
   str[0]=j%8+'A';
   str[1]=j/8+'1';
  }
 else
  {
   str[0]='H'-j%8;
   str[1]='8'-j/8;
  }
 str[2]=0;
 return str;
}

#ifdef SPULENVERSION

void warte_bis_alle_sensoren_gelesen()
{
 uchar kanal, i, j;
 uint summe;
 for(anregpin=(1<<PC0),j=0; j<64; anregpin <<= 1, j+=8)
  {
   milliwait(10); //TODO: ev. optimale Pause einstellen
   for(kanal=0;kanal<8;kanal++)
    {
     ad_init(kanal); while(adamlaufen) {} //erste A/D-Wandlung verwerfen
     uint n=2; //Anzahl Aufsummierungen
     for(summe=0,i=0;i<n;i++)
      {
       ad_start(); while(adamlaufen) {} // A/D-Wandlung abwarten
       summe += messwert;
      }
     sensorfeld[j+kanal] = (summe+n/2)/n; //Mittelwert von n Messungen
    }
  }
 anregpin=0;
}

#else

void warte_bis_alle_sensoren_gelesen()
{
 static uchar zaehlervorher=0;
 while(sensorlesezyklus==zaehlervorher) milliwait(1);
 zaehlervorher=sensorlesezyklus;
 return;
}

#endif

bool fastgleich(uint a,uint b,uint delta)
{
 if(a==b) return true;
 return (a <= b+delta && a+delta >= b);
}

bool fastgleichSpulenversion(uint a,uint b)
{
 if(a==b) return true;
 //if(a<b) {return !(a < b*7/8);}
 //return !(b < a*7/8);
 //if(a<b) {return !(a < b*PROZGRENZ/100);} //100*1024 -> zu gross, 50*1024 -> ok fuer uint
 if(a<b) {return !(a < b*(PROZGRENZ/2)/50);}
 return !(b < a*(PROZGRENZ/2)/50);
}

bool feldfastgleich(uint *f1,uint *f2,uint delta)
{
 for(int8_t j=0;j<64;j++) if(!fastgleich(f1[j],f2[j],delta)) return false;
 return true;
}

void sensorfeld_einlesen(uint *feld)
{
 warte_bis_alle_sensoren_gelesen();
 for(int8_t j=0;j<64;j++) feld[j]=sensorfeld[j];
}

void feld_kopieren(uint *f1,volatile uint *f2)
{
 for(int8_t j=0;j<64;j++) *f1++ = *f2++;
}

void sensorfeld_einlesen_entprellt(uint *feld)
{
 static uint feld1[64],feld2[64];
 static uint *p1=NULL,*p2=NULL;
#ifdef SPULENVERSION
 uint delta=16;
#else
 uint delta=4;
#endif
#ifdef ENTPRELLPAUSE
 milliwait(ENTPRELLPAUSE);
#endif
 if(p1==NULL)
  {
   warte_bis_alle_sensoren_gelesen();
   p1=feld1; p2=feld2; feld_kopieren(p1,sensorfeld); //erstes Feld einlesen
  }
 //for(int8_t n=0;n<2;) //die Sensoren einlesen bis 2 mal mit erstem Feld uebereinstimmend
 for(int8_t n=0;n<1;) //die Sensoren einlesen bis 1 mal mit erstem Feld uebereinstimmend
  {
   warte_bis_alle_sensoren_gelesen();
   feld_kopieren(p2,sensorfeld); //naechstes Feld einlesen
   uint* h=p1; p1=p2; p2=h; //Zeiger vertauschen
   if(!feldfastgleich(p1,p2,delta)) {n=0; continue;} //wenn ungleich aelteres Feld ueberschreiben
   else n++;
  }
 feld_kopieren(feld,p2); //Resultat zurueckgeben
}

#ifdef SPULENVERSION
void sensorfeld_einlesen_digital(int8_t *feld)
{
 uint feld1[64],wert;
 sensorfeld_einlesen_entprellt(feld1);
 for(int8_t j=0;j<64;j++)
  {
   //wert = (feld1[j] < sensorCalibLeer[j]*7/8) ? 1 : 0;
   wert = (feld1[j] < sensorCalibLeer[j]*(PROZGRENZ/2)/50) ? 1 : 0;
   if(brettlage==SCHWARZ) feld[j] = wert;
   else                feld[63-j] = wert;
   //feld umsortiert speichern wenn Brettlage umgekehrt
  }
}
#else
void sensorfeld_einlesen_digital(int8_t *feld)
{
 uint feld1[64],diff;
 sensorfeld_einlesen_entprellt(feld1);
 for(int8_t j=0;j<64;j++)
  {
   diff = (feld1[j]>=sensorCalibLeer[j]) ? 0 : sensorCalibLeer[j] - feld1[j];
   if(brettlage==SCHWARZ) feld[j] = (diff>minDiff) ? 1 : 0;
   else          feld[63-j] = (diff>minDiff) ? 1 : 0;
   //feld umsortiert speichern wenn Brettlage umgekehrt
  }
}
#endif

/*
void lcd_print_felddiff(uint *feld1,uint *feld2,int8_t max) //Test
{
 int8_t i,j,z=1;
 uchar c;
 lcd_goto(z,1);
 for(i=j=0;j<max;j++)
  {
   if(feld1[j]>feld2[j]+10) c=((feld1[j]-feld2[j])/10)%10 + '0';
   else if(feld2[j]>feld1[j]+10) c=((feld2[j]-feld1[j])/10)%10 + 'A';
   else c='0';
   lcd_writec(c);
   if(++i==16 && j<max-1) {lcd_goto(++z,1); i=0;}
  }
}
*/

bool ist_feld_leer(const int8_t *feld)
{
 for(int8_t i=0;i<64;i++) if(feld[i]!=0) return false;
 return true;
}

int8_t feld_vergleich(const int8_t *feld1,const int8_t *feld2)
{
 for(int8_t i=0;i<64;i++) if(feld1[i]!=feld2[i]) return (feld1[i]>feld2[i]) ? 1 : -1;
 return 0;
}

void calib_anzeige(int8_t wahl)
{
 lcd_clear();
 if(wahl==1)      lcd_printf("Leerbrett calibrieren...");
 else if(wahl==2) lcd_printf("Grundstellung calibrieren...");
 else             lcd_printf("Sensorentests...");
}

void aufsummiert_einlesen(uint* feld1,uint* feld2,uint8_t n)
{
 for(int8_t k=1;k<n;k++) //total n mal einlesen
  {
   sensorfeld_einlesen(feld1);
   for(int8_t i=0;i<64;i++) feld2[i] += feld1[i]; //und aufsummieren
  }
 for(int8_t i=0;i<64;i++) feld2[i] /= n; //Durchschnitt von n mal Einlesen
}

void calibrieren(int8_t wahl)
{
 uint feld1[64],feld2[64];
 int8_t j;
 calib_anzeige(wahl);
 milliwait(500);
 sensorfeld_einlesen(feld1);
 sensorfeld_einlesen(feld2);
 /*
 for(int i=0;i<50;i++) //test 50*0.2 = 10 Sec
  {
   sensorfeld_einlesen(feld2);
   lcd_print_felddiff(feld1,feld2,64); //test
  }
 calib_anzeige(wahl);
 */
 if(wahl==1) //Leeres Brett
  {
   aufsummiert_einlesen(feld1,feld2,16);
   for(j=0;j<64;j++) sensorCalibLeer[j]=feld2[j];
   calibLeer_im_eeprom_speichern();
   zeile4.write("fertig.");
  }
 else if(wahl==2) //mit Grundaufstellung
  {
   int8_t errcnt=0;
   int8_t lasterr=64;//test
   uint diff=32000; //mindestens Differenz zwischen leerem Feld und Figur auf Feld
#ifdef SPULENVERSION
   uint delta=16;//TODO
#else
   uint delta=4;
#endif
   aufsummiert_einlesen(feld1,feld2,16);
   for(j=0;j<64;j++)
    {
     if(!fastgleich(feld1[j],feld2[j],delta)) {errcnt++; lasterr=j;}
     if(j>=16 && j<64-16)
      {
       if(!fastgleich(sensorCalibLeer[j],feld2[j],delta)) {errcnt++; lasterr=j;}
      }
     else
      {
#ifdef SPULENVERSION
       uint d;
       if(feld2[j] >= sensorCalibLeer[j]) {d=32000; errcnt++; lasterr=j;}//ignorieren wenn Wert zu gross
       else d = (sensorCalibLeer[j]-feld2[j])*100L/sensorCalibLeer[j]; //Unterschied in Prozent
#else
       uint d=sensorCalibLeer[j]-feld2[j];
#endif
       if(d<diff) diff=d; //minimale Differenz ermitteln
      }
    }
#ifdef SPULENVERSION
   //minDiff = diff*2/4; //minimale Differenz mit viel Toleranz speichern
   minDiff = diff; //minimale Differenz ohne Toleranz
#else
   minDiff = diff*3/4; //minimale Differenz mit etwas Toleranz speichern
#endif
   lcd_goto(4,1);
   lcd_print1("minDiff=",minDiff);
   if(errcnt==0)
    {
#ifndef SPULENVERSION //TODO
     minDiff_im_eeprom_speichern();
#endif
     lcd_write(" ok.");
    }
   //else  lcd_print1("er=",errcnt);
   else  {lcd_goto(1,1); lcd_print1("er=",errcnt); lcd_print1("nr=",lasterr);}//test
   lcd_write(" ");
  }
 else //sonst Sensorentests
  {
   for(;;)
    {
     sensorfeld_einlesen(feld2);
     for(j=0;j<64;j++)
#ifdef SPULENVERSION
      if(!fastgleichSpulenversion(feld1[j],feld2[j]))
#else
      if(!fastgleich(feld1[j],feld2[j],minDiff))
#endif
       {
	for(uint8_t k=0;k<2;k++)
	 {
	  if(k==0) {lcd_write_zeile(2,feld2[j]>feld1[j]?"Figur weg    ":"Figur gesetzt");}
	  lcd_goto(3,1); lcd_write(feldname(j)); lcd_print1(" ADwert:",feld2[j]); lcd_write("  ");
	  lcd_goto(4,1); //AD-Werte von drumherum auch anzeigen:
	  if(j%8>0) {lcd_print1(" ",feld2[j-1]);}
	  if(j%8<7) {lcd_print1(" ",feld2[j+1]);}
	  if(j>=8)  {lcd_print1(" ",feld2[j-8]);}
	  if(j<64-8) {lcd_print1(" ",feld2[j+8]);}
	  lcd_write("   ");
	  milliwait(500);
	  sensorfeld_einlesen(feld2);
	 }
	sensorfeld_einlesen(feld1);
	break;
       }
     if((tastegedrueckt&TASTE_MITTE)!=0)
      {
       lcd_clear();
       warte_losgelassen_mitte();
       return;
      }
    }
  }
 secwait(10);
 lcd_clear();
}

/****************************** Fehler-Behandlungen ***********************************/
static uint8_t fehlerflags=0;
static int8_t nrfehlersensor= -1;
static int8_t nrwrongfig= -1;
#define FEHLER_RAM             1
#define FEHLER_FIGUR_GESETZT   2 
#define FEHLER_SENSOREN        4
#define FEHLER_ROCHADESENSOR   16

static int8_t rochadefehlernr=0;//genauere Angabe bei falsch gemachter Rochade
//#define ROCHFEHLER1 "Koenig hat schon mal gezogen" //neu 19.9.24
#define ROCHFEHLER1 "Koenig schon gez" //auf 16 Zeichen beschraenkt
#define ROCHFEHLER2 "Koen.Turm falsch"
#define ROCHFEHLER3 "Turm c-o falsch"
#define ROCHFEHLER4 "Turm g-o falsch"
#define ROCHFEHLER5 "Koenig falsch"

void fehler_zu_wenig_ram()
{
 fehlerflags |= FEHLER_RAM;
}

const char* feldstr(int8_t nr)  //nr schon mit brettlage korrigiert
{                               //wenn noch nicht: feldname() verwenden
 static char str[3];
 if(nr== -1) {str[0]='-'; str[1]='1';}
 else if(nr>=64 && nr<100) {str[0]=nr/10+'0'; str[1]=nr%10+'0';}
 else if(nr<0 || nr>=64) {str[0]=str[1]='?';}
 else {str[0] = (nr%8)+'A'; str[1] = (nr/8)+'1';}
 str[2]=0;
 return str;
}

void fehler_auf_lcd_anzeigen()
{
 if(fehlerflags&FEHLER_RAM)                lcd_write4("zu wenig RAM");
 else if(fehlerflags&FEHLER_FIGUR_GESETZT) lcd_write4("neue Fig gsetzt?");
 else if(fehlerflags&FEHLER_SENSOREN)      lcd_write4s("Sensorfehler %s ",feldstr(nrfehlersensor));
 else if(fehlerflags&FEHLER_ROCHADESENSOR)
  {lcd_write_zeile(1,"Rochade-Fehler:");
   switch(rochadefehlernr) //neu 19.9.24
    {case 1: lcd_write_zeile(2,ROCHFEHLER1); break;
     case 2: lcd_write_zeile(2,ROCHFEHLER2); break;
     case 3: lcd_write_zeile(2,ROCHFEHLER3); break;
     case 4: lcd_write_zeile(2,ROCHFEHLER4); break;
     case 5: default: lcd_write_zeile(2,ROCHFEHLER5);
    }
  }
 else lcd_write4("unbek. Fehler");
}

/********************************** Schach-Engine *************************************/
static int8_t spielsuchtiefe=0;

#ifdef STACKTEST
static int freezaehler=0;

void* mymalloc(int n){
 void* p=malloc(n);
 if(p!=NULL) freezaehler++;
 return p;
}

void myfree(void* p)
{
 if(p!=NULL) freezaehler--;
 free(p);
}

#define malloc mymalloc
#define free myfree
#endif

#define AVRSCHACH
#include "schach-engine.h"

// weitere Vordeklarationen:
void uart_startstellung_senden(Brett& brett);
bool check_stellung(Brett& brett);
// Vordeklarationen fuer 3 mal gleiche Stellung und 50-Zug-Regel:
void stellung_speichern_reset();
void stellung_speichern(Brett& brett,bool resetflag);
int stellung_vergleich(Brett& brett);
bool fuenfzig_zug_regel();

#ifdef MITUART
Zug besterzug_stockfish(Brett& brett,Zug& letzterzug,uchar farbe,int8_t suchtiefe);
#endif

void letzte_stellung_von_eeprom_einlesen(Brett& brett)
{
 for(int8_t i=0;i<64;i++) brett[i]=eeprom_read_byte(&eeLetzteStellung[i]);
 if(!check_stellung(brett)) {brett.grundstellung();}
}
 
void stellung_im_eeprom_speichern(Brett& brett)
{
 for(int8_t i=0;i<64;i++)
   if(eeprom_read_byte(&eeLetzteStellung[i]) != brett[i])
     eeprom_write_byte(&eeLetzteStellung[i], brett[i]);
}

static int8_t computerfarbe=WEISS,spielerfarbe=SCHWARZ;

class LcdZugliste //Anzeige der bisher gemachten Zuege auf ersten 3 Zeilen des LCD-Moduls
{
public:
 int8_t zeile;
 char txt1[17],txt2[17],txt3[17];
 char *ptxt;
 int zugnr;
 LcdZugliste() {zeile=0; txt1[0]=txt2[0]=txt3[0]=0; ptxt=txt1; zugnr=0;}
 void reset();
 void nextline();
 void scrollen();
 //void write(int nr,uchar farbe,Zug& zug);
 void write(uchar farbe,Zug& zug);
 void refresh();
 void zugzuruecknehmen();
};
static LcdZugliste lcdzugliste;

void LcdZugliste::reset()
{
 zeile=0; txt1[0]=txt2[0]=txt3[0]=0; ptxt=txt1;
 for(int8_t j=1;j<=3;j++) lcd_write_zeile(j,txt1);
 zugnr=0;
}

void LcdZugliste::nextline()
{
 if(zeile==3) scrollen();
 else
  {
   zeile++;
   if(zeile==1)      ptxt=txt1;
   else if(zeile==2) ptxt=txt2;
   else              ptxt=txt3;
  }
}

void LcdZugliste::scrollen()
{
 strcpy(txt1,txt2); lcd_write_zeile(1,txt1);
 strcpy(txt2,txt3); lcd_write_zeile(2,txt2);
 txt3[0]=0;         lcd_write_zeile(3,txt3);
}

void LcdZugliste::zugzuruecknehmen()
{
 if(zeile==0) return; //kein vorheriger Zug
 if(txt3[0]!=0)      {txt3[0]=0; ptxt = (spielerfarbe==WEISS) ? txt3 : txt2;}
 else if(txt2[0]!=0) {txt2[0]=0; ptxt = (spielerfarbe==WEISS) ? txt2 : txt1;}
 else {txt1[0]=0; ptxt=txt1;}
 if(spielerfarbe==SCHWARZ)
  {int n=strlen(ptxt)-6;
   if(n>=0) ptxt[n]=0;
  }
 zeile--;
 zugnr--;
}

void LcdZugliste::refresh() //neu zeichnen
{
 lcd_write_zeile(1,txt1);
 lcd_write_zeile(2,txt2);
 lcd_write_zeile(3,txt3);
 zeile4.refresh();
}

//void LcdZugliste::write(int nr,uchar farbe,Zug& zug) //nr: Zugnummern (1, 2, 3, ...), farbe: WEISS oder SCHWARZ, zug: Halbzug
void LcdZugliste::write(uchar farbe,Zug& zug) //nr: Zugnummern (1, 2, ...), farbe: WEISS oder SCHWARZ, zug: Halbzug
{
 int8_t i;
 //if(nr<=1) {nr=1; if(farbe==WEISS) reset();}
 if(farbe==WEISS)
  {
   //zugnr=nr;
   zugnr++;
   nextline();
   itoa(ptxt,zugnr);
   i=strlen(ptxt);
   ptxt[i++]=':';
  }
 else i=strlen(ptxt);
 ptxt[i++]=' ';
 strcpy(&ptxt[i],zug.print()); i+=5;
 lcd_goto(zeile,1); lcd_write(ptxt);
}

/************************************ Schach960 ***************************************/
#ifdef SCHACH960
#define ROCHADE_FLAG 0x80
static int wahrscheinlich_rochade960=0;
#endif
char koenig_startpos='E';
char turm1_startpos='A';
char turm2_startpos='H';

static uint zufallsgen=0; //wird in taste_abwarten() gesetzt

int8_t zufall(int8_t max) //Zufallszahl von 0 bis max-1
{
 /* Variante1: (nicht wirklich zufaellig)
 uint z=TCNT1;
 microwait(z&0xFF);
 z=TCNT1;
 return z%max;
 */
 /* Variante2: */
 uint z=zufallsgen%max, zneu=zufallsgen/max;
 zufallsgen=zneu;
 return z;
}

void zufalls_startstellung(char *stellung) //stellung muss Platz fuer 9 Zeichen haben inkl. Nullbyte
{
 int8_t i,j;
 for(i=0;i<8;i++) stellung[i]=' ';
 i = zufall(4)*2; //Laeufer auf schwarzem Feld
 stellung[i]='L';
 i = zufall(4)*2+1; //Laeufer auf weissem Feld
 stellung[i]='L';
 i = zufall(6); //Dame setzen
 for(j=0; stellung[j]!=' ' || i>0; j++)
  {if(stellung[j]==' ') i--;}
 stellung[j]='D';
 i = zufall(5); //1.Springer setzen
 for(j=0; stellung[j]!=' ' || i>0; j++)
  {if(stellung[j]==' ') i--;}
 stellung[j]='S';
 i = zufall(4); //2.Springer setzen
 for(j=0; stellung[j]!=' ' || i>0; j++)
  {if(stellung[j]==' ') i--;}
 stellung[j]='S';
 for(j=0; stellung[j]!=' '; j++) {}
 stellung[j]='T'; //1.Turm sezen
 //turm1_startpos='A'+j; //erst in grundstellung960() machen
 for(j=0; stellung[j]!=' '; j++) {}
 stellung[j]='K'; //Koenig sezen
 //koenig_startpos='A'+j; //erst in grundstellung960() machen
 for(j=0; stellung[j]!=' '; j++) {}
 stellung[j]='T'; //2.Turm sezen
 //turm2_startpos='A'+j; //erst in grundstellung960() machen
 stellung[8]=0;
}

bool check_stellung(Brett& brett)
{
 for(int8_t i=0;i<64;i++)
  {
   uchar fig=(brett[i]&FIGUR);
   if(fig==0 && brett[i]!=LEER) return false;
   if(fig==7) return false;
   if((brett[i]&RESERVEBITS)!=0) return false;
  }
 uchar flag_koenig=0;
 for(uint8_t i=0;i<8;i++)
  {
   uchar figw = brett[i]&FIGUR,    figs = brett[i+56]&FIGUR;
   uchar flagw = brett[i]&MOVFLAG, flags = brett[i+56]&MOVFLAG;
   if((figw==KOENIG && flagw!=0) || (figs==KOENIG && flags!=0))
    {flag_koenig=1; koenig_startpos='A'+i;}
   if((figw==TURM && flagw!=0) || (figs==TURM && flags!=0))
    {if(flag_koenig==0) turm1_startpos='A'+i; else turm2_startpos='A'+i;}
  }
 return true;
}

/********************************** Hauptprogramm *************************************/
//static int8_t computerfarbe=WEISS,spielerfarbe=SCHWARZ; //schon weiter oben

void auf_taste_warten(uchar taste)
{
 tastegedrueckt=0; //eventuell schon zuvor gedrueckte Tasten ignorieren
 for(;;)
  {
   milliwait(100);
   if((tastegedrueckt&taste)!=0)
      {
       while((entprelltetasten&taste)==0) {} //wait until button released
       tastegedrueckt &= ~taste; //reset pressed state for the button
       return;
      }
  }
}

uchar taste_abwarten(uchar maske)
{
 uchar t1=0, t2;
 while((entprelltetasten&TASTE_ALLE)!=TASTE_ALLE) {zufallsgen++;}
 tastegedrueckt=0;
 for(;;)
  {
   while((tastegedrueckt&maske)==0) {zufallsgen++;}
   //milliwait(20);
   milliwait(8);//test
   t2 = tastegedrueckt&maske;
   if(t2!=0 && t2==t1) break;
   t1=t2;
  }
 return t2;
}

bool stellung_pruefen(int8_t *feld,Brett& brett)
{
 nrwrongfig = -1;
 for(int8_t j=0;j<64;j++)
  {
   if(feld[j]==0) {if(brett[j]!=LEER) {nrwrongfig=j; return false;}}
   else           {if(brett[j]==LEER) {nrwrongfig=j; return false;}}
  }
 return true;
}

void stellung_fehler()
{
 lcd_write_zeile (1,"Stellung-Fehler");
 lcd_write_zeiles(2,"justier Figur %s",feldstr(nrwrongfig));
 lcd_write_zeiles(3,"oder Sensor %s ",feldstr(nrfehlersensor));
 lcd_write4("dann ok-Taste");
 auf_taste_warten(TASTE_MITTE);
}

void stellung_pruefen_und_korrigieren()
{
 int8_t feld1[64];
 for(;;)
  {
   sensorfeld_einlesen_digital(feld1);
   if(stellung_pruefen(feld1,spielbrett)) break;
   stellung_fehler();
   lcdzugliste.refresh();
   secwait(1);
  }
}

char *get_brettzeile(uchar brettzeile)
{
 static char zeile[9];
 brettzeile--;
 for(uchar i=0; i<8; i++)
  {char ch=0;
   uchar c=spielbrett[brettzeile*8+i];
   switch(c&FIGUR)
    {case BAUER:    ch='B'; break;
     case SPRINGER: ch='S'; break;
     case LAEUFER:  ch='L'; break;
     case TURM:     ch='T'; break;
     case DAME:     ch='D'; break;
     case KOENIG:   ch='K'; break;
     default: ch='.';
    }
   if((c&FARBE)==SCHWARZ) ch += 'a'-'A';
   zeile[i]=ch;
  }
 zeile[8]=0;
 return zeile;
}

//bool rochade(uchar von,uchar nach) //ist es Koenigszug einer Rochade?
bool rochade(uchar *pvon,uchar *pnach) //ist es Koenigszug einer Rochade? neu 19.9.24: Zeiger auf Parmeter
{
 uint8_t von = *pvon, nach = *pnach;
 if((spielbrett[von]&FIGUR)!=KOENIG) return false;
#ifdef SCHACH960
 if(nach&ROCHADE_FLAG) return true;
 if((spielbrett[von]&MOVFLAG)==0) {rochadefehlernr=1; return false;} //Koenig hat schon mal gezogen
 uchar cpos,gpos;
 if((spielbrett[von]&FARBE)==WEISS) {cpos=2; gpos=6;} else {cpos=56+2; gpos=56+6;}
 if(wahrscheinlich_rochade960) //Koenig und eigener Turm vom Brett genommen
  {//und wenn Koenig nach C oder G gezogen, dann ist es ein Rochadezug:
   if(nach==cpos || nach==gpos) return true;
   //oder wenn Turm nach D oder F gezogen, dann ist es ein Rochadezug:
   if(nach==cpos+1) {*pnach=cpos; return true;} //neu 19.9.24
   if(nach==gpos-1) {*pnach=gpos; return true;} //neu 19.9.24
   rochadefehlernr=2;//Koenig oder Turm auf falsche Position fuer Rochade gesetzt
   //TODO: ev noch fehlerflags setzen? neu 19.9.24
#ifdef DEBUG3
   lcd_print1("Err nach=",nach); lcd_writec(' '); secwait(1);//test
#endif
   return false;
  }
 //wenn Koenig mehr als 1 Feld weit nach C oder G gezogen, dann ist es ein Rochadezug:
 if((nach==cpos && von!=cpos-1 && von!=cpos+1) || (nach==gpos && nach!=gpos-1 && nach!=gpos+1))
  {return true;}
 return false;
#else
 return (nach==von+2 || nach==von-2); //Koenig hat 2 Felder gezogen --> Rochade
#endif
}

void rochade_fertig_machen(uchar nach,uchar farbe)
{
 uchar turmvon,turmnach;
 lcd_write4("Rochade");
#ifdef SCHACH960
 if(wahrscheinlich_rochade960)
  {
   uchar cpos,gpos; //neu 19.9.24
   if(farbe==WEISS) {cpos=2; gpos=6;} else {cpos=56+2; gpos=56+6;}
   nach &= ~ROCHADE_FLAG;
   if(nach==cpos+1 || nach==gpos-1) //wurde Turm zuerst abgestellt?
    {uchar koenignach; //neu 19.9.24
#ifdef DEBUG3
     lcd_print1("Turm=",nach); lcd_writec(' '); secwait(1);//test
#endif
     turmnach=nach;
     if(nach==cpos+1) {nach=cpos; zug_aus_sensoren_ermitteln(NULL,&koenignach,farbe);}
     else             {nach=gpos; zug_aus_sensoren_ermitteln(NULL,&koenignach,farbe);}
#ifdef DEBUG3
     lcd_print1("Koenig=",koenignach); lcd_writec(' '); secwait(1);//test
#endif
     if(koenignach!=nach) {fehlerflags|=FEHLER_ROCHADESENSOR; rochadefehlernr=5;}
    }
   else zug_aus_sensoren_ermitteln(NULL,&turmnach,farbe);
#ifdef DEBUG3
   lcd_print1("turmnach=",turmnach); lcd_writec(' '); secwait(1);//test
#endif
   if(turmnach==cpos) {turmnach=cpos+1;}//test
   if(turmnach==gpos) {turmnach=gpos-1;}//test
   wahrscheinlich_rochade960=0; //ist sicher eine Rochade
   //nach &= ~ROCHADE_FLAG;
  }
 else zug_aus_sensoren_ermitteln(&turmvon,&turmnach,farbe);
#else
 zug_aus_sensoren_ermitteln(&turmvon,&turmnach,farbe);
#endif
 if(nach==2 || nach==58)
      {if(turmnach!=nach+1) {rochadefehlernr=3; fehlerflags|=FEHLER_ROCHADESENSOR;}}
 else {if(turmnach!=nach-1) {rochadefehlernr=4; fehlerflags|=FEHLER_ROCHADESENSOR;}}
}

bool enpassant(uchar von,uchar nach) //ist es ein en-passant-Zug?
{
 if((spielbrett[von]&FIGUR)!=BAUER || spielbrett[nach]!=LEER) return false;
 if(nach==von+8 || nach==von-8) return false;
 uchar gegnerfarbe=WEISS+SCHWARZ-(spielbrett[von]&FARBE);
 if(nach==von+7 || nach==von-9) return (spielbrett[von-1]==BAUER+ENPASSFLAG+gegnerfarbe);
 if(nach==von+9 || nach==von-7) return (spielbrett[von+1]==BAUER+ENPASSFLAG+gegnerfarbe);
 return false;
}

void enpassant_fertig_machen(uchar von,uchar nach,uchar farbe,int8_t *feld1,int8_t *feld2)
{
 lcd_write4("En Passant");
 uchar schlagfeld = (nach==von+7 || nach==von-9) ? von-1 : von+1;
 secwait(1);//test
 for(int i=0; i < 30*2; i++) //maximal etwa 30 Sec warten bis geschlagene Figur genommen
  {
   sensorfeld_einlesen_digital(feld2);
   if(feld2[schlagfeld]==0) break;
   milliwait(500);
  }
}

void stellung_anzeigen()
{
 uchar brettzeile=1, taste=0;
 while(taste!=TASTE_MITTE)
  {
   lcd_clear(); lcd_printf("Stellung:");
   lcd_print(2,"%d: %s", brettzeile, get_brettzeile(brettzeile));
   lcd_write3("weiter: Rechte");
   lcd_write4("fertig: ok-Taste");
   taste=taste_abwarten(TASTE_ALLE);
   if(taste==TASTE_RECHTS) {if(++brettzeile==9) brettzeile=1;}
   else                    {if(--brettzeile==0) brettzeile=8;}
  }
}

void zug_aus_sensoren_ermitteln(uchar *von,uchar *nach,uchar farbe)
{
 //*von=12; *nach=12+16; return; //test E2-E4
 int8_t feld1[64],feld2[64];
 int8_t j, geschlagen=64, fehler=0;
 nrfehlersensor = -1;//bisher kein Fehler
 sensorfeld_einlesen_digital(feld1);
#ifdef SCHACH960
 //von==NULL kommt nur vor um bei Rochade den Turm noch abzustellen
 if(von!=NULL)
 {
#endif
 do {
  sensorfeld_einlesen_digital(feld2);
  for(j=0;j<64;j++)
   {
    if(feld2[j]!=feld1[j])
     {
      if(feld2[j]==0) //Figur weggenommen
          {
	   feld1[j]=feld2[j];
	   if((spielbrett[j]&FARBE)==farbe) {*von=j;}
	   else //der Roboterarm wird zuerst geschlagene Figur nehmen, ausser bei enpassant
	     {geschlagen=j; j=64;}
	  }
      else //Figur gesetzt? aber keine abgehoben - oder Figur von leerem Feld abgehoben?
          {
	   if(++fehler>=3) {fehlerflags|=FEHLER_SENSOREN; nrfehlersensor=j; return;}
	   j=64;
	  }
      milliwait(200);
      break;
     }
    }//ende for
  if(j==64)
   {//neu ab Version 0.8: Tastendruck pruefen fuer ev. Zug zuruecknehmen
    if(taste_gedrueckt(TASTE_ALLE))
     {uchar taste=0;
      while(taste!=TASTE_MITTE && taste!=TASTE_LINKS)
       {lcd_clear(); lcd_printf("Hilfe: Zug zuruecknehmen: Linke");
	lcd_write3("Stellung: Rechte");
	lcd_write4("weiter: ok-Taste");
	taste=taste_abwarten(TASTE_ALLE);
	if(taste==TASTE_RECHTS) stellung_anzeigen();
	else if(taste==TASTE_LINKS && farbe==spielerfarbe)
	 {
	  bool ok=zug_zuruecknehmen();
	  lcdzugliste.refresh();
	  secwait(3);
	  if(ok) lcd_printf("Vorherige Stellung wieder setzen Dann ok-Taste ");
	  else lcd_write3("weiter: ok-Taste");
	  //TODO: ev. Stellung anzeigen mit aktuellem spielbrett
	  taste=taste_abwarten(TASTE_MITTE);
	  lcdzugliste.refresh();
	  warte_taste_losgelassen();
	  *von=1; *nach=1; //fuer Erkennung Zug zuruecknehmen
	  return;
	 }
	warte_taste_losgelassen();
       }
      lcdzugliste.refresh();
      secwait(3);
     }
   }
 } while(j==64);
 //Jetzt steht in *von die Startposition der gezogenen Figur,
#ifdef SCHACH960
 }
#endif
 //und in feld1 ist die Stellung bevor Figur wieder abgestellt.
 do {
  sensorfeld_einlesen_digital(feld2);
  for(j=0;j<64;j++)
   {
    if(feld2[j]!=feld1[j])
     {
      feld1[j]=feld2[j];
      if(feld2[j]==1) //Figur abgesetzt
       {*nach=j;
       }
      else //Figur weggenommen, also Figur geschlagen?
       {if(geschlagen==64) geschlagen=j;
	else {fehlerflags|=FEHLER_SENSOREN; nrfehlersensor=j;}
	//if((spielbrett[j]&FARBE)==farbe) //eigene Figur geschlagen?
	//   {} //dieser Fehler wird schon mit Ueberpruefung auf gueltigen Zug abgefangen
#ifdef SCHACH960
	if(von!=NULL && (spielbrett[j]&FARBE)==farbe) //eigene Figur geschlagen?
	 {//wahrscheinlich Koenig und eigener Turm vom Brett genommen
	  wahrscheinlich_rochade960=1;
	  geschlagen=64; //nicht wirklich eine Figur geschlagen
	 }
#endif
	j=64; //Figur noch nicht wieder abgestellt
       }
      break;
     }
   }//ende for
 } while(j==64);
 //in feld1 ist jetzt Stellung nachdem Figur wieder abgestellt und ev. weggenommene entfernt
#ifdef SCHACH960
 if(von!=NULL)
  {
   if(geschlagen!=64 && geschlagen!= *nach) fehlerflags|=FEHLER_SENSOREN;
   if(*nach == *von && wahrscheinlich_rochade960==0)  fehlerflags|=FEHLER_SENSOREN;
   if(fehlerflags!=0) return;
   //if(rochade(*von,*nach))
   if(rochade(von,nach))
    {rochade_fertig_machen(*nach,farbe); *nach |= ROCHADE_FLAG;}
   else if(enpassant(*von,*nach)) enpassant_fertig_machen(*von,*nach,farbe,feld1,feld2);
  }
#else
 if(geschlagen!=64 && geschlagen!= *nach) fehlerflags|=FEHLER_SENSOREN;
 if(*nach == *von)  fehlerflags|=FEHLER_SENSOREN;
 if(fehlerflags!=0) return;
 //if(rochade(*von,*nach)) rochade_fertig_machen(*nach,farbe);
 if(rochade(von,nach)) rochade_fertig_machen(*nach,farbe);
 else if(enpassant(*von,*nach)) enpassant_fertig_machen(*von,*nach,farbe,feld1,feld2);
#endif
 return;
}

void zug_aus_sensoren_ermitteln_mitfehlertest(uchar *von,uchar *nach,uchar farbe)
{
 rochadefehlernr=0;
 stellung_pruefen_und_korrigieren();
 for(;;)
  {
   for(int8_t i=0;i<2;)
    {
     fehlerflags=0;
     zug_aus_sensoren_ermitteln(von,nach,farbe);
     if(fehlerflags==0) return;
     fehler_auf_lcd_anzeigen(); secwait(3);
     if(fehlerflags==FEHLER_ROCHADESENSOR || rochadefehlernr!=0)
      {
       //fehler_auf_lcd_anzeigen(); secwait(3);
       switch(rochadefehlernr)
	{
	 case 1: lcd_printf("Rochade? Koenig hat schon mal gezogen."); break;
	 case 2: case 5: lcd_printf("Rochade? Bitte Koenig korrekt ziehen."); break;
	 //case 3: case 4:
	 default: lcd_printf("Bitte Turm von  Rochade korrekt ziehen."); break;
	}
       lcd_write4("dann ok-Taste");
       auf_taste_warten(TASTE_MITTE);
       lcdzugliste.refresh();
       fehlerflags=0;
       return;
      }
     if(++i==2) break; //nach 2-maligem Fehler Schlaufe abbrechen
     if(*von != *nach) {lcd_write4("zurueck,ok-Taste"); auf_taste_warten(TASTE_MITTE);}
     lcd_write4("nochmals ziehen");
    }
   lcd_clear();
   //lcd_printf("Sensoren-Fehler Bitte justiere  Figuren.");
   //lcd_write4("dann ok-Taste");
   //auf_taste_warten(TASTE_MITTE);
   lcd_printf("Sensor-Fehler:  justiere Figuren");
   lcd_write3("dann ok-Taste");
   lcd_write4("odr Hilfe: Linke");
   uchar taste=taste_abwarten(TASTE_LINKS|TASTE_MITTE);
   if(taste!=TASTE_MITTE) stellung_anzeigen();
   lcdzugliste.refresh();
   secwait(1); lcd_write4("neu ziehen");//test
   warte_taste_losgelassen();
  }
}

void zug_von_hand_machen(Zug& zug) //wird aufgerufen wenn der Computerzug gemacht werden soll
{
 uchar von,nach,fehlerflag=0;
 timer0_init();
 for(;;)
  {
   zug_aus_sensoren_ermitteln_mitfehlertest(&von, &nach, spielbrett[zug.von&IMASK]&FARBE);
#ifdef SCHACH960
   if(von==zug.von && nach==zug.nach) break;
#ifdef DEBUG2
   lcd_print(1,"von=%d nach=%d ",von,nach);//test
   lcd_print(2,"z.v=%d z.na=%d ",zug.von,zug.nach);//test
   lcd_write_zeile(3,"DEBUG2 ok-Taste");//test
   auf_taste_warten(TASTE_MITTE);//test
   lcd_write_zeile(3," ");//test
#endif
   if((spielbrett[zug.von&IMASK]&FIGUR)!=KOENIG &&
      von==(zug.von&IMASK) && nach==(zug.nach&IMASK)) break; //mit IMASK damit Promoflag bei Bauernzug geloescht?
#else
   if(von==(zug.von&IMASK) && nach==(zug.nach&IMASK)) break; //wieso IMASK ??
#endif
   fehlerflag=1;
   Zug falscher; falscher.set(von,nach);
   lcd_goto(4,1); lcd_write("Zug falsch "); lcd_write(zug.print());
#ifdef SCHACH960
   auf_taste_warten(TASTE_MITTE);//test
   secwait(3);//test
   lcd_goto(4,1); lcd_write("falscher:"); lcd_write(falscher.print());//test
   auf_taste_warten(TASTE_MITTE);//test
#endif
   secwait(3);
   lcd_write4("zurueck dann ok");
   auf_taste_warten(TASTE_MITTE);
   lcd_goto(4,1); lcd_write(" ziehe "); lcd_write(zug.print()); lcd_write("    ");
  }
 if(fehlerflag)
  {
   lcd_write4("Ok richtiger Zug"); //provi.
   milliwait(1000);
   lcd_write4(zug.print());
   milliwait(1000);
  }
 timer0_stop();
}

void suart_parameter_senden(char c1,char c2)
{
 char txt[5]; //z.B. "F:0\n";
 int8_t i=0;
 txt[i++]=c1; txt[i++]=':'; txt[i++]=c2;
 txt[i++]='\n'; txt[i]=0;
#ifdef DEBUG
 txt[i-1]=0; lcd_write4s("suart:%s",txt); txt[i-1]='\n';
#endif
 suart_write(txt);
 while(suart_getc() != '\n') {} //Antwort abwarten
}

char get_figurabk(uchar v)
{
 //englische Figurenabkuerzungen, erster und letzter Wert ungueltig:
 char figurabk[8]={'N','P','N','B','R','Q','K','N'};
 int8_t i=spielbrett[v]&FIGUR;
 return figurabk[i];
}

char getfig(uchar x)
{
 char figurabk[8]={'?','B','S','L','T','D','K','?'}; //TODO: englische statt deutsche Abkuerzungen?
 char fig=figurabk[x&FIGUR];
 if((x&FARBE)==SCHWARZ) fig += 'a'-'A'; //Kleinbuchstaben fuer Schwarz, Grossbuchstaben fuer Weiss
 return fig;
}

void suart_zug_senden(Zug& zug) //Zug vom Roboter machen lassen
{
 uchar v=(zug.von&IMASK), n=(zug.nach&IMASK);
 bool schlagflag = (spielbrett[n] != LEER); //flag setzen wenn an Zielposition eine Figur steht
 bool enpassflag = enpassant(v, n);
 if(enpassflag) schlagflag=true; //bei en passant schlagflag ebenfalls setzen
 //von Zug::print() kopiert und angepasst:
 char zugtext[16];
 char promofigurabk[4]={'N','B','R','Q'}; //englische Figurenabkuerzungen
 uchar bauerumwandlung=0;
 int8_t i=0;
 suart_parameter_senden('G', (brettlage==SCHWARZ)?'1':'0'); //Sicherstellen dass Roboter Brettlage kennt
 //if(rochade(zug.von, zug.nach))
 if(rochade(&zug.von, &zug.nach)) //neu 19.9.24
  {
   bool grosse = (n&7)==2; //grosse Rochade wenn Koenig nach C gezogen wird
   if(grosse) {strcpy(zugtext,"o-o-o"); i=5;} //grosse Rochade
   else       {strcpy(zugtext,"o-o"); i=3;}   //kleine Rochade
   //Sicherstellen dass Roboter Computerfarbe kennt:
   suart_parameter_senden('F', (computerfarbe==SCHWARZ)?'1':'0');
#ifdef SCHACH960
   suart_parameter_senden('K', koenig_startpos);
   suart_parameter_senden('T', (grosse) ? turm1_startpos : turm2_startpos);
#endif
  }
 else
  {
   char figur=get_figurabk(v);
   suart_parameter_senden('P', figur); //Figur die gezogen wird an Roboter senden
   zugtext[i++] = 'A'+(v%8);  zugtext[i++] = '1'+(v/8);
   if(schlagflag) zugtext[i++] = 'x'; else zugtext[i++] = '-';
   zugtext[i++] = 'A'+(n%8); zugtext[i++] = '1'+(n/8);
   if(zug.von&PROMOFLAG) zugtext[i++] = bauerumwandlung = promofigurabk[zug.nach>>6];
   if(enpassflag) {zugtext[i++]='e'; zugtext[i++]='.'; zugtext[i++]='p';zugtext[i++]='.';}
  }
 if(bauerumwandlung) //bei Bauerumwandlung: Hinweis an Benutzer
  {
   const char *txt="Dame setzen";
   if(bauerumwandlung=='R')      txt="Turm setzen";
   else if(bauerumwandlung=='B') txt="Laeufer setzen";
   else if(bauerumwandlung=='N') txt="Springer setzen";
   lcd_write4(txt);
  }
 zugtext[i++] = '\n';
 zugtext[i] = 0;
 suart_write(zugtext);
 uchar c=0;
 while(c!='\n') {c=suart_getc();} //Antwort abwarten
}

bool bauer_umwandlung(Brett& brett,uchar from,uchar to,uchar farbe)
{
 if((brett[from]&FIGUR)!=BAUER) return false;
 return ((farbe==WEISS && to>=64-8) || (farbe==SCHWARZ && to<8)); //gegnerische Grundlinie erreicht?
}

Zug spielerzug(Brett& brett,uchar farbe)
{
 Zugliste zugliste;
 Zug zug;
 alle_moeglichen_zuege(brett,farbe,&zugliste,0);
 if(zugliste.istleer()) {zug.von=zug.nach=0; return zug;}//kein Zug mehr moeglich
 timer0_init();
 for(;;)
  {
   uchar errflag=0;
   uchar von,nach;
   zug_aus_sensoren_ermitteln_mitfehlertest(&von,&nach,farbe);
   zug.set(von,nach);
   if(von==1 && nach==1) //Zug zuruecknehmen gemacht?
    {timer0_stop(); return zug;}
   if(bauer_umwandlung(brett,von,nach,farbe)) zug.setpromo(DAME);//DAME zum Gueltigkeit des Zuges testen
   if(zugliste.zugsuchen(zug)==NULL) errflag=2; //ungueltig wenn nicht in Liste vorhanden
   if(errflag==0) break;
   //if(errflag==1) printf("Eingabefehler - Eingabe-Beispiel: E2-E4 oder ? zum Stellung zeigen\n");
   if(errflag==2)
    {
     lcd_goto(3,1); lcd_write(zug.print()); lcd_write(" Ung\365ltig");
     lcd_write4("zur\365ck, ok-Taste"); //sollte \xF5 auch gehen?
     auf_taste_warten(TASTE_MITTE);
#ifdef DEBUG2
     lcd_print(3,"von=%d nach=%d ",von,nach);//test
     auf_taste_warten(TASTE_MITTE);//test
#endif     
     lcdzugliste.refresh();
    }
  }
 if(zug.getpromo()!=0) //Bauerumwandlung?
  {
   string17 texte[4]={"Dame    ","Turm    ","Springer","Laeufer "};
   int8_t wahl=lcd_auswahlfrage("BauerUmwandlung:",4,texte);
   if(wahl!=0)
    {
     uchar fig;
     switch(wahl)
      {case 2: fig=TURM; break;
       case 3: fig=SPRINGER; break;
       case 4: fig=LAEUFER; break;
       default: fig=DAME; break;
      }
     zug.setpromo(fig);
    }
   lcdzugliste.refresh();
  }
 timer0_stop();
 return zug;
}

void stockfish_fertig(Zug &zug,const char *str);

void fertig(Zug &letzterzug,const char *str)
{
 lcd_write4i("nach%3d Zuegen ",lcdzugliste.zugnr);
 lcd_printf(str);
#ifdef MITUART
 if(uart_ok) stockfish_fertig(letzterzug,str);
#endif
 auf_taste_warten(TASTE_MITTE);
}

#ifdef SPULENFREQUENZTEST
void frequenz_justieren(int n) //test
{
 OCR0A += n;
}

void delay_sec(int n)
{
 for(;n>0;--n)
  {
   for(int i=0;i<500;i++)
    {_delay_ms(1);}
  }
}
#endif

int main(void)
{
 // PortA alles Eingaenge fuer A/D-Wandler
 PORTB = 0x1F; //PortB untere 5 Bits Pullups fuer Tasten
 DDRB  = 0x20; //PortB alles Eingaenge ausser 6.Bit fuer LCD-Beleuchtung, oder hier TestLED
 PORTC = 0x00; //PortC alle auf Low fuer inaktiv (N-Chann-Transistoren)
 DDRC  = 0xFF; //PortC alles Ausgaenge
 // PortD wird fuer LCD benutzt
 
 //timer0_init(); //nur bei Bedarf ein- und aus-schalten
 timer1_init();
 sei(); //Interrupt einschalten

#ifdef SPULENFREQUENZTEST
 timer0_init();
 sei();
 anregpin=(1<<PC0);
 delay_sec(10);
 for(;;)
  {
   /*
   delay_sec(5);
   frequenz_justieren(1); //test
   delay_sec(5);
   frequenz_justieren(-2); //test
   delay_sec(5);
   frequenz_justieren(1); //test
   */
  }
#endif
 
 lcd_init();
 PORTB |= (1<<5); //Hintergrundbeleuchung ein

#ifdef SPULENTEST
 timer0_init();
 char text[32];
 uint werta[8],wertb[8],wertc[8];
 //uint wertd[8],werte[8],wertf[8],wertg[8],werth[8];
 uint t0,t1,sensorzeit;
 for(uint i=0;i<20;i++)  //etwa 60 Sekunden lang Werte aller Sensoren anzeigen
  {
   cli(); t0=millisec; sei();
   warte_bis_alle_sensoren_gelesen();
   cli(); t1=millisec; sei();
   sensorzeit = (t1>t0) ? t1-t0 : t1+1000-t0;
   for(int j=0;j<8;j++)
    {
     werta[j] = sensorfeld[j*8];
     wertb[j] = sensorfeld[j*8+1];
     wertc[j] = sensorfeld[j*8+2];
     /*
     wertd[j] = sensorfeld[j*8+3];
     werte[j] = sensorfeld[j*8+4];
     wertf[j] = sensorfeld[j*8+5];
     wertg[j] = sensorfeld[j*8+6];
     werth[j] = sensorfeld[j*8+7];
     */
    }
   sprintf(text,"A:%4d%4d%4d  ",werta[0],werta[1],werta[2]); lcd_goto(1,1); lcd_write(text);
   sprintf(text,"B:%4d%4d%4d  ",wertb[0],wertb[1],wertb[2]); lcd_goto(2,1); lcd_write(text);
   sprintf(text,"C:%4d%4d%4d  ",wertc[0],wertc[1],wertc[2]); lcd_goto(3,1); lcd_write(text);
   sprintf(text,"messzeit %d ms ",sensorzeit); lcd_goto(4,1); lcd_write(text);
   milliwait(500);
   /*
   sprintf(text,"%4d%4d%4d%4d",werta[0],werta[1],werta[2],werta[3]); lcd_goto(1,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertb[0],wertb[1],wertb[2],wertb[3]); lcd_goto(2,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertc[0],wertc[1],wertc[2],wertc[3]); lcd_goto(3,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertd[0],wertd[1],wertd[2],wertd[3]); lcd_goto(4,1); lcd_write(text);
   milliwait(500);
   sprintf(text,"%4d%4d%4d%4d",werta[4],werta[5],werta[6],werta[7]); lcd_goto(1,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertb[4],wertb[5],wertb[6],wertb[7]); lcd_goto(2,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertc[4],wertc[5],wertc[6],wertc[7]); lcd_goto(3,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertd[4],wertd[5],wertd[6],wertd[7]); lcd_goto(4,1); lcd_write(text);
   milliwait(500);
   
   sprintf(text,"E:%4d%4d%4d  ",werte[0],werte[1],werte[2]); lcd_goto(1,1); lcd_write(text);
   sprintf(text,"F:%4d%4d%4d  ",wertf[0],wertf[1],wertf[2]); lcd_goto(2,1); lcd_write(text);
   sprintf(text,"G:%4d%4d%4d  ",wertg[0],wertg[1],wertg[2]); lcd_goto(3,1); lcd_write(text);
   sprintf(text,"messzeit %d ms ",sensorzeit); lcd_goto(4,1); lcd_write(text);
   milliwait(500);
   sprintf(text,"%4d%4d%4d%4d",werte[0],werte[1],werte[2],werte[3]); lcd_goto(1,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertf[0],wertf[1],wertf[2],wertf[3]); lcd_goto(2,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertg[0],wertg[1],wertg[2],wertg[3]); lcd_goto(3,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",werth[0],werth[1],werth[2],werth[3]); lcd_goto(4,1); lcd_write(text);
   milliwait(500);
   sprintf(text,"%4d%4d%4d%4d",werte[4],werte[5],werte[6],werte[7]); lcd_goto(1,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertf[4],wertf[5],wertf[6],wertf[7]); lcd_goto(2,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",wertg[4],wertg[5],wertg[6],wertg[7]); lcd_goto(3,1); lcd_write(text);
   sprintf(text,"%4d%4d%4d%4d",werth[4],werth[5],werth[6],werth[7]); lcd_goto(4,1); lcd_write(text);
   milliwait(500);
   */
  }
 timer0_stop();
#endif
 
 //lcd_goto(1,1); lcd_write("Schach 0.0      "); //1.Zeile (16 Zeichen)
 lcd_goto(1,1); lcd_write("Schach "); lcd_write(VERSION); lcd_write("      "); //1.Zeile

#ifdef UARTTEST
 uart0_init();
 secwait(10);
 for(int i=0;i<30000;i++)
  {
   if(i==0 || uart_check())
    {
     char txt[40];
     uchar c=0,j=0;
     if(uart_check())
      {while(c!='\n') {c=uart_getc(); if(c>=' ' && j<39) txt[j++]=c;} //Zeichen empfangen
      }
     else {txt[0]='A'; txt[1]='B'; txt[2]='C'; j=3;}
     txt[j]=0;
     lcd_goto(2,1); lcd_write("uart: "); lcd_write(txt); lcd_writec(' ');
     for(j=0;(c=txt[j])>=' ';j++) uart_putc(c); //Zeichen zuruecksenden
     uart_putc('\n');
     i=0;
    }
   milliwait(1);
  }
#endif

#ifdef MITUART
 uart0_init();
 lcd_goto(2,1); 
 uart_ok=uart_verbindung_aufbauen(); //Verbindung zum Computer fuer Stockfish
 if(uart_ok) {lcd_write_fill("uart ok"); stockflag=true;}
 else        {lcd_keine_UART_Verbindung(); uart_stopp();}
 secwait(2);
 suart_init(); //Soft-UART
 suart_ok=suart_verbindung_aufbauen(); //Verbindung zu robo.cc fuer Motorensteuerung
 if(suart_ok) {lcd_write_fill("suart ok");}
 else         {lcd_write("keine SUART-Verb"); secwait(4);}
 secwait(2);
#endif

 calib_von_eeprom_einlesen(); //Calibrationswerte fuer Sensoren
 spielsuchtiefe=spielstaerke_von_eeprom_einlesen();
 lcd_goto(2,1);
 uint version=eeprom_read_word(&eeversion);
 if(version==100)
  {
   uint8_t status=eeprom_read_byte(&eeCalibStatus);
   lcd_print1("CalibStatus=",status); lcd_write("   ");
   lcd_goto(3,1); lcd_print1("Spielstaerke=",spielsuchtiefe);
  }
 else
  {
   if(version==0xFFFF) lcd_write("EEPROM leer");
   else lcd_print1("eeVersion=",version);
  }
#ifdef UMLAUTTEST
 for(uchar c0=0xD0;;c0+=0x10)
  {
   lcd_goto(4,1);
   for(uchar c=c0,i=0;i<16;c++,i++) lcd_writec(c);
   auf_taste_warten(TASTE_MITTE);
   if(c0==0xF0) break;
  }
 lcd_goto(4,1); lcd_write("Ae..Ue: \xE1\xEF\xF5");
 auf_taste_warten(TASTE_MITTE);
 // Umrechnung Hex-Octal: 0xE1 = 1110 0001 = 11 100 001=oktal 341 
 lcd_goto(4,1); lcd_write("ae..ue: \341\357\365");
 auf_taste_warten(TASTE_MITTE);
 lcd_write4("zur\365ck, ok-Taste");
#endif
 secwait(5); //Startmeldung 5 Sekunden stehen lassen

#ifdef STACKTEST
 lcd_goto(4,1); lcd_print1("SP=",SP);
 secwait(3);
 lcd_goto(4,1); lcd_print1("freezaehler=",freezaehler);
 secwait(3);
#endif

 int8_t modus=0;
#define MODUS_HAUPTMENU   0
#define MODUS_FARBWAHL      1 //Spiel starten
#define MODUS_SPIELSTAERKE  2
#define MODUS_STELLUNG_SETZEN 3
#define MODUS_CALIB           4
#define MODUS_ROBOTCALIB      5
#define MODUS_SEITENWAHL      6
#define MODUS_SCHACH960       7
#define MODUS_RESERVE         8 //Reserve
#define MODUS_SPIELEN         10

 int8_t stellungsflag=0;
 string17 weissSchwarz[2]={"Weiss  ","Schwarz"};

 while(1) //Hauptschlaufe
  {
   milliwait(100);
   if(modus==MODUS_HAUPTMENU)
    {
     string17 texte[5]={"Spiel starten ", //Reihenfoge muss mit MODUS-Nummerierung uebereinstimmen
			"Spielst\341rke   ",
			"neue Stellung ",
			"Calib Sensoren",
			"Calib Robot   "};
     int8_t wahl = lcd_auswahlfrage("Hauptmenu:",5,texte);
     //if(wahl==1) modus=MODUS_FARBWAHL; //zum Spiel starten zuerst Farbe auswaehlen lassen
     //else if(wahl==2) ...
     modus=wahl; //Ist einfacher so, da MODUS-Nummerierung schon uebereinstimmt.
    }
   else if(modus==MODUS_SPIELSTAERKE)
    {
#ifdef MITUART
     string17 texte[10]={"0  ","1  ","2  ","3  ","4  ","5  ","6  ","10 ","20 ","30 "};
     int8_t n=0; while(n<10 && atoi(texte[n++])!=spielsuchtiefe) {}
     int8_t wahl=lcd_auswahlfrage("Spielst\341rke:",10,texte,n);
     if(wahl!=0) spielsuchtiefe=atoi(texte[wahl-1]);
#else
     string17 texte[8]={"0  ","1  ","2  ","3  ","4  ","5 ","6 ","7 "};
     int8_t wahl=lcd_auswahlfrage("Spielst\341rke:",8,texte,spielsuchtiefe+1);
     if(wahl!=0) spielsuchtiefe=wahl-1;
#endif
     lcd_goto(4,1); lcd_print1("Suchtiefe=",spielsuchtiefe); lcd_write("     ");
     milliwait(500);
#ifdef MITUART
     if(spielsuchtiefe>MAXEIGENEENGINE) stockflag=true;
     else
      {string17 text[2]={"Eigene   ","Stockfish"};
       wahl=lcd_auswahlfrage("Schach-Engine:",2,text,stockflag?2:1);
       if(wahl>=1) stockflag=(wahl==2);
      }
#endif
     modus=0;
    }
   else if(modus==MODUS_FARBWAHL)
    {
     //string17 texte[2]={"Weiss  ","Schwarz"};
     //int8_t wahl=lcd_auswahlfrage("Deine Farbwahl:",2,texte);
     int8_t wahl=lcd_auswahlfrage("Deine Farbwahl:",2,weissSchwarz);
     if(wahl==0) modus=MODUS_HAUPTMENU;
     else
      {
       spielerfarbe = (wahl==1)?WEISS:SCHWARZ;
       computerfarbe = WEISS+SCHWARZ-spielerfarbe;
       if(computerfarbe==SCHWARZ) suart_write("F:1\n"); else suart_write("F:0\n");
       //modus=MODUS_SPIELEN;
       modus=MODUS_SEITENWAHL;
      }
    }
   else if(modus==MODUS_SEITENWAHL)
    {
     //string17 texte[2]={"Weiss  ","Schwarz"};
     //int8_t wahl=lcd_auswahlfrage("Seitenwahl:",2,texte);
     int8_t wahl=lcd_auswahlfrage("Seitenwahl:",2,weissSchwarz);
     if(wahl==0) modus=MODUS_HAUPTMENU;
     else
      {
       brettlage = (wahl==1)?WEISS:SCHWARZ;
       if(brettlage==SCHWARZ) suart_write("G:1\n"); else suart_write("G:0\n");
       modus=MODUS_SPIELEN;
      }
    }
   else if(modus==MODUS_ROBOTCALIB)
    {
     string17 texte[4]={"A1XYPOS ","H8XYPOS ","KIPPFAKT","ZSENSOR "};
     int8_t wahl=lcd_auswahlfrage("Auswahl:",4,texte);
     char vorzeichen=0, cstr[20], antwort[20];
     if(wahl==1 || wahl==2)
      {
       strcpy(cstr,"CALIB:A1X+2\n");
       if(wahl==2) {cstr[6]='H'; cstr[7]='8';}
       cstr[9]=0; lcd_write4(cstr); //Kommando ohne Parameter auf LCD anzeigen
       //TODO: Verschiedene Schrittweiten, bisher nur 1mm
       for(char flag=1;;)
	{
	 milliwait(200);
	 if((tastegedrueckt&TASTE_RECHTS)!=0)     {vorzeichen='-'; cstr[8]='X'; warte_losgelassen_rechts();}
	 else if((tastegedrueckt&TASTE_LINKS)!=0) {vorzeichen='+'; cstr[8]='X'; warte_losgelassen_links();}
	 else if((tastegedrueckt&TASTE_OBEN)!=0)  {vorzeichen='-'; cstr[8]='Y'; warte_losgelassen_oben();}
	 else if((tastegedrueckt&TASTE_UNTEN)!=0) {vorzeichen='+'; cstr[8]='Y'; warte_losgelassen_unten();}
	 else if((tastegedrueckt&TASTE_MITTE)!=0)
	  {// wenn als erstes Mitte-Taste gedrueckt: wie Taste unten
	    if(flag) {vorzeichen='+'; cstr[8]='Y'; warte_losgelassen_mitte();}
	    else break; //sonst Schleife verlassen
	   }
	 else vorzeichen=0;
	 if(vorzeichen!=0)
	  {
	   flag=0; //Flag fuer ersten Tastendruck ruecksetzen
	   cstr[9]=vorzeichen; 
	   suart_write_wait(cstr,antwort);
	   lcd_write4(antwort);
	  }
	} //Ende for-Schlaufe
      }
     else if(wahl==3 || wahl==4)
      {
       uint8_t v;
       if(wahl==3) {strcpy(cstr,"CALIB:KIPPFAKT+5\n"); v=14;} //TODO: Schrittweite aendern?
       else        {strcpy(cstr,"CALIB:ZSPOS+1\n"); v=11;}
       cstr[v]=0; lcd_write4(cstr);//Kommando ohne Parameter auf LCD anzeigen
       for(char flag=1;;)
	{
	 milliwait(200);
	 if((tastegedrueckt&TASTE_OBEN)!=0)  {vorzeichen='+'; warte_losgelassen_oben();}
	 else if((tastegedrueckt&TASTE_UNTEN)!=0) {vorzeichen='-'; warte_losgelassen_unten();}
	 else if((tastegedrueckt&TASTE_MITTE)!=0)
	  {// wenn als erstes Mitte-Taste gedrueckt: wie Taste unten
	    if(flag) {vorzeichen='-'; warte_losgelassen_mitte();}
	    else break; //sonst Schleife verlassen
	   }
	 else vorzeichen=0;
	 if(vorzeichen!=0)
	  {
	   flag=0; //Flag fuer ersten Tastendruck ruecksetzen
	   cstr[v]=vorzeichen;
	   suart_write_wait(cstr,antwort);
	   lcd_write4(antwort);
	  }
	} //Ende for-Schlaufe
      }
     //else if(wahl==5 ... falls weitere Kommandos
     if(wahl!=0)
      {
       lcd_write4("CALIB:ok\n");//test
       warte_losgelassen_mitte();
       suart_write_wait("CALIB:ok\n",antwort);
       lcd_write4(antwort); secwait(3);
      }
     modus=MODUS_HAUPTMENU;
    }
   else if(modus==MODUS_SPIELEN)
    {
     spielstaerke_in_eeprom_speichern(spielsuchtiefe);
     if(stellungsflag==0) spielbrett.grundstellung();
#ifdef MITUART
     uchar alte_spielsuchtiefe=spielsuchtiefe;
     if(stockflag && !uart_ok)
      {uart_start();
       uart_ok=uart_verbindung_aufbauen();
       lcd_goto(3,1); 
       if(uart_ok)
	{
	 lcd_write_fill("uart ok");
	}
       else
	{lcd_keine_UART_Verbindung();
	 uart_stopp();
	 if(spielsuchtiefe>MAXEIGENEENGINE) spielsuchtiefe=MAXEIGENEENGINE;
	}
       secwait(1);
      }
     if(stockflag && uart_ok)
      {
       uart_write("suchtiefe:"); uart_print(spielsuchtiefe); uart_putc('\n');
       milliwait(10); //provi. keine Antwort abwarten bei Suchtiefe setzen
#ifdef SCHACH960
       if(stellungsflag!=0)
	  {uart_startstellung_senden(spielbrett);
	   lcd_write_zeile(4,"Stellg. gesendet");
	  }
#endif
      }
#endif //MITUART
     Zug zug; zug.set(0,0);
#ifdef STACKTEST
     lcd_goto(4,1); lcd_print1("SP=",SP);
     secwait(1);
#endif
     //lcd_clear();
     lcdzugliste.reset();
     zeile4.write("Weiss am Zug ...");
     if(spielerfarbe==WEISS)
      {
       zug=spielerzug(spielbrett,spielerfarbe);
       zug_wirklich_machen(spielbrett,zug);
       lcdzugliste.write(spielerfarbe,zug);
      }
     bool geschlagenflag,bauernflag,koenigsflag; //Bauer gezogen, erster Koenigszug
     for(;;)
      {
#ifdef STACKTEST
       lcd_goto(4,1); lcd_print1("SP=",SP); lcd_print1("fr=",freezaehler);
       secwait(3);
#endif
       timer0_init();
       stellung_pruefen_und_korrigieren();
       timer0_stop();
       zeile4.write("Computer rechnet"); //Computer ist am Zug
       //if(spielsuchtiefe==0) milliwait(500);//test
#ifdef GESCHWINDIGKEITSTEST
       long startzeit = ((tage*24L+stunden)*60+min)*60+sec;
#endif

       //Zug letzterzug=zug;
       Zug keinzug; keinzug.von=keinzug.nach=0;
#ifdef MITUART
       //lcd_write_zeile(3,"besterzug()");//test
       if(stockflag && uart_ok) zug=besterzug_stockfish(spielbrett,zug,computerfarbe,spielsuchtiefe);
       else
#endif
        zug=besterzug(spielbrett,computerfarbe,spielsuchtiefe);

#ifdef GESCHWINDIGKEITSTEST
       long endzeit = ((tage*24L+stunden)*60+min)*60+sec;
       lcd_printz1L(4,"Zeit:",endzeit-startzeit," sec"); secwait(1);
       auf_taste_warten(TASTE_MITTE);
#endif
       if(fehlerflags!=0) {fehler_auf_lcd_anzeigen(); secwait(3); fehlerflags=0;}
       if(zug.von==zug.nach) {fertig(keinzug,"Computer kann keinen Zug mehr machen - Unentschieden"); break;}
       lcdzugliste.write(computerfarbe,zug);
       zeile4.write((computerfarbe==WEISS)?"Weiss zieht ...":"Schwarz zieht...");
       if(suart_ok) suart_zug_senden(zug); //hier Zug vom Roboter machen lassen
       else      zug_von_hand_machen(zug);
       geschlagenflag = (spielbrett[zug.nach] != LEER);
       bauernflag = (((spielbrett[zug.von&IMASK])&FIGUR) == BAUER);
       koenigsflag = (((spielbrett[zug.von&IMASK])&(FIGUR|MOVFLAG)) == (MOVFLAG|KOENIG));
       zug_wirklich_machen(spielbrett,zug);
       if(spielbrett.gewonnen(computerfarbe)) {fertig(keinzug,"Schachmatt - Computer hat gewonnen"); break;}
       if(stellung_vergleich(spielbrett) >= 2) {fertig(keinzug,"3 mal gleiche Stellung - unentschieden"); break;}
       if(fuenfzig_zug_regel()) {fertig(keinzug,"50 Zug-Regel - unentschieden"); break;}
       stellung_speichern(spielbrett, geschlagenflag||bauernflag||koenigsflag);
       //Spieler ist am Zug:
       do {
        zeile4.write((spielerfarbe==WEISS)?"Weiss am Zug ...":"Schwarz am Zug..");
        zug=spielerzug(spielbrett,spielerfarbe);
       } while(zug.von==1 && zug.nach==1); //Zug zuruecknehmen gemacht?
       if(zug.von==zug.nach) {fertig(keinzug,"Du kannst keinen Zug mehr machen - Unentschieden"); break;}
       geschlagenflag = (spielbrett[zug.nach] != LEER);
       bauernflag = (((spielbrett[zug.von&IMASK])&FIGUR) == BAUER);
       koenigsflag = (((spielbrett[zug.von&IMASK])&(FIGUR|MOVFLAG)) == (MOVFLAG|KOENIG));
       if(geschlagenflag||bauernflag||koenigsflag) stellung_speichern_reset();
       zug_wirklich_machen(spielbrett,zug);
       lcdzugliste.write(spielerfarbe,zug);
       //Korrektur 5.4.2024: zug statt letzterzug:
       if(spielbrett.gewonnen(spielerfarbe)) {fertig(zug,"Computer ist Schachmatt - Du hast gewonnen"); break;}
       stellungsflag=0;
      }
#ifdef MITUART
     if(stockflag && !uart_ok) spielsuchtiefe=alte_spielsuchtiefe;
#endif
     modus=0;
    }
   else if(modus==MODUS_CALIB)
    {
     timer0_init();
     string17 texte[3]={"leeres Brett ","Startstellung","Sensortests  "};
     int8_t wahl=lcd_auswahlfrage("Calibrieren:",3,texte);
     if(wahl!=0) calibrieren(wahl);
     modus=0;
     timer0_stop();
    }
   else if(modus==MODUS_STELLUNG_SETZEN) //eine Stellung setzen
    {
     int8_t wahl=lcd_auswahlfrage("Seitenwahl:",2,weissSchwarz);
     brettlage = (wahl<=1)?WEISS:SCHWARZ;
#ifdef SCHACH960
     uchar analyseflag=0;//test
#endif
     string17 texte[4]={"ab Leerbrett ","ab Startstlg.","Schach960    ","gesp. Stellg."};
     wahl=lcd_auswahlfrage("Stellung setzen:",4,texte);
     if(wahl!=0)
      {
       int8_t i,feld1[64],feld2[64];
       timer0_init();
       lcd_clear();
       if(wahl==1)
	{lcd_printf("alle Figuren entfernen");
	 spielbrett.clear();
	}
       else if(wahl==2)
	{lcd_printf("Startstellung setzen");
	 for(i=0;i<64;i++) feld2[i] = (i<16 || i>=64-16) ? 1 : 0;
	 spielbrett.grundstellung();
	}
       else if(wahl==4)
	{lcd_printf("Gesp. Stellung wird vom EEPROM geladen...");
	 letzte_stellung_von_eeprom_einlesen(spielbrett);
	 for(i=0;i<64;i++) feld2[i] = (spielbrett[i]==LEER) ? 0 : 1;
	 milliwait(500);
	 //secwait(3);//test
	 lcd_printf("Stellung setzen (anzeigen mit TASTE_RECHTS)");
	 lcd_print(4,"fertig: MITTE   ");
	 taste_abwarten(TASTE_RECHTS);
	 int j=0; uchar taste=0;
	 do {
	  lcd_print(4,"A%c-H%c: ",j+'1',j+'1');
	  for(i=0;i<8;i++) lcd_writec(getfig(spielbrett[i+j*8]));
	  taste=taste_abwarten(TASTE_RECHTS|TASTE_MITTE);
	  j++; j&=7;
	 } while(taste!=TASTE_MITTE);
	}
       else //if(wahl==3) //Schach960
	{
	 char text[32]; strcpy(text,"Schach960 setze:");
	 lcd_write_zeile(1,"Zufallsstellung?");
	 lcd_write_zeile(2,"ja: ok-Taste");
	 lcd_write_zeile(3,"nein:andre Taste");
	 uchar taste=taste_abwarten(TASTE_ALLE);
	 lcd_write_zeile(3," ");
	 if(taste==TASTE_MITTE)
	  {
	   //TODO: ev. besserer Zufallsgenerator
	   zufalls_startstellung(&text[16]); text[24]=0;
	  }
	 else
	  {//Beispielstellungen Schach960:
	   const char *test1="STSLLKTD"; uchar flag1=1; //1=fuer weitere Figuren setzen/entfernen, 0=ohne
	   const char *test2="SSLDTKTL"; uchar flag2=1;
	   const char *test3="TSKDLLTS"; uchar flag3=1;
	   lcd_write_zeile(1,"Teststellung:");
	   lcd_write_zeiles(2,"%s: Linke",test1);
	   lcd_write_zeiles(3,"%s: Rechte",test2);
	   lcd_write_zeiles(4,"%s: Obere",test3);
	   taste=taste_abwarten(TASTE_ALLE);
	   //provisorisch: Stellung zum testen:
	   if(taste==TASTE_RECHTS)      {strcpy(&text[16],test1); analyseflag=flag1;}
	   else if(taste==TASTE_LINKS)  {strcpy(&text[16],test2); analyseflag=flag2;}
	   else //if(taste==TASTE_OBEN)
	                                {strcpy(&text[16],test3); analyseflag=flag3;}
	   lcd_clear();
	  }
	 lcd_printf(text);
	 for(i=0;i<64;i++) feld2[i] = (i<16 || i>=64-16) ? 1 : 0;
	 spielbrett.grundstellung960(&text[16]);
	 for(;;)  //warten bis mindestens 1 Figur entfernt
	  {sensorfeld_einlesen_digital(feld1);
	   if(feld_vergleich(feld1,feld2)!=0) break;
	  }
	}
       bool ok=false;
       do {
	sensorfeld_einlesen_digital(feld1);
	if(wahl==1) ok = ist_feld_leer(feld1);
	else        ok = (feld_vergleich(feld1,feld2)==0);
       } while(!ok);
       milliwait(500);//test
       if(wahl!=4)
	{
	 int8_t i,wahl1,wahl2;
	 if(wahl==3 && analyseflag==0) lcd_printf("weiter mit ok-Taste");
	 else lcd_printf("jetzt Figuren setzen, beenden mit ok-Taste");
	 do {
	  sensorfeld_einlesen_digital(feld2);
	  for(i=0;i<64;i++) if(feld2[i]!=feld1[i]) break;
	  if(i<64)
	   {if(feld2[i]==1)
	     {string17 figuren[6]={"Bauer   ","Turm    ","Springer","L\341ufer  ","Dame    ","K\357nig   "};
	      string17 farben[2]={"weiss  ","schwarz"};
	      wahl1=lcd_auswahlfrage("Figur gesetzt:",6,figuren);
	      wahl2=lcd_auswahlfrage("Farbe:",2,farben);
	      uchar figur=0,farbe=0,flag=0;
	      switch(wahl1) {case 1: figur=BAUER; break;  case 2: figur=TURM;  break; case 3: figur=SPRINGER; break;
	                     case 4: figur=LAEUFER; break;  case 5: figur=DAME;  break; case 6: figur=KOENIG; break;
	                    }
	      if(wahl1!=0)
	       {farbe=(wahl2==1)?WEISS:SCHWARZ;
#ifdef SCHACH960
		string17 status[2]={"true","false"};
		if(figur==KOENIG || figur==TURM)
		 {wahl2=lcd_auswahlfrage("MOVFLAG:",2,status);
		  if(wahl2==1) flag=MOVFLAG;
		 }
#else
		if(figur==KOENIG) {if((farbe==WEISS && i==4) || (farbe==SCHWARZ && i==60)) flag=MOVFLAG;}
		if(figur==TURM) {if((farbe==WEISS && (i==0 || i==7)) || (farbe==SCHWARZ && (i==63 || i==56))) flag=MOVFLAG;}
#endif
		spielbrett[i]=figur+farbe+flag;
	       }
	      lcd_printf("n\341chste Figur setzen oder ok-Taste");//provi.
	     }
	    else
	     {spielbrett[i]=LEER;
	      lcd_write4("Figur entfernt");//provi.
	      //TODO
	     }
	    feld1[i]=feld2[i];
	   }
	  ok = (tastegedrueckt&TASTE_MITTE)!=0;
	  if(ok)
	   {
	    warte_losgelassen_mitte(); //warte auf Taste losgelassen
	    i=koenigposition(spielbrett,WEISS);
	    if((spielbrett[i]&(FIGUR+FARBE))!=KOENIG+WEISS)
	     {lcd_printf("Weisser Koenig fehlt noch!"); ok=false;}
	    i=koenigposition(spielbrett,SCHWARZ);
	    if((spielbrett[i]&(FIGUR+FARBE))!=KOENIG+SCHWARZ)
	     {lcd_printf("Schwarzer Koenig fehlt noch!"); ok=false;}
	   }
	 } while(!ok);
	 string17 janein[2]={"yes","no "};
	 wahl1=lcd_auswahlfrage("save in EEPROM ?",2,janein);
	 if(wahl1==1) stellung_im_eeprom_speichern(spielbrett);//TODO
	}
       stellungsflag=1; //gewuenschte Stellung fertig gesetzt
       timer0_stop();
       //Stellung ueberpruefen (sollte immer ok sein), und koenig_startpos ... turm2_startpos setzen:
       if(!check_stellung(spielbrett)) {lcd_print(1,"Fehler:Stellung?",0,0); taste_abwarten(TASTE_MITTE);}
       warte_losgelassen_mitte(); //warte auf Taste losgelassen
      }
     modus=0;
    }
   else
    {
     lcd_write_zeile(1,"Err: unbek.Modus");
     lcd_goto(2,1); lcd_print1("modus=",modus);
     secwait(10); modus=0; //test
    }
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

/** Stockfisch benutzen fuer starke Spielstaerke **/
#ifdef MITUART
void uart_readline(char *text)
{
 uchar c,i;
 for(i=1; i<80 && (c=uart_getc())!='\n';)
  {
   if(c>=' ') {*text++ = c; i++;}
  }
 *text = 0;
}

void uart_startstellung_senden(Brett& brett)
{
 uart_write("startstellung:");
 for(int8_t i=0;i<64;i++)
  {
   uart_putc(' ');
   uart_print(brett[i]);
  }
 uart_putc('\n');
 char text[80];
 uart_readline(text); //Antwort von Stockfisch lesen
}

Zug besterzug_stockfish(Brett& brett,Zug& letzterzug,uchar farbe,int8_t suchtiefe)
{
 Zugliste zugliste;
 Zug zug;
 //lcd_write_zeile(1,"best_stockfish");//test
 alle_moeglichen_zuege(brett,farbe,&zugliste,0);
 if(zugliste.istleer()) {zug.set(0,0); return zug;}
 uart_write("letzterzug: ");
 uart_print(letzterzug.von);
 uart_putc(' ');
 uart_print(letzterzug.nach);
 uart_putc(' ');
 uart_write("Brett:");
 for(int8_t i=0;i<64;i++)
  {
   uart_putc(' ');
   uart_print(brett[i]);
  }
 uart_putc('\n'); //Letzter Zug und Aufstellung an Stockfisch senden
 //lcd_write_zeile(2,"uart gesendet");//test
 char text[80];
 uart_readline(text); //Antwort von Stockfisch lesen
 //lcd_write_zeile(3,"empfang uart");//test
 int von,nach;
 //lcd_write_zeile(3,text);//test
 sscanf(text,"%d %d",&von,&nach);
#ifdef DEBUG2
 lcd_print(1,"stock:%d %d  ",von,nach);
 if(von&0xC0 || nach&0xC0) secwait(2);//test
#endif
 zug.set(von,nach);
 return zug;
}

void stockfish_fertig(Zug &zug,const char *str)
{
 char txt[32];
 sprintf(txt,"%d %d ",zug.von,zug.nach); //letzter Zug bei Spielende noch uebermitteln
 uart_putc('Q'); uart_write(txt); uart_write(str); uart_putc('\n');
 uart_stopp();
}
#endif //MITUART

/******************** Test auf 3 mal gleiche Stellung *********************/
#define MAXSTELL (50*64)
static uchar stellungen[MAXSTELL]; //Platz fuer 50 Stellungen
static int istell=0; //Zeiger auf erste gespeicherte Stellung
static int jstell=0; //Zeiger auf naechste zu speichernde Stellung

#define MAXSTELL2 (4*64)
static uchar stellungen2[MAXSTELL2]; //Platz fuer 4 Stellungen zum ev. Zug zuruecknehmen
static int jstell2=0; //Zeiger auf naechste zu speichernde Stellung

void stellung_speichern_reset()
{
 istell=jstell=0;
}

void stellung_speichern(Brett& brett,bool resetflag)
{
 if(resetflag) {istell=jstell=0;}
 for(int i=0;i<64;i++) {stellungen[jstell+i] = brett[i];}
 if((jstell+=64)==MAXSTELL) {jstell=0; istell=64;}
 else if(jstell==istell)
  {
   if((istell+=64)==MAXSTELL) istell=0;
  }
 if(jstell2==MAXSTELL2)
  {
   jstell2 -= 64;
   for(int j=0; j<=jstell2; j+=64) //Stellungen nachschieben, aelteste verwerfen
    for(int i=j;i<j+64;i++) stellungen2[i]=stellungen2[i+64];
  }
 for(int i=0;i<64;i++) {stellungen2[jstell2++] = brett[i];}
}

int stellung_vergleich(Brett& brett)
{
 int ngleiche=0;
 for(int i=istell; i!=jstell; )
  {
   int k;
   for(k=0;k<64;k++)
    {
     if(stellungen[i+k] != brett[k]) break;
    }
   if(k==64) ngleiche++;
   if((i+=64)==MAXSTELL) i=0;
  }
 return ngleiche;
}

bool fuenfzig_zug_regel()
{
 return (istell>jstell);
}

bool zug_zuruecknehmen()
{
 if(jstell2<128) //noch keine vorherigen Stellungen gespeichert?
  {
   lcd_printf("Zug zuruecknehmen nicht moeglich.");
   secwait(3);//TODO
   zeile4.write("ruecknehm. nicht");
   return false;
  }
 lcdzugliste.zugzuruecknehmen();
 jstell2-=64;
 int j=jstell2-64;
 for(int k=0;k<64;k++)
  {
   spielbrett[k] = stellungen2[j+k];
  }
 if(stockflag && uart_ok) uart_write("Zug zurueckgenommen\n");
 zeile4.write("Zug zurueckgen. ");
 return true;
}
