/* serusb.cc       letzte Aenderungen: 2.1.2017, 26.2.2023
  Serielle Schnittstelle (UART) von Microcontroller ueber USB mit Linux verbinden
  und Stockfish benutzen
*/
#define VERSION "Version 0.2"
/*
History:
31.12.2016	 Erstellung aus ser.cc (RP)
26.2.2023  0.2	 Aufforderung zum USB einstecken zugefuegt

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <termios.h>
#include <unistd.h>

void sonderzeichen_ersetzen(char *zeile); //fuer Modus 6

/*************************** kleinkram ***************************/
bool getline(FILE *fp,char *s,int lim)
{		// liest eine Textzeile oder maximal lim Zeichen
		// und ersetzt den Zeilentrenner durch 0
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
	*s++ = c;
 *s=0;
 return (c!=EOF); // TRUE wenn erfolgreich, FALSE wenn Fileende
}

int ascitodigit(int c)
{		// wandelt ein ASCI-Zeichen in eine 1-stellige Hex-Zahl
 if(!isxdigit(c)) return 0;
 if(isupper(c))      c = c-'A'+10;
 else if(islower(c)) c = c-'a'+10;
 else c -= '0';
 return c;
}

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
  }
}

/*********************** Serielle Schnittstelle ***********************/
//static char sername[80]="/dev/ttyS0";
static char sername[80]="/dev/ttyUSB0";
//static char sername[80]="/dev/tty.serial-0001";
static int baudrate=19200;
static int fdr,fdw;

static int baudcodliste[]={50,B50, 75,B75, 110,B110, 134,B134, 150,B150,
   300,B300, 600,B600, 1200,B1200, 1800,B1800, 2400,B2400, 4800,B4800,
   9600,B9600, 19200,B19200, 38400,B38400,
   57600,B57600,
   115200,B115200,
   230400,B230400,
   460800,B460800, 500000,B500000, 576000,B576000,
   921600,B921600, 1000000,B1000000, 1152000,B1152000, 1500000,B1500000,
   2000000,B2000000, 2500000,B2500000, 3000000,B3000000, 3500000,B3500000,
   4000000,B4000000,
   0,0};

int ibaud(int x) //Baudcode in Baudrate umrechnen
{
 int *co,a;
 for(co=baudcodliste;(a= *co++)!=0;co++)
    if(x== *co) return a;
 printf("Fehler: unbekannter Baudratecode %d\n",x);
 return 0;
}

int baudi(int x) //Baudrate in Baudcode umrechnen
{
 int *co,b,min=0;
 for(co=baudcodliste;(b= *co++)>0;co++)
     if(x==b) return *co;
     else if(x>b) {min= *co;}
 printf("Warnung: unbekannte Baudrate %d, auf %d abgerundet\n",x,min);
 return min;
}

void noecho(int fd)
{
 struct termios buf;
 if(tcgetattr(fd,&buf)<0)
   {printf("tcgetattr() Error\n"); return;}
 buf.c_lflag &= ~ECHO;
 if(baudrate!=0)
     {cfsetispeed(&buf,baudi(baudrate));
      cfsetospeed(&buf,baudi(baudrate));
     }
 if(tcsetattr(fd,TCSAFLUSH,&buf)<0)
   {printf("tcsetattr() Error\n"); return;}
 int a=cfgetispeed(&buf),b=cfgetospeed(&buf);
 if(!argflag['Q']) printf("Baudraten: %d %d\n",ibaud(a),ibaud(b));
}

void seriell_open()
{
 fdr=open(sername,O_RDWR);
 for(int i=0,imax=1000; i<imax && fdr<0;i++)
  {
   if(i<3) printf("bitte %s einstecken\n",sername);
   sleep(1);
   fdr=open(sername,O_RDWR);
  }
 if(fdr<0) {printf("Fehler: fdr = %d (Zugriffsberechtigung %s ?)\n",fdr,sername); exit(0);}
	   //wenn dieser Fehler kommt sollte /dev/ttyS0 ueberprueft werden
	   //Zugriffsberechtigung freigeben: chmod a+rw /dev/ttyS0
	   //oder vielleicht /dev/ttyS1
 noecho(fdr);
 fdw=fdr;
}

void seriell_close()
{
 close(fdr);
}

void seriell_readline(char *str,int max)
{
 int i;
 char *s,c;
 for(i=1,s=str;i<max;)
   {
    int n=read(fdr,&c,1);
    if(n<=0) printf("Fehler: read() --> %d\n",n);//test
    if(c=='\n') break;
    if(c==0) printf("\\0");//test
    else if(c=='\r') ;//CR ignorieren, auskommentieren wenn CR mit gespeichert werden soll
    else if(c<' ') printf("'\\%d'",c);//test
    else
      {*s++ = c; i++;}
   }
 /*
 if(s!=str && s[-1] == '\r') //nicht noetig wenn CR ignoriert wurde
   {--s;
    printf("CR empfangen\n");//test
   }
 */
 *s=0;
}

void seriell_writeline(const char *str)
{
 int n=strlen(str);
 char *wbuf=new char[n+3];
 if(argflag['C']) sprintf(wbuf,"%s\r\n",str);
 else sprintf(wbuf,"%s\n",str);
 n=write(fdw,wbuf,(argflag['C'])?n+2:n+1);
 if(n<=0) printf("Fehler: write() --> %d\n",n);//test
 delete wbuf;
}

/************************* Schach-Engine ******************************/
#include "schach-engine.h"
static int8_t spielsuchtiefe=10;
static bool geschlagenflag=false;

//Vordeklarationen:
void stockfish_init();
void fenbrett_aktualisieren(Brett& brett,Zug& letzterzug);
void stockfish_beenden();

/************************* Kommunikation mit Stockfish ******************************/
/*
 Zur Kommunikation mit Stockfish wird das UCI (Universal Chess Interface) verwendet.
 Das UCI-Protokoll ist z.B. unter www.shredderchess.com zu finden.
*/
/*
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
*/

const int MAXFEN=200;
const int MAXZ=240;
static int fd1=0,fd2=0;
static char zeile[MAXZ];
const char *myfifo1="myfifo1", *myfifo2="myfifo2";

int read1(int fd) //lies 1 Zeichen von einer Pipe
{
 char c;
 read(fd,&c,1);
 return c&0xFF;
}

bool readp(int fd,char *s,int lim)
{		// lies von einer Pipe eine Zeile
		// und ersetze den Zeilentrenner durch 0
 int c=0;
 while(--lim && (c=read1(fd))!='\n')
	*s++ = c;
 *s='\0';
 return (c!=EOF); // TRUE wenn erfolgreich, FALSE wenn Fileende
}

void printp(int fd,const char *s)
{
 write(fd,s,strlen(s));
}

void printp1(int fd,const char *s,int p1)
{
 char str[MAXZ];
 sprintf(str,s,p1);
 write(fd,str,strlen(str));
}

void printp1s(int fd,const char *s,const char *p1)
{
 char str[MAXZ];
 sprintf(str,s,p1);
 write(fd,str,strlen(str));
}

void fifos_init()
{
 mkfifo(myfifo1, 0666);
 mkfifo(myfifo2, 0666);
 int pid;
 if((pid=fork())!=0)
  {
   system("stockfish <myfifo1 >myfifo2");
   exit(0); //Fork beenden
  }
 fd1=open(myfifo1, O_WRONLY);
 fd2=open(myfifo2, O_RDONLY);
 if(argflag['V']) printf("fifos geoeffnet: fd1=%d fd2=%d\n",fd1,fd2);
 printp(fd1,"uci\n");
 if(argflag['V']) printf("uci an fd1 gesendet\n");
 bool ok=false;
 while(readp(fd2,zeile,200))
  {
   if(strncmp(zeile,"uciok",5)==0) {ok=true; break;}
   //fprintf(stderr,"Stockfish: %s\n",zeile);//test
  }
 if(!ok)
  {
   fprintf(stderr,"Error: kein uciok empfangen.\n");
   stockfish_beenden();
  }
}

class Zugstock
{
public:
 uchar von,nach;
 uchar promotfig;
 void set(uchar from,uchar to) {von=from; nach=to; promotfig=0;}
 const char *print();
};

const char *Zugstock::print()
{
 static char zugtext[6];
 zugtext[0] = 'A'+(von%8);  zugtext[1] = '1'+(von/8);
 //zugtext[2] = '-'; 
 zugtext[2] = 'A'+(nach%8); zugtext[3] = '1'+(nach/8);
 char c;
 switch(promotfig) //fuer Umwandlung von Bauer
  {case DAME:     c='q'; break; //Queen
   case SPRINGER: c='n'; break; //kNight
   case TURM:     c='r'; break; //Rook
   case LAEUFER:  c='b'; break; //Bishop
   default: c=0;
  }
 zugtext[4] = c;
 zugtext[5] = 0;
 return zugtext;
}

class Fen
{
public:
 char fen[MAXFEN];
 int halbzug,zugnummer;
 void grundstellung();
 //void fen2brett();
 void brett2fen(Brett& brett,Zug& letzterzug);
 //void print() {fen2brett(); brett.print();}
 Zugstock besterzug(int farbe,int suchtiefe);
 Zugstock spielerzug(int farbe);
};
static Fen fenbrett;

void Fen::grundstellung()
{
 strcpy(fen,"rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1");
 halbzug=0,zugnummer=1;
 printp(fd1,"position startpos\n");
}

char fig_eng_abk(uchar u)
{
 char c = u&FIGUR;
 switch(c)
  {case DAME:     c='q'; break; //Queen
   case SPRINGER: c='n'; break; //kNight
   case TURM:     c='r'; break; //Rook
   case LAEUFER:  c='b'; break; //Bishop
   case KOENIG:   c='k'; break; //King
   case BAUER:    c='p'; break; //Pawn
   default: c='1';
  }
 if((u&FARBE)==WEISS) c -= 'a'-'A';
 return c;
}

void Fen::brett2fen(Brett& brett,Zug& letzterzug)
{
 int i,j,k,zahl;
 uchar c;
 char *f=fen;
 uchar farbeamzug;
 if(letzterzug.nach==letzterzug.von) farbeamzug=WEISS;
 else
  {
   uchar fig=(brett[letzterzug.nach]&FIGUR);
   farbeamzug = WEISS+SCHWARZ-(brett[letzterzug.nach]&FARBE);
   if(fig==BAUER || geschlagenflag) halbzug=0; //provi.
   else halbzug++;
   if(farbeamzug==WEISS) zugnummer++;
  }
 // Beispiel Grundstellung:
// "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1"
 for(j=64-8;j>=0;)
  {
   for(zahl=0,k=0;k<8;k++)
    {
     c=brett[j++];
     if(c==LEER) zahl++;
     else
      {if(zahl!=0) {*f++ = zahl+'0'; zahl=0;}
       *f++ = fig_eng_abk(c);
      }
    }
   if(zahl!=0) *f++ = zahl+'0';
   if((j-=16)>=0) *f++ = '/';
  }
 *f++ = ' ';
 *f++ = (farbeamzug==WEISS)?'w':'b';
 *f++ = ' ';
 bool flag=false;
 if(brett[4]==KOENIG+WEISS+MOVFLAG)
  {
   if(brett[7]==TURM+WEISS+MOVFLAG) {*f++ = 'K'; flag=true;}
   if(brett[0]==TURM+WEISS+MOVFLAG) {*f++ = 'Q'; flag=true;}
  }
 if(brett[60]==KOENIG+SCHWARZ+MOVFLAG)
  {
   if(brett[63]==TURM+SCHWARZ+MOVFLAG) {*f++ = 'k'; flag=true;}
   if(brett[56]==TURM+SCHWARZ+MOVFLAG) {*f++ = 'q'; flag=true;}
  }
 if(flag==false) *f++ = '-';
 *f++ = ' ';
 flag=false;
 c='-';
 for(j=24,i=0;i<8;j++,i++)
  if((brett[j]&(FIGUR|ENPASSFLAG))==(BAUER|ENPASSFLAG)) {*f++ = 'a'+i%8; *f++ = '3'; flag=true;}
 if(flag==false)
  for(j=32,i=0;i<8;j++,i++)
   if((brett[j]&(FIGUR|ENPASSFLAG))==(BAUER|ENPASSFLAG)) {*f++ = 'a'+i%8; *f++ = '6'; flag=true;}
 if(flag==false) *f++ = '-';
 *f++ = ' ';
 sprintf(f,"%d %d",halbzug,zugnummer);
 if(!argflag['Q']) printf("fen='%s'\n",fen);
}

Zugstock Fen::besterzug(int farbe,int suchtiefe)
{
 Zugstock zug;
 printp1s(fd1,"position fen %s\n",fen);
 printp1(fd1,"go depth %d\n",suchtiefe);
 if(argflag['V']) printf("go depth %d\n",suchtiefe);//test
 while(readp(fd2,zeile,200))
    {
     //fprintf(stderr,"Stockfish: %s\n",zeile);//test
     if(strncmp(zeile,"bestmove",8)==0)
      {
       if(argflag['V']) printf("Bester Zug: %s\n",&zeile[9]);//test
       break;
      }
    }
 const char *p = &zeile[9];
 int c1= *p++;
 int c2= *p++;
 if(c1>='a' && c1<='h') c1 -= 'a'-'A';
 if(c1<'A' || c1>'H') printf("Fehler c1='%c'\n",c1);//test
 if(c2<'1' || c2>'8') printf("Fehler c2='%c'\n",c2);//test
 zug.von = c1-'A'+(c2-'1')*8;
 c1= *p++;
 c2= *p++;
 if(c1>='a' && c1<='h') c1 -= 'a'-'A';
 if(c1<'A' || c1>'H') printf("Fehler c1='%c'\n",c1);//test
 if(c2<'1' || c2>'8') printf("Fehler c2='%c'\n",c2);//test
 zug.nach = c1-'A'+(c2-'1')*8;
 if((c1= *p++)>='A')
  {uchar c=0;
   if(c1>='A' && c1<='Z') c1 += 'a'-'A';
   switch(c1)
    {case 'q': c=DAME; break; //Queen
     case 'n': c=SPRINGER; break; //kNight
     case 'r': c=TURM; break; //Rook
     case 'b': c=LAEUFER; break; //Bishop
    }
   zug.promotfig=c;
  }
 else zug.promotfig=0;
 return zug;
}

void stockfish_init()
{
 fenbrett.grundstellung();
 fifos_init();
}

Zug besterzug_stockfish(Brett& brett,Zug& letzterzug,uchar farbe,int suchtiefe)
{
 Zugliste zugliste;
 Zug zug;
 alle_moeglichen_zuege(brett,farbe,&zugliste,0);
 if(zugliste.istleer()) {zug.set(0,0); return zug;}
 fenbrett.brett2fen(brett,letzterzug);
 Zugstock zugst=fenbrett.besterzug(farbe,suchtiefe);
 zug.set(zugst.von,zugst.nach);
 if(zugst.promotfig!=0) zug.setpromo(zugst.promotfig);
 return zug;
}

void fenbrett_aktualisieren(Brett& brett,Zug& letzterzug)
{
 fenbrett.brett2fen(brett,letzterzug);
}

void stockfish_beenden()
{
 printp(fd1,"quit\n");
 close(fd1);
 close(fd2);
}

/************************* Hauptprogramm ******************************/

Zug besterzug_stock(Brett& brett,Zug& letzterzug)
{
 uchar farbe;
 if(letzterzug.von==letzterzug.nach) farbe=WEISS; //erster Zug ist mit Weiss
 else farbe = WEISS+SCHWARZ-(brett[letzterzug.nach]&FARBE); //sonst andere Farbe als vorheriger Zug
 return besterzug_stockfish(brett,letzterzug,farbe,spielsuchtiefe);
}

char *mysscanf(char *s,const char *cstr,int *p1)
{
 int n=0;
 while(*s==' ') s++;
 while(isdigit(*s)) {n=10*n+(*s)-'0'; s++;}
 *p1 = n;
 return s;
}

int main(int argc,char *argv[])
{
 char name[80],zeile[80],rzeile[400];
 int i,j,wahl=1,c=0;
 name[0]=0;
 for(j=0,i=1;i<argc;i++)
	{if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
	 else	{if(++j==1) strcpy(name,argv[i]);
	         else if(j==2) sscanf(argv[i],"%d",&baudrate);
	}	}
 if(argflag['?'] || j>MAXARG)
	{printf("serusb  %s\n",VERSION);
	 printf("Anwendung: serusb [-flags] [/dev/ttyUSB1] [Baudrate]\n");
	 printf("      voreingestellt ist %s %d\n",sername,baudrate);
	 printf("   flags: c=CR vor LF senden\n");
	 printf("          t=Tests mit Auswahl\n");
	 printf("          v=Verbose, also mit Testausdrucken\n");
	 printf("          q=quiet, also ganz ohne Testausdrucke\n");
	 exit(0);
	}
 if(j>=1)
     {if(isdigit(*name)) sprintf(sername,"/dev/ttyS%c",*name);
      else sprintf(sername,"%s",name);
     }
 if(argflag['V']) printf("'%s' oeffnen\n",sername);
 seriell_open();
 stockfish_init();
 for(;;)
 {
  if(argflag['T'])
   {
    printf("Modus-Auswahl:\n");
    printf("0  Programm verlassen\n");
    printf("1  Zeilenweise schreiben/lesen\n");
    printf("2  Zeilenweise lesen/schreiben (auf A automatisch mit B antworten)\n");
    printf("3  Einzelzeichen schreiben/lesen\n");
    printf("4  Einzelzeichen lesen/schreiben\n");
    printf("5  je 100000 Zeichen senden\n");
    printf("6  Nur Zeilen senden. Nicht-ASCII-Zeichen mit 0x12 senden\n");
    printf("Auswahl:"); getline(stdin,zeile,80); sscanf(zeile,"%d",&wahl);
    if(wahl<1) break;
   }
  else //sonst: es werden mit dem Schachcomputer Daten ausgetauscht
   {
    wahl=10;
    seriell_readline(rzeile,400);
    if(argflag['V']) printf("empfangen: '%s'\n",rzeile);
    if(strcmp(rzeile,"A")==0) //Einschaltkommando vom Schachcomputer empfangen?
     {
      seriell_writeline("B"); //Antwort auf Einschaltkommando 
      if(argflag['V']) printf("mit 'B' geantwortet\n");
     }
    else if(strcmp(rzeile,"Q")==0) //Beenden-Kommando vom Schachcomputer empfangen?
     {
      //if(argflag['V']) printf("Programm wird regulaer beendet.\n");
      //break;
      if(!argflag['Q']) printf("Partie beendet.\n");//test
     }
    else if(strncmp(rzeile,"suchtiefe:",10)==0) //setzen der Suchtiefe
     {
      int n;
      sscanf(&rzeile[10],"%d",&n); spielsuchtiefe=n;
      if(argflag['V']) printf("spielsuchtiefe auf %d gesetzt\n",spielsuchtiefe);
     }
    else if(strncmp(rzeile,"letzterzug:",11)==0) //fraegt Schachcomputer nach dem besten Zug mit Stockfish?
     {
      Zug letzterzug;
      int von,nach,n;
      sscanf(&rzeile[11],"%d %d",&von,&nach);
      if(argflag['V']) printf("letzter Zug: von=%d nach=%d\n",von,nach);//test
      letzterzug.set(von,nach);
      char *s;
      Brett brett;
      for(s= &rzeile[11]; *s!=':' && *s!=0; s++) {}//ueberlesen von "Brett:"
      if(*s==0) fprintf(stderr,"Error: fehlende Brett-Informationen\n");
      else
       {
	s++;
	if(argflag['V']) printf("Brett einlesen:\n");//test
	for(i=0;i<64;i++) {s=mysscanf(s,"%d",&n); brett[i]=n;}
	if(argflag['V'])
	 {for(i=0;i<64;i++) printf("%d ",brett[i]);//test
	  printf("\n");
	 }
	Zug zug=besterzug_stock(brett,letzterzug);
	sprintf(zeile,"%d %d",zug.von,zug.nach);
	if(argflag['V']) printf("seriell_writeline(\"%s\")\n",zeile);
	seriell_writeline(zeile);
       }
     }
    else fprintf(stderr,"Error: unbekanntes Kommando empfangen: '%s'\n",rzeile);
   }
  if(wahl==1)
    {for(;;)
	{printf("(q=Abbruch) senden:"); getline(stdin,zeile,80);
	 if(*zeile=='q') break;
	 if(*zeile!=0) seriell_writeline(zeile);
	 do {
	  seriell_readline(rzeile,400);
	  printf("  zeile empfangen:'%s'\n",rzeile);
	 }while(*rzeile==0);
	}
    }
  else if(wahl==2)
    {for(;;)
	{do {
	  seriell_readline(rzeile,400);
	  if(*rzeile==0) printf("    Leerzeile empfangen\n");
	  else printf("   Zeile empfangen:'%s'\n",rzeile);
	 }while(*rzeile==0);
	 if(strcmp(rzeile,"A")==0)
	  {
	   seriell_writeline("B");
	   printf("'B' automatisch gesendet\n");
	  }
	 else
	  {
	   printf("(q=Abbruch) senden:"); getline(stdin,zeile,80);
	   if(*zeile=='q') break;
	   if(*zeile!=0) seriell_writeline(zeile);
	  }
	}
    }
  else if(wahl==3)
    {for(;;)
	{printf("(q=Abbruch) senden:"); c=getc(stdin);
	 if(c=='q') break;
	 write(fdw,&c,1);
	 read(fdr,&c,1);
	 printf("Zeichen  empfangen: 0x%02X\n",c);
	}
    }
  else if(wahl==4)
    {for(;;)
	{read(fdr,&c,1);
	 printf("Zeichen  empfangen: 0x%02X\n",c);
	 printf("(q=Abbruch) senden:"); c=getc(stdin);
	 if(c=='q') break;
	 write(fdw,&c,1);
	}
    }
  else if(wahl==5)
    {for(;;)
	{printf("(q=Abbruch) senden:"); c=getc(stdin);
	 if(c=='q') break;
	 for(i=0;i<100000;i++) write(fdw,&c,1);
	}
    }
  else if(wahl==6)
    {for(;;)
	{printf("senden:"); getline(stdin,zeile,80);
	 sonderzeichen_ersetzen(zeile);
	 seriell_writeline(zeile);
	}
    }
  if(wahl<=6) getc(stdin);
 }
 if(argflag['V']) printf("stockfish_beenden();\n");//test
 stockfish_beenden();
 if(argflag['V']) printf("seriell_close();\n");//test
 seriell_close();
 if(argflag['V']) printf("return 0;\n");//test
 return 0;
}/* ende von main */

void sonderzeichen_ersetzen(char *zeile)
{
 //Sonderzeichen der Form 0x12 als entsprechendes Zeichen setzen
 char *s;
 while(*zeile!=0)
  {
   if(zeile[0]=='0' && zeile[1]=='x' && isxdigit(zeile[2]) && isxdigit(zeile[3]))
    {
     zeile[0] = (ascitodigit(zeile[2])<<4) + ascitodigit(zeile[3]);
     zeile++;
     for(s=zeile;s[3]!=0;s++) {s[0]=s[3];} //restliche Zeichen nachruecken
     s[0]=0;
    }
   else
    zeile++;
  }
}
