/* robot.cc       letzte Aenderungen: 1.4.2023, 5.4.2024, 25.9.2024
Programm um Schrittmotoren des Schachroboters anzusteuern
*/
#define VERSION "0.6"
#define EEVERSION 50 //Version der EEPROM-Belegung
/*
 Prozessor: Atmega1284P, 20MHz Quarz
 Schaltung: schaltung/robot.cc

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 1.1.2023       Erstellung
23.2.2023  0.2  einstellbare Brettlage
 1.3.2023  0.3  Abhebhoehe abhaengig von aktueller Figur und Art des Zuges
 2.3.2023  0.4  Diagonales Fahren auch in andere Richtungen als 45 Grad
                Alte Testoptionen entfernt
10.3.2023  0.5  EEPROM verwenden fuer A1XPOS, A1YPOS, ...
1.4.2023        weiterauf[] nochmals angepasst
30.3.2023  0.6  Anpassungen fuer Schach960
25.9.2024       weiterauf[] fuer Springer von 40 auf 45 erhoeht
*/

#define F_CPU 20000000UL
#define XSENSORPOS 0  //auf 0 lassen und A1XPOS anpassen
#define YSENSORPOS 0  //auf 0 lassen und A1YPOS anpassen
#define ZSENSORPOS 20 //Hoehe des Z-Sensors ueber dem Brett
#define SCHNELLER 4  //Schrittmotoren schneller laufen lassen (2=langsam, 4=normal, 8=schnell)
#define GREIFER_LANGSAMER  8  //1=sehr schnell, 2=schnell, 4=normal, 8=langsam, 40=sehr langsam
#define ALFAKORR  //bei leicht schraegem Brett um Winkel alfa korrigieren
//#define MOT3_NIE_AUS  //Motor fuer z-Richtung nie ganz ausschalten
//#define AUTOCALIB 10  //nach jedem sovielten Zug neu calibrieren
#define SCHACH960

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdio.h>
//#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include <util/delay.h>
#include <avr/eeprom.h>

#define KIPPFAKTOR 80L //Korrektur fuer Neigung wenn Arm weit ausgefahren
#define ZEILE1KORR  2  //Korrektur von y in mm wegen leichtem Kippen nach hinten auf erster Zeile
#define YZEILE2 (a1ypos_steps+STEPS_PRO_FELD) //Kippkorrektur erst ab Zeile2 anwenden

// nur eine der folgenden 2 Zeilen nicht auskommentieren:
//#define DIAGONAL_MOVES //nur 45-Grad-Diagonalen unterstuetzt
#define DIAGONAL_MOVES3 //beliebige Richtungen

//#define DIAGONALTEST
//#define DIAGONALTEST2
//#define DIAGONALTEST3
//#define KIPPTEST  //D1-D8 und zurueck zum Kippfaktor testen
//#define CALIBTEST

#define DEBUG  //zur Fehlersuche
//#define DEBUG2 //Schach960-Fehlersuche
//#define UARTTEST  //UART testen

/**** Vordeklarationen ****/
#define uchar uint8_t
#define uint uint16_t
#define ulong uint32_t
void schrittmotoren_steps();
void werte_calibrieren(const char *zeile);

/**** Zeitbehandlung ****/
// von nimm.cc uebernommen
static volatile uint millisec=0;
static volatile uchar sec=0,min=0,stunden=0;
static volatile uchar tage=25,monat=2; //aktuelles Datum
static volatile uint jahr=2023;
static char monatstage[13]={31,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=5; //0=Montag ... 6=Sonntag
//static volatile char sommerzeit=1; //0=Winterzeit 1=Sommerzeit

void schaltjahrtest()
{
 if(jahr%4==0 && (jahr%100!=0 || jahr%400==0)) monatstage[2]=29;
 else monatstage[2]=28;
}
/*
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
void schaltjahr_und_wochentagberechnen()
{
 schaltjahrtest();
 wochentagberechnen();
}
*/

// fuer genaue Uhr: Quarz moeglichst genau ausmessen und hier eintragen:
const ulong Hz=20000000; //exakt 20MHz

//#define Rest1 (Hz%1000) //jede Sekunde zu korrigierender Fehler
#define NormalerCompwert (Hz/(1000*SCHNELLER)-1)

void timer1_init()
{
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
 set_sleep_mode(SLEEP_MODE_IDLE);
 sleep_enable(); //Schlaf-Modus waehlen, aber noch nicht schlafen
}

ISR(TIMER1_COMPA_vect)
{
 schrittmotoren_steps();
 static uchar j=0;
 if(++j==SCHNELLER)  j=0;
 else return;
 uint ms=millisec;
#ifdef Rest1
 uchar mehr;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
#else
 ++ms;
#endif
 sei();
 if(ms==1000)
  {ms=0;
#ifdef Rest1
   mehr=0;
#endif
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
       /*
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
       */
      }
    }
  }
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
#ifdef Rest1
 OCR1A=NormalerCompwert+mehr;
#endif
 //tastaturcheck();
}

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) {sleep_mode();}//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
  {sleep_mode();}//auf n. Millisekunde warten
}

void secwait(uchar n) //n Sekunden warten
{
 while(n!=0) {milliwait(1000); --n;}
}

void microwait(uchar n) //mindestens n Microsekunden warten
{
 while(--n!=0) _delay_us(1);
}

/************************** Serielle Schnittstelle **************************/
#define TXUNDRX  //senden und empfangen, auskommentieren wenn nur empfangen
#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 20000000"
#define F_CPU 20000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 9600UL      // Baudrate

// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart1_init()
{
 UBRR1 = UBRR_VAL;
#ifdef TXUNDRX
 UCSR1B = (1<<RXEN1)|(1<<TXEN1); // UART TX und RX einschalten
#else
 UCSR1B = (1<<RXEN1); // nur UART RX einschalten
#endif
 UCSR1C = (3<<UCSZ10); // Asynchron 8N1
 // UCSR1C = (3<<UCSZ10)|(1<<USBS1); // Asynchron 8N2
}

#ifdef TXUNDRX
void uart_putc(unsigned char c)
{
 while (!(UCSR1A & (1<<UDRE1))) {milliwait(1);}// warten bis Senden moeglich
 UDR1 = c;                      // sende Zeichen
 milliwait(3);//test
}

void uart_print(const char *text)
{
 for(int i=0; text[i] != 0; i++)
  {
   uart_putc(text[i]);
  }
}

void uart_printf(const char *cstr,int p1,int p2=0)
{
 char text[32];
 sprintf(text,cstr,p1,p2);
 for(int i=0; text[i] != 0; i++)
  {
   uart_putc(text[i]);
  }
}
#endif

unsigned char uart_getc(void)
{
 while(!(UCSR1A & (1<<RXC1))) {milliwait(1);}// warten bis Zeichen verfuegbar
 return UDR1; // Zeichen aus UDR an Aufrufer zurueckgeben
}

bool uart_check()
{
 return (UCSR1A & (1<<RXC1)) ;//true wenn Zeichen verfuegbar
}

char* uart_zeile_empfangen()
{
 static char zeile[32];
 uint8_t i;
 char c=' ';
 for(i=0;i<31 && c!=0;)
  {
   c=uart_getc();
   if(c=='\r') continue; //CR ignorieren
   if(c=='\n') c=0; //mit LF beenden
   zeile[i++]=c;
  }
 zeile[i]=0;
 return zeile;
}

/***************************** Port-Belegungen *******************************/
#define LED_PORT PORTB
#define LED_DDR  DDRB
#define LED0     PB5
#define LED1     PB4
#define LED0_ON  LED_PORT |= (1<<LED0)
#define LED0_OFF LED_PORT &= ~(1<<LED0)
#define LED1_ON  LED_PORT |= (1<<LED1)
#define LED1_OFF LED_PORT &= ~(1<<LED1)

#define MOT1_PORT PORTA
#define MOT1_DDR  DDRA
#define MOT1_PIN  PINA
#define MOT1_SEL  4
#define MOT1_DIR  5
#define MOT1_STEP 6
#define MOT1_SENS 7
#define MOT1_SENS2 0 //bisher unbenutzt
#define MOT1_SENS3 1 //bisher unbenutzt
// 0-3 fuer ADC0-3 falls noch A/D-Wandlung gebraucht

#define MOT2_PORT PORTB
#define MOT2_DDR  DDRB
#define MOT2_PIN  PINB
#define MOT2_SEL  0
#define MOT2_DIR  1
#define MOT2_STEP 2
#define MOT2_SENS 3
// B5-B6 sind fuer Programmer belegt

#define MOT3_PORT PORTC
#define MOT3_DDR  DDRC
#define MOT3_PIN  PINC
#define MOT3_SEL  2
#define MOT3_DIR  3
#define MOT3_STEP 4
#define MOT3_SENS 5
// C0-C1  SCL und SDA falls noch I2C gebraucht wir

#define MOT4_PORT PORTD
#define MOT4_DDR  DDRD
#define MOT4_PIN  PIND
#define MOT4_SEL  4
#define MOT4_DIR  5
#define MOT4_STEP 6
#define MOT4_SENS 7
// D0-D1 reserviert fuer UART0
// D2-D3 verwendet fuer UART1  //koennten ev. auch fuer INT0 INT1 Interrupts verwendet werden

#define TAST_PORT PORTC
#define TASTE1    6 //PCINT22
#define TASTE2    7 //PCINT23

void motor_onoff(uchar nr,bool on)
{
 uchar flags = (nr==0 || nr>4) ? 0x0F : (1<<(nr-1));
 if(flags&0x08)
  {if(on) MOT4_PORT &= ~(1<<MOT4_SEL);
   else   MOT4_PORT |= (1<<MOT4_SEL);
  }
 if(flags&0x04)
  {if(on) MOT3_PORT &= ~(1<<MOT3_SEL);
#ifndef MOT3_NIE_AUS
   else   MOT3_PORT |= (1<<MOT3_SEL);
#endif
  }
 if(flags&0x02)
  {if(on) MOT2_PORT &= ~(1<<MOT2_SEL);
   else   MOT2_PORT |= (1<<MOT2_SEL);
  }
 if(flags&1)
  {if(on) MOT1_PORT &= ~(1<<MOT1_SEL);
   else   MOT1_PORT |= (1<<MOT1_SEL);
  }
}

/********************************** PCINT ************************************/
volatile uint8_t got_pcint=0;
volatile bool notstopp=false;

ISR(PCINT2_vect)
{
 if((PINC&(1<<6))==0)      {got_pcint = 22; notstopp=true;} //Taste1 gedrueckt
 else if((PINC&(1<<7))==0) {got_pcint = 23; notstopp=true;} //Taste2 gedrueckt
 //TODO: vorlaeufig beide Tasten fuer Notstopp verwendet
}

void pcint_init()
{
 TAST_PORT |= (1<<TASTE1); //Pullup setzen
 TAST_PORT |= (1<<TASTE2); //Pullup setzen
 PCMSK2 = (1<<PCINT22)|(1<<PCINT23); //Auswahl der aktiven PC-Interrupts
 PCICR = (1<<PCIE2); //PCINT23 bis PCINT16 ermoeglichen
}

/************************* Variablen und Konstanten **************************/
static int xposition=0, yposition=0, zposition=0; //aktuelle Position in Steps
static volatile int steps1=0, steps2=0, steps3=0, steps4=0; //zu fahrende Schritte
#define VMAX 10000L       //maximale Geschwindigkeit beim Fahren auf Achsen
#define VMAX2 (VMAX*2) //groessere Geschw. der Motoren wenn gleich schnell fahren in Diagonalen
static volatile uint vmot1=VMAX, vmot2=VMAX, vmot3=VMAX; //Schrittmotor-Geschwindigkeiten

#define STEPS_PRO_METER 40000L
#define FELDGROESSE_IN_MM 38
#define STEPS_PRO_FELD (FELDGROESSE_IN_MM*STEPS_PRO_METER/1000)
#define A1XPOS_MM 67 //x-Position von Mitte des A1 Feldes
#define A1YPOS_MM 11 //y-Position von Mitte des A1 Feldes
#define A1XPOS_STEPS (STEPS_PRO_METER*A1XPOS_MM/1000)
#define A1YPOS_STEPS (STEPS_PRO_METER*A1YPOS_MM/1000)
#define CM  (STEPS_PRO_METER/100)
#define MM  (STEPS_PRO_METER/1000)
#define CM9 (9*CM) //Greifer ueber hoechsten Figuren
#define CM4 (4*CM) //Greifer nur minimal hochheben

/************************************ EEPROM ******************************************/
uint16_t a1xpos_steps=A1XPOS_STEPS, a1ypos_steps=A1YPOS_STEPS;
uint16_t h8xpos=0, h8ypos=0; //in main() initialisieren
int16_t zsensorpos=ZSENSORPOS;
uint16_t kippfaktor=KIPPFAKTOR;
float sinusalfa=0.0;

//mit aktueller Compilerversion stehen die Daten in umgekehrter Reihenfolge im EEPROM!
float eeSinusalfa EEMEM;
uint16_t eeZsensorpos EEMEM;
uint16_t eeKIPPFAKTOR EEMEM;
uint16_t eeA1YPOS_STEPS EEMEM;
uint16_t eeA1XPOS_STEPS EEMEM;
uint eeversion EEMEM;

void eeversion_setzen()
{
 if(eeprom_read_word(&eeversion)!=EEVERSION)
   eeprom_write_word(&eeversion, EEVERSION);
}

void a1xypos_von_eeprom_einlesen()
{
 uint version=eeprom_read_word(&eeversion);
 if(version==EEVERSION)
  {
   a1xpos_steps=eeprom_read_word(&eeA1XPOS_STEPS);
   a1ypos_steps=eeprom_read_word(&eeA1YPOS_STEPS);
  }
}

void kippfaktor_von_eeprom_einlesen()
{
 uint version=eeprom_read_word(&eeversion);
 if(version==EEVERSION)
  {
   kippfaktor=eeprom_read_word(&eeKIPPFAKTOR);
   if(kippfaktor>400) kippfaktor=KIPPFAKTOR; //bei unrealistischem Wert auf Default setzen
  }
}

void zsensorpos_von_eeprom_einlesen()
{
 uint version=eeprom_read_word(&eeversion);
 if(version==EEVERSION)
  {
   zsensorpos=eeprom_read_word(&eeZsensorpos);
   if(zsensorpos>100) zsensorpos=ZSENSORPOS; //bei unrealistischem Wert auf Default setzen
  }
}

void sinusalfa_von_eeprom_einlesen()
{
 uint version=eeprom_read_word(&eeversion);
 if(version==EEVERSION)
  {
   //eeprom_read_block(&sinusalfa,&eeSinusalfa,sizeof(float));
   sinusalfa=eeprom_read_float(&eeSinusalfa);
   if(sinusalfa>0.1 || sinusalfa < -0.1) sinusalfa=0;//bei unrealistischem Wert auf 0 setzen
  }
}

void werte_in_eeprom_speichern()
{
 eeversion_setzen();
 if(eeprom_read_word(&eeA1XPOS_STEPS)!=a1xpos_steps)
   eeprom_write_word(&eeA1XPOS_STEPS,a1xpos_steps);
 if(eeprom_read_word(&eeA1YPOS_STEPS)!=a1ypos_steps)
   eeprom_write_word(&eeA1YPOS_STEPS,a1ypos_steps);
 if(eeprom_read_word(&eeKIPPFAKTOR)!=kippfaktor)
   eeprom_write_word(&eeKIPPFAKTOR,kippfaktor);
 if((int)eeprom_read_word(&eeZsensorpos)!=zsensorpos)
   eeprom_write_word(&eeZsensorpos,zsensorpos);
 if(eeprom_read_float(&eeSinusalfa)!=sinusalfa)
   eeprom_write_float(&eeSinusalfa,sinusalfa);
}

void sinusalfa_berechnen()
{
#ifdef ALFAKORR
 float d = 7*STEPS_PRO_FELD;
 float dx = float(h8xpos-a1xpos_steps) - d;
 float dy = d - float(h8ypos-a1ypos_steps);
 sinusalfa = (dx+dy)/2 / d; //Durchschnitt von dx und dy verwenden
#endif
}

/******************************* Hauptprogramm *******************************/
#define sensorcheck(pin,bit) ((pin&bit)==0) //wenn Sensoren Lowaktiv
//#define sensorcheck(pin,bit) ((pin&bit)!=0) //wenn Sensoren Highaktiv

void fehlermeldung(int n,int s) //n mal pro Sekunde blinken, s Sekunden lang, und Motoren aus
{
 int wartezeit=500/n;
 motor_onoff(0,false); //alle Motoren aus
 s *= n;
 for(int i=0;i<s;i++)
  {
   LED0_ON;
   milliwait(wartezeit);
   LED0_OFF; //schnell Blinkende LED zur Fehleranzeige
   milliwait(wartezeit);
  }
}

#ifdef SCHACH960
void fehlerwarnung(int n,int s) //n mal pro Sekunde blinken, s Sekunden lang, maximal 10
{
 int wartezeit=500/n;
 if(s>10) s=10;
 s *= n;
 for(int i=0;i<s;i++)
  {
   LED0_ON;
   milliwait(wartezeit);
   LED0_OFF; //schnell Blinkende LED zur Fehleranzeige
   milliwait(wartezeit);
  }
}
#endif

void schrittmotoren_steps() //von Interrupt-Routine aufzurufen
{
 if(notstopp)
  {
   MOT1_PORT |= (1<<MOT1_SEL); MOT2_PORT |= (1<<MOT2_SEL);
   MOT3_PORT |= (1<<MOT3_SEL); MOT4_PORT |= (1<<MOT4_SEL); //alle Motoren aus
   steps1=steps2=steps3=steps4=0;
   return;
  }
 static uint mot1takt=0, mot2takt=0, mot3takt=0;
 if(steps1!=0 && (mot1takt+=vmot1) >= VMAX2)
  {
   mot1takt -= VMAX2;
   if(steps1<0)
    {MOT1_PORT |= (1<<MOT1_DIR);
     //auf Sensor am x-Anschlag pruefen:
     if(steps2<=0 && sensorcheck(MOT1_PIN,(1<<MOT1_SENS))) {steps1=0; steps2=0;}
     else  {steps1++; MOT1_PORT |= (1<<MOT1_STEP);}
    }
   else
    {MOT1_PORT &= ~(1<<MOT1_DIR);
     steps1--; MOT1_PORT |= (1<<MOT1_STEP);
    }
   microwait(10);
   MOT1_PORT &= ~(1<<MOT1_STEP);
  }
 if(steps2!=0 && (mot2takt+=vmot2) >= VMAX2)
  {
   mot2takt -= VMAX2;
   if(steps2>0)
    {MOT2_PORT &= ~(1<<MOT2_DIR);
     //auf Sensor am y-Anschlag pruefen:
     if(steps1<=0 && sensorcheck(MOT2_PIN,(1<<MOT2_SENS))) {steps2=0; steps1=steps3=0;}
     else  {steps2--; MOT2_PORT |= (1<<MOT2_STEP);}
    }
   else
    {MOT2_PORT |= (1<<MOT2_DIR);
     steps2++; MOT2_PORT |= (1<<MOT2_STEP);
    }
   microwait(10);
   MOT2_PORT &= ~(1<<MOT2_STEP);
  }
 if(steps3!=0 && (mot3takt+=vmot3) >= VMAX2)
  {
   mot3takt -= VMAX2;
   if(steps3<0)
    {MOT3_PORT |= (1<<MOT3_DIR);
     //auf Sensor am z-Anschlag pruefen:
     if(sensorcheck(MOT3_PIN,(1<<MOT3_SENS))) {steps3=0; steps1=steps2=0;}
     else  {steps3++; MOT3_PORT |= (1<<MOT3_STEP);}
    }
   else
    {MOT3_PORT &= ~(1<<MOT3_DIR);
     steps3--; MOT3_PORT |= (1<<MOT3_STEP);
    }
   microwait(10);
   MOT3_PORT &= ~(1<<MOT3_STEP);
  }
 static uchar mot4takt=0; //Greifer-Motor etwas langsamer
 if(steps4!=0 && ++mot4takt==GREIFER_LANGSAMER)
  {
   mot4takt=0;
   if(steps4<0)
    {MOT4_PORT |= (1<<MOT4_DIR);
     //Sensor fuer "Greifer zu" pruefen:
     //if(sensorcheck(MOT4_PIN, (1<<MOT4_SENS))) {steps4=0;}
     static uint8_t sens4flag=0; //erst beim zweiten Mal den Sensor erkennen
     if(sensorcheck(MOT4_PIN, (1<<MOT4_SENS)))
      {
       //wenn 2 mal erkannt: anhalten und flag loeschen:
       if(++sens4flag==2) {sens4flag=0; steps4=0;}
      }
     else {steps4++; MOT4_PORT |= (1<<MOT4_STEP); sens4flag=0;}
    }
   else
    {MOT4_PORT &= ~(1<<MOT4_DIR);
     steps4--; MOT4_PORT |= (1<<MOT4_STEP);
    }
   microwait(10);
   MOT4_PORT &= ~(1<<MOT4_STEP);
  }
}

void motorwait()
{
 while(steps1!=0 || steps2!=0 || steps3!=0 || steps4!=0)
  {milliwait(10);}
}

void movemot1(int steps)
{
 cli(); steps1 += steps; sei();
}
void movemot2(int steps)
{
 cli(); steps2 += steps; sei();
}
void movemot3(int steps)
{
 MOT3_PORT &= ~(1<<MOT3_SEL); //Motor3 ein
 cli(); steps3 += steps; sei();
}
void movemot4(int steps)
{
 MOT4_PORT &= ~(1<<MOT4_SEL); //Motor4 ein
 cli(); steps4 += steps; sei();
}

void movex(int steps)
{
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 cli(); steps1 += steps; steps2 += steps; sei();
 xposition += steps;
}

void movey(int steps)
{
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 MOT3_PORT &= ~(1<<MOT3_SEL); //Motor3 ein
 cli(); steps1 += steps; //movemot1(steps);
 steps2 -= steps;        //movemot2(-steps);
 steps3 -= steps; sei(); //movemot3(-steps);
 yposition += steps;
}

void movez(int steps)
{
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 movemot3(steps);
 zposition += steps;
}

/****************************** Diagonal-Fahrten ******************************/
static bool alfakorrflag=true;
#ifdef ALFAKORR

float pytagoras(float a,float b) {return sqrt(a*a+b*b);}

void alfakorr(int &xwert,int &ywert)
{
 if(alfakorrflag && sinusalfa!=0 && xwert>=(int)a1xpos_steps && ywert>=(int)a1ypos_steps)
  {
   float c1,c,sinbeta,cosbeta;
   c1 = pytagoras(xwert-(int)a1xpos_steps, ywert-(int)a1ypos_steps);
   if(c1<CM) return;//bei sehr kleinem Wert keine Korrektur noetig
   sinbeta = (xwert-a1xpos_steps)/c1;
   cosbeta = (ywert-a1ypos_steps)/c1;
   c = c1*sinusalfa;
   ywert -= int(sinbeta*c);
   xwert += int(cosbeta*c);
  }
}
#endif //ALFAKORR

#ifdef DIAGONAL_MOVES
void movexy(int xsteps,int ysteps)
{
 #ifdef ALFAKORR
 alfakorr(xsteps,ysteps);
 #endif
 xsteps -= xposition;
 ysteps -= yposition;
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 MOT3_PORT &= ~(1<<MOT3_SEL); //Motor3 ein
 vmot3=VMAX/2; //Motor3 auf halbe Geschwindigkeit setzen
 if(xsteps>0 && ysteps>0)
  {
   int h = (xsteps<ysteps) ? xsteps : ysteps; //h=kleinerer Wert
   cli(); steps1 = h*2;
          steps3 = -h; sei();
   motorwait();
   xsteps -= h; ysteps -= h;
   xposition += h; yposition += h;
  }
 else if(xsteps<0 && ysteps<0)
  {
   int h = (-xsteps < -ysteps) ? -xsteps : -ysteps; //h=kleinerer absoluter Wert
   cli(); steps1 = -h*2;
           steps3 = h; sei();
   motorwait();
   xsteps += h; ysteps += h;
   xposition -= h; yposition -= h;
  }
 else if(xsteps>0 && ysteps<0)
  {
   int h = (xsteps < -ysteps) ? xsteps : -ysteps; //h=kleinerer absoluter Wert
   cli(); steps2 = h*2;
           steps3 = h; sei();
   motorwait();
   xsteps -= h; ysteps += h;
   xposition += h; yposition -= h;
  }
 else if(xsteps<0 && ysteps>0)
  {
   int h = (-xsteps < ysteps) ? -xsteps : ysteps; //h=kleinerer absoluter Wert
   cli(); steps2 = -h*2;
           steps3 = -h; sei();
   motorwait();
   xsteps += h; ysteps -= h;
   xposition -= h; yposition += h;
  }
 vmot3=VMAX; //wieder normale Geschwindigkeit
 if(xsteps!=0) {movex(xsteps); motorwait();}
 if(ysteps!=0) {movey(ysteps); motorwait();}
}
#endif //DIAGONAL_MOVES

#ifdef DIAGONAL_MOVES3
uint abs(int n) {return (n<0) ? -n : n;}
ulong sqri(int n) {return (long)n*(long)n;}//Quadrat von int --> ulong
ulong sqrtu(ulong n) {return (ulong)(sqrtf((float)n)+0.5);}//Wurzel aus Ganzzahl

#define ARMHOEHE 16 //Hoehe des Arms in cm

int kippkorry_berechnen(int ypos)
{
 int kippkorry=0;
#ifdef YZEILE2
 if(ypos > YZEILE2)
  kippkorry =  (long)kippfaktor*ypos*((ypos-YZEILE2)/CM)/(ARMHOEHE*1000);
#else
 if(ypos > (int)a1ypos_steps)
  kippkorry =  (long)kippfaktor*ypos*(ypos/CM)/(ARMHOEHE*1000);
#endif
 return kippkorry;
}

void movexy(int xsteps,int ysteps)
{
#ifdef ALFAKORR
 alfakorr(xsteps,ysteps);
#endif
#ifdef KIPPKORRY
 ysteps += kippkorry_berechnen(ysteps);
#endif
 xsteps -= xposition;
 ysteps -= yposition;
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 MOT3_PORT &= ~(1<<MOT3_SEL); //Motor3 ein
 if(ysteps==0)      {movex(xsteps); motorwait();}
 else if(xsteps==0) {movey(ysteps); motorwait();}
 else
  {
   int s1=0,s2=0,s3=0; //steps fuer die 3 Motoren
   uint v1,v2,v3;      //Geschwindigkeiten
   s1 += xsteps;  s2 += xsteps;                //Motorensteps gemaess Fahrt in x-Richtung
   s1 += ysteps;  s2 -= ysteps;  s3 -= ysteps; //Motorensteps gemaess Fahrt in y-Richtung
   uint as1=abs(s1), as2=abs(s2), as3=abs(s3);
   uint smax = as1;
   if(as2 > smax) smax=as2;
   if(as3 > smax) smax=as3;

   //smax anpassen, so dass Geschwindigkeit in Diagonalen gleich wie auf Achsen:
   ulong strecke=sqrtu(sqri(xsteps)+sqri(ysteps));
   ulong uh=abs(xsteps);  if(abs(ysteps) > uh) uh=abs(ysteps);
   smax = smax * uh / strecke;
   
   v1 = as1*VMAX/smax; if(v1<1) v1=1;
   v2 = as2*VMAX/smax; if(v2<1) v2=1;
   v3 = as3*VMAX/smax; if(v3<1) v3=1;
   cli(); //Motorwerte unterbrechungsfrei setzen:
   steps1=s1; steps2=s2; steps3=s3;
   vmot1=v1;  vmot2=v2;  vmot3=v3;
   sei(); //Interrupts wieder zulassen
   motorwait();
   vmot1=vmot2=vmot3=VMAX;//wieder normale Geschwindigkeiten
   xposition += xsteps; yposition += ysteps;
  }
}
#endif //DIAGONAL_MOVES3
/***************************** :Diagonal-Fahrten ******************************/
 
void movezabs(int steps)
{
#ifdef YZEILE2
 if(yposition > YZEILE2) steps += (long)kippfaktor*(yposition-YZEILE2)/1000;
#else
 if(yposition > 0) steps += (long)kippfaktor*yposition/1000;
#endif
 steps -= zposition;
 MOT1_PORT &= ~(1<<MOT1_SEL); //Motor1 ein
 MOT2_PORT &= ~(1<<MOT2_SEL); //Motor2 ein
 movemot3(steps);
 zposition += steps;
 motorwait(); 
}

uint8_t greiferstatus=0; //1=zu, 2=auf, 0=ausgeschaltet

void greifer_zu()
{
 movemot4(-800); //maximal 1/2 Umdrehung bis Sensor anspricht
 motorwait();
 greiferstatus=1;
 milliwait(200);//test
}

void greifer_auf(int ns=110) //ns = Anzahl Schritte, Defaultwert  (200) ev. anpassen
{
 if(greiferstatus==0) greifer_zu();
 //milliwait(500);//testgreifer
 movemot4(ns);
 motorwait();
 greiferstatus=2;
 //milliwait(100);//test
}

void greifer_off()
{
 motor_onoff(4,false);
 greiferstatus=0;
}

static uint8_t brettlage=1; //SCHWARZ; //schwarze Seite zum Spieler zeigend
static char aktfigur=0;
static uint abhebhoehe=CM9;

// Greiferparameter:
char figurabk[6]   =         {'P','N','B','R','Q','K'}; //englische Figurenabkuerzungen
static uint8_t kopfhoehe[6] = {30, 42, 45, 35, 56, 63}; //Hoehe der Figuren in mm
//static uint8_t weiterauf[6] = {30, 40, 35, 40, 45, 45}; //weiter oeffnen Figurabhaengig
//static uint8_t weiterauf[6] = {35, 40, 40, 40, 45, 45}; //weiter oeffnen Figurabhaengig
static uint8_t weiterauf[6] = {35, 45, 40, 40, 45, 45}; //weiter oeffnen Figurabhaengig
//static uint8_t sockeldurchm[6]={16, 22, 19, 18, 24, 24}; //Sockeldurchmesser in mm

uint letter_xwert(char c)
{
 c = toupper(c);
#ifdef SCHACH960
 if(c=='J')
  {// Figur auf rechten Rand ziehen (mit 'I' auf linken Rand)
   int x = a1xpos_steps; x -= STEPS_PRO_FELD; if(x<0) x=0;
   return x;
  }
#endif
 if(c<'A' || c>'I') c='A'; //bei Fehler auf gueltigen Wert setzen
 uint x = c-'A';
 if(brettlage==0 && c != 'I')  x = 7-x; //wenn weisse Seite beim Spieler
 x = x*STEPS_PRO_FELD + a1xpos_steps;
 return x;
}

uint ziffer_ywert(char c)
{
 if(c<'1' || c>'9') c='1'; //bei Fehler auf gueltigen Wert setzen
 c -= '1';
 if(brettlage==0)  c = 7-c; //wenn weisse Seite beim Spieler
 uint y = c*STEPS_PRO_FELD + a1ypos_steps;
#ifdef ZEILE1KORR
 if(c==0) y -= ZEILE1KORR*MM;
#endif
 return y;
}

void figur_vom_brett_nehmen(uint x,uint y)
{
 movexy(x,y);
 //Figur erfassen:
 int8_t k; for(k=0; k<6 && figurabk[k] != aktfigur; k++) {}
 greifer_auf(); greifer_auf(weiterauf[k]);//erst mal weit oeffnen
 //uint z = 30*MM; //bis auf Hoehe Bauernkopf fahren
 uint z = (kopfhoehe[k]/2)*MM; //bis auf halbe Figurhoehe fahren
 if(z<30*MM) z=30*MM; //aber mindestens 30mm
 movezabs(z);
 greifer_auf(-50-weiterauf[k]); //(-80); //Greifer etwas wieter zu zum Figur positionieren
 movezabs(20*MM); //bis knapp ueber Sockel fahren
 greifer_auf(weiterauf[k]); //dann wieder ganz oeffnen
 movezabs(1*MM); //knapp ueber dem Brett
 milliwait(500);//test: warten bis fertig gewackelt
 greifer_zu();
 z = abhebhoehe;
 movezabs(z); //entsprechend nach oben fahren
}

void figur_abstellen(uint x,uint y)
{
 movexy(x,y);
 movezabs(1*MM); //auf knapp ueber dem Brett fahren
 greifer_auf();
 movezabs(CM9); //uber hoechste Figuren nach oben fahren
 greifer_off();
}

void figur_abstellen2(uint x,uint y) //beim Abstellen geschlagener Figur: etwas hoeher loslassn
{
 movexy(x,y);
 movezabs(5*MM);
 movexy(x+14*MM,y); //ev. vorher geschlagene Figur wegschieben, TODO: groesserer Wert wenn Mechanisch moeglich
 greifer_auf();
 movezabs(CM9);
 greifer_off();
}

void feld_anfahren(const char* str)
{
 uint x = letter_xwert(str[0]);
 uint y = ziffer_ywert(str[1]);
 movexy(x,y);
}

void figur_ziehen(const char* zeile)
{
 uint8_t i=0;
 uint x = letter_xwert(zeile[i++]);
 uint y = ziffer_ywert(zeile[i++]);
 figur_vom_brett_nehmen(x,y);
 if(zeile[i]=='-' || zeile[i]=='x' || zeile[i]=='X') i++;
 x = letter_xwert(zeile[i++]);
 y = ziffer_ywert(zeile[i++]);
 figur_abstellen(x,y);
}

void calibrieren(int ywert=2*CM)
{
 if(ywert!=0) {movey(ywert); motorwait();}
 movex(-32767); motorwait();
 movey(-32767); motorwait();
 greifer_zu();
 movez(-32767); motorwait();
 xposition=XSENSORPOS*MM; yposition=YSENSORPOS*MM; zposition=zsensorpos*MM;
 //static int test=1000; if(test) {milliwait(test); test-=500;}//test
 milliwait(500);//test
 movez(CM9-zposition); //auf 9cm ueber dem Brett nach oben fahren
 motorwait();
 greifer_auf();
 greifer_off();
 motor_onoff(0,false); //alle Motoren aus
}

#define WEISS 0
#define SCHWARZ 1

#ifdef SCHACH960
static char koenig_startpos='E';
static char turm_startpos=0; //wenn nicht Schach960: 'A' bei c-Rochade, 'H' bei g-Rochade
#endif

int main(void)
{
 uint8_t computerfarbe=WEISS;
 uint x,y;
 LED_DDR = (1<<LED0) | (1<<LED1);
 MOT1_DDR |= (1<<MOT1_SEL)|(1<<MOT1_DIR)|(1<<MOT1_STEP); //Ausgaenge setzen
 MOT2_DDR |= (1<<MOT2_SEL)|(1<<MOT2_DIR)|(1<<MOT2_STEP); //Ausgaenge setzen
 MOT3_DDR |= (1<<MOT3_SEL)|(1<<MOT3_DIR)|(1<<MOT3_STEP); //Ausgaenge setzen
 MOT4_DDR |= (1<<MOT4_SEL)|(1<<MOT4_DIR)|(1<<MOT4_STEP); //Ausgaenge setzen
 MOT1_PORT |= (1<<MOT1_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT2_PORT |= (1<<MOT2_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT3_PORT |= (1<<MOT3_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT4_PORT |= (1<<MOT4_SEL); //Motor deaktiviert, (SEL ist Low-aktiv)
 MOT1_PORT |= (1<<MOT1_SENS); //Pullup fuer Sensor
 MOT2_PORT |= (1<<MOT2_SENS); //Pullup fuer Sensor
 MOT3_PORT |= (1<<MOT3_SENS); //Pullup fuer Sensor
 MOT4_PORT |= (1<<MOT4_SENS); //Pullup fuer Sensor
 MOT1_PORT |= (1<<MOT1_SENS2); //Pullup fuer zusaetzlichen Sensor //TODO
 MOT1_PORT |= (1<<MOT1_SENS3); //Pullup fuer zusaetzlichen Sensor //TODO

 timer1_init(); //Uhr und Schrittmotorentakt
 sei(); //Interrupts einschalten
 uart1_init(); //Serielle Schnittstelle
 secwait(1);
 pcint_init(); //fuer Tasten-Interrupts

 // Variablen initialisieren:
 a1xpos_steps=A1XPOS_STEPS; a1ypos_steps=A1YPOS_STEPS;
 a1xypos_von_eeprom_einlesen();
 h8xpos = a1xpos_steps+7*STEPS_PRO_FELD;
 h8ypos = a1ypos_steps+7*STEPS_PRO_FELD;
 kippfaktor_von_eeprom_einlesen();
 sinusalfa_von_eeprom_einlesen();
 zsensorpos_von_eeprom_einlesen();
 //TODO: noch andere Variablen

 //auf Startposition fahren:
 movez(7*CM); motorwait();
 movey(2*CM); motorwait();
 movex(2*CM); motorwait();
 calibrieren();

#ifdef UARTTEST
 movez(4*CM-zposition); //auf 4cm ueber dem Brett nach oben fahren
 motorwait();
 motor_onoff(0,false); //alle Motoren aus
 for(;;)
  {
   uart_putc('a');
   //uart_getc();
    secwait(1);//test
  }
#endif
 
#ifdef DIAGONALTEST
 movexy(a1xpos_steps+2*STEPS_PRO_FELD, a1ypos_steps+STEPS_PRO_FELD); motorwait();
 secwait(3);
 movexy(xposition+2*STEPS_PRO_FELD, yposition); motorwait(); secwait(3);
 movexy(xposition, yposition+2*STEPS_PRO_FELD); motorwait(); secwait(3);
 while(1)
  {
   movexy(xposition + 2*STEPS_PRO_FELD, yposition + 3*STEPS_PRO_FELD); secwait(10);
   movexy(xposition - 2*STEPS_PRO_FELD, yposition - 3*STEPS_PRO_FELD); secwait(10);
   movexy(xposition - 2*STEPS_PRO_FELD, yposition + 2*STEPS_PRO_FELD); secwait(10);
   movexy(xposition + 2*STEPS_PRO_FELD, yposition - 2*STEPS_PRO_FELD); secwait(10);
  }
#endif

#ifdef DIAGONALTEST2
 movexy(a1xpos_steps+3*STEPS_PRO_FELD, a1ypos_steps+3*STEPS_PRO_FELD); motorwait();
 secwait(3);
 while(1)
  {
   movexy(xposition + 2*STEPS_PRO_FELD, yposition + STEPS_PRO_FELD); secwait(10);
   movexy(xposition - 2*STEPS_PRO_FELD, yposition - STEPS_PRO_FELD); secwait(10);

   movexy(xposition + STEPS_PRO_FELD, yposition + 2*STEPS_PRO_FELD); secwait(10);
   movexy(xposition - STEPS_PRO_FELD, yposition - 2*STEPS_PRO_FELD); secwait(10);
   
   movexy(xposition - STEPS_PRO_FELD, yposition + 2*STEPS_PRO_FELD); secwait(10);
   movexy(xposition + STEPS_PRO_FELD, yposition - 2*STEPS_PRO_FELD); secwait(10);
   
   movexy(xposition - 2*STEPS_PRO_FELD, yposition + STEPS_PRO_FELD); secwait(10);
   movexy(xposition + 2*STEPS_PRO_FELD, yposition - STEPS_PRO_FELD); secwait(10);
  }
#endif

#ifdef DIAGONALTEST3
 feld_anfahren("A1"); secwait(2); 
 while(1)
  {
   feld_anfahren("A8"); secwait(2); feld_anfahren("A1"); secwait(2);
   feld_anfahren("B8"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("C8"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("D8"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("E8"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("F8"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("G8"); secwait(2); feld_anfahren("A1"); secwait(2);
   
   feld_anfahren("H8"); secwait(2); feld_anfahren("A1"); secwait(2);

   feld_anfahren("H7"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("H6"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("H5"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("H4"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("H3"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("H2"); secwait(2); feld_anfahren("A1"); secwait(2); 
   feld_anfahren("H1"); secwait(2); feld_anfahren("A1"); secwait(2); 
  }
#endif

 const char *zeile="";
 uart_putc('\n'); //bereit fuer ersten Zug
 while(1) //Hauptschlaufe
  {
   while(notstopp) {set_sleep_mode(SLEEP_MODE_PWR_DOWN); milliwait(100);}//TODO
#ifdef KIPPTEST
   static int testflag=1;
   if(testflag)
    {
     /*
     if(testflag==1) zeile="D1-D8";
     if(testflag==2) zeile="D8-D7";
     if(testflag==3) zeile="D7-D6";
     if(testflag==4) zeile="D6-D5";
     if(testflag==5) zeile="D5-D4";
     if(testflag==6) zeile="D4-D3";
     if(testflag==7) zeile="D3-D2";
     if(testflag==8) zeile="D2-D1";
     */
     if(testflag==1) zeile="A1-A8";
     if(testflag==2) zeile="A8-B8";
     if(testflag==3) zeile="B8-C8";
     if(testflag==4) zeile="C8-E8";
     if(testflag==5) zeile="E8-F8";
     if(testflag==6) zeile="F8-G8";
     if(testflag==7) zeile="G8-H8";
     if(testflag==8) zeile="H8-A1";
     /* */
     if(++testflag == 9) testflag=1;
     uart_print(zeile); uart_putc('\n');
    } else
#endif
   //UART abfragen und entsprechend reagieren:
   zeile=uart_zeile_empfangen();
   //Form der Antwort: "E2-E4" oder z.B. "e2xe4"
   //x = Schlagen von Figuren
   //Spezialzuege: en passant Beispiel: "d4xe3e.p.",
   //              Rochaden: "o-o" "o-o-o" (kleine = g-Rochade, grosse = c-Rochade)
   //              Bauerumwandlung: "b7-b8Q" (Q=Queen, R=Turm, B=Laeufer, N=Springer)
   if(strlen(zeile) < 3)
    {
#ifdef DEBUG
     if(zeile[0]!='A')
      {
       if(zeile[0]==0) fehlermeldung(3,1); //bei leerer Zeile mit 3 Hz 1 Sekunde lang blinken
       else if(zeile[0]==':') fehlermeldung(3,2); //bei fehlendem ersten Zeichen von Parameter: mit 3 Hz 2 Sekunde lang blinken
       else fehlermeldung(5,6);  //bei unbekanntem Fehler 5 mal pro Sekunde blinken, 6 Sekunden lang
       secwait(1);
      }
#endif
     uart_putc('B'); uart_putc('\n'); //Antwort auf Verbindungsaufbau
     continue;
    }
   if(zeile[1]==':')
    {//Parameter setzen:
     if(zeile[0]=='F') computerfarbe=zeile[2]-'0';
     else if(zeile[0]=='G') brettlage=zeile[2]-'0';
     else if(zeile[0]=='P') aktfigur=zeile[2]; //Figur die gezogen wird
#ifdef SCHACH960
     else if(zeile[0]=='K') koenig_startpos=zeile[2]; //Startposition von Koenig fuer Rochade
     else if(zeile[0]=='T') turm_startpos = zeile[2]; //Startposition von Turm fuer Rochade
#endif
     //TODO: eventuell weitere Parameter
     uart_putc('\n'); //bereit fuer naechsten Befehl
     continue;
    }
   if(strncmp(zeile,"CALIB:",6)==0)
    {
     //Kommandos-Beispiele: "CALIB:A1X+2", "CALIB:ok"
     werte_calibrieren(&zeile[6]);
     //uart_putc('\n'); //schon in werte_calibrieren() gemacht
     continue;
    }
   LED1_ON;//test: rote LED leuchten lassen solange Motoren laufen
   if(zeile[2]=='x' || zeile[2]=='X') //Figur schlagen?
    {
     char c1,c2;
 #ifdef DEBUG2
     //testblinken bevor Figur geschlagen wird:
     fehlerwarnung(3,10); //n mal pro Sekunde blinken, s Sekunden lang
 #endif
     if(strncmp(&zeile[5],"e.p.",4)==0) //en passant?
          {c1=zeile[3]; c2=zeile[1];}
     else {c1=zeile[3]; c2=zeile[4];}
     if((brettlage==WEISS && c1=='A') || (brettlage==SCHWARZ && c1=='H'))
          abhebhoehe=CM4;
     else abhebhoehe=CM9;
     x = letter_xwert(c1);
     y = ziffer_ywert(c2);
     figur_vom_brett_nehmen(x,y);
     x = letter_xwert('i'); //geschlagene Figur neben das Brett stellen
     figur_abstellen2(x,y);
    }
   if(zeile[0]=='o') //Rochade?
    {
     abhebhoehe=CM4;
#ifdef SCHACH960
 #ifdef DEBUG2
     /*
      {//test: calibrieren vor Rochade
       if(brettlage==1)  {x = letter_xwert('A');  y = ziffer_ywert('1');}
       else              {x = letter_xwert('H');  y = ziffer_ywert('8');}
       movexy(x,y);
       movey(2*CM); motorwait();
       calibrieren(0);
      }
     */
     if(turm_startpos==0) //test
       fehlerwarnung(3,5); //n mal pro Sekunde blinken, s Sekunden lang
     else
       fehlerwarnung(1,5); //n mal pro Sekunde blinken, s Sekunden lang
 #endif
     if(turm_startpos!=0)
     {
      char cg,df,z,zug1[6],zug2[6],zug3[6];
      bool grosse = (strncmp(zeile,"o-o-o",5)==0);
      if(grosse) {cg='C'; df='D';} else {cg='G'; df='F';}
      if(computerfarbe==WEISS) z='1'; else z='8';
      if( (grosse && koenig_startpos>'D' && turm_startpos!='C') ||
	 (!grosse && koenig_startpos<'F' && turm_startpos!='G') )
       {
	sprintf(zug1,"%c%c-%c%c",koenig_startpos,z, cg,z);
	sprintf(zug2,"%c%c-%c%c",turm_startpos,z,   df,z);
	figur_ziehen(zug1); abhebhoehe=CM9; figur_ziehen(zug2);
       }
      else
       {abhebhoehe=CM9;
	char zwischenpos='I'; //Linker Rand
	if((brettlage==0 && cg=='G') || (brettlage!=0 && cg=='C')) zwischenpos='J'; //ev. Rechter Rand
	sprintf(zug1,"%c%c-%c%c",koenig_startpos,z, zwischenpos,z);
	sprintf(zug2,"%c%c-%c%c",turm_startpos,z,   df,z);
	sprintf(zug3,"%c%c-%c%c",zwischenpos,z, cg,z);
	figur_ziehen(zug1); figur_ziehen(zug2); figur_ziehen(zug3);
       }
      turm_startpos=0;
     }
     else
#endif
     if(strncmp(zeile,"o-o-o",5)==0)
      {//grosse Rochade:
       if(computerfarbe==WEISS)
	    {figur_ziehen("e1-c1"); abhebhoehe=CM9; figur_ziehen("a1-d1");}
       else {figur_ziehen("e8-c8"); abhebhoehe=CM9; figur_ziehen("a8-d8");}
      }
     else
      {//kleine Rochade:
       if(computerfarbe==WEISS)
	    {figur_ziehen("e1-g1"); abhebhoehe=CM9; figur_ziehen("h1-f1");}
       else {figur_ziehen("e8-g8"); abhebhoehe=CM9; figur_ziehen("h8-f8");}
      }
    }
   else //gewoehnlicher Zug
    {
     if(aktfigur=='N') abhebhoehe=CM9; //Springer immer auf Hoehe ueber allen Figuren heben
     else              abhebhoehe=CM4; //sonst nur wenig anheben
     figur_ziehen(zeile);
#ifndef CALIBTEST
     if(strlen(zeile)==6) //bei Bauerumwandlung: kalibrieren damit genug Zeit zum Dame setzen
      {
       if(brettlage==1)  {x = letter_xwert('A');  y = ziffer_ywert('1');}
       else              {x = letter_xwert('H');  y = ziffer_ywert('8');}
       movexy(x,y);
       movey(2*CM); motorwait();
       calibrieren(0);
      }
#endif
    }
#ifdef CALIBTEST
   // jedes mal kalibrieren:
   if(brettlage==1)  {x = letter_xwert('A');  y = ziffer_ywert('2');}
   else              {x = letter_xwert('H');  y = ziffer_ywert('7');}
   movexy(x,y);
   calibrieren(0);
#else
   //Arm zurueckfahren nach abgeschlossenem Zug:
   //movey(ziffer_ywert(brettlage==1 ? '1' : '8') - yposition);
   movey(-5*MM - yposition); yposition=0; //bis zum y-Sensor
   motorwait();
   milliwait(200);//test
  #ifdef AUTOCALIB
   static int recalib=0;//nach jedem 10. Zug neu calibrieren
   if(++recalib >= AUTOCALIB) {calibrieren(); recalib=0;}
  #endif
#endif
   motor_onoff(0,false); //alle Motoren aus
   LED1_OFF;//test: rote LED aus wenn Motoren aus
#ifdef KIPPTEST
   if(testflag)   secwait(2);//test
#endif
   uart_putc('\n'); //bereit fuer naechsten Zug
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

void werte_calibrieren(const char *zeile)
{
 //  Kommandos: "CALIB:<variable><+/-><Anzahl halbe mm>"
 //             "CALIB:KIPPFAKT<+/-><Zahl>"
 //wenn fertig: "CALIB:ok"
 //das "CALIB:" wurde in zeile schon entfernt
 uchar err=0;
 static uchar lastcommand=0; //1=A1XY, 2=H8XY, 3=Kippfaktor, 4=ZSPOS (zsensorpos)
 static uchar antwort=0;
 brettlage=SCHWARZ;
 if(strncmp(zeile,"A1X",3)==0 || strncmp(zeile,"A1Y",3)==0)
  {
   // Beispiel: "A1X+8" a1x um 4 mm erhoehen
   // Beispiel: "A1X-1" a1x um 1/2 mm erniedrigen
   uint16_t steps = (zeile[4]-'0')*STEPS_PRO_METER/2000;
   if(lastcommand==1) //erst ab zweitem Aufruf Position wirklich aendern
    {
     if(zeile[3]=='+')      {if(zeile[2]=='X') a1xpos_steps += steps; else a1ypos_steps += steps;}
     else if(zeile[3]=='-') {if(zeile[2]=='X') a1xpos_steps -= steps; else a1ypos_steps -= steps;}
     else err=1;
    }
   else alfakorrflag=false; //Winkelkorrektur ausschalten
   if(err==0)
    {
     movexy(a1xpos_steps,a1ypos_steps);
     if(lastcommand==0) {movezabs(5*MM); lastcommand=antwort=1;}
    }
  }
 else if(strncmp(zeile,"H8X",3)==0 || strncmp(zeile,"H8Y",3)==0)
  {
   uint16_t steps = (zeile[4]-'0')*STEPS_PRO_METER/2000;
   if(lastcommand==2) //erst ab zweitem Aufruf Position wirklich aendern
    {
     if(zeile[3]=='+')      {if(zeile[2]=='X') h8xpos += steps; else h8ypos += steps;}
     else if(zeile[3]=='-') {if(zeile[2]=='X') h8xpos -= steps; else h8ypos -= steps;}
     else err=2;
    }
   else
    {
     sinusalfa=0; //bisherige Winkel-Korrektur loeschen
     h8xpos = a1xpos_steps+7*STEPS_PRO_FELD; //h8xypos auf theoretischen Wert setzen
     h8ypos = a1ypos_steps+7*STEPS_PRO_FELD;
    }
   if(err==0)
    {
     movexy(h8xpos,h8ypos);
     if(lastcommand==0) {movezabs(5*MM); lastcommand=antwort=2;}
    }
  }
 else if(strncmp(zeile,"KIPPFAKT",8)==0)
  {
   // Beispiel: "KIPPFAKT+5" //kippfaktor um 5 erhoehehen
   uint16_t diff = zeile[9]-'0'; if(isdigit(zeile[10])) diff = diff*10+(zeile[10]-'0');//max. 2stellige Zahl
   if(lastcommand==3) //erst ab zweitem Aufruf Kippfaktor wirklich aendern
    {
     if(zeile[8]=='+')      kippfaktor += diff;
     else if(zeile[8]=='-') {if(diff>kippfaktor) kippfaktor=0; else kippfaktor -= diff;}
     else err=3;
    }
   else
    {
     feld_anfahren("D2"); movezabs(20*MM);
     movezabs(1*MM); secwait(2);//auf 1mm ueber dem Brett fahren
     movezabs(20*MM);
     feld_anfahren("D8");
     lastcommand=antwort=3;
    }
   if(err==0)
    {
     //movezabs(20*MM); //test: zuerst etwas hochfahren
     movezabs(1*MM); //auf 1mm ueber dem Brett fahren (falls Kippfaktor stimmt)
    }
  }
 else if(strncmp(zeile,"ZSPOS",5)==0)
  {
   // Beispiele: "ZSPOS+1", "ZSPOS-1"
   int diff = zeile[6]-'0';
   if(lastcommand==4) //erst ab zweitem Aufruf zsensorpos wirklich aendern
    {
     if(zeile[5]=='+')      {}
     else if(zeile[5]=='-') {diff = -diff;}
     else err=4;
     if(err==0) {movez(diff*MM); zsensorpos -= diff;}
    }
   else
    {
     movezabs(CM9);
     feld_anfahren("A2"); movezabs(20*MM); secwait(1);
     movezabs(1*MM); secwait(1);
     lastcommand=antwort=4;
    }
  }
 else if(strncmp(zeile,"ok",2)==0)
  {
   if(lastcommand==1) {alfakorrflag=true;} //Winkelkorrektur wieder einschalten
   if(lastcommand==2) {sinusalfa_berechnen();} //neue Winkelkorrektur
   werte_in_eeprom_speichern();
   movez(CM9-zposition); motorwait();
   calibrieren();
   lastcommand=0;
  }
 else err=5;
 if(err)
  {
   fehlermeldung(4,1); //4 mal pro Sekunde blinken, 1 Sekunde lang
   uart_printf("Error%d",err);
  }
 else
  {
   if(antwort==1) uart_printf("A1XY:%d %d", a1xpos_steps/MM, a1ypos_steps/MM);
   else if(antwort==2) uart_printf("H8XY:%d %d", h8xpos/MM, h8ypos/MM);
   else if(antwort==3) uart_printf("KIPPFAKT:%d", kippfaktor);
   else if(antwort==4) uart_printf("ZSPOS:%d",zsensorpos);
  }
 uart_putc('\n'); //bereit fuer naechsten Befehl
}
