/* xyzstl.cc			letzte Aenderung: 13.10.2013 */
#define VERSION "Version 0.1"
/*
Uebersetzen auf Unix (Linux):
> make  ;siehe makefile

 Kurzbeschreibung: Molekueldateien.xyz umwandeln in STL-Datei

History:
12.10.2013	Erstellung (RP)
*/

#include <stdio.h>
#include <iostream>
#include <stdlib.h>
#include <string.h>
#include "vektor3dklasse.cc"

/************************* Vordeklarationen ***************************/
void kugeln_speichern(FILE *fp);
void kugel_merken(Vektor3d& kugelmittelpunkt,double radius);
void kugel_speichern(FILE *fp,Vektor3d& kugelmittelpunkt,double radius);

/************************* Globale Variablen **************************/
static int workbenchflag=0;
static int gradstep=5; //Aufloesung fuer Kugelannaeherungen

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
   if((c=='a' || c=='A') && isdigit(s[0]))
    {
     sscanf(s,"%d",&gradstep);
     while(isdigit(s[0])) s++;
    }
  }
}

/*************************** Kleinkram ********************************/
bool istja(char *s)
{
 int c= *s;
 return (c=='j' || c=='J' || c=='y' || c=='Y');
}

char *ohneletztenpunkt(char *name)  /* die Endung .xxx loeschen */
{
 char c=0,*s; int i;
 for(i=0,s=name;*s!=0;s++,i++) ;
 while(i>0 && (c= *s)!='.')
        {--i; --s;}
 if(c=='.') *s='\0';
 return name;
}

char *anhaengen(char *s,const char *t)  /* s + t --> s */
{
 char *p;
 for(p=s;*p!=0;p++) ;
 for(;*p++ = *t++;) ;
 return s;
}

char *endungersetzen(char *neu,const char *name,const char *endung)
{
 strcpy(neu,name); ohneletztenpunkt(neu);
 return anhaengen(neu,endung);
}

/*********************** Dateien einlesen *****************************/
const int MAXATOME=10000;

const int NELE=27*26;
typedef char STRING[80];

class Elemente  //uebernommen von Molz3d
{
 double radius[NELE];
 double vdwradius[NELE]; //Vanderwaals-Radi
 STRING farbe[NELE];
 int ind(int c1,int c2=0)
  {if(c1<'A' || c1>'Z') return 0;
   if(c2==0) return 26*(c1-'A');
   if(c2<'a' || c2>'z') return 0;
   return 26*(c1-'A')+c2-'a'+1;
  }
public:
 Elemente()
  {int n;
   //elemente_init();
   for(int i=0;i<NELE;i++) {vdwradius[i]=radius[i]=1.0; strcpy(farbe[i],"Plum");}
   // Atomradien in Angstroem (1Angstrom=0.1nm)
   radius[n=ind('C')]=0.772; //Kohlenstoff Radius in Angstrom (0.1nm)
   strcpy(farbe[n],"Gray50"); //Farbe fuer Povray
   radius[n=ind('H')]=0.373; strcpy(farbe[n],"White"); //Wasserstoff
   radius[n=ind('B')]=0.795; strcpy(farbe[n],"Gray60"); //Bor
   radius[n=ind('N')]=0.549; strcpy(farbe[n],"Blue"); //Stickstoff
   radius[n=ind('O')]=0.604; strcpy(farbe[n],"Red"); //Sauerstoff
   radius[n=ind('F')]=0.709; strcpy(farbe[n],"rgb<0.8,1,0.5>"); //Fluor
   radius[n=ind('N','a')]=1.858; strcpy(farbe[n],"Gray40"); //Natrium
   radius[n=ind('C','l')]=0.994; strcpy(farbe[n],"Green"); //Chlor
   radius[n=ind('B','r')]=1.145; strcpy(farbe[n],"rgv<0.7,0.7,0>"); //Brom
   radius[n=ind('I')]=1.331; strcpy(farbe[n],"rgb<0.5,1,1>"); //Jod
   radius[n=ind('J')]=1.331; strcpy(farbe[n],"rgb<0.5,1,1>"); //Jod
   radius[n=ind('S')]=1.035; strcpy(farbe[n],"Yellow"); //Schwefel
   radius[n=ind('P')]=1.105; strcpy(farbe[n],"rgb<1,0,1>"); //Phosphor
   // Van-der-Waals-Radien in Angstrom:
   vdwradius[n=ind('C')]=1.70; //Kohlenstoff
   vdwradius[n=ind('H')]=1.10; //Wasserstoff
   vdwradius[n=ind('B')]=1.92; //Bor
   vdwradius[n=ind('N')]=1.55; //Stickstoff
   vdwradius[n=ind('O')]=1.52; //Sauerstoff
   vdwradius[n=ind('F')]=1.47; //Fluor
   vdwradius[n=ind('N','a')]=2.27; //Natrium
   vdwradius[n=ind('C','l')]=1.75; //Chlor
   vdwradius[n=ind('B','r')]=1.85; //Brom
   vdwradius[n=ind('I')]=1.98; //Jod
   vdwradius[n=ind('J')]=1.98; //Jod
   vdwradius[n=ind('S')]=1.80; //Schwefel
   vdwradius[n=ind('P')]=1.80; //Phosphor
  }
 double radi(int c1,int c2) {return radius[ind(c1,c2)];}
 double radi_vdw(int c1,int c2) {return vdwradius[ind(c1,c2)];}
 char* farb(char *sym) {return farbe[ind(sym[0],sym[1])];}
};
static Elemente gelemente;

double atomradius(char *sym)
{
 return gelemente.radi(sym[0],sym[1]);
}

double atomradius_vdw(char *sym)
{
 return gelemente.radi_vdw(sym[0],sym[1]);
}

bool xyzeinlesen(FILE *fp)
{
 char atomname[80],sym[4],*s,c1,c2;
 int natome=0,nbindungen=0;
 double radius;
 Vektor3d v;
 fscanf(fp,"%d %d",&natome,&nbindungen);
 if(natome<1 || natome>MAXATOME)
  {
   if(natome>0) printf("Fehler: in xyz-Datei zu viele Atome. (%d)\n",natome);
   else printf("Fehler: in xyz-Datei fehlende Anzahl Atome. (%d)\n",natome);
   return false;
  }
 for(int i=0;i<natome;i++)
  {
   //fscanf(fp,"%s %lf %lf %lf",atomname,&x,&y,&z);
   fscanf(fp,"%s %lf %lf %lf",atomname,&v.x,&v.z,&v.y); //anderes Koordinatensystem fuer STL-Format
   c1=atomname[0]; c2=atomname[1];
   sym[0]=c1; sym[1]=c2; sym[2]=0;
   radius=atomradius_vdw(sym)*10; // 10 mm soll 1 Angstroem entsprechen
   v *= 10.0; //Vektor auch in mm umrechnen
   kugel_merken(v,radius);
  }
 return true;
}

/************** Klassen und Hauptteil des Programms *******************/

const double scalx=1; //Scaling factor to get millimeters in x-Axis
const double scaly=1; //Scaling factor to get millimeters in y-Axis
const double scalz=1; //Scaling factor to get millimeters in z-Axis
const double ddr=1e-4; //Kleiner Wert um Rundungsfehler zu vermeiden

class Kugel
{
public:
 Vektor3d pos;
 double radius;
 Kugel() {radius=0; pos.x=pos.y=pos.z=0;}
 void init(Vektor3d& mittelpunkt,double r) {pos=mittelpunkt; radius=r;}
 double getradius() {return radius;}
 bool istinnerhalb(Vektor3d& p);
 bool istausserhalb(Vektor3d& p);
 Vektor3d schnittpunkt(Vektor3d p1,Vektor3d p2);
};

bool Kugel::istinnerhalb(Vektor3d& p)
{
 return (abs(p-pos) < radius-ddr);
}

bool Kugel::istausserhalb(Vektor3d& p)
{
 return (abs(p-pos) > radius+ddr);
}

Vektor3d Kugel::schnittpunkt(Vektor3d p1,Vektor3d p2) //p1 soll innerhalb der Kugel liegen, p2 ausserhalb der Kugel
{
 double A,a,b,c;
 Vektor3d v,q;           //v=Hilfsvektor, q=gesuchter Schnittpunkt
 p1 -= pos; p2 -= pos; //wir setzen das Koordinatensystem so dass der Kugelmittelpunkt am Nullpunkt ist
 v = p2-p1;
 // Herleitung der Loesungsformel:
 // q = (p2-p1)*A + p1 = v*A + p1   der Punkt q liegt auf der Strecke p1_p2, A muss eine Zahl zwischen 0 und 1 sein
 // r^2 = x^2 + y^2 + z^2           Kugelgleichung
 // q.x = v.x*A + p1.x
 // q.y = v.y*A + p1.y
 // q.z = v.z*A + p1.z
 // Wir haben also 4 Gleichungen mit 4 Unbekannten: A, q.x, q.y, q.z
 // Die letzten 3 Gleichungen in die Kugelgleichung eingesetzt ergibt eine Gleichung mit nur noch einer Unbekannten: A
 // Umformung auf die Form der allgemeinen Quadratischen Gleichung ergibt die Loesung.
 a = v.x*v.x + v.y*v.y + v.z*v.z;
 b = (v.x*p1.x + v.y*p1.y + v.z*p1.z)*2;
 c = p1.x*p1.x + p1.y*p1.y + p1.z*p1.z - radius*radius;
 A = (sqrt(b*b-4*a*c)-b)/(2*a); //allgemeine Loesungsformel fuer Quadratische Gleichung (hier nur positives Resultat sinnvoll)
 q = v*A + p1;
 return (q+pos); //Resultat wieder im vorherigen Koordinatensystem zurueckgeben
}

static Kugel kugel_liste[MAXATOME]; //provi.
static int anzahl_kugeln=0;
static int j_aktuelle_kugel=0;

void dreieck(FILE *fp,Vektor3d& p1,Vektor3d& p2,Vektor3d& p3,int j0=0)
{
 int j;
 Vektor3d q1,q2,normalenvektor;
 //if(innentestflag) //test
 for(j=j0;j<anzahl_kugeln;j++)
  {
   if(j!=j_aktuelle_kugel)
    {
     // Es gibt 12 Moeglichkeiten wie ein Dreieck eine Kugel schneiden kann.
     // Begruendung: jeder der 3 Eckpunkte kann innerhalb, ausserhalb oder auf der Kugeloberflaeche liegen,
     // es gibt also total 3^3 = 27 Moeglichkeiten. Davon fallen die Varianten weg wo alle Punkte ausserhalb oder
     // auf der Oberflaeche liegen, also 3^2 = 8 Varianten. Auch die Varianten wo alle punkte innerhalb oder auf
     // der Oberflaeche liegen, also nochmals 3^2 = 8 fallen weg. Wobei wir jetzt die Variante wo alle Punkte auf
     // der Oberflaeche liegen 2 mal gezaehlt haben, also 1 weniger abziehen.
     // Also verbleiben 27-(2*8-1) = 12 Moeglichkeiten.
     // Hier also die entsprechenden if-Abfragen um alle Moeglichkeiten abzudecken.
     if(kugel_liste[j].istinnerhalb(p1))
      {
       if(kugel_liste[j].istausserhalb(p2))
	{
	 q1=kugel_liste[j].schnittpunkt(p1,p2);
	 if(kugel_liste[j].istausserhalb(p3)) //nur p1 innerhalb, p2 und p3 ausserhalb (1. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p1,p3);
	   dreieck(fp,q1,p2,p3,++j);
	   dreieck(fp,q1,p3,q2,j); return;
	  }
	 else if(kugel_liste[j].istinnerhalb(p3)) //p1 und p3 innerhalb, p2 ausserhalb (2. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p3,p2);
	   dreieck(fp,q1,p2,q2,++j); return;
	  }
	 else //p1 innerhalb, p3 auf der Kugeloberflaeche, p2 ausserhalb (3. Moeglichkeit)
	  {
	   dreieck(fp,q1,p2,p3,++j); return;
	  }
	}
       else if(kugel_liste[j].istinnerhalb(p2)) //p1 und p2 innerhalb
	{
	 if(kugel_liste[j].istausserhalb(p3)) //p1 und p2 innerhalb, p3 ausserhalb (4. Moeglichkeit)
	  {
	   q1=kugel_liste[j].schnittpunkt(p1,p3);
	   q2=kugel_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,q1,q2,p3,++j); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
       else //p1 innerhalb, p2 auf Kugeloberflaeche
	{
	 if(kugel_liste[j].istausserhalb(p3)) //p1 innerhalb, p2 auf Oberflaeche, p3 ausserhalb (5. Moeglichkeit)
	  {
	   q1=kugel_liste[j].schnittpunkt(p1,p3);
	   dreieck(fp,q1,p2,p3,++j); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
      } //Ende if istinnerhalb(p1)

     else if(kugel_liste[j].istinnerhalb(p2))
      {
       if(kugel_liste[j].istausserhalb(p1))
	{
	 q1=kugel_liste[j].schnittpunkt(p2,p1);
	 if(kugel_liste[j].istausserhalb(p3)) //nur p2 innerhalb, p1 und p3 ausserhalb (6. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,p1,q1,q2,++j);
	   dreieck(fp,p1,q2,p3,j); return;
	  }
	 else if(kugel_liste[j].istinnerhalb(p3)) //p2 und p3 innerhalb, p1 ausserhalb (7. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p3,p1);
	   dreieck(fp,p1,q1,q2,++j); return;
	  }
	 else //p2 innerhalb, p3 auf der Kugeloberflaeche, p1 ausserhalb (8. Moeglichkeit)
	  {
	   dreieck(fp,p1,q1,p3,++j); return;
	  }
	}
       else //p2 innerhalb, p1 auf Kugeloberflaeche
	{
	 if(kugel_liste[j].istausserhalb(p3)) //p2 innerhalb, p1 auf Oberflaeche, p3 ausserhalb (9. Moeglichkeit)
	  {
	   q1=kugel_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,p1,q1,p3,++j); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
      } //Ende if istinnerhalb(p2)

     else if(kugel_liste[j].istinnerhalb(p3))
      {
       if(kugel_liste[j].istausserhalb(p1))
	{
	 q1=kugel_liste[j].schnittpunkt(p3,p1);
	 if(kugel_liste[j].istausserhalb(p2)) //p3 innerhalb, p1 und p2 ausserhalb (10. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p3,p2);
	   dreieck(fp,p1,p2,q2,++j);
	   dreieck(fp,p1,q2,q1,j); return;
	  }
	 else //p3 innerhalb, p2 auf Oberflaeche, p1 ausserhalb (11. Moeglichkeit)
	  {
	   dreieck(fp,p1,p2,q1,++j); return;
	  }
	}
       else if(kugel_liste[j].istausserhalb(p2)) //p3 innerhalb, p1 auf Oberflaeche, p2 ausserhalb (12. Moeglichkeit)
	{
	 q1=kugel_liste[j].schnittpunkt(p3,p2);
	 dreieck(fp,p1,p2,q1,++j); return;
	}
       else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	{
	 return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
      } //Ende if istinnerhalb(p3)
    } //Ende if j!=j_aktuelle_kugel
  } //Ende der for-j-Schlaufe
 q1=p2-p1;
 q2=p3-p2;
 normalenvektor=vektorkreuzprodukt(q1,q2);
 fprintf(fp," facet normal %.3f %.3f %.3f\n", normalenvektor.x, normalenvektor.y, normalenvektor.z);
 fprintf(fp,"  outer loop\n");
 fprintf(fp,"   vertex %.3f %.3f %.3f\n", p1.x*scalx, p1.y*scaly, p1.z*scalz);
 fprintf(fp,"   vertex %.3f %.3f %.3f\n", p2.x*scalx, p2.y*scaly, p2.z*scalz);
 fprintf(fp,"   vertex %.3f %.3f %.3f\n", p3.x*scalx, p3.y*scaly, p3.z*scalz);
 fprintf(fp,"  endloop\n");
 fprintf(fp," endfacet\n");
}

void viereck(FILE *fp,Vektor3d& p1,Vektor3d& p2,Vektor3d& p3,Vektor3d& p4)
{
 dreieck(fp,p1,p2,p3);
 dreieck(fp,p1,p3,p4);
}

/************************* Hauptprogramm ******************************/
main(int argc,char *argv[])
{
 char quellname[80],zielname[80],antw[40];
 FILE *fp1,*fp2;
 int i,j,c;
 quellname[0]=zielname[0]=0;
 if(argc<=0)
  {/* es wurde von WorkBench (GUI, Desktop, ...) gestartet */
   workbenchflag=1;
  }
 else
  {/* es wurde von Shell (Terminal, Konsole, ...) gestartet */
   for(j=0,i=1;i<argc;i++)
	{if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
	 else	{if(++j==1) strcpy(quellname,argv[i]);
	         else if(j==2) strcpy(zielname,argv[i]);
	}	}
  }
 if(argflag['?'] || j<1 || j>MAXARG)
	{printf("xyzstl  %s\n",VERSION);
	 printf("Anwendung: xyzstl Quelle.xyz [Ziel.stl] [-Flags]\n");
	 printf("  Flags: a=Aufloesung setzen (Schrittweit in ganzen Grad)\n");
	 printf("         b=Binaer speichern (noch nicht unterstuetzt)\n");
	 printf("         v=Verbose, mit Testausgaben\n");
	 printf("Beispiel: xyzstl -a20 -vb test.xyz test.stl\n");
	 exit(0);
	}
 if(gradstep<1) gradstep=1;
 else if(gradstep>30) gradstep=30;
 else while(180/gradstep*gradstep != 180) gradstep++;
 if(argflag['V']) printf("gradstep=%d\n",gradstep);
 fp1=fopen(quellname,"r");
 if(fp1==NULL) {printf("kann '%s' nicht oeffnen.\n",quellname); exit(0);}
 if(xyzeinlesen(fp1)==false) {printf("konnte Molekuel nicht einlesen.\n"); exit(0);}
 if(*zielname==0)
  {
   endungersetzen(zielname,quellname,".stl");
   if((fp2=fopen(zielname,"r"))!=NULL)
    {
     fclose(fp2);
     printf("'%s' schon vorhanden. Ueberschreiben?",zielname); scanf("%s",antw);
     if(!istja(antw)) {printf("nein - nichts gespeichert.\n"); return 0;}
    }
  }
 fp2=fopen(zielname,"w");
 if(fp2==NULL) {printf("konnte '%s' nicht erstellen.\n",zielname); exit(0);}
 kugeln_speichern(fp2);
 fclose(fp2);
 return 0;
}/* ende von main */

void kugeln_speichern(FILE *fp2)
{
 int j;
 for(j=0;j<anzahl_kugeln;j++)
  {
   j_aktuelle_kugel=j;
   kugel_speichern(fp2, kugel_liste[j].pos, kugel_liste[j].getradius());
  }
 fprintf(fp2,"endsolid unnamed\n");
}

void kugel_merken(Vektor3d& kugelmittelpunkt,double r)
{
 int i=anzahl_kugeln;
 if(i==MAXATOME) {fprintf(stderr,"zu viele Kugeln\n"); return;} //provi.
 kugel_liste[i].init(kugelmittelpunkt,r);
 anzahl_kugeln++;
}

void kugel_speichern(FILE *fp,Vektor3d& kugelmittelpunkt,double r)
{
 static int ntekugel=0;
 Vektor3d p1,p2,p3,p4,v1,v2,normalenvektor;
 int    theta0,phi0,dtheta0,dphi0; //Werte in Grad
 double theta, phi, dtheta, dphi, sintheta,sintheta2,costheta2; //Werte in Rad
 ntekugel += 1;
 dtheta0=dphi0=gradstep;
 //fprintf(fp,"solid kugel%d\n",ntekugel);
 if(ntekugel==1) fprintf(fp,"solid unnamed\n");
 for(theta0=180;theta0>0;theta0-=dtheta0) //gehe vom Suedpol zum Nordpol in ganzzahligen Schritten
  for(phi0= 0;phi0<360;phi0+=dphi0)       //gehe einmal rundrum
   {
    theta=theta0*GRAD; phi=phi0*GRAD; dtheta=dtheta0*GRAD; dphi=dphi0*GRAD;
    sintheta=sin(theta);
    p1=p2=p3=kugelmittelpunkt;
    //printf("theta = %d Grad = %lf, phi = %d Grad = %lf\n",theta0,theta,phi0,phi);//test
    if(theta0==180)
     { //einzelnes Dreieck am Suedpol
      sintheta2=sin(theta-dtheta);
      costheta2=cos(theta-dtheta);
      p1.z -= r;
      p2.x += r*sintheta2*cos(phi);
      p2.y += r*sintheta2*sin(phi);
      p2.z += r*costheta2;
      p3.x += r*sintheta2*cos(phi+dphi);
      p3.y += r*sintheta2*sin(phi+dphi);
      p3.z = p2.z;
      dreieck(fp,p1,p3,p2);
     }
    else if(theta0==dtheta0)
     { //einzelnes Dreieck am Nordpol
      p1.z += r;
      p2.x += r*sintheta*cos(phi);
      p2.y += r*sintheta*sin(phi);
      p2.z += r*cos(theta);
      p3.x += r*sintheta*cos(phi+dphi);
      p3.y += r*sintheta*sin(phi+dphi);
      p3.z = p2.z;
      dreieck(fp,p1,p2,p3);
     }
    else
     {// wenn nicht am Pol dann jeweils ein Viereck
      // Formeln aus  http://de.wikipedia.org/wiki/Kugelkoordinaten
      sintheta2=sin(theta-dtheta);
      costheta2=cos(theta-dtheta);

      p1.x += r*sintheta*cos(phi);
      p1.y += r*sintheta*sin(phi);
      p1.z += r*cos(theta);

      p2.x += r*sintheta*cos(phi+dphi);
      p2.y += r*sintheta*sin(phi+dphi);
      p2.z = p1.z;

      p3.x += r*sintheta2*cos(phi+dphi);
      p3.y += r*sintheta2*sin(phi+dphi);
      p3.z += r*costheta2;

      p4=kugelmittelpunkt;
      p4.x += r*sintheta2*cos(phi);
      p4.y += r*sintheta2*sin(phi);
      p4.z = p3.z;
      viereck(fp,p1,p2,p3,p4);
     }
   }
 //fprintf(fp,"endsolid kugel%d\n",ntekugel);
}
