/* xyzstl.cc			letzte Aenderung: 16.10.2013 */
#define VERSION "Version 0.2"
/*
Uebersetzen auf Unix (Linux):
> make  ;siehe makefile

 Kurzbeschreibung: Molekueldateien.xyz umwandeln in STL-Datei

History:
12.10.2013	Erstellung (RP)
*/

#include <stdio.h>
#include <iostream>
#include <stdlib.h>
#include <string.h>
#include "vektor3dklasse.cc"

/************************* Vordeklarationen ***************************/
void kugel_merken(Vektor3d& kugelmittelpunkt,double radius);
void kugel_speichern(FILE *fp,Vektor3d& kugelmittelpunkt,double radius);
void alle_kugeln_speichern(FILE *fp);
void zylinder_speichern(FILE *fp,Vektor3d pos,double radius,Vektor3d achse);
void alle_zylinder_speichern(FILE *fp);

/************************* Globale Variablen **************************/
static int workbenchflag=0;
static int gradstep=5; //Aufloesung fuer Kugelannaeherungen

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
   if((c=='a' || c=='A') && isdigit(s[0]))
    {
     sscanf(s,"%d",&gradstep);
     while(isdigit(s[0])) s++;
    }
  }
}

/*************************** Kleinkram ********************************/
bool istja(char *s)
{
 int c= *s;
 return (c=='j' || c=='J' || c=='y' || c=='Y');
}

char *ohneletztenpunkt(char *name)  /* die Endung .xxx loeschen */
{
 char c=0,*s; int i;
 for(i=0,s=name;*s!=0;s++,i++) ;
 while(i>0 && (c= *s)!='.')
        {--i; --s;}
 if(c=='.') *s='\0';
 return name;
}

char *anhaengen(char *s,const char *t)  /* s + t --> s */
{
 char *p;
 for(p=s;*p!=0;p++) ;
 for(;*p++ = *t++;) ;
 return s;
}

char *endungersetzen(char *neu,const char *name,const char *endung)
{
 strcpy(neu,name); ohneletztenpunkt(neu);
 return anhaengen(neu,endung);
}

int getline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
	*s++ = c;
 *s='\0';
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

/************** Klassen und Hauptteil des Programms *******************/

const double scalx=1; //Scaling factor to get millimeters in x-Axis
const double scaly=1; //Scaling factor to get millimeters in y-Axis
const double scalz=1; //Scaling factor to get millimeters in z-Axis
const double ddr=1e-4; //Kleiner Wert um Rundungsfehler zu vermeiden

class Kugel
{
public:
 Vektor3d pos;
 double radius;
 Kugel() {radius=0; pos.x=pos.y=pos.z=0;}
 void init(Vektor3d& mittelpunkt,double r) {pos=mittelpunkt; radius=r;}
 double getradius() {return radius;}
 bool istinnerhalb(Vektor3d& p);
 bool istausserhalb(Vektor3d& p);
 Vektor3d schnittpunkt(Vektor3d p1,Vektor3d p2);
};

bool Kugel::istinnerhalb(Vektor3d& p)
{
 return (abs(p-pos) < radius-ddr);
}

bool Kugel::istausserhalb(Vektor3d& p)
{
 return (abs(p-pos) > radius+ddr);
}

Vektor3d Kugel::schnittpunkt(Vektor3d p1,Vektor3d p2) //p1 soll innerhalb der Kugel liegen, p2 ausserhalb der Kugel
{
 double A,a,b,c;
 Vektor3d v,q;           //v=Hilfsvektor, q=gesuchter Schnittpunkt
 p1 -= pos; p2 -= pos; //wir setzen das Koordinatensystem so dass der Kugelmittelpunkt am Nullpunkt ist
 v = p2-p1;
 // Herleitung der Loesungsformel:
 // q = (p2-p1)*A + p1 = v*A + p1   der Punkt q liegt auf der Strecke p1_p2, A muss eine Zahl zwischen 0 und 1 sein
 // r^2 = x^2 + y^2 + z^2           Kugelgleichung
 // q.x = v.x*A + p1.x
 // q.y = v.y*A + p1.y
 // q.z = v.z*A + p1.z
 // Wir haben also 4 Gleichungen mit 4 Unbekannten: A, q.x, q.y, q.z
 // Die letzten 3 Gleichungen in die Kugelgleichung eingesetzt ergibt eine Gleichung mit nur noch einer Unbekannten: A
 // Umformung auf die Form der allgemeinen Quadratischen Gleichung ergibt die Loesung.
 a = v.x*v.x + v.y*v.y + v.z*v.z;
 b = (v.x*p1.x + v.y*p1.y + v.z*p1.z)*2;
 c = p1.x*p1.x + p1.y*p1.y + p1.z*p1.z - radius*radius;
 A = (sqrt(b*b-4*a*c)-b)/(2*a); //allgemeine Loesungsformel fuer Quadratische Gleichung (hier nur positives Resultat sinnvoll)
 q = v*A + p1;
 return (q+pos); //Resultat wieder im vorherigen Koordinatensystem zurueckgeben
}

//static Kugel kugel_liste[MAXATOME]; //provi.
static Kugel *kugel_liste=NULL;
static int anzahl_kugeln=0;
static int j_aktuelle_kugel= -1;

class Zylinder
{
public:
 Vektor3d pos;
 double radius;
 Vektor3d achse; //Richtung und Laenge der Achse (jeweils halbe Laenge vom Mittelpunkt aus in beide Richtungen)
 Zylinder() {radius=0;}
 void init(Vektor3d mittelpunkt,double r,Vektor3d a) {pos=mittelpunkt; radius=r; achse=a;}
 double getradius() {return radius;}
 void drehwinkel_berechnen(double* sinw1,double* cosw1,double* sinw2,double* cosw2);
 bool istinnerhalb(Vektor3d p);
 bool istausserhalb(Vektor3d p);
 Vektor3d schnittpunkt(Vektor3d p1,Vektor3d p2);
};

void Zylinder::drehwinkel_berechnen(double* sinw1,double* cosw1,double* sinw2,double* cosw2)
{
 double drehwinkel1,drehwinkel2;
 Vektor3d v;
 v = achse;
 drehwinkel1 = atan2(v.y,v.x); //Winkel um den gedreht werden muss damit v in die x-z-Ebene zu liegen kommt
 *sinw1=sin(drehwinkel1); *cosw1=cos(drehwinkel1);
 v.drehenyx(*sinw1,*cosw1);
 drehwinkel2 = atan2(v.x,v.z); //Winkel um den weiter gedreht werden muss damit v auf die z-Achse zu liegen kommt
 *sinw2=sin(drehwinkel2); *cosw2=cos(drehwinkel2);
 //v.drehenxz(*sinw2,*cosw2);
 //Verwendung:
 // vorwaerts drehen:  p.drehenyx(sinw1,cosw1); p.drehenxz(sinw2,cosw2);
 //  zurueck  drehen:  p.drehenzx(sinw2,cosw2); p.drehenxy(sinw1,cosw1);
}

bool Zylinder::istinnerhalb(Vektor3d p)
{
 double r=radius-ddr;
 double sinw1,cosw1,sinw2,cosw2;
 double halbelaenge=abs(achse)*0.5;
 p -= pos;
 drehwinkel_berechnen(&sinw1,&cosw1,&sinw2,&cosw2);
 p.drehenyx(sinw1,cosw1); p.drehenxz(sinw2,cosw2);
 if(p.z>halbelaenge || p.z< -halbelaenge) return false;
 return (p.x*p.x+p.y*p.y < r*r);
}

bool Zylinder::istausserhalb(Vektor3d p)
{
 double r=radius+ddr;
 double sinw1,cosw1,sinw2,cosw2;
 double halbelaenge=abs(achse)*0.5;
 p -= pos;
 drehwinkel_berechnen(&sinw1,&cosw1,&sinw2,&cosw2);
 p.drehenyx(sinw1,cosw1); p.drehenxz(sinw2,cosw2);
 if(p.z>halbelaenge || p.z< -halbelaenge) return true;
 return (p.x*p.x+p.y*p.y > r*r);
}

Vektor3d Zylinder::schnittpunkt(Vektor3d p1,Vektor3d p2) //p1 soll innerhalb des Zylinders liegen, p2 ausserhalb
{
 double A,a,b,c;
 Vektor3d p1a,v1,v2,q;           //v1,v2=Hilfsvektoren, q=gesuchter Schnittpunkt
 double sinw1,cosw1,sinw2,cosw2;
 double halbelaenge=abs(achse)*0.5;
 p1a = p1; //urspruenglicher Vektor brauchen wir am Schluss nochmals
 v1 = p2-p1;
 p1 -= pos; p2 -= pos; //wir setzen das Koordinatensystem so dass der Zylindermittelpunkt am Nullpunkt ist
 drehwinkel_berechnen(&sinw1,&cosw1,&sinw2,&cosw2); //und drehen so zurecht, dass Zylinderachse mit z-Achse uebereinstimmt
 p1.drehenyx(sinw1,cosw1); p1.drehenxz(sinw2,cosw2); p1.z=0;
 p2.drehenyx(sinw1,cosw1); p2.drehenxz(sinw2,cosw2); p2.z=0;
 // Jetzt Schnittpunkt (t) der Linie von p1 nach p2 mit dem Kreis suchen.
 // r^2 = t.x^2 + t.y^2
 // t.x = (p2-p1).x*A + p1.x
 // t.y = (p2-p1).y*A + p1.y
 // v2 = (p2-p1)
 // r^2 = (v2.x*A + p1.x)^2 + (v2.y*A + p1.y)^2
 //     = (v2.x^2+v2.y^2)*A^2 + 2*(v2.x*p1.x+v2.y*p1.y)*A + p1.x^2 + p1.y^2
 // 0 = (v2.x^2+v2.y^2)*A^2 + 2*(v2.x*p1.x+v2.y*p1.y)*A + p1.x^2 + p1.y^2 - r^2
 v2 = p2-p1;
 a = v2.x*v2.x+v2.y*v2.y;
 b = (v2.x*p1.x+v2.y*p1.y)*2;
 c = p1.x*p1.x+p1.y*p1.y-radius*radius;
 A = (sqrt(b*b-4*a*c)-b)/(2*a); //allgemeine Loesungsformel fuer Quadratische Gleichung (hier nur positives Resultat sinnvoll)
 q = v1*A + p1a;
 return q;
}

static Zylinder *zylinder_liste=NULL;
static int anzahl_zylinder=0;
static int j_aktueller_zylinder= -1;

void dreieck(FILE *fp,Vektor3d& p1,Vektor3d& p2,Vektor3d& p3,int j0=0)
{
 int j;
 Vektor3d q1,q2,normalenvektor;
 for(j=j0;j<anzahl_kugeln;j++)
  {
   if(j!=j_aktuelle_kugel)
    {
     // Es gibt 12 Moeglichkeiten wie ein Dreieck eine Kugel schneiden kann.
     // Begruendung: jeder der 3 Eckpunkte kann innerhalb, ausserhalb oder auf der Kugeloberflaeche liegen,
     // es gibt also total 3^3 = 27 Moeglichkeiten. Davon fallen die Varianten weg wo alle Punkte ausserhalb oder
     // auf der Oberflaeche liegen, also 3^2 = 8 Varianten. Auch die Varianten wo alle punkte innerhalb oder auf
     // der Oberflaeche liegen, also nochmals 3^2 = 8 fallen weg. Wobei wir jetzt die Variante wo alle Punkte auf
     // der Oberflaeche liegen 2 mal gezaehlt haben, also 1 weniger abziehen.
     // Also verbleiben 27-(2*8-1) = 12 Moeglichkeiten.
     // Hier also die entsprechenden if-Abfragen um alle Moeglichkeiten abzudecken.
     if(kugel_liste[j].istinnerhalb(p1))
      {
       if(kugel_liste[j].istausserhalb(p2))
	{
	 q1=kugel_liste[j].schnittpunkt(p1,p2);
	 if(kugel_liste[j].istausserhalb(p3)) //nur p1 innerhalb, p2 und p3 ausserhalb (1. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p1,p3);
	   dreieck(fp,q1,p2,p3,++j);
	   dreieck(fp,q1,p3,q2,j); return;
	  }
	 else if(kugel_liste[j].istinnerhalb(p3)) //p1 und p3 innerhalb, p2 ausserhalb (2. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p3,p2);
	   dreieck(fp,q1,p2,q2,++j); return;
	  }
	 else //p1 innerhalb, p3 auf der Kugeloberflaeche, p2 ausserhalb (3. Moeglichkeit)
	  {
	   dreieck(fp,q1,p2,p3,++j); return;
	  }
	}
       else if(kugel_liste[j].istinnerhalb(p2)) //p1 und p2 innerhalb
	{
	 if(kugel_liste[j].istausserhalb(p3)) //p1 und p2 innerhalb, p3 ausserhalb (4. Moeglichkeit)
	  {
	   q1=kugel_liste[j].schnittpunkt(p1,p3);
	   q2=kugel_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,q1,q2,p3,++j); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
       else //p1 innerhalb, p2 auf Kugeloberflaeche
	{
	 if(kugel_liste[j].istausserhalb(p3)) //p1 innerhalb, p2 auf Oberflaeche, p3 ausserhalb (5. Moeglichkeit)
	  {
	   q1=kugel_liste[j].schnittpunkt(p1,p3);
	   dreieck(fp,q1,p2,p3,++j); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
      } //Ende if istinnerhalb(p1)

     else if(kugel_liste[j].istinnerhalb(p2))
      {
       if(kugel_liste[j].istausserhalb(p1))
	{
	 q1=kugel_liste[j].schnittpunkt(p2,p1);
	 if(kugel_liste[j].istausserhalb(p3)) //nur p2 innerhalb, p1 und p3 ausserhalb (6. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,p1,q1,q2,++j);
	   dreieck(fp,p1,q2,p3,j); return;
	  }
	 else if(kugel_liste[j].istinnerhalb(p3)) //p2 und p3 innerhalb, p1 ausserhalb (7. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p3,p1);
	   dreieck(fp,p1,q1,q2,++j); return;
	  }
	 else //p2 innerhalb, p3 auf der Kugeloberflaeche, p1 ausserhalb (8. Moeglichkeit)
	  {
	   dreieck(fp,p1,q1,p3,++j); return;
	  }
	}
       else //p2 innerhalb, p1 auf Kugeloberflaeche
	{
	 if(kugel_liste[j].istausserhalb(p3)) //p2 innerhalb, p1 auf Oberflaeche, p3 ausserhalb (9. Moeglichkeit)
	  {
	   q1=kugel_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,p1,q1,p3,++j); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
      } //Ende if istinnerhalb(p2)

     else if(kugel_liste[j].istinnerhalb(p3))
      {
       if(kugel_liste[j].istausserhalb(p1))
	{
	 q1=kugel_liste[j].schnittpunkt(p3,p1);
	 if(kugel_liste[j].istausserhalb(p2)) //p3 innerhalb, p1 und p2 ausserhalb (10. Moeglichkeit)
	  {
	   q2=kugel_liste[j].schnittpunkt(p3,p2);
	   dreieck(fp,p1,p2,q2,++j);
	   dreieck(fp,p1,q2,q1,j); return;
	  }
	 else //p3 innerhalb, p2 auf Oberflaeche, p1 ausserhalb (11. Moeglichkeit)
	  {
	   dreieck(fp,p1,p2,q1,++j); return;
	  }
	}
       else if(kugel_liste[j].istausserhalb(p2)) //p3 innerhalb, p1 auf Oberflaeche, p2 ausserhalb (12. Moeglichkeit)
	{
	 q1=kugel_liste[j].schnittpunkt(p3,p2);
	 dreieck(fp,p1,p2,q1,++j); return;
	}
       else //alle Dreiecks-Punkte innerhalb der Kugel oder auf der Kugeloberflaeche
	{
	 return; //ignorieren wenn gesamtes Dreieck innerhalb einer andern Kugel
	}
      } //Ende if istinnerhalb(p3)
    } //Ende if j!=j_aktuelle_kugel
  } //Ende der for-j-Schlaufe

 // Gleiches fuer Ueberschneidungen mit Zylindern
 if(j_aktueller_zylinder<0) // aber nur Kugeln koennen einen Zylinder schneiden
  {
   j0=anzahl_kugeln; //es muss nicht nochmals auf Ueberschneidungen mit Kugeln geprueft werden
   for(j=0;j<anzahl_zylinder;j++)
   {
    if(zylinder_liste[j].istinnerhalb(p1))
      {
       if(zylinder_liste[j].istausserhalb(p2))
	{
	 q1=zylinder_liste[j].schnittpunkt(p1,p2);
	 if(zylinder_liste[j].istausserhalb(p3)) //nur p1 innerhalb, p2 und p3 ausserhalb (1. Moeglichkeit)
	  {
	   q2=zylinder_liste[j].schnittpunkt(p1,p3);
	   dreieck(fp,q1,p2,p3,j0);
	   dreieck(fp,q1,p3,q2,j0); return;
	  }
	 else if(zylinder_liste[j].istinnerhalb(p3)) //p1 und p3 innerhalb, p2 ausserhalb (2. Moeglichkeit)
	  {
	   q2=zylinder_liste[j].schnittpunkt(p3,p2);
	   dreieck(fp,q1,p2,q2,j0); return;
	  }
	 else //p1 innerhalb, p3 auf der Zylinderoberflaeche, p2 ausserhalb (3. Moeglichkeit)
	  {
	   dreieck(fp,q1,p2,p3,j0); return;
	  }
	}
       else if(zylinder_liste[j].istinnerhalb(p2)) //p1 und p2 innerhalb
	{
	 if(zylinder_liste[j].istausserhalb(p3)) //p1 und p2 innerhalb, p3 ausserhalb (4. Moeglichkeit)
	  {
	   q1=zylinder_liste[j].schnittpunkt(p1,p3);
	   q2=zylinder_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,q1,q2,p3,j0); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb des Zylinders oder auf der Zylinderoberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb eines andern Zylinders
	}
       else //p1 innerhalb, p2 auf Zylinderoberflaeche
	{
	 if(zylinder_liste[j].istausserhalb(p3)) //p1 innerhalb, p2 auf Oberflaeche, p3 ausserhalb (5. Moeglichkeit)
	  {
	   q1=zylinder_liste[j].schnittpunkt(p1,p3);
	   dreieck(fp,q1,p2,p3,j0); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb des Zylinders oder auf der Zylinderoberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb eines andern Zylinders
	}
      } //Ende if istinnerhalb(p1)

     else if(zylinder_liste[j].istinnerhalb(p2))
      {
       if(zylinder_liste[j].istausserhalb(p1))
	{
	 q1=zylinder_liste[j].schnittpunkt(p2,p1);
	 if(zylinder_liste[j].istausserhalb(p3)) //nur p2 innerhalb, p1 und p3 ausserhalb (6. Moeglichkeit)
	  {
	   q2=zylinder_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,p1,q1,q2,j0);
	   dreieck(fp,p1,q2,p3,j0); return;
	  }
	 else if(zylinder_liste[j].istinnerhalb(p3)) //p2 und p3 innerhalb, p1 ausserhalb (7. Moeglichkeit)
	  {
	   q2=zylinder_liste[j].schnittpunkt(p3,p1);
	   dreieck(fp,p1,q1,q2,j0); return;
	  }
	 else //p2 innerhalb, p3 auf der Zylinderoberflaeche, p1 ausserhalb (8. Moeglichkeit)
	  {
	   dreieck(fp,p1,q1,p3,j0); return;
	  }
	}
       else //p2 innerhalb, p1 auf Zylinderoberflaeche
	{
	 if(zylinder_liste[j].istausserhalb(p3)) //p2 innerhalb, p1 auf Oberflaeche, p3 ausserhalb (9. Moeglichkeit)
	  {
	   q1=zylinder_liste[j].schnittpunkt(p2,p3);
	   dreieck(fp,p1,q1,p3,j0); return;
	  }
	 else //alle Dreiecks-Punkte innerhalb des Zylinders oder auf der Zylinderoberflaeche
	   return; //ignorieren wenn gesamtes Dreieck innerhalb eines andern Zylinders
	}
      } //Ende if istinnerhalb(p2)

     else if(zylinder_liste[j].istinnerhalb(p3))
      {
       if(zylinder_liste[j].istausserhalb(p1))
	{
	 q1=zylinder_liste[j].schnittpunkt(p3,p1);
	 if(zylinder_liste[j].istausserhalb(p2)) //p3 innerhalb, p1 und p2 ausserhalb (10. Moeglichkeit)
	  {
	   q2=zylinder_liste[j].schnittpunkt(p3,p2);
	   dreieck(fp,p1,p2,q2,j0);
	   dreieck(fp,p1,q2,q1,j0); return;
	  }
	 else //p3 innerhalb, p2 auf Oberflaeche, p1 ausserhalb (11. Moeglichkeit)
	  {
	   dreieck(fp,p1,p2,q1,j0); return;
	  }
	}
       else if(zylinder_liste[j].istausserhalb(p2)) //p3 innerhalb, p1 auf Oberflaeche, p2 ausserhalb (12. Moeglichkeit)
	{
	 q1=zylinder_liste[j].schnittpunkt(p3,p2);
	 dreieck(fp,p1,p2,q1,j0); return;
	}
       else //alle Dreiecks-Punkte innerhalb des Zylinders oder auf der Zylinderoberflaeche
	{
	 return; //ignorieren wenn gesamtes Dreieck innerhalb eines andern Zylinders
	}
      } //Ende if istinnerhalb(p3)
   } //Ende der for-j-Schlaufe
  } //Ende if j_aktueller_zylinder

 q1=p2-p1;
 q2=p3-p2;
 normalenvektor=vektorkreuzprodukt(q1,q2);
 fprintf(fp," facet normal %.3f %.3f %.3f\n", normalenvektor.x, normalenvektor.y, normalenvektor.z);
 fprintf(fp,"  outer loop\n");
 fprintf(fp,"   vertex %.3f %.3f %.3f\n", p1.x*scalx, p1.y*scaly, p1.z*scalz);
 fprintf(fp,"   vertex %.3f %.3f %.3f\n", p2.x*scalx, p2.y*scaly, p2.z*scalz);
 fprintf(fp,"   vertex %.3f %.3f %.3f\n", p3.x*scalx, p3.y*scaly, p3.z*scalz);
 fprintf(fp,"  endloop\n");
 fprintf(fp," endfacet\n");
}

void viereck(FILE *fp,Vektor3d& p1,Vektor3d& p2,Vektor3d& p3,Vektor3d& p4)
{
 dreieck(fp,p1,p2,p3);
 dreieck(fp,p1,p3,p4);
}

/*********************** Dateien einlesen *****************************/
//const int MAXATOME=10000; //provi.

const int NELE=27*26;
typedef char STRING[80];

class Elemente  //uebernommen von Molz3d
{
 double radius[NELE];
 double vdwradius[NELE]; //Vanderwaals-Radi
 STRING farbe[NELE];
 int ind(int c1,int c2=0)
  {if(c1<'A' || c1>'Z') return 0;
   if(c2==0) return 26*(c1-'A');
   if(c2<'a' || c2>'z') return 0;
   return 26*(c1-'A')+c2-'a'+1;
  }
public:
 Elemente()
  {int n;
   //elemente_init();
   for(int i=0;i<NELE;i++) {vdwradius[i]=radius[i]=1.0; strcpy(farbe[i],"Plum");}
   // Atomradien in Angstroem (1Angstrom=0.1nm)
   radius[n=ind('C')]=0.772; //Kohlenstoff Radius in Angstrom (0.1nm)
   strcpy(farbe[n],"Gray50"); //Farbe fuer Povray
   radius[n=ind('H')]=0.373; strcpy(farbe[n],"White"); //Wasserstoff
   radius[n=ind('B')]=0.795; strcpy(farbe[n],"Gray60"); //Bor
   radius[n=ind('N')]=0.549; strcpy(farbe[n],"Blue"); //Stickstoff
   radius[n=ind('O')]=0.604; strcpy(farbe[n],"Red"); //Sauerstoff
   radius[n=ind('F')]=0.709; strcpy(farbe[n],"rgb<0.8,1,0.5>"); //Fluor
   radius[n=ind('N','a')]=1.858; strcpy(farbe[n],"Gray40"); //Natrium
   radius[n=ind('C','l')]=0.994; strcpy(farbe[n],"Green"); //Chlor
   radius[n=ind('B','r')]=1.145; strcpy(farbe[n],"rgv<0.7,0.7,0>"); //Brom
   radius[n=ind('I')]=1.331; strcpy(farbe[n],"rgb<0.5,1,1>"); //Jod
   radius[n=ind('J')]=1.331; strcpy(farbe[n],"rgb<0.5,1,1>"); //Jod
   radius[n=ind('S')]=1.035; strcpy(farbe[n],"Yellow"); //Schwefel
   radius[n=ind('P')]=1.105; strcpy(farbe[n],"rgb<1,0,1>"); //Phosphor
   // Van-der-Waals-Radien in Angstrom:
   vdwradius[n=ind('C')]=1.70; //Kohlenstoff
   vdwradius[n=ind('H')]=1.10; //Wasserstoff
   vdwradius[n=ind('B')]=1.92; //Bor
   vdwradius[n=ind('N')]=1.55; //Stickstoff
   vdwradius[n=ind('O')]=1.52; //Sauerstoff
   vdwradius[n=ind('F')]=1.47; //Fluor
   vdwradius[n=ind('N','a')]=2.27; //Natrium
   vdwradius[n=ind('C','l')]=1.75; //Chlor
   vdwradius[n=ind('B','r')]=1.85; //Brom
   vdwradius[n=ind('I')]=1.98; //Jod
   vdwradius[n=ind('J')]=1.98; //Jod
   vdwradius[n=ind('S')]=1.80; //Schwefel
   vdwradius[n=ind('P')]=1.80; //Phosphor
  }
 double radi(int c1,int c2) {return radius[ind(c1,c2)];}
 double radi_vdw(int c1,int c2) {return vdwradius[ind(c1,c2)];}
 char* farb(char *sym) {return farbe[ind(sym[0],sym[1])];}
};
static Elemente gelemente;

double atomradius(char *sym)
{
 return gelemente.radi(sym[0],sym[1]);
}

double atomradius_vdw(char *sym)
{
 return gelemente.radi_vdw(sym[0],sym[1]);
}

bool xyzeinlesen(FILE *fp)
{
 char atomname[80],sym[4],*s,c1,c2;
 int i, natome=0, nbindungen=0;
 double radius;
 double zylinderradius=2; //Zylinder-Radius in mm
 Vektor3d v;
 fscanf(fp,"%d %d",&natome,&nbindungen);
 //if(natome<1 || natome>MAXATOME)
 if(natome<1)
  {
   //if(natome>0) printf("Fehler: in xyz-Datei zu viele Atome. (%d)\n",natome); else 
   printf("Fehler: in xyz-Datei fehlende Anzahl Atome. (%d)\n",natome);
   return false;
  }
 kugel_liste=new Kugel[natome];
 if(argflag['S'] || argflag['T']) zylinder_liste=new Zylinder[nbindungen];
 for(i=0;i<natome;i++)
  {
   //fscanf(fp,"%s %lf %lf %lf",atomname,&x,&y,&z);
   fscanf(fp,"%s %lf %lf %lf",atomname,&v.x,&v.z,&v.y); //anderes Koordinatensystem fuer STL-Format
   c1=atomname[0]; c2=atomname[1];
   sym[0]=c1; sym[1]=c2; sym[2]=0;
   if(argflag['S']) radius=zylinderradius*1.5;
   else if(argflag['T']) radius=atomradius(sym)*10; 
   else radius=atomradius_vdw(sym)*10; // 10 mm soll 1 Angstroem entsprechen
   v *= 10.0; //Vektor auch in mm umrechnen
   kugel_merken(v,radius);
  }
 if(argflag['S'] || argflag['T'])
  {
   char zeile[40];
   int n,n1,n2,n3;
   Vektor3d pos1,pos2;
   getline(fp,zeile,40); //letzte Zeile von Atompositionen fertig einlesen
   for(i=0;i<nbindungen;i++)
    {
     if(getline(fp,zeile,40))
      {
       if((n=sscanf(zeile,"%d %d %d",&n1,&n2,&n3))>=2)
	{
	 pos1=kugel_liste[n1].pos;
	 pos2=kugel_liste[n2].pos;
	 if(argflag['V'] && n==3) printf("Mehrfachbindungen noch nicht ausgewertet (%d %d %d)\n",n1,n2,n3);//test
	 zylinder_liste[anzahl_zylinder++].init((pos1+pos2)*0.5,zylinderradius,pos2-pos1);
	}
       else printf("Fehler in xyz-Datei: fehlende Zahl fuer Bindungsbeschreibung: '%s'\n",zeile);
      }
     else printf("Fehler in xyz-Datei: fehlende Zeile fuer Bindung.\n");
    }
  }
 return true;
}

/************************* Hauptprogramm ******************************/
main(int argc,char *argv[])
{
 char quellname[80],zielname[80],antw[40];
 FILE *fp1,*fp2;
 int i,j,c;
 quellname[0]=zielname[0]=0;
 if(argc<=0)
  {/* es wurde von WorkBench (GUI, Desktop, ...) gestartet */
   workbenchflag=1;
  }
 else
  {/* es wurde von Shell (Terminal, Konsole, ...) gestartet */
   for(j=0,i=1;i<argc;i++)
	{if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
	 else	{if(++j==1) strcpy(quellname,argv[i]);
	         else if(j==2) strcpy(zielname,argv[i]);
	}	}
  }
 if(argflag['?'] || j<1 || j>MAXARG)
	{printf("xyzstl  %s\n",VERSION);
	 printf("Anwendung: xyzstl Quelle.xyz [Ziel.stl] [-Flags]\n");
	 printf("  Flags: a=Aufloesung setzen (Schrittweit in ganzen Grad)\n");
	 printf("         b=Binaer speichern\n");
	 printf("         s=Stab-Modell\n");
	 printf("         t=Kugeln und Staebe (Ball and Sticks)-Modell\n");
	 printf("         k=Kalotten-Modell (Voreinstellung)\n");
	 printf("         v=Verbose, mit Testausgaben\n");
	 printf("Beispiel: xyzstl alanin.xyz alanin.stl -a10 -vbk\n");
	 exit(0);
	}
 if(argflag['S'])
  {
   if(argflag['T'] || argflag['K']) {printf("Fehler: nur eine der drei Optionen s t oder k erlaubt.\n"); exit(0);}
   if(argflag['V']) printf("Stab-Modell\n");
  }
 else if(argflag['T'])
  {
   if(argflag['K']) {printf("Fehler: nur eine der drei Optionen s t oder k erlaubt.\n"); exit(0);}
   if(argflag['V']) printf("Kugeln-und-Staebe-Modell\n");
  }
 else if(argflag['V']) printf("Kalotten-Modell\n");
 if(gradstep<1) gradstep=1;
 else if(gradstep>30) gradstep=30;
 else while(180/gradstep*gradstep != 180) gradstep++;
 if(argflag['V']) printf("gradstep=%d\n",gradstep);
 fp1=fopen(quellname,"r");
 if(fp1==NULL) {printf("kann '%s' nicht oeffnen.\n",quellname); exit(0);}
 if(xyzeinlesen(fp1)==false) {printf("konnte Molekuel nicht einlesen.\n"); exit(0);}
 if(*zielname==0)
  {
   endungersetzen(zielname,quellname,".stl");
   if((fp2=fopen(zielname,"r"))!=NULL)
    {
     fclose(fp2);
     printf("'%s' schon vorhanden. Ueberschreiben?",zielname); scanf("%s",antw);
     if(!istja(antw)) {printf("nein - nichts gespeichert.\n"); return 0;}
    }
  }
 fp2=fopen(zielname,"w");
 if(fp2==NULL) {printf("konnte '%s' nicht erstellen.\n",zielname); exit(0);}
 fprintf(fp2,"solid unnamed\n");
 alle_kugeln_speichern(fp2);
 if(argflag['S'] || argflag['T']) alle_zylinder_speichern(fp2);
 fprintf(fp2,"endsolid unnamed\n");
 fclose(fp2);
 if(kugel_liste!=NULL) delete[] kugel_liste;
 if(zylinder_liste!=NULL) delete[] zylinder_liste;
 return 0;
}/* ende von main */

void alle_kugeln_speichern(FILE *fp2)
{
 int j;
 for(j=0;j<anzahl_kugeln;j++)
  {
   j_aktuelle_kugel=j;
   kugel_speichern(fp2, kugel_liste[j].pos, kugel_liste[j].getradius());
  }
 j_aktuelle_kugel= -1;
}

void alle_zylinder_speichern(FILE *fp2)
{
 int j;
 for(j=0;j<anzahl_zylinder;j++)
  {
   j_aktueller_zylinder=j;
   zylinder_speichern(fp2, zylinder_liste[j].pos, zylinder_liste[j].getradius(), zylinder_liste[j].achse);
  }
 j_aktueller_zylinder= -1;
}

void kugel_merken(Vektor3d& kugelmittelpunkt,double r)
{
 int i=anzahl_kugeln;
 //if(i==MAXATOME) {fprintf(stderr,"zu viele Kugeln\n"); return;} //provi.
 kugel_liste[i].init(kugelmittelpunkt,r);
 anzahl_kugeln++;
}

void kugel_speichern(FILE *fp,Vektor3d& kugelmittelpunkt,double r)
{
 //static int ntekugel=0;
 Vektor3d p1,p2,p3,p4,v1,v2,normalenvektor;
 int    theta0,phi0,dtheta0,dphi0; //Werte in Grad
 double theta, phi, dtheta, dphi, sintheta,sintheta2,costheta2; //Werte in Rad
 //ntekugel += 1;
 dtheta0=dphi0=gradstep;
 for(theta0=180;theta0>0;theta0-=dtheta0) //gehe vom Suedpol zum Nordpol in ganzzahligen Schritten
  for(phi0= 0;phi0<360;phi0+=dphi0)       //gehe einmal rundrum
   {
    theta=theta0*GRAD; phi=phi0*GRAD; dtheta=dtheta0*GRAD; dphi=dphi0*GRAD;
    sintheta=sin(theta);
    p1=p2=p3=kugelmittelpunkt;
    //printf("theta = %d Grad = %lf, phi = %d Grad = %lf\n",theta0,theta,phi0,phi);//test
    if(theta0==180)
     { //einzelnes Dreieck am Suedpol
      sintheta2=sin(theta-dtheta);
      costheta2=cos(theta-dtheta);
      p1.z -= r;
      p2.x += r*sintheta2*cos(phi);
      p2.y += r*sintheta2*sin(phi);
      p2.z += r*costheta2;
      p3.x += r*sintheta2*cos(phi+dphi);
      p3.y += r*sintheta2*sin(phi+dphi);
      p3.z = p2.z;
      dreieck(fp,p1,p3,p2);
     }
    else if(theta0==dtheta0)
     { //einzelnes Dreieck am Nordpol
      p1.z += r;
      p2.x += r*sintheta*cos(phi);
      p2.y += r*sintheta*sin(phi);
      p2.z += r*cos(theta);
      p3.x += r*sintheta*cos(phi+dphi);
      p3.y += r*sintheta*sin(phi+dphi);
      p3.z = p2.z;
      dreieck(fp,p1,p2,p3);
     }
    else
     {// wenn nicht am Pol dann jeweils ein Viereck
      // Formeln aus  http://de.wikipedia.org/wiki/Kugelkoordinaten
      sintheta2=sin(theta-dtheta);
      costheta2=cos(theta-dtheta);

      p1.x += r*sintheta*cos(phi);
      p1.y += r*sintheta*sin(phi);
      p1.z += r*cos(theta);

      p2.x += r*sintheta*cos(phi+dphi);
      p2.y += r*sintheta*sin(phi+dphi);
      p2.z = p1.z;

      p3.x += r*sintheta2*cos(phi+dphi);
      p3.y += r*sintheta2*sin(phi+dphi);
      p3.z += r*costheta2;

      p4=kugelmittelpunkt;
      p4.x += r*sintheta2*cos(phi);
      p4.y += r*sintheta2*sin(phi);
      p4.z = p3.z;
      viereck(fp,p1,p2,p3,p4);
     }
   }
}

void zylinder_speichern(FILE *fp,Vektor3d pos,double zylinderradius,Vektor3d achse)
{
 Vektor3d p1,p2,p3,p4,v1,v2,normalenvektor;
 int    phi0,dphi0; //Werte in Grad
 double phi, dphi, sintphi, cosphi; //Werte in Rad
 double drehwinkel1,drehwinkel2,sinw1,cosw1,sinw2,cosw2;
 double halbelaenge=abs(achse)*0.5;
 drehwinkel1 = atan2(achse.y,achse.x); //Winkel um den gedreht werden muss damit achse in die x-z-Ebene zu liegen kommt
 sinw1=sin(drehwinkel1); cosw1=cos(drehwinkel1);
 achse.drehenyx(sinw1,cosw1);
 drehwinkel2 = atan2(achse.x,achse.z); //Winkel um den weiter gedreht werden muss damit achse auf die z-Achse zu liegen kommt
 sinw2=sin(drehwinkel2); cosw2=cos(drehwinkel2);
 //so zurueckdrehen:
 // p.drehenzx(sinw2,cosw2);
 // p.drehenxy(sinw1,cosw1);
 dphi0=gradstep;
 for(phi0=0;phi0<360;phi0+=dphi0)       //gehe einmal rundrum
  {
   if(phi0==0)
    {
     p1.x = p4.x = zylinderradius;
     p1.y = p4.y = 0;
     p4.z = halbelaenge;
     p1.z = -halbelaenge;
     p1.drehenzx(sinw2,cosw2); p1.drehenxy(sinw1,cosw1);
     p4.drehenzx(sinw2,cosw2); p4.drehenxy(sinw1,cosw1);
     p1 += pos;
     p4 += pos;
    }
   else {p1=p2; p4=p3;}
   phi=(phi0+dphi0)*GRAD;
   p2.x = p3.x = zylinderradius*cos(phi);
   p2.y = p3.y = zylinderradius*sin(phi);
   p3.z = halbelaenge;
   p2.z = -halbelaenge;
   p2.drehenzx(sinw2,cosw2); p2.drehenxy(sinw1,cosw1);
   p3.drehenzx(sinw2,cosw2); p3.drehenxy(sinw1,cosw1);
   p2 += pos;
   p3 += pos;
   viereck(fp,p1,p2,p3,p4);
  }
}
