/* tarech.cc       letzte Aenderung: 5.12.2010
Selbstbau-Taschenrechner
*/
#define VERSION "0.08"
/*
 Prozessor: Atmega1284P, 16MHz Quarz
 Schaltung: tarech.png, tastatur5x6/tastatur5x6.png
   die 3 LEDs auf PB0...PB2 sind noch unbenutzt (nicht angeschlossen)

   LCD-Anzeige:
   Text-LCD: 4-Zeilen LCD an PortA (in aktueller Version benutzt)
   (Erweiterung mit Grafik-LCD koennte mit PortA + PortB gemacht werden)

   Tastatur: PD0 (RxD)

   Akku-Spannung messen: PA0 (ADC0) 47k auf GND, 75k auf VCC
                         somit Akkuspannung = 2.6 * gemessene Spannung

   Echtzeituhr DS1307: PC0 (SCL) und PC1 (SDA) jetzt fuer I2C benutzt.
   myUSBtoUART: PD2 (RxD1), PD3 (TxD1)  (nur benutzbar ab ATmega1284P)

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 19.9.2010	Erstellung, ATmega32, Uhr von Lichtwecker uebernommen (vereinfacht)
 6.11.          neue Tastenbelegung
 7.11.    0.04  Fliesszahl-Klasse, Wechsel auf ATmega644
 14.11.   0.05  Gold-Taste nicht mehr global aktiv, Gold und Blue schliessen sich jetzt
                gegenseitig aus. Gold-Red und Blue-Red fuer weitere Spezialfunktionen reserviert.
		Echtzeituhr (RTC) eingebaut. Provisorische Version eines Kalenders.
 22.11.   0.06  e^x und ln(x)
 27.11.   0.07  Winkelfunktionen (sin, cos, tan, und Umkehrfunktionen asin, acos, atan)
 5.12.10  0.08  Anpassung an ATmega1284P

*/
#include <avr/io.h>
//#include <avr/iom1284p.h>  //test
#include <avr/interrupt.h>
#include <stdio.h>
//#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>
#include "ulong.h"
#include "fliesszahlklasse.cc"

#define F_CPU 16000000UL
#define UARTVARIANTE //Methode fuer Tastatur-Anschluss
//#define GRAFIK_LCD //Wenn Grafik-LCD angeschlossen ist
#define TEXT_LCD   //Sonst mit Text-LCD

//#define RTCDEBUG  //zur Fehlersuche

/**** Vordeklarationen ****/
void tastaturcheck();
char tastaturabfrage();
//void variable_feiertage_berechnen();
void glcd_printzeile(int zeile,const char *cstr,
		     int p1=0,int p2=0,int p3=0,int p4=0);
void tarech_init();
void tarech_tastenreaktion(uchar taste);
void tarech_refresh();
//Fliesszahl calcln2();
Fliesszahl log(Fliesszahl x);  //natuerlicher Logarithmus
Fliesszahl exp(Fliesszahl x);  //e^x

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=35,stunden=19;
static volatile uchar tage=20,monat=11; //aktuelles Datum
static volatile uint jahr=2010;
static char monatstage[13]={31,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=6; //0=Montag ... 6=Sonntag
static volatile char sommerzeit=0; //0=Winterzeit 1=Sommerzeit

void schaltjahrtest()
{
 if(jahr%4==0 && (jahr%100!=0 || jahr%400==0)) monatstage[2]=29;
 else monatstage[2]=28;
}
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
void schaltjahr_und_wochentagberechnen()
{
 schaltjahrtest();
 wochentagberechnen();
 //variable_feiertage_berechnen();
}

/** Feiertage: ** /
const char NFEST=6;//Anzahl feste Feiertage
const char festfeiertag[NFEST]={1, 2, 1, 1, 25, 26};
const char festfeiermon[NFEST]={1, 1, 5, 8, 12, 12};
const char NVARI=4;//Anzahl von Ostern abhaengige Feiertage
static char varifeiertag[NVARI];
static char varifeiermon[NVARI];
void variable_feiertage_berechnen()
{
 int a,b,c,k,p,q,M,N;
 char d,e,tag,i;
 const char mon=3;
 //Osterformel nach Gauss:
 a=jahr%19;
 b=jahr%4;
 c=jahr%7;
 k=jahr/100;
 p=(8*k+13)/25;
 q=k/4;
 M=(15+k-p-q)%30;
 N=(4+k-q)%7;
 d=(19*a+M)%30;
 e=(2*b+4*c+6*d+N)%7;
 tag=22+d+e;
 varifeiertag[0]=tag+1; varifeiermon[0]=mon; //Ostermontag
 varifeiertag[1]=tag-2; varifeiermon[1]=mon; //Karfreitag
 varifeiertag[2]=tag+39; varifeiermon[2]=mon; //Auffahrt
 varifeiertag[3]=tag+50; varifeiermon[3]=mon; //Pfingstmontag
 for(i=0;i<NVARI;i++)
   while(varifeiertag[i]>monatstage[varifeiermon[i]])
    {varifeiertag[i] -= monatstage[varifeiermon[i]]; varifeiermon[i]++;}
}

bool istfeiertag()
{
 char i;
 for(i=0;i<NFEST;i++)
   if(tage==festfeiertag[i] && monat==festfeiermon[i]) return true;
 for(i=0;i<NVARI;i++)
   if(tage==varifeiertag[i] && monat==varifeiermon[i]) return true;
 return false;
}
/ **/

// Quarz moeglichst genau ausmessen und hier eintragen
// in Hertz und Tausendstel Hertz:

//const ulong Hz=16000000,MilliHz=0; //exakt 16MHz
//Beispiel: 16MHz ist nach n Tagen z sec vorgegangen:
//nach n Tagen sind also 16000000*(3600*24*n+z) Takte vergangen
//Frequenz ist also:
// x = (3600*24*n+z)*16000000 / (3600*24*n)
//   = 16000000 + z*16000000/(3600*24*n)
const ulong Hz=16002000,MilliHz=0;

const uint Rest1 = Hz%1000;//jede Sekunde zu korrigierender Fehler
//const uint Rest2 = MilliHz*6/100;//jede Minute
//const uint Rest3 = ((MilliHz*6-Rest2*100)*60+50)/100;//jede Stunde
const uint NormalerCompwert=Hz/1000-1;
//der Compiler sollte bis hier noch keinen Code erzeugt haben.
//(der Quarz ist wahrscheinlich zu wenig stabil als dass Rest2 und Rest3
//eine Rolle spielen wuerden. Also MilliHz vorlaeufig auf 0 lassen.)

//die Namen *_vect sind in /usr/lib/avr/include/avr/iom1284p.h zu finden
ISR(TIMER1_COMPA_vect)
{
 uchar mehr;
 uint ms=millisec;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0;
   //if(++sec>=60-Rest2) mehr=1; else mehr=0;
   mehr=0;//einfacher wenn Rest2 und Rest3 sowieso ==0
   if(++sec==60)
    {sec=0;
     //if(++min>=60-Rest3) mehr=1; else mehr=0;
     if(++min==60)
      {min=0; mehr=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
      }
    }
  }
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
 tastaturcheck();
}

//Bei einem unerwarteten Interrupt nichts machen:
//SIGNAL (__vector_default) {}
//Oder besser wirklich einfach nur ein reti machen:
void __attribute__ ((naked)) __vector_default(void) {__asm__ ("reti");}
//Ohne diese Funktion wird nach 0 gesprungen, was einem Reset entspricht.

void milliwait(int n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) ;//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
   ;//auf n. Millisekunde warten
}

void secwait(int8 n) //n Sekunden warten
{
 while(--n>=0) milliwait(1000);
}

/*************** Routinen fuer AD-Wandler ****************/
//#define F_ADC 200000
//#define ADPRESCALER int(log2(F_CPU/F_ADC)+0.999)
#define ADPRESCALER 7 //entspricht einem Vorteiler von 2^7 = 128 --> 16MHz/128 = 125kHz

void adwandler_init()
{
 //fuer ATmega644:
 DIDR0 |= (1<<ADC0D); //Digital-Input des ADC0-Pins ausschalten
 ADMUX = (1<<REFS1) | (1<<REFS0) | 0; //Referenz 2.534V, ADC0
 ADCSRA = (1<<ADEN) | (1<<ADSC) | ADPRESCALER; //Einschalten und erste Messung machen
 while (ADCSRA & (1<<ADSC)) ;//Ende der Wandlung abwarten
}

int adwandler() //einzelne Messung machen, Rueckgabe im Bereich 0...1023
{
 ADCSRA |= (1<<ADSC); //Messung starten
 while (ADCSRA & (1<<ADSC)) ;//Ende der Wandlung abwarten
 return ADC;
}

/************** Routinen fuer LCD-Ansteuerung ************/
char microwait(int n) //etwa n Microsekunden warten
{
 char vx=0;
 while(n>0)
  {vx+=n; --n;}
 return vx;
}

#ifdef GRAFIK_LCD
#include "glcd.cc"
#else
#include "lcd4zeilen.cc"
inline void glcd_init() {lcd_init();}
inline void glcd_goto(uchar zeile,uchar spalte) {lcd_goto(zeile,spalte);}
inline void glcd_write(char c) {lcd_write(c);}
inline void glcd_write(const char *txt) {lcd_write(txt);}
#endif

void glcd_writedd(const char *cstr,int p1,int p2)
{
 char glcd_puffer[40];
 sprintf(glcd_puffer,cstr,p1,p2);
 glcd_write(glcd_puffer);
}

void lcd_gowrite(char zeile,char spalte,const char* text)
{
 glcd_goto(zeile,spalte); glcd_write(text);
}

void lcd_gowrite20(char zeile,const char* text)
{
 char txt[22],*s,i;
 for(i=0,s=txt; (*s = *text++);s++,i++) ;
 for(;i<20;i++) *s++ = ' ';//Zeile mit Leerstellen auffuellen
 *s=0;
 lcd_goto(zeile,1); glcd_write(txt);
}

static int8 hexmodus=0;

int48 divmod(int8 *prest,int48 x,int8 teiler)
{
 //diese Routine koennte man mit Assembler wesentlich optimieren
 int48 result=x/teiler;
 x %= teiler;
 *prest = getint8(x);
 return result;
}

void string_umkehren(char *str)
{
 int8 n=strlen(str),c;
 char *s,*p;
 for(s=str,p= &str[n-1]; s<p; )
   {c = *s;
    *s++ = *p;
    *p-- = c;
   }
}

const char Hochzeichen=0x1D; //0x1D oder '^' oder ... abhaengig von verwendetem LCD-Display

void hochzeichen_ersetzen(char *s)
{
 for(;*s;s++) {if(*s=='^') *s=Hochzeichen;}
}

const Fliesszahl einhalbes(1,2);

void sprintf_d(char *str,Fliesszahl x)
{
 long ix,dx,dxmax;
 int ex=0;
 bool isneg;
 char vorz=' ';
 int8 zehn=(hexmodus)?16:10, logdxmax, logdxmax3, rest;
 int48 ix48;
 Fliesszahl einhunderstel(1,100),tausend(1000),zigtausend(99000);
 //printf("Testpunkt1: x = %s\n",x.str());
 isneg = x.isnegativ();
 if(isneg) {vorz='-'; x = -x;}
 ix48=getint48(x);
 if(x==ix48) //wenn es eine Ganze Zahl ist
   {char str1[24], *p1=str1;
    do
      {ix48 = divmod(&rest,ix48,zehn);
       if(rest>9) *p1++ = rest + 'A' - 10;
       else       *p1++ = rest + '0';
      }
    while(ix48!=0);
    *p1 = 0;
    string_umkehren(str1);
    sprintf(str,"%c%16s",vorz,str1);
    return;
   }
 if(x.istnan())
   {sprintf(str,"NAN");
    return;
   }
 if(hexmodus)
   {einhunderstel=Fliesszahl(1,256);
    tausend=Fliesszahl(0x1000);
    zigtausend=Fliesszahl(0xFF000);
    dxmax=0x1000000; logdxmax=6; logdxmax3=3*6;
   }
 else
   {//einhunderstel=Fliesszahl(1,100);
    //tausend=Fliesszahl(1000);
    //zigtausend=Fliesszahl(99000);
    dxmax=1000000000; logdxmax=9; logdxmax3=3*9;
   }
 if(x<=einhunderstel)
   {
    while(x.getexp() < -logdxmax3)  //fuer Geschwindigkeitsoptimierung bei grossen Exponenten
       {ex -= logdxmax; x *= dxmax;}
    while(x < 1)
      {ex -= 3; x *= tausend;} //kleine Zahlen fuer Darstellung der Form 12.3e-6 vorbereiten
   }
 else if(x>=zigtausend)
   {
    while(x.getexp() > logdxmax3)  //fuer Geschwindigkeitsoptimierung bei grossen Exponenten
       {ex += logdxmax; x /= dxmax;}
    while(x >= tausend)
      {ex += 3; x /= tausend;} //grosse Zahlen fuer Darstellung der Form 12.3e6 vorbereiten
   }
 //printf("Testpunkt2: x = %s ex=%d\n",x.str(),ex);
 ix=getint32(x);
 x -= ix;
 dx = getint32(x*dxmax+einhalbes); // +0.5 ist fuer korrekte Rundung
 if(dx==dxmax)               //wenn beim Runden auf eine Ganze Zahl aufgerundet wurde
   {dx=0;  ix++;}                 //dann entsprechend den Ganzzahligen Anteil korrigieren
 if(hexmodus)
   {if(ex==0)     sprintf(str,"%c%5lX.%06lX ",vorz,ix,dx);
    else if(ex>0) sprintf(str,"%c%3lX.%06lX*10%c%X ",vorz,ix,dx,Hochzeichen,ex);
    else          sprintf(str,"%c%3lX.%06lX*10%c-%X ",vorz,ix,dx,Hochzeichen,-ex);
   }
 else
   {if(ex==0) sprintf(str,"%c%5ld.%09ld ",vorz,ix,dx);
    else      sprintf(str,"%c%3ld.%09lde%d ",vorz,ix,dx,ex);
   }
}

void lcd_gowrite_d(char zeile,Fliesszahl x)
{
 char text[24];
 sprintf_d(text,x);
 lcd_gowrite20(zeile,text);
}

//Globale Variablen:
int8 hauptmodus=0;
const int8 STARTMODUS=0,TASCHENRECHNER=1,KALENDER=2;//Gueltige Werte fuer modus

void zeitauflcdanzeigen()
{
 char txt[40];
 glcd_goto(4,1);
 sprintf(txt,"%02d:%02d:%02d ",stunden,min,sec);
 glcd_write(txt);
 sprintf(txt,"%02d.%02d.%d",tage,monat,jahr);
 glcd_write(txt);
}

void batteriespannung_auf_lcd_anzeigen()
{
 int adwert,vorkomma,nachkomma;
 const int vref=2534; //Referenzspannung des AD-Wandlers (idealerweise nachmessen)
 adwert=adwandler(); //Messwert Bereich 0...1023
 adwert = ( ((long)adwert*vref+512)/1024 //Messwert in Millivolt umrechnen
	    * (75+47) + 47/2 ) / 47;     //Spannungsteiler 75k zu 47k beruecksichtigen
 vorkomma  = adwert/1000;
 nachkomma = adwert%1000;
 glcd_goto(3,1); glcd_write("Batt: "); glcd_writedd("%d.%03d V",vorkomma,nachkomma);
}

#ifdef UARTVARIANTE
void uart_init();
#endif

/****************** Dinge zum Uhr stellen und in RTC speichern *************************/
#ifdef RTCDEBUG
void lcd_printf3(const char *cstr,int n1,int n2,int n3)
{
 char buffer[40];
 sprintf(buffer,cstr,n1,n2,n3);
 lcd_write(buffer);
}

void lcd_goprintf3(int zeile,int spalte,const char *cstr,int n1,int n2,int n3)
{
 lcd_goto(zeile,spalte);
 lcd_printf3(cstr,n1,n2,n3);
}

void testpunkt(const char *cstr,int n)
{
 lcd_goprintf3(1,1,cstr,n,0,0);
 milliwait(1000);
}

void lcd_printerror(uchar err)
{
 int8 i;
 lcd_goprintf3(1,1,"err=%d  ",err,0,0);
 for(i=0;i<10;i++) milliwait(1000);
}
#endif

#include "i2c.c"
#include "rtc.c"
static int JAHRHUNDERT=2000;
static char rtc[10]; // {Sekunden, Minuten, Stunden, Wochentag, Tag, Monat, Jahr, Control, Jahrhund., Jahr}

char bin2bcd(char c)
{
 return ((c/10)<<4)+c%10;
}

char bcd2bin(char c)
{
 return ((c>>1)&0x78)+((c>>3)&0x1E)+(c&0x0F);
}

void zeit_in_rtc_speichern()
{
 uchar err, sek;
 for(sek=sec;sek==sec;) ;//auf Sekundenwechsel warten
 rtc[0] = bin2bcd(sec);
 rtc[1] = bin2bcd(min);
 rtc[2] = bin2bcd(stunden);
 rtc[3] = wochentag;
 rtc[4] = bin2bcd(tage);
 rtc[5] = bin2bcd(monat);
 rtc[6] = bin2bcd(jahr%100);
 rtc[7] = 0x00; //0x80=OUT=Level wenn SQWE nicht gesetzt, 0x10=SQWE=Frequenz an Pin 7 ausgeben, 0x03=RS
                //RS definiert die Frequenz wenn SQWE gesetzt ist: 0=1Hz, 1=4096, 2=8192, 3=32768 Hz
 //rtc[7] = (min&1)?0x13:0x00; //test: 32768 Hz Ausgabe wenn bei ungerader Minute gespeichert wird
 rtc[8] = jahr/100; //so koennte man noch das Jahrhundert speichern.
 rtc[9] = jahr%100; //das aktuell Jahr muesste man aber auch noch mit speichern.
 if(rtc[0] != bin2bcd(sec)) err=5; //falls Sekunde inzwischen gewechselt hat ist etwas faul.
 else err = rtc_write(rtc,10);
#ifdef RTCDEBUG
 if(err) lcd_printerror(err);
 else lcd_gowrite(4,1,"in RTC gespeichert");
#endif
}

void zeit_von_rtc_lesen()
{
 uchar err=0;
 char sek,sek2;
 //char wotag; //nicht wirklich gebraucht
 sek2 = sek = rtc_readsecond();
 if(sek==ERROR_CODE) err=2;
 else
  for(int i=0;sek2==sek && i<1000;i++)
      {milliwait(2);
       sek2 = rtc_readsecond(); //auf Sekundenwechsel warten
       if(sek2==ERROR_CODE) {err=3; break;}
      }
 cli(); millisec=0; sei(); //millisec auf Sekundenwechsel der RTC synchronisieren
 if(!err)
      {sec = bcd2bin(sek2);
       milliwait(2);
       err = rtc_read(rtc,10); //gesamte Zeit und Datum von RTC einlesen
       if(!err && (sec != bcd2bin(rtc[0]))) err=4; //Wenn Sekunde inzwischen gewechselt hat ist was faul.
      }
 if(err)
      {
#ifdef RTCDEBUG
       lcd_printerror(err);
#endif
      }
 else
      {min     = bcd2bin(rtc[1]);
       stunden = bcd2bin(rtc[2]);
       //wotag   = rtc[3]; if(wotag>6) wotag=6;
       tage    = bcd2bin(rtc[4]);
       monat   = bcd2bin(rtc[5]);
       JAHRHUNDERT = rtc[8]*100;                  //wenn das Jahrhundert gespeichert wurde
       if(rtc[6]==0 && rtc[9]==99) JAHRHUNDERT++; //ueberpruefen ob inzwischen kein Jahrhundertwechsel war
       // geht aber nur wenn beim Abschalten immer die RTC beschrieben wird, zumindest jedes Jahr mal
       jahr    = JAHRHUNDERT + bcd2bin(rtc[6]);
#ifdef RTCDEBUG
       lcd_gowrite(1,1,"RTC eingelesen");
#endif
      }
 schaltjahr_und_wochentagberechnen();
 //if(wotag!=wochentag) irgendwas_falsch();
}

/********************************** Hauptprogramm *************************************/
int main(void)
{
 char c=' ',pos=1,startsec;
 int8 i;

#ifdef ATMEGA32
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
#else
 //Timer-Einstellungen fuer ATmega644:
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
#endif

 sei(); //Interrupt einschalten

 milliwait(50);
 glcd_init();
 adwandler_init();
#ifdef UARTVARIANTE
 uart_init();
#endif
 twi_init();

 lcd_gowrite(1,1,"Tarech Version "); glcd_write(VERSION);
 lcd_gowrite(2,1,"Press Gold to start");//test
 batteriespannung_auf_lcd_anzeigen();
 zeit_von_rtc_lesen();
 zeitauflcdanzeigen();
 startsec = sec+5; if(startsec>=60) startsec -= 60;

 while(1) //Hauptschlaufe
  {//int altsec=0;
   if(hauptmodus==STARTMODUS)
    {//if(altsec!=sec) {zeitauflcdanzeigen(); altsec=sec;}
     batteriespannung_auf_lcd_anzeigen();
     zeitauflcdanzeigen();
     if(sec==startsec) c='a'; else c=tastaturabfrage();
     if(c!=0) //bei Tastendruck oder 5 Sec nach einschalten in Taschenrechner-Modus wechseln
      {glcd_goto(1,pos); //Test: Tastatureingabe auf 1.Zeile darstellen
       glcd_write(c);
       if(++pos==21) pos=1;
       if(c=='a')
	{pos=1; hauptmodus=TASCHENRECHNER;
	 tarech_init();
	 tarech_tastenreaktion('='); //Starten mit "Enter"
	}
      }
    }
   else if(hauptmodus==TASCHENRECHNER)
    {
     if((c=tastaturabfrage())!=0)
       tarech_tastenreaktion(c);
    }
   else if(hauptmodus==KALENDER)
    {
     static int tagoffset=0;
     lcd_gowrite(1,1,"Mo Di Mi Do Fr Sa So ");
     lcd_goto(2,1);
     for(i=0;i<14;i++)
       {int tg = tage+i-wochentag+tagoffset;
	int8 mo = monat;
	while(tg<1) {tg += monatstage[--mo]; if(mo==0) mo=12;} //provi. (noch Jahr beruecksichtigen)
	while(tg>monatstage[mo])
	            {tg -= monatstage[mo++]; if(mo==13) mo=1;} //provi. (noch Jahr beruecksichtigen)
	if(i==7) lcd_goto(3,1);
	glcd_writedd("%2d ",tg,0);
       }
     zeitauflcdanzeigen();
     c=tastaturabfrage();
     if(c=='a')
	{hauptmodus=TASCHENRECHNER;
	 tarech_refresh();
	}
     switch(c)     //Tastenauswertung fuer Kalender-Modus
       {case '+': tagoffset += 14; //naechste 2 Wochen darstellen
        CASE '-': tagoffset -= 14; //vorherige 2 Wochen darstellen
        CASE '=': tagoffset = 0; //aktuelle Woche
       }
    }
  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

/*** Abfrage der Matrix-Tastatur (mit Tiny2313 aufgebaut) ***/
#ifdef UARTVARIANTE

#ifndef F_CPU
#warning "F_CPU war noch nicht definiert, wird nun nachgeholt mit 16000000"
#define F_CPU 16000000UL  // Systemtakt in Hz - Definition als unsigned long
                         // Ohne ergeben sich unten Fehler in der Berechnung
#endif
 
#define BAUD 8192UL      // Baudrate
 
// Berechnungen
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1)   // clever runden
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     // Reale Baudrate
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD) // Fehler in Promille, 1000=kein Fehler

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
  #error Systematischer Fehler der Baudrate groesser 1% und damit zu hoch! 
#endif

void uart_init()
{
#ifdef ATMEGA32
 UBRRH = UBRR_VAL >> 8;
 UBRRL = UBRR_VAL & 0xFF;
 //UCSRB |= (1<<TXEN); // UART TX einschalten
 UCSRB |= (1<<RXEN); // UART RX einschalten
 UCSRC = (1<<URSEL)|(3<<UCSZ0); // Asynchron 8N1 Atmega32
#else
 //fuer ATmega644:
 UBRR0 = UBRR_VAL;
 //UCSR0B = (1<<RXEN0)|(1<<TXEN0); // UART TX und RX einschalten
 UCSR0B = (1<<RXEN0);  // UART RX einschalten
 UCSR0C = (3<<UCSZ00); // Asynchron 8N1 Atmega644
#endif
}

/*
void uart_putc(unsigned char c)
{
 while (!(UCSR0A & (1<<UDRE0))) ;// warten bis Senden moeglich
 UDR = c;                      // sende Zeichen
}
char uart_getc(void)
{
 while (!(UCSR0A & (1<<RXC0))) ;// warten bis Zeichen verfuegbar
 return UDR0; // Zeichen aus UDR an Aufrufer zurueckgeben
}
*/

static char uart_puffer[8];
static uint8 uart_i1=0,uart_i2=0;
void tastaturcheck()
{
 if(!(UCSR0A & (1<<RXC0))) return;// warten bis Zeichen verfuegbar
 if(((uart_i2+1)&0x07)==uart_i1) return;//Puffer voll, noch nichts tun
 uart_puffer[uart_i2++]=UDR0;
 uart_i2 &= 0x07;
}
char tastaturabfrage()
{
 char c=0;
 if(uart_i1!=uart_i2)
  {c=uart_puffer[uart_i1++]; uart_i1 &= 0x07;}
 return c;
}

#else
/*
Datenuebertragung auf 1 Leitung:
-------------------------------

                | 1.Bit |  2. Bit  |   ...  9. Bit |    Rueckmeldung

-----+ Startbit +---+   +---+      +---            +----+   +--
     |  3 ms    |1ms|1ms|1ms| 2ms  |   ...         |    |1ms|
     |          |   |   |   |      |               |    |   |
     +----------+   +---+   +------+             --+    +---+

Das Startbit definiert die Laenge eines zu uebertragenden Bits und ist
3 Einheiten lang (z.B. 3ms). Eine 0 wird durch einen 2 Einheiten langen
Puls uebertragen, eine 1 durch einen 1 Einheit langen Puls.
(Eine 1 also ein kurzer negativer Puls, eine 0 ein langer Puls)
Wenn das erste Datenbit eine 1 ist wird ein Kommando fuer das LCD, sonst
ein darzustellendes Zeichen uebertragen. Die folgenden 8 Bit ergeben
ein Byte beginnend mit dem hoechstwertigen Bit.
Fuer eine schnelle Uebertragung muss am Ende die Datenrichtung umgekehrt
werden, und der LCD-Kontroller gibt einen 1 Einheiten langen Puls zurueck.
(Fuer langsame Uebertragung kann man darauf verzichten und einfach eine
genuegend lange Warteschlaufe einbauen.)
*/
static char taste_ready=0,taste_c=0;
void tastaturcheck()
{
 char d0;
 d0=PIND&1;
 if(d0==1) return;
 char i,j,ok=0;
 int startbit,kurzesbit,langesbit,sehrlangesbit,tmitte,t1,daten;
 for(j=0;ok==0 && j<3;j++)
  {for(startbit=0;(PIND&1)==0;startbit++)
     microwait(10);//Laenge des Startbits messen
   langesbit=startbit*2/3;
   kurzesbit=startbit/3;
   sehrlangesbit=langesbit+(kurzesbit+1)/2;
   tmitte=(langesbit+kurzesbit+1)/2;
   for(i=0,daten=0,ok=1;i<9;i++)
    {for(t1=0;(PIND&1)!=0;t1++)
      {microwait(10);//Low abwarten
       if(t1>sehrlangesbit) {ok=0; daten=0; break;} //Abbruch bei Fehler
      }
     if(ok==0) break;
     for(t1=0;(PIND&1)==0;t1++) microwait(10);//Bitlaenge messen
     daten <<= 1;
     if(t1<tmitte) daten++;
     else if(t1>sehrlangesbit) {ok=0; daten=0; break;} //Abbruch bei Fehler
    }
  }
 if((d0=daten&0xFF)!=0) {taste_ready=1; taste_c=d0;}
 return;
}
char tastaturabfrage()
{
 char c=0;
 if(taste_ready)
  {c=taste_c; taste_ready=0;}
 return c;
}
#endif

/*************** Uebernommen von vereinfachtem Vektar ****************/

//(3.9.08) Vermeidung von Konflikten mit anderweitig definiertem abs:
#ifdef abs
#undef abs
#endif
#define abs meinabs

//const Fliesszahl einhalbes(1,2); //schon weiter oben
const Fliesszahl NAN(1,0); //ungueltige Zahl
//const Fliesszahl PI=atan(1.0)*4;
const Fliesszahl PI("3.14159265358979323846",0); //Eulersche Zahl
const Fliesszahl ZWEIPI=2*PI;
const Fliesszahl LN2("0.69314718055994531",0); //natuerlicher Logarithmus von 2
//const Fliesszahl LN2=calcln2(); //wird auch nicht genauer
//const Fliesszahl LN10("2.302585092994",0);
//const Fliesszahl ee("2.718281828459",0); //Basis zum natuerlichen Logarithmus
const Fliesszahl ee=exp(1); //Basis zum natuerlichen Logarithmus
const Fliesszahl LN10=log(10); //natuerlicher Logarithmus von 10
//const Fliesszahl WURZEL2("1.41421356237",0);
const Fliesszahl GRAD=PI/180;
const Fliesszahl Lichtgeschwindigkeit(299792458,1000); //in km/s
const Fliesszahl Erdbeschleunigung(980665,100000); //in m/s^2
const Fliesszahl Avogadrokonstante("6.022142",23); //in Teilchen/mol

/************************* Vordeklarationen ***************************/
int keyget(int* in,int *asci,ULONG* rawcode);

/**************************** Kleinkram *******************************/
inline bool ist(const char*t,const char*s) {return strcmp(t,s)==0;}
inline Fliesszahl abs(Fliesszahl x) {return x<0?-x:x;}

//static int hexmodus=0; //vorgezogen wegen lcd_gowrite_d()

void lcd_goprint(int8 zeile,char *str)
{
 char c;
 lcd_goto(zeile,1);
 while(strlen(str) > 20)
   {
    c=str[20]; str[20]=0;
    lcd_write(str);
    str[20]=c;
    str = &str[20];
    if(++zeile > 4) zeile=1;
    lcd_goto(zeile,1);
   }
 lcd_write(str);
}

void wprintf(int zeile,const char *cs,const char *p1,const char *p2=NULL)
{
 char str[80];
 sprintf(str,cs,p1,p2);
 hochzeichen_ersetzen(str);
 lcd_goprint(zeile,str);
}

void wprintf(int zeile,const char *cs,int p1,int p2=0)
{
 char str[80];
 sprintf(str,cs,p1,p2);
 hochzeichen_ersetzen(str);
 lcd_goprint(zeile,str);
}

void minus_entfernen(char *s)
{
 char *t= &s[1];
 while((*s++ = *t++)) ;
}
void minus_einfuegen(char *s)
{
 char c='-',c2;
 while(c)
   {c2= *s; *s++ = c; c=c2;}
 *s=0;
}

int s_index(const char *s1,int n1,const char *s2,int n2)
			/* Sucht den String s2 innerhalb von s1 und  */
{			/* gibt die Position zurueck (nicht gefunden = -1) */
 int i,c,j;
 const char *p1,*p2;
 if(n2<=0) return 0;	/* leerer String ist immer enthalten */
 for(i=0;;i++)
	{if(i>=n1) return -1; /* nicht gefunden */
	 c= *s1++;
	 if(c== *s2)
		{for(p1=s1,p2=s2,j=1; j<n2; j++)
			{c= *++p2;
			 if(*p1++!=c) break; /* noch nicht gefunden */
			}
		 if(j==n2) break; /* gefunden */
		}
	}
 return i;
}

/************************* Fehlermeldungen ****************************/
class Fehlermeldung
{
public:
 int errflag;
 Fehlermeldung() {errflag=0;}
 //void error(const char *s,int p1,int p2=0);
 void error(const char *s,const char *p1=NULL);
 void say(const char *s,const char *p1=NULL,const char *p2=NULL);
 // void clear() {color(0); fillbox(x0,y0,x2,y2); wfont(1);}
 void clear();
 void pause() {secwait(3);}
};

/*
void Fehlermeldung::error(const char *s,int p1,int p2)
{
 clear();
 wprintf(4,s,p1,p2);
 if(*s!=' ') errflag=1; else errflag=0;
}
*/

void Fehlermeldung::error(const char *s,const char *p1)
{
 clear();
 wprintf(4,s,p1);
 if(*s!=' ') errflag=1; else errflag=0;
 pause();
}

void Fehlermeldung::say(const char *s,const char *p1,const char *p2)
{
 clear();
 wprintf(1,s,p1,p2);
 errflag=0;
 pause();
}

void Fehlermeldung::clear()
{
 wprintf(4,"                    ",0,0);
 errflag=0;
}

/************************ Globale Variablen ***************************/
//const int TYP_SHIFT=1,TYP_GOLD=2,TYP_NORMAL=0,TYP_HEXTASTE=3;

static Fehlermeldung meldung;
static Fliesszahl regist[10];

/***************************** Klassen ********************************/
const int8 STAPELSIZE=4;

class Stapel
{
 Fliesszahl mem[STAPELSIZE];
 int8 imax;
public:
 Stapel();
 void draw();
 void push(Fliesszahl a);
 Fliesszahl pop();
 Fliesszahl rol(bool);
 Fliesszahl get(int8 j);
 void set(int8 j,Fliesszahl a) {if(j>=imax) j=imax-1; mem[j]=a;}
};
Stapel::Stapel()
{
 imax=STAPELSIZE;
 for(int8 i=0;i<imax;i++) mem[i]=0;
 //for(int8 i=0;i<imax;i++) mem[i]=i+einzehntel;//test
}
void Stapel::draw()
{
 for(uint8 i=0;i<4;i++)
   lcd_gowrite_d(4-i,get(i));
}
Fliesszahl Stapel::get(int8 j)
{
 if(j>=imax) j=imax-1;
 return mem[j];
}
void Stapel::push(Fliesszahl a)
{
 for(int8 i=imax-1;i>0;i--) mem[i]=mem[i-1];
 mem[0]=a;
}
Fliesszahl Stapel::pop()
{
 Fliesszahl a=mem[0];
 for(int8 i=1;i<imax;i++) mem[i-1]=mem[i];
 return a;
}
Fliesszahl Stapel::rol(bool rev)
{
 Fliesszahl a;
 int8 i;
 if(rev)
   {for(a=mem[i=imax-1]; i>0; i--)  mem[i]=mem[i-1];
    mem[0]=a;
   }
 else
   {a=mem[0];
    for(i=1;i<imax;i++) mem[i-1]=mem[i];
    mem[imax-1]=a;
   }
 return a;
}

#define EINGABFELD 40

class Eingabefeld
{
 bool aktiv,reop;
 char str[EINGABFELD];
 int8 cursor,nx;
 Stapel *ps;
public:
 bool help;
 Eingabefeld() {aktiv=reop=help=0; nx=0;}
 void init(Stapel *p) {ps=p;}
 void open(int8 n) {nx=n; str[cursor=0]=0; aktiv=true; reop=0;}
 //void reopen(int8 n);
 void write(int8 c);
 void write(const char *s) {int8 c;while((c= *s++)!=0) write(c);}
 Fliesszahl close();
 void draw();
 bool istaktiv() {return aktiv;}
};

/*
void Eingabefeld::reopen(int8 n) //funktioniert nicht
{
 Fliesszahl v;
 if(aktiv) close();
 nx=n;
 v=ps->get(0);
 sprintf_d(str,v);
 cursor=strlen(str);
 if(ist(str,"0")) str[cursor=0]=0;
 aktiv=reop=true;
}
*/

bool ishexdigit(int8 c)
{
 if(c>='A') return (c<='F' || (c>='a' && c<='f'));
 return isdigit(c);
}
bool isdigitx(int8 c)
{
 if(hexmodus && c<='F' && c>='A') return true;
 return isdigit(c);
}
int8 hexdigit(int8 c)
{
 if(c>='a') return c-'a'+10;
 if(c>='A') return c-'A'+10;
 return c-'0';
}

/*
Fliesszahl pow(Fliesszahl x,int i)
{
 Fliesszahl z=1.0;
 if(i>0)  {do {z *= x;} while(--i!=0);}
 else if(i<0) {do {z /= x;} while(++i!=0);}
 return z;
}
*/
inline Fliesszahl pow(Fliesszahl x,int i) {return zhochn(x,i);}

void hexsscanf(char *str,Fliesszahl* px)
{
 Fliesszahl x=0,y,v=1;
 const char *s=str;
 int c,ex;
 if(*s=='-') {v= -1; s++;}
 for(;(c= *s++)!=0 && ishexdigit(c);)
   x=x*16+hexdigit(c);
 if(c=='.')
   {Fliesszahl y(1,16);
    for(;(c= *s++)!=0 && ishexdigit(c);y/=16)
       x += y*hexdigit(c);
   }
 if(c=='*' && (c= *s++)=='1' && (c= *s++)=='0' && (c= *s++)==Hochzeichen)
   {int vorz=1;
    if(*s=='-') {vorz= -1; s++;}
    for(ex=0;(c= *s++)!=0 && ishexdigit(c);)
       ex=ex*16+hexdigit(c);
    x *= pow(16,ex*vorz);
   }
 *px=v*x;
}

void dezsscanf(char *str,Fliesszahl* px)
{
 Fliesszahl x=0,v=1;
 const char *s=str;
 int c,ex;
 if(*s=='-') {v= -1; s++;}
 for(;(c= *s++)!=0 && isdigit(c);)
   x=x*10+(c-'0');
 if(c=='.')
   {Fliesszahl y(1,10);
    for(;(c= *s++)!=0 && isdigit(c);y/=10)
       x += y*(c-'0');
   }
 if(c=='e' || c=='E' ||
    (c=='*' && (c= *s++)=='1' && (c= *s++)=='0' && (c= *s++)==Hochzeichen))
   {int vorz=1;
    if(*s=='-') {vorz= -1; s++;}
    for(ex=0;(c= *s++)!=0 && isdigit(c);)
       ex=ex*10+(c-'0');
    x *= pow(10,ex*vorz);
   }
 *px=v*x;
}

Fliesszahl Eingabefeld::close()
{
 Fliesszahl x;
 x=ps->get(0);
 if(reop)
   {reop=0;}
 else
   {if(*str==0) x=0;
    else if(hexmodus) hexsscanf(str,&x);
    else dezsscanf(str,&x);
    ps->set(0,x);
   }
 aktiv=0;
 return x;
}

void Eingabefeld::write(int8 c)
{
 static int8 emi=0;
 if(!aktiv) open(nx);
 reop=0;
 if(c=='\b')
   {if(cursor==0) return;//Feld leer
    str[--cursor]=0;
    return;
   }
 //if(cursor >= sizeof(str)-1) return;//Feld voll
 if(cursor >= EINGABFELD-1) return;//Feld voll
 if(c=='-')
   {int i;
    for(i=cursor;i>0 && str[--i]!='+';)
      {if(!hexmodus) {if(str[i]=='e' && (emi=1-emi)==1) {i++; break;}}
       else {if(str[i]==Hochzeichen && (emi=1-emi)==1) {i++; break;}}
      }
    if(str[i]=='+') str[i]='-';
    else if(str[i]=='-') {minus_entfernen(&str[i]); --cursor;}
    else {minus_einfuegen(&str[i]); ++cursor;}
    return;
   }
 if(hexmodus==0 && c=='e' && cursor==0) str[cursor++]='1';
 str[cursor++]=c;
 str[cursor]=0;
}
void Eingabefeld::draw()
{
 if(aktiv)
   {char text[22];
    sprintf(text,"%18s  ",str);
    lcd_gowrite(4,1,text);
   }
}

class Taste
{
 const char **text0,**textnorm,**textblue,**textgold,**textred;
 //int typ;
public:
 //Taste() {typ=TYP_NORMAL; text0=NULL;}
 void init(const char**t1,const char**t2,const char**t3,const char**t4)
        {text0=textnorm=t1; textblue=t2; textgold=t3; textred=t4;
	 //typ=TYP_NORMAL;
	}
 int press(int8,Eingabefeld*,Stapel*);
 void shift(int8 i) {text0 = i==0?textnorm:(i==1?textblue:(i==2?textgold:textred));}
};

//Die Tasten "Red", "Blue" und "Gold" muessen bei allen Tastenbelegungen
//an der gleichen Position sein, die andern Tasten koennen beliebig
//angeordnet werden.
//In aktueller Version muss "Gold" an 1.Stelle und "Blue" an 2.Stelle stehen.

/*
Tasten wie sie gesendet werden:
 {'a','b','c','d','e',
  'A','B','C','D','/',
  '7','8','9','E','*',
  '4','5','6','F','-',
  '1','2','3','N','+',
  '0','.','I','J','='}
 */
static const char *tastenname[]=
 {"Gold","Blue","Red", "POP", "CORR",
  "A",   "B",   "C",   "D",   "/",
  "7",   "8",   "9",   "E",   "*",
  "4",   "5",   "6",   "F",   "-",
  "1",   "2",   "3",   "EXP", "+",
  "0",   ".",   "+/-", "x<>y","ENTR"
 };
static const char *tastenname_gold[]=
 {"Gold","Blue","Red", "POP", "CORR",
  "sin", "cos", "tan", "SQRT"," DEG",
  "7",   "8",   "9",   "LN",  "*",
  "4",   "5",   "6",   "LOG", "-",
  "1",   "HMS", "3",   "EXP", "+",
  "0",   ".",   "+/-", "x<>y","ENTR"
 };
static const char *tastenname_blue[]=
 {"Gold","Blue","Red", "ROL^","ROLv",
  "asin","acos","atan","x^2", "RAD",
  "NA",  "g",   "c",   "e^x", " DEL",
  "Pi",  "|x|", "Int", " 10^x"," EDIT",
  "Time","HR",  "3",   "y^x", "+",
  " Date"," 1/x","HEX", " Dez","ENTR"
 };
static const char *tastenname_red[]=
 {"Gold","Blue","Red", "Help","CORR",
  "Run", "Lear","LBL", "brEQ","brNE",
  "Kale","MGW", "elem","Kons","brPL",
  "STO", "RCL", "STOx","RCLx","brMI",
  "Tset","2",   "3",   "EXP",  "+",
  " Dset",".",   "+/-", "x<>y", "ENTR"
 };
static const char *tasten_hilfe[]=
 {
  "HMS","Fliesszahl umrechnen in Std:Min:Sec",
  "HR","von Std:Min:Sek in Fliesszahl umrechnen",
  "RAD","Umschaltung auf Rad fuer Winkelberechnungen (180Grad==Pi)",
  "DEG","Umschaltung auf Grad fuer Winkelberechnungen",
  "Pi","Eulersche Kreiszahl 3.14159265...",
  "SQRT","Wurzel",
  " 1/x","Kehrwert",
  "x^2","x quadrieren",
  "sin","sin(x)  Sinus",
  "cos","cos(x)  Cosinus",
  "tan","tan(x)  Tangens",
  "asin","arcsin(x)  Umkehrfunktion zu Sinus",
  "acos","arccos(x)  Umkehrfunktion zu Cosinus",
  "atan","arctan(x)  Umkehrfunktion zu Tangens",
  "LN","ln(x)  Natuerlicher Logarithmus",
  "LOG","log(x)  ZehnerLogarithmus",
  "POP","x wird geloescht, das oberste Stapelelement dubliziert",
  "ROLv","der Stapel wird rotiert",
  "ROL^","der Stapel wird nach oben rotiert",
  "RCL","Register 0 wird gelesen",
  "STO","x wird in Register 0 gespeichert",
  "RCLx","Register x wird gelesen, x muss zwischen 0 und 9 sein",
  "STOx","y wird in Register x gespeichert, x muss zwischen 0 und 9 sein",
  "+/-","Vorzeichen wechseln",
  ".","Dezimalpunkt",
  "x<>y","x und y vertauschen",
  "|x|","Betrag von x",
  "-","y-x --> x",
  "CORR","Zahleneingabe korrigieren",
  "EXP","Exponent eingeben",
  "*","Multiplikation",
  /*
  " e","Basis des natuerlichen Logarithmus 2.71828...",
  "POLAR","Umrechnung von Polarkoordinaten (alfa,phi,Betrag) in XYZ",
  "RALOP","Umrechnung in Polarkoordinaten, X=alfa Y=phi Z=Betrag",
  */
  "HEX","Hexadezimal-Modus einschalten",
  " Dez","Dezimal-Modus einschalten (HEX aus)",
  "Time","aktuelle Zeit als HH.MMSS",
  " Date","aktuelles Datum als JJJJ.MMTT",
  "Tset","Zeit setzen HH.MMSS",
  " Dset","Datum setzen JJJJ.MMTT",
  "Kale","in Kalender-Modus wechseln",
  "MGW","Molekulargewicht berechnen",
  NULL,NULL
 };

class Tastenfeld
{
 int nreihen,nspalten;
 Taste tasten;
 //int blue;
public:
 Tastenfeld()
   {nreihen=6; nspalten=5;
    tasten.init(tastenname,tastenname_blue,tastenname_gold,tastenname_red);
   }
 void press(int8 i,Eingabefeld *ef,Stapel *ps);
 void shift(int8 j) {tasten.shift(j);}
};

void Tastenfeld::press(int8 i,Eingabefeld *ef,Stapel *ps)
{
 tasten.press(i,ef,ps);
}

int gettastennummer(const char *txt)
{
 int i,n=sizeof(tastenname)/sizeof(char*);
 const char **name=tastenname;
 for(i=0;i<n;i++)
   {if(strcmp(txt,name[i])==0) return i;
   }
 return -1;
}

#define RADIAN 0
#define DEGREE 1
static int8 winkelmodus=DEGREE; //oder RADIAN

Fliesszahl rad(Fliesszahl x)
{
 if(winkelmodus==DEGREE) x*=GRAD;
 return x;
}

Fliesszahl grad(Fliesszahl x)
{
 if(winkelmodus==DEGREE) x/=GRAD;
 return x;
}

static int8 eing=0;

class Vektar
{
 Stapel stapel;
 Eingabefeld eingabefeld;
 Tastenfeld tafeld;
public:
 //Vektar() {}
 void init();
 void press(int8 i);
 void shift(int8 j) {tafeld.shift(j);}
 bool eingabeaktiv() {return eingabefeld.istaktiv();}
 void refresh();
};
void Vektar::init()
{
 eingabefeld.init(&stapel);
}
void Vektar::press(int8 i)
{
 tafeld.press(i,&eingabefeld,&stapel);
 refresh();
}
void Vektar::refresh()
{
 if(eing==0) stapel.draw();
 eingabefeld.draw();
}
static Vektar vektar;

void gethelp(const char *frage)
{
 const char **p,*s,*antw=NULL;
 for(p=tasten_hilfe; (s= *p++)!=NULL; p++)
   {if(strcmp(s,frage)==0) {antw = *p; break;}
   }
 if(antw==NULL) meldung.say("'%s' no help yet ",frage);
 else meldung.say("'%s' %s ",frage,antw);
}

/********************* Mathematische Funktionen **************************/
Fliesszahl exp(Fliesszahl x)
{
 // e^x = Summe(n=1...)[ x^n/n! ]
 Fliesszahl result(1),glied,alt(0);
 int n;
 uint16_t iy;
 bool neg;
 if(x<0) {x= -x; neg=true;}
 else neg=false;
 if(x>8) {iy=getint16(x); x -= iy; if(iy>1500) return neg?0:NAN;}
 else iy=0;
 for(n=1, glied=x; result!=alt && n<200; n++,glied *= x/n) //provi. Grenze 200 sollte nicht noetig sein
   {alt = result;
    result += glied;
   }
 //regist[8]=n;//test
 if(iy!=0) result *= pow(ee,iy);
 if(neg) return 1/result;
 return result;
}

/**
Fliesszahl calcln2()  //natuerlicher Logarithmus von 2
{
 // ln(x) = Summe(k=0...)[ 2/(2k+1) * ((x-1)/(x+1))^(2k+1) ]
 // wobei ln(x) = m*ln(2) + ln(x/2^m)
 Fliesszahl result(0),alt(1),faktor1(2,3),faktor2(1,9);
 int zweik1;
 for(zweik1=1; result!=alt; zweik1+=2)
   {alt = result;
    result += faktor1/zweik1;
    faktor1 *= faktor2;
   }
 return result;
}
**/

Fliesszahl log(Fliesszahl x)  //natuerlicher Logarithmus
{
 // ln(x) = Summe(k=0...)[ 2/(2k+1) * ((x-1)/(x+1))^(2k+1) ]
 // wobei ln(x) = m*ln(2) + ln(x/2^m)
 Fliesszahl result(0),alt(1),faktor1,faktor2;
 int zweik1,m;
 if(x.istnull() || x<0) return NAN;//Fehler
 m=x.getexp()-1;
 x.setexp(1);
 faktor1 = (x-1)/(x+1);
 faktor2 = faktor1*faktor1;
 faktor1 += faktor1;
 for(zweik1=1; result!=alt && zweik1<200; zweik1+=2)  //provi. Grenze 200 sollte nicht noetig sein
   {alt = result;
    result += faktor1/zweik1;
    faktor1 *= faktor2;
   }
 //regist[9]=zweik1;//test
 return result+LN2*m;
}

Fliesszahl pow(Fliesszahl x,Fliesszahl y)
{
 int16 iy=getint16(y);
 if(y==iy) return pow(x,iy);
 if(x.istnull()) return (y<0)?NAN:x;
 if(x<0) return NAN;
 return exp(log(x)*y);
}

Fliesszahl sqrt(Fliesszahl x)
{
 Fliesszahl result=1,r0=x;
 int i;
 i=x.getexp()/2;
 if (i > 1)      while(i>0) {result *= 2; --i;}
 else if(i < -1) while(i<0) {result /= 2; i++;}
 i=0;
 do
   {r0 = result;
    result = (r0+x/r0)/2;
   }
 while(result!=r0 && ++i<100); //ev. waere ++i<100 nicht noetig ?
 return result;
}

Fliesszahl x2hms(Fliesszahl x,int16& h,int8& m,int8& s,int8 hundert)
{
 const Fliesszahl dx(1,1000000);
 h = getint16(x);    x = (x-h)*hundert; if(x<0) x=0;
 m = getint8(x+dx);  x = (x-m)*hundert; if(x<0) x=0;
 s = getint8(x+dx);
 return x;
}

Fliesszahl hms(Fliesszahl x)  //in Std:Min:Sek umrechnen, provisorisch
{
 Fliesszahl result;
 int16 std;
 int8 min,s,neg=0;
 if(x<0) {x = -x; neg=1;}
 x = x2hms(x,std,min,s,60);
 result = std + (min + x/100) / 100;
 if(neg) result = -result;
 return result;
}

Fliesszahl hr(Fliesszahl x)  //von Std:Min:Sek zurueckrechnen, provisorisch
{
 Fliesszahl result;
 int16 std;
 int8 min,s,neg=0;
 if(x<0) {x = -x; neg=1;}
 x = x2hms(x,std,min,s,100);
 result = std + (min + x/60) / 60;
 if(neg) result = -result;
 return result;
}

Fliesszahl sin(Fliesszahl x)
{
 // sin(x) = Summe(n=0...)[ (-1)^n*x^(2n+1)/(2n+1)! ] = x/1 - x^3/3! + x^5/5! - ...
 Fliesszahl result(0),glied,alt(1),x2;
 int k; //k=2n+1
 if(x >= PI || x <= -PI)  x -= getint48(x/PI)*PI;
 x2 = -x*x;
 for(k=1, glied=x; result!=alt; k++, glied *= x2/k/(k+1), k++)
   {alt = result;
    result += glied;
   }
 //regist[7]=k;//test
 return result;
}

Fliesszahl cos(Fliesszahl x)
{
 // cos(x) = Summe(n=0...)[ (-1)^n*x^(2n)/(2n)! ] = 1 - x^2/2! + x^4/4! - ...
 Fliesszahl result(0),glied(1),alt(2),x2;
 int k; //k=2n
 if(x<0) x = -x;
 if(x >= ZWEIPI)  x -= getint48(x/ZWEIPI)*ZWEIPI;
 x2 = -x*x;
 for(k=0; result!=alt; k++, glied *= x2/k/(k+1), k++)
   {alt = result;
    result += glied;
   }
 //regist[8]=k;//test
 return result;
}

Fliesszahl tan(Fliesszahl x)
{
 return sin(x)/cos(x);
}

Fliesszahl atan(Fliesszahl x)
{
 // arctan(x) = Summe(n=0...)[ (-1)^n*x^(2n+1)/(2n+1) ] = x/1 - x^3/3 + x^5/5 - ...
 Fliesszahl result(0),glied,alt(1),x2,m(1);
 int k; //k=2n+1
 bool neg;
 if(x<0) {x = -x; neg=true;}
 else neg=false;
 while(x >= einhalbes)
  {x = (sqrt(x*x+1)-1)/x;
   m += m;
  }
 x2 = -x*x;
 for(k=1, glied=x; result!=alt && k<1000; k+=2, glied *= x2)
   {alt = result;
    result += glied/k;
   }
 regist[9]=k;//test
 result *= m;
 if(neg) result = -result;
 return result;
}

Fliesszahl asin(Fliesszahl x)
{
 if(x==1) return PI/2;
 if(x>1 || -x>1) return NAN;
 return atan(x/sqrt(1-x*x));
}

Fliesszahl acos(Fliesszahl x)
{
 if(x.istnull()) return PI/2;
 if(x>1 || -x>1) return NAN;
 return atan(sqrt(1-x*x)/x);
}

/*************************** Hauptprogramm *******************************/
//static int8 eing=0;  //schon vergezogen

int Taste::press(int8 j,Eingabefeld *ef,Stapel *stap)
{
 static int8 blue=0; // 1 = Taste Blue aktiv, 2 = Taste Gold aktiv, 4 = Taste Red aktiv
 static int8 nx=0;
 int16 c;
 Fliesszahl x,y,z;
 const char *text=text0[j];
 c= *text;
 if(ef->help) {gethelp(text); ef->help=0; return blue;}
 if(ist(text,"Blue")) return blue ^= 1;
 if(ist(text,"Gold")) return blue ^= 2;
 if(ist(text,"Red"))  return blue ^= 4;
 if(ist(text,"ENTR"))
   {if(eing) {ef->close(); eing=0;}
    else stap->push(stap->get(0));
    return blue;
   }
 if(eing==0)
   {if(isdigitx(c) || c=='.' || ist(text,"EXP"))
      {stap->push(0); stap->draw(); ef->open(nx); eing=1; //Eingabemodus einschalten
      }
    /* funktioniert nicht:
    else if(ist(text,"CORR"))
      {ef->reopen(nx); eing=1; //Eingabemodus einschalten
       return blue;
      }
    */
   }
 if(eing==1)
   {if(isdigitx(c)||c=='.'||ist(text,"EXP")||ist(text,"+/-")||ist(text,"CORR"))
      {//Reaktion auf Tasten im Eingabemodus:
       if(ist(text,"+/-")) c='-';
       else if(ist(text,"EXP")) {c='e'; if(hexmodus){ef->write("*10");c=Hochzeichen;}}
       else if(ist(text,"CORR")) c='\b';
       ef->write(c);
       return blue;
      }
    else
      {ef->close(); eing=0;} //Eingabemodus beenden
   }
 if(ist(text,"+/-")) {x=stap->pop(); stap->push(-x);}
 else if(ist(text,"POP")) stap->pop();
 else if(ist(text,"ROLv")) stap->rol(0);
 else if(ist(text,"ROL^")) stap->rol(1);
 else if(ist(text,"+")) {x=stap->pop(); y=stap->pop(); stap->push(y+x);}
 else if(ist(text,"-")) {x=stap->pop(); y=stap->pop(); stap->push(y-x);}
 else if(ist(text,"/")) {x=stap->pop(); y=stap->pop(); stap->push(y/x);}
 else if(ist(text,"*"))	{x=stap->pop(); y=stap->pop(); stap->push(y*x);}
 else if(ist(text,"sin")) {x=stap->pop(); stap->push(sin(rad(x)));}
 else if(ist(text,"cos")) {x=stap->pop(); stap->push(cos(rad(x)));}
 else if(ist(text,"tan")) {x=stap->pop(); stap->push(tan(rad(x)));}
 else if(ist(text,"asin")) {x=stap->pop(); stap->push(grad(asin(x)));}
 else if(ist(text,"acos")) {x=stap->pop(); stap->push(grad(acos(x)));}
 else if(ist(text,"atan")) {x=stap->pop(); stap->push(grad(atan(x)));}
 else if(ist(text,"SQRT")) {x=stap->pop(); stap->push(sqrt(x));}
 else if(ist(text,"x^2")) {x=stap->pop(); stap->push(x*x);}
 else if(ist(text,"x<>y"))
	{x=stap->pop(); y=stap->pop(); stap->push(x); stap->push(y);}
 else if(ist(text,"|x|"))
	{x=stap->pop(); x=abs(x); stap->push(x);}
 else if(ist(text,"Int"))
	{x=stap->pop(); x=getint48(x); stap->push(x);}
 else if(ist(text,"Pi")) {stap->push(PI);}
 else if(ist(text,"NA")) {stap->push(Avogadrokonstante);}
 else if(ist(text,"g")) {stap->push(Erdbeschleunigung);}
 else if(ist(text,"c")) {stap->push(Lichtgeschwindigkeit);}
 //else if(ist(text," e")) {stap->push(ee);}
 else if(ist(text,"e^x")) {x=stap->pop(); stap->push(exp(x));}
 else if(ist(text," 10^x")) {x=stap->pop(); stap->push(pow(10,x));}
 else if(ist(text,"LN")) {x=stap->pop(); stap->push(log(x));}
 else if(ist(text,"LOG")) {x=stap->pop(); stap->push(log(x)/LN10);}
 else if(ist(text,"y^x")) {x=stap->pop(); y=stap->pop(); stap->push(pow(y,x));}
 //else if(ist(text,"y^1/x")) {x=stap->pop(); y=stap->pop(); stap->push(pow(y,1/x));}
 else if(ist(text," 1/x")) {x=stap->pop(); stap->push(1/x);}
 else if(ist(text,"RAD")) winkelmodus=RADIAN;
 else if(ist(text," DEG")) winkelmodus=DEGREE;
 else if(ist(text,"STO")) regist[0]=stap->get(0);
 else if(ist(text,"RCL")) stap->push(regist[0]);
 else if(ist(text,"STOx"))
	{x=stap->pop(); c=getint16(x);
	 if(x==c && c>=1 && c<=9) regist[c]=stap->get(0);
	 else {stap->push(x); meldung.error("only register 1 to 9 available");}
	}
 else if(ist(text,"RCLx"))
	{x=stap->pop(); c=getint16(x);
	 if(x==c && c>=1 && c<=9) stap->push(regist[c]);
	 else {stap->push(x); meldung.error("only register 1 to 9 available");}
	}
 else if(ist(text,"Help"))
	{meldung.say("press key for help"); ef->help=1;}
 else if(ist(text,"HEX"))
   {hexmodus=1;}
 else if(ist(text," Dez"))
   {hexmodus=0;}
 else if(ist(text,"HMS"))
   {x=stap->pop();
    stap->push(hms(x));
   }
 else if(ist(text,"HR"))
   {x=stap->pop();
    stap->push(hr(x));
   }
 else if(ist(text,"Time"))
   {x = stunden + (min + Fliesszahl(sec)/100)/100;
    stap->push(x);
   }
 else if(ist(text," Date"))
   {x = jahr + (monat + Fliesszahl(tage)/100)/100;
    stap->push(x);
   }
 else if(ist(text,"Tset"))
   {int h; int8 m,s;
    x=stap->pop();
    x2hms(x,h,m,s,100);
    cli(); stunden=h; min=m; sec=s; sei();
    zeit_in_rtc_speichern();
   }
 else if(ist(text," Dset"))
   {int j; int8 m,t;
    x=stap->pop();
    x2hms(x,j,m,t,100);
    cli(); jahr=j; monat=m; tage=t; sei();
    schaltjahr_und_wochentagberechnen();
    zeit_in_rtc_speichern();
   }
 else if(ist(text,"Kale"))
   {hauptmodus=KALENDER;
   }
 //neue Funktionen hier einfuegen

 else meldung.error("Err:'%s'",text);
 return blue;
}

void tarech_init()
{
 vektar.init();
 //meldung.error("Ready");
}

char tastennummer(uchar c)
{
 char str[6];
 switch(c)
  {case 'a': strcpy(str,"Gold"); break;
   case 'b': strcpy(str,"Blue"); break;
   case 'c': strcpy(str,"Red"); break;
   case 'd': strcpy(str,"POP"); break;
   case 'e': strcpy(str,"CORR"); break;
   case '=': strcpy(str,"ENTR"); break;
   case 'I': strcpy(str,"+/-"); break;
   case 'J': strcpy(str,"x<>y"); break;
   case 'N': strcpy(str,"EXP"); break;
   default: str[0]=c; str[1]=0;
  }
 return gettastennummer(str);
}

void tarech_tastenreaktion(uchar taste)
{
 static uchar gold=0,blue=0,red=0;
 int8 i,j;
 if(taste)
  {i=tastennummer(taste);
   //wprintf(3,"i=%d  ",i);//test
   if(i==0) {gold ^= 1;}
   if(i==1) {blue ^= 1;}
   if(i==2) {red  ^= 1;}
   if(gold)  j = 2; else j=0;
   if(blue) {j = 1; gold=0;} //Gold ungueltig wenn Blue gedrueckt
   if(red)   j = 3; //vorlaeufig Gold und Blue ungueltig wenn Red gedrueckt
   //if(red)  j += 3; //Gold-Red (j=5) oder Blue-Red (j=4) fuer weiter Spezialfunktionen

   if(i>2) //Gold-, Blue- und Red-Tasten sind schon ausgwertet
     {vektar.shift(j);
      vektar.press(i);
      //red=blue=0; //Blue und Red sind nur fuer einen Tastendruck gueltig
      gold=red=blue=0; //Gold, Blue und Red sind nur fuer einen Tastendruck gueltig
     }
  }
 glcd_goto(1,19);
 glcd_write((gold) ? 'G' : (blue)?'B':' ');
 glcd_write((red) ?'R':' ');
 if(vektar.eingabeaktiv())
      glcd_goto(4,19); //Cursor hinter letztem eingegebenen Zeichen
 else glcd_goto(4,20); //sonst Cursor auf letzter Zeile ganz rechts
}

void tarech_refresh()
{
 vektar.refresh();
 tarech_tastenreaktion(0);
}
