/* fliesszahlklasse.cc                               letzte Aenderung: 27.11.2010
Rechnen mit Fliesszahlen auf AVR-Controllern
Anwendung: einfgen in C++ Programmen

Version: 0.06

 History:
 27.l1.10     Versionsnummer (0.06) von tarech.cc uebernommen
              exponent nur noch 12 Bit, restliche Bits fuer NAN und HMS-Rechnung
	      oder einfacher: Flags seperat (dafuer mehr RAM-Speicher benoetigt)
	      Fehler in operator+=: bei Addition von 0 gab es signifikante Rundungsfehler.
	      Korrigiert durch Abfrage mit istnull().
	      Einfuehrung von ZEROFLAG: sollte z.B. istnull() schneller machen.

*/

#ifndef FLIESSZAHLKLASSE_CC
#define FLIESSZAHLKLASSE_CC

typedef int8_t int8;
typedef unsigned char uint8;

//#if sizeof(int)==2
#ifdef TESTLINUX
typedef short int16; //stimmt mit Linux Kubuntu
#else
typedef int int16; //stimmt mit AVR
#endif

//#if sizeof(int)==4
#ifdef TESTLINUX
typedef int int32; //koennte unter Linux stimmen
#else
typedef long int32; //stimmt mit AVR
#endif

#define LONGMULT
#include "int48.cc"

class Fliesszahl
{
 int48 mantisse; //Vorzeichenbehaftete Zahl im Bereich -1.0 bis 1.0 (inklusive -1.0 aber ohne 1.0)
                 //Beispiel:  0.5 = 0b01000...
                 //Beispiel: 0.75 = 0b01100...
                 //Beispiel: -0.5 = 0b11000...
                 //Beispiel: -0.75= 0b10100...
                 //Beispiel:   fast 1.0 = 0b01111...111
                 //Beispiel:  fast -1.0 = 0b10000...001
                 //Beispiel: genau -1.0 = 0b10000...000
 int16 exponent; //12 Bit vorzeichenbehafteter Zweierexponent so dass:  Zahl = mantisse * 2^exponent
                 //2 hoechstwertige Bits fuer Ueberlauf und NAN
                 //weiteres Bit fuer HMS-Rechnung: Trick waere nur bei Darstellung z.B. 12:15:30
                 //sonst aber normale Fliesszahl  z.B. 12.258333333
                 //1 Bit Zeroflag (wenn die Zahl gleich Null ist)
 int8 flags;   //Einfacher zum programmieren, dafuer 9 Bytes statt nur 8 pro Fliesszahl
#define OVERFLOW     0x80
#define NANFLAG      0x40
#define NANOVERFLOW  0xC0
#define HMSFLAG      0x20
#define ZEROFLAG     0x10
 //int16 getexpflags(int16* flags) {*flags=exponent&0xF000; return getexp();}
 void normieren(); //Mantisse in den Bereich -1.0 bis -0.5 oder 0.5 bis <1.0 bringen
 bool rshift()                 //Mantisse vorzeichengerecht nach rechts schieben,
   {exponent++;                //Exponent erhoehen,
    return mantisse.rshift();} //Rueckgabewert false wenn Zahl 0 wird. (ZEROFLAG aber noch nicht gesetzt)
public:
 Fliesszahl(long a=0) {mantisse=a; exponent=47; flags=0; normieren();}
 Fliesszahl(int48 a) {mantisse=a; exponent=47; flags=0; normieren();}
 Fliesszahl(long a,long b)
   {mantisse=a; exponent=47; flags=0; normieren();
    if(b==0) {exponent=0x7FF; flags |= NANOVERFLOW;} //Bei Division durch 0 Ueberlauf und NAN setzen
    else {Fliesszahl z(b); (*this)/=z;}
   }
 Fliesszahl(const char *s,int exp); //zum eingeben von z.B. ("12.34",6) fuer 12.34 Millionen
 Fliesszahl& operator=(Fliesszahl v) {mantisse=v.mantisse; exponent=v.exponent; flags=v.flags; return *this;}
 Fliesszahl& operator+=(Fliesszahl v);
 Fliesszahl& operator-=(Fliesszahl v);
 Fliesszahl& operator*=(Fliesszahl z);
 Fliesszahl& operator/=(Fliesszahl z);
 friend Fliesszahl operator+(Fliesszahl,Fliesszahl);
 friend Fliesszahl operator-(Fliesszahl,Fliesszahl); //Subtraktions-Operator
 friend Fliesszahl operator-(Fliesszahl); //Vorzeichen-Operator
 friend Fliesszahl operator*(Fliesszahl,Fliesszahl);
 friend Fliesszahl operator/(Fliesszahl,Fliesszahl);
 friend Fliesszahl abs(Fliesszahl);
 bool isnegativ() {return mantisse.isnegativ();}
 //bool istnull()   {return mantisse==0;}
 bool istnull()   {return (flags&ZEROFLAG)!=0;} //schnellere Variante
 bool istnan()    {return (flags&NANFLAG)!=0;}
 bool operator!=(char z);
 bool operator==(long z);
 bool operator<(long z);
 bool operator>(long z);
 bool operator<=(Fliesszahl v);
 bool operator>=(Fliesszahl v);
 bool operator==(Fliesszahl v);
 bool operator!=(Fliesszahl v) {return !((*this)==v);}
 Fliesszahl& operator<<=(int n);
 Fliesszahl& operator>>=(int n);
 friend int8 getint8(Fliesszahl z);
 //operator int() {return getint16(*this);}
 friend int16 getint16(Fliesszahl z);
 friend int32 getint32(Fliesszahl z);
 friend int48 getint48(Fliesszahl z);
 //friend long long getlonglong(Fliesszahl z) {return getlonglong(getint48(z));}
#ifdef TESTSTR
 char *str();
#endif
 //int16 getexp() {int16 n=exponent&0xFFF; return (n&0x0800) ? (n|0xF000) : n;}
 //void setexp(int16 n) {exponent = n&0xFFF;}
 int16 getexp() {return exponent;}
 void setexp(int16 n) {exponent = n;}
};

Fliesszahl::Fliesszahl(const char *s,int exp)
{
 char c;
 bool neg=false;
 int j,k;
 mantisse=0; exponent=47; flags=0;
 if(*s=='-') {neg=true; s++;}
 for(j=0,k= -1; (c= *s++)!=0 && j<14 && (isdigit(c) || c=='.');)
   {
    if(c=='.') k=j;
    else {mantisse = mantisse*10 + (c-'0'); j++;}
   }
 normieren();
 if(k<0) k=j;
 j = j-k-exp;
 if(j>0)
   for(;j>0;--j) (*this) /= 10;
 else if(j<0)
   for(;j<0;j++) (*this) *= 10;
 if(neg) (*this) = -(*this);
}

void Fliesszahl::normieren()
{
 if(mantisse==0) {exponent=0; flags &= ~NANOVERFLOW; flags |= ZEROFLAG; return;}
 else flags &= ~ZEROFLAG;
 if(!mantisse.isnegativ())
   while((mantisse[5]&0x40)==0) //positive Zahl schieben bis das 2. Bit gesetzt ist
     {
      mantisse<<=1; exponent--; //schieben und entsprechend Exponent anpassen
     }
 else
   while((mantisse[5]&0x60)==0x60) //negative Zahl schieben bis 2. oder 3. Bit Null ist
     {
      mantisse<<=1; exponent--; //schieben und entsprechend Exponent anpassen
      mantisse.setnegbit();     //bei negativer Zahl immer 1. Bit setzen
     }
 if(exponent > 0x7FF) //Maximale Zahlengroesse --> Exponent auf Maximalwert beschraenken
   {exponent=0x7FF; flags |= NANOVERFLOW;}
 else if(exponent < -0x7FF) //Minimale Zahlengroesse --> Zahl auf 0 setzen
   {mantisse=0; flags &= ~NANOVERFLOW; flags |= ZEROFLAG; exponent=0;}
}

Fliesszahl& Fliesszahl::operator+=(Fliesszahl v)
{
 bool neg1,neg2,neg3;
 if(v.istnull()) return (*this);
 if(istnull())   return (*this)=v;
 if(exponent!=v.exponent)
   {
    while(v.exponent<exponent) {if(v.rshift()==0) return (*this);}
    while(exponent<v.exponent) {if ( rshift()==0) return (*this)=v;}
   }
 neg1 = isnegativ();
 neg2 = v.isnegativ();
 mantisse += v.mantisse;
 neg3 = isnegativ();
 if(!neg1 && !neg2 && neg3) //Cy beim Addieren von 2 positiven Zahlen
   {
    mantisse>>=1; //Bits schieben, hoechstwertiges auf 0 setzen
    exponent++;
   }
 else if(neg1 && neg2 && !neg3) //Cy beim Addieren von 2 negativen Zahlen
   {
    mantisse>>=1; mantisse.setnegbit(); //Bits schieben, hoechstwertiges auf 1 setzen
    exponent++;
   }
 else normieren();
 return (*this);
}

Fliesszahl operator+(Fliesszahl v1,Fliesszahl v2)
{
 return v1+=v2;
}

Fliesszahl& Fliesszahl::operator-=(Fliesszahl v)
{
 v = -v;
 return (*this) += v;
}

Fliesszahl operator-(Fliesszahl v1,Fliesszahl v2)
{
 return v1-=v2;
}

Fliesszahl operator-(Fliesszahl v1)
{
 char c;
 c = (v1.isnegativ()) ? 1 : 0;
 v1.mantisse = -v1.mantisse;
 if(c==1 && (v1.isnegativ())) //Ueberlauf beim Negieren, kann eigentlich nur mit -1.0 passieren
   {
    v1.mantisse>>=1; //v1.mantisse[5] &= ~0x80;
    v1.exponent++;
   }
 return v1;
}

Fliesszahl abs(Fliesszahl v)	//Betrag
{
 if(v.isnegativ()) v = -v;
 return v;
}

bool Fliesszahl::operator!=(char z)
{
 if(z==0) return !istnull();
 Fliesszahl x=(*this);
 if(z<0 && isnegativ())
   {
    x = -x;
    z = -z;
   }
 if(z>0 && !isnegativ())
   {
    return (x.mantisse[5]!=z || x.exponent!=8 || //erste 8 Bits muessten uebereinstimmen, restlich Bits auf 0
	    x.mantisse[4]!=0 || x.mantisse[3]!=0 || x.mantisse[2]!=0 || x.mantisse[1]!=0 || x.mantisse[0]!=0);
   }
 return true;
}

bool Fliesszahl::operator==(long z)
{
 Fliesszahl v(z);
 v -= (*this);
 return v.istnull();
}

bool Fliesszahl::operator>(long z)
{
 if(z==0) return (mantisse!=0 && !isnegativ());
 Fliesszahl v(z);
 v -= (*this);
 return v.isnegativ();
}

bool Fliesszahl::operator<(long z)
{
 if(z==0) return isnegativ();
 return !((*this)>z || (*this)==z);
}

bool Fliesszahl::operator<=(Fliesszahl v)
{
 bool neg=isnegativ(), neg2=v.isnegativ();
 Fliesszahl x=(*this);
 if(neg)
   {if(!neg2) return true;
    x = -x;  v = -v;
   }
 x -= v;
 if(x.istnull()) return true;
 if(neg) return !x.isnegativ();
 return x.isnegativ();
}

bool Fliesszahl::operator>=(Fliesszahl v)
{
 bool neg=isnegativ(), neg2=v.isnegativ();
 Fliesszahl x=(*this);
 if(neg)
   {if(!neg2) return false;
    x = -x;  v = -v;
   }
 x -= v;
 if(x.istnull()) return true;
 if(neg) return x.isnegativ();
 return !x.isnegativ();
}

bool Fliesszahl::operator==(Fliesszahl v)
{
 v -= (*this);
 return v.istnull();
}

/*
Fliesszahl& Fliesszahl::operator<<=(int n)
{
 return (*this);
}

Fliesszahl& Fliesszahl::operator>>=(int n)
{
 return (*this);
}
*/

bool ukleinergleich(int48 a,int48 b)  //a<=b Vorzeichenlos gerechnet
{
 int8 i;
 for(i=6;--i>=0;)
   {
    if((a[i]&0xFF) < (b[i]&0xFF)) return true;
    if((a[i]&0xFF) > (b[i]&0xFF)) return false;
   }
 return true;
}

Fliesszahl& Fliesszahl::operator/=(Fliesszahl z)
{
 int8 neg=0,j;
 int48 result(0);
 if(z.istnull()) //Division durch Null?
   {flags |= NANOVERFLOW; flags &= ~ZEROFLAG; exponent |= 0x7FF; return (*this);}
 if(isnegativ()) //linke Zahl negativ?
   {(*this) = -(*this);  neg=1;} //ja: positiv machen
 if(z.isnegativ()) //rechte Zahl negativ?
   {z = -z;  neg ^= 1;} //ja: positiv machen

 exponent -= z.exponent;
 for(j=48;;)
   {if(ukleinergleich(z.mantisse,mantisse))
       {
	result[0] |= 1;
	mantisse -= z.mantisse;
       }
    --j;
    if((result.isnegativ()) || mantisse==0) break;
    mantisse <<= 1;
    result <<= 1;
   }
 if(result.isnegativ())
  {
   //result += 1; //bei letztem Bit korrekt runden (waere logischer)
   result += 2; //bei letztem Bit aufrunden (scheint aber so besser zu sein)
   result >>= 1; j++;
  }
 exponent += j;
 mantisse=result;

 if(neg) (*this) = -(*this); //eventuell Vorzeichen setzen
 normieren();
 return (*this);
}

Fliesszahl operator/(Fliesszahl v1,Fliesszahl v2)
{
 return v1/=v2;
}

Fliesszahl& Fliesszahl::operator*=(Fliesszahl z)
{
 int8 neg=0,j;
 int96 res96(0);
 if(isnegativ()) //linke Zahl negativ?
   {(*this) = -(*this);  neg=1;} //ja: positiv machen
 if(z.isnegativ()) //rechte Zahl negativ?
   {z = -z;  neg ^= 1;} //ja: positiv machen

 exponent += z.exponent + 1;
 res96=multu48(mantisse,z.mantisse);
 for(j=0;j<6;j++) mantisse[j]=res96.uc[j+6];

 if(neg) (*this) = -(*this); //eventuell Vorzeichen setzen
 normieren();
 return (*this);
}

Fliesszahl operator*(Fliesszahl v1,Fliesszahl v2)
{
 return v1 *= v2;
}

Fliesszahl zhochn(Fliesszahl z,int n)
{
 Fliesszahl result=1;
 bool neg;
 if(n<0) {n = -n; neg=true;}
 else neg=false;
 while(n>0)
   {
    if(n&1) result *= z;
    if((n>>=1)==0) break;
    z *= z;
   }
 if(neg) return 1/result;
 return result;
}

int8 getint8(Fliesszahl z)
{
 int8 result=0,cy=0;
 bool neg=0;
 if(z.isnegativ()) {z = -z; neg=true;}
 if(z!=0 && z.exponent>0)
   {
    do
     {cy |= result&0x80;
      result <<= 1;
      if(z.mantisse[5]&0x40) result++;
      z.mantisse <<= 1;
     }
    while(--z.exponent>0);
    if(cy) result=0x7F; //Maximalwert bei Ueberlauf
   }
 if(neg) result = -result;
 return result;
}

int16 getint16(Fliesszahl z)
{
 int16 result=0;
 int8 cy=0;
 bool neg=0;
 if(z.isnegativ()) {z = -z; neg=true;}
 if(z!=0 && z.exponent>0)
   {
    do
     {if(result&0x8000) cy=1;
      result <<= 1;
      if(z.mantisse[5]&0x40) result++;
      z.mantisse <<= 1;
     }
    while(--z.exponent>0);
    if(cy) result=0x7FFF; //Maximalwert bei Ueberlauf
   }
 if(neg) result = -result;
 return result;
}

int32 getint32(Fliesszahl z)
{
 int32 result=0;
 int8 cy=0;
 bool neg=0;
 if(z.isnegativ()) {z = -z; neg=true;}
 if(z!=0 && z.exponent>0)
   {
    do
     {if(result&0x80000000) cy=1;
      result <<= 1;
      if(z.mantisse[5]&0x40) result++;
      z.mantisse <<= 1;
     }
    while(--z.exponent>0);
    if(cy) result=0x7FFFFFFF; //Maximalwert bei Ueberlauf
   }
 if(neg) result = -result;
 return result;
}

int48 getint48(Fliesszahl z)
{
 int48 result=0;
 int8 cy=0;
 bool neg=0;
 if(z.isnegativ()) {z = -z; neg=true;}
 if(z!=0 && z.exponent>0)
   {
    do
     {if(result.isnegativ()) cy=1;
      result <<= 1;
      if(z.mantisse[5]&0x40) ++result;
      z.mantisse <<= 1;
     }
    while(--z.exponent>0);
    if(cy) result[5]=0x7F; //fast Maximalwert bei Ueberlauf
   }
 if(neg) result = -result;
 return result;
}

#ifdef TESTSTR
char *Fliesszahl::str()
{
 //char *txt=new char[40]; //geht so nicht auf AVR
 char *txt=new_char40();
 char puf[32],*s;
 Fliesszahl z=(*this),z4;
 int k,zexp=z.exponent*3/10; //Zehnerexponent = Zweierexponent * log(2)
 int8 c;

 //Wenn wir durch 10^zexp dividieren ist die Zahl etwa im Bereich +-1 bis +-9.999
 z /= zhochn(10,zexp);
 //for(int i=0;i<zexp;i++) z/=10; //test wenn zhochn() nicht geht

 s = puf;
 if(z.isnegativ()) {*s++ = '-'; z = -z;}
 for(int8 i=0;i<12;i++) //12 Stellen drucken
   {
    k = getint8(z);
    if(k>9) c = '?';//Fehler
    else if(k<0) c = '#';//Fehler
    else c = k+'0';
    *s++ = c;
    if(i==0) *s++ = '.';
    z -= k;
    //z *= 10;
    z += z; z4 = z+z+z+z; z += z4; //Prov. ersatz fuer z *= 10
   }
 *s=0;
 sprintf(txt,"%se%d",puf,zexp);
 return txt;
}
#endif

#endif
