//**************************************************************
//    ******** FUNCTIONS FOR I2C COMMUNICATION  *******
//**************************************************************
//Controller:	ATmega128 (16Mhz Crystal)
//Compiler:		ICCAVR
//Author:		CC Dharmani, Chennai (INDIA)
//			www.dharmanitech.com
//Date:			Aug 2008
// Anpassung an ATmega8: 31.10.2010

#include <avr/io.h>
#include "i2c.h"

//*************************************************
//Function to start i2c communication
//*************************************************
unsigned char i2c_start(void)
{
    TWCR = (1<<TWINT) | (1<<TWSTA) | (1<<TWEN); //Send START condition
    while (!(TWCR & (1<<TWINT)));   	        //Wait for TWINT flag set. This indicates that the
		  				//START condition has been transmitted
    if ((TWSR & 0xF8) == START)			//Check value of TWI Status Register
	return(0);
    else
	return(1);
}

//*************************************************
//Function for repeat start condition
//*************************************************
unsigned char i2c_repeatStart(void)
{
    TWCR = (1<<TWINT) | (1<<TWSTA) | (1<<TWEN); //Send START condition
    while (!(TWCR & (1<<TWINT)));   		//Wait for TWINT flag set. This indicates that the
		  				//START condition has been transmitted
    if ((TWSR & 0xF8) == REPEAT_START)		//Check value of TWI Status Register
	return(0);
    else
	return(1);
}

//**************************************************
//Function to transmit address of the slave
//*************************************************
unsigned char i2c_sendAddress(unsigned char address)
{
   unsigned char STATUS;
   
   if((address & 0x01) == 0)  STATUS = MT_SLA_ACK;
   else                       STATUS = MR_SLA_ACK; 
   
   TWDR = address; 
   TWCR = (1<<TWINT) | (1<<TWEN);  //Load SLA_W into TWDR Register. Clear TWINT bit
                                   //in TWCR to start transmission of address
   while (!(TWCR & (1<<TWINT)));   //Wait for TWINT flag set. This indicates that the
   		 		   //SLA+W has been transmitted, and
				   //ACK/NACK has been received.
   if ((TWSR & 0xF8) == STATUS)	   //Check value of TWI Status Register
       return(0);
   else
       return(1);
}

//**************************************************
//Function to transmit a data byte
//*************************************************
unsigned char i2c_sendData(unsigned char data)
{
   TWDR = data; 
   TWCR = (1<<TWINT) | (1<<TWEN);	//Load SLA_W into TWDR Register. Clear TWINT bit
   		  			//in TWCR to start transmission of data
   while (!(TWCR & (1<<TWINT)));	//Wait for TWINT flag set. This indicates that the
   		 		   	//data has been transmitted, and
					//ACK/NACK has been received.
   if ((TWSR & 0xF8) != MT_DATA_ACK)    //Check value of TWI Status Register
      return(1);
   else
      return(0);
}

//*****************************************************
//Function to receive a data byte and send ACKnowledge
//*****************************************************
unsigned char i2c_receiveData_ACK(void)
{
  unsigned char data;
  
  TWCR = (1<<TWEA) | (1<<TWINT) | (1<<TWEN);
  
  while (!(TWCR & (1<<TWINT)));	   	   //Wait for TWINT flag set. This indicates that the
   		 		   					   //data has been received
  if ((TWSR & 0xF8) != MR_DATA_ACK)    //Check value of TWI Status Register
   	  return(ERROR_CODE);
  
  data = TWDR;
  return(data);
}

//******************************************************************
//Function to receive the last data byte (no acknowledge from master
//******************************************************************
unsigned char i2c_receiveData_NACK(void)
{
  unsigned char data;
  
  TWCR = (1<<TWINT) | (1<<TWEN);
  
  while (!(TWCR & (1<<TWINT)));	   	   //Wait for TWINT flag set. This indicates that the
   		 		   					   //data has been received
  if ((TWSR & 0xF8) != MR_DATA_NACK)    //Check value of TWI Status Register
   	  return(ERROR_CODE);
  
  data = TWDR;
  return(data);
}

//**************************************************
//Function to end the i2c communication
//*************************************************   	
void i2c_stop(void)
{
  TWCR = (1<<TWINT) | (1<<TWEN) | (1<<TWSTO);	  //Transmit STOP condition
}  

//**************************************************
//Function to initialize the i2c communication
//Angepasst an ATmega8 und ATmega644
//*************************************************   	
void twi_init(void)
{
  TWCR = 0x00; //disable twi

#ifndef F_CPU
//#define F_CPU 3686400
#define F_CPU 16000000
#endif
#define F_SCL 100000

#if F_CPU >= 16000000
#define TWPS 1  //bei 16MHz
#define VIER_HOCH_TWPS 4
#else
#define TWPS 0  //bei 3.6864MHz
#define VIER_HOCH_TWPS 1
#endif

  TWBR = (F_CPU/F_SCL-16+VIER_HOCH_TWPS) //bei F_CPU=3686400 ergibt das 10 (ohne Rundung 10.43)
         / (2*VIER_HOCH_TWPS);           //bei 16 MHz und TWPS=1 hingegen geht es auf
  TWSR = TWPS;

  TWAR = 0x00; //set slave address
  TWCR = (1<<TWEA) | (1<<TWEN); //enable twi
}
