/* lcd_treiber.c
Dies ist ein Treiber fuer Grafik-LCD-Module mit 128x64 Pixeln
Bisher unterstuetzte Typen sind EADOGM, EADOGL und LCD-Modul auf dem MK3-Board

Version 0.4   letzte Aenderungen: 15.1.2012
Autor:     Rolf Pfister
Homepage:  www.rolfp.ch
Copyright: Opensource Freeware

Initialisierung fuer das MK3-LCD wurde von hier uebernommen: www.Mikrocontroller-4U.de

Installation:
-------------
- lcd_treiber.h   Editieren und Anpassungen ans Zielsystem machen
- lcd_treiber.c und lcd_treiber.h  an gewuenschte Stelle kopieren

Oder statt umkopieren vielleicht sowas machen:
> ln -s /gesamter_Pfad/lcd_treiber.c lcd_treiber.c
> ln -s /gesamter_Pfad/lcd_treiber.h lcd_treiber.h

Benutzung im Hauptprogramm:
---------------------------
#include "lcd_treiber.h"    //Wenn lcd_treiber seperat compiliert werden soll
#include "lcd_treiber.c"    //Wenn lcd_treiber gleich mit dem Hauptprogramm compiliert werden soll

Beispiele wie es jeweils compiliert wird sind im makefile zu finden.
Wenn die voreingestellten Wartefunktionen benutzt werden sollen, muss
der Treiber mindestens mit Optimierung -O1 compiliert werden.

History:
--------
5.11.2011    Erstellung (Rolf Pfister)
3.12.2011    Linien zeichnen verbessert
10.12.2011   Unterstuetzung fuer LCD-Simulator eingebaut
13.12.2011   Senkrechte Linien optimiert. Gefuellte Vierecke implementiert (fillbox).
15.1.2012    Kleine Fehler korrigiert. (Version 0.4)

*/

#include "lcd_treiber.h"
#ifndef LCD_TREIBER_C
#define LCD_TREIBER_C

//------------------------------------------------------------------------------
//------------------ Globale Variablen im Treiber ------------------------------
static uint8_t cursor_x=0,cursor_y=0; //aktuelle Cursor-Position

#ifdef LINIENZEICHNEN
static int8_t drawmodus=JAM1;
#endif

#ifdef FONTTABELLE
static uint8_t text_zeilenabstand=10; //Zeilenabstand fuer Text
void setze_zeilenabstand(uint8_t n) {text_zeilenabstand=n;}
#endif

#if BILDPUFFER>=1
#define LCD_MAXJ (LCD_MAXY/8 * LCD_MAXX)
static uchar bildpuffer1[LCD_MAXJ];
static uint16_t j1=0;
/*
Aufbau des Bildspeichers:
 Das erste Bit des ersten Bytes (Bit 0, LSB) ist die linke obere Ecke.
 Das ganze erste Byte entspricht den erste 8 Pixel in der ersten Spalte des Bildes.
 Das naechste Byte ist die naechste Spalte. 128 Bytes machen dann die ganze Reihe
 aus. Im Ganzen hat es dann 8 solche 8 Pixel dicke Reihen, also Bildhoehe von 64.
*/
#endif
#if BILDPUFFER>=2
static uchar bildpuffer2[LCD_MAXJ]; //(provisorisch: noch nicht benutzt)
static uint16_t j2=0;
#endif

//-----------------------------------------------------------------------------
// Wartefunktionen
// Idealerweise stellt der Benutzer die Wartefunktionen zur Verfuegung und
// macht die entsprechenden defines in lcd_treiber.h.
// Andernfalls werden die folgenden Funktionen benutzt.

#ifndef LCD_SIMULATOR

#ifndef MICROWAIT
#include <util/delay.h>
void MICROWAIT(uint n)  //etwa n Microsekunden warten
{
 while(n>=38)
  {_delay_us(38); n-=38;}
 while(n>=2)
  {_delay_us(2); --n;}
}
#endif

#ifndef MILLIWAIT
#include <util/delay.h>
void MILLIWAIT(uint n)  //mindestens n Millisekunden warten
{
 while(n>0)
  {_delay_ms(1); --n;}
}
#endif

#endif

//-----------------------------------------------------------------------------
//------------------ Grundfunktionen mit dem EADOG ----------------------------
#ifdef LCD_EADOG

void spi_init()
{
 // schon in lcd_init() gemacht:
 //MOSI_DDR |= (1<<MOSI_BIT); //MOSI auf Ausgang setzen
 //SCK_DDR  |= (1<<SCK_BIT);  //SCK auf Ausgang setzen

#ifdef ATMEGA644
 PORTB |= (1<<4); DDRB |= (1<<4); //provisorisch: sicherstellen dass Funkmodul aus ist.
#endif

#ifndef SOFTWARE_SPI

//fuer volle Geschwindigkeit:
 //SPSR = (1<<SPI2X); //Doppelte Geschwindigkeit 
 SPCR = (1<<SPE) | (1<<MSTR) | (1<<CPOL) | (1<<CPHA); //SPI einschalten, Master, SCK=Lowactiv, fck/2 (zusammen mit SPI2X)
 //entgegen den Angaben im Elektor-Buch muss offenbar CPHA auch gesetzt sein!

//fuer kleinste Geschwindigkeit:
 //SPCR = (1<<SPE) | (1<<MSTR) | (1<<CPOL) | (1<<CPHA) | (1<<SPR1) | (1<<SPR0); //SPI einschalten, Master, SCK=Lowactiv, fck/128

 //Einstellbare Geschwindigkeiten:
 // SPR1-0: 00=f/4, 01=f/16, 10=f/64, 11=f/128
 // Verdoppelung wenn SPI2X in SPSR gesetzt wird.

#endif
}

void spi_transmit(char c)
{
#ifdef SOFTWARE_SPI
 int8_t i;
 for(i=0;i<8;i++,c<<=1)
  {if(c&0x80) MOSI_PORT |= (1<<MOSI_BIT);
   else       MOSI_PORT &= ~(1<<MOSI_BIT);
   _delay_us(2);
   SCK_PORT &= ~(1<<SCK_BIT);
   _delay_us(2);
   SCK_PORT |= (1<<SCK_BIT);
 }
#else

 SPDR = c;                  //SPI-Uebertragung starten
 while(!(SPSR&(1<<SPIF))) ; //Warten bis Uebertragung fertig ist

#endif
}

void lcdSendCmd(uchar c)  // Commando senden
{
 A0_PORT &= ~(1<<A0_BIT); // sendet Control-Werte
 CS_PORT &= ~(1<<CS_BIT); // Chip-Select aktivieren (auf 0 setzen)
 spi_transmit(c);
 CS_PORT |= (1<<CS_BIT);  // Chip-Select ausschalten
}

void lcdSendData(uchar c) // Datenbyte senden
{
 A0_PORT |= (1<<A0_BIT);  // sendet Daten
 CS_PORT &= ~(1<<CS_BIT); // Chip-Select aktivieren (auf 0 setzen)
 spi_transmit(c);
 CS_PORT |= (1<<CS_BIT);  // Chip-Select ausschalten
#if BILDPUFFER>=1
 if(j1<LCD_MAXJ) bildpuffer1[j1++] = c;
#endif
}

inline void lcdSendDataCur(uchar c) // Datenbyte senden und Cursor erhoehen
{
 lcdSendData(c);
 cursor_x++;
}

uchar lcdReadData()        // Liest ein Byte an Cursor-Position aus
{
#if BILDPUFFER>=1
 return bildpuffer1[j1];
#else
 return 0; //ohne Bildpuffer koennen die Daten nicht erhalten werden.
#endif
}

void glcd_init()	// Grafik-LCD initialisieren 
{
#ifdef HLED_PORT
  HLED_DDR  |= (1<<HLED_BIT);  //Port fuer Hintergrundlicht auf Ausgang
  HLED_PORT &= ~(1<<HLED_BIT); //Hintergrundlicht aus
#endif

  RST_PORT |= (1<<RST_BIT); RST_DDR |= (1<<RST_BIT); //Bits auf High setzen, und als Ausgang schalten
  CS_PORT  |= (1<<CS_BIT);  CS_DDR  |= (1<<CS_BIT);
  A0_PORT  |= (1<<A0_BIT);  A0_DDR  |= (1<<A0_BIT);
  SCK_PORT |= (1<<SCK_BIT); SCK_DDR |= (1<<SCK_BIT);
  MOSI_PORT |= (1<<MOSI_BIT); MOSI_DDR |= (1<<MOSI_BIT);

  /* Test: Bits durchtesten:
  CS_PORT = (1<<CS_BIT);
  MILLIWAIT(1000);
  RST_PORT |= (1<<RST_BIT);
  MILLIWAIT(1000);
  A0_PORT |= (1<<A0_BIT);
  MILLIWAIT(1000);
  SCK_PORT |= (1<<SCK_BIT);
  MILLIWAIT(1000);
  MOSI_PORT |= (1<<MOSI_BIT);
  MILLIWAIT(1000);
  */

  spi_init(); //SPI einschalten

  // Hardware-Reset Sequenz:
  MILLIWAIT(60);
  RST_PORT &= ~(1<<RST_BIT);
  MILLIWAIT(10);
  RST_PORT |= (1<<RST_BIT);
  MILLIWAIT(10);

  /* Test: andere Hardware-Reset Sequenz:
  MILLIWAIT(60);
  CS_PORT &= ~(1<<CS_BIT); // Chip-Select aktivieren (auf 0 setzen)
  MILLIWAIT(1);
  RST_PORT &= ~(1<<RST_BIT);
  MILLIWAIT(10);
  RST_PORT |= (1<<RST_BIT);
  MILLIWAIT(1);
  CS_PORT |= (1<<CS_BIT);
  MILLIWAIT(20);
  */

  // LCD-Init Sequenz:
  lcdSendCmd(0x40);
  lcdSendCmd(0xA1);
  lcdSendCmd(0xC0);
  lcdSendCmd(0xA6);
  lcdSendCmd(0xA2);
#ifdef LADUNGSPUMPE_OFF
  lcdSendCmd(0x2B); //Ladungspumpe nicht einschalten
#else
  lcdSendCmd(0x2F); //Ladungspumpe einschalten
  lcdSendCmd(0xF8);
  lcdSendCmd(0x00);
#endif
  lcdSendCmd(0x27);
  lcdSendCmd(0x81);
  lcdSendCmd(0x10);
  lcdSendCmd(0xAC);
  lcdSendCmd(0x00);
  lcdSendCmd(0xAF); //Display on
  glcd_cursor(0,0);
}

void glcd_off() //LCD ausschalten
{
 lcdSendCmd(0xAE); //Display off
 lcdSendCmd(0xA5); //Power off
}

//Ende ifdef LCD_EADOG
#endif

//-----------------------------------------------------------------------------
//------------------ Grundfunktionen mit dem MK3 ------------------------------
#ifdef LCD_MK3

void glcd_init()	// Grafik-LCD initialisieren 
{
#ifdef HLED_PORT
  HLED_DDR |= (1<<HLED_BIT);   //Port fuer Hintergrundlicht auf Ausgang
  HLED_PORT &= ~(1<<HLED_BIT); //Hintergrundlicht aus
#endif

  // Daten-Port
  // Als Eingang (ohne Pull-UP)
  MK3_DDR_LCDD=0x00;  
  MK3_PORT_LCDD=0;

  // Control-Port
  // Als Ausgang
  MK3_DDR_LCDC=0xff;
  MK3_PORT_LCDC=0xC3;
  MK3_PORT_LCDC |= LCD_PS;   // Mode=Parallel
  MK3_PORT_LCDC |= LCD_MI;   // Mode=6800
  MK3_PORT_LCDC &= ~LCD_HB;  // Beleuchtung AUS
  MK3_PORT_LCDC &= ~LCD_CLK; // Clk=Lo
  MK3_PORT_LCDC &= ~LCD_WR;  // Write
  MK3_PORT_LCDC &= ~LCD_RS;  // Control
  MK3_PORT_LCDC |= LCD_RST;  // Reset=OFF
  MK3_PORT_LCDC |= LCD_CS;   // CS=Disable

  MILLIWAIT(60);	

  // Hardware-Reset Sequenz
  MK3_PORT_LCDC &= ~LCD_RST;
  MILLIWAIT(10);	
  MK3_PORT_LCDC |= LCD_RST;
  MILLIWAIT(10);

  // LCD-Init Sequenz	 
  lcdSendCmd(0xE2);
  lcdSendCmd(0xA0);
  lcdSendCmd(0xC8);
  lcdSendCmd(0xA2);
  lcdSendCmd(0x2F);
  lcdSendCmd(0x26);
  lcdSendCmd(0xF8);
  lcdSendCmd(0x00);
  lcdSendCmd(0x81);
  lcdSendCmd(0x09);
  lcdSendCmd(0xE0);	
  lcdSendCmd(0xAF);
  glcd_cursor(0,0);
}

void lcdSend1(uchar c)     // sendet Werte zum Display
{
  MK3_PORT_LCDC &= ~LCD_WR; // Write
  MK3_PORT_LCDD = c;        // Daten anlegen
  MK3_DDR_LCDD  = 0xFF;     // Port als Ausgang  
  MK3_PORT_LCDC &= ~LCD_CS; // CS=enable
  MK3_PORT_LCDC |= LCD_CLK; // CLK=HI
  __asm volatile ("nop \n nop \n nop \n");  // mini Pause
  MK3_PORT_LCDC &= ~LCD_CLK;// CLK=LO
  MK3_PORT_LCDC |= LCD_CS;  // CS=disable
}

void lcdSendData(uchar c) // zum schreiben von daten ab Cursor
{
 MK3_PORT_LCDC |= LCD_RS; // sendet Daten-Werte
 lcdSend1(c);
#if BILDPUFFER>=1
 if(j1<LCD_MAXJ) bildpuffer1[j1++] = c;
#endif
}

inline void lcdSendDataCur(uchar c) // Datenbyte senden und Cursor erhoehen
{
 lcdSendData(c);
 cursor_x++;
}

void lcdSendCmd(uchar c)
{
 MK3_PORT_LCDC &= ~LCD_RS; // sendet Control-Werte
 lcdSend1(c);
}

uchar lcdReadData()  // liest ein byte an Cursor-Pos aus
{

#if BILDPUFFER>=1
 return bildpuffer1[j1];
#else

 uchar data;
 MK3_DDR_LCDD = 0x00;      // Port auf Eingang
 MK3_PORT_LCDC |= LCD_RS;  // sendet Daten-Werte
 MK3_PORT_LCDC |= LCD_WR;  // mode = read
 MK3_PORT_LCDC &= ~LCD_CS; // CS=enable
 MK3_PORT_LCDC |= LCD_CLK; // CLK=HI
 __asm volatile ("nop \n nop \n nop \n");  // mini Pause
 MK3_PORT_LCDC &= ~LCD_CLK;// CLK=LO
 MK3_PORT_LCDC |= LCD_CLK; // CLK=HI
 __asm volatile ("nop \n nop \n nop \n");  // mini Pause
 data = MK3_PIN_LCDD;      // daten lesen
 MK3_PORT_LCDC &= ~LCD_CLK;// CLK=LO
 MK3_PORT_LCDC |= LCD_CS;  // CS=disable			
 MK3_DDR_LCDD = 0xFF;      // Port auf Ausgang
 return data;

#endif
}

void glcd_stop() //test
{
 MK3_PORT_LCDC |= LCD_CS; // Chip-Select abschalten (auf 1 setzen)
}

//Ende Grundfunktioen mit dem LCD_MK3
#endif

//-----------------------------------------------------------------------------
//------------------ Grundfunktionen mit dem LCD-Simulator ----------------------------
#ifdef LCD_SIMULATOR

#ifndef __cplusplus
#error "der Simulator sollte mit c++ compiliert werden"
#endif
#include <stdio.h>
#include <iostream>
#include <stdlib.h>
#include <xtekplot1.h>

#define XMAX (1024+8)
#ifdef YZUSATZ
//zusaetzliche Fensterhoehe um ausser LCD noch andere Sachen zu simulieren.
#define YMAX (512+8+YZUSATZ)
#else
#define YMAX (512+8)
#endif
#define TIEFE 8

// Fuer Menu im Simulator um Programm zu beenden:
static int exitflag=0;
void menu_exit() {exitflag=1;}

void rasterzeichnen() //nur im LCD-Simulator um die simulierten Pixel zu markieren
{
 double x,y;
 color(15); //Graue Linien zeichnen
 for(y=0;y<=64;y++)
  {plot(0.0,y,PENUP); plot(128.0,y,PENDOWN);} //waagrechte Linien
 for(x=0;x<=128;x++)
  {plot(x,0.0,PENUP); plot(x,64.0,PENDOWN);} //senkrechte Linien
 color(1);
}

static int simcursor_x=0, simcursor_y=0; //Interne Zaehler im LCD

void lcdSendCmd(uchar c)  // Commando senden
{
 if((c&0xF0)==0x10) simcursor_x = ((c&0x0F)<<4)+(simcursor_x&0x0F);
 else if((c&0xF0)==0x00) simcursor_x = (c&0x0F)+(simcursor_x&0xF0);
 else if((c&0xF8)==0xB0) simcursor_y = (c&0x07)<<3;
 else if(c==0xAF) rasterzeichnen();//Display on
 else if(c==0xAE) //Display off
  {screenclear(); term_refresh();
   printf("lcdSendCmd() Display off\n");
  }
 else if(c==0xA5) //Power off
  {term_exit();
   printf("lcdSendCmd() Power off\n");
  }
}

void pixelsetzen(int x,int y) //nur in lcdSendData() zu verwenden
{
 fillbox(x,64-y-1,x+1,64-y);
}

void pixelloeschen(int x,int y) //nur in lcdSendData() zu verwenden
{
 color(0);
 fillbox(x,64-y-1,x+1,64-y);
 color(1);
}

void menu_raster_aus()
{
 int x,y,j=0,c,i;
#if BILDPUFFER>=1
 for(y=0;y<LCD_MAXY;y+=8)
   {
    for(x=0;x<LCD_MAXX;x++)
     {
      c=bildpuffer1[j++];
      for(i=0;i<8;i++)
       {if((c&(1<<i))!=0) pixelsetzen(x,y+i);
	else            pixelloeschen(x,y+i);
       }
     }
   }
#endif
}
void menu_raster_ein() {menu_raster_aus(); rasterzeichnen();}

void lcdSendData(uchar c) // Datenbyte senden
{
 if(simcursor_x<0x80)
  for(int i=0,y=simcursor_y,maske=1;i<8;i++,y++,maske<<=1)
   {
    if(c&maske) pixelsetzen(simcursor_x,y);
    else  pixelloeschen(simcursor_x,y);
   }
 if(simcursor_x<0x83) simcursor_x++;
#if BILDPUFFER>=1
 if(j1<LCD_MAXJ) bildpuffer1[j1++] = c;
#endif
}

inline void lcdSendDataCur(uchar c) // Datenbyte senden und Cursor erhoehen
{
 lcdSendData(c);
 cursor_x++;
}

uchar lcdReadData()        // Liest ein Byte an Cursor-Position aus
{
#if BILDPUFFER>=1
 return bildpuffer1[j1];
#else
 return 0; //ohne Bildpuffer koennen die Daten nicht erhalten werden.
#endif
}

void glcd_init()	// Grafik-LCD initialisieren 
{
 //int maxcol;
 int breite,hoehe,tiefe,visklasse;
 double xmin=-4.0,ymin=-4.0,xmax=128+4,ymax=64+4;
#ifdef YZUSATZ
 ymin -= YZUSATZ/8;
#endif
 //tek_setdebug(1);//test
 getmaxsize(&breite,&hoehe,&tiefe,&visklasse);
 if(tiefe>TIEFE) tiefe=TIEFE;
 if(breite>XMAX) breite=XMAX;
 if(hoehe>YMAX) hoehe=YMAX;
 //maxcol=(1<<TIEFE);
 setsize(breite,hoehe,tiefe);
 setmenu(1,"File");
 setmenu(1,"Raster ein",&menu_raster_ein);
 setmenu(1,"Raster aus",&menu_raster_aus);
 setmenu(1,"Exit",&menu_exit);
 inital(xmin,ymin,xmax,ymax); /* Grafikfenster oeffnen */
 lcdSendCmd(0xAF); //Display on
 glcd_cursor(0,0);
}

void glcd_off() //LCD ausschalten
{
 lcdSendCmd(0xAE); //Display off
 lcdSendCmd(0xA5); //Power off
}

//Ende ifdef LCD_SIMULATOR
#endif

//-----------------------------------------------------------------------------
//--------------- Funktionen fuer alle Grafik-LCDs ----------------------------

//-----------------------------------------------------------------------------
#ifdef HLED_PORT

void glcd_licht(bool on)  // Hintergrundlicht Ein/Aus-Schalten
{
 if(on==false) HLED_PORT &= ~(1<<HLED_BIT); // Beleuchtung AUS
 else          HLED_PORT |= (1<<HLED_BIT);  // Beleuchtung EIN
}

#endif

//-----------------------------------------------------------------------------
void glcd_clear()      // Loeschen des LCD
{
 uint8_t x,y;

 for(y=0;y<LCD_MAXY;y+=8)
  {
   glcd_cursor(0,y);
   for(x=0;x<LCD_MAXX;x++)
      lcdSendData(0);
  }
 glcd_cursor(0,0);
}

//-----------------------------------------------------------------------------
void glcd_cursor(uint8_t x,uint8_t y)  // Setzen des Cursors
{
  // X = X-Koordinate in Pixel (0...MAXX-1)
  // Y = Y-Koordinate in Pixel (0...MAXY-1)

  // X-Koordinate
  lcdSendCmd(0x10+(x>>4));
  lcdSendCmd(0x00+(x&0x0F));

  // Y-Koordinate
  lcdSendCmd(0xB0+(y>>3));

#if BILDPUFFER>=1
  j1 = (y>>3)*LCD_MAXX + x;
#endif

  cursor_x=x;
  cursor_y=y;  //aktuelle Cursor-Position speichern
}

//-----------------------------------------------------------------------------
//--------------------- Hoehere Funktionen ------------------------------------
#ifdef LINIENZEICHNEN
void glcd_drawmode(int8_t mode)        //Zeichnungsmodus setzen: JAM0, JAM1, oder COMPLEMENT
{
 drawmodus=mode;
}

void punktsetzen(uint8_t maske)  //in glcd_lineto() und glcd_setzepunkt() zu benutzen
{
 uint8_t c=lcdReadData();
 switch(drawmodus)
  {
   case JAM2: //vorlaeufig wie COMPLEMENT, eigentlich fuer Farb-LCDs reserviert
   case COMPLEMENT:    c ^= maske;  break; //Pixelwert umkehren
   case JAM0:          c &= ~maske; break; //Pixel loeschen
   case JAM1: default: c |= maske;         //Pixel setzen
  }
 lcdSendData(c);
}

void glcd_setzepunkt(uint8_t x,uint8_t y)  //Einzelner Grafik-Punkt setzen
{
 glcd_cursor(x,y);
 punktsetzen(1<<(y&7));
}

uint8_t maske7(uint8_t y)
{
 uint8_t maske,m;
 maske = m = 1<<y;
 while(++y<8)
  maske |= (m<<=1);
 return maske;
}

uint8_t maske8(uint8_t y1,uint8_t y2)
{
 uint8_t maske,m;
 maske = m = 1<<y1;
 while(++y1<=y2)
  maske |= (m<<=1);
 return maske;
}

void senkrechte_linie(uint8_t y1,uint8_t y2)  //Senkrechte Linie von y1 bis y2 zeichnen (y2 muss groesser y1 sein),
{                                             //inklusive letzter Punkt. Cursor steht danach an undefinierter Position
 uint8_t maske,page1,page2;
 page1 = y1 & ~7;
 page2 = y2 & ~7;
 maske=maske7(y1&7);
 while(page1!=page2)
  {
   glcd_cursor(cursor_x,page1);
   punktsetzen(maske);
   y1 = page1 += 8;
   maske=0xFF;
  }
 maske=maske8(y1&7,y2&7);
 glcd_cursor(cursor_x,page1);
 punktsetzen(maske);
}

void dicke_senkrechte_linie(uint8_t y1,uint8_t y2,uint8_t dx) //Senkrechte Linie von y1 bis y2 zeichnen, mit Dicke dx,
{                                                             //y2 muss groesser y1 und dx muss mindestens 1 sein.
 uint8_t maske,page1,page2,x,x1,x2;
 x1=cursor_x; x2=x1+dx;
 page1 = y1 & ~7;
 page2 = y2 & ~7;
 maske=maske7(y1&7);
 while(page1!=page2)
  {
   for(x=x1;x<x2;x++)
    {
     glcd_cursor(x,page1);
     punktsetzen(maske);
    }
   y1 = page1 += 8;
   maske=0xFF;
  }
 maske=maske8(y1&7,y2&7);
 for(x=x1;x<x2;x++)
  {
   glcd_cursor(x,page1);
   punktsetzen(maske);
  }
}

void glcd_lineto(uint8_t x,uint8_t y)  //Linie vom aktuellen Cursor bis zur neuen Position zeichnen
{                                      //Ohne letzter Punkt zu setzen (besser fuer Linienzuege)
 uint8_t maske;

 if(y==cursor_y)  //waagrechte Linie ?
  {
   maske = 1<<(y&7);
   if(cursor_x<x)
      {
       do {punktsetzen(maske);}
       while(++cursor_x!=x);
      }
   else if(cursor_x>x)
      {
       do {punktsetzen(maske); glcd_cursor(--cursor_x,y);}
       while(cursor_x!=x);
      }
  }

 else if(x==cursor_x)  //senkrechte Linie ?
  {
   if(y-1>cursor_y)      senkrechte_linie(cursor_y,y-1);
   else if(y+1<cursor_y) senkrechte_linie(y+1,cursor_y);
   glcd_cursor(x,y);
  }

 else  //sonst schraege Linie
  {
   //Bresenham-Algorithmus Kompakte Variante (von Wikipedia uebernommen):
   int8_t dx =   (x>=cursor_x) ? x-cursor_x : cursor_x-x,   sx = cursor_x<x ? 1 : -1;
   int8_t dy = -((y>=cursor_y) ? y-cursor_y : cursor_y-y),  sy = cursor_y<y ? 1 : -1; 
   int err = dx+dy, e2; /* error value e_xy */
   maske = 1<<(cursor_y&7);
   while(cursor_x!=x || cursor_y!=y)
    {
     punktsetzen(maske);
     e2 = 2*err;
     if(e2 > dy) {err += dy; cursor_x += sx; } /* e_xy+e_x > 0 */
     if(e2 < dx) {err += dx; cursor_y += sy; maske = 1<<(cursor_y&7);} /* e_xy+e_y < 0 */
     glcd_cursor(cursor_x,cursor_y);
    }
  }
}

void glcd_lineto_last(uint8_t x,uint8_t y)  //Linie vom aktuellen Cursor bis zur neuen Position inklusive Letztem Punkt
{
 glcd_lineto(x,y);
 glcd_setzepunkt(x,y);
}

void glcd_drawbox(uint8_t x1,uint8_t y1,uint8_t x2,uint8_t y2) //Viereck zeichnen
{
 glcd_moveto(x1,y1);
 glcd_lineto(x2,y1);
 glcd_lineto(x2,y2);
 glcd_lineto(x1,y2);
 glcd_lineto(x1,y1);
}

//Ende LINIENZEICHNEN
#endif

#ifdef FIGURENFUELLEN
void glcd_fillbox(uint8_t x1,uint8_t y1,uint8_t x2,uint8_t y2) //gefuelltes Viereck zeichnen, es muss x1!=x2 und y1!=y2 sein
{
 uint8_t h;
 if(x1>x2) {h=x1; x1=x2; x2=h;} //sicherstellen dass x2>x1
 if(y1>y2) {h=y1; y1=y2; y2=h;} //sicherstellen dass y2>y1
 glcd_moveto(x1,y1);
 dicke_senkrechte_linie(y1,y2,x2-x1+1);
}
//Ende FIGURENFUELLEN
#endif

#ifdef FONTTABELLE

#define FONTBREITE 5
#define FONTHOEHE  8

/************************************************************************/
/* Filename: font90_8x5                                                 */
/* Version : 1.00                                                       */
/* Date    : 08.02.2010                                                 */
/* Owner   : UB                                                         */
/* leicht abgeaendert: RP                                               */
/*----------------------------------------------------------------------*/
/* Remark  : Font-File zum LCD-Modul                                    */
/*           Schriftgroesse : 8x5 bit                                   */
/*                            wobei ein Zeichen aus 5x7 Pixel besteht   */
/*----------------------------------------------------------------------*/
/*                                                                      */
/*  z.B. Ascii 82 = R -> 82,0x7F,0x09,0x19,0x29,0x46,                   */
/*                                                                      */
/*                  (linke obere Ecke = LSB                             */
/*                                                                      */
/*                       12345 (Spalte)                                 */
/*                                                                      */
/*                  01h  ####:         = 0x7F (Spalte-1)                */
/*                  02h  #:::#         = 0x09 (Spalte-2)                */
/*                  04h  #:::#         = 0x19 (Spalte-3)                */
/*                  08h  ####:         = 0x29 (Spalte-4)                */
/*                  10h  #:#::         = 0x46 (Spalte-5)                */
/*                  20h  #::#:                                          */
/*                  40h  #:::#                                          */
/*                  80h  :::::                                          */
/*                                                                      */
/************************************************************************/

typedef struct {              // def von Font 8x5
//char ascii;                 // ascii-nr des buchstabens
  char b0;
  char b1;
  char b2;
  char b3;
  char b4;
} Font8x5_t;


Font8x5_t font8x5[] = {
  {/*32,*/0x00,0x00,0x00,0x00,0x00}, // Space
  {/*33,*/0x00,0x00,0x4F,0x00,0x00}, // !
  {/*34,*/0x00,0x07,0x00,0x07,0x00}, // "
  {/*35,*/0x14,0x7F,0x14,0x7F,0x14}, // #
  {/*36,*/0x24,0x2A,0x7F,0x2A,0x12}, // $
  {/*37,*/0x63,0x13,0x08,0x64,0x63}, // %
  {/*38,*/0x30,0x4E,0x5D,0x26,0x50}, // &
  {/*39,*/0x04,0x02,0x01,0x00,0x00}, // '
  {/*40,*/0x00,0x1C,0x22,0x41,0x00}, // (
  {/*41,*/0x00,0x41,0x22,0x1C,0x00}, // )
  {/*42,*/0x14,0x08,0x3E,0x08,0x14}, // *
  {/*43,*/0x08,0x08,0x3E,0x08,0x08}, // +
  {/*44,*/0xC0,0x30,0x00,0x00,0x00}, // ,
  {/*45,*/0x08,0x08,0x08,0x08,0x08}, // -
  {/*46,*/0x00,0x60,0x60,0x00,0x00}, // .
  {/*47,*/0x60,0x10,0x08,0x04,0x03}, // /
  {/*48,*/0x3E,0x41,0x41,0x3E,0x00}, // 0
  {/*49,*/0x00,0x42,0x7F,0x40,0x00}, // 1
  {/*50,*/0x72,0x49,0x49,0x49,0x46}, // 2
  {/*51,*/0x21,0x41,0x49,0x4D,0x33}, // 3
  {/*52,*/0x18,0x14,0x12,0x7F,0x10}, // 4
  {/*53,*/0x47,0x45,0x45,0x45,0x39}, // 5
  {/*54,*/0x38,0x4C,0x4A,0x49,0x30}, // 6
  {/*55,*/0x01,0x71,0x09,0x05,0x03}, // 7
  {/*56,*/0x36,0x49,0x49,0x49,0x36}, // 8
  {/*57,*/0x06,0x49,0x29,0x19,0x0E}, // 9
  {/*58,*/0x00,0x00,0x44,0x00,0x00}, // :
  {/*59,*/0x40,0x34,0x00,0x00,0x00}, // ;
  {/*60,*/0x08,0x14,0x22,0x41,0x41}, // <
  {/*61,*/0x14,0x14,0x14,0x14,0x14}, // =
  {/*62,*/0x41,0x41,0x22,0x14,0x08}, // >
  {/*63,*/0x02,0x01,0x51,0x09,0x06}, // ?
  {/*64,*/0x3E,0x41,0x5D,0x55,0x5E}, // @
  {/*65,*/0x7E,0x09,0x09,0x09,0x7E}, // A
  {/*66,*/0x7F,0x49,0x49,0x49,0x36}, // B
  {/*67,*/0x3E,0x41,0x41,0x41,0x22}, // C
  {/*68,*/0x7F,0x41,0x41,0x22,0x1C}, // D
  {/*69,*/0x7F,0x49,0x49,0x49,0x41}, // E
  {/*70,*/0x7F,0x09,0x09,0x09,0x01}, // F
  {/*71,*/0x3E,0x41,0x41,0x49,0x7A}, // G
  {/*72,*/0x7F,0x08,0x08,0x08,0x7F}, // H
  {/*73,*/0x00,0x41,0x7F,0x41,0x00}, // I
  {/*74,*/0x20,0x40,0x41,0x3F,0x01}, // J
  {/*75,*/0x7F,0x08,0x14,0x22,0x41}, // K
  {/*76,*/0x7F,0x40,0x40,0x40,0x40}, // L
  {/*77,*/0x7F,0x02,0x0C,0x02,0x7F}, // M
  {/*78,*/0x7F,0x06,0x08,0x30,0x7F}, // N
  {/*79,*/0x3E,0x41,0x41,0x41,0x3E}, // O
  {/*80,*/0x7F,0x09,0x09,0x09,0x06}, // P
  {/*81,*/0x3E,0x41,0x51,0x21,0x5E}, // Q
  {/*82,*/0x7F,0x09,0x19,0x29,0x46}, // R
  {/*83,*/0x46,0x49,0x49,0x49,0x31}, // S
  {/*84,*/0x01,0x01,0x7F,0x01,0x01}, // T
  {/*85,*/0x3F,0x40,0x40,0x40,0x3F}, // U
  {/*86,*/0x1F,0x20,0x40,0x20,0x1F}, // V
  {/*87,*/0x7F,0x20,0x18,0x20,0x7F}, // W
  {/*88,*/0x63,0x14,0x08,0x14,0x63}, // X
  {/*89,*/0x07,0x08,0x78,0x08,0x07}, // Y
  {/*90,*/0x61,0x51,0x49,0x45,0x43}, // Z
  {/*91,*/0x7F,0x41,0x41,0x00,0x00}, // [
  {/*92,*/0x02,0x04,0x08,0x10,0x20}, // \ (Backslash)
  {/*93,*/0x00,0x00,0x41,0x41,0x7F}, // ]
  {/*94,*/0x04,0x02,0x01,0x02,0x04}, // ^
  {/*95,*/0x80,0x80,0x80,0x80,0x80}, // _
  {/*96,*/0x00,0x01,0x02,0x04,0x00}, // `
  {/*97,*/0x38,0x44,0x44,0x28,0x7C}, // a
  {/*98,*/0x7F,0x28,0x44,0x44,0x38}, // b
  {/*99,*/0x38,0x44,0x44,0x44,0x00}, // c
 {/*100,*/0x38,0x44,0x44,0x28,0x7F}, // d
 {/*101,*/0x38,0x54,0x54,0x54,0x18}, // e
 {/*102,*/0x00,0x08,0x7E,0x09,0x00}, // f
 {/*103,*/0x18,0xA4,0xA4,0x98,0x7C}, // g
 {/*104,*/0x7F,0x08,0x04,0x04,0x78}, // h
 {/*105,*/0x00,0x44,0x7D,0x40,0x00}, // i
 {/*106,*/0x40,0x80,0x80,0x7D,0x00}, // j
 {/*107,*/0x7F,0x10,0x18,0x24,0x42}, // k
 {/*108,*/0x00,0x41,0x7F,0x40,0x00}, // l
 {/*109,*/0x7C,0x04,0x78,0x04,0x78}, // m
 {/*110,*/0x7C,0x04,0x04,0x78,0x00}, // n
 {/*111,*/0x38,0x44,0x44,0x44,0x38}, // o
 {/*112,*/0xFC,0x18,0x24,0x24,0x18}, // p
 {/*113,*/0x18,0x24,0x24,0x18,0xFC}, // q
 {/*114,*/0x7C,0x08,0x04,0x04,0x04}, // r
 {/*115,*/0x48,0x54,0x54,0x54,0x24}, // s
 {/*116,*/0x00,0x04,0x3F,0x44,0x00}, // t
 {/*117,*/0x3C,0x40,0x40,0x40,0x7C}, // u
 {/*118,*/0x0C,0x30,0x40,0x30,0x0C}, // v
 {/*119,*/0x3C,0x40,0x20,0x40,0x3C}, // w
 {/*120,*/0x44,0x28,0x10,0x28,0x44}, // x
 {/*121,*/0x1C,0xA0,0xA0,0xA0,0x7C}, // y
 {/*122,*/0x44,0x64,0x54,0x4C,0x44}, // z
 {/*123,*/0x08,0x36,0x41,0x00,0x00}, // {
 {/*124,*/0x00,0x00,0x77,0x00,0x00}, // |
 {/*125,*/0x00,0x00,0x41,0x36,0x08}, // }
 {/*126,*/0x10,0x08,0x08,0x08,0x04}, // ~
#define VIERECKZEICHEN 127
 {/*127,*/0xFF,0x81,0x81,0x81,0xFF}, // Viereck fuer nicht darstellbare Zeichen

#ifdef UTF8ZEICHEN
#define GRADZEICHEN 128
 {0x0E,0x11,0x11,0x11,0x0E}, // ^o
#define AUMLAUT 129
 {0x78,0x15,0x14,0x15,0x78}, // "A
#define OUMLAUT 130
 {0x7C,0x45,0x44,0x45,0x7C}, // "O
#define UUMLAUT 131
 {0x3C,0x41,0x40,0x41,0x3C}, // "U
#define aUMLAUT 132
 {0x38,0x45,0x44,0x29,0x7C}, // "a
#define oUMLAUT 133
 {0x38,0x45,0x44,0x45,0x38}, // "o
#define uUMLAUT 134
 {0x3C,0x41,0x40,0x41,0x7C}, // "u
#define SHARPSS 135
 {0x7E,0x01,0x49,0x4C,0x32}, // Deusches Scharfes S
#define EUROZEICHEN 136
 {0x3E,0x55,0x55,0x41,0x22}, // Eurozeichen
#define DREIFACHBINDUNG 137
 {0x2A,0x2A,0x2A,0x2A,0x2A}, // Wie = aber mit 3 Strichen
//weitere Zeichen definierbar bis maximal 225
#endif

};
const uint8_t ANZAHLZEICHEN = sizeof(font8x5)/sizeof(Font8x5_t);  // Anzahl der Zeichen

//------------------------------------------------------------------
void glcd_printc(uchar c) // schreibe ein Zeichen erhoehe Cursor
{                         // (muss uchar sein weil auch Nicht-Ascii-Zeichen unterstuetzt werden sollen)
 uint8_t n;
#ifdef UTF8ZEICHEN
 static uchar u1=0,u2=0; //zum Zwischenspeichern der ersten 2 Bytes eines UTF8-Zeichens
#endif

 if(c<' ') return; //Steuerzeichen ignorieren
 n = c-' ';

#ifdef UTF8ZEICHEN
 if(u1!=0 || n>=ANZAHLZEICHEN)
   {
    if(u1==0)        //ist es erstes Byte eines UTF8-Zeichens?
     {u1=c; return;} //ja: speichern und return
    if(u1==0xC2)
     {
      if(c==0xB0) n=GRADZEICHEN;
      //else if... //hier weitere 2-Byte UTF8-Zeichen definierbar
      else n=VIERECKZEICHEN; //Viereck fuer unbekanntes Zeichen
     }
    else if(u1==0xC3)
     {
      if(c==0x84)      n=AUMLAUT;
      else if(c==0x96) n=OUMLAUT;
      else if(c==0x9C) n=UUMLAUT;
      else if(c==0xA4) n=aUMLAUT;
      else if(c==0xB6) n=oUMLAUT;
      else if(c==0xBC) n=uUMLAUT;
      else if(c==0x9F) n=SHARPSS;
      //else if... //hier weitere 2-Byte UTF8-Zeichen definierbar
      else n=VIERECKZEICHEN;
     }
    else if(u1==0xE2)
     {
      if(u2==0)        //ist es zweites Byte eines 3-Byte UTF8-Zeichens?
       {u2=c; return;} //ja: speichern und return
      if(u2==0x82)
       {
	if(c==0xAC) n=EUROZEICHEN;
	//else if... //hier weitere 3-Byte UTF8-Zeichen definierbar
	else n=VIERECKZEICHEN; //unbekanntes Zeichen
       }
      //else if(u2==0x... //hier waeren weitere 3-Byte UTF8-Zeichen mit anderem 2.Byte definierbar
      else n=VIERECKZEICHEN;
     }
    else  //andernfalls unbekanntes Zeichen
     {
      n=VIERECKZEICHEN;
     }
    u1=u2=0; //beide ruecksetzen fuers naechste Zeichen
    if(n==VIERECKZEICHEN         //Wenn ein ungueltiges UTF8-Zeichen erkannt wurde,
       && c-' '<ANZAHLZEICHEN)   //aber das aktuelle Zeichen ein gueltiges Zeichen ist,
                     {
		      n = c;     //dann dieses Zeichen uebernehmen. (und das ungueltige UTF8 ignorieren)
		     }
    n -= ' ';
   }

#else

 if(n >= ANZAHLZEICHEN)
   {
    n=VIERECKZEICHEN-' '; //Viereck fuer nicht darstellbare Zeichen
   }

#endif

 glcd_printb(font8x5[n].b0);
 glcd_printb(font8x5[n].b1);
 glcd_printb(font8x5[n].b2);
 glcd_printb(font8x5[n].b3);
 glcd_printb(font8x5[n].b4);
 glcd_printb(0x00);
}

void glcd_printb(uchar c) // Schreibe ein Byte eines Buchstabens
{
 if((cursor_y&7)==0)      // wenn y durch 8 teilbar ist
   {
    lcdSendDataCur(c);    // dann einfach Daten senden und Cursor erhoehen
   }
 else                     //sonst etwas komplizierter:
   {
    uint8_t byte1, byte2, x=cursor_x, y=cursor_y;
    uint16_t word1;

    byte1 = lcdReadData();     // erstes Byte vom LCD lesen
    glcd_cursor(x,y+8);        // Cursor auf zweites Byte setzen
    byte2 = lcdReadData();     // zweites Byte vom LCD lesen
    word1 = byte1+(byte2<<8);  // die beiden Bytes in 16-Bit speichern
    word1 &= ~(0x00FF<<(y&7)); // entsprechende Bits im 16-Bit-Wert loeschen
    word1 |= c<<(y&7);         // das zu schreibende Byte dort einfuegen
    byte1 = word1&0xFF;        // Der 16-Bit-Wert wieder in die beiden Bytes kopieren
    byte2 = word1>>8;
    lcdSendData(byte2);        // zweites Byte ans LCD schicken
    glcd_cursor(x,y);          // wieder aktueller Cursor setzen
    lcdSendDataCur(byte1);     // erstes Byte ans LCD schicken und Cursor erhoehen
   }
}

void glcd_print(const char *ptr) // zum schreiben von texten ab Cursor
{
 char c;

 while((c= *ptr++)!=0)
  {
   // Sonderzeichen behandeln:
   if(c=='\t') c=' '; //Tabulator durch Leerzeichen ersetzen
   else if(c=='\r')
    {
     if(*ptr=='\n')  c = *ptr++;  //CR LF durch LF ersezen (LF = Zeilentrenner)
     else c = '\n';  //einzelnes CR auch als Zeilentrenner behandeln
    }

   // Zeilentrenner behandeln:
   if(c=='\n' || cursor_x >= LCD_MAXX-FONTBREITE)  //Zeilentrennzeichen, oder Zeile voll?
      {
       glcd_cursor(0,cursor_y+text_zeilenabstand); //ja: neue Zeile
      }

   glcd_printc(c);
  }

}

//Ende #ifdef FONTTABELLE
#endif

//Ende #ifndef LCD_TREIBER_C
#endif
