/* tarech2.cc       letzte Aenderung: 29.8.2016
Selbstbau-Taschenrechner
*/
#define VERSION "0.16"
/*
 Prozessor: Atmega2560, 16MHz Quarz
 Schaltung: tarech2.png

   Grafik-LCD EADOGL an PortB:
     PB1 (SCK)  = SCL
     PB2 (MOSI) = SI
     PB4        = CS
     PB5        = RST
     PB6        = A0
     PB7        = Hintergrundbeleuchtung

   Tastatur: 5*6-Tastenmatrix an PE0..PE4 und PH0..PH5

   Ein/Aus-Schalter: Taster an PD5 (L-aktiv), Ausgang an PD4 (L=Ausschalten)

   Akku-Spannung messen:
     PF0 (ADC0) 50k auf GND, 150k auf VCC
     somit Akkuspannung = 3 * gemessene Spannung + Spannungsabfall am IRF6920

   Echtzeituhr DS1307: PD0 (SCL) und PD1 (SDA) fuer I2C benutzt.

   myUSBtoUART: PJ0 (RxD3), PJ1 (TxD3) direkt auf dem MyAvrStamp
   Micro-SD: SS (PB0), MISO (PB3), MOSI (PB2), SCK (PB1) direkt auf dem MyAvrStamp

 Autor: Rolf Pfister
 Copyright: Freeware

 History:
 1.1.2012      Erstellung ausgehend von tarech.cc (Version 0.08, ATmega1284P)
 11.1.12       Molekulargewichts-Berechnung
 11.2.12       Behandlung von Ein/Aus-Transistor beim Einschalten (Ausschalten fehlt noch)
 29.2.12       Anpassung an neue Version von mgkern.c
 14.4.12       mgkernf2.c erneuert
 24.9.12       Test fuer 64KB Problem, provisorisch geloest mit "make install3" (siehe makefile)
               Tastenmatrix-Abfrage fuer MK3-Board-Variante eingebaut
	       MGW-Rechner: beim abschalten aktuelle Formel im EEPROM speichern
 28.9.12       Tippfehler korrigiert
 2.10.12       Register vom Taschenrechner auch im EEPROM speichern
               Hochzeichen angepasst
 4.10.12  0.14 Geraet ausschalten bei laengerem Nichtbenutzen.
               Fehler in tastaturabfrage() korrigiert. Bei nicht gedrueckter Taste: return 0;
 13.10.12 0.15 MGW-Rechner: mehrere Formeln im EEPROM
 29.10.        Tippfehler im Kommentar korrigiert und Versionsnummer 0.15 gesetzt
 29.8.16  0.16 Fehler bei acos() korrigiert.

*/
//#define TEST64KB //test
//#define OHNEMGW //test

//#define MGWFLOAT //fuer Molekulargewichts-Berechnungen float verwenden
                 //Auskommentieren um genauere Fliesszahl-Klasse zu verwenden

//#define MATRIXQUER //fuer Tastenmatrix im Querformat. Fuer Hochformat auskommentieren.

#define F_CPU 16000000UL

#include <stdio.h>
//#include <stdlib.h>
//#include <math.h>
#include <string.h>
#include <ctype.h>

//#define RTCDEBUG  //zur Fehlersuche
//#define DEBUG  1
//#define DEBUG1
//#define FLIESSZAHLTEST

#ifdef LCD_SIMULATOR
static const char* tasten_text[30] =
 {
  "Gold","Blue","Red","POP","Cor","/",
  "7",   "8",   "9",  "A",  "E",  "*",
  "4",   "5",   "6",  "B",  "F",  "-",
  "1",   "2",   "3",  "C",  "Exp","+",
  "0",   ".",  "+/-", "D",  "<>", "Ent"
 };
static const char* tasten_textgoldblue[30] =
 {
  "",        "",        "",        "    PUSH","    ROLv","Deg  Rad",
  "    N_A ","MW    g ","      c ","Sin Asin","ln   e^x","     Del",
  "     Pi ","Iso  |x|","     Int","Cos Acos","log 10^x","    Edit",
  "    Time","HMS  HR ","",        "Tan Atan","     y^x","",
  "    Date","    1/x ","     Hex","Sqrt x^2","     Dez",""
 };
static const char* tasten_textgreenred[30] =
 {
  "",         "Caps     ","Dig/Let  "," Help    ","       ","\"\' BRNE",
  "<=   Kale","=>   MGW "," Elemente"," Run/Stop"," Konst ","&~  BRPL",
  "ij   STO ","nm   RCL ","rq   STOx","t   Lear ","w  RCLx","z=  BRMI",
  "hg   Tset","kl       ","op       ","s   LBL  ","uv     ","xy  BREQ",
  "([{< Dset",",:;!?    ",">}])     ","@#$%     ","\\|_^  ",""
 };
#define PIN_SIMULATION
#define TASTEN_SIMULATION
#define TASTEN_BESCHRIFTUNG
#define UHR_SIMULATION uhr_simulation
#define ZEITERHOEHEN_SIMULATION zeiterhoehen_simulation
#include "lcdsim.cc"
#else
#include <avr/io.h>
#include <avr/interrupt.h>
#include "ulong.h"
#endif

#include "lcd_treiber.c"  //Treiber fuer Grafik-LCD 128x64
#include "fliesszahlklasse.cc"

/**** Vordeklarationen ****/
void init_matrix();
void tastaturcheck();
char tastaturabfrage();
int8 tastaturabfragenr();
//void variable_feiertage_berechnen();
void glcd_printzeile(int zeile,const char *cstr,
		     int p1=0,int p2=0,int p3=0,int p4=0);
void tarech_init();
void tarech_tastenreaktion(uchar taste);
void tarech_refresh();
bool tarech_in_eingabemodus();
void aktuelle_formel_im_eeprom_speichern();
void aktuelle_formel_vom_eeprom_einlesen();
void stapel_vom_eeprom_einlesen(uint8 i,Fliesszahl *pz);
void tarech_stapel_im_eeprom_speichern();
void tarech_regist_vom_eeprom_einlesen();
void tarech_regist_im_eeprom_speichern();
//Fliesszahl calcln2();
Fliesszahl log(Fliesszahl x);  //natuerlicher Logarithmus
Fliesszahl exp(Fliesszahl x);  //e^x
void mgrechner_init();
void mgrechner_tastenreaktion(uint8 tastenr);
Fliesszahl mgw_ausrechnen(char *text,char *flags);
void mg_error_reset();
void mgw_formel_rotieren();
void geraet_ausschalten();

/**** Zeitbehandlung ****/
static volatile uint millisec=0;
static volatile uchar sec=0,min=35,stunden=19;
static volatile uchar tage=20,monat=11; //aktuelles Datum
static volatile uint jahr=2010;
static char monatstage[13]={31,31,28,31,30,31,30,31,31,30,31,30,31};
static volatile char wochentag=6; //0=Montag ... 6=Sonntag
static volatile char sommerzeit=0; //0=Winterzeit 1=Sommerzeit

bool istschaltjahr(int y)
{
 return (y%4==0 && (y%100!=0 || y%400==0));
}
void schaltjahrtest()
{
 if(istschaltjahr(jahr)) monatstage[2]=29;
 else monatstage[2]=28;
}
void wochentagberechnen()
{
 int j=jahr-2001,t,m;
 t=j*365+j/4-j/100+j/400+tage+6;
 for(m=1;m<monat;m++) t+=monatstage[m];
 wochentag=t%7;
}
void schaltjahr_und_wochentagberechnen()
{
 schaltjahrtest();
 wochentagberechnen();
 //variable_feiertage_berechnen();
}

/** Feiertage: ** /
const char NFEST=6;//Anzahl feste Feiertage
const char festfeiertag[NFEST]={1, 2, 1, 1, 25, 26};
const char festfeiermon[NFEST]={1, 1, 5, 8, 12, 12};
const char NVARI=4;//Anzahl von Ostern abhaengige Feiertage
static char varifeiertag[NVARI];
static char varifeiermon[NVARI];
void variable_feiertage_berechnen()
{
 int a,b,c,k,p,q,M,N;
 char d,e,tag,i;
 const char mon=3;
 //Osterformel nach Gauss:
 a=jahr%19;
 b=jahr%4;
 c=jahr%7;
 k=jahr/100;
 p=(8*k+13)/25;
 q=k/4;
 M=(15+k-p-q)%30;
 N=(4+k-q)%7;
 d=(19*a+M)%30;
 e=(2*b+4*c+6*d+N)%7;
 tag=22+d+e;
 varifeiertag[0]=tag+1; varifeiermon[0]=mon; //Ostermontag
 varifeiertag[1]=tag-2; varifeiermon[1]=mon; //Karfreitag
 varifeiertag[2]=tag+39; varifeiermon[2]=mon; //Auffahrt
 varifeiertag[3]=tag+50; varifeiermon[3]=mon; //Pfingstmontag
 for(i=0;i<NVARI;i++)
   while(varifeiertag[i]>monatstage[varifeiermon[i]])
    {varifeiertag[i] -= monatstage[varifeiermon[i]]; varifeiermon[i]++;}
}

bool istfeiertag()
{
 char i;
 for(i=0;i<NFEST;i++)
   if(tage==festfeiertag[i] && monat==festfeiermon[i]) return true;
 for(i=0;i<NVARI;i++)
   if(tage==varifeiertag[i] && monat==varifeiermon[i]) return true;
 return false;
}
/ **/

// Quarz moeglichst genau ausmessen und hier eintragen
// in Hertz und Tausendstel Hertz:

const ulong Hz=16000000,MilliHz=0; //exakt 16MHz
//Beispiel: 16MHz ist nach n Tagen z sec vorgegangen:
//nach n Tagen sind also 16000000*(3600*24*n+z) Takte vergangen
//Frequenz ist also:
// x = (3600*24*n+z)*16000000 / (3600*24*n)
//   = 16000000 + z*16000000/(3600*24*n)
//const ulong Hz=16002000,MilliHz=0;

const uint Rest1 = Hz%1000;//jede Sekunde zu korrigierender Fehler
//const uint Rest2 = MilliHz*6/100;//jede Minute
//const uint Rest3 = ((MilliHz*6-Rest2*100)*60+50)/100;//jede Stunde
const uint NormalerCompwert=Hz/1000-1;
//der Compiler sollte bis hier noch keinen Code erzeugt haben.
//(der Quarz ist wahrscheinlich zu wenig stabil als dass Rest2 und Rest3
//eine Rolle spielen wuerden. Also MilliHz vorlaeufig auf 0 lassen.)

#ifdef LCD_SIMULATOR
void uhr_simulation()
{
 struct tm *tm;
 time_t timer;
 time(&timer);
 tm=localtime(&timer);
 sec=tm->tm_sec;
 min=tm->tm_min;
 stunden=tm->tm_hour;
 tage=tm->tm_mday;
 monat=tm->tm_mon+1;
 jahr=tm->tm_year+1900;
 schaltjahr_und_wochentagberechnen();
}

void zeiterhoehen_simulation(int n)
{
 millisec += n;
 if(millisec>=1000)
  {millisec-=1000;
   if(++sec==60)
    {sec=0;
     if(++min==60)
      {min=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
      }
    }
  }
}
#else

//die Namen *_vect sind in /usr/lib/avr/include/avr/iomxx0_1.h oder iom2560.h zu finden
ISR(TIMER1_COMPA_vect)
{
 uchar mehr;
 uint ms=millisec;
 if(++ms>=1000-Rest1) mehr=1; else mehr=0;
 sei();
 if(ms==1000)
  {ms=0;
   //if(++sec>=60-Rest2) mehr=1; else mehr=0;
   mehr=0;//einfacher wenn Rest2 und Rest3 sowieso ==0
   if(++sec==60)
    {sec=0;
     //if(++min>=60-Rest3) mehr=1; else mehr=0;
     if(++min==60)
      {min=0; mehr=0;
       if(++stunden==24)
	{stunden=0; ++tage; //neuertag=1;
	 if(++wochentag==7) wochentag=0; 
	 if(tage>monatstage[monat])
	  {tage=1;
	   if(++monat==13) //Jahreswechsel
	     {monat=1; ++jahr; sei(); schaltjahrtest();
	      //variable_feiertage_berechnen();
	     }
	  }
	}
       else //Test auf Sommerzeit-Umschaltungen
	{if(sommerzeit==0)
	  {if(stunden==2 && monat==3 && wochentag==6 && tage>(31-7))
	    {sommerzeit=1; stunden++;} //am letzten Sonntag im Maerz
	  }
	 else //if(sommerzeit==1)
	  {if(stunden==3 && monat==10 && wochentag==6 && tage>(31-7))
	    {sommerzeit=0; --stunden;} //am letzten Sonntag im Oktober
	  }
	}
      }
    }
  }
 //else if((ms&0x0F)==1) {}//etwa alle 16ms
 //else if((ms&0xFF)==0xE7) {}//alle ca 250ms
 millisec=ms;
 OCR1A=NormalerCompwert+mehr;
 //tastaturcheck(); //nur fuer UART-Tastatur
}

void milliwait(uint n) //n Millisekunden warten 1000 >= n >= 1
{
 uint ms;
 cli(); ms=millisec; sei();
 while(ms==millisec) ;//auf 1. Millisekunde warten
 if((ms+=n)>=1000) ms-=1000;
 while(ms!=millisec || ms!=millisec) //zweimal lesen um cli() sei() zu sparen
   ;//auf n. Millisekunde warten
}

#endif //ifdef LCD_SIMULATOR else

void secwait(int8 n) //n Sekunden warten
{
 while(--n>=0) milliwait(1000);
}

/*************** Routinen fuer AD-Wandler ****************/
//#define F_ADC 200000
//#define ADPRESCALER int(log2(F_CPU/F_ADC)+0.999)
#define ADPRESCALER 7 //entspricht einem Vorteiler von 2^7 = 128 --> 16MHz/128 = 125kHz

#ifdef LCD_SIMULATOR
void adwandler_init() {}
int adwandler() {return 512;} //einzelne Messung machen, Rueckgabe im Bereich 0...1023
#else

void adwandler_init()
{
 //fuer ATmega2560:
 DIDR0 |= (1<<ADC0D); //Digital-Input des ADC0-Pins ausschalten
 ADMUX = (1<<REFS1) | (1<<REFS0) | 0; //Referenz ca 2.56V, ADC0
 ADCSRA = (1<<ADEN) | (1<<ADSC) | ADPRESCALER; //Einschalten und erste Messung machen
 while (ADCSRA & (1<<ADSC)) ;//Ende der Wandlung abwarten
}

int adwandler() //einzelne Messung machen, Rueckgabe im Bereich 0...1023
{
 ADCSRA |= (1<<ADSC); //Messung starten
 while (ADCSRA & (1<<ADSC)) ;//Ende der Wandlung abwarten
 return ADC;
}

#endif //LCD_SIMULATOR else

/************** Routinen fuer LCD-Ansteuerung ************/

void glcd_goto(uchar zeile,uchar spalte)
{
 const uint8_t text_zeichenabstand=FONTBREITE+1; //= 6 = Breite eines 5x7-Zeichens + 1 Pixel Abstand
 uint8_t x=(spalte-1)*text_zeichenabstand;
 if(zeile<=6)
  glcd_moveto(x,(zeile-1)*text_zeilenabstand);
 else
  glcd_moveto(x,LCD_MAXY-FONTHOEHE);
}
inline void glcd_write(char c) {glcd_printc(c);}
inline void glcd_write(const char *txt) {glcd_print(txt);}

void glcd_writedd(const char *cstr,int p1,int p2)
{
 char glcd_puffer[40];
 sprintf(glcd_puffer,cstr,p1,p2);
 glcd_write(glcd_puffer);
}

void lcd_gowrite(char zeile,char spalte,const char* text)
{
 glcd_goto(zeile,spalte); glcd_write(text);
}

void lcd_gowrite21(char zeile,const char* text)
{
 char txt[22],*s,i;
 for(i=0,s=txt; i<21 && (*s = *text++);s++,i++) ;
 for(;i<21;i++) *s++ = ' ';//Zeile mit Leerstellen auffuellen
 *s=0;
 glcd_goto(zeile,1); glcd_write(txt);
}

static int8 hexmodus=0;

int48 divmod(int8 *prest,int48 x,int8 teiler)
{
 //diese Routine koennte man mit Assembler wesentlich optimieren
 int48 result=x/teiler;
 x %= teiler;
 *prest = getint8(x);
 return result;
}

void string_umkehren(char *str)
{
 int8 n=strlen(str),c;
 char *s,*p;
 for(s=str,p= &str[n-1]; s<p; )
   {c = *s;
    *s++ = *p;
    *p-- = c;
   }
}

#ifdef TEXT_LCD
const char Hochzeichen=0x1D; //0x1D oder '^' oder ... abhaengig von verwendetem LCD-Display
void hochzeichen_ersetzen(char *s)
{
 for(;*s;s++) {if(*s=='^') *s=Hochzeichen;}
}
#else
const char Hochzeichen='^'; //0x1D oder '^' oder ... abhaengig von verwendetem LCD-Display
#endif

const Fliesszahl einhalbes(1,2);

void sprintf_d(char *str,Fliesszahl x)
{
 long ix,dx,dxmax;
 int ex=0;
 bool isneg;
 char vorz=' ';
 int8 zehn=(hexmodus)?16:10, logdxmax, logdxmax3, rest;
 int48 ix48;
 Fliesszahl einhunderstel(1,100),tausend(1000),zigtausend(99000);
 //printf("Testpunkt1: x = %s\n",x.str());
 isneg = x.isnegativ();
 if(isneg) {vorz='-'; x = -x;}
 ix48=getint48(x);
 if(x==ix48) //wenn es eine Ganze Zahl ist
   {char str1[24], *p1=str1;
    do
      {ix48 = divmod(&rest,ix48,zehn);
       if(rest>9) *p1++ = rest + 'A' - 10;
       else       *p1++ = rest + '0';
      }
    while(ix48!=0);
    *p1 = 0;
    string_umkehren(str1);
    sprintf(str,"%c%16s",vorz,str1);
    return;
   }
 if(x.istnan())
   {sprintf(str,"NAN");
    return;
   }
 if(hexmodus)
   {einhunderstel=Fliesszahl(1,256);
    tausend=Fliesszahl(0x1000);
    zigtausend=Fliesszahl(0xFF000);
    dxmax=0x1000000; logdxmax=6; logdxmax3=3*6;
   }
 else
   {//einhunderstel=Fliesszahl(1,100);
    //tausend=Fliesszahl(1000);
    //zigtausend=Fliesszahl(99000);
    dxmax=1000000000; logdxmax=9; logdxmax3=3*9;
   }
 if(x<=einhunderstel)
   {
    while(x.getexp() < -logdxmax3)  //fuer Geschwindigkeitsoptimierung bei grossen Exponenten
       {ex -= logdxmax; x *= dxmax;}
    while(x < 1)
      {ex -= 3; x *= tausend;} //kleine Zahlen fuer Darstellung der Form 12.3e-6 vorbereiten
   }
 else if(x>=zigtausend)
   {
    while(x.getexp() > logdxmax3)  //fuer Geschwindigkeitsoptimierung bei grossen Exponenten
       {ex += logdxmax; x /= dxmax;}
    while(x >= tausend)
      {ex += 3; x /= tausend;} //grosse Zahlen fuer Darstellung der Form 12.3e6 vorbereiten
   }
 //printf("Testpunkt2: x = %s ex=%d\n",x.str(),ex);
 ix=getint32(x);
 x -= ix;
 dx = getint32(x*dxmax+einhalbes); // +0.5 ist fuer korrekte Rundung
 if(dx==dxmax)               //wenn beim Runden auf eine Ganze Zahl aufgerundet wurde
   {dx=0;  ix++;}                 //dann entsprechend den Ganzzahligen Anteil korrigieren
 if(hexmodus)
   {if(ex==0)     sprintf(str,"%c%5lX.%06lX ",vorz,ix,dx);
    else if(ex>0) sprintf(str,"%c%3lX.%06lX*10%c%X ",vorz,ix,dx,Hochzeichen,ex);
    else          sprintf(str,"%c%3lX.%06lX*10%c-%X ",vorz,ix,dx,Hochzeichen,-ex);
   }
 else
   {if(ex==0) sprintf(str,"%c%5ld.%09ld ",vorz,ix,dx);
    else      sprintf(str,"%c%3ld.%09lde%d ",vorz,ix,dx,ex);
   }
}

void lcd_gowrite_d(char zeile,Fliesszahl x)
{
 char text[24];
 sprintf_d(text,x);
 lcd_gowrite21(zeile,text);
}

//Globale Variablen:
int8 hauptmodus=0;
const int8 STARTMODUS=0,TASCHENRECHNER=1,KALENDER=2,MGRECHNER=3;//Gueltige Werte fuer modus

void zeitauflcdanzeigen(uint8_t zeile=4)
{
 char txt[40];
 glcd_goto(zeile,1);
 sprintf(txt,"%02d:%02d:%02d ",stunden,min,sec);
 glcd_write(txt);
 sprintf(txt,"%02d.%02d.%d",tage,monat,jahr);
 glcd_write(txt);
}

void batteriespannung_auf_lcd_anzeigen()
{
 int adwert,vorkomma,nachkomma;
 const int vref=2579; //Referenzspannung des AD-Wandlers (idealerweise nachmessen)
 adwert=adwandler(); //Messwert Bereich 0...1023

 //adwert = ((long)adwert*vref+512)/1024 //Messwert in Millivolt umrechnen
 //                      	    * 4; //Spannungsteiler 150k zu 50k beruecksichtigen
 adwert = ((long)adwert*vref+(512/4))/(1024/4);//Variante mit weniger Rundungsfehler

 vorkomma  = adwert/1000;
 nachkomma = adwert%1000;
 glcd_goto(3,1); glcd_write("Batt: "); glcd_writedd("%d.%03d V",vorkomma,nachkomma);
}

void monatjahrauflcd(uint8_t zeile,int8 mo,int ja)
{
 const char* monatsnamen[12]=
    {"Januar","Februar","März","April","Mai","Juni","Juli","August","September","Oktober","November","Dezember"};
 // {"Januar","Februar","Maerz","April","Mai","Juni","Juli","August","September","Oktober","November","Dezember"};
 char txt[40];
 glcd_goto(zeile,1);
 sprintf(txt,"%10s %04d      ",monatsnamen[mo-1],ja);
 glcd_write(txt);
}

/****************** Dinge zum Uhr stellen und in RTC speichern *************************/
#ifndef LCD_SIMULATOR
#ifdef RTCDEBUG
void lcd_printf3(const char *cstr,int n1,int n2,int n3)
{
 char buffer[40];
 sprintf(buffer,cstr,n1,n2,n3);
 glcd_write(buffer);
}

void lcd_goprintf3(int zeile,int spalte,const char *cstr,int n1,int n2,int n3)
{
 glcd_goto(zeile,spalte);
 lcd_printf3(cstr,n1,n2,n3);
}

void testpunkt(const char *cstr,int n)
{
 lcd_goprintf3(1,1,cstr,n,0,0);
 milliwait(1000);
}

void lcd_printerror(uchar err)
{
 int8 i;
 lcd_goprintf3(1,1,"err=%d  ",err,0,0);
 for(i=0;i<10;i++) milliwait(1000);
}
#endif

#include "i2c.c"
#include "rtc.c"

static int JAHRHUNDERT=2000;
static char rtc[10]; // {Sekunden, Minuten, Stunden, Wochentag, Tag, Monat, Jahr, Control, Jahrhund., Jahr}

char bin2bcd(char c)
{
 return ((c/10)<<4)+c%10;
}

char bcd2bin(char c)
{
 return ((c>>1)&0x78)+((c>>3)&0x1E)+(c&0x0F);
}

void zeit_in_rtc_speichern()
{
 uchar err, sek;
 for(sek=sec;sek==sec;) ;//auf Sekundenwechsel warten
 rtc[0] = bin2bcd(sec);
 rtc[1] = bin2bcd(min);
 rtc[2] = bin2bcd(stunden);
 rtc[3] = wochentag;
 rtc[4] = bin2bcd(tage);
 rtc[5] = bin2bcd(monat);
 rtc[6] = bin2bcd(jahr%100);
 rtc[7] = 0x00; //0x80=OUT=Level wenn SQWE nicht gesetzt, 0x10=SQWE=Frequenz an Pin 7 ausgeben, 0x03=RS
                //RS definiert die Frequenz wenn SQWE gesetzt ist: 0=1Hz, 1=4096, 2=8192, 3=32768 Hz
 //rtc[7] = (min&1)?0x13:0x00; //test: 32768 Hz Ausgabe wenn bei ungerader Minute gespeichert wird
 rtc[8] = jahr/100; //so koennte man noch das Jahrhundert speichern.
 rtc[9] = jahr%100; //das aktuell Jahr muesste man aber auch noch mit speichern.
 if(rtc[0] != bin2bcd(sec)) err=5; //falls Sekunde inzwischen gewechselt hat ist etwas faul.
 else err = rtc_write(rtc,10);
#ifdef RTCDEBUG
 if(err) lcd_printerror(err);
 else lcd_gowrite(4,1,"in RTC gespeichert");
#endif
}

void zeit_von_rtc_lesen()
{
 uchar err=0;
 char sek,sek2;
 //char wotag; //nicht wirklich gebraucht
 sek2 = sek = rtc_readsecond();
 if(sek==ERROR_CODE) err=2;
 else
  for(int i=0;sek2==sek && i<1000;i++)
      {milliwait(2);
       sek2 = rtc_readsecond(); //auf Sekundenwechsel warten
       if(sek2==ERROR_CODE) {err=3; break;}
      }
 cli(); millisec=0; sei(); //millisec auf Sekundenwechsel der RTC synchronisieren
 if(!err)
      {sec = bcd2bin(sek2);
       milliwait(2);
       err = rtc_read(rtc,10); //gesamte Zeit und Datum von RTC einlesen
       if(!err && (sec != bcd2bin(rtc[0]))) err=4; //Wenn Sekunde inzwischen gewechselt hat ist was faul.
      }
 if(err)
      {
#ifdef RTCDEBUG
       lcd_printerror(err);
#endif
      }
 else
      {min     = bcd2bin(rtc[1]);
       stunden = bcd2bin(rtc[2]);
       //wotag   = rtc[3]; if(wotag>6) wotag=6;
       tage    = bcd2bin(rtc[4]);
       monat   = bcd2bin(rtc[5]);
       JAHRHUNDERT = rtc[8]*100;                  //wenn das Jahrhundert gespeichert wurde
       if(rtc[6]==0 && rtc[9]==99) JAHRHUNDERT++; //ueberpruefen ob inzwischen kein Jahrhundertwechsel war
       // geht aber nur wenn beim Abschalten immer die RTC beschrieben wird, zumindest jedes Jahr mal
       jahr    = JAHRHUNDERT + bcd2bin(rtc[6]);
#ifdef RTCDEBUG
       lcd_gowrite(1,1,"RTC eingelesen");
#endif
      }
 schaltjahr_und_wochentagberechnen();
 //if(wotag!=wochentag) irgendwas_falsch();
}

#else  //von ifndef LCD_SIMULATOR
#include <time.h>
void zeit_in_rtc_speichern() {}
void zeit_von_rtc_lesen() {uhr_simulation();}
#endif //Ende von ifndef LCD_SIMULATOR

/********************************** Hauptprogramm *************************************/
#ifdef DEBUG
int testwait(int n)
{
 volatile int i,j,k;
 for(i=0;i<n;i++)
  for(j=0;j<n;j++)
   k=i+j;
 return k;
}
#endif

int main(void)
{
 char c=' ',pos=1,startsec;
 int8 i;
 int8 lichtstatus=1;
 long wartezeit=0;

#ifndef LCD_SIMULATOR
#ifdef ATMEGA2560
 //Ein/Aus-Schalter nur auf wirklichem Geraet:
 DDRD  |= (1<<4); //PD4 als Ausgang fuer Ein/Aus-Transistor
 PORTD |= (1<<4); //und auf "Ein" schalten
 PORTD |= (1<<5); //Pullupwiderstand an PD5 fuer Ein/Aus-Taster
#define EINAUSTASTER_VORHANDEN
#ifndef LCD_MK3
#define EINAUSAKTIV
 //Pullup-Widerstaende fuer LCD auf wirklichem Geraet:
 CS_DDR   |= (1<<CS_BIT);  //DDRB  |= (1<<4);
 RST_DDR  |= (1<<RST_BIT); //DDRB  |= (1<<5);
 CS_PORT  |= (1<<CS_BIT);  //PORTB |= (1<<4);
 RST_PORT |= (1<<RST_BIT); //PORTB |= (1<<5);
#endif
#endif
#endif

#ifdef DEBUG
 DDRB |= (1<<7); //PB7 als Ausgang fuer Hintergrundlicht
 for(i=0;i<3;i++) { //3 mal blinken
  PORTB |= (1<<7); //Hintergrundlicht ein
  testwait(300);
  PORTB &= ~(1<<7); //Hintergrundlicht aus
  testwait(300);
 }
#endif

#ifndef LCD_SIMULATOR
 //Timer-Einstellungen fuer ATmega32, ATmega644, ATmega128, oder ATmega2560:
 TCCR1B = (1<<WGM12)|(1<<CS10); //CTC-OCR1A-Modus, Prescaler=1
 OCR1A = NormalerCompwert;
 TCNT1 = 0; //Startwert des Timers
#ifdef ATMEGA32
 TIMSK = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
#else
 TIMSK1 = 1<<OCIE1A; //Timer1 Interrupts bei Vergleichswert
#endif
#endif

 sei(); //Interrupt einschalten

 init_matrix();
 milliwait(50);
#ifdef DEBUG
 milliwait(950);
 for(i=0;i<3;i++) { //3 mal im Sekundentakt blinken
  glcd_licht(1); //Hintergrundlicht ein
  milliwait(500);
  glcd_licht(0); //Hintergrundlicht aus
  milliwait(500);
 }
 glcd_licht(1); //Hintergrundlicht ein
#endif
 glcd_init();
#ifdef DEBUG
 for(i=0;i<3;i++) { //3 mal schnell blinken
  glcd_licht(1); //Hintergrundlicht ein
  milliwait(200);
  glcd_licht(0); //Hintergrundlicht aus
  milliwait(200);
 }
#endif
 setze_zeilenabstand(10);
 adwandler_init();
 twi_init();

 glcd_clear();
 glcd_licht(1);
 lcd_gowrite(1,1,"Tarech2 Version "); glcd_write(VERSION);
#ifndef EINAUSAKTIV
 lcd_gowrite(2,1,"Press Gold to start");
#endif
 batteriespannung_auf_lcd_anzeigen();

#ifdef FLIESSZAHLTEST
 char text[40];//test
 sprintf(text,"Test1: %d ",(int)sizeof(Fliesszahl));//test
 lcd_gowrite(2,1,text);//test
#endif

 aktuelle_formel_vom_eeprom_einlesen(); //die zuletzt im MGW-Rechner eingegebene Formel wieder vom EEPROM einlesen
 tarech_regist_vom_eeprom_einlesen(); //die Register-Werte vom Taschenrechner vom EEPROM einlesen

#ifndef LCD_MK3
 zeit_von_rtc_lesen();
#endif
 zeitauflcdanzeigen();
 startsec = sec+5; if(startsec>=60) startsec -= 60;

 while(1) //Hauptschlaufe
  {
#ifndef DEBUG1
   if(hauptmodus==STARTMODUS)
    {
     wartezeit=0;
     batteriespannung_auf_lcd_anzeigen();
     zeitauflcdanzeigen();
#ifdef EINAUSAKTIV
     if((PIND&(1<<5))!=0) //wurde Ein/Aus-Taster nach dem Einschalten wieder losgelassen?
        {pos=1; hauptmodus=TASCHENRECHNER; //ja: Modus auf Taschenrechner schalten
	 tarech_init();
	}
#else
     if(sec==startsec) c='a'; else c=tastaturabfrage();
     if(c!=0) //bei Tastendruck oder 5 Sec nach einschalten in Taschenrechner-Modus wechseln
      {glcd_goto(1,pos); //Test: Tastatureingabe auf 1.Zeile darstellen
       glcd_write(c);
       if(++pos==21) pos=1;
       if(c=='a')
	{pos=1; hauptmodus=TASCHENRECHNER;
#ifdef LCD_SIMULATOR
	 printf("Testpunkt3: tarech_init() startet\n");//test
#endif
	 tarech_init();
#ifdef LCD_SIMULATOR
	 printf("Testpunkt4: tarech_init() fertig\n");//test
#endif
	}
      }
#endif
    }
   else if(hauptmodus==TASCHENRECHNER)
    {
     if((c=tastaturabfrage())!=0)
       {tarech_tastenreaktion(c);
	wartezeit=0;
       }
     if(hauptmodus==MGRECHNER) //wurde im Taschenrechner MGW gedrueckt?
       {mgrechner_init(); //ja: entsprechende Initialisierung machen
	wartezeit=0;
       }
    }
   else if(hauptmodus==KALENDER)
    {
     static int tagoffset=0;
     int ja;      //momentanes Jahr
     int8 mo,feb; //momentaner Monat, Anzahl Tage im Februar vom momentanen Jahr
     static int8 flag=0;
     lcd_gowrite(1,1,"Mo Di Mi Do Fr Sa So ");
     glcd_goto(2,1);
     for(i=0;i<28;i++)
       {int tg = tage+i-wochentag+tagoffset;
	mo = monat; ja=jahr; feb=monatstage[2];
	while(tg<1)
	  {--mo;
	   tg += mo==2 ? feb : monatstage[mo];
	   if(mo==0) {mo=12; --ja; feb=istschaltjahr(ja)?29:28;}
	  }
	while(tg > (mo==2 ? feb : monatstage[mo]))
	  {tg -= mo==2 ? feb : monatstage[mo];
	   if(++mo==13) {mo=1; ja++; feb=istschaltjahr(ja)?29:28;}
	  }
	if(i==7) glcd_goto(3,1);
	else if(i==14) glcd_goto(4,1);
	else if(i==21) glcd_goto(5,1);
	glcd_writedd("%2d ",tg,0);
       }
     if(flag==0) zeitauflcdanzeigen(7);
     else monatjahrauflcd(7,mo,ja);
     c=tastaturabfrage();
     if(c>0) wartezeit=0;
     if(c=='a')                     //Gold-Taste gedrueckt?
        {hauptmodus=TASCHENRECHNER; //ja: zum Taschenrechner zurueckkehren
	 glcd_clear();
	 tarech_refresh();
	}
     switch(c)     //Tastenauswertung fuer Kalender-Modus
       {case '+': tagoffset += 7; flag=1; // 1 Woche weiter
        CASE '-': tagoffset -= 7; flag=1; // 1 Woche zurueck
        CASE '*': tagoffset += 28; flag=1; //naechste 4 Wochen darstellen
        CASE '/': tagoffset -= 28; flag=1; //vorherige 4 Wochen darstellen
        CASE '=': tagoffset = 0; flag=0; //aktuelle Woche
       }
    }
   else if(hauptmodus==MGRECHNER)
    {
     if((c=tastaturabfragenr())!=0)
      {wartezeit=0;
       if(c==1) //Gold-Taste gedrueckt?
	{
	 hauptmodus=TASCHENRECHNER; //ja: zum Taschenrechner zurueckkehren
	 glcd_clear();
	 tarech_refresh();
	}
       else
         mgrechner_tastenreaktion(c);
      }
    }
#else //von ifndef DEBUG1
   printf("DEBUG1: sec=%d millisec=%d\n",sec,millisec);
#endif //von ifndef DEBUG1 else
#ifdef LCD_SIMULATOR
   MILLIWAIT(20);
#endif

#ifdef EINAUSTASTER_VORHANDEN
   if(hauptmodus!=STARTMODUS && (PIND&(1<<5))==0) //Ein/Aus-Taster gedrueckt?
    {
     for(i=0;(PIND&(1<<5))==0;i++) milliwait(100);//warten bis Taster wieder losgelassen
     milliwait(100);
     if(i<5) //nur kurzer Tastendruck?
       glcd_licht(lichtstatus^=1); //Hintergrundlicht ein/ausschalten
     else
       geraet_ausschalten();
     wartezeit=0;
    }
#endif

   if(++wartezeit>20*3500L) //wenn einige Zeit (ca. 20 Sek) inaktiv,
        glcd_licht(lichtstatus=0); //dann Hintergrundlicht ausschalten
   if(wartezeit>=120*3500L) //wenn lange (ca. 2 Min) inaktiv,
       geraet_ausschalten(); //dann Geraet ausschalten

  }//Ende Hauptschlaufe
 return 0;//wird nie erreicht
}

void geraet_ausschalten()
{
 aktuelle_formel_im_eeprom_speichern(); //falls veraendert, wird die eingegebene Formel vom MGW-Rechner im EEPROM gespeichert
 if(tarech_in_eingabemodus()) tarech_tastenreaktion('='); //falls aktive Eingabezeile, diese mit Enter abschliessen
 tarech_stapel_im_eeprom_speichern(); //Stapel vom Taschenrechner im EEPROM speichern
 tarech_regist_im_eeprom_speichern(); //und auch Register, falls veraendert im EEPROM speichern
 glcd_licht(0); //Hintergrundlicht ausschalten
 milliwait(50); //warten bis EEPROM speichern sicher fertig
 PORTD &= ~(1<<4); //Ein/Aus-Transistor auf "Aus" schalten
 while(1)  {  }    //Anhalten
}

/*** Abfrage der Matrix-Tastatur ***/
#ifdef LCD_MK3
// wenn auf dem MK3-Board: Tastenmatrix an F2-F7 und E0-E4
#include "mk3_tastenmatrix.c"

#else

#define PORT_MATRIXOUT PORTH //Mit Spalten verbundener Port
#define DDR_MATRIXOUT  DDRH

#define PORT_MATRIXIN  PORTE //Mit Reihen verbundener Port
#define PIN_MATRIXIN   PINE
#define DDR_MATRIXIN   DDRE
#define NREIHEN 5
#define NSPALTEN 6

#ifdef LCD_SIMULATOR
#include "matrixsim.cc"
#endif

void init_matrix()
{
 PORT_MATRIXOUT=0xFF; //Pullup-Widerstaende setzen
 PORT_MATRIXIN=0xFF;  //Pullup-Widerstaende setzen
 DDR_MATRIXOUT=0; //Ausgabe-Leitungen vorlaeufig auf Eingang setzen
 DDR_MATRIXIN=0;  //Eingabe-Leitungen auf Eingang setzen
#ifdef LCD_SIMULATOR
 matrix_init();
#endif
}

void matrixwait() //so in etwa 100us warten
{
#ifdef LCD_SIMULATOR
 MILLIWAIT(1);
#else
 volatile int8 x=0;
 for(int8 i=0;i<100;i++)
  x=i;
#endif
}

uint8_t matrix_einlesen()
{
 uint8_t spalte,reihe,nr=0,reihe_mal_n;
 for(spalte=0; spalte<NSPALTEN; spalte++)
  {
   DDR_MATRIXOUT = (1<<spalte);   //nur aktuelle Spalte DDR auf Ausgang
   PORT_MATRIXOUT = ~(1<<spalte); //nur aktuelle Spalte PORT auf 0 setzen
   matrixwait();
   for(reihe=0,reihe_mal_n=0; reihe<NREIHEN; reihe++,reihe_mal_n+=NSPALTEN)
    {
     if((PIN_MATRIXIN&(1<<reihe))==0)
      {nr=reihe_mal_n+spalte+1; break;}
    }
  }
 DDR_MATRIXOUT = 0;
 PORT_MATRIXOUT = 0xFF;
 return nr;
}

#endif

static char taste_ready=0,taste_nr=0;
//static char tasten_puffer[8]; //Tastenpuffer falls es sonst zu langsam reagiert
//static uint8 tasten_i1=0,tasten_i2=0; //Schreib- und Lese-Zeiger im Tastenpuffer
#define MAXTASTCOD 30
//Definition der Tastenanordnung. (Tastenbelegungen siehe weiter unten)
#ifdef MATRIXQUER
   //Fuer Tastenmatrix im Querformat:
static uint8_t tastencode[MAXTASTCOD] =
 {
  'a','b','c','d','e','/',
  '7','8','9','A','E','*',
  '4','5','6','B','F','-',
  '1','2','3','C','N','+',
  '0','.','I','D','J','='
 };
#else
   //Fuer Tastenmatrix im Hochformat:
static uint8_t tastencode[MAXTASTCOD] =
 {
  'a','A','7','4','1','0',
  'b','B','8','5','2','.',
  'c','C','9','6','3','I',
  'd','D','E','F','N','J',
  'e','/','*','-','+','='
 };
#endif

void tastaturcheck()
{
 static int8 loslassflag=1;
 uint8_t nr=matrix_einlesen();
 if(loslassflag)
  {if(nr==0 && matrix_einlesen()==0) //zweimal eine Null gelesen?
    loslassflag=0; //ja: Taste wurde losgelassen, naechstes Mal wieder auf Tastendruck warten.
  }
 else
  {if(nr!=0 && nr==matrix_einlesen() && nr<=MAXTASTCOD) //gleicher Wert bei erneutem Einlesen?
    {taste_ready=1; taste_nr=nr; //ja: entprellte Taste speichern
     loslassflag=1; //und naechstes Mal auf Loslassen der Taste warten.
    }
  }
}

int8 tastaturabfragenr()
{
 int8 nr=0;
 tastaturcheck();
 if(taste_ready)
  {nr=taste_nr; taste_ready=0;}
 return nr;
}

char tastaturabfrage()
{
 int8 nr=tastaturabfragenr();
 if(nr==0) return 0;
 return tastencode[nr-1];
}

/*************** Uebernommen von vereinfachtem Vektar ****************/

//(3.9.08) Vermeidung von Konflikten mit anderweitig definiertem abs:
#ifdef abs
#undef abs
#endif
#define abs meinabs

//(1.1.12) Vermeidung von Konflikten mit in c++ definiertem NAN:
#ifdef NAN
#undef NAN
#endif

//const Fliesszahl einhalbes(1,2); //schon weiter oben
const Fliesszahl NAN(1,0); //ungueltige Zahl
//const Fliesszahl PI=atan(1.0)*4;
const Fliesszahl PI("3.14159265358979323846",0); //Eulersche Zahl
const Fliesszahl ZWEIPI=2*PI;
const Fliesszahl LN2("0.69314718055994531",0); //natuerlicher Logarithmus von 2
//const Fliesszahl LN2=calcln2(); //wird auch nicht genauer
//const Fliesszahl LN10("2.302585092994",0);
//const Fliesszahl ee("2.718281828459",0); //Basis zum natuerlichen Logarithmus
const Fliesszahl ee=exp(Fliesszahl(1)); //Basis zum natuerlichen Logarithmus
const Fliesszahl LN10=log(Fliesszahl(10)); //natuerlicher Logarithmus von 10
//const Fliesszahl WURZEL2("1.41421356237",0);
const Fliesszahl GRAD=PI/180;
const Fliesszahl Lichtgeschwindigkeit(299792458,1000); //in km/s
const Fliesszahl Erdbeschleunigung(980665,100000); //in m/s^2
const Fliesszahl Avogadrokonstante("6.022142",23); //in Teilchen/mol

/************************* Vordeklarationen ***************************/
int keyget(int* in,int *asci,ULONG* rawcode);

/**************************** Kleinkram *******************************/
inline bool ist(const char*t,const char*s) {return strcmp(t,s)==0;}
inline Fliesszahl abs(Fliesszahl x) {return x<0?-x:x;}

//static int hexmodus=0; //vorgezogen wegen lcd_gowrite_d()

void lcd_goprint(uint8_t zeile,char *str)
{
 const uint8_t M=20; //Anzahl benutzte Zeichen pro Zeile
 const uint8_t Z=6;  //Anzahl Textzeilen auf dem LCD
 glcd_goto(zeile,1);
 while(strlen(str) > M)
   {
    char c;
    c=str[M]; str[M]=0;
    glcd_write(str);
    str[M]=c;
    str = &str[M];
    if(++zeile > Z) zeile=1;
    glcd_goto(zeile,1);
   }
 glcd_write(str);
}

void wprintf(int zeile,const char *cs,const char *p1,const char *p2=NULL)
{
 char str[80];
 sprintf(str,cs,p1,p2);
#ifdef TEXT_LCD
 hochzeichen_ersetzen(str);
#endif
 lcd_goprint(zeile,str);
}

void wprintf(int zeile,const char *cs,int p1,int p2=0)
{
 char str[80];
 sprintf(str,cs,p1,p2);
#ifdef TEXT_LCD
 hochzeichen_ersetzen(str);
#endif
 lcd_goprint(zeile,str);
}

void minus_entfernen(char *s)
{
 char *t= &s[1];
 while((*s++ = *t++)) ;
}

void minus_einfuegen(char *s)
{
 char c='-',c2;
 while(c)
   {c2= *s; *s++ = c; c=c2;}
 *s=0;
}

int s_index(const char *s1,int n1,const char *s2,int n2)
			/* Sucht den String s2 innerhalb von s1 und  */
{			/* gibt die Position zurueck (nicht gefunden = -1) */
 int i,c,j;
 const char *p1,*p2;
 if(n2<=0) return 0;	/* leerer String ist immer enthalten */
 for(i=0;;i++)
	{if(i>=n1) return -1; /* nicht gefunden */
	 c= *s1++;
	 if(c== *s2)
		{for(p1=s1,p2=s2,j=1; j<n2; j++)
			{c= *++p2;
			 if(*p1++!=c) break; /* noch nicht gefunden */
			}
		 if(j==n2) break; /* gefunden */
		}
	}
 return i;
}

/************************* Fehlermeldungen ****************************/
class Fehlermeldung
{
public:
 int errflag;
 Fehlermeldung() {errflag=0;}
 void error(const char *s,const char *p1=NULL);
 void say(const char *s,const char *p1=NULL,const char *p2=NULL);
 void say1(const char *s);
 void clear();
};

void Fehlermeldung::error(const char *s,const char *p1)
{
 clear();
 wprintf(4,s,p1);
 if(*s!=' ') errflag=1; else errflag=0;
 secwait(3);
}

void Fehlermeldung::say(const char *s,const char *p1,const char *p2)
{
 clear();
 wprintf(1,s,p1,p2);
 errflag=0;
 secwait(8);
}

void Fehlermeldung::say1(const char *s)
{
 clear();
 wprintf(5,s,0);
 errflag=0;
}

void zeileclear(uint8_t zeile)
{
 glcd_goto(zeile,1);
 glcd_write("                     ");
}

void Fehlermeldung::clear()
{
 for(uint8_t i=4;i<=6;i++) zeileclear(i);
 errflag=0;
}

/************************ Globale Variablen ***************************/
//const int TYP_SHIFT=1,TYP_GOLD=2,TYP_NORMAL=0,TYP_HEXTASTE=3;
#define ANZAHLREGIST 10

static Fehlermeldung meldung;
static Fliesszahl regist[ANZAHLREGIST];

/***************************** Klassen ********************************/
const int8 STAPELSIZE=4;

class Stapel
{
 Fliesszahl mem[STAPELSIZE];
 int8 imax;
public:
 Stapel();
 void draw();
 void push(Fliesszahl a);
 Fliesszahl pop();
 Fliesszahl rol(bool);
 Fliesszahl get(int8 j);
 void set(int8 j,Fliesszahl a) {if(j>=imax) j=imax-1; mem[j]=a;}
};
Stapel::Stapel()
{
 imax=STAPELSIZE;
 //for(int8 i=0;i<imax;i++) mem[i]=i+einzehntel;//test
 //for(int8 i=0;i<imax;i++) mem[i]=0;
 for(int8 i=0;i<imax;i++) stapel_vom_eeprom_einlesen(i,&mem[i]);
}
void Stapel::draw()
{
 for(uint8 i=0;i<4;i++)
   lcd_gowrite_d(4-i,get(i));
}
Fliesszahl Stapel::get(int8 j)
{
 if(j>=imax) j=imax-1;
 return mem[j];
}
void Stapel::push(Fliesszahl a)
{
 for(int8 i=imax-1;i>0;i--) mem[i]=mem[i-1];
 mem[0]=a;
}
Fliesszahl Stapel::pop()
{
 Fliesszahl a=mem[0];
 for(int8 i=1;i<imax;i++) mem[i-1]=mem[i];
 return a;
}
Fliesszahl Stapel::rol(bool rev)
{
 Fliesszahl a;
 int8 i;
 if(rev)
   {for(a=mem[i=imax-1]; i>0; i--)  mem[i]=mem[i-1];
    mem[0]=a;
   }
 else
   {a=mem[0];
    for(i=1;i<imax;i++) mem[i-1]=mem[i];
    mem[imax-1]=a;
   }
 return a;
}

#define EINGABFELD 40

class Eingabefeld
{
 bool aktiv,reop;
 char str[EINGABFELD];
 int8 cursor,nx;
 Stapel *ps;
public:
 bool help;
 Eingabefeld() {aktiv=reop=help=0; nx=0;}
 void init(Stapel *p) {ps=p;}
 void open(int8 n) {nx=n; str[cursor=0]=0; aktiv=true; reop=0;}
 //void reopen(int8 n);
 void write(int8 c);
 void write(const char *s) {int8 c;while((c= *s++)!=0) write(c);}
 Fliesszahl close();
 void draw();
 bool istaktiv() {return aktiv;}
};

/*
void Eingabefeld::reopen(int8 n) //funktioniert nicht
{
 Fliesszahl v;
 if(aktiv) close();
 nx=n;
 v=ps->get(0);
 sprintf_d(str,v);
 cursor=strlen(str);
 if(ist(str,"0")) str[cursor=0]=0;
 aktiv=reop=true;
}
*/

bool ishexdigit(int8 c)
{
 if(c>='A') return (c<='F' || (c>='a' && c<='f'));
 return isdigit(c);
}
bool isdigitx(int8 c)
{
 if(hexmodus && c<='F' && c>='A') return true;
 return isdigit(c);
}
int8 hexdigit(int8 c)
{
 if(c>='a') return c-'a'+10;
 if(c>='A') return c-'A'+10;
 return c-'0';
}

// Vermeidung von Konflikt mit in c++ eingebauter Funktion:
#define pow mypow

/*
Fliesszahl pow(Fliesszahl x,int i)
{
 Fliesszahl z=1.0;
 if(i>0)  {do {z *= x;} while(--i!=0);}
 else if(i<0) {do {z /= x;} while(++i!=0);}
 return z;
}
*/
inline Fliesszahl pow(Fliesszahl x,int i) {return zhochn(x,i);}

void hexsscanf(char *str,Fliesszahl* px)
{
 Fliesszahl x=0,y,v=1;
 const char *s=str;
 int c,ex;
 if(*s=='-') {v= -1; s++;}
 for(;(c= *s++)!=0 && ishexdigit(c);)
   x=x*16+hexdigit(c);
 if(c=='.')
   {Fliesszahl y(1,16);
    for(;(c= *s++)!=0 && ishexdigit(c);y/=16)
       x += y*hexdigit(c);
   }
 if(c=='*' && (c= *s++)=='1' && (c= *s++)=='0' && (c= *s++)==Hochzeichen)
   {int vorz=1;
    if(*s=='-') {vorz= -1; s++;}
    for(ex=0;(c= *s++)!=0 && ishexdigit(c);)
       ex=ex*16+hexdigit(c);
    x *= pow(16,ex*vorz);
   }
 *px=v*x;
}

void dezsscanf(char *str,Fliesszahl* px)
{
 Fliesszahl x=0,v=1;
 const char *s=str;
 int c,ex;
 if(*s=='-') {v= -1; s++;}
 for(;(c= *s++)!=0 && isdigit(c);)
   x=x*10+(c-'0');
 if(c=='.')
   {Fliesszahl y(1,10);
    for(;(c= *s++)!=0 && isdigit(c);y/=10)
       x += y*(c-'0');
   }
 if(c=='e' || c=='E' ||
    (c=='*' && (c= *s++)=='1' && (c= *s++)=='0' && (c= *s++)==Hochzeichen))
   {int vorz=1;
    if(*s=='-') {vorz= -1; s++;}
    for(ex=0;(c= *s++)!=0 && isdigit(c);)
       ex=ex*10+(c-'0');
    x *= pow(10,ex*vorz);
   }
 *px=v*x;
}

Fliesszahl Eingabefeld::close()
{
 Fliesszahl x;
 x=ps->get(0);
 if(reop)
   {reop=0;}
 else
   {if(*str==0) x=0;
    else if(hexmodus) hexsscanf(str,&x);
    else dezsscanf(str,&x);
    ps->set(0,x);
   }
 aktiv=0;
 return x;
}

void Eingabefeld::write(int8 c)
{
 static int8 emi=0;
 if(!aktiv) open(nx);
 reop=0;
 if(c=='\b')
   {if(cursor==0) return;//Feld leer
    str[--cursor]=0;
    return;
   }
 //if(cursor >= sizeof(str)-1) return;//Feld voll
 if(cursor >= EINGABFELD-1) return;//Feld voll
 if(c=='-')
   {int i;
    for(i=cursor;i>0 && str[--i]!='+';)
      {if(!hexmodus) {if(str[i]=='e' && (emi=1-emi)==1) {i++; break;}}
       else {if(str[i]==Hochzeichen && (emi=1-emi)==1) {i++; break;}}
      }
    if(str[i]=='+') str[i]='-';
    else if(str[i]=='-') {minus_entfernen(&str[i]); --cursor;}
    else {minus_einfuegen(&str[i]); ++cursor;}
    return;
   }
 if(hexmodus==0 && c=='e' && cursor==0) str[cursor++]='1';
 str[cursor++]=c;
 str[cursor]=0;
}
void Eingabefeld::draw()
{
 if(aktiv)
   {char text[22];
    sprintf(text,"%18s  ",str);
    lcd_gowrite(4,1,text);
   }
}

class Taste
{
 const char **text0,**textnorm,**textblue,**textgold,**textred;
 //int typ;
public:
 //Taste() {typ=TYP_NORMAL; text0=NULL;}
 void init(const char**t1,const char**t2,const char**t3,const char**t4)
        {text0=textnorm=t1; textblue=t2; textgold=t3; textred=t4;
	 //typ=TYP_NORMAL;
	}
 int press(int8,Eingabefeld*,Stapel*);
 void shift(int8 i) {text0 = i==0?textnorm:(i==1?textblue:(i==2?textgold:textred));}
};

//Die Tasten "Red", "Blue" und "Gold" muessen bei allen Tastenbelegungen
//an der gleichen Position sein, die andern Tasten koennen beliebig
//angeordnet werden.
//In aktueller Version muss "Gold" an 1.Stelle und "Blue" an 2.Stelle stehen.

/*
Tasten wie sie von der UART-Tastatur bei der ersten Version (tarech.cc) gesendet werden:
 {'a','b','c','d','e',
  'A','B','C','D','/',
  '7','8','9','E','*',
  '4','5','6','F','-',
  '1','2','3','N','+',
  '0','.','I','J','='}
Tastenanordnung neu in tarech2.cc:
 {'a','b','c','d','e','/',
  '7','8','9','A','E','*',
  '4','5','6','B','F','-',
  '1','2','3','C','N','+',
  '0','.','I','D','J','='}
Diese Tastenanordnung kann bei Bedarf weiter oben veraendert werden.
Die folgenden Tabellen koennen in der urspruenglichen Anordnung belassen werden.
*/
static const char *tastenname[]=
 {"Gold","Blue","Red", "POP", "CORR",
  "A",   "B",   "C",   "D",   "/",
  "7",   "8",   "9",   "E",   "*",
  "4",   "5",   "6",   "F",   "-",
  "1",   "2",   "3",   "EXP", "+",
  "0",   ".",   "+/-", "x<>y","ENTR"
 };
static const char *tastenname_gold[]=
 {"Gold","Blue","Red", "POP", "CORR",
  "sin", "cos", "tan", "SQRT"," DEG",
  "7",   "MW",  "9",   "LN",  "*",
  "4",   "Iso", "6",   "LOG", "-",
  "1",   "HMS", "3",   "EXP", "+",
  "0",   ".",   "+/-", "x<>y","ENTR"
 };
static const char *tastenname_blue[]=
 {"Gold","Blue","Red", "ROL^","ROLv",
  "asin","acos","atan","x^2", "RAD",
  "NA",  "g",   "c",   "e^x", " DEL",
  "Pi",  "|x|", "Int", " 10^x"," EDIT",
  "Time","HR",  "3",   "y^x", "+",
  " Date"," 1/x","HEX", " Dez","ENTR"
 };
static const char *tastenname_red[]=
 {"Gold","Blue","Red", "Help","CORR",
  "Run", "Lear","LBL", "D",   "brNE",
  "Kale","MGW", "elem","Kons","brPL",
  "STO", "RCL", "STOx","RCLx","brMI",
  "Tset","2",   "3",   "EXP", "brEQ",
  " Dset",".",   "+/-", "x<>y", "ENTR"
 };
static const char *tasten_hilfe[]=
 {
  "HMS","Fliesszahl umrechnen in Std:Min:Sec",
  "HR","von Std:Min:Sek in Fliesszahl umrechnen",
  "RAD","Umschaltung auf Rad fuer Winkelberechnungen (180Grad==Pi)",
  " DEG","Umschaltung auf Grad fuer Winkelberechnungen",
  "Pi","Eulersche Kreiszahl 3.14...",
  "NA","Avogadrokonstante = Anzahl Molekuele pro Mol",
  "SQRT","Wurzel",
  " 1/x","Kehrwert",
  "x^2","x quadrieren",
  "sin","sin(x)  Sinus",
  "cos","cos(x)  Cosinus",
  "tan","tan(x)  Tangens",
  "asin","arcsin(x)  Umkehrfunktion zu Sinus",
  "acos","arccos(x)  Umkehrfunktion zu Cosinus",
  "atan","arctan(x)  Umkehrfunktion zu Tangens",
  "LN","ln(x)  Natuerlicher Logarithmus",
  "LOG","log(x)  ZehnerLogarithmus",
  "POP","x wird geloescht, das oberste Stapelelement dubliziert",
  "ROLv","der Stapel wird rotiert",
  "ROL^","der Stapel wird nach oben rotiert",
  "RCL","Register 0 wird gelesen",
  "STO","x wird in Register 0 gespeichert",
  "RCLx","Register x wird gelesen, (x von 0 bis 9)",
  "STOx","y wird in Register x gespeichert, (x von 0 bis 9)",
  "+/-","Vorzeichen wechseln",
  ".","Dezimalpunkt",
  "x<>y","x und y vertauschen",
  "|x|","Betrag von x",
  "Int","Ganzzahliger Anteil von x",
  "-","y-x --> x",
  "CORR","Zahleneingabe korrigieren",
  "EXP","Exponent eingeben",
  "*","Multiplikation",
  "e^x","2.718... hoch x",
  /*
  "POLAR","Umrechnung von Polarkoordinaten (alfa,phi,Betrag) in XYZ",
  "RALOP","Umrechnung in Polarkoordinaten, X=alfa Y=phi Z=Betrag",
  */
  "HEX","Hexadezimal-Modus einschalten",
  " Dez","Dezimal-Modus einschalten (HEX aus)",
  "Time","aktuelle Zeit als HH.MMSS",
  " Date","aktuelles Datum als JJJJ.MMTT",
  "Tset","Zeit setzen HH.MMSS",
  " Dset","Datum setzen JJJJ.MMTT",
  "Kale","in Kalender-Modus wechseln",
  "MGW","Molekulargewicht berechnen",
  "MW","Resultat vom MGW-Rechner: Durchschnittliches Molekulargewicht",
  "Iso","Resultat vom MGW-Rechner: Isotopenreines Molekulargewicht",
  NULL,NULL
 };

class Tastenfeld
{
 int nreihen,nspalten;
 Taste tasten;
 //int blue;
public:
 Tastenfeld()
   {nreihen=6; nspalten=5;
    tasten.init(tastenname,tastenname_blue,tastenname_gold,tastenname_red);
   }
 void press(int8 i,Eingabefeld *ef,Stapel *ps);
 void shift(int8 j) {tasten.shift(j);}
};

void Tastenfeld::press(int8 i,Eingabefeld *ef,Stapel *ps)
{
 tasten.press(i,ef,ps);
}

int gettastennummer(const char *txt)
{
 int i,n=sizeof(tastenname)/sizeof(char*);
 const char **name=tastenname;
 for(i=0;i<n;i++)
   {if(strcmp(txt,name[i])==0) return i;
   }
 return -1;
}

#define RADIAN 0
#define DEGREE 1
static int8 winkelmodus=DEGREE; //oder RADIAN

Fliesszahl rad(Fliesszahl x)
{
 if(winkelmodus==DEGREE) x*=GRAD;
 return x;
}

Fliesszahl grad(Fliesszahl x)
{
 if(winkelmodus==DEGREE) x/=GRAD;
 return x;
}

static int8 eing=0;
static Fliesszahl tarech_stapel_t(1);
static Fliesszahl tarech_stapel_z(2);
static Fliesszahl tarech_stapel_y(3);
static Fliesszahl tarech_stapel_x(4);

class Vektar
{
 Stapel stapel;
 Eingabefeld eingabefeld;
 Tastenfeld tafeld;
public:
 //Vektar() {}
 void init();
 void press(int8 i);
 void shift(int8 j) {tafeld.shift(j);}
 bool eingabeaktiv() {return eingabefeld.istaktiv();}
 void refresh();
 Fliesszahl pop() {return stapel.pop();}
};
void Vektar::init()
{
 eingabefeld.init(&stapel);
 /* in Constructor von Stapel gemacht
 stapel.push(tarech_stapel_t); //Stapelwerte vom EEPROM wieder in den Stapel schieben
 stapel.push(tarech_stapel_z);
 stapel.push(tarech_stapel_y);
 stapel.push(tarech_stapel_x);
 */
}
void Vektar::press(int8 i)
{
 tafeld.press(i,&eingabefeld,&stapel);
 refresh();
}
void Vektar::refresh()
{
 if(eing==0) stapel.draw();
 eingabefeld.draw();
}
static Vektar vektar;

void gethelp(const char *frage)
{
 const char **p,*s,*antw=NULL;
 for(p=tasten_hilfe; (s= *p++)!=NULL; p++)
   {if(strcmp(s,frage)==0) {antw = *p; break;}
   }
 if(antw==NULL) meldung.say("'%s' no help yet ",frage);
 else meldung.say("'%s' %s ",frage,antw);
}

bool tarech_in_eingabemodus()
{
 return (hauptmodus==TASCHENRECHNER && vektar.eingabeaktiv());
}

/********************* Mathematische Funktionen **************************/
Fliesszahl exp(Fliesszahl x)
{
 // e^x = Summe(n=1...)[ x^n/n! ]
 Fliesszahl result(1),glied,alt(0);
 int n;
 uint16_t iy;
 bool neg;
 if(x<0) {x= -x; neg=true;}
 else neg=false;
 if(x>8) {iy=getint16(x); x -= iy; if(iy>1500) return neg?0:NAN;}
 else iy=0;
 for(n=1, glied=x; result!=alt && n<200; n++,glied *= x/n) //provi. Grenze 200 sollte nicht noetig sein
   {alt = result;
    result += glied;
   }
 //regist[8]=n;//test
 if(iy!=0) result *= pow(ee,iy);
 if(neg) return 1/result;
 return result;
}

/**
Fliesszahl calcln2()  //natuerlicher Logarithmus von 2
{
 // ln(x) = Summe(k=0...)[ 2/(2k+1) * ((x-1)/(x+1))^(2k+1) ]
 // wobei ln(x) = m*ln(2) + ln(x/2^m)
 Fliesszahl result(0),alt(1),faktor1(2,3),faktor2(1,9);
 int zweik1;
 for(zweik1=1; result!=alt; zweik1+=2)
   {alt = result;
    result += faktor1/zweik1;
    faktor1 *= faktor2;
   }
 return result;
}
**/

Fliesszahl log(Fliesszahl x)  //natuerlicher Logarithmus
{
 // ln(x) = Summe(k=0...)[ 2/(2k+1) * ((x-1)/(x+1))^(2k+1) ]
 // wobei ln(x) = m*ln(2) + ln(x/2^m)
 Fliesszahl result(0),alt(1),faktor1,faktor2;
 int zweik1,m;
 if(x.istnull() || x<0) return NAN;//Fehler
 m=x.getexp()-1;
 x.setexp(1);
 faktor1 = (x-1)/(x+1);
 faktor2 = faktor1*faktor1;
 faktor1 += faktor1;
 for(zweik1=1; result!=alt && zweik1<200; zweik1+=2)  //provi. Grenze 200 sollte nicht noetig sein
   {alt = result;
    result += faktor1/zweik1;
    faktor1 *= faktor2;
   }
 //regist[9]=zweik1;//test
 return result+LN2*m;
}

Fliesszahl pow(Fliesszahl x,Fliesszahl y)
{
 int16 iy=getint16(y);
 if(y==iy) return pow(x,iy);
 if(x.istnull()) return (y<0)?NAN:x;
 if(x<0) return NAN;
 return exp(log(x)*y);
}

Fliesszahl sqrt(Fliesszahl x)
{
 Fliesszahl result=1,r0=x;
 int i;
 i=x.getexp()/2;
 if (i > 1)      while(i>0) {result *= 2; --i;}
 else if(i < -1) while(i<0) {result /= 2; i++;}
 i=0;
 do
   {r0 = result;
    result = (r0+x/r0)/2;
   }
 while(result!=r0 && ++i<100); //ev. waere ++i<100 nicht noetig ?
 return result;
}

Fliesszahl x2hms(Fliesszahl x,int16& h,int8& m,int8& s,int8 hundert)
{
 const Fliesszahl dx(1,1000000);
 h = getint16(x);    x = (x-h)*hundert; if(x<0) x=0;
 m = getint8(x+dx);  x = (x-m)*hundert; if(x<0) x=0;
 s = getint8(x+dx);
 return x;
}

Fliesszahl hms(Fliesszahl x)  //in Std:Min:Sek umrechnen, provisorisch
{
 Fliesszahl result;
 int16 std;
 int8 min,s,neg=0;
 if(x<0) {x = -x; neg=1;}
 x = x2hms(x,std,min,s,60);
 result = std + (min + x/100) / 100;
 if(neg) result = -result;
 return result;
}

Fliesszahl hr(Fliesszahl x)  //von Std:Min:Sek zurueckrechnen, provisorisch
{
 Fliesszahl result;
 int16 std;
 int8 min,s,neg=0;
 if(x<0) {x = -x; neg=1;}
 x = x2hms(x,std,min,s,100);
 result = std + (min + x/60) / 60;
 if(neg) result = -result;
 return result;
}

Fliesszahl sin(Fliesszahl x)
{
 // sin(x) = Summe(n=0...)[ (-1)^n*x^(2n+1)/(2n+1)! ] = x/1 - x^3/3! + x^5/5! - ...
 Fliesszahl result(0),glied,alt(1),x2;
 int k; //k=2n+1
 if(x >= PI || x <= -PI)  x -= getint48(x/PI)*PI;
 x2 = -x*x;
 for(k=1, glied=x; result!=alt; k++, glied *= x2/k/(k+1), k++)
   {alt = result;
    result += glied;
   }
 //regist[7]=k;//test
 return result;
}

Fliesszahl cos(Fliesszahl x)
{
 // cos(x) = Summe(n=0...)[ (-1)^n*x^(2n)/(2n)! ] = 1 - x^2/2! + x^4/4! - ...
 Fliesszahl result(0),glied(1),alt(2),x2;
 int k; //k=2n
 if(x<0) x = -x;
 if(x >= ZWEIPI)  x -= getint48(x/ZWEIPI)*ZWEIPI;
 x2 = -x*x;
 for(k=0; result!=alt; k++, glied *= x2/k/(k+1), k++)
   {alt = result;
    result += glied;
   }
 //regist[8]=k;//test
 return result;
}

Fliesszahl tan(Fliesszahl x)
{
 return sin(x)/cos(x);
}

Fliesszahl atan(Fliesszahl x)
{
 // arctan(x) = Summe(n=0...)[ (-1)^n*x^(2n+1)/(2n+1) ] = x/1 - x^3/3 + x^5/5 - ...
 Fliesszahl result(0),glied,alt(1),x2,m(1);
 int k; //k=2n+1
 bool neg;
 if(x<0) {x = -x; neg=true;}
 else neg=false;
 while(x >= einhalbes)
  {x = (sqrt(x*x+1)-1)/x;
   m += m;
  }
 x2 = -x*x;
 for(k=1, glied=x; result!=alt && k<1000; k+=2, glied *= x2)
   {alt = result;
    result += glied/k;
   }
 //regist[9]=k;//test
 result *= m;
 if(neg) result = -result;
 return result;
}

Fliesszahl asin(Fliesszahl x)
{
 if(x==1) return PI/2;
 if(x>1 || -x>1) return NAN;
 return atan(x/sqrt(1-x*x));
}

Fliesszahl acos(Fliesszahl x)
{
 if(x.istnull()) return PI/2;
 if(x>1 || -x>1) return NAN;
 x=atan(sqrt(1-x*x)/x);
 if(x<0) x += PI;
 return x;
}

/*************************** Hauptprogramm *******************************/
//static int8 eing=0;  //schon vergezogen
static Fliesszahl mgwnorm(0),mgwiso(0); //Resultate vom MGW-Rechner, zum in den Taschenrechner uebernehmen

int Taste::press(int8 j,Eingabefeld *ef,Stapel *stap)
{
 static int8 blue=0; // 1 = Taste Blue aktiv, 2 = Taste Gold aktiv, 4 = Taste Red aktiv
 static int8 nx=0;
 int16 c;
 Fliesszahl x,y,z;
 const char *text=text0[j];
 c= *text;
 if(ef->help) {gethelp(text); ef->help=0; return blue;}
 if(ist(text,"Blue")) return blue ^= 1;
 if(ist(text,"Gold")) return blue ^= 2;
 if(ist(text,"Red"))  return blue ^= 4;
 if(ist(text,"ENTR"))
   {if(eing) {ef->close(); eing=0;}
    else stap->push(stap->get(0));
    return blue;
   }
 if(eing==0)
   {if(isdigitx(c) || c=='.' || ist(text,"EXP"))
      {stap->push(0); stap->draw(); ef->open(nx); eing=1; //Eingabemodus einschalten
      }
    /* funktioniert nicht:
    else if(ist(text,"CORR"))
      {ef->reopen(nx); eing=1; //Eingabemodus einschalten
       return blue;
      }
    */
   }
 if(eing==1)
   {if(isdigitx(c)||c=='.'||ist(text,"EXP")||ist(text,"+/-")||ist(text,"CORR"))
      {//Reaktion auf Tasten im Eingabemodus:
       if(ist(text,"+/-")) c='-';
       else if(ist(text,"EXP")) {c='e'; if(hexmodus){ef->write("*10");c=Hochzeichen;}}
       else if(ist(text,"CORR")) c='\b';
       ef->write(c);
       return blue;
      }
    else
      {ef->close(); eing=0;} //Eingabemodus beenden
   }
 if(ist(text,"+/-")) {x=stap->pop(); stap->push(-x);}
 else if(ist(text,"POP")) stap->pop();
 else if(ist(text,"ROLv")) stap->rol(0);
 else if(ist(text,"ROL^")) stap->rol(1);
 else if(ist(text,"+")) {x=stap->pop(); y=stap->pop(); stap->push(y+x);}
 else if(ist(text,"-")) {x=stap->pop(); y=stap->pop(); stap->push(y-x);}
 else if(ist(text,"/")) {x=stap->pop(); y=stap->pop(); stap->push(y/x);}
 else if(ist(text,"*"))	{x=stap->pop(); y=stap->pop(); stap->push(y*x);}
 else if(ist(text,"sin")) {x=stap->pop(); stap->push(sin(rad(x)));}
 else if(ist(text,"cos")) {x=stap->pop(); stap->push(cos(rad(x)));}
 else if(ist(text,"tan")) {x=stap->pop(); stap->push(tan(rad(x)));}
 else if(ist(text,"asin")) {x=stap->pop(); stap->push(grad(asin(x)));}
 else if(ist(text,"acos")) {x=stap->pop(); stap->push(grad(acos(x)));}
 else if(ist(text,"atan")) {x=stap->pop(); stap->push(grad(atan(x)));}
 else if(ist(text,"SQRT")) {x=stap->pop(); stap->push(sqrt(x));}
 else if(ist(text,"x^2")) {x=stap->pop(); stap->push(x*x);}
 else if(ist(text,"x<>y"))
	{x=stap->pop(); y=stap->pop(); stap->push(x); stap->push(y);}
 else if(ist(text,"|x|"))
	{x=stap->pop(); x=abs(x); stap->push(x);}
 else if(ist(text,"Int"))
	{x=stap->pop(); x=getint48(x); stap->push(x);}
 else if(ist(text,"Pi")) {stap->push(PI);}
 else if(ist(text,"NA")) {stap->push(Avogadrokonstante);}
 else if(ist(text,"g")) {stap->push(Erdbeschleunigung);}
 else if(ist(text,"c")) {stap->push(Lichtgeschwindigkeit);}
 //else if(ist(text," e")) {stap->push(ee);}
 else if(ist(text,"e^x")) {x=stap->pop(); stap->push(exp(x));}
 else if(ist(text," 10^x")) {x=stap->pop(); stap->push(pow(10,x));}
 else if(ist(text,"LN")) {x=stap->pop(); stap->push(log(x));}
 else if(ist(text,"LOG")) {x=stap->pop(); stap->push(log(x)/LN10);}
 else if(ist(text,"y^x")) {x=stap->pop(); y=stap->pop(); stap->push(pow(y,x));}
 //else if(ist(text,"y^1/x")) {x=stap->pop(); y=stap->pop(); stap->push(pow(y,1/x));}
 else if(ist(text," 1/x")) {x=stap->pop(); stap->push(1/x);}
 else if(ist(text,"RAD")) winkelmodus=RADIAN;
 else if(ist(text," DEG")) winkelmodus=DEGREE;
 else if(ist(text,"STO")) regist[0]=stap->get(0);
 else if(ist(text,"RCL")) stap->push(regist[0]);
 else if(ist(text,"STOx"))
	{x=stap->pop(); c=getint16(x);
	 if(x==c && c>=1 && c<=9) regist[c]=stap->get(0);
	 else {stap->push(x); meldung.error("only register 1 to 9 available");}
	}
 else if(ist(text,"RCLx"))
	{x=stap->pop(); c=getint16(x);
	 if(x==c && c>=1 && c<=9) stap->push(regist[c]);
	 else {stap->push(x); meldung.error("only register 1 to 9 available");}
	}
 else if(ist(text,"Help"))
	{meldung.say1("press key for help"); ef->help=1;}
 else if(ist(text,"HEX"))
   {hexmodus=1;}
 else if(ist(text," Dez"))
   {hexmodus=0;}
 else if(ist(text,"HMS"))
   {x=stap->pop();
    stap->push(hms(x));
   }
 else if(ist(text,"HR"))
   {x=stap->pop();
    stap->push(hr(x));
   }
 else if(ist(text,"Time"))
   {x = stunden + (min + Fliesszahl(sec)/100)/100;
    stap->push(x);
   }
 else if(ist(text," Date"))
   {x = jahr + (monat + Fliesszahl(tage)/100)/100;
    stap->push(x);
   }
 else if(ist(text,"Tset"))
   {int h; int8 m,s;
    x=stap->pop();
    x2hms(x,h,m,s,100);
    cli(); stunden=h; min=m; sec=s; sei();
#ifndef LCD_MK3
    zeit_in_rtc_speichern();
#endif
   }
 else if(ist(text," Dset"))
   {int j; int8 m,t;
    x=stap->pop();
    x2hms(x,j,m,t,100);
    cli(); jahr=j; monat=m; tage=t; sei();
    schaltjahr_und_wochentagberechnen();
#ifndef LCD_MK3
    zeit_in_rtc_speichern();
#endif
   }
 else if(ist(text,"Kale"))
   {hauptmodus=KALENDER;
   }
 else if(ist(text,"MGW"))
   {hauptmodus=MGRECHNER;
   }
 else if(ist(text,"MW")) {stap->push(mgwnorm);}
 else if(ist(text,"Iso")) {stap->push(mgwiso);}
 //neue Funktionen hier einfuegen

 else meldung.error("Err:'%s'",text);
 return blue;
}

void tarech_init()
{
 vektar.init();
 vektar.refresh();
 tarech_tastenreaktion(0);
}

char tastennummer(uchar c)
{
 char str[6];
 switch(c)
  {case 'a': strcpy(str,"Gold"); break;
   case 'b': strcpy(str,"Blue"); break;
   case 'c': strcpy(str,"Red"); break;
   case 'd': strcpy(str,"POP"); break;
   case 'e': strcpy(str,"CORR"); break;
   case '=': strcpy(str,"ENTR"); break;
   case 'I': strcpy(str,"+/-"); break;
   case 'J': strcpy(str,"x<>y"); break;
   case 'N': strcpy(str,"EXP"); break;
   default: str[0]=c; str[1]=0;
  }
 return gettastennummer(str);
}

void tarech_tastenreaktion(uchar taste)
{
 static uchar gold=0,blue=0,red=0;
 int8 i,j;
 if(taste)
  {i=tastennummer(taste);
   //wprintf(3,"i=%d  ",i);//test
   if(i==0) {gold ^= 1;}
   if(i==1) {blue ^= 1;}
   if(i==2) {red  ^= 1;}
   if(gold)  j = 2; else j=0;
   if(blue) {j = 1; gold=0;} //Gold ungueltig wenn Blue gedrueckt
   if(red)   j = 3; //vorlaeufig Gold und Blue ungueltig wenn Red gedrueckt
   //if(red)  j += 3; //Gold-Red (j=5) oder Blue-Red (j=4) fuer weitere Spezialfunktionen

   if(i>2) //Gold-, Blue- und Red-Tasten sind schon ausgwertet
     {vektar.shift(j);
      vektar.press(i);
      //red=blue=0; //Blue und Red sind nur fuer einen Tastendruck gueltig
      gold=red=blue=0; //Gold, Blue und Red sind nur fuer einen Tastendruck gueltig
     }
  }
 glcd_goto(1,19);
 glcd_write((gold) ? 'G' : (blue)?'B':' ');
 glcd_write((red) ?'R':' ');
 if(vektar.eingabeaktiv())
      glcd_goto(4,19); //Cursor hinter letztem eingegebenen Zeichen
 else glcd_goto(4,20); //sonst Cursor auf letzter Zeile ganz rechts
}

void tarech_refresh()
{
 vektar.refresh();
 tarech_tastenreaktion(0);
}

/***************************** Molekulargewichts-Rechner *********************************/
#ifdef MATRIXQUER
static const char *tasten_green[]=
 {
  //"Gold","Caps","LetDig","Help","CORR","/",
  " ",   " ",   " ",     " ",   " ",   "\"\'",
  "7",   "8",   "9",     "A",   "E",   "&~",
  "IJ",  "NM",  "RQ",    "BT",  "FW",  "Z=",
  "HG",  "KL",  "OP",	 "CS",  "UV",  "XY",
  "({[<",",:;!?",")}]>", "D@#$%","\\|_^","ENTR"
 };
#else
static const char *tasten_green[]=
 {
  " ",  "A",    "7",   "IJ",  "HG",  "({[<",
  " ",  "BT",   "8",   "NM",  "KL",  ",:;!?",
  " ",  "CS",   "9",   "RQ",  "OP",  ")}]>",
  " ", "D@#$%", "E",   "FW",  "UV",  "\\|_^",
  " ",  "\"\'", "&~",  "Z=",  "XY",  "ENTR"
 };
#endif

#define MAXF 126
#define MAXF2 128
static char mgwtext[MAXF2]="",tmptext[MAXF2]="  ";
static uint8 jmgw=0; //Cursor innerhalb mgwtext
static uint8 formel_nr=0; //aktuelle Formel-Nummer

void mgrechner_init()
{
 glcd_clear();
 lcd_gowrite(1,1,"MGW-Calculator");
 mgrechner_tastenreaktion(1); //noetig um vom EEPROM eingelesene Formel direkt anzuzeigen
}

bool exor(bool a,bool b)
{
 return ((a && !b) || (!a && b));
}

#ifdef MATRIXQUER
const uint8 mgwtastenbelegung[30]={0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29};
#else
const uint8 mgwtastenbelegung[30]={0,9,6,12,18,24,1,15,7,13,19,25,2,21,8,14,20,26,3,27,10,16,22,28,4,5,11,17,23,29};
#endif

//static Fliesszahl mgwnorm(0),mgwiso(0); //schon weiter oben definiert zum in den Taschenrechner uebernehmen

void mgrechner_tastenreaktion(uint8 tastenr)
{
 const uint DT=500; //falls Doppelklick nicht gut erkannt, dann noch anpassen
 static uint zeitstempel=0;
 static uint8 k=0,ivorher=0;
 static uint8 modus=0; //0=Texteingabe-Grossbuchstaben, 1=Kleinbuchstaben, 2=Zahlen,
#define MODUS_KLEINBIT 1
#define MODUS_DIGITBIT 2
#define MODUS_CAPSBIT  4    //Gross/Klein-Wechsel fuer naechste Taste
#define MODUS_DIGLOCKBIT 8  //Zahlen/Buchstabe-Wechsel fuer naechste Taste

 char c,h;
 uint8_t i,x1,y1,tastenr1;
 bool resultatflag=0;
 char flags[3]="I4"; //provi. Flags sollte von Anwender eingestellt werden koennen
                     //Erster Buchstabe wird weiter unten automatisch auf 'I' oder 'D' gesetzt, andere Flags
                     //sollten irgendwie vom Anwender gesetzt werden koennen.
 tastenr1 = --tastenr; //Nummer vom Bereich 1-30 in Bereich 0-29 aendern
 tastenr=mgwtastenbelegung[tastenr];
 if(mgwtext[0]==0) {jmgw=0; modus=0; k=0; ivorher=0;} //Initialisierungen fuer neuen Text
 //Test: Tastennummer von gedrueckten Tasten anzeigen:
 //if(jmgw<MAXF) {mgwtext[jmgw++]=tastenr/10+'0'; mgwtext[jmgw++]=tastenr%10+'0'; mgwtext[jmgw]=0;}//test
 switch(tastenr)
  {
   case 0: //Gold-Taste
     break;
   case 1: //Caps-Taste
     if(modus&MODUS_CAPSBIT)
       {modus ^= MODUS_KLEINBIT; modus &= ~MODUS_CAPSBIT;} //Wechsel zwischen Gross- und Klein-Buchstaben
     else modus |= MODUS_CAPSBIT; //Nur naechste Taste Grosss/Klein-Wechsel
     tastenr=0; break;
   case 2: //Dig/Lett-Taste
     if(modus&MODUS_DIGLOCKBIT)
       {modus ^= MODUS_DIGITBIT; modus &= ~MODUS_DIGLOCKBIT;} //Wechsel zwischen Buchstaben und Zahlen
     else modus |= MODUS_DIGLOCKBIT; //Nur naechste Taste Grosss/Klein-Wechsel
     tastenr=0; break;
   case 3: //POP/Help-Taste
     glcd_goto(2,1); glcd_write(tmptext); //Cursor am Ende der momentanen Formel loeschen
     mgw_formel_rotieren();
     tastenr=0; break;
   case 4: //Corr-Taste
     if(jmgw>0)
       {--jmgw;
	for(i=jmgw;mgwtext[i]!=0;i++) //Zeichen links vom Cursor loeschen
	     mgwtext[i]=mgwtext[i+1];          //und Text nachruecken.
       }
     tastenr=0; break;
   case 6: //Pfeil-Links-Taste
     if((modus&MODUS_DIGITBIT)==0)
       {
        if(jmgw>0) --jmgw;
        tastenr=0;
       }
     break;
   case 7: //Pfeil-Rechts-Taste
     if((modus&MODUS_DIGITBIT)==0)
       {
        if(jmgw<MAXF && mgwtext[jmgw]!=0) ++jmgw;
        tastenr=0;
       }
     break;
   case 29: //Enter-Taste
     mg_error_reset();
     flags[0]='D'; //normales, mittleres Molekulargewicht
     mgwnorm=mgw_ausrechnen(mgwtext,flags);
     flags[0]='I'; //isotopenreines Molekulargewicht
     mgwiso=mgw_ausrechnen(mgwtext,flags);
     resultatflag=true;
     tastenr=0; break;
  }
 if(tastenr==0)
  {
   zeitstempel=0;
  }
 else //Alle andern Tasten auswerten
  {
   uint zeit;
   bool doppelklick;
   zeit=((uint)(sec)<<10)+millisec+DT; //eigentlich sec*1000, sec<<10 ist aber kuerzer?
   if(zeitstempel>zeit) zeitstempel -= 60<<10; //Korrektur bei Minutenwechsel
   //printf("Zeitdifferenz = %d\n",zeit-zeitstempel);//test
   if(exor((modus&MODUS_DIGITBIT)!=0,(modus&MODUS_DIGLOCKBIT)!=0))
     {
      c=tastencode[tastenr1]; //Stimmt fuer Ziffern, Punkt und +*-/
      if(c=='J') c=' '; //Taste <> als Blank verwenden
      if(c=='N') c=128; //Taste Exp als Gradzeichen verwenden
      if(c=='I') c=137; //Taste +/- als Dreifachbindung verwenden
      modus &= ~MODUS_DIGLOCKBIT;
      doppelklick=false;
     }
   else
     {
      if(jmgw>0 && zeit<zeitstempel+DT && tastenr==ivorher) //Doppelklick?
	{
	 if(++k==strlen(tasten_green[tastenr1])) k=0;
	 doppelklick=true;
	}
      else
	{
	 k=0;
	 doppelklick=false;
	}
      c=tasten_green[tastenr1][k];
      if(exor((modus&MODUS_KLEINBIT)!=0,(modus&MODUS_CAPSBIT)!=0)
	 && c>='A' && c<='Z')
	c += 'a'-'A';
      modus &= ~MODUS_CAPSBIT;
     }
   if(doppelklick)
     {
      if(((h=mgwtext[jmgw-1])>='a' && h<='z') //war zu ueberschreibendes ein Kleinbuchstabe?
	 && c>='A' && c<='Z')
	c += 'a'-'A'; //ja: neuer auch ein Kleinbuchstabe
      mgwtext[jmgw-1]=c;
     }
   else if(jmgw<MAXF)
     {
      for(i=jmgw;(h=mgwtext[i])!=0;i++)
	{mgwtext[i]=c; c=h;}
      mgwtext[i]=c; mgwtext[i+1]=0;
      jmgw++;
     }
   zeitstempel=zeit;
  }
 ivorher=tastenr;
 
 sprintf(tmptext,"%s  ",mgwtext);
 glcd_goto(2,1);
 glcd_write(tmptext);
 glcd_drawmode(COMPLEMENT);
 y1=text_zeilenabstand;
 for(i=jmgw;i>=(127/(FONTBREITE+1));)
   {
    i -= (127/(FONTBREITE+1));
    y1 += text_zeilenabstand;
   }
 x1=i*(FONTBREITE+1);
 glcd_fillbox(x1,y1,x1+FONTBREITE-1,y1+FONTHOEHE-1);
 glcd_drawmode(MODE_NORMAL);
 if(resultatflag)
   {
      lcd_gowrite_d(6,mgwiso); //Isotop-MG auf Zeile 6
      lcd_gowrite_d(5,mgwnorm); //Normales MG auf Zeile 5
   }
}

#ifndef OHNEMGW
Fliesszahl mg_main(char *zeile,char *zeile2,int max1,int max2,const char *flags);
#endif

Fliesszahl mgw_ausrechnen(char *text,char* flags)
{
 Fliesszahl z;
 //static char flags[3]="D4"; //provi. Flags sollte von Anwender eingestellt werden koennen
 //static char flags[3]="D9"; //test (zum Genauigkeit austesten)
#ifdef OHNEMGW
 z=Fliesszahl("123.4567",0);//test
#else
 char text2[MAXF2+40];
 z=mg_main(text,text2,MAXF,MAXF+40,flags);
#endif
 return z;
}

//------------------------- aus mg.cc kopiert -------------------------
//Anpassung weil es im AVR kein double gibt:
#ifdef MGWFLOAT
//ergibt kuerzeren Code, aber Zahlen nicht ganz so genau
#define double float
#else
//etwas laengeren Code, dafuer Zahlen sehr genau
#define double Fliesszahl
#endif

//#define SUMMENFORMELN
#include "mgkern.h"

/** globale Parameter **/
#ifndef OHNEMGW
static int8 actinium_warnung=0; //fuer Warnung um Actinium nicht mit Acetyl zu verwechseln
static char fehlermeldung[80],warnstr[80];

void mg_error_reset()
{
 error_reset(fehlermeldung,warnstr);
 lcd_gowrite21(4," ");
 lcd_gowrite21(5," ");
}

Fliesszahl mg_main(char *zeile,char *zeile2,int max1,int max2,const char *flags)
{
 double mg;
 char *s,cstr[40];
 int8 i,n;

 //Flags: Ix=nur häufigste Isotope verwenden
 //       Dx=nur Durchschnitt rechnen
 //       x=Anzahl Nachkommastellen (z.B. 4)
 i=flags[1]-'0'; if(i>9 || i<1) i=4;
 sprintf(cstr,"%%.%df",i);

 //error_reset(fehlermeldung,warnstr);

 formel_normieren(zeile,zeile2,max1,max2);
 //if(strcmp(zeile,zeile2)!=0) printf("Normierte Formel: %s\n",zeile2);
 if((n=klammertest(zeile2))!=0)
   sprintf(fehlermeldung,"fehlende Klammer %s ",n<0?"auf":"zu");
 //actinium_warnung=0; //wird schon in error_reset gemacht
 if(flags[0]=='I') mg=mol_weight(zeile2,&s,ISO);
 else              mg=mol_weight(zeile2,&s,0);
 if(fehlermeldung[0]==0 && actinium_warnung>0)
   sprintf(fehlermeldung,"Remark: Ac = Actinium, Ace = Acetyl ");

 if(fehlermeldung[0]!=0)
  {
   if(strlen(fehlermeldung)>21)
    {
     lcd_gowrite21(4,fehlermeldung);
     lcd_gowrite21(5,&fehlermeldung[21]);
    }
   else
    {
     //lcd_gowrite21(4," ");
     lcd_gowrite21(5,fehlermeldung);
    }
   secwait(1);
  }

#ifdef MGWFLOAT
 char tmp[40];
 sprintf(tmp,cstr,mg); //Umwandeln von float in Fliesszahl
 return Fliesszahl(tmp,0);
#else
 return mg;
#endif
}

/*
void fehler0(const char *text,int c)
{
 sprintf(fehlermeldung,"%s %c ",text,c);
}
void fehler1(const char *text)
{
 sprintf(fehlermeldung,"%s ",text);
}
void fehler2(const char *text1,const char *text2)
{
 sprintf(fehlermeldung,"%s%s ",text1,text2);
}
*/
//------------------------- aus mg.cc kopiert -------------------------

//---------------------- angepasstes mgkern.c -------------------------
#ifdef MGWFLOAT
#include "mgkern.c"
#else
#define FLIESSZAHLKLASSE

// Anpassungen fuer avr-gcc wegen komischen Fehlermeldungen:
__extension__ typedef int __guard __attribute__((mode (__DI__)));
extern "C" int __cxa_guard_acquire(__guard *);
extern "C" void __cxa_guard_release (__guard *);
extern "C" void __cxa_guard_abort (__guard *);
int __cxa_guard_acquire(__guard *g) {return !*(char *)(g);};
void __cxa_guard_release (__guard *g) {*(char *)g = 1;};
void __cxa_guard_abort (__guard *) {};

#include "mgkernf2.c"
#endif
//---------------------- angepasstes mgkern.c -------------------------
#endif //OHNEMGW

//--------------------- Test fuer 64KB Problem ------------------------
#ifdef TEST64KB
// Unterprogramm anhaengen um das gesamte Programm groesser 64KB zu machen
void test_64kb()
{
  __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n"); //8 nops belegen 16 Bytes
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n"); //32 Bytes
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n"); //64 Bytes
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n"); //128 Bytes
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n"); //256 Bytes
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n");
 __asm volatile ("nop \n nop \n nop \n nop \n nop \n nop \n nop \n nop \n"); //512 Bytes
}
#endif
//--------------------- Test fuer 64KB Problem ------------------------

//------------------------- EEPROM Routinen ---------------------------
void fliesszahl_vom_eeprom_einlesen(uchar* eezahl,void* zahl);
void fliesszahl_im_eeprom_speichern(uchar* eezahl,void* zahl);

#ifdef LCD_SIMULATOR
#define EEMEM
uchar eeprom_read_byte(const uchar* s) {return s[0];}
void eeprom_write_byte(uchar* s,uchar c) {s[0]=c;}
#else
#include <avr/eeprom.h>
#endif

static uint8_t ee_formel_nr[2] EEMEM;        // [0] entspricht formel_nr, [1] noch als Reserve
static uchar ee_mgwformeln[4*(MAXF2)] EEMEM; // 4 Chemische Formeln im EEPROM verwalten (fuer mehr, dann SD-Karte verwenden)

//static Fliesszahl ee_stapel[STAPELSIZE] EEMEM; //geht so nicht, weil Klassen-Constructor auf RAM angewendet wuerde
static uchar ee_stapel[STAPELSIZE*sizeof(Fliesszahl)] EEMEM;
static uchar ee_regist[ANZAHLREGIST*sizeof(Fliesszahl)] EEMEM;
static uchar ee_mgwnorm[sizeof(Fliesszahl)] EEMEM;
static uchar ee_mgwiso[sizeof(Fliesszahl)] EEMEM;

void string_vom_eeprom_einlesen(const uint8_t* eeadr,char *buffer,uint8 max)
{
 char c, c0=0xFF; //0xFF fuer String-Abschluss im EEPROM benutzt.
   //Es wird nicht wie ueblich mit 0 abgeschlossen weil, wenn das EEPROM geloescht wird,
   //dann ist gesamter Inhalt jeweils 0xFF. Deshalb 0xFF als String-Abschluss benutzt,
   //so ist es bei geloeschtem EEPROM automatisch ein leerer String.
 for(jmgw=0;(c=eeprom_read_byte(eeadr+jmgw))!=c0 && jmgw<max-1;jmgw++)
     *buffer++ = c;
 *buffer = 0; //der String im RAM jedoch wie ueblich mit 0 abschliessen.
}

void string_im_eeprom_speichern(uint8_t* eeadr,const char *buffer,uint8 max)
{
 char buffer2[max],c;
 string_vom_eeprom_einlesen(eeadr,buffer2,max);
 if(strcmp(buffer,buffer2)!=0) //nur speichern wenn unterschiedlich zum schon gespeicherten Wert
  {
   uint8 i;
   for(i=0;(c=buffer[i])!=0 && i<max-1;i++)
     eeprom_write_byte(eeadr+i, c);
   eeprom_write_byte(eeadr+i, 0xFF); //String im EEPROM mit 0xFF abschliessen
   //Es wird nicht wie ueblich mit 0 abgeschlossen weil, wenn das EEPROM geloescht wird,
   //dann ist gesamter Inhalt jeweils 0xFF. Deshalb 0xFF als String-Abschluss benutzt,
   //so ist es bei geloeschtem EEPROM automatisch ein leerer String.
  }
}

void formel_speichern()
{
 string_im_eeprom_speichern(&ee_mgwformeln[formel_nr*MAXF2],mgwtext,MAXF2);
}

void aktuelle_formel_im_eeprom_speichern()
{
 //aktuelle im MG-Rechner eingegebene Formel im EEPROM speichern
 if(formel_nr != eeprom_read_byte(ee_formel_nr))
   eeprom_write_byte(ee_formel_nr,formel_nr);
 formel_speichern();
 fliesszahl_im_eeprom_speichern(ee_mgwnorm,&mgwnorm);
 fliesszahl_im_eeprom_speichern(ee_mgwiso,&mgwiso);
}

void formel_einlesen()
{
 string_vom_eeprom_einlesen(&ee_mgwformeln[formel_nr*MAXF2],mgwtext,MAXF2);
}

void aktuelle_formel_vom_eeprom_einlesen()
{
 //vom MG-Rechner gespeicherte Formel wieder vom EEPROM einlesen
 formel_nr=eeprom_read_byte(ee_formel_nr);
 if(formel_nr<0 || formel_nr>=4) formel_nr=0;
 formel_einlesen();
 fliesszahl_vom_eeprom_einlesen(ee_mgwnorm,&mgwnorm);
 fliesszahl_vom_eeprom_einlesen(ee_mgwiso,&mgwiso);
}

void mgw_formel_rotieren()  // Zwischen den MGW-Formeln im EEPROM rotieren
{
 formel_speichern();
 if(++formel_nr==4) formel_nr=0;
 formel_einlesen();
}

void fliesszahl_vom_eeprom_einlesen(uchar* eezahl,void* zahl)
{
 uchar *buffer=(uchar*)zahl;
 bool leerflag=true; //Test auf geloeschtes EEPROM
 uint8 i,imax=sizeof(Fliesszahl);
#ifdef LCD_SIMULATOR
 printf("Testpunkt: fliesszahl_vom_eeprom_einlesen() imax=%d\n",imax);//test
#endif
 for(i=0;i<imax;i++)
   {
    buffer[i]=eeprom_read_byte(&eezahl[i]);
    if(buffer[i]!=0xFF) leerflag=false; //im geloeschten EEPROM sind alle Bytes auf 0xFF
   }
 if(leerflag)
   {
    for(i=0;i<imax-1;i++)  buffer[i]=0; //Buffer mit 0 fuellen fuer Fliesszahl 0
#ifdef LCD_SIMULATOR
    buffer[i]=0;
#else
    buffer[i]=ZEROFLAG; //Fliesszahl auf exakt 0 setzen
#endif
   }
#ifdef LCD_SIMULATOR
 printf("Testpunkt: fliesszahl_vom_eeprom_einlesen() fertig\n");//test
#endif
}

void fliesszahl_im_eeprom_speichern(uchar* eezahl,void* zahl)
{
 uchar *buffer=(uchar*)zahl;
 uchar buffer2[sizeof(Fliesszahl)];
 bool gleichflag;
 uint8 i;
 fliesszahl_vom_eeprom_einlesen(eezahl,buffer2);
 for(i=0,gleichflag=true;i<sizeof(Fliesszahl);i++)
   if(buffer[i]!=buffer2[i]) gleichflag=false;
 if(gleichflag==false)
  {
   for(i=0;i<sizeof(Fliesszahl);i++)
     eeprom_write_byte(&eezahl[i],buffer[i]);
  }
}

void stapel_vom_eeprom_einlesen(uint8 i,Fliesszahl *pz) //Zuletzt gespeicherte Werte vom EEPROM wieder einlesen
{
#ifndef LCD_SIMULATOR
 fliesszahl_vom_eeprom_einlesen(&ee_stapel[i*sizeof(Fliesszahl)],pz);
#endif
}

void tarech_stapel_im_eeprom_speichern() //Stapel vom Taschenrechner im EEPROM speichern
{
 Fliesszahl zahl;
 for(int8 i=0;i<STAPELSIZE;i++)
  {
   zahl=vektar.pop();
   fliesszahl_im_eeprom_speichern(&ee_stapel[i*sizeof(Fliesszahl)],&zahl);
  }
}

void tarech_regist_vom_eeprom_einlesen() //Zuletzt gespeicherte Register-Werte vom EEPROM wieder einlesen
{
#ifndef LCD_SIMULATOR
 for(int8 i=0;i<ANZAHLREGIST;i++)
   fliesszahl_vom_eeprom_einlesen(&ee_regist[i*sizeof(Fliesszahl)],&regist[i]);
#endif
}

void tarech_regist_im_eeprom_speichern() //Rgister vom Taschenrechner im EEPROM speichern
{
 for(int8 i=0;i<ANZAHLREGIST;i++)
  {
   fliesszahl_im_eeprom_speichern(&ee_regist[i*sizeof(Fliesszahl)],&regist[i]);
  }
}
