/* otekplot1.h  letzte Aenderungen: 27.12.2021, 28.12.2021, 14.1.2022
Einfache 2D-Grafiken mit OpenGL zeichnen.
Gleiche Routinen wie in xtekplot1, so dass Programme, die mit xtekplot1
geschrieben wurden, mit moeglichst wenig Aenderungen auch mit OpenGL laufen.
Und somit auch unter Windows.

Im Hauptprogramm muss es so eingefuegt werden:
#define TEKPLOT1_IMPLEMENTATION
#include "otekplot1.h"
In weiteren Quelldateien jeweils ohne TEKPLOT1_IMPLEMENTATION einfuegen.

Wenn glversion ermittelt werden soll:
int main(int argc,char *argv[])
{
 check_glversion(argc,argv);
 ...
}

*/
#ifndef OTEKPLOT1_H
#define OTEKPLOT1_H

//#define SHADERS_OLD  //mit alter Version 120 ausprobieren

#ifdef TEKPLOT1_IMPLEMENTATION
const char *tekplot_version="Version 0.25";
const char *tekplot_fenstername="OpenGL-Tekplot1 Version 0.25";

#ifndef _WIN32
#ifndef SHADERS_OLD
#define CHECK_GL_VERSION  //provisorisch
#endif
#endif

#endif
/*
Autor: Rolf Pfister
Copyright: Opensource Freeware

letzte Aenderungen:
13.12.2020  0.1  Erstellung
2.1.2021         Grundfunktionen mit einfachem Spiel getestet
20.1.2021        setmenu() realisiert (bisher ohne setsubmenu())
29.1.2021        separate Vertex-Liste fuer Menu
3.2.2021    0.2  Requester, nochmals separate Vertex-Liste, als Mesh realisiert
19.2.2021   0.21 class Mesh verbessert, alle Zeichenoperationen mit Meshs
14.11.2021  0.22 Fadenkreuz, tek_grein()
30.11.2021  0.23 neue Fadenkreuz-Funktionen: fadenmodus, faden_ix[], ...
21.12.2021       Wenn shaders/basic.vs oder .fs nicht gefunden, dann
                 entsprechende Datei direkt als String einfuegen.
27.12.2021  0.24 Vergessene Aenderungen (Fadenkreuzmodus) vom 11.12. wieder restauriert
28.12.2021       Statustext eingefuegt
12.1.2022   0.25 Bessere Tastatureingabe mit SDL_TEXT

TODO:
 - Submenu
*/

//#define _DEBUG  //bei Bedarf im makefile definiert

#include <fstream>
#include <iostream>

#define GLEW_STATIC
#include <GL/glew.h>

#define SDL_MAIN_HANDLED
#ifdef _WIN32
#include <SDL.h>
#pragma comment(lib, "SDL2.lib")
#pragma comment(lib, "glew32s.lib")
#pragma comment(lib, "opengl32.lib")
#pragma warning(disable : 4996) //Warnungen fuer sscanf() ignorieren
#else
#include <SDL2/SDL.h>
#endif
#include <math.h>
#include <string.h>

#ifdef TEKPLOT1_IMPLEMENTATION
#define STB_IMAGE_IMPLEMENTATION
#endif
#include "libs/stb_image.h"
#include "libs/glm/glm.hpp" //opengl-Mathe-Bibliothek
#include "libs/glm/ext/matrix_transform.hpp"
#include "libs/glm/gtc/matrix_transform.hpp"

#include<cstdint>
#include<vector>

typedef int8_t int8;
typedef int16_t int16;
typedef int32_t int32;
typedef int64_t int64;
typedef uint8_t uint8;
typedef uint16_t uint16;
typedef uint32_t uint32;
typedef uint64_t uint64;
typedef float float32;
typedef double float64;

struct Vertex {
 float32 x, y, z; //Koordinaten
 float32 r,g,b,a; //Farbe
 Vertex() {x=y=z=0; r=g=b=0.5f; a=1.0f;}
 Vertex(float32 xf, float32 yf, float32 zf, int32 color) {setxyzcol(xf,yf,zf,color);}
 void setxyzcol(float32 xf, float32 yf, float32 zf, int32 color) {
  x=xf; y=yf; z=zf;
  r=(color&0xFF)/255.0f;
  g=((color>>8)&0xFF)/255.0f;
  b=((color>>16)&0xFF)/255.0f;
  a=1.0f;
 }
};

class VertexBuffer {
public:
 VertexBuffer(void* data, uint32 numVertices) {
    glGenVertexArrays(1, &vao);
    glBindVertexArray(vao);
    glGenBuffers(1, &bufferId);
    //irgendwie wird da ein Puffer im Speicher der Grafikkarte angelegt?
    glBindBuffer(GL_ARRAY_BUFFER, bufferId);
    glBufferData(GL_ARRAY_BUFFER, numVertices * sizeof(Vertex), data, GL_STATIC_DRAW);

    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex), (void*)offsetof(struct Vertex, x));
    glEnableVertexAttribArray(1);
    glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, sizeof(Vertex), (void*)offsetof(struct Vertex, r));

    glBindVertexArray(0);
 }
 ~VertexBuffer() {glDeleteBuffers(1, &bufferId); glDeleteVertexArrays(1, &vao);}
 void bind() {glBindVertexArray(vao);}
 void unbind() {glBindVertexArray(0);}
private:
 GLuint bufferId;
 GLuint vao;
};

class IndexBuffer {
public:
 IndexBuffer(void* data, uint32 numIndices, uint8 elementSize) {
    //Puffer im Speicher der Grafikkarte anlegen:
    glGenBuffers(1, &bufferId);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferId);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, numIndices * (uint32)elementSize, data, GL_STATIC_DRAW);
 }
 ~IndexBuffer() {glDeleteBuffers(1, &bufferId);}
 void bind() {glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferId);}
 void unbind() {glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);}
private:
 GLuint bufferId;
};

class Shader {
public:
 Shader(const char* vertexShaderFilename, const char* fragmentShaderFilename);
 virtual ~Shader();
 void bind();
 void unbind();
 GLuint getShaderId() {return shaderId;}

private:
 GLuint compile(std::string shaderSource, GLenum type);
 std::string parse(const char* filename);
 GLuint createShader(const char* vertexShaderFilename, const char* fragmentShaderFilename);
 GLuint shaderId;
};

char *mystrncpy(char *dest,const char *src,int max);
char *mystrncpy2(char *dest,const char *src,int max);

#ifdef TEKPLOT1_IMPLEMENTATION
Shader::Shader(const char* vertexShaderFilename, const char* fragmentShaderFilename)
{
 shaderId = createShader(vertexShaderFilename, fragmentShaderFilename);
}
Shader::~Shader() {glDeleteProgram(shaderId);}
void Shader::bind() {glUseProgram(shaderId);}
void Shader::unbind() {glUseProgram(0);}

GLuint Shader::compile(std::string shaderSource, GLenum type)
{
 GLuint id = glCreateShader(type);
 const char* src = shaderSource.c_str();
 glShaderSource(id, 1, &src, 0);
 glCompileShader(id);
 
 int result;
 glGetShaderiv(id, GL_COMPILE_STATUS, &result);
 if(result != GL_TRUE)
  {
   int length = 0;
   glGetShaderiv(id, GL_INFO_LOG_LENGTH, &length);
   char* message = new char[length];
   glGetShaderInfoLog(id, length, &length, message);
   if(type==GL_VERTEX_SHADER)
    std::cout << "Vertex Shader compile error: " << message << std::endl;
   else
    std::cout << "Fragment Shader compile error: " << message << std::endl;
   delete[] message;
   return 0;
  }
 return id;
}

FILE *myfopen(const char *filename,const char *rw)
{
 FILE* fp=NULL;
#ifdef _WIN32
 fopen_s(&fp, filename, rw);
#else
 fp = fopen(filename, rw);
#endif
 return fp;
}

char *mystrncpy2(char *dest,const char *src,int max)
{
 const char *s=src;
 char *d=dest;
 for(int i=1; i<max && *s!=0; i++)  {*d++ = *s++;}
 *d = 0;
 if(*s!=0) fprintf(stderr,"Error: in mystrncpy(.., src=\"%s\", max=%d) --> dest=\"%s\"\n",src,max,dest);//test
 return dest;
}

char *mystrncpy(char *dest,const char *src,int max)
{
 const char *s=src;
 char *d=dest;
 for(int i=1; i<max && *s!=0; i++)  {*d++ = *s++;}
 *d = 0;
 return dest;
}

std::string Shader::parse(const char* filename)
{
 std::string contents;
 FILE* file=myfopen(filename,"rb");
 if(file==NULL)
  {
   if(strcmp(filename,"shaders/basic.vs")==0)
    {contents="#version 330 core\n\
layout(location=0) in vec3 a_position;\n\
layout(location=1) in vec4 a_color;\n\
out vec4 v_color;\n\
void main() {\n\
 gl_Position = vec4(a_position, 1.0f);\n\
 v_color = a_color;\n}\n";
     return contents;
    }
   else if(strcmp(filename,"shaders/basic.fs")==0)
    {contents="#version 330 core\n\
layout(location=0) out vec4 f_color;\n\
in vec4 v_color;\n\
void main() {f_color = v_color;}\n";
     return contents;
    }
   else if(strcmp(filename,"shaders/basic120.vs")==0)
    {contents="#version 120\n\
attribute vec3 a_position;\n\
attribute vec4 a_color;\n\
varying vec4 v_color;\n\
void main() {\n\
 gl_Position = vec4(a_position, 1.0f);\n\
 v_color = a_color;\n}\n";
     return contents;
    }
   else if(strcmp(filename,"shaders/basic120.fs")==0)
    {contents="#version 120\n\
varying vec4 v_color;\n\
void main() {gl_FragColor = v_color;}\n";
     return contents;
    }
   std::cout << "File " << filename << " not found\n";
   return NULL;
  }
 fseek(file, 0, SEEK_END);
 size_t filesize = ftell(file);
 rewind(file);
 contents.resize(filesize);
 fread(&contents[0], 1, filesize, file);
 fclose(file);
 return contents;
}

GLuint Shader::createShader(const char* vertexShaderFilename, const char* fragmentShaderFilename)
{
 std::string vertexShaderSource = parse(vertexShaderFilename);
 std::string fragmentShaderSource = parse(fragmentShaderFilename);

 GLuint program = glCreateProgram();
 GLuint vs = compile(vertexShaderSource, GL_VERTEX_SHADER);
 GLuint fs = compile(fragmentShaderSource, GL_FRAGMENT_SHADER);

 glAttachShader(program, vs);
 glAttachShader(program, fs);
 glLinkProgram(program);

 glDetachShader(program, vs);
 glDetachShader(program, fs);
 glDeleteShader(vs);
 glDeleteShader(fs);

 return program;
}

#ifdef _DEBUG
void openGLDebugCallback(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const void* userParam)
{
 if(severity==GL_DEBUG_SEVERITY_HIGH)
  {
   std::cout << "[OpenGL Error] " << message << std::endl;
  }
 else if(severity==GL_DEBUG_SEVERITY_MEDIUM)
  {
   std::cout << "[OpenGL Warning] " << message << std::endl;
  }
 else
  {
   //std::cout << "[OpenGL Message] " << message << std::endl;
  }
}
#endif
#endif

/************************ von xtekplot1.h uebernommen: *************************/
struct descr {int16 n,pad; const char *s;};

/* allgemeine Funktionen: */
void inital(double,double,double,double);
void q2plot(double,double,int);
void term_exit();
void term_refresh();
void term();
void getmaxsize(int*,int*,int*,int*); //vor inital() zu verwenden
void getmaxsize(int*,int*); //nur nach inital() verwendbar
void setsize(int,int,int);
void color(int); //Farbe als 0xBBGGRR
void rgbcolor(int r,int g,int b);
//void setcolor(int farbnr,int r,int g,int b);
//void getcolor(int farbnr,int *r,int *g,int *b);//nur zu testzwecken
//void qpunkt(float* x,float* y);
void tek_line(int n,double* xf,double* yf);
void tek_grein(double* x,double* y);
void qgrein(float* x,float* y);
void qgclear();
void q2esc(char* str);
void plotsave(const char* s);
void bildplot(double x,double y,int pen);
void bildopen(const char *name);
void bildclose();
void tek_pen(const char* x);
void iplotline(int x1,int y1,int x2,int y2,int farbe);
#define UP 3
#define DOWN 2
#define PENUP 3
#define PENDOWN 2
void ipunkt(int x,int y,int farbe);
void ipunkte(void* p,int n,int farbe);
void imausklick(int taste,int *x,int *y);
void mausklick(int taste,double *x,double *y);
int mausposition(double *x,double *y);
int imausposition(int *x,int *y);
int maustasten();
void Delay(int sec50tel);
#define LIMAUS		0x040 /* auf linke Maustaste warten */
#define REMAUS		0x400 /* auf rechte Maustaste warten */
#define MIMAUS		0x100 /* mittlere Maustaste */
#define SCROLL_UP      	0x800 /* Mausrad rauf */
#define SCROLL_DOWN    0x1000 /* Mausrad runter */
#define MAUSMASKE	0x540 /* provisorisch: bisher nur 2 unterstuetzt */
#define MAUSUP		0x001 /* auf Loslassen der Maustaste warten */
#define MAUSDOWN	0x002 /* auf Druecken der Maustaste warten */
#define MAUSUDMASKE	0x003
void iviereck(int x1,int y1,int x2,int y2,int farbe);
void drawmode(int mode);
#define JAM1 1
#define JAM2 2
#define COMPLEMENT 3
#define INVERSEVID 4
void setxsync(int mode);
void screenclear(int farbe=0);
void show_total();
void iplot(int,int,int);
void iline(int,int*,int*);
void idrawcircle(int,int,int,int);
void ifillcircle(int,int,int,int);
void idrawbox(int x1,int y1,int x2,int y2);
void ifillbox(int x1,int y1,int x2,int y2);
void drawcircle(double,double,double,double);
void fillcircle(double,double,double,double);
void drawbox(double x1,double y1,double x2,double y2);
void fillbox(double x1,double y1,double x2,double y2);
void moveto(int,int);
void lineto(int,int);
void waitTOF(),waitBOF();
void koordpix2user(int i,int j,double* x,double* y);
void qkoorduser2pix(double x,double y,int *i,int *j);
void koorduser2pix(double x,double y,int *i,int *j);
void deltakoorduser2pix(double x,double y,int *i,int *j);
void deltakoordpix2user(int i,int j,double *x,double *y);

//TODO: neue Vordeklarationen hier einfuegen
void get_xyminmax(double *x1,double *y1,double *x2,double *y2);
int get_menubalkenhoehe();
int get_menuleistenhoehe(); //das selbe wie menubalkenhoehe
int get_menuzeilenhoehe(); //Zeilenhoehe der aufgeklappten Menus
void check_glversion(int argc,char **argv);

/* xmenu.h */
#define MAXMENU 500
#define MAXMAINMENU 50
//typedef void (*funkzeiger)(long);
typedef void (*funkzeiger)(int);
typedef void (*funkzeig)(void);
void setmenu(int n,...);
void setsubmenu(int n,...);
int waitmenu(int waitflag);
void changemenu(long idnr,const char *neuertext);
//void menu_font_anfordern();
#define STATICGRAY      0
#define GRAYSCALE       1
#define STATICCOLOR     2
#define PSEUDOCOLOR     3
#define TRUECOLOR       4
#define DIRECTCOLOR     5
/*** neu ab TEKPLOT1.C ***/
void getsize(int*,int*,int*,int*);
void tek_setdebug(int);
int itextsize(int bx,int hy,const char *font=NULL,int style=0);
double textsize(double bx,double hy,const char *font=NULL,int style=0);
#define BOLD		1
#define ITALIC		2
#define UNDERLINE	4
void fillpolygon(int n,double *xf,double *yf);
void ifillpolygon(int n,int *xf,int *yf);
int requester_input(int,...);
double rundezahl(double,double);
void tek_punkt(double,double);
void tek_punkt(double,double,int farbnr);
int keyget(int *n,int *asci,uint32 *rawcode); /* noch nicht endgueltig */
/*
#define AUTOMENU (-1)
void automenu_loop();
*/
void tek_flush();
void setzergbtabelle(int n,uint8 neuetabelle[][3]);
double get_tekplot_version();
double getaspect();
double getraspect();
void setaspect(double);
void setmaxfarben(int max);

void ischrift(int x,int y,const char *,double winkel=0.);
void ischrift_utf8(int x,int y,const char *str,double winkel=0.);
void schrift(double x,double y,const char *,double winkel=0.);
void schrift_utf8(double x,double y,const char *,double winkel=0.);
void inital_new(double xmin=0.,double ymin=0.,double xmax=0.,double ymax=0.);
int janeinrequester(const char *text,const char *jatext=NULL,const char *neintext=NULL);
int janeinrequester_utf8(const char *text,const char *jatext=NULL,const char *neintext=NULL);
void getmenuids(long idnr,long *idfeld,int max=0);
void set_funktions(funkzeig pre,funkzeig rel=NULL,
			funkzeig exp=NULL,funkzeig mot=NULL);
#define MAX_DATEINAME_LAENGE 256
#define MAX_PFADNAME_LAENGE 1024
int nachfilenamefragen(const char *str,char *name,int max,int flgs=0,
	char *filter=NULL, //hier darf kein const sein!
	const char *oktext=NULL,const char *canceltext=" CANCEL",
	const char *parenttext=" Parent",const char *diskstext="  Disks",
	int nfilt=0);
void splitpfad(const char *name,char *pfad,char *file,int max,int max2=0);
char *joinpfad(const char *pfad,const char *file,char *name=NULL,int max=0);
void fullscreen_modus(int modus=1);
void borderless_modus(int modus=2);
int menu_switch(long id,long *ids,int k,int max,char **text,int k1=0,int k2=0);

void set_tektitel(const char*);
int changebuffer(int,int);
void bpencolor(int farbnr);
void rgbbpencolor(int r,int g,int b);
void utf8_to_isolatin1(char *ziel,const char *von,int max);

//TODO: Pixelbilder speichern/laden  (tek_xcreateimage() und tek_xputimage())
//typedef void* XImage;
//XImage *tek_xcreateimage(int xmax,int ymax);
//void tek_xputimage(XImage *image,int x0,int y0,int ix0,int iy0,int xmax,int ymax);

//bei set_menu_font() muss als user_dy die Pixelgroesse in User-Koordinaten angegeben werden
int set_menu_font(double user_dy,int bx,int hy,const char *fontname=NULL);

//bei set_requ_font() ist user_dy nicht noetig, da dies erst nach inital() benutzt werden darf
int set_requ_font(int bx,int hy,const char *fontname=NULL);

void set_redraw_all_function(funkzeig fun);

/* neue Funktionen hier einfuegen */

/* wahrscheinlich nicht mehr gebraucht: * /
#ifdef USE_INTERNAS
extern Drawable tekplot_rambild;
extern int tekplot_rambildflag,tekplot_refreshflag,tekplot_initalflag,
		tekplot_iffflag,tekplot_postflag;
extern char *tekplot_bildname, *tekplot_bildname_default;
#endif
/ * */

#define SPRACHE_DEUTSCH 1
#define SPRACHE_ENGLISH 2

/* Taste um Menu ein/aus zu schalten: ALT * /
// Codes noch von xtekplot:
#define MENUTASTE 0xFFE9
#define TABTASTE  0xFF09
#define SHIFTTASTE 0xFFE1
// Spezialtaste fuer schnelle Menu-Anwahl ueber Tastatur: ESC
#define AMIGATASTE 0xFF1B
*/
// neue Codes in OpenGL:
#define TABTASTE  SDLK_TAB

/************************ :von xtekplot1.h uebernommen *************************/

/********************************* Mesh-Klasse *********************************/

class Mesh {  //Variante fuer 2D-Grafik
private:
 VertexBuffer* vertexBuffer; //Puffer fuer alle Punkte
 IndexBuffer *indexBuffer; //Puffer zum Dreiecke zeichnen, jeweils 3 Indices pro Dreieck
 IndexBuffer *indexBufferLines; //zum Linien zeichnen, jeweils 2 Indices pro Linie
 IndexBuffer *indexBufferPoints; //zum einzelne Punkte zeichnen, 1 Index pro Punkt
 std::vector<Vertex> vertices; //alle Punkte
 std::vector<uint32> triangIndices; //Indexe auf vertices fuer Dreiecke
 std::vector<uint32> lineIndices; //jeweils 2 Punkte pro Linie
 std::vector<uint32> pointIndices; //Indexe auf vertices fuer einzelne Punkte
public:
 uint32 numVertices; //Anzahl aller Punkte
 uint32 numTriangIndices, numLineIndices, numPointIndices;

 Mesh() {
  vertexBuffer = NULL;
  indexBuffer = indexBufferLines = indexBufferPoints = NULL;
  numVertices = 0;
  numTriangIndices = numLineIndices = numPointIndices = 0;
 }
 
 ~Mesh() {delBuffers();}

 void addVertex(Vertex vertex) {vertices.push_back(vertex); numVertices++;}
 void addVertex(float x,float y,float z,uint32 col) {
  Vertex vertex(x,y,z,col); vertices.push_back(vertex); numVertices++;
 }
 void changeVertex(float x,float y,float z,uint32 col) {
  if(numVertices==0) {fprintf(stderr,"Error in changeVertex: no vertices\n"); return;}
  vertices[numVertices-1].setxyzcol(x,y,z,col);
 }
 void addTriangIndex(uint32 index) {triangIndices.push_back(index); numTriangIndices++;}
 void addLineIndex(uint32 index)  {lineIndices.push_back(index); numLineIndices++;}
 void addPointIndex(uint32 index) {pointIndices.push_back(index); numPointIndices++;}
 void addPointIndex() {pointIndices.push_back(numVertices); numPointIndices++;}
 void addLine() {
  if(numVertices==0) {fprintf(stderr,"Error in addLine: no vertices\n"); return;}
  addLineIndex(numVertices-1); //Startpunkt und
  addLineIndex(numVertices);   //Endpunkt der Linie
 }
 
 void delBuffers() {
  if(vertexBuffer!=NULL) delete vertexBuffer;
  if(indexBuffer!=NULL)  delete indexBuffer;
  if(indexBufferLines!=NULL) delete indexBufferLines;
  if(indexBufferPoints!=NULL) delete indexBufferPoints;
  vertexBuffer = NULL;
  indexBuffer = indexBufferLines = indexBufferPoints = NULL;
 }

 void clear() {
  for(uint32 i=0;i<numVertices;i++) vertices.pop_back();
  numVertices=0;
  for(uint32 i=0;i<numTriangIndices;i++) triangIndices.pop_back();
  for(uint32 i=0;i<numLineIndices;i++) lineIndices.pop_back();
  for(uint32 i=0;i<numPointIndices;i++) pointIndices.pop_back();
  numTriangIndices = numLineIndices = numPointIndices = 0;
  delBuffers();
 }
 void genbuffers();
 void render();
};

#ifdef TEKPLOT1_IMPLEMENTATION
void Mesh::genbuffers()
{
 delBuffers();
 if(numVertices!=0)
  {
   if(numTriangIndices==0) indexBuffer=NULL;
   else indexBuffer = new IndexBuffer(&triangIndices[0], numTriangIndices, sizeof(uint32));
   if(numLineIndices==0) indexBufferLines=NULL;
   else indexBufferLines = new IndexBuffer(&lineIndices[0], numLineIndices, sizeof(uint32));
   if(numPointIndices==0) indexBufferPoints=NULL;
   else indexBufferPoints= new IndexBuffer(&pointIndices[0], numPointIndices, sizeof(uint32));
   vertexBuffer = new VertexBuffer(&vertices[0], numVertices);
   vertexBuffer->unbind();
  }
}

void Mesh::render()
{
 if(numVertices>0)
  {
   vertexBuffer->bind();
   if(numTriangIndices>0)
    {
     indexBuffer->bind();
     glDrawElements(GL_TRIANGLES, numTriangIndices, GL_UNSIGNED_INT, 0);
     indexBuffer->unbind();
    }
   if(numLineIndices>0)
    {
     indexBufferLines->bind();
     glDrawElements(GL_LINES, numLineIndices, GL_UNSIGNED_INT, 0);
     indexBufferLines->unbind();
    }
   if(numPointIndices>0)
    {
     indexBufferPoints->bind();
     glDrawElements(GL_POINTS, numPointIndices, GL_UNSIGNED_INT, 0);
     indexBufferPoints->unbind();
    }
  }
}
#endif

/************* Klasse fuer Tekplot-Funktionen und globale Variablen ************/
#define BUTTON_PRESELECT 1

class Button {   //eine kleine Box mit kurzem Text
public:
 int x1,y1,x2,y2; //Box in Pixel
 char text[64];
 uint8 flags;
 
 Button() { x1 = y1 = 0; x2 = y2 = 40; flags = 0; text[0] = 0; }
 void set(const char *txt) {mystrncpy2(text,txt,64);}
 void setpos(int x,int y) {x2=x2-x1+x; y2=y2-y1+y; x1=x; y1=y;}
 void setsize(int x,int y) {x2=x1+x; y2=y1+y;}
};

bool ist_in_box(Button *bu, int mausx, int mausy);
#ifdef TEKPLOT1_IMPLEMENTATION
bool ist_in_box(Button *bu, int mausx, int mausy)
{
 if(bu==NULL) return false;
 return (mausx > bu->x1 && mausx < bu->x2 && mausy > bu->y1 && mausy < bu->y2);
}
#endif

/********** Schrift *************/
#include "fonts/myfonts.h"

//Default-Fonts hier setzen:
#ifdef TEKPLOT1_IMPLEMENTATION
static Font menu_font(myfont12x16, 12, 16, 1, 32); //Speicher Breite Hoehe Typ Bytesprozeichen
static Font requ_font(myfont12x16, 12, 16, 1, 32); //Speicher Breite Hoehe Typ Bytesprozeichen
Font *menufont= &menu_font;
Font *requfont= &requ_font;
//Font *aktfont= &default_font; //schon in fonts/myfonts.h gemacht
//TODO: Font-Skalierung abhaengig von Bildschirmaufloesung in inital() machen
#endif

/****** Globale Variablen: ******/
#ifdef TEKPLOT1_IMPLEMENTATION
//tekplot_version und tekplot_fenstername schon weiter oben definiert.
int tekplot_breite,tekplot_hoehe,tekplot_tiefe;
int32 tek_mausx=0,tek_mausy=0,maustasten_zustand=0;
//FILE *tekplot_fpbild=NULL;
//const int SCXRAND=6,SCYRAND=6;//provi.
const int SCXRAND=0, SCYRAND=0;//provi.

// fuer plotsave():
static FILE *fpbild=NULL;
static int qsaveflag=0; //plotsave eingeschaltet
//static int qsave1flag=0;//plotsave bei inital() einschalten?
static bool postflag=false;
#endif

struct NewGadget; //wird in nachfilenamefragen.h definiert

class Tekplot
{
 double xmin,ymin,xmax,ymax,xs,ys,dx,dy;//fuer User-Koordinaten
 double xscalOpengl, yscalOpengl; //fuer Umrechnung nach OpenGL-Koordinaten
 int breite,hoehe;//Fenstergroesse (Bereich in dem man zeichnen kann)
 int max_fensterbreite,max_fensterhoehe;//Maximalwerte, in inital() gesetzt
 int neue_menubalkenhoehe; //gesetzt falls menufont vor inital() gesetzt
 int menubalkenhoehe; //falls Menu vorhanden: Fenster entsprechend hoeher
 int menuzeilenhoehe; //Zeilenhoehe in aufgeklappten Menus
 //menuleistenhoehe ist identisch zu menubalkenhoehe
 int visualklasse,aktuellefarbe; //Farbe als 0xBBGGRR
 int hintergrundfarbe; //Farbe als 0xBBGGRR
 funkzeig redraw_all_func; //Zeiger auf eine Funktion ohne Parameter
 funkzeiger menu_funkfeld[MAXMENU]; //Zeiger auf Funktionen mit int als Parameter
 //funkzeiger menu_sub_funkfeld[MAXMENU];
 int menu_debug; //0=ohne debugging, 2=debugging, (1=reserviert fuer automenu)
 int menu_anzahl; //Anzahl Hauptmenus
 int menu_i; //Index auf naechstes Menu das mit setmenu() gesetzt werden kann
 //int menu_isub,menu_k,j1i_max; //Parameter fuer submenu() bisher nicht verwendet
 //int menu_istsub[MAXMENU],menu_sub_ifu[MAXMENU];
 int menu_aufgeklappt; //Nummer des aufgeklappten Hauptmenus mit 1 beginnend
 int menu_ix0, menu_iy0; //Startwerte zum Menubalken beschriften
 char *menu_textfeld[MAXMENU];
 char statustext[400];
 int greinflag,greinx,greiny;//fuer tek_grein()
 
 int fadenflag,fadenx,fadeny,fadencolor;//fuer Fadenkreuz
 int fadenmodus; //0=aus, 1=einzelnes Fadenkreuz, 2=doppeltes Fadenkreuz, 3=Integral-Linie,
                 //4=Linienzug, 5=Linienzug+Kreuz, ...//TODO
 int faden_ix[10],faden_iy[10],faden_npkt;//TODO
 int xzoom_flg,yzoom_flg; //fuer Spezielle Zoom-Modi
#define FADEN_MAXMODUS 2 //TODO

 void menu_reset();
 void menu_zeichnen();
 int menu_aufklappbox[4]; //Eckpunkte in Pixel von aufgeklapptem Rechteck
 int menu_mainboxlist[MAXMAINMENU+1];
 void menu_klapper(int mausx,int mausy); //wenn in Menuleiste geklickt
 void menu_klapper_mark(int mausx,int mausy); //wenn Maus im Menu bewegt
 bool maus_im_menu(int x,int y); //true wenn Position im aufgeklappten Menu
 void redraw_all(); //alles neu zeichnen
 int menuvorhanden();
 bool fadenkreuzvorhanden() {return (fadenflag==1);}
 void neuefenstergroesse();
 long testlong;//test seltsamer Fehler in mouse_press wenn dies nicht gesetzt!
 funkzeig mouse_press, mouse_release, win_expose, mouse_motion;
 SDL_Window* window;
 //Camera camera; //fuer 2D-Grafik nicht gebraucht?
 int screenmodus; //1=Vollbildmodus, 2=Borderless (ohne Vollbild)
 bool buttonEnter; //in Requester gebraucht
 int oldpen; //wird gesetzt wenn q2plot() benutzt
 
 int lastDrawFunction;
#define DF_PUNKT 1
#define DF_LINE  2
#define DF_TRIANG 3
 
 Mesh userMesh;
 Mesh menuMesh;
public:
 void get_xyminmax(double *x1,double *y1,double *x2,double *y2) {*x1=xmin; *y1=ymin; *x2=xmax; *y2=ymax;}
 int get_menubalkenhoehe() {return menubalkenhoehe;}
 int get_menuzeilenhoehe() {return menuzeilenhoehe;}
 bool requester_aktiv;
 SDL_Event requester_event; //Ereignisse fuer aktiven Requester
 Mesh requesterMesh;
 Mesh *aktMesh;
 void set_requesterVertices();
 void unset_requesterVertices();
 int da1=1,da2=2; //kleiner Abstand und doppelter Wert, abhaengig von Bildschirmaufloesung

 //#define ZWERT_MIN (-1.0f)
#define ZWERT_MIN (-0.99998f)
#define ZWERT_MAX 0.99998f
#define ZWERT_DELTA (1.0f/(1<<16)) //reicht fuer 65536 Zeichenbefehle
//#define ZWERT_DELTA (1.0f/(1<<24)) //wenn mehr benoetigt (ca. 16 Mio, nicht auf allen Grafikkarten?)
#define MENU_ZWERT_START (ZWERT_MIN+(20000*ZWERT_DELTA))
#define FADEN_ZWERT_START (ZWERT_MIN+(20500*ZWERT_DELTA))
#define REQUESTER_ZWERT_START (ZWERT_MIN+(10000*ZWERT_DELTA))
 void zwert_overflow(const char *text);
 float32 zwert; //muss zwischen -1.0 und +1.0 liegen (-1.0=vorderste Ebene, +1.0=hinterste)
 void zwert_decrement() {
  zwert -= ZWERT_DELTA;
  if(zwert<ZWERT_MIN) {fprintf(stderr,"Error: zwert underflow %f\n",zwert); zwert+=ZWERT_DELTA;}
 }

 int zflag=0; //Verhalten, wann bei Zeichenoperation der zwert erniedrigt werden soll
#define ZFLAG_AUTO 0    //bei Wechsel zwischen Punkt-, Linie-, Dreieck-zeichnen
#define ZFLAG_IMMER 1   //bei jeder neuen Zeichenoperation
#define ZFLAG_NIEMALS 2 //nie, wenn noetig direkt mit zwert -= ZWERT_DELTA machen
#define ZFLAG_ALWAYS 1
#define ZFLAG_NEVER 2

 float32 menu_mark_zwert;
 double get_dx() {return dx;}
 double get_dy() {return dy;}
 float32 get_zwert() {return zwert;}
 void set_zwert(float32 z) {zwert=z;}
 int initalflag; //sollte zwischen inital() und term_refresh() gesetzt sein
 int exitflag; //0 oder 1
 int scxrand, /* 2*Screenrandbreite */
     scyrand; /* Geraeteabhaengig ! (nicht verwechseln mit border) */
 Tekplot();
 int inital(double xmi,double ymi,double xma,double yma); //Ruckgabewert 0 wenn ok
 void inital_new(double xmi,double ymi,double xma,double yma);
 void term_refresh();
 void term_exit();
 //q2plot() statt plot() verwedet um Verwechslungen mit Systemfunktionen zu vermeiden
 void q2plot(double x,double y,int pen);
 void koordpix2user(int i,int j,double* x,double* y);
 void koordpix2userf(int i,int j,float32* x,float32* y);
 void qkoorduser2pix(double x,double y,int *i,int *j);
 void koorduser2pix(double x,double y,int *i,int *j);
 void koorduser2mauspix(double x1,double y1,double x2,double y2,int *box);
 void deltakoorduser2pix(double x,double y,int *i,int *j);
 void deltakoordpix2user(int i,int j,double *x,double *y);
 void koorduser2opengl(double x,double y,float32* xf,float32* yf);
 int waitmenu(int waitflag);
 void bildrefresh() {term_refresh();}
 void rgbcolor(int r,int g,int b);
 void color(int farbe);
 void tek_grein(double*px,double*py);
 void tek_fadenkreuz(int modus);
 void tek_faden_read(double*px,double*py);
 //void tek_faden_setmodus(int m);
 void tek_faden_setflags(int xzoom,int yzoom) {xzoom_flg=xzoom; yzoom_flg=yzoom;}
 void tek_faden_setxy(int x,int y) {faden_ix[faden_npkt]=x; faden_iy[faden_npkt++]=y;}
 void tek_faden_setxy2(int x,int y,int x2,int y2)
  {faden_ix[faden_npkt]=x; faden_iy[faden_npkt++]=y; faden_ix[faden_npkt]=x2; faden_iy[faden_npkt++]=y2;}
 void tek_faden_setpoint1(int x,int y) {faden_ix[0]=x; faden_iy[0]=y; faden_npkt=1;}
 void tek_faden_clearpoints() {faden_npkt=0;}
 int janeinrequester(const char *text,const char *jatext,const char *neintext);
 void getsize(int*br,int*ho,int*ti,int*v);
 void setsize(int br,int ho,int ti);
 void setmenu(int n,...);
 void setsubmenu(int n,...);
 void menu_aufruf(int id);
 void ipunkt(int ix,int iy);
 void punkt(double x,double y);
 void mausklickEvent(int ix,int iy);
 void mausmoveEvent(int ix,int iy);
 void fillbox(double x1,double y1,double x2,double y2);
 void ifillbox(int ix1,int iy1,int ix2,int iy2);
 void idrawbox(int ix1,int iy1,int ix2,int iy2);
 void drawbox(double x1,double y1,double x2,double y2);
 void screenclear(int color);
 void set_funktions(funkzeig pre,funkzeig rel,funkzeig exp,funkzeig mot);
 int set_menu_font(double user_dy,int bx,int hy,const char *fontname);
 int set_requ_font(int bx,int hy,const char *fontname);
 void menu_umrahmen(int ix1,int iy1,int ix2,int iy2);
 void set_screenmodus(int modus);
 void set_redraw_all_function(funkzeig fun) {redraw_all_func=fun;}
 void set_uservertices();
 void set_menuvertices(bool resetflag);
 int messageBox(const char *titel, const char *message, int width, int height,
		Button* button1, Button* button2, Button* button3);
 void fadenkreuz_zeichnen();
 void getmenuids(long idnr,long *idfeld,int max);
 void changemenu(long idnr,const char *neuertext);
 void set_statustext(const char *s) {strncpy(statustext,s,400); statustext[400-1]=0; menu_zeichnen();}
};

#ifdef TEKPLOT1_IMPLEMENTATION
static Tekplot haupt;//Haupt-Tekplotfenster

Tekplot::Tekplot()
{
 dx=dy=0; //kann erst in inital() richtig gesetzt werden
 initalflag=exitflag=0;
 tekplot_breite=breite=1220; tekplot_hoehe=hoehe=680;//provi.
 tekplot_tiefe=24; //auf neueren Computern immer 24
 greinflag=fadenflag=fadenmodus=0; fadenx=fadeny=0; fadencolor=0x7F7F7F;//TODO
 faden_npkt=0;
 for(int i=0;i<10;i++) {faden_ix[i]=faden_iy[i]=0;}
 xzoom_flg=yzoom_flg=0; //Spezielle Zoom-Modi aus
 tek_mausx=tek_mausy=0;
 maustasten_zustand=0;
 neue_menubalkenhoehe=menubalkenhoehe=0;
 menu_ix0=10; menu_iy0= -10; //Startwerte fuer 12x16-Font
 scxrand=SCXRAND; scyrand=SCYRAND;
 visualklasse=TRUECOLOR; //??
 hintergrundfarbe=0; //schwarzer Hintergrund als default
 //aktuellefarbe=0; //Schwarz als Zeichenfarbe bei weissem Hintergrund
 aktuellefarbe=0xFFFFFF; //Weiss als Zeichenfarbe bei schwarzem Hintergrund
 buttonEnter=false;
 
 menu_anzahl = menu_i = menu_debug = 0;
 //menu_isub = menu_k = 0;
 for(int i=0;i<MAXMENU;i++)
  {
   menu_funkfeld[i]=NULL;
   menu_textfeld[i]=NULL;
  }
 menu_aufgeklappt=0;
 for(int i=0;i<4;i++) menu_aufklappbox[i]=0;
 menu_mark_zwert = MENU_ZWERT_START; //Menu auf fast vorderster Ebene
 statustext[0]=0; //leerer Statustext

 aktMesh = &userMesh;
 oldpen=0;
 zwert=0;
 lastDrawFunction=0;
 screenmodus=0;
 mouse_press=NULL; mouse_release=NULL; win_expose=NULL; mouse_motion=NULL;
 redraw_all_func=NULL;
 requester_aktiv=false;
}

void Tekplot::set_uservertices()
{
 aktMesh = &userMesh;
}

void Tekplot::set_menuvertices(bool resetflag)
{
 aktMesh = &menuMesh;
 if(resetflag)
  {
   menuMesh.clear();
  }
}

void Tekplot::set_requesterVertices()
{
 aktMesh = &requesterMesh;
 requesterMesh.clear();
}

void Tekplot::unset_requesterVertices()
{
 aktMesh = &userMesh;
}

void Tekplot::set_screenmodus(int modus)
{
 screenmodus=modus;
 if(screenmodus&1)
  {
   int ti,vi;
   getmaxsize(&breite,&hoehe,&ti,&vi);
   tekplot_breite = breite;
   tekplot_hoehe = hoehe;
  }
}

void Tekplot::screenclear(int color)
{
 hintergrundfarbe=color;
 userMesh.clear();
 oldpen=0;
 lastDrawFunction=0;
 zwert = ZWERT_MAX;
 if(menuvorhanden()) menu_zeichnen();
}

//Schnittstelle fuer C-Aufrufe:
//void term() {haupt.term_refresh();} //veraltet
void term_refresh() {haupt.term_refresh();}
void term_exit() {haupt.term_exit();}
void inital(double xmi,double ymi,double xma,double yma)
  {
   int err=haupt.inital(xmi,ymi,xma,yma);
   if(err) printf("Error in inital(): %d\n",err);
  }
void inital_new(double xmi,double ymi,double xma,double yma)
  {haupt.inital_new(xmi,ymi,xma,yma);}
void screenclear(int color) {haupt.screenclear(color);}
int waitmenu(int waitflag) {return haupt.waitmenu(waitflag);}
void q2plot(double x,double y,int pen) {haupt.q2plot(x,y,pen);}
void ipunkt(int ix,int iy) {haupt.ipunkt(ix,iy);}
//void punkt(double x,double y) {haupt.punkt(x,y);}
void tek_punkt(double x,double y) {haupt.punkt(x,y);}
void tek_punkt(double x,double y,int farbnr) {
 haupt.rgbcolor(farbnr&0xFF,(farbnr>>8)&0xFF,(farbnr>>24)&0xFF);
 haupt.punkt(x,y);
}
void fillbox(double x1,double y1,double x2,double y2) {haupt.fillbox(x1,y1,x2,y2);}
void ifillbox(int x1,int y1,int x2,int y2) {haupt.ifillbox(x1,y1,x2,y2);}
void drawbox(double x1,double y1,double x2,double y2) {haupt.drawbox(x1,y1,x2,y2);}
void idrawbox(int x1,int y1,int x2,int y2) {haupt.idrawbox(x1,y1,x2,y2);}
void koordpix2user(int i,int j,double *x,double *y) {haupt.koordpix2user(i,j,x,y);}
void deltakoordpix2user(int i,int j,double *x,double *y)
  {haupt.deltakoordpix2user(i,j,x,y);}
void deltakoorduser2pix(double x,double y,int *i,int *j)
  {haupt.deltakoorduser2pix(x,y,i,j);}
void set_funktions(funkzeig pre,funkzeig rel,funkzeig exp,funkzeig mot)
  {haupt.set_funktions(pre,rel,exp,mot);}
void fullscreen_modus(int modus) {haupt.set_screenmodus(modus);}
void borderless_modus(int modus) {haupt.set_screenmodus(modus);}
void set_redraw_all_function(funkzeig fun) {haupt.set_redraw_all_function(fun);}
void zwert_decrement() {haupt.zwert_decrement();}
void get_xyminmax(double *x1,double *y1,double *x2,double *y2) {haupt.get_xyminmax(x1,y1,x2,y2);}
int get_menubalkenhoehe() {return haupt.get_menubalkenhoehe();}
int get_menuleistenhoehe() {return haupt.get_menubalkenhoehe();}
int get_menuzeilenhoehe() {return haupt.get_menuzeilenhoehe();}
void getmenuids(long idnr,long *idfeld,int max) {haupt.getmenuids(idnr,idfeld,max);}
void changemenu(long idnr,const char *neuertext) {haupt.changemenu(idnr,neuertext);}
void set_statustext(const char *s) {haupt.set_statustext(s);}

//TODO: neue Funktionen fuer C-Schnittstelle einfuegen

#ifdef _WIN32
#include <chrono>
#include <thread>
void usleep(int n)
{
 //Wartefunktion auch unter Windows
 std::this_thread::sleep_for(std::chrono::milliseconds(n/1000));
}
#else
#include <unistd.h>
#endif

void Delay(int sec50tel)
{
 while(sec50tel>0) {usleep(20000); --sec50tel;}
}

bool fastgleich(double a,double b)
{
 const double d=1e-10;
 return (a>b-d && a<b+d);
}

#ifdef _DEBUG
#define DEBUG(n,x) x
#else
#define DEBUG(n,x) 
#endif

//Koordinatenumrechnungen wie in xtekplot1:
void Tekplot::koordpix2user(int i,int j,double* x,double* y)
{
 *x = xmin + i/xs;
 // *y = ymin + (hoehe-1-j+menubalkenhoehe)/ys;
 // da hoehe in Wirklichkeit (hoehe+menubalkenhoehe) ist, faellt die Korrektur um menubalkenhoehe heraus.
 *y = ymin + (hoehe-1-j)/ys;
}
void Tekplot::koordpix2userf(int i,int j,float32* x,float32* y)
{
 *x = (float32)(xmin + i/xs);
 *y = (float32)(ymin + (hoehe-1-j)/ys);
}
void Tekplot::qkoorduser2pix(double x,double y,int *i,int *j)
{       //quick = ohne Ueberpruefung der Grenzen
 x=(x-xmin)*xs;
 y=(y-ymin)*ys;
 *i=(int)(x+0.5);
 *j = hoehe - 1 - (int)(y+0.5);
}
void Tekplot::koorduser2pix(double x,double y,int *i,int *j)
{       //mit Ueberpruefung der Grenzen
 int z;
 x=(x-xmin)*xs;
 y=(y-ymin)*ys;
 if(x>0.0) {*i=(int)(x+0.5); if(*i>=breite) *i=breite-1;}
 else	   {*i=0;}
 if(y>0.0) {z=(int)(y+0.5); if(z>=hoehe) z=hoehe-1;}
 else	   {z=0;}
 *j = hoehe-1-z;
}

void Tekplot::koorduser2mauspix(double x1,double y1,double x2,double y2,int *box)
{
 //fuer mauspixel geht y-Koordinate von oben nach unten
 int fensterhoehe = tekplot_hoehe+menubalkenhoehe;
 box[0] = (int)((x1-xmin)*xs+0.5);
 box[1] = fensterhoehe - (int)((y1-ymin)*ys+0.5);
 box[2] = (int)((x2-xmin)*xs+0.5);
 box[3] = fensterhoehe - (int)((y2-ymin)*ys+0.5);
}

void Tekplot::deltakoorduser2pix(double x,double y,int *i,int *j)
{
 if((x*=xs)>0.) *i=(int)(x+0.5);
 else		*i=(int)(x-0.5);
 if((y*=ys)>0.) *j=(int)(y+0.5);
 else		*j=(int)(y-0.5);
}
void Tekplot::deltakoordpix2user(int i,int j,double *x,double *y)
{
 *x=i/xs;
 *y=j/ys;
}

void Tekplot::q2plot(double x,double y,int pen)
{
 float32 xf,yf;
 koorduser2opengl(x, y, &xf, &yf); //Skalierung auf OpenGL-Koordinaten
 if(qsaveflag && (pen==PENUP || pen==PENDOWN))
  {
   bildplot(x,y,pen);
  }
 if(pen==PENUP)
  {
   if(oldpen==PENUP)
    {
     printf("Warning: twice PENUP, ignoring first\n");
     aktMesh->changeVertex(xf,yf,zwert,aktuellefarbe);
    }
   else
    {
     if(zflag==ZFLAG_ALWAYS || (zflag!=ZFLAG_NEVER && lastDrawFunction != DF_LINE))
      {
       if(zwert > ZWERT_MIN+ZWERT_DELTA) zwert -= ZWERT_DELTA;
       else zwert_overflow("q2plot()");
      }
     //printf("q2plot: zwert=%f\n",zwert);//test
     lastDrawFunction = DF_LINE;
     aktMesh->addVertex(xf,yf,zwert,aktuellefarbe);
    }
  }
 else //if(pen==PENDOWN)
  {
   if(oldpen==0)
    {
     printf("Error in plot(): missing PENUP\n");
    }
   else
    {
     aktMesh->addLine();
    }
   aktMesh->addVertex(xf,yf,zwert,aktuellefarbe);
  }
 oldpen=pen;
}

void Tekplot::ipunkt(int ix,int iy)
{
 double xf,yf;
 koordpix2user(ix,iy,&xf,&yf);
 punkt(xf,yf);
}

void Tekplot::punkt(double x,double y)
{
 oldpen=0;
 //erste Punkte verdecken eventuell spaeter gezeichnete!
 if(zflag==ZFLAG_ALWAYS || (zflag!=ZFLAG_NEVER && lastDrawFunction!=DF_PUNKT))
  {
   if(zwert > ZWERT_MIN+ZWERT_DELTA) zwert -= ZWERT_DELTA;
   else zwert_overflow("punkt()");
   //printf("punkt: zwert=%f\n",zwert);//test
  }
 lastDrawFunction = DF_PUNKT;
 float32 xf,yf;
 koorduser2opengl(x, y, &xf, &yf); //Skalierung auf OpenGL-Koordinaten
 aktMesh->addPointIndex(); //Index auf den Punkt im Mesh anfuegen
 aktMesh->addVertex(xf, yf, zwert, aktuellefarbe); //und Punkt selbst anfuegen
}

void Tekplot::koorduser2opengl(double x, double y, float32* xf, float32* yf)
{
 *xf = (x-xmin)*xscalOpengl - 1.0;
 *yf = (y-ymin)*yscalOpengl - 1.0;
}

void Tekplot::idrawbox(int ix1,int iy1,int ix2,int iy2)
{
 double x1,y1,x2,y2;
 koordpix2user(ix1,iy1,&x1,&y1);
 koordpix2user(ix2,iy2,&x2,&y2);
 drawbox(x1,y1,x2,y2);
}

void Tekplot::ifillbox(int ix1,int iy1,int ix2,int iy2)
{
 double x1,y1,x2,y2;
 koordpix2user(ix1,iy1,&x1,&y1);
 koordpix2user(ix2,iy2,&x2,&y2);
 fillbox(x1,y1,x2,y2);
}

void Tekplot::zwert_overflow(const char *text)
{
 fprintf(stderr,"Error: %s zwert overflow %f\n",text,zwert);
 zwert=0;//test
}

void Tekplot::fillbox(double x1,double y1,double x2,double y2)
{
 float32 x1f, y1f, x2f, y2f;
 oldpen=0;
 //erste Flaechen verdecken eventuell spaeter gezeichnete!
 //if(zflag==ZFLAG_ALWAYS || (zflag!=ZFLAG_NEVER && lastDrawFunction != DF_TRIANG))
 if(zflag!=ZFLAG_NEVER)
  {
   if(zwert > ZWERT_MIN+ZWERT_DELTA) zwert -= ZWERT_DELTA;
   else zwert_overflow("fillbox()");
   //printf("fillbox: zwert = %f\n",zwert);//test
  }
 lastDrawFunction = DF_TRIANG;
 koorduser2opengl(x1, y1, &x1f, &y1f); //Skalierung auf OpenGL-Koordinaten
 koorduser2opengl(x2, y2, &x2f, &y2f); //Skalierung auf OpenGL-Koordinaten
 uint32 i=aktMesh->numVertices;
 aktMesh->addVertex(x1f, y1f, zwert, aktuellefarbe); 
 aktMesh->addVertex(x1f, y2f, zwert, aktuellefarbe); 
 aktMesh->addVertex(x2f, y2f, zwert, aktuellefarbe); 
 aktMesh->addVertex(x2f, y1f, zwert, aktuellefarbe);
 aktMesh->addTriangIndex(i); //erstes Dreieck
 aktMesh->addTriangIndex(i+1);
 aktMesh->addTriangIndex(i+2);
 aktMesh->addTriangIndex(i+2); //zweites Dreieck
 aktMesh->addTriangIndex(i+3);
 aktMesh->addTriangIndex(i);
}

void Tekplot::drawbox(double x1,double y1,double x2,double y2)
{
 float32 x1f, y1f, x2f, y2f;
 oldpen=0;
 //erste Linien verdecken eventuell spaeter gezeichnete!
 if(zflag==ZFLAG_ALWAYS || (zflag!=ZFLAG_NEVER && lastDrawFunction != DF_LINE))
  {
   if(zwert > ZWERT_MIN+ZWERT_DELTA) zwert -= ZWERT_DELTA;
   else zwert_overflow("drawbox()");
   //printf("drawbox: zwert=%f\n",zwert);//test
  }
 lastDrawFunction = DF_LINE;
 koorduser2opengl(x1, y1, &x1f, &y1f); //Skalierung auf OpenGL-Koordinaten
 koorduser2opengl(x2, y2, &x2f, &y2f); //Skalierung auf OpenGL-Koordinaten
 uint32 i=aktMesh->numVertices;
 aktMesh->addVertex(x1f, y1f, zwert, aktuellefarbe); 
 aktMesh->addVertex(x1f, y2f, zwert, aktuellefarbe); 
 aktMesh->addVertex(x2f, y2f, zwert, aktuellefarbe); 
 aktMesh->addVertex(x2f, y1f, zwert, aktuellefarbe); 
 aktMesh->addLineIndex(i); aktMesh->addLineIndex(i+1); //erste Linie des Vierecks
 aktMesh->addLineIndex(i+1); aktMesh->addLineIndex(i+2); //2. Linie des Vierecks
 aktMesh->addLineIndex(i+2); aktMesh->addLineIndex(i+3); //3. Linie des Vierecks
 aktMesh->addLineIndex(i+3); aktMesh->addLineIndex(i); //4. Linie des Vierecks
}

#ifndef CHECK_GL_VERSION
void check_glversion(int argc,char **argv) {}
#endif

#ifdef CHECK_GL_VERSION

const char *tmpname =  "glversion.tmp";
static int glversion=0;

int getGLversion() //GL_VERSION ermitteln
{
 return glversion;
}

void check_glversion(int argc,char **argv)
{
 //printf("check_glversion(argc=%d,...)\n",argc);//test
 //printf(" argv[0]=%s\n",argv[0]);//test
 //if(argc>=2) printf(" argv[1]=%s\n",argv[1]);//test
 if(argc==2 && strcmp(argv[1],"GLVERSION")==0)
  {
   int er=SDL_VideoInit(NULL);
   if(er) {printf("Error initializing SDL: %s\n", SDL_GetError()); exit(1);}
   SDL_Window* window=SDL_CreateWindow("Fenster0", 50, 50, 100, 100, SDL_WINDOW_OPENGL);
   SDL_GLContext context = SDL_GL_CreateContext(window);
   const char *gl_vers = (const char*)glGetString(GL_VERSION);
   if(gl_vers==NULL) {printf("Error glGetString(GL_VERSION) -> NULL\n"); exit(1);}
   float gl_version; sscanf(gl_vers,"%f",&gl_version);
   SDL_GL_DeleteContext(context);
   SDL_DestroyWindow(window);
   SDL_VideoQuit();
   SDL_Quit();
   int glvers = (int)(gl_version*100+0.0001);
   FILE *fp=myfopen(tmpname,"w");
   fprintf(fp,"%d\n",glvers);
   fclose(fp);
   exit(0);
  }
 else
  {
   char str[400];
   sprintf(str,"%s GLVERSION",argv[0]);
   system(str);
   FILE *fp=myfopen(tmpname,"r");
   if(fp==NULL) {printf("Error opening %s\n",tmpname); return;}
   char tmp[8]; int i,c;
   for(i=0; i<8-1 && (c=getc(fp))!=EOF && c>=' '; i++)
     {tmp[i] = c;}
   tmp[i]=0;
   fclose(fp);
   sscanf(tmp,"%d",&glversion);
   remove(tmpname);
  }
}

/*
int getGLversion() //GL_VERSION ermitteln
{
 int glvers=0;
 const char *tmpname =  "glversion.tmp";
 system("./getGLversion >glversion.tmp"); //test
 //system("getGLversion >glversion.tmp"); //Unschoen, aber bisher
 FILE *fp=myfopen(tmpname,"r");           //keine andere Loesung gefunden.
 if(fp==NULL) {printf("Error opening %s\n",tmpname); return 0;}
 char tmp[8]; int i,c;
 for(i=0; i<8-1 && (c=getc(fp))!=EOF && c>=' '; i++)
   {tmp[i] = c;}
 tmp[i]=0;
 fclose(fp);
 sscanf(tmp,"%d",&glvers);
 remove(tmpname);
 return glvers;
}
*/

/*
int getGLversion() //test: GL_VERSION ermitteln
{
 // geht nicht:
 //int er=SDL_Init(SDL_INIT_EVERYTHING);
 int er=SDL_VideoInit(NULL);
 if(er) {printf("Error initializing SDL: %s\n", SDL_GetError()); exit(1);}
 //uint32 flags = SDL_WINDOW_OPENGL | SDL_WINDOW_BORDERLESS;
 static SDL_Window* window=SDL_CreateWindow("Fenster0", 50, 50, 100, 100, SDL_WINDOW_OPENGL);
 static SDL_GLContext context = SDL_GL_CreateContext(window);
 const char *gl_vers = (const char*)glGetString(GL_VERSION);
 if(gl_vers==NULL) {printf("Error glGetString(GL_VERSION) -> NULL\n"); exit(1);}
 float gl_version; sscanf(gl_vers,"%f",&gl_version);
 SDL_GL_DeleteContext(context);
 SDL_DestroyWindow(window);
 SDL_VideoQuit();
 SDL_Quit();
 int version = (int)(gl_version*100+0.0001);
 printf("GL version = %d\n",version);//test
 return version;
 // version wird zwar richtig gelesen, aber das MESA_OVERRIDE geht danach nicht mehr.
}
*/

#endif //CHECK_GL_VERSION

void sdl_init()
{
#ifdef CHECK_GL_VERSION
 static int startflag=1;
 if(startflag)
  {
   int glvers;
#ifdef MESA_GL_VERSION_OVERRIDE
   glvers=0;
#else
   glvers=getGLversion();
   printf("glvers=%d\n",glvers);//test
   //printf("weiter mit Enter "); getc(stdin); printf("\n");//test
#endif
   if(glvers<330)
    {
     SDL_setenv("MESA_GL_VERSION_OVERRIDE", "3.3", 1);
     SDL_setenv("MESA_GLSL_VERSION_OVERRIDE", "330", 1);
     printf("MESA_GL_VERSION_OVERRIDE auf \"3.3\" gesetzt.\n");//test
    }
   startflag=0;
  }
#endif
 int er=SDL_Init(SDL_INIT_EVERYTHING);
 if(er) printf("Error initializing SDL: %s\n", SDL_GetError());
 SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
 SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
 SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
 SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE, 8);
 SDL_GL_SetAttribute(SDL_GL_BUFFER_SIZE, 32);
 er=SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
 if(er!=0) printf("SDL_GL_SetAttribute()--> \"%s\"\n",SDL_GetError());
}

int Tekplot::inital(double xmi,double ymi,double xma,double yma)
{
 //DEBUG(1,printf("inital(%f, %f, %f, %f)\n",xmi,ymi,xma,yma));//test
 xmin=xmi; ymin=ymi; xmax=xma; ymax=yma;
 if(menuvorhanden())
  {// Menuleistenhoehe definieren:
   menuzeilenhoehe = (neue_menubalkenhoehe>0) ? neue_menubalkenhoehe : 34;
   menubalkenhoehe = (menuzeilenhoehe>26) ? menuzeilenhoehe : 26;
   int bux=menufont->get_ischrift_breite(), buy=menufont->get_ischrift_hoehe();
   menu_ix0 = bux/2+4;
   menu_iy0 = (menuzeilenhoehe-buy)/2 + buy - menuzeilenhoehe;
   //printf("menu_ix0=%d _iy0=%d\n",menu_ix0,menu_iy0);//test
  }
 if(initalflag==0)
   {initalflag=1;}
 else
   {printf("Warnung in inital(): initalflag schon gesetzt.\n");}
 sdl_init();
 getmaxsize(&max_fensterbreite,&max_fensterhoehe);
#ifdef _DEBUG
 printf("max_fenster: %d %d\n",max_fensterbreite,max_fensterhoehe);//test
#endif
 da1=1; //kleiner Abstand, abhaengig von Bildschirmaufloesung
 if(max_fensterbreite>1920) da1=3;
 else if(max_fensterbreite>640) da1=2;
 da2 = 2*da1;
 
 uint32 flags = SDL_WINDOW_OPENGL;
 int fensterbreite = tekplot_breite;
 int fensterhoehe = tekplot_hoehe+menubalkenhoehe;
 if(screenmodus&1)
  {
   //bei Borderless oder Fullscreen mit Alt Q oder Alt F4 stoppen
   flags |= SDL_WINDOW_FULLSCREEN_DESKTOP;
   getmaxsize(&fensterbreite,&fensterhoehe);
   tekplot_breite = breite = fensterbreite;
   tekplot_hoehe = hoehe = fensterhoehe-menubalkenhoehe;
  }
 if(screenmodus&2)
  {
   flags |= SDL_WINDOW_BORDERLESS;
  }

 //SDL_Window* window; //in der Klasse definiert
 window = SDL_CreateWindow(tekplot_fenstername, SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED,
			   fensterbreite, fensterhoehe, flags);
 //printf("window = 0x%lX\n",(long)window);//test
 SDL_GLContext glContext = SDL_GL_CreateContext(window);
 //printf("glContext = 0x%lX\n",(long)glContext);//test
 std::cout << "OpenGL version: " << glGetString(GL_VERSION) << std::endl;
 
 GLenum err = glewInit();
 if(err != GLEW_OK)
  {
   std::cout << "glewInit Error: " << glewGetErrorString(err) << std::endl;
   return -1;
  }

 err=0;
 if(SDL_GL_GetSwapInterval()!=1) //falls nicht schon gesetzt:
   err=SDL_GL_SetSwapInterval(1); //setze Wartemethode auf VSYNC
#ifdef _DEBUG
 if(err!=0) printf("SetSwapInterval()--> \"%s\"\n",SDL_GetError());//test
#endif
 
 glEnable(GL_DEPTH_TEST);
 //glEnable(GL_CULL_FACE);//nur vordere Seite von Dreiecken zeichnen
 //glFrontFace(GL_CW);//falsch herum definierte Dreiecke (normal GL_CCW)

#ifdef _DEBUG
 SDL_GL_SetAttribute(SDL_GL_CONTEXT_FLAGS, SDL_GL_CONTEXT_DEBUG_FLAG);
 glEnable(GL_DEBUG_OUTPUT);
 glEnable(GL_DEBUG_OUTPUT_SYNCHRONOUS);
 glDebugMessageCallback(openGLDebugCallback, 0);
#endif

#ifdef SHADERS_OLD
 Shader shader("shaders/basic120.vs", "shaders/basic120.fs");
#else
 Shader shader("shaders/basic.vs", "shaders/basic.fs");
#endif
 shader.bind();

 neuefenstergroesse(); //Skalierfaktoren an aktuelle Fenstergroesse anpassen
 menufont->init(dx,dy);
 requfont->init(dx,dy);
 aktfont->init(dx,dy);
 if(menuvorhanden()) menu_zeichnen(); //auch in screenclear() machen
 zwert = ZWERT_MAX;
 return 0;
}

void Tekplot::inital_new(double xmi,double ymi,double xma,double yma)
{
 //DEBUG(1,printf("inital_new()\n"));//test
 if(!fastgleich(xmi,xma) && !fastgleich(ymi,yma))
    {xmin=xmi; ymin=ymi; xmax=xma; ymax=yma;}
 initalflag=1;
 neuefenstergroesse(); //Skalierfaktoren anpassen
 menufont->init(dx,dy); //Fonts mit neuen Skalierfaktoren
 requfont->init(dx,dy);
 aktfont->init(dx,dy);
}

void Tekplot::fadenkreuz_zeichnen()
{
 set_requesterVertices();
 zwert = FADEN_ZWERT_START; //auf fast vorderster Ebene zeichnen
 int zflag_vorher=zflag;
 zflag=ZFLAG_NEVER; //test
 double xfad,yfad;
 if(fadenmodus!=0)
  {
   color(fadencolor);
   if(fadenmodus==2 && faden_npkt!=0)
    {
     koordpix2user(faden_ix[0],faden_iy[0]-menubalkenhoehe,&xfad,&yfad);
     if(!yzoom_flg) {q2plot(xfad,ymin,PENUP); q2plot(xfad,ymax,PENDOWN);} //Senkrechte Linie
     if(!xzoom_flg) {q2plot(xmin,yfad,PENUP); q2plot(xmax,yfad,PENDOWN);} //Waagrechte Linie
    }
   if(fadenmodus<=2)
    {
     koordpix2user(fadenx,fadeny-menubalkenhoehe,&xfad,&yfad);
     if(!yzoom_flg) {q2plot(xfad,ymin,PENUP); q2plot(xfad,ymax,PENDOWN);} //Senkrechte Linie
     if(!xzoom_flg) {q2plot(xmin,yfad,PENUP); q2plot(xmax,yfad,PENDOWN);} //Waagrechte Linie
    }
   else if(fadenmodus==3)
    {
     double x1,y1,x2,y2;
     koordpix2user(fadenx-6,fadeny-menubalkenhoehe-6,&x1,&y1);
     koordpix2user(fadenx+6,fadeny-menubalkenhoehe+6,&x2,&y2);
     q2plot(x1,y1,PENUP); q2plot(x2,y2,PENDOWN);
     q2plot(x1,y2,PENUP); q2plot(x2,y1,PENDOWN);
     if(faden_npkt!=0)
      {
       double x0,y0;
       koordpix2user(faden_ix[0],faden_iy[0]-menubalkenhoehe,&x0,&y0);
       koordpix2user(fadenx,fadeny-menubalkenhoehe,&xfad,&yfad);
       q2plot(x0,y0,PENUP); q2plot(xfad,yfad,PENDOWN);
      }
    }
   else if(fadenmodus==4)
    {
     int ix0=fadenx, iy0=fadeny-menubalkenhoehe;
     for(int i=1;i<faden_npkt;i++)
      {
       double x1,y1,x2,y2;
       koordpix2user(ix0+faden_ix[i-1], iy0+faden_iy[i-1], &x1, &y1);
       koordpix2user(ix0+faden_ix[i], iy0+faden_iy[i], &x2, &y2);
       q2plot(x1,y1,PENUP); q2plot(x2,y2,PENDOWN);
      }
    }
   else if(fadenmodus==5)
    {
     int ix0=fadenx, iy0=fadeny-menubalkenhoehe;
     for(int i=1;i<faden_npkt;i++)
      {
       double x1,y1,x2,y2;
       koordpix2user(ix0+faden_ix[i-1], iy0+faden_iy[i-1], &x1, &y1);
       koordpix2user(ix0+faden_ix[i], iy0+faden_iy[i], &x2, &y2);
       q2plot(x1,y1,PENUP); q2plot(x2,y2,PENDOWN);
      }
     koordpix2user(fadenx,fadeny-menubalkenhoehe,&xfad,&yfad);
     q2plot(xfad,ymin,PENUP); q2plot(xfad,ymax,PENDOWN); //Senkrechte Linie
     q2plot(xmin,yfad,PENUP); q2plot(xmax,yfad,PENDOWN); //Waagrechte Linie
    }
  }
 //zwert -= ZWERT_DELTA;
 set_uservertices();
 zflag=zflag_vorher;
 zwert=ZWERT_MAX;
 rgbcolor(255,255,255); //defaultfarbe: weiss bei schwarzem Hintergrund
}

void Tekplot::term_refresh()
{
 //DEBUG(1,printf("term_refresh()\n"));//test
 if(greinflag)
  {
   //fadenkreuz_zeichnen(); //in mausmoveEvent() machen
  }
 if(initalflag!=0)
  {
   //eventuell ausstehende Zeichenbefehle abschliessen: sollte mit OpenGL nicht vorkommen
   //if(qsaveflag) {bildclose(); qsave1flag=1; qsaveflag=0;} //neu 4.4.95
   if(qsaveflag) {bildclose(); qsaveflag=0;}
   initalflag=0;
  }
 waitmenu(0);
}

void Tekplot::term_exit()
{
 //DEBUG(1,printf("term_exit()\n"));//test
 if(initalflag)
   {
    term_refresh();
    initalflag=0;
   }
 //TODO: Resourcen freigeben?
 SDL_DestroyWindow(window);
 //SDL_Quit(); //passiert bei Programmende automatisch
}

//#define PERF_TEST //test

int Tekplot::waitmenu(int waitflag)
{
 //Rueckgabewert: 0=normal, 1=exitflag gesetzt, 2=event an Requester weitergeben
#ifdef PERF_TEST
 float32 perfCounterFrequency = SDL_GetPerformanceFrequency();
 uint64 lastCounter = SDL_GetPerformanceCounter();
#endif

 for(bool close=false; !close;) //das Programm wird meistens in dieser Schlaufe sein
  {
   SDL_Event event; //zum Ereignisse abfragen
   while(close==false && SDL_PollEvent(&event)) //Events-Schlaufe
    {
     if(event.type == SDL_QUIT) {exitflag=1; close = true; break;}
     if(event.type==SDL_MOUSEMOTION)
      {
       tek_mausx = event.motion.x;
       tek_mausy = event.motion.y;
       if(requester_aktiv) {requester_event=event; return 2;}
       else if(menu_aufgeklappt) menu_klapper_mark(tek_mausx, tek_mausy);
       else  mausmoveEvent(tek_mausx, tek_mausy);
      }
     else if(event.type==SDL_MOUSEBUTTONDOWN)
      {
       switch(event.button.button)
	{
	 case SDL_BUTTON_LEFT:  maustasten_zustand |= LIMAUS; break;
	 case SDL_BUTTON_RIGHT: maustasten_zustand |= REMAUS; break;
	 case SDL_BUTTON_MIDDLE: maustasten_zustand |= MIMAUS; break;
	 default: printf("unbekannte Maustaste gedrueckt: 0x%X\n",event.button.button);
	}
       if(requester_aktiv) {requester_event=event; return 2;}
       if(greinflag==1 && fadenflag==0) greinflag=0;
       if(event.button.button==SDL_BUTTON_LEFT && tek_mausy < menubalkenhoehe)
	{
	 //printf("Maus in menu geklickt\n");//test
	 menu_klapper(tek_mausx, tek_mausy);
	}
       else if(event.button.button==SDL_BUTTON_LEFT
	       && maus_im_menu(tek_mausx,tek_mausy))
	{
	 //printf("Maus im aufgeklappten Menu geklickt\n");//test
	 int id = ((tek_mausy - menubalkenhoehe)/menuzeilenhoehe + 1)*menu_anzahl
	          + (menu_aufgeklappt-1);
	 menu_aufgeklappt=0;
	 menu_zeichnen();
	 menu_aufruf(id);
	 //redraw_all(); //test
	}
       else if(event.button.button==SDL_BUTTON_LEFT && menu_aufgeklappt)
	{
	 menu_aufgeklappt=0;
	 menu_zeichnen();
	}
       else
	{
	 mausklickEvent(tek_mausx,tek_mausy);
	 if(mouse_press!=NULL)
	  {
	   //printf("mouse_press() soll aufgerufen werden: 0x%lX\n",(long)mouse_press);//test
	   //seltsamer Fehler: Adresse aendert sich manchmal!
	   (*mouse_press)();
	  }
	 close=true;
	}
      }
     else if(event.type==SDL_MOUSEBUTTONUP)
      {
       switch(event.button.button)
	{
	 case SDL_BUTTON_LEFT:  maustasten_zustand &= ~LIMAUS; break;
	 case SDL_BUTTON_RIGHT: maustasten_zustand &= ~REMAUS; break;
	 case SDL_BUTTON_MIDDLE: maustasten_zustand &= ~MIMAUS; break;
	 default: printf("unbekannte Maustaste losgelassen: 0x%X\n",event.button.button);
	}
       if(requester_aktiv) {requester_event=event; return 2;}
       if(greinflag==1 && fadenflag==0) greinflag=0;
       if(event.button.button==SDL_BUTTON_LEFT && maus_im_menu(tek_mausx,tek_mausy))
	{
	 //printf("Maus im aufgeklappten Menu losgelassen\n");//test
	 int id = ((tek_mausy - menubalkenhoehe)/menuzeilenhoehe + 1)*menu_anzahl
	          + (menu_aufgeklappt-1);
	 menu_aufgeklappt=0;
	 menu_zeichnen();
	 menu_aufruf(id);
	}
       else if(mouse_release!=0) (*mouse_release)();
      }
     else if(event.type==SDL_KEYDOWN)
      {
       if(requester_aktiv) {requester_event=event; return 2;}
#ifdef _DEBUG
       if(event.key.keysym.mod != 0)//test
	 printf("event.key.keysym.mod = 0x%X\n",event.key.keysym.mod);//test
#endif
       //Definitionen von LALT und RALT in SDL_keycode.h sind falsch!
       //if(event.key.keysym.mod & (KMOD_LALT|KMOD_RALT))
       if(event.key.keysym.mod & KMOD_ALT)
	{
	 int taste=event.key.keysym.sym;
	 if(taste=='q') {exitflag=1; close=true;} //Programm mit ALT q verlassen
#ifdef _DEBUG
	 if(taste>' ' && taste<='z') printf("Taste %c mit Alt-Taste gedrueckt\n",taste);//test
	 else printf("Taste 0x%02X mit Alt-Taste gedrueckt\n",taste);//test
#endif
	}
       else
	{
	 switch(event.key.keysym.sym)
	  {
	   //case SDLK_w: printf("w gedrueckt\n"); break;//test
	   case SDLK_KP_ENTER: case SDLK_RETURN: buttonEnter=true; break;
	   //default: printf("Taste '%c' 0x%X gedrueckt\n",event.key.keysym.sym,event.key.keysym.sym);//test
	  }
	}
      }
     else if(event.type==SDL_TEXTINPUT)
      {
       //printf("SDL_TEXTINPUT: event.text.text=\"%s\"\n", event.text.text);//test
       if(requester_aktiv) {requester_event=event; return 2;}
#ifdef _DEBUG
       if(strlen(event.text.text)==1) //normale Taste?
	{
	 int taste = event.text.text[0];
	 printf("Taste '%c' gedrueckt\n",taste);//test
	}
       else
	{
	 printf("Sondertaste gedrueckt: \"%s\"\n",event.text.text);//test
	}
#endif
      }
     else if(event.type==SDL_KEYUP)
      {
       if(requester_aktiv) {requester_event=event; return 2;}
       switch(event.key.keysym.sym)
	  {
	   //case SDLK_w: printf("w losgelassen\n"); break;//test
	   case SDLK_KP_ENTER: case SDLK_RETURN: buttonEnter=false; break;
	  }
      }
     else if(event.type==SDL_TEXTEDITING)
      {
#ifdef _DEBUG
       printf("SDL_TEXTEDITING: event.edit.text=\"%s\"\n", event.edit.text);//test
#endif
      }
     else if(event.type==SDL_WINDOWEVENT)
      {
#ifdef _DEBUG
       printf("SDL_WINDOWEVENT\n");//test
#endif
      }
     else if(event.type==SDL_KEYMAPCHANGED)
      {
#ifdef _DEBUG
       printf("SDL_KEYMAPCHANGED\n");//test
#endif
      }
     else if(event.type==SDL_AUDIODEVICEADDED)
      {
#ifdef _DEBUG
       printf("SDL_AUDIODEVICEADDED\n");//test
#endif
      }
     else if(event.type==SDL_MOUSEWHEEL)
      {
       //printf("Mausrad\n");//test
      }
     else printf("event.type=0x%X\n",event.type);//test
    } //Ende Event-Schlaufe

   //Puffer anlegen fuer Anwender-Zeichnung:
   userMesh.genbuffers();
   //Puffer anlegen fuer Menu-Zeichnung:
   menuMesh.genbuffers();
   //Puffer anlegen fuer Requester-Zeichnung:
   requesterMesh.genbuffers();

#ifdef PERF_TEST
   uint64 zeichnenCounter = SDL_GetPerformanceCounter();
#endif

   if(hintergrundfarbe==0)
    glClearColor(0.0f, 0.0f, 0.0f, 1.0f); //schwarzer Hintergrund
   else
   {
    float32 r = (hintergrundfarbe&0xFF)/255.0f;
    float32 g = ((hintergrundfarbe>>8)&0xFF)/255.0f;
    float32 b = ((hintergrundfarbe>>16)&0xFF)/255.0f;
    glClearColor(r, g, b, 1.0f); //1.0f ist alpha-wert
   }
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

   userMesh.render();
   menuMesh.render();
   requesterMesh.render();

#ifdef PERF_TEST
   uint64 fertigzeichnenCounter = SDL_GetPerformanceCounter();
#endif

   SDL_GL_SwapWindow(window);

#ifdef PERF_TEST
   uint64 endCounter = SDL_GetPerformanceCounter();
   uint64 counterElapsed = endCounter - lastCounter;
   float32 delta = (float32)counterElapsed / perfCounterFrequency;
   uint32 fps = (uint32)(1.0 / delta); //Frames Pro Sekunde
   float32 zzeit = (fertigzeichnenCounter-zeichnenCounter)/perfCounterFrequency;
   int32 zzeitus = zzeit*1e6;
   //std::cout << perfCounterFrequency;//test
   std::cout << "Zeichnenzeit: " << zzeitus << "us  FPS: " << fps << "\n";
   lastCounter = endCounter;
#endif
   if(waitflag==0) close=true;
  }

 return exitflag;
}

void Tekplot::mausklickEvent(int ix,int iy)
{
 //DEBUG(1,printf("mausklickEvent(%d,%d)\n",ix,iy));//test
 if(greinflag)
  {greinx=ix; greiny=iy; fadenflag=0; fadenx=fadeny=0;}
 else if(faden_npkt==0 && fadenmodus>=2)
  {
   faden_ix[0]=ix; faden_iy[0]=iy; faden_npkt=1;
  }
}
void Tekplot::mausmoveEvent(int ix,int iy)
{
 if(greinflag || fadenflag)
   {
    if(fadenflag) {fadenx=ix; fadeny=iy; fadenkreuz_zeichnen();}
    else greinflag=0;
   }
 //DEBUG(1,printf("mausmoveEvent(%d,%d)\n",ix,iy));//test
 if(mouse_motion!=NULL) (*mouse_motion)();
}

void Tekplot::neuefenstergroesse()
{
 //aktuelle Groesse des Fensters ermitteln:
 int w,h;
 SDL_GetWindowSize(window,&w,&h);
 //printf("SDL_GetWindowSize() --> w=%d h=%d\n",w,h);//test
 tekplot_breite = breite = w;
 tekplot_hoehe = hoehe = h-menubalkenhoehe;

 xs = (breite-1)/(xmax-xmin); //Skalierfaktor fuer Umrechnung in Pixel
 dx = 1.0/xs; //Breite eines Pixels in Userkoordinaten
 ys = (hoehe-1)/(ymax-ymin);
 dy = 1.0/ys; //Hoehe eines Pixels in Userkoordinaten

 //Skalierfaktor fuer Umrechnung in OpenGL-Koordinaten:
 //x-Koordinate von OpenGL geht von -1.0 bis 1.0, von links nach rechts
 //y-Koordinate von OpenGL geht von -1.0 bis 1.0, von unten nach oben
 xscalOpengl = 2.0/(xmax-xmin);
 if(menubalkenhoehe==0)
  yscalOpengl = 2.0/(ymax-ymin);
 else
  yscalOpengl = (2.0/(hoehe+menubalkenhoehe)*hoehe) / (ymax-ymin);
}

void Tekplot::rgbcolor(int r,int g,int b)
{
 int bgr=(b<<16)+(g<<8)+r;
 if(qsaveflag && postflag)
  {
   bildplot(0.,0.,4);
   if(hintergrundfarbe==0 && r==g && r==b) //bei schwarzem Hintergrund: schwarz / weiss fuer Linien vertauscht
        fprintf(fpbild,"%lf %lf %lf setrgbcolor\n", 1.0-r/255.0, 1.0-g/255.0, 1.0-b/255.0);
   else fprintf(fpbild,"%lf %lf %lf setrgbcolor\n", r/255.0, g/255.0, b/255.0);
  }
 //altefarbe=aktuellefarbe;
 aktuellefarbe=bgr;
}

void Tekplot::color(int bgr)
{
 //altefarbe=aktuellefarbe;
 //aktuellefarbe=bgr;
 rgbcolor(bgr&0xFF, (bgr>>8)&0xFF, (bgr>>16)&0xFF);
}

void Tekplot::tek_grein(double*px,double*py)
{
 tek_fadenkreuz(1);
 greinflag=1;
 do {waitmenu(0);} while(greinflag==1);
 koordpix2user(greinx,greiny-menubalkenhoehe,px,py);
 tek_fadenkreuz(0);
}

void Tekplot::tek_fadenkreuz(int modus)
{
 fadenflag = (modus>0) ? 1 : 0;
 if(fadenmodus!=modus)
  {
   if(modus==1) {faden_ix[0]=faden_iy[0]=0; faden_npkt=0;}
   else if(modus==2) {faden_ix[0]=fadenx; faden_iy[0]=fadeny; faden_npkt=1;}
   else if(modus==3) {faden_npkt=0;}
   else if(modus==4)
    {
     if(faden_npkt==0 || faden_npkt>10) printf("fadenkreuz-Fehler %d\n",faden_npkt);//test
    }
   else if(modus==5)
    {
     if(faden_npkt==0 || faden_npkt>10) printf("fadenkreuz-Fehler %d\n",faden_npkt);//test
    }
   //else if... //TODO
   fadenmodus=modus;
  }
 fadenkreuz_zeichnen();
}

void Tekplot::tek_faden_read(double*px,double*py)
{
 koordpix2user(fadenx, fadeny-menubalkenhoehe, px, py);
}

void rgbcolor(int r,int g,int b) {haupt.rgbcolor(r,g,b);}
void color(int farbe) {haupt.color(farbe);}
void waitTOF() {waitmenu(0);}
void waitBOF() {waitmenu(0);}
void tek_grein(double*px,double*py) {haupt.tek_grein(px,py);}
void tek_fadenkreuz(int onflag) {haupt.tek_fadenkreuz(onflag);}
void tek_faden_read(double* px, double* py) {haupt.tek_faden_read(px,py);}
void tek_faden_setflags(int xzoom,int yzoom) {haupt.tek_faden_setflags(xzoom,yzoom);}
void tek_faden_setxy(int x,int y) {haupt.tek_faden_setxy(x,y);}
void tek_faden_setxy2(int x,int y,int x2,int y2) {haupt.tek_faden_setxy2(x,y,x2,y2);}
void tek_faden_setpoint1(int x,int y) {haupt.tek_faden_setpoint1(x,y);}
void tek_faden_clearpoints() {haupt.tek_faden_clearpoints();}
int janeinrequester(const char *text,const char *jatext,const char *neintext)
  {return haupt.janeinrequester(text,jatext,neintext);}

void getmaxsize(int*br,int*ho)
{
 SDL_DisplayMode dm;
 int displayIndex=0;
 if(SDL_GetDesktopDisplayMode(displayIndex,&dm)!=0)
     {printf("Error in getmaxsize() in SDL_GetDesktopDisplayMode(): %s\n",SDL_GetError());}
 *br = dm.w;
 *ho = dm.h;
}

void getmaxsize(int*br,int*ho,int*ti,int*vis)
{
 sdl_init();
 SDL_DisplayMode dm;
 int displayIndex=0;
 if(SDL_GetDesktopDisplayMode(displayIndex,&dm)!=0)
     {printf("Error SDL_GetDesktopDisplayMode(): %s\n",SDL_GetError());}
 *br = dm.w;
 *ho = dm.h;
 *ti = 24; //je 8 Bit fuer die 3 Grundfarben
 *vis = TRUECOLOR; //nicht mehr wirklich noetig?
}

void Tekplot::getsize(int*br,int*ho,int*ti,int*v)
{
 *br=breite; *ho=hoehe; *ti=tekplot_tiefe; *v=visualklasse;
}
void Tekplot::setsize(int br,int ho,int dep)
{
 if(dep!=24)
  {
   printf("Warning: ungewoehnliche Tiefe: %d (sollte 24 sein)\n",dep);
   dep=24;
  }
 tekplot_breite=breite=br;
 tekplot_hoehe=hoehe=ho;
 tekplot_tiefe=dep;
 //DEBUG(1,printf("setsize() --> breite=%d hoehe=%d tiefe=%d\n",br,ho,dep));
}
void setsize(int br,int ho,int ti) {haupt.setsize(br,ho,ti);}
void getsize(int*br,int*ho,int*ti,int*v) {haupt.getsize(br,ho,ti,v);}

void Tekplot::set_funktions(funkzeig pre,funkzeig rel,funkzeig exp,funkzeig mot)
{
 mouse_press = pre;
 mouse_release = rel;
 win_expose = exp;
 mouse_motion = mot;
}

/*********************************** Schrift ***********************************/
/* schon weiter oben gemacht:
#include "fonts/myfonts.h"

//Default-Fonts hier setzen:
static Font menu_font(myfont12x16, 12, 16, 1, 32); //Speicher Breite Hoehe Typ Bytesprozeichen
static Font requ_font(myfont12x16, 12, 16, 1, 32); //Speicher Breite Hoehe Typ Bytesprozeichen
Font *menufont= &menu_font;
Font *requfont= &requ_font;
*/

void schrift(double x,double y,const char *text,double winkel)
{
 if(aktfont->missingInit()) aktfont->init(haupt.get_dx(),haupt.get_dy());
 aktfont->schrift(x,y,text,winkel,false);
}
void schrift_utf8(double x,double y,const char *text,double winkel)
{
 if(aktfont->missingInit()) aktfont->init(haupt.get_dx(),haupt.get_dy());
 aktfont->schrift(x,y,text,winkel,true);
}

void ischrift(int x,int y,const char *text,double winkel)
{
 if(aktfont->missingInit()) aktfont->init(haupt.get_dx(),haupt.get_dy());
 aktfont->ischrift(x,y,text,winkel,false);
}
void ischrift_utf8(int x,int y,const char *text,double winkel)
{
 if(aktfont->missingInit()) aktfont->init(haupt.get_dx(),haupt.get_dy());
 aktfont->ischrift(x,y,text,winkel,true);
}

double textsize(double bx,double hy,const char *font,int style)
{
 if(aktfont->missingInit()) aktfont->init(haupt.get_dx(),haupt.get_dy());
 return aktfont->textsize(bx,hy,font,style);
}

int itextsize(int bx,int hy,const char *font,int style)
{
 if(aktfont->missingInit()) aktfont->init(haupt.get_dx(),haupt.get_dy());
 return aktfont->itextsize(bx,hy,font,style);
}

int set_menu_font(double user_dy,int bx,int hy,const char *fontname)
{return haupt.set_menu_font(user_dy,bx,hy,fontname);}

int Tekplot::set_menu_font(double user_dy,int bx,int hy,const char *fontname)
{
 if(dy==0)
  {
   dx=dy=user_dy;
   menufont->init(dx,dy);
  }
 else if(user_dy!=dy)
  {printf("Error: set_menu_font() wrong user_dy: dy=%f user_dy=%f\n",dy,user_dy); return 0;}
 int h=menufont->font_aendern(fontname);
 if(h==0) return 0; //0=Fehler
 hy=menufont->itextsize(bx,hy,NULL,0);
 neue_menubalkenhoehe = 2*hy+2;
 printf("Tekplot::set_menu_font() -> hy=%d neue_menubalkenhoehe=%d\n",hy,neue_menubalkenhoehe);//test
 return hy;//Fonthoehe als Rueckgabewert
}
 
int set_requ_font(int bx,int hy,const char *fontname)
{return haupt.set_requ_font(bx,hy,fontname);}

int Tekplot::set_requ_font(int bx,int hy,const char *fontname)
{
 if(requfont->missingInit()) {requfont->init(dx,dy);}
 int h=requfont->font_aendern(fontname);
 if(h==0) return 0; //0=Fehler
 hy=requfont->itextsize(bx,hy,NULL,0);
 return hy;//Fonthoehe als Rueckgabewert
}

void setmenu(int n,...)
{
 va_list ap;
 int i;
 const int M=10;
 char *s[M];
 funkzeiger f[M];
 va_start(ap,n);
 if(n>M) printf("vorlaeufig nur %d Menus erlaubt.\n",M);
 if(n>=1 && n<=M)
   {for(i=0;i<n;i++) s[i]=va_arg(ap,char*);
    for(i=0;i<n;i++) f[i]=va_arg(ap,funkzeiger);
   }
 switch(n)
   {case 1: haupt.setmenu(n,s[0],f[0]); break;
    case 2: haupt.setmenu(n,s[0],s[1],f[0],f[1]); break;
    case 3: haupt.setmenu(n,s[0],s[1],s[2],f[0],f[1],f[2]); break;
    case 4: haupt.setmenu(n,s[0],s[1],s[2],s[3],f[0],f[1],f[2],f[3]); break;
    case 5: haupt.setmenu(n,s[0],s[1],s[2],s[3],s[4],f[0],f[1],f[2],f[3],f[4]);
	    break;
    case 6: haupt.setmenu(n,s[0],s[1],s[2],s[3],s[4],s[5],
			    f[0],f[1],f[2],f[3],f[4],f[5]); break;
    case 7: haupt.setmenu(n,s[0],s[1],s[2],s[3],s[4],s[5],s[6],
			    f[0],f[1],f[2],f[3],f[4],f[5],f[6]); break;
    case 8: haupt.setmenu(n,s[0],s[1],s[2],s[3],s[4],s[5],s[6],s[7],
			    f[0],f[1],f[2],f[3],f[4],f[5],f[6],f[7]); break;
    case 9: haupt.setmenu(n,s[0],s[1],s[2],s[3],s[4],s[5],s[6],s[7],s[8],
			    f[0],f[1],f[2],f[3],f[4],f[5],f[6],f[7],f[8]); break;
    case 10: haupt.setmenu(n,s[0],s[1],s[2],s[3],s[4],s[5],s[6],s[7],s[8],s[9],
			     f[0],f[1],f[2],f[3],f[4],f[5],f[6],f[7],f[8],f[9]); break;
    default: haupt.setmenu(n);
   }
 va_end(ap);
}

void setsubmenu(int n,...)
{
 va_list ap;
 int i;
 const int M=10;
 char *s[M];
 funkzeiger f[M];
 va_start(ap,n);
 printf("Testpunkt: setsubmenu(n=%d, ...)\n",n);//test
 printf(" setsubmenu() geht noch nicht\n");//test
 //TODO
 va_end(ap);
}

void Tekplot::getmenuids(long idnr,long *idfeld,int max)
{
 if(idnr==0) //Hauptmenus
  {
   /*
   int i,j,ji;
   if(max==0 || max>menu_anzahl) max=menu_anzahl;
   for(i=0;i<max;i++)
    for(j=1,ji=menu_anzahl+i; j<menu.ntext[i]; j++,ji+=menu_anzahl)
     {idfeld[ji] = -ji-1;} //negative Nummern fuer Hauptmenus
   */
   //einfacher wenn nur Haupmenus:
   if(max==0 || max>menu_i) max=menu_i;
   for(int i=0; i<max; i++)
    {
     idfeld[i] = -i-1;
    }
  }
 else //Submenus
  {
   //TODO
   //positive Nummern fuer Submenus?
  }
}

void Tekplot::changemenu(long idnr,const char *neuertext)
{
 //printf("Testpunkt: changemenu(idnr=%ld, neuertext=\"%s\")\n",idnr,neuertext);//test
 //TODO
 if(idnr>=0) {printf("Fehler in changemenu(): bisher nur HauptmenuIDs erlaubt. idnr=%ld\n",idnr); return;}
 int i = -idnr-1;
 strcpy(menu_textfeld[i], neuertext);
}
#endif //TEKPLOT1_IMPLEMENTATION

/******************************* Menus ***********************************/
#ifdef AUTOMENU
void automenu_exit() {haupt.exitflag=1;}
void automenu_loop()
{
 if(haupt.initalflag) term_refresh();
 while(haupt.exitflag==0 && haupt.waitmenu(1)==0) {}
 haupt.exitflag=0;
}
#endif

/* Diese Variablen sind jetzt in der Tekplot-Klasse:
static funkzeiger menu_funkfeld[MAXMENU],menu_sub_funkfeld[MAXMENU];
int		menu_debug=0;
static int	menu_anzahl=0,menu_i=0,menu_isub=0,menu_k=0,
		menu_istsub[MAXMENU],menu_sub_ifu[MAXMENU];
static char	*menu_textfeld[MAXMENU];
*/

#ifdef TEKPLOT1_IMPLEMENTATION
void Tekplot::menu_reset()
{
 menu_anzahl=menu_i=0;
 //menu_isub=menu_k=0;
 menubalkenhoehe=0;
}

int Tekplot::menuvorhanden() {return menu_anzahl>0;}

void Tekplot::setmenu(int n,...)
{
/*
Beispiel:
 setmenu(3,"File","Funktionen",     "Hilfe"); // Menuleiste
 setmenu(3,"Quit","erste Funktion", "Ueber",  // Menupunkte
           &quit, &erste,           &ueber);  // Zeiger auf Funktionen
 setmenu(2,"Exit","zweite Funktion",&exit,&zweite);
 setmenu(2,NULL,  "dritte Funktion",NULL,&dritte);
*/
 va_list ap;
 int i,k;
 if(n<=0)
   {
#ifdef AUTOMENU
    if(n==AUTOMENU)
      {setmenu(1,"File"); setmenu(1,"Exit",automenu_exit);}
    else
#endif
      printf("setmenu(): menu_debug=%d\n",menu_debug= -n);
    return;
   }
 if(menu_i+n > MAXMENU)
    {printf("maximale Menu-Anzahl ueberschritten\n"); return;}
 va_start(ap,n);
 if(menu_i==0)
  {
   menu_anzahl=n;
   if(menu_debug) printf("menu_anzahl=%d\n",menu_anzahl);//test
   //for(i=0;i<MAXMENU;i++) menu_istsub[i]=menu_sub_ifu[i]=0;
   //menu_isub=0; menu_k=0;
   for(i=0;i<n;i++) {menu_textfeld[menu_i++] = va_arg(ap,char*);}
  }
 else
  {
   if(menu_debug) printf("menu_i=%d\n",menu_i);//test
   for(k=menu_i,i=0; i<n; i++)
    {menu_textfeld[k++] = va_arg(ap,char*);} //n Parameter in menu_textfeld[] eintragen
   //for(k=menu_i-menu_isub,i=0; i<n; i++) //menu_isub bisher immer auf 0
   for(k=menu_i,i=0; i<n; i++)
    {
     menu_funkfeld[k++] = va_arg(ap,funkzeiger); //naechste n Parameter = Zeiger auf Funktionen
                                                 //in menu_funkfeld[] eintragen
     //amigatast.put(k-1,-k); //provisorisch auskommentiert (von amitekplot uebernommen?)
     //printf("menu_funkfeld[%d]=0x%08X\n",k-1,menu_funkfeld[k-1]);//test
    }
   for(i=n;i<menu_anzahl;i++)
    {menu_textfeld[menu_i+i]=NULL; menu_funkfeld[k++]=NULL;} //unbenutztes mit NULL fuellen
   menu_i += menu_anzahl; //Index fuer naechsten Aufruf von setmenu()
   //if(menu_sub_ifu[menu_k]) menu_k++; //? Parameter fuer submenu ?
  }
 va_end(ap);
}

void Tekplot::menu_aufruf(int id)
{
 funkzeiger funk;
 //int j=id-40000; ??
 //if(j<menu_anzahl || j>j1i_max) ??
 //printf("menu_aufruf(%d) j=%d\n",id,j);//test
 int j=id;
 if(j<menu_anzahl || j>=MAXMENU)
  {printf("Error in menu_aufruf() j=%d ungueltig\n",j); return;}
 funk=menu_funkfeld[j];
 //printf("funk=0x%08X\n",funk);//test
 if(funk!=NULL) (*funk)(j);
}

void Tekplot::menu_umrahmen(int ix1,int iy1,int ix2,int iy2)
{
 double x1,y1,x2,y2;
 koordpix2user(ix1,iy1,&x1,&y1);
 koordpix2user(ix2,iy2,&x2,&y2);
 zwert -= ZWERT_DELTA;
 rgbcolor(80,80,255); //dunkelblau umrahmt
 drawbox(x1,y1,x2,y2);
 rgbcolor(0,0,0); //schwarze Schrift fuer Menu
}

void Tekplot::menu_klapper(int mausx,int mausy) //aufgerufen wenn mit Maus in Menuleiste geklickt
{
 //printf("menuklapper(mausx=%d,mausy=%d)\n",mausx,mausy);//test
 if(menu_aufgeklappt)
  {menu_aufgeklappt=0;} //offenes Menu wieder schliessen
 int ix=menu_ix0; //iy=menu_iy0;
 int bux=menufont->get_ischrift_breite();
 ix -= bux/2; if(ix<0) ix=0;
 menu_mainboxlist[0] = ix;
 for(int i=0; i<menu_anzahl; i++)
  {
   const char *text=menu_textfeld[i];
   int ixnext = ix + (strlen(text)+2)*bux; //x-Position vom naechsten Text
   menu_mainboxlist[i+1] = ixnext;
   if(mausx>=ix && mausx<ixnext)
    {
     menu_aufgeklappt = i+1;
     break;
    }
   ix=ixnext;
  }
 menu_zeichnen();
}

bool isinbox(int x,int y,int *box)
{
 return (x>box[0] && x<box[2] && y>box[1] && y<box[3]);
}

void Tekplot::menu_klapper_mark(int mausx,int mausy) //aufgerufen wenn Maus in Menuleiste bewegt
{
 if(menu_aufgeklappt<1) {printf("Error menu_klapper_mark()\n"); return;}//test
 static int klappindex = -1;
 static int loeschflag = 0;
 bool isinboxflag=false;
 if(mausy<=menubalkenhoehe)
  {
   int x1=menu_mainboxlist[menu_aufgeklappt-1];
   int x2=menu_mainboxlist[menu_aufgeklappt];
   if(mausx>=x1 && mausx<=x2)
    {
     if(loeschflag==0) return;//noch im markierten Hauptmenu -> nichts zu tun
    }
   else
    {
     menu_klapper(mausx,mausy); //Menu neu aufklappen
     klappindex = -1; //im aufgeklappten Teil noch nichts markiert
     loeschflag=0;
    }
  }
 else isinboxflag = isinbox(mausx,mausy+1,menu_aufklappbox);
 if(isinboxflag || loeschflag>=1)
  {
   //Menupunkt im aufgeklappten Teil, auf dem Maus steht markieren:
   int ix1=menu_aufklappbox[0]+da1, ix2=menu_aufklappbox[2]-da1, iy1,iy2;
   int neuer_klappindex = (!isinboxflag) ? -1 : (mausy-menubalkenhoehe)/menuzeilenhoehe;
   if(klappindex!=neuer_klappindex)
    {
     if(menubalkenhoehe!=menuzeilenhoehe)
       printf("menubalkenhoehe=%d ..zeilenhoehe=%d\n",menubalkenhoehe,menuzeilenhoehe);//test
     set_menuvertices(false);
     float zwert_vorher=zwert;
     zwert = menu_mark_zwert;
     if(klappindex>=0)
      {
       zwert -= ZWERT_DELTA;
       rgbcolor(180,180,255); //hellblau = Hintergrund des Klappteils -> Rahmen loeschen
       iy1 = da1+klappindex*menuzeilenhoehe;
       iy2 = iy1+menuzeilenhoehe-da2;
       idrawbox(ix1,iy1,ix2,iy2);
       if(loeschflag>0) loeschflag--;
      }
     klappindex = neuer_klappindex;
     if(klappindex>=0)
      {
       zwert -= ZWERT_DELTA;
       rgbcolor(90,90,255); //dunkelblau fuer Rahmen um aktuellen Menupunkt
       iy1 = da1+klappindex*menuzeilenhoehe;
       iy2 = iy1+menuzeilenhoehe-da2;
       idrawbox(ix1,iy1,ix2,iy2);
       loeschflag++;//wieder loeschen wenn Maus aus Box hinausfaehrt
      }
     menu_mark_zwert = zwert;
     zwert=zwert_vorher;
     rgbcolor(255,255,255); //defaultfarbe: weiss bei schwarzem Hintergrund
     set_uservertices();
    }
  }
 else klappindex = -1;
}

bool Tekplot::maus_im_menu(int mausx,int mausy)
{
 //true wenn im aufgeklappten Menu
 if(menu_aufgeklappt==0) return false;
 return isinbox(mausx,mausy,menu_aufklappbox);
}

void Tekplot::redraw_all()
{
 if(redraw_all_func==NULL)
  {
   fprintf(stderr,"Missing redraw_all_function\n");
   inital_new(0,0,0,0);
   screenclear(hintergrundfarbe);
   term_refresh();
   return;
  }
 (*redraw_all_func)();
}

char *radio_button_provisorisch(const char *text)
{
 static char neutext[80];
 strncpy(neutext,&text[1],80); neutext[80-1]=0;
 switch(neutext[0])
  {case 'Q': neutext[0]='X'; break; //provisorisch statt gefuelltes Viereck
   case 'O': neutext[0]='O'; break; //provisorisch statt gefuellter Kreis
   case 'R': neutext[0]='V'; break; //provisorisch statt Haken-Markierung
   default: neutext[0]=' ';
  }
 return neutext;
}

void Tekplot::menu_zeichnen()
{
 //leere Menuleiste zeichnen:
 set_menuvertices(true);
 zwert = MENU_ZWERT_START; //Menu auf fast vorderster Ebene
 int zflag_vorher=zflag;
 zflag=ZFLAG_NEVER; //test
 double hy,bx;
 deltakoordpix2user(0, menubalkenhoehe, &bx, &hy);
 rgbcolor(180,180,255); //hellblau
 fillbox(xmin,ymax, xmax,ymax+hy);
 rgbcolor(0,0,0); //schwarze Schrift fuer Menu
 
 //Hauptmenu zeichnen:
 zwert -= ZWERT_DELTA;
 //printf("Hauptmenu zeichnen: zwert=%f\n",zwert);//test
 int ix=menu_ix0, iy=menu_iy0;
 int bux=menufont->get_ischrift_breite(), buy=menufont->get_ischrift_hoehe();//Font Breite und Hoehe
 //printf("menu_zeichnen() menu_aufgeklappt=%d\n",menu_aufgeklappt);//test
 int klappnr = (menu_aufgeklappt==0) ? 0 : menu_aufgeklappt+menu_anzahl-1;
 ix -= bux/2; if(ix<0) ix=0;
 menu_mainboxlist[0]=ix;
 for(int i=0; i<menu_anzahl; i++)
  {
   const char *text=menu_textfeld[i];
   menufont->ischrift(ix+bux/2,iy,text,0,0);
   ix += (strlen(text)+2)*bux; //x-Position vom naechsten Text
   menu_mainboxlist[i+1] = ix; //speichern fuer eventuell aufgeklapptes Menu
   if(i==menu_aufgeklappt-1)
    {menu_umrahmen(menu_mainboxlist[i], 0, ix, 1-menubalkenhoehe);}
  }

 if(statustext[0]!=0) //Statustext hinter letztem Hauptmenu anzeigen
  menufont->ischrift(ix+bux,iy,statustext,0,0);
 
 if(menu_aufgeklappt) //eventuell aufgeklapptes Menu zeichnen
  {
   double x1,y1,x2,y2,br,ho;
   int maxbr=0,nk=0;
   for(int i=klappnr; i<menu_i && menu_textfeld[i]!=NULL; i+=menu_anzahl)
    {
     int n = (strlen(menu_textfeld[i])+2)*bux;
     if(n>maxbr) maxbr=n;
     nk++;
    }
   ix = menu_mainboxlist[klappnr-menu_anzahl];  iy = menu_iy0+menuzeilenhoehe;
   deltakoordpix2user(ix, 0, &br, &ho);
   x1=xmin+br; y1=ymax;
   deltakoordpix2user(maxbr, menuzeilenhoehe*nk, &br, &ho);
   x2=x1+br; y2=y1-ho;
   koorduser2mauspix(x1,y1,x2,y2,menu_aufklappbox);
   rgbcolor(180,180,255); //hellblau aufgeklappter Menuteil
   fillbox(x1,y1,x2,y2);
   zwert -= ZWERT_DELTA;
   rgbcolor(90,90,255); //dunkelblau umrahmt
   drawbox(x1,y1,x2,y2);
   zwert -= ZWERT_DELTA;
   rgbcolor(0,0,0); //schwarze Schrift
   for(int i=klappnr; i<menu_i && menu_textfeld[i]!=NULL; i+=menu_anzahl)
    {
     const char *text=menu_textfeld[i];
     if(text[0]=='%') text=radio_button_provisorisch(text);
     menufont->ischrift(ix+bux/2,iy,text,0,0);
     iy += menuzeilenhoehe;
    }
  }
 menu_mark_zwert = zwert;
 set_uservertices();
 zflag=zflag_vorher;
 zwert=ZWERT_MAX;
 rgbcolor(255,255,255); //defaultfarbe: weiss bei schwarzem Hintergrund
}

/***************************** Requester *********************************/

/* Requester in eigenem SDL-Fenster: geht nicht
int test_messageBox()
{
 uint32 flags = SDL_WINDOW_OPENGL; //| SDL_WINDOW_BORDERLESS;
 SDL_Window* window=SDL_CreateWindow(titel, 200, 200, //TODO: Position prov. 200 200
					    width, height, flags);
 SDL_GLContext context = SDL_GL_CreateContext(window);
 //TODO: ins Fenster zeichnen
 //TODO: Events abwarten
 printf("messageBox() noch provisorisch - weiter mit Enter "); getc(stdin);//test
 //SDL_GL_DeleteContext(context);
 SDL_DestroyWindow(window);
 return 0;
}
*/

int Tekplot::messageBox(const char *titel, const char *message, int width, int height,
			Button* button1, Button* button2, Button* button3)
{
 // allgemeines Unterprogramm fuer alle Requester
 //printf("messageBox(%s, %s, %d, %d, ...)\n",titel,message,width,height);//test
 int antwort=0; //Rueckgabewert: 0=ungueltig (bei Fehler), sonst 1, 2, oder 3
 
 // Inhalt des Subfensters zeichnen:
 set_requesterVertices();
 zwert = REQUESTER_ZWERT_START; //Requester auf fast vorderster Ebene
 int zflag_vorher=zflag;
 zflag=ZFLAG_NEVER;//test
 int ix0=100, iy0=100;//TODO: ev. variable Position des Requesters?
 rgbcolor(180,180,255); //hellblauer Hintergrund
 ifillbox(ix0, iy0, ix0+width, iy0+height);
 zwert -= ZWERT_DELTA;
 rgbcolor(150,150,255); //etwas dunkleres hellblau fuer Knoepfe (Buttons)
 if(button1==NULL)
  {fprintf(stderr,"Error: missing button1 in messageBox()\n"); zflag=zflag_vorher; return 0;}
 ifillbox(button1->x1+ix0, button1->y1+iy0, button1->x2+ix0, button1->y2+iy0);
 if(button2!=NULL)
   ifillbox(button2->x1+ix0, button2->y1+iy0, button2->x2+ix0, button2->y2+iy0);
 if(button3!=NULL)
   ifillbox(button3->x1+ix0, button3->y1+iy0, button3->x2+ix0, button3->y2+iy0);
 zwert -= ZWERT_DELTA;
 //Umrahmungen der Knoepfe zeichnen:
 rgbcolor(80,80,200); //blaue Umrahmungen
 int xa,xb,ya,yb;
 int da=da1; //kleine Distanz abhaengig von Bildschirmaufloesung (1 bis 3)
 idrawbox(xa=button1->x1+ix0, ya=button1->y1+iy0, xb=button1->x2+ix0, yb=button1->y2+iy0);
 if(button1->flags & BUTTON_PRESELECT) idrawbox(xa+da, ya+da, xb-da, yb-da);
 if(button2!=NULL)
  {
   idrawbox(xa=button2->x1+ix0, ya=button2->y1+iy0, xb=button2->x2+ix0, yb=button2->y2+iy0);
   if(button2->flags & BUTTON_PRESELECT) idrawbox(xa+da, ya+da, xb-da, yb-da);
  }
 if(button3!=NULL)
  {
   idrawbox(xa=button3->x1+ix0, ya=button3->y1+iy0, xb=button3->x2+ix0, yb=button3->y2+iy0);
   if(button3->flags & BUTTON_PRESELECT) idrawbox(xa+da, ya+da, xb-da, yb-da);
  }
 zwert -= ZWERT_DELTA;
 //Beschriftungen zeichnen:
 int bux=requfont->get_ischrift_breite(), buy=requfont->get_ischrift_hoehe(); //Font Breite und Hoehe
 int xrand = (width-strlen(titel)*bux)/2;
 requfont->itextsize(bux,buy,NULL,0);
 rgbcolor(80,80,200); //blaue Schrift fuer Titel
 requfont->ischrift(ix0+xrand, iy0+buy+1, titel,0,0);
 char c, message1[80]; mystrncpy2(message1,message,80);
 rgbcolor(0,0,0); //schwarze Schrift
 int i=strlen(message1);
 while(i>1 && ((c=message1[i-1])=='\n' || c=='\r' || c==' '))
  {message1[--i]=0;} //Zeilentrenner und Leerstellen an Zeilenende loeschen
 xrand = (width-strlen(message1)*bux)/2;
 requfont->ischrift(ix0+xrand, iy0+height/2-buy/2, message1, 0,0);
 int x = (button1->x2 - button1->x1 - strlen(button1->text)*bux)/2 + button1->x1;
 int y = (button1->y2 - button1->y1 - buy)/2 + buy + button1->y1;
 requfont->ischrift(ix0+x, iy0+y, button1->text, 0,0);
 if(button2!=NULL)
  {
   x = (button2->x2 - button2->x1 - strlen(button2->text)*bux)/2 + button2->x1;
   y = (button2->y2 - button2->y1 - buy)/2 + buy + button2->y1;
   requfont->ischrift(ix0+x, iy0+y, button2->text, 0,0);
  }
 if(button3!=NULL)
  {
   x = (button3->x2 - button3->x1 - strlen(button3->text)*bux)/2 + button3->x1;
   y = (button3->y2 - button3->y1 - buy)/2 +buy + button3->y1;
   requfont->ischrift(ix0+x, iy0+y, button3->text, 0,0);
  }
 
 unset_requesterVertices();
 zwert=ZWERT_MAX;
 zflag=zflag_vorher;
 rgbcolor(255,255,255); //defaultfarbe: weiss bei schwarzem Hintergrund
 // Fertig mit Zeichnen vom Inhalt des Subfensters

 requester_aktiv=true;
 for(;exitflag==0 && antwort==0;)
  {
   int flag=waitmenu(0);
   if(flag==0) continue;
   if(flag==2) //dann Events abfragen
    {
     if(requester_event.type==SDL_MOUSEBUTTONDOWN)
      {
       if(maustasten_zustand & LIMAUS)
	{
	 int mx=tek_mausx-ix0, my=tek_mausy-menubalkenhoehe-iy0;
	 //printf("Mausklick: x=%d y=%d\n",mx,my);//test
	 //usleep(200000); //test 0.2sec warten
	 //testen ob ein Button angeklickt:
	 if(ist_in_box(button1, mx, my)) {antwort=1; break;}
	 if(ist_in_box(button2, mx, my)) {antwort=2; break;}
	 if(ist_in_box(button3, mx, my)) {antwort=3; break;}
	}
      }
     else if(requester_event.type==SDL_KEYDOWN)
      {
       switch(requester_event.key.keysym.sym)
	{
	 case SDLK_KP_ENTER: case SDLK_RETURN: buttonEnter=true; break;
	}
       if(buttonEnter)
	{
	 if(button1->flags & BUTTON_PRESELECT) {antwort=1; break;}
	 if(button2->flags & BUTTON_PRESELECT) {antwort=2; break;}
	 if(button3->flags & BUTTON_PRESELECT) {antwort=3; break;}
	}
      }
    }//Ende Events abfragen
  }
 requester_aktiv=false;
 requesterMesh.clear();
 return antwort; //Rueckgabewert ist Nummer des angeklickten Buttons, also 1, 2, oder 3.
}

int Tekplot::janeinrequester(const char *text,const char *jatext,const char *neintext)
{
 int antwort=0;
 int width,height;
 Button knopf1,knopf2;
 if(jatext==NULL) knopf1.set(" OK "); else knopf1.set(jatext);
 if(neintext!=NULL) knopf2.set(neintext);
 knopf1.flags=BUTTON_PRESELECT;
 int bux=requfont->get_ischrift_breite();
 width = (strlen(text)+2)*bux;
 if(width<200) width=200;
 height = width*4/10;
 if(width>tekplot_breite) width=tekplot_breite;
 if(height>tekplot_hoehe) height=tekplot_hoehe;
 int posx=0, posy=height*2/3; //TODO: optimale y-Position berechnen
 int knopfbreite = (strlen(knopf1.text)+1)*bux, knopfhoehe=bux*2;
 if(neintext==NULL)
  {posx = width/2 - knopfbreite/2;}
 else
  {
   int breite2 = (strlen(knopf2.text)+1)*bux;
   if(breite2>knopfbreite) knopfbreite=breite2;
   posx = width/3 - knopfbreite/2;
   knopf2.setpos(width*2/3 - knopfbreite/2, posy);
   knopf2.setsize(knopfbreite, knopfhoehe);
  }
 knopf1.setpos(posx, posy);
 knopf1.setsize(knopfbreite, knopfhoehe);
 antwort = messageBox("JaneinRequester", text, width, height,
 		      &knopf1,  (neintext==NULL) ? NULL : &knopf2,  NULL);
 return (antwort==1)?1:0;
}

#define PIXPROBUCHST_X requfont->get_ischrift_breite()
#define PIXPROBUCHST_Y requfont->get_ischrift_hoehe()

/* besser ersetzen durch: const_cast<char*>("Text")
char *strKopie(const char *str)
{
 int n=strlen(str)+1;
 char *kopie=new char[n];
 //printf("strKopie(\"%s\") n=%d\n",str,n);//test
 mystrncpy2(kopie,str,n);
 return kopie;
}
*/

/**************************** nachfilenamefragen() ****************************/
// nachfilenamefragen(...) und requester_input(...)
#include "nachfilenamefragen.h"

/************************* plotsave() bildplot() ******************************/
//static FILE *fpbild=NULL; //weiter oben
static double psxcal,psycal,psxmin,psymin;
double psxkor(double x) {return (x-psxmin)*psxcal;}
double psykor(double y) {return (y-psymin)*psycal;}
#define initalflag haupt.initalflag

void plotsave(const char* s)
{
 DEBUG(1,printf("plotsave('%s')\n",s));//test
 if((s[0]=='A' && s[1]=='U') || (s[0]=='O' && s[1]=='F'))
  {
   if(qsaveflag) bildclose();
   //qsaveflag=qsave1flag=0;
   qsaveflag=0;
  }
 else
  {
   qsaveflag=1;
   const char *bn;
   char bildname[200];
   bn=getenv("PLOTSAVE");
   if(bn!=NULL && strlen(bn)>=200) {printf("Fehler: $PLOTSAVE zu lang: \"%s\"\n",bn); bn=NULL;}
   if(bn==NULL) bn="tmpbild.ps";
   strcpy(bildname,bn);
   postflag = (strncmp(s,"POST",4)==0);
   if(!postflag) {printf("Fehler: plotsave(\"%s\") statt plotsave(\"POST\")\n",s);}
   //if(initalflag) bildopen(bildname);
   //else {qsave1flag=1; qsaveflag=0;} //neu 4.4.95
   bildopen(bildname);
  }
}

#ifndef MIT_MYBILDOPEN
//TODO: verbesserte Variante von showitlight.cc auch hier uebernehmen?
void bildopen(const char* name)
{
 FILE *fp;
 double x1,y1,x2,y2;
 //double ymini,ymaxi,xmini,xmaxi,ystep;
 get_xyminmax(&x1,&y1,&x2,&y2);
 DEBUG(1,printf("bildopen('%s')\n",name));
 fp=fpbild=fopen(name,"w");
 if(fp==NULL) {printf("Fehler beim Oeffnen von '%s'\n",name); qsaveflag=0; return;}
 if(postflag)
   {
    fprintf(fp,"%%!PS-Adobe-2.0 EPSF-1.2\n");
    fprintf(fp,"%%%%Creator: xtekplot1 %s\n",tekplot_version);
#define BBXMAX 558
#define BBYMAX 750
#define BBU 10
#define BBL 38
#define BBR 19
#define BBO 6
#define BBDY 3
#define BBUCHSTABENBREITE 4
#define GH 16
    psxcal=BBYMAX/(x2-x1); psxmin=x1;
    psycal=BBXMAX/(y2-y1); psymin=y1;
    fprintf(fp,"%%%%BoundingBox: %d %d %d %d\n",
				-BBU,-BBR,BBXMAX+BBO,BBYMAX+BBL);
    fprintf(fp,"save\n");
    fprintf(fp,"/m /moveto load def\n");
    fprintf(fp,"/d /lineto load def\n");
    fprintf(fp,"/s /stroke load def\n");
    fprintf(fp,"/dsm {2 copy lineto stroke moveto} bind def\n");
    fprintf(fp,"/Helvetica findfont 15 scalefont setfont\n");
    fprintf(fp,"0 %d translate  -90 rotate\n",BBYMAX);
    return;
   }
 else printf("Fehler in bildopen(): fehlendes postflag.\n");//test
}

/*
static void anschreiben(FILE* fp,double wert,int x,int y,int strokeflag)
{				 // Werte in Postscriptbild anschreiben
 char str[32];
 int dx;
 if(wert<10.) sprintf(str,"%.3lf",wert);
 else  sprintf(str,"%.2lf",wert);
 dx=(BBUCHSTABENBREITE*strlen(str))/2;
 if(strokeflag) putc('s',fp);
 fprintf(fp," %d %d m (%s) show\n",x-dx,y-BBDY,str);
}
*/

void bildplot(double x,double y,int pen)
{
 static int ncount=0, pen2count=0;
 DEBUG(1,printf("bildplot(%lf,%lf,%d)\n",x,y,pen));
 //if(postflag)
   {
    static int lzugi=0;
    if(pen==2)
	{fprintf(fpbild," %.2lf %.2lf d",psxkor(x),psykor(y));
	 if(++pen2count>=100)  {pen2count=0; fprintf(fpbild,"sm");}
	 ++lzugi;
	}
    else
	{if(lzugi>1) {fprintf(fpbild," s\n"); lzugi=1; ncount= -1;}
	 if(pen==3) fprintf(fpbild," %.2lf %.2lf m ",psxkor(x),psykor(y));
	 pen2count=0;
	}
    if(++ncount>=4)  {ncount=0; fprintf(fpbild,"\n");}
    return;
   }
}

void bildclose()
{
 DEBUG(1,printf("bildclose()\n"));
 bildplot(0.,0.,4);
 fprintf(fpbild," showpage\nrestore\n");
 fclose(fpbild); fpbild=NULL;
 qsaveflag=0;
}
#endif //MIT_MYBILDOPEN

/********************* Ende von plotsave() bildplot() *************************/

#endif //TEKPLOT1_IMPLEMENTATION

//um Verwechslungen mit Systemfunktionen zu vermeiden:
#define plot(x,y,pen) q2plot(x,y,pen)

#endif //OTEKPLOT1_H
