/* uvshow.cc		    letzte Aenderungen: 13.12.2021 */
#define VERSION "2.16"
/*
Autor: Rolf Pfister
Lizenz: Opensource Freeware

UV Auswerteprogramm mit Verwendung von OpenGL

History:
13.12.2021  Version 2.16  Erstellung aus uvshow.c

*/

#include "otekplot1.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "showitlight.h"
#ifndef PI
#define PI 3.14159265358979
#endif
#include "ulong.h"

#ifdef _WIN32
char *unconst(const char *s)
{
 int n=strlen(s)+1;
 char *str=new char[n];
 strcpy(str,s);
 return str;
}
#else
#define unconst(s) (char*)(s)
#endif

#define MAXARG 3
#define MAXSPEKT 15
#define OD		1
#define EPSILON		2
#define LOGEPSILON	4
#define TRANSMISSION	8
#define SIGMA		0x10
#define EPSILONMASK	0x16 /* (EPSILON | LOGEPSILON | SIGMA) */
#define NM ('N'+256*'M')
#define CM ('C'+256*'M')
#define IN ('I'+256*'N')
/*#define TR ('T'+256*'R') veraltet: Transmission wird jetzt in mody gefhrt */

#define cfree free

#define UVALT	1
#define UVOD2	2
#define SPEX2	3
#define CASE break;case
#define DEFAULT break;default

#ifndef true
#define true 1
#define false 0
#endif

struct uv_kopf
{       LONG    start,ende,delta,slit;   /* Werte in PicoMeter (0.001nm) */
        WORD    period;	  /* Daempfung in Millisekunden */
        WORD    multi;	  /* Anzahl Multiscans, Daten aber bereits dividiert */
        LONG    conz;     /* Konzentration * Schichtdicke in NanoMol/l*cm    */
	LONG	yfaktor;  /* Die y-Werte sind um diesen Faktor vergroessert  */
};

/*********************** Vordeklarationen ********************************/
#define getline mygetline
int getline(FILE *fp,char *s,int lim);
double powi(double x,int i);
int hatpunkt(char *s);
void ohnepunkt(char *s);
char *machgross(char *s);
void uvshow(char *filename,char *substanzname,float **pyfeld,float **pyhfeld,
       WORD *mody,WORD *modp,WORD *irev,LONG *modus);
int wordlesen(FILE *fp);
LONG longlesen(FILE *fp);
int gepacktlesen(FILE *fp,LONG *y);
void uvschreib(char *filename,float *yfeld,WORD *mody,WORD *irev);
void zeilelesen(char *str,int max);
void error(int n);
#define CANT_OPEN_NEW_FILE 1
#define ZU_WENIG_RAM 2
int istvorhanden(char *name);

int uvshowfor(char *filename,char *substanzname,
	      float *yfeld,float *hfeld,WORD *mody2,WORD *modp,WORD *irev);

void inkatalog(char *filename,char *substanzname,struct uv_kopf *kopf);
void intspeichern(FILE *fp,int y);
void longspeichern(FILE *fp,LONG y);
void gepacktspeichern(FILE *fp,LONG y);
void plotsave_setzen(char *filename);
void funk_anpass();
void spex_parameter_lesen(int c,FILE *fp);
#ifdef VAX_OR_ALPHA
void kill(char*);
#endif
void anhaengen(char *str,char *s,int max);
void anhaengen2(char *str,char *s,int max);
char *spektrum_typ(int n);
void splitpfad(const char *name,char *pfad,char *file,int max,int max2);
void mehrere_files_laden(char *name);
void spex_spektrum_speichern(char *filename,float *yfeld,WORD *mody,WORD *irev,
		   char *snam, struct uv_kopf *kopf);
void spex_par_clear();
void spex_par_copy(int i,char *zeile);

/*************************** globale Parameter ************************/
char filename[80],spename[200];
typedef char STRING[120];
STRING substanzname[MAXSPEKT];
typedef char STRING80[80];
STRING80 fname2[MAXSPEKT-1];
int spektrumtyp[MAXSPEKT],nspe= -1;

static STRING80
	OLGA_TEXT[MAXSPEKT-1], //"Emission, Exciting: ..."  neu: 24.7.2000
	LASER_TEXT[MAXSPEKT-1]; //".. Laserpulses at ..Hz with ..ms Gate"
static bool lasermodus=false;

#define OLGA 1
#define MOPS 2
#define CORRECTED 4
float *hauptfeld=NULL,*hilfsfeld=NULL,*PNULL=NULL;
int feldgroesse;
WORD	mody, /* Darstellungseinheit in y-Richtung (OD TRANSMISSION EPSILON) */
	modp, /* Darstellungseinheit in x-Richtung (CM oder NM) */
	irev; /* Richtung der Darstellung */
LONG modus=4, minus1= -1, npkt;

/* in Fortran: COMMON /ISAM/ ISTART,IEND,DX,PER,SLIT,CONZ,SNAM */
//extern  //mit oder ohne?, ausprobieren
//struct _isam {WORD ISTART,IEND; float DX,PER,SLIT,CONZ; char SNAM[20];} ISAM;
struct _isam {WORD ISTART,IEND; float DX,PER,SLIT,CONZ; char SNAM[20];};
_isam ISAM;

static FILE *fp=NULL;/* fuer ausgemessene Punkte und Integrale */
static int uvformat, katalogflag=0, testflag=0;
static int ntesspektrum=3;
static double yfaktor2=0.0;

void feldalloc(int n,float **feld,float **hfeld)
{
 if(*feld!=NULL) cfree(*feld);
 if(*hfeld!=NULL) cfree(*hfeld);
 *feld = (float*)calloc(n,sizeof(float)); *hfeld = (float*)calloc((MAXSPEKT-1)*n,sizeof(float));
 if(*feld==NULL || *hfeld==NULL) error(ZU_WENIG_RAM);
 feldgroesse=n;
 return;
}

/***************************** Hauptprogramm **************************/
#include "olga-eichkurve.c"
char argflag['Z'+1];
int altkor=0;

void setargflags(char *s)
{
 int c;
 while((c= *s++)!=0)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
  }
}


int main(int argc,char *argv[])
{
 int i,j;
 for(*filename= *fname2[0]=0,i=j=1;i<argc;i++)
   if(*argv[i]=='-') setargflags(argv[i]);
   else switch(j++)
	{case 1:strcpy(filename,argv[i]); break;
	 case 2:strcpy(fname2[0],argv[i]); break;
	 default:break;
        }
 if(argc>MAXARG+1 || filename[0]=='?' || argflag['?'])
	{printf("UVSHOW  Version %s\n",VERSION);
	 printf("Anwendung: UVSHOW [-Flags] File1 [File2]\n");
	 printf("    Flags: O=OD  E=Epsilon C=cm-1  R=Rueckwaerts\n");
	 printf("           I=Sigma, Absorptionsquerschnitt / 10^{-20}\n");
	 printf("           T=Transmission  L=log(Epsilon)\n");
	 printf("           K=in Katalog aufnemen  D=Debug\n");
	 printf("           X = SPEX-Eichkurven-Korrektur\n");
	 printf("           A = Alte unlesbare Datei korrigiert lesen\n");
	 printf("           P = Postscriptbild speichern, dann exit\n");
	 printf("           S = Speichere im neusten Format, dann exit\n");
	 printf("           S2 = Speichere auch wenn UVOD2 schon gesetzt\n");
	 exit(0);
	}
 if(*filename==0) {printf("File:"); scanf("%s",filename); getc(stdin);}
 if(!istvorhanden(filename)) exit(0);
 // plotsave_setzen(filename);//test
 if(argflag['E']) mody=EPSILON;
 else if(argflag['I']) mody=SIGMA;
 else if(argflag['L']) mody=LOGEPSILON;
 else if(argflag['T']) mody=TRANSMISSION;
 else mody=OD;
 if(argflag['R']) irev=1; else irev=0;
 if(argflag['C']) modp=CM; else modp=NM;
 if(argflag['K']) katalogflag=1;
 if(argflag['D'] || argflag['V']) testflag=1;
 if(argflag['A'])
  {if(argflag['0']) altkor=0;
   else if(argflag['1']) altkor=1;
   else if(argflag['2']) altkor=2;
   else	{printf("Korrekturversuch fuer alte Datei\n");
	 printf(" 0 = Daten vermutlich noch vollstaendig\n");
	 printf(" 1 = erstes Byte jedes Records zerstoert\n");
	 printf(" 2 = noch unbekannte Fehler suchen\n");
	 printf(" 3 = noch mehr Testausdrucke\n");
	 printf("Korrekturmethode:"); scanf("%d",&altkor);
  }	}
 npkt=modus;
 feldalloc(8000,&hauptfeld,&hilfsfeld);
 if(*fname2[0]!=0 && istvorhanden(fname2[0]))
  {uvshow(fname2[0],substanzname[1],&hilfsfeld,&PNULL,&mody,&modp,&irev,&minus1);
   uvshow(filename,substanzname[0],&hauptfeld,&hilfsfeld,&mody,&modp,&irev,&npkt);
  }
 else
  {npkt=modus=2; ntesspektrum=2;
   uvshow(filename,substanzname[0],&hauptfeld,&PNULL,&mody,&modp,&irev,&npkt);
  }
 return 0;
}/* ende von main */

void file_exit()
{
 if(fp!=NULL) fclose(fp);
 exit(0);
}

bool istscript(char *name)
{
 char c;
 if(*name=='@') return true;
 while((c= *name++)!=0)
   if(c=='.' && strncmp(name,"scr",3)==0) return true;
 return false;
}

void file_load_mu(char *muname)
{  //wenn muname gesetzt ist, soll das entsprechende Spektrum geladen werden
   //sonst wird nach Dateiname gefragt.
 float *hf;
// lower_window();
 if(ntesspektrum>MAXSPEKT)
	printf("Anzahl Spektren auf %d beschrnkt\n",MAXSPEKT);
 else
  {int ntes2=ntesspektrum-2;
   //printf("%d. Spektrum laden\n",ntesspektrum);
   //printf(" File:"); scanf("%s",fname2[ntes2]);
   if(muname!=NULL)
     {strcpy(fname2[ntes2],muname);
      printf("%d. Spektrum = %s\n",ntesspektrum,fname2[ntes2]);
     }
   else
     {char scratch[80],pfad[80],name[200];
      static char filter[80]="*.uv";
      int ok;
      sprintf(scratch,"lade %d. Spektrum",ntesspektrum);
      strcpy(name,".uv");
      ok=nachfilenamefragen(scratch,name,200,
			    0,filter,NULL," CANCEL"," Parent","  Disks",80);
      if(!ok) {refresh(); return;}
      if(strlen(name)<80 && istvorhanden(name)) strcpy(fname2[ntes2],name);
      else splitpfad(name,pfad,fname2[ntes2],80,80);
     }
   if(istscript(fname2[ntes2]))
	mehrere_files_laden(fname2[ntes2]);
   else
   {if(fname2[ntes2][1]==0)
	{janeinrequester("Eingaben fuer Funktionsanpassung in Shell-Fenster",
			 " OK ",NULL);
	 lower_window(); funk_anpass();
	 raise_window();
	}
    else if(istvorhanden(fname2[ntes2]))
	{hf= &hilfsfeld[(ntesspektrum-2)*npkt];
	 uvshow(fname2[ntes2],substanzname[ntesspektrum-1],
		  &hf,&PNULL,&mody,&modp,&irev,&minus1);
	 modus=4;
	 if(ntesspektrum>2) modus+=ntesspektrum;
	 new_hfeld(hilfsfeld,(int*)&modus);
	 ntesspektrum++;
  }}	}
// raise_window();
 refresh();
}

void file_load()
{
 file_load_mu(NULL);
}

void file_save()
{
 char antw[80],c;
 if(argflag['S'] && uvformat>=UVOD2 && !argflag['2']) return;
 lower_window();
 if(!argflag['S'])
  {printf("Speichern unter '%s' ?",filename); scanf("%s",antw);
   if((c=toupper(antw[0]))=='N')
	{printf("Filename:"); scanf("%s",filename);}
   else if(c!='J')
	strcpy(filename,antw);
#ifdef VAX_OR_ALPHA
   else kill(filename);
#endif
  }
#ifdef VAX_OR_ALPHA
 else kill(filename);
#endif
 printf("Saving '%s'\n",filename);
 uvschreib(filename,hauptfeld,&mody,&irev);
 raise_window();
}

FILE *fopen_uvshowdat()
{
 FILE *fp;
 fp=fopen("UVSHOW.DAT","w");
 if(fp==NULL) printf("Fehler beim Oeffnen von 'UVSHOW.DAT'\n");
 else fprintf(fp,"%s  %s\n",substanzname[0],filename);
 return fp;
}

/*************************** user_prog ***********************************/
const char *mody2str(int mody)  /* aus mody Klartext machen */
{
 const char *s;
 if(mody==EPSILON) s="Epsilon";
 else if(mody==LOGEPSILON) s="log(Epsilon)";
 else if(mody==TRANSMISSION) s="% Transmission";
 else if(mody==SIGMA) s="$cross section / 10^{-20} cm^2";
 else s="OD";
 return s;
}

void get_delta(float *z,float grenz[4])
{
 const char *eps,*nmcm;
 printf("get_delta: x1=%lf y1=%lf  x2=%lf y2=%lf\n",
	grenz[0],grenz[1],grenz[2],grenz[3]);
 eps=mody2str(mody);
 if(modp==CM) nmcm="cm-1"; else nmcm="nm";
 printf("%7.2lf bis %7.2lf   xDifferenz=%7.2lf %s  yDifferenz=%7.5lf %s\n",
	grenz[0],grenz[2],grenz[0]-grenz[2],nmcm, grenz[1]-grenz[3],eps);
 if(fp==NULL)
   {if((fp=fopen_uvshowdat())==NULL) return;
    fprintf(fp,"     xBereich            xDifferenz            yBereich       \
  yDifferenz\n");
   }
 fprintf(fp,"%7.2lf bis %7.2lf  %7.2lf %s    %9.5lf bis %9.5lf  %9.5lf %s\n",
	 grenz[0],grenz[2],grenz[0]-grenz[2],nmcm,
	 grenz[1],grenz[3],grenz[1]-grenz[3],eps);
}

void int_save(float *z,float *xspitz,float *yspitz,float grenz[4])
{
 const char *eps,*nmcm;
 eps=mody2str(mody);
 if(modp==CM) nmcm="cm-1"; else nmcm="nm";
 printf("%7.2lf bis %7.2lf%s  Integral=%lg  Maximum: %7.2lf %s  %s=%7.5lf\n",
	grenz[0],grenz[2],nmcm,*z,*xspitz,nmcm,eps,*yspitz);
 printf("Integral: %lg\n",*z);
 if(fp==NULL)
   {if((fp=fopen_uvshowdat())==NULL) return;
    fprintf(fp," Integrierter Bereich      Integral     Spitze        %s\n",
		eps);
   }
 fprintf(fp,"%7.2lf bis %7.2lf%s    %lg    %7.2lf%s    %7.5lf\n",
	grenz[0],grenz[2],nmcm,*z,*xspitz,nmcm,*yspitz);
}

void get_punkt(float *x,float *y)
{
 const char *nm,*eps;
 static double letzter=0.;
 printf("GetPoint: x=%lg  y=%lg\n",*x,*y);
 if(fp==NULL) {if((fp=fopen_uvshowdat())==NULL) return;}
 if(modp==CM) nm="cm-1"; else nm="nm";
 eps=mody2str(mody);
 if(letzter!=0.)
	fprintf(fp,"%lg%s  %s=%lg  delta=%lg\n",*x,nm,eps,*y,*x-letzter);
 else fprintf(fp,"%lg%s  %s=%lg\n",*x,nm,eps,*y);
 if(testflag) printf("%lg%s  %s=%lg  delta=%lg\n",*x,nm,eps,*y,*x-letzter);
 letzter= *x;
}

void texteladen(char **titel,char **xachse,char **yachse)
{
 if(modus<4)
  {if(!lasermodus) sprintf(spename,
    "%s\n%s  %s\n%.2lf ... %.2lf\nScanrate:%.3lf  NScans:%d\nSlit:%.2lf  Period:%.2lf",
    substanzname[0],machgross(filename),spektrum_typ(nspe), ISAM.ISTART/20.,ISAM.IEND/20.,
    ISAM.DX/20.,(int)ISAM.PER,
    ISAM.SLIT,(ISAM.PER-((int)ISAM.PER))*100.);
   else sprintf(spename,
    "%s\n%s  %s\n%.2lf ... %.2lf\nScanrate:%.3lf  NScans:%d\n%s",
    substanzname[0],machgross(filename),spektrum_typ(nspe), ISAM.ISTART/20.,ISAM.IEND/20.,
    ISAM.DX/20.,(int)ISAM.PER,
    LASER_TEXT[0]);
   *titel=spename;
  }
 else	{int max,i; char *str,*t,*s;
	 sprintf(spename,"%s %s",machgross(filename),spektrum_typ(0));
	 str=(char*)calloc(max=strlen(spename)+(ntesspektrum-1)*80+1,1);
	 if(str==NULL) error(ZU_WENIG_RAM);
	 for(t=str,s=spename;(i = *t++ = *s++)&& i!='\n' && i!='\r';) ;
	 *--t=0;
	 if(*--t<' ') *t=0; /* hinter Dateiname noch substanzname haengen */
	 anhaengen2(str,substanzname[0],max);
	 for(i=2;i<ntesspektrum;i++)
		{anhaengen(str,fname2[i-2],max);
		 anhaengen2(str,substanzname[i-1],max);
		}
	 *titel=str;
	}
/* if(modp==CM) *xachse="Wavenumber (cm<EH.8)-1<EXHX))"; */
 if(modp==CM)	*xachse=unconst("$Wavenumber / cm^{-1}");
 else		*xachse=unconst("Wavelength / nm");
 if(mody==OD) *yachse=unconst(" "); /* OD nicht anschreiben */
 else *yachse = (char*)mody2str(mody);
}

void user_prog(int *fun,POINTER p1,POINTER p2,POINTER p3,POINTER p4)
{
 switch(*fun)
        {case 1: case 2: int_save((float*)p1,(float*)p2,(float*)p3,(float*)p4); break;
	 case 3: case 4: get_punkt((float*)p1,(float*)p2); break;
	 case 5: texteladen((char**)p1,(char**)p2,(char**)p3); break;
	 case 6: get_delta((float*)p1,(float*)p2); break;
	 default: printf("user_prog(%d,%06lx,%06lx,%06lx,%06lx)\n",
	                            *fun,(long)p1,(long)p2,(long)p3,(long)p4);
	}
}

char *machgross(char *s)
{
 char *t;
 int c;
 t=s;
 while((c= *t)!=0)  {*t++ = toupper(c);}
 return s;
}

float *cmumrech(float *start,float *ende,float **pyfeld,int *npkt)
{
 float *yhfeld,*yfeld;
 int i,i2,j;
 double a,b,d,x;
 static double A,B,D;
 static int N, erstesflag=1;
 yfeld= *pyfeld;
 yhfeld=(float *)calloc(*npkt,sizeof(float));
 if(yhfeld==NULL) error(ZU_WENIG_RAM);
 for(i=0;i<(*npkt);i++) yhfeld[i]=yfeld[i];
 a= *start;  b= *ende;  d=(*start-(*ende))/((*npkt)-1);
 if(erstesflag)
	{A=1e7/a; B=1e7/b;
	 D=1e7/(a-d)-A;
	 N=(B-A)/D+0.5;
	 B=A+N*D;
	}
 if(testflag)
	 printf("a=%lf b=%lf d=%lf npkt=%d\nA=%lf B=%lf D=%lf N=%d\n",
			a,b,d,*npkt,A,B,D,N);
 if(N>feldgroesse) {feldalloc(N,&hauptfeld,&hilfsfeld); yfeld= *pyfeld;}
 for(j=0;j<N;j++)
	{i=x=(a-1e7/(A+j*D))/d; i2=i+1;
	 if(i<0 || i2>=(*npkt)) yfeld[j]=0.;
	 else yfeld[j]=yhfeld[i]+(yhfeld[i2]-yhfeld[i])*(x-i);
	}
 cfree(yhfeld);
 *start=A;
 *ende=B;
 *npkt=N+1;
 if(testflag) printf("neues npkt=%d\n",*npkt);
 erstesflag=0;
 return yfeld;
}

float *cmumrech(double *start,double *ende,float **pyfeld,int *npkt)
{
 float fstart=(float)(*start), fende=(float)(*ende);
 float *yfeld=cmumrech(&fstart,&fende,pyfeld,npkt);
 *start = fstart;
 *ende = fende;
 return yfeld;
}

/********* Erweiterung 13.8.2010 zum Lesen vom UVPC-SPC-Format **********/
bool hatendung(char *s,char *e)
{
 int c,i;
 while((c= *s++)!=0)
  {if(c==e[0])
    {for(i=1;e[i]!=0 && s[i-1]!=0 && e[i]==s[i-1];i++) {}
     if(e[i]==0 && s[i-1]==0) return true;
    }
  }
 return false;
}
float floatlesen(int* pj,FILE *fp1,int h,char* he)
{
 int j= *pj,k;
 union {char c[4]; float y;} y;
 for(k=0;k<4;k++,j++)
   y.c[k] = (j<h) ? he[j] : getc(fp1);
 *pj=j;
 return y.y;
}
double doublelesen(int* pj,FILE *fp1,int h,char* he)
{
 int j= *pj,k;
 union {char c[8]; double y;} y;
 for(k=0;k<8;k++,j++)
   y.c[k] = (j<h) ? he[j] : getc(fp1);
 *pj=j;
 if(y.c[6]==0 || y.c[7]==0 || y.y!=y.y) y.y= 1e20;//ungueltige Zahl
 return y.y;
}
int textsuchen(int* pj,FILE *fp,char *text)
{
 int i=0,j= *pj,c,c0;
 int position=j;
 while((c0=text[i])!=0)
  {if((c=getc(fp))==EOF) return 0;//nicht gefunden
   j++;
   if(c==c0)
    {i++;}
   else
    {position=j;
     if(i>0)
       {if(c==text[0]) {position=j-1; i=1;}
	else {i=0;}
       }
    }
  }
 *pj=j;
 return position;//gefunden
}

void textblocklesen(int *pj,FILE *fp,char *zeile)
{
 int j= *pj,i,c,n,imax=80-1;
 n=getc(fp); j++;
 for(i=0;i<n;i++)
  {c=getc(fp); j++;
   if(i<imax) *zeile++ = c;
  }
 *zeile=0;
 *pj=j;
}

/********************** Anpassung an Format 'UVOD2   ' ******************/
#define VERTAUSCHE(x,y) h=x; x=y; y=h
#define WRITE "wb"
#define READ  "rb"
//LONG idfix(double x) {if(x>=0.) return (LONG)(x+0.5); else return (LONG)(x-0.5);}
long idfix(double x);

static struct uv_kopf kopf;

double mody_umrechnung(double y,int mody,double conz)
{						/* y enthlt den OD-Wert und */
 if(mody & EPSILONMASK)				/* wird dann gemss mody auf */
	{y /= conz;				/* die gewnschte Einheit    */
	 if(mody==LOGEPSILON)			/* umgerechnte.              */
		{if(y < 0.001) y=0.001;
		 y=log10(y); /* log(Epsilon) */
		}
	 else if(mody==SIGMA)
		{y *= 0.3823;}
	}
 else if(mody==TRANSMISSION)
	y=100.*pow(10.,-y); /* Prozent Transmission */
 return y;
}
void isam_setzen(WORD istart,WORD iend,char *substanzname)
{
 int i;
 ISAM.ISTART=istart; ISAM.IEND=iend;
 ISAM.DX=kopf.delta/1e3*20.; ISAM.PER=1.+kopf.period/1e3/100.;
 ISAM.SLIT=kopf.slit/1e3; ISAM.CONZ=kopf.conz/1e9;
 for(i=0;i<19 && substanzname[i]>=' ';i++)
   ISAM.SNAM[i]=substanzname[i];
 ISAM.SNAM[i]=0;
}

void uvshow(char *filename,char *substanzname,float **pyfeld,float **pyhfeld,
       WORD *mody,WORD *modp,WORD *irev,LONG *modus)
{
 FILE *fp;
 char format[10],zeile[80],*s;
 int i,c;
 LONG x,y,j;
 int modus2,npkt;
 float start,ende,ymin,ymax,*py,*yfeld,*yhfeld;
 double yfakt,konz,h;
 fp=fopen(filename,"r");
 c=getc(fp); ungetc(c,fp);
 if(isdigit(c) || c=='"') strcpy(format,"XY      ");
 else if(hatendung(filename,".spc")) strcpy(format,"SPC     ");
 else
  for(i=0;i<8;i++) format[i]=getc(fp);
 format[8]=0;
 if(testflag) printf("Format=%s\n",format);
 if(strcmp(format,"UVOD2   ")==0)
	{int nfehler=0;
	 uvformat=UVOD2;
	 for(s=substanzname;(c=getc(fp))>=' ';)  *s++ = c;
	 *s=0;
	 kopf.start=longlesen(fp); kopf.ende=longlesen(fp);
	 kopf.delta=longlesen(fp); kopf.slit=longlesen(fp);
	 kopf.period=wordlesen(fp); kopf.multi=wordlesen(fp);
	 kopf.conz=longlesen(fp); konz=kopf.conz/1e9;
	 kopf.yfaktor=longlesen(fp); yfakt=(double)kopf.yfaktor;
	 start=kopf.start/1e3; ende=kopf.ende/1e3;
	 npkt=(kopf.start-kopf.ende)/kopf.delta+1;
	 printf("'%s'\n",substanzname);
	 printf("gescannter Bereich : %.2lf bis %.2lf nm\n",start,ende);
	 printf("Scanrate: %.3lf nm/s   Anzahl Scans: %d\n",
		kopf.delta/1e3,kopf.multi);
	 printf("Slit: %.2lf nm    Period: %.1lf s\n",
		kopf.slit/1e3,kopf.period/1e3);
	 if(konz==0.)	{printf("conz*d: unbekannt\n");
			 if(*mody & EPSILONMASK) *mody=OD;
			}
	 else printf("conz*d: %lg cm*Mol/Liter\n",konz);
	 ymin=0.; ymax=0.001;
	 if(npkt>feldgroesse) feldalloc(npkt,&hauptfeld,&hilfsfeld);
	 yfeld= *pyfeld; yhfeld= *pyhfeld;
	 for(py=yfeld,x=kopf.start; x>=kopf.ende; x-=kopf.delta,py++)
	   {if(gepacktlesen(fp,&y)==EOF) nfehler++;
	    *py=mody_umrechnung(y/yfakt,*mody,konz);
	    if(*py<ymin) ymin= *py;
	    if(*py>ymax) ymax= *py;
	   }
	 h=powi(10.,(int)(log10(ymax)+9.)-9);
	 ymax=h*(int)(ymax/h+1.); /* ymax auf runde Zahl aufrunden */
	 if(ymin>(-ymax/35.)) ymin= -ymax/35.;
	 printf("\n  ymin=%lf  ymax=%lf\n",ymin,ymax);
	 if(nfehler) printf("ERROR:'%s' hat %d Lesefehler\n",filename,nfehler);
	 for(j=0;getc(fp)!=EOF;j++)  ;
	 if(j) printf("WARNUNG:'%s' hat %d ueberfluessige Bytes\n",filename,j);
	 fclose(fp);
	 /* provisorisch: ISAM-Feld setzen */
	 isam_setzen(start*20.,ende*20.,substanzname);
	 if(*modp==CM) yfeld = cmumrech(&start,&ende,pyfeld,&npkt);
	 if(*irev==0)
	   {for(i=0;i<npkt/2;i++) {j=npkt-1-i; VERTAUSCHE(yfeld[i],yfeld[j]);}
	    VERTAUSCHE(start,ende);
	   }
	 if(katalogflag) inkatalog(filename,substanzname,&kopf);
	 if(*modus>=0)
	   {modus2= *modus; *modus=npkt;
	    if(argflag['P']) modus2+=AUTOPLOTEXIT;
	    if(argflag['S']) modus2+=AUTOSAVEEXIT;
	    if(testflag) printf("showit(%lf,%lf,%lf,%lf,%d,%d,%08lX,%08lX\n",
				 start,ymin,ende,ymax,npkt,modus2,(long)yfeld,(long)yhfeld);
	    showit(&start,&ymin,&ende,&ymax,&npkt,&modus2,yfeld,yhfeld);
	   }
	}
 else if(strcmp(format,"SPEX2   ")==0)
	{int nfehler=0,willcorr=0;
	 LONG period;
	 uvformat=SPEX2; corrflag=0;
	 kopf.conz=0; yfaktor2=0.0;
	 while((c=getc(fp))!=0xFF && c!=EOF)
	   spex_parameter_lesen(c,fp);
	 willcorr=(argflag['X'] && corrflag==0);
	 period = (kopf.period<0) ? -1000*kopf.period : kopf.period;
	 if(testflag)
	   {if(willcorr) printf("OLGA-Eichkurve wird angewendet\n");
	    else printf("es wird keine Eichkurve angewendet\n");
	   }
	 if(yfaktor2!=0.0)
	   {yfakt=yfaktor2;
	    kopf.yfaktor=(yfakt<2e3)?idfix(1e6*yfakt):idfix(yfakt);
	   }
	 else
	   {kopf.yfaktor=1000*period*kopf.multi;
	    yfakt=(double)kopf.yfaktor*1e-6; //Counts/sec
	    printf("yfaktor=%lf\n",yfakt);//test
	   }
	 konz=kopf.conz/1e9;
	 start=kopf.start/1e3; ende=kopf.ende/1e3;
	 npkt=(kopf.start-kopf.ende)/kopf.delta+1;
	 printf("'%s'\n",substanzname);
	 printf("gescannter Bereich : %.2lf bis %.2lf nm\n",start,ende);
	 printf("Scanrate: %.3lf nm/s   Anzahl Scans: %d\n",
		kopf.delta/1e3,kopf.multi);
	 printf("Slit: %.2lf mm    Period: %.1lf s\n",
		kopf.slit/1e3,period/1e3);
	 if(konz==0.)	{printf("conz*d: unbekannt\n");
			 if(*mody & EPSILONMASK) *mody=OD;
			}
	 else printf("conz*d: %lg cm*Mol/Liter\n",konz);
	 ymin=0.; ymax=0.001;
	 if(npkt>feldgroesse) feldalloc(npkt,&hauptfeld,&hilfsfeld);
	 yfeld= *pyfeld; yhfeld= *pyhfeld;
	 for(py=yfeld,x=kopf.start; x>=kopf.ende; x-=kopf.delta,py++)
	   {if(gepacktlesen(fp,&y)==EOF) nfehler++;
	    if(willcorr)
	         *py=correct(x/1e3,mody_umrechnung(y/yfakt,*mody,konz));
	    else *py=mody_umrechnung(y/yfakt,*mody,konz);
	    if(*py<ymin) ymin= *py;
	    if(*py>ymax) ymax= *py;
	   }
	 h=powi(10.,(int)(log10(ymax)+9.)-9);
	 ymax=h*(int)(ymax/h+1.); /* ymax auf runde Zahl aufrunden */
	 if(ymin>(-ymax/35.)) ymin= -ymax/35.;
	 printf("\n  ymin=%lf  ymax=%lf\n",ymin,ymax);
	 if(nfehler) printf("ERROR:'%s' hat %d Lesefehler\n",filename,nfehler);
	 for(j=0;getc(fp)!=EOF;j++)  ;
	 if(j) printf("WARNUNG:'%s' hat %d ueberfluessige Bytes\n",filename,j);
	 fclose(fp);
	 /* provisorisch: ISAM-Feld setzen */
	 isam_setzen(start*20.,ende*20.,substanzname);
	 if(*modp==CM) yfeld=cmumrech(&start,&ende,pyfeld,&npkt);
	 if(*irev==0)
	   {for(i=0;i<npkt/2;i++) {j=npkt-1-i; VERTAUSCHE(yfeld[i],yfeld[j]);}
	    VERTAUSCHE(start,ende);
	   }
	 if(katalogflag) inkatalog(filename,substanzname,&kopf);
	 if(*modus>=0)
	   {modus2= *modus; *modus=npkt;
	    if(argflag['P']) modus2+=AUTOPLOTEXIT;
	    if(argflag['S']) modus2+=AUTOSAVEEXIT;
	    if(testflag) printf("showit(%lf,%lf,%lf,%lf,%d,%d,%08lX,%08lX\n",
				 start,ymin,ende,ymax,npkt,modus2,(long)yfeld,(long)yhfeld);
	    showit(&start,&ymin,&ende,&ymax,&npkt,&modus2,yfeld,yhfeld);
	   }
	}
 else if(strcmp(format,"XY      ")==0)
	{double xw,yw,xstart,xende,dx;
	 int tauschflag=0,nzeilenkommentar=0;
	 do
	  {getline(fp,zeile,80); c= *zeile;
	   if(c=='"') {nzeilenkommentar++; if(testflag) printf("%s\n",zeile);}
	  }
	 while(c=='"');
	 sscanf(zeile,"%lf%*c%lf",&xstart,&yw);
	 for(npkt=0,i=2;i==2;npkt++)
	   i=fscanf(fp,"%lf%*c%lf",&xende,&yw);
	 fp=freopen(filename,"r",fp); //fclose(fp); fp=fopen(filename,"r");
	 for(i=0;i<nzeilenkommentar;i++) getline(fp,zeile,80);
	 dx=(xende-xstart)/(npkt-1);
	 printf("Start=%lf Ende=%lf   Anzahl Messpunkte = %d  (Step=%lf)\n",
		xstart,xende,npkt,dx);
	 if(xende>xstart)
	  {dx=xende; xende=xstart; xstart=dx; //xstart mit xende vertauschen
	   dx=(xende-xstart)/(npkt-1);
	   tauschflag=1;
	  }
	 kopf.start=(LONG)(xstart*1e3+0.5);
	 kopf.ende=(LONG)(xende*1e3+0.5);
	 kopf.delta=(kopf.start-kopf.ende)/(npkt-1);
	 kopf.conz=0; konz=kopf.conz/1e9;
	 kopf.slit=0; kopf.period=0; kopf.multi=1; kopf.yfaktor=1; //test
	 start=kopf.start/1e3; ende=kopf.ende/1e3;
	 ymin=0.; ymax=0.001;
	 if(npkt>feldgroesse) feldalloc(npkt,&hauptfeld,&hilfsfeld);
	 yfeld= *pyfeld; yhfeld= *pyhfeld;
	 if(tauschflag) py= &yfeld[npkt-1];
	 else py=yfeld;
	 for(x=kopf.start; x>=kopf.ende; x-=kopf.delta,(tauschflag)?py--:py++)
	   {fscanf(fp,"%lf%*c%lf",&xw,&yw);
	    *py=mody_umrechnung(yw,*mody,konz);
	    if(*py<ymin) ymin= *py;
	    if(*py>ymax) ymax= *py;
	   }
	 h=powi(10.,(int)(log10(ymax)+9.)-9);
	 ymax=h*(int)(ymax/h+1.); /* ymax auf runde Zahl aufrunden */
	 if(ymin>(-ymax/35.)) ymin= -ymax/35.;
	 printf("\n  ymin=%lf  ymax=%lf\n",ymin,ymax);
	 fclose(fp);
	 /* provisorisch: ISAM-Feld setzen */
	 isam_setzen(start*20.,ende*20.,substanzname);
	 if(*modp==CM) yfeld=cmumrech(&start,&ende,pyfeld,&npkt);
	 if(*irev==0)
	   {for(i=0;i<npkt/2;i++) {j=npkt-1-i; VERTAUSCHE(yfeld[i],yfeld[j]);}
	    VERTAUSCHE(start,ende);
	   }
	 if(*modus>=0)
	   {modus2= *modus; *modus=npkt;
	    if(argflag['P']) modus2+=AUTOPLOTEXIT;
	    if(argflag['S']) modus2+=AUTOSAVEEXIT;
	    if(testflag) printf("showit(%lf,%lf,%lf,%lf,%d,%d,%08lX,%08lX\n",
				start,ymin,ende,ymax,npkt,modus2,(long)yfeld,(long)yhfeld);
	    showit(&start,&ymin,&ende,&ymax,&npkt,&modus2,yfeld,yhfeld);
	   }
	}
 else if(strcmp(format,"SPC     ")==0)
       {double xw,yw,xstart,xende,dx;
	int j;
	const int h2=512;
	char he[h2];
	for(i=0;i<h2;i++) he[i]=getc(fp);
	if(c==0)
	{printf("UVPC-SPC Format\n");
	 printf("comment = \"");
	 for(i=0x27,s=substanzname;i<h2 && (c=he[i])!=0;i++)
	   {if(i<0x27+120-1) *s++ = c;//max. 120 Zeichen speichern
	    putc(c,stdout);
	   }
	 *s=0;
	 putc('\"',stdout); putc('\n',stdout);
	 printf("time date = ");
	 for(i=0x64;i<h2 && (c=he[i])!=0;i++)
	   putc(he[i],stdout);
	 putc(' ',stdout);
	 for(i++;i<h2 && (c=he[i])!=0;i++)
	   putc(c,stdout);
	 putc('\n',stdout);
	 npkt=(he[0x76]&0xFF)+(he[0x77]<<8); //Anzahl Messpunkte
	 j=10; xstart=floatlesen(&j,fp,h2,he);
	 xende=floatlesen(&j,fp,h2,he);
	 dx=(xende-xstart)/(npkt-1);
	 printf("Start=%lf Ende=%lf   Anzahl Messpunkte = %d  (Step=%lf)\n",
		xstart,xende,npkt,dx);
	 kopf.start=(LONG)(xstart*1e3+0.5);
	 kopf.ende=(LONG)(xende*1e3+0.5);
	 kopf.delta=(kopf.start-kopf.ende)/(npkt-1);
	 kopf.conz=0; konz=kopf.conz/1e9;
	 kopf.slit=0; kopf.period=0; kopf.multi=1; kopf.yfaktor=1; //test
	 start=kopf.start/1e3; ende=kopf.ende/1e3;
	 ymin=0.; ymax=0.001;
	 if(npkt>feldgroesse) feldalloc(npkt,&hauptfeld,&hilfsfeld);
	 yfeld= *pyfeld; yhfeld= *pyhfeld;
	 for(i=0,j=0x78,xw=xstart,py=yfeld; i<npkt; i++,xw+=dx,py++)
	   {yw=floatlesen(&j,fp,h2,he);
	    *py=mody_umrechnung(yw,*mody,konz);
	    if(*py<ymin) ymin= *py;
	    if(*py>ymax) ymax= *py;
	   }
	}
	else if(c==0xD0 && (he[1]&0xFF)==0xCF)
	{int jmax;
	 printf("UV2450-SPC Format\n");
	 for(j=h2;j<0x2000;j++) getc(fp);//Schrott ueberlesen
	 jmax=textsuchen(&j,fp,"Wavelength Range");
	 if(jmax==0)
	  {printf("spc-Format-Error: \"Wavelength Range\" not found.\n");
	   fclose(fp); return;
	  }
	 while(++j && (c=getc(fp))!=':' && c!=EOF) ;//naechster Doppelpunkt
	 textblocklesen(&j,fp,zeile);
	 sscanf(zeile,"%lf to %lf",&xende,&xstart);
	 if(testflag) printf("'%s' xstart=%lf xende=%lf\n",zeile,xstart,xende);
	 for(;;)
	  {textblocklesen(&j,fp,zeile);
	   if(*zeile==0) break;
	   if(strcmp(zeile,"Sampling Interval:")==0)
	    {textblocklesen(&j,fp,zeile);
	     sscanf(zeile,"%lf",&dx);
	     if(testflag) printf("dx=%lf\n",dx);//test
	     break;
	    }
	  }
	 jmax=(j+0xFF)&0xFFFFFF00; //auf runde Adresse aufrunden
	 double yw4[4];
	 for(i=0;i<100;i++)
	  {int ok,k;
	   for(;j<jmax;j++) getc(fp);
	   for(k=0;k<4;k++) yw4[k]=doublelesen(&j,fp,h2,he);
	   if(testflag) printf("%04X: yw=%lf\n",j-4*8,yw4[0]);
	   for(k=0,ok=1;k<4;k++)
	    {if(!(yw4[k]<=4.0 && yw4[k]>= -1.0)) ok=0;}
	   if(ok) break;
	   //jmax=(j+0x0F)&0xFFFFFFF0; //auf runde Adresse aufrunden
	  }
	 if(i>=100)
	  {printf("spc-Format-Error: keine vernuenftigen Werte gefunden.\n");
	   fclose(fp); return;
	  }
	 if(xstart>xende && dx>0) dx = -dx;
	 npkt=(int)((xende-xstart)/dx+1.5);
	 printf("Start=%lf Ende=%lf   Anzahl Messpunkte = %d  (Step=%lf)\n",
		xstart,xende,npkt,dx);
	 kopf.start=(LONG)(xstart*1e3+0.5);
	 kopf.ende=(LONG)(xende*1e3+0.5);
	 kopf.delta=(kopf.start-kopf.ende)/(npkt-1);
	 kopf.conz=0; konz=kopf.conz/1e9;
	 kopf.slit=0; kopf.period=0; kopf.multi=1; kopf.yfaktor=1; //test
	 start=kopf.start/1e3; ende=kopf.ende/1e3;
	 //printf("j=0x%04X\n",j);//test
	 ymin=0.; ymax=0.001;
	 if(npkt>feldgroesse) feldalloc(npkt,&hauptfeld,&hilfsfeld);
	 yfeld= *pyfeld; yhfeld= *pyhfeld;
	 for(i=0,j=0x2800,xw=xstart,py=yfeld; i<npkt; i++,xw+=dx,py++)
	   {if(i<4) yw=yw4[i]; else yw=doublelesen(&j,fp,h2,he);
	    *py=mody_umrechnung(yw,*mody,konz);
	    if(*py<ymin) ymin= *py;
	    if(*py>ymax) ymax= *py;
	   }
	}
	else
	{printf("unbekanntes SPC-Format: 0x%02X %02X\n",c,he[1]&0xFF);
	 fclose(fp); return;
	}
	//Vorbereitung und Aufruf von showit() unabhaengig vom Format:
	h=powi(10.,(int)(log10(ymax)+9.)-9);
	ymax=h*(int)(ymax/h+1.); /* ymax auf runde Zahl aufrunden */
	if(ymin>(-ymax/35.)) ymin= -ymax/35.;
	printf("\n  ymin=%lf  ymax=%lf\n",ymin,ymax);
	fclose(fp);
	/* provisorisch: ISAM-Feld setzen */
	isam_setzen(start*20.,ende*20.,substanzname);
	if(*modp==CM) yfeld=cmumrech(&start,&ende,pyfeld,&npkt);
	if(*irev==0)
	   {for(i=0;i<npkt/2;i++) {j=npkt-1-i; VERTAUSCHE(yfeld[i],yfeld[j]);}
	    VERTAUSCHE(start,ende);
	   }
	if(*modus>=0)
	   {modus2= *modus; *modus=npkt;
	    if(argflag['P']) modus2+=AUTOPLOTEXIT;
	    if(argflag['S']) modus2+=AUTOSAVEEXIT;
	    if(testflag) printf("showit(%lf,%lf,%lf,%lf,%d,%d,%08lX,%08lX\n",
				start,ymin,ende,ymax,npkt,modus2,(long)yfeld,(long)yhfeld);
	    showit(&start,&ymin,&ende,&ymax,&npkt,&modus2,yfeld,yhfeld);
	   }
       }
 else	{WORD mody2= *mody;
	 fclose(fp);
	 uvformat=UVALT; printf("Altes Fileformat\n");
//	 substanzname[0]=' '; substanzname[1]=0;// provi.
//	 if(katalogflag) inkatalog(filename,substanzname,NULL);
//	 uvshowfor(filename,substanzname,*pyfeld,*pyhfeld,mody,modp,irev,modus);
// neue Variante (ab 2.02): Aufruf von showit nicht mehr in uvshowfor
	 for(i=0;i<120;i++) substanzname[i]=' ';
/*
#ifndef unix
	 if(argflag['A']) // versuche unloesbares Problem zu loesen
	   uvshowfor_c(filename,substanzname,*pyfeld,*pyhfeld,&mody2,modp,irev);
	 else
#endif
*/
	   uvshowfor(filename,substanzname,*pyfeld,*pyhfeld,&mody2,modp,irev);
	 for(i=0;i<120-1;i++) if(substanzname[i]<' ') break;
	 substanzname[i]=0;
	 kopf.start=ISAM.ISTART*1000/20; kopf.ende=ISAM.IEND*1000/20;
	 kopf.delta=(int)(ISAM.DX*1e3+0.5)/20; kopf.slit=ISAM.SLIT*1e3;
	 kopf.period=(ISAM.PER-((int)ISAM.PER))*100.*1e3;
	 kopf.multi=(int)ISAM.PER;
	 kopf.conz=ISAM.CONZ*1e9;
	 kopf.yfaktor=8192*kopf.multi;
	 yfakt=(double)kopf.yfaktor;
	 konz=kopf.conz/1e9;
	 start=kopf.start/1e3; ende=kopf.ende/1e3;
	 npkt=(kopf.start-kopf.ende)/kopf.delta+1;
	 if(testflag) printf("start=%lf ende=%lf npkt=%ld\n",start,ende,(long)npkt);
	 if(kopf.conz==0 && (*mody & EPSILONMASK)) *mody=OD;
	 ymin=0.; ymax=0.001;
	 yfeld= *pyfeld; yhfeld= *pyhfeld;
	 for(py=yfeld,x=kopf.start; x>=kopf.ende; x-=kopf.delta,py++)
	   {if(*py<ymin) ymin= *py;
	    if(*py>ymax) ymax= *py;
	   }
	 h=powi(10.,(int)(log10(ymax)+9.)-9);
	 ymax=h*(int)(ymax/h+1.); /* ymax auf runde Zahl aufrunden */
	 if(ymin>(-ymax/35.)) ymin= -ymax/35.;
	 printf("ymin=%lf  ymax=%lf\n",ymin,ymax);
	 if(katalogflag) inkatalog(filename,substanzname,&kopf);
	 if(*modus>=0)
	   {modus2= *modus; *modus=npkt;
	    if(argflag['P']) modus2+=AUTOPLOTEXIT;
	    if(argflag['S']) modus2+=AUTOSAVEEXIT;
	    if(testflag) printf("showit(%lf,%lf,%lf,%lf,%d,%d,%08lX,%08lX\n",
				ende,ymin,start,ymax,npkt,modus2,(long)yfeld,(long)yhfeld);
	    showit(&ende,&ymin,&start,&ymax,&npkt,&modus2,yfeld,yhfeld);
	   }
/* :ende der neuen Variante */
	}
}
int wordlesen(FILE *fp)
{
 int c1,c2; WORD x;
 c1=getc(fp); c2=getc(fp);
 x=(c1<<8)+c2;
 return x;
}
LONG longlesen(FILE *fp)
{
 LONG c1,c2,c3,c4,x;
 c1=getc(fp); c2=getc(fp); c3=getc(fp); c4=getc(fp);
 x=(c1<<24)+(c2<<16)+(c3<<8)+c4;
 return x;
}

int gepacktlesen(FILE *fp,LONG *y)
{
 /* Wenn erstes Bit gesetzt (c & 0x80 == 0x80), dann sind die restlichen 7 Bits
    die Differenz zum vorherigen Wert.
    Wenn das erste Bit nicht, aber das 2.Bit gesetzt ist (c & 0xC0 == 0x40),
    dann sind die restlichen 6 Bits + das naechste Byte die Differenz zum
    vorherigen Wert.
    Wenn weder das erste noch das 2. Bit gesetzt sind (c & 0xC0 == 0), dann
    sind die restlichen 6 Bits + die naechsten 3 Bytes ein absoluter Wert.
 */
 static LONG letzterwert=0;
 LONG dy;
 int c,c2,c3,c4;
 if((c=getc(fp))==EOF) {*y=0; return EOF;}
 if(c & 0x80)
	{dy=c & 0x7F; if(dy & 0x40) dy |= 0xFFFFFF80;
	 *y=letzterwert+=dy;
	}
 else if(c & 0x40)
	{c2=getc(fp);
	 dy=((c & 0x3F)<<8)+c2; if(dy & 0x2000) dy |= 0xFFFFC000;
	 *y=letzterwert+=dy; c=c2;
	}
 else
	{c2=getc(fp); c3=getc(fp); c4=getc(fp);
	 dy=(c<<24)+(c2<<16)+(c3<<8)+c4; if(dy & 0x20000000) dy |= 0xC0000000;
	 *y=letzterwert=dy; c=c4;
	}
 return c;
}

double Log10a(double x)
{
 if(x<=0.) return -5.;
 return log10(x);
}

static int packflag=0;

void spektrum_speichern(char *filename,float *yfeld,WORD *mody,WORD *irev,
		   char *snam, struct uv_kopf *kopf)
{
 LONG x,y=0,j,i,npkt;
 FILE *fp;
 double yfak,konz;
 fp=fopen(filename,WRITE);
 if(fp==NULL) {printf("Fehler beim oeffnen von '%s'\n",filename); exit(0);}
 fprintf(fp,"UVOD2   "); /* Dateiformat = UV Optische Dichte Version 2 */
 fprintf(fp,"%s\n",snam);
 longspeichern(fp,kopf->start); longspeichern(fp,kopf->ende);
 longspeichern(fp,kopf->delta); longspeichern(fp,kopf->slit);
 intspeichern(fp,kopf->period); intspeichern(fp,kopf->multi);
 longspeichern(fp,kopf->conz);  longspeichern(fp,kopf->yfaktor);
 yfak=(double)kopf->yfaktor; konz=(double)kopf->conz;
 npkt=(kopf->start-kopf->ende)/kopf->delta+1;
 packflag=0;
 for(i=0,x=kopf->start; x>=kopf->ende; x-=kopf->delta,i++)
	{if(*irev==0) j=npkt-1-i; else j=i;
	 switch(*mody)
	  {case OD:	    y=idfix(yfeld[j]*yfak);
	   CASE EPSILON:    y=idfix(yfeld[j]*yfak*konz);
	   CASE LOGEPSILON: y=idfix(pow(10.,yfeld[j])*yfak*konz);
	   CASE SIGMA:      y=idfix(yfeld[j]*yfak*konz/0.3823);
	   CASE TRANSMISSION:y=idfix(-Log10a(yfeld[j]/100.)*yfak);
	   DEFAULT: printf("unbekanntes mody=%d\n",*mody);
	  }
	 gepacktspeichern(fp,y);
	}
 fclose(fp);
 return;
}

void uvschreib(char *filename,float *yfeld,WORD *mody,WORD *irev)
{
 //LONG *ylfeld;
 char *s;
 int i;
 switch(uvformat)
  {/*case UVALT: uvschrei(filename,yfeld,mody,irev); (vor 2.02)
     CASE UVOD2: s=substanzname[0];*/
   case UVALT: printf("Speichern im Format UVOD2\n");
   case UVOD2: s=substanzname[0];
	       if(ntesspektrum>2)
			{for(i=0;i<ntesspektrum-1;i++)
				printf("'%s'\n",substanzname[i]);
			 do {printf("Neue Bemerkungen:"); zeilelesen(s,120);}
			 while(*s==0);
			}
	       spektrum_speichern(filename,yfeld,mody,irev,s,&kopf);
   CASE SPEX2: s=substanzname[0];
	       if(ntesspektrum>2)
			{for(i=0;i<ntesspektrum-1;i++)
				printf("'%s'\n",substanzname[i]);
			 do {printf("Neue Bemerkungen:"); zeilelesen(s,120);}
			 while(*s==0);
			}
	       spex_spektrum_speichern(filename,yfeld,mody,irev,s,&kopf);
   DEFAULT: printf("unbekanntes UV-Format %d!\n",uvformat);
  }
}

/** uebernommen von uv.c: **/
void intspeichern(FILE *fp,int y) {putc(y>>8,fp);  putc(y,fp);}
void longspeichern(FILE *fp,LONG y)
{
 putc((int)(y>>24),fp); putc((int)(y>>16),fp);
 putc((int)(y>>8),fp);  putc((int)(y),fp);
}
void gepacktspeichern(FILE *fp,LONG y)
{
 static LONG letzterwert=0; LONG dy;
 dy=y-letzterwert;
 if(packflag==0 || dy>0x1FFF || dy<(-0x2000))
        {putc((int)((y>>24)&0x3F),fp); putc((int)(y>>16),fp);
         putc((int)(y>>8),fp);  putc((int)(y),fp);
        }
 else if(dy>0x3F || dy<(-0x40))
        {putc((int)((dy>>8)&0x3F)+0x40,fp); putc((int)(dy),fp);}
 else   putc((int)(dy&0x7F)+0x80,fp);
 letzterwert=y; packflag=1;
}

/************************ kleinkram **************************/
double powi(double x,int i)  /* Potenzieren einer Fliesszahl mit Integer */
{
 double z;
 if(i<0) return 1./powi(x,-i);
 for(z=1.;i>0;)
	{if(i & 1) z*=x;
	 if(i>>=1) x*=x;
	}
 return z;
}

void zeilelesen(char *str,int max)
{
 int c;
 if((c=getc(stdin))=='\n') c=getc(stdin);
 while(c!='\n' && --max) {*str++=c; c=getc(stdin);}
 *str=0;
 while(c!='\n') c=getc(stdin);
}

int hatpunkt(char *s)
{
 int c;
 while((c= *s++)!=0) if(c=='.') return 1;
 return 0;
}
void ohnepunkt(char *s)
{
 while(*s && *s!='.') s++;
 *s=0;
}

int getline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n' && c!=0)
	*s++ = c;
 *s='\0';
 if(lim==0)
   while(c!='\n' && c!=0 && c!=EOF) c=getc(fp);
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

/********************* Katalogdatei erstellen ***********************/
void inkatalog(char *filename,char *substanzname,struct uv_kopf *kopf)
{
 FILE *fp;
 double konz;
 fp=fopen("uvkatalog.dat","a"); if(fp==NULL) return;
 fprintf(fp,"%s\t'%s'\n",filename,substanzname);
 if(kopf)
	{fprintf(fp,"\t\t%.2lf bis %.2lf nm  ",kopf->start/1e3,kopf->ende/1e3);
	 fprintf(fp,"%.2lf nm/s  %d Scans\n",kopf->delta/1e3,kopf->multi);
	 fprintf(fp,"\t\tSlit: %.2lf nm   Period: %.1lf s\n",
		kopf->slit/1e3,kopf->period/1e3);
	 konz=kopf->conz/1e9;
	 if(konz==0.) fprintf(fp,"\t\tconz*d: unbekannt\n");
	 else fprintf(fp,"\t\tconz*d: %lg cm*Mol/Liter\n",konz);
	}
 fclose(fp);
}

/************************* ab Version 2.03 *******************************/
/* Problem:
	wenn ein altes UVFile gepackt wurde, oder mit SCOPY umkopiert
	dann ist RecordFormat immer 'Stream_LF' und RecordAttributes
	'Carriage return carriage control' gesetzt, was das Fortran
	offenbar nicht vertrgt.
	Dabei wurden anscheinend einige Bytes mit 0A berschrieben.
*/

void call_klar(FILE *fp);
void call_alp(FILE *fp,float *yfeld,int nstart,int nend,double start,
	      double ende,double ymin,double ymax,int modp,int mody,int nsmot);
void call_spitz(int n1,int n2,LONG *L);

int uvshowfor(char *filename,char *substanzname,
	      float *yfeld,float *hfeld,WORD *mody,WORD *modp,WORD *irev)
{
/** Umsetzung der Fortranroutine UVSHOWFOR.FOR **/
 int modpa;
 double start,ende,ymin,ymax,h;
 int nstart,nend,nsmot,npkt,i,j;
 FILE *fp;
 fp=fopen(filename,READ); if(fp==NULL) return 0;
 call_klar(fp);
 start=ISAM.ISTART/20.;
 ende=ISAM.IEND/20.;
 ymax=0.;
 if(start>10000) start=1e7/start;
 if(start>1000) start/=10.;
 if(ende>10000) ende=1E7/ende;
 if(ende>1000) ende/=10.;
 nsmot= -1;
 nstart=(ISAM.ISTART-start*20.)/ISAM.DX+1.5;
 nend=(ISAM.ISTART-ende*20.)/ISAM.DX+1.5;
 ymin=0.0;
 modpa=NM;
 call_alp(fp,yfeld,nstart,nend,start,ende,ymin,ymax,modpa,*mody,nsmot);
 fclose(fp);
 for(i=0;i<20;i++) substanzname[i]=ISAM.SNAM[i];
 substanzname[i]=0;
 npkt=nstart-nend;
 if(npkt<0) npkt= -npkt;
 npkt++;
 //if(*modp==CM) cmumrech(&start,&ende,yfeld,&npkt);
 if(*modp==CM) cmumrech(&start,&ende,&yfeld,&npkt);
 if(*irev==0)
	{for(i=0;i<npkt/2;i++) {j=npkt-1-i; VERTAUSCHE(yfeld[i],yfeld[j]);}
	 VERTAUSCHE(start,ende);
	}
 return 1;
}

void read_word(FILE *fp,WORD *x)
{
 int c1,c2;
 c1=getc(fp)&0xFF; c2=getc(fp);
 *x=c1+(c2<<8);
}
int read_word_uns(FILE *fp,WORD *x)
{
 int c1,c2,uns;
 c1=getc(fp); c2=getc(fp);
 if(altkor!=0)
	{if(c1!=0x0A) /* fehlendes 0A */
		{ungetc(c2,fp); c2=c1;
		 printf("Warnung: fehlendes 0x0A  c2=%02X\n",c2);
		 uns=0x80; c1=0x7F;
		}
	 else if(c2==0x0A) {c1=c2; c2=getc(fp); uns=0;}
	 else	{uns=0x80; c1=0x7F;}
	}
 else uns=0;
 *x=(c1&0xFF)+(c2<<8);
 return uns;
}
void read_float(FILE *fp,float *x)
{
 union {char c[4]; float z;} xu;
 xu.c[0]=getc(fp); xu.c[1]=getc(fp);
 xu.c[2]=getc(fp); xu.c[3]=getc(fp);
 *x=xu.z;
}
void read_lf(FILE *fp)
{
 int c;
 if(altkor==0) return;
 c=getc(fp);
 if(c!=0x0D)
	{printf("Warnung: c=%02X statt erwartetem 0D\n",c&0xFF);
	 if(altkor==3)
	 {while((c=getc(fp))!=EOF && c!=0x0D) printf(" %02X",c&0xFF);
	  printf("\n");
	  while((c=getc(fp))!=EOF && c!=0x0D) printf(" %02X",c&0xFF);
	  printf("\n");
	  return;
	 }
	 switch(c&0xFF)
	  {case 0x0A: ungetc(c,fp); return;
	   default: if((c=getc(fp))==0x0D) printf("naechstes war 0D\n");
		    else {ungetc(c,fp); printf("naechstes Byte: %02X\n",c);}
	  }
	 if(altkor==2)
	 {while((c=getc(fp))!=EOF && c!=0x0D) printf(" %02X",c&0xFF);
	  printf("\n");
	  while((c=getc(fp))!=EOF && c!=0x0D) printf(" %02X",c&0xFF);
	  printf("\n");
	 }
	}
}

void call_klar(FILE *fp)
{
 int i,n;
 WORD startunsicherheit;
 for(i=0;i<20;i++) ISAM.SNAM[i]=getc(fp);
 if(ISAM.SNAM[0]==0x0A) ISAM.SNAM[0]='_'; /* zerstrtes Zeichen ersetzen */
 ISAM.SNAM[19]=0;
 read_lf(fp);
 startunsicherheit=read_word_uns(fp,&ISAM.ISTART);
 read_word(fp,&ISAM.IEND);
 read_float(fp,&ISAM.DX);
 read_float(fp,&ISAM.PER);
 read_float(fp,&ISAM.SLIT);
 read_float(fp,&ISAM.CONZ);
 if(ISAM.IEND<3600)
	{ISAM.ISTART*=2; ISAM.IEND*=2; ISAM.DX*=2.; startunsicherheit*=2;}
 if(altkor!=0 || startunsicherheit!=0)
  {if(ISAM.ISTART<=ISAM.IEND) ISAM.ISTART += 0x7F;
   printf("Start: ca.%d +- %d  Ende=%d\n",
	ISAM.ISTART,startunsicherheit,ISAM.IEND);
   printf("richtiger Startwert:"); scanf("%d",&n); ISAM.ISTART=n;
  }
 printf("\n%s\n",ISAM.SNAM);
 printf("GESCANNTER BEREICH: %.2lf BIS %.2lf nm\n",
			ISAM.ISTART/20.,ISAM.IEND/20.);
 printf("SCANRATE : %.3lf      ANZAHL SCANS: %d\n",ISAM.DX/20.,(int)(ISAM.PER));
 printf("SLIT: %.2lf          PERIOD: %.1lf\n",
	ISAM.SLIT, (ISAM.PER-(int)(ISAM.PER))*100.);
 printf("KONZENTRATION*D: ");
 if(ISAM.CONZ==0.) printf("UNBEKANNT\n");
 else printf("%lg MOL/LITER *CM\n",ISAM.CONZ);
 printf("\n");
}

double absdoub(double x) {if(x<0.) return -x;  return x;}

void call_alp(FILE *fp,float *yfeld,int nstart,int nend,double start,
	      double ende,double ymin,double ymax,int modp,int mody,int nsmot)
{
 int i,j;
 LONG min,max;
 LONG istep,NE,NS;
 double faktor,X,Z,hs,xrand; //xmin,xmax;
 WORD wo;
 LONG L[8000];
 faktor=8192.*(int)(ISAM.PER);
/* if(ISAM.CONZ==0. || mody==1) mody=OD;  korrigiert am 3.11.95 */
 if((mody==EPSILON || mody==LOGEPSILON || mody==SIGMA) && ISAM.CONZ==0.)
	mody=OD;
 if(testflag) printf("call_alp(..mody=%d..)\n",mody);
 if(start < ende) istep= -1;
 else istep=1;
 if(modp==CM || modp==IN) {start=1E7/start; ende=1E7/ende;}
 for(i=(nstart-1)/10; i<=(nend-1)/10; i+=istep)
	{int i10=i*10,uns;
	 read_lf(fp);
	 uns=read_word_uns(fp,&wo); L[i10]=(LONG)wo;
	 for(j=1;j<10;j++)
		{read_word(fp,&wo); L[i10+j]=(LONG)wo;}
	 if(uns && i>0)
		{L[i10]=(L[i10-1]+L[i10+1])/2;
		}
	}
 if(istep == -1) call_spitz(nend,nstart,L);
 else	call_spitz(nstart,nend,L);
 min=2147483647; max= -2147483647;
 for(i=nstart;i<=nend;i++)
	{if(L[i] < min) min=L[i];
	 if(L[i] > max) max=L[i];
	}
 if(ymax==0.)
	{ymax=max/faktor;
	 if(modp==IN) ymax *= absdoub(start-ende)/2.;
	 ymax=mody_umrechnung(ymax,mody,ISAM.CONZ);
	 hs=pow(10.,(int)(log10(ymax)+9.)-9);
	 ymax=(int)(ymax/hs+1.)*hs; /* Aufrunden auf hbsche Zahl */
	 printf("ymax=%lf\n",ymax);
	}
 ymin=min/faktor;
 ymin=mody_umrechnung(ymin,mody,ISAM.CONZ);
 if(ymin > -ymax/35.) ymin= -ymax/35.;
 xrand=ISAM.DX/5.;
 if(start>1E4) xrand=2.;
 if(start<ende) xrand= -xrand;
 //xmin=ende-xrand; xmax=start+xrand;
 Z=0.;
 if(istep==1) {NS=nstart; NE=nend;}
 else	{NS=nend; NE=nstart;}
 for(i=NS-1;i<NE;i++)
	{X=(ISAM.ISTART-i*ISAM.DX)/20.;
	 if(modp!=IN)
		{Z=L[i]/faktor;
		 if(modp==CM) X=1E7/X;
/*		 if(modp==TR) Z=100.*pow(10.,-Z); alte Transmissionbehandlung */
		}
	else	{Z=Z+L[i]*(1E7/X-1E7/(X+ISAM.DX/20.))/faktor;
		 X=1E7/X;
		}
	 yfeld[i]=mody_umrechnung(Z,mody,ISAM.CONZ);
	}
}

void call_spitz(int n1,int n2,LONG *L)
{
 LONG iplus=0,L1,L0;
 int i;
 for(i=n1;i<n2;i++)
	{L1=L[i];
	 L0=L[i-1]-iplus;
	 if(L1 < L0-32768) iplus += 65536;
	 else if(L1>L0+32768) iplus -= 65536;
	 L[i]=L1+iplus;
	}
}

/************************ ab Version 2.04 ********************************/
void anhaengen(char *str,char *s,int max)
{
 char *t;
 int i;
 if(s==NULL) return;
 for(i=1,t=str;*t!=0;t++,i++) ;
 if(++i<max) *t++ = '\n';
 while(i<max && (*t++ = *s++)) ;
 if(i==max) *t=0;
}
void anhaengen2(char *str,char *s,int max)
{
 char *t;
 int i;
 if(s==NULL) return;
 for(i=1,t=str;*t!=0;t++,i++) ;
 if(++i<max) *t++ = ' ';
 if(++i<max) *t++ = ' ';
 while(++i<max && (*t++ = *s++)) ;
 if(i>=max) *t=0;
}

void error(int n)
{
 switch(n)
  {case ZU_WENIG_RAM: printf("ERROR: zu wenig RAM\n"); exit(n);
   case CANT_OPEN_NEW_FILE: printf("ERROR: cant open new file\n"); return;
   default: printf("ERROR: %d\n",n); exit(0);
  }
}

FILE *super_fopen(char *name,char *zugriff)
{
 FILE *fp;
 char *liste[]={".uv",".uvx",".spex",NULL}, **s;
 if(fp=fopen(name,zugriff)) return fp;
 if(!hatpunkt(name))
   for(s=liste;*s!=NULL;s++)
        {strcat(name,*s);
         if(fp=fopen(name,zugriff)) return fp;
         ohnepunkt(name);
        }
 return NULL;
}

int istvorhanden(char *name)
{
 FILE *fp;
 fp=super_fopen(name,"r");
 if(fp==NULL) {printf("'%s' nicht gefunden\n",name); return 0;}
 fclose(fp);
 return 1;
}

/** geht leider nicht !
#ifdef VAX_OR_ALPHA
void setenv(char *name,char *s)
{
 char str[80];
// sprintf(str,"def %s %s",name,s);
 sprintf(str,"setenv %s %s",name,s);
 printf("%s\n",str);// test
 system(str);
}
#endif

void plotsave_setzen(char *filename)
{
 char str[80];
 strcpy(str,filename);
 ohnepunkt(str);
 strcat(str,".ps");
 setenv("PLOTSAVE",str);
}
**/

/************************ ab Version 2.05 ********************************/
typedef double DOUBFUNK(double);
typedef DOUBFUNK *FUNKZEIGER;
void anpassung_machen(float *hf0,float *hf,FUNKZEIGER funktion);
void anpass_variablen_setzen();

#define MAXPOLYN 20

struct _funk_params
	{/*double x0,y0,a1,a2,aj[MAXPOLYN];*/
	 double x0,aj[MAXPOLYN];
	 int n;
	} funk_params;

#define N1 funk_params.n
#define AJ funk_params.aj
#define X0 funk_params.x0
/*
#define Y0 funk_params.y0
#define A1 funk_params.a1
#define A2 funk_params.a2
*/
#define Y0 funk_params.aj[0]
#define A1 funk_params.aj[1]
#define A2 funk_params.aj[2]

double funk_quadrat(double x)
{
 double xwert=x-X0;
 return A1*xwert*xwert+Y0;
}
double funk_polynom(double x)
{
 double xwert=x-X0,z;
 int i;
 for(i=1,z=Y0;i<N1;i++)
	z += AJ[i]*powi(xwert,i);
 return z;
}
double funk_exp(double x)
{
 double xwert=A1*(x-X0);
 if(xwert>85.1956) return 1e37;
 return A2*exp(xwert)+Y0;
}
double funk_1overx(double x)
{
 double xwert=x-X0;
 if(xwert==0.) return 1e37;
 return A1/xwert+Y0;
}
double funk_gauss(double x)
{
 double xwert=x-X0;
 xwert=A1*xwert*xwert;
 if(xwert>85.1956) return 1e37;
 return A2*exp(xwert)+Y0;
}

void funk_anpass()
{
 float *hf,*hf0;
 double x,dx;
 static double sigma=0.,d=0.;
 static FUNKZEIGER funktion=NULL;
 static int neuflag=1,altes_n='2';
 int i,funkdefflag=0,n;
 char *antw=fname2[ntesspektrum-2];
 if((n= *antw)!='1')
  {printf("  Funktionen anpassen:\n");
   if(n!='?') printf("?: interaktive Auswahl\n");
   printf("1: letztes Spektrum verwerfen\n");
   printf("2: Quadrat.Funktion: y=A*(x-x0)^2+C\n");
   printf("3: Polynom Funktion: y=Aj*(x-x0)^j+C\n");
   printf("4: Exponential Funk: y=A2*e^(A1*(x-x0))+C\n");
   printf("5: Kehrwertfunktion: y=A*1/(x-x0)+C\n");
   printf("6: Gaussglocke: y= A2 * e^(A1*(x-x0)) + C\n");
   printf("          oder: y= 1/(sigma*sqrt(2PI)) * e^(-1/2d*(x-x0))\n");
   printf("P: Parameter ndern\n");
   printf("N: Neue Kurve erzeugen\n");
   printf("A: Anpassungs-Variablen setzen, dann Anpassung vornehmen\n");
   printf("0: Anpassung vornehmen\n");
  }
 if(n=='?') {printf("Auswahl:"); scanf("%s",antw);}
 else printf("Auswahl:%c\n",n);
 machgross(antw); n= *antw;
 if(n=='P')
	{if(--ntesspektrum<2) ntesspektrum=2;
	 neuflag=1; n=altes_n;
	 printf("FunktionAuswahl = %c\n",n);
	}
 switch(n)
  {case '1':
	ntesspektrum-=2;
	if(ntesspektrum<1) ntesspektrum=1;
	neuflag=1;
	break;
   case '2':
	funktion=funk_quadrat; N1=2;
	printf("Konstanten eingeben:\n");
	printf("     Faktor A=?"); scanf("%lf",&A1);
	funkdefflag=1;
	break;
   case '3':
	funktion=funk_polynom;
	printf("Konstanten eingeben:\n");
	printf("Hchste Polynompotenz:"); scanf("%d",&N1); N1++;
	if(N1>MAXPOLYN) printf("N1 auf %d beschrnkt\n",N1=MAXPOLYN);
	for(i=1;i<N1;i++)
	 {printf("    Faktor A%d=?",i); scanf("%lf",&AJ[i]);}
	funkdefflag=1;
	break;
   case '4':
	funktion=funk_exp; N1=3;
	printf("Konstanten eingeben:\n");
	printf("    Faktor A1=?"); scanf("%lf",&A1);
	printf("    Faktor A2=?"); scanf("%lf",&A2);
	funkdefflag=1;
	break;
   case '5':
	funktion=funk_1overx; N1=2;
	printf("Konstanten eingeben:\n");
	printf("     Faktor A=?"); scanf("%lf",&A1);
	funkdefflag=1;
	break;
   case '6':
	funktion=funk_gauss; N1=3;
	printf("Konstanten eingeben:\n");
	if(A2>0.) sigma=1./A2/sqrt(PI+PI); else sigma=0.;
	if(A1<0.) d=sqrt(-0.5/A1); else d=0.;
	printf("  Eingestellt: A1=%lf oder d=%lf  A2=%lf oder sigma=%lf\n",
		A1,d,A2,sigma);
	printf("    1. Faktor (A1 oder d) :"); scanf("%lf",&A1);
	printf("    2. Faktor A2 :"); scanf("%lf",&A2);
	sigma=1./A2/sqrt(PI+PI);
	if(A1<0.) d=sqrt(-0.5/A1);
	else {d=A1; A1= -0.5/(d*d);}
	printf(" A2=%lf Sigma=%lf  A1=%lf d=%lf\n",A2,sigma,A1,d);
	funkdefflag=1;
	break;
   case 'A':
	anpass_variablen_setzen();
   case '0':
	if(--ntesspektrum<2 || funktion==NULL)
	  {printf("zuerst Funktion definieren\n"); break;}
	hf= &hilfsfeld[(ntesspektrum-2)*npkt];
	if(ntesspektrum>=3) hf0= &hilfsfeld[(ntesspektrum-3)*npkt];
	else hf0= hauptfeld;
	anpassung_machen(hf0,hf,funktion);
	neuflag=1;
	break;
   case 'N':
	neuflag=1;
	break;
   default: ntesspektrum--;
  }
 if(funkdefflag)
	{if(antw[1]=='N') neuflag=1;
	 if(neuflag==0) ntesspektrum--;
	 neuflag=0;
	 hf= &hilfsfeld[(ntesspektrum-2)*npkt];
	 printf(" (eingestellt: C=%lf x0=%lf\n",Y0,X0);
		 printf("      (OD)  C=?"); scanf("%lf",&Y0);
	 if(modp==CM) /* ab Version 2.06 */
		 printf("     (cm-1) x0=?");
	 else	 printf("      (nm)  x0=?");
	 scanf("%lf",&X0);
	 printf("Substanzname:"); scanf("%s",substanzname[ntesspektrum-1]);
	 if(modp==CM) /* ab Version 2.06 */
	   {double x2,h;
	    x=1e10/kopf.start; x2=1e10/kopf.ende;
	    if(irev==0) {VERTAUSCHE(x,x2);}
	    dx=(x2-x)/npkt;
	    for(i=0;i<npkt;i++,x+=dx)
		 hf[i]=(*funktion)(x);
	   }
	 else
	   for(i=0,x=kopf.ende*0.001,dx=kopf.delta*0.001;i<npkt;i++,x+=dx)
		hf[i]=(*funktion)(x);
	 altes_n=n;
	}
 if(n!='1') printf("\n");
 if(ntesspektrum<2) modus=1;
 else modus=4;
 if(ntesspektrum>2) modus+=ntesspektrum;
 new_hfeld(hilfsfeld,(int*)&modus);
 ntesspektrum++;
}

/********************** Anpassungsalgorythmus ****************************/
#ifdef VAX_OR_ALPHA
double random()         /* Zufallszahl zwischen -1. und 1. */
{
 return (1. - rand()/1073741824.);
}
#endif

static struct _funk_params alte_params;
static double anpass_delta=1e-6;
static int anpass_imax=100,anpass_stark=0;

#define input(s,x,s2) printf(s,x);scanf(s2,&x);

void anpass_variablen_setzen()
{
 input("(%lf) anpass_delta:",anpass_delta,"%lf");
 input("(%d) anpass_imax:",anpass_imax,"%d");
 input("(%d) anpass_stark: (0,1) :",anpass_stark,"%d");
}

void parameter_aendern(double delta)
{
 int j;
 alte_params.x0=X0;
 X0 += random()*delta;
 for(j=0;j<N1;j++)
	{alte_params.aj[j]=AJ[j];
	 AJ[j] += random()*delta;
	}
}
void parameter_stark_aendern(double delta)
{
 int j;
 alte_params.x0=X0;
 X0 *= 1.+random()*delta;
 for(j=0;j<N1;j++)
	{alte_params.aj[j]=AJ[j];
	 AJ[j] *= 1.+random()*delta;
	}
}
void parameter_zurueck()
{
 int j;
 X0=alte_params.x0;
 for(j=0;j<N1;j++) AJ[j]=alte_params.aj[j];
}
void parameter_testausdruck()
{
 int j;
 printf(" X0=%lg  ",X0);
 for(j=0;j<N1;j++) printf("A%d=%lg  ",j,AJ[j]);
 printf("\n");
}

double chiquadrat(float *hf0,FUNKZEIGER funktion)
{
 double chiq=0.,chi,x,y,dx;
 int i;
 for(i=0,x=kopf.ende*0.001,dx=kopf.delta*0.001;i<npkt;i++,x+=dx)
   if((y= *hf0++)!=0.)
	{chi=y-(*funktion)(x);
	 chiq += chi*chi;
	}
 return chiq;
}

void anpassung_machen(float *hf0,float *hf,FUNKZEIGER funktion)
{
 int i,flag,besser;
 double x,dx,abweichung,abw2,delta;
/*
  funk_params irgendwie anpassen so dass (*funktion)(x) mglichst gut
  mit hf0[] bereinstimmt
*/
 printf("Parameter vor Anpassung:");/* test */
 parameter_testausdruck();
 if(anpass_stark)
  {abweichung=chiquadrat(hf0,funktion);
   do for(besser=0,delta=1.;delta>=anpass_delta;delta/=2.)
	{parameter_stark_aendern(delta);
	 abw2=chiquadrat(hf0,funktion);
	 if(abw2>=abweichung) {parameter_zurueck();}
	 else	{abweichung=abw2; besser++;}
	}
   while(besser);
  }
 do
 {abweichung=chiquadrat(hf0,funktion);
  for(besser=0,delta=1.;delta>=anpass_delta;delta/=2.)
   {
    for(flag=0,i=0;i<anpass_imax;i++)
	{parameter_aendern(delta);
	 abw2=chiquadrat(hf0,funktion);
	 if(abw2>=abweichung) parameter_zurueck();
	 else	{abweichung=abw2;
		 if(++flag==1) {parameter_testausdruck(); besser++;}
		}
	}
   }
 } while(besser>=anpass_imax);
 printf("Parameter nach Anpassung:");/* test */
 parameter_testausdruck();
 for(i=0,x=kopf.ende*0.001,dx=kopf.delta*0.001;i<npkt;i++,x+=dx)
		hf[i]=(*funktion)(x);
}

/************************* ab Version 2.08 *******************************/
#ifdef VAX_OR_ALPHA
void kill(char* name)
{
 char str[256];
 sprintf(str,"del %s;*",name);
 system(str);
}
#endif

/*************************** SPEX2-Format ********************************/
LONG picometer(char *zeile)
{
 char *s;
 int c,c2;
 double x=0;
 for(s=zeile;(c= *s++) && (isdigit(c) || c=='.' || c==' ');) ;
 if(c!=0) {c=toupper(c); c2=toupper(*s);}
 sscanf(zeile,"%lf",&x);
 if(c=='A' && c2=='U') x*=0.1; //von Angstrom in nm umrechnen
 else if(c=='C' && c2=='M') x=1e7/x; //von cm-1 in nm umrechnen
 return (LONG)(1000*x+0.5);
}
long nlesen(char *zeile)
{
 long x=0;
 sscanf(zeile,"%ld",&x);
 return x;
}
double xlesen(char *zeile)
{
 double x=0;
 sscanf(zeile,"%lf",&x);
 return x;
}

static char* einheiten[] =
 {"COUNTS","CTS","Counts","counts",
  "CPS","CTS/S","CTS/SEC","COUNTS/SEC",
  "OD","OPT.DENS","OPTDENS","OptischeDichte",
  "Torr","TORR","torr","mmHg",
  "mTorr","milliTorr","MILLITORR","millitorr",
  "mbar","millibar","MBAR","hectopascal",
  "mol/l","molproliter","mol","MOL/L",
  NULL
 };
#define UNIT_COUNTS 1
#define UNIT_CPS   2
#define UNIT_OD    3
#define UNIT_TORR  4
#define UNIT_MTORR 5
#define UNIT_MBAR  6
#define UNIT_MOL   7

#ifdef __cplusplus
double molproliter(double x,int einh,double t=298)
#else
double molproliter(double x,int einh,double t)
#endif
{
 double z;
 const double k=0.0160;
 switch(einh)
  {case UNIT_MTORR: x*=0.001;
   case UNIT_TORR: z=x*k/t;
   CASE UNIT_MBAR: z=x*(k/1.333)/t;
   CASE UNIT_MOL: default: z=x;
  }
 return z;
}

double xlesen2(char *zeile,int *einheit)
{
 double x=0;
 char *s,*p;
 int einh,i,c;
 sscanf(zeile,"%lf",&x);
 for(s=zeile;(c= *s++) && isspace(c);) ;
 if(isdigit(c) || c=='-')
   {while(c!=0 && !isspace(c)) c= *s++;
    if(c!=0) c= *s++;
   }
 if(c==0)  einh=0;
 else
   {for(--s,i=0;(p=einheiten[i])!=NULL;i++)
      if(strcmp(p,s)==0) break;
    einh=i/4+1;
   }
 *einheit=einh;
 return x;
}

#include "spex2.h"

void spex_parameter_lesen(int c,FILE *fp)
{
 static double temp=298,path=1,conz=0;
 double x;
 char zeile[120],txt[80];
 int einheit;
 static int rep_number=1; //neu 24.7.2000
 static double rep_time=0.1, mes_time=0.01; //neu 24.7.2000
 getline(fp,zeile,120);
 if(c<0 || c>=30) printf("Fehler: spexstring %d ungueltig\n",c);
 else if(c==19 || c==18 || c==2 || c==23)
   printf("%2d %s  %s\n",c,ptext[c],zeile);//wichtigste SPEX-Parameter zeigen
 switch(c)
  {case 0:  ++nspe;
	    if(strncmp(zeile,"OLGA",4)==0) spektrumtyp[nspe]=OLGA;
	    else if(strncmp(zeile,"MOPS",4)==0) spektrumtyp[nspe]=MOPS;
	    else spektrumtyp[nspe]=0;
	    if(strlen(zeile)>=6 && (zeile[5]=='c' || zeile[5]=='C'))
	      {spektrumtyp[nspe] += CORRECTED; corrflag=1;}
	    if(argflag['X'])
	     {if((spektrumtyp[nspe]&OLGA)==0)
	       {printf("kann bisher nur OLGA-Spektren korrigieren\n");
	        corrflag=1;
	       }
	      else if(corrflag)
		printf("OLGA corrected - Eichkurve wurde schon beruecksichtigt\n");
	     }
	    spex_par_clear();
	    // hier kein "break;" !
   case 1:  temp=298.0; path=1.0; conz=0.0;
   CASE 2:  strcpy(substanzname[nspe],zeile);
   CASE 5:  sprintf(OLGA_TEXT[nspe],"Exciting: %s",zeile); //neu 24.7.2000
   CASE 7:  temp=xlesen(zeile);
   CASE 8:  conz=xlesen2(zeile,&einheit);
	    kopf.conz=idfix(1e9*molproliter(conz,einheit,temp));
   CASE 18: path=xlesen(zeile);
	    if(path<1e-9)
	      {if(strncmp(zeile,"Gas",3)==0) path=9.8;
	       else path=1.0;
	      }
	    else printf("path > 1e-9\n");
	    kopf.conz=idfix(1e9*molproliter(conz,einheit,temp)*path);
   CASE 20: kopf.start=picometer(zeile);
   CASE 21: kopf.ende=picometer(zeile);
   CASE 22: kopf.delta=picometer(zeile);
   CASE 16: kopf.slit=picometer(zeile);//provi.
   CASE 23: strcpy(txt,OLGA_TEXT[nspe]); sprintf(OLGA_TEXT[nspe],"%s, %s",zeile,txt);
   CASE 24: kopf.multi=nlesen(zeile);
   CASE 25: yfaktor2=xlesen(zeile);
   CASE 26: sscanf(zeile,"%d",&rep_number);//nur relevant mit LASERMODUS==ON
   CASE 27: sscanf(zeile,"%lf",&rep_time);//nur relevant mit LASERMODUS==ON
   CASE 28: x=xlesen(zeile); mes_time=x;
	    kopf.period=idfix((x>32.767)?-x:1000*x);
   CASE 29: if(strcmp("ON",zeile)==0 || strcmp("on",zeile)==0)
	     {lasermodus=true;
	      if(rep_time>1) sprintf(LASER_TEXT[nspe],
		      "%d Laserpulses at %.2lfHz with %.0lfms Gate",
		      rep_number, 1.0/rep_time, 1e3*mes_time/rep_number);
	      else sprintf(LASER_TEXT[nspe],
		      "%d Laserpulses at %.0lfHz with %.0lfms Gate",
		      rep_number, 1.0/rep_time, 1e3*mes_time/rep_number);
	     }
  }
 spex_par_copy(c,zeile);
}

char *spektrum_typ(int n)
{
 static char txt[80];
 switch(spektrumtyp[n])
   {case OLGA: return OLGA_TEXT[n]; //"OLGA";
    case MOPS: return "MOPS";
    case OLGA+CORRECTED: sprintf(txt,"%s, corrected",OLGA_TEXT[n]); return txt;
//"OLGA corr";
    case MOPS+CORRECTED: return "MOPS corr";
   }
 return " ";
}

void mehrere_files_laden(char *name)
{
 char scriptname[80],muname[80];
 FILE *fp;
 int i;
 if(*name=='@') name++;
 if(!hatpunkt(name)) sprintf(scriptname,"%s.script",name);
 else strcpy(scriptname,name);
 if(!(fp=fopen(scriptname,"r")))
	{printf("'%s' nicht gefunden\n",scriptname); return;}
 fscanf(fp,"%d",&i);
 if(i<1 || i>=MAXSPEKT)
       {printf("Scriptfile muss als erste Zahl Anzahl Spektren enthalten !\n");
	printf(" (i=%d ist ungueltig, i sollte zwischen 1 und %d liegen)\n",
		i,MAXSPEKT);
	return;
       }
 while(--i>=0)
  {fscanf(fp,"%s",muname);
   file_load_mu(muname);
  }
 fclose(fp);
}

/******************** speichern im SPEX2-Format **************************/
static STRING80 spex_parameter[31];

void spex_par_clear()
{
 int j;
 for(j=0;j<31;j++) *spex_parameter[j]=0;
}

void spex_par_copy(int i,char *zeile)
{
 int j,c;
 char *s=(char*)spex_parameter[i];
 for(j=1;j<80 && (c= *zeile++);j++) *s++ = c;
 *s=0;
}

int istzahl(char *s)
{
 return isdigit(*s) || (*s++ =='-' && isdigit(*s));
}

void spex_spektrum_speichern(char *filename,float *yfeld,WORD *mody,WORD *irev,
		   char *snam, struct uv_kopf *kopf)
{
 FILE *fp;
 double yfak; //,konz;
 LONG npkt,j;
 if(testflag) printf("spex_spektrum_speichern('%s',yfeld,mody=%d,irev=%d,snam='%s')\n",
		     filename,*mody,*irev,snam);//test
 spex_par_copy(1,filename);
 spex_par_copy(2,snam);
 fp=fopen(filename,WRITE);
 if(fp==NULL) {printf("Fehler beim oeffnen von '%s'\n",filename); return;}
 fprintf(fp,"SPEX2   "); /* Dateiformat = SPEX Version 2 */
 for(j=0;j<31;j++)
   if(*spex_parameter[j]) fprintf(fp,"%c%s%c",j,(char*)spex_parameter[j],0);
 putc(255,fp);
 packflag=0;
 if(istzahl(spex_parameter[25])) sscanf(spex_parameter[25],"%lf",&yfak);
 else yfak=1.; //provi.: noch nicht getestet!
 if(yfak==0.) yfak=1.;
 packflag=0;
 npkt=(kopf->start-kopf->ende)/kopf->delta+1;
 for(j=npkt; --j>=0;)
     gepacktspeichern(fp,idfix(yfak*yfeld[j]));
 fclose(fp);
}
