/* xyshow.cc		letzte Aenderungen: 14.1.2022 */
#define VERSION "Version 1.56"
/*
 OpenGL-Version von xyshow
 xy-Dateien anzeigen, und auch NMR-Spektren

Autor: Rolf Pfister
Lizenz: Opensource Freeware

History, Version, Remarks:
31.12.2021  1.55  Erstellung aus xyshow.c Version 1.55
14.1.2022   1.56  Anpassungen an Windows gemacht

*/

#include "otekplot1.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "showitlight.h"
#include "ulong.h"
#ifndef PI
#define PI 3.14159265358979
#endif
#define cfree free

#include <sys/types.h>
#include <sys/stat.h>
#define MAXARG 2
#define MAXSPEKT 100
inline long idfix(double x) {return (x>=0.)?x+0.5:x-0.5;}

#define CONTROLZ 0x1A
#define NEUES_NMR_2018 //xmin und xmax zuerst in "procs" suchen

#ifdef _WIN32
#define UNCONST
#endif
#ifdef UNCONST
char *unconst(const char *s)
{
 int n=strlen(s)+1;
 char *str=new char[n];
 strcpy(str,s);
 return str;
}
#else
#define unconst(s) (char*)(s)
#endif

/************************ Vordeklarationen *******************************/
#define getline mygetline
int getline(FILE *fp,char *s,int lim);
int nextc(FILE *fp);
FILE *neuoeffnen(FILE *fp,char *name);
void incmumrechnen(float **feld,int *npkt,float *xmin,float *xmax,int flag);
int starkverschieden(double a,double b);
int daten_einlesen(char *name,float *feld,int gxflag);
double tmspeaksuchen(FILE *fp);
double tmspeaksuchenb(FILE *fp,double xmin,double ddx);
void file_load_mu(char *muname);
double lesezahl(FILE *fp);
void mehrere_files_laden(char *name);

/*************************** globale Parameter ************************/
char dateiname[80],dateiname2[80],argflag['Z'+1];
FILE *fpw=NULL;/* fuer ausgemessene Punkte und Integrale */
static int npoints=0; //Anzahl Stuetz-Punkte
static int methode=1; //Methode zum Interpolieren (1=linear)
static int modus;
static bool msbfirst=1;//neu ab Version 1.51
static int u1neu=2;

void setflags(char *s)
{
 int c;
 while((c= *s++)!=0)
  {if((c=toupper(c))<='Z')  argflag[c]=1;
   if(c=='N') {u1neu=atoi(s); while(isdigit(*s)) s++;}
  }
}

/*** fuer mehrere Spektren: ***/
static float gxmin=0,gxmax=0,gymin=0,gymax=0;
static double gdx;
static int nspektren=0;
static float *hauptfeld=NULL,*hilfsfeld=NULL;//Felder zur Uebergabe ins Showit
typedef char STRING40[40];
typedef char STRING80[80];
STRING80 filename[MAXSPEKT];//Dateiname zu jedem Spektrum
typedef char STRING[120];
STRING substanzname[MAXSPEKT];//Substanzname zu jedem Spektrum
void substanzname_init()
 {int i; for(i=0;i<MAXSPEKT;i++) sprintf(substanzname[i],"Spektrum%d",i+1);}

/***************************** Kleinkram ******************************/
#define CANT_OPEN_NEW_FILE 1
#define ZU_WENIG_RAM 2

void error(int n)
{
 switch(n)
  {case ZU_WENIG_RAM: printf("ERROR: zu wenig RAM\n"); exit(1);
   case CANT_OPEN_NEW_FILE: printf("ERROR: cant open new file\n"); return;
   default: printf("ERROR: %d\n",n); exit(1);
  }
}

int hatpunkt(const char *s)
{
 int c;
 while((c= *s++)!=0) {if(c=='.') return 1;}
 return 0;
}
void ohnepunkt(char *s)
{
 while(*s && *s!='.') s++;
 *s=0;
}

FILE *super_fopen(char *name,const char *zugriff)
{
 FILE *fp;
 //char *liste[]={unconst(".xy"),NULL}, **s;
 const char *liste[]={".xy",NULL}, **s;
 if((fp=fopen(name,zugriff))!=NULL) return fp;
 if(!hatpunkt(name))
   for(s=liste;*s!=NULL;s++)
        {strcat(name,*s);
         if((fp=fopen(name,zugriff))!=NULL) return fp;
         ohnepunkt(name);
        }
 return NULL;
}

//#define TEST_WIN32
#define TEST_STAT

//#ifdef _WIN32
#ifdef TEST_WIN32
#define _stat stat //test
int letzteszeichen(const char *str)
{
 int n=strlen(str);
 return (str[n-1]&0xFF);
}

bool istordner(const char *name)
{
#ifdef TEST_STAT
 struct _stat buf;
 if(_stat(name,&buf)<0) return 0;
 return ((buf.st_mode & _S_IFDIR)!=0);
#else
 //TODO
 //privisorisch: wenn name keinen Punkt enthaelt oder mit z.B. ".001" oder
 //              mit "/" endet, dann ist es wahrscheinlich ein Ordner
 if(!hatpunkt(name)) return true;
 if(letzteszeichen(name)=='/') return true;
 const char *s;
 for(s=name; *s != '.'; s++) {}
 return (strlen(s)>=4 && isdigit(s[1]));
#endif
}
bool istlink(const char *dateiname)
{
#ifdef TEST_STAT
 struct _stat buf;
 if(_stat(dateiname,&buf)<0) return 0;
 return ((buf.st_mode & (_S_IFDIR|_S_IFREG))==0);
#else
 return false; //TODO
#endif
}
#else
bool istordner(const char *name)
{
 struct stat buf;
 if(lstat(name,&buf)<0) return 0;
 return (S_ISDIR(buf.st_mode));
//fuer Inhalt von buf siehe "man lstat"
//oder more /usr/include/sys/stat.h
}
bool istlink(const char *dateiname)
{
 struct stat buf;
 if(lstat(dateiname,&buf)<0) return 0;
 if((buf.st_mode & S_IFMT)==S_IFLNK) return 1;
 return 0;
}
#endif

int istvorhanden(char *name)
{
 if(istordner(name)) return 1;
 FILE *fp;
 fp=super_fopen(name,"rb");
 if(fp==NULL) return 0;
 fclose(fp);
 return 1;
}

bool hatendung(const char *name,const char *endung)
{
 int n=strlen(name)-strlen(endung);
 if(n<0) return 0;
 return strcmp(&name[n],endung)==0;
}

bool istscript(char *name)
{
 char c;
 if(*name=='@') return 1;
 while((c= *name++)!=0)
   if(c=='.' && strncmp(name,"scr",3)==0) return 1;
 return 0;
}

int filelaenge(char *name)
{
 int n=0;
#ifdef _WIN32
 FILE* fp=fopen(name,"rb");
 if(fp==NULL) return 0;
 while(getc(fp)!=EOF) {n++;}
 fclose(fp);
#else
 struct stat buf;
 if(lstat(name,&buf)<0) return -1;
 if(!S_ISREG(buf.st_mode)) return -2;
 //fuer Inhalt von buf siehe "man lstat"
 //if(flag==BLOCKS) n=buf.st_blocks;
 //else if(flag==BSIZE) n=buf.st_blksize;
 //else
 n=buf.st_size;
#endif
 return n;
}

int automsbfirst(char *name)
{
 unsigned char z[8];
 int i, ord=0;
 FILE *fp=fopen(name,"rb");
 if(fp==NULL) return 0;
 for(i=0;i<8;i++) z[i]=getc(fp);
 if((z[0]==0 || z[0]==0xFF) && (z[4]==0 || z[4]==0xFF)) ord=1;
 fclose(fp);
 return ord;
}

void feldalloc(int n,float **pfeld)
{
 *pfeld=(float*)calloc(n,sizeof(float));
 if(*pfeld==NULL) error(ZU_WENIG_RAM);
 return;
}

void anhaengen(char *str,char *s,int max)
{
 char *t;
 int i;
 if(s==NULL) return;
 for(i=1,t=str;*t!=0;t++,i++) ;
 if(++i<max) *t++ = '\n';
 while(i<max && (*t++ = *s++)) ;
 if(i==max) *t=0;
}
void anhaengen2(char *str,char *s,int max)
{
 char *t;
 int i;
 if(s==NULL) return;
 for(i=1,t=str;*t!=0;t++,i++) ;
 if(++i<max) *t++ = ' ';
 if(++i<max) *t++ = ' ';
 while(++i<max && (*t++ = *s++)) ;
 if(i>=max) *t=0;
}

double anpassen(double *px0,float xmin,float xmax)
{
 double dx=gdx,x0=gxmin;
 if((gxmax>gxmin && xmax<xmin) ||
    (gxmax<gxmin && xmax>xmin))  {dx= -gdx; x0=gxmax;}
 *px0=x0;
 return dx;
}

double parser(FILE *fp,const char *str)
{
 int c=0,i=0,n=strlen(str);
 double x;
 while(c!=EOF && i!=n)
   {while((c=getc(fp))!=EOF && c!= *str) {}
    for(i=1;i<n;i++)
      if(str[i]!=(c=getc(fp))) break;
   }
 fscanf(fp,"%lf",&x);
 return x;
}

#define index2 myindex2 //TODO

int index2(const char *s1,const char *s2) // Sucht den String s2 innerhalb von s1 und
{			     // gibt Position zurueck (nicht gefunden: -1)
 int i,c;
 const char *p1,*p2;
 if(*s2==0) return 0;   // leerer String ist immer enthalten
 for(i=0;;i++)
	{if((c= *s1++)==0) return -1; // nicht gefunden
	 if(c== *s2)
	        {for(p1=s1,p2=s2; (c= *++p2)!=0;)
			if(*p1++!=c) break; // noch nicht gefunden
		 if(c==0) break; // gefunden
		}
	}
 return i;
}

/***************************** Hauptprogramm **************************/
int main(int argc,char *argv[])
{
 check_glversion(argc,argv);
 int i,j,c,n;
 for(dateiname[0]=0,j=0,i=1;i<argc;i++)
   {if((c= *argv[i])=='-' || c=='?') setflags(argv[i]);
    else if(++j==1) strcpy(dateiname,argv[i]);
    else if(j==2) strcpy(dateiname2,argv[i]);
   }
 if(argflag['?'] || j>MAXARG)
	{printf("xyshow  %s\n",VERSION);
	 printf("Anwendung: xyshow Datei [Datei2] [-Flags]\n");
	 printf("   Flags: R=Rueckwaerts  S=Skalierung\n");
	 printf("          C=von nm in cm-1 umrechnen\n");
	 printf("          G=Grenzen neu setzen\n");
	 printf("          A=alles Automatisch\n");
	 printf("          H=Hilfe fuer Dateiformat\n");
	 printf("          V=Verbose, also mit Testausdrucken\n");
	 printf("          T=TMS-Peak suchen, Grenzen automatisch neu setzen\n");
	 printf("          U=Referenz-Peak suchen, Grenzen automatisch neu setzen\n");
	 printf("          N2=NMR 2. Spektrum (z.B. 13C-NMR) benutzen\n");
	 printf("          P=Datei in aktuellem Format speichern, dann exit\n");
	 printf("          W=Byteorder umkehren (falls automsbfirst misslingt)\n");
	 exit(0);
	}
 if(argflag['H'])
  {printf("xyshow  %s\n",VERSION);
   printf("Bisher unterstuetzte Dateiformate:\n\n");
   printf(" Primitives Format:\n");
   printf("  x1 y1  x2 y2  ...\n\n");
   printf(" Primitives Format mit Grenzwertangabe:\n");
   printf("  xmin=.. xmax=.. nx=..\n");
   printf("  ymin=.. ymax=.. dy=..\n");
   printf("  x1 y1  x2 y2  ...\n");
   printf(" Fuer ymax sollte eine Zahl groesser als maximaler Eingabewert,\n");
   printf(" fuer ymin eine Zahl kleiner minimaler Eingabewert,\n");
   printf(" und fuer dy die Differenz ymax-ymin gewaehlt werden.\n");
   printf(" Wenn die y-Eingabewerte umgerechnet werden sollen, kann man ymin\n");
   printf(" und dy entsprechend veraendern (z.B. bei eingescannten Kurven)\n");
   printf(" \n");
   printf(" Gekennzeichnetes Format:\n");
   printf("  xydata11   ;Kommentar hinter Strichpunkt\n");
   printf("  xmin=.. xmax=.. nx=..\n");
   printf("  ymin=.. ymax=.. dy=..\n");
   printf("  nkurven    ;Anzahl Kurven (ganze Zahl)\n");
   printf("  npu1       ;Kurve1 ;Anzahl Punktepaare\n");
   printf("  x1 y1  x2 y2  ...\n");
   printf("  npu2       ;Kurve2 ;Anzahl Punktepaare der zweiten Kurve\n");
   printf("  x1 y1  x2 y2  ...\n");
   printf("  ...\n");
   printf(" nx darf auch groesser sein als wirklich vorhandene Punkte.\n");
   printf("\nSeit Version 1.42 werden auch noch NMR-Daten unterstuetzt:\n");
   printf(" (Ausgelesen von ordner/1/pdata/1/1r und peak.txt oder proc)\n");
   exit(0);
  }
 substanzname_init();
 if(dateiname[0]==0) {printf("Datei:"); scanf("%s",dateiname);}
 if(istscript(dateiname)) {printf("Script-Datei direkt lesen geht (noch) nicht\nLoesung: nur erste Datei laden, dann mit file_load die Script-Datei.\n"); exit(0);}
 n=daten_einlesen(dateiname,hauptfeld,1);
 if(n>0) nspektren=n;
 if(*dateiname2!=0)
   {if(hilfsfeld==NULL) feldalloc((MAXSPEKT-1)*npoints,&hilfsfeld);
    n=daten_einlesen(dateiname2,&hilfsfeld[(nspektren-1)*npoints],1);
    if(n>0)  nspektren += n;
   }
 if(nspektren==0) {printf("Fehler: keine Spektren gefunden\n"); exit(0);}
 if(nspektren>=2)
	{if(nspektren==2) modus=4; else modus=4+nspektren;}
 else	{modus=3;}
 if(argflag['P']) modus+=AUTOSAVEEXIT;
 showit(&gxmin,&gymin,&gxmax,&gymax,&npoints,&modus,hauptfeld,hilfsfeld);
 //printf("Dieser Punkt wird nie erreicht.\n");//test
 return 0;
}/* ende von main */

void file_exit()
{
 if(fpw!=NULL) fclose(fpw);
 exit(0);
}

void file_load_mu(char *muname)
{  //wenn muname gesetzt ist, soll das entsprechende Spektrum geladen werden
   //sonst wird nach Dateiname gefragt.
 char scratch[80],pfad[400],name[400];
 static char filter[80]="*.xy";
 int n=nspektren,ok;
 //lower_window();
 if(nspektren>=MAXSPEKT)
   {printf("Anzahl Spektren auf MAXSPEKT=%d beschraenkt\n",MAXSPEKT);
    return;
   }
 //printf("%d. Spektrum laden\n",n+1);
 if(muname!=NULL)
     {strcpy(filename[n],muname);
      printf("%d. Spektrum = %s\n",n+1,filename[n]);
     }
 else
     {sprintf(scratch,"lade %d. Spektrum",n+1);
      strcpy(name,"neu.xy");
      ok=nachfilenamefragen(scratch,name,400,
			    0,filter,NULL," CANCEL"," Parent","  Disks",80);
      if(!ok) {printf("Cancel - nichts geladen.\n"); return;}
      if(strlen(name)>=80 || !istvorhanden(name))
       {char name2[100];
	splitpfad(name,pfad,name2,400,100);
	if(strlen(name2)>=80)
	   {printf("Dateiname zu lang - nicht geladen.\n"); return;}
	strcpy(filename[n],name2);
       }
      else strcpy(filename[n],name);
      //printf("Test1: substanzname[%d]='%s'\n",n,substanzname[n]);//test
     }
 if(istscript(filename[n]))
     mehrere_files_laden(filename[n]);
     //printf("Script-Datei lesen und mehrere_files_laden() geht noch nicht.\n");//provi.
 /*
    //dies in uvshow.c benutzt:
    else if(filename[n][1]==0)
    {janeinrequester("Eingaben fuer Funktionsanpassung in Shell-Fenster",
                     " OK ",NULL);
     lower_window(); funk_anpass();
     raise_window();
    }
 */
 else if(istvorhanden(filename[n]))
    {
     if(hilfsfeld==NULL) feldalloc((MAXSPEKT-1)*npoints,&hilfsfeld);
     n=daten_einlesen(filename[n],&hilfsfeld[(n-1)*npoints],0);
     if(n>0) n=nspektren+=n;
     else n=nspektren;
     if(n>2) modus=4+n; else if(n==2) modus=4; else modus=3;
     new_hfeld(hilfsfeld,&modus);
    }
 else printf("file_load_mu() \"%s\" nicht gefunden.\n",filename[n]);
// raise_window();
 refresh();
}

void mehrere_files_laden(char *name)
{
 char scriptname[80],muname[80];
 FILE *fp;
 int i;
 if(*name=='@') name++;
 if(!hatpunkt(name)) sprintf(scriptname,"%s.script",name);
 else strcpy(scriptname,name);
 if(!(fp=fopen(scriptname,"r")))
	{printf("script '%s' nicht gefunden\n",scriptname); return;}
 fscanf(fp,"%d",&i);
 if(i<1 || i>=MAXSPEKT)
       {printf("Scriptfile muss als erste Zahl Anzahl Spektren enthalten !\n");
	printf(" (i=%d ist ungueltig, i sollte zwischen 1 und %d liegen)\n",
		i,MAXSPEKT);
	return;
       }
 while(--i>=0)
  {fscanf(fp,"%s",muname);
   file_load_mu(muname);
  }
 fclose(fp);
}

void file_load()
{
 file_load_mu(NULL);
}

void file_save()
{
 static char savefilename[200],antw[40]="ja",antw2[40]="nein";
 STRING40 str[4];
 FILE *fp;
 double x,y;
 int i,j,jmax,k,ok=0,nspe=nspektren;
 bool mitkopf=0;
 if(argflag['P'])
   {mitkopf=1; ok=1;
    sprintf(savefilename,"autosave.xy");
    jmax=nspektren; nspe=0;
   }
 else
   {ok=requester_input(4,
       "Welche Kurve (1,2, ... 0=Alle)","%d","%d\n",&nspe,
       "Mit Kopfdaten (ja/nein)       ","%s","%s\n",antw,
       "Substanznamen setzen (ja/nein)","%s","%s\n",antw2,
       "Dateiname zum Speichern       ","%s","%s",savefilename);
   }
 if(ok)
   {printf("Saving '%s'\n",savefilename);
    fp=fopen(savefilename,"w");
    if(fp==NULL) {printf("kann %s nicht erstellen.\n",savefilename); return;}
    mitkopf=(*antw!='n' && *antw!='N') || nspe==0;
    if(mitkopf)
      {fprintf(fp,"xydata11  ;erstellt von xyshow %s\n",VERSION);
       fprintf(fp,"xmin=%lf xmax=%lf nx=%d\n",gxmin,gxmax,npoints);
       fprintf(fp,"ymin=%lf ymax=%lf dy=%lf\n",gymin,gymax,gymax-gymin);
       fprintf(fp,"%d\n",(nspe==0)?nspektren:1);
       if(nspe==0) {jmax=nspektren; k=0;} else {jmax=1; k=nspe-1;}
       if(*antw2!='n' && *antw2!='N')
	 for(j=k;j<jmax+k;j+=4)
	   {for(i=0;i<4;i++)
	      sprintf(str[i],"Substanzname %d               ",j+i+1);
	    requester_input(jmax+k-j>4?4:jmax+k-j,
			    str[0],"%s","%s\n",substanzname[j],
			    str[1],"%s","%s\n",substanzname[j+1],
			    str[2],"%s","%s\n",substanzname[j+2],
			    str[3],"%s","%s\n",substanzname[j+3]);
	   }
      }
    else
      {jmax=1; k=0;}
    for(j=k;j<jmax+k;j++)
      {if(mitkopf) fprintf(fp,"%d  ;%s\n",npoints,substanzname[j]);
       for(i=0;i<npoints;i++)
	 {y=(j==0)?hauptfeld[i]:hilfsfeld[(j-1)*npoints+i];
	  x=gxmin+i*gdx;
	  fprintf(fp,"%lg %lg\n",x,y);
	 }
      }
    fclose(fp);
   }
 else
   printf("cancel - nichts gespeichert.\n");
}

FILE *fopen_xyshowdat()
{
 FILE *fp;
 fp=fopen("xyshow.dat","w");
 if(fp==NULL) printf("Fehler beim Oeffnen von 'xyshow.dat'\n");
 else fprintf(fp,"%s\n",dateiname); //dateiname ist aktueller Filename
 return fp;
}

void get_punkt(float *x,float *y)
{
 printf("GetPoint: x=%lg  y=%lg\n",*x,*y);
 if(fpw==NULL) {if((fpw=fopen_xyshowdat())==NULL) return;}
 fprintf(fpw,"x=%lg  y=%lg\n",*x,*y);
}

void int_save(float *z,float *xspitz,float *yspitz,float grenz[]) //float grenz[4];
{
 static int titelflag=1;
 printf("%lg bis %lg  Integral=%lg  Maximum: x=%lg  y=%lg\n",
	grenz[0],grenz[2],*z,*xspitz,*yspitz);
 printf("Integral: %lg\n",*z);
 if(fpw==NULL)
   {if((fpw=fopen_xyshowdat())==NULL) return;}
 if(titelflag)
   {fprintf(fpw," Integrierter Bereich      Integral     Spitze        y\n");
    titelflag=0;
   }
 fprintf(fpw,"%lg bis %lg    %lg    x=%lg    y=%lg\n",
	grenz[0],grenz[2],*z,*xspitz,*yspitz);
}

void texte_laden(char **titel,char **xachse,char **yachse)
{
 static char str[MAXSPEKT*120];
 if(modus<4)
   sprintf(str,"%s  %s",filename[0],substanzname[0]);
 else
   {int i,max=MAXSPEKT*120-1;
    str[0]=0;
    for(i=0;i<nspektren;i++)
      {anhaengen(str,filename[i],max);
       anhaengen2(str,substanzname[i],max);
      }
   }
 *titel=str;
 *xachse=unconst("x-Achse");
 *yachse=unconst("y-Achse");
}

void get_delta(float *z,float grenz[4]) {}

void user_prog(int *fun,POINTER p1,POINTER p2,POINTER p3,POINTER p4)
{
 switch(*fun)	{ case 1: case 2: int_save((float*)p1,(float*)p2,(float*)p3,(float*)p4); break;
			  case 3: case 4: get_punkt((float*)p1,(float*)p2); break;
			  case 5: texte_laden((char**)p1,(char**)p2,(char**)p3); break;
			  case 6: get_delta((float*)p1,(float*)p2); break;
		}
}

void incmumrechnen(float **feld,int *npkt,float *xmin,float *xmax,int flag)
{
 //Nur wenn flag==1 neue Punkteanzahl und neues Feld anlegen,
 //sonst npkt und Feldgroesse unveraendert lassen.
 float *neufeld,*altfeld;
 int nx,j,i,i2;
 double a,b,d,x,A,B,D;
 altfeld= *feld;
 a= *xmin; b= *xmax; d=(a-b)/((*npkt)-1);
 if(argflag['V']) printf("a=%lf b=%lf d=%lf npkt=%d\n",a,b,d,*npkt);//test
 A=1e7/a; B=1e7/b;
 if(argflag['G'])
  {printf("Grenzen: xmin=%lf xmax=%lf\n",A,B);
   printf("neue Grenzen eingeben:\n\txmin:"); scanf("%lf",&A);
   printf("\txmax:"); scanf("%lf",&B);
  }
 if(a>b) D=1e7/(a-d)-1e7/a;
 else    D=1e7/b-1e7/(b+d);
 if(D>1.0 || D<-1.0) D=(int)D;
 if(flag==1)
   nx=(int)((B-A)/D+0.5)+1;
 else
   nx= *npkt;
 B=A+(nx-1)*D;
 if(argflag['V']) printf("A=%lf B=%lf D=%lf nx=%d\n",A,B,D,nx);//test
 neufeld=(float *)calloc(nx,sizeof(float));
 for(j=0;j<nx;j++)
        {i=x=(a-1e7/(A+j*D))/d; i2=i+1;
         if(i<0 || i2>=(*npkt)) neufeld[j]=0.;
         else neufeld[j]=altfeld[i]+(altfeld[i2]-altfeld[i])*(x-i);
        }
 *xmin=A;
 *xmax=B;
 if(flag==1)
   {*npkt=nx;
    cfree(altfeld);
    *feld=neufeld;
   }
}

/*************************** kleinkram ***************************/
int getline(FILE *fp,char *s,int lim)
{               /* liest eine Textzeile oder maximal lim Zeichen */
                /* und ersetzt den Zeilentrenner durch 0         */
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
        *s++ = c;
 *s='\0';
 return (c!=EOF);       /* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

int nextc(FILE *fp)
{
 int c;
 while((c=getc(fp))!=EOF && isspace(c)) ;
 return c;
}

FILE *neuoeffnen(FILE *fp,char *name)
{
 fclose(fp);
 if((fp=fopen(name,"rb"))==NULL)
        {printf("kann '%s' nicht mehr finden\n",name); exit(1);}
 return fp;
}

int starkverschieden(double a,double b)
{
 double v;
 if(a==b) return 0;
 if(a==0 || b==0) return 1;
 if((v=a/b)<0.9 || v>1.1) return 1;
 return 0;
}

double tmspeaksuchen(FILE *fp)
{
 double x,y,ymax,xmax=0.0;
 int c,n;
 for(n=0;(c=nextc(fp))!=EOF && c!=CONTROLZ;)
	{ungetc(c,fp);
	 fscanf(fp,"%lf%*c%lf",&x,&y);
	 if(x> -0.5 && x<0.5)
	   {if(n==0 || y>ymax) {ymax=y; xmax=x;}
	    n++;
	   }
	}
 return xmax;
}
double tmspeaksuchenb(FILE *fp,double xmin,double ddx)
{
 double x,y,ymax,xmax=0.0;
 int c,n,i;
 for(i=n=0;(c=getc(fp))!=EOF;i++)
	{ungetc(c,fp);
	 x=xmin+i*ddx;
	 y=lesezahl(fp);
	 if(x> -0.5 && x<0.5)
	   {if(n==0 || y>ymax) {ymax=y; xmax=x;}
	    n++;
	   }
	}
 return xmax;
}
double referenzpeaksuchenb(int r,double d,FILE *fp,double xmin,double ddx)
{
 double x,y,ymax,xsoll,xmax,xo,xu;
 int c,n,i,ok=1;
 switch(r)
   {case 1: xsoll=2.50; break; //DMSO 1H-NMR
    case 2: xsoll=2.05; break; //Aceton
    case 3: xsoll=7.26; break; //CDCl3
    case 4: xsoll=4.79; break; //D2O
    case 11: xsoll=39.52; break; //DMSO 13C-NMR
    case 12: xsoll=19.84; break; //Aceton
    case 13: xsoll=77.16; break; //CDCl3
    case 14: xsoll=0; ok=0; break; //D2O
    default: xsoll=0;
      ok=requester_input(2,"unbekannte Referenz","%d","%d\n",&r,
			 "Bitte Sollwert eingeben:","%lf","%lf",&xsoll);
   }
 if(!ok) return 0.0;
 xmax=xsoll;
 xu=xmax-d; xo=xmax+d;
 if(argflag['V']) printf("Suche Referenzpeak bei %.2lf +- %.1lf\n",xsoll,d);
 for(i=n=0;(c=getc(fp))!=EOF;i++)
	{ungetc(c,fp);
	 x=xmin+i*ddx;
	 y=lesezahl(fp);
	 if(x>xu && x<xo)
	   {if(n==0 || y>ymax) {ymax=y; xmax=x;}
	    n++;
	   }
	}
 return xmax-xsoll;
}

int daten_einlesen(char *name,float *feld,int gxflag)
{
 //Rueckgabewert >=1 wenn erfolgreich (= Anzahl eingelesene Spektren)
 int i,j,nx=0,c,nspektr=0,cmflag=(feld==NULL)?1:0;
 FILE *fp=NULL;
 int rueckflag=0,syist1=0;
 float xmin,xmax,ymin,ymax,h,*fe;
 double x0,dx,dy,sy,y0,x,y,shift=0.;
 char str[80];
 if(istordner(name) || (istlink(name) && !hatendung(name,".xy")))
   {
    printf("daten_einlesen() Testpunkt NMRordner\n");//test
    sprintf(str,"NMRordner");
   }
 else
   {if((fp=fopen(name,"rb"))==NULL)
        {printf("daten_einlesen() '%s' nicht gefunden\n",name); return -1;}
    if(feld!=NULL || argflag['V']) printf("%s wird eingelesen\n",name);
    c=getc(fp); ungetc(c,fp);
    if((c<' ' && c!=0x0A && c!=0x0D && c!=0x09) || c>'z') //Binaerdatei
      sprintf(str,"NMRbinaer");
    else
      getline(fp,str,80);
   }
 if(strncmp(str,"xmin=",5)==0) /* Datei mit Kopfdaten */
 {sscanf(str,"xmin=%f xmax=%f %*cx=%d",&xmin,&xmax,&nx);
  fscanf(fp,"\nymin=%f ymax=%f %*cy=%lf",&ymin,&ymax,&dy);
  if(nx<2 || dy==0) {printf("Fehlerhafte Kopfdaten.\n");fclose(fp);return -1;}
  sy=(ymax-ymin)/dy;
  if(feld==NULL)
   {dx=(xmax-xmin)/(nx-1); x0=xmin;
    if(!argflag['A'] || argflag['V'])
     {printf("xmin=%f xmax=%f ymin=%f ymax=%f\n",xmin,xmax,ymin,ymax);
      printf("nx=%d dy=%lg\n",nx,dy);
      if(argflag['V']) printf("dx=%lf sy=%lf\n",dx,sy);
     }
    feldalloc(nx,&hauptfeld);
    if((feld=hauptfeld)==NULL) {fclose(fp); return -1;}
   }
  else
    {nx=npoints; dx=anpassen(&x0,xmin,xmax);//Anpassen an verhandenes Feld
    }
  for(i=0;i<nx;i++) feld[i]=0.0;
  if(sy<1.001 && sy>0.999) syist1=1;
  for(c=getc(fp);c!=EOF && c!=CONTROLZ;)
	{ungetc(c,fp);
	 if(fscanf(fp,"%lf%*c%lf",&x,&y)!=2)
	   {printf("Fehler bei fscanf()\n"); break;}
	 if(!syist1) y=sy*(y-ymin);
	 j=idfix((x-x0)/dx);
	 if(j>=0 && j<nx) feld[j]=y;
	 else printf("Fehler in Datensatz: x=%lf y=%lf\n",x,y);
	 //if(argflag['V']) printf("j=%d x=%lf y=%lf\n",j,x,y);//test
	 do {c=getc(fp); if(c==EOF) break;} while(isspace(c));
	}
  if(!syist1) {ymax=(ymax-ymin)*sy; ymin=0.0;}
  nspektr=1;
 }
 else if(strncmp(str,"xydata11",8)==0) /* erweiterte Kopfdaten */
 {int npunktpaar,nsp,j1,n1,n2,n;
  double yvorher;
  getline(fp,str,80); sscanf(str,"xmin=%f xmax=%f %*cx=%d",&xmin,&xmax,&nx);
  getline(fp,str,80); sscanf(str,"ymin=%f ymax=%f %*cy=%lf",&ymin,&ymax,&dy);
  if(nx<2 || dy==0) {printf("Fehlerhafte Kopfdaten.\n");fclose(fp);return -1;}
  if(feld!=NULL)
    {nx=npoints; dx=anpassen(&x0,xmin,xmax);}//Anpassen an verhandenes Feld
  else
    {dx=(xmax-xmin)/(nx-1); x0=xmin;}
  sy=(ymax-ymin)/dy;
  if(!argflag['A'] || argflag['V'])
   {printf("xmin=%f xmax=%f ymin=%f ymax=%f\n",xmin,xmax,ymin,ymax);
    printf("nx=%d dy=%lg\n",nx,dy);
    if(argflag['V']) printf("dx=%lf sy=%lf\n",dx,sy);
   }
  do getline(fp,str,80); while(*str==';' || *str=='\n');
  sscanf(str,"%d",&nspektr);
  if(nspektr<1)
    {printf("Fehlerhafte Kopfdaten: nspektren muss >=1 sein.\n");
     fclose(fp); return -1;
    }
  if(nspektr+nspektren>MAXSPEKT)
    {printf("Anzahl Spektren auf MAXSPEKT=%d beschraenkt\n",MAXSPEKT);
     nspektr=MAXSPEKT-nspektren;
     if(nspektr<1) {fclose(fp); return 0;}
    }
  if(feld==NULL)
    {int nxtotal;
     feldalloc(nxtotal=nx*MAXSPEKT,&hauptfeld);
     if((feld=hauptfeld)==NULL) {fclose(fp); return -1;}
     hilfsfeld = &feld[nx];
     for(i=0;i<nxtotal;i++) feld[i]=0.0;//gesamtes Feld+Hilfsfeld loeschen
    }
  else for(i=0;i<nx;i++) feld[i]=0.0;
  if(sy<1.001 && sy>0.999) syist1=1;
  for(nsp=0;nsp<nspektr;nsp++)
   {do getline(fp,str,80); while(*str==';' || *str<' ');
    if(argflag['V']) printf("%s\n",str);/*test*/
    sscanf(str,"%d",&npunktpaar);
    if(argflag['V']) printf("nspektr=%d nsp=%d npunktpaar=%d\n",
			    nspektr,nsp,npunktpaar);/*test*/
    if(npunktpaar>1 && (n=index2(str,";"))>0) //wenn Name im Kommentar: als
       strcpy(substanzname[nspektren+nsp],&str[n+1]); //Substanzname speichern.
    for(fe= &feld[nsp*nx],yvorher=0.,j1=i=0;i<npunktpaar;i++,yvorher=y)
	{do c=getc(fp); while(isspace(c));
	 ungetc(c,fp);
	 fscanf(fp,"%lf%*c%lf",&x,&y);
	 if(!syist1) y=sy*(y-ymin);
	 j=idfix((x-x0)/dx);
	 //if(argflag['V']) printf("j=%d x=%lf y=%lf\n",j,x,y);//test
	 if(j>=0 && j<nx)
		{for(n2=1,n1=j-j1; j1<j; n2++,n1--)
			 fe[j1++]=(yvorher*n1+y*n2)/(n1+n2);
		 fe[j1++]=y;
		}
	 else printf("Ausserhalb des definierten Bereichs: x=%lf y=%lf\n",x,y);
	}
    //while(j1<nx) fe[j1++]=y; /*test*/
   }
  if(!syist1) {ymax=(ymax-ymin)*sy; ymin=0.0;}
 }
 else if(strncmp(str,"NMR",3)==0) /* rohe NMR-Datei */
 {double ddx;
  char *pfadname,vollername[200];
  if(strncmp(str,"NMRord",6)==0) /* Ordner mit NMR-Dateien */
    {int u1=1;
     //if(argflag['N']) for(i='2';i<='9';i++) if(argflag[i]) {u1=i-'0';break;}
     if(argflag['N']) u1=u1neu;
     if(argflag['V']) printf("NMR-Ordner.\n");
     pfadname=name;
#ifdef NEUES_NMR_2018
     sprintf(vollername,"%s/%d/pdata/1/procs",pfadname,u1);
     name=vollername;
     fp=fopen(name,"rb");
     if(fp)
      {xmin=parser(fp,"ABSF1="); xmax=parser(fp,"ABSF2=");}
     else {
#endif
     sprintf(vollername,"%s/%d/pdata/1/peak.txt",pfadname,u1);
     name=vollername;
     fp=fopen(name,"rb");
     if(!fp)
      {sprintf(vollername,"%s/%d/pdata/1/peaklist.xml",pfadname,u1);
       name=vollername;
       fp=fopen(name,"rb");
      }
     if(fp)
      {xmin=parser(fp,"F1="); xmax=parser(fp,"F2=");}
     else
      {sprintf(vollername,"%s/%d/pdata/1/proc",pfadname,u1);
       fp=fopen(name,"rb");
       if(fp==NULL) {printf("%s nicht gefunden.\n",name); return -1;}
       xmin=parser(fp,"F1P="); xmax=parser(fp,"F2P=");
      }
#ifdef NEUES_NMR_2018
     }
#endif
     if(xmin>xmax) rueckflag=1;
     fclose(fp);
     sprintf(vollername,"%s/%d/pdata/1/1r",pfadname,u1);
     //sprintf(vollername,"%s/%d/pdata/1/1i",pfadname,u1);//test
     name=vollername;
     nx=filelaenge(name)/8;//neu ab Version 1.51
     msbfirst=automsbfirst(name);
     if(argflag['W']) msbfirst=1-msbfirst;
     if((fp=fopen(name,"rb"))==NULL)
       {printf("%s nicht gefunden.\n",name); return -1;}
    }
  else
    {printf("binaere Daten: vermutlich 1H-NMR\n");
     printf("Besser nur den Ordner angeben, xyshow sollte dann 1r automatisch\n");
     printf("finden, und auch die Grenzen aus peak.txt automatisch einlesen.\n");
     xmax= -4.285913; xmin=16.538271; rueckflag=1;
    }
  ymin= -0.01; ymax=0.01;
  if(nx==0) {printf("Fehler: nx==0, auf 16384 gesetzt.\n"); nx=16384;}//test
  ddx=(xmax-xmin)/(nx-1);
  if(argflag['V']) printf("xmin=%f xmax=%f nx=%d\n",xmin,xmax,nx);
  if(argflag['T'])
   {shift= -tmspeaksuchenb(fp,xmin,ddx);
    if(argflag['V']) printf("TMS-Peak bei %lf\n",-shift);
    xmin += shift; xmax += shift;
    fp=neuoeffnen(fp,name);
   }
  else if(argflag['U'])
   {int n=1,r=0;
    double d=0.5;
    char zeile[80]="1H";
    if(argflag['N']) {strcpy(zeile,"13C"); d=2.5;}
    requester_input(3,
	"Welches Referenzsignal? (1=DMSO, 2=Aceton, 3=CDCl3, 4=D2O) ?",
		    "%d","%d\n",&n,
		    "1H-NMR oder 13C-NMR ?","%s","%s\n",zeile,
		    " Suchbereich ","%.2lf","%lf",&d);
    sscanf(zeile,"%d",&r);
    if(r==13) n+=10;
    shift= -referenzpeaksuchenb(n,d,fp,xmin,ddx);
    if(argflag['V']) printf("Referenz-Peak um %lf verschoben\n",-shift);
    xmin += shift; xmax += shift;
    fp=neuoeffnen(fp,name);
   }
  else if(argflag['G'] && !argflag['C'])
   {printf("xmin=%f xmax=%f nx=%d\n",xmin,xmax,nx);
    printf("Verschieben in x-Richtung:"); scanf("%lf",&shift);
    if(feld==NULL && !argflag['A'])
      {printf("neue Grenzen eingeben (Verschiebung beruecksichtigt)\n");
       printf("\txmin:"); scanf("%f",&xmin);
       printf("\txmax:"); scanf("%f",&xmax);
      }
    else
      {xmin += shift; xmax += shift;}
   }
  if(feld==NULL)
    {feldalloc(nx,&hauptfeld);
     if((feld=hauptfeld)==NULL) {fclose(fp); return -1;}
     dx=ddx; x0=xmin;
    }
  else
    {nx=npoints; dx=anpassen(&x0,xmin,xmax);}//Anpassen an verhandenes Feld
  for(i=0;i<nx;i++)
      {y=lesezahl(fp);
       x=xmin+i*ddx;
       j=idfix((x-x0)/dx);
       //printf("i=%d j=%d x=%lf y=%lf\n",i,j,x,y);//test
       if(j>=0 && j<nx) feld[j]=y;
       //else printf("Fehler bei feld[j]=y; i=%d j=%d nx=%d\n",i,j,nx);//test
       if(y<ymin) ymin=y;
       else if(y>ymax) ymax=y;
      }
  if(argflag['V']) printf("ymin=%f ymax=%f\n",ymin,ymax);
  nspektr=1;
 }
 else if(strncmp(str,"xydata",6)==0)
 {printf("xyshow unterstuetzt Format '%s' nicht. Siehe xyshow -h\n",str);
  fclose(fp); return -1;
 }
 else /* xy-Format ohne Kopfdaten */
 {double deltax=0.0,sumdx=0.0,dxmin=0,dxmax=0;
  double firstx=0,newx=0;
  int starkflag=0;
  fp=neuoeffnen(fp,name);
  for(nx=0;(c=nextc(fp))!=EOF && c!=CONTROLZ;nx++,newx=x)
	{ungetc(c,fp);
	 fscanf(fp,"%lf%*c%lf",&x,&y);
	 if(nx==0)
	  {xmin=xmax=x; ymin=ymax=y; firstx=x;}
	 else if(nx==1)
	  {dxmin=dxmax=x-firstx;}
	 else
	  {if(x<xmin) xmin=x; else if(x>xmax) xmax=x;
	   if(y<ymin) ymin=y; else if(y>ymax) ymax=y;
	   if((dx=x-newx)<dxmin) dxmin=dx;
	   else if(dx>dxmax) dxmax=dx;
	  }
	}
  if(nx<2)
    {printf("Datei enthaelt zu wenig Punktpaare.\n"); fclose(fp); return -1;}
  sumdx=newx-firstx;
  if(sumdx<0) {sumdx= -sumdx; rueckflag=1;}
  if(feld==NULL)
    {if(starkverschieden(dxmin,dxmax))
      {printf("Es werden konstante x-Abstaende vorausgesetzt. nx erhoehen.\n");
       starkflag=1;
      }
     deltax=sumdx/(nx-1);
     nx=idfix((xmax-xmin)/deltax)+1;
     dx=x0=0; //richtige Werte erst weiter unten berechnen
    }
  else
    {nx=npoints; dx=anpassen(&x0,xmin,xmax);}//Anpassen an verhandenes Feld
  fp=neuoeffnen(fp,name);
  if(!argflag['A'] || argflag['V'])
    {printf("xmin=%f xmax=%f  ymin=%f ymax=%f\n",xmin,xmax,ymin,ymax);
     if(feld==NULL) printf("nx=%d\n",nx);
    }
  if(argflag['T'])
   {shift= -tmspeaksuchen(fp);
    if(argflag['V']) printf("TMS-Peak bei %lf\n",-shift);
    xmin += shift; xmax += shift;
    fp=neuoeffnen(fp,name);
   }
  else if(argflag['G'] && !argflag['C'])
   {printf("Verschieben in x-Richtung:"); scanf("%lf",&shift);
    if(feld==NULL && !argflag['A'])
      {printf("neue Grenzen eingeben (Verschiebung beruecksichtigt)\n");
       printf("\txmin:"); scanf("%f",&xmin);
       printf("\txmax:"); scanf("%f",&xmax);
       nx=idfix((xmax-xmin)/deltax)+1;
       if(argflag['V']) printf("neu berechnetes nx=%d\n",nx);
      }
    else
      {xmin += shift; xmax += shift;}
   }
  if(starkflag && !argflag['A'])
   {printf("neues nx:"); scanf("%d",&nx);
    printf("Methode fuer Berechnung von Zwischenwerten:\n");
    printf(" 0 = Zwischenwerte auf 0.0 setzen\n");
    printf(" 1 = lineare Interpolation\n");
    printf(" 2 = Treppen\n");
    printf("Methode:"); scanf("%d",&methode);
    if(methode<0 || methode>2) methode=1;//ungueltige Methode korrigieren
   }
  if(nx<2) {printf("Error nx=%d - should be >=2\n",nx); return -1;}//test
  if(dx==0) {dx=(xmax-xmin)/(nx-1); x0=xmin;}
  if(argflag['V']) printf("dx=%lf\n",dx);//test
  if(feld==NULL)
    {feldalloc(nx,&hauptfeld);
     if((feld=hauptfeld)==NULL) {fclose(fp); return -1;}
    }
  if(methode==0) for(i=0;i<nx;i++) feld[i]=0.;
  for(c=getc(fp),i=(rueckflag==1)?nx-1:0;c!=EOF && c!=CONTROLZ;)
	{ungetc(c,fp);
	 fscanf(fp,"%lf%*c%lf",&x,&y); x+=shift;
	 j=idfix((x-x0)/dx);
	 if(methode==0)
		{if(j>=0 && j<nx) feld[j]=y;}
	 else if(methode==1)
	       {if(rueckflag==0)
		  {if(i!=j)
		    {y0=(i>0)?feld[i-1]:0.; dy=(y-y0)/(j-(i-1));
		     while(i<j && i<nx) {feld[i++]=(y0+=dy);}
		    }
		   if(i==j && i<nx) feld[i++]=y;
		  }
	        else /*if(rueckflag==1)*/
		  {if(i!=j)
		    {y0=(i<nx-1)?feld[i+1]:0.; dy=(y-y0)/((i+1)-j);
		     while(i>j && i>=0) {feld[i--]=(y0+=dy);}
		    }
		   if(i==j && i>=0) feld[i--]=y;
	       }  }
	 else /*if(methode==2)*/
	       {if(rueckflag==0)
			{while(i<=j && i<nx) feld[i++]=y;}
	        else	{while(i>=j && i>=0) feld[i--]=y;}
	       }
	 do c=getc(fp); while(isspace(c));
	}
  nspektr=1;
 }
 fclose(fp);
 npoints=nx;
 if((argflag['R'] && rueckflag==0) || (!argflag['R'] && rueckflag==1))
     {for(i=0,j=nx-1;i<j;i++,j--) {h=feld[i]; feld[i]=feld[j]; feld[j]=h;}
      h=xmin; xmin=xmax; xmax=h;
      dx = -dx;
     }
 if(argflag['S'])
   {double faktor;
    printf("Skalierungsfaktor:"); scanf("%lf",&faktor);
    if(faktor!=0. && faktor!=1.)
      {faktor=1.0/faktor;
       for(i=0;i<npoints;i++) feld[i] *= faktor;
       ymin *= faktor;
       ymax *= faktor;
      }
   }
 if(argflag['C']) incmumrechnen(&feld,&npoints,&xmin,&xmax,cmflag);
 if(gxflag)
   {if(gxmin==gxmax) {gxmin=xmin; gxmax=xmax; gdx=dx;}
    if(gymin==gymax) {gymin=ymin; gymax=ymax;}
    else
      {if(ymin<gymin) gymin=ymin;
       if(ymax>gymax) gymax=ymax;
      }
   }
 for(i=0;i<nspektr;i++)
   {strcpy(filename[nspektren+i],name);
    //printf("Test2: substanzname[%d]='%s'\n",nspektren+i,substanzname[nspektren+i]);//test
   }
 return nspektr;
}

/************ NMR-Spektrum direkt einlesen ***************/
double lesezahl(FILE *fp)
{
/* dies geht offenbar nicht:
 LONG x[2];
 fread(x,4,1,fp);
 fread(&x[1],4,1,fp);
 return (x[0]+x[1])/(2*256.0);
*/
/* auch dies nicht auf allen Maschinen:
 int i;
 union {unsigned char b[8]; long x[2];} z;
 for(i=8;i>0;) z.b[--i]=getc(fp);
 return (z.x[0]+z.x[1])/(2*256.0);
*/
 LONG z[2];
 int i,j;
 if(msbfirst)
  for(j=0;j<2;j++)
   {z[j]=getc(fp);
    for(i=0;i<3;i++)
      z[j]=(z[j]<<8)+getc(fp);
   }
 else
  for(j=0;j<2;j++)
   {int c[4];
    for(i=0;i<4;i++) c[i]=getc(fp);
    z[j]=c[3];
    for(i=2;i>=0;--i)
      z[j]=(z[j]<<8)+c[i];
   }
 return (z[0]+z[1])/(2*256.0);
}
