/* objmodelexporter.cc			letzte Aenderung: 11.3.2022 */
#define VERSION "Version 0.3"
/*

 Modell im OBJ-Format einlesen und als BMF-Datei speichern (eigenes Format von Pilzschaf)
 Beschreibung der Formate:
 OBJ: Mit Blender exportiert als name.obj
 MTL: wird von Blender automatisch als name.mtl gespeichert
 BMF: bmf-dateiformat.txt, ohne Versionsnummer: Variante von Pilzschaf

Die Einheiten von OpenGL in Spiel1 sind als Meter definiert.
Die Einheiten in OBJ auch als Meter?

History:
17.2.2022	Erstellung aus stlmodelexporter.cc (RP)
1.3.2022   0.2  Version 2 vom BMF-Format: zusaetzlich ambient in Material
                und texturfilename, normtexturfilename auch in Material
8.3.2022   0.3  Ambient und Specular wenn von Blender stammend mit
                Diffuse multipliziert. Einlesen von Indexen verbessert.
11.3.2022       Diffuse doch nicht multiplizieren! Noch als Option drin gelassen
                Fehler korrigiert: hatte immer multipliziert (option|FLAG war falsch, richtig: option&FLAG)

*/
//#define TEST_VERGLEICH

#include <stdio.h>
#include <iostream>
#include <vector>
#include <stdlib.h>
#include <string.h>
#include "vektor3fklasse.cc"
//#include "../libs/glm/glm.hpp"
#include "objmodelexporter.h"

typedef uint16_t uint16;
typedef uint32_t uint32;
//typedef uint64_t uint64;

/************************* eigene Klassen *****************************/
struct Uvs {
 float x,y;
 Uvs() {x=y=0;}
};

class Dreieck {
public:
 float normalen[3];
 float vertex1[3];
 float vertex2[3];
 float vertex3[3];
 Uvs uvs1,uvs2,uvs3;
};

class MTLmaterial {
public:
 char name[200]; //z.B. "Material.001"
 float ambient[3]; //Ka  Farbe bei Umgebungslicht
 float diffuse[3]; //Kd  Grundfarbe (wenn beleuchtet)
 float specular[3]; //Ks  Spiegelnde Farbe
 float emissive[3]; //Ke  Selbstleuchtend
 float shininess; //Ns  Spiegelunskonzentration: grosser Wert macht kleinen Spieglungsfleck
 float ni; //Ni  Optical Density (1.0 = Licht wird nicht gebrochen) (bisher unbenutzt)
 float dissolve;//d  (1.0=undurchsichtig, 0.0=vollkommen transparent) (bisher unbenutzt)
 int illum; //illum  (TODO bisher unbenutzt)
 char texturfile[120]; //map_Kd oder map_d Textur (JPG- oder PNG-Dateiname ohne Pfad)
 char normtexturfile[120]; //map_Bump  Textur fuer Normalen-Vektoren //TODO
 //char speculartexturfile[120]; //map_Ks //TODO
 //char dissolvetexturfile[120]; //map_d //TODO
 MTLmaterial() {reset();}
 void reset()
   {name[0]=0; shininess=16.0f; ni=dissolve=1.0f; illum=0;
    for(int i=0;i<120;i++) texturfile[i]=normtexturfile[i]=0;
   }
};

class BMFmesh {
public:
 uint32 numDreiecke;
 std::vector<Dreieck> dreiecke;
 int smooth; //0=off 1=mit Glaettung //TODO: ev. auch groessere Nummern
 MTLmaterial mat;
 BMFmesh() {numDreiecke=0; smooth=0;}
 void init(MTLmaterial m)
 {
  mat=m;
  //if(numDreiecke>0) dreiecke.erase(dreiecke.begin(),dreiecke.begin()+(numDreiecke-1));//test
  dreiecke.clear(); //alle dreiecke loeschen
  numDreiecke=0;
 }
};

/********************** Globale Variablen *****************************/
float offset[3] = {0,0,0}; //Offset in Meter beim Speichern zu beruecksichtigen
float scalefaktor = 1.0; //Skalierfaktor, beim Einlesen der Daten zu beruecksichtigen
int option_smooth=0; //0=gemaess OBJ-Datei setzen, 1=immer setzen, 2=immer aus
int bmfversion=200; //zu schreibende BMF-Version (200 bedeutet Version 2.00)
bool texturflag=true; //Texturen beruecksichtigen falls bmfversion>=200
#define AMBIENT_MULT 1
#define SPECULAR_MULT 2
//int option_mult= AMBIENT_MULT|SPECULAR_MULT; //ambient und specular mit diffuse multiplizieren
int option_mult= AMBIENT_MULT; //ambient mit diffuse multiplizieren
bool autosmoothflag=false; //TODO: autosmooth beruecksichtigen
float autosmooth=(30.0*GRAD); //TODO: maximaler Winkel zwischen Normalen die gemittelt werden soll

/************************* Vordeklarationen ***************************/
void obj2bmf(FILE *fp1,FILE *fpmtl,FILE *fp2);
void save_bmf(FILE *fp,uint32 numMeshes,uint32 numUvs,std::vector<BMFmesh> mesh);
void save_mesh(FILE *fp,uint32 numUvs,BMFmesh &mesh);
//void output_uint64(FILE *fp, uint64 u);
void output_uint32(FILE *fp, uint32 u);
void output_string(FILE *fp, const char *name);

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];

void setargflags(const char *s)
{
 int c;
 while((c= *s++)!=0)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
   if(c=='X')
    {
     sscanf(s,"%f",&scalefaktor);
     //sscanf(s,"%lf",&scalefaktor); //wenn als double definiert
     while(isdigit(*s) || *s=='.') {s++;}
    }
   else if(c=='S') {sscanf(s,"%d",&option_smooth);}
   else if(c=='W')
    {
     sscanf(s,"%f",&autosmooth); autosmooth *= GRAD;
     autosmoothflag=true;
    }
   else if(c=='F')
    {
     sscanf(s,"%d",&bmfversion);
     if(bmfversion<10) bmfversion *= 100;
     texturflag = (bmfversion>=200);
    }
  }
}

/*************************** Kleinkram ********************************/
int getline(FILE *fp,char *s,int lim)
{		/* liest eine Textzeile oder maximal lim Zeichen */
		/* und ersetzt den Zeilentrenner durch 0         */
 int c=0;
 while(--lim && (c=getc(fp))!=EOF && c!='\n')
  {
   if(c>=' ')  *s++ = c;
  }
 *s=0;
 return (c!=EOF);	/* TRUE wenn erfolgreich, FALSE wenn Fileende */
}

bool schonvorhanden(const char *name)
{
 FILE *fp=fopen(name,"rb");
 if(fp==NULL) return false;
 fclose(fp);
 return true;
}

bool ist_objname(const char *zeile)
{
 int n = strlen(zeile) - 4;
 return (n>0 && strcmp(&zeile[n],".obj")==0);
}

void drei_floats_einlesen(const char *zeile, float *data)
{
 float x,y,z;
 int n=sscanf(zeile,"%f%*c%f%*c%f",&x,&y,&z);
 if(n!=3) {printf("Fehler in drei_floats_einlesen(): zu wenige Parameter\n");}
 data[0]=x;
 data[1]=y;
 data[2]=z;
}

void zwei_floats_einlesen(const char *zeile, float *data)
{
 float x,y;
 int n=sscanf(zeile,"%f%*c%f",&x,&y);
 if(n!=2) {printf("Fehler in zwei_floats_einlesen(): zu wenige Parameter\n");}
 data[0]=x;
 data[1]=y;
}

/************************* Hauptprogramm ******************************/
void mtllibsuchen(const char *quellname,char *quellname2,int max)
{
 quellname2[0]=0; 
 FILE *fp=fopen(quellname,"r"); if(fp==NULL) {return;}
 char zeile[400], *s;
 while(getline(fp,zeile,400))
  {
   for(s=zeile; *s==' ' || *s=='\t'; s++) {}
   if(strncmp(s,"mtllib",6)==0)
    {
     for(s= &zeile[6]; *s==' ' || *s=='\t'; s++) {}
     sscanf(s,"%s",quellname2);
     fclose(fp);
     return;
    }
  }
 fclose(fp);
}

int main(int argc,char *argv[])
{
 char quellname[400], quellname2[400], zielname[400];
 FILE *fp1=NULL, *fp2=NULL, *fpmtl=NULL;
 int i,j=0,c;
 quellname[0]=zielname[0]=0;
 for(i=1;i<argc;i++)
  {
   if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
   else if(++j==1) strncpy(quellname,argv[i],200);
   else if(j==2) strncpy(zielname,argv[i],200);
  }
 if(argflag['?'] || argflag['H'] || j<1 || j>MAXARG)
  {
   printf("%s  %s\n",argv[0],VERSION);
   printf("Anwendung: %s [-Flags] Quelle.obj [Ziel.bmf]\n",argv[0]);
   printf("  Flags: v = Verbose (gespraechig)\n");
   printf("         x = Skalierfaktor, z.B: verkleinern: -x0.1 vergroessern: -x2.5\n");
   printf("         s1 = Smooth fuer abgerundete Kanten (1: ein, 2: aus, sonst automatisch)\n");//TODO
   printf("         w  = maximaler Winkel zwischen Normalen beim Smoothen (Beispiel: -w%.1f)\n",autosmooth/GRAD);
   printf("         f1 = alte BMF-Version schreiben\n");
   printf("         m0 = Auswahl ob Ambient und Specular mit Diffuse multipliziert werden soll\n");
   printf("              (1=Ambient, 2=Specular, 3=beide, 0=keines, Voreinstellung: %d)\n",option_mult);
   if(argflag['H']==0)
    {printf("         h = ausfuehrlichere Hilfe\n");
    }
   else
    {
     printf("  Neben name.obj sollte auch noch name.mtl vorhanden sein.\n");
     printf("  Beides wird beim Exportieren als obj mit Blender gespeichert.\n");
     printf("  Die MTL-Datei kann man auch kopieren und editieren. Dann in\n");
     printf("  der OBJ-Datei die Zeile \"mtllib ...\" anpassen.\n");
    }
   exit(0);
  }
 //if(argflag['S'] && option_smooth==0) option_smooth=1;
 if(!ist_objname(quellname))
  {printf("Fehler: Eingabedatei muss eine .obj-Datei sein\n"); return 1;}
 mtllibsuchen(quellname,quellname2,400);
 if(quellname2[0]==0)
  {
   for(i=0; (c=quellname[i])!=0 && c!='.'; i++)
    {quellname2[i]=quellname[i];}
   strcpy(&quellname2[i],".mtl");
  }
 if(*zielname==0)
  {
   for(i=0; (c=quellname[i])!=0 && c!='.'; i++)
    {zielname[i]=quellname[i];}
   strcpy(&zielname[i],".bmf");
  }
 if(argflag['V']) printf("quellname=\"%s\" quellname2=\"%s\" zielname=\"%s\"\n",quellname,quellname2,zielname);//test
 if((fp1=fopen(quellname,"r"))==NULL)
  {printf("Datei \"%s\" nicht vorhanden\n",quellname); return 1;}
 if((fpmtl=fopen(quellname2,"r"))==NULL)
  {printf("Datei \"%s\" nicht vorhanden\n",quellname2); fclose(fp1); return 1;}
 if((fp2=fopen(zielname,"wb"))==NULL)
  {printf("Konnte Datei \"%s\" nicht erstellen\n",zielname); fclose(fp1); fclose(fpmtl); return 2;}
 obj2bmf(fp1,fpmtl,fp2);
 fclose(fp2); fclose(fpmtl); fclose(fp1);
 printf("Datei \"%s\" erstellt.\n",zielname);
 return 0;
}// ende von main

uint16 input_uint16(FILE *fp)
{
 uint16 c;
 c = getc(fp) & 0xFF;
 c += ((getc(fp)&0xFF)<<8);
 return c;
}

uint32 input_uint32(FILE *fp)
{
 uint32 c;
 c = getc(fp) & 0xFF;
 c += ((getc(fp)&0xFF)<<8);
 c += ((getc(fp)&0xFF)<<16);
 c += ((getc(fp)&0xFF)<<24);
 return c;
}

float input_float(FILE *fp)
{
 union {float f; uint32 c;} fc;
 fc.c = getc(fp) & 0xFF;
 fc.c += ((getc(fp)&0xFF)<<8);
 fc.c += ((getc(fp)&0xFF)<<16);
 fc.c += ((getc(fp)&0xFF)<<24);
 return fc.f;
}

void input_vektor(FILE *fp,float* vektor,float scale)
{
 // Umrechnung Koordinatensystem von STL zum Koordinatensystem von OpenGL:
 vektor[2] = input_float(fp)*scale; //Achse z nach vorne in OpenGL, ist Achse x nach rechts in STL
 vektor[0] = input_float(fp)*scale; //Achse x nach rechts in OpengGL, ist Achse y nach hinten in STL
 vektor[1] = input_float(fp)*scale; //senkrechte Achse ist in OpenGL y, in STL aber z
}

int getmatnr(std::vector<MTLmaterial> mtlmat,const char *nam) //wenn nicht gefunden: -1
{
 int nr=0;
 for(;nr<(int)mtlmat.size();nr++)
  {
   if(strcmp(mtlmat[nr].name,nam)==0) return nr;
  }
 return -1;
}

int vtn_lesen(const char *str, uint* n1,uint* n2,uint* n3,uint* n4,uint* n5,
	      uint* n6,uint* n7,uint* n8,uint* n9) //Vertex/Textur/Normalen indexe einlesen
{
 //Beispiel1: f 2/1/1 3/2/1 1/3/1  #jeweils (Vertex/Texturkoord/Normalen) also n1/n2/n3 n4/n5/n6 n7/n8/n9
 //Beispiel2: f 2//1 3//1 1//1     #keine Texturkoordinaten vorhanden
 //Beispiel3: f 2/1 3/2 1/3        #keine Normalen vorhanden
 //Beispiel4: f 2 3 1              #nur Vertexe vorhanden
 int n=sscanf(str,"%d",n1);
 if(n!=1) return 0;//Fehler
 while(isdigit(*str)) str++;
 if(*str==' ')
  {//gemaess Beispiel4 einlesen:
   n += sscanf(&str[1],"%d %d",n4,n7);
   *n2 = *n3 = *n5 = *n6 = *n8 = *n9 = 0;//nicht benutzte auf 0 setzen
   if(n==3) return 9;//9=alles ok
   else return n;//sonst Fehler
  }
 else if(*str=='/' && str[1]=='/')
  {//gemaess Beispiel2 einlesen:
   n += sscanf(&str[2],"%d %d//%d %d//%d",n3, n4,n6, n7,n9);
   *n2 = *n5 = *n8 = 0;//nicht benutzte auf 0 setzen
   if(n==6) return 9;
  }
 else if(*str=='/')
  {
   str++;
   n += sscanf(str,"%d",n2);
   if(n!=2) return n;//Fehler
   while(isdigit(*str)) {str++;}
   if(*str=='/')
    {//gemaess Beispiel1 einlesen:
     n += sscanf(&str[1],"%d %d/%d/%d %d/%d/%d",n3, n4,n5,n6, n7,n8,n9);
     return n;
    }
   //gemaess Beispiel3 einlesen:
   n += sscanf(&str[1],"%d %d/%d %d/%d",n2, n4,n5, n7,n8);
   *n3 = *n6 = *n9 = 0;//nicht benutzte auf 0 setzen
   if(n==6) return 9;
  }
 else
  {return 0;}//Fehler
 return n;
}

void obj2bmf(FILE *fp1,FILE *fpmtl,FILE *fp2)
{
 int c;
 uint32 i;
 char zeile[400], *ze;
 if(argflag['V']) printf("OBJ-Datei wird nach BMF konvertiert...\n");//test

 std::vector<MTLmaterial> mtlmat;
 MTLmaterial mtlmatneu;
 int matnr=0,maxmatnr=0;
 while(getline(fpmtl,zeile,400)) //MTL-Datei durchgehen
  {
   if(*zeile==0 || *zeile=='#') continue;
   if(strncmp(zeile,"newmtl",6)==0)
    {ze = &zeile[6];
     while(*ze==' ' || *ze=='\t') ze++; //Leerstellen ueberlesen
     if(mtlmatneu.name[0]!=0) {mtlmat.push_back(mtlmatneu); mtlmatneu.reset();}
     sscanf(ze,"%s",mtlmatneu.name);
     matnr=getmatnr(mtlmat,mtlmatneu.name);
     if(matnr>=0) printf("Fehler: \"%s\" schon vorhanden\n",mtlmatneu.name);
    }
   else if(strncmp(zeile,"Ns",2)==0) {sscanf(&zeile[2],"%f", &mtlmatneu.shininess);}
   else if(strncmp(zeile,"Ni",2)==0) {sscanf(&zeile[2],"%f", &mtlmatneu.ni);}
   else if(strncmp(zeile,"d ",2)==0) {sscanf(&zeile[2],"%f", &mtlmatneu.dissolve);}
   else if(strncmp(zeile,"illum",5)==0) {sscanf(&zeile[5],"%d", &mtlmatneu.illum);}
   else if(strncmp(zeile,"Ka",2)==0)
    {drei_floats_einlesen(&zeile[2], mtlmatneu.ambient);}
   else if(strncmp(zeile,"Kd",2)==0)
    {drei_floats_einlesen(&zeile[2], mtlmatneu.diffuse);}
   else if(strncmp(zeile,"Ks",2)==0)
    {drei_floats_einlesen(&zeile[2], mtlmatneu.specular);}
   else if(strncmp(zeile,"Ke",2)==0)
    {drei_floats_einlesen(&zeile[2], mtlmatneu.emissive);}
   else if(strncmp(zeile,"map_Kd",6)==0)
    {//texturfile einlesen (nur Filename):
     const char *s,*s1;
     for(s= &zeile[6]; *s==' ' || *s=='\t' ; s++) {}
     for(s1=s ;*s!=0 ; s++)
      {
       if(*s=='/') {s1= &s[1];}
      }
     char *t=mtlmatneu.texturfile;
     int i;
     for(s=s1,i=1; i<120 && *s!=0; i++) {*t++ = *s++;}
     *t=0;
     if(i>=120) printf("Fehler: texturfilename zu lang: \"%s\"\n",mtlmatneu.texturfile);
     else if(argflag['V']) printf("texturfile: \"%s\"\n",mtlmatneu.texturfile);
    }
   else if(strncmp(zeile,"map_Bump",8)==0)
    {
     //TODO  Normalentextur
     printf("TODO: Normalentextur noch nicht unterstuetzt\n");//test
    }
   else if(strncmp(zeile,"map_Ks",6)==0)
    {
     //TODO  Speculartextur
     printf("TODO: Speculartextur noch nicht unterstuetzt\n");//test
    }
   else if(strncmp(zeile,"map_d",5)==0)
    {
     //TODO  dissolve-textur?
     printf("TODO: dissolve-textur noch nicht unterstuetzt\n");//test
    }
   else printf("Unbekannte Zeile in .mtl-Datei: \"%s\"\n",zeile);//test
  }
 if(mtlmatneu.name[0]!=0) {mtlmat.push_back(mtlmatneu); mtlmatneu.reset();}
 maxmatnr=mtlmat.size();
 if(argflag['V']) printf("%d Materials eingelesen\n",maxmatnr);//test

 uint32 numVertices=0,  numVn=0;
 std::vector<Vektor3f> vertices; //alle Eckpunkte
 std::vector<Vektor3f> vn; //Normalen-Vektoren fuer alle Dreiecksflaechen
 uint32 numUvs=0; //Anzahl Texturkoordinaten
 std::vector<Uvs> uvs; //alle Texturkoordinaten
 
 uint32 numMeshes=0;
 std::vector<BMFmesh> bmfmesh;
 float ftmp[3];
 Vektor3f vertex;
 BMFmesh mesh;
 bool firstmesh=true;
 while(getline(fp1,zeile,400)) //OBJ-Datei durchgehen
  {
   if(*zeile==0 || *zeile=='#') continue; //Leerzeilen und Kommentarzeilen ignorieren
   if(strncmp(zeile,"mtllib",6)==0) continue; //TODO: ev. pruefen ob name stimmt
   if(strncmp(zeile,"o ",2)==0)
    {if(argflag['V']) printf("Objekt: %s wird eingelesen\n",&zeile[2]);}//test
   else if(strncmp(zeile,"v ",2)==0)
    {//Eckpunkte (vertices) einlesen
     drei_floats_einlesen(&zeile[2], ftmp);
     vertex.x=ftmp[0]*scalefaktor; vertex.y=ftmp[1]*scalefaktor; vertex.z=ftmp[2]*scalefaktor;
     vertices.push_back(vertex);
     numVertices++;
     //printf("vertex.x=%f vertex.y=%f vertex.z=%f\n",vertex.x, vertex.y, vertex.z);//test
    }
   else if(strncmp(zeile,"vt ",3)==0)
    {//Tekturkoordinaten einlesen
     if(bmfversion>=200 && texturflag)
      {
       Uvs uv;
       zwei_floats_einlesen(&zeile[3], &uv.x);
       uvs.push_back(uv);
       numUvs++;
      }
    }
   else if(strncmp(zeile,"vn ",3)==0)
    {//Normalenvektoren einlesen
     drei_floats_einlesen(&zeile[2], ftmp);
     vertex.x=ftmp[0]; vertex.y=ftmp[1]; vertex.z=ftmp[2];
     vn.push_back(vertex);
     numVn++;
    }
   else if(strncmp(zeile,"usemtl",6)==0)
    {
     char *s;
     for(s= &zeile[6]; *s==' ' || *s=='\t'; s++) {}
     matnr=getmatnr(mtlmat,s);
     if(matnr<0) {printf("Fehler: Material \"%s\" nicht gefunden\n",s); matnr=0;}//test
     //printf("meshnr %ld: numVertices=%d matnr=%d\n",numMeshes,numVertices,matnr);//test
     if(firstmesh) firstmesh=false;
     else {bmfmesh.push_back(mesh); numMeshes++;}
     mesh.init(mtlmat[matnr]);
     //for(uint i=0;i<numVertices;i++) //test
     // printf("vertices[%d]={%f %f %f}\n",i,vertices[i].x,vertices[i].y,vertices[i].z);//test
    }
   else if(strncmp(zeile,"s ",2)==0)//Glaettung der Normalenvektoren
    {
     if(option_smooth==0)
      {
       if(zeile[2]=='0' || strncmp(&zeile[2],"off",3)==0) mesh.smooth=0;
       else mesh.smooth=1; //TODO: fuer "ein" waehren mehrere Nummern moeglich
      }
     else if(option_smooth==1) mesh.smooth=1;
    }
   else if(strncmp(zeile,"f ",2)==0)
    {
     //Beispiel: f 2/1/2 3/2/1 1/3/1  #jeweils v/vt/vn  (Vertex/Texturkoord/Normalen)
     uint n,n1=0,n2=0,n3=0,n4=0,n5=0,n6=0,n7=0,n8=0,n9=0;
     //n=sscanf(&zeile[2],"%d/%d/%d %d/%d/%d %d/%d/%d",&n1,&n2,&n3,&n4,&n5,&n6,&n7,&n8,&n9);//geht so nicht
     n=vtn_lesen(&zeile[2],&n1,&n2,&n3,&n4,&n5,&n6,&n7,&n8,&n9);
     if(n!=9) printf("zu wenige Zahlen fuer Dreieck: \"%s\"\n",zeile);//test
     if(n1>numVertices || n4>numVertices || n7>numVertices) printf("Fehler: Indexe zu gross: %d %d %d\n",n1,n4,n7);//test
     if(n1<1 || n4<1 || n7<1) printf("Fehler: Indexe sind falsch: %d %d %d\n",n1,n4,n7);//test
     else {n1--; n4--; n7--;}
     if(bmfversion>=200)
      {
       if(n2>numUvs || n5>numUvs || n8>numUvs) printf("Fehler: Uvs-Indexe zu gross: %d %d %d\n",n2,n5,n8);//test
       if(n2<1 || n5<1 || n8<1)
	{
	 //printf("keine Texturkoordinaten: Indexe 2,5,8: %d %d %d\n",n2,n5,n8);//test
	 texturflag=false;
	}
       else {n2--; n5--; n8--;}
      }
     if(n3>numVn || n6>numVn || n9>numVn) printf("Fehler: Normalen-Indexe zu gross: %d %d %d\n",n3,n6,n9);//test
     if(n3<1 || n6<1 || n9<1) printf("Fehler: Normalen-Indexe falsch: %d %d %d\n",n3,n6,n9);//test
     else {n3--; n6--; n9--;}
     Dreieck d;
     d.vertex1[0] = vertices[n1].x;  d.vertex1[1] = vertices[n1].y;  d.vertex1[2] = vertices[n1].z;
     d.vertex2[0] = vertices[n4].x;  d.vertex2[1] = vertices[n4].y;  d.vertex2[2] = vertices[n4].z;
     d.vertex3[0] = vertices[n7].x;  d.vertex3[1] = vertices[n7].y;  d.vertex3[2] = vertices[n7].z;
     if(n3==n6 && n6==n9)
      {d.normalen[0]=vn[n3].x; d.normalen[1]=vn[n3].y; d.normalen[2]=vn[n3].z;}
     else
      {
       //printf("Test: Durchschnitt der Normalen-Vektoren von 3 Eckpunkten\n");
       d.normalen[0] = (vn[n3].x+vn[n6].x+vn[n9].x)/3; //Durchschnitt der Normalen-Vektoren der 3 Eckpunkte
       d.normalen[1] = (vn[n3].y+vn[n6].y+vn[n9].y)/3;
       d.normalen[2] = (vn[n3].z+vn[n6].z+vn[n9].z)/3;
      }
     /*
     if(mesh.numDreiecke<10) //test
       {printf("Dreieck: %f %f %f  %f %f %f  %f %f %f\n",d.vertex1[0],d.vertex1[1],d.vertex1[2],
	     d.vertex2[0],d.vertex2[1],d.vertex2[2],
	     d.vertex3[0],d.vertex3[1],d.vertex3[2]);}//test
     */
     if(texturflag && numUvs!=0)
      {
       //n2,n5,n8 = Texturkoordinaten-Indexe, speichern:
       d.uvs1 = uvs[n2];
       d.uvs2 = uvs[n5];
       d.uvs3 = uvs[n8];
      }
     mesh.dreiecke.push_back(d);
     mesh.numDreiecke++;
    }
   else
    {printf("Unbekannte Zeile in OBJ-Datei: \"%s\"\n",zeile);}//test
  }
 if(argflag['V']) printf("letzter mesh: numVertices=%d\n",numVertices);//test
 bmfmesh.push_back(mesh); numMeshes++;
 save_bmf(fp2,numMeshes,numUvs,bmfmesh);
}

/*********************** Neue Datei schreiben *************************/
struct Position {
 float x,y,z;
};

void output_uint32(FILE *fp, uint32 u)
{
 for(int i=0;i<4;i++)
  {
   fputc(u&0xFF, fp);
   u >>= 8;
  }
}

/*
void output_uint64(FILE *fp, uint64 u)
{
 for(int i=0;i<8;i++)
  {
   fputc(u&0xFF, fp);
   u >>= 8;
  }
}
*/

void output_float(FILE *fp, float fx)
{
 union {float f; uint32 c;} fc;
 fc.f = fx;
 fputc(fc.c&0xFF, fp);
 fputc((fc.c>>8)&0xFF, fp);
 fputc((fc.c>>16)&0xFF, fp);
 fputc((fc.c>>24)&0xFF, fp);
}

void output_vec3(FILE *fp, float* data)
{
 output_float(fp, data[0]);
 output_float(fp, data[1]);
 output_float(fp, data[2]);
}

void output_string(FILE *fp, const char* str)
{
 int c;
 while((c= *str++)!=0)
  {fputc(c,fp);}
 fputc(0,fp);
}

Position get_vertex(float *p)
{
 Position vertex;
 vertex.x = p[0];
 vertex.y = p[1];
 vertex.z = p[2];
 return vertex;
}

Vektor3f get_vektor3f(float *p)
{
 Vektor3f vekt;
 vekt.x = p[0];
 vekt.y = p[1];
 vekt.z = p[2];
 return vekt;
}

uint32 vertex_suchen(uint32 imax, std::vector<Position>& positions, Position& vertex)
{
 for(uint32 i=0;i<imax;i++)
  {
   if(positions[i].x == vertex.x
      && positions[i].y == vertex.y
      && positions[i].z == vertex.z)
    //TODO: Rundungsfehler ausgleichen
    return i+1;
  }
 return 0;
}

bool istgleich(float a,float b)
{
 if(a==b) return true;
 const float dd=1e-4;
 if(a>b)
  {if(a < b+dd) return true;}
 else
  {if(b < a+dd) return true;}
 return false;
}

bool istgleich(Position& a,Position b)
{
 return (istgleich(a.x, b.x) && istgleich(a.y, b.y) && istgleich(a.z, b.z));
}

void save_bmf(FILE *fp,uint32 numMeshes,uint32 numUvs,std::vector<BMFmesh> mesh)
{
 if(bmfversion>=200)
  {
   fprintf(fp,"BMF%2d.%02d",bmfversion/100,bmfversion%100);
  }
 output_uint32(fp,numMeshes);
 output_uint32(fp,numUvs);
 if(argflag['V']) printf("numMeshes=%d\n",numMeshes);//test
 if(argflag['V']) printf("Anzahl Texturkoordinaten: numUvs = %d\n",numUvs);//test
 for(uint32 i=0; i<numMeshes; i++)
  {
   if(argflag['V']) printf("mesh %d wird gespeichert\n",i);//test
   save_mesh(fp, numUvs, mesh[i]);
  }
}

void multcopy(int multflag, float* ziel, float* quelle, float* diffuse)
{
 if(multflag==0)
  {
   for(int i=0;i<3;i++) {ziel[i]=quelle[i];}
  }
 else
  for(int i=0;i<3;i++)
    {ziel[i] = quelle[i]*diffuse[i];}
}

void save_mesh(FILE *fp,uint32 numUvs,BMFmesh &mesh)
{
 uint32 i, index, index1, index2, index3;
 uint32 next_index=1;
 uint32 numVertices=0;
 uint32 numIndices=0;
 std::vector<Position> positions;
 //std::vector<Position> normals;
 std::vector<Vektor3f> normals;
 std::vector<Uvs> dreieckuvs;
 std::vector<uint32> indices;
 Position vertex1,vertex2,vertex3;
 Uvs uvs1,uvs2,uvs3;
 
 if(bmfversion>=200)
  {
   float ambient[3];
   multcopy((option_mult&AMBIENT_MULT), ambient, mesh.mat.ambient, mesh.mat.diffuse);
   output_vec3(fp, ambient);
  }
 output_vec3(fp, mesh.mat.diffuse);
 float specular[3];
 multcopy((option_mult&SPECULAR_MULT), specular, mesh.mat.specular, mesh.mat.diffuse);
 output_vec3(fp, specular);
 output_vec3(fp, mesh.mat.emissive);
 output_float(fp, mesh.mat.shininess);
 if(bmfversion>=200)
  {
   output_string(fp, mesh.mat.texturfile);
   output_string(fp, mesh.mat.normtexturfile);
  }
#ifdef TEST_VERGLEICH //test
 printf("mesh.mat.ambient={%f %f %f}\n",mesh.mat.ambient[0], mesh.mat.ambient[1], mesh.mat.ambient[2]);
 printf("mesh.mat.diffuse={%f %f %f}\n",mesh.mat.diffuse[0], mesh.mat.diffuse[1], mesh.mat.diffuse[2]);
 printf("mesh.mat.specular={%f %f %f}\n",mesh.mat.specular[0], mesh.mat.specular[1], mesh.mat.specular[2]);
 printf("mesh.mat.emissive={%f %f %f}\n",mesh.mat.emissive[0], mesh.mat.emissive[1], mesh.mat.emissive[2]);
 printf("mesh.mat.shininess=%f\n",mesh.mat.shininess);
#endif
 if(argflag['V']) printf("numDreiecke=%d\n",mesh.numDreiecke);//test
 
 if((numUvs==0 || !mesh.smooth) && !autosmoothflag)
 {//wenn keine Texturkoordinaten verwendt oder kein Smooth: einfachere, sparsamere Variante:
 for(i=0;i<mesh.numDreiecke;i++) //Normalenvektoren anpassen, Vertex inklusive Texturkoordinaten
  {                              //fuer jeden Eckpunkt pro Dreieck erstellen.
   Vektor3f normalvek = get_vektor3f(mesh.dreiecke[i].normalen);
   vertex1 = get_vertex(mesh.dreiecke[i].vertex1);
   uvs1 = mesh.dreiecke[i].uvs1;
   if(mesh.smooth) index1 = vertex_suchen(numVertices, positions, vertex1);
   else            index1 = 0;
   if(index1==0)
    {
     positions.push_back(vertex1); numVertices++;
     index1 = next_index++;
     normals.push_back(normalvek);
     dreieckuvs.push_back(uvs1);
    }
   else
    {normals[index1-1] += normalvek;} //aufsummieren der Normalen-Vektoren
   indices.push_back(index1-1);
   
   vertex2 = get_vertex(mesh.dreiecke[i].vertex2);
   uvs2 = mesh.dreiecke[i].uvs2;
   if(mesh.smooth) index2 = vertex_suchen(numVertices, positions, vertex2);
   else            index2 = 0;
   if(index2==0)
    {
     positions.push_back(vertex2); numVertices++;
     index2 = next_index++;
     normals.push_back(normalvek);
     dreieckuvs.push_back(uvs2);
    }
   else
    {normals[index2-1] += normalvek;}
   indices.push_back(index2-1);
   
   vertex3 = get_vertex(mesh.dreiecke[i].vertex3);
   uvs3 = mesh.dreiecke[i].uvs3;
   if(mesh.smooth) index3 = vertex_suchen(numVertices, positions, vertex3);
   else            index3 = 0;
   if(index3==0)
    {
     positions.push_back(vertex3); numVertices++;
     index3 = next_index++;
     normals.push_back(normalvek);
     dreieckuvs.push_back(uvs3);
    }
   else
    {normals[index3-1] += normalvek;}
   indices.push_back(index3-1);
   
   numIndices += 3;
  }
 } else {
 //wenn Texturkoordinaten verwendet und Smooth: kompliziertere Variante:
 for(i=0;i<mesh.numDreiecke;i++) //Vertex inklusive Texturkoordinaten
  {                              //fuer jeden Eckpunkt pro Dreieck erstellen.
   Vektor3f normalvek = get_vektor3f(mesh.dreiecke[i].normalen);
   vertex1 = get_vertex(mesh.dreiecke[i].vertex1);
   uvs1 = mesh.dreiecke[i].uvs1;
   positions.push_back(vertex1); numVertices++;
   index1 = next_index++;
   normals.push_back(normalvek);
   dreieckuvs.push_back(uvs1);
   indices.push_back(index1-1);
   
   vertex2 = get_vertex(mesh.dreiecke[i].vertex2);
   uvs2 = mesh.dreiecke[i].uvs2;
   positions.push_back(vertex2); numVertices++;
   index2 = next_index++;
   normals.push_back(normalvek);
   dreieckuvs.push_back(uvs2);
   indices.push_back(index2-1);
   
   vertex3 = get_vertex(mesh.dreiecke[i].vertex3);
   uvs3 = mesh.dreiecke[i].uvs3;
   positions.push_back(vertex3); numVertices++;
   index3 = next_index++;
   normals.push_back(normalvek);
   dreieckuvs.push_back(uvs3);
   indices.push_back(index3-1);
   
   numIndices += 3;
  }
 //jetzt noch Smoothen der Normalenvektoren:
 std::vector<Vektor3f> neuNormals;
 for(i=0;i<numIndices;i++)
  {
   Vektor3f normalvek = normals[i];
   Position pos = positions[i];
   for(uint32 j=0;j<numIndices;j++)
    {
     if(j!=i && istgleich(positions[j],pos))
      {
       if(!autosmoothflag || vektorwinkel2(normalvek,normals[j]) <= autosmooth)
       normalvek += normals[j];
      }
    }
   neuNormals.push_back(normalvek);
  }
 for(i=0;i<numIndices;i++)
   {normals[i] = neuNormals[i];}
 neuNormals.clear();
 }
 
 if(mesh.smooth)
  {
   //die Normalenvektoren wieder zu Eiheitsvektoren machen, entspricht Durchschnitt von Aufsummierung
   for(i=0;i<next_index-1;i++) {normals[i].einheitsvektor();}
  }
 if(numVertices != positions.size())
  printf("da stimmt was nicht: numVertices: %d %ld\n",numVertices,positions.size()); //test
 //printf("positions.size(): %ld\n",positions.size());//test
 //printf("indices.size(): %ld\n",indices.size());//test
 if(argflag['V']) printf("normals.size(): %d\n",(int)normals.size());//test
 if(argflag['V']) printf("numVertices=%d\n",numVertices);//test
 if(argflag['V']) printf("numIndices=%d\n",numIndices);//test
 output_uint32(fp,numVertices);
 output_uint32(fp,0);//Reserve (in erster Version war numVertices uint64)
 output_uint32(fp,numIndices);
 output_uint32(fp,0);//Reserve (in erster Version war numIndices uint64)
 for(uint32 i=0; i<numVertices; i++)
  {
#ifdef TEST_VERGLEICH
   printf("positions[i]={%f %f %f}\n",positions[i].x, positions[i].y, positions[i].z);//test
   printf("  normals[i]={%f %f %f}\n",normals[i].x, normals[i].y, normals[i].z);//test
#endif
   output_float(fp, positions[i].x+offset[0]);
   output_float(fp, positions[i].y+offset[1]);
   output_float(fp, positions[i].z+offset[2]);
   output_float(fp, normals[i].x);
   output_float(fp, normals[i].y);
   output_float(fp, normals[i].z);
   if(bmfversion>=200 && numUvs!=0)
    {
     output_float(fp, dreieckuvs[i].x);
     output_float(fp, dreieckuvs[i].y);
     //if(i<10) printf("uvs[%d]={%f %f}\n", i, dreieckuvs[i].x, dreieckuvs[i].y);//test
    }
  }
 for(uint32 i=0; i<numIndices; i++)
  {
   output_uint32(fp, indices[i]);
  }
}
