/* hello.cc  HelloWorld fuer OpenCL
Multiplikation von 2 Zahlenfeldern
Beschreibung auf http://www.rolfp.ch/computer/parallelprogramme/teil3.html
*/

#include <stdio.h>
#include <thread>
#include "hello.h"
#include "myopencl.h"

#define NUR_OPTIONEN
#include "hello_kernel.cc"
#undef NUR_OPTIONEN

//#define DEBUG

static char kernel_name[200]="hello_kernel.cc";

void opt_kernel_name_setzen(int opt)
{
 if(opt==0) sprintf(kernel_name,"kernelversionen/hello_kernel-naiv.cc");
 else sprintf(kernel_name,"kernelversionen/hello_kernel-opt%d.cc",opt);
}

#define XMAX 100000
#define YMAX 12000
#define ZMAX XMAX

#define FLOP (2L*XMAX*YMAX) //soviele "Floating OPerations" werden gebraucht

// Serielle Variante:
void hello_calc(const float *A,const float *B,float *C,int na,int nb,int nc)
{
 // na..nc = Groessen der Felder
 int z=0;
 for(int x=0;x<na;x++)
  {
   float sum=0;
   for(int y=0;y<nb;y++)
     sum += A[x]*B[y];
   C[z++] = sum;
   if(z>nc) {printf("Fehler: Feld C zu klein. z=%d\n",z); return;}
  }
}

int arg_maxreg=0; //Falls Grafikkarte nur wenige Register hat (0=beliebig viele)
int variante=0;
#ifndef OPTIMIERUNG
#define OPTIMIERUNG 0
#endif
int optimierung=OPTIMIERUNG;
int arg_TPB=0;

void argscan(int argc,char **argv)
{
 bool ok=true;
 for(int i=1;i<argc;i++)
  {
   if(isdigit(*argv[i])) sscanf(argv[i],"%d",&variante);
   else if(*argv[i]=='-')
    {if(argv[i][1]=='r') sscanf(&argv[i][2],"%d",&arg_maxreg);
     else if(argv[i][1]=='o')
      {
       sscanf(&argv[i][2],"%d",&optimierung);
       opt_kernel_name_setzen(optimierung);
      }
     else if(argv[i][1]=='t') sscanf(&argv[i][2],"%d",&arg_TPB);
     else ok=false;
    }
   else ok=false;
  }
 if(!ok)
  {
   printf("Aufruf:\n%s [-Optionen] <Variante>\n",argv[0]);
   printf(" Variante: 0=alles, 1=nur CPU, 2=nur OpenCL\n");
   printf(" Optionen:\n");
   printf("   r=Beschraenkung auf z.B. 32 Register: -r32\n");
   printf("   o=Optimierungs-Version ausprobieren (1...5)\n");
   printf("   t=TPB setzen (Threads Per Block)\n");
   exit(0);
  }
}

int main(int argc,char **argv)
{
 int i;
 bool ok=true;
 float *ma,*mb,*mc; //Felder A B C
 ma = new float[XMAX]; //Feld A
 mb = new float[YMAX]; //Feld B
 mc = new float[ZMAX]; //Resultat-Feld
 if(argc>=2) argscan(argc,argv);
 for(i=0;i<XMAX;i++)
   ma[i] = (random()&0xFFFF)/float(0xFFFF)*10; //Zufallszahl zwischen 0 und 10
 for(i=0;i<YMAX;i++)
   mb[i] = (random()&0xFFFF)/float(0xFFFF)*10; //Zufallszahl zwischen 0 und 10
 for(i=0;i<ZMAX;i++) mc[i]=0; //Zielfeld loeschen
 printf("%d Zufallszahlen erstellt.\n",XMAX+YMAX);

 if(variante==1 || variante==0)
  {printf("\nSeriell auf CPU gerechnet:\n");
   stoppuhr_reset();
   hello_calc(ma,mb,mc,XMAX,YMAX,ZMAX);
   int usec=stoppuhr_read();
   printf("Rechenzeit CPU: ");
   zeit_print(usec,FLOP);
  }
 if(variante==2 || variante==0)
  {
   printf("\nAuf Grafikkarte mit OpenCL gerechnet:\n");
   for(i=0;i<ZMAX;i++) mc[i]=0; //Zielfeld loeschen
   copy_to_device(ma,mb,mc,XMAX,YMAX,ZMAX);
   ok=kernel_compilieren(kernel_name,arg_maxreg);
#ifdef DEBUG
   if(ok) printf("Kernel \"%s\" erfolgreich compiliert.\n",kernel_name);
#endif
   if(!ok) {printf("Fehler3: kernel_compilieren() misslungen.\n"); exit(1);}
   stoppuhr_reset();
#ifdef DEBUG
   printf("optimierung=%d TPB=%d\n",optimierung,arg_TPB);//test
#endif
   ok=kernel_aufrufen(optimierung,arg_TPB);
   if(ok)
    {
     int usec=stoppuhr_read();
     copy_from_device(mc,ZMAX);
     int usec2=stoppuhr_read();
     printf("Rechenzeit mit OpenCL: "); zeit_print(usec,FLOP);
     printf("  mit zurueckkopieren: "); zeit_print(usec2);
    }
   device_speicher_freigeben();
  }
 if(ok)
  {
   printf("\nErgebnisfeld: "); //einige Ergebnisse anzeigen
   for(int i=0;i<3;i++) printf("%f ",mc[i]);
   printf(".... ");
   for(int i=ZMAX-3;i<ZMAX;i++) printf("%f ",mc[i]);
   printf("\n");
   //Ergebnisse ueberpruefen:
   float *mtest=new float[ZMAX];
   for(i=0;i<ZMAX;i++) mtest[i]=0; //neue Zielfeld loeschen
   hello_calc(ma,mb,mtest,XMAX,YMAX,ZMAX);
   for(i=0;i<ZMAX;i++)
    if(!fastgleich(mc[i],mtest[i]))
     {printf("Differenz an Position %d: %f statt %f\n",i,mc[i],mtest[i]);
      ok=false; break;
     }
   if(ok) printf("Ergebnisse erfolgreich ueberprueft.\n");
   else  {printf("Fehlerhafte Ergebnisse!\n");}
   delete[] mtest;
  }
 delete[] ma;
 delete[] mb;
 delete[] mc;
 return 0;
}

/****************************** Stoppuhr: ******************************/
#include <time.h>
#include <sys/time.h>
static struct timeval stoppuhr_tv0, stoppuhr_tv;
static struct timezone stoppuhr_tz0, stoppuhr_tz;

void stoppuhr_reset()
{
 gettimeofday(&stoppuhr_tv0,&stoppuhr_tz0);
}

int stoppuhr_read() //Zeit seit letztem stoppuhr_reset() in usec
{
 int usec,sec;
 gettimeofday(&stoppuhr_tv,&stoppuhr_tz);
 usec = stoppuhr_tv.tv_usec - stoppuhr_tv0.tv_usec;
 if((sec=stoppuhr_tv.tv_sec-stoppuhr_tv0.tv_sec)!=0) usec += sec*1000000;
 return usec;
}

void zeit_print(int usec,long flop)
{
 if(usec>=500000)
      printf("%d.%03d sec",usec/1000000,(usec/1000)%1000);
 else printf("%d.%03d ms",usec/1000,usec%1000);
 if(flop>0)
  {
   double gflops=flop/(usec*1e-6)/1e9;
   if(gflops<2)        printf("  (%.3f GFLOPS)\n",gflops);
   else if(gflops<200) printf("  (%.1f GFLOPS)\n",gflops);
   else                printf("  (%.0f GFLOPS)\n",gflops);
  }
 else printf("\n");
}
/**************************** Ende Stoppuhr ****************************/

bool fastgleich(float a,float b)
{
 const float fasteins=0.9999, guteins=1.0001;
 if(a==b) return true;
 if(a==0 || b==0) return false;
 double eins=a/b;
 return (eins>=fasteins && eins<=guteins);
}

void transponieren(float *matrix,int breite,int hoehe)
{
 float *neu=new float[breite*hoehe];
 for(int x=0;x<breite;x++)
  for(int y=0;y<hoehe;y++)
   neu[x*hoehe+y] = matrix[y*breite+x];
 for(int i=0;i<breite*hoehe;i++)
   matrix[i]=neu[i];
 delete[] neu;
}
