;-----------------------------------------------------------------------------
;* scatter328.asm  wie scatter.asm aber fuer ATmega328P
;*                              	 letzte Aenderung: 26.3.2010
;* Version: 1.04
;* Autor: Rolf Pfister
;* Copyright: Freeware
;* Prozessor: Atmega328P, 8MHz
;* Schaltschema: scatter.png oder scatter3.png ohne MOC und ohne S2
;*	 PortB: B0, B1, B2 mit LEDs verbunden, B4 Basis auf NPN-Transistor
;*       PortC:	C4 ueber 100 Ohm mit Taster verbunden, wenn gedrueckt == 0
;*	 PortD:	D2..D7 fuer LCD-Anzeige
;*   Variante 1: ADC0 (Pin 23), Messspannung nach Verstaerkung mit OP
;*
;* History:
;* 27.11.2009	Erstellung aus thermologger.asm
;* 30.11.09	Version 1.0, Verwendung von C4 fuer Tastschalter
;*		(statt "PIND, 2" jetzt also "PINC, 4")
;*		LCD-Modul angeschlossen an D2...D7
;* 2.12.09	Fehler in gepacktspeichern korrigiert
;* 3.12.09	Version 1.01, Kondensator beim Detektor und deshalb groessere
;*		Wartezeit (300 statt 200 ms) bis Messlicht stabil ist.
;* 4.12.09	Version 1.02,  Noch nicht getestet!
;*		Erweiterung zum Magnetruehrer schalten:
;*		B4 auf weiteren Transistor, Diode im MOC3040 ueber 100 Ohm
;*		angesteuert. scatter3.png (aktuelle Schaltung noch ohne MOC)
;* 5.1.2010	1.03 neue Version. Groessere Aenderungen gemacht.
;*		Programm vereinfacht und laufende Mittelung eingebaut:
;*		AD-Wandler soll immer laufen, Wert und summierter Wert
;*		im RAM gespeichert (sensor1adwert und summeadwert, max N=64)
;* 7.1.2010	Probleme beim Schreiben ins Flash: pruefen ob die Daten auch
;*		stimmen. Korrektur in adcomplete: r30:r31 nicht mehr global.
;* 8.1.2010	Ueberpruefung von spm mittels lpm funktioniert jetzt.
;*		Trick war RWWSRE zu setzen.
;* 6.2.10	Aenderungen fuer ATmega328P
;* 25.3.10 1.04	Anpassung fuer Schuetteln zwischendurch
;*		(mit Schuttelmaschine, .../tiny13a/.../servo.asm)
;*		Leitung PB4 ueber BC550 verwendet (urspruenglich fuer Magnetr)
;* 26.3.10	Auf negative Logik umgestellt (cbi PORTB,4 mit sbi vertauscht)
;*
;-----------------------------------------------------------------------------
.include "m328Pdef.inc" 	; Definitionsdatei fuer den Prozessortyp
.equ SRAMSTART=SRAM_START	;0x60 beim Atmega8, 0x100 beim Atmega328P
;.define MYAVR		;zum Austesten auf myAVR-Board mit 3.6864MHz
.equ	ANZAHLSENSOREN=1	;(vom thermologger uebernommen)
.equ	SENSORTYP=83		;Phototransistor BPX38
.equ	ANSCHLUSSVARIANTE=1	;Verstaerkt mit BC550 oder OP, 10kOhm am Emitt.
;.equ	ANSCHLUSSVARIANTE2=3	;Anschluss des 2.Sensors
.equ	SENSORSPEICHERN=1	;1=1.Sensor, 2=Wert vom 2.Sensor speichern
;.equ	DATENFORMAT=0		;0=ungepackt = 2 Byte pro Messpunkt
.equ	DATENFORMAT=1	;1 oder 2=Einfaches packen = 0.5 bis 2 Byte pro Mess.
;.define EEPROMVERSION
.define FLASHVERSION
.define BOOTBLOCK2K
;.define BOOTBLOCK1K
;; zum testen die folgenden 2 Zeilen auskommentiert:
.define LCDANZEIGE	;Anzeige des Messwertes auf einem LCD-Modul
;.define STACKERROR	;Pruefen auf Stack-Fehler
;.define MAGNETRUEHRER  ;falls Magnetruehrer ein/aus-schalten gebraucht wird
.define SCHUETTLER	;wenn Schuettelmaschine gebraucht wird
;.define MULDIV         ;falls Multiplikation und Division gebraucht wird
;.define DEBUGGING

;-----------------------------------------------------------------------------
;Datenformat:
; 1 Byte DATENFORMAT
; 1 Byte SENSORTYP
; 1 Byte ANSCHLUSSVARIANTE
; 1 Byte Messperiode = Anzahl Sekunden zwischen 2 Messungen
; 2 Byte N = Anzahl Messpunkte
;Wenn DATENFORMAT==0: N*2 Byte ungepackte Daten
;sonst je nach Packvariante weniger als N*2 Bytes
;
;DATENFORMAT==1:
; Wenn erstes Bit gesetzt, dann sind die naechsten 3 Bits
; die Differenz zum vorherigen Wert (Bereich -4 bis +3).
; Wenn das erste Bit nicht, aber das 2.Bit gesetzt ist,
; dann sind die naechsten 6 Bits +-4 die Differenz zum vorherigen Wert.
; (+-4 heisst: wenn Diff. positiv noch 4 addieren, wenn negativ 4 subtrahieren)
; Somit Wertebereich 4 bis 35 und -5 bis -36.
; Wenn weder das erste noch das 2. Bit gesetzt sind (c & 0xC0 == 0), dann
; sind die naechsten 10 Bits ein absoluter Wert.
; Also kurz zusammengefasst:
;       1X -> 4Bit, davon 3Bit Differenzwert (-4 ... +3)
;       01 -> 8Bit, davon 6Bit Differenzwert (-36 ... -5 und +4 ... +35)
;       00 -> 12Bit, davon 10Bit Absolutwert vorzeichenlos (0 ... 1023)
; Die Werte sind Messwerte von AD-Wandlung.
;
;DATENFORMAT==2:
; Wie 1 aber mit 14Bit-Absolutwerten:
;       00 -> 16Bit, davon 14Bit Absolutwert vorzeichenlos (0 ... 16383)
; Die Werte sind GradCelsius*10 (z.B. 357 bedeutet 35.7 Grad)
;
;-----------------------------------------------------------------------------
;Speicherbelegung:
;Global in Interrupts benutzt:
; r11:r10 Millisekunden, r12 Sekunden, r13 Minuten, r14 Stunden, r15 Tage
; r4=Anzahl Aufsummierungen in AD-Wandler
;Global benutzt:
; r5: Vorheriges Nibble beim gepackten Speichern (Gueltig=r5&0x10)
; r7:r6 letzter Messwert
; r8 = Zwischenspeicher fuer imflashspeichern
; r9 = Flags:
;   r9 Bit0 = Flag fuer erste Page fuer imflashspeichern
;   r9 Bit1 = Nicht-Speichern-Flag (um einen Messwert mal nicht zu speichern)
; r23:r22 Restlicher Platz in Byte
; r25:r24 Anzahl bisher gespeicherte Messpunkte
; r27:r26 == XH:XL = Zeiger auf zu speichernde Daten
; r29:r28 == YH:YL = Zeiger auf am Schluss zu speichernde Anzahl Messpunkte
; 
;im Hauptprogramm:
; r17:r16 fuer kurze Wartezeit (immer innerhalb 1...255)
; r16 auch als temporaere Variable
; r1=Messperiode Anzeige mit LEDs, 1=rot 2=gruen 4=gelb
; r2=Flag fuer Mittelung, 0=ohne, 1=mit
; r18=Sensor der gespeichert und auf LCD angezeigt werden soll
; r19=Messperiode in Sekunden
; r3=1
; r20,r21 Sollzeit
; r31:r30 == ZH:ZL = Zeiger zum umkopieren (in page1zurueckkopieren)
; 
;-----------------------------------------------------------------------------

.ifdef MYAVR
.equ QUARTZ=3686400	;zum Austesten mit myAVR-Board
.else
.equ QUARTZ=8000000	;Takt in MHz
.endif

.equ PAGESIZEB = PAGESIZE*2 ; PAGESIZEB is page size in BYTES, not words

;; fuer den ATmega8
;.ifdef BOOTBLOCK2K
;.org LARGEBOOTSTART
;.endif
;.ifdef BOOTBLOCK1K
;.org THIRDBOOTSTART
;.endif

;; fuer den ATmega328P
.ifdef BOOTBLOCK4K
.org LARGEBOOTSTART
.endif
.ifdef BOOTBLOCK2K
.org THIRDBOOTSTART
.endif
.ifdef BOOTBLOCK1K
.org SECONDBOOTSTART
.endif

;-----------------------------------------------------------------------------
; Variablen im SRAM:
.equ lcdpos=SRAMSTART+PAGESIZEB		;LCD-Position
.equ magnetschaltzeit=SRAMSTART+PAGESIZEB+1
.equ sollwertH=SRAMSTART+PAGESIZEB+2	;Sollwert fuer Thermostat
.equ istwertL=SRAMSTART+PAGESIZEB+3
.equ istwertH=SRAMSTART+PAGESIZEB+4	;Istwert fuer Thermostat
.equ widerstandL=SRAMSTART+PAGESIZEB+5
.equ widerstandH=SRAMSTART+PAGESIZEB+6
.equ sensor2gradL=SRAMSTART+PAGESIZEB+7
.equ sensor2gradH=SRAMSTART+PAGESIZEB+8 ;Wert von Sensor2 in GradCelsius*10
.equ lcdn0=SRAMSTART+PAGESIZEB+9
.equ lcdn1=SRAMSTART+PAGESIZEB+10
.equ lcdn2=SRAMSTART+PAGESIZEB+11
.equ lcdn3=SRAMSTART+PAGESIZEB+12	;Zahlen fuers LCD
.equ sensor1adwertL=SRAMSTART+PAGESIZEB+13
.equ sensor1adwertH=SRAMSTART+PAGESIZEB+14 ;Wert von AD-Wandler fuer Sensor1
.equ summeadwertL=SRAMSTART+PAGESIZEB+15
.equ summeadwertH=SRAMSTART+PAGESIZEB+16   ;Summe ueber letzte N Werte
.equ nmittel=SRAMSTART+PAGESIZEB+17        ;N (maximal==64)
.equ ringpuffer=SRAMSTART+PAGESIZEB+18     ;Platz fuer 64 2-Byte-Werte
.equ ringpufferende=SRAMSTART+PAGESIZEB+18+128 ;18+128=146
.equ ringrL=SRAMSTART+PAGESIZEB+146      ;Lesezeiger auf Ringpuffer
.equ ringrH=SRAMSTART+PAGESIZEB+147      ;Lesezeiger auf Ringpuffer
.equ ringwL=SRAMSTART+PAGESIZEB+148      ;Schreibzeiger auf Ringpuffer
.equ ringwH=SRAMSTART+PAGESIZEB+149      ;Schreibzeiger auf Ringpuffer
.equ adcsummeL=SRAMSTART+PAGESIZEB+150 ;nur in adcomplete gebraucht
.equ adcsummeH=SRAMSTART+PAGESIZEB+151 ;nur in adcomplete gebraucht
.equ flashpuffer=SRAMSTART+PAGESIZEB+152 ;Puffer zum SPM ueberpruefen
.equ zpufferL=SRAMSTART+2*PAGESIZEB+152  ;Zeiger auf den Puffer
.equ zpufferH=SRAMSTART+2*PAGESIZEB+153  ;Zeiger auf den Puffer
.equ adrunflag=SRAMSTART+2*PAGESIZEB+154   ;ev. neues hier eintragen!!
;.equ neu=SRAMSTART+2*PAGESIZEB+155   ;ev. neues hier eintragen!!
.equ errorflag=SRAMSTART+2*PAGESIZEB+155 ;letztes im SRAM benutztes Byte
.equ errorflag2=SRAMSTART+2*PAGESIZEB+156 ;letztes im SRAM benutztes Byte
;falls errorflag dann vom Stack ueberschriben wird --> Fehlerabbruch

;-----------------------------------------------------------------------------
; Reset und Interruptvectoren	;VNr. Beschreibung
begin:	jmp	main		; 1   Power On Reset
	jmp leererint		; 2   Int0-Interrupt
	jmp leererint		; 3   Int1-Interrupt
	jmp leererint		; 4   PCINT0 Pin Change Interrupt
	jmp leererint		; 5   PCINT1 Pin Change Interrupt
	jmp leererint		; 6   PCINT2 Pin Change Interrupt
	jmp leererint		; 7   WDT Watchdog Iime-out
	jmp leererint		; 8   TIMER2 COMPA Compare Match A
	jmp leererint		; 9   TIMER2 COMPB Compare Match B
	jmp leererint		; 10  TIMER2 OVF Overflow
	jmp leererint		; 11  TC1 Capture
	jmp leererint		; 12  TC1 Compare Match A
	jmp leererint		; 13  TC1 Compare Match B
	jmp leererint		; 14  TC1 Overflow
	jmp leererint		; 15  TC0 Compare Match A
	jmp leererint		; 16  TC0 Compare Match B
	jmp tc0over		; 17  TC0 Overflow
	jmp leererint		; 18  SPI, STC = Serial Transfer Complete
	jmp leererint		; 19  UART Rx Complete
	jmp leererint		; 20  UART Data Register Empty
	jmp leererint		; 21  UART Tx Complete
	jmp adcomplete		; 22  ADC Conversion Complete
	jmp leererint		; 23  EEPROM ready
	jmp leererint		; 24  Analog Comperator
	jmp leererint		; 25  TWI (I2C) Serial Interface
	jmp leererint		; 26  SPM READY Store Program Memory ready
leererint:
	reti
;-----------------------------------------------------------------------------
; Start, Power On, Reset
main:	ldi r16, RAMEND&0xFF
        out SPL, r16  	     ; Init Stackpointer L
	ldi r16, RAMEND>>8
        out SPH, r16  	     ; Init Stackpointer H

;; Init-Code
	ldi r16, 0xFF		;Alle Leitungen auf Ausgang
        out DDRB, r16		;Data Direction Register B setzen
;	ldi r16, 0x10
;	out PORTB, r16		;alle LEDs loeschen, Magnetruehrer ein
	ldi r16, 0x00		;positive Logik: 0x10, negative: 0x00
	out PORTB, r16		;alle LEDs loeschen, Schuettler in Aus-Position
	ldi r17, 0x10		;Pullup fuer Taster
	out PORTC, r17		;Pullup-Widerstand an C4 fuer Taster
        ldi r17, 0xFC
        out DDRD, r17		;6 Obere Leitungen PortD auf Ausgang (fuer LCD)

;; Timer ist auf dem ATmega328P anders als beim ATmeag8:
	ldi r17, 0x01
	sts TIMSK0, r17		;Timer/Counter0 Overflow einschalten
	ldi r17, 3		;1=nicht teilen, 2=durch 8, 3=64, 4=256, 5=1024
	out TCCR0B, r17		;Vorteiler auf 64 setzen
;Bei 8MHz wird somit in 8us Schritten gezaehlt (bei 3.6864MHz etwa 8*2.17us)
;Um alle 1000usec einen Interrupt zu erhalten muss der Zaehler jeweils bei
;256-125=131 gestartet werden.
;(bei 3.6864MHz 256-57.6=198.4 oder jeweils 2*57 und 3*58 us)
.ifdef MYAVR
.equ	ZAEHLER =198		; 131 bei 8MHz, 198 oder 199 bei 3.6864MHz
.else
.equ	ZAEHLER =131		; 131 bei 8MHz, 198 oder 199 bei 3.6864MHz
.endif
	ldi r17, ZAEHLER
	out TCNT0, r17

.ifdef FLASHVERSION
	ldi r17, (1<<IVCE)	;Veraendern von IVSEL vorankuendigen
	out MCUCR, r17
	ldi r17, (1<<IVSEL)	;Interrupttabelle auch bei Bootadresse
	out MCUCR, r17
.endif
	clr r9			; Flags loeschen
	clr r10
	clr r11			; r11r10 = Millisec
	clr r12			; Sekunden
	clr r13			; Minuten
	clr r14			; Stunden
	clr r15			; Tage, auf 0 setzen
	sts errorflag, r15	; errorflag auf 0 setzen
	sts errorflag2, r15	; errorflag2 auf 0 setzen
	sei			;Interrupt erlauben

; AD-Wandler initialisieren:
	ldi r16, 8		;Anzahl Aufsummierungen 
	mov r4, r16		;fuer AD-Wandler
	ldi XH, 0
	ldi XL, 0		;X auf Anfang im EEPROM oder FLASH setzen
;	ldi r16, 0x40		;ADC0 und Referenz=AVCC (+ externem Condenser)
;	ldi r16, 0xC0		;ADC0 und Referenz=2.56V, beim 328P ca 1.095V
	ldi r16, 0x00		;ADC0 und Referenz=extern, 2.6V mit Spannungs-
				;teiler 2.2k, 2.0k realisiert.
	sts ADMUX, r16
;Prescaler: 5=32 6=64 7=128, 8MHz/64=125kHz (muss im Bereich 50-200 liegen)
;AD-Wandlung braucht jeweils 13 Takte also 104usec
.ifdef MYAVR
.equ	PRESCALER=5
.else
.equ	PRESCALER=6
.endif
;r16 = Prescaler+0b11001000 (ADEN+ADSC+ADIE = Enable,Einzelstart,Interrupt)
;ADIF (0x10) wird automatisch vom AD-Wandler gesetzt wenn Wandlung fertig.
	ldi r16, 0b11001000 + PRESCALER
	sts ADCSRA, r16		;Prescaler und Enable
		;ab hier wird also der AD-Wandler dauernd laufen
	clr r1			;Start-Parameter
	inc r1			;Start mit Roter LED
	mov r3, r1		;r3=1
	mov r2, r1		;r2=1 -> mit Mittelung (0=ohne)

.ifdef EEPROMVERSION
	.equ MAX=512		;Atmega8 hat 512 Bytes = 256 WORDs EEPROM
.endif
.ifdef BOOTBLOCK4K
	.equ MAX=LARGEBOOTSTART*2
.endif
.ifdef BOOTBLOCK2K
	.equ MAX=THIRDBOOTSTART*2
.endif
.if DATENFORMAT==0
	ldi r22, low(MAX-6)	;Freier Speicher minus 6 Bytes Kopfdaten 
	ldi r23, high(MAX-6)
.else ;if DATENFORMAT<=2
	ldi r22, low(MAX-8) ;bei gepacktem Speichern noch 2 Reserve
			;zum am Schluss noch ev. letztes Nibble zu speichern
			;(2 weil manchmal gleich 2 Byte gespeichert werden)
	ldi r23, high(MAX-8)
.endif
	clr r17			;r17=0
	ldi r18, SENSORSPEICHERN
	clr r5			;Nibble fuer gepacktes Speichern loeschen
.ifdef LCDANZEIGE
	rcall lcd_init
.endif
;-----------------------------------------------------------------------------
startloop:
	rcall ledportb		;ausgewaehlte LED leuchten lassen
;	mov r16, r12		;test sekunden auf LEDs anzeigen
;	out PORTB, r16		;test
.ifdef LCDANZEIGE
	rcall lcd_r1
.endif
	sbic PINC, 4		;Skip if Bit Clear, Taster gedrueckt?
	rjmp startloop		;nein--> warten
	ldi r16, 250
	rcall milliwait		;250ms warten
	sbis PINC, 4		;Skip if Bit Set, Taster losgelassen?
	rjmp evlangertastendruck	;nein--> langer Tastendruck
kurzertastendruck:
	lsl r1			;naechste LED
	sbrs r1, 3		;0x08 erreicht ?
	rjmp startloop		;nein-->
	mov r1, r3		;ja: r1=1
	eor r2, r3		;r2 zwischen 1 und 0 wechseln
	rjmp startloop
evlangertastendruck:	
	rcall milliwait		;250ms warten
	sbic PINC, 4		;Skip if Bit Clear, Taster noch gedrueckt?
	rjmp kurzertastendruck	;nein-> doch nur kurzer Tastendruck
langertastendruck:
	rcall ledportbclear
	rcall milliwait		;250ms warten
	rcall ledportb
	rcall milliwait		;250ms warten
	sbis PINC, 4		;Skip if Bit Set, Taster losgelassen?
	rjmp langertastendruck	;nein--> warten
aufganzesekundewarten:
	mov r20, r12		;r20 = Startzeit Sekunden
aufsecwarten:
	cp r20, r12
	breq aufsecwarten
	mov r20, r12		;r20 = Startzeit Sekunden
	mov r21, r13		;r21 = Startzeit Minuten

; r1 --> r2 = Messperiode in Anzahl Sekunden:
	sbrc r1, 0		;Skip if Bit in Register Clear, Rot gewaehlt?
	rjmp auswahl1		;ja-->
	sbrc r1, 1		;Skip if Bit in Register Clear, Gruen gewaehlt?
	rjmp auswahl2		;ja-->
auswahl3:
	ldi r16, 10		;Schaltzeit Magnetruehrer
	ldi r19, 60		;60 Sekunden Messperiode
	rjmp auswahlfertig
auswahl1:
	ldi r16, 0		;Schaltzeit Magnetruehrer deaktivieren
	ldi r19, 1		;1 Sekunde Messperiode
	rjmp auswahlfertig
auswahl2:
	ldi r16, 2		;Schaltzeit Magnetruehrer
	ldi r19, 10		;10 Sekunden Messperiode
auswahlfertig:
	sts magnetschaltzeit, r16
	mov r16, r19
	tst r2
	brne ausw3
	ldi r16, 0		;N=0 --> ohne Mittelung
ausw3:	sts nmittel, r16	;N fuer laufender Mittelwert
	rcall ringinit

;Datenkopf schreiben:
	ldi r16, DATENFORMAT
	rcall bytespeichern
	ldi r16, SENSORTYP
	rcall bytespeichern
.if SENSORSPEICHERN==2
	ldi r16, ANSCHLUSSVARIANTE2
.else
	ldi r16, ANSCHLUSSVARIANTE
.endif
	rcall bytespeichern
	mov r16, r19
	rcall bytespeichern	;Messperiode in Sekunden
	clr r24
	clr r25			;Anzahl Messpunkte = 0
	mov YH,XH		;Y = Zeiger zum am Schluss Anzahl
	mov YL,XL		;Messpunkte zu speichern.
	ldi r16, 0xFF
	rcall bytespeichern	;Anzahl Messpunkte erst mal auf ungueltigen
	rcall bytespeichern	;Wert (0xFFFF) setzen.
;	clr r6			;Letzter Messwert auf 0 setzen nicht noetig
;	clr r7			;weil erster Messwert eh Absolutwert.
	rcall admessung		;AD-Wert lesen und ev. aufsummieren -> r16:r17 
messloop:
	rcall messung		;AD-Wert von r16:r17 speichern
	add r20, r19		;neue Sollzeit = alte Sollzeit + Messperiode
	cpi r20, 60		;r20<60 ?
	brlt warteloop		;ja-->
	subi r20, 60		;nein: 60 subtrahieren
	;braucht da noch Anpassung falls Messperiode groesser 60 sein kann!
	inc r21			;und Minuten erhoehen
	cpi r21, 60
	brne warteloop
	clr r21
warteloop:
	mov r17, r12
nextsec:
	sbis PINC, 4		;Skip if Bit Set, Taster gedrueckt?
	rjmp evvorzeitig	;ja--> ev. vorzeitig beenden
	lds r16, errorflag
	tst r16
	brne fehlerabbruch
	lds r16, errorflag2
	tst r16
	brne fehlerabbruch2
	cp r17, r12
	breq nextsec		;auf naechste Sekunde warten
	rcall admessung		;AD-Wert lesen und ev. aufsummieren -> r16:r17
.ifdef MAGNETRUEHRER
	rcall magnetruehrer
.endif
.ifdef SCHUETTLER
	rcall schuetteln
.endif
	cp r20, r12		;Sollzeit Sekunden erreicht?
	cpc r21, r13		;Sollzeit Minuten erreicht?
	brne warteloop
.ifdef SCHUETTLER
	rcall adwandler_ein
.endif
	tst r23			;Speicher voll: restlicher Platz == 0?
	brne messloop		;nein-->
	tst r22
	brne messloop		;nein-->
	rjmp mfertig
fehlerabbruch2:
.ifdef STACKERROR
	rcall print_stack	;Ausgabe "Stack" auf LCD
.endif
fehlerabbruch:
	ldi r16, 0x11
.ifdef LCDANZEIGE
	rcall lcd_go
	rcall print_error	;Ausgabe "Error" auf LCD
.endif
	rjmp halteloop
evvorzeitig:
	ldi r16, 250
	rcall milliwait1	;250ms warten
	sbis PINC, 4		;Skip if Bit Set, Taster immer noch gedrueckt?
	rcall milliwait1	;ja: nochmals 250ms warten
	sbic PINC, 4		;Taster immer noch gedrueckt?
	rjmp nextsec		;nein--> dochnicht
mfertig:
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp mfert2		;nein --> kein vorheriges Nibble
	mov r16, r5
	swap r16
	andi r16, 0xF0		;nicht unbedingt noetig (unteres Nibble wird
				;eh verworfen), aber leichter zum debuggen.
	rcall bytespeichern	;letztes Nibble auch noch speichern
mfert2:
.ifdef EEPROMVERSION
	mov XH, YH
	mov XL, YL
	mov r16, r25
	rcall eepromstore
	mov r16, r24
	rcall eepromstore	;Anzahl Messpunkte speichern
.else
	ldi r16, 0xFF
	rcall endeimflashspeichern	;Rest der Page mit 0xFF fuellen
; Erste Page wurde schon in imflashspeichern ins SRAM kopiert
	ldi XL, low(SRAMSTART)
	ldi XH, high(SRAMSTART)
	add XL, YL
	adc XH, YH
	st X+, r25		;Anzahl Messpunkte im SRAM
	st X+, r24		;noch richtig eintragen.
	rcall page1zurueckkopieren
.endif
halteloop:
	lsl r1
	sbrc r1, 3		;0x08 erreicht ?
	mov r1, r3		;ja: r1=1
	rcall ledportb
	rcall milliwait255	;255ms warten
	rjmp halteloop

ledportb:
	push r16
	in r16, PORTB
	andi r16, 0xF8		;obere Bits unveraendert lassen
	or r16, r1		;untere 3 Bits = ausgewaehlte LED
	rjmp ledp1
ledportbclear:
	push r16
	in r16, PORTB
	andi r16, 0xF8		;obere Bits unveraendert lassen
ledp1:	out PORTB, r16
	pop r16
	ret
	
.ifdef FLASHVERSION

.else
bytespeichern:
eepromstore:
;	st X+, r16	;fuer test mit SRAM
;	ret		;fuer test mit SRAM
eeL1:	sbic EECR,1
	rjmp eeL1	;warte bis EEPROM bereit
	out EEARH,XH
	out EEARL,XL
	out EEDR, r16
	sbi EECR, 2	;EEMWE-Bit setzen
	sbi EECR, 1	;EEWE-Bit setzen
	adiw XL, 1
	ret
.endif

page1zurueckkopieren:		;Page vom SRAM zurueck ins FLASH kopieren
	ldi XL, 0
	ldi XH, 0
	ldi ZL, low(SRAMSTART)
	ldi ZH, high(SRAMSTART)
	ldi r17, PAGESIZEB
kopL2:	ld r16, Z+
	rcall imflashspeichern
	dec r17
	brne kopL2
	ret

;-----------------------------------------------------------------------------
; Interrupt-Routine
; Registerbelegungen:
; r11:r10 Millisekunden, r12 Sekunden, r13 Minuten, r14 Stunden, r15 Tage
tc0over:push r16
	push r17
	in r17, SREG	;Statusregister sichern
	push r17
	ldi r17, ZAEHLER
	out TCNT0, r17
	ldi r16, 0xE8
	ldi r17, 0x03	;r17:r16 = 1000 = 0x03E8
	inc r10		;Millisekunden erhoehen
	brne L1
	inc r11
L1:	cp r10, r16
	cpc r11, r17
	brne end1	;fertig wenn msec!=1000
	clr r10
	clr r11		;Millisekunden auf 0 setzen
	inc r12		;Sekunden erhoehen
	ldi r16, 60
	cp r12, r16
	brne end1	;fertig wenn sec!=60
	clr r12		;Sekunden auf 0 setzen
	inc r13		;Minuten erhoehen
	cp r13, r16
	brne end1	;fertig wenn min!=60
	clr r13		;Minuten auf 0 setzen
	inc r14		;Stunden erhoehen
	ldi r16, 24
	cp r14, r16
	brne end1	;fertig wenn std!=24
	clr r14		;Stunden auf 0 setzen
	inc r15		;Tage erhoehen
end1:
pop17reti:
	pop r17
	out SREG, r17	;Statusregister zurueckholen
	pop r17
	pop r16
	reti

;-----------------------------------------------------------------------------
; Interrupt-Routine fuer AD-Wandler
; Rueckgabewert: sensor1adwert
adcomplete:
	push r16
	push r17
	in r17, SREG	;Statusregister sichern
	push r17
	push r30
	push r31
	lds r17, ADCL	;L muss zuerst gelesen werden!
	lds r16, ADCH
	lds r30, adcsummeL
	lds r31, adcsummeH
	add r30, r17
	adc r31, r16	;Summe += Messwert
	dec r4		;letzter Messert?
	brne _end3	;nein-->
	ldi r17, 3
durch8:	asr r31		;Durchschnitt berechnen:
	ror r30		;Summe /= 8
	dec r17
	brne durch8
_L1:	mov r17, r30
	mov r16, r31
	sts sensor1adwertL, r17 	;Messwert im RAM
	sts sensor1adwertH, r16		;zwischenspeichern.
	clr r30
	clr r31
	ldi r17, 8
	mov r4, r17		;Anzahl Aufsummierungen wieder setzen
_end3:	lds r17, adrunflag
	tst r17
	breq _end4
;	lds r17, ADCSRA
;	ori r17, (1<<ADSC)
	ldi r17, 0b11001000 + PRESCALER
	sts ADCSRA, r17		;Starte naechste AD-Wandlung
;	sbi ADCSRA, 6		;Starte naechste AD-Wandlung Varante ATmega8
_end4:	sts adcsummeL, r30
	sts adcsummeH, r31
	pop r31
	pop r30
	rjmp pop17reti

;adwandler_aus:
;	push r17
;	clr r17
;	sts adrunflag, r17
;	pop r17
;	ret

adwandler_ein:
	push r17
;	lds r17, ADCSRA
;	ori r17, (1<<ADSC)
	ldi r17, 0b11001000 + PRESCALER
	sts adrunflag, r17
	sts ADCSRA, r17		;Starte naechste AD-Wandlung
;	sbi ADCSRA, 6		;Starte naechste AD-Wandlung Varante ATmega8
	pop r17
	ret

;-----------------------------------------------------------------------------
; Routinen fuer Ring-Puffer (FIFO-Speicher)
; mit ringpush wird hineingeschrieben, mit ringpop wieder zurueckgeholt
ringinit:			; Ringpuffer initialisieren, r16 zerstoert
	push XH
	push XL
	push r17
	push r18
	push r2
	push r3
	ldi XL, low(ringpuffer)
	ldi XH, high(ringpuffer)
	sts ringrL, XL
	sts ringrH, XH
	sts ringwL, XL
	sts ringwH, XH		; wenn ringr==ringw ist Puffer leer
	lds r16, sensor1adwertH
	lds r17, sensor1adwertL
	clr r2
	clr r3			; summe=0
	lds r18, nmittel
	tst r18			; sicherheithalber auf 0 testen
	breq ringi2		; N==0 --> keine Anfangswerte speichern
ringi1:	rcall ringpush		; N mal Anfangs-AD-Wert speichern
	add r2, r17
	adc r3, r16		; summe += Startwert
	dec r18
	brne ringi1
ringi2:	sts summeadwertL, r2
	sts summeadwertH, r3
	pop r3
	pop r2
	pop r18
	pop r17
	rjmp popxret
ringzeigercheck:	;XH:XL auf ringpufferende pruefen, und wenn erreicht..
	cpi XL, low(ringpufferende)
	brne ring1
	cpi XH, high(ringpufferende)
	brne ring1
	ldi XL, low(ringpuffer)	 ; ..wieder auf Ringanfang setzen.
	ldi XH, high(ringpuffer)
ring1:	ret
ringpush:			; r16:r17 in den Ringpuffer schreiben
	push XH
	push XL
	lds XL, ringwL
	lds XH, ringwH
	st X+, r16
	st X+, r17
	rcall ringzeigercheck
	sts ringwL, XL
	sts ringwH, XH
	rjmp popxret
ringpop:			; r16:r17 aus dem Ringpuffer auslesen
	push XH
	push XL
	lds XL, ringrL
	lds XH, ringrH
	ld r16, X+
	ld r17, X+
	rcall ringzeigercheck
	sts ringrL, XL
	sts ringrH, XH
popxret:
	pop XL
	pop XH
	ret

;-----------------------------------------------------------------------------
; Subroutine gepacktspeichern(UWORD r16:r17)
; r25:r24 = Anzahl bisherige Messpunkte, unveraendert
; r5 = vorheriges Nibble falls Bit 4 gesetzt, wird aktualisiert
; r7:r6 = alter Messwert, wird dann aktualisiert
; r16:r17 = neuer Messwert, geht verloren, ja wirklich r16=H r17=L
gepacktspeichern:
	push r1
	push r2
	mov r1, r16
	mov r2, r17		;Aktueller Messwert: r1:r2 = r16:r17
	tst r24			;Anzahl bisherige Messpunkte == 0 ?
	brne packstart		;nein -->
	tst r25
	breq pack0		;ja --> ungepackter Absolutwert
packstart:
;       1X -> 4Bit, davon 3Bit Differenzwert (-4 ... +3)
;       01 -> 8Bit, davon 6Bit Differenzwert (-36 ... -5 und +4 ... +35)
;       00 -> 12Bit, davon 10Bit Absolutwert (0 ... 1023)
;mit DATENFORMAT==2:
;       00 -> 16Bit, davon 14Bit Absolutwert (0 ... 16383)
	sub r17, r6	;r7:r6 = alter Wert, (r1:r2 = neuer Wert)
	sbc r16, r7	;r16:r17 = Differenz
	tst r16
	breq pack9p	;positive Differenz weniger als 9 Bit -->
	cpi r16, 0xFF	;negative Differenz weniger als 9 Bit ?
	brne pack0	;nein --> ungepackter Absolutwert
pack9:	cpi r17, 0xFC	;passt negativer Wert in 3 Bit?
	brcc pack1	;ja --> in 4 Bit = 1Nibble packen
	subi r17, -4	;Diff.Wertebereich erhoehen
	cpi r17, 0xE0	;passt negativer Wert in 6 Bit?
	brcs pack0	;nein --> Absolutwert speichern
	rjmp pack2	;ja --> in 2 Nibbles packen
pack9p:	cpi r17, 0x04	;passt positiver Wert in 3 Bit?
	brcs pack1	;ja --> in 1Nibble packen
	subi r17, 4	;Diff.Wertebereich erhoehen
	cpi r17, 0x20	;passt positiver Wert in 6 Bit?
	brcc pack0	;nein --> Absolutwert speichern
pack2:			;in 8 Bit = 2Nibble packen
	andi r17,0x3F	;6 Nutzbits
	ori r17, 0x40	;01 als erste 2 Bits
	sbrs r5, 4	;Bit 4 in r5 gesetzt?
	rjmp pack2k	;nein --> kein vorheriges Nibble
	swap r5		;vorheriges Nibble zuerst..
	mov r16, r5
	andi r16, 0xF0	 ;und zusammen mit..
	push r17
	swap r17
	andi r17, 0x0F 	  ;erstem Nibble..
	add r16, r17
	pop r17
	rcall bytespeichern2  ;abspeichern.
	andi r17,0x0F		;zweites Nibble
	ori r17, 0x10		;als gueltig markieren
	mov r5, r17		;und nach r5 kopieren.
	rjmp packret
pack2k:	mov r16, r17		;genau 1 Byte speichern
	rcall bytespeichern2
	rjmp packret
pack1:	;in 4 Bit packen
	andi r17,0x07
	ori r17, 0x08
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack1k		;nein --> kein vorheriges Nibble
	swap r5			;ja: vorheriges Nibble einfuegen
	mov r16, r5
	andi r16, 0xF0
	add r16, r17
	rcall bytespeichern2
	clr r5			;r5 als ungueltig markieren
	rjmp packret
pack1k:	ori r17, 0x10		;als gueltig markieren
	mov r5, r17		;und in r5 kopieren
packret:mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
	rjmp packret2
.if DATENFORMAT==2
pack0:				;16Bit Absolutwert (=r1:r2) speichern
	mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack04		;nein --> kein vorheriges Nibble
	ldi r17, 0x18		;ja: schiebe Nibble r5 -> r1:r2 -> r5
paL2:	lsr r5
	ror r1
	ror r2
	ror r17
	brcc paL2
	swap r17
	mov r5, r17		;uebriges Nibble
pack04:	mov r16, r1
	rcall bytespeichern2 	;vorheriges und oberstes Nibble speichern
	mov r16, r2
	rcall bytespeichern2	;2 mittlere Nibbles speichern
.else
pack0:				;12Bit Absolutwert (=r1:r2) speichern
	sbrs r5, 4		;Bit 4 in r5 gesetzt?
	rjmp pack03		;nein --> kein vorheriges Nibble
	;andi r1, 0x03	;nur noetig wenn obere Bits in r1 nicht schon 0 waeren.
	mov r16, r5		;ja: Nibble r5 -> oberes Nibble r16:r17
	clr r5
	swap r16		;r16 = r5<<4
	andi r16, 0xF0
	add r16, r1
	rcall bytespeichern2 	;2 obere Nibbles zuerst speichern
	mov r16, r2
	rcall bytespeichern2	;2 untere Nibbles danach speichern
	rjmp packret
pack03:				;3 Nibbles speichern
	mov r7, r1
	mov r6, r2		;letzter Messwert r7:r6 = r1:r2 = neuer Wert
	ldi r16, 4
paL1:	lsl r2
	rol r1
	dec r16
	brne paL1		;r1:r2 um 4 Bit nach links schieben
	mov r16, r1
	rcall bytespeichern2	;erste 2 Nibbles wirklich speichern
	mov r5, r2
	swap r5			;drittes Nibble im r5
	set			;T-Flag setzen
	bld r5, 4		;in r5 Bit4=T == r5 als gueltig markieren
.endif
packret2:
	pop r2
	pop r1
	ret

;-----------------------------------------------------------------------------
milliwait5:
	ldi r16, 5
milliwait1:			;Wartezeit: r16, 1 bis 255 msec
	push r16
	push r17
	clr r17
	rjmp miL1
milliwait255:
	ldi r16, 255
milliwait:			;Wartezeit: r17:r16 msec, min 1 max 1000
	push r16
	push r17
miL1:	push r18
	push r19
	mov r18, r10
loop1:	cp  r18, r10
	breq loop1		;1. Millisekunde abwarten (0 bis 1 ms)
	mov r18, r17		; Oberstes Bit von r17 nach r18 kopieren
	andi r17, 0x7F		; und in r17 loeschen.
	add r16, r10
	adc r17, r11		;sollzeit = Wartezeit + aktuelle Millisek
	subi r16, low(1001)
	sbci r17, high(1001)	;sollzeit -= 1; sollzeit -= 1000;
	brpl loop2		;wenn sollzeit>=0 -->
	subi r16, low(-1000)	;sonst haben wir 1000 zuviel subtrahiert
	sbci r17, high(-1000)	;also wieder 1000 addieren (soll -= -1000)
loop2:	mov r19, r10
	subi r19, -8
	cp r16, r10
	cpc r17, r11		;auf sollzeit warten
	cpc r16, r10		;falls zwischen 1. u 2. cp ein Interrupt war
	brne loop2
pop19ret: pop r19
pop18ret: pop r18
pop17ret: pop r17
pop16ret: pop r16
	  ret

admessung:	; AD-Wert lesen, ev. laufendes Mittel berechnen
		; Rueckgabewert: r16:r17
	push r0
	push r2
	push r3
	push r18
	push r19
	push r20
	lds r18, adrunflag
	tst r18			;wenn adwandler aus ist
	breq admalt		;--> nur alter Wert lesen
	lds r17, sensor1adwertL
	lds r16, sensor1adwertH
	lds r20, nmittel
	cpi r20, 2		; N<2 ?
	brcs mit1		; ja: --> ohne Mittelung
	lds r19, summeadwertL
	lds r18, summeadwertH
	mov r3, r16
	mov r2, r17		; r3:r2 = r16:r17
	rcall ringpop
	sub r19, r17
	sbc r18, r16		; summe -= N-ter vorheriger Wert
	mov r16, r3
	mov r17, r2
	rcall ringpush
	add r2, r19
	adc r3, r18		; summe += adwert
	sts summeadwertL, r2
	sts summeadwertH, r3
admalt:	lds r2, summeadwertL
	lds r3, summeadwertH
	lds r20, nmittel
	rcall divmodr20		; r3:r2 /= r20, r0=Rest
	mov r16, r3
	mov r17, r2		; Rueckgabewert=Summe/N
mit1:
.ifdef LCDANZEIGE
	clr r18			;lcd-Position unveraendert lassen
	rcall lcd_r16r17
.endif
	pop r20
	pop r19
	pop r18
	pop r3
	pop r2
	pop r0
return:	ret
	
;-----------------------------------------------------------------------------
; Messpunkt vom vorherigen admessung speichern
; Eingabewert: r16:r17
; Rueckgabewert: -
messung:
	push r16
	push r17
	push r18
	push r19
	mov r18, r16
	mov r19, r17
	lds r17, adrunflag
	tst r17			;wenn adwandler aus ist
	breq rpop19ret		;dann nichts machen.
	ldi r17, 2		;LED 2 mal blinken
blinkloop1:
	ldi r16, 50
	rcall milliwait1	;50ms warten
	rcall ledportb
	rcall milliwait1	;50ms warten
	rcall ledportbclear
	dec r17
	brne blinkloop1		;Blinkloop braucht also 2*2*50 = 200ms
	mov r16, r18
	mov r17, r19
.if DATENFORMAT==0
	rcall bytespeichern2 	;H zuerst speichern
	mov r16, r17
	rcall bytespeichern2	;L danach speichern
.else ;if DATENFORMAT<=2
	rcall gepacktspeichern
.endif
	adiw r24, 1		;r25:r24 += 1, Anzahl gespeicherte Messpunkte
	mov r16, r18
	mov r17, r19
	ldi r18, 1		;Flag fuer lcd-Position erhoehen
.ifdef LCDANZEIGE
	rcall lcd_r16r17
.endif
rpop19ret:
	rjmp pop19ret

;-----------------------------------------------------------------------------
; Routinen zum Speichern im Flash

bytespeichern2:		;Byte speichern und den Zaehler r23:r22 erniedrigen
	subi r22, 1
	sbci r23, 0
			;Speicher voll? (sollte eigentlich nicht passieren)
	brcs return	;ja --> nichts machen
bytespeichern:
imflashspeichern:		;1 Byte im Flash speichern
	;; zuerst Zwischenspeichern und dann 2 Byte aufs mal speichern
	;; jeweils Page loeschen bevor das erste Word gespeichert wird.
	;; r16 = zu speicherndes Byte
	;; XH:XL = Zeiger im Flash, wird um 1 erhoeht
	.def zwischenspeicher=r8
; oder falls kein Register belegt werden soll:
;	.equ ZWISCHENSPEICHER=SRAMSTART+256 ;2Byte im SRAM
	push r0
	push r1
	push r16
	push r17
	push ZL
	push ZH
; zuerst noch in einen Puffer kopieren:
	lds ZL, zpufferL
	lds ZH, zpufferH
	clr r17
	cpi XL, 0
	cpc XH, r17		  ;erster Aufruf?
	breq zpu1
	ldi r17, high(flashpuffer+PAGESIZEB)
	cpi ZL, low(flashpuffer+PAGESIZEB)
	cpc ZH, r17		  ;oder schon bei Pufferende?
	brne zpu2
zpu1:	ldi ZL, low(flashpuffer)  ;ja: dann zpuffer auf 
	ldi ZH, high(flashpuffer) ;    Puffer-Anfang setzen
zpu2:	st Z+, r16
	sts zpufferL, ZL
	sts zpufferH, ZH
; fertig mit in einen Puffer kopieren.
	sbrc r9, 0		;schreiben wir in erste Page?
	rjmp imfL1		;nein-->
	ldi ZL, low(SRAMSTART)	;ja: auch noch ins SRAM kopieren
	ldi ZH, high(SRAMSTART)
	add ZL, XL
	adc ZH, XH
	st Z+, r16
	cpi ZH, high(SRAMSTART+PAGESIZEB) ;erste Page voll?
	brcs imfL1		    ;nein-->
	cpi ZL, low(SRAMSTART+PAGESIZEB) ;erste Page voll?
	brcs imfL1		    ;nein-->
	inc r9			    ;ja: r9=1 (bzw Bit0 in r9 auf 1 setzen)
imfL1:	mov ZL, XL
	mov ZH, XH
	sbrs ZL, 0		;Z ungerade?
	rjmp zwischenspeichern	;nein--> erstes Byte zwischenspeichern
wordspeichern:
	dec ZL			;immer auf gerade Adresse 2 Bytes speichern
	mov r0, zwischenspeicher ;oder lds r0, ZWISCHENSPEICHER
	mov r1, r16
	ldi r17, (1<<SELFPRGEN)
	rcall do_spm17
	adiw ZH:ZL, 2
	mov r17, ZL
	andi r17, PAGESIZEB-1	;wird beim naechsten Mal neue Page erreicht?
	brne incxandreturn	;nein-->
	subi ZL, low(PAGESIZEB)	;ja:  execute page write
	sbci ZH, high(PAGESIZEB) ;restore pointer
	ldi r17, (1<<PGWRT) + (1<<SELFPRGEN)
	rcall do_spm17
	rcall spm_ueberpruefen
	sbrs r16, 0
.ifdef LCDANZEIGE
	rcall print_error17
.endif
	rjmp incxandreturn
zwischenspeichern:
	mov zwischenspeicher, r16	;oder sts ZWISCHENSPEICHER, r16
	mov r17, ZL
	andi r17, PAGESIZEB-1	;neue Page?
	brne incxandreturn	;nein-->
	ldi r17, (1<<PGERS) + (1<<SELFPRGEN) ;ja: neue Page zuerst loeschen
	rcall do_spm17
incxandreturn:
	adiw XH:XL, 1
	pop ZH
	pop ZL
	pop r17
	pop r16
	pop r1
	pop r0
	ret
do_spm17:	;r17 ins SPMCR speichern und spm durchfuehren, r16 zerstoert
spmwait: in r16, SPMCSR	;; check for previous SPM complete
	sbrc r16, SELFPRGEN
	rjmp spmwait
	in r16, SREG
	push r16	;; disable interrupts if enabled, store status
	cli
;eewait:	sbic EECR, EEWE		; auf Ende von allfaelligem
;	rjmp eewait		; EEPROM Schreibzyklus warten
	out SPMCSR, r17	;; SPM timed sequence
	spm
	pop r16 ;; restore SREG (to enable interrupts if originally enabled)
	out SREG, r16
	ret
endeimflashspeichern:		;Rest der Page mit r16 auffuellen
	push r17
eflashloop:
	mov r17, XL
	andi r17, PAGESIZEB-1	;neue Page erreicht?
	breq eflashfertig	;ja-->
	rcall imflashspeichern
	rjmp eflashloop
eflashfertig:
	pop r17
	ret

spm_ueberpruefen:	;Flash ab Stelle ZH:ZL um PAGESIZEB Bytes nachpruefen
			;Rueckgabe: r16=okflag, r17=restliche Bytes, Z erhoeht
	push YL
	push YH
; Das ist zwar im Beispiel in AVR_Instruction_Set.pdf nicht drin,
; braucht es aber offenbar:
	ldi r17, (1<<RWWSRE)+(1<<SELFPRGEN)
	rcall do_spm17		; Flash-Speicher wieder zum Lesen freigeben

.ifdef DEBUGGING
	push ZL
	push ZH
	ldi YL, low(flashpuffer)
	ldi YH, high(flashpuffer)
	ldi r16, 0x21
	rcall lcd_go
	ld r16, Y+
	rcall lcd_r16hex
	ld r16, Y+
	rcall lcd_r16hex
	lpm r16, Z+
	rcall lcd_r16hex
	lpm r16, Z+
	rcall lcd_r16hex
	pop ZH
	pop ZL
.endif

	ldi YL, low(flashpuffer)
	ldi YH, high(flashpuffer)
	ldi r17, PAGESIZEB	; PAGESIZEB==64 beim Atmega8
rdloop:	lpm r0, Z+
	ld r16, Y+
	cpse r0, r16
	rjmp err
	dec r17
	brne rdloop
 	ldi r16, 1		; true == ok
	rjmp popyret
err:	clr r16			; false
popyret:pop YH
	pop YL
	ret

.ifdef LCDANZEIGE
string_L1:
	rcall lcd_write
print_string:	    ;String in ZH:ZL auf lcd ausgeben, Z erhoeht, r16 zerst.
	lpm r16, Z+
	tst r16
	brne string_L1
	ret

.ifdef STACKERROR
Stack:	.db "Stack",0
print_stack:
	push ZL
	push ZH
	ldi r16, 'S'
	sts errorflag, r16
	ldi ZL, low(Stack<<1)
	ldi ZH, high(Stack<<1)
	rcall print_string
	pop ZH
	pop ZL
	ret
.endif
;Error:	.db "Error",0
Error:	.db "Err",0
print_error17:			; r17 ausgeben und 'Error' auf LCD
	mov r16, r17
	rcall lcd_r16dez
print_error:
	ldi r16, 'E'
	sts errorflag, r16
	push ZL
	push ZH
	ldi ZL, low(Error<<1)
	ldi ZH, high(Error<<1)
	rcall print_string
	pop ZH
	pop ZL
	ret
.endif

.ifdef LCDANZEIGE
;-----------------------------------------------------------------------------
; Anzeige auf LCD-Modul
microwait:			; etwa r16 usec warten
	push r16
	rjmp mic_L1
microwait1:			; etwa 1 usec warten
	push r16
	push r17
	ldi r16, 4
	rjmp mic_L2
;microwait50:
;	push r16
;	ldi r16, 50
mic_L1: push r17
	add r16, r16
	add r16, r16
mic_L2:	dec r16
	brne mic_L2
	pop r17
	pop r16
	ret

lcd_ziffer:		; void lcd_ziffer(int n) ;fuer 0<=n<=9
	ori r16, 0x30
	cpi r16, 0x3A	; oder fuer Hex-Ziffern bis n<=15
	brcs lcd_write
	subi r16, -7	; fuer Hex-Ziffern z.B. 3A --> 41=='A'
lcd_write:		;void lcd_write(char r16=c)
	sbi PORTD, 2
	rjmp lcd_send

lcd_cmd:	;void lcd_cmd(char cmd)
	cbi PORTD, 2
	rcall microwait1
lcd_send:	; lcd_send(char data) ;r16=data
	push r17
	push r18
	in r17, PORTD	;rs=PORTD
	andi r17, 4	;rs &= 4;
	ori r17, 3		; fuer Pullup-Widerstaende
	mov r18, r16	;char tmp=data
	andi r18, 0xF0	;tmp &= 0xF0
	or r18, r17	;tmp |= rs
	out PORTD, r18	;PORTD=tmp
	rcall microwait1
	sbi PORTD, 3	;PORTD |= (1<<3); //sbi(PORTD,3)
	rcall microwait1
	cbi PORTD, 3	;PORTD &= ~(1<<3); //cbi(PORTD,3)
	mov r18, r16	;tmp=data
	andi r18, 0x0F	;tmp &= 0x0F
	swap r18	;tmp <<= 4
	or r18, r17	;tmp |= rs
	out PORTD, r18	;PORTD=tmp
	rcall microwait1
	sbi PORTD, 3	;PORTD |= (1<<3); //sbi(PORTD,3)
	rcall microwait1
	cbi PORTD, 3	;PORTD &= ~(1<<3); //cbi(PORTD,3)
	ldi r16, 1
	rcall milliwait1
	pop r18
	pop r17
	ret

lcd_clear:
	push r16
	ldi r16, 0x01
lcd_cmd_wait2:
	rcall lcd_cmd
	ldi r16, 2
	rcall milliwait1
	pop r16
	ret
lcd_home:
	push r16
	ldi r16, 0x02
	rjmp lcd_cmd_wait2

lcd_on:	push r16
	ldi r16, 0x0E
lcd_cmd_2:
	rcall lcd_cmd
	pop r16
	ret
lcd_off:
	push r16
	ldi r16, 0x08
	rjmp lcd_cmd_2

lcd_init:
	push r16
	;out DDRD,0xFC	;DDRD=0xFF; //Port D auf Ausgang
	ldi r16, 3	;PORTD=0; (3 = +PullupWiderstaende)
	out PORTD, r16
	ldi r16, 50
	rcall milliwait	;//Warten bis LCD-Controller gebootet hat
	ldi r16, 0x23		; (3 = +PullupWiderstaende)
	out PORTD, r16	;PORTD=0x20; //4-Bit-Modus einschalten
	rcall microwait1
	sbi PORTD, 3	;PORTD |= (1<<3); //sbi(PORTD,3); //Enable-Impuls
	rcall microwait1
	cbi PORTD, 3	;PORTD &= ~(1<<3); //cbi(PORTD,3);
	rcall milliwait5
	ldi r16, 0x28	;lcd_cmd(0x28);
	rcall lcd_cmd	;  //Funktions-Set: 2 Zeilen, 5*7 Matrix, 4Bit
	rcall lcd_off
	rcall lcd_clear
	ldi r16, 0x06	;lcd_cmd(0x06); // Entry Mode
	rcall lcd_cmd
	rcall lcd_on
	ldi r16, 0x11
	sts lcdpos, r16
	pop r16
	ret

lcd_go:
	push r17
	mov r17, r16
	swap r16
	rcall lcd_goto
	pop r17
	ret
lcd_goto:		;lcd_goto(r16=row,r17=col)
	dec r16		;row--
	andi r16, 0x01	;row &= 0x01; //nur Zeile 0 oder Zeile 1 zulassen
	swap r16
 	add r16, r16		
 	add r16, r16	;row <<= 6; //Zeile nach Bit 6 bringen
	dec r17		;col--;
	andi r17, 0x0F	;col &= 0x0F; //nur Bereich 0 bis 15 zulassen
	or r16, r17	;char tmp=row|col; //Adresse bilden
	ori r16, 0x80	;tmp |= 0x80; //Cursor setzen
	rcall lcd_cmd	;lcd_cmd(tmp);
	ret

lcd_r1:		;Ziffer in r1, und wenn r2 gesetzt 'M', auf LCD anzeigen
	push r16
	push r17
	ldi r16, 1
	ldi r17, 1
	rcall lcd_goto
	mov r16, r1
	andi r16, 0x0F
	rcall lcd_ziffer
	ldi r16, ' '
	rcall lcd_write
	ldi r16, ' '
	tst r2			; Mittelwertflag gesetzt?
	breq _ohne		; nein-->
	ldi r16, 'M'
_ohne:	rcall lcd_write
	rjmp pop17ret

lcd_r16dez:
	ldi r17, 10
	rjmp lcd_r16
lcd_r16hex:
	ldi r17, 16
lcd_r16:	;r16 auf LCD, Dezimal oder Hexadezimal ohne fuehrende Nullen
		;Eingabe: r16=Zahl, r17=10 oder r17=16 fuer Hexadezimal
	push r0
	push r2
	push r3
	push r16
	push r17
	push r20
	push XL
	push XH
	clr r3
	mov r2, r16
	mov r20, r17		;Basis (10 oder 16)
	ldi XL, low(lcdn0)
	ldi XH, high(lcdn0)
	ldi r17, 1		;Zaehler fuer Anzahl Ziffern
	rjmp lc16L2
lc16L1:	rcall divmodr20		;durch 10 dividieren (oder durch 16)
	st X+, r0
	inc r17
lc16L2:	cp r2, r20		;Zahl>=10 ?
	brcc lc16L1		;ja--> weiter dividieren
	mov r16, r2
	rcall lcd_ziffer
	rjmp lc16L4
lc16L3:	ld r16, -X
	rcall lcd_ziffer
lc16L4:	dec r17
	brne lc16L3
	pop XH
	pop XL
	rjmp popr20ret

lcd_r16r17:	;Zahl r16:r17 auf LCD anzeigen, Wenn r18==1 lcdpos erhoehen
	push r0
	push r2
	push r3
	push r16
	push r17
	push r20
	mov r3, r16
	mov r2, r17
	lds r16, lcdpos
	rcall lcd_go
lcd_L0:	ldi r20, 10
	rcall divmodr20
	sts lcdn0, r0
	rcall divmodr20
	sts lcdn1, r0
	rcall divmodr20
	sts lcdn2, r0
	rcall divmodr20
	sts lcdn3, r0
	rcall divmodr20
	mov r16, r0
	tst r16			; wenn fuehrende Ziffer keine 0 ist..
	breq lcd_L1
	ori r16, 0x30		; ..dann die Ziffer selbst..
	rjmp lcd_L2
lcd_L1:	ldi r16, 0x20		; ..sonst ' ' ausgeben.
lcd_L2:	rcall lcd_write
	lds r16, lcdn3
	rcall lcd_ziffer
	lds r16, lcdn2
	rcall lcd_ziffer
	lds r16, lcdn1
	rcall lcd_ziffer
	lds r16, lcdn0
	rcall lcd_ziffer
	tst r18			; r18==0 ?
	breq lcd_L4		; ja--> lcdpos nicht veraendern
	lds r16, lcdpos
	cpi r16, 0x11
	breq lcd_L11
	cpi r16, 0x18
	breq lcd_L18
	cpi r16, 0x21
	breq lcd_L21
	ldi r16, 0x11
	rjmp lcd_L3
lcd_L11:ldi r16, 0x18
	rjmp lcd_L3
lcd_L18:
.ifdef DEBUGGING
	ldi r16, 0x11
.else
	ldi r16, 0x21
.endif
	rjmp lcd_L3
lcd_L21:ldi r16, 0x28
lcd_L3:	sts lcdpos, r16
lcd_L4:
popr20ret:
	pop r20
	pop r17
	pop r16
popr3ret:	
	pop r3
	pop r2
	pop r0
	ret

.endif
;------- Ende LCD-Anzeige ------------

;-----------------------------------------------------------------------------
; divmodr20
;  16Bit vorzeichenlose Zahl dividieren durch 8Bit vorzeichenlosen Teiler
; Eingabe: r3:r2 = Zahl (16Bit), r20 = Teiler (8Bit) (unveraendert)
; Ausgabe: r3:r2 = Resultat, r0 = Rest
;-----------------------------------------------------------------------------
divmodr20:
	push r17
	clr r0
	ldi r17, 16
divL1:	lsl r2
	rol r3
	rol r0
	cp r0, r20
	brcs divL2
	sub r0, r20
	inc r2
divL2:	dec r17
	brne divL1
	pop r17
	ret

.ifdef MULDIV
mul16:		;r3:r2:r1:r0 = r1:r0 * r3:r2
	push r16
	push r17
	push r18
	push r19
	push r6
	push r7
	mov r16, r0
	mov r17, r1		; AB
	mov r18, r2
	mov r19, r3		; CD
	mul r17, r19		;A*C
	mov r2, r0		; EFGH = AB*CD --> EF=A*C GH=B*D EFG+=A*D+B*C
	mov r3, r1		; r3r2 = EF=A*C
	mul r16, r18		;B*D
	mov r6, r0
	mov r7, r1		; r6r7 = GH=B*D
	mul r17, r18		;A*D
	clr r17
	add r5, r0		; EFG+=A*D
	adc r2, r1
	adc r3, r17
	mul r16, r19		;B*C
	add r5, r0		; EFG+=B*C
	adc r2, r1
	adc r3, r17
	mov r0, r6
	mov r1, r7
	pop r7
	pop r6
	rjmp pop19ret

div16:			;r3:r2:r1:r0 /= r17:r16, r17:r16 = Rest
	push r18
	push r19
	push r20
	clr r18
	clr r19
	ldi r20, 32
div16L1:lsl r0
	rol r1
	rol r2
	rol r3
	rol r18
	rol r19
	cp  r18, r16
	cpc r19, r17
	brcs div16L2
	sub r18, r16
	sbc r19, r17
	inc r0
div16L2:dec r20
	brne div16L1
	mov r16, r18
	mov r17, r19		;r17:r16 = Rest
	pop r20
	pop r19
	pop r18
	ret
.endif

.ifdef MAGNETRUEHRER
magnetruehrer:			; Magnetruehrer ein/aus-schalten
;r21:r20 = Sollzeit  r13:r12 = aktuelle Min:Sek
	push r16
	push r20
	push r21
	lds r16, magnetschaltzeit
	tst r16
	breq _magret
	sub r20, r16		;Sollzeit fuer "Magnetruehrer ausschalten"
	sbci r21, 0
	brcc _magL1
	ldi r21, 59
_magL1:	cp r20, r12		;Sollzeit Sekunden erreicht?
	cpc r21, r13		;Sollzeit Minuten erreicht?
	brne _magret
	cbi PORTB, 5		;Magnetruehrer aus
_magret:pop r21
	pop r20
	pop r16
	ret
.endif

.ifdef SCHUETTLER
schuetteln:			; Schuettler ev laufen lassen
;r21:r20 = Sollzeit  r13:r12 = aktuelle Min:Sek
	push r16
	push r20
	push r21
	subi r20, 3		;Sollzeit -= 3 Sek Schuettelzeit
	brcc _magL1
	subi r20, -60		; 60 Sek zuzaehlen
	subi r21, 1		; dafuer 1 Min abziehen
	brcc _magL1
	ldi r21, 59
_magL1:	sub r20, r12		;Sollzeit Sekunden erreicht?
	sbc r21, r13		;Sollzeit Minuten erreicht?
	brne _magret		;nein-->
	sts adrunflag, r21	;adwandler_aus
	sbi PORTB, 4		;Puls fuer Schuettelvorgang senden
	rcall milliwait5
	cbi PORTB, 4
_magret:pop r21
	pop r20
	pop r16
	ret
.endif
