/* mscalc.cc			letzte Aenderung: 12.11.2014 */
#define VERSION "Version 0.3"
/*
Uebersetzen auf Unix (Linux):
> make  ;siehe makefile

 Kurzbeschreibung: Massenspektrum (MS) Isotopenaufspaltung berechnen

History:
11.11.2014      Erstellung aus mssimu.cc (RP)

*/

#include <stdio.h>
//#include <iostream>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <xtekplot1.h>

#define SUMMENFORMELN
#include "mgkern.c"

#define XMAX 1024
#define YMAX  820
//#define TIEFE 24
#define TIEFE 8

#define SCHWARZAUFWEISS //auskommentieren fuer weiss auf schwarz

/************************* Vordeklarationen ***************************/
double mg(const char *formel);

/**************** Routinen zur Parameterauswertung ********************/
#define MAXARG 2
static char argflag[128];
void setargflags(char *s)
{
 int c;
 while(c= *s++)
  {if(c>='a' && c<='z')  c -= 'a'-'A';
   argflag[c&127]=1;
  }
}

/*************************** kleinkram ***************************/

inline int idfix(double x) {return int((x>=0.) ? x+0.5 : x-0.5);} //Runden

/******************** Wahrscheinlichkeitsrechnung *********************/
struct elements {const char *sym; double m1,m2, w1,w2;};
elements element[]=
{
 //"C",	12.0000,13.00335, 0.9893,0.0107,  //Durchschnitt: 12.0107
 "C",	12.0000,13.00335, 0.989, 0.011,   //Durchschnitt: 12.0110
 "H",   1.00794,2.0141, 0.999885,0.000115,
 "N",   14.0003,15.0001, 0.99636,0.00364,
 "O",   15.9949,17.9992, 0.99757,0.00205,  //noch nicht beruecksichtigt: ^17O 16.9991 0.00038
 //"O",   15.9949,17.9992, 0.99757,0.00243,  //test
 "S",   31.9721,33.9679, 0.9499,0.0425,    //noch nicht: ^33S 32.9715 0.0075, ^36S 35.9671 0.0001
 //"S",   31.9721,33.9679, 0.9499,0.0501,    //test
 "B",   11.0093,10.0129, 0.801,0.199,      //Haeufigstes Isotop zuerst
 //"B",   10.0129,11.0093, 0.199,0.801,      //test: Isotop mit kleinerer Masse zuerst
 "Br",  78.9183,80.9163, 0.5069,0.4931,
 "Cl",  34.9688,36.9659, 0.7576,0.2424,
 "Cu",  62.9296,64.9278, 0.6915,0.3085,
 "Li",   7.0160,6.01512, 0.9241,0.0759,
 "Mg",  23.9850,25.9826, 0.7899,0.1101,  //^25Mg 24.9858 0.1000
 "Si",  27.9769,28.9765, 0.92223,0.04685, //^30Si 29.9738 0.03092
 "K",   38.9637,40.9618, 0.93258,0.067302,
 "Ca",  39.9626,43.9555, 0.96941,0.02086, //diverse weitere Isotope noch nicht beruecksichtigt
 "Fe",  55.9349,53.9396, 0.91754,0.05845,
 "Ni",  57.9353,59.9308, 0.680769,0.262231,
 "Zn",  63.9291,65.9260, 0.48268,0.27975, //^68Zn 67.9248 0.19024
 "Se",  79.9165,77.9173, 0.4961,0.2377, //diverse weitere Isotope noch nicht beruecksichtigt
 "Ag", 106.9051,108.9047, 0.51839,0.48161,
 "Ti",  47.9479, 45.9526, 0.7372,0.0825, //diverse weitere Isotope noch nicht beruecksichtigt
 "V",   50.9440, 49.9472, 0.9975,0.0025,
 "Cr",  51.9405, 52.9406, 0.83789,0.09501,
 "Ga",  68.9256, 70.9247, 0.60108,0.39892,
 "Ge",  73.9212, 71.9221, 0.3672,0.2731,
 //bisher alle Elemente bis Brom, ausser den Edelgasen, beruecksichtigt
 //aber bisher nur jeweils 2 Isotope pro Element beruecksichtigt.
 NULL,	0.,0.,		0.,0
};

struct wahrtab {double a,b,m1,m2;};
wahrtab wahr['z'*256+'Z'];
void init_wahr(void)
{
 elements *p;
 int index;
 for(index=0;index<('z'*256+'Z');index++)
	{
	 wahr[index].a=1.0; wahr[index].b=0.0;
	 wahr[index].m1=0; wahr[index].m2=0;
	}
 for(p=element;p->sym!=NULL;p++)
	{index=256*tolower(p->sym[1])+toupper(p->sym[0]);
	 wahr[index].a=p->w1; wahr[index].b=p->w2;
	 wahr[index].m1=p->m1; wahr[index].m2=p->m2;
	}
}

double *pas3eck(int n) //Pascalsches Dreieck
{
 int i,j;
 double *x=new double[(n+1)*(n+1)];
 for(i=0;i<n;i++)
  for(j=0;j<n;j++)
	if(i==0 || j==0) x[i+j*n]=1;
	else	x[i+j*n]=x[i-1+j*n]+x[i+(j-1)*n];
 return x;
}

struct NisoWahrtab {char sym[4]; double massedifferenz; double w[20];}; //Wahrscheinlichkeitstabelle fuer N Isotopenmarkierte
static NisoWahrtab isowahrtab[20];
static int isoanzahl=0;

void isowahrtab_init()
{
 for(int i=0;i<20;i++)
  {
   isowahrtab[i].sym[0]=0;
   isowahrtab[i].massedifferenz=0;
   isowahrtab[i].w[0]=1;
   isowahrtab[i].w[1]=0;
  }
}

void calcisowahr(const char *sym,int n)
{
 int i,j,index;
 double *feld;
 double wa,wb; //Wahrscheinlichkeiten fuer die 2 haeufigsten Isotope
 if(isoanzahl==20)
   {
     static int flag=0;
     if(flag==0) printf("Zu viele Atome. %s nicht mehr beruecksichtigt\n",sym);
     flag=1;
     return;
   }
 if(strlen(sym)>2) {printf("Fehler: ungueltiges Atom '%s'\n",sym); return;}
 strcpy(isowahrtab[isoanzahl].sym,sym);
 feld=pas3eck(n+1);
 if(argflag['V'])
  {printf("Wahrscheinlichkeitsberechnung fuer %d %s Atome\n",n,sym);//test
   for(j=n,i=0;j>=0 && i<20;i++,j--) printf(" %.0f",feld[i+j*(n+1)]); //test
   printf("\n");//test
  }
 index=256*tolower(sym[1])+toupper(sym[0]);
 wa=wahr[index].a; wb=wahr[index].b;
 isowahrtab[isoanzahl].massedifferenz = wahr[index].m2 - wahr[index].m1; //kann auch negativ sein
 if(argflag['V']) printf("Massedifferenz = %f\n",isowahrtab[isoanzahl].massedifferenz); //test
 for(j=n,i=0;j>=0 && i<20;i++,j--)
  {
   isowahrtab[isoanzahl].w[i] = pow(wa,j)*pow(wb,i)*feld[i+j*(n+1)];
   if(argflag['V']) printf(" %lf%%",100*isowahrtab[isoanzahl].w[i] ); //test
  }
 if(argflag['V']) printf("\n");//test
 isoanzahl++;
 delete feld;
}

/********************** Spektrometer-Simulation ***********************/
#define XSTEP 20
#define AUFLOESUNG (1.0/XSTEP)
#define BEREICH 20
#define IMAX (BEREICH*XSTEP) //Messbereich Multipliziert mit Anzahl aufgeloesten Punkten pro Masseneinheit
#define IMAX3 (3*IMAX)

class Spektrometer
{
 double molekuelmasse,startmasse,endmasse,startx,endx;
 double detektorfeld[IMAX3];
public:
 Spektrometer() {startmasse=0.0; endmasse=BEREICH;}
 void init(const char *formel,int bereichsflag);
 void injekt(double masse,double wahrscheinlichkeit);
 void drawxachse();
 void draw();
 void rahmenzeichnen();
 void clear();
};

static Spektrometer spektrum;

const double xmin= -1.0, ymin= -0.1, xmax=BEREICH+1.0, ymax=1.05;
const double ymark10= -0.06, ymark5= -0.05, ymark1= -0.04, ymark0= -0.03;
const double schrift_bx=xmax/80.0,schrift_hy=ymax/40.0;

void Spektrometer::init(const char *formel,int bereichsflag)
{
 int i,anzahl;
 char *sym;
 init_wahr();
 molekuelmasse = mg(formel);
 printf("Molmasse isotopenrein ist %f\n",molekuelmasse);
 startx = (int)molekuelmasse-1;
 endx = startx+BEREICH;
 startmasse = startx-BEREICH;
 endmasse = startx+2*BEREICH;
 if(bereichsflag!=0)
   {startx=startmasse; endx=endmasse;}
 if(argflag['V']) printf("Messbereich: %f ... %f\n",startmasse,endmasse);//test
 if(argflag['V']) printf("Anzeigebereich: %f ... %f\n",startx,endx);//test
 for(i=0;i<IMAX3;i++) detektorfeld[i]=0;
 //printf("bx=%f hy=%f\n",schrift_bx,schrift_hy);//test
 textsize(schrift_bx,schrift_hy);

 //Aus der Formel fuer jede Atomsorte die Wahrscheinlichkeiten berechnen,
 //dass n Atome davon Isotope sein koennten:
 isoanzahl=0;
 isowahrtab_init();
 while(*formel!=0)
  {
   sym=getsymbol(&formel);
   if(isdigit(*formel)) anzahl=getnumber(&formel);
   else anzahl=1;
   calcisowahr(sym,anzahl);
  }

 if(argflag['V']) printf("Anzahl Atomsorten: isoanzahl=%d\n",isoanzahl);//test
 //printf("sizeof(int)=%ld sizeof(ulong)=%ld\n",sizeof(int),sizeof(ulong));//test
}

void Spektrometer::clear()
{
 int i;
 for(i=0;i<IMAX3;i++) detektorfeld[i]=0;
}

void Spektrometer::injekt(double masse,double wahrscheinlichkeit)
{
 double x,zuf,sumw;
 const double mgH=1.007825;
 int i,n,index;
 char isotop[40];
 masse += molekuelmasse + mgH; //Mit H+ Jonisiertes Molekuel

 if(masse>=startmasse && masse<=endmasse)
  for(i=0,x=startmasse;i<IMAX3;i++,x+=AUFLOESUNG)
   {
    if(x>=masse)
      {
       detektorfeld[i] += wahrscheinlichkeit;
       break;
      }
   }
}

void Spektrometer::draw()
{
 int i,pen=PENUP;
 double x,y,max=100;
#ifdef SCHWARZAUFWEISS
 color(1);
#else
 color(0);
#endif
 fillbox(0.05,ymark0+0.005,BEREICH-0.05,ymax-0.005); //Bereich innerhalb Rahmen loeschen
#ifdef SCHWARZAUFWEISS
 color(0);
#else
 color(1);
#endif

 for(i=0;i<IMAX3;i++) if(detektorfeld[i]>max) max=detektorfeld[i];
 int i1=IMAX3/(endmasse-startmasse)*(startx-startmasse);
 int i2=IMAX3/(endmasse-startmasse)*(endx-startmasse);
 for(i=i1;i<i2;i++)
  {
   y = detektorfeld[i]/max;
   x = BEREICH/(double)(i2-i1)*(i-i1);
   plot(x,y,pen);
   pen=PENDOWN;
  }
}

void Spektrometer::drawxachse()
{
 double xp,xw,y; //xp=Zeichnen-Korrdinaten, xw=Wirkliche x-Werte
 int ix;
 char text[40];
 //Zeichnen-Koordinaten immer 0.0 bis BEREICH (z.B. 20.0)
 plot(0.0,ymark0,PENUP); plot(BEREICH,ymark0,PENDOWN); //Achse zeichnen
 for(xw=startx;xw<=endx;xw+=1.0) //Markierstriche zeichnen
  {
   xp=BEREICH/(endx-startx)*(xw-startx);
   ix=xw;
   if(ix%5==0)
    {
     sprintf(text,"%.0f",xw);
     schrift(xp-strlen(text)*schrift_bx/2,ymark10-1.1*schrift_hy,text);
     y = (ix%10==0) ? ymark10 : ymark5;
    }
   else y=ymark1;
   plot(xp,y,PENUP); plot(xp,ymark0,PENDOWN);
  }
}

void Spektrometer::rahmenzeichnen()
{
 plot(0.0,ymark0,PENUP); plot(0.0,ymax,PENDOWN); plot(BEREICH,ymax,PENDOWN); plot(BEREICH,ymark0,PENDOWN);
}

void refresh()
{
 spektrum.draw();
 spektrum.rahmenzeichnen();
 spektrum.drawxachse();
}

/******************** Molekulargewicht-Berechnung *********************/
double mg(const char *formel)
{
 char str[200],*s;
 strcpy(str,formel);
 //error_reset(errortext,warningtext);
 return mol_weight(str,&s,ISO);
}

/************************* Menu Behandlung ****************************/
static int exitflag=0;
void menu_exit() {exitflag=1;}
void menu_refresh() {waitTOF(); inital_new(); refresh(); term_refresh();}

/************************* Hauptprogramm ******************************/
main(int argc,char *argv[])
{
 char formel[400];
 formel[0]=0;
 int i,j,c,col=0,maxcol,anzahlmessungen=10000;
 int breite,hoehe,tiefe,visklasse;
 //printf("Test: sizeof(uint)=%lu sizeof(ulong)=%lu sizeof(UINT64)=%lu\n",sizeof(uint),sizeof(ulong),sizeof(UINT64));
 //printf("Test: sizeof(float)=%lu sizeof(double)=%lu\n",sizeof(float),sizeof(double));
 if(argc<=0)
   ;/* es wurde von WorkBench gestartet */
 else
   /* es wurde von der Shell gestartet */
   for(j=0,i=1;i<argc;i++)
	{if((c= *argv[i])=='-' || c=='?') setargflags(argv[i]);
	 else	{if(++j==1) strcpy(formel,argv[i]);
                 else if(j==2) sscanf(argv[i],"%d",&anzahlmessungen);
	}	}
 if(argflag['?'] || j>MAXARG)
	{printf("mscalc  %s\n",VERSION);
	 printf("Anwendung: mscalc [-Flags] Summenformel [AnzahlMessungen]\n");
	 printf("   Flags: v = verbose, einige Testausdrucke machen\n");
	 printf("          b = groesserer Bereich darstellen\n");
	 printf("          g = zuerst genaeherte Berechnungen machen\n");
	 printf("   Voreinstellung fuer AnzahlMessungen ist %d\n",anzahlmessungen);
	 printf("   (wird nur fuer schoenere Grafische Darstellung gebraucht)\n");
	 exit(0);
	}
 //tek_setdebug(1);//test

 if(*formel==0)
  {printf("Summenformel:"); do i=scanf("%s",formel); while(i!=1);}

 char *summenformel=getsummenformel(formel,NULL);
 if(strcmp(summenformel,formel)!=0)
   printf("Summenformel: %s\n",summenformel);

 getmaxsize(&breite,&hoehe,&tiefe,&visklasse);
 if(tiefe>TIEFE) tiefe=TIEFE;
 if(breite>XMAX) breite=XMAX;
 if(hoehe>YMAX) hoehe=YMAX;
 maxcol=(1<<TIEFE);
 setsize(breite,hoehe,tiefe);
 setmenu(1,"File");
 setmenu(1,"Refresh",&menu_refresh);
 setmenu(1,"Exit",&menu_exit);
 inital(xmin,ymin,xmax,ymax); /* Grafikfenster oeffnen */

#ifdef SCHWARZAUFWEISS
 //fuer schwarz auf weiss (default ist weiss auf schwarz):
 if(TIEFE==24)
  {screenclear(0xFFFFFF); rgbcolor(0x00,0x00,0x00);}
 else
  {screenclear(1); color(0);}
#endif

 int bereichsflag = argflag['B'] ? 1 : 0;
 spektrum.init(summenformel,bereichsflag);
 spektrum.rahmenzeichnen();
 spektrum.drawxachse();
 spektrum.draw(); //noch leeres Spektrum zeichnen
 term_refresh();
 inital_new();

 double multiplikator=1,summe;
 double proz[20];
 int jmax;

 if(argflag['G'])
 { //nur angenaeherte Berechnungen:
 printf("\nEs wird das Spektrum von %s naeherungsweise berechnet.\n",summenformel);
 for(i=1;i<isoanzahl;i++)
   multiplikator *= isowahrtab[i].w[0];
 for(j=0;j<20;j++)
   proz[j] = multiplikator*isowahrtab[0].w[j];
 printf("Verteilung mit nur erstem Atom:\n");//test
 for(j=0;j<20 && proz[j]>=0.00005;j++) printf(" %.2f%%",proz[j]*100);//test
 jmax=j;
 printf("\n");//test

 for(j=0;j<jmax;j++)
  {
   spektrum.injekt(j,proz[j]*anzahlmessungen);
  }
 refresh();

 for(i=0;i<100;i++) waitTOF(); //etwa 5 Sekunden warten
 spektrum.clear(); //Spektrum loeschen
 refresh();
 for(i=0;i<50;i++) waitTOF(); //test etwa 1 Sekunde warten

 printf("\nNaeherung nur fuer Formeln der Form CxHxNxOxSx\n");
 if(isoanzahl==5)
 {
  for(j=1;j<jmax;j++)
  {
   summe=0;
   for(int k=1;k<=2;k++)
    {
     multiplikator = isowahrtab[0].w[0];
     for(i=1;i<isoanzahl;i++)
       multiplikator *= isowahrtab[i].w[(i==k)?j:0];
     summe += multiplikator;
    }
   if((j%2)==0)
    {
     for(int k=3;k<=4;k++)
      {
       multiplikator = isowahrtab[0].w[0];
       for(i=1;i<isoanzahl;i++)
	 multiplikator *= isowahrtab[i].w[(i==k)?(j/2):0];
       summe += multiplikator;
      }
    }
   proz[j] += summe;
  }
 printf("Gesamte Verteilung (Naeherung):\n");//test
 for(j=0;j<20 && proz[j]>=0.00005;j++) printf(" %.2f%%",proz[j]*100);//test
 printf("\n");//test

 for(j=0;j<jmax;j++)
  {
   spektrum.injekt(j,proz[j]*anzahlmessungen);
   //spektrum.draw(); waitTOF();//test
  }
 refresh();

 for(i=0;i<100;i++) waitTOF(); //etwa 5 Sekunden warten
 spektrum.clear(); //Spektrum loeschen
 refresh();
 for(i=0;i<50;i++) waitTOF(); //test etwa 1 Sekunde warten
 } //Ende Naeherung fuer CxHxNxOxSx
 else printf("Falsche Formel fuer diese Naeherung.\n");
 } //Ende von genaeherten Berechnungen

 printf("\nEs wird das Spektrum von %s moeglichst exakt berechnet.\n",summenformel);
 printf("Einschraenkung: es werden jeweils nur die beiden haeufisten Isotope pro Atom beruecksichtigt.\n");
 if(isoanzahl>10)
  printf("das geht bisher nur mit maximal 10 Atomsorten.\n");
 else
  {
   int i0,i1,i2,i3,i4,i5,i6,i7,i8,i9;
   double w0=0,w1=0,w2=0,w3=0,w4=0,w5=0,w6=0,w7=0,w8=0,w9=0;
   double m0=0,m1=0,m2=0,m3=0,m4=0,m5=0,m6=0,m7=0,m8=0,m9=0,masse,wahrscheinlichkeit;
   const double dd=0.000001; //kleinste noch darzustellende Wahrscheinlichkeit
   double summewahrsch=0;//test
   double verteilung[100],verteilungneg[100];
   int imin=0,imax=0;
   for(i=0;i<100;i++) verteilung[i]=verteilungneg[i]=0;
   for(i0=0;i0<20 && (w0=isowahrtab[0].w[i0])>dd;i0++)
    {m0=isowahrtab[0].massedifferenz*i0;
     for(i1=0;i1<20 && (w1=isowahrtab[1].w[i1])>dd;i1++)
      {m1=isowahrtab[1].massedifferenz*i1;
       for(i2=0;i2<20 && (w2=isowahrtab[2].w[i2])>dd;i2++)
	{m2=isowahrtab[2].massedifferenz*i2;
	 for(i3=0;i3<20 && (w3=isowahrtab[3].w[i3])>dd;i3++)
	  {m3=isowahrtab[3].massedifferenz*i3;
	   for(i4=0;i4<20 && (w4=isowahrtab[4].w[i4])>dd;i4++)
	    {m4=isowahrtab[4].massedifferenz*i4;
	     for(i5=0;i5<20 && (w5=isowahrtab[5].w[i5])>dd;i5++)
	      {m5=isowahrtab[5].massedifferenz*i5;
	       for(i6=0;i6<20 && (w6=isowahrtab[6].w[i6])>dd;i6++)
		{m6=isowahrtab[6].massedifferenz*i6;
		 for(i7=0;i7<20 && (w7=isowahrtab[7].w[i7])>dd;i7++)
		  {m7=isowahrtab[7].massedifferenz*i7;
		   for(i8=0;i8<20 && (w8=isowahrtab[8].w[i8])>dd;i8++)
		    {m8=isowahrtab[8].massedifferenz*i8;
	   for(i9=0;i9<20 && (w9=isowahrtab[9].w[i9])>dd;i9++)
	    {m9=isowahrtab[9].massedifferenz*i9;
	     wahrscheinlichkeit = w0*w1*w2*w3*w4*w5*w6*w7*w8*w9;
	     summewahrsch += wahrscheinlichkeit;//test
	     masse = m0+m1+m2+m3+m4+m5+m6+m7+m8+m9;
	     if(argflag['V'] && wahrscheinlichkeit>=0.01)
	       printf("masse=%f wahrscheinlichkeit=%f%%\n",masse,wahrscheinlichkeit*100);//test
	     spektrum.injekt(masse,wahrscheinlichkeit*anzahlmessungen);
	     //spektrum.draw(); waitTOF();//test
	     i=idfix(masse);
	     if(i<imin) imin=i;
	     if(i>imax) imax=i;
	     if(i>=0)
	       verteilung[i] += wahrscheinlichkeit;
	     else
	       verteilungneg[-i] += wahrscheinlichkeit;
	    }
	            }
	          }
	        }
	      }
	    }
	  }
	}
      }
     spektrum.draw(); waitTOF();//test
    }
   if(argflag['V'])
    {printf("Summe aller Wahrscheinlichkeiten: %f%%\n",summewahrsch*100);//test
     printf("(ist kleiner 100%% wenn O oder S enthalten ist, da nur 2 hauefigste Isotope beruecksichtigt)\n");//test
    }
   for(i=imin;i<=imax;i++)
    {
     if(i<0)
      {if(verteilungneg[-i]>=0.0001) printf("M-%d: %f%%\n",-i,verteilungneg[-i]*100);}
     else
      {if(verteilung[i]>=0.0001) printf("M+%d: %f%%\n",i,verteilung[i]*100);}
    }
  }

 refresh();
 printf("fertig.\n");

 term_refresh();
 while(exitflag==0 && waitmenu(1)==0)
	;// auf Benutzereingaben warten
 term_exit();
 free(summenformel);
 return 0;
}/* ende von main */
