PNG (Portable Network Graphics) Specification, Version 1.2

  ------------------------------------------------------------------------
Previous page
Next page
Table of contents
  ------------------------------------------------------------------------

15. Appendix: Sample CRC Code

The following sample code represents a practical implementation of the CRC
(Cyclic Redundancy Check) employed in PNG chunks. (See also ISO 3309
[ISO-3309] or ITU-T V.42 [ITU-T-V42] for a formal specification.)

The sample code is in the ANSI C programming language. Non C users may find
it easier to read with these hints:

&
     Bitwise AND operator.

^
     Bitwise exclusive-OR operator.

>>
     Bitwise right shift operator. When applied to an unsigned quantity, as
     here, right shift inserts zeroes at the left.

!
     Logical NOT operator.

++
     N++ increments the variable N.

0xNNN
     0x introduces a hexadecimal (base 16) constant. Suffix L indicates a
     long value (at least 32 bits).

  ------------------------------------------------------------------------

   /* Table of CRCs of all 8-bit messages. */
   unsigned long crc_table[256];

   /* Flag: has the table been computed? Initially false. */
   int crc_table_computed = 0;

   /* Make the table for a fast CRC. */
   void make_crc_table(void)
   {
     unsigned long c;
     int n, k;

     for (n = 0; n < 256; n++) {
       c = (unsigned long) n;
       for (k = 0; k < 8; k++) {
         if (c & 1)
           c = 0xedb88320L ^ (c >> 1);
         else
           c = c >> 1;
       }
       crc_table[n] = c;
     }
     crc_table_computed = 1;
   }

   /* Update a running CRC with the bytes buf[0..len-1]--the CRC
      should be initialized to all 1's, and the transmitted value
      is the 1's complement of the final running CRC (see the
      crc() routine below)). */

   unsigned long update_crc(unsigned long crc, unsigned char *buf,
                            int len)
   {
     unsigned long c = crc;
     int n;

     if (!crc_table_computed)
       make_crc_table();
     for (n = 0; n < len; n++) {
       c = crc_table[(c ^ buf[n]) & 0xff] ^ (c >> 8);
     }
     return c;
   }

   /* Return the CRC of the bytes buf[0..len-1]. */
   unsigned long crc(unsigned char *buf, int len)
   {
     return update_crc(0xffffffffL, buf, len) ^ 0xffffffffL;
   }

  ------------------------------------------------------------------------
  ------------------------------------------------------------------------
Previous page
Next page
Table of contents
  ------------------------------------------------------------------------
